﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using Microsoft.AspNetCore.Razor.Language.Intermediate;
using Microsoft.CodeAnalysis.Text;
using Roslyn.Test.Utilities;
using Xunit;

namespace Microsoft.AspNetCore.Razor.Language.CodeGeneration;

public class DefaultDocumentWriterTest
{
    [Fact] // This test covers the whole process including actual hashing.
    public void WriteDocument_EndToEnd_WritesChecksumAndMarksAutoGenerated()
    {
        // Arrange
        var document = new DocumentIntermediateNode();

        var codeDocument = TestRazorCodeDocument.CreateEmpty();
        var options = RazorCodeGenerationOptions.Default;

        var target = CodeTarget.CreateDefault(codeDocument, options);
        var writer = new DefaultDocumentWriter(target, options);

        // Act
        var result = writer.WriteDocument(codeDocument, document);

        // Assert
        var csharp = result.Text.ToString();
        AssertEx.AssertEqualToleratingWhitespaceDifferences(
            """
            #pragma checksum "test.cshtml" "{8829d00f-11b8-4213-878b-770e8597ac16}" "f1945cd6c19e56b3c1c78943ef5ec18116907a4ca1efc40a57d48ab1db7adfc5"
            // <auto-generated/>
            #pragma warning disable 1591
            #pragma warning restore 1591

            """,
            csharp);
    }

    [Fact]
    public void WriteDocument_SHA1_WritesChecksumAndMarksAutoGenerated()
    {
        // Arrange
        var sourceText = SourceText.From("", checksumAlgorithm: SourceHashAlgorithm.Sha1);
        var source = RazorSourceDocument.Create(sourceText, RazorSourceDocumentProperties.Create("test.cshtml", null));

        var projectEngine = RazorProjectEngine.CreateEmpty();
        var codeDocument = projectEngine.CreateCodeDocument(source);

        var document = new DocumentIntermediateNode();

        var options = codeDocument.CodeGenerationOptions;
        var target = CodeTarget.CreateDefault(codeDocument, options);
        var writer = new DefaultDocumentWriter(target, options);

        // Act
        var result = writer.WriteDocument(codeDocument, document);

        // Assert
        var csharp = result.Text.ToString();
        AssertEx.AssertEqualToleratingWhitespaceDifferences(
            """
            #pragma checksum "test.cshtml" "{ff1816ec-aa5e-4d10-87f7-6f4963833460}" "da39a3ee5e6b4b0d3255bfef95601890afd80709"
            // <auto-generated/>
            #pragma warning disable 1591
            #pragma warning restore 1591

            """,
            csharp);
    }

    [Fact]
    public void WriteDocument_SHA256_WritesChecksumAndMarksAutoGenerated()
    {
        // Arrange
        var sourceText = SourceText.From("", checksumAlgorithm: SourceHashAlgorithm.Sha256);
        var source = RazorSourceDocument.Create(sourceText, RazorSourceDocumentProperties.Create("test.cshtml", null));

        var projectEngine = RazorProjectEngine.CreateEmpty();
        var codeDocument = projectEngine.CreateCodeDocument(source);

        var document = new DocumentIntermediateNode();

        var options = codeDocument.CodeGenerationOptions;
        var target = CodeTarget.CreateDefault(codeDocument, options);
        var writer = new DefaultDocumentWriter(target, options);

        // Act
        var result = writer.WriteDocument(codeDocument, document);

        // Assert
        var csharp = result.Text.ToString();
        AssertEx.AssertEqualToleratingWhitespaceDifferences(
            """
            #pragma checksum "test.cshtml" "{8829d00f-11b8-4213-878b-770e8597ac16}" "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855"
            // <auto-generated/>
            #pragma warning disable 1591
            #pragma warning restore 1591

            """,
            csharp);
    }

    [Fact]
    public void WriteDocument_Empty_SuppressChecksumTrue_DoesnotWriteChecksum()
    {
        // Arrange
        var document = new DocumentIntermediateNode();

        var codeDocument = TestRazorCodeDocument.CreateEmpty();
        var options = RazorCodeGenerationOptions.Default.WithFlags(suppressChecksum: true);

        var target = CodeTarget.CreateDefault(codeDocument, options);
        var writer = new DefaultDocumentWriter(target, options);

        // Act
        var result = writer.WriteDocument(codeDocument, document);

        // Assert
        var csharp = result.Text.ToString();
        AssertEx.AssertEqualToleratingWhitespaceDifferences(
@"// <auto-generated/>
#pragma warning disable 1591
#pragma warning restore 1591
",
            csharp);
    }

    [Fact]
    public void WriteDocument_WritesNamespace()
    {
        // Arrange
        var document = new DocumentIntermediateNode();
        var builder = IntermediateNodeBuilder.Create(document);
        builder.Add(new NamespaceDeclarationIntermediateNode()
        {
            Content = "TestNamespace",
        });

        var codeDocument = TestRazorCodeDocument.CreateEmpty();
        var options = RazorCodeGenerationOptions.Default;

        var target = CodeTarget.CreateDefault(codeDocument, options);
        var writer = new DefaultDocumentWriter(target, options);

        // Act
        var result = writer.WriteDocument(codeDocument, document);

        // Assert
        var csharp = result.Text.ToString();
        AssertEx.AssertEqualToleratingWhitespaceDifferences(
            """
            #pragma checksum "test.cshtml" "{8829d00f-11b8-4213-878b-770e8597ac16}" "f1945cd6c19e56b3c1c78943ef5ec18116907a4ca1efc40a57d48ab1db7adfc5"
            // <auto-generated/>
            #pragma warning disable 1591
            namespace TestNamespace
            {
                #line hidden
            }
            #pragma warning restore 1591

            """,
            csharp);
    }

    [Fact]
    public void WriteDocument_WritesClass()
    {
        // Arrange
        var document = new DocumentIntermediateNode();
        var builder = IntermediateNodeBuilder.Create(document);
        builder.Add(new ClassDeclarationIntermediateNode()
        {
            Modifiers = { "internal" },
            BaseType = new BaseTypeWithModel("TestBase"),
            Interfaces = [IntermediateNodeFactory.CSharpToken("IFoo"), IntermediateNodeFactory. CSharpToken("IBar")],
            TypeParameters = [
                new TypeParameter() { ParameterName = "TKey" },
                new TypeParameter() { ParameterName = "TValue" },
            ],
            ClassName = "TestClass"
        });

        var codeDocument = TestRazorCodeDocument.CreateEmpty();
        var options = RazorCodeGenerationOptions.Default;

        var target = CodeTarget.CreateDefault(codeDocument, options);
        var writer = new DefaultDocumentWriter(target, options);

        // Act
        var result = writer.WriteDocument(codeDocument, document);

        // Assert
        var csharp = result.Text.ToString();
        AssertEx.AssertEqualToleratingWhitespaceDifferences(
            """
            #pragma checksum "test.cshtml" "{8829d00f-11b8-4213-878b-770e8597ac16}" "f1945cd6c19e56b3c1c78943ef5ec18116907a4ca1efc40a57d48ab1db7adfc5"
            // <auto-generated/>
            #pragma warning disable 1591
            internal class TestClass<TKey,TValue> : TestBase, IFoo, IBar
            {
            }
            #pragma warning restore 1591

            """,
            csharp);
    }

    [Fact]
    public void WriteDocument_WithNullableContext_WritesClass()
    {
        // Arrange
        var document = new DocumentIntermediateNode();
        var builder = IntermediateNodeBuilder.Create(document);
        builder.Add(new ClassDeclarationIntermediateNode()
        {
            Modifiers = { "internal" },
            BaseType = new BaseTypeWithModel("TestBase"),
            Interfaces = [IntermediateNodeFactory.CSharpToken("IFoo"), IntermediateNodeFactory.CSharpToken("IBar")],
            TypeParameters = [
                new TypeParameter() { ParameterName = "TKey" },
                new TypeParameter() { ParameterName = "TValue" },
            ],
            ClassName = "TestClass",
            NullableContext = true
        });

        var codeDocument = TestRazorCodeDocument.CreateEmpty();
        var options = RazorCodeGenerationOptions.Default;

        var target = CodeTarget.CreateDefault(codeDocument, options);
        var writer = new DefaultDocumentWriter(target, options);

        // Act
        var result = writer.WriteDocument(codeDocument, document);

        // Assert
        var csharp = result.Text.ToString();
        AssertEx.AssertEqualToleratingWhitespaceDifferences(
            """
            #pragma checksum "test.cshtml" "{8829d00f-11b8-4213-878b-770e8597ac16}" "f1945cd6c19e56b3c1c78943ef5ec18116907a4ca1efc40a57d48ab1db7adfc5"
            // <auto-generated/>
            #pragma warning disable 1591
            #nullable restore
            internal class TestClass<TKey,TValue> : TestBase, IFoo, IBar
            #nullable disable
            {
            }
            #pragma warning restore 1591

            """,
            csharp);
    }

    [Fact]
    public void WriteDocument_WritesClass_ConstrainedGenericTypeParameters()
    {
        // Arrange
        var document = new DocumentIntermediateNode();
        var builder = IntermediateNodeBuilder.Create(document);
        builder.Add(new ClassDeclarationIntermediateNode()
        {
            Modifiers = { "internal" },
            BaseType = new BaseTypeWithModel("TestBase"),
            Interfaces = [IntermediateNodeFactory.CSharpToken("IFoo"), IntermediateNodeFactory.CSharpToken("IBar")],
            TypeParameters = [
                new TypeParameter() { ParameterName = "TKey", Constraints = "where TKey : class" },
                new TypeParameter() { ParameterName = "TValue", Constraints = "where TValue : class" },
            ],
            ClassName = "TestClass"
        });

        var codeDocument = TestRazorCodeDocument.CreateEmpty();
        var options = RazorCodeGenerationOptions.Default;

        var target = CodeTarget.CreateDefault(codeDocument, options);
        var writer = new DefaultDocumentWriter(target, options);

        // Act
        var result = writer.WriteDocument(codeDocument, document);

        // Assert
        var csharp = result.Text.ToString();
        AssertEx.AssertEqualToleratingWhitespaceDifferences(
            """
            #pragma checksum "test.cshtml" "{8829d00f-11b8-4213-878b-770e8597ac16}" "f1945cd6c19e56b3c1c78943ef5ec18116907a4ca1efc40a57d48ab1db7adfc5"
            // <auto-generated/>
            #pragma warning disable 1591
            internal class TestClass<TKey,TValue> : TestBase, IFoo, IBar
            where TKey : class
            where TValue : class
            {
            }
            #pragma warning restore 1591

            """,
            csharp);
    }

    [Fact]
    public void WriteDocument_WritesMethod()
    {
        // Arrange
        var document = new DocumentIntermediateNode();
        var builder = IntermediateNodeBuilder.Create(document);
        builder.Add(new MethodDeclarationIntermediateNode()
        {
            Modifiers = { "internal", "virtual", "async", },
            MethodName = "TestMethod",
            Parameters =
            {
                new MethodParameter()
                {
                    Modifiers = { "readonly", "ref" },
                    ParameterName = "a",
                    TypeName = "int"
                },
                new MethodParameter()
                {
                    ParameterName = "b",
                    TypeName = "string"
                }
            },
            ReturnType = "string"
        });

        var codeDocument = TestRazorCodeDocument.CreateEmpty();
        var options = RazorCodeGenerationOptions.Default;

        var target = CodeTarget.CreateDefault(codeDocument, options);
        var writer = new DefaultDocumentWriter(target, options);

        // Act
        var result = writer.WriteDocument(codeDocument, document);

        // Assert
        var csharp = result.Text.ToString();
        AssertEx.AssertEqualToleratingWhitespaceDifferences(
            """
            #pragma checksum "test.cshtml" "{8829d00f-11b8-4213-878b-770e8597ac16}" "f1945cd6c19e56b3c1c78943ef5ec18116907a4ca1efc40a57d48ab1db7adfc5"
            // <auto-generated/>
            #pragma warning disable 1591
            #pragma warning disable 1998
            internal virtual async string TestMethod(readonly ref int a, string b)
            {
            }
            #pragma warning restore 1998
            #pragma warning restore 1591

            """,
            csharp);
    }

    [Fact]
    public void WriteDocument_WritesField()
    {
        // Arrange
        var document = new DocumentIntermediateNode();
        var builder = IntermediateNodeBuilder.Create(document);
        builder.Add(new FieldDeclarationIntermediateNode()
        {
            Modifiers = { "internal", "readonly", },
            FieldName = "_foo",
            FieldType = "string",
        });

        var codeDocument = TestRazorCodeDocument.CreateEmpty();
        var options = RazorCodeGenerationOptions.Default;

        var target = CodeTarget.CreateDefault(codeDocument, options);
        var writer = new DefaultDocumentWriter(target, options);

        // Act
        var result = writer.WriteDocument(codeDocument, document);

        // Assert
        var csharp = result.Text.ToString();
        AssertEx.AssertEqualToleratingWhitespaceDifferences(
            """
            #pragma checksum "test.cshtml" "{8829d00f-11b8-4213-878b-770e8597ac16}" "f1945cd6c19e56b3c1c78943ef5ec18116907a4ca1efc40a57d48ab1db7adfc5"
            // <auto-generated/>
            #pragma warning disable 1591
            internal readonly string _foo;
            #pragma warning restore 1591

            """,
            csharp);
    }

    [Fact]
    public void WriteDocument_WritesProperty()
    {
        // Arrange
        var document = new DocumentIntermediateNode();
        var builder = IntermediateNodeBuilder.Create(document);
        builder.Add(new PropertyDeclarationIntermediateNode()
        {
            Modifiers = { "internal", "virtual", },
            PropertyName = "Foo",
            PropertyType = IntermediateNodeFactory.CSharpToken("string"),
            PropertyExpression = "default"
        });

        var codeDocument = TestRazorCodeDocument.CreateEmpty();
        var options = RazorCodeGenerationOptions.Default;

        var target = CodeTarget.CreateDefault(codeDocument, options);
        var writer = new DefaultDocumentWriter(target, options);

        // Act
        var result = writer.WriteDocument(codeDocument, document);

        // Assert
        var csharp = result.Text.ToString();
        AssertEx.AssertEqualToleratingWhitespaceDifferences(
            """
            #pragma checksum "test.cshtml" "{8829d00f-11b8-4213-878b-770e8597ac16}" "f1945cd6c19e56b3c1c78943ef5ec18116907a4ca1efc40a57d48ab1db7adfc5"
            // <auto-generated/>
            #pragma warning disable 1591
            internal virtual string Foo => default;
            #pragma warning restore 1591

            """,
            csharp);
    }
}
