/*
  DB Mixer
  ========
  Description: 
    a DJ Mixer style GUI interface to the DBMix system.

	Copyright (c) 1999, 2000 Robert Michael S Dean

	Author: Robert Michael S Dean
	Version: 1.0


   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public Licensse as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

 */


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <limits.h>
#include <fcntl.h>
#include <signal.h>
#include <gtk/gtk.h>
#include <sys/shm.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <xmms/plugin.h>
#include <dbchannel.h>
#include <dbaudiolib.h>
#include <dbdebug.h>
#include <dbsoundcard.h>
#include <sys/time.h>
#include <glib.h>

#include "dbmixer.h"

local_channel * fader_ch1, * fader_ch2 = NULL;
/*GtkAdjustment * fader_adj1 = NULL,* fader_adj2 = NULL;*/
int fader_index1 = 0,fader_index2 = 0;
GtkAdjustment * fader_adj, * autofade_adj;


extern int * channel_indexes;
extern local_channel * local_channels;
extern dbfsd_data * sysdata;
extern channel_widgets * widgets;

GtkButton *punch_button_left, * punch_button_right;
int punch_button_left_id = 0, punch_button_right_id = 1;

#if 0
static const char * fadeleft_data[] = {
	"16 16 3 1",
	"       c None",
	".      c #000000000000",
	"X      c #0000FFFF0000",
	"................",
	".X..............",
	".XX.............",
	".XXX............",
	".XXXX...........",
	".XXXXX..........",
	".XXXXXX.........",
	".XXXXXXX........",
	".XXXXXXXX.......",
	".XXXXXXXXX......",
	".XXXXXXXXXX.....",
	".XXXXXXXXXXX....",
	".XXXXXXXXXXXX...",
	".XXXXXXXXXXXXX..",
	".XXXXXXXXXXXXXX.",
	"................"};

static const char * fadecenter_data[] = {
	"16 16 3 1",
	"       c None",
	".      c #000000000000",
	"X      c #0000FFFF0000",
	"................",
	"..............X.",
	".............XX.",
	"............XXX.",
	"...........XXXX.",
	"..........XXXXX.",
	".........XXXXXX.",
	"........XXXXXXX.",
	".......XXXXXXXX.",
	"......XXXXXXXXX.",
	".....XXXXXXXXXX.",
	"....XXXXXXXXXXX.",
	"...XXXXXXXXXXXX.",
	"..XXXXXXXXXXXXX.",
	".XXXXXXXXXXXXXX.",
	"................"};

static const char * faderight_data[] = {
	"16 16 3 1",
	"       c None",
	".      c #000000000000",
	"X      c #0000FFFF0000",
	"................",
	"..............X.",
	".............XX.",
	"............XXX.",
	"...........XXXX.",
	"..........XXXXX.",
	".........XXXXXX.",
	"........XXXXXXX.",
	".......XXXXXXXX.",
	"......XXXXXXXXX.",
	".....XXXXXXXXXX.",
	"....XXXXXXXXXXX.",
	"...XXXXXXXXXXXX.",
	"..XXXXXXXXXXXXX.",
	".XXXXXXXXXXXXXX.",
	"................"};

#endif

/***********************************************************************/
/*             autofade functions for crossfader                       */
/***********************************************************************/

gint fade_time_base;
float fade_time;

typedef struct autofade_s
{
	struct timeval start_tv;
	int count;
	int last;
	int start;
	int end;
	int direction;
} autofade;
 
autofade af;


/*
The following three functions implement auto crossfading.

This took about a week of banging my head against the wall before 
a friend (Ryan Bedwell) pointed out to me that linux is not a realtime 
OS.  The problem was that all crossfades took either 0.001, 2 or 4 seconds,
instead of being linearally mapped against the autofade spd. slider. 
(i.e. setting the slider to a value of 1.5 should have caused a fade of
3 seconds, but in reality took only 2.)

So the algorithm now samples the time at every iteration and will adjust
the loop counters and crossfade value appropriately if a timeout was missed.


11/12/00 - rewrote audofade callbacks so that they use gtk timeouts instead
of the good ole' reliable select() call.  Because gtk interrupts processing
each time new input was recieved, there was some fade jumping occuring. For
example, say you fade left to right(0 to 200).  When the fader hits 20, you
click the button for fade right to left. The fader in response, fades from 
20 back to 0. Becuase the first fade was interrupted to accomplish the
second fade, when the fader hits 0 and the right to left fade routine exits,
the first routine starts where it was interrupted and fades from 20 to 200.

I had solved this 95% with some loop flags and use of the volatile keyword
on shared variables, but the gtk_timeout method is more elegant, and works
100% of the time.  =)
 */

int autofade_callback(gpointer data)
{
	static struct timeval tv;
	static struct timezone tz;
	int  sleep_seconds;
	int  sleep_usec;
	int i;
	int time;
	unsigned int elapsed;

	/* get current fade speed */
	time = (int)((float)fade_time * (float)1000);
	sleep_seconds = (int)(time / 200) / 1000;
	sleep_usec = (int)(time/200) * 1000;

	if (sleep_usec == 0)
	{
		sleep_usec = 500;
	}

	if(af.count > 0)
	{
		/* get time */
		gettimeofday(&tv,&tz);

		/* get time elapsed since fade began */
		elapsed = ((tv.tv_sec - af.start_tv.tv_sec) * 1000 * 1000) +
			(tv.tv_usec - af.start_tv.tv_usec);
				
		i = elapsed / sleep_usec;
		
		/* update iterator */
		af.count -= (i - af.last);
		af.last = i;
		
		/* calculate new value and make sure it is in range */
		i = af.start + (af.direction * i);

		if(af.direction == 1)
		{
			if (i > af.end) i = af.end;
		}
		else
		{
			if(i < af.end) i = af.end;
		}

		/* set value */
		gtk_adjustment_set_value(GTK_ADJUSTMENT(fader_adj),i);
		while (gtk_events_pending()) gtk_main_iteration();	

		return TRUE;
	}
	else
	{
		af.last = 0;
		return FALSE;
	}
}



void autofade_left_to_right(void * arg)
{
	int faderpos = fader_adj->value;
	int  sleep_usec;
	int time;
	static struct timezone start_tz;
	
	time = (int)((float)autofade_adj->value * (float) (fade_time_base * 1000));
	
	sleep_usec = (time/200) % 1000;
			
	gettimeofday(&af.start_tv,&start_tz);
	af.count = 200 - faderpos;
	af.direction = 1;
	af.start = faderpos;
	af.last = 0;
	af.end = 200;

	send_msg(fader_ch2,DBMSG_UNMUTE,0);
	send_msg(fader_ch2,DBMSG_PLAY,0);

	gtk_timeout_add(sleep_usec,(GtkFunction)autofade_callback,&af);
		
	return;
}	


void autofade_right_to_left(void * arg)
{
	int faderpos = fader_adj->value;
	int  sleep_usec;
	int time;
	static struct timezone start_tz;
	
	time = (int)((float)autofade_adj->value * (float) (fade_time_base * 1000));
	
	sleep_usec = (time/200) % 1000;
		
	gettimeofday(&af.start_tv,&start_tz);
	af.count = faderpos;
	af.direction = -1;
	af.start = faderpos;
	af.last = 0;
	af.end = 0;

	send_msg(fader_ch1,DBMSG_UNMUTE,0);
	send_msg(fader_ch1,DBMSG_PLAY,0);

	gtk_timeout_add(sleep_usec,autofade_callback,&af);
		
	return;
}	


void autofade_to_center (void * arg)
{
	int faderpos = fader_adj->value;
	int  sleep_usec;
	int time;
	static struct timezone start_tz;

	if(faderpos == 100) return;

	time = (int)((float)autofade_adj->value * (float) (fade_time_base * 1000));
	
	sleep_usec = (time/200) % 1000;

	gettimeofday(&af.start_tv,&start_tz);

	af.end = 100;

	if (faderpos < 100)
	{
		af.count = 100 - faderpos;
		af.direction = 1;
		af.start = faderpos;
		af.last = 0;
	}
	else
	{
		af.count = faderpos - 100;
		af.direction = -1;
		af.start = faderpos;
		af.last = 0;
	}

	gtk_timeout_add(sleep_usec,autofade_callback,&af);
}




/***********************************************************************/
/*             callback functions for crossfader                       */
/***********************************************************************/

void update_fader()
{
	/* not implemented. */
}


void crossfader_scale_changed(GtkAdjustment * adj)
{
	int gain;

	if(adj->value < 100)
	{
		gain = ((float) (100 - widgets[fader_index2].adj->value) * (float)adj->value) / DBAUDIO_MAX_VOLUME;

	    /* convert to 0-128 scale */
		gain = (gain * DBAUDIO_INTERNAL_MAX_VOLUME) / DBAUDIO_MAX_VOLUME;

		fader_ch2->right_gain = fader_ch2->left_gain = gain;

		return;
	}

	if(adj->value == 100)
	{
	    gain = ((float) (100 - widgets[fader_index1].adj->value) * (float)adj->value) / DBAUDIO_MAX_VOLUME;

		gain = (gain * DBAUDIO_INTERNAL_MAX_VOLUME) / DBAUDIO_MAX_VOLUME;

		fader_ch2->right_gain = fader_ch2->left_gain = gain;

		fader_ch1->right_gain = fader_ch1->left_gain = gain;

		return;
	}

	if(adj->value > 100)
	{
		gain = normalize_scale(((float)(100 - widgets[fader_index1].adj->value) * (float)(100 - (adj->value - 100))) / DBAUDIO_MAX_VOLUME);

		gain = (gain * DBAUDIO_INTERNAL_MAX_VOLUME) / DBAUDIO_MAX_VOLUME;

		fader_ch1->right_gain = fader_ch1->left_gain = gain;

		return;
	}
}


void fader_left_select(GtkWidget *w, gint * data)
{
	gint index;

	index = (gint)*data;
	fader_ch1 = &(local_channels[index]);
	fader_index1 = index;
	crossfader_scale_changed(fader_adj);
}


void fader_right_select(GtkWidget *w, gint * data)
{
	gint index;

	index = (gint)*data;
	fader_ch2 = &(local_channels[index]);
	fader_index2 = index;
	crossfader_scale_changed(fader_adj);
}


void punch_button_pressed(GtkWidget *w, gchar* data)
{
	local_channel * ch;
	gint index;

	if(*data)
	{
		index = fader_index2;
	}
	else
	{
		index = fader_index1;
	}

	ch = &(local_channels[channel_indexes[index]]);	
	
	widgets[index].tempvol = ch->left_gain;
	ch->left_gain = ch->right_gain = 
		normalize_scale(100 - widgets[index].adj->value);
}


void punch_button_released(GtkWidget *w, gchar* data)
{
	local_channel * ch;
	gint index;

	if(*data)
	{
		index = fader_index2;
	}
	else
	{
		index = fader_index1;
	}

	ch = &(local_channels[channel_indexes[index]]);	
	
	ch->left_gain = ch->right_gain = widgets[index].tempvol;
}


void autofade_left_clicked(GtkWidget *w, gchar * data)
{
	autofade_right_to_left(NULL);
}


void autofade_center_clicked(GtkWidget *w, gchar * data)
{
	autofade_to_center(NULL);
}


void autofade_right_clicked(GtkWidget *w, gchar * data)
{
	autofade_left_to_right(NULL);
}

void autofade_scale_changed(GtkAdjustment * adj)
{
	fade_time = adj->value;
}


/***********************************************************************/
/*              widget creation functions for crossfader               */
/***********************************************************************/

void Create_Autofade_Scale(GtkBox * box)
{
	GtkWidget * auto_scale;
	GtkWidget * label;

	label = (GtkWidget *) gtk_label_new(AUTOFADE_STR);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(box),label,FALSE,FALSE,0);

	autofade_adj = (GtkAdjustment *) gtk_adjustment_new(fade_time_base,0.1,FADE_TIME_MAX,0.1,1.0,0.0);
	auto_scale = (GtkWidget *) gtk_hscale_new(GTK_ADJUSTMENT(autofade_adj));
	gtk_scale_set_digits((GtkScale*)auto_scale,1);

	gtk_signal_connect (GTK_OBJECT(autofade_adj), "value_changed", 
						GTK_SIGNAL_FUNC(autofade_scale_changed),NULL);

	gtk_box_pack_start(GTK_BOX(box),auto_scale,FALSE,FALSE,0);
	gtk_widget_show(auto_scale);
}


GtkWidget* Create_Fader(GtkWindow* win)
{
	GtkWidget * opt, * menu, * item;   /* pointer to fill in channel selector*/
	char str[20];
	int i;
	GtkWidget * fader_scale; /* cross fader slider */
	gint      * index;
	GtkWidget * fader_box;
	GtkWidget * label;
	GtkWidget * separator;

	fade_time = DEFAULT_FADE_TIME_BASE;

	fader_box = gtk_hbox_new(FALSE,0);
	gtk_widget_show(fader_box);

  	fader_adj = (GtkAdjustment *) gtk_adjustment_new(0.0,0.0,200.0,1.0,10.0,0.0);

	Debug("Creating crossfader left input selector...");

	/* add left punch button */
	{
	   punch_button_left = (GtkButton *) gtk_button_new();
	   label = gtk_label_new(PUNCH_STR);
	   gtk_container_add(GTK_CONTAINER(punch_button_left),label);
	   gtk_box_pack_start(GTK_BOX(fader_box),(GtkWidget*)punch_button_left,FALSE,FALSE,0);
	   gtk_signal_connect (GTK_OBJECT(punch_button_left), "pressed", 
						   GTK_SIGNAL_FUNC(punch_button_pressed),&(punch_button_left_id));

	   gtk_signal_connect (GTK_OBJECT(punch_button_left), "released", 
						   GTK_SIGNAL_FUNC(punch_button_released),&(punch_button_left_id));

	   gtk_widget_show(label);	   
	   gtk_widget_show((GtkWidget*)punch_button_left);
	}

	/* crossfader left side input select */
	{
		opt = gtk_option_menu_new();
		menu = gtk_menu_new();
		
		for(i = 0; i < sysdata->num_channels;i++)
		{
			index = (gint*) malloc(sizeof(gint));
			*index = i;
			sprintf(str,"%d",local_channels[i].index + 1);
			item = make_menu_item (str,GTK_SIGNAL_FUNC(fader_left_select),
								   index);

			gtk_menu_append(GTK_MENU (menu), item);
		}
		
		/* add option list to channel selector */
		gtk_option_menu_set_menu(GTK_OPTION_MENU (opt), menu);
		gtk_box_pack_start(GTK_BOX (fader_box), opt, FALSE, FALSE, 10);
		gtk_widget_show(opt);
	}

	Debug("Creating crossfader...");

	fader_scale = gtk_hscale_new(GTK_ADJUSTMENT(fader_adj));
	gtk_scale_set_digits((GtkScale*)fader_scale,0);
	
	gtk_box_pack_start(GTK_BOX(fader_box),fader_scale,TRUE,TRUE,10);
	gtk_widget_show(fader_scale);

	Debug("Creating crossfader right input selector...");

	/* crossfader right side input select */
	{
		opt = gtk_option_menu_new();
		menu = gtk_menu_new();
		
		/* add options to channel selector */
		for(i = 0; i < sysdata->num_channels;i++)
		{
			index = (gint*) malloc(sizeof(gint));
			*index = i;
			sprintf(str,"%d",local_channels[i].index + 1);
			item = make_menu_item(str,GTK_SIGNAL_FUNC(fader_right_select),
								   index);
			gtk_menu_append(GTK_MENU (menu), item);
		}
		
		/* add option list to channel selector */
		gtk_menu_set_active(GTK_MENU(menu),1);
		gtk_option_menu_set_menu(GTK_OPTION_MENU (opt), menu);
		gtk_box_pack_start (GTK_BOX (fader_box), opt, FALSE, FALSE, 10);
		gtk_widget_show(opt);
	}

	/* add right punch button */
	{
	   punch_button_right = (GtkButton *) gtk_button_new();
	   label = gtk_label_new(PUNCH_STR);
	   gtk_container_add(GTK_CONTAINER(punch_button_right),label);
	   gtk_box_pack_start(GTK_BOX(fader_box),(GtkWidget*)punch_button_right,FALSE,FALSE,0);
	   gtk_signal_connect (GTK_OBJECT(punch_button_right), "pressed", 
						   GTK_SIGNAL_FUNC(punch_button_pressed),&(punch_button_right_id));

	   gtk_signal_connect (GTK_OBJECT(punch_button_right), "released", 
						   GTK_SIGNAL_FUNC(punch_button_released),&(punch_button_right_id));

	   gtk_widget_show(label);	   
	   gtk_widget_show((GtkWidget*)punch_button_right);
	}

	gtk_signal_connect (GTK_OBJECT(fader_adj), "value_changed", 
						GTK_SIGNAL_FUNC(crossfader_scale_changed),NULL);

	separator = gtk_vseparator_new();
	gtk_box_pack_start(GTK_BOX(fader_box),separator,FALSE,FALSE,5);
	gtk_widget_show(separator);

	/* add autofade buttons */
	{
		GtkWidget * button;

		Debug("adding autofade buttons...");
		/* add fade left button */
		button = (GtkWidget *) gtk_button_new();
		label = gtk_label_new(" < ");
		gtk_container_add(GTK_CONTAINER(button),label);
		gtk_widget_show(label);
		gtk_widget_show(button);
		gtk_box_pack_start(GTK_BOX(fader_box),button,FALSE,FALSE,0);
		gtk_signal_connect (GTK_OBJECT(button), "clicked", 
							GTK_SIGNAL_FUNC(autofade_left_clicked),NULL);
		
		/* add fade center button */
		button = (GtkWidget *) gtk_button_new();
		label = gtk_label_new(" /\\ ");
		gtk_container_add(GTK_CONTAINER(button),label);
		gtk_widget_show(label);
		gtk_widget_show(button);
		gtk_box_pack_start(GTK_BOX(fader_box),button,FALSE,FALSE,0);

		gtk_signal_connect(GTK_OBJECT(button), "clicked", 
							GTK_SIGNAL_FUNC(autofade_center_clicked),NULL);

		/* add fade right button */
		button = (GtkWidget *) gtk_button_new();
		label = gtk_label_new(" > ");
		gtk_container_add(GTK_CONTAINER(button),label);
		gtk_widget_show(label);
		gtk_widget_show(button);
		gtk_box_pack_start(GTK_BOX(fader_box),button,FALSE,FALSE,0);

		gtk_signal_connect (GTK_OBJECT(button), "clicked", 
							GTK_SIGNAL_FUNC(autofade_right_clicked),NULL);
	}

	/* init fader to the first channel */
	fader_ch1 = &(local_channels[0]);
	fader_ch2 = &(local_channels[1]);
	fader_index1 = 0;
	fader_index2 = 1;

	return fader_box;
}
