#!/bin/bash

EASY_VERSION="1.1"

bb=`tput bold`
nn=`tput sgr0`

# help
if [ $# -eq 0 ]; then
   echo "${bb}-------------------------------------------------------------------${nn}"
   echo "${bb}Creates sof-file for kmo_illumination_flat-recipe and executes it.${nn}"
   echo " "
   echo "- The environment variable ${bb}KMOS_CALIB${nn} pointing to a path has to be"
   echo "  set in order to find the needed static calibration frame WAVE_BAND."
   echo "  Otherwise the execution of this script will fail!"
   echo "- If XCAL, YCAL and LCAL frames aren't existing in the working directory"
   echo "  the script is searching as well KMOS_CALIB in order to find them."
   echo "- As argument an arbitrary exposure of an executed "
   echo "  ${bb}KMOS_spec_cal_calunitflat${nn} template must be provided."
   echo "  All matching exposures are collected automatically (Only FLAT,LAMP at an"
   echo "  angle of 0deg)."
   echo "- (Optional) If as 2nd argument ${bb}sof${nn} is provided, then only"
   echo "  the sof-file is generated but not executed."
   echo " "
   echo "${bb}Usage:${nn}"
   echo "  easySPARK_illumination_flat.sh <filename with full path>"
   echo "    or"
   echo "  easySPARK_illumination_flat.sh <filename with full path> sof"
   echo " "
   echo "Version: "$EASY_VERSION
   echo "${bb}-------------------------------------------------------------------${nn}"
   exit
fi

# init
sofOnly=0
printVers=0

# extract all arguments beginning with KMOS, eventually imethod and sof
while [ "$1" ] ; do
    tmp=$(basename $1)
    if [[ $tmp == KMOS* ]]; then
        # file-check
        if [ ! -e $1 ]; then
            echo "${bb}ERROR:${nn} File doesn't exist! ($1)"
            exit
        fi
        fileName=$1
    else
        if [[ $tmp == sof ]]; then
            sofOnly=1
        elif [[ $tmp == vers ]]; then
            printVers=1
        fi
    fi
    shift
done

if [ $printVers = 1 ]; then
    echo $EASY_VERSION
    exit
fi

# check if gawk is existing
if command -v gawk >/dev/null; then
   :
else
    echo "'gawk' has not been found on your system!"
    OS=$(uname -s)
    if [ $OS != "Linux" ]; then
        echo "You seem using a MacOS, please use e.g. MacPorts to install gawk."
    fi
    echo "Aborting now.!"
fi

if [ -z "${KMOS_CALIB}" ]; then
   echo "${bb}ERROR:${nn} environment variable KMOS_CALIB is not defined! (see help)"
   exit
fi

#check if file is of correct template
tplId=$(dfits $fileName |\
        fitsort -d tpl.id |\
        gawk '{print $2}')
if [ $tplId != KMOS_spec_cal_calunitflat ]; then
   echo "${bb}ERROR${nn}: File is of wrong template: $tplId instead of KMOS_spec_cal_calunitflat !"
   exit
fi

# extract TPL.START of ths file
tplStartTime=$(dfits $fileName |\
               fitsort -d tpl.start |\
               gawk '{print $2}')

# get files with matching tplStartTime and FLAT_LAMP
dataDir=$(dirname "$fileName")
matchingFiles=$(dfits $dataDir/KMOS*.fits | \
                fitsort -d tpl.start dpr.type ocs.rot.naangle |\
       	        grep $tplStartTime |\
                grep FLAT,LAMP |\
                grep -P "\s+0.0" | \
                gawk '{print $1}' |\
                tr "\n" " ")

# populate sof-file with FLAT_SKY_FLAT frames
sofFile="illumination_flat_$tplStartTime.sof"
cat > $sofFile </dev/null
for file in $matchingFiles; do
    echo "$file                FLAT_SKY_FLAT" >> $sofFile
done

# get grating id of first FLAT_SKY_FLAT frame
gratingUpper=$(dfits $matchingFiles |\
               fitsort -d ins.grat1.id |\
               head -n1 |\
               gawk '{print $2}')

# add static calibration frames
echo "$KMOS_CALIB/kmos_wave_band.fits       WAVE_BAND" >> $sofFile

# add band-specific calibration frames
fnXcal=xcal_$gratingUpper$gratingUpper$gratingUpper.fits
fnYcal=ycal_$gratingUpper$gratingUpper$gratingUpper.fits
fnLcal=lcal_$gratingUpper$gratingUpper$gratingUpper.fits

if [ ! -e $fnXcal ]; then
   fnXcal=$KMOS_CALIB"/"$fnXcal
fi
if [ ! -e $fnYcal ]; then
   fnYcal=$KMOS_CALIB"/"$fnYcal
fi
if [ ! -e $fnLcal ]; then
   fnLcal=$KMOS_CALIB"/"$fnLcal
fi

echo "$fnXcal            XCAL" >> $sofFile
echo "$fnYcal            YCAL" >> $sofFile
echo "$fnLcal            LCAL" >> $sofFile

echo ""
echo "${bb}------------------ easySPARK_illumination_flat ---------------------${nn}"
echo "${bb}INPUT${nn}"
echo "filename:  "$fileName
echo "with "
echo "TPL.START: "$tplStartTime
echo "GRATING:   "$gratingUpper
echo "EXPTIME:   "$exptime
echo ""
echo "${bb}OUTPUT${nn}"
echo $sofFile" with:"
cat $sofFile
echo ""

# execute kmo_illumination_flat
if [ $sofOnly = 0 ]; then
   echo "${bb}NOW CALCULATING ILLUMINATION_FLAT CORRECTION:${nn}"
   echo "esorex kmo_illumination_flat "$sofFile
   
   esorex kmo_illumination_flat $sofFile
fi
echo "${bb}----------------------------------------------------------${nn}"
