/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/
#ifndef ARRAY_H
#define ARRAY_H

#include <vector>
#include <cstddef>
#include "compare.h"

template <typename T>
using Varray = std::vector<T>;

template <typename T>
using Varray2D = std::vector<std::vector<T>>;

template <typename T>
using Varray3D = std::vector<std::vector<std::vector<T>>>;

template <typename T>
using Varray4D = std::vector<std::vector<std::vector<std::vector<T>>>>;

#define VARRAY_2D(T, P2D, N, M) \
  Varray2D<T> P2D(N);           \
  if ((N))                      \
    for (size_t i = 0; i < (size_t)(N); ++i) P2D[i].resize(M);

#define IX2D(y, x, nx) ((y) * (nx) + (x))

template <typename T>
inline void
varrayFree(std::vector<T> &v)
{
  v.clear();
  v.shrink_to_fit();
}

/*
template <class T, size_t ROW, size_t COL<
using Matrix = std::array<std::array<T, COL>, ROW>;
*/

// clang-format off
#define MADDMN(x, y) (DBL_IS_EQUAL((x), missval1) || DBL_IS_EQUAL((y), missval2) ? missval1 : (x) + (y))
#define MSUBMN(x, y) (DBL_IS_EQUAL((x), missval1) || DBL_IS_EQUAL((y), missval2) ? missval1 : (x) - (y))
#define MMULMN(x, y) (DBL_IS_EQUAL((x), 0.) || DBL_IS_EQUAL((y), 0.) \
                      ? 0  : DBL_IS_EQUAL((x), missval1) || DBL_IS_EQUAL((y), missval2) ? missval1 : (x) * (y))
#define MDIVMN(x, y) (DBL_IS_EQUAL((x), missval1) || DBL_IS_EQUAL((y), missval2) || DBL_IS_EQUAL((y), 0.) ? missval1 : (x) / (y))
#define MPOWMN(x, y) (DBL_IS_EQUAL((x), missval1) || DBL_IS_EQUAL((y), missval2) ? missval1 : std::pow((x), (y)))
#define MSQRTMN(x) (DBL_IS_EQUAL((x), missval1) || (x) < 0 ? missval1 : std::sqrt(x))

#define ADD(x, y) ((x) + (y))
#define SUB(x, y) ((x) - (y))
#define MUL(x, y) ((x) * (y))
#define DIV(x, y) (IS_EQUAL((y), 0.) ? missval1 : (x) / (y))
#define POW(x, y) std::pow((x), (y))
#define SQRT(x) std::sqrt(x)

#define ADDM(x, y) (IS_EQUAL((x), missval1) || IS_EQUAL((y), missval2) ? missval1 : (x) + (y))
#define SUBM(x, y) (IS_EQUAL((x), missval1) || IS_EQUAL((y), missval2) ? missval1 : (x) - (y))
#define MULM(x, y)                            \
  (IS_EQUAL((x), 0.) || IS_EQUAL((y), 0.) ? 0 \
                                          : IS_EQUAL((x), missval1) || IS_EQUAL((y), missval2) ? missval1 : (x) * (y))
#define DIVM(x, y) (IS_EQUAL((x), missval1) || IS_EQUAL((y), missval2) || IS_EQUAL((y), 0.) ? missval1 : (x) / (y))
#define POWM(x, y) (IS_EQUAL((x), missval1) || IS_EQUAL((y), missval2) ? missval1 : std::pow((x), (y)))
#define SQRTM(x) (IS_EQUAL((x), missval1) || (x) < 0 ? missval1 : std::sqrt(x))

#define ADDMN(x, y) FADDMN(x, y, missval1, missval2)
#define SUBMN(x, y) FSUBMN(x, y, missval1, missval2)
#define MULMN(x, y) FMULMN(x, y, missval1, missval2)
#define DIVMN(x, y) FDIVMN(x, y, missval1, missval2)
#define POWMN(x, y) FPOWMN(x, y, missval1, missval2)
#define SQRTMN(x) FSQRTMN(x, missval1)

static inline double FADDMN(const double x, const double y, const double missval1, const double missval2)
{
  return MADDMN(x, y);
}
static inline double FSUBMN(const double x, const double y, const double missval1, const double missval2)
{
  return MSUBMN(x, y);
}
static inline double FMULMN(const double x, const double y, const double missval1, const double missval2)
{
  return MMULMN(x, y);
}
static inline double FDIVMN(const double x, const double y, const double missval1, const double missval2)
{
  return MDIVMN(x, y);
}
static inline double FPOWMN(const double x, const double y, const double missval1, const double missval2)
{
  return MPOWMN(x, y);
}
static inline double FSQRTMN(const double x, const double missval1)
{
  return MSQRTMN(x);
}
// clang-format on

const char *fpe_errstr(int fpeRaised);

void varrayMinMaxSum(size_t len, const Varray<double> &array, double &rmin, double &rmax, double &rsum);
void varrayMinMaxSum_f(size_t len, const Varray<float> &array, double &rmin, double &rmax, double &rsum);
size_t varrayMinMaxSumMV(size_t len, const Varray<double> &array, double missval, double &rmin, double &rmax, double &rsum);
size_t varrayMinMaxSumMV_f(size_t len, const Varray<float> &array, double missval, double &rmin, double &rmax, double &rsum);
void varrayMinMaxMean(size_t len, const Varray<double> &array, double &rmin, double &rmax, double &rmean);
size_t varrayMinMaxMeanMV(size_t len, const Varray<double> &array, double missval, double &rmin, double &rmax, double &rmean);

void arrayMinMaxMask(size_t len, const double *array, const Varray<int> &mask, double &rmin, double &rmax);

void arrayAddArray(size_t len, double *array1, const double *array2);
void arrayAddArrayMV(size_t len, double *array1, const double *array2, double missval);

template <typename T>
void
varrayFill(const size_t len, T *v, const T value)
{
  for (size_t i = 0; i < len; ++i) v[i] = value;
}

template <typename T>
void
varrayFill(Varray<T> &v, const T value)
{
  varrayFill(v.size(), v.data(), value);
}

template <typename T>
void
varrayFill(const size_t len, Varray<T> &v, const T value)
{
  varrayFill(len, v.data(), value);
}

template <typename T>
void
arrayCopy(const size_t len, const T *array1, T *array2)
{
  for (size_t i = 0; i < len; ++i) array2[i] = array1[i];
}

template <typename T>
void
varrayCopy(const size_t len, const T &array1, T &array2)
{
  for (size_t i = 0; i < len; ++i) array2[i] = array1[i];
}

size_t arrayNumMV(size_t len, const double *array, double missval);
size_t varrayNumMV(size_t len, const Varray<double> &array, double missval);
size_t varrayNumMV_f(size_t len, const Varray<float> &array, double missval);

std::pair<double, double> varrayMinMax(size_t len, const Varray<double> &v);
std::pair<double, double> varrayMinMax(size_t len, const double *array);
std::pair<double, double> varrayMinMax(const Varray<double> &v);
std::pair<double, double> varrayMinMaxMV(size_t len, const Varray<double> &array, double missval);
std::pair<double, double> varrayMinMaxMV(size_t len, const double *array, double missval);
std::pair<double, double> varrayMinMaxMV(size_t len, const double *array, double missval, size_t &nvals);

double varrayMin(size_t len, const Varray<double> &v);
double varrayMax(size_t len, const Varray<double> &v);
double varrayRange(size_t len, const Varray<double> &v);
double varrayMinMV(size_t len, const Varray<double> &v, double missval);
double varrayMaxMV(size_t len, const Varray<double> &v, double missval);
double varrayRangeMV(size_t len, const Varray<double> &v, double missval);

double varraySum(size_t len, const Varray<double> &v);
double varraySumMV(size_t len, const Varray<double> &v, double missval);

double varrayMean(size_t len, const Varray<double> &v);
double varrayMeanMV(size_t len, const Varray<double> &v, double missval);
double varrayWeightedMean(size_t len, const Varray<double> &v, const Varray<double> &w, double missval);
double varrayWeightedMeanMV(size_t len, const Varray<double> &v, const Varray<double> &w, double missval);

double varrayAvgMV(size_t len, const Varray<double> &v, double missval);
double varrayWeightedAvgMV(size_t len, const Varray<double> &v, const Varray<double> &w, double missval);

#endif  //  ARRAY_H
