/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Remapeta     remapeta          Model to model level interpolation
*/

#include <cdi.h>

#include "dmemory.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "readline.h"
#include "hetaeta.h"
#include "after_vertint.h"
#include "stdnametable.h"
#include "util_string.h"
#include "timer.h"
#include "const.h"
#include "cdo_options.h"
#include "cdo_zaxis.h"
#include "cdi_lockedIO.h"

static void
setmissval(long nvals, const Varray<int> &imiss, double missval, double *array)
{
  if (!imiss.empty())
    for (long i = 0; i < nvals; ++i)
      if (imiss[i]) array[i] = missval;
}

static void
corr_hum(long gridsize, double *q, double q_min)
{
  for (long i = 0; i < gridsize; ++i)
    {
      if (q[i] < q_min) q[i] = q_min;
    }
}

static long
ncctop(double cptop, long nlev, long nlevp1, const double *vct_a, const double *vct_b)
{
  /*
    Description:
    Defines highest level *ncctop* where condensation is allowed.

    Author:

    E. Roeckner, MPI, October 2001
  */
  long nctop = 0;
  Varray<double> zph(nlevp1), zp(nlev);
  // double    cptop  =  1000.;   /* min. pressure level for cond. */

  /* half level pressure values, assuming 101320. Pa surface pressure */

  for (long jk = 0; jk < nlevp1; ++jk)
    {
      const double za = vct_a[jk];
      const double zb = vct_b[jk];
      zph[jk] = za + zb * 101320.;
    }

  /* full level pressure */

  for (long jk = 0; jk < nlev; ++jk) zp[jk] = (zph[jk] + zph[jk + 1]) * 0.5;

  /* search for pressure level cptop (Pa) */

  for (long jk = 0; jk < nlev; ++jk)
    {
      nctop = jk;
      if (zp[jk] >= cptop) break;
    }

  return nctop;
}

static void
vctFromFile(const char *filename, int *nvct, Varray<double> &vct2)
{
  char line[1024], *pline;
  int i = 0;
  constexpr int maxvct = 8192;

  auto fp = fopen(filename, "r");
  if (fp == nullptr)
    {
      perror(filename);
      exit(EXIT_FAILURE);
    }

  vct2.resize(maxvct);

  while (readline(fp, line, 1024))
    {
      if (line[0] == '#' || line[0] == '\0') continue;

      pline = line;
      int num = (int) strtod(pline, &pline);
      if (pline == nullptr) cdoAbort("Format error in VCT file %s!", filename);
      if (num != i) cdoWarning("Inconsistent VCT file, entry %d is %d.", i, num);

      if (i + maxvct / 2 >= maxvct - 1) cdoAbort("Too many values in VCT file!");

      vct2[i] = strtod(pline, &pline);
      if (pline == nullptr) cdoAbort("Format error in VCT file %s!", filename);

      vct2[i + maxvct / 2] = strtod(pline, &pline);

      i++;
    }

  fclose(fp);

  const int nvct2 = 2 * i;
  const int nlevh2 = i - 1;

  for (i = 0; i < nlevh2 + 1; ++i) vct2[i + nvct2 / 2] = vct2[i + maxvct / 2];

  vct2.resize(nvct2);

  *nvct = nvct2;
}

static void
vertSum(Varray<double> &sum, const double *var3d, size_t gridsize, size_t nlevel)
{
  for (size_t i = 0; i < gridsize; ++i) sum[i] = 0;

  for (size_t k = 0; k < nlevel; ++k)
    for (size_t i = 0; i < gridsize; ++i)
      {
        sum[i] += var3d[k * gridsize + i];
      }
}

static void
vertSumw(Varray<double> &sum, const double *var3d, size_t gridsize, size_t nlevel, const Varray<double> &deltap)
{
  for (size_t i = 0; i < gridsize; ++i) sum[i] = 0;

  for (size_t k = 0; k < nlevel; ++k)
    for (size_t i = 0; i < gridsize; ++i)
      {
        sum[i] += var3d[k * gridsize + i] * deltap[k * gridsize + i];
      }
}

void
vlist_hybrid_vct(int vlistID, int *rzaxisIDh, int *rnvct, Varray<double> &vct, int *rnhlevf)
{
  int zaxisIDh = -1;
  int nhlevf = 0;
  int nvct = 0;

  bool lhavevct = false;
  const int nzaxis = vlistNzaxis(vlistID);
  for (int i = 0; i < nzaxis; i++)
    {
      const auto zaxisID = vlistZaxis(vlistID, i);
      const auto nlevel = zaxisInqSize(zaxisID);

      if (zaxisInqType(zaxisID) == ZAXIS_HYBRID && nlevel > 1)
        {
          nvct = zaxisInqVctSize(zaxisID);
          if (nlevel == (nvct / 2 - 1))
            {
              if (!lhavevct)
                {
                  lhavevct = true;
                  zaxisIDh = zaxisID;
                  nhlevf = nlevel;

                  vct.resize(nvct);
                  zaxisInqVct(zaxisID, vct.data());
                }
            }
          else
            {
              if (Options::cdoVerbose) cdoPrint("nlevel = (nvct1/2 - 1): nlevel = %d", nlevel);
              if (nlevel < (nvct / 2 - 1))
                cdoPrint("z-axis %d has only %d of %d hybrid sigma pressure levels!", i + 1, nlevel, (nvct / 2 - 1));
            }
        }
    }

  *rzaxisIDh = zaxisIDh;
  *rnvct = nvct;
  *rnhlevf = nhlevf;
}

#define MAX_VARS3D 1024

void *
Remapeta(void *process)
{
  size_t nfis2gp = 0;
  int nrecs;
  int i;
  int iv;
  int varID, levelID;
  int nvars3D = 0;
  int sgeopotID = -1, tempID = -1, sqID = -1, psID = -1, lnpsID = -1;
  char varname[CDI_MAX_NAME], stdname[CDI_MAX_NAME];
  double *single2;
  Varray<double> fis2;
  double *t1 = nullptr, *q1 = nullptr;
  double *t2 = nullptr, *q2 = nullptr;
  double *tscor = nullptr, *pscor = nullptr, *secor = nullptr;
  size_t nmiss, nmissout = 0;
  bool ltq = false;
  bool lfis2 = false;
  int varids[MAX_VARS3D];
  Varray<int> imiss;
  int timer_hetaeta = 0;
  double **vars1 = nullptr, **vars2 = nullptr;
  double minval, maxval;
  double missval = 0;
  double cconst = 1.E-6;
  double cptop = 0; /* min. pressure level for cond. */

  if (Options::Timer) timer_hetaeta = timer_new("Remapeta_hetaeta");

  cdoInitialize(process);

  // clang-format off
  const auto REMAPETA  = cdoOperatorAdd("remapeta",   0, 0, "VCT file name");
  const auto REMAPETAS = cdoOperatorAdd("remapeta_s", 0, 0, "VCT file name");
  const auto REMAPETAZ = cdoOperatorAdd("remapeta_z", 0, 0, "VCT file name");
  // clang-format on

  const int operatorID = cdoOperatorID();

  operatorInputArg(cdoOperatorEnter(operatorID));

  const char *envstr = getenv("REMAPETA_PTOP");
  if (envstr)
    {
      double fval = atof(envstr);
      if (fval > 0)
        {
          cptop = fval;
          //	  if ( Options::cdoVerbose )
          cdoPrint("Set REMAPETA_PTOP to %g", cptop);
        }
    }

  int nvct2 = 0;
  Varray<double> vct2;
  vctFromFile(cdoOperatorArgv(0).c_str(), &nvct2, vct2);
  const int nhlevf2 = nvct2 / 2 - 1;

  const double *a2 = vct2.data();
  const double *b2 = vct2.data() + nvct2 / 2;

  if (Options::cdoVerbose)
    for (i = 0; i < nhlevf2 + 1; ++i) cdoPrint("vct2: %5d %25.17f %25.17f", i, vct2[i], vct2[nvct2 / 2 + i]);

  const auto streamID1 = cdoOpenRead(0);

  if (operatorArgc() == 2)
    {
      lfis2 = true;

      const char *fname = cdoOperatorArgv(1).c_str();
      int streamID = streamOpenReadLocked(fname);
      const auto vlistID1 = streamInqVlist(streamID);

      streamInqRecord(streamID, &varID, &levelID);
      const auto gridID = vlistInqVarGrid(vlistID1, varID);
      nfis2gp = gridInqSize(gridID);

      fis2.resize(nfis2gp);

      streamReadRecord(streamID, fis2.data(), &nmiss);

      if (nmiss)
        {
          missval = vlistInqVarMissval(vlistID1, varID);
          imiss.resize(nfis2gp);
          for (size_t i = 0; i < nfis2gp; ++i) imiss[i] = DBL_IS_EQUAL(fis2[i], missval);

          nmissout = nmiss;
        }

      // check range of surface_geopotential
      arrayMinMaxMask(nfis2gp, fis2.data(), imiss, minval, maxval);
      if (minval < MIN_FIS || maxval > MAX_FIS)
        cdoWarning("%s out of range (min=%g max=%g)!", var_stdname(surface_geopotential), minval, maxval);

      if (minval < -1.e10 || maxval > 1.e10) cdoAbort("%s out of range!", var_stdname(surface_geopotential));

      streamClose(streamID);
    }

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto gridID = vlistGrid(vlistID1, 0);
  if (gridInqType(gridID) == GRID_SPECTRAL) cdoAbort("Spectral data unsupported!");

  const auto gridsize = vlist_check_gridsize(vlistID1);

  const auto zaxisID2 = zaxisCreate(ZAXIS_HYBRID, nhlevf2);

  {
    Varray<double> lev2(nhlevf2);
    for (int i = 0; i < nhlevf2; ++i) lev2[i] = i + 1;
    zaxisDefLevels(zaxisID2, lev2.data());
  }

  if (nvct2 == 0) cdoAbort("Internal problem, vct2 undefined!");
  zaxisDefVct(zaxisID2, nvct2, vct2.data());

  auto surfaceID = zaxisFromName("surface");

  int zaxisIDh = -1;
  int nvct1 = 0;
  int nhlevf1 = 0;
  Varray<double> vct1;
  vlist_hybrid_vct(vlistID1, &zaxisIDh, &nvct1, vct1, &nhlevf1);

  vlist_change_hybrid_zaxis(vlistID1, vlistID2, zaxisIDh, zaxisID2);

  const auto nzaxis = vlistNzaxis(vlistID1);
  for (int i = 0; i < nzaxis; i++)
    {
      const auto zaxisID = vlistZaxis(vlistID1, i);
      const auto nlevel = zaxisInqSize(zaxisID);
      if (zaxisInqType(zaxisID) == ZAXIS_HYBRID && nlevel == 1) vlistChangeZaxisIndex(vlistID2, i, surfaceID);
    }

  const double *a1 = vct1.data();
  const double *b1 = vct1.data() + nvct1 / 2;
  if (Options::cdoVerbose)
    for (i = 0; i < nvct1 / 2; ++i) cdoPrint("vct1: %5d %25.17f %25.17f", i, vct1[i], vct1[nvct1 / 2 + i]);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  if (zaxisIDh == -1) cdoWarning("No 3D variable with hybrid sigma pressure coordinate found!");

  const auto nvars = vlistNvars(vlistID1);

  for (varID = 0; varID < nvars; varID++)
    {
      const auto gridID = vlistInqVarGrid(vlistID1, varID);
      const auto zaxisID = vlistInqVarZaxis(vlistID1, varID);
      const auto nlevel = zaxisInqSize(zaxisID);

      auto code = vlistInqVarCode(vlistID1, varID);
      /* code = -1; */
      if (code <= 0 || code == 255)
        {
          vlistInqVarName(vlistID1, varID, varname);
          cstrToLowerCase(varname);

          vlistInqVarStdname(vlistID1, varID, stdname);
          cstrToLowerCase(stdname);

          code = echamcode_from_stdname(stdname);

          if (code == -1)
            {
              //                                            ECHAM                         ECMWF
              // clang-format off
              if (sgeopotID == -1 && (cstrIsEqual(varname, "geosp") || cstrIsEqual(varname, "z"))) code = 129;
              else if (tempID == -1 && (cstrIsEqual(varname, "st") || cstrIsEqual(varname, "t"))) code = 130;
              else if (psID == -1 && (cstrIsEqual(varname, "aps") || cstrIsEqual(varname, "ps"))) code = 134;
              else if (lnpsID == -1 && (cstrIsEqual(varname, "lsp") || cstrIsEqual(varname, "lnsp"))) code = 152;
              else if (sqID == -1 && (cstrIsEqual(varname, "q"))) code = 133;
              // clang-format on
            }
        }

      // clang-format off
      if      (code == 129 && nlevel == 1) sgeopotID = varID;
      else if (code == 134 && nlevel == 1) psID = varID;
      else if (code == 152 && nlevel == 1) lnpsID = varID;
      else if (code == 130 && nlevel == nhlevf1) tempID = varID;
      else if (code == 133 && nlevel == nhlevf1) sqID = varID;
      // clang-format on

      if (gridInqType(gridID) == GRID_SPECTRAL && zaxisInqType(zaxisID) == ZAXIS_HYBRID)
        cdoAbort("Spectral data on model level unsupported!");

      if (gridInqType(gridID) == GRID_SPECTRAL) cdoAbort("Spectral data unsupported!");

      if (zaxisInqType(zaxisID) == ZAXIS_HYBRID && zaxisIDh != -1 && nlevel == nhlevf1)
        {
          if (!(code == 130 || code == 133)) varids[nvars3D++] = varID;
        }
      else
        {
          if (code == 130) tempID = -1;
          if (code == 133) sqID = -1;
        }
    }

  if (Options::cdoVerbose)
    {
      cdoPrint("Found:");
      if (tempID != -1) cdoPrint("  %s", var_stdname(air_temperature));
      if (psID != -1) cdoPrint("  %s", var_stdname(surface_air_pressure));
      if (lnpsID != -1) cdoPrint("  LOG(%s)", var_stdname(surface_air_pressure));
      if (sgeopotID != -1) cdoPrint("  %s", var_stdname(surface_geopotential));
      if (sqID != -1) cdoPrint("  %s", var_stdname(specific_humidity));
    }

  if (tempID != -1 && sqID != -1)
    {
      ltq = true;
    }
  else
    {
      if (tempID != -1) cdoAbort("Temperature without humidity unsupported!");
      if (sqID != -1) cdoAbort("Humidity without temperature unsupported!");
    }
  /*
  if ( ltq == false )
    {
      cdoWarning("Temperature and Humidity not found!");
    }
  */
  if (operatorID == REMAPETA)
    {
    }

  Varray<double> sum1, sum2;
  if (operatorID == REMAPETAS || operatorID == REMAPETAZ)
    {
      sum1.resize(gridsize);
      sum2.resize(gridsize);
    }

  Varray<double> deltap1, deltap2;
  Varray<double> half_press1, half_press2;
  if (operatorID == REMAPETAZ)
    {
      deltap1.resize(gridsize * nhlevf1);
      deltap2.resize(gridsize * nhlevf2);
      half_press1.resize(gridsize * (nhlevf1 + 1));
      half_press2.resize(gridsize * (nhlevf2 + 1));
    }

  Varray<double> array(gridsize);

  Varray<double> fis1(gridsize);
  Varray<double> ps1(gridsize);

  if (!lfis2) fis2.resize(gridsize);
  if (lfis2 && gridsize != nfis2gp) cdoAbort("Orographies have different grid size!");

  Varray<double> ps2(gridsize);

  if (ltq)
    {
      tscor = (double *) Malloc(gridsize * sizeof(double));
      pscor = (double *) Malloc(gridsize * sizeof(double));
      secor = (double *) Malloc(gridsize * sizeof(double));

      t1 = (double *) Malloc(gridsize * nhlevf1 * sizeof(double));
      q1 = (double *) Malloc(gridsize * nhlevf1 * sizeof(double));

      t2 = (double *) Malloc(gridsize * nhlevf2 * sizeof(double));
      q2 = (double *) Malloc(gridsize * nhlevf2 * sizeof(double));
    }

  if (nvars3D)
    {
      vars1 = (double **) Malloc(nvars * sizeof(double *));
      vars2 = (double **) Malloc(nvars * sizeof(double *));

      for (varID = 0; varID < nvars3D; ++varID)
        {
          vars1[varID] = (double *) Malloc(gridsize * nhlevf1 * sizeof(double));
          vars2[varID] = (double *) Malloc(gridsize * nhlevf2 * sizeof(double));
        }
    }

  if (zaxisIDh != -1 && sgeopotID == -1)
    {
      if (ltq) cdoWarning("%s not found - set to zero!", var_stdname(surface_geopotential));

      varrayFill(fis1, 0.0);
    }

  int presID = lnpsID;
  if (zaxisIDh != -1 && lnpsID == -1)
    {
      if (psID == -1)
        cdoAbort("%s not found!", var_stdname(surface_air_pressure));
      else
        presID = psID;
    }

  if (Options::cdoVerbose)
    {
      if (presID == lnpsID)
        cdoPrint("using LOG(%s)", var_stdname(surface_air_pressure));
      else
        cdoPrint("using %s", var_stdname(surface_air_pressure));
    }

  if (Options::cdoVerbose) cdoPrint("nvars3D = %d   ltq = %d", nvars3D, (int) ltq);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          const auto zaxisID = vlistInqVarZaxis(vlistID1, varID);
          const auto nlevel = zaxisInqSize(zaxisID);
          const auto offset = gridsize * levelID;
          cdoReadRecord(streamID1, array.data(), &nmiss);

          if (zaxisIDh != -1)
            {
              if (varID == sgeopotID)
                varrayCopy(gridsize, array, fis1);
              else if (varID == presID)
                {
                  if (lnpsID != -1)
                    for (size_t i = 0; i < gridsize; ++i) ps1[i] = std::exp(array[i]);
                  else if (psID != -1)
                    varrayCopy(gridsize, array, ps1);
                }
              else if (ltq && varID == tempID)
                arrayCopy(gridsize, array.data(), t1 + offset);
              else if (ltq && varID == sqID)
                arrayCopy(gridsize, array.data(), q1 + offset);
              /* else if ( zaxisID == zaxisIDh ) */
              else if (zaxisInqType(zaxisID) == ZAXIS_HYBRID && nlevel == nhlevf1)
                {
                  for (i = 0; i < nvars3D; ++i)
                    if (varID == varids[i]) break;

                  if (i == nvars3D) cdoAbort("Internal error, 3D variable not found!");

                  arrayCopy(gridsize, array.data(), vars1[i] + offset);
                }
              else
                {
                  cdoDefRecord(streamID2, varID, levelID);
                  cdoWriteRecord(streamID2, array.data(), nmiss);
                }
            }
          else
            {
              cdoDefRecord(streamID2, varID, levelID);
              cdoWriteRecord(streamID2, array.data(), nmiss);
            }
        }

      if (zaxisIDh != -1)
        {
          // check range of ps_prog
          arrayMinMaxMask(gridsize, ps1.data(), imiss, minval, maxval);
          if (minval < MIN_PS || maxval > MAX_PS) cdoWarning("Surface pressure out of range (min=%g max=%g)!", minval, maxval);

          // check range of geop
          arrayMinMaxMask(gridsize, fis1.data(), imiss, minval, maxval);
          if (minval < MIN_FIS || maxval > MAX_FIS) cdoWarning("Orography out of range (min=%g max=%g)!", minval, maxval);
        }

      if (!lfis2)
        for (size_t i = 0; i < gridsize; i++) fis2[i] = fis1[i];

      if (ltq)
        {
          varID = tempID;
          int nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
          for (levelID = 0; levelID < nlevel; levelID++)
            {
              const auto offset = gridsize * levelID;
              single2 = t1 + offset;

              arrayMinMaxMask(gridsize, single2, imiss, minval, maxval);
              if (minval < MIN_T || maxval > MAX_T)
                cdoWarning("Input temperature at level %d out of range (min=%g max=%g)!", levelID + 1, minval, maxval);
            }

          varID = sqID;
          nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
          for (levelID = 0; levelID < nlevel; levelID++)
            {
              const auto offset = gridsize * levelID;
              single2 = q1 + offset;

              corr_hum(gridsize, single2, MIN_Q);

              arrayMinMaxMask(gridsize, single2, imiss, minval, maxval);
              if (minval < MIN_Q || maxval > MAX_Q)
                cdoWarning("Input humidity at level %d out of range (min=%g max=%g)!", levelID + 1, minval, maxval);
            }
        }

      if (nvars3D || ltq)
        {
          if (Options::Timer) timer_start(timer_hetaeta);
          hetaeta(ltq, gridsize, imiss.data(), nhlevf1, a1, b1, fis1.data(), ps1.data(), t1, q1, nhlevf2, a2, b2, fis2.data(),
                  ps2.data(), t2, q2, nvars3D, vars1, vars2, tscor, pscor, secor);
          if (Options::Timer) timer_stop(timer_hetaeta);
        }

      const long nctop = (cptop > 0) ? ncctop(cptop, (long) nhlevf2, (long) nhlevf2 + 1, a2, b2) : 0;

      if (zaxisIDh != -1 && sgeopotID != -1)
        {
          varID = sgeopotID;
          levelID = 0;
          setmissval(gridsize, imiss, missval, fis2.data());
          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, fis2.data(), nmissout);
        }

      if (zaxisIDh != -1 && lnpsID != -1)
        for (size_t i = 0; i < gridsize; ++i) ps2[i] = std::log(ps2[i]);

      if (zaxisIDh != -1 && presID != -1)
        {
          varID = presID;
          levelID = 0;
          setmissval(gridsize, imiss, missval, ps2.data());
          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, ps2.data(), nmissout);
        }

      if (ltq)
        {
          varID = tempID;
          int nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID2, varID));
          for (levelID = 0; levelID < nlevel; levelID++)
            {
              const auto offset = gridsize * levelID;
              single2 = t2 + offset;

              arrayMinMaxMask(gridsize, single2, imiss, minval, maxval);
              if (minval < MIN_T || maxval > MAX_T)
                cdoWarning("Output temperature at level %d out of range (min=%g max=%g)!", levelID + 1, minval, maxval);

              setmissval(gridsize, imiss, missval, single2);
              cdoDefRecord(streamID2, varID, levelID);
              cdoWriteRecord(streamID2, single2, nmissout);
            }

          varID = sqID;
          nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID2, varID));
          for (levelID = 0; levelID < nlevel; levelID++)
            {
              const auto offset = gridsize * levelID;
              single2 = q2 + offset;

              corr_hum(gridsize, single2, MIN_Q);

              if (levelID < nctop)
                for (size_t i = 0; i < gridsize; ++i) single2[i] = cconst;

              arrayMinMaxMask(gridsize, single2, imiss, minval, maxval);
              if (minval < MIN_Q || maxval > MAX_Q)
                cdoWarning("Output humidity at level %d out of range (min=%g max=%g)!", levelID + 1, minval, maxval);

              setmissval(gridsize, imiss, missval, single2);
              cdoDefRecord(streamID2, varID, levelID);
              cdoWriteRecord(streamID2, single2, nmissout);
            }
        }

      for (iv = 0; iv < nvars3D; ++iv)
        {
          varID = varids[iv];

          const auto nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID2, varID));

          if (operatorID == REMAPETAS)
            {
              vertSum(sum1, vars1[iv], gridsize, nhlevf1);
              vertSum(sum2, vars2[iv], gridsize, nhlevf2);
            }
          else if (operatorID == REMAPETAZ)
            {
              presh(nullptr, half_press1.data(), vct1.data(), ps1.data(), nhlevf1, gridsize);
              for (int k = 0; k < nhlevf1; ++k)
                for (size_t i = 0; i < gridsize; ++i)
                  {
                    deltap1[k * gridsize + i] = half_press1[(k + 1) * gridsize + i] - half_press1[k * gridsize + i];
                    deltap1[k * gridsize + i] = std::log(deltap1[k * gridsize + i]);
                  }
              vertSumw(sum1, vars1[iv], gridsize, nhlevf1, deltap1);

              presh(nullptr, half_press2.data(), vct2.data(), ps1.data(), nhlevf2, gridsize);
              for (int k = 0; k < nhlevf2; ++k)
                for (size_t i = 0; i < gridsize; ++i)
                  {
                    deltap2[k * gridsize + i] = half_press2[(k + 1) * gridsize + i] - half_press2[k * gridsize + i];
                    deltap2[k * gridsize + i] = std::log(deltap2[k * gridsize + i]);
                  }
              vertSumw(sum2, vars2[iv], gridsize, nhlevf2, deltap2);
            }

          for (levelID = 0; levelID < nlevel; levelID++)
            {
              const size_t offset = gridsize * levelID;
              single2 = vars2[iv] + offset;

              if (operatorID == REMAPETAS || operatorID == REMAPETAZ)
                {
                  /*
                  for ( size_t i = 0; i < gridsize; ++i )
                    if ( i %100 == 0 )
                      printf("%d %g %g %g %g %g\n",i, single2[i], sum1[i],
                  sum2[i], sum1[i]/sum2[i], single2[i]*sum1[i]/sum2[i]);
                  */
                  for (size_t i = 0; i < gridsize; ++i) single2[i] = single2[i] * sum1[i] / sum2[i];
                }

              setmissval(gridsize, imiss, missval, single2);
              cdoDefRecord(streamID2, varID, levelID);
              cdoWriteRecord(streamID2, single2, nmissout);
            }
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  if (nvars3D)
    {
      for (varID = 0; varID < nvars3D; varID++)
        {
          Free(vars2[varID]);
          Free(vars1[varID]);
        }
      Free(vars2);
      Free(vars1);
    }

  if (ltq)
    {
      Free(q2);
      Free(t2);
      Free(q1);
      Free(t1);
      Free(secor);
      Free(pscor);
      Free(tscor);
    }

  cdoFinish();

  return nullptr;
}
