/**
 * Copyright (C) 2009-2013 Paul Fretwell - aka 'Footleg' (drfootleg@gmail.com)
 * 
 * This file is part of Cave Converter.
 * 
 * Cave Converter is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Cave Converter is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Cave Converter.  If not, see <http://www.gnu.org/licenses/>.
 */
package footleg.cavesurvey.data.model;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import footleg.cavesurvey.cmdline.CaveConverter;

/**
 * Class representing a complete cave survey data model. This can consist of 
 * one or more cave surveys.
 *  
 * @author      Footleg <drfootleg@gmail.com>
 * @version     2013.07.18                                (ISO 8601 YYYY.MM.DD)
 * @since       1.6                                       (The Java version used)
 */
public class CaveSurvey {
	
	private List<SurveySeries> survey;
	
	/**
	 * 
	 */
	public CaveSurvey() {
		//Create new list of survey series to hold data
		survey = new ArrayList<SurveySeries>();
	}

	public int size() {
		return survey.size();
	}

	public boolean isEmpty() {
		return survey.isEmpty();
	}

	public boolean contains(Object o) {
		return survey.contains(o);
	}

	public Iterator<SurveySeries> iterator() {
		return survey.listIterator();
	}

	public boolean add(SurveySeries e) {
		return survey.add(e);
	}

	public boolean remove(SurveySeries o) {
		return survey.remove(o);
	}

	public void clear() {
		survey.clear();
	}

	public SurveySeries get(int index) {
		return survey.get(index);
	}

	public SurveySeries set(int index, SurveySeries element) {
		return survey.set(index, element);
	}

	public void add(int index, SurveySeries element) {
		survey.add(index, element);
	}

	public SurveySeries remove(int index) {
		return survey.remove(index);
	}

	public ListIterator<SurveySeries> listIterator() {
		return survey.listIterator();
	}

	public ListIterator<SurveySeries> listIterator(int index) {
		return survey.listIterator(index);
	}

	/**
	 * Generates LRUD data from splays for all legs in all series
	 */
	public void generateLRUDfromSplays() {
		CaveConverter.logMessage("Generating LRUD data from splays...");
		
		//Loop through all series
		ListIterator<SurveySeries> seriesIterator = survey.listIterator();
		while ( seriesIterator.hasNext() ) {
			SurveySeries series = seriesIterator.next();

			//Process series and then recursively call inner series
			processLRUDfromSplays( series );
		}
	}

	/**
	 * Process the legs in this series to generate LRUD data from splays,
	 * then loop through all inner series recursively to process them too.
	 * @param series The survey series to process
	 */
	private void processLRUDfromSplays( SurveySeries series ) {
		//Process series and then recursively call inner series
		series.generateLRUDFromSplays();
		
		//TODO Fix the way the option to remove splays used for LRUD is triggered, and don't do it by default.
		//series.removeSplaysUsedForLRUD(); 
		
		//Loop through all inner series
		ListIterator<SurveySeries> seriesIterator = series.getInnerSeriesList().listIterator();
		while ( seriesIterator.hasNext() ) {
			SurveySeries innerSeries = seriesIterator.next();
			processLRUDfromSplays( innerSeries );
		}
	}

}
