//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Tests/Functional/Consistence/ConsistenceTests.cpp
//! @brief     Implements Core consistence tests through gtest macros.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

// These tests check whether pairs of reference data sets, generated by
// some other tests, coincide within reasonable numeric error margins.

#include "Device/Data/Datafield.h"
#include "Device/Histo/SimulationResult.h"
#include "Sample/StandardSamples/CylindersAndPrismsBuilder.h"
#include "Sample/StandardSamples/PlainMultiLayerBySLDBuilder.h"
#include "Sim/Simulation/ScatteringSimulation.h"
#include "Sim/Simulation/SpecularSimulation.h"
#include "Tests/GTestWrapper/google_test.h"
#include "Tests/SimFactory/MakeSimulations.h"
#include <string>

int compareTwoReferences(const std::string& name0, const std::string& name1, double limit);

TEST(Consistence, SpecularWithSlicing)
{
    EXPECT_TRUE(compareTwoReferences("SpecularWithSlicing1", "SpecularWithSlicing2", 2e-10));
    EXPECT_TRUE(compareTwoReferences("SpecularWithSlicing1", "SpecularWithSlicing3", 2e-10));
}

TEST(Consistence, InstrumentDefinitionComparison)
{
    EXPECT_TRUE(compareTwoReferences("InstrumentDefinitionComparison_0",
                                     "InstrumentDefinitionComparison_Q", 2e-10));
}

TEST(Consistence, TOFResolutionComparison)
{
    EXPECT_TRUE(
        compareTwoReferences("TOFResolutionComparison_TP", "TOFResolutionComparison_TR", 2e-10));
}

TEST(Consistence, PolarizedQAngleReflectivity)
{
    EXPECT_TRUE(compareTwoReferences("BasicSpecularPP", "PolarizedQAngleReflectivityPP_Q", 2e-10));
    EXPECT_TRUE(compareTwoReferences("BasicSpecularMM", "PolarizedQAngleReflectivityMM_Q", 2e-10));
}

TEST(Consistence, PolarizedScalarSpinFlip)
{
    auto* sample = ExemplarySamples::createPlainMultiLayerBySLD();
    auto simulation = test::makeSimulation::BasicYPolarizedSpecular(*sample, "PM", false);
    const SimulationResult result = simulation->simulate();
    for (auto r : result.flatVector())
        EXPECT_EQ(r, 0);
}

TEST(Consistence, PolarizedScalarSpinFlipParticles)
{
    auto* sample = ExemplarySamples::createCylindersAndPrisms();
    auto simulation = test::makeSimulation::MiniZPolarizedGISAS(*sample, "PM");
    const SimulationResult result = simulation->simulate();
    for (auto r : result.flatVector())
        EXPECT_EQ(r, 0);
}
