//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Mask/MaskEditorToolbar.cpp
//! @brief     Implements class MaskEditorToolbar
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Mask/MaskEditorToolbar.h"
#include "Base/Util/Assert.h"
#include "GUI/View/Mask/MaskEditorActions.h"
#include "GUI/View/Tool/mainwindow_constants.h"
#include <QButtonGroup>
#include <QLabel>
#include <QRadioButton>
#include <QStyle>
#include <QToolButton>
#include <QVariant>

MaskEditorToolbar::MaskEditorToolbar(MaskEditorActions* editorActions, QWidget* parent)
    : QToolBar(parent)
    , m_editorActions(editorActions)
    , m_activityButtonGroup(new QButtonGroup(this))
    , m_maskValueGroup(new QButtonGroup(this))
{
    setIconSize(QSize(GUI::Constants::toolbar_icon_size, GUI::Constants::toolbar_icon_size));
    setProperty("_q_custom_style_disabled", QVariant(true));

    setup_selection_group();
    //    setup_maskvalue_group();
    setup_shapes_group();
    setup_maskmodify_group();
    setup_extratools_group();

    connect(m_activityButtonGroup, &QButtonGroup::idClicked, this,
            &MaskEditorToolbar::onActivityGroupChange);
    connect(m_maskValueGroup, &QButtonGroup::idClicked, this,
            &MaskEditorToolbar::onMaskValueGroupChange);

    m_previousActivity = currentActivity();
}

//! Handles ZOOM requests from MaskGraphicsView while user press and holds
//! space bar. As soon as space bar is released, activity is returned to previous state.
void MaskEditorToolbar::onChangeActivityRequest(MaskEditorFlags::Activity value)
{
    if (value == MaskEditorFlags::PREVIOUS_MODE)
        setCurrentActivity(m_previousActivity);
    else {
        m_previousActivity = currentActivity();
        setCurrentActivity(value);
    }
    emit activityModeChanged(currentActivity());
}

void MaskEditorToolbar::onActivityGroupChange(int value)
{
    Q_UNUSED(value);
    emit activityModeChanged(currentActivity());
}

void MaskEditorToolbar::onMaskValueGroupChange(int value)
{
    Q_UNUSED(value);
    emit maskValueChanged(MaskEditorFlags::MaskValue(value));
}

void MaskEditorToolbar::onPresentationTypePressed()
{
    emit presentationTypeRequest(MaskEditorFlags::MASK_PRESENTER);
}

void MaskEditorToolbar::onPresentationTypeReleased()
{
    emit presentationTypeRequest(MaskEditorFlags::MASK_EDITOR);
}

void MaskEditorToolbar::setup_selection_group()
{
    auto* panButton = new QToolButton(this);
    panButton->setIcon(QIcon(":/images/hand-right.svg"));
    panButton->setToolTip("Pan/zoom mode (space)");
    panButton->setCheckable(true);
    panButton->setChecked(true);
    addWidget(panButton);

    auto* resetViewButton = new QToolButton(this);
    resetViewButton->setIcon(QIcon(":/images/camera-metering-center.svg"));
    resetViewButton->setToolTip("Reset pan/zoom to initial state");
    addWidget(resetViewButton);
    connect(resetViewButton, &QToolButton::clicked, m_editorActions,
            &MaskEditorActions::resetViewRequest);

    add_separator();

    auto* selectionButton = new QToolButton(this);
    selectionButton->setIcon(QIcon(":/Mask/images/maskeditor_arrow.svg"));
    selectionButton->setToolTip("Select/modify mask");
    selectionButton->setCheckable(true);
    addWidget(selectionButton);

    m_activityButtonGroup->addButton(panButton, MaskEditorFlags::PAN_ZOOM_MODE);
    m_activityButtonGroup->addButton(selectionButton, MaskEditorFlags::SELECTION_MODE);
}

void MaskEditorToolbar::setup_maskvalue_group()
{
    addWidget(new QLabel(" "));

    auto* maskTrueButton = new QRadioButton(this);
    maskTrueButton->setText("masked");
    maskTrueButton->setToolTip("Set mask to True (area is excluded from the simulation)");
    maskTrueButton->setCheckable(true);
    maskTrueButton->setChecked(true);
    addWidget(maskTrueButton);

    addWidget(new QLabel(" "));

    auto* maskFalseButton = new QRadioButton(this);
    maskFalseButton->setToolTip("Set mask to False (area stays in the simulation)");
    maskFalseButton->setCheckable(true);
    maskFalseButton->setText("active");
    addWidget(maskFalseButton);

    m_maskValueGroup->addButton(maskTrueButton, MaskEditorFlags::MASK_ON);
    m_maskValueGroup->addButton(maskFalseButton, MaskEditorFlags::MASK_OFF);
}

void MaskEditorToolbar::setup_shapes_group()
{
    auto* roiButton = new QToolButton(this);
    roiButton->setIcon(QIcon(":/Mask/images/maskeditor_roi.svg"));
    roiButton->setToolTip("Create region of interest");
    roiButton->setCheckable(true);
    addWidget(roiButton);

    auto* rectangleButton = new QToolButton(this);
    rectangleButton->setIcon(QIcon(":/Mask/images/maskeditor_rectangle.svg"));
    rectangleButton->setToolTip("Create rectangle mask");
    rectangleButton->setCheckable(true);
    addWidget(rectangleButton);

    auto* polygonButton = new QToolButton(this);
    polygonButton->setIcon(QIcon(":/Mask/images/maskeditor_polygon.svg"));
    polygonButton->setToolTip("Create polygon mask");
    polygonButton->setCheckable(true);
    addWidget(polygonButton);

    auto* verticalLineButton = new QToolButton(this);
    verticalLineButton->setIcon(QIcon(":/Mask/images/maskeditor_verticalline.svg"));
    verticalLineButton->setToolTip("Create vertical line mask");
    verticalLineButton->setCheckable(true);
    addWidget(verticalLineButton);

    auto* horizontalLineButton = new QToolButton(this);
    horizontalLineButton->setIcon(QIcon(":/Mask/images/maskeditor_horizontalline.svg"));
    horizontalLineButton->setToolTip("Create horizontal line mask");
    horizontalLineButton->setCheckable(true);
    addWidget(horizontalLineButton);

    auto* ellipseButton = new QToolButton(this);
    ellipseButton->setIcon(QIcon(":/Mask/images/maskeditor_ellipse.svg"));
    ellipseButton->setToolTip("Create ellipse mask");
    ellipseButton->setCheckable(true);
    addWidget(ellipseButton);

    //    QToolButton *maskAllButton = new QToolButton(this);
    //    maskAllButton->setIcon(QIcon(":/Mask/images/maskeditor_maskall.svg"));
    //    maskAllButton->setToolTip("Create masked area covering whole detector plane\n"
    //                              "Will be placed beneath all masks. Only one instance is
    //                              allowed.");
    //    maskAllButton->setCheckable(true);
    //    addWidget(maskAllButton);

    m_activityButtonGroup->addButton(roiButton, MaskEditorFlags::ROI_MODE);
    m_activityButtonGroup->addButton(rectangleButton, MaskEditorFlags::RECTANGLE_MODE);
    m_activityButtonGroup->addButton(polygonButton, MaskEditorFlags::POLYGON_MODE);
    m_activityButtonGroup->addButton(verticalLineButton, MaskEditorFlags::VERTICAL_LINE_MODE);
    m_activityButtonGroup->addButton(horizontalLineButton, MaskEditorFlags::HORIZONTAL_LINE_MODE);
    m_activityButtonGroup->addButton(ellipseButton, MaskEditorFlags::ELLIPSE_MODE);
    //    m_activityButtonGroup->addButton(maskAllButton, MaskEditorFlags::MASKALL_MODE);
    add_separator();
}

void MaskEditorToolbar::setup_maskmodify_group()
{
    ASSERT(m_editorActions);
    addAction(m_editorActions->bringToFrontAction());
    addAction(m_editorActions->sendToBackAction());
    add_separator();
}

void MaskEditorToolbar::setup_extratools_group()
{
    auto* presentationButton = new QToolButton(this);
    presentationButton->setIcon(QIcon(":/Mask/images/maskeditor_lightbulb.svg"));
    presentationButton->setToolTip("Press and hold to see mask results.");
    addWidget(presentationButton);

    connect(presentationButton, &QToolButton::pressed, this,
            &MaskEditorToolbar::onPresentationTypePressed);
    connect(presentationButton, &QToolButton::released, this,
            &MaskEditorToolbar::onPresentationTypeReleased);

    auto* propertyPanelButton = new QToolButton(this);
    propertyPanelButton->setIcon(QIcon(":/Mask/images/maskeditor_toolpanel.svg"));
    propertyPanelButton->setToolTip("Open panel with additional properties");
    addWidget(propertyPanelButton);

    connect(propertyPanelButton, &QToolButton::clicked, m_editorActions,
            &MaskEditorActions::propertyPanelRequest);
    add_separator();
}

void MaskEditorToolbar::add_separator()
{
    addWidget(new QLabel(" "));
    addSeparator();
    addWidget(new QLabel(" "));
}

MaskEditorFlags::Activity MaskEditorToolbar::currentActivity() const
{
    return MaskEditorFlags::EActivityType(m_activityButtonGroup->checkedId());
}

void MaskEditorToolbar::setCurrentActivity(MaskEditorFlags::Activity value)
{
    m_activityButtonGroup->button(value)->setChecked(true);
}
