!{\src2tex{textfont=tt}}
!!****f* ABINIT/setmesh
!!
!! NAME
!! setmesh
!!
!! FUNCTION
!! Calculate the size of the FFT grid for the GW calculation.
!!
!! COPYRIGHT
!! Copyright (C) 1999-2007 ABINIT group (GMR, VO, LR, RWG, YMN)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  gmet(3,3) = reciprocal space metric.
!!  gvec(3,max(npwwfn,npwsigx)) = G-vectors array.
!!  npwsigx = size of the dielectric or self-energy matrix.
!!  npwwfn = number of G-vectors in the wavefunctions.
!!
!! OUTPUT
!!  n1 ,n2, n3  = size of FFT grid.
!!  n1d,n2d,n3d = size of the augmented FFT grid.
!!  nr = n1d*n2d*n3d = number of points on the augmented FFT grid.
!!
!! NOTE
!!  In principle, the FFT grid can be augmented to improve the efficiency of the FFT
!!   (see ~abinit/doc/input_variables/vargs.htm#ngfft).
!!  This is now actually done "on the fly" in the GW code (see e.g. fftwfn.f), so that setmesh
!!  always returns n1d=n1, n2d=n2, and n3d=n3.
!!
!!  Three methods have been coded for the calculation of the mesh (see parameter "method" below):
!!    method=1 roughly takes the FFT box that encloses the larger of the two spheres of radius
!!      aliasing_factor*rwfn and rsigx, where rwfn ans rsigx are the radius of the spheres
!!      with npwwfn and npwsigx planewaves respectively.
!!    method=2 calculates the optimal FFT grid that allows aliasing only outside the sphere of the
!!      npwsigx planewaves (finer than method=1 with aliasing_factor=1).
!!    method=3 calculates the FFT grid needed to expand the density.
!!      (even finer than method=2, roughly corresponds to method=1 with aliasing_factor=2).
!!  At the present time method=1 is enforced. The use of method=2 or method=3 requires a change
!!  in the call to setmesh in screening.F90 and sigma.F90 (see these routines).
!!
!! PARENTS
!!      mrgscr,screening,sigma
!!
!! CHILDREN
!!      sizefft,wrtout
!!
!! SOURCE
#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine setmesh(gmet,gvec,n1,n2,n3,n1d,n2d,n3d,npwsigx,npwwfn,nr,method)

 use defs_basis

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi
 use interfaces_15gw, except_this_one => setmesh
#endif
!End of the abilint section

 implicit none
 
!Arguments ------------------------------------
!scalars
 integer,intent(in) :: npwsigx,npwwfn,method
 integer,intent(out) :: n1,n1d,n2,n2d,n3,n3d,nr
!arrays
 integer,intent(in) :: gvec(3,max(npwwfn,npwsigx))
 real(dp),intent(in) :: gmet(3,3)
 
!Local variables ------------------------------
!scalars
 integer,parameter :: aliasing_factor=1  
! integer,parameter ::method=1
 integer :: ig,ig1,ig1max,ig2,ig2max,ig3,ig3max,m1,m2,m3,mm1,mm2,mm3
 real(dp) :: ecuteff,ecutsigx,ecutwfn,g1,g2,g3,gsq,gsqmax,reff,rsigx,rwfn
 character(len=500) :: message
 
!*************************************************************************

 !calculate the limits of the sphere of npwwfn G-vectors in each direction
 m1=-1
 m2=-1
 m3=-1
 do ig=1,npwwfn
  m1=max(m1,abs(gvec(1,ig)))
  m2=max(m2,abs(gvec(2,ig)))
  m3=max(m3,abs(gvec(3,ig)))
 end do

 write(message,'(a,i5,a,3i4)') ' setmesh: npwwfn = ',npwwfn,' ; m1,m2,m3 = ',m1,m2,m3
 call wrtout(6,message,'COLL')

 !calculate the limits of the sphere of npwsigx G-vectors in each direction
 mm1=-1
 mm2=-1
 mm3=-1
 do ig=1,npwsigx
  mm1=max(mm1,abs(gvec(1,ig)))
  mm2=max(mm2,abs(gvec(2,ig)))
  mm3=max(mm3,abs(gvec(3,ig)))
 end do

 !ensure that G+G0 will fit into the FFT grid, where G is any of the
 !npwsigx vectors and G0 is (i,j,k) [-2<i,j,k<2]
 mm1=mm1+2
 mm2=mm2+2
 mm3=mm3+2

 write(message,'(a,i5,a,3i4)')' setmesh: npweps/npwsigx = ',npwsigx,' ; mm1,mm2,mm3 = ',mm1,mm2,mm3
 call wrtout(6,message,'COLL')

 if (method==1) then

  m1=m1*aliasing_factor
  m2=m2*aliasing_factor
  m3=m3*aliasing_factor

 else if ((method==2).or.(method==3)) then

! calculate the radius of the sphere of npwwfn G-vectors.
! of course we may directly input ecutwfn and not recalculate it...
  ecutwfn=-1.0
  do ig=1,npwwfn
   g1=real(gvec(1,ig))
   g2=real(gvec(2,ig))
   g3=real(gvec(3,ig))
   gsq=gmet(1,1)*g1**2+gmet(2,2)*g2**2+gmet(3,3)*g3**2+ &
&                 2.0*(gmet(1,2)*g1*g2+gmet(1,3)*g1*g3+gmet(2,3)*g2*g3)
   ecutwfn=max(ecutwfn,gsq)
  end do
  rwfn=sqrt(ecutwfn)
  ecutwfn=2.0*ecutwfn*pi**2

  write(6,'(a,f7.3,a)') ' setmesh: calculated ecutwfn  = ',ecutwfn,' [Ha]'

! calculate the radius of the sphere of npwsigx G-vectors.
! of course we may directly input ecutsigx and not recalculate it...
  ecutsigx=-1.0
  do ig=1,npwsigx
   g1=real(gvec(1,ig))
   g2=real(gvec(2,ig))
   g3=real(gvec(3,ig))
   gsq=gmet(1,1)*g1**2+gmet(2,2)*g2**2+gmet(3,3)*g3**2+ &
&                 2.0*(gmet(1,2)*g1*g2+gmet(1,3)*g1*g3+gmet(2,3)*g2*g3)
   ecutsigx=max(ecutsigx,gsq)
  end do
  rsigx=sqrt(ecutsigx)
  ecutsigx=2.0*ecutsigx*pi**2

  write(std_out,'(a,f7.3,a)') ' setmesh: calculated ecutsigx = ',ecutsigx,' [Ha]'

! in the calculation of the GW self-energy or of the RPA dielectric matrix,
! we will make products Rho_12=Phi_1*Phi_2 of wavefunctions whose Fourier
! coefficients lie in the sphere of radius rwfn. Such products will have non
! vanishing Fourier coefficients in the whole sphere of radius 2*rwfn. However,
! we only need the Fourier coefficients of Rho_12 that lie in the sphere of radius
! rsigx. we can thus allow aliasing outside that sphere, so that the FFT box may
! only enclose a sphere of radius:
  reff=min(0.5*rsigx+rwfn,2.0*rwfn)

! extreme case: this should yield back the GS FFT grid if full wavefunctions are
! considered in the GW calculation...
  if (method==3) reff=2.0*rwfn

  ecuteff=0.5*(pi*reff)**2
  write(std_out,'(a,f7.3,a)') ' setmesh: calculated ecuteff  = ',ecuteff,' [Ha]'

! we now search the limits of that sphere in each direction...
  gsqmax=reff**2
  ig1max=2*m1+1
  ig2max=2*m2+1
  ig3max=2*m3+1
  m1=-1
  m2=-1
  m3=-1
  do ig1=0,ig1max
   g1=real(ig1)
   do ig2=0,ig2max
    g2=real(ig2)
    do ig3=0,ig3max
     g3=real(ig3)
     gsq=gmet(1,1)*g1**2+gmet(2,2)*g2**2+gmet(3,3)*g3**2+ &
&                   2.0*(gmet(1,2)*g1*g2+gmet(1,3)*g1*g3+gmet(2,3)*g2*g3)

     if (gsq>gsqmax) cycle
     m1=max(m1,ig1)
     m2=max(m2,ig2)
     m3=max(m3,ig3)
    end do
   end do
  end do

 else

  stop 'method > 3 is not allowed in setmesh (see ~abinit/src/5gw/setmesh.F90)'

 end if

!warn on low npwwfn
 if(m1<mm1.or.m2<mm2.or.m3<mm3) then
  write(message, '(10a)' ) ch10,&
&   ' setmesh: COMMENT -',ch10,&
&   '  Note that npwwfn is small with respect to npweps',ch10,&
&   '  or with respect to npwsigx.',ch10,&
&   '  Such a small npwwfn is a waste:',ch10,&
&   '  You could raise npwwfn without loss in cpu time.'
  call wrtout(std_out, message,'COLL')
 end if

!keep the largest of the m's/mm's.
 m1=max(m1,mm1)
 m2=max(m2,mm2)
 m3=max(m3,mm3)

!determine the size of FFT grid (ideally n=2*m+1 but this could
!not be allowed by the fft library)
 call sizefft(m1,n1)
 call sizefft(m2,n2)
 call sizefft(m3,n3)

 write(message,'(a,i3,a,i3,a,i3)') ' setmesh: mesh size selected = ',n1,'x',n2,'x',n3
 call wrtout(6,message,'COLL')
 call wrtout(ab_out,message,'COLL')

!write(std_out,'(a)') ' setmesh: warning ! mesh size enforced by the user !'
!write(ab_out,'(a)')  ' setmesh: warning ! mesh size enforced by the user !'
!n1=20
!n2=20
!n3=20
!write(std_out,'(a,i3,a,i3,a,i3)') ' setmesh: mesh size enforced by the user = ',n1,'x',n2,'x',n3
!write(ab_out,'(a,i3,a,i3,a,i3)')  ' setmesh: mesh size enforced by the user = ',n1,'x',n2,'x',n3

 n1d=n1
 n2d=n2
 n3d=n3

 !calculate the number of FFT grid points
 nr=n1d*n2d*n3d
 write(message,'(a,i8,a)')' setmesh: total size of FFT array nr = ',nr,ch10
 call wrtout(6,message,'COLL')
 call wrtout(ab_out,message,'COLL')

end subroutine setmesh
!!***
