(* \chaptertitle{Io}{IO Support for zlib} *)
(*
 * IO - Abstract input/output
 * Copyright (C) 2003 Nicolas Cannasse
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version,
 * with the special exception on linking described in file LICENSE.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *)

(* Modified by Coherent Graphics Ltd *)

open Utility

type input =
  {mutable in_read : unit -> char;
   mutable in_input : Istring.t -> int -> int -> int;
   mutable in_close : unit -> unit}

exception No_more_input
exception Input_closed

(* -------------------------------------------------------------- *)
(* API *)

let create_in read input close =
  {in_read = read;
   in_input = input;
   in_close = close}

let nread i n =
	if n < 0 then invalid_arg "IO.nread";
	if n = 0 then
		Istring.create 0
	else
	let s = Istring.create n in
	let l = ref n in
	let p = ref 0 in
	try
		while !l > 0 do
			let r = i.in_input s !p !l in
			if r = 0 then raise No_more_input;
			p := !p + r;
			l := !l - r;
		done;
		s
	with
		No_more_input as e ->
                        dpr "2A";
			if !p = 0 then raise e;
			Istring.sub s 0 !p

let close_in i =
	let f _ = raise Input_closed in
	i.in_close();
	i.in_read <- f;
	i.in_input <- f;
	i.in_close <- f

let read_all i =
	let maxlen = 1024 in
	let str = ref [] in
	let pos = ref 0 in
	let rec loop() =
		let s = nread i maxlen in
		str := (s,!pos) :: !str;
		pos := !pos + Istring.length s;
		loop()
	in
	try
		loop()
	with
		No_more_input ->
                        dpr "2B";
			let buf = Istring.create !pos in
			List.iter (fun (s,p) ->
				Istring.blit s 0 buf p (Istring.length s)
			) !str;
			buf

(* -------------------------------------------------------------- *)
(* BINARY APIs *)

exception Overflow of string

let read_byte i = int_of_char (i.in_read())

let read_ui16 i =
	let ch1 = read_byte i in
	let ch2 = read_byte i in
	ch1 lor (ch2 lsl 8)


