/*
 * Copyright 2013-2016 Canonical Ltd.
 *
 * This file is part of webbrowser-app.
 *
 * webbrowser-app is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * webbrowser-app is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

// system
#include <cerrno>
#include <cstring>
#include <sys/apparmor.h>

// Qt
#include <QtCore/QMetaObject>
#include <QtCore/QtGlobal>
#include <QtGui/QTouchDevice>
#include <QtNetwork/QNetworkInterface>
#include <QtQml/QQmlComponent>
#include <QtQml/QQmlContext>
#include <QtQml/QQmlEngine>
#include <QtQml/QtQml>
#include <QtQuick/QQuickWindow>

// local
#include "browserapplication.h"
#include "config.h"
#include "favicon-fetcher.h"
#include "meminfo.h"
#include "mime-database.h"
#include "qquickshortcut_p.h"
#include "session-storage.h"
#include "webbrowser-window.h"

#include "TouchRegistry.h"
#include "Ubuntu/Gestures/Direction.h"
#include "Ubuntu/Gestures/DirectionalDragArea.h"
#include "Unity/InputInfo/qdeclarativeinputdevicemodel_p.h"

BrowserApplication::BrowserApplication(int& argc, char** argv)
    : QApplication(argc, argv)
    , m_engine(0)
    , m_window(0)
    , m_component(0)
    , m_webbrowserWindowProxy(0)
{
    m_arguments = arguments();
    m_arguments.removeFirst();
}

BrowserApplication::~BrowserApplication()
{
    if (m_webbrowserWindowProxy) {
        m_webbrowserWindowProxy->setWindow(NULL);
    }
    delete m_window;
    delete m_webbrowserWindowProxy;
    delete m_component;
    delete m_engine;
}

QString BrowserApplication::inspectorPort() const
{
    QString port;
    Q_FOREACH(const QString& argument, m_arguments) {
        if (argument == "--inspector") {
            // default port
            port = QString::number(REMOTE_INSPECTOR_PORT);
            break;
        }
        if (argument.startsWith("--inspector=")) {
            port = argument.split("--inspector=")[1];
            break;
        }
    }
    return port;
}

QString BrowserApplication::inspectorHost() const
{
    QString host;
    Q_FOREACH(QHostAddress address, QNetworkInterface::allAddresses()) {
        if (!address.isLoopback() && (address.protocol() == QAbstractSocket::IPv4Protocol)) {
            host = address.toString();
            break;
        }
    }
    return host;
}

#define MAKE_SINGLETON_FACTORY(type) \
    static QObject* type##_singleton_factory(QQmlEngine* engine, QJSEngine* scriptEngine) { \
        Q_UNUSED(engine); \
        Q_UNUSED(scriptEngine); \
        return new type(); \
    }

MAKE_SINGLETON_FACTORY(MemInfo)
MAKE_SINGLETON_FACTORY(MimeDatabase)
MAKE_SINGLETON_FACTORY(Direction)

bool BrowserApplication::initialize(const QString& qmlFileSubPath
                                    , const QString& appId)
{
    Q_ASSERT(m_window == 0);

    if (appId.isEmpty()) {
        qCritical() << "Cannot initialize the runtime environment: "
                       "no application id detected.";
        return false;
    }

    if (m_arguments.contains("--help") || m_arguments.contains("-h")) {
        printUsage();
        return false;
    }

    // Ensure that application-specific data is written where it ought to.
    QStringList appIdParts = appId.split('_');

    QCoreApplication::setApplicationName(appIdParts.first());
    QCoreApplication::setOrganizationDomain(QCoreApplication::applicationName());

    // Get also the the first two components of the app ID: <package>_<app>,
    // which is needed by Online Accounts.
    QString unversionedAppId = QStringList(appIdParts.mid(0, 2)).join('_');

    // Ensure only one instance of the app is running.
    // For webapps using the container as a launcher, the predicate that
    // is used to determine if this running instance is a duplicate of
    // a running one, is based on the current APP_ID.
    // The app id is formed as: <package name>_<app name>_<version>

    // Where the <package name> is specified in the the manifest.json as
    // "appName" and is specific for the whole click package.

    // The <app name> portion is based on the desktop file name and is a short
    // app name. This name is meaningful when more than one desktop file is
    // found in a given click package.

    // IMPORTANT:
    // 1. When a click application contains more than one desktop file
    // the bundle is considered a single app from the point of view of the
    // cache and resource file locations. THOSE FILES ARE THEN SHARED between
    // the instances.
    // 2. To make sure that if more than one desktop file is found in a click package,
    // those apps are not considered the same instance, the instance existance predicate
    // is based on the <package name> AND the <app name> detailed above.
    if (m_singleton.run(m_arguments, appId)) {
        connect(&m_singleton, SIGNAL(newInstanceLaunched(const QStringList&)),
                SLOT(onNewInstanceLaunched(const QStringList&)));
    } else {
        return false;
    }

    bool runningConfined = true;
    char* label;
    char* mode;
    if (aa_getcon(&label, &mode) != -1) {
        if (strcmp(label, "unconfined") == 0) {
            runningConfined = false;
        }
        free(label);
    } else if (errno == EINVAL) {
        runningConfined = false;
    }

    QString devtoolsPort = inspectorPort();
    QString devtoolsHost = inspectorHost();
    bool inspectorEnabled = !devtoolsPort.isEmpty();
    if (inspectorEnabled) {
        qputenv("UBUNTU_WEBVIEW_DEVTOOLS_HOST", devtoolsHost.toUtf8());
        qputenv("UBUNTU_WEBVIEW_DEVTOOLS_PORT", devtoolsPort.toUtf8());
    }

    const char* uri = "webbrowsercommon.private";
    qmlRegisterType<FaviconFetcher>(uri, 0, 1, "FaviconFetcher");
    qmlRegisterSingletonType<MemInfo>(uri, 0, 1, "MemInfo", MemInfo_singleton_factory);
    qmlRegisterSingletonType<MimeDatabase>(uri, 0, 1, "MimeDatabase", MimeDatabase_singleton_factory);
    qmlRegisterType<SessionStorage>(uri, 0, 1, "SessionStorage");
    qmlRegisterType<QQuickShortcut>(uri, 0, 1, "Shortcut");

    const char* gesturesUri = "Ubuntu.Gestures";
    qmlRegisterSingletonType<Direction>(gesturesUri, 0, 1, "Direction", Direction_singleton_factory);
    qmlRegisterType<DirectionalDragArea>(gesturesUri, 0, 1, "DirectionalDragArea");

    const char* inputInfoUri = "Unity.InputInfo";
    qmlRegisterType<QDeclarativeInputDeviceModel>(inputInfoUri, 0, 1, "InputDeviceModel");
    qmlRegisterType<QInputDevice>(inputInfoUri, 0, 1, "InputInfo");

    m_engine = new QQmlEngine;
    connect(m_engine, SIGNAL(quit()), SLOT(quit()));
    if (!isRunningInstalled()) {
        m_engine->addImportPath(UbuntuBrowserImportsDirectory());
    }

    qmlEngineCreated(m_engine);

    QQmlContext* context = m_engine->rootContext();
    context->setContextProperty("__runningConfined", runningConfined);
    context->setContextProperty("unversionedAppId", unversionedAppId);

    m_component = new QQmlComponent(m_engine);
    m_component->loadUrl(QUrl::fromLocalFile(UbuntuBrowserDirectory() + "/" + qmlFileSubPath));
    if (!m_component->isReady()) {
        qWarning() << m_component->errorString();
        return false;
    }
    m_webbrowserWindowProxy = new WebBrowserWindow();
    context->setContextProperty("webbrowserWindowProxy", m_webbrowserWindowProxy);

    QObject* browser = m_component->beginCreate(context);
    m_window = qobject_cast<QQuickWindow*>(browser);
    m_webbrowserWindowProxy->setWindow(m_window);

    m_window->installEventFilter(new TouchRegistry(this));

    browser->setProperty("developerExtrasEnabled", inspectorEnabled);
    browser->setProperty("forceFullscreen", m_arguments.contains("--fullscreen"));

    bool hasTouchScreen = false;
    Q_FOREACH(const QTouchDevice* device, QTouchDevice::devices()) {
        if (device->type() == QTouchDevice::TouchScreen) {
            hasTouchScreen = true;
        }
    }
    browser->setProperty("hasTouchScreen", hasTouchScreen);

    return true;
}

void BrowserApplication::onNewInstanceLaunched(const QStringList& arguments) const
{
    QVariantList urls;
    Q_FOREACH(const QString& argument, arguments) {
        if (!argument.startsWith(QStringLiteral("-"))) {
            QUrl url = QUrl::fromUserInput(argument);
            if (url.isValid()) {
                urls.append(url);
            }
        }
    }
    QMetaObject::invokeMethod(m_window, "openUrls", Q_ARG(QVariant, QVariant(urls)));
    m_window->requestActivate();
}

void BrowserApplication::qmlEngineCreated(QQmlEngine*)
{}

int BrowserApplication::run()
{
    Q_ASSERT(m_window != 0);

    if (m_arguments.contains("--fullscreen")) {
        m_window->showFullScreen();
    } else if (m_arguments.contains("--maximized")) {
        m_window->showMaximized();
    } else {
        m_window->show();
    }
    return exec();
}

QList<QUrl> BrowserApplication::urls() const
{
    QList<QUrl> urls;
    Q_FOREACH(const QString& argument, m_arguments) {
        if (!argument.startsWith("-")) {
            QUrl url = QUrl::fromUserInput(argument);
            if (url.isValid()) {
                urls.append(url);
            }
        }
    }
    return urls;
}
