#! /usr/bin/python3
# -*- coding: utf-8 -*-

# Copyright(C) 2013 Mark Tully <markjtully@gmail.com>
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

from gi.repository import GLib, Gio
from gi.repository import Unity
import gettext
import os
import sqlite3

APP_NAME = 'unity-scope-gourmet'
LOCAL_PATH = '/usr/share/locale/'
gettext.bindtextdomain(APP_NAME, LOCAL_PATH)
gettext.textdomain(APP_NAME)
_ = gettext.gettext

GROUP_NAME = 'com.canonical.Unity.Scope.Recipes.Gourmet'
UNIQUE_PATH = '/com/canonical/unity/scope/recipes/gourmet'

SEARCH_HINT = _('Search Gourmet Recipe Manager')
NO_RESULTS_HINT = _('Sorry, there are no Gourmet recipe results that match your search.')
PROVIDER_CREDITS = _('')
SVG_DIR = '/usr/share/icons/unity-icon-theme/places/svg/'
PROVIDER_ICON = SVG_DIR + 'service-yelp.svg'
DEFAULT_RESULT_ICON = SVG_DIR + 'result-help.svg'
DEFAULT_RESULT_MIMETYPE = 'x-scheme-handler/man'
DEFAULT_RESULT_TYPE = Unity.ResultType.DEFAULT
GOURMET_DB = os.getenv("HOME") + '/.gourmet/recipes.db'
GOURMET_EXECUTABLE = '/usr/bin/gourmet'

SQL_QUERY = '''SELECT recipe.id, recipe.title, recipe.instructions, recipe.modifications, recipe.cuisine, recipe.rating,
               recipe.preptime, recipe.cooktime, recipe.yields, recipe.yield_unit, recipe.image,
               recipe.link, GROUP_CONCAT(COALESCE(ingredients.amount, '') || ' ' || COALESCE(ingredients.unit, '') || ' ' || ingredients.item)
               FROM recipe INNER JOIN ingredients ON recipe.id = ingredients.recipe_id
               WHERE recipe.title LIKE '%%%s%%' AND recipe.deleted = '0' 
               GROUP BY recipe.id'''

STARS = {0: '☆☆☆☆☆',
         1: '✮☆☆☆☆',
         2: '★☆☆☆☆',
         3: '★✮☆☆☆',
         4: '★★☆☆☆',
         5: '★★✮☆☆',
         6: '★★★☆☆',
         7: '★★★✮☆',
         8: '★★★★☆',
         9: '★★★★✮',
         10: '★★★★★'}

c1 = {'id': 'recipes',
      'name': _('Recipes'),
      'icon': SVG_DIR + 'group-installed.svg',
      'renderer': Unity.CategoryRenderer.VERTICAL_TILE}
CATEGORIES = [c1]

FILTERS = []

META0 = {'id': 'instructions',
         'type': 's',
         'field': Unity.SchemaFieldType.OPTIONAL}
META1 = {'id': 'likes',
         'type': 's',
         'field': Unity.SchemaFieldType.OPTIONAL}
META2 = {'id': 'preptime',
         'type': 's',
         'field': Unity.SchemaFieldType.OPTIONAL}
META3 = {'id': 'cooktime',
         'type': 's',
         'field': Unity.SchemaFieldType.OPTIONAL}
META4 = {'id': 'cuisine',
         'type': 's',
         'field': Unity.SchemaFieldType.OPTIONAL}
META5 = {'id': 'ingredients',
         'type': 's',
         'field': Unity.SchemaFieldType.OPTIONAL}
META6 = {'id': 'yields',
         'type': 's',
         'field': Unity.SchemaFieldType.OPTIONAL}
EXTRA_METADATA = [META0, META1, META2, META3, META4, META5, META6]


def duration(seconds):
    if isinstance(seconds, int):
        import time
        if seconds > 3600:
            duration = time.strftime(_('%H hours %M minutes'), time.gmtime(seconds))
        else:
            duration = time.strftime(_('%M minutes'), time.gmtime(seconds))
        return duration
    else:
        return ""


def gourmet_search(path, query, search):
    try:
        conn = sqlite3.connect(path)
        cursor = conn.cursor()
        print("Searching for Gourmet recipes")
        query = query % search
        cursor.execute(query)
        results = cursor.fetchall()
        cursor.close()
        return results
    except:
        return []


def search(search, filters):
    '''
    Search for recipes matching the search string
    '''
    results = []
    if not os.path.exists('/tmp/unity-scope-gourmet'):
        os.mkdir('/tmp/unity-scope-gourmet')

    gourmet_results = gourmet_search(GOURMET_DB, SQL_QUERY, search)
    for row in gourmet_results:
        i = row[0]
        title = row[1]
        instructions = row[2]
        comments = row[3]
        cuisine = row[4]
        likes = STARS[row[5]]
        preptime = str(duration(row[6]))
        cooktime = str(duration(row[7]))
        yields = str(row[8]).replace('.0', ' ') + str(row[9])
        uri = row[11]
        ingredients = row[12].replace('.0 ', ' ')
        ingredients = ingredients.replace('  ', ' ')

        # add http:// if necessary
        if not (uri.startswith("http://")) and not (uri == ''):
            uri = 'http://%s' % uri
        if uri == '':
            uri = 'gourmet'

        # Give each cached image file the unique ID of its recipe
        icon_hint = DEFAULT_RESULT_ICON
        if row[10]:
            open('/tmp/unity-scope-gourmet/icon' + str(i), 'wb').write(row[10])
            icon_hint = 'file:///tmp/unity-scope-gourmet/icon%s' % str(i)
        else:
            if os.path.exists('/tmp/unity-scope-gourmet/icon' + str(i)):
                os.remove('/tmp/unity-scope-gourmet/icon' + str(i))

        results.append({'uri': uri,
                        'icon': icon_hint,
                        'title': title,
                        'likes': GLib.Variant('s', likes),
                        'comments' : comments,
                        'instructions': GLib.Variant('s', instructions),
                        'cuisine': GLib.Variant('s', cuisine),
                        'preptime': GLib.Variant('s', preptime),
                        'cooktime': GLib.Variant('s', cooktime),
                        'ingredients': GLib.Variant('s', ingredients),
                        'yields': GLib.Variant('s', yields)})
    return results


def activate(result, metadata, action):
        parameters = [GOURMET_EXECUTABLE]
        GLib.spawn_async(parameters)
        return Unity.ActivationResponse(handled=Unity.HandledType.HIDE_DASH, goto_uri=None)


class Preview(Unity.ResultPreviewer):
    '''
    Creates the preview for the result
    '''
    def do_run(self):
        '''
        Create a preview and return it
        '''
        preview = Unity.GenericPreview.new(self.result.title, '', None)
        preview.props.image_source_uri = self.result.icon_hint
        preview.props.subtitle = self.result.metadata['likes'].get_string()
        preview.props.description_markup = _('<b>Yields:</b>\t\t\t\t%s\n' % self.result.metadata['yields'].get_string())
        preview.props.description_markup += _('<b>Preparation Time:</b>\t%s\n' % self.result.metadata['preptime'].get_string())
        preview.props.description_markup += _('<b>Cooking Time:</b>\t\t%s\n' % self.result.metadata['cooktime'].get_string())
        preview.props.description_markup += _('<b>Cuisine:</b>\t\t\t%s\n\n' % self.result.metadata['cuisine'].get_string())
        preview.props.description_markup += _('<b>Ingredients:</b>\n%s\n\n' % self.result.metadata['ingredients'].get_string().replace(',', ', '))
        preview.props.description_markup += _('<b>Instructions</b>\n\n%s' % self.result.metadata['instructions'].get_string())
        preview.props.description_markup += _('\n\n\n%s' % self.result.comment)
        show_action = Unity.PreviewAction.new("open", _("Open"), None)
        preview.add_action(show_action)
        return preview

# Classes below this point establish communication
# with Unity, you probably shouldn't modify them.


class MySearch(Unity.ScopeSearchBase):
    def __init__(self, search_context):
        super(MySearch, self).__init__()
        self.set_search_context(search_context)

    def do_run(self):
        '''
        Adds results to the model
        '''
        try:
            result_set = self.search_context.result_set
            for i in search(self.search_context.search_query,
                            self.search_context.filter_state):
                if not 'uri' in i or not i['uri'] or i['uri'] == '':
                    continue
                if not 'icon' in i or not i['icon'] or i['icon'] == '':
                    i['icon'] = DEFAULT_RESULT_ICON
                if not 'mimetype' in i or not i['mimetype'] or i['mimetype'] == '':
                    i['mimetype'] = DEFAULT_RESULT_MIMETYPE
                if not 'result_type' in i or not i['result_type'] or i['result_type'] == '':
                    i['result_type'] = DEFAULT_RESULT_TYPE
                if not 'category' in i or not i['category'] or i['category'] == '':
                    i['category'] = 0
                if not 'title' in i or not i['title']:
                    i['title'] = ''
                if not 'comment' in i or not i['comment']:
                    i['comment'] = ''
                if not 'dnd_uri' in i or not i['dnd_uri'] or i['dnd_uri'] == '':
                    i['dnd_uri'] = i['uri']
                i['provider_credits'] = GLib.Variant('s', PROVIDER_CREDITS)
                result_set.add_result(**i)
        except Exception as error:
            print(error)


class Scope(Unity.AbstractScope):
    def __init__(self):
        Unity.AbstractScope.__init__(self)

    def do_get_search_hint(self):
        return SEARCH_HINT

    def do_get_schema(self):
        '''
        Adds specific metadata fields
        '''
        schema = Unity.Schema.new()
        if EXTRA_METADATA:
            for m in EXTRA_METADATA:
                schema.add_field(m['id'], m['type'], m['field'])
        #FIXME should be REQUIRED for credits
        schema.add_field('provider_credits', 's', Unity.SchemaFieldType.OPTIONAL)
        return schema

    def do_get_categories(self):
        '''
        Adds categories
        '''
        cs = Unity.CategorySet.new()
        if CATEGORIES:
            for c in CATEGORIES:
                cat = Unity.Category.new(c['id'], c['name'],
                                         Gio.ThemedIcon.new(c['icon']),
                                         c['renderer'])
                cs.add(cat)
        return cs

    def do_get_filters(self):
        '''
        Adds filters
        '''
        fs = Unity.FilterSet.new()
        #if FILTERS:
        #
        return fs

    def do_get_group_name(self):
        return GROUP_NAME

    def do_get_unique_name(self):
        return UNIQUE_PATH

    def do_create_search_for_query(self, search_context):
        se = MySearch(search_context)
        return se

    def do_activate(self, result, metadata, action):
        '''
        What to do when a resut is clicked
        '''
        activation = activate(result, metadata, action)
        return activation

    def do_create_previewer(self, result, metadata):
        '''
        Creates a preview when a resut is right-clicked
        '''
        result_preview = Preview()
        result_preview.set_scope_result(result)
        result_preview.set_search_metadata(metadata)
        return result_preview


def load_scope():
    return Scope()
