# -*- coding: utf-8 -*-
#
# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""QArrow tests."""

from PyQt4.QtGui import QStyle
from twisted.internet import defer
from twisted.trial.unittest import TestCase

from ubuntu_sso.qt.arrow import QArrow

# pylint: disable=W0212,C0103


class FakeStyle(object):
    """Fake a Qt style."""

    def __init__(self):
        """Create instance."""
        self.called = []

    def __call__(self, *args, **kwargs):
        """Instance callable."""
        return self

    def drawPrimitive(self, primitive, opt, painter, widget):
        """Fake a draw primitive."""
        self.called.append(('drawPrimitive', primitive, opt, painter, widget))


class QArrowTestCase(TestCase):
    """Test the QArrow class."""

    @defer.inlineCallbacks
    def setUp(self):
        """Set the tests."""
        yield super(QArrowTestCase, self).setUp()
        self.valid_dirs = (QArrow.UP, QArrow.DOWN,
                           QArrow.LEFT, QArrow.RIGHT)
        self.style = FakeStyle()

    def test_correct_init(self):
        """Test the init using the correct direction."""
        for direction in self.valid_dirs:
            arrow = QArrow(direction)
            self.assertEqual(direction, arrow.direction)

    def test_wrong_init(self):
        """Test the init using the wrong direction."""
        self.assertRaises(ValueError, QArrow, 'test')

    def test_set_direction_correct(self):
        """Test the set direction with the correct value."""
        arrow = QArrow(QArrow.RIGHT)
        for direction in self.valid_dirs:
            arrow.direction = direction
            self.assertEqual(direction, arrow.direction)

    def test_set_direction_wrong(self):
        """Test set direction wrong."""
        arrow = QArrow(QArrow.RIGHT)
        self.assertRaises(ValueError, setattr, arrow, 'direction', 'wrong')

    def test_paint_event(self):
        """Test the paint event."""
        directions_map = {QArrow.UP: QStyle.PE_IndicatorArrowUp,
            QArrow.DOWN: QStyle.PE_IndicatorArrowDown,
            QArrow.LEFT: QStyle.PE_IndicatorArrowLeft,
            QArrow.RIGHT: QStyle.PE_IndicatorArrowRight}

        arrow = QArrow(QArrow.RIGHT)
        self.patch(arrow, 'style', self.style)

        for key, value in directions_map.items():
            self.style.called = []
            arrow.direction = key
            arrow.paintEvent(None)
            self.assertIn(value, self.style.called[0])
