/* Copyright (c) 2018 Dovecot authors, see the included COPYING file */

#include "test-lib.h"
#include "murmurhash3.h"

struct murmur3_test_vectors {
	const char *input;
	size_t len;
	uint32_t seed;

	/* murmurhash3_128() produces a different output on ILP32 and LP64
	   systems (by design).  Therefore, we must use different expected
	   results based on what system we're on.  We define both all the
	   time, but use the below pre-processor magic to select which
	   version we'll use. */
	uint8_t result_ilp32[MURMURHASH3_128_RESULTBYTES]; /* fits all results */
	uint8_t result_lp64[MURMURHASH3_128_RESULTBYTES]; /* fits all results */
#ifdef _LP64
#define result result_lp64
#else
#define result result_ilp32
#endif
};

static void test_murmurhash3_algorithm(const char *name,
				       void (*func)(const void*,size_t,uint32_t,unsigned char[]),
				       size_t result_size,
				       const struct murmur3_test_vectors *vectors,
				       unsigned int tests)
{
	test_begin(t_strdup_printf("murmurhash3 (%s)", name));

	for(unsigned int i = 0; i < tests; i++) {
		unsigned char result[result_size];
		func(vectors[i].input, vectors[i].len, vectors[i].seed, result);
		test_assert_idx(memcmp(result, vectors[i].result, sizeof(result)) == 0, i);
	}

	test_end();
}

static void test_murmurhash3_32(void)
{
	/* murmurhash3_32() produces the same output on both ILP32 and LP64
	   systems, so use the same expected outputs for both */
	struct murmur3_test_vectors vectors[] = {
		{ "", 0, 0, { 0, }, { 0, } },
		{ "", 0, 0x1,
		  { 0x51, 0x4E, 0x28, 0xB7, },
		  { 0x51, 0x4E, 0x28, 0xB7, } },
		{ "", 0, 0xFFFFFFFF,
		  { 0x81, 0xF1, 0x6F, 0x39, },
		  { 0x81, 0xF1, 0x6F, 0x39, } },
		{ "\0\0\0\0", 4, 0,
		  { 0x23, 0x62, 0xF9, 0xDE, },
		  { 0x23, 0x62, 0xF9, 0xDE, } },
		{ "aaaa", 4, 0x9747b28c,
		  { 0x5A, 0x97, 0x80, 0x8A, },
		  { 0x5A, 0x97, 0x80, 0x8A, } },
		{ "aaa", 3, 0x9747b28c,
		  { 0x28, 0x3E, 0x01, 0x30, },
		  { 0x28, 0x3E, 0x01, 0x30, } },
		{ "aa", 2, 0x9747b28c,
		  { 0x5D, 0x21, 0x17, 0x26, },
		  { 0x5D, 0x21, 0x17, 0x26, } },
		{ "a", 1, 0x9747b28c,
		  { 0x7F, 0xA0, 0x9E, 0xA6, },
		  { 0x7F, 0xA0, 0x9E, 0xA6, } },
		{ "abcd", 4, 0x9747b28c,
		  { 0xF0, 0x47, 0x86, 0x27, },
		  { 0xF0, 0x47, 0x86, 0x27, } },
		{ "abc", 3, 0x9747b28c,
		  { 0xC8, 0x4A, 0x62, 0xDD, },
		  { 0xC8, 0x4A, 0x62, 0xDD, } },
		{ "ab", 2, 0x9747b28c,
		  { 0x74, 0x87, 0x55, 0x92, },
		  { 0x74, 0x87, 0x55, 0x92, } },
		{ "Hello, world!", 13, 0x9747b28c,
		  { 0x24, 0x88, 0x4C, 0xBA, },
		  { 0x24, 0x88, 0x4C, 0xBA, } },
		{
		  "\xcf\x80\xcf\x80\xcf\x80\xcf\x80\xcf\x80\xcf\x80\xcf\x80\xcf\x80",
		  16,
		  0x9747b28c,
		  { 0xD5, 0x80, 0x63, 0xC1, },
		  { 0xD5, 0x80, 0x63, 0xC1, }
		}, /* 8 U+03C0 (Greek Small Letter Pi) */
		{
		  "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
		  "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
		  "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
		  "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
		  "aaaaaaaaaaaaaaaaaaaa",
		  256,
		  0x9747b28c,
		  { 0x37, 0x40, 0x5B, 0xDC, },
		  { 0x37, 0x40, 0x5B, 0xDC, }
		},
	};

	test_murmurhash3_algorithm("murmurhash3_32", murmurhash3_32,
				   MURMURHASH3_32_RESULTBYTES,
				   vectors, N_ELEMENTS(vectors));
}

static void test_murmurhash3_128(void)
{
	/* murmurhash3_128() produces a different output on ILP32 and LP64
	   systems (by design). */
	struct murmur3_test_vectors vectors[] = {
		{ "", 0, 0x00000000, { 0, }, { 0, }},
		{ "", 0, 0x00000001,
		  { 0x88, 0xc4, 0xad, 0xec, 0x54, 0xd2, 0x01, 0xb9,
		    0x54, 0xd2, 0x01, 0xb9, 0x54, 0xd2, 0x01, 0xb9 },
		  { 0x46, 0x10, 0xab, 0xe5, 0x6e, 0xff, 0x5c, 0xb5,
		    0x51, 0x62, 0x2d, 0xaa, 0x78, 0xf8, 0x35, 0x83 }},
		{ "", 0, 0xffffffff,
		  { 0x05, 0x1e, 0x08, 0xa9, 0x98, 0x9d, 0x49, 0xf7,
		    0x98, 0x9d, 0x49, 0xf7, 0x98, 0x9d, 0x49, 0xf7 },
		  { 0x6a, 0xf1, 0xdf, 0x4d, 0x9d, 0x3b, 0xc9, 0xec,
		    0x85, 0x74, 0x21, 0x12, 0x1e, 0xe6, 0x44, 0x6b }},
		{ "\0\0\0\0", 4, 0x00000000,
		  { 0xcc, 0x06, 0x6f, 0x1f, 0x9e, 0x51, 0x78, 0x40,
		    0x9e, 0x51, 0x78, 0x40, 0x9e, 0x51, 0x78, 0x40 },
		  { 0xcf, 0xa0, 0xf7, 0xdd, 0xd8, 0x4c, 0x76, 0xbc,
		    0x58, 0x96, 0x23, 0x16, 0x1c, 0xf5, 0x26, 0xf1 }},
		{ "aaaa", 4, 0x9747b28c,
		  { 0x36, 0x80, 0x4c, 0xef, 0x2a, 0x61, 0xc2, 0x24,
		    0x2a, 0x61, 0xc2, 0x24, 0x2a, 0x61, 0xc2, 0x24 },
		  { 0xb4, 0xe0, 0xa5, 0xf7, 0x5e, 0x64, 0x9b, 0xf0,
		    0xa5, 0xd3, 0xe8, 0xe9, 0x03, 0x8c, 0x56, 0x9f }},
		{ "aaa", 3, 0x9747b28c,
		  { 0x83, 0x83, 0x89, 0xbe, 0x9a, 0xad, 0x7f, 0x88,
		    0x9a, 0xad, 0x7f, 0x88, 0x9a, 0xad, 0x7f, 0x88 },
		  { 0x8e, 0xa5, 0xe3, 0x7a, 0xe4, 0xc7, 0x46, 0x6b,
		    0xf9, 0x25, 0xbe, 0xf0, 0x35, 0xdc, 0x93, 0x1c }},
		{ "aa", 2, 0x9747b28c,
		  { 0xdf, 0xbe, 0x4a, 0x86, 0x4a, 0x9c, 0x35, 0x0b,
		    0x4a, 0x9c, 0x35, 0x0b, 0x4a, 0x9c, 0x35, 0x0b },
		  { 0x12, 0xa6, 0x98, 0xa9, 0xbe, 0xe5, 0xbb, 0x1f,
		    0xe9, 0x36, 0x30, 0xff, 0x5e, 0x26, 0x94, 0x01 }},
		{ "a", 1, 0x9747b28c,
		  { 0x08, 0x4e, 0xf9, 0x44, 0x21, 0xa1, 0x18, 0x6e,
		    0x21, 0xa1, 0x18, 0x6e, 0x21, 0xa1, 0x18, 0x6e },
		  { 0x5c, 0xe8, 0xd8, 0x51, 0x2d, 0xb2, 0x5a, 0x1d,
		    0x9e, 0x6d, 0xab, 0x0f, 0x92, 0x08, 0xf0, 0x04 }},
		{ "abcd", 4, 0x9747b28c,
		  { 0x47, 0x95, 0xc5, 0x29, 0xce, 0xc1, 0x88, 0x5e,
		    0xce, 0xc1, 0x88, 0x5e, 0xce, 0xc1, 0x88, 0x5e },
		  { 0x49, 0xb4, 0x70, 0x9e, 0xac, 0x55, 0x37, 0x91,
		    0x8a, 0x7e, 0x67, 0xe7, 0xe9, 0xd3, 0xa7, 0xbb }},
		{ "abc", 3, 0x9747b28c,
		  { 0xd6, 0x35, 0x9e, 0xaf, 0x48, 0xfc, 0x3a, 0xc3,
		    0x48, 0xfc, 0x3a, 0xc3, 0x48, 0xfc, 0x3a, 0xc3 },
		  { 0x37, 0x43, 0x63, 0x0d, 0xbf, 0xc3, 0xce, 0xdc,
		    0xcd, 0xe0, 0xa2, 0x34, 0x20, 0xb5, 0x04, 0xbf }},
		{ "ab", 2, 0x9747b28c,
		  { 0x38, 0x37, 0xd7, 0x95, 0xc7, 0xfe, 0x58, 0x96,
		    0xc7, 0xfe, 0x58, 0x96, 0xc7, 0xfe, 0x58, 0x96 },
		  { 0x84, 0x34, 0xee, 0xad, 0x1a, 0x44, 0x28, 0x0b,
		    0x7e, 0xb9, 0x33, 0xe7, 0x63, 0xce, 0x37, 0x2b }},
		{ "Hello, world!", 13, 0x9747b28c,
		  { 0x75, 0x6d, 0x54, 0x60, 0xbb, 0x87, 0x22, 0x16,
		    0xb7, 0xd4, 0x8b, 0x7c, 0x53, 0xc8, 0xc6, 0x36 },
		  { 0xed, 0xc4, 0x85, 0xd6, 0x62, 0xa8, 0x39, 0x2e,
		    0xf8, 0x5e, 0x7e, 0x76, 0x31, 0xd5, 0x76, 0xba }},
		{
		  "\xcf\x80\xcf\x80\xcf\x80\xcf\x80\xcf\x80\xcf\x80\xcf\x80\xcf\x80",
		  16,
		  0x9747b28c,
		  { 0xaf, 0x2a, 0xd3, 0x25, 0x3a, 0x74, 0xdf, 0x88,
		    0x38, 0xcc, 0x75, 0x34, 0xf1, 0x97, 0xcc, 0x0d },
		  { 0x96, 0xea, 0x5b, 0xd8, 0xc0, 0x36, 0x1a, 0x1f,
		    0xf8, 0xb7, 0x2b, 0xd0, 0x09, 0x4b, 0xe1, 0x7b }
		},
		{
		  "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
		  "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
		  "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
		  "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
		  "aaaaaaaaaaaaaaaaaaaa",
		  256,
		  0x9747b28c,
		  { 0xd3, 0xf2, 0xb7, 0xbb, 0xf6, 0x66, 0xc0, 0xcc,
		    0xd4, 0xa4, 0x00, 0x60, 0x5e, 0xc8, 0xd3, 0x2a },
		  { 0x07, 0xbd, 0x95, 0x7c, 0xa5, 0xde, 0xc1, 0xc4,
		    0xc4, 0xd8, 0xbb, 0x8d, 0x1f, 0x6c, 0xee, 0x55 }
		},
	};

	test_murmurhash3_algorithm("murmurhash3_128", murmurhash3_128,
				   MURMURHASH3_128_RESULTBYTES,
				   vectors, N_ELEMENTS(vectors));
}

void test_murmurhash3(void)
{
	test_murmurhash3_32();
	test_murmurhash3_128();
}
