// Copyright (c) .NET Foundation. All rights reserved.
// Licensed under the Apache License, Version 2.0. See License.txt in the project root for license information.

using FluentAssertions;
using NuGet.Protocol.Model;
using NuGet.Versioning;
using Xunit;

namespace NuGet.Protocol.Tests.Converters
{
    public class PackageVulnerabilityInfoConverterTests
    {
        [Fact]
        public void Deserialize_NormalJson_ReturnsValidObject()
        {
            string json = @"{
    ""url"": ""https://cve.test/1"",
    ""severity"": 2,
    ""versions"": ""(, 2.0.0)""
}";
            var result = json.FromJson<PackageVulnerabilityInfo>();

            result.Should().NotBeNull();
            result.Url.OriginalString.Should().Be("https://cve.test/1");
            result.Severity.Should().Be(PackageVulnerabilitySeverity.High);
            result.Versions.Should().Be(new VersionRange(maxVersion: new NuGetVersion(2, 0, 0), includeMaxVersion: false));
        }

        [Theory]
        [InlineData(null)]
        [InlineData(-1)]
        [InlineData(-2)]
        [InlineData(4)]
        public void Deserialize_JsonWithInvalidSeverity_ReturnsUnknownSeverity(int? severity)
        {
            string json = severity.HasValue
                ? $@"{{""url"":""https://cve.test/1"", ""versions"": ""(, 2.0.0)"", ""severity"": {severity.Value}}}"
                : @"{""url"":""https://cve.test/1"", ""versions"": ""(, 2.0.0)""}";

            var result = json.FromJson<PackageVulnerabilityInfo>();

            result.Should().NotBeNull();
            result.Severity.Should().Be(PackageVulnerabilitySeverity.Unknown);
        }

        [Fact]
        public void Serialize_NormalObject_ReturnsJson()
        {
            var vulnerabilityInfo = new PackageVulnerabilityInfo(
                new System.Uri("https://cve.test/1"),
                PackageVulnerabilitySeverity.High,
                new VersionRange(maxVersion: new NuGetVersion(2, 0, 0), includeMaxVersion: false));

            string json = vulnerabilityInfo.ToJson();

            const string expected = @"{""url"":""https://cve.test/1"",""severity"":2,""versions"":""(, 2.0.0)""}";
            Assert.Equal(expected, json);
        }
    }
}
