/*
 * Copyright (C) 2002 Stefan Holst
 * Copyright (C) 2025 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA.
 *
 * parsing mediamarks
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <dirent.h>
#include <sys/stat.h>
#include <libgen.h>

#include "common.h"
#include "../xine-toolkit/xitk.h"  /* XITK_NAME_MAX, xitk_lock_* */
#include "../xine-toolkit/cfg_parse.h"
#include "../kbindings.h"
#include "../mediamark.h"
#include "../playlist.h"
#include "../event.h"

#include "utils.h"
#include "oxine.h"
#include "otk.h"
#include "mediamarks.h"

#include <xine.h>
#include <xine/xineutils.h>
#include <xine/xmlparser.h>

/*
#define LOG
*/

/** playlist gui ***************************************************/

#define PL_ACTION_PLAY 1
#define PL_ACTION_REMOVE 2

typedef struct {
  oxine_session_t session;
  int playing, itool;
  otk_widget_t *list;
  mmk_string_list_t glist;
  const char *tools[3];
} oxine_playlist_session_t;

static void oxine_playlist_close (oxine_playlist_session_t *session) {
  session->session.oxine->main_window = NULL;
  session->session.oxine->info_window = NULL;
  otk_clear (session->session.oxine->otk);
  session->list = NULL;
}

static int oxine_playlist_change (oxine_playlist_session_t *session, int pos, int force);

static void playlist_play_cb (otk_widget_t *list, void *data, int action, int value) {
  oxine_playlist_session_t *session = (oxine_playlist_session_t *) data;
  oxine_t *oxine = session->session.oxine;
  gGui_t *gui = oxine->gui;

  if (list != session->list)
    return;
  if (action != OTK_ACT_VALUE)
    return;

  if (session->itool == PL_ACTION_PLAY) {
    if (!gui_playlist_play (gui, value)) {
      oxine_playlist_close (session);
    } /* else play_error(oxine); */
  }
  if (session->itool == PL_ACTION_REMOVE) {
    playlist_delete_entry (gui, value);
    if (oxine_playlist_change (session, value, 0))
      otk_draw_all (session->session.oxine->otk);
  }
}

static int oxine_playlist_change (oxine_playlist_session_t *session, int pos, int force) {
  otk_widget_t *list = session->list;
  gGui_t *gui = session->session.oxine->gui;

  /* printf ("gui.oxine.playlist.change (session=%p, list=%p, gen=%u).\n", (void *)session, (void *)session->list, session->gen); */
  if (!gui_playlist_get_strings (session->session.oxine->gui, &session->glist)) {
    if ((session->glist.cur == session->playing) && !force)
      return 0;
  }
  session->playing = session->glist.cur;
  if (gui->verbosity >= 2)
    printf ("gui.oxine.playlist.changed (%d, %d).\n", session->glist.cur, session->glist.used);
  otk_set_user_array (session->list, session->glist.ident, session->glist.used, session->glist.cur,
    playlist_play_cb, session);
  otk_list_set_pos (session->list, (pos >= 0) ? pos : session->glist.cur);

  otk_set_focus (list);
  return 1;
}

static void playlist_tool_changed (otk_widget_t *w, void *data, int action, int value) {
  oxine_playlist_session_t *session = (oxine_playlist_session_t *) data;

  (void)w;
  if (action == OTK_ACT_VALUE)
    session->itool = value;
}

static void playlist_to_mmk (void *data) {
  oxine_playlist_session_t *session = (oxine_playlist_session_t*) data;

  oxine_win_main (session->session.oxine, oxine_win_id_mediamarks, oxine_win_action_show);
}

static void playlist_to_main (void *data) {
  oxine_session_t *session = (oxine_session_t *) data;

  oxine_win_main (session->oxine, oxine_win_id_main, oxine_win_action_show);
}

static void playlist_clear_cb (void *data) {
  oxine_playlist_session_t *session = (oxine_playlist_session_t *) data;

  playlist_action (session->session.oxine->gui, PLAYLIST_CLEAR);
  session->session.main (session->session.oxine, oxine_win_id_playlist, oxine_win_action_update);
}

static int oxine_playlist_main (oxine_t *oxine, oxine_win_id_t id, oxine_win_action_t action) {
  oxine_playlist_session_t *session;
  if (id != oxine_win_id_playlist)
    return 0;
  if (!oxine)
    return 0;
  xitk_container (session, oxine->sessions[oxine_win_id_playlist], session);
  if (!session)
    return 0;
  switch (action) {
    case oxine_win_action_show:
      {
        otk_widget_t *b, *list_window;
        /* printf ("gui.oxine.playlist.reentry (session=%p, list=%p, gen=%u).\n",
           (void *)session, (void *)session->list, session->gen); */
        xitk_lock (oxine->xitk, 1);
        if (oxine->info_window && oxine->media_info_close_cb) {
          oxine->media_info_close_cb(oxine);
        }
        xitk_lock (oxine->xitk, 0);
        oxine_playlist_close (session);
        oxine->main_window = otk_window_new (oxine->otk, NULL, 20, 130, 215, 420);
        otk_label_new (oxine->main_window, 108, 60, OTK_ALIGN_CENTER|OTK_ALIGN_BOTTOM, _("Tool:"));
        b = otk_selector_new (oxine->main_window, 10, 80, 195, 60, session->tools, 2, playlist_tool_changed, session);
        otk_set_focus (b);
        otk_selector_set (b, session->itool);
        /*
        otk_button_new (oxine->main_window, 10, 150, 195, 60, "load", playlist_load_cb, session);
        otk_button_new (oxine->main_window, 10, 210, 195, 60, "Save", playlist_save_cb, session);
        */
        otk_button_new (oxine->main_window, 10, 170, 195, 60, _("Media"), playlist_to_mmk, session);
        otk_button_new (oxine->main_window, 10, 270, 195, 60, _("Clear"), playlist_clear_cb, session);
        otk_button_new (oxine->main_window, 10, 340, 195, 60, _("Main Menu"), playlist_to_main, &session->session);
        list_window = otk_window_new (oxine->otk, NULL, 245, 130, 535, 420);
        session->list = otk_list_new(list_window, 10, 15, 523, 390, NULL, session);
        oxine_playlist_change (session, -1, 1);
        otk_draw_all(oxine->otk);
        return 1;
      }
    case oxine_win_action_hide:
      oxine_playlist_close (session);
      return 1;
    case oxine_win_action_update:
      if (!session->list)
        return 0;
      if (!oxine_playlist_change (session, -1, 0))
        return 0;
      otk_draw_all (oxine->otk);
     return 1;
    case oxine_win_action_delete:
      oxine_playlist_close (session);
      oxine->sessions[oxine_win_id_playlist] = NULL;
      gui_playlist_free_strings (session->session.oxine->gui, &session->glist);
      ho_free (session);
      return 1;
    default: return 0;
  }
}

void oxine_playlist_init (oxine_t *oxine) {
  oxine_playlist_session_t *session = ho_new (oxine_playlist_session_t);
  if (!session)
    return;
  session->session.oxine = oxine;
  session->session.main = oxine_playlist_main;
  session->itool = 1;
  session->playing = -1;
  session->tools[0] = _("Play");
  session->tools[1] = _("Remove");
  session->tools[2] = NULL;

  gui_playlist_init_strings (&session->glist);
  oxine->sessions[oxine_win_id_playlist] = &session->session;
}

void oxine_playlist_update (oxine_t *oxine) {
  if (!oxine)
    return;
  if (xitk_lock (oxine->xitk, 200))
    return;
  if (oxine->last_win == oxine_win_id_playlist)
    oxine_playlist_main (oxine, oxine_win_id_playlist, oxine_win_action_update);
  /* printf ("gui.oxine.playlist.update (session=%p, list=%p, gen=%u).\n", (void *)session, (void *)session->list, session->gen); */
  xitk_lock (oxine->xitk, 0);
}

/** mediamarks gui *************************************************/

#define TYPE_NONE (0)
#define TYPE_DIR  (1)
#define TYPE_REG  (2)
#define TYPE_RDIR (3)
#define TYPE_RREG (4)
#define TYPE_UP   (5)
#define TYPE_MULTIPLE (6)
#define TYPE_M3U  (7)

/* sort methods 0 is default*/
#define PLAYITEM_SORT_FILES (1)
#define PLAYITEM_SORT_DIRFILES (0)

#define MM_ACTION_PLAY (1)
#define MM_ACTION_ADD (2)

typedef struct playitem_s playitem_t;

struct playitem_s {
  xitk_dnode_t node;
  playitem_t *parent;
  const char *title, *mrl;
  sized_list_t sub;
  int   type, lastpart;
  char buf[1];
};

typedef struct {
  oxine_session_t session;
  otk_widget_t   *list;
  int             listpos, itool;
  playitem_t     *backpath;
  const char      *tools[3];
  char            ilabel[64];
} oxine_mmk_session_t;

static void oxine_mmk_close (oxine_mmk_session_t *session) {
  session->session.oxine->main_window = NULL;
  session->session.oxine->info_window = NULL;
  otk_clear (session->session.oxine->otk);
  session->list = NULL;
}

/* private functions */
static int file_is_m3u(const char *mrl);


/* body of mediamarks functions */

/** implements case insensitive lengthlexicographic order. */
static int _playitem_cmp (playitem_t *a, playitem_t *b, int type) {
  /* printf("%s %s\n", a, b);
  stat(ita->mrl, &filestat1);
  stat(itb->mrl, &filestat2);*/
  if (type == PLAYITEM_SORT_DIRFILES) {
    int r = ((b->type == TYPE_DIR) || (b->type == TYPE_RDIR))
          - ((a->type == TYPE_DIR) || (a->type == TYPE_RDIR));
    if (r)
      return r;
  }
  return strcasecmp (a->mrl + a->lastpart, b->mrl + b->lastpart);
}

static void _playitem_sort_into (playitem_t *item, playitem_t *list, int type) {
  playitem_t *t;
  int a = 0, b = list->sub.num, m;

  if (!item->title)
    return;

  if (b > 0) {
    do {
      xitk_dnode_t *node;
      int d;

      m = (a + b) >> 1;
      node = sized_list_get (&list->sub, m);
      xitk_container (t, node, node);
      d = _playitem_cmp (item, t, type);
      if (d < 0) {
        b = m;
      } else {
        a = m + 1;
      }
    } while (a < b);
  }
  sized_list_insert (&list->sub, &item->node, a);
}

static playitem_t *playitem_new (int type, const char *title, const char *mrl, playitem_t *parent, int append) {
  size_t lm;
  playitem_t *item;

  if (!title)
    title = "";
  if (!mrl)
    mrl = "";
  lm = strlen (mrl) + 1;
  if ((title >= mrl) && (title < mrl + lm)) {
    item = malloc (sizeof (*item) + lm);
    if (!item)
      return NULL;
    memcpy (item->buf, mrl, lm);
    item->mrl = item->buf;
    item->lastpart = title - mrl;
    item->title = item->buf + item->lastpart;
  } else {
    char *q;
    size_t lt = strlen (title) + 1;
    item = malloc (sizeof (*item) + lt + lm);
    if (!item)
      return NULL;
    item->lastpart = 0; /** << can be set later to speed up mrl cmp. */
    q = item->buf;
    item->title = q; memcpy (q, title, lt); q += lt;
    item->mrl   = q; memcpy (q, mrl, lm); /* q += lm; */
  }
  xitk_dnode_init (&item->node);
  sized_list_init (&item->sub);
  item->parent = parent;
  item->type = type;

  if (append && parent)
    sized_list_insert (&parent->sub, &item->node, parent->sub.num);

  return item;
}

static void free_subtree (playitem_t *item, int this1too);

static void _free_subtree (xitk_dnode_t *node) {
  playitem_t *item;
  xitk_container (item, node, node);
  if (item->type != TYPE_UP)
    free_subtree (item, 0);
  free (item);
}

static void free_subtree (playitem_t *item, int thistoo) {
  if (!item)
    return;
  sized_list_free (&item->sub, _free_subtree);
  if (thistoo) {
    xitk_dnode_remove (&item->node);
    free (item);
  }
}

static playitem_t *playitem_load (xml_node_t *node, playitem_t *parent) {
  const char *title = NULL, *mrl = NULL;
  int _type = 0;

  while (node) {
    if (!strcasecmp (node->name, "title")) {
      title = node->data;
#ifdef LOG
      printf ("mediamarks: title = %s\n", title);
#endif
    } else if (!strcasecmp (node->name, "ref")) {
      const char *type = xml_parser_get_property (node, "type");
      mrl = xml_parser_get_property (node, "href");
      if (type) {
	if (!strcasecmp(type, "multiple")) {
          _type = TYPE_MULTIPLE;
	}
      } else {
        struct stat st;
	st.st_mode = 0;
	stat (mrl, &st);
	if (S_ISDIR (st.st_mode)) {
          _type = TYPE_RDIR;
	} else {
          _type = TYPE_REG;
          if (file_is_m3u (mrl))
            _type = TYPE_M3U;
	}
      }

#ifdef LOG
      printf ("mediamarks: mrl   = %s\n", mrl);
#endif
    } else if (!strcasecmp (node->name, "time")) {
    } else {
      printf ("mediamarks: error while loading mediamarks file, unknown node '%s'\n", node->name);
    }
    node = node->next;
  }

  return playitem_new (_type, title, mrl, parent, 1);
}

static int parse_multiple (oxine_t *oxine, const char *mrl, playitem_t *parent) {
  int i = 0, num = 0;
#if XINE_MAJOR_VERSION < 1 || (XINE_MAJOR_VERSION == 1 && XINE_MINOR_VERSION < 2)
  char **str;
#else
  const char * const *str;
#endif

  str = xine_get_autoplay_mrls (oxine->xine, mrl, &num);
  if (num <= 0)
    return 0;

  free_subtree (parent, 0);

  while ((i < num) && playitem_new (TYPE_REG, str[i], str[i], parent, 1))
    i++;
  return 1;
}

static int read_directory(oxine_t *oxine, const char *dname, playitem_t *parent) {
  char buf[XITK_PATH_MAX + XITK_NAME_MAX + 2], *add, tbuf[512], *title;
  DIR *dir;
  struct dirent *ent;
  playitem_t *item;

  free_subtree (parent, 0);

  /* xine_config_lookup_entry (oxine->xine, "oxine.sort_type", &entry); */
#if 0
  printf("Processing %s\n", dir);
#endif
  dir = opendir (dname);
  if (!dir) {
    int e = errno;
    printf ("gui.oxine.mediamarks.dir.read.failed (%s, %d (%s)).\n", dname, e, strerror (e));
    return 0;
  }
  add = buf + strlcpy (buf, dname, XITK_PATH_MAX + 1);
  if (add > buf + XITK_PATH_MAX + 1)
    add = buf + XITK_PATH_MAX + 1;
  memcpy (add, "/", 2); add++;

  while ((ent = readdir (dir))) {
    struct stat st;
    int type = 0;

    if (ent->d_name[0] == '.') {
#ifdef SHOW_HIDDEN_FILES
      if (!ent->d_name[1] || ((ent->d_name[1] == '.') && !ent->d_name[2]))
#endif
        continue;
    }

    if (strlcpy (add, ent->d_name, XITK_NAME_MAX + 1) < 1)
      continue;
    if (stat (buf, &st) < 0)
      continue;

    if (file_is_m3u (buf)) {
      title = tbuf;
      *title++ = '[';
      title += strlcpy (title, ent->d_name, sizeof (tbuf) - 3);
      if (title > tbuf + sizeof (tbuf) - 2)
        title = tbuf + sizeof (tbuf) - 2;
      memcpy (title, "]", 2);
      title = tbuf;
      type = TYPE_M3U;
    } else if (S_ISDIR (st.st_mode)) {
      title = tbuf;
      *title++ = '[';
      title += strlcpy (title, ent->d_name, sizeof (tbuf) - 3);
      if (title > tbuf + sizeof (tbuf) - 2)
        title = tbuf + sizeof (tbuf) - 2;
      memcpy (title, "]", 2);
      title = tbuf;
      type = TYPE_RDIR;
    } else if (S_ISREG (st.st_mode)) {
      title = add;
      type = TYPE_RREG;
    } else {
      continue;
    }
    item = playitem_new (type, title, buf, parent, 0);
    if (!item)
      continue;
#if 0
    printf ("mrl   : %s\n", item->mrl);
    printf ("title : %s\n", item->title);
    printf ("type  : %d\n", item->type);
    printf ("ei %d\n", oxine->mm_sort_type);
#endif
    item->lastpart = add - buf;
    _playitem_sort_into (item, parent, oxine->mm_sort_type);
  }
  closedir (dir);
  return 1;
}

static int file_is_m3u(const char *mrl) {

#ifdef M3U_AS_SUBDIR

  FILE *file;
  char **line;
  int *n;
  int a;

  file = fopen(mrl, "r");
  if(!file) return 0;

  n = ho_new(size_t);
  line = ho_new(char *);

  *line = NULL;
  *n = 0;
  a = getline(line, n, file);

  fclose(file);

  if(a<=0) {
    ho_free(n);
    ho_free(line);
    return 0;
  }
  if(!strncmp(*line, "#EXTM3U", 7)) {
    ho_free(n);
    ho_free(line);
    return 1;
  }

  ho_free(n);
  ho_free(line);

  if( strstr(mrl,".m3u") != NULL || strstr(mrl,".M3U") )
    return 1;
  else
    return 0;

#else

  (void)mrl;
  return 0; /* m3u is a normal file here, let xine-ui handle it as playlist */

#endif
}

static void parse_m3u(const char *mrl, playitem_t *parent) {
  char *text, *line, *next;
  size_t len = 2 << 20;

  text = next = xitk_cfg_load (mrl, &len);
  if (!next)
    return;
  while ((line = xitk_cfg_next_line (&next))) {
    if (line[0] == '#')
      continue;
    playitem_new (TYPE_REG, basename (line), line, parent, 1);
  }
  xitk_cfg_unload (text);
}

static void read_subs(xml_node_t *node, playitem_t *parent) {
  playitem_t *item = NULL;

  while (node) {
    if (!strcasecmp (node->name, "entry")) {
      item = playitem_load (node->child, parent);
    } else if(!strcasecmp (node->name, "sub")) {
      char title[256];

      snprintf(title, 255, "[%s]", xml_parser_get_property (node, "name"));
      item = playitem_new (TYPE_DIR, title, NULL, parent, 1);
      read_subs (node->child, item);
    }
    node=node->next;
  }
}

static int read_mediamarks (playitem_t *parent, const char *mrl) {
  xml_node_t *node;
  char *text;
  size_t size = 2 << 20;
  int r = 0;

  text = xitk_cfg_load (mrl, &size);
  if (!text)
    return 0;

  xml_parser_init_R (xml_parser_t *xml, text, size, XML_PARSER_CASE_INSENSITIVE);
  if (xml_parser_build_tree_R (xml, &node) < 0) {
    printf ("gui.oxine.mediamarks.xml.parse.failed (%s).\n", mrl);
  } else {
    if (strcasecmp (node->name, "oxinemm")) {
      printf ("gui.oxine.mediamarks.read.invalid (root node must be OXINEMM).\n");
    } else {
      read_subs (node->child, parent);
      r = 1;
    }
    xml_parser_free_tree (node);
  }
  xml_parser_finalize_R (xml);
  xitk_cfg_unload (text);
  return r;
}

static void *mediamarks_get_ptr (xitk_dnode_t *node, otk_get_ptr_what_t what) {
  playitem_t *item;

  xitk_container (item, node, node);
  switch (what) {
    case OTK_GET_PTR_ITEM: return item;
    case OTK_GET_PTR_TEXT: return (void *)(const void *)item->title;
    default: return NULL;
  }
}

static const char * const upstr = "[..]";

static void changelist (oxine_mmk_session_t *session) {
  playitem_t *item;
  xitk_dnode_t *node;
  /** add 1 up entry if needed and not already there. */
  if (session->backpath->parent) {
    item = NULL;
    node = session->backpath->sub.list.head.next;
    if (node->next) {
      xitk_container (item, node, node);
      if (item->title != upstr)
        item = NULL;
    }
    if (!item) {
      item = playitem_new (TYPE_UP, upstr, session->backpath->mrl, session->backpath, 0);
      if (item) {
        item->title = upstr;
        sized_list_insert (&session->backpath->sub, &item->node, 0);
      }
    }
  }
  otk_set_user_list (session->list, &session->backpath->sub, mediamarks_get_ptr);
  otk_list_set_pos (session->list,0);
  otk_set_focus (session->list);
}

static void mmk_tool_changed (otk_widget_t *w, void *data, int action, int value) {
  oxine_mmk_session_t *session = (oxine_mmk_session_t *) data;

  (void)w;
  if (action == OTK_ACT_VALUE)
    session->itool = value;
}

static void set_ilabel (oxine_mmk_session_t *session) {
  snprintf (session->ilabel, sizeof (session->ilabel), "%s (%d)",
    _("List"), session->session.oxine->gui->playlist.num);
}

static void mediamarks_play_cb(void *data, void *entry) {
  oxine_mmk_session_t *session = (oxine_mmk_session_t *) data;
  oxine_t *oxine = session->session.oxine;
  playitem_t *item = (playitem_t*) entry;

  session->listpos = otk_list_get_pos (session->list);
#ifdef LOG
  printf("mediamarks: mediamarks_play_cb %s\n", item->mrl);
  fflush(NULL);
#endif

  if (item->type == TYPE_DIR) {
    session->backpath = item;
    changelist (session);
    otk_draw_all(oxine->otk);
  } else if (item->type == TYPE_RDIR) {
    if (read_directory (oxine, item->mrl, item)) {
      session->backpath = item;
      changelist (session);
      otk_draw_all(oxine->otk);
    }
  } else if (item->type == TYPE_UP) {
    session->backpath = session->backpath->parent;
    changelist (session);
    otk_draw_all(oxine->otk);
  } else if (session->itool == MM_ACTION_PLAY) {
    if ((item->type == TYPE_REG)||(item->type == TYPE_RREG)) {
      set_ilabel (session);
      printf("mediamarks: playing %s\n", item->mrl);
/*
      if (is_movie(item->mrl)) {
        oxine_mmk_close (session);
        if (oxine->filename)
          ho_free(oxine->filename);
        oxine->filename=ho_strdup(item->mrl);
        select_subtitle_cb(get_dirname(oxine->filename), oxine, 0);
      } else
*/
      if (odk_open_and_play (oxine->odk, item->mrl)) {
        oxine_mmk_close (session);
      } else {
        /* play_error (oxine); */
      }
    } else if (item->type == TYPE_M3U) {
      parse_m3u (item->mrl, item);
      session->backpath = item;
      changelist (session);
      otk_draw_all(oxine->otk);
    } else if (item->type == TYPE_MULTIPLE) {
      if (parse_multiple (oxine, item->mrl, item)) {
        session->backpath = item;
        changelist (session);
        otk_draw_all(oxine->otk);
      } else
        printf("mediamarks: error getting autoplay mrls\n");
    } else printf("mediamarks: %s is of unknown type\n", item->mrl);
  } else if (session->itool == MM_ACTION_ADD) {
    if ((item->type == TYPE_REG)||(item->type == TYPE_RREG)) {
      odk_enqueue(session->session.oxine->odk, item->mrl); /* FIXME: item->title unused */
      set_ilabel(session);
      otk_draw_all(session->session.oxine->otk);
    }
  }
}

static void mediamarks_to_playlist (void *data) {
  oxine_mmk_session_t *session = (oxine_mmk_session_t*) data;

  oxine_win_main (session->session.oxine, oxine_win_id_playlist, oxine_win_action_show);
}

static int oxine_mmk_main (oxine_t *oxine, oxine_win_id_t id, oxine_win_action_t action) {
  oxine_mmk_session_t *session;
  if (id != oxine_win_id_mediamarks)
    return 0;
  if (!oxine)
    return 0;
  xitk_container (session, oxine->sessions[oxine_win_id_mediamarks], session);
  if (!session)
    return 0;
  switch (action) {
    case oxine_win_action_show:
      if (!session->backpath) {
        char mmpath[XITK_PATH_MAX + XITK_NAME_MAX];
        playitem_t *root = playitem_new (TYPE_DIR, "[mediamarks]", NULL, NULL, 0);
        snprintf (mmpath, sizeof (mmpath), "%s/.xine/oxine/mediamarks", xine_get_homedir ());
        if (!read_mediamarks (root, mmpath)) {
          lprintf ("trying to load system wide mediamarks\n");
          snprintf (mmpath, sizeof (mmpath), "%s/mediamarks", XINE_OXINEDIR);
          if (!read_mediamarks (root, mmpath)) {
            free_subtree (root, 1);
            return 0;
          }
        }
        session->backpath = root;
      }
      {
        otk_widget_t *b, *list_window;
        xitk_lock (oxine->xitk, 1);
        if (oxine->info_window && oxine->media_info_close_cb)
          oxine->media_info_close_cb (oxine);
        xitk_lock (oxine->xitk, 0);
        oxine->main_window = otk_window_new (oxine->otk, NULL, 20, 130, 225, 420);
        otk_label_new (oxine->main_window, 108, 60, OTK_ALIGN_CENTER|OTK_ALIGN_BOTTOM, _("Tool:"));
        b = otk_selector_new (oxine->main_window, 10, 80, 195, 60, session->tools, 2, mmk_tool_changed, session);
        otk_set_focus (b);
        otk_selector_set (b, session->itool);
        set_ilabel (session);
        otk_button_new (oxine->main_window, 10, 170, 195, 60, session->ilabel, mediamarks_to_playlist, session);
        /* otk_label_new (oxine->main_window, 108, 280, OTK_ALIGN_CENTER|OTK_ALIGN_BOTTOM, session->ilabel); */
        otk_button_new (oxine->main_window, 10, 340, 195, 60, _("Main Menu"), playlist_to_main, &session->session);
        list_window = otk_window_new (oxine->otk, NULL, 245, 130, 535, 420);
        session->list = otk_list_new(list_window, 10, 15, 523, 390, mediamarks_play_cb, session);
        changelist (session);
        otk_list_set_pos (session->list, session->listpos);
      }
      otk_draw_all (oxine->otk);
      return 1;
    case oxine_win_action_hide:
      oxine_mmk_close (session);
      return 1;
    case oxine_win_action_update:
     return 1;
    case oxine_win_action_delete:
      oxine_mmk_close (session);
      oxine->sessions[oxine_win_id_mediamarks] = NULL;
      {
        playitem_t *item = session->backpath;
        while (item && item->parent)
          item = item->parent;
        free_subtree (item, 1);
        session->backpath = NULL;
      }
      ho_free (session);
      return 1;
    default: return 0;
  }
}

void oxine_mediamarks_init (oxine_t *oxine) {
  oxine_mmk_session_t *session;

  if (!oxine)
    return;
  session = ho_new (oxine_mmk_session_t);
  if (!session)
    return;

  session->session.oxine = oxine;
  session->session.main = oxine_mmk_main;
  session->backpath = NULL;
  session->itool = 1;
  session->listpos = 0;
  session->tools[0] = _("Play");
  session->tools[1] = _("Add");
  session->tools[2] = NULL;

  oxine->sessions[oxine_win_id_mediamarks] = &session->session;
}
