/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 * MRL Browser
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <errno.h>

#include "common.h"
#include "config_wrapper.h"
#include "xine-toolkit/backend.h"
#include "xine-toolkit/mrlbrowser.h"
#include "mediamark.h"
#include "mrl_browser.h"
#include "file_browser.h"
#include "osd.h"
#include "panel.h"
#include "playlist.h"
#include "actions.h"
#include "event.h"
#include "errors.h"

struct xui_mrlb_st {
  gGui_t *gui;
  xitk_widget_t *w;
  xitk_window_t *xwin;

  const char **filter_names;
};

/*
 *
 */
void mrl_browser_change_skins (xui_mrlb_t *mrlb, int synthetic) {
  (void)synthetic;
  if (mrlb && mrlb->w) {
    xitk_mrlbrowser_change_skins (mrlb->w, mrlb->gui->skin_config);
    /* this annoys the user, and it interferes with widget list repaint.
    gui_raise_window (mrlb->gui, mrlb->xwin); */
  }
}

/*
 *
 */
static void _mrl_browser_kill (xitk_widget_t *w, void *data) {
  xui_mrlb_t *mrlb = data;

  (void)w;
  if (mrlb) {
    if (mrlb->xwin) {
      xitk_rect_t wr;

      wr.width = 0;
      xitk_window_get_window_position (mrlb->xwin, &wr);
      if (wr.width) {
        config_update_num (mrlb->gui->xine, "gui.mrl_browser_x", wr.x);
        config_update_num (mrlb->gui->xine, "gui.mrl_browser_y", wr.y);
      }
      mrlb->xwin = NULL;
    }
    filebrowser_ext_list_unget (mrlb->gui, &mrlb->filter_names);
    mrlb->gui->mrlb = NULL;
    free (mrlb);
  }
}

static void _mrl_browser_rpwin (void *data, xitk_window_t *xwin) {
  xui_mrlb_t *mrlb = (xui_mrlb_t *)data;
  gui_raise_window (mrlb->gui, xwin);
}

static int _mrl_browser_input (void *data, const xitk_be_event_t *event) {
  xui_mrlb_t *mrlb = (xui_mrlb_t *)data;
  return gui_handle_be_event (mrlb->gui, event);
}

static int _mrl_browser_match (void *data, const char *name, uint32_t n) {
  xui_mrlb_t *mrlb = (xui_mrlb_t *)data;
  return filebrowser_ext_match (mrlb->gui, name, n);
}

static void _mrlb_add (xitk_widget_t *w, void *data, const xine_mrl_t *mrl) {
  xui_mrlb_t *mrlb = data;

  (void)w;
  if (mrlb && mrl) {
    uint32_t mode;
    /* user may want to add multiple items, so dont upfront/focus
     * an already open playlist editor. */
    if (!mrlb->gui->plwin)
      playlist_main (XUI_W_ON, mrlb->gui);
    /* just add this entry, do not play immediateöy. */
    gui_playlist_lock (mrlb->gui);
    mode = mrlb->gui->playlist.control;
    mrlb->gui->playlist.control = mode | PLAYLIST_CONTROL_IGNORE;
    gui_playlist_unlock (mrlb->gui);
    gui_dndcallback (mrlb->gui, mrl->mrl);
    gui_playlist_lock (mrlb->gui);
    mrlb->gui->playlist.control = mode;
    gui_playlist_unlock (mrlb->gui);
  }
}

/*
 * Callback called by mrlbrowser on play event.
 */
static void _mrlb_play (xitk_widget_t *w, void *data, const xine_mrl_t *mrl) {
  xui_mrlb_t *mrlb = data;

  (void)w;
  if (mrlb && mrl) {
    const char *_mrl = mrl->mrl, *_ident;

    {
      const uint8_t *p = (const uint8_t *)_mrl;
      _ident = _mrl;
      while (1) {
        while (*p > '/')
          p++;
        if ((p[0] == '/') && p[1])
          _ident = (const char *)p + 1;
        else if (!*p)
          break;
        p++;
      }
    }

    if (mrl_look_like_playlist (mrlb->gui, _mrl) && gui_playlist_add_item (mrlb->gui, _mrl, 1, GUI_ITEM_TYPE_PLAYLIST, 1)) {
      gui_current_set_index (mrlb->gui, GUI_MMK_CURRENT);
      playlist_update_playlist (mrlb->gui);
    } else {
      mediamark_t *mmk = &mrlb->gui->mmk;
      gui_playlist_lock (mrlb->gui);
      mediamark_clear (mmk);
      mediamark_set_str_val (&mmk, _mrl, MMK_VAL_MRL);
      mediamark_set_str_val (&mmk, _ident, MMK_VAL_IDENT);
      gui_playlist_unlock (mrlb->gui);
    }

    osd_hide (mrlb->gui);
    panel_playback_ctrl (mrlb->gui->panel, gui_playlist_play_current (mrlb->gui) ? 0 : 1);
    panel_message (mrlb->gui->panel, NULL);
  }
}

/*
 *
 */
void mrl_browser_main (xitk_widget_t *mode, void *data) {
  gGui_t *gui = data;
  xui_mrlb_t *mrlb;
  const char * const *ip_availables;

  if (!gui)
    return;

  mrlb = gui->mrlb;
  if (mode == XUI_W_OFF) {
    if (!mrlb)
      return;
    xitk_widgets_delete (&mrlb->w, 1);
    return;
  } else if (mode == XUI_W_ON) {
    if (mrlb) {
      if (mrlb->w) {
        mrlb->gui->nongui_error_msg = NULL;
        xitk_widgets_state (&mrlb->w, 1, XITK_WIDGET_STATE_VISIBLE, XITK_WIDGET_STATE_VISIBLE);
        xitk_window_raise_window (mrlb->xwin);
        xitk_window_set_input_focus (mrlb->xwin);
      }
      return;
    }
  } else { /* toggle */
    if (mrlb) {
      xitk_widgets_delete (&mrlb->w, 1);
      return;
    }
  }

  mrlb = calloc (1, sizeof (*mrlb));
  if (!mrlb)
    return;
  mrlb->gui = gui;

  mrlb->filter_names = filebrowser_ext_list_get (mrlb->gui);

  ip_availables = xine_get_browsable_input_plugin_ids (mrlb->gui->xine);

  {
    xitk_mrlbrowser_widget_t mb = {
        .nw = { .userdata = mrlb, .skin_element_name = "MrlBG" },

        .reparent_window = _mrl_browser_rpwin,
        .layer_above  = gui_layer_above (mrlb->gui, NULL),
        .icon         = mrlb->gui->icon,
        .x = 200,
        .y = 100,
        .resource_class = "xine",
        .winfocus = { .skin_element_name = "MrlWF" },

        .origin = { .skin_element_name = "MrlCurOrigin" },

        .input_cb = _mrl_browser_input,

        .select = {
          .skin_element_name = "MrlSelect",
          .callback = _mrlb_add
        },

        .play = {
          .skin_element_name = "MrlPlay",
          .callback = _mrlb_play
        },

        .dismiss = {
          .skin_element_name = "MrlDismiss",
        },

        .kill = { .callback = _mrl_browser_kill },
 
        .ip_availables = ip_availables,
        .ip_name = {
          .button = { .skin_element_name = "MrlPlugNameBG" },
          .label = { .skin_element_name  = "MrlPlugLabel" }
        },

        .xine = mrlb->gui->xine,

        .browser = {
          .nw = { .skin_element_name = "MrlItemBtn", .userdata = mrlb },
          .arrow_up = { .skin_element_name = "MrlUp" },
          .slider = { .skin_element_name = "SliderMrl" },
          .arrow_dn = { .skin_element_name = "MrlDn" },
          .arrow_left = { .skin_element_name  = "MrlLeft" },
          .slider_h = { .skin_element_name = "SliderHMrl" },
          .arrow_right = { .skin_element_name = "MrlRight" },
          .browser = {
            .num_entries = 0,
            .entries  = NULL,
          },
          .callback = NULL
        },

        .combo = { .skin_element_name = "MrlFilt" },

        .filter = { .names = mrlb->filter_names, .match = _mrl_browser_match }
    };

    gui_load_window_pos (mrlb->gui, "mrl_browser", &mb.x, &mb.y);

    mb.window_title = mb.resource_name = _("xine MRL Browser");
    mb.select.caption = _("Add");
    mb.dismiss.caption = _("Dismiss");

    /* NOTE: pgettext () may be a macro that does not work as initializer. */
    /* TRANSLATORS: only ASCII characters (skin) */
    mb.ip_name.label.label_str  = pgettext ("skin", "Source:");

    mrlb->w = xitk_mrlbrowser_create (mrlb->gui->xitk, &mb, mrlb->gui->skin_config);
  }

  mrlb->xwin = xitk_mrlbrowser_get_window (mrlb->w);

  mrlb->gui->mrlb = mrlb;
}

void open_mrlbrowser_from_playlist (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  gGui_t *gui = data;

  (void)w;
  (void)state;
  (void)modifier;
  mrl_browser_main (XUI_W_ON, gui);
}

