// @HEADER
// ************************************************************************
//
//               Rapid Optimization Library (ROL) Package
//                 Copyright (2014) Sandia Corporation
//
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact lead developers:
//              Drew Kouri   (dpkouri@sandia.gov) and
//              Denis Ridzal (dridzal@sandia.gov)
//
// ************************************************************************
// @HEADER

/** \file
    \brief  Contains definitions for Poisson optimal control.
    \author Created by D. Ridzal and D. Kouri.
 */

#ifndef USE_HESSVEC 
#define USE_HESSVEC 1
#endif

#ifndef ROL_POISSONCONTROL_HPP
#define ROL_POISSONCONTROL_HPP

#include "ROL_StdVector.hpp"
#include "ROL_Objective.hpp"

namespace ROL {
namespace ZOO {

/** \brief Poisson distributed control.
 */
template<class Real>
class Objective_PoissonControl : public Objective<Real> {

typedef std::vector<Real>  vector;
typedef Vector<Real>       V;
typedef StdVector<Real>    SV;  

typedef typename vector::size_type uint;

private:
  Real alpha_;

  Teuchos::RCP<const vector> getVector( const V& x ) {
    using Teuchos::dyn_cast;
    return dyn_cast<const SV>(x).getVector();
  }

  Teuchos::RCP<vector> getVector( V& x ) {
    using Teuchos::dyn_cast;
    return dyn_cast<SV>(x).getVector();
  }

public:

  Objective_PoissonControl(Real alpha = 1.e-4) : alpha_(alpha) {}

  void apply_mass(Vector<Real> &Mz, const Vector<Real> &z ) {

    using Teuchos::RCP;
    RCP<const vector> zp = getVector(z);
    RCP<vector> Mzp = getVector(Mz);

    uint  n = zp->size();
    Real h = 1.0/((Real)n+1.0);
    for (uint i=0; i<n; i++) {
      if ( i == 0 ) {
        (*Mzp)[i] = h/6.0*(4.0*(*zp)[i] + (*zp)[i+1]);
      }
      else if ( i == n-1 ) {
        (*Mzp)[i] = h/6.0*((*zp)[i-1] + 4.0*(*zp)[i]);
      }
      else {
        (*Mzp)[i] = h/6.0*((*zp)[i-1] + 4.0*(*zp)[i] + (*zp)[i+1]);
      }
    }
  }

  void solve_poisson(Vector<Real> & u, const Vector<Real> & z) {

    using Teuchos::RCP;
    using Teuchos::rcp;

    RCP<vector> up = getVector(u);

    uint  n = up->size();
    Real h = 1.0/((Real)n+1.0);
    SV b( Teuchos::rcp( new vector(n,0.0) ) );
    RCP<vector> bp = getVector(b);
    apply_mass(b,z);

    Real d   =  2.0/h;
    Real o   = -1.0/h;
    Real m   = 0.0;
    vector c(n,o);
    c[0]     = c[0]/d;
    (*up)[0] = (*bp)[0]/d;
    for ( uint i = 1; i < n; i++ ) {
      m        = 1.0/(d - o*c[i-1]);
      c[i]     = c[i]*m;
      (*up)[i] = ( (*bp)[i] - o*(*up)[i-1] )*m;
    }
    for ( uint i = n-1; i > 0; i-- ) {
      (*up)[i-1] = (*up)[i-1] - c[i-1]*(*up)[i];
    }
  }

  Real evaluate_target(Real x) {
    Real val = 1.0/3.0*std::pow(x,4.0) - 2.0/3.0*std::pow(x,3.0) + 1.0/3.0*x + 8.0*alpha_;
    return val;
  }

  Real value( const Vector<Real> &z, Real &tol ) {

    using Teuchos::RCP;
    using Teuchos::rcp;
    RCP<const vector> zp = getVector(z);
    uint n    = zp->size();
    Real h    = 1.0/((Real)n+1.0);
    // SOLVE STATE EQUATION
    SV u( rcp( new vector(n,0.0) ) );
    solve_poisson(u,z);
    RCP<vector> up = getVector(u);

    Real val  = 0.0;
    Real res  = 0.0;
    Real res1 = 0.0;
    Real res2 = 0.0;
    Real res3 = 0.0;
    for (uint i=0; i<n; i++) {
      res = alpha_*(*zp)[i];
      if ( i == 0 ) {
        res *= h/6.0*(4.0*(*zp)[i] + (*zp)[i+1]);
        res1 = (*up)[i]-evaluate_target((Real)(i+1)*h);
        res2 = (*up)[i+1]-evaluate_target((Real)(i+2)*h);
        res += h/6.0*(4.0*res1 + res2)*res1;
      }
      else if ( i == n-1 ) {
        res *= h/6.0*((*zp)[i-1] + 4.0*(*zp)[i]);
        res1 = (*up)[i-1]-evaluate_target((Real)(i)*h);
        res2 = (*up)[i]-evaluate_target((Real)(i+1)*h);
        res += h/6.0*(res1 + 4.0*res2)*res2;
      }
      else {
        res *= h/6.0*((*zp)[i-1] + 4.0*(*zp)[i] + (*zp)[i+1]);
        res1 = (*up)[i-1]-evaluate_target((Real)(i)*h);
        res2 = (*up)[i]-evaluate_target((Real)(i+1)*h);
        res3 = (*up)[i+1]-evaluate_target((Real)(i+2)*h);
        res += h/6.0*(res1 + 4.0*res2 + res3)*res2;
      }
      val += 0.5*res;
    }
    return val;
 }

  void gradient( Vector<Real> &g, const Vector<Real> &z, Real &tol ) {

    using Teuchos::RCP;
    using Teuchos::rcp;
    RCP<const vector> zp = getVector(z);
    RCP<vector> gp = getVector(g);

    uint  n = zp->size();
    Real h = 1.0/((Real)n+1.0);

    // SOLVE STATE EQUATION
    SV u( rcp( new vector(n,0.0) ) );
    solve_poisson(u,z);
    RCP<vector> up = getVector(u);

    // SOLVE ADJOINT EQUATION
    StdVector<Real> res( Teuchos::rcp( new std::vector<Real>(n,0.0) ) );
    RCP<vector> rp = getVector(res);

    for (uint i=0; i<n; i++) {
      (*rp)[i] = -((*up)[i]-evaluate_target((Real)(i+1)*h));
    }

    SV p( rcp( new vector(n,0.0) ) );
    solve_poisson(p,res);
    RCP<vector> pp = getVector(p);

    Real res1 = 0.0;
    Real res2 = 0.0;
    Real res3 = 0.0;
    for (uint i=0; i<n; i++) {
      if ( i == 0 ) {
        res1 = alpha_*(*zp)[i] - (*pp)[i];
        res2 = alpha_*(*zp)[i+1] - (*pp)[i+1];
        (*gp)[i] = h/6.0*(4.0*res1 + res2);
      }
      else if ( i == n-1 ) {
        res1 = alpha_*(*zp)[i-1] - (*pp)[i-1];
        res2 = alpha_*(*zp)[i] - (*pp)[i];
        (*gp)[i] = h/6.0*(res1 + 4.0*res2);
      }
      else {
        res1 = alpha_*(*zp)[i-1] - (*pp)[i-1];
        res2 = alpha_*(*zp)[i] - (*pp)[i];
        res3 = alpha_*(*zp)[i+1] - (*pp)[i+1];
        (*gp)[i] = h/6.0*(res1 + 4.0*res2 + res3);
      }
    }
  }
#if USE_HESSVEC
  void hessVec( Vector<Real> &hv, const Vector<Real> &v, const Vector<Real> &z, Real &tol ) {

    using Teuchos::RCP;
    using Teuchos::rcp;
    RCP<const vector> zp = getVector(z);
    RCP<const vector> vp = getVector(v);
    RCP<vector> hvp = getVector(hv);

    uint  n = zp->size();
    Real h = 1.0/((Real)n+1.0);

    // SOLVE STATE EQUATION
    SV u( rcp( new vector(n,0.0) ) );
    solve_poisson(u,v);
    RCP<vector> up = getVector(u);

    // SOLVE ADJOINT EQUATION
    SV p( rcp( new vector(n,0.0) ) );

    solve_poisson(p,u);
    RCP<vector> pp = getVector(p);  

    Real res1 = 0.0;
    Real res2 = 0.0;
    Real res3 = 0.0;
    for (uint i=0; i<n; i++) {
      if ( i == 0 ) {
        res1 = alpha_*(*vp)[i] + (*pp)[i];
        res2 = alpha_*(*vp)[i+1] + (*pp)[i+1];
        (*hvp)[i] = h/6.0*(4.0*res1 + res2);
      }
      else if ( i == n-1 ) {
        res1 = alpha_*(*vp)[i-1] + (*pp)[i-1];
        res2 = alpha_*(*vp)[i] + (*pp)[i];
        (*hvp)[i] = h/6.0*(res1 + 4.0*res2);
      }
      else {
        res1 = alpha_*(*vp)[i-1] + (*pp)[i-1];
        res2 = alpha_*(*vp)[i] + (*pp)[i];
        res3 = alpha_*(*vp)[i+1] + (*pp)[i+1];
        (*hvp)[i] = h/6.0*(res1 + 4.0*res2 + res3);
      }
    }
  }
#endif
};

template<class Real>
void getPoissonControl( Teuchos::RCP<Objective<Real> > &obj,
                        Teuchos::RCP<Vector<Real> >    &x0,
                        Teuchos::RCP<Vector<Real> >    &x ) {
  // Problem dimension
  int n = 512;

  // Get Initial Guess
  Teuchos::RCP<std::vector<Real> > x0p = Teuchos::rcp(new std::vector<Real>(n,0.0));
  for (int i=0; i<n; i++) {
    (*x0p)[i] = 0.0;
  }
  x0 = Teuchos::rcp(new StdVector<Real>(x0p));

  // Get Solution
  Teuchos::RCP<std::vector<Real> > xp = Teuchos::rcp(new std::vector<Real>(n,0.0));
  Real h = 1.0/((Real)n+1.0), pt = 0.0;
  for( int i = 0; i < n; i++ ) {
    pt = (Real)(i+1)*h;
    (*xp)[i] = 4.0*pt*(1.0-pt);
  }
  x = Teuchos::rcp(new StdVector<Real>(xp));

  // Instantiate Objective Function
  obj = Teuchos::rcp(new Objective_PoissonControl<Real>);
}

} // End ZOO Namespace
} // End ROL Namespace

#endif
