#ifndef FIBERS_H
#define FIBERS_H

#include <dolfin.h>
#include <cmath>

class FiberFieldBase: public Function
{
protected:
  double theta, phi;
  double ct, st, cp, sp;

public:
  FiberFieldBase(const FunctionSpace & V, double theta, double phi):
    Function(V), theta(theta), phi(phi)
  {
    ct = cos(theta);
    st = sin(theta);
    cp = cos(phi);
    sp = sin(phi);
  }
};

class FiberField_f: public FiberFieldBase
{
public:
  FiberField_f(const FunctionSpace & V, double theta, double phi):
    FiberFieldBase(V, theta, phi)
  {}

  void eval(double* values, const double* x) const
  {
    // Fiber direction
    values[0] = ct;
    values[1] = st;
    values[2] = 0;
  }
};

class FiberField_s: public FiberFieldBase
{
public:
  FiberField_s(const FunctionSpace & V, double theta, double phi):
    FiberFieldBase(V, theta, phi)
  {}

  void eval(double* values, const double* x) const
  {
    // Sheet direction
    values[0] = -st;
    values[1] =  ct;
    values[2] = 0;
  }
};

class FiberField_n: public FiberFieldBase
{
public:
  FiberField_n(const FunctionSpace & V, double theta, double phi):
    FiberFieldBase(V, theta, phi)
  {}

  void eval(double* values, const double* x) const
  {
    double v[6];

    // Fiber direction
    v[0] = ct;
    v[1] = st;
    v[2] = 0;

    // Sheet direction
    v[3] = -st;
    v[4] =  ct;
    v[5] = 0;

    // Sheet normal direction
    values[0] =  v[1]*v[5] - v[2]*v[4];
    values[1] = -v[5]*v[0] + v[2]*v[3];
    values[2] =  v[0]*v[4] - v[1]*v[3];
  }
};

class FiberField: public FiberFieldBase
{
public:
  FiberField(const FunctionSpace & V, double theta, double phi):
    FiberFieldBase(V, theta, phi)
  {}

  void eval(double* values, const double* x) const
  {
    // Fiber direction
    values[0] = ct;
    values[1] = st;
    values[2] = 0;

    // Sheet direction
    values[3] = -st;
    values[4] =  ct;
    values[5] = 0;

    // Sheet normal direction
    values[6] =  values[1]*values[5] - values[2]*values[4];
    values[7] = -values[5]*values[0] + values[2]*values[3];
    values[8] =  values[0]*values[4] - values[1]*values[3];
  }
};

#endif
