% -----------------------------------------------------------------------------
%  (C) Altran Praxis Limited
% -----------------------------------------------------------------------------
% 
%  The SPARK toolset is free software; you can redistribute it and/or modify it
%  under terms of the GNU General Public License as published by the Free
%  Software Foundation; either version 3, or (at your option) any later
%  version. The SPARK toolset is distributed in the hope that it will be
%  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
%  Public License for more details. You should have received a copy of the GNU
%  General Public License distributed with the SPARK toolset; see file
%  COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
%  the license.
% 
% =============================================================================


%###############################################################################
% PURPOSE
%-------------------------------------------------------------------------------
% Provides access to all configurable switches. These may be configured by
% the user on the command line.
%###############################################################################


%###############################################################################
% MODULE
%###############################################################################
:- module(data__switches, [get_source_of_switch/2,
                           add_source_of_switch/2,
                           get_switch_input_file/1,
                           add_switch_input_file/1,
                           get_switch_log/1,
                           add_switch_log/1,
                           get_switch_wrap/1,
                           add_switch_wrap/1,
                           get_switch_verbose/1,
                           add_switch_verbose/1,
                           get_switch_user_rules/1,
                           add_switch_user_rules/1,
                           get_switch_plain/1,
                           add_switch_plain/1,
                           get_switch_typecheck_only/1,
                           add_switch_typecheck_only/1,
                           get_switch_renum/1,
                           add_switch_renum/1,
                           get_switch_simplification/1,
                           add_switch_simplification/1,
                           get_switch_standardisation/1,
                           add_switch_standardisation/1,
                           get_switch_rule_substitution/1,
                           add_switch_rule_substitution/1,
                           get_switch_nocontradiction_hunt/1,
                           add_switch_nocontradiction_hunt/1,
                           get_switch_substitution_elimination/1,
                           add_switch_substitution_elimination/1,
                           get_switch_expression_reduction/1,
                           add_switch_expression_reduction/1,
                           get_switch_complexity_limit/1,
                           add_switch_complexity_limit/1,
                           get_switch_depth_limit/1,
                           add_switch_depth_limit/1,
                           get_switch_inference_limit/1,
                           add_switch_inference_limit/1,
                           get_switch_help/1,
                           add_switch_help/1,
                           get_switch_version/1,
                           add_switch_version/1,
                           get_switch_empty/1,
                           add_switch_empty/1,
                           get_switch_usage/1,
                           add_switch_usage/1,
                           get_switch_deadpaths/1,
                           add_switch_deadpaths/1,
                           get_switch_hyp_limit/1,
                           add_switch_hyp_limit/1,
                           prune_source_of_switch/1,
                           prune_switch_log/0,
                           prune_switch_wrap/0,
                           prune_switch_verbose/0,
                           prune_switch_user_rules/0,
                           prune_switch_plain/0,
                           prune_switch_typecheck_only/0,
                           prune_switch_renum/0,
                           prune_switch_simplification/0,
                           prune_switch_standardisation/0,
                           prune_switch_rule_substitution/0,
                           prune_switch_nocontradiction_hunt/0,
                           prune_switch_substitution_elimination/0,
                           prune_switch_expression_reduction/0,
                           prune_switch_complexity_limit/0,
                           prune_switch_depth_limit/0,
                           prune_switch_inference_limit/0,
                           prune_switch_help/0,
                           prune_switch_version/0,
                           prune_switch_empty/0,
                           prune_switch_usage/0,
                           prune_switch_deadpaths/0,
                           prune_switch_hyp_limit/0,

                           plain/1,
                           complexity_limit/1,
                           inference_limit/1,
                           depth_limit/1,
                           substitution_elimination/1,
                           substitution_elimination/2,
                           simplification/1,
                           simplification/2,
                           contradiction_hunt/1,
                           contradiction_hunt/2,
                           expression_reduction/1,
                           expression_reduction/2,
                           standardisation/1,
                           standardisation/2,
                           rule_substitution/1,
                           rule_substitution/2,
                           renumber_conclusions/1,
                           typecheck_only/1,
                           nowrap_output_files/0,
                           standardise_in_infer/1,
                           typechecking_during_load/1]).


%###############################################################################
% DEPENDENCIES
%###############################################################################

:- use_module('data__formats.pro', [add_state/2,
                                    add_type/2]).

%###############################################################################
% TYPES
%###############################################################################

:- add_type('SwitchSource',
            [auto_set,
             user_set]).

:- add_type('OnOrOff',
            [on,
             off]).

:- add_type('LogFile',
            [no_log_file,
             yes_log_file,
             provided_log_file('LogFile_Atom')]).

:- add_type('SelectVCs',
            [all,
             none,
             exclude('IntList')]).

:- add_type('UsageFile',
            [no_usage_file,
             yes_usage_file,
             provided_usage_file('UsageFile_Atom')]).

:- add_type('Switch',
            [switch_input_file,
             switch_log,
             switch_wrap,
             switch_verbose,
             switch_user_rules,
             switch_plain,
             switch_typecheck_only,
             switch_renum,
             switch_simplification,
             switch_standardisation,
             switch_rule_substitution,
             switch_nocontradiction_hunt,
             switch_substitution_elimination,
             switch_expression_reduction,
             switch_complexity_limit,
             switch_depth_limit,
             switch_inference_limit,
             switch_help,
             switch_version,
             switch_usage,
             switch_empty,
             switch_deadpaths]).

%###############################################################################
% DATA
%###############################################################################

:- add_state(get_source_of_switch,
             get_source_of_switch('Switch', 'SwitchSource')).
:- dynamic(get_source_of_switch/2).

:- add_state(get_switch_input_file,
             get_switch_input_file('InputFile_Atom')).
:- dynamic(get_switch_input_file/1).

:- add_state(get_switch_log,
             get_switch_log('LogFile')).
:- dynamic(get_switch_log/1).

:- add_state(get_switch_wrap,
             get_switch_wrap('OnOrOff')).
:- dynamic(get_switch_wrap/1).

:- add_state(get_switch_verbose,
             get_switch_verbose('OnOrOff')).
:- dynamic(get_switch_verbose/1).

:- add_state(get_switch_user_rules,
             get_switch_user_rules('OnOrOff')).
:- dynamic(get_switch_user_rules/1).

:- add_state(get_switch_plain,
             get_switch_plain('OnOrOff')).
:- dynamic(get_switch_plain/1).

:- add_state(get_switch_typecheck_only,
             get_switch_typecheck_only('OnOrOff')).
:- dynamic(get_switch_typecheck_only/1).

:- add_state(get_switch_renum,
             get_switch_renum('OnOrOff')).
:- dynamic(get_switch_renum/1).

:- add_state(get_switch_simplification,
             get_switch_simplification('SelectVCs')).
:- dynamic(get_switch_simplification/1).

:- add_state(get_switch_standardisation,
             get_switch_standardisation('SelectVCs')).
:- dynamic(get_switch_standardisation/1).

:- add_state(get_switch_rule_substitution,
             get_switch_rule_substitution('SelectVCs')).
:- dynamic(get_switch_rule_substitution/1).

:- add_state(get_switch_nocontradiction_hunt,
             get_switch_nocontradiction_hunt('SelectVCs')).
:- dynamic(get_switch_nocontradiction_hunt/1).

:- add_state(get_switch_substitution_elimination,
             get_switch_substitution_elimination('SelectVCs')).
:- dynamic(get_switch_substitution_elimination/1).

:- add_state(get_switch_expression_reduction,
             get_switch_expression_reduction('SelectVCs')).
:- dynamic(get_switch_expression_reduction/1).

:- add_state(get_switch_complexity_limit,
             get_switch_complexity_limit('Int')).
:- dynamic(get_switch_complexity_limit/1).

:- add_state(get_switch_depth_limit,
             get_switch_depth_limit('Int')).
:- dynamic(get_switch_depth_limit/1).

:- add_state(get_switch_inference_limit,
             get_switch_inference_limit('Int')).
:- dynamic(get_switch_inference_limit/1).

:- add_state(get_switch_help,
             get_switch_help('OnOrOff')).
:- dynamic(get_switch_help/1).

:- add_state(get_switch_version,
             get_switch_version('OnOrOff')).
:- dynamic(get_switch_version/1).

:- add_state(get_switch_empty,
             get_switch_empty('OnOrOff')).
:- dynamic(get_switch_empty/1).

:- add_state(get_switch_usage,
             get_switch_usage('UsageFile')).
:- dynamic(get_switch_usage/1).

:- add_state(get_switch_deadpaths,
             get_switch_deadpaths('OnOrOff')).
:- dynamic(get_switch_deadpaths/1).

:- add_state(get_switch_hyp_limit,
             get_switch_hyp_limit('Int')).

:- dynamic(get_switch_hyp_limit/1).

%###############################################################################
% PREDICATES
%###############################################################################


%===============================================================================
% Add.
%===============================================================================

add_source_of_switch(Switch, SwitchSource):-
    assert(get_source_of_switch(Switch, SwitchSource)),
    !.

add_switch_input_file(InputFile_Atom):-
    assert(get_switch_input_file(InputFile_Atom)),
    !.

add_switch_log(LogFile):-
    assert(get_switch_log(LogFile)),
    !.

add_switch_wrap(OnOrOff):-
    assert(get_switch_wrap(OnOrOff)),
    !.

add_switch_verbose(OnOrOff):-
    assert(get_switch_verbose(OnOrOff)),
    !.

add_switch_user_rules(OnOrOff):-
    assert(get_switch_user_rules(OnOrOff)),
    !.

add_switch_plain(OnOrOff):-
    assert(get_switch_plain(OnOrOff)),
    !.

add_switch_typecheck_only(OnOrOff):-
    assert(get_switch_typecheck_only(OnOrOff)),
    !.

add_switch_renum(OnOrOff):-
    assert(get_switch_renum(OnOrOff)),
    !.

add_switch_simplification(SelectVCs):-
    assert(get_switch_simplification(SelectVCs)),
    !.

add_switch_standardisation(SelectVCs):-
    assert(get_switch_standardisation(SelectVCs)),
    !.

add_switch_rule_substitution(SelectVCs):-
    assert(get_switch_rule_substitution(SelectVCs)),
    !.

add_switch_nocontradiction_hunt(SelectVCs):-
    assert(get_switch_nocontradiction_hunt(SelectVCs)),
    !.

add_switch_substitution_elimination(SelectVCs):-
    assert(get_switch_substitution_elimination(SelectVCs)),
    !.

add_switch_expression_reduction(SelectVCs):-
    assert(get_switch_expression_reduction(SelectVCs)),
    !.

add_switch_complexity_limit(Int):-
    assert(get_switch_complexity_limit(Int)),
    !.

add_switch_depth_limit(Int):-
    assert(get_switch_depth_limit(Int)),
    !.

add_switch_inference_limit(Int):-
    assert(get_switch_inference_limit(Int)),
    !.

add_switch_help(OnOrOff):-
    assert(get_switch_help(OnOrOff)),
    !.

add_switch_version(OnOrOff):-
    assert(get_switch_version(OnOrOff)),
    !.

add_switch_empty(OnOrOff):-
    assert(get_switch_empty(OnOrOff)),
    !.

add_switch_usage(UsageFile):-
    assert(get_switch_usage(UsageFile)),
    !.

add_switch_deadpaths(OnOrOff):-
    assert(get_switch_deadpaths(OnOrOff)),
    !.

add_switch_hyp_limit(Limit_Int):-
    assert(get_switch_hyp_limit(Limit_Int)),
    !.

%===============================================================================


%===============================================================================
% Prune.
%===============================================================================

prune_source_of_switch(Switch):-
    retract(get_source_of_switch(Switch, _SwitchSource)),
    !.

prune_switch_log:-
    retract(get_switch_log(_LogFile)),
    !.

prune_switch_wrap:-
    retract(get_switch_wrap(_OnOrOff)),
    !.

prune_switch_verbose:-
    retract(get_switch_verbose(_OnOrOff)),
    !.

prune_switch_user_rules:-
    retract(get_switch_user_rules(_OnOrOff)),
    !.

prune_switch_plain:-
    retract(get_switch_plain(_OnOrOff)),
    !.

prune_switch_typecheck_only:-
    retract(get_switch_typecheck_only(_OnOrOff)),
    !.





prune_switch_renum:-
    retract(get_switch_renum(_OnOrOff)),
    !.

prune_switch_simplification:-
    retract(get_switch_simplification(_SelectVCs)),
    !.

prune_switch_standardisation:-
    retract(get_switch_standardisation(_SelectVCs)),
    !.

prune_switch_rule_substitution:-
    retract(get_switch_rule_substitution(_SelectVCs)),
    !.

prune_switch_nocontradiction_hunt:-
    retract(get_switch_nocontradiction_hunt(_SelectVCs)),
    !.

prune_switch_substitution_elimination:-
    retract(get_switch_substitution_elimination(_SelectVCs)),
    !.

prune_switch_expression_reduction:-
    retract(get_switch_expression_reduction(_SelectVCs)),
    !.

prune_switch_complexity_limit:-
    retract(get_switch_complexity_limit(_Int)),
    !.

prune_switch_depth_limit:-
    retract(get_switch_depth_limit(_Int)),
    !.

prune_switch_inference_limit:-
    retract(get_switch_inference_limit(_Int)),
    !.

prune_switch_help:-
    retract(get_switch_help(_OnOrOff)),
    !.

prune_switch_version:-
    retract(get_switch_version(_OnOrOff)),
    !.

prune_switch_empty:-
    retract(get_switch_empty(_OnOrOff)),
    !.

prune_switch_usage:-
    retract(get_switch_usage(_SwitchFile)),
    !.

prune_switch_deadpaths:-
    retract(get_switch_deadpaths(_OnOrOff)),
    !.

prune_switch_hyp_limit:-
    retract(get_switch_hyp_limit(_Limit)),
    !.

%===============================================================================


%===============================================================================
% Refactor.
%===============================================================================

plain(STATUS):-
    get_switch_plain(STATUS).

complexity_limit(VALUE):-
    get_switch_complexity_limit(VALUE).

inference_limit(VALUE):-
    get_switch_inference_limit(VALUE).

depth_limit(VALUE):-
    get_switch_depth_limit(VALUE).




substitution_elimination(on):-
    get_switch_substitution_elimination(all).

substitution_elimination(VC, off):-
    get_switch_substitution_elimination(exclude(IntList)),
    member(VC,IntList).


simplification(on):-
    get_switch_simplification(all).

simplification(VC, off):-
    get_switch_simplification(exclude(IntList)),
    member(VC,IntList).

contradiction_hunt(off):-
    get_switch_nocontradiction_hunt(all).

contradiction_hunt(VC, off):-
    get_switch_nocontradiction_hunt(exclude(IntList)),
    member(VC,IntList).

expression_reduction(on):-
    get_switch_expression_reduction(all).

expression_reduction(VC, off):-
    get_switch_expression_reduction(exclude(IntList)),
    member(VC,IntList).

standardisation(on):-
    get_switch_standardisation(all).

standardisation(VC, off):-
    get_switch_standardisation(exclude(IntList)),
    member(VC,IntList).

rule_substitution(on):-
    get_switch_rule_substitution(all).

rule_substitution(VC, off):-
    get_switch_rule_substitution(exclude(IntList)),
    member(VC,IntList).

renumber_conclusions(STATUS):-
    get_switch_renum(STATUS).

typecheck_only(STATUS):-
    get_switch_typecheck_only(STATUS).

nowrap_output_files:-
    get_switch_wrap(off).


standardise_in_infer(off).
typechecking_during_load(on).

%===============================================================================


%###############################################################################
% END-OF-FILE
