#!/bin/sh
#
# Astyle code formatting. This file is an adapted version of the pre-commit
# file of the SDF3  project. It can be found here:
# http://www.es.ele.tue.nl/sdf3/manuals/development/files/pre-commit
#
# Changes by Matthias Petri.

if git rev-parse --verify HEAD >/dev/null 2>&1
then
    against=HEAD
else
    # Initial commit: diff against an empty tree object
    against=4b825dc642cb6eb9a060e54bf8d69288fbee4904
fi

# If you want to allow non-ascii filenames set this variable to true.
allownonascii=$(git config hooks.allownonascii)

# Cross platform projects tend to avoid non-ascii filenames; prevent
# them from being added to the repository. We exploit the fact that the
# printable range starts at the space character and ends with tilde.
if [ "$allownonascii" != "true" ] &&
    # Note that the use of brackets around a tr range is ok here, (it's
    # even required, for portability to Solaris 10's /usr/bin/tr), since
    # the square bracket bytes happen to fall in the designated range.
    test "$(git diff --cached --name-only --diff-filter=A -z $against |
      LC_ALL=C tr -d '[ -~]\0')"
then
    echo "Error: Attempt to add a non-ascii file name."
    echo
    echo "This can cause problems if you want to work"
    echo "with people on other platforms."
    echo
    echo "To be portable it is advisable to rename the file ..."
    echo
    echo "If you know what you are doing you can disable this"
    echo "check using:"
    echo
    echo "  git config hooks.allownonascii true"
    echo
    exit 1
fi

CLANGFORMAT_BIN=@CLANG_FORMAT@

if [ -x "$CLANGFORMAT_BIN" ];
then
    echo "--Formatting source code USING clang-format: $CLANGFORMAT_BIN --"

    CLANG_FORMAT_PARAMS="-i"

    files=`git-diff-index --diff-filter=ACMR --name-only -r --cached $against --`
    for file in $files; do
        x=`echo $file |grep -E '^.*(\.cc|\.h|\.cpp|\.c|\.hpp|\.hpp\.cmake|\.cpp\.cmake)$'`
        if test "x$x" != "x"; then
        $CLANGFORMAT_BIN $CLANG_FORMAT_PARAMS $file
        git add $file
        fi
    done

    echo "--Formatting source code done--"
else
	echo "WARNING: clang-format not found. not formatting code!"
fi
