/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file Module.cxx
    \author Romain BONDUE
    \date 18/09/2005
    \note The design is not the best, but there isn't many supported drivers
          so it suffices for now. I can always improve it later. :) */
#include <fstream>
#include <limits>
#include <string>
#include <cstring> // strcpy()

extern "C"{
#include <sys/ioctl.h> // SIOCGIFCONF
}

#include "Module.h"
#include "Parameters.h"
#include "CWE17Module.h"
#include "RTModules.h"
#include "ErrorsCode.h"
#include "SystemTools.h" // Ioctl()



namespace
{
    const char* RT2400Name ("RT2400PCI");
    const char* RT2500Name ("RT2500 Wireless");
    const char* RT2570Name ("RT2500USB WLAN");
    const char* RT61Name ("RT61 Wireless");
    const char* RT73Name ("RT73 WLAN");

    const std::string ProfileFilePrefix ("_profiles.xml");

} // anonymous namespace



void nsCore::GetWirelessDevice (std::vector<nsWireless::CDeviceInfo>& NameVec)
                                                throw (nsErrors::CException)
{
    std::ifstream Is ("/proc/net/dev");
    if (!Is)
    {
        Is.clear();
        Is.open ("/proc/net/wireless"); // Try the deprecated old file.
    }
    if (Is)
        try
        {
            Is.exceptions (std::ios_base::failbit | std::ios_base::badbit);
            Is.ignore (std::numeric_limits<int>::max(), '\n');
            Is.ignore (std::numeric_limits<int>::max(), '\n');

            // <wireless.h> is included through WE17Driver.h (and is needed).
            ::iwreq Data;
            for (std::string Buffer ; Is >> Buffer ; )
            {
                const std::string::size_type Pos = Buffer.rfind (':');
                    // Ignore ':'.
                if (Pos < Buffer.size())
                    Buffer.resize (Pos);
                else
                {
                    char C;
                    Is >> C;
                }
                // Check if it's a valid wireless device by getting the name.
                if (Buffer.size() < IFNAMSIZ - 1)
                {
                    Buffer.copy (Data.ifr_ifrn.ifrn_name, IFNAMSIZ);
                    Data.ifr_ifrn.ifrn_name [Buffer.size()] = '\0';
                    try
                    {
                        nsSystem::Ioctl (SIOCGIWNAME, &Data, "");
                        NameVec.push_back (nsWireless::CDeviceInfo (Buffer,
                                                                Data.u.name));
                    }
                    catch (const nsErrors::CSystemExc&) {} // Error ignored.
                    Is.ignore (std::numeric_limits<int>::max(), '\n');
                }
            }
        }
        catch (const std::ios_base::failure& Exc)
        {       // A failbit exception is thrown if Is.eof().
            if (!Is.eof())
                throw nsErrors::CException ("Error parsing /proc/net/dev"
                                            " (or /proc/net/wireless).",
                                            nsErrors::ProcParsing);
        }
    if (NameVec.empty()) // Ask the kernel directly, the downside of this, is
    {                    // that unconfigured interfaces are not reported.
        char Buffer [1024]; // Size from iwlib.c
        ::ifconf NetConf;
        NetConf.ifc_len = sizeof (Buffer);
        NetConf.ifc_buf = Buffer;
        nsSystem::Ioctl (SIOCGIFCONF, &NetConf, "Can't get interfaces through"
                                                " SIOCGIFCONF.");
        const ::ifreq* pNetFreq (NetConf.ifc_req);
        ::iwreq Data;
        for (size_t I (NetConf.ifc_len / sizeof (::ifreq)) ; I-- ; ++pNetFreq)
        {
            std::strcpy (Data.ifr_ifrn.ifrn_name, pNetFreq->ifr_name);
            try
            {
                nsSystem::Ioctl (SIOCGIWNAME, &Data, "");
                NameVec.push_back (nsWireless::CDeviceInfo (pNetFreq->ifr_name,
                                                            Data.u.name));
            }
            catch (const nsErrors::CSystemExc&) {} // Error ignored.
        }
    }

} // GetWirelessDevice()


bool nsCore::IsSupported (const nsWireless::CDeviceInfo& Device) throw()
{
    return Device.GetProtoName() == RT2400Name ||
           Device.GetProtoName() == RT2500Name ||
           Device.GetProtoName() == RT2570Name ||
           Device.GetProtoName() == RT61Name ||
           Device.GetProtoName() == RT73Name;

} // IsSupported()


nsCore::IModule* nsCore::MakeModule (const nsWireless::CDeviceInfo& Device,
                                     nsGUI::CGtkGUI* pUI)
                                throw (nsErrors::CException, std::bad_alloc)
{
    const std::string ProfilesFileName (Device.GetProtoName() +
                                                            ProfileFilePrefix);
    if (Device.GetProtoName() == RT2400Name)
        return new CRT2400Module (Device.GetDeviceName(), ProfilesFileName);
    if (Device.GetProtoName() == RT2500Name)
        return new CRT2500Module (Device.GetDeviceName(), ProfilesFileName,
                                  pUI);
    if (Device.GetProtoName() == RT2570Name)
        return new CRT2570Module (Device.GetDeviceName(), ProfilesFileName,
                                  pUI);
    if (Device.GetProtoName() == RT61Name)
        return new CRT61Module (Device.GetDeviceName(), ProfilesFileName, pUI);
    if (Device.GetProtoName() == RT73Name)
        return new CRT73Module (Device.GetDeviceName(), ProfilesFileName, pUI);
    return new CWE17Module (Device.GetDeviceName(), ProfilesFileName);

} // MakeModule()


nsCore::ICLIModule* nsCore::MakeCLIModule
                                        (const nsWireless::CDeviceInfo& Device)
                                    throw (nsErrors::CException, std::bad_alloc)
{
    const std::string ProfilesFileName (Device.GetProtoName() +
                                                            ProfileFilePrefix);
    if (Device.GetProtoName() == RT2400Name)
        return new CRT2400CLIModule (Device.GetDeviceName(), ProfilesFileName);
    if (Device.GetProtoName() == RT2500Name)
        return new CRT2500CLIModule (Device.GetDeviceName(), ProfilesFileName);
    if (Device.GetProtoName() == RT2570Name)
        return new CRT2570CLIModule (Device.GetDeviceName(), ProfilesFileName);
    if (Device.GetProtoName() == RT61Name)
        return new CRT61CLIModule (Device.GetDeviceName(), ProfilesFileName);
    if (Device.GetProtoName() == RT73Name)
        return new CRT73CLIModule (Device.GetDeviceName(), ProfilesFileName);
    return new CWE17CLIModule (Device.GetDeviceName(), ProfilesFileName);

} // MakeModule()
