#ifndef _RHEO_SMART_POINTER_H
#define _RHEO_SMART_POINTER_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// smart pointers
//
// author: Pierre.Saramito@imag.fr
//
// date: 27 january 2000, updated 11 may 2012.
//
/*Class:smart_pointer
NAME:  @code{smart_pointer}, @code{smart_pointer_clone} - reference counted safe pointer with true copy semantic 
@clindex smart_pointer
@clindex smart_pointer_clone
@cindex reference counting
@cindex shallow copy
@cindex smart pointer
DESCRIPTION:       
  @noindent
  Here is a convenient way to implement a true copy semantic,
  by using shallow copies and reference counting, in order to 
  minimise memory copies.
  This concept is generally related to the @dfn{smart pointer}
  method for managing memory.

  @noindent
  The true semantic copy is defined as follows: if an object 
  @code{A} is assigned to
  @code{B}, such as @code{A = B}, every further modification on @code{A} or @code{B}
  does not modify the other.

  @noindent
  Notice that this class differs from the @code{boost::shared\_ptr} class
  that implements safe pointers without the true copy semantic.

CLONE VARIANT:       
  @noindent
  The @code{smart_pointer_clone} variant uses a @code{T* T::clone() const} member
  function instead of the usual @code{T::T()} copy constructor for obtaining 
  a true copy of the data.
  This variant is motivated as follows:
  when using hierarchies of derived classes (also known as polymorphic classes),
  the usual copy is not possible because c++ copy constructors cannot be virtual,
  so you cannot make a copy this way. 
  This is a well-known problem with C++'s implementation of polymorphism.
 
  @noindent
  We uses a solution to the non-virtual copy constructor problem which is
  suggested by Ellis and Stroustrup in "The Annotated LRM". 
  The solution is to require the @code{T} class to provide a virtual clone method for every
  class which makes a copy using new and the correct copy constructor,
  returning the result as a pointer to the superclass @code{T}.
  Each subclass of @code{T} overloads this function with its own variant
  which copies its own type. Thus the copy operation is now virtual and
  furthermore is localised to the individual subclass. 

NOCOPY VARIANT:       
  @noindent
  This variant of the smart pointer is designed for use on objects that
  cannot (or must not) be copied. An example would be when managing an 
  object that contains, say, a file handle. It is essential that this
  not be copied because then you get the problem of deciding which copy
  is responsible for closing the file. To avoid the problem, wrap the file
  handle in a class and then manage a unique instance of it using a 
  @code{smart_pointer_nocopy}.
  This ensures that the file handle cannot be copied and is closed when the last alias is destroyed.
 
  @noindent
  The interface to the nocopy variant is the same as @code{smart_pointer}
  but with all operations that perform copying forbidden.
  In fact, because all three variants are instances of a common superclass,
  the forbidden methods do exist but will cause an error and exit if they are called.
 
  The following modifiers cannot be used because they use copying
  of the pointed-to object and will thereore cause an error:
  @example
    T* operator-> ();
    T& operator* ();
    T* pointer ();
    T& data ();
  @end example
EFERENCES:
 @example
 [1] A. Geron and F. Tawbi,
    Pour mieux developper avec C++ : design pattern, STL, RTTI et smart pointers,
    InterEditions, 1999. Page 118.
 [2] STLplus, http://stlplus.sourceforge.net/stlplus3/docs/smart_ptr.html
    for the clone and nocopy variants.
 @end example
End:
*/

#include "rheolef/compiler.h"

// -----------------------------------------------------------------------
// smart_pointer_base<T,Copy>
// -----------------------------------------------------------------------
namespace rheolef { 
//<smart_pointer:
template <class T, class C>
class smart_pointer_base {
public:

// allocators:

  smart_pointer_base (T* p = 0);
  smart_pointer_base (const smart_pointer_base<T,C>&);
  smart_pointer_base<T,C>& operator= (const smart_pointer_base<T,C>&);
  ~smart_pointer_base ();

// accessors:

  const T* pointer    () const;
  const T& data       () const;
  const T* operator-> () const;
  const T& operator*  () const;

// modifiers:

  T* pointer    ();
  T& data       ();
  T* operator-> ();
  T& operator*  ();

// implementation:
private:
  struct counter {
    T*  _p;
    int _n;
    counter (T* p = 0);
    ~counter ();
    int operator++ ();
    int operator-- ();
  };
  counter *_count;
#ifndef TO_CLEAN
public:
  int reference_counter() const { return _count != 0 ? _count->_n : -1; }
#endif // TO_CLEAN
};
//>smart_pointer:
// -----------------------------------------------------------------------
// counter: inlined
// -----------------------------------------------------------------------
template <class T, class C>
inline
smart_pointer_base<T,C>::counter::counter (T* p)
 : _p(p), _n(1)
{
}
template <class T, class C>
inline
smart_pointer_base<T,C>::counter::~counter ()
{
    delete_macro(_p);
}
template <class T, class C>
inline
int
smart_pointer_base<T,C>::counter::operator++ ()
{
    return ++_n;
}
template <class T, class C>
inline
int
smart_pointer_base<T,C>::counter::operator-- ()
{ 
    if (--_n != 0) return _n;
    delete_macro(this);
    return 0;
}
// -----------------------------------------------------------------------
// smart_pointer_base: inlined
// -----------------------------------------------------------------------
template <class T, class C>
inline
smart_pointer_base<T,C>::smart_pointer_base (T* p)
: _count(new_macro(counter(p)))
{
}
template <class T, class C>
inline
smart_pointer_base<T,C>::smart_pointer_base (const smart_pointer_base& sp)
 : _count(sp._count)
{
    ++(*_count);
}
template <class T, class C>
inline
smart_pointer_base<T,C>::~smart_pointer_base ()
{
     if (_count != 0) { --(*_count); }
}
template <class T, class C>
inline
smart_pointer_base<T,C>&
smart_pointer_base<T,C>::operator= (const smart_pointer_base& sp)
{
    if (_count != sp._count) { 
	--(*_count);
	_count = sp._count;
	++(*_count);
    }
    return *this;
}
template <class T, class C>
inline
const T*
smart_pointer_base<T,C>::pointer () const
{
    return _count -> _p;
}
template <class T, class C>
inline
const T&
smart_pointer_base<T,C>::data () const
{
    return *pointer();
}
template <class T, class C>
inline
const T*
smart_pointer_base<T,C>::operator-> () const
{
    return pointer();
}
template <class T, class C>
inline
const T&
smart_pointer_base<T,C>::operator* () const
{
    return data();
}
template <class T, class C>
inline
T*
smart_pointer_base<T,C>::pointer ()
{
    // here is tthe true copy semantic:
    if (_count -> _p == 0) return 0;
    if (_count -> _n > 1) {
   	--(_count-> _n);
	T* q = C()(*(_count -> _p));
       	_count = new_macro (counter(q));
    }	
    return _count -> _p;
}
template <class T, class C>
inline
T*
smart_pointer_base<T,C>::operator-> ()
{
     return pointer();
}
template <class T, class C>
inline
T&
smart_pointer_base<T,C>::data ()
{
    return *pointer();
}
template <class T, class C>
inline
T&
smart_pointer_base<T,C>::operator* ()
{
    return data();
}
// -----------------------------------------------------------------------
// copy functors implementing the three possible copy semantics
// -----------------------------------------------------------------------
namespace details {

// constructor_copy uses the copy constructor of the object - used for simple types
template <typename T>
struct constructor_copy {
  T* operator() (const T& data) throw() { return new_macro(T(data)); }
};

// clone_copy uses the clone method of the object - used for polymorphic types
template <typename T>
struct clone_copy {
  T* operator() (const T& from) throw() { return from.clone(); }
};

// no_copy throws an exception - used for types that cannot be copied
template <typename T>
struct no_copy {
  T* operator() (const T& from) {
    error_macro ("no_copy functor called (illegal copy)");
    return 0;
  }
};

} // end namespace stlplus
// -----------------------------------------------------------------------
// smart_pointer<T>
// -----------------------------------------------------------------------
template <typename T>
class smart_pointer : public smart_pointer_base<T, details::constructor_copy<T> > {
    typedef details::constructor_copy<T>     C;
    typedef smart_pointer_base<T,C>          base;
  public:
    smart_pointer (T* p = 0) : base (p) {}
    smart_pointer (const smart_pointer<T>& x) : base(x) {}
    smart_pointer<T>& operator= (const smart_pointer<T>& x) {
	base::operator= (x); return *this; }
    ~smart_pointer() {}
};
// -----------------------------------------------------------------------
// smart_pointer_clone<T>
// -----------------------------------------------------------------------
template <typename T>
class smart_pointer_clone : public smart_pointer_base<T, details::clone_copy<T> > {
    typedef details::clone_copy<T>     C;
    typedef smart_pointer_base<T,C>    base;
  public:
    smart_pointer_clone (T* p = 0) : base (p) {}
    smart_pointer_clone (const smart_pointer_clone<T>& x) : base(x) {}
    smart_pointer_clone<T>& operator= (const smart_pointer_clone<T>& x) {
	base::operator= (x); return *this; }
    ~smart_pointer_clone() {}
};
// -----------------------------------------------------------------------
// smart_pointer_nocopy<T>
// -----------------------------------------------------------------------
template <typename T>
class smart_pointer_nocopy : public smart_pointer_base<T, details::no_copy<T> > {
    typedef details::no_copy<T>        C;
    typedef smart_pointer_base<T,C>    base;
  public:
    smart_pointer_nocopy (T* p = 0) : base (p) {}
    smart_pointer_nocopy (const smart_pointer_nocopy<T>& x) : base(x) {}
    smart_pointer_nocopy<T>& operator= (const smart_pointer_nocopy<T>& x) {
	base::operator= (x); return *this; }
    ~smart_pointer_nocopy() {}
};

}// namespace rheolef
#endif // _RHEO_SMART_POINTER_H
