///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// i/o in gmsh format
//
#include "rheolef/field.h"
using namespace rheolef;
using namespace std;

ostream&
field::put_gmsh (ostream& out) const
{
    // I don't known how to use gmsh node data format for mesh adaptation 
    // -> use old-fashionned .pos format instead
#define USE_GMSH_POS
#ifdef USE_GMSH_POS
    // documentation of the file format:
    // http://www.geuz.org/gmsh/doc/texinfo/gmsh.html#Post_002dprocessing-commands
    check_macro (get_approx() == "P1", "gmsh field output: P1 approximation expected");
    check_macro (n_component() == 1, "gmsh field output: non-scalar field not yet supported for pos-gmsh file");
    const geo& g = get_geo();
    geo::const_iterator  pK = g.begin();
    geo::const_iterator_vertex  p = g.begin_node();
    out << setprecision(numeric_limits<Float>::digits10)
        << "View \"data\" {" << endl;
    for (size_t k = 0; k < g.size(); k++, pK++) {
	const geo_element& K = *pK;
	switch (K.name()) {
	  case 'p' : out << "SP("; break;
	  case 'e' : out << "SL("; break;
	  case 't' : out << "ST("; break;
	  case 'q' : out << "SQ("; break;
	  case 'T' : out << "SS("; break;
	  case 'P' : out << "SI("; break;
	  case 'H' : out << "SH("; break;
	  default: {
	    warning_macro ("unexpected element type `" << K.name() << "'");
	    warning_macro ("element was " << K);
	    error_macro ("cannot recover");
	    break;
	  }
	}
	for (size_t i = 0; i < K.size(); i++) {
	  const point& x = p[K[i]];
	  for (size_t j = 0; j < 3; j++) {
	    out << x[j];
	    if (j != 2 || i != K.size()-1)
		out << ", ";
	  }
	}
	out << "){";
	for (size_t i = 0; i < K.size(); i++) {
	   out << at(K[i]);
	   if (i != K.size()-1) 
	       out << ",";
	}
	out << "};" << endl;
    }
    out << "};" << endl;
#else // !USE_GMSH_POS
    // documentation of the file format:
    // http://www.geuz.org/gmsh/doc/texinfo/gmsh.html#MSH-ASCII-file-format
    check_macro (get_approx() == "P1", "gmsh field output: P1 approximation expected, `"
        << get_approx() << "' found");
    out << gmsh
        << setprecision(numeric_limits<Float>::digits10)
        << get_geo()
        << "$NodeData" << endl
        << "1" << endl
        << "  hlocal" << endl
        << "0" << endl
        << "3" << endl
        << "  0" << endl
        << "  1" << endl
        << "  " << size() << endl;
    for (size_type dof = 0; dof < size(); dof++) {
        out << dof+1 << " " << at(dof) << endl;
    }
    out << "$EndNodeData" << endl;
#endif // USE_GMSH_POS
    return out;
}
