// usage:
//   demo2 square.geo | demo2_metric 

#include "rheolef/rheolef.h"
using namespace rheolef;
using namespace std;

Float coef = 1;
Float err = 1;
Float hmax = 0.1;
Float hmin = 0.0000005;
Float u_range = 1;

// metric: isotropic and for P1 initial field
// return 1/(h^2)
Float 
make_metric_iso_P1 (const Float& e, const Float& h_prec)
{
    Float inv_h2_prec = 1/(h_prec*h_prec);
    Float inv_h2;
    if (1 + fabs(err)     == Float(1) ||
        1 + fabs(u_range) == Float(1) ||
        1 + fabs(u_range) == Float(1)) {
	
	inv_h2 = 1/(hmax*hmax);
    
    } else {
        inv_h2 = (inv_h2_prec*e*e)/(err*coef*coef);
        inv_h2 = min(inv_h2, 1/(hmin*hmin));
    }
    return inv_h2;
}
void
build_previous_isotrope_metric (
    const geo&              omega_h,
    vector<Float>::iterator h)
{
    typedef geo::size_type size_type;
    geo::const_iterator_node p = omega_h.begin_node();
    geo::const_iterator iter_elt = omega_h.begin();
    geo::const_iterator last_elt = omega_h.end();
    while (iter_elt != last_elt) {
	const geo_element& K = *iter_elt++;
        for (size_type e = 0; e < K.n_edge(); e++) {
	    size_type i1 = K.subgeo_vertex (1, e, 0);
	    size_type i2 = K.subgeo_vertex (1, e, 1);
	    Float h12 = norm(p[i1] - p[i2]);
	    h [i1] = min(h[i1], h12);
	    h [i2] = min(h[i2], h12);
        }
    }
}
string get_approx_grad (const string &Pk)
{
    if (Pk == "P1") return "P0";
    if (Pk == "P2") return "P1d";
    cerr << "unexpected approximation " << Pk << endl;
    exit (1);
}
string get_approx_err (const string &Pk)
{
    if (Pk == "P1") return "P1d";
    if (Pk == "P2") return "P2d";
    cerr << "unexpected approximation " << Pk << endl;
    exit (1);
}
enum format_type { 
	format_field,
	format_bamg
	};
void
usage(const char* prog)
{
    cerr << "usage: " << get_basename(prog) << " " 
         << "[-coef float] "
         << "[-err float] "
         << "[-hmin float] "
         << "[-hmax float] "
         << "[-n-proj int] "
         << "[-n-smooth int] "
         << "[-field|-bamg] "
         << "[-estimator] "
         << "[-verbose] "
         << "[-noverbose] "
	 << endl;
    exit(1);
}
int main(int argc, char**argv)
{
    typedef geo::size_type size_type;

    format_type format = format_field;
    bool do_verbose = true;
    size_type n_proj   = 1;
    size_type n_smooth = 0;
    int digits10 = numeric_limits<Float>::digits10;
    cout << setprecision(digits10);

    bool output_estimator = false;

    // scan the command line parameters
    for (int i = 1; i < argc; i++) {
             if (strcmp(argv[i], "-err") == 0)  { err = atof(argv[++i]); }
        else if (strcmp(argv[i], "-coef") == 0) { coef = atof(argv[++i]); }
        else if (strcmp(argv[i], "-hmin") == 0) { hmin = atof(argv[++i]); }
        else if (strcmp(argv[i], "-hmax") == 0) { hmax = atof(argv[++i]); }
        else if (strcmp(argv[i], "-n-proj") == 0) { n_proj = size_type(atoi(argv[++i])); }
        else if (strcmp(argv[i], "-n-smooth") == 0) { n_smooth = size_type(atoi(argv[++i])); }
        else if (strcmp(argv[i], "-field") == 0) { format = format_field; }
        else if (strcmp(argv[i], "-bamg") == 0) { format = format_bamg; }
        else if (strcmp(argv[i], "-ndigit") == 0) { digits10 = atoi(argv[++i]); }
        else if (strcmp(argv[i], "-estimator") == 0) { output_estimator = true; }
        else if (strcmp(argv[i], "-verbose") == 0) { do_verbose = true; }
        else if (strcmp(argv[i], "-noverbose") == 0) { do_verbose = false; }
	else usage(argv[0]);
    }
    n_proj   = max(n_proj, size_type(1));
    if (do_verbose) {
	cerr << "err " << err << endl
	     << "coef " << coef << endl
	     << "hmin " << hmin << endl
	     << "hmax " << hmax << endl
	     << "n_proj " << n_proj << endl
	     << "n_smooth " << n_smooth << endl
	     ;
    }
    // the approximate solution
    field uh;
    cin >> uh;
    u_range = fabs(uh.max() - uh.min());

    // get space Vh where uh belives and mesh
    space Vh      = uh.get_space();
    geo   omega_h = Vh.get_geo();
    size_type dim = omega_h.dimension();

    // the space of gradients
    string Pk = Vh.get_approx();
    string approx_grad = get_approx_grad(Pk);
    space Th (omega_h, approx_grad);

    // the space of errors
    string approx_err = get_approx_err(Pk);
    space Eh (omega_h, approx_err);

    // build d_dxi operators
    form d_dx [3];
    for (size_type i = 0; i < dim; i++) {
	char d_dxi [10];
	sprintf (d_dxi, "d_dx%d", int(i+1));
	d_dx [i] = form(Vh, Th, d_dxi);
    }
    // build invert of mass on Th 
    form inv_mt (Th, Th, "inv_mass");

    // compute the gradient field of uh (P0)
    field duh_dx [3];
    for (size_type i = 0; i < dim; i++) {
        duh_dx [i] = inv_mt*(d_dx[i]*uh);
    }
    // projector:
    //   build projection: P(k-1) discontinuous gradient in Pk continuous
    //   and re-project in P(k-1) discontinuous
    form mv   (Vh, Vh, "mass");
    form proj (Th, Vh, "mass");
    ssk<Float> fact_mv = ldlt(mv.uu);

    field g0h [3];
    for (size_type i = 0; i < dim; i++) {
	g0h [i] = duh_dx [i];
    }
    field p1h [3];
    for (size_type i = 0; i < dim; i++) {
	p1h [i] = field(Vh);
    }
    for (size_type k = 0; k < n_proj; k++) {
    
        for (size_type i = 0; i < dim; i++) {

	    // project gradient Pk continuous
            p1h[i].u = fact_mv.solve(proj.uu*g0h[i].u);

            // re-project P0
            g0h [i] = inv_mt*proj.trans_mult(p1h[i]);
        }
    }
    // send the projected gradient Pk-continuous in Pk-discontinuous
    form inv_me (Eh, Eh, "inv_mass");
    form pr1    (Vh, Eh, "mass");
    field ph [3];
    for (size_type i = 0; i < dim; i++) {
	ph [i] = inv_me*(pr1*p1h[i]);
    }
    // send the gradient P(k-1)-dicontinuous in Pk-discontinuous
    form pr0    (Th, Eh, "mass");
    field gh [3];
    for (size_type i = 0; i < dim; i++) {
	gh [i] = inv_me*(pr0*duh_dx[i]);
    }
    // estimated error by residual of proj operation
    field e1h [3];
    for (size_type i = 0; i < dim; i++) {
        e1h[i] = abs(ph[i] - gh[i]);
    }
    // make it scalar : isotrope metric
    field e0h (Eh, 0.);
    for (size_type i = 0; i < dim; i++) {
        e0h = e0h + e1h[i]*e1h[i];
    }
    e0h = sqrt(e0h);

    // project-it Pk continuous
    field eh (Vh);
    eh.u = fact_mv.solve(pr1.uu.trans_mult(e0h.u));
    
    if (output_estimator) {
        cout << rheo << abs(eh);
	return 0;
    }
    // get previous isotrope metric : for Vh = P1 only !!
    field h_prec (Vh);
    build_previous_isotrope_metric (omega_h, h_prec.u.begin());

    // build the metric
    field mh (Vh);
    for (size_type idof = 0; idof < eh.size(); idof++) {
        mh.at(idof) = make_metric_iso_P1 (eh.at(idof), h_prec.at(idof));
    }
    // smooth the metric
    for (size_type k = 0; k < n_smooth; k++) {
	
        // project P0
        field m0h = inv_mt*proj.trans_mult(mh);
	
	// re-project P1
        mh.u = fact_mv.solve(proj.uu*m0h.u);
    }
    // output the isotrope metric for bamg
    if (format == format_bamg) {
        cout << mh.size() << " 3" << endl;
        for (size_type idof = 0; idof < mh.size(); idof++) {
            cout << mh.at(idof) << " 0 " << mh.at(idof) << endl;
        }
    } else {
        cout << rheo << mh;
    }
    return 0;
}
