#ifndef _RHEO_COMPILER_H
#define _RHEO_COMPILER_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// compiler-dependent part for rheolef implementation
//
// author: Pierre.Saramito@imag.fr
//
// date: 27 january 2000
//

#include "rheolef/config.h"   /* as generated by configure */

#include <cmath>
namespace rheolef {
using std::cos;
using std::sin;
using std::tan;
using std::acos;
using std::asin;
using std::atan;
using std::cosh;
using std::sinh;
using std::tanh;
using std::exp;
using std::log;
using std::log10;
using std::sqrt;
using std::abs;
using std::fabs;
using std::floor;
using std::ceil;
using std::atan2;
using std::pow;
using std::fmod;
} // namespace rheolef


#include <complex>
namespace rheolef {
/// @brief helper for std::complex<T>: get basic T type
template<class T> struct float_traits                   { typedef T                              type; };
template<class T> struct float_traits<std::complex<T> > { typedef typename float_traits<T>::type type; };
} // namespace rheolef

// for DEC/Compacq cxx: use standard iostreams
#ifdef _RHEOLEF_HAVE_DEC_CXX
# ifndef __USE_STD_IOSTREAM
#  define __USE_STD_IOSTREAM 1
# endif
#endif

// for CRAY C++ STL library: in <iterator.h>, need it
#ifdef _RHEOLEF_HAVE_CRAY_CXX
#define __STL_USE_NAMESPACES
#define __STL_CLASS_PARTIAL_SPECIALIZATION
#endif

#ifdef _RHEOLEF_HAVE_IOS
# include <ios>
#elif defined(_RHEOLEF_HAVE_IOS_H)
# include <ios.h>
#endif

// buggy g++ on SunOS defines _X !!
#ifdef _X
#undef _X
#endif
#ifdef _RHEOLEF_HAVE_ISTREAM
# include <istream>
#elif defined(_RHEOLEF_HAVE_ISTREAM_H)
# include <istream.h>  /* ioskit need it without std:: for kai */
#endif

#ifdef _RHEOLEF_HAVE_OSTREAM
# include <ostream>
#elif defined(_RHEOLEF_HAVE_OSTREAM_H)
# include <ostream.h>
#endif

#ifdef _RHEOLEF_HAVE_IOSTREAM
# include <iostream>
#else
# include <iostream.h>  /* ioskit need it */
#endif

#ifdef _RHEOLEF_HAVE_FSTREAM
# include <fstream>
#elif defined(_RHEOLEF_HAVE_FSTREAM_H)
# include <fstream.h>
#endif

#ifdef _RHEOLEF_HAVE_IOMANIP
# include <iomanip>
#elif defined(_RHEOLEF_HAVE_IOMANIP_H)
# include <iomanip.h>
#endif

// ==================================================================================================
// macro utilities:
//	fatal_macro
//	error_macro
//	trace_macro
//	check_macro
//	assert_macro
//
# define fatal_macro(message) \
	{ std::cerr << "fatal(" << __FILE__ << "," << __LINE__ << "): " << message << std::endl; exit(1); }
# define error_macro(message) \
	fatal_macro(message)
# define warning_macro(message) \
	{ std::cerr << "warning(" << __FILE__ << "," << __LINE__ << "): " << message << std::endl; }
#  define check_macro(ok_condition, message) \
        { if (!(ok_condition)) fatal_macro(message); }

# ifdef _RHEOLEF_PARANO
#  ifdef  NDEBUG
#    undef  NDEBUG
#  endif
#  define trace_macro(message) \
        { std::cerr << "trace(" << __FILE__ << "," << __LINE__ << "): " << message << std::endl; }
#  define assert_macro(ok_condition, message) \
	check_macro(ok_condition,message)
# else // not _RHEOLEF_PARANO
#  ifndef NDEBUG
#    define NDEBUG
#  endif
#  define trace_macro(message)
#  define assert_macro(ok_condition, message)
# endif // _RHEOLEF_PARANO

// ==================================================================================================

#ifdef _RHEOLEF_HAVE_LIMITS
#include <limits>
#else
#error numeric_limits<T> C++ class not available in standard library: please, update your compiler !
// substitution for numeric_limits<T> class
// #include "rheolef/numeric_limits.h"
#endif // _RHEOLEF_HAVE_LIMITS

#include "rheolef/numeric_flags.h"

// Float type used for the library pre-instanciation
#if defined(_RHEOLEF_HAVE_CLN)

# include "rheolef/bigfloat.h"
namespace rheolef {
typedef bigfloat<_RHEOLEF_DIGITS10> Float;
} // namespace rheolef
#elif defined(_RHEOLEF_HAVE_DOUBLEDOUBLE)

# define  DD_INLINE
# include "rheolef/doubledouble.h"
namespace rheolef {
typedef doubledouble Float;
} // namespace rheolef

#elif defined(_RHEOLEF_HAVE_LONG_DOUBLE)

namespace rheolef {
typedef long double Float;
} // namespace rheolef

#else

namespace rheolef {
typedef double Float;
} // namespace rheolef

#endif // FLOAT: CLN, DOUBLEDOUBLE, LONG_DOUBLE, ...

/* max size for char* filename; TODO: may disapear from code... */
#define STRLENMAX       1024


#ifdef _RHEOLEF_HAVE_MEMORY_H
#include <memory.h> // KAI C++
#endif

#if !defined(_RHEOLEF_HAVE_SQR_DOUBLE) && !defined(_RHEOLEF_HAVE_DOUBLEDOUBLE)
namespace rheolef {
static inline double sqr (const double& x) { return (x*x); }
} // namespace rheolef
#endif

#if defined(_RHEOLEF_HAVE_ITERATOR)
#include <iterator>	// DEC C++
#elif defined(_RHEOLEF_HAVE_ITERATOR_H)
#include <iterator.h>
#endif

// there is no std on string...
#if defined(_RHEOLEF_HAVE_STRING)
#include <string>
#elif defined(_RHEOLEF_HAVE_BSTRING_H)
#include <bstring.h>
#elif defined(_RHEOLEF_HAVE_MSTRING_H)
#include <mstring.h>
#endif

// greater/less, ect...
#if defined(_RHEOLEF_HAVE_FUNCTIONAL)
#include <functional> // DEC C++
#elif defined(_RHEOLEF_HAVE_FUNCTIONAL_H)
#include <functional.h> // KAI C++
#elif defined(_RHEOLEF_HAVE_FUNCTION_H)
#include <function.h> // GNU C++
#endif

// inner_product, ect...
#if defined(_RHEOLEF_HAVE_NUMERIC)
#include <numeric> // standard c++
#elif defined(_RHEOLEF_HAVE_NUMERIC_H)
#include <numeric.h> // KAI C++
#endif

#if defined(_RHEOLEF_HAVE_ALGORITHM)
#include <algorithm> // standard 
#elif defined(_RHEOLEF_HAVE_ALGO_H)
#include <algo.h> // old GNU C++
#endif

namespace rheolef {
using std::min;
using std::max;
#ifdef TO_CLEAN
// TODO: promote<T1,T2>::type max(T1,T2) {...}
// conflict with max(field_nonlinear_expr<E>,int)
template <class T> T max (T x, int y) { return x > y ? x : T(y); }
template <class T> T max (int x, T y) { return x > y ? T(x) : y; }
template <class T> T min (T x, int y) { return x < y ? x : T(y); }
template <class T> T min (int x, T y) { return x < y ? T(x) : y; }
// conflict with ginac::abs
template <class T> T abs (T x)        { return (x > T(0) ? x : -x); }
#endif // TO_CLEAN
} // namespace rheolef

#if defined(_RHEOLEF_HAVE_CSTLIB)
# include <cstdlib>
#else
# include <stdlib.h>
#endif

#if defined(_RHEOLEF_HAVE_CSTLIB)
# include <cstdio>
#else
# include <stdio.h>
#endif

#if defined(_RHEOLEF_HAVE_CLIMITS)
# include <climits>    /* MAXINT, HUGE */
#else
# include <limits.h>   /* MAXINT, HUGE */
#endif

#ifdef _RHEOLEF_HAVE_UTILITY
#include <utility>
#elif defined(_RHEOLEF_HAVE_UTILITY_H)
#include <utility.h>    // pair with KAI C++
#endif

#if defined(_RHEOLEF_HAVE_VECTOR)
#include <vector>
#else
#include <vector.h>
#endif

#if defined(_RHEOLEF_HAVE_LIST)
#include <list>
#else
#include <list.h>
#endif

#if defined(_RHEOLEF_HAVE_SET)
#include <set>
#else
#include <set.h>
#endif

#if defined(_RHEOLEF_HAVE_MAP)
#include <map>
#else
#include <map.h>
#endif

// by default ublas::vector<T>::iterator has no [i] accessors
#if defined(_RHEOLEF_HAVE_BOOST)
#define BOOST_UBLAS_USE_INDEXED_ITERATOR
#endif

// macros wrapper for new and delette
// macros wrapper for new and delette
// when debuging with malloc_dbg
//
# ifdef _RHEOLEF_HAVE_DMALLOC /* C++ malloc debug */

#include <dmalloc.h> // link also with -ldmalloc
#include "rheolef/dmalloc_return.h"  // from dmalloc lib

// defined in "util/dmallocxx/dmallocc.cc"
// put it in global namespace:
#define USE_NEW_INLINE
#ifdef USE_NEW_INLINE
inline
void *operator new(size_t size, const char *const file, int line)
{
  if (size == 0) return (void*)0;
  return dmalloc_malloc(file, line, size, DMALLOC_FUNC_NEW,
			0 /* no alignment */, 0 /* no xalloc messages */);
}
inline
void *operator new[](size_t size,  const char *const file, int line)
{
  if (size == 0) return (void*)0;
  return dmalloc_malloc(file, line, size, DMALLOC_FUNC_NEW_ARRAY,
			0 /* no alignment */, 0 /* no xalloc messages */);
}
inline
void
operator delete(void *pnt)
{
  char	*file;
  GET_RET_ADDR(file);
  dmalloc_free(file, 0, pnt, DMALLOC_FUNC_DELETE);
}
inline
void
operator delete[](void *pnt)
{
  char	*file;
  GET_RET_ADDR(file);
  dmalloc_free(file, 0, pnt, DMALLOC_FUNC_DELETE_ARRAY);
}
#else // USE_NEW_INLINE
void *operator new(size_t, const char *const, int);
void *operator new[](size_t,  const char *const, int);
void operator delete(void *);
void operator delete[](void *);
#endif // USE_NEW_INLINE
# define 	new_macro(obj)            new (__FILE__ , __LINE__) obj
# define 	new_tab_macro(typ, n)     (new (__FILE__ , __LINE__) typ [(n)])
# define        delete_macro(ptr)         { if (ptr) delete (ptr); }
# define        delete_tab_macro(ptr)     { if (ptr) delete [] (ptr); }

# else // ! _RHEOLEF_HAVE_DMALLOC : standard c++ new, delete

# define 	new_macro(obj)            (new obj)
# define 	new_tab_macro(typ, n)     (new typ [(n)])
# define        delete_macro(ptr)         { if (ptr) delete (ptr); }
# define        delete_tab_macro(ptr)     { if (ptr) delete [] (ptr); }

# endif // _RHEOLEF_HAVE_DMALLOC

#endif // _RHEO_COMPILER_H
