% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aux_outer.R
\name{pmp.bma}
\alias{pmp.bma}
\title{Posterior Model Probabilities}
\usage{
pmp.bma(bmao, oldstyle = FALSE)
}
\arguments{
\item{bmao}{A bma object (see argument \code{nmodel} in \code{\link{bms}}),
alternatively an object of class \code{\link{topmod}}}

\item{oldstyle}{For normal use, leave this at \code{FALSE}. It is an
argument for compatibility with older BMS versions - see section 'Notes'}
}
\value{
the result is a matrix, its row names describe the model binaries\cr
There are two columns in the matrix: \item{PMP (Exact)}{posterior model
probabilities based on the posterior likelihoods of the best models in
\code{bmao} } \item{PMP (MCMC)}{posterior model probabilities of the best
models in \code{bmao} based on their MCMC frequencies, relative to all
models encountered by \code{bmao} - see 'Details' }
}
\description{
Returns the posterior model probabilites for the best models encountered by
a 'bma' object
}
\details{
A call to bms with an MCMC sampler (e.g.
\code{bms(datafls,mcmc="bd",nmodel=100)} uses a Metropolis-Hastings
algorithm to sample through the model space - and the frequency of how often
models are drawn converges to the distribution of their posterior marginal
likelihoods.  While sampling, each 'bma' object stores the best models
encountered by its sampling chain with their marginal likelihood and their
MCMC frequencies.

\code{pmp.bma} then allows for comparing the posterior model probabilities
(PMPs) for the two different methods, similar to \code{\link{plotConv}}.  It
calculates the PMPs based on marginal likelihoods (first column) and the
PMPs based on MCMC frequencies (second column) for the best x models stored
in the bma object.

The correlation of the two columns is an indicator of how well the MCMC
sampler has converged to the actual PMP distribution - it is therefore also
given in the output of \code{\link{summary.bma}}.

The second column is slightly different in case the \code{\link{bms}}
argument \code{mcmc} was set to \code{mcmc="enumeration"}: In this case, the
second column is also based on marginal likelihoods. The correlation between
the two columns is therefore one.
}
\note{
The second column thus shows the PMPs of the best models relative to
all models the call to \code{\link{bms}} has sampled through (therefore
typically the second column adds up to less than one).  The first column
relates to the likelihoods of the best models, therefore it would add up to
1.  In order estimate for their marginal likelihoods with respect to the
other models (the ones not retained in the best models), these PMP aadding
up to one are multiplied with the sum of PMP of the best models accroding to
MCMC frequencies.  Therefore, the two columns have the same column sum.

CAUTION: In package versions up to \code{BMS 0.2.5}, the first column was
indeed set always equal to one. This behaviour can still be mimicked by
setting \code{oldstyle=TRUE}.
}
\examples{


## sample BMA for growth dataset, MCMC sampler
data(datafls)
mm=bms(datafls[,1:10],nmodel=20, mcmc="bd")

## mmodel likelihoods and MCMC frequencies of best 20 models
print(mm$topmod)

pmp.bma(mm)
#first column: posterior model prob based on model likelihoods,
#  relative to best models in 'mm'
#second column: posterior model prob based MCMC frequencies,
#  relative to all models encountered by 'mm'

#consequently, first column adds up to one
#second column shows how much of the sampled model space is
# contained in the best models
colSums(pmp.bma(mm))


#correlation betwwen the two shows how well the sampler converged
cor(pmp.bma(mm)[,1],pmp.bma(mm)[,2])

#is the same as given in summary.bma
summary(mm)["Corr PMP"]

#plot the two model probabilites
plotConv(mm)

#equivalent to the following chart
plot(pmp.bma(mm)[,2], type="s")
lines(pmp.bma(mm)[,1],col=2)


#moreover, note how the first column is constructed
liks=exp(mm$top$lik())
liks/sum(liks)
pmp.bma(mm)[,1] #these two are equivalent



#the example above does not converge well,
#too few iterations and best models
# this is already better, but also not good
mm=bms(datafls[,1:10],burn=2000,iter=5000,nmodel=200)


# in case the sampler has been 'enumeration' instead of MCMC,
# then both matrix columns are of course equivalent
mm=bms(datafls[,1:10],nmodel=512,mcmc="enumeration")
cor(pmp.bma(mm)[,1],pmp.bma(mm)[,2])
colSums(pmp.bma(mm))


}
\seealso{
\code{\link{plotConv}} for plotting \code{pmp.bma},
\code{\link{pmpmodel}} to obtain the PMP for any individual model,
\code{\link{bms}} for sampling bma objects

Check \url{http://bms.zeugner.eu} for additional help.
}
\keyword{utilities}
