// ---------------------------------------------------------------------------
//
//  This file is part of PermLib.
//
// Copyright (c) 2009-2011 Thomas Rehn <thomas@carmen76.de>
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of the author may not be used to endorse or promote products
//    derived from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// ---------------------------------------------------------------------------


#include <permlib/common.h>
#include <permlib/permutation.h>
#include <permlib/bsgs.h>
#include <permlib/transversal/schreier_tree_transversal.h>
#include <permlib/construct/schreier_sims_construction.h>
#include <permlib/construct/random_schreier_sims_construction.h>
#include <permlib/generator/bsgs_random_generator.h>

#include <iostream>

int main(int argc, char *argv[]) {
	using namespace permlib;
	
	// we use elementary permutations
	typedef Permutation PERM;
	// and Schreier tree transversals
	typedef SchreierTreeTransversal<PERM> TRANSVERSAL;
	
	// our group will have degree 10
	const unsigned long n = 10;
	
	// group generators
	std::list<PERM::ptr> groupGenerators;
	PERM::ptr gen1(new PERM(n, std::string("1 3 5 7 9 10 2 4 6 8")));
	groupGenerators.push_back(gen1);
	PERM::ptr gen2(new PERM(n, std::string("1 5")));
	groupGenerators.push_back(gen2);
	PERM::ptr gen3(new PERM(n, std::string("2 6 5 9, 3 7 10 4 8")));
	groupGenerators.push_back(gen3);
	PERM::ptr gen4(new PERM(n, std::string("4 8 7 10")));
	groupGenerators.push_back(gen4);
	
	// BSGS construction
	SchreierSimsConstruction<PERM, TRANSVERSAL> schreierSims(n);
	BSGS<PERM, TRANSVERSAL> bsgs = schreierSims.construct(groupGenerators.begin(), groupGenerators.end());
	std::cout << "Group " << bsgs << std::endl;
	
	// In the following we try to find a small generating set for the group.
	// Our goal is to find a set of two or three generators.
	// In this case <$gen1, $gen2> already generate the group, but we may find
	//  other generators by the following randomized construction.
	
	const boost::uint64_t groupOrder = bsgs.order();
	bool haveSmallGeneratingSet = false;
	std::list<PERM::ptr> subgroupGenerators;
	
	for (unsigned int i = 2; i <= 3; ++i) {
		for (unsigned int j = 0; j < 5; ++j) {
			
			// Try #j times to build a subgroup from #i randomly chosen group elements.
			// Then check the order of this subgroup.
			// If it is the order of the original group, we found a small generating set.
			
			subgroupGenerators.clear();
			for (unsigned int c = 0; c < i; ++c) {
				subgroupGenerators.push_back(PERM::ptr(new PERM(bsgs.random())));
			}
			RandomSchreierSimsConstruction<PERM, TRANSVERSAL> randomConstruction(n, new BSGSRandomGenerator<PERM,TRANSVERSAL>(bsgs), 0, 10, 100);
			bool guaranteedBSGS = false;
			BSGS<PERM, TRANSVERSAL> subgroupBSGS = randomConstruction.construct(subgroupGenerators.begin(), subgroupGenerators.end(), guaranteedBSGS);
			if (subgroupBSGS.order() == groupOrder) {
				haveSmallGeneratingSet = true;
				break;
			}
		}
		if (haveSmallGeneratingSet)
			break;
	}
	
	if (haveSmallGeneratingSet) {
		std::cout << "Group generated by:" << std::endl;
		BOOST_FOREACH(const PERM::ptr& gen, subgroupGenerators) {
			std::cout << *gen << std::endl;
		}
	} else {
		std::cout << "Could not find small generating set." << std::endl;
	}
	
	return 0;
}
