/*	Conductor_Matrix

PIRL CVS ID: Conductor_Matrix.java,v 1.17 2012/04/16 06:04:11 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package PIRL.Conductor.Maestro;

import	PIRL.Conductor.Colors;

import	org.jdesktop.swingx.JXTable;

import	javax.swing.JTable;
import	javax.swing.table.TableCellRenderer;
import	javax.swing.JPanel;
import	javax.swing.JLabel;
import	javax.swing.BorderFactory;
import	javax.swing.border.Border;
import	javax.swing.border.BevelBorder;
import	javax.swing.Box;
import	javax.swing.ListSelectionModel;
import	javax.swing.SwingConstants;
import	java.awt.Component;
import	java.awt.Color;
import	java.awt.Font;
import	java.awt.GridBagLayout;
import	java.awt.GridBagConstraints;
import	java.awt.Dimension;
import	java.awt.Graphics;
import	java.awt.FontMetrics;


/**	A <i>Conductor_Matrix</i> is a table view of a Conductor_Matrix_Model..
<p>
	@author	Bradford Castalia, UA/PIRL
	@version 1.17
*/
public class Conductor_Matrix
	extends JXTable
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.Maestro.Conductor_Matrix (1.17 2012/04/16 06:04:11)";


/**	Index for arrays of processing state information.
*/
public static final int
	RUNNING_INDEX		= Conductor_Matrix_Model.RUNNING,
	POLLING_INDEX		= Conductor_Matrix_Model.POLLING,
	WAITING_INDEX		= Conductor_Matrix_Model.WAITING,
	HALTED_INDEX		= Conductor_Matrix_Model.HALTED,
	TOTAL_INDEX			= Conductor_Matrix_Model.TOTAL;

public static final Color
	STATE_COLORS[][] =
		{
		{Colors.RUNNING_STATE,
			Colors.Selected_Color (true, Colors.RUNNING_STATE)},
		{Colors.POLLING_STATE,
			Colors.Selected_Color (true, Colors.POLLING_STATE)},
		{Colors.WAITING_STATE,
			Colors.Selected_Color (true, Colors.WAITING_STATE)},
		{Colors.HALTED_STATE,
			Colors.Selected_Color (true, Colors.HALTED_STATE)},
		{Colors.TABLE,
			Colors.TABLE_SELECTED}
		};

/**	Processing state label annotations.
*/
public static final String
	STATE_ANNOTATIONS[] = {"r", "p", "w", "h", "t"};

/**	Bit flags for processing state count selection.
*/
public static final int
	RUNNING_COUNT		= 1 << RUNNING_INDEX,
	POLLING_COUNT		= 1 << POLLING_INDEX,
	WAITING_COUNT		= 1 << WAITING_INDEX,
	HALTED_COUNT		= 1 << HALTED_INDEX,
	TOTAL_COUNT			= 1 << TOTAL_INDEX,
	ALL_COUNTS			= RUNNING_COUNT
						| POLLING_COUNT
						| WAITING_COUNT
						| HALTED_COUNT
						| TOTAL_COUNT;

private int
	Counts_Shown		= 0;

private boolean
	State_Annotate		= false;

private Font
	Emphasis_Font;

Conductor_Matrix_Cell_Renderer
	Renderer;


private static final String
	NL					= System.getProperty ("line.separator");


//  DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTOR	= 1 << 0,
	DEBUG_ACCESSORS		= 1 << 1,
	DEBUG_RENDERER		= 1 << 2,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
public Conductor_Matrix
	(
	Conductor_Matrix_Model	matrix_model
	)
{
super (matrix_model);
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println (">>> Conductor_Matrix");

setColumnSelectionAllowed (false);
setRowSelectionAllowed (true);
setSelectionMode (ListSelectionModel.SINGLE_SELECTION);
//setAutoResizeMode (JTable.AUTO_RESIZE_LAST_COLUMN);
//setShowVerticalLines (false);
setBackground (Colors.TABLE);
getTableHeader ().setBackground (Colors.TABLE_HEADER);
/*
	WARNING: These are JXTable specific methods.
*/
setColumnControlVisible (true);
setSortable (true);

Renderer = new Conductor_Matrix_Cell_Renderer ();
setDefaultRenderer (String.class, Renderer);
setDefaultRenderer (Conductor_Matrix_Model.Count.class, Renderer);

setRowHeight (Renderer.Cell_Panel.getPreferredSize ().height);

Emphasis_Font = getFont ().deriveFont (Font.BOLD);
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println ("<<< Conductor_Matrix");
}

/*==============================================================================
	Accessors
*/
/**	Test if Conductor processing state count annotations are enabled.
<p>
	@return	true if processing state count annotations will be
		displayed; false otherwise.
	@see	#State_Annotate(boolean)
*/
public boolean State_Annotate ()
{return State_Annotate;}

/**	Enable or disable Conductor processing state count annotations.
<p>
	When enable each Conductor processing state count label includes a
	single character annotation corresponding to the first character
	of its state name: running, polling, waiting, halted and total.
<p>
	When disabled the annotatations are not included in the label.
<p>
	@param	enable	If true processing state count annotations will be
		displayed. If false annotations are not displayed.
	@return	This Conductor_Matrix.
*/
public Conductor_Matrix State_Annotate
	(
	boolean	enable
	)
{
if (State_Annotate != enable)
	{
	State_Annotate = enable;
	repaint ();
	}
return this;
}

/**	Get the Conductor processing state counts that are shown in the table.
<p>
	@return	A bit mask indicating which Conductor processing state
		counts are shown in the table.
	@see	#Counts_Shown(int)
*/
public int Counts_Shown ()
{return Counts_Shown;}

/**	Set the Conductor processing state counts that are shown in the table.
<p>
	Conductor processing state counts to be shown are selected by a bit
	mask composed of the logic OR-ing of the following values:
<dl>
<dt>{@link #RUNNING_COUNT}
<dd>Conductors that are running: Source records are currently being processed.

<dt>{@link #POLLING_COUNT}
<dd>Conductors that are polling: The Conductor is running but no unprocessed
	source records are available from the database so the Conductor is
	periodically pollling for additional unprocessed source records.

<dt>{@link #WAITING_COUNT}
<dd>Conductors that are waiting: The Conductor has stopped processing
	source records by request, or it has yet to start, and is waiting to
	be told to start running.

<dt>{@link #HALTED_COUNT}
<dd>Conductors that have halted: The Conductor has stopped processing
	due to the sequential source record processing failures limit having
	been reached or an error condition was encountered.

<dt>{@link #TOTAL_COUNT}
<dd>The total number of Conductors of the corresponding name on the
	corresponding Theater.

<dt>{@link #ALL_COUNTS}
<dd>All of the above counts. This is the default.
</dl>
<p>
	@param	counts_shown	A bit mask int value that is the OR-ing of
		processing state count selection flags. If zero {@link
		#ALL_COUNTS} will be used.
	@return	This Conductor_Matrix.
*/
public Conductor_Matrix Counts_Shown
	(
	int		counts_shown
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">>> Conductor_Matrix.Counts_Shown: " + counts_shown);
if (Renderer.Counts_Shown (counts_shown))
	repaint ();
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("<<< Conductor_Matrix.Counts_Shown");
return this;
}


/*	If not provided by the base class.
public int convertRowIndexToModel
	(
	int		view_row
	)
{return view_row;}
*/

/*==============================================================================
	Conductor_Matrix_Cell_Renderer
*/
private class Conductor_Matrix_Cell_Renderer
	implements TableCellRenderer
{
private JPanel
	Cell_Panel;
private JLabel
	State_Labels[]		= new JLabel[TOTAL_INDEX + 1];
private JLabel
	Name_Label			= new JLabel (),
	Empty_Label			= new JLabel ();
private Color
	Background_Color;

private static final int
	BORDER_WIDTH = 2;


public Conductor_Matrix_Cell_Renderer ()
{
//	Conductor names label.
Name_Label.setOpaque (true);

//	Empty cell label.
Empty_Label.setOpaque (true);

//	Panel to contain the processing state counts cells.
Cell_Panel = new JPanel (new GridBagLayout ());
Cell_Panel.setOpaque (true);
Cell_Panel.setBackground (Colors.TABLE);

//	Processing state counts cell labels.
JLabel
	label;
for (int
		index = 0;
		index < TOTAL_INDEX;
		index++)
	{
	label = new JLabel (" ", SwingConstants.RIGHT);
	label.setVerticalAlignment (SwingConstants.CENTER);
	label.setOpaque (true);
	label.setBorder
		(BorderFactory.createCompoundBorder
		(BorderFactory.createBevelBorder
			(BevelBorder.RAISED, Color.WHITE, Color.GRAY),
		 BorderFactory.createLineBorder
		 	(STATE_COLORS[index][0], BORDER_WIDTH)));
	State_Labels[index] = label;
	}

label = new JLabel (" ", SwingConstants.RIGHT);
label.setVerticalAlignment (SwingConstants.CENTER);
label.setOpaque (true);
label.setBorder
	(BorderFactory.createCompoundBorder
		(BorderFactory.createBevelBorder
			(BevelBorder.RAISED, Color.WHITE, Color.GRAY),
		 BorderFactory.createEmptyBorder
			(BORDER_WIDTH, BORDER_WIDTH, BORDER_WIDTH, BORDER_WIDTH)));
State_Labels[TOTAL_INDEX] = label;

//	Assemble the initial contents of the cell panel.
Counts_Shown (ALL_COUNTS);
}


public boolean Counts_Shown
	(
	int		counts_shown
	)
{
if ((DEBUG & DEBUG_RENDERER) != 0)
	System.out.println
		(">>> Conductor_Matrix_Cell_Renderer.Counts_Shown: " + counts_shown);
counts_shown &= ALL_COUNTS;
if (counts_shown == 0)
	counts_shown = TOTAL_COUNT;

if (Counts_Shown == counts_shown)
	{
	if ((DEBUG & DEBUG_RENDERER) != 0)
		System.out.println
			("<<< Conductor_Matrix_Cell_Renderer.Counts_Shown: false");
	return false;
	}

if ((DEBUG & DEBUG_RENDERER) != 0)
	System.out.println
		("    New Counts_Shown = " + counts_shown);
Counts_Shown = counts_shown;

Cell_Panel.removeAll ();
GridBagConstraints
	location = new GridBagConstraints ();
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
int
	index = -1;
while (++index <= TOTAL_INDEX)
	if ((Counts_Shown & (1 << index)) != 0)
		Cell_Panel.add (State_Labels[index], location);

if ((DEBUG & DEBUG_RENDERER) != 0)
	System.out.println
		("<<< Conductor_Matrix_Cell_Renderer.Counts_Shown: true");
return true;
}


public Component getTableCellRendererComponent
	(
	JTable	table,
	Object	value,
	boolean	selected,
	boolean	focused,
	int		row,
	int		column
	)
{
if ((DEBUG & DEBUG_RENDERER) != 0)
	System.out.println
		(">>> Conductor_Matrix.getTableCellRendererComponent:" + NL
		+"    Value - " + value + NL
		+"    Selected - " + selected + NL
		+"    Table row " + row + " column " + column);
row = ((JXTable)table).convertRowIndexToModel (row);
column = table.convertColumnIndexToModel (column);
if ((DEBUG & DEBUG_RENDERER) != 0)
	System.out.println
		("    Model row " + row + " column " + column);
if (value == null)
	value = "null";

Component
	component = null;
if ((row % 2) != 0)
	Background_Color = selected ?
		Colors.ALTERNATE_SELECTED : Colors.ALTERNATE;
else if (column == 0)
	Background_Color = selected ?
		Colors.TABLE_HEADER_SELECTED : Colors.TABLE_HEADER;
else
	Background_Color = selected ?
		Colors.TABLE_SELECTED : Colors.TABLE;

if (column == 0)
	{
	component = Name_Label;
	Name_Label.setText (value.toString ());
	Name_Label.setBackground (Background_Color);
	}
else
if (value instanceof Conductor_Matrix_Model.Count)
	{
	if ((DEBUG & DEBUG_RENDERER) != 0)
		System.out.println
			("    Counts_Shown: " + Counts_Shown
				+ (((Counts_Shown & RUNNING_COUNT) != 0) ? " RUNNING" : "")
				+ (((Counts_Shown & POLLING_COUNT) != 0) ? " POLLING" : "")
				+ (((Counts_Shown & WAITING_COUNT) != 0) ? " WAITING" : "")
				+ (((Counts_Shown & HALTED_COUNT) != 0) ? " HALTED" : "")
				+ (((Counts_Shown & TOTAL_COUNT) != 0) ? " TOTAL" : "")
				);
	Conductor_Matrix_Model.Count
		count = (Conductor_Matrix_Model.Count)value;
	if (count.Counts[TOTAL_INDEX] == 0)
		{
		component = Empty_Label;
		Empty_Label.setBackground (Background_Color);
		}
	else
		{
		component = Cell_Panel;
		int
			index = -1;
		while (++index <= TOTAL_INDEX)
			if ((Counts_Shown & (1 << index)) != 0)
				Format (index, count.Counts[index], selected);
		}
	}
else
	{
	component = Name_Label;
	Name_Label.setText (value.toString ());
	Name_Label.setBackground (Background_Color);
	}

if ((DEBUG & DEBUG_RENDERER) != 0)
	System.out.println
		("<<< Conductor_Matrix.getTableCellRendererComponent");
return component;
}


private void Format
	(
	int		index,
	int		value,
	boolean	selected
	)
{
String
	text;
Color
	color = Background_Color;
if (value == 0)
	text = " ";
else
	{
	text = String.valueOf (value);
	if (index < TOTAL_INDEX)
		color = selected ? STATE_COLORS[index][1] : STATE_COLORS[index][0];
	}
if (State_Annotate)
	text += STATE_ANNOTATIONS[index];

State_Labels[index].setText (text);
State_Labels[index].setBackground (color);
}

}	//	Conductor_Table_Cell_Renderer


}
