/* Nessus
 * Copyright (C) 1998 - 2001 Renaud Deraison
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2,
 * as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * In addition, as a special exception, Renaud Deraison
 * gives permission to link the code of this program with any
 * version of the OpenSSL library which is distributed under a
 * license identical to that listed in the included COPYING.OpenSSL
 * file, and distribute linked combinations including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * this file, you may extend this exception to your version of the
 * file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 */

/**
 * @file
 * Module that deals with representation of (collections of) NVTs.
 */

#include <includes.h>
#include <glib.h>
#include "nessus_plugin.h"

/**
 * @brief Static hash table to keep 'interned' strings.
 */
static GHashTable* interned_strings = NULL;

/**
 * @brief 'Interns' a string by adding a duplicate of it to a static hash table.
 */
static char *
intern_string(char * s)
{
  char *interned;

  if (!interned_strings)
    {
      interned_strings = g_hash_table_new(g_str_hash, g_str_equal);
    }

  interned = g_hash_table_lookup(interned_strings, s);
  if (interned)
    return interned;

  interned = estrdup(s);

  g_hash_table_insert(interned_strings, interned, interned);
  return interned;
}


struct nessus_plugin *
nessus_plugin_new(char * oid,
                  char * name,
                  char * category,
                  char * copyright,
                  char * description,
                  char * summary,
                  char * family,
                  char * version,
                  char * cve,
                  char * bid,
                  char * xrefs,
                  char * sign_key_ids,
                  char * script_tags
                  )
{
 struct nessus_plugin * np = malloc(sizeof(*np));

 if ( np == NULL ) abort();
 bzero(np, sizeof(*np));
 np->oid = cache_inc(oid);
 np->md5sum = NULL;

 np->name = cache_inc(name); 
 np->category = cache_inc(category);
 np->copyright = cache_inc(copyright);

 np->description = intern_string(description);

 np->summary = intern_string(summary);
 np->family  = cache_inc(family);
 np->version = cache_inc(version);
 np->cve     = cache_inc(cve);
 np->bid     = cache_inc(bid);
 np->xrefs   = cache_inc(xrefs);
 np->script_tags = cache_inc(script_tags);
 np->sign_key_ids = cache_inc(sign_key_ids);
 np->next    = NULL;
 return np;
}


/**
 * @brief Set the md5 sum of the plugin (parameter is copied).
 * 
 * @param plugin The NVT to get a new md5sum.
 * @param md5sum New md5sum of a plugin (will be copied).
 */
void
nessus_plugin_set_md5sum(struct nessus_plugin * plugin, const char * md5sum)
{
  efree(&plugin->md5sum);
  if (md5sum != NULL)
    plugin->md5sum = estrdup(md5sum);
}


/**
 * @brief Returns the first plugin in the plugin set \<plugins\> with the name
 * @brief \<name\>.
 * 
 * @param plugins List of plugins.
 * @param name    Name of plugin to find.
 * 
 * @return First plugin in plugin set with given name.
 */
struct nessus_plugin *
nessus_plugin_get_by_name(struct nessus_plugin * plugins, char * name)
{
 while ( plugins != NULL )
	{
 	if ( strcmp(plugins->name, name) == 0 ) return plugins;
	plugins = plugins->next;
	}

 return NULL;
}

/**
 * @brief Returns the first plugin in the plugin set \<plugins\> with the OID
 * @brief \<oid\>.
 * 
 * @param plugins List of plugins.
 * @param name    OID of plugin to find.
 * 
 * @return First plugin in plugin set with given name.
 */
struct nessus_plugin *
nessus_plugin_get_by_oid(struct nessus_plugin * plugins, const char * oid)
{
 while ( plugins != NULL )
 	{
	if (!strcmp(plugins->oid, oid)) return plugins;
	plugins = plugins->next;
	}

  return NULL;
}

char *
nessus_plugin_get_description(struct nessus_plugin * plugin)
{
  return plugin->description;
}

/**
 * @brief Returns a duplicate of a plugin.
 * 
 * Properties will be 'interned' and included in the cache.
 * Preferences are deep- copied.
 * 
 * @param plugin The plugin to duplicate.
 * 
 * @return Duplicate of plugin parameter.
 */
struct nessus_plugin*
nessus_plugin_duplicate(struct nessus_plugin *plugin)
{
  struct nessus_plugin * copy = nessus_plugin_new(plugin->oid,
      plugin->name, plugin->category, plugin->copyright,
      nessus_plugin_get_description(plugin), plugin->summary, plugin->family,
      plugin->version, plugin->cve, plugin->bid, plugin->xrefs, 
      plugin->sign_key_ids, plugin->script_tags);
  copy->enabled = plugin->enabled;
  nessus_plugin_set_md5sum(copy, plugin->md5sum);
  if (plugin->plugin_prefs != NULL)
  {
    copy->plugin_prefs = emalloc(sizeof(struct arglist));
    arg_dup(copy->plugin_prefs, plugin->plugin_prefs);
  }
  return copy;
}


void nessus_plugin_free(struct nessus_plugin * plugins)
{
 /** @TODO TBD */
}
