/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTL_CORE_TIME_CONTROLLER_
#define _GTL_CORE_TIME_CONTROLLER_

#include <GTLCore/Export.h>

namespace GTLCore {
  /**
    * @ingroup GTLCore_TimeMachine
    *
    * This class control the value of the time changed.
    */
  class GTLCORE_EXPORT TimeController {
    GTL_NO_COPY(TimeController);
    public:
      TimeController();
      virtual ~TimeController();
      /**
        * @return a value between 0 and 1, dependant of the time.
        */
      virtual double valueAt(double time) const = 0;
  };
  /**
    * @ingroup GTLCore_TimeMachine
    *
    * This class control the time lineary.
    */
  class GTLCORE_EXPORT LinearTimeController : public TimeController {
    GTL_NO_COPY(LinearTimeController);
    public:
      /**
        * @p _length of the time interval
        * @p _restart the number of time the controller is reset, if set to -1 it always restart to 0
        *             after reaching _length, otherwise once the time is superior to
        *             (_restart * _length) the value is constant to 1.0
        */
      LinearTimeController(double _length, int _restart = -1);
      virtual ~LinearTimeController();
      virtual double valueAt(double time) const;
    private:
      struct Private;
      Private* const d;
  };
  /**
    * @ingroup GTLCore_TimeMachine
    *
    * This class control the time with a sinusoid.
    */
  class GTLCORE_EXPORT SinusoidalTimeController : public TimeController {
    GTL_NO_COPY(SinusoidalTimeController);
    public:
      /**
        * @param _waveLength
        * @param _phasis
        */
      SinusoidalTimeController(double _waveLength, double _phasis);
      virtual ~SinusoidalTimeController();
      virtual double valueAt(double time) const;
    private:
      struct Private;
      Private* const d;
  };
}

#endif
