/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "TimeController.h"

#include <cfloat>
#include "Math.h"

using namespace GTLCore;

TimeController::TimeController()
{
}

TimeController::~TimeController()
{
}

struct LinearTimeController::Private
{
  double length;
  int restart;
  double stoplength;
};

LinearTimeController::LinearTimeController(double _length, int _restart) : d(new Private)
{
  d->length = _length;
  d->stoplength = (_restart == -1) ? FLT_MAX : _length * (_restart + 1);
  d->restart = _restart;
}

LinearTimeController::~LinearTimeController()
{
  delete d;
}

double LinearTimeController::valueAt(double time) const
{
  if(time >= d->stoplength)
  {
    return 1.0;
  } else {
    time = fmod(time, d->length);
    return time / d->length;
  }
}

struct SinusoidalTimeController::Private {
  double waveLength;
  double phasis;
};

SinusoidalTimeController::SinusoidalTimeController(double _waveLength, double _phasis) : d(new Private)
{
  d->waveLength = 2 * M_PI / _waveLength;
  d->phasis = _phasis;
}

SinusoidalTimeController::~SinusoidalTimeController()
{
  delete d;
}

double SinusoidalTimeController::valueAt(double time) const
{
  return sin(time * d->waveLength + d->phasis) * 0.5 + 0.5;
}
