/*
 ** Copyright 2004-2010 - EdenWall Technologies
 ** Written by Eric Leblond <eric.leblond@inl.fr>
 **            Vincent Deffontaines <vincent@inl.fr>
 **            Pierre Chifflier <chifflier@edenwall.com>
 ** INL http://www.inl.fr/
 **
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, version 2 of the License.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <nuclient.h>
#include <QtGui>

#include "systray.h"
#include "systray.moc"
#include "auth_dlg.h"
#include "preferences.h"
#include "logs.h"

extern "C" {
typedef void (*log_callback_t)(int area, int priority, char *format, va_list args);

extern log_callback_t nubase_log_set_callback(log_callback_t cb);

extern int log_engine;
extern int debug_level;
}

static NuLogsWindow *_logs_window = NULL;

static void log_callback(int area, int priority, char *format, va_list args)
{
	if (_logs_window != NULL) {
		_logs_window->addLine(area, priority, format, args);
	}
}

/*
 * See http://doc.trolltech.com/4.3/mainwindows-application.html
 */

NuAppSystray::NuAppSystray() : timer(this)
{
	QCoreApplication* app = QCoreApplication::instance();
	parse_cmdline_options(app->argc(), app->argv());

	logs_dlg = new NuLogsWindow();
	_logs_window = logs_dlg;
	pref_dlg = new NuAppPreferences();
	auth_dlg = new NuAppAuthDialog(this);
	QObject::connect(this, SIGNAL(activated(QSystemTrayIcon::ActivationReason)), this, SLOT(onClick(QSystemTrayIcon::ActivationReason)));
	QObject::connect(this, SIGNAL(showAuthDlg()), auth_dlg, SLOT(show()));

	// Create actions
	//  connect
	connectAct = new QAction(tr("&Connect"), this);
	connectAct->setStatusTip(tr("Connect to the authentication server"));
	connect(connectAct, SIGNAL(triggered()), auth_dlg, SLOT(show()));

	//  disconnect
	disconnectAct = new QAction(tr("&Disconnect"), this);
	disconnectAct->setStatusTip(tr("Disconnect from the authentication server"));
	connect(disconnectAct, SIGNAL(triggered()), this, SLOT(stop_auth()));

	//  preferences
	preferencesAct = new QAction(tr("&Preferences"), this);
	preferencesAct->setStatusTip(tr("Preferences"));
	connect(preferencesAct, SIGNAL(triggered()), pref_dlg, SLOT(show()));

	//  logs
	logsAct = new QAction(tr("&Log messages"), this);
	logsAct->setStatusTip(tr("Show log messages"));
	connect(logsAct, SIGNAL(triggered()), logs_dlg, SLOT(show()));

	//  about
	aboutAct = new QAction(tr("&About"), this);
	aboutAct->setStatusTip(tr("About"));
	connect(aboutAct, SIGNAL(triggered()), this, SLOT(popup_about()));

	//  exit
	exitAct = new QAction(tr("&Quit"), this);
	exitAct->setStatusTip(tr("Quit application"));
	connect(exitAct, SIGNAL(triggered()), qApp, SLOT(quit()));

	createTrayIcon();
	settings.sync();

	// Display the authentication dialog
	auth_dlg->show();

	nubase_log_set_callback(log_callback);
	log_engine = 3;
	debug_level = 9;

#ifdef HAVE_DEFAULT_TLS_FUNC
	// Load defaults settings
	if(strcmp(settings.value("hostname").toString().toStdString().c_str(), "") == 0)
	{
		printf("Loading default params\n");
		//suppress_fqdn_verif |= nu_client_default_suppress_fqdn_verif();
		const char* default_cafile = nu_client_default_tls_ca();
		if (default_cafile)
			settings.setValue("ca", default_cafile);
		const char* default_cert = nu_client_default_tls_cert();
		if (default_cert)
			settings.setValue("cert", default_cert);
		const char* default_key = nu_client_default_tls_key();
		if (default_key)
			settings.setValue("key", default_key);
		const char* default_crl = nu_client_default_tls_crl();
		if (default_crl)
			settings.setValue("crl", default_crl);
		const int default_suppress_fqdn = nu_client_default_suppress_fqdn_verif();
		if (default_suppress_fqdn)
			settings.setValue("suppress_fqdn", default_suppress_fqdn);
		if(default_cafile || default_cert || default_key)
			settings.setValue("use_certificate", true);
	}
#endif

#ifdef HAVE_DEFAULT_HOSTNAME_FUNC
	// Load defaults settings
	if(strcmp(settings.value("hostname").toString().toStdString().c_str(), "") == 0)
	{
		const char* nuauth_ip = nu_client_default_hostname();
		if(nuauth_ip != NULL)
			settings.setValue("hostname", nuauth_ip);

		const char* nuauth_port = nu_client_default_port();
		if(nuauth_port != NULL)
			settings.setValue("port", nuauth_port);
		else
			settings.setValue("port", DEFAULT_NUAUTH_PORT);
	}
#endif

#if defined(HAVE_DEFAULT_HOSTNAME_FUNC) || !defined(HAVE_NUAUTH_IP)
	if(strcmp(settings.value("hostname").toString().toStdString().c_str(), "") == 0)
		pref_dlg->show();
#endif
}

NuAppSystray::~NuAppSystray()
{
	stop_auth();
}

void NuAppSystray::createTrayIcon()
{
	if ( ! QSystemTrayIcon::isSystemTrayAvailable() ) {
		fprintf(stderr, "Your environnement doesn't support a system tray icon...\n");
		fprintf(stderr, "Exiting ...\n");
		exit(EXIT_FAILURE);
	}

	trayIconMenu = new QMenu(/*this*/);
	trayIconMenu->addAction(connectAct);
	connectAct->setEnabled(true);
	trayIconMenu->addAction(disconnectAct);
	disconnectAct->setEnabled(false);
	trayIconMenu->addAction(logsAct);
	trayIconMenu->addAction(preferencesAct);
	trayIconMenu->addSeparator();
	trayIconMenu->addAction(aboutAct);
	trayIconMenu->addSeparator();
	trayIconMenu->addAction(exitAct);

	QIcon icon = QIcon(":/images/nuapplet2-stopped.png");
	setIcon(icon);
	show();
	setContextMenu(trayIconMenu);
}

void NuAppSystray::usage(void)
{
	fprintf(stderr, "usage: nuapplet2 [options...]\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "Options:\n");
	fprintf(stderr, "  -H HOSTNAME: nuauth host address\n");
	fprintf(stderr, "  -p PORT: nuauth port number\n");
	fprintf(stderr, "  -U USERNAME: set default username\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "Certificate options:\n");
	fprintf(stderr, "  -C CERTFILE: certificate filename\n");
	fprintf(stderr, "  -A AUTHFILE: authority certificate filename\n");
	fprintf(stderr, "  -K KEYFILE:  key filename\n");
	fprintf(stderr, "  -R CRLFILE: crl filename\n");
	fprintf(stderr, "  -N: suppress error if server FQDN does not match certificate CN.\n");
	fprintf(stderr, "\n");
	exit(EXIT_FAILURE);
}

void NuAppSystray::parse_cmdline_options(int argc, char **argv)
{
	/* Parse all command line arguments */
	int ch;
	int opterr = 0;
	while ((ch = getopt(argc, argv, "H:p:U:C:K:A:R:N")) != -1) {
		switch (ch) {
		case 'H':
			settings.setValue("hostname", QString(optarg));
			break;
		case 'p':
			settings.setValue("port", QString(optarg));
			break;
		case 'U':
			settings.setValue("username", QString(optarg));
			break;
		case 'C':
			settings.setValue("cert", QString(optarg));
			break;
		case 'K':
			settings.setValue("key", QString(optarg));
			break;
		case 'A':
			settings.setValue("ca", QString(optarg));
			break;
		case 'R':
			settings.setValue("crl", QString(optarg));
			break;
		case 'N':
			settings.setValue("suppress_fqdn", 1);
		default:
			usage();
		}
	}
}

void NuAppSystray::onClick(QSystemTrayIcon::ActivationReason reason)
{
	if( reason == QSystemTrayIcon::Trigger && !timer.isRunning())
	{
		// Display the authentication dialog
		emit showAuthDlg();
	}
}

void NuAppSystray::start_auth(QString username, QString password)
{
	if(!timer.isRunning())
	{
		timer.setUserInfos(username, password);
		timer.start();
		showMessage(tr("Warning"), tr("Nuapplet isn't totally strict about TLS verifications.\nRun it from a terminal to see any warning that could happen."), Warning, 5000);
	}
}

void NuAppSystray::stop_auth()
{
	// Close Nuauth connection
	timer.askStop();
	timer.wait(ULONG_MAX);
}

void NuAppSystray::tray_set_connected()
{
	disconnectAct->setEnabled(true);
	connectAct->setEnabled(false);

	//printf("Icon running\n");
	QIcon icon = QIcon(":/images/nuapplet2-running.png");
	setIcon(icon);
}

void NuAppSystray::tray_set_trying()
{
	QIcon icon = QIcon(":/images/nuapplet2-trying.png");
	setIcon(icon);
	//printf("Icon trying\n");
}

void NuAppSystray::tray_set_stopped()
{
	// Enable / disable menu entries
	disconnectAct->setEnabled(false);
	connectAct->setEnabled(true);

	// Change the icon in the systray
	QIcon icon = QIcon(":/images/nuapplet2-stopped.png");
	setIcon(icon);
	//printf("Icon stopped\n");
}

/**
 * Stupid wrapper
 */
const char *wrap_error_message(const char *str)
{
	if (strncmp(str,"A TLS packet with unexpected length was received.",49) == 0) {
		return("Access denied");
	}
	if (strncmp(str,"Success.",9) == 0) {
		return NULL;
	}
	return(str);
}

void NuAppSystray::tray_report_error()
{
	QString msg = tr("Unable to connect.\nCheck your username and your password are correct.");
	QString str_err = "";
	if(wrap_error_message(timer.GetNuclientError()))
	{
		str_err = QString("\nError is: ") + QString(wrap_error_message(timer.GetNuclientError()));
	}
	logs_dlg->show();
	QMessageBox::critical(NULL, tr("Error"), msg + str_err, QMessageBox::Ok);
}

void NuAppSystray::tray_report_connected()
{
	showMessage(tr("Connection succeeded"), tr("You are now connected to ") + settings.value("hostname").toString(), Information,3000 );
	printf("Connnected\n");
}

void NuAppSystray::tray_report_disconnected()
{
	showMessage(tr("Warning"), tr("You have been disconnected\nReason: ") + QString(timer.GetNuclientError()), Warning,5000 );
	printf("Disconnected\n");
}

void NuAppSystray::popup_about()
{
	QMessageBox msgBox;
	msgBox.setWindowTitle(tr("About"));
	msgBox.setIcon(QMessageBox::Information);
	msgBox.setTextFormat(Qt::RichText);
	msgBox.setText(tr("NuApplet %1 edited by <a href=\"http://www.edenwall.com/\">EdenWall Technologies</a>.<br>Distributed under the <a href=\"http://www.gnu.org/copyleft/gpl.html\">GPL</a><br>").arg(PROJECT_VERSION_STR) + "<img src=\":/images/edenwall.png\"/>");
	msgBox.exec();
}

