/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2010 Sun Microsystems, Inc.
 */

package org.netbeans.modules.cnd.modelimpl.trace;

import java.io.IOException;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collection;
import org.netbeans.modules.cnd.api.model.CsmFile;
import org.netbeans.modules.cnd.api.model.CsmModelAccessor;
import org.netbeans.modules.cnd.api.model.CsmProject;
import org.netbeans.modules.cnd.api.model.services.CsmCacheManager;
import org.netbeans.modules.cnd.api.model.services.CsmStandaloneFileProvider;
import org.netbeans.modules.cnd.api.model.util.CsmTracer;
import org.netbeans.modules.cnd.api.project.NativeFileItem;
import org.netbeans.modules.cnd.api.project.NativeFileItemSet;
import org.netbeans.modules.cnd.api.project.NativeProject;
import org.netbeans.modules.cnd.debug.CndDiagnosticProvider;
import org.netbeans.modules.cnd.modelimpl.content.file.ReferencesIndex;
import org.netbeans.modules.cnd.modelimpl.csm.core.*;
import org.netbeans.modules.cnd.utils.FSPath;
import org.netbeans.modules.cnd.utils.cache.CndFileUtils;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.util.Lookup;
import org.openide.util.NbBundle;
import org.openide.util.lookup.ServiceProvider;
import org.openide.windows.OutputWriter;
import static org.netbeans.modules.cnd.modelimpl.trace.Bundle.*;

/**
 *
 * @author vv159170
 */

public final class CodeModelDiagnostic {
    public static void dumpProjectContainers(PrintStream ps, CsmProject prj, boolean dumpFiles) {
        ProjectBase.dumpProjectContainers(ps, prj, dumpFiles);
    }

    public static void dumpFileContainer(CsmProject project, OutputWriter out) {
        ProjectBase.dumpFileContainer(project, out);
    }

    public static void dumpProjectGrapthContainer(CsmProject project, OutputWriter out) {
        ProjectBase.dumpProjectGrapthContainer(project, out);
    }

    @ServiceProvider(service = CndDiagnosticProvider.class, position = 1000)
    public final static class StandAloneProviderTrace implements CndDiagnosticProvider {

        @NbBundle.Messages({"StandAloneProviderTrace.displayName=Standalone Files Information"})
        @Override
        public String getDisplayName() {
            return StandAloneProviderTrace_displayName();
        }

        @Override
        public void dumpInfo(Lookup context, PrintWriter printOut) {
            printOut.printf("====CsmStandaloneFileProviders info:%n");// NOI18N
            for (CsmStandaloneFileProvider sap : Lookup.getDefault().lookupAll(CsmStandaloneFileProvider.class)) {
                if (sap instanceof CsmStandaloneFileProviderImpl) {
                    ((CsmStandaloneFileProviderImpl) sap).dumpInfo(printOut);
                } else {
                    printOut.printf("UNKNOWN FOR ME [%s] %s%n", sap.getClass().getName(), sap.toString());// NOI18N
                }
            }
        }
    }

    @ServiceProvider(service = CndDiagnosticProvider.class, position = 1100)
    public final static class FileTrace implements CndDiagnosticProvider {

        @NbBundle.Messages({"FileTrace.displayName=General File Information"})
        @Override
        public String getDisplayName() {
            return FileTrace_displayName();
        }

        @Override
        public void dumpInfo(Lookup context, PrintWriter printOut) {
            printOut.printf("====Files info:%nGlobal ParseCount=%d%n", FileImpl.getLongParseCount());// NOI18N
            Collection<? extends CsmFile> allFiles = context.lookupAll(CsmFile.class);
            for (CsmFile csmFile : allFiles) {
                if (csmFile instanceof FileImpl) {
                    ((FileImpl) csmFile).dumpInfo(printOut);
                } else if (csmFile instanceof FileSnapshot) {
                    ((FileSnapshot) csmFile).dumpInfo(printOut);
                } else {
                    printOut.printf("UNKNOWN FOR ME [%s] %s%n", csmFile.getClass().getName(), csmFile.toString());// NOI18N
                }
            }
            Collection<? extends DataObject> dobs = context.lookupAll(DataObject.class);
            if (!dobs.isEmpty()) {
                boolean foundItem = false;
                for(DataObject dob : dobs) {
                    NativeFileItemSet nfis = dob.getLookup().lookup(NativeFileItemSet.class);
                    if (nfis != null) {
                        foundItem = true;
                        printOut.printf("NativeFileItemSet has %d elements%n", nfis.getItems().size());// NOI18N
                        int ind = 0;
                        for (NativeFileItem item : nfis.getItems()) {
                            printOut.printf("[%d] NativeFileItem %s of class %s%n", ++ind, item.getAbsolutePath(), item.getClass().getName());// NOI18N
                            NativeProject nativeProject = item.getNativeProject();
                            printOut.printf(" from project %s [%s]%n", nativeProject.getProjectDisplayName(), nativeProject.getProjectRoot());// NOI18N
                            printOut.printf("\tLang=%s Flavor=%s excluded=%s%n", item.getLanguage(), item.getLanguageFlavor(), item.isExcluded());// NOI18N
                            printOut.print("\tUser Include Paths:\n");// NOI18N
                            for (FSPath path : item.getUserIncludePaths()) {
                                String msg = CndFileUtils.isLocalFileSystem(path.getFileSystem()) ? path.getPath() : path.getURL().toString();
                                FileObject valid = path.getFileObject();
                                if (valid != null && !valid.isValid()) {
                                    valid = null;
                                }
                                printOut.printf("\t\t%s%s%n", msg, valid == null ? "[invalid]" : "");// NOI18N
                            }
                            if (!item.getIncludeFiles().isEmpty()) {
                                printOut.print("\tUser Include Files:\n");// NOI18N
                                for (String path : item.getIncludeFiles()) {
                                    String msg = path;
                                    printOut.printf("\t\t%s%s%n", msg, "");// NOI18N
                                }
                            }
                            printOut.print("\tUser Macros:\n");// NOI18N
                            for (String macro : item.getUserMacroDefinitions()) {
                                printOut.printf("\t\t%s%n", macro);// NOI18N
                            }
                            printOut.print("\tSystem Include Paths:\n");// NOI18N
                            for (FSPath path : item.getSystemIncludePaths()) {
                                String msg = CndFileUtils.isLocalFileSystem(path.getFileSystem()) ? path.getPath() : path.getURL().toString();
                                FileObject valid = path.getFileObject();
                                if (valid != null && !valid.isValid()) {
                                    valid = null;
                                }
                                printOut.printf("\t\t%s%s%n", msg, valid == null ? "[invalid]" : "");// NOI18N
                            }
                            printOut.print("\tSystem Macros:\n");// NOI18N
                            for (String macro : item.getSystemMacroDefinitions()) {
                                printOut.printf("\t\t%s%n", macro);// NOI18N
                            }
                        }
                    }
                }
                if(!foundItem) {
                    printOut.printf("no NativeFileItemSet in %s%n", context);// NOI18N
                }
            } else {
                printOut.printf("no file object in lookup%n");// NOI18N
            }
        }
    }

    @ServiceProvider(service = CndDiagnosticProvider.class, position = 1200)
    public final static class PPStatesTrace implements CndDiagnosticProvider {

        @NbBundle.Messages({"PPStatesTrace.displayName=Preprocessor States"})
        @Override
        public String getDisplayName() {
            return PPStatesTrace_displayName();
        }

        @Override
        public void dumpInfo(Lookup context, PrintWriter printOut) {
            printOut.printf("====Files info:%nGlobal ParseCount=%d%n", FileImpl.getLongParseCount());// NOI18N
            Collection<? extends CsmFile> allFiles = context.lookupAll(CsmFile.class);
            for (CsmFile csmFile : allFiles) {
                if (csmFile instanceof FileImpl) {
                    ((FileImpl) csmFile).dumpPPStates(printOut);
                } else {
                    printOut.printf("UNKNOWN FOR ME [%s] %s%n", csmFile.getClass().getName(), csmFile.toString());// NOI18N
                }
            }
        }
    }

    @ServiceProvider(service = CndDiagnosticProvider.class, position = 1250)
    public final static class IncludePPStatesTrace implements CndDiagnosticProvider {

        @NbBundle.Messages({"IncludePPStatesTrace.displayName=Included Preprocessor States"})
        @Override
        public String getDisplayName() {
            return IncludePPStatesTrace_displayName();
        }

        @Override
        public void dumpInfo(Lookup context, PrintWriter printOut) {
            printOut.printf("====Files info:%nGlobal ParseCount=%d%n", FileImpl.getLongParseCount());// NOI18N
            Collection<? extends CsmFile> allFiles = context.lookupAll(CsmFile.class);
            for (CsmFile csmFile : allFiles) {
                if (csmFile instanceof FileImpl) {
                    ((FileImpl) csmFile).dumpIncludePPStates(printOut);
                } else {
                    printOut.printf("UNKOWN FOR ME [%s] %s%n", csmFile.getClass().getName(), csmFile.toString());// NOI18N
                }
            }
        }
    }

    @ServiceProvider(service = CndDiagnosticProvider.class, position = 1300)
    public final static class ModelProjectsTrace implements CndDiagnosticProvider {

        @NbBundle.Messages({"ModelProjectsTrace.displayName=Model Projects"})
        @Override
        public String getDisplayName() {
            return ModelProjectsTrace_displayName();
        }

        @Override
        public void dumpInfo(Lookup context, PrintWriter printOut) {
            printOut.printf("====ModelImpl:%n");// NOI18N
            ModelImpl.instance().dumpInfo(printOut, false);
            printOut.printf("====Libraries:%n"); //NOI18N
            LibraryManager.dumpInfo(printOut, false);
            printOut.printf("====Files count size summary:%n"); //NOI18N
            dumpProjectFilesInfo(printOut);
        }

        private void dumpProjectFilesInfo(PrintWriter printOut) {
            Collection<CsmProject> projects = CsmModelAccessor.getModel().projects();
            for (CsmProject project : projects) {
                dumpProjectFilesInfo(project, printOut, false);
                for (CsmProject lib : project.getLibraries()) {
                    dumpProjectFilesInfo(lib, printOut, false);
                }
            }
        }

        private void dumpProjectFilesInfo(CsmProject project, PrintWriter printOut, boolean printList) {
            Collection<CsmFile> sourceFiles = project.getSourceFiles();
            Collection<CsmFile> headerFiles = project.getHeaderFiles();
            printOut.printf("%s%n", project.getDisplayName());// NOI18N
            printOut.printf("   %,d source files; %,d header files; %,d total files%n", // NOI18N
                    sourceFiles.size(), headerFiles.size(), sourceFiles.size() + headerFiles.size());
            long totalSize = 0;
            long maxSize = 0;
            for (CsmFile file : sourceFiles) {
                if (printList) {
                    printOut.printf("\t%s%n", file.getAbsolutePath()); // NOI18N
                }
                FileObject fo = file.getFileObject();
                if (fo != null && fo.isValid()) {
                    totalSize += fo.getSize();
                    maxSize = Math.max(maxSize, fo.getSize());
                }
            }
            for (CsmFile file : headerFiles) {
                if (printList) {
                    printOut.printf("\t%s%n", file.getAbsolutePath()); // NOI18N
                }
                FileObject fo = file.getFileObject();
                if (fo != null && fo.isValid()) {
                    totalSize += fo.getSize();
                    maxSize = Math.max(maxSize, fo.getSize());
                }
            }
            printOut.printf("   total files size: %,d KBytes;  max file size: %,d KBytes%n", kilobytes(totalSize), kilobytes(maxSize)); // NOI18N
        }
        private static long kilobytes(long num) {
            return ((num % 1024) < 512) ? num/1024 : num/1024+1;
        }

    }

    @ServiceProvider(service = CndDiagnosticProvider.class, position = 1350)
    public final static class ModelProjectsContainers implements CndDiagnosticProvider {

        @NbBundle.Messages({"ModelProjectsContainers.displayName=Model Projects File Containers"})
        @Override
        public String getDisplayName() {
            return ModelProjectsContainers_displayName();
        }

        @Override
        public void dumpInfo(Lookup context, PrintWriter printOut) {
            printOut.printf("====ModelImpl:%n");// NOI18N
            ModelImpl.instance().dumpInfo(printOut, true);
            printOut.printf("====Libraries:%n"); //NOI18N
            LibraryManager.dumpInfo(printOut, true);
        }
    }

    @ServiceProvider(service = CndDiagnosticProvider.class, position = 1325)
    public final static class ModelProjectsIndex implements CndDiagnosticProvider {

        @NbBundle.Messages({"ModelProjectsIndex.displayName=Model Projects Index"})
        @Override
        public String getDisplayName() {
            return ModelProjectsIndex_displayName();
        }

        @Override
        public void dumpInfo(Lookup context, PrintWriter printOut) {
            printOut.printf("====Model Projects Index:%n");// NOI18N
            ReferencesIndex.dumpInfo(printOut);
        }
    }

    @ServiceProvider(service = CndDiagnosticProvider.class, position = 1375)
    public final static class ModelFileIndex implements CndDiagnosticProvider {

        @NbBundle.Messages({"ModelFileIndex.displayName=File References Index"})
        @Override
        public String getDisplayName() {
            return ModelTrace_displayName();
        }

        @Override
        public void dumpInfo(Lookup context, PrintWriter printOut) {
            printOut.printf("====File Indices:%n");// NOI18N
            Collection<? extends CsmFile> allFiles = context.lookupAll(CsmFile.class);
            for (CsmFile csmFile : allFiles) {
                if (csmFile instanceof FileImpl) {
                    ((FileImpl) csmFile).dumpIndex(printOut);
                } else if (csmFile instanceof FileSnapshot) {
                    ((FileSnapshot) csmFile).dumpIndex(printOut);
                } else {
                    printOut.printf("UNKNOWN FOR ME [%s] %s%n", csmFile.getClass().getName(), csmFile.toString());// NOI18N
                }
            }
        }
    }

    @ServiceProvider(service = CndDiagnosticProvider.class, position = 1400)
    public final static class FileImplModelTrace implements CndDiagnosticProvider {

        @NbBundle.Messages({"FileImplModelTrace.displayName=File Code Model"})
        @Override
        public String getDisplayName() {
            return FileImplModelTrace_displayName();
        }

        @Override
        public void dumpInfo(Lookup context, PrintWriter printOut) throws IOException {
            CsmCacheManager.enter();
            try {
                Collection<? extends CsmFile> allFiles = context.lookupAll(CsmFile.class);
                for (CsmFile csmFile : allFiles) {
                    new CsmTracer(printOut).dumpModel(csmFile);
                }
            } finally {
                CsmCacheManager.leave();
            }
        }
    }

    @ServiceProvider(service = CndDiagnosticProvider.class, position = 1400)
    public final static class FileImplASTTrace implements CndDiagnosticProvider {

        @NbBundle.Messages({"FileImplASTTrace.displayName=File AST"})
        @Override
        public String getDisplayName() {
            return FileImplASTTrace_displayName();
        }

        @Override
        public void dumpInfo(Lookup context, PrintWriter printOut) {
            Collection<? extends CsmFile> allFiles = context.lookupAll(CsmFile.class);
            for (CsmFile csmFile : allFiles) {
                if(csmFile instanceof FileImpl) {
                    ASTFrameEx frame = new ASTFrameEx(csmFile.getName().toString(), ((FileImpl) csmFile).debugParse());
                    frame.setVisible(true);
                }
            }
        }
    }

    @ServiceProvider(service = CndDiagnosticProvider.class, position = 1500)
    public final static class ProjectDeclarationsTrace implements CndDiagnosticProvider {

        @NbBundle.Messages({"ProjectDeclarationsTrace.displayName=Project Declaration Containers (Huge size)"})
        @Override
        public String getDisplayName() {
            return ProjectReferencesTrace_displayName();
        }

        @Override
        public void dumpInfo(Lookup context, PrintWriter printOut) throws IOException{
            Collection<CsmProject> projects = new ArrayList<>(context.lookupAll(CsmProject.class));
            if (projects.isEmpty()) {
                CsmFile file = context.lookup(CsmFile.class);
                if (file != null) {
                    CsmProject project = file.getProject();
                    if (project instanceof ProjectBase) {
                        projects.add(project);
                    }
                }
            }
            PrintStream ps = CsmTracer.toPrintStream(printOut);
            for (CsmProject prj : projects) {
                if (prj instanceof ProjectBase) {
                    dumpProjectContainers(ps, prj, false);
                }
            }
        }
    }

    @ServiceProvider(service = CndDiagnosticProvider.class, position = 1600)
    public final static class ModelTrace implements CndDiagnosticProvider {

        @NbBundle.Messages({"ModelTrace.displayName=Project Code Model (Huge size)"})
        @Override
        public String getDisplayName() {
            return ModelTrace_displayName();
        }

        @Override
        public void dumpInfo(Lookup context, PrintWriter printOut) throws IOException {
            Collection<CsmProject> projects = new ArrayList<>(context.lookupAll(CsmProject.class));
            if (projects.isEmpty()) {
                CsmFile file = context.lookup(CsmFile.class);
                if (file != null) {
                    CsmProject project = file.getProject();
                    if (project != null) {
                        projects.add(project);
                    }
                }
            }
            CsmCacheManager.enter();
            try {
                for (CsmProject prj : projects) {
                    new CsmTracer(printOut).dumpModel(prj);
                }
            } finally {
                CsmCacheManager.leave();
            }
        }
    }

    @ServiceProvider(service = CndDiagnosticProvider.class, position = 1600)
    public final static class ProjectReferencesTrace implements CndDiagnosticProvider {

        @NbBundle.Messages({"ProjectReferencesTrace.displayName=Project References"})
        @Override
        public String getDisplayName() {
            return ProjectReferencesTrace_displayName();
        }

        @Override
        public void dumpInfo(Lookup context, PrintWriter printOut) {
            Collection<CsmProject> projects = new ArrayList<>(context.lookupAll(CsmProject.class));
            if (projects.isEmpty()) {
                CsmFile file = context.lookup(CsmFile.class);
                if (file != null) {
                    CsmProject project = file.getProject();
                    if (project != null) {
                        projects.add(project);
                    }
                }
            }
            printOut.println("References:"); // NOI18N
            for (CsmProject prj : projects) {
                printOut.print(prj.getName() + " : "); // NOI18N
                int refsNumber = 0;
                for (CsmFile file : prj.getAllFiles()) {
                    refsNumber += ((FileImpl)file).getReferences().size();
                }
                printOut.println(refsNumber);
            }
        }
    }

    @ServiceProvider(service = CndDiagnosticProvider.class, position = 1400)
    public final static class OffsetToPositionProvider implements CndDiagnosticProvider {

        @NbBundle.Messages({"OffsetToPositionProvider.displayName=Offset to Position"})
        @Override
        public String getDisplayName() {
            return OffsetToPositionProvider_displayName();
        }

        @Override
        public void dumpInfo(Lookup context, PrintWriter printOut) {
            Collection<? extends CsmFile> allFiles = context.lookupAll(CsmFile.class);
            for (CsmFile csmFile : allFiles) {
                if (csmFile instanceof FileImpl) {
                    OffsetToPositionFrame frame = new OffsetToPositionFrame((FileImpl) csmFile);
                    frame.setVisible(true);
                }
            }
        }
    }
}
