/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.tomcat5;

import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLEncoder;
import java.util.StringTokenizer;
import javax.enterprise.deploy.shared.ActionType;
import javax.enterprise.deploy.shared.CommandType;
import javax.enterprise.deploy.shared.StateType;
import javax.enterprise.deploy.spi.Target;
import javax.enterprise.deploy.spi.TargetModuleID;
import javax.enterprise.deploy.spi.exceptions.OperationUnsupportedException;
import javax.enterprise.deploy.spi.status.ClientConfiguration;
import javax.enterprise.deploy.spi.status.DeploymentStatus;
import javax.enterprise.deploy.spi.status.ProgressListener;
import javax.enterprise.deploy.spi.status.ProgressObject;
import org.netbeans.modules.tomcat5.config.gen.Context;
import org.netbeans.modules.tomcat5.config.gen.Engine;
import org.netbeans.modules.tomcat5.config.gen.Host;
import org.netbeans.modules.tomcat5.config.gen.SContext;
import org.netbeans.modules.tomcat5.config.gen.Server;
import org.netbeans.modules.tomcat5.config.gen.Service;
import org.netbeans.modules.tomcat5.progress.ProgressEventSupport;
import org.netbeans.modules.tomcat5.progress.Status;
import org.openide.util.RequestProcessor;
import org.openide.util.NbBundle;
import java.io.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.netbeans.modules.tomcat5.util.TomcatProperties;

/** Implemtation of management task that provides info about progress
 *
 * @author  Radim Kubacki
 */
public class TomcatManagerImpl implements ProgressObject, Runnable {
    
    /** RequestProcessor processor that serializes management tasks. */
    private static RequestProcessor rp;
    
    /** Returns shared RequestProcessor. */
    private static synchronized RequestProcessor rp () {
        if (rp == null) {
            rp = new RequestProcessor ("Tomcat management", 1); // NOI18N
        }
        return rp;
    }

    /** Support for progress notifications. */
    private ProgressEventSupport pes;
    
    /** Command that is executed on running server. */
    private String command;
    
    /** Output of executed command (parsed for list commands). */
    private String output;
    
    /** Command type used for events. */
    private CommandType cmdType;
    
    /** InputStream of application data. */
    private InputStream istream;
    
    private TomcatManager tm;
    
    /** Has been the last access to tomcat manager web app authorized? */
    private boolean authorized;
    
    /** TargetModuleID of module that is managed. */
    private TomcatModule tmId;
    
    private static final Logger LOGGER = Logger.getLogger(TomcatManagerImpl.class.getName());
    
    public TomcatManagerImpl (TomcatManager tm) {
        this.tm = tm;
        pes = new ProgressEventSupport (this);
    }

    public void deploy (Target t, InputStream is, InputStream deplPlan) {
        Context ctx;
        try {
            ctx = Context.createGraph(deplPlan);
        } catch (RuntimeException e) {
            String msg = NbBundle.getMessage(TomcatManagerImpl.class, "MSG_DeployBrokenContextXml");
            pes.fireHandleProgressEvent(null, new Status(ActionType.EXECUTE, cmdType, msg, StateType.FAILED));
            return;
        }
        String ctxPath = ctx.getAttributeValue ("path");   // NOI18N
        tmId = new TomcatModule (t, ctxPath);
        command = "deploy?path=" + encodePath(tmId.getPath()); // NOI18N
        cmdType = CommandType.DISTRIBUTE;
        String msg = NbBundle.getMessage(TomcatManagerImpl.class, "MSG_DeploymentInProgress");
        pes.fireHandleProgressEvent (null, new Status (ActionType.EXECUTE, cmdType, msg, StateType.RUNNING));
        istream = is;
        rp ().post (this, 0, Thread.NORM_PRIORITY);
    }
    
    /** Deploys WAR file or directory to Tomcat using deplPlan as source 
     * of conetx configuration data.
     */
    public void install (Target t, File wmfile, File deplPlan) {
        // WAR file
        String docBase = wmfile.toURI ().toASCIIString ();
        if (docBase.endsWith ("/")) { // NOI18N
            docBase = docBase.substring (0, docBase.length ()-1);
        }
        if (wmfile.isFile ()) {
            // WAR file
            docBase = "jar:"+docBase+"!/"; // NOI18N
        }
        // config or path
        String ctxPath = null;
        try {
            if (!deplPlan.exists ()) {
                if (wmfile.isDirectory ()) {
                    ctxPath = "/"+wmfile.getName ();    // NOI18N
                }
                else {
                    ctxPath = "/"+wmfile.getName ().substring (0, wmfile.getName ().lastIndexOf ('.'));    // NOI18N
                }
                tmId = new TomcatModule (t, ctxPath); // NOI18N
                command = "deploy?update=true&path="+encodePath(ctxPath)+"&war="+docBase; // NOI18N
            }
            else {
                FileInputStream in = new FileInputStream (deplPlan);
                Context ctx;
                try {
                    ctx = Context.createGraph(in);
                } catch (RuntimeException e) {
                    String msg = NbBundle.getMessage(TomcatManagerImpl.class, "MSG_DeployBrokenContextXml");
                    pes.fireHandleProgressEvent(null, new Status(ActionType.EXECUTE, cmdType, msg, StateType.FAILED));
                    return;
                }
                //PENDING #37763
//                tmId = new TomcatModule (t, ctx.getAttributeValue ("path")); // NOI18N
//                command = "install?update=true&config="+deplPlan.toURI ()+ // NOI18N
//                    "&war="+docBase; // NOI18N
                if (wmfile.isDirectory ()) {
                    ctxPath = "/"+wmfile.getName ();    // NOI18N
                }
                else {
                    ctxPath = "/"+wmfile.getName ().substring (0, wmfile.getName ().lastIndexOf ('.'));    // NOI18N
                }
                ctxPath = ctx.getAttributeValue ("path");
                tmId = new TomcatModule (t, ctxPath); // NOI18N
                command = "deploy?update=true&path="+encodePath(tmId.getPath())+"&war="+docBase; // NOI18N
            }
            
            // call the command
            cmdType = CommandType.DISTRIBUTE;
            String msg = NbBundle.getMessage(TomcatManagerImpl.class, "MSG_DeploymentInProgress");
            pes.fireHandleProgressEvent (null, new Status (ActionType.EXECUTE, cmdType, msg, StateType.RUNNING));
            
            rp ().post (this, 0, Thread.NORM_PRIORITY);
        }
        catch (java.io.FileNotFoundException fnfe) {
            pes.fireHandleProgressEvent (null, new Status (ActionType.EXECUTE, cmdType, fnfe.getLocalizedMessage (), StateType.FAILED));
        }
        
    }
    
    public void initialDeploy (Target t, File contextXml, File dir) {
        try {
            FileInputStream in = new FileInputStream (contextXml);
            Context ctx = Context.createGraph (in);
            String docBaseURI = dir.getAbsoluteFile().toURI().toASCIIString();
            String docBase = dir.getAbsolutePath ();
            String ctxPath = ctx.getAttributeValue ("path");
            this.tmId = new TomcatModule (t, ctxPath, docBase); //NOI18N
            String tmpContextXml = createTempContextXml(docBase, ctx);
            if (tm.isTomcat50()) {
                command = "deploy?config=" + tmpContextXml + "&war=" + docBaseURI; // NOI18N
            } else {
                command = "deploy?config=" + tmpContextXml + "&path=" + encodePath(tmId.getPath()); // NOI18N
            }
            cmdType = CommandType.DISTRIBUTE;
            String msg = NbBundle.getMessage(TomcatManagerImpl.class, "MSG_DeploymentInProgress");
            pes.fireHandleProgressEvent (null, new Status (ActionType.EXECUTE, cmdType, msg, StateType.RUNNING));
            rp ().post (this, 0, Thread.NORM_PRIORITY);
        } catch (java.io.IOException ioex) {
            pes.fireHandleProgressEvent (null, new Status (ActionType.EXECUTE, cmdType, ioex.getLocalizedMessage (), StateType.FAILED));
        } catch (RuntimeException e) {
            String msg = NbBundle.getMessage(TomcatManagerImpl.class, "MSG_DeployBrokenContextXml");
            pes.fireHandleProgressEvent(null, new Status(ActionType.EXECUTE, cmdType, msg, StateType.FAILED));
        }
    }

    public void remove(TomcatModule tmId) {
        // remove context from server.xml
        Server server = tm.getRoot();
        if (server != null && removeContextFromServer(server, tmId.getPath())) {
            File f = null;
            try {
                f = tm.getTomcatProperties().getServerXml();
                server.write(f);
            } catch (Exception e) {
                // cannot save changes
                pes.fireHandleProgressEvent(tmId, new Status (ActionType.EXECUTE, 
                        CommandType.UNDEPLOY, 
                        NbBundle.getMessage(TomcatManagerImpl.class, "MSG_ServerXml_RO", f.getAbsolutePath()),
                        StateType.FAILED));                
                return;
            }
        }
        this.tmId = tmId;
        command = "undeploy?path="+encodePath(tmId.getPath()); // NOI18N
        cmdType = CommandType.UNDEPLOY;
        String msg = NbBundle.getMessage(TomcatManagerImpl.class, "MSG_UndeploymentInProgress");
        pes.fireHandleProgressEvent (null, new Status (ActionType.EXECUTE, cmdType, msg, StateType.RUNNING));
        rp ().post (this, 0, Thread.NORM_PRIORITY);        
    }

    /**
     * Remove context with the specified path from the Server tree.
     * Look for the first appearance of the service and host element.
     * (ide currently does not support multiple service and host elements).
     */
    private boolean removeContextFromServer(Server server, String path) {
        // root web application is specified as an empty string
        if (path.equals("/")) path = ""; // NOI18N
        Service[] service = server.getService();
        if (service.length > 0) {
            Engine engine = service[0].getEngine();
            if (engine != null) {
                Host[] host = engine.getHost();
                if (host.length > 0) {                    
                    SContext[] sContext = host[0].getSContext();
                    for (int i = 0; i < sContext.length; i++) {
                        if (sContext[i].getAttributeValue("path").equals(path)) { // NOI18N
                            host[0].removeSContext(sContext[i]);
                            return true;
                        }                        
                    }
                }
            }
        }
        return false;
    }
    
    /** Starts web module. */
    public void start (TomcatModule tmId) {
        this.tmId = tmId;
        command = "start?path="+encodePath(tmId.getPath()); // NOI18N
        cmdType = CommandType.START;
        String msg = NbBundle.getMessage(TomcatManagerImpl.class, "MSG_StartInProgress");
        pes.fireHandleProgressEvent (null, new Status (ActionType.EXECUTE, cmdType, msg, StateType.RUNNING));
        rp ().post (this, 0, Thread.NORM_PRIORITY);
    }
    
    /** Stops web module. */
    public void stop (TomcatModule tmId) {
        this.tmId = tmId;
        command = "stop?path="+encodePath(tmId.getPath()); // NOI18N
        cmdType = CommandType.STOP;
        String msg = NbBundle.getMessage(TomcatManagerImpl.class, "MSG_StopInProgress");
        pes.fireHandleProgressEvent (null, new Status (ActionType.EXECUTE, cmdType, msg, StateType.RUNNING));
        rp ().post (this, 0, Thread.NORM_PRIORITY);
    }
    
    /** Reloads web module. */
    public void reload (TomcatModule tmId) {
        this.tmId = tmId;
        command = "reload?path="+encodePath(tmId.getPath()); // NOI18N
        cmdType = CommandType.REDEPLOY;
        String msg = NbBundle.getMessage(TomcatManagerImpl.class, "MSG_ReloadInProgress");
        pes.fireHandleProgressEvent (null, new Status (ActionType.EXECUTE, cmdType, msg, StateType.RUNNING));
        rp ().post (this, 0, Thread.NORM_PRIORITY);
    }
    
    public void incrementalRedeploy (TomcatModule tmId) {
        try {
            this.tmId = tmId;
            String docBase = tmId.getDocRoot ();
            assert docBase != null;
            String docBaseURI = new File (docBase).toURI().toASCIIString();
            File contextXml = new File (docBase + "/META-INF/context.xml"); // NO18N
            FileInputStream in = new FileInputStream (contextXml);
            Context ctx = Context.createGraph (in);
            String tmpContextXml = createTempContextXml(docBase, ctx);
            if (tm.isTomcat50()) {
                command = "deploy?config=" + tmpContextXml + "&war=" + docBaseURI; // NOI18N
            } else {
                command = "deploy?config=" + tmpContextXml + "&path=" + encodePath(tmId.getPath()); // NOI18N
            }
            cmdType = CommandType.DISTRIBUTE;
            String msg = NbBundle.getMessage(TomcatManagerImpl.class, "MSG_DeployInProgress");
            pes.fireHandleProgressEvent (null, new Status (ActionType.EXECUTE, cmdType, msg, StateType.RUNNING));
            rp ().post (this, 0, Thread.NORM_PRIORITY);
        } catch (java.io.IOException ioex) {
            pes.fireHandleProgressEvent (null, new Status (ActionType.EXECUTE, cmdType, ioex.getLocalizedMessage (), StateType.FAILED));
        } catch (RuntimeException e) {
            String msg = NbBundle.getMessage(TomcatManagerImpl.class, "MSG_DeployBrokenContextXml");
            pes.fireHandleProgressEvent(null, new Status(ActionType.EXECUTE, cmdType, msg, StateType.FAILED));
            return;
        }
    }
    
    /**
     * Translates a context path string into <code>application/x-www-form-urlencoded</code> format.
     */
    private static String encodePath(String str) {
        try {
            StringTokenizer st = new StringTokenizer(str, "/"); // NOI18N
            if (!st.hasMoreTokens()) {
                return str;
            }
            StringBuilder result = new StringBuilder();
            while (st.hasMoreTokens()) {
                result.append("/").append(URLEncoder.encode(st.nextToken(), "UTF-8")); // NOI18N
            }
            return result.toString();
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e); // this should never happen
        }
    }
    
    /**
     * Create a temporary copy of context.xml and set a docBase attribute 
     * in it. This does not modify the existing context.xml.
     *
     * @return properly escaped URL (<code>application/x-www-form-urlencoded</code>) in string form
     */
    private String createTempContextXml(String docBase, Context ctx) throws IOException {
        File tmpContextXml = File.createTempFile("context", ".xml"); // NOI18N
        tmpContextXml.deleteOnExit();
        if (!docBase.equals (ctx.getAttributeValue ("docBase"))) { //NOI18N
            ctx.setAttributeValue ("docBase", docBase); //NOI18N
            FileOutputStream fos = new FileOutputStream (tmpContextXml);
            ctx.write (fos);
            fos.close ();
        }
        // http://www.netbeans.org/issues/show_bug.cgi?id=167139
        URL url = tmpContextXml.toURI().toURL();
        String ret = URLEncoder.encode(url.toString(), "UTF-8"); // NOI18N
        return ret;
    }
    
    /** Lists web modules.
     * This method runs synchronously.
     * @param target server target
     * @param state one of ENUM_ constants.
     *
     * @throws IllegalStateException when access to tomcat manager has not been
     * authorized and therefore list of target modules could not been retrieved
     */
    TargetModuleID[] list (Target t, int state) throws IllegalStateException {
        command = "list"; // NOI18N
        run ();
        if (!authorized) {
            // connection to tomcat manager has not been authorized
            String errMsg = NbBundle.getMessage(TomcatManagerImpl.class, "MSG_AuthorizationFailed",
                    tm.isTomcat70() ? "manager-script" : "manager");
            IllegalStateException ise = new IllegalStateException(errMsg);
            throw (IllegalStateException)ise.initCause(new AuthorizationException());
        }
        // PENDING : error check
        java.util.List modules = new java.util.ArrayList ();
        boolean first = true;
        StringTokenizer stok = new StringTokenizer (output, "\r\n");    // NOI18N
        while (stok.hasMoreTokens ()) {
            String line = stok.nextToken ();
            if (first) {
                first = false;
            }
            else {
                StringTokenizer ltok = new StringTokenizer (line, ":"); // NOI18N
                try {
                    String ctx = ltok.nextToken ();
                    String s = ltok.nextToken ();
                    String tag = ltok.nextToken ();
                    String path = null;
                    //take the rest of line as path (it can contain ':')
                    // #50410 - path information is missing in the Japanese localization of Tomcat Manager
                    if (ltok.hasMoreTokens()) {
                        path = line.substring (ctx.length () + s.length () + tag.length () + 3);
                    }
                    if ("running".equals (s)
                    &&  (state == TomcatManager.ENUM_AVAILABLE || state == TomcatManager.ENUM_RUNNING)) {
                        modules.add (new TomcatModule (t, ctx, path));
                    }
                    if ("stopped".equals (s)
                    &&  (state == TomcatManager.ENUM_AVAILABLE || state == TomcatManager.ENUM_NONRUNNING)) {
                        modules.add (new TomcatModule (t, ctx, path));
                    }
                } catch (java.util.NoSuchElementException e) {
                    // invalid value
                    LOGGER.log(Level.FINE, line, e);
                    System.err.println(line);
                    e.printStackTrace();
                }
            }
        }
        return (TargetModuleID []) modules.toArray (new TargetModuleID[modules.size ()]);
    }
    
    /** Queries Tomcat server to get JMX beans containing management information
     * @param param encoded parameter(s) for query
     * @return server output
     */
    public String jmxProxy (String query) {
        command = "jmxproxy/"+query; // NOI18N
        run ();
        // PENDING : error check
        return output;
    }
    
    /** JSR88 method. */
    public ClientConfiguration getClientConfiguration (TargetModuleID targetModuleID) {
        return null; // PENDING
    }
    
    /** JSR88 method. */
    public DeploymentStatus getDeploymentStatus () {
        return pes.getDeploymentStatus ();
    }
    
    /** JSR88 method. */
    public TargetModuleID[] getResultTargetModuleIDs () {
        return new TargetModuleID [] { tmId };
    }
    
    /** JSR88 method. */
    public boolean isCancelSupported () {
        return false;
    }
    
    /** JSR88 method. */
    public void cancel () 
    throws OperationUnsupportedException {
        throw new OperationUnsupportedException ("cancel not supported in Tomcat deployment"); // NOI18N
    }
    
    /** JSR88 method. */
    public boolean isStopSupported () {
        return false;
    }
    
    /** JSR88 method. */
    public void stop () throws OperationUnsupportedException {
        throw new OperationUnsupportedException ("stop not supported in Tomcat deployment"); // NOI18N
    }
    
    /** JSR88 method. */
    public void addProgressListener (ProgressListener l) {
        pes.addProgressListener (l);
    }
    
    /** JSR88 method. */
    public void removeProgressListener (ProgressListener l) {
        pes.removeProgressListener (l);
    }
    
    /** Executes one management task. */
    public synchronized void run () {
        LOGGER.log(Level.FINE, command);
        pes.fireHandleProgressEvent (tmId, new Status (ActionType.EXECUTE, cmdType, command /* message */, StateType.RUNNING));
        
        output = "";
        authorized = true;
        
        int retries = 4;
        
        // similar to Tomcat's Ant task
        URLConnection conn = null;
        InputStreamReader reader = null;
        
        URL urlToConnectTo = null;
        
        boolean failed = false;
        String msg = null;
        while (retries >= 0) {
            retries = retries - 1;
            try {

                // Create a connection for this command
                String uri = tm.getPlainUri ();
                String withoutSpaces = (uri + command).replaceAll(" ", "%20");  //NOI18N
                urlToConnectTo = new URL(withoutSpaces);
                
                if (Boolean.getBoolean("org.netbeans.modules.tomcat5.LogManagerCommands")) { // NOI18N
                    String message = "Tomcat 5 sending manager command: " + urlToConnectTo;
                    Logger.getLogger(TomcatManagerImpl.class.getName()).log(Level.FINE, null, new Exception(message));
                }

                conn = urlToConnectTo.openConnection();
                HttpURLConnection hconn = (HttpURLConnection) conn;

                // Set up standard connection characteristics
                hconn.setAllowUserInteraction(false);
                hconn.setDoInput(true);
                hconn.setUseCaches(false);
                if (istream != null) {
                    hconn.setDoOutput(true);
                    hconn.setRequestMethod("PUT");   // NOI18N
                    hconn.setRequestProperty("Content-Type", "application/octet-stream");   // NOI18N
                } else {
                    hconn.setDoOutput(false);
                    hconn.setRequestMethod("GET"); // NOI18N
                }
                hconn.setRequestProperty("User-Agent", // NOI18N
                                         "NetBeansIDE-Tomcat-Manager/1.0"); // NOI18N
                // Set up an authorization header with our credentials
                TomcatProperties tp = tm.getTomcatProperties();
                String input = tp.getUsername () + ":" + tp.getPassword ();
                String auth = new String(Base64.encode(input.getBytes()));
                hconn.setRequestProperty("Authorization", // NOI18N
                                         "Basic " + auth); // NOI18N

                // Establish the connection with the server
                hconn.connect();
                int respCode = hconn.getResponseCode();
                if (respCode == HttpURLConnection.HTTP_UNAUTHORIZED 
                    || respCode == HttpURLConnection.HTTP_FORBIDDEN) {
                    // connection to tomcat manager has not been allowed
                    authorized = false;
                    String errMsg = NbBundle.getMessage(TomcatManagerImpl.class, "MSG_AuthorizationFailed");
                    pes.fireHandleProgressEvent (null, new Status (ActionType.EXECUTE, cmdType, errMsg, StateType.FAILED));
                    return;
                }
                if (Boolean.getBoolean("org.netbeans.modules.tomcat5.LogManagerCommands")) { // NOI18N
                    int code = hconn.getResponseCode();
                    String message = "Tomcat 5 receiving response, code: " + code;
                    System.out.println(message);
                    Logger.getLogger(TomcatManagerImpl.class.getName()).log(Level.INFO, null, new Exception(message));
                }
                // Send the request data (if any)
                if (istream != null) {
                    BufferedOutputStream ostream =
                        new BufferedOutputStream(hconn.getOutputStream(), 1024);
                    byte buffer[] = new byte[1024];
                    while (true) {
                        int n = istream.read(buffer);
                        if (n < 0) {
                            break;
                        }
                        ostream.write(buffer, 0, n);
                    }
                    ostream.flush();
                    ostream.close();
                    istream.close();
                }

                // Process the response message
                reader = new InputStreamReader(hconn.getInputStream(),"UTF-8"); //NOI18N
                retries = -1;
                StringBuffer buff = new StringBuffer();
                String error = null;
                msg = null;
                boolean first = !command.startsWith ("jmxproxy");   // NOI18N
                while (true) {
                    int ch = reader.read();
                    if (ch < 0) {
                        output += buff.toString ()+"\n";    // NOI18N
                        break;
                    } else if ((ch == '\r') || (ch == '\n')) {
                        String line = buff.toString();
                        buff.setLength(0);
                        LOGGER.log(Level.FINE, line);
                        if (first) {
                            // hard fix to accept the japanese localization of manager app
                            String japaneseOK="\u6210\u529f"; //NOI18N
                            msg = line;
                            // see issue #62529
                            if (line.indexOf("java.lang.ThreadDeath") != -1) { // NOI18N
                                String warning = NbBundle.getMessage(TomcatManagerImpl.class, "MSG_ThreadDeathWarning");
                                pes.fireHandleProgressEvent(
                                    tmId, 
                                    new Status(ActionType.EXECUTE, cmdType, warning, StateType.RUNNING)
                                );
                            } else if (!(line.startsWith("OK -") || line.startsWith(japaneseOK))) { // NOI18N
                                error = line;
                            }
                            first = false;
                        }
                        output += line+"\n";    // NOI18N
                    } else {
                        buff.append((char) ch);
                    }
                }
                if (buff.length() > 0) {
                    LOGGER.log(Level.FINE, buff.toString());
                }
                if (error != null) {
                    LOGGER.log (Level.INFO, "TomcatManagerImpl connecting to: " + urlToConnectTo, error); // NOI18N
                    pes.fireHandleProgressEvent (tmId, new Status (ActionType.EXECUTE, cmdType, error, StateType.FAILED));
                    failed = true;
                }
                if (msg == null) {
                    msg = buff.toString();
                }
            } catch (Exception e) {
                if (retries < 0) {
                    LOGGER.log(Level.INFO, "TomcatManagerImpl connecting to: " + urlToConnectTo, e); // NOI18N
                    pes.fireHandleProgressEvent (tmId, new Status (ActionType.EXECUTE, cmdType, e.getLocalizedMessage (), StateType.FAILED));
                    failed = true;
                }
                // throw t;
            } finally {
                if (reader != null) {
                    try {
                        reader.close();
                    } catch (java.io.IOException ioe) { // ignore this
                    }
                    reader = null;
                }
                if (istream != null) {
                    try {
                        istream.close();
                    } catch (java.io.IOException ioe) { // ignore this
                    }
                    istream = null;
                }                
            }
            if (retries >=0) {
                try {
                    Thread.sleep(3000);
                } catch (InterruptedException e) {}
            }
        } // while
        if (!failed) {
            pes.fireHandleProgressEvent (tmId, new Status (ActionType.EXECUTE, cmdType, msg, StateType.COMPLETED));
        }
    }
}
