/*
*   Copyright (C) 2008 Nicola Gigante <nicola.gigante@gmail.com>
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License as published by
*   the Free Software Foundation; either version 2.1 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
*/

#ifndef HELPER_SUPPORT_H
#define HELPER_SUPPORT_H

#include <QObject>
#include <QVariant>

#include <kauthcore_export.h>

#define KAUTH_HELPER_MAIN(ID, HelperClass) \
    int main(int argc, char **argv) { return KAuth::HelperSupport::helperMain(argc, argv, ID, new HelperClass()); }

namespace KAuth
{

/**
 * @brief Support class with some KAUTHCORE_EXPORT methods useful to the helper's code
 *
 * This class provides the API to write the helper tool that executes your actions.
 * You don't create instances of HelperSupport. Instead, you use its KAUTHCORE_EXPORT methods.
 *
 * This them you can notify the application of progress in your action's execution
 * and you can check if the application asked you to terminate it.
 *
 * @since 4.4
 */
namespace HelperSupport
{
/**
 * @brief Send a progressStep signal to the caller application
 *
 * You can use this method to notify progress information about the
 * action execution. When you call this method, the KAuth::ExecuteJob
 * object associated with the current action will emit the KJob::percent(KJob*, unsigned long)
 * signal. The meaning of the integer passed here is totally application dependent,
 * but you'll want to use it as a sort of percentage.
 * If you need to be more expressive, use the other overload which takes a QVariantMap
 *
 * @param step The progress indicator
 */
KAUTHCORE_EXPORT void progressStep(int step);

/**
* @brief Send a progressStep signal to the caller application
*
* You can use this method to notify progress information about the
* action execution. When you call this method, the KAuth::ExecuteJob
* object associated with the current action will emit the progressStep(QVariantMap)
* signal. The meaning of the data passed here is totally application dependent.
*
* If you only need a simple percentage value, use the other overload which takes an int.
*
* @param data The progress data
*/
KAUTHCORE_EXPORT void progressStep(const QVariantMap &data);

/**
 * @brief Check if the caller asked the helper to stop the execution
 *
 * This method will return true if the helper has been asked to stop the
 * execution of the current action. If this happens, your helper should
 * return (NOT exit). The meaning of the data you return in this case is
 * application-dependent.
 *
 * It's good practice to check it regularly if you have a long-running action
 *
 * @see ExecuteJob::kill
 * @return true if the helper has been asked to stop, false otherwise
 */
KAUTHCORE_EXPORT bool isStopped();

/**
 * @brief Method that implements the main function of the helper tool. Do not call directly
 *
 * This method is called in the proper way by the code generated by the KAUTH_HELPER_MAIN(),
 * which creates a main() function for the helper tool.
 * macro. You shouldn't call this method directly.
 *
 * @param argc The argc parameter from the main() function.
 * @param argv The argv parameter from the main() function.
 * @param id The helper ID as passed to the macro
 * @param responder The responder object for the helper. The macro passes a default-constructed,
 *                  heap-allocated object of the class specified as the last macro parameter
 */
KAUTHCORE_EXPORT int helperMain(int argc, char **argv, const char *id, QObject *responder);
} // namespace HelperSupport

} // namespace Auth

#endif
