      DOUBLE PRECISION FUNCTION slRCC (TDB, UT1, WL, U, V)
*+
*     - - - -
*      R C C
*     - - - -
*
*  Relativistic clock correction:  the difference between proper time at
*  a point on the surface of the Earth and coordinate time in the Solar
*  System barycentric space-time frame of reference.
*
*  The proper time is terrestrial time, TT;  the coordinate time is an
*  implementation of barycentric dynamical time, TDB.
*
*  Given:
*    TDB      d     TDB (MJD: JD-2400000.5)
*    UT1      d     universal time (fraction of one day)
*    WL       d     clock longitude (radians west)
*    U        d     clock distance from Earth spin axis (km)
*    V        d     clock distance north of Earth equatorial plane (km)
*
*  Returned:
*    The clock correction, TDB-TT, in seconds:
*
*    .  TDB is coordinate time in the solar system barycentre frame
*       of reference, in units chosen to eliminate the scale difference
*       with respect to terrestrial time.
*
*    .  TT is the proper time for clocks at mean sea level on the
*       Earth.
*
*  Notes:
*
*  1  The argument TDB is, strictly, the barycentric coordinate time;
*     however, the terrestrial time TT can in practice be used without
*     any significant loss of accuracy.
*
*  2  The result returned by slRCC comprises a main (annual)
*     sinusoidal term of amplitude approximately 0.00166 seconds, plus
*     planetary and lunar terms up to about 20 microseconds, and diurnal
*     terms up to 2 microseconds.  The variation arises from the
*     transverse Doppler effect and the gravitational red-shift as the
*     observer varies in speed and moves through different gravitational
*     potentials.
*
*  3  The geocentric model is that of Fairhead & Bretagnon (1990), in
*     its full form.  It was supplied by Fairhead (private
*     communication) as a FORTRAN subroutine.  The original Fairhead
*     routine used explicit formulae, in such large numbers that
*     problems were experienced with certain compilers (Microsoft
*     Fortran on PC aborted with stack overflow, Convex compiled
*     successfully but extremely slowly).  The present implementation is
*     a complete recoding, with the original Fairhead coefficients held
*     in a table.  To optimise arithmetic precision, the terms are
*     accumulated in reverse order, smallest first.  A number of other
*     coding changes were made, in order to match the calling sequence
*     of previous versions of the present routine, and to comply with
*     Starlink programming standards.  The numerical results compared
*     with those from the Fairhead form are essentially unaffected by
*     the changes, the differences being at the 10^-20 sec level.
*
*  4  The topocentric part of the model is from Moyer (1981) and
*     Murray (1983).  It is an approximation to the expression
*     ( v / c ) . ( r / c ), where v is the barycentric velocity of
*     the Earth, r is the geocentric position of the observer and
*     c is the speed of light.
*
*  5  During the interval 1950-2050, the absolute accuracy of is better
*     than +/- 3 nanoseconds relative to direct numerical integrations
*     using the JPL DE200/LE200 solar system ephemeris.
*
*  6  The IAU definition of TDB was that it must differ from TT only by
*     periodic terms.  Though practical, this is an imprecise definition
*     which ignores the existence of very long-period and secular
*     effects in the dynamics of the solar system.  As a consequence,
*     different implementations of TDB will, in general, differ in zero-
*     point and will drift linearly relative to one other.
*
*  7  TDB was, in principle, superseded by new coordinate timescales
*     which the IAU introduced in 1991:  geocentric coordinate time,
*     TCG, and barycentric coordinate time, TCB.  However, slRCC
*     can be used to implement the periodic part of TCB-TCG.
*
*  References:
*
*  1  Fairhead, L., & Bretagnon, P., Astron.Astrophys., 229, 240-247
*     (1990).
*
*  2  Moyer, T.D., Cel.Mech., 23, 33 (1981).
*
*  3  Murray, C.A., Vectorial Astrometry, Adam Hilger (1983).
*
*  4  Seidelmann, P.K. et al, Explanatory Supplement to the
*     Astronomical Almanac, Chapter 2, University Science Books
*     (1992).
*
*  5  Simon J.L., Bretagnon P., Chapront J., Chapront-Touze M.,
*     Francou G. & Laskar J., Astron.Astrophys., 282, 663-683 (1994).
*
*  P.T.Wallace   Starlink   7 May 2000
*
*  Copyright (C) 2000 Rutherford Appleton Laboratory
*
*  License:
*    This program is free software; you can redistribute it and/or modify
*    it under the terms of the GNU General Public License as published by
*    the Free Software Foundation; either version 2 of the License, or
*    (at your option) any later version.
*
*    This program is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*    GNU General Public License for more details.
*
*    You should have received a copy of the GNU General Public License
*    along with this program (see SLA_CONDITIONS); if not, write to the
*    Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
*    Boston, MA  02110-1301  USA
*
*  Copyright (C) 1995 Association of Universities for Research in Astronomy Inc.
*-

      IMPLICIT NONE

      DOUBLE PRECISION TDB,UT1,WL,U,V

      DOUBLE PRECISION D2PI,D2R
      PARAMETER (D2PI=6.283185307179586476925287D0,
     :           D2R=0.0174532925199432957692369D0)

      DOUBLE PRECISION T,TSOL,W,ELSUN,EMSUN,D,ELJ,ELS,
     :                 WT,W0,W1,W2,W3,W4,WF,WJ

* -----------------------------------------------------------------------
*
*  Fairhead and Bretagnon canonical coefficients
*
*  787 sets of three coefficients.
*
*  Each set is amplitude (microseconds)
*              frequency (radians per Julian millennium since J2000),
*              phase (radians).
*
*  Sets   1-474 are the T**0 terms,
*   "   475-679  "   "  T**1   "
*   "   680-764  "   "  T**2   "
*   "   765-784  "   "  T**3   "
*   "   785-787  "   "  T**4   "  .
*
      DOUBLE PRECISION FAIRHD(3,787)
      INTEGER I,J
      DATA ((FAIRHD(I,J),I=1,3),J=  1, 10) /
     : 1656.674564D-6,    6283.075849991D0, 6.240054195D0,
     :   22.417471D-6,    5753.384884897D0, 4.296977442D0,
     :   13.839792D-6,   12566.151699983D0, 6.196904410D0,
     :    4.770086D-6,     529.690965095D0, 0.444401603D0,
     :    4.676740D-6,    6069.776754553D0, 4.021195093D0,
     :    2.256707D-6,     213.299095438D0, 5.543113262D0,
     :    1.694205D-6,      -3.523118349D0, 5.025132748D0,
     :    1.554905D-6,   77713.771467920D0, 5.198467090D0,
     :    1.276839D-6,    7860.419392439D0, 5.988822341D0,
     :    1.193379D-6,    5223.693919802D0, 3.649823730D0 /
      DATA ((FAIRHD(I,J),I=1,3),J= 11, 20) /
     :    1.115322D-6,    3930.209696220D0, 1.422745069D0,
     :    0.794185D-6,   11506.769769794D0, 2.322313077D0,
     :    0.447061D-6,      26.298319800D0, 3.615796498D0,
     :    0.435206D-6,    -398.149003408D0, 4.349338347D0,
     :    0.600309D-6,    1577.343542448D0, 2.678271909D0,
     :    0.496817D-6,    6208.294251424D0, 5.696701824D0,
     :    0.486306D-6,    5884.926846583D0, 0.520007179D0,
     :    0.432392D-6,      74.781598567D0, 2.435898309D0,
     :    0.468597D-6,    6244.942814354D0, 5.866398759D0,
     :    0.375510D-6,    5507.553238667D0, 4.103476804D0 /
      DATA ((FAIRHD(I,J),I=1,3),J= 21, 30) /
     :    0.243085D-6,    -775.522611324D0, 3.651837925D0,
     :    0.173435D-6,   18849.227549974D0, 6.153743485D0,
     :    0.230685D-6,    5856.477659115D0, 4.773852582D0,
     :    0.203747D-6,   12036.460734888D0, 4.333987818D0,
     :    0.143935D-6,    -796.298006816D0, 5.957517795D0,
     :    0.159080D-6,   10977.078804699D0, 1.890075226D0,
     :    0.119979D-6,      38.133035638D0, 4.551585768D0,
     :    0.118971D-6,    5486.777843175D0, 1.914547226D0,
     :    0.116120D-6,    1059.381930189D0, 0.873504123D0,
     :    0.137927D-6,   11790.629088659D0, 1.135934669D0 /
      DATA ((FAIRHD(I,J),I=1,3),J= 31, 40) /
     :    0.098358D-6,    2544.314419883D0, 0.092793886D0,
     :    0.101868D-6,   -5573.142801634D0, 5.984503847D0,
     :    0.080164D-6,     206.185548437D0, 2.095377709D0,
     :    0.079645D-6,    4694.002954708D0, 2.949233637D0,
     :    0.062617D-6,      20.775395492D0, 2.654394814D0,
     :    0.075019D-6,    2942.463423292D0, 4.980931759D0,
     :    0.064397D-6,    5746.271337896D0, 1.280308748D0,
     :    0.063814D-6,    5760.498431898D0, 4.167901731D0,
     :    0.048042D-6,    2146.165416475D0, 1.495846011D0,
     :    0.048373D-6,     155.420399434D0, 2.251573730D0 /
      DATA ((FAIRHD(I,J),I=1,3),J= 41, 50) /
     :    0.058844D-6,     426.598190876D0, 4.839650148D0,
     :    0.046551D-6,      -0.980321068D0, 0.921573539D0,
     :    0.054139D-6,   17260.154654690D0, 3.411091093D0,
     :    0.042411D-6,    6275.962302991D0, 2.869567043D0,
     :    0.040184D-6,      -7.113547001D0, 3.565975565D0,
     :    0.036564D-6,    5088.628839767D0, 3.324679049D0,
     :    0.040759D-6,   12352.852604545D0, 3.981496998D0,
     :    0.036507D-6,     801.820931124D0, 6.248866009D0,
     :    0.036955D-6,    3154.687084896D0, 5.071801441D0,
     :    0.042732D-6,     632.783739313D0, 5.720622217D0 /
      DATA ((FAIRHD(I,J),I=1,3),J= 51, 60) /
     :    0.042560D-6,  161000.685737473D0, 1.270837679D0,
     :    0.040480D-6,   15720.838784878D0, 2.546610123D0,
     :    0.028244D-6,   -6286.598968340D0, 5.069663519D0,
     :    0.033477D-6,    6062.663207553D0, 4.144987272D0,
     :    0.034867D-6,     522.577418094D0, 5.210064075D0,
     :    0.032438D-6,    6076.890301554D0, 0.749317412D0,
     :    0.030215D-6,    7084.896781115D0, 3.389610345D0,
     :    0.029247D-6,  -71430.695617928D0, 4.183178762D0,
     :    0.033529D-6,    9437.762934887D0, 2.404714239D0,
     :    0.032423D-6,    8827.390269875D0, 5.541473556D0 /
      DATA ((FAIRHD(I,J),I=1,3),J= 61, 70) /
     :    0.027567D-6,    6279.552731642D0, 5.040846034D0,
     :    0.029862D-6,   12139.553509107D0, 1.770181024D0,
     :    0.022509D-6,   10447.387839604D0, 1.460726241D0,
     :    0.020937D-6,    8429.241266467D0, 0.652303414D0,
     :    0.020322D-6,     419.484643875D0, 3.735430632D0,
     :    0.024816D-6,   -1194.447010225D0, 1.087136918D0,
     :    0.025196D-6,    1748.016413067D0, 2.901883301D0,
     :    0.021691D-6,   14143.495242431D0, 5.952658009D0,
     :    0.017673D-6,    6812.766815086D0, 3.186129845D0,
     :    0.022567D-6,    6133.512652857D0, 3.307984806D0 /
      DATA ((FAIRHD(I,J),I=1,3),J= 71, 80) /
     :    0.016155D-6,   10213.285546211D0, 1.331103168D0,
     :    0.014751D-6,    1349.867409659D0, 4.308933301D0,
     :    0.015949D-6,    -220.412642439D0, 4.005298270D0,
     :    0.015974D-6,   -2352.866153772D0, 6.145309371D0,
     :    0.014223D-6,   17789.845619785D0, 2.104551349D0,
     :    0.017806D-6,      73.297125859D0, 3.475975097D0,
     :    0.013671D-6,    -536.804512095D0, 5.971672571D0,
     :    0.011942D-6,    8031.092263058D0, 2.053414715D0,
     :    0.014318D-6,   16730.463689596D0, 3.016058075D0,
     :    0.012462D-6,     103.092774219D0, 1.737438797D0 /
      DATA ((FAIRHD(I,J),I=1,3),J= 81, 90) /
     :    0.010962D-6,       3.590428652D0, 2.196567739D0,
     :    0.015078D-6,   19651.048481098D0, 3.969480770D0,
     :    0.010396D-6,     951.718406251D0, 5.717799605D0,
     :    0.011707D-6,   -4705.732307544D0, 2.654125618D0,
     :    0.010453D-6,    5863.591206116D0, 1.913704550D0,
     :    0.012420D-6,    4690.479836359D0, 4.734090399D0,
     :    0.011847D-6,    5643.178563677D0, 5.489005403D0,
     :    0.008610D-6,    3340.612426700D0, 3.661698944D0,
     :    0.011622D-6,    5120.601145584D0, 4.863931876D0,
     :    0.010825D-6,     553.569402842D0, 0.842715011D0 /
      DATA ((FAIRHD(I,J),I=1,3),J= 91,100) /
     :    0.008666D-6,    -135.065080035D0, 3.293406547D0,
     :    0.009963D-6,     149.563197135D0, 4.870690598D0,
     :    0.009858D-6,    6309.374169791D0, 1.061816410D0,
     :    0.007959D-6,     316.391869657D0, 2.465042647D0,
     :    0.010099D-6,     283.859318865D0, 1.942176992D0,
     :    0.007147D-6,    -242.728603974D0, 3.661486981D0,
     :    0.007505D-6,    5230.807466803D0, 4.920937029D0,
     :    0.008323D-6,   11769.853693166D0, 1.229392026D0,
     :    0.007490D-6,   -6256.777530192D0, 3.658444681D0,
     :    0.009370D-6,  149854.400134205D0, 0.673880395D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=101,110) /
     :    0.007117D-6,      38.027672636D0, 5.294249518D0,
     :    0.007857D-6,   12168.002696575D0, 0.525733528D0,
     :    0.007019D-6,    6206.809778716D0, 0.837688810D0,
     :    0.006056D-6,     955.599741609D0, 4.194535082D0,
     :    0.008107D-6,   13367.972631107D0, 3.793235253D0,
     :    0.006731D-6,    5650.292110678D0, 5.639906583D0,
     :    0.007332D-6,      36.648562930D0, 0.114858677D0,
     :    0.006366D-6,    4164.311989613D0, 2.262081818D0,
     :    0.006858D-6,    5216.580372801D0, 0.642063318D0,
     :    0.006919D-6,    6681.224853400D0, 6.018501522D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=111,120) /
     :    0.006826D-6,    7632.943259650D0, 3.458654112D0,
     :    0.005308D-6,   -1592.596013633D0, 2.500382359D0,
     :    0.005096D-6,   11371.704689758D0, 2.547107806D0,
     :    0.004841D-6,    5333.900241022D0, 0.437078094D0,
     :    0.005582D-6,    5966.683980335D0, 2.246174308D0,
     :    0.006304D-6,   11926.254413669D0, 2.512929171D0,
     :    0.006603D-6,   23581.258177318D0, 5.393136889D0,
     :    0.005123D-6,      -1.484472708D0, 2.999641028D0,
     :    0.004648D-6,    1589.072895284D0, 1.275847090D0,
     :    0.005119D-6,    6438.496249426D0, 1.486539246D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=121,130) /
     :    0.004521D-6,    4292.330832950D0, 6.140635794D0,
     :    0.005680D-6,   23013.539539587D0, 4.557814849D0,
     :    0.005488D-6,      -3.455808046D0, 0.090675389D0,
     :    0.004193D-6,    7234.794256242D0, 4.869091389D0,
     :    0.003742D-6,    7238.675591600D0, 4.691976180D0,
     :    0.004148D-6,    -110.206321219D0, 3.016173439D0,
     :    0.004553D-6,   11499.656222793D0, 5.554998314D0,
     :    0.004892D-6,    5436.993015240D0, 1.475415597D0,
     :    0.004044D-6,    4732.030627343D0, 1.398784824D0,
     :    0.004164D-6,   12491.370101415D0, 5.650931916D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=131,140) /
     :    0.004349D-6,   11513.883316794D0, 2.181745369D0,
     :    0.003919D-6,   12528.018664345D0, 5.823319737D0,
     :    0.003129D-6,    6836.645252834D0, 0.003844094D0,
     :    0.004080D-6,   -7058.598461315D0, 3.690360123D0,
     :    0.003270D-6,      76.266071276D0, 1.517189902D0,
     :    0.002954D-6,    6283.143160294D0, 4.447203799D0,
     :    0.002872D-6,      28.449187468D0, 1.158692983D0,
     :    0.002881D-6,     735.876513532D0, 0.349250250D0,
     :    0.003279D-6,    5849.364112115D0, 4.893384368D0,
     :    0.003625D-6,    6209.778724132D0, 1.473760578D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=141,150) /
     :    0.003074D-6,     949.175608970D0, 5.185878737D0,
     :    0.002775D-6,    9917.696874510D0, 1.030026325D0,
     :    0.002646D-6,   10973.555686350D0, 3.918259169D0,
     :    0.002575D-6,   25132.303399966D0, 6.109659023D0,
     :    0.003500D-6,     263.083923373D0, 1.892100742D0,
     :    0.002740D-6,   18319.536584880D0, 4.320519510D0,
     :    0.002464D-6,     202.253395174D0, 4.698203059D0,
     :    0.002409D-6,       2.542797281D0, 5.325009315D0,
     :    0.003354D-6,  -90955.551694697D0, 1.942656623D0,
     :    0.002296D-6,    6496.374945429D0, 5.061810696D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=151,160) /
     :    0.003002D-6,    6172.869528772D0, 2.797822767D0,
     :    0.003202D-6,   27511.467873537D0, 0.531673101D0,
     :    0.002954D-6,   -6283.008539689D0, 4.533471191D0,
     :    0.002353D-6,     639.897286314D0, 3.734548088D0,
     :    0.002401D-6,   16200.772724501D0, 2.605547070D0,
     :    0.003053D-6,  233141.314403759D0, 3.029030662D0,
     :    0.003024D-6,   83286.914269554D0, 2.355556099D0,
     :    0.002863D-6,   17298.182327326D0, 5.240963796D0,
     :    0.002103D-6,   -7079.373856808D0, 5.756641637D0,
     :    0.002303D-6,   83996.847317911D0, 2.013686814D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=161,170) /
     :    0.002303D-6,   18073.704938650D0, 1.089100410D0,
     :    0.002381D-6,      63.735898303D0, 0.759188178D0,
     :    0.002493D-6,    6386.168624210D0, 0.645026535D0,
     :    0.002366D-6,       3.932153263D0, 6.215885448D0,
     :    0.002169D-6,   11015.106477335D0, 4.845297676D0,
     :    0.002397D-6,    6243.458341645D0, 3.809290043D0,
     :    0.002183D-6,    1162.474704408D0, 6.179611691D0,
     :    0.002353D-6,    6246.427287062D0, 4.781719760D0,
     :    0.002199D-6,    -245.831646229D0, 5.956152284D0,
     :    0.001729D-6,    3894.181829542D0, 1.264976635D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=171,180) /
     :    0.001896D-6,   -3128.388765096D0, 4.914231596D0,
     :    0.002085D-6,      35.164090221D0, 1.405158503D0,
     :    0.002024D-6,   14712.317116458D0, 2.752035928D0,
     :    0.001737D-6,    6290.189396992D0, 5.280820144D0,
     :    0.002229D-6,     491.557929457D0, 1.571007057D0,
     :    0.001602D-6,   14314.168113050D0, 4.203664806D0,
     :    0.002186D-6,     454.909366527D0, 1.402101526D0,
     :    0.001897D-6,   22483.848574493D0, 4.167932508D0,
     :    0.001825D-6,   -3738.761430108D0, 0.545828785D0,
     :    0.001894D-6,    1052.268383188D0, 5.817167450D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=181,190) /
     :    0.001421D-6,      20.355319399D0, 2.419886601D0,
     :    0.001408D-6,   10984.192351700D0, 2.732084787D0,
     :    0.001847D-6,   10873.986030480D0, 2.903477885D0,
     :    0.001391D-6,   -8635.942003763D0, 0.593891500D0,
     :    0.001388D-6,      -7.046236698D0, 1.166145902D0,
     :    0.001810D-6,  -88860.057071188D0, 0.487355242D0,
     :    0.001288D-6,   -1990.745017041D0, 3.913022880D0,
     :    0.001297D-6,   23543.230504682D0, 3.063805171D0,
     :    0.001335D-6,    -266.607041722D0, 3.995764039D0,
     :    0.001376D-6,   10969.965257698D0, 5.152914309D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=191,200) /
     :    0.001745D-6,  244287.600007027D0, 3.626395673D0,
     :    0.001649D-6,   31441.677569757D0, 1.952049260D0,
     :    0.001416D-6,    9225.539273283D0, 4.996408389D0,
     :    0.001238D-6,    4804.209275927D0, 5.503379738D0,
     :    0.001472D-6,    4590.910180489D0, 4.164913291D0,
     :    0.001169D-6,    6040.347246017D0, 5.841719038D0,
     :    0.001039D-6,    5540.085789459D0, 2.769753519D0,
     :    0.001004D-6,    -170.672870619D0, 0.755008103D0,
     :    0.001284D-6,   10575.406682942D0, 5.306538209D0,
     :    0.001278D-6,      71.812653151D0, 4.713486491D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=201,210) /
     :    0.001321D-6,   18209.330263660D0, 2.624866359D0,
     :    0.001297D-6,   21228.392023546D0, 0.382603541D0,
     :    0.000954D-6,    6282.095528923D0, 0.882213514D0,
     :    0.001145D-6,    6058.731054289D0, 1.169483931D0,
     :    0.000979D-6,    5547.199336460D0, 5.448375984D0,
     :    0.000987D-6,   -6262.300454499D0, 2.656486959D0,
     :    0.001070D-6, -154717.609887482D0, 1.827624012D0,
     :    0.000991D-6,    4701.116501708D0, 4.387001801D0,
     :    0.001155D-6,     -14.227094002D0, 3.042700750D0,
     :    0.001176D-6,     277.034993741D0, 3.335519004D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=211,220) /
     :    0.000890D-6,   13916.019109642D0, 5.601498297D0,
     :    0.000884D-6,   -1551.045222648D0, 1.088831705D0,
     :    0.000876D-6,    5017.508371365D0, 3.969902609D0,
     :    0.000806D-6,   15110.466119866D0, 5.142876744D0,
     :    0.000773D-6,   -4136.910433516D0, 0.022067765D0,
     :    0.001077D-6,     175.166059800D0, 1.844913056D0,
     :    0.000954D-6,   -6284.056171060D0, 0.968480906D0,
     :    0.000737D-6,    5326.786694021D0, 4.923831588D0,
     :    0.000845D-6,    -433.711737877D0, 4.749245231D0,
     :    0.000819D-6,    8662.240323563D0, 5.991247817D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=221,230) /
     :    0.000852D-6,     199.072001436D0, 2.189604979D0,
     :    0.000723D-6,   17256.631536341D0, 6.068719637D0,
     :    0.000940D-6,    6037.244203762D0, 6.197428148D0,
     :    0.000885D-6,   11712.955318231D0, 3.280414875D0,
     :    0.000706D-6,   12559.038152982D0, 2.824848947D0,
     :    0.000732D-6,    2379.164473572D0, 2.501813417D0,
     :    0.000764D-6,   -6127.655450557D0, 2.236346329D0,
     :    0.000908D-6,     131.541961686D0, 2.521257490D0,
     :    0.000907D-6,   35371.887265976D0, 3.370195967D0,
     :    0.000673D-6,    1066.495477190D0, 3.876512374D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=231,240) /
     :    0.000814D-6,   17654.780539750D0, 4.627122566D0,
     :    0.000630D-6,      36.027866677D0, 0.156368499D0,
     :    0.000798D-6,     515.463871093D0, 5.151962502D0,
     :    0.000798D-6,     148.078724426D0, 5.909225055D0,
     :    0.000806D-6,     309.278322656D0, 6.054064447D0,
     :    0.000607D-6,     -39.617508346D0, 2.839021623D0,
     :    0.000601D-6,     412.371096874D0, 3.984225404D0,
     :    0.000646D-6,   11403.676995575D0, 3.852959484D0,
     :    0.000704D-6,   13521.751441591D0, 2.300991267D0,
     :    0.000603D-6,  -65147.619767937D0, 4.140083146D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=241,250) /
     :    0.000609D-6,   10177.257679534D0, 0.437122327D0,
     :    0.000631D-6,    5767.611978898D0, 4.026532329D0,
     :    0.000576D-6,   11087.285125918D0, 4.760293101D0,
     :    0.000674D-6,   14945.316173554D0, 6.270510511D0,
     :    0.000726D-6,    5429.879468239D0, 6.039606892D0,
     :    0.000710D-6,   28766.924424484D0, 5.672617711D0,
     :    0.000647D-6,   11856.218651625D0, 3.397132627D0,
     :    0.000678D-6,   -5481.254918868D0, 6.249666675D0,
     :    0.000618D-6,   22003.914634870D0, 2.466427018D0,
     :    0.000738D-6,    6134.997125565D0, 2.242668890D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=251,260) /
     :    0.000660D-6,     625.670192312D0, 5.864091907D0,
     :    0.000694D-6,    3496.032826134D0, 2.668309141D0,
     :    0.000531D-6,    6489.261398429D0, 1.681888780D0,
     :    0.000611D-6, -143571.324284214D0, 2.424978312D0,
     :    0.000575D-6,   12043.574281889D0, 4.216492400D0,
     :    0.000553D-6,   12416.588502848D0, 4.772158039D0,
     :    0.000689D-6,    4686.889407707D0, 6.224271088D0,
     :    0.000495D-6,    7342.457780181D0, 3.817285811D0,
     :    0.000567D-6,    3634.621024518D0, 1.649264690D0,
     :    0.000515D-6,   18635.928454536D0, 3.945345892D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=261,270) /
     :    0.000486D-6,    -323.505416657D0, 4.061673868D0,
     :    0.000662D-6,   25158.601719765D0, 1.794058369D0,
     :    0.000509D-6,     846.082834751D0, 3.053874588D0,
     :    0.000472D-6,  -12569.674818332D0, 5.112133338D0,
     :    0.000461D-6,    6179.983075773D0, 0.513669325D0,
     :    0.000641D-6,   83467.156352816D0, 3.210727723D0,
     :    0.000520D-6,   10344.295065386D0, 2.445597761D0,
     :    0.000493D-6,   18422.629359098D0, 1.676939306D0,
     :    0.000478D-6,    1265.567478626D0, 5.487314569D0,
     :    0.000472D-6,     -18.159247265D0, 1.999707589D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=271,280) /
     :    0.000559D-6,   11190.377900137D0, 5.783236356D0,
     :    0.000494D-6,    9623.688276691D0, 3.022645053D0,
     :    0.000463D-6,    5739.157790895D0, 1.411223013D0,
     :    0.000432D-6,   16858.482532933D0, 1.179256434D0,
     :    0.000574D-6,   72140.628666286D0, 1.758191830D0,
     :    0.000484D-6,   17267.268201691D0, 3.290589143D0,
     :    0.000550D-6,    4907.302050146D0, 0.864024298D0,
     :    0.000399D-6,      14.977853527D0, 2.094441910D0,
     :    0.000491D-6,     224.344795702D0, 0.878372791D0,
     :    0.000432D-6,   20426.571092422D0, 6.003829241D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=281,290) /
     :    0.000481D-6,    5749.452731634D0, 4.309591964D0,
     :    0.000480D-6,    5757.317038160D0, 1.142348571D0,
     :    0.000485D-6,    6702.560493867D0, 0.210580917D0,
     :    0.000426D-6,    6055.549660552D0, 4.274476529D0,
     :    0.000480D-6,    5959.570433334D0, 5.031351030D0,
     :    0.000466D-6,   12562.628581634D0, 4.959581597D0,
     :    0.000520D-6,   39302.096962196D0, 4.788002889D0,
     :    0.000458D-6,   12132.439962106D0, 1.880103788D0,
     :    0.000470D-6,   12029.347187887D0, 1.405611197D0,
     :    0.000416D-6,   -7477.522860216D0, 1.082356330D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=291,300) /
     :    0.000449D-6,   11609.862544012D0, 4.179989585D0,
     :    0.000465D-6,   17253.041107690D0, 0.353496295D0,
     :    0.000362D-6,   -4535.059436924D0, 1.583849576D0,
     :    0.000383D-6,   21954.157609398D0, 3.747376371D0,
     :    0.000389D-6,      17.252277143D0, 1.395753179D0,
     :    0.000331D-6,   18052.929543158D0, 0.566790582D0,
     :    0.000430D-6,   13517.870106233D0, 0.685827538D0,
     :    0.000368D-6,   -5756.908003246D0, 0.731374317D0,
     :    0.000330D-6,   10557.594160824D0, 3.710043680D0,
     :    0.000332D-6,   20199.094959633D0, 1.652901407D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=301,310) /
     :    0.000384D-6,   11933.367960670D0, 5.827781531D0,
     :    0.000387D-6,   10454.501386605D0, 2.541182564D0,
     :    0.000325D-6,   15671.081759407D0, 2.178850542D0,
     :    0.000318D-6,     138.517496871D0, 2.253253037D0,
     :    0.000305D-6,    9388.005909415D0, 0.578340206D0,
     :    0.000352D-6,    5749.861766548D0, 3.000297967D0,
     :    0.000311D-6,    6915.859589305D0, 1.693574249D0,
     :    0.000297D-6,   24072.921469776D0, 1.997249392D0,
     :    0.000363D-6,    -640.877607382D0, 5.071820966D0,
     :    0.000323D-6,   12592.450019783D0, 1.072262823D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=311,320) /
     :    0.000341D-6,   12146.667056108D0, 4.700657997D0,
     :    0.000290D-6,    9779.108676125D0, 1.812320441D0,
     :    0.000342D-6,    6132.028180148D0, 4.322238614D0,
     :    0.000329D-6,    6268.848755990D0, 3.033827743D0,
     :    0.000374D-6,   17996.031168222D0, 3.388716544D0,
     :    0.000285D-6,    -533.214083444D0, 4.687313233D0,
     :    0.000338D-6,    6065.844601290D0, 0.877776108D0,
     :    0.000276D-6,      24.298513841D0, 0.770299429D0,
     :    0.000336D-6,   -2388.894020449D0, 5.353796034D0,
     :    0.000290D-6,    3097.883822726D0, 4.075291557D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=321,330) /
     :    0.000318D-6,     709.933048357D0, 5.941207518D0,
     :    0.000271D-6,   13095.842665077D0, 3.208912203D0,
     :    0.000331D-6,    6073.708907816D0, 4.007881169D0,
     :    0.000292D-6,     742.990060533D0, 2.714333592D0,
     :    0.000362D-6,   29088.811415985D0, 3.215977013D0,
     :    0.000280D-6,   12359.966151546D0, 0.710872502D0,
     :    0.000267D-6,   10440.274292604D0, 4.730108488D0,
     :    0.000262D-6,     838.969287750D0, 1.327720272D0,
     :    0.000250D-6,   16496.361396202D0, 0.898769761D0,
     :    0.000325D-6,   20597.243963041D0, 0.180044365D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=331,340) /
     :    0.000268D-6,    6148.010769956D0, 5.152666276D0,
     :    0.000284D-6,    5636.065016677D0, 5.655385808D0,
     :    0.000301D-6,    6080.822454817D0, 2.135396205D0,
     :    0.000294D-6,    -377.373607916D0, 3.708784168D0,
     :    0.000236D-6,    2118.763860378D0, 1.733578756D0,
     :    0.000234D-6,    5867.523359379D0, 5.575209112D0,
     :    0.000268D-6, -226858.238553767D0, 0.069432392D0,
     :    0.000265D-6,  167283.761587465D0, 4.369302826D0,
     :    0.000280D-6,   28237.233459389D0, 5.304829118D0,
     :    0.000292D-6,   12345.739057544D0, 4.096094132D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=341,350) /
     :    0.000223D-6,   19800.945956225D0, 3.069327406D0,
     :    0.000301D-6,   43232.306658416D0, 6.205311188D0,
     :    0.000264D-6,   18875.525869774D0, 1.417263408D0,
     :    0.000304D-6,   -1823.175188677D0, 3.409035232D0,
     :    0.000301D-6,     109.945688789D0, 0.510922054D0,
     :    0.000260D-6,     813.550283960D0, 2.389438934D0,
     :    0.000299D-6,  316428.228673312D0, 5.384595078D0,
     :    0.000211D-6,    5756.566278634D0, 3.789392838D0,
     :    0.000209D-6,    5750.203491159D0, 1.661943545D0,
     :    0.000240D-6,   12489.885628707D0, 5.684549045D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=351,360) /
     :    0.000216D-6,    6303.851245484D0, 3.862942261D0,
     :    0.000203D-6,    1581.959348283D0, 5.549853589D0,
     :    0.000200D-6,    5642.198242609D0, 1.016115785D0,
     :    0.000197D-6,     -70.849445304D0, 4.690702525D0,
     :    0.000227D-6,    6287.008003254D0, 2.911891613D0,
     :    0.000197D-6,     533.623118358D0, 1.048982898D0,
     :    0.000205D-6,   -6279.485421340D0, 1.829362730D0,
     :    0.000209D-6,  -10988.808157535D0, 2.636140084D0,
     :    0.000208D-6,    -227.526189440D0, 4.127883842D0,
     :    0.000191D-6,     415.552490612D0, 4.401165650D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=361,370) /
     :    0.000190D-6,   29296.615389579D0, 4.175658539D0,
     :    0.000264D-6,   66567.485864652D0, 4.601102551D0,
     :    0.000256D-6,   -3646.350377354D0, 0.506364778D0,
     :    0.000188D-6,   13119.721102825D0, 2.032195842D0,
     :    0.000185D-6,    -209.366942175D0, 4.694756586D0,
     :    0.000198D-6,   25934.124331089D0, 3.832703118D0,
     :    0.000195D-6,    4061.219215394D0, 3.308463427D0,
     :    0.000234D-6,    5113.487598583D0, 1.716090661D0,
     :    0.000188D-6,    1478.866574064D0, 5.686865780D0,
     :    0.000222D-6,   11823.161639450D0, 1.942386641D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=371,380) /
     :    0.000181D-6,   10770.893256262D0, 1.999482059D0,
     :    0.000171D-6,    6546.159773364D0, 1.182807992D0,
     :    0.000206D-6,      70.328180442D0, 5.934076062D0,
     :    0.000169D-6,   20995.392966449D0, 2.169080622D0,
     :    0.000191D-6,   10660.686935042D0, 5.405515999D0,
     :    0.000228D-6,   33019.021112205D0, 4.656985514D0,
     :    0.000184D-6,   -4933.208440333D0, 3.327476868D0,
     :    0.000220D-6,    -135.625325010D0, 1.765430262D0,
     :    0.000166D-6,   23141.558382925D0, 3.454132746D0,
     :    0.000191D-6,    6144.558353121D0, 5.020393445D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=381,390) /
     :    0.000180D-6,    6084.003848555D0, 0.602182191D0,
     :    0.000163D-6,   17782.732072784D0, 4.960593133D0,
     :    0.000225D-6,   16460.333529525D0, 2.596451817D0,
     :    0.000222D-6,    5905.702242076D0, 3.731990323D0,
     :    0.000204D-6,     227.476132789D0, 5.636192701D0,
     :    0.000159D-6,   16737.577236597D0, 3.600691544D0,
     :    0.000200D-6,    6805.653268085D0, 0.868220961D0,
     :    0.000187D-6,   11919.140866668D0, 2.629456641D0,
     :    0.000161D-6,     127.471796607D0, 2.862574720D0,
     :    0.000205D-6,    6286.666278643D0, 1.742882331D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=391,400) /
     :    0.000189D-6,     153.778810485D0, 4.812372643D0,
     :    0.000168D-6,   16723.350142595D0, 0.027860588D0,
     :    0.000149D-6,   11720.068865232D0, 0.659721876D0,
     :    0.000189D-6,    5237.921013804D0, 5.245313000D0,
     :    0.000143D-6,    6709.674040867D0, 4.317625647D0,
     :    0.000146D-6,    4487.817406270D0, 4.815297007D0,
     :    0.000144D-6,    -664.756045130D0, 5.381366880D0,
     :    0.000175D-6,    5127.714692584D0, 4.728443327D0,
     :    0.000162D-6,    6254.626662524D0, 1.435132069D0,
     :    0.000187D-6,   47162.516354635D0, 1.354371923D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=401,410) /
     :    0.000146D-6,   11080.171578918D0, 3.369695406D0,
     :    0.000180D-6,    -348.924420448D0, 2.490902145D0,
     :    0.000148D-6,     151.047669843D0, 3.799109588D0,
     :    0.000157D-6,    6197.248551160D0, 1.284375887D0,
     :    0.000167D-6,     146.594251718D0, 0.759969109D0,
     :    0.000133D-6,   -5331.357443741D0, 5.409701889D0,
     :    0.000154D-6,      95.979227218D0, 3.366890614D0,
     :    0.000148D-6,   -6418.140930027D0, 3.384104996D0,
     :    0.000128D-6,   -6525.804453965D0, 3.803419985D0,
     :    0.000130D-6,   11293.470674356D0, 0.939039445D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=411,420) /
     :    0.000152D-6,   -5729.506447149D0, 0.734117523D0,
     :    0.000138D-6,     210.117701700D0, 2.564216078D0,
     :    0.000123D-6,    6066.595360816D0, 4.517099537D0,
     :    0.000140D-6,   18451.078546566D0, 0.642049130D0,
     :    0.000126D-6,   11300.584221356D0, 3.485280663D0,
     :    0.000119D-6,   10027.903195729D0, 3.217431161D0,
     :    0.000151D-6,    4274.518310832D0, 4.404359108D0,
     :    0.000117D-6,    6072.958148291D0, 0.366324650D0,
     :    0.000165D-6,   -7668.637425143D0, 4.298212528D0,
     :    0.000117D-6,   -6245.048177356D0, 5.379518958D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=421,430) /
     :    0.000130D-6,   -5888.449964932D0, 4.527681115D0,
     :    0.000121D-6,    -543.918059096D0, 6.109429504D0,
     :    0.000162D-6,    9683.594581116D0, 5.720092446D0,
     :    0.000141D-6,    6219.339951688D0, 0.679068671D0,
     :    0.000118D-6,   22743.409379516D0, 4.881123092D0,
     :    0.000129D-6,    1692.165669502D0, 0.351407289D0,
     :    0.000126D-6,    5657.405657679D0, 5.146592349D0,
     :    0.000114D-6,     728.762966531D0, 0.520791814D0,
     :    0.000120D-6,      52.596639600D0, 0.948516300D0,
     :    0.000115D-6,      65.220371012D0, 3.504914846D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=431,440) /
     :    0.000126D-6,    5881.403728234D0, 5.577502482D0,
     :    0.000158D-6,  163096.180360983D0, 2.957128968D0,
     :    0.000134D-6,   12341.806904281D0, 2.598576764D0,
     :    0.000151D-6,   16627.370915377D0, 3.985702050D0,
     :    0.000109D-6,    1368.660252845D0, 0.014730471D0,
     :    0.000131D-6,    6211.263196841D0, 0.085077024D0,
     :    0.000146D-6,    5792.741760812D0, 0.708426604D0,
     :    0.000146D-6,     -77.750543984D0, 3.121576600D0,
     :    0.000107D-6,    5341.013788022D0, 0.288231904D0,
     :    0.000138D-6,    6281.591377283D0, 2.797450317D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=441,450) /
     :    0.000113D-6,   -6277.552925684D0, 2.788904128D0,
     :    0.000115D-6,    -525.758811831D0, 5.895222200D0,
     :    0.000138D-6,    6016.468808270D0, 6.096188999D0,
     :    0.000139D-6,   23539.707386333D0, 2.028195445D0,
     :    0.000146D-6,   -4176.041342449D0, 4.660008502D0,
     :    0.000107D-6,   16062.184526117D0, 4.066520001D0,
     :    0.000142D-6,   83783.548222473D0, 2.936315115D0,
     :    0.000128D-6,    9380.959672717D0, 3.223844306D0,
     :    0.000135D-6,    6205.325306007D0, 1.638054048D0,
     :    0.000101D-6,    2699.734819318D0, 5.481603249D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=451,460) /
     :    0.000104D-6,    -568.821874027D0, 2.205734493D0,
     :    0.000103D-6,    6321.103522627D0, 2.440421099D0,
     :    0.000119D-6,    6321.208885629D0, 2.547496264D0,
     :    0.000138D-6,    1975.492545856D0, 2.314608466D0,
     :    0.000121D-6,     137.033024162D0, 4.539108237D0,
     :    0.000123D-6,   19402.796952817D0, 4.538074405D0,
     :    0.000119D-6,   22805.735565994D0, 2.869040566D0,
     :    0.000133D-6,   64471.991241142D0, 6.056405489D0,
     :    0.000129D-6,     -85.827298831D0, 2.540635083D0,
     :    0.000131D-6,   13613.804277336D0, 4.005732868D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=461,470) /
     :    0.000104D-6,    9814.604100291D0, 1.959967212D0,
     :    0.000112D-6,   16097.679950283D0, 3.589026260D0,
     :    0.000123D-6,    2107.034507542D0, 1.728627253D0,
     :    0.000121D-6,   36949.230808424D0, 6.072332087D0,
     :    0.000108D-6,  -12539.853380183D0, 3.716133846D0,
     :    0.000113D-6,   -7875.671863624D0, 2.725771122D0,
     :    0.000109D-6,    4171.425536614D0, 4.033338079D0,
     :    0.000101D-6,    6247.911759770D0, 3.441347021D0,
     :    0.000113D-6,    7330.728427345D0, 0.656372122D0,
     :    0.000113D-6,   51092.726050855D0, 2.791483066D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=471,480) /
     :    0.000106D-6,    5621.842923210D0, 1.815323326D0,
     :    0.000101D-6,     111.430161497D0, 5.711033677D0,
     :    0.000103D-6,     909.818733055D0, 2.812745443D0,
     :    0.000101D-6,    1790.642637886D0, 1.965746028D0,

*  T
     :  102.156724D-6,    6283.075849991D0, 4.249032005D0,
     :    1.706807D-6,   12566.151699983D0, 4.205904248D0,
     :    0.269668D-6,     213.299095438D0, 3.400290479D0,
     :    0.265919D-6,     529.690965095D0, 5.836047367D0,
     :    0.210568D-6,      -3.523118349D0, 6.262738348D0,
     :    0.077996D-6,    5223.693919802D0, 4.670344204D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=481,490) /
     :    0.054764D-6,    1577.343542448D0, 4.534800170D0,
     :    0.059146D-6,      26.298319800D0, 1.083044735D0,
     :    0.034420D-6,    -398.149003408D0, 5.980077351D0,
     :    0.032088D-6,   18849.227549974D0, 4.162913471D0,
     :    0.033595D-6,    5507.553238667D0, 5.980162321D0,
     :    0.029198D-6,    5856.477659115D0, 0.623811863D0,
     :    0.027764D-6,     155.420399434D0, 3.745318113D0,
     :    0.025190D-6,    5746.271337896D0, 2.980330535D0,
     :    0.022997D-6,    -796.298006816D0, 1.174411803D0,
     :    0.024976D-6,    5760.498431898D0, 2.467913690D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=491,500) /
     :    0.021774D-6,     206.185548437D0, 3.854787540D0,
     :    0.017925D-6,    -775.522611324D0, 1.092065955D0,
     :    0.013794D-6,     426.598190876D0, 2.699831988D0,
     :    0.013276D-6,    6062.663207553D0, 5.845801920D0,
     :    0.011774D-6,   12036.460734888D0, 2.292832062D0,
     :    0.012869D-6,    6076.890301554D0, 5.333425680D0,
     :    0.012152D-6,    1059.381930189D0, 6.222874454D0,
     :    0.011081D-6,      -7.113547001D0, 5.154724984D0,
     :    0.010143D-6,    4694.002954708D0, 4.044013795D0,
     :    0.009357D-6,    5486.777843175D0, 3.416081409D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=501,510) /
     :    0.010084D-6,     522.577418094D0, 0.749320262D0,
     :    0.008587D-6,   10977.078804699D0, 2.777152598D0,
     :    0.008628D-6,    6275.962302991D0, 4.562060226D0,
     :    0.008158D-6,    -220.412642439D0, 5.806891533D0,
     :    0.007746D-6,    2544.314419883D0, 1.603197066D0,
     :    0.007670D-6,    2146.165416475D0, 3.000200440D0,
     :    0.007098D-6,      74.781598567D0, 0.443725817D0,
     :    0.006180D-6,    -536.804512095D0, 1.302642751D0,
     :    0.005818D-6,    5088.628839767D0, 4.827723531D0,
     :    0.004945D-6,   -6286.598968340D0, 0.268305170D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=511,520) /
     :    0.004774D-6,    1349.867409659D0, 5.808636673D0,
     :    0.004687D-6,    -242.728603974D0, 5.154890570D0,
     :    0.006089D-6,    1748.016413067D0, 4.403765209D0,
     :    0.005975D-6,   -1194.447010225D0, 2.583472591D0,
     :    0.004229D-6,     951.718406251D0, 0.931172179D0,
     :    0.005264D-6,     553.569402842D0, 2.336107252D0,
     :    0.003049D-6,    5643.178563677D0, 1.362634430D0,
     :    0.002974D-6,    6812.766815086D0, 1.583012668D0,
     :    0.003403D-6,   -2352.866153772D0, 2.552189886D0,
     :    0.003030D-6,     419.484643875D0, 5.286473844D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=521,530) /
     :    0.003210D-6,      -7.046236698D0, 1.863796539D0,
     :    0.003058D-6,    9437.762934887D0, 4.226420633D0,
     :    0.002589D-6,   12352.852604545D0, 1.991935820D0,
     :    0.002927D-6,    5216.580372801D0, 2.319951253D0,
     :    0.002425D-6,    5230.807466803D0, 3.084752833D0,
     :    0.002656D-6,    3154.687084896D0, 2.487447866D0,
     :    0.002445D-6,   10447.387839604D0, 2.347139160D0,
     :    0.002990D-6,    4690.479836359D0, 6.235872050D0,
     :    0.002890D-6,    5863.591206116D0, 0.095197563D0,
     :    0.002498D-6,    6438.496249426D0, 2.994779800D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=531,540) /
     :    0.001889D-6,    8031.092263058D0, 3.569003717D0,
     :    0.002567D-6,     801.820931124D0, 3.425611498D0,
     :    0.001803D-6,  -71430.695617928D0, 2.192295512D0,
     :    0.001782D-6,       3.932153263D0, 5.180433689D0,
     :    0.001694D-6,   -4705.732307544D0, 4.641779174D0,
     :    0.001704D-6,   -1592.596013633D0, 3.997097652D0,
     :    0.001735D-6,    5849.364112115D0, 0.417558428D0,
     :    0.001643D-6,    8429.241266467D0, 2.180619584D0,
     :    0.001680D-6,      38.133035638D0, 4.164529426D0,
     :    0.002045D-6,    7084.896781115D0, 0.526323854D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=541,550) /
     :    0.001458D-6,    4292.330832950D0, 1.356098141D0,
     :    0.001437D-6,      20.355319399D0, 3.895439360D0,
     :    0.001738D-6,    6279.552731642D0, 0.087484036D0,
     :    0.001367D-6,   14143.495242431D0, 3.987576591D0,
     :    0.001344D-6,    7234.794256242D0, 0.090454338D0,
     :    0.001438D-6,   11499.656222793D0, 0.974387904D0,
     :    0.001257D-6,    6836.645252834D0, 1.509069366D0,
     :    0.001358D-6,   11513.883316794D0, 0.495572260D0,
     :    0.001628D-6,    7632.943259650D0, 4.968445721D0,
     :    0.001169D-6,     103.092774219D0, 2.838496795D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=551,560) /
     :    0.001162D-6,    4164.311989613D0, 3.408387778D0,
     :    0.001092D-6,    6069.776754553D0, 3.617942651D0,
     :    0.001008D-6,   17789.845619785D0, 0.286350174D0,
     :    0.001008D-6,     639.897286314D0, 1.610762073D0,
     :    0.000918D-6,   10213.285546211D0, 5.532798067D0,
     :    0.001011D-6,   -6256.777530192D0, 0.661826484D0,
     :    0.000753D-6,   16730.463689596D0, 3.905030235D0,
     :    0.000737D-6,   11926.254413669D0, 4.641956361D0,
     :    0.000694D-6,    3340.612426700D0, 2.111120332D0,
     :    0.000701D-6,    3894.181829542D0, 2.760823491D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=561,570) /
     :    0.000689D-6,    -135.065080035D0, 4.768800780D0,
     :    0.000700D-6,   13367.972631107D0, 5.760439898D0,
     :    0.000664D-6,    6040.347246017D0, 1.051215840D0,
     :    0.000654D-6,    5650.292110678D0, 4.911332503D0,
     :    0.000788D-6,    6681.224853400D0, 4.699648011D0,
     :    0.000628D-6,    5333.900241022D0, 5.024608847D0,
     :    0.000755D-6,    -110.206321219D0, 4.370971253D0,
     :    0.000628D-6,    6290.189396992D0, 3.660478857D0,
     :    0.000635D-6,   25132.303399966D0, 4.121051532D0,
     :    0.000534D-6,    5966.683980335D0, 1.173284524D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=571,580) /
     :    0.000543D-6,    -433.711737877D0, 0.345585464D0,
     :    0.000517D-6,   -1990.745017041D0, 5.414571768D0,
     :    0.000504D-6,    5767.611978898D0, 2.328281115D0,
     :    0.000485D-6,    5753.384884897D0, 1.685874771D0,
     :    0.000463D-6,    7860.419392439D0, 5.297703006D0,
     :    0.000604D-6,     515.463871093D0, 0.591998446D0,
     :    0.000443D-6,   12168.002696575D0, 4.830881244D0,
     :    0.000570D-6,     199.072001436D0, 3.899190272D0,
     :    0.000465D-6,   10969.965257698D0, 0.476681802D0,
     :    0.000424D-6,   -7079.373856808D0, 1.112242763D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=581,590) /
     :    0.000427D-6,     735.876513532D0, 1.994214480D0,
     :    0.000478D-6,   -6127.655450557D0, 3.778025483D0,
     :    0.000414D-6,   10973.555686350D0, 5.441088327D0,
     :    0.000512D-6,    1589.072895284D0, 0.107123853D0,
     :    0.000378D-6,   10984.192351700D0, 0.915087231D0,
     :    0.000402D-6,   11371.704689758D0, 4.107281715D0,
     :    0.000453D-6,    9917.696874510D0, 1.917490952D0,
     :    0.000395D-6,     149.563197135D0, 2.763124165D0,
     :    0.000371D-6,    5739.157790895D0, 3.112111866D0,
     :    0.000350D-6,   11790.629088659D0, 0.440639857D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=591,600) /
     :    0.000356D-6,    6133.512652857D0, 5.444568842D0,
     :    0.000344D-6,     412.371096874D0, 5.676832684D0,
     :    0.000383D-6,     955.599741609D0, 5.559734846D0,
     :    0.000333D-6,    6496.374945429D0, 0.261537984D0,
     :    0.000340D-6,    6055.549660552D0, 5.975534987D0,
     :    0.000334D-6,    1066.495477190D0, 2.335063907D0,
     :    0.000399D-6,   11506.769769794D0, 5.321230910D0,
     :    0.000314D-6,   18319.536584880D0, 2.313312404D0,
     :    0.000424D-6,    1052.268383188D0, 1.211961766D0,
     :    0.000307D-6,      63.735898303D0, 3.169551388D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=601,610) /
     :    0.000329D-6,      29.821438149D0, 6.106912080D0,
     :    0.000357D-6,    6309.374169791D0, 4.223760346D0,
     :    0.000312D-6,   -3738.761430108D0, 2.180556645D0,
     :    0.000301D-6,     309.278322656D0, 1.499984572D0,
     :    0.000268D-6,   12043.574281889D0, 2.447520648D0,
     :    0.000257D-6,   12491.370101415D0, 3.662331761D0,
     :    0.000290D-6,     625.670192312D0, 1.272834584D0,
     :    0.000256D-6,    5429.879468239D0, 1.913426912D0,
     :    0.000339D-6,    3496.032826134D0, 4.165930011D0,
     :    0.000283D-6,    3930.209696220D0, 4.325565754D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=611,620) /
     :    0.000241D-6,   12528.018664345D0, 3.832324536D0,
     :    0.000304D-6,    4686.889407707D0, 1.612348468D0,
     :    0.000259D-6,   16200.772724501D0, 3.470173146D0,
     :    0.000238D-6,   12139.553509107D0, 1.147977842D0,
     :    0.000236D-6,    6172.869528772D0, 3.776271728D0,
     :    0.000296D-6,   -7058.598461315D0, 0.460368852D0,
     :    0.000306D-6,   10575.406682942D0, 0.554749016D0,
     :    0.000251D-6,   17298.182327326D0, 0.834332510D0,
     :    0.000290D-6,    4732.030627343D0, 4.759564091D0,
     :    0.000261D-6,    5884.926846583D0, 0.298259862D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=621,630) /
     :    0.000249D-6,    5547.199336460D0, 3.749366406D0,
     :    0.000213D-6,   11712.955318231D0, 5.415666119D0,
     :    0.000223D-6,    4701.116501708D0, 2.703203558D0,
     :    0.000268D-6,    -640.877607382D0, 0.283670793D0,
     :    0.000209D-6,    5636.065016677D0, 1.238477199D0,
     :    0.000193D-6,   10177.257679534D0, 1.943251340D0,
     :    0.000182D-6,    6283.143160294D0, 2.456157599D0,
     :    0.000184D-6,    -227.526189440D0, 5.888038582D0,
     :    0.000182D-6,   -6283.008539689D0, 0.241332086D0,
     :    0.000228D-6,   -6284.056171060D0, 2.657323816D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=631,640) /
     :    0.000166D-6,    7238.675591600D0, 5.930629110D0,
     :    0.000167D-6,    3097.883822726D0, 5.570955333D0,
     :    0.000159D-6,    -323.505416657D0, 5.786670700D0,
     :    0.000154D-6,   -4136.910433516D0, 1.517805532D0,
     :    0.000176D-6,   12029.347187887D0, 3.139266834D0,
     :    0.000167D-6,   12132.439962106D0, 3.556352289D0,
     :    0.000153D-6,     202.253395174D0, 1.463313961D0,
     :    0.000157D-6,   17267.268201691D0, 1.586837396D0,
     :    0.000142D-6,   83996.847317911D0, 0.022670115D0,
     :    0.000152D-6,   17260.154654690D0, 0.708528947D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=641,650) /
     :    0.000144D-6,    6084.003848555D0, 5.187075177D0,
     :    0.000135D-6,    5756.566278634D0, 1.993229262D0,
     :    0.000134D-6,    5750.203491159D0, 3.457197134D0,
     :    0.000144D-6,    5326.786694021D0, 6.066193291D0,
     :    0.000160D-6,   11015.106477335D0, 1.710431974D0,
     :    0.000133D-6,    3634.621024518D0, 2.836451652D0,
     :    0.000134D-6,   18073.704938650D0, 5.453106665D0,
     :    0.000134D-6,    1162.474704408D0, 5.326898811D0,
     :    0.000128D-6,    5642.198242609D0, 2.511652591D0,
     :    0.000160D-6,     632.783739313D0, 5.628785365D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=651,660) /
     :    0.000132D-6,   13916.019109642D0, 0.819294053D0,
     :    0.000122D-6,   14314.168113050D0, 5.677408071D0,
     :    0.000125D-6,   12359.966151546D0, 5.251984735D0,
     :    0.000121D-6,    5749.452731634D0, 2.210924603D0,
     :    0.000136D-6,    -245.831646229D0, 1.646502367D0,
     :    0.000120D-6,    5757.317038160D0, 3.240883049D0,
     :    0.000134D-6,   12146.667056108D0, 3.059480037D0,
     :    0.000137D-6,    6206.809778716D0, 1.867105418D0,
     :    0.000141D-6,   17253.041107690D0, 2.069217456D0,
     :    0.000129D-6,   -7477.522860216D0, 2.781469314D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=661,670) /
     :    0.000116D-6,    5540.085789459D0, 4.281176991D0,
     :    0.000116D-6,    9779.108676125D0, 3.320925381D0,
     :    0.000129D-6,    5237.921013804D0, 3.497704076D0,
     :    0.000113D-6,    5959.570433334D0, 0.983210840D0,
     :    0.000122D-6,    6282.095528923D0, 2.674938860D0,
     :    0.000140D-6,     -11.045700264D0, 4.957936982D0,
     :    0.000108D-6,   23543.230504682D0, 1.390113589D0,
     :    0.000106D-6,  -12569.674818332D0, 0.429631317D0,
     :    0.000110D-6,    -266.607041722D0, 5.501340197D0,
     :    0.000115D-6,   12559.038152982D0, 4.691456618D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=671,680) /
     :    0.000134D-6,   -2388.894020449D0, 0.577313584D0,
     :    0.000109D-6,   10440.274292604D0, 6.218148717D0,
     :    0.000102D-6,    -543.918059096D0, 1.477842615D0,
     :    0.000108D-6,   21228.392023546D0, 2.237753948D0,
     :    0.000101D-6,   -4535.059436924D0, 3.100492232D0,
     :    0.000103D-6,      76.266071276D0, 5.594294322D0,
     :    0.000104D-6,     949.175608970D0, 5.674287810D0,
     :    0.000101D-6,   13517.870106233D0, 2.196632348D0,
     :    0.000100D-6,   11933.367960670D0, 4.056084160D0,
     :    4.322990D-6,    6283.075849991D0, 2.642893748D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=681,690) /
     :    0.406495D-6,       0.000000000D0, 4.712388980D0,
     :    0.122605D-6,   12566.151699983D0, 2.438140634D0,
     :    0.019476D-6,     213.299095438D0, 1.642186981D0,
     :    0.016916D-6,     529.690965095D0, 4.510959344D0,
     :    0.013374D-6,      -3.523118349D0, 1.502210314D0,
     :    0.008042D-6,      26.298319800D0, 0.478549024D0,
     :    0.007824D-6,     155.420399434D0, 5.254710405D0,
     :    0.004894D-6,    5746.271337896D0, 4.683210850D0,
     :    0.004875D-6,    5760.498431898D0, 0.759507698D0,
     :    0.004416D-6,    5223.693919802D0, 6.028853166D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=691,700) /
     :    0.004088D-6,      -7.113547001D0, 0.060926389D0,
     :    0.004433D-6,   77713.771467920D0, 3.627734103D0,
     :    0.003277D-6,   18849.227549974D0, 2.327912542D0,
     :    0.002703D-6,    6062.663207553D0, 1.271941729D0,
     :    0.003435D-6,    -775.522611324D0, 0.747446224D0,
     :    0.002618D-6,    6076.890301554D0, 3.633715689D0,
     :    0.003146D-6,     206.185548437D0, 5.647874613D0,
     :    0.002544D-6,    1577.343542448D0, 6.232904270D0,
     :    0.002218D-6,    -220.412642439D0, 1.309509946D0,
     :    0.002197D-6,    5856.477659115D0, 2.407212349D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=701,710) /
     :    0.002897D-6,    5753.384884897D0, 5.863842246D0,
     :    0.001766D-6,     426.598190876D0, 0.754113147D0,
     :    0.001738D-6,    -796.298006816D0, 2.714942671D0,
     :    0.001695D-6,     522.577418094D0, 2.629369842D0,
     :    0.001584D-6,    5507.553238667D0, 1.341138229D0,
     :    0.001503D-6,    -242.728603974D0, 0.377699736D0,
     :    0.001552D-6,    -536.804512095D0, 2.904684667D0,
     :    0.001370D-6,    -398.149003408D0, 1.265599125D0,
     :    0.001889D-6,   -5573.142801634D0, 4.413514859D0,
     :    0.001722D-6,    6069.776754553D0, 2.445966339D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=711,720) /
     :    0.001124D-6,    1059.381930189D0, 5.041799657D0,
     :    0.001258D-6,     553.569402842D0, 3.849557278D0,
     :    0.000831D-6,     951.718406251D0, 2.471094709D0,
     :    0.000767D-6,    4694.002954708D0, 5.363125422D0,
     :    0.000756D-6,    1349.867409659D0, 1.046195744D0,
     :    0.000775D-6,     -11.045700264D0, 0.245548001D0,
     :    0.000597D-6,    2146.165416475D0, 4.543268798D0,
     :    0.000568D-6,    5216.580372801D0, 4.178853144D0,
     :    0.000711D-6,    1748.016413067D0, 5.934271972D0,
     :    0.000499D-6,   12036.460734888D0, 0.624434410D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=721,730) /
     :    0.000671D-6,   -1194.447010225D0, 4.136047594D0,
     :    0.000488D-6,    5849.364112115D0, 2.209679987D0,
     :    0.000621D-6,    6438.496249426D0, 4.518860804D0,
     :    0.000495D-6,   -6286.598968340D0, 1.868201275D0,
     :    0.000456D-6,    5230.807466803D0, 1.271231591D0,
     :    0.000451D-6,    5088.628839767D0, 0.084060889D0,
     :    0.000435D-6,    5643.178563677D0, 3.324456609D0,
     :    0.000387D-6,   10977.078804699D0, 4.052488477D0,
     :    0.000547D-6,  161000.685737473D0, 2.841633844D0,
     :    0.000522D-6,    3154.687084896D0, 2.171979966D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=731,740) /
     :    0.000375D-6,    5486.777843175D0, 4.983027306D0,
     :    0.000421D-6,    5863.591206116D0, 4.546432249D0,
     :    0.000439D-6,    7084.896781115D0, 0.522967921D0,
     :    0.000309D-6,    2544.314419883D0, 3.172606705D0,
     :    0.000347D-6,    4690.479836359D0, 1.479586566D0,
     :    0.000317D-6,     801.820931124D0, 3.553088096D0,
     :    0.000262D-6,     419.484643875D0, 0.606635550D0,
     :    0.000248D-6,    6836.645252834D0, 3.014082064D0,
     :    0.000245D-6,   -1592.596013633D0, 5.519526220D0,
     :    0.000225D-6,    4292.330832950D0, 2.877956536D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=741,750) /
     :    0.000214D-6,    7234.794256242D0, 1.605227587D0,
     :    0.000205D-6,    5767.611978898D0, 0.625804796D0,
     :    0.000180D-6,   10447.387839604D0, 3.499954526D0,
     :    0.000229D-6,     199.072001436D0, 5.632304604D0,
     :    0.000214D-6,     639.897286314D0, 5.960227667D0,
     :    0.000175D-6,    -433.711737877D0, 2.162417992D0,
     :    0.000209D-6,     515.463871093D0, 2.322150893D0,
     :    0.000173D-6,    6040.347246017D0, 2.556183691D0,
     :    0.000184D-6,    6309.374169791D0, 4.732296790D0,
     :    0.000227D-6,  149854.400134205D0, 5.385812217D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=751,760) /
     :    0.000154D-6,    8031.092263058D0, 5.120720920D0,
     :    0.000151D-6,    5739.157790895D0, 4.815000443D0,
     :    0.000197D-6,    7632.943259650D0, 0.222827271D0,
     :    0.000197D-6,      74.781598567D0, 3.910456770D0,
     :    0.000138D-6,    6055.549660552D0, 1.397484253D0,
     :    0.000149D-6,   -6127.655450557D0, 5.333727496D0,
     :    0.000137D-6,    3894.181829542D0, 4.281749907D0,
     :    0.000135D-6,    9437.762934887D0, 5.979971885D0,
     :    0.000139D-6,   -2352.866153772D0, 4.715630782D0,
     :    0.000142D-6,    6812.766815086D0, 0.513330157D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=761,770) /
     :    0.000120D-6,   -4705.732307544D0, 0.194160689D0,
     :    0.000131D-6,  -71430.695617928D0, 0.000379226D0,
     :    0.000124D-6,    6279.552731642D0, 2.122264908D0,
     :    0.000108D-6,   -6256.777530192D0, 0.883445696D0,
     :    0.143388D-6,    6283.075849991D0, 1.131453581D0,
     :    0.006671D-6,   12566.151699983D0, 0.775148887D0,
     :    0.001480D-6,     155.420399434D0, 0.480016880D0,
     :    0.000934D-6,     213.299095438D0, 6.144453084D0,
     :    0.000795D-6,     529.690965095D0, 2.941595619D0,
     :    0.000673D-6,    5746.271337896D0, 0.120415406D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=771,780) /
     :    0.000672D-6,    5760.498431898D0, 5.317009738D0,
     :    0.000389D-6,    -220.412642439D0, 3.090323467D0,
     :    0.000373D-6,    6062.663207553D0, 3.003551964D0,
     :    0.000360D-6,    6076.890301554D0, 1.918913041D0,
     :    0.000316D-6,     -21.340641002D0, 5.545798121D0,
     :    0.000315D-6,    -242.728603974D0, 1.884932563D0,
     :    0.000278D-6,     206.185548437D0, 1.266254859D0,
     :    0.000238D-6,    -536.804512095D0, 4.532664830D0,
     :    0.000185D-6,     522.577418094D0, 4.578313856D0,
     :    0.000245D-6,   18849.227549974D0, 0.587467082D0 /
      DATA ((FAIRHD(I,J),I=1,3),J=781,787) /
     :    0.000180D-6,     426.598190876D0, 5.151178553D0,
     :    0.000200D-6,     553.569402842D0, 5.355983739D0,
     :    0.000141D-6,    5223.693919802D0, 1.336556009D0,
     :    0.000104D-6,    5856.477659115D0, 4.239842759D0,
     :    0.003826D-6,    6283.075849991D0, 5.705257275D0,
     :    0.000303D-6,   12566.151699983D0, 5.407132842D0,
     :    0.000209D-6,     155.420399434D0, 1.989815753D0 /
* -----------------------------------------------------------------------



*  Time since J2000.0 in Julian millennia.
      T=(TDB-51544.5D0)/365250D0

* -------------------- Topocentric terms -------------------------------

*  Convert UT1 to local solar time in radians.
      TSOL = MOD(UT1,1D0)*D2PI - WL

*  FUNDAMENTAL ARGUMENTS:  Simon et al 1994

*  Combine time argument (millennia) with deg/arcsec factor.
      W = T / 3600D0

*  Sun Mean Longitude.
      ELSUN = MOD(280.46645683D0+1296027711.03429D0*W,360D0)*D2R

*  Sun Mean Anomaly.
      EMSUN = MOD(357.52910918D0+1295965810.481D0*W,360D0)*D2R

*  Mean Elongation of Moon from Sun.
      D = MOD(297.85019547D0+16029616012.090D0*W,360D0)*D2R

*  Mean Longitude of Jupiter.
      ELJ = MOD(34.35151874D0+109306899.89453D0*W,360D0)*D2R

*  Mean Longitude of Saturn.
      ELS = MOD(50.07744430D0+44046398.47038D0*W,360D0)*D2R

*  TOPOCENTRIC TERMS:  Moyer 1981 and Murray 1983.
      WT =  +0.00029D-10*U*SIN(TSOL+ELSUN-ELS)
     :      +0.00100D-10*U*SIN(TSOL-2D0*EMSUN)
     :      +0.00133D-10*U*SIN(TSOL-D)
     :      +0.00133D-10*U*SIN(TSOL+ELSUN-ELJ)
     :      -0.00229D-10*U*SIN(TSOL+2D0*ELSUN+EMSUN)
     :      -0.0220 D-10*V*COS(ELSUN+EMSUN)
     :      +0.05312D-10*U*SIN(TSOL-EMSUN)
     :      -0.13677D-10*U*SIN(TSOL+2D0*ELSUN)
     :      -1.3184 D-10*V*COS(ELSUN)
     :      +3.17679D-10*U*SIN(TSOL)

* --------------- Fairhead model ---------------------------------------

*  T**0
      W0=0D0
      DO I=474,1,-1
         W0=W0+FAIRHD(1,I)*SIN(FAIRHD(2,I)*T+FAIRHD(3,I))
      END DO

*  T**1
      W1=0D0
      DO I=679,475,-1
         W1=W1+FAIRHD(1,I)*SIN(FAIRHD(2,I)*T+FAIRHD(3,I))
      END DO

*  T**2
      W2=0D0
      DO I=764,680,-1
         W2=W2+FAIRHD(1,I)*SIN(FAIRHD(2,I)*T+FAIRHD(3,I))
      END DO

*  T**3
      W3=0D0
      DO I=784,765,-1
         W3=W3+FAIRHD(1,I)*SIN(FAIRHD(2,I)*T+FAIRHD(3,I))
      END DO

*  T**4
      W4=0D0
      DO I=787,785,-1
         W4=W4+FAIRHD(1,I)*SIN(FAIRHD(2,I)*T+FAIRHD(3,I))
      END DO

*  Multiply by powers of T and combine.
      WF=T*(T*(T*(T*W4+W3)+W2)+W1)+W0

*  Adjustments to use JPL planetary masses instead of IAU.
      WJ=     0.00065D-6  * SIN(   6069.776754D0   *T + 4.021194D0   ) +
     :        0.00033D-6  * SIN(    213.299095D0   *T + 5.543132D0   ) +
     :      (-0.00196D-6  * SIN(   6208.294251D0   *T + 5.696701D0   ))+
     :      (-0.00173D-6  * SIN(     74.781599D0   *T + 2.435900D0   ))+
     :        0.03638D-6*T*T

* -----------------------------------------------------------------------

*  Final result:  TDB-TT in seconds.
      slRCC=WT+WF+WJ

      END
