/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 * 
 * Contributions from:
 *     Ryan Wagoner <ryan@wgnrs.dynu.com>, 2006
 *     David Hinkle <hinkle@cipafilter.com>, 2007
 *
 * Released under the GPL v2. */

#include "imspector.h"

#include <libpq-fe.h>

#define PLUGIN_NAME "PostgreSQL IMSpector logging plugin"
#define PLUGIN_SHORT_NAME "PostgreSQL"

#define CREATE_TABLE "CREATE TABLE messages ( " \
	"id serial primary key, " \
	"\"timestamp\" timestamp with time zone default now(), " \
	"clientaddress varchar, " \
	"protocolname varchar, " \
	"outgoing int default 0, " \
	"type int default 0, " \
	"localid varchar, " \
	"remoteid varchar, " \
	"filtered int default 0, " \
	"categories varchar, " \
	"eventdata text )"

#define INSERT_STATEMENT "INSERT INTO messages " \
	"(timestamp, clientaddress, protocolname, outgoing, type, localid, remoteid, filtered, categories, eventdata) " \
	"VALUES (timestamptz 'epoch' + $1 * INTERVAL '1 second', $2, $3, $4, $5, $6, $7, $8, $9, $10)"
#define NO_FIELDS 10

extern "C"
{
	bool initloggingplugin(struct loggingplugininfo &ploggingplugininfo,
		class Options &options, bool debugmode);
	void closeloggingplugin(void);
	int logevents(std::vector<struct imevent> &imevents);
};

PGconn *conn = NULL;

char timestamp[STRING_SIZE];
char clientaddress[STRING_SIZE]; 
char protocolname[STRING_SIZE]; 
char outgoing[STRING_SIZE];
char type[STRING_SIZE];
char localid[STRING_SIZE]; 
char remoteid[STRING_SIZE]; 
char filtered[STRING_SIZE];
char categories[STRING_SIZE];
char eventdata[BUFFER_SIZE]; 

const char *paramvalues[NO_FIELDS] = { timestamp, clientaddress, protocolname, outgoing, type, localid, remoteid, filtered, categories, eventdata };

bool localdebugmode = false;
bool connected = false;
int retries = 0;

std::string connect_string;

std::vector<struct imevent> pgsqlevents;

bool connectpgsql(void);

bool initloggingplugin(struct loggingplugininfo &loggingplugininfo,
	class Options &options, bool debugmode)
{
	connect_string = options["pgsql_connect"];

	if (connect_string.empty()) return false;

	localdebugmode = debugmode;

	loggingplugininfo.pluginname = PLUGIN_NAME;
	
	return connected = connectpgsql();
}

void closeloggingplugin(void)
{
	PQfinish(conn);
	conn = NULL;

	return;
}

/* The main plugin function. See loggingplugin.cpp. */
int logevents(std::vector<struct imevent> &imevents)
{	
	PGresult *res;

	/* store imevents locally in case of sql connection error */
	for (std::vector<struct imevent>::iterator i = imevents.begin(); i != imevents.end(); i++)
		pgsqlevents.push_back(*i);

	/* if not connected try and connect */
	if (!connected)
	{
		retries++;
		
		if ((retries <= 2) || ((retries % 10) == 0))
		{
			if ((connected = connectpgsql()))
			{
				syslog(LOG_NOTICE, PLUGIN_SHORT_NAME ": Reconnected to database, "\
					"pending events will now be logged");
				retries = 0;
			}
			/* still not able to connect, user will get the connection attempt errors,
			 * lets not fill log with uneeded error messages */
			else
			{
				debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Still not able to connect", retries);
				return 0;
			}
		}
		else
		{
			debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Connection to server dead; " \
				"queued events: %d retries: %d", pgsqlevents.size(), retries);
			return 0;
		}
	}
	
	/* try and process local imevents */
	while (pgsqlevents.size())
	{
		struct imevent imevent = pgsqlevents.front();
			
		snprintf(timestamp, STRING_SIZE, "%ld", imevent.timestamp);
		strncpy(clientaddress, imevent.clientaddress.c_str(), STRING_SIZE - 1);
		strncpy(protocolname, imevent.protocolname.c_str(), STRING_SIZE - 1);
		snprintf(outgoing, STRING_SIZE, "%d", imevent.outgoing);
		snprintf(type, STRING_SIZE, "%d", imevent.type);
		strncpy(localid, imevent.localid.c_str(), STRING_SIZE - 1);
		strncpy(remoteid, imevent.remoteid.c_str(), STRING_SIZE - 1);
		snprintf(filtered, STRING_SIZE, "%d", imevent.filtered);
		strncpy(categories, imevent.categories.c_str(), STRING_SIZE - 1);
		strncpy(eventdata, imevent.eventdata.c_str(), BUFFER_SIZE - 1);

		/* we're connected insert the data */
		if (connected)
		{
			debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Connected, so logging one event");

			res = PQexecParams(conn, INSERT_STATEMENT,
				NO_FIELDS,       /* nine param */
				NULL,            /* let the backend deduce param type */
				paramvalues,
				NULL,            /* don't need param lengths since text */
				NULL,            /* default to all text params */
				0);              /* ask for binary results */
				
			if (PQresultStatus(res) != PGRES_COMMAND_OK) 
			{
				syslog(LOG_ERR, PLUGIN_SHORT_NAME ": PQexecParams(), Error: %s", PQerrorMessage(conn));
					
				PQclear(res);
				PQfinish(conn);
				conn = NULL;
				connected = false;
				
				debugprint(localdebugmode, PLUGIN_SHORT_NAME ": Connection lost");
				
				return 1;
			}
			else
			{
				PQclear(res);
				pgsqlevents.erase(pgsqlevents.begin());
			}
		}
	}

	return 0;
}

bool connectpgsql(void)
{
	PGresult *res;

	conn = PQconnectdb(connect_string.c_str());
	
	/* Check to see that the backend connection was successfully made */
	if (PQstatus(conn) != CONNECTION_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't connect to database, Error: %s", PQerrorMessage(conn));
		PQfinish(conn);
		conn = NULL;
		return false;
	}

	res = PQexec(conn, "SELECT tablename FROM pg_tables WHERE tablename='messages';");

	if (PQresultStatus(res) != PGRES_TUPLES_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": PQexec(), Error: %s", PQerrorMessage(conn));
		PQclear(res);
		PQfinish(conn);
		conn = NULL;
		return false;
	}

	if (PQntuples(res) == 1)
	{
		PQclear(res);
		return true;
	}

	PQclear(res);

	res = PQexec(conn, CREATE_TABLE);

	if (PQresultStatus(res) != PGRES_COMMAND_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't create table, Error: %s", PQerrorMessage(conn));
		PQclear(res);
		PQfinish(conn);
		conn = NULL;
		return false;
	}

	PQclear(res);
	
	return true;
}
