/* rcfile.c

   Functions for use with the XML resource file facility.

   Copyright (C) 2002 Tim Stadelmann

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */



/** HEADERS **/

/* autoconf header */
#if HAVE_CONFIG_H
#  include <config.h>
#endif

/* ISO C headers */
#include <errno.h>
#include <string.h>

/* POSIX headers */
#include <sys/types.h>
#include <sys/wait.h>

/* UNIX headers */
#include <iconv.h>
#include <langinfo.h>

/* XML headers */
#include <libxml/parser.h>
#include <libxml/tree.h>

/* program headers */
#include "hotswap.h"
#include "rcfile.h"

#if ENABLE_NLS
#  include <libintl.h>
#  define _(Text) gettext (Text)
#else
#  define _(Text) Text
#endif

/* the path of the resource file */
#ifndef RCFILE
#define RCFILE "/etc/hotswaprc"
#endif /* RCFILE */



/** GLOBAL VARIABLES **/

static xmlDocPtr doc;	        /* a pointer to the document structure */
static xmlNodePtr root;		/* a pointer to the root node */
static iconv_t descriptor;	/* for data set conversion */
static char *buffer = NULL;     /* a buffer for charset conversion */
static size_t size = 0;		/* the buffer size */
static const size_t block = 256; /* the allocation block size */



/** FUNCTIONS **/

/* Load and parse the resource file if it exists.  */

void
init_rcfile ()
{
  /* Open a descriptor for converting from UTF-8 to the current
     character set.  */
  descriptor = iconv_open (nl_langinfo (CODESET), "UTF-8");

  if (settings.verbose)
    {
      printf ("%s: ", program_name);
      printf (_("trying to open the resource file (%s)\n"), RCFILE);
    }

  /* Try to open the resource file.  */
  doc = xmlParseFile (RCFILE);

  if (doc)
    {
      if (settings.verbose)
	{
	  printf ("%s: ", program_name);
	  puts (_("reading the resource file"));
	}
      root = xmlDocGetRootElement (doc);
    }
}


/* Find the first node with the given name in the list.  Return NULL
   if no node was found.  Note that the names of the tags are all
   valid UTF-8 strings that do not require conversion. */

xmlNodePtr
find_node (xmlNodePtr list, xmlChar *name)
{
  xmlNodePtr i;			/* an iterator */

  i = list;
  while (i != NULL)
    { 
      if (strcmp (i->name, name) == 0)
	break;
      i = i->next;
    }
  return i;
}


/* Get the named script from the record with the given device name.
   Return 0 if no script was found.  Note that the names of the tags
   are all valid UTF-8 strings and do not require conversion.  */

int
run_script (char *device, script_t script)
{
  xmlNodePtr i;			/* an iterator */
  xmlNodePtr name_node;		/* points to a <name> node */
  xmlNodePtr script_node;	/* points to a script node */
  xmlChar *contents;		/* the contents of a name node */
  xmlChar *script_name;		/* the name of the requested script */
  unsigned int offset = 0;	/* the offset of the device string */
  int found;			/* whether the device was found */
  pid_t pid;			/* the process ID of the forked instance */
  char *in, *out;
  size_t in_left, out_left;

  /* Return immediately if no resource file was found.  */
  if (!doc)
    return 0;

  /* Determine the name of the script. */
  switch (script)
    {
    case HOTSWAP_POST_INSERT:
      script_name = "post-insert";
      break;
    case HOTSWAP_PRE_REMOVE:
      script_name = "pre-remove";
      break;
    case HOTSWAP_POST_REMOVE:
      script_name = "post-remove";
      break;
    default:
      fprintf (stderr, "%s: ", program_name);
      fputs (_("invalid script requested\n"), stderr);
      exit (EXIT_FAILURE);
    }

  /* Find the correct device node.  */
  i = find_node(root->xmlChildrenNode, "device");
  while (i != NULL)
    {
      name_node = find_node (i->xmlChildrenNode, "name");
      if (!name_node) continue;
      
      /* Get the contents of the name node.  */
      contents = xmlNodeListGetString (doc, name_node->xmlChildrenNode, 1);
   
      /* Deal with white space at the start of contents.  */
      while (isspace (contents[offset])) offset++;
      
      /* Compare with the device name, which is (reasonably) assumed
	 to be ASCII and therefore valid UTF-8.  The use of strncmp
	 deals with white space at the end of contents.  A flag is
	 used to make it easy to free the string at the right
	 point.  */
      found = strncmp (contents + offset, device, strlen (device)) == 0;
      xmlFree (contents);

      if (found)
	break;

      i = find_node (i->next, "device");
    }
  
  /* Indicate that no suitable device node was found.  */
  if (!i) return 0;

  /* Find the correct script node.  */
  script_node = find_node (i->xmlChildrenNode, script_name);
      
  /* Indicate that no script node was found.  */
  if (!script_node) return 0;

  /* Get the contents of the script node.  */
  contents = xmlNodeListGetString (doc, script_node->xmlChildrenNode, 1);

  /* Convert the contents from UTF-8 to the character set of the
     current locale using iconv.  Allocate block additional bytes
     whenever there is not enough space left in the conversion
     buffer.  */

  in = contents;
  in_left = strlen (contents) + 1; /* Account for the trailing '\0'.  */
  out = buffer;
  out_left = size;

  while (in_left > 0)
    {
      if (out_left == 0)
	{
	  size += block;
	  out_left += block;
	  buffer = realloc (buffer, size);
	  if (!buffer)
	    {
	      fprintf (stderr, "%s: ", program_name);
	      perror (_("could not convert script\n"));
	      exit (EXIT_FAILURE);
	    }
	  out = buffer + size - out_left;
	}
      iconv (descriptor, &in, &in_left, &out, &out_left);
    }

  /* Free the original string.  */
  xmlFree (contents);

  /* Execute the script.  */
  pid = fork ();
  if (pid == -1)
    {
      fprintf (stderr, "%s: ", program_name);
      perror (_("could not execute script\n"));
      exit (EXIT_FAILURE);
    }
  else if (pid == 0)
    { 
      execl ("/bin/sh", "/bin/sh", "-c", buffer, NULL);

      /* only called if execl failed */
      fprintf (stderr, "%s: ", program_name);
      fputs (_("could not execute script\n"), stderr);
      exit (EXIT_FAILURE);
    }
  else 
    {
      waitpid (pid, NULL, 0);
    }
      
  return 1;
}
