/* -*- mode: C++; c-basic-offset: 2; indent-tabs-mode: nil -*- */
/*
 *  Main authors:
 *     Stefano Gualandi <stefano.gualandi@gmail.com>
 *     Christian Schulte <schulte@gecode.org>
 *
 *  Copyright:
 *     Stefano Gualandi, 2013
 *     Christian Schulte, 2014
 *
 *  Last modified:
 *     $Date: 2015-03-17 16:09:39 +0100 (Tue, 17 Mar 2015) $ by $Author: schulte $
 *     $Revision: 14447 $
 *
 *  This file is part of Gecode, the generic constraint
 *  development environment:
 *     http://www.gecode.org
 *
 *  Permission is hereby granted, free of charge, to any person obtaining
 *  a copy of this software and associated documentation files (the
 *  "Software"), to deal in the Software without restriction, including
 *  without limitation the rights to use, copy, modify, merge, publish,
 *  distribute, sublicense, and/or sell copies of the Software, and to
 *  permit persons to whom the Software is furnished to do so, subject to
 *  the following conditions:
 *
 *  The above copyright notice and this permission notice shall be
 *  included in all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 *  LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 *  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#include <gecode/driver.hh>

#include <gecode/int.hh>

using namespace Gecode;

/// Instance data
namespace {

  /// Instances
  extern const int* mbpp[];
  /// Instance names
  extern const char* name[];

  /// A wrapper class for instance data
  class Spec {
  protected:
    /// Raw instance data
    const int* data;
    /// Instance name
    const char* n;
    /// Find instance by name \a s
    static const int* find(const char* s) {
      for (int i=0; name[i] != NULL; i++)
        if (!strcmp(s,name[i]))
          return mbpp[i];
      return NULL;
    }
  public:
    /// Initialize
    Spec(const char* s) : data(find(s)), n(s) {}
    /// Whether a valid specification has been found
    bool valid(void) const {
      return data != NULL;
    }
    /// Return dimension
    int dimension(void) const {
      return data[0];
    }
    /// Return instance name
    const char* iname(void) const {
      return n;
    }
    /// Return number of items
    int items(void) const {
      return data[1];
    }
    /// Return number of bins
    int bins(void) const {
      return data[2];
    }
    /// Return item size for item \a i in dimension \a k
    int size(int i, int k) const {
      return data[3+i*dimension()+k];
    }
    /// Return maximal capacity of bin \a j
    int capacity(int j) const {
      return data[3+items()*dimension()+j];
    }
  };

}

/**
 * \brief %Example: Multi-dimensional bin packing
 *
 * \ingroup Example
 *
 */
class MultiBinPacking : public Script {
protected:
  /// Specification
  const Spec spec;
  /// Bin for each item
  IntVarArray bin;  
  /// Load for each bin
  IntVarArray load;
public:
  /// Model variants
  enum {
    MODEL_SINGLE, ///< Use single-dimensional bin-packing constraints
    MODEL_MULTI   ///< Use multi-dimensional bin-packing constraint
  };
  /// Branching to use for the model
  enum {
    BRANCH_DEGREE, ///< Use degree for branching
    BRANCH_AFC     ///< Use AFC for branching
  };
  /// Symmetry breaking to use for the model
  enum {
    SYMMETRY_NONE,        ///< Do not use symmetry breaking
    SYMMETRY_MAX_CONFLICT ///< Use a maximal conflict set of items
  };
  /// Actual model
  MultiBinPacking(const InstanceOptions& opt) 
    : Script(opt), spec(opt.instance()),
      bin(*this, spec.items(), 0, spec.bins()-1), 
      load(*this, spec.bins()*spec.dimension(), 0, Int::Limits::max) 
  { 

    switch (opt.model()) {
    case MODEL_SINGLE:
      {
        // Constrain load variables in every dimension
        for (int k=0; k<spec.dimension(); k++)
          for (int j=0; j<spec.bins(); j++)
            rel(*this, load[j*spec.dimension()+k] <= spec.capacity(k));

        // Post bin-packing constraint for each dimension
        for (int k=0; k<spec.dimension(); k++) {
          IntVarArgs l(spec.bins());
          for (int j=0; j<spec.bins(); j++)
            l[j]=load[j*spec.dimension()+k];
          IntArgs s(spec.items());
          for (int i=0; i<spec.items(); i++)
            s[i]=spec.size(i,k);

          binpacking(*this, l, bin, s);
        }
      }
      break;
    case MODEL_MULTI:
      {
        // Every bin has the same capacity for a given dimension
        IntArgs c(spec.dimension());
        for (int k=0; k<spec.dimension(); k++)
          c[k] = spec.capacity(k);
        
        // Item size for each dimension
        IntArgs s(spec.items()*spec.dimension());
        for (int i=0; i<spec.items(); i++) 
          for (int k=0; k<spec.dimension(); k++) 
            s[i*spec.dimension()+k] = spec.size(i,k);
        
        // Post multidimensional binpacking constraint
        IntSet mc = binpacking(*this, spec.dimension(), load, bin, s, c);
        
        // Symmetry breaking
        if (opt.symmetry() == SYMMETRY_MAX_CONFLICT) {
          // Use a maximal conflict set by assigning items to fixed bins
          IntVarArgs bc(mc.size());
          int j=0;
          for (IntSetValues i(mc); i(); ++i)
            bc[j++] = bin[i.val()];
          assign(*this, bc, INT_ASSIGN_MIN());
        }
      }
      break;
    }

    // Branching strategy 
    switch (opt.branching()) {
    case BRANCH_DEGREE:
      branch(*this, bin, INT_VAR_DEGREE_SIZE_MAX(), INT_VAL_MIN());
      break;
    case BRANCH_AFC:
      branch(*this, bin, INT_VAR_AFC_SIZE_MAX(opt.decay()), INT_VAL_MIN());
      break;
    }
  }
  /// Constructor for cloning \a s
  MultiBinPacking(bool share, MultiBinPacking& s) 
    : Script(share,s), spec(s.spec) {
    bin.update(*this, share, s.bin);
    load.update(*this, share, s.load);
  }
  /// Perform copying during cloning
  virtual Space* copy(bool share) {
    return new MultiBinPacking(share, *this);
  }
  /// Print solution
  virtual void
  print(std::ostream& os) const {
    os << "\tInstance: " << spec.iname() << std::endl;
    if (bin.assigned()) {
      for (int j=0; j<spec.bins(); j++) {
        os << "\tbin[" << j << "]: ";
        for (int i=0; i<spec.items(); i++)
          if (bin[i].val() == j)
            os << i << " ";
        os << "\n\t\tloads: ";
        for (int k=0; k<spec.dimension(); k++)
          os << load[j*spec.dimension()+k] << " ";
        os << std::endl;
      }
    } else {
      os << "\tbins: " << bin << std::endl
         << "\tload: " << load << std::endl;
    }
  }
};

/** \brief Main-function
 *  \relates MultiBinPacking
 */
int
main(int argc, char* argv[]) {
  InstanceOptions opt("MultiBinPacking");
  opt.model(MultiBinPacking::MODEL_MULTI);
  opt.model(MultiBinPacking::MODEL_SINGLE, "single", 
            "use single-dimensional bin-packing constraints");
  opt.model(MultiBinPacking::MODEL_MULTI, "multi", 
            "use single-dimensional bin-packing constraints");
  opt.branching(MultiBinPacking::BRANCH_AFC);
  opt.branching(MultiBinPacking::BRANCH_DEGREE, "degree");
  opt.branching(MultiBinPacking::BRANCH_AFC, "afc");
  opt.decay(0.99);
  opt.symmetry(MultiBinPacking::SYMMETRY_MAX_CONFLICT);
  opt.symmetry(MultiBinPacking::SYMMETRY_NONE, "none");
  opt.symmetry(MultiBinPacking::SYMMETRY_MAX_CONFLICT, "max-conflict");
  opt.instance("6-18-6-20_1");
  opt.solutions(1);
  opt.parse(argc,argv);
  if (!Spec(opt.instance()).valid()) {
    std::cerr << "Error: unkown instance" << std::endl;
    return 1;
  }
  Script::run<MultiBinPacking,DFS,InstanceOptions>(opt);
  return 0;
}

namespace {

  /*
   * Instances are taken from:
   * B. Kell and W.-J. van Hoeve: An MDD approach to multidimensional
   * bin packing. In CP-AI-OR 2013, pages 128-143.
   *
   * The data set is slightly simplified as all bins have
   * the same capacity.
   *
   */

  const int i6_18_6_0_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    991, 82, 474, 622, 924, 336,
    352, 812, 283, 319, 931, 640,
    784, 371, 3, 889, 405, 439,
    913, 3, 703, 815, 678, 939,
    8, 67, 17, 203, 122, 373,
    947, 17, 69, 227, 840, 1,
    102, 321, 670, 49, 719, 845,
    929, 903, 575, 69, 502, 814,
    716, 361, 24, 57, 263, 427,
    608, 885, 998, 454, 824, 292,
    359, 111, 442, 28, 813, 817,
    328, 632, 926, 899, 622, 677,
    124, 587, 370, 198, 763, 625,
    372, 276, 11, 709, 146, 930,
    15, 391, 252, 302, 758, 262,
    980, 869, 37, 234, 830, 353,
    937, 65, 977, 897, 644, 573,
    886, 74, 972, 180, 375, 121,
    // Bin capacity
    1726, 1138, 1301, 1192, 1860, 1578
  };
  const int i6_18_6_1_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    606, 299, 157, 660, 727, 103,
    529, 203, 712, 116, 661, 342,
    65, 129, 520, 886, 411, 492,
    957, 236, 895, 479, 979, 434,
    576, 259, 441, 644, 459, 963,
    873, 346, 878, 345, 631, 633,
    365, 464, 673, 908, 574, 810,
    558, 66, 703, 265, 178, 418,
    86, 500, 251, 786, 799, 716,
    456, 66, 998, 835, 937, 645,
    736, 892, 369, 193, 671, 894,
    782, 319, 941, 876, 481, 585,
    595, 998, 787, 656, 29, 742,
    122, 254, 644, 691, 919, 403,
    734, 970, 193, 726, 832, 415,
    944, 805, 439, 60, 345, 745,
    295, 341, 93, 415, 826, 742,
    458, 556, 580, 226, 158, 574,
    // Bin capacity
    1640, 1297, 1730, 1645, 1788, 1794
  };
  const int i6_18_6_2_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    692, 649, 382, 962, 680, 271,
    468, 850, 406, 147, 508, 718,
    387, 273, 95, 278, 419, 698,
    571, 433, 997, 702, 581, 271,
    30, 620, 792, 13, 192, 719,
    930, 429, 435, 956, 830, 258,
    939, 879, 183, 616, 277, 430,
    649, 582, 695, 471, 416, 737,
    864, 519, 188, 526, 325, 737,
    108, 556, 469, 121, 692, 700,
    113, 774, 613, 769, 310, 571,
    979, 765, 219, 418, 564, 9,
    412, 991, 270, 993, 595, 434,
    459, 44, 820, 37, 12, 496,
    103, 913, 662, 603, 210, 339,
    925, 708, 644, 204, 632, 710,
    17, 990, 640, 135, 236, 105,
    368, 139, 725, 741, 884, 52,
    // Bin capacity
    1533, 1890, 1570, 1478, 1422, 1404
  };
  const int i6_18_6_3_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    795, 146, 36, 947, 622, 134,
    995, 787, 204, 826, 57, 842,
    605, 624, 592, 547, 669, 651,
    59, 273, 768, 35, 727, 625,
    824, 149, 906, 547, 438, 617,
    644, 446, 259, 702, 254, 903,
    952, 483, 7, 802, 965, 88,
    477, 317, 243, 25, 605, 554,
    626, 832, 620, 816, 196, 386,
    443, 787, 585, 965, 339, 295,
    653, 478, 483, 478, 388, 130,
    705, 601, 268, 249, 251, 493,
    67, 272, 773, 178, 753, 359,
    367, 83, 595, 409, 285, 187,
    273, 465, 703, 829, 810, 600,
    273, 481, 311, 306, 725, 215,
    81, 67, 601, 517, 673, 132,
    229, 311, 194, 272, 588, 706,
    // Bin capacity
    1557, 1306, 1399, 1623, 1605, 1360
  };
  const int i6_18_6_4_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    545, 359, 497, 530, 866, 890,
    680, 981, 580, 952, 582, 32,
    481, 136, 246, 128, 867, 530,
    402, 166, 537, 605, 58, 788,
    705, 120, 298, 715, 796, 260,
    688, 223, 913, 518, 195, 382,
    943, 205, 999, 447, 204, 69,
    338, 252, 164, 880, 962, 470,
    401, 995, 793, 949, 359, 142,
    621, 653, 360, 319, 120, 121,
    484, 720, 59, 20, 609, 26,
    295, 319, 171, 297, 62, 492,
    591, 804, 727, 753, 710, 100,
    167, 911, 166, 226, 878, 20,
    748, 482, 563, 699, 422, 682,
    781, 170, 456, 99, 754, 119,
    577, 541, 736, 744, 445, 687,
    91, 346, 345, 72, 325, 490,
    // Bin capacity
    1654, 1454, 1493, 1552, 1598, 1092
  };
  const int i6_18_6_5_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    294, 179, 774, 547, 633, 354,
    419, 494, 985, 667, 532, 36,
    49, 190, 51, 617, 539, 362,
    969, 129, 379, 67, 854, 537,
    92, 898, 91, 919, 511, 211,
    246, 21, 119, 785, 522, 264,
    190, 197, 429, 568, 109, 290,
    112, 76, 434, 613, 26, 463,
    353, 90, 563, 80, 375, 516,
    74, 519, 173, 99, 491, 530,
    615, 396, 803, 586, 563, 792,
    47, 299, 598, 398, 479, 29,
    526, 287, 657, 915, 883, 615,
    626, 792, 953, 982, 542, 608,
    117, 24, 879, 26, 906, 304,
    256, 44, 425, 739, 743, 996,
    921, 666, 728, 105, 930, 832,
    654, 971, 541, 244, 291, 470,
    // Bin capacity
    1148, 1098, 1677, 1568, 1738, 1437
  };
  const int i6_18_6_6_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    430, 737, 558, 543, 17, 578,
    186, 276, 217, 264, 312, 577,
    819, 148, 818, 690, 462, 654,
    952, 200, 283, 440, 417, 239,
    854, 218, 190, 884, 94, 99,
    583, 769, 838, 732, 421, 306,
    189, 374, 744, 315, 690, 956,
    39, 536, 327, 513, 702, 285,
    770, 137, 832, 429, 352, 384,
    574, 902, 497, 462, 598, 578,
    788, 871, 424, 811, 249, 742,
    317, 421, 894, 680, 176, 106,
    671, 15, 480, 834, 808, 118,
    315, 182, 706, 10, 664, 713,
    318, 638, 339, 238, 963, 23,
    126, 751, 644, 12, 361, 179,
    698, 833, 600, 269, 288, 240,
    42, 722, 402, 256, 911, 380,
    // Bin capacity
    1532, 1543, 1731, 1481, 1500, 1265
  };
  const int i6_18_6_7_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    963, 590, 95, 369, 523, 249,
    6, 820, 263, 228, 70, 798,
    58, 195, 844, 334, 214, 777,
    790, 31, 448, 917, 932, 858,
    767, 98, 91, 297, 207, 598,
    376, 273, 519, 313, 732, 83,
    733, 45, 871, 812, 735, 13,
    515, 541, 953, 88, 858, 124,
    625, 500, 977, 476, 990, 192,
    445, 406, 628, 73, 234, 838,
    959, 773, 551, 463, 753, 111,
    445, 934, 689, 321, 256, 473,
    910, 468, 958, 353, 910, 729,
    645, 824, 181, 287, 373, 578,
    494, 779, 788, 933, 599, 655,
    11, 48, 605, 655, 728, 440,
    211, 416, 258, 864, 374, 655,
    834, 855, 946, 439, 882, 770,
    // Bin capacity
    1746, 1533, 1902, 1467, 1850, 1595
  };
  const int i6_18_6_8_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    298, 499, 960, 874, 113, 531,
    888, 102, 212, 612, 76, 55,
    35, 33, 336, 623, 10, 927,
    700, 932, 292, 409, 734, 783,
    402, 136, 686, 523, 744, 455,
    252, 678, 417, 809, 769, 110,
    642, 330, 903, 404, 161, 524,
    14, 453, 297, 96, 755, 386,
    599, 908, 14, 132, 97, 660,
    76, 371, 674, 395, 845, 23,
    485, 804, 757, 985, 469, 252,
    255, 241, 453, 518, 522, 272,
    350, 650, 429, 214, 52, 407,
    139, 825, 123, 189, 503, 11,
    520, 232, 769, 699, 371, 154,
    675, 108, 296, 572, 805, 50,
    985, 306, 631, 998, 582, 335,
    17, 37, 355, 432, 107, 274,
    // Bin capacity
    1320, 1377, 1549, 1708, 1389, 1118
  };
  const int i6_18_6_9_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    254, 282, 179, 506, 480, 193,
    940, 262, 86, 853, 581, 448,
    301, 360, 369, 928, 747, 554,
    305, 390, 593, 222, 170, 60,
    368, 563, 123, 844, 480, 77,
    775, 501, 444, 268, 788, 764,
    298, 420, 713, 728, 95, 528,
    107, 693, 925, 365, 674, 617,
    631, 435, 99, 650, 94, 627,
    111, 944, 568, 217, 459, 962,
    215, 877, 623, 749, 464, 418,
    232, 565, 214, 188, 849, 110,
    360, 664, 715, 141, 554, 877,
    492, 375, 967, 237, 996, 650,
    366, 344, 314, 120, 787, 943,
    425, 48, 374, 818, 722, 700,
    237, 705, 114, 811, 729, 195,
    847, 669, 270, 349, 921, 413,
    // Bin capacity
    1320, 1653, 1398, 1634, 1924, 1660
  };
  const int i6_18_6_10_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    945, 260, 752, 199, 42, 153,
    230, 411, 49, 631, 535, 307,
    37, 319, 78, 699, 461, 303,
    614, 978, 570, 398, 885, 939,
    558, 699, 822, 305, 576, 226,
    0, 80, 483, 456, 198, 750,
    208, 982, 481, 92, 80, 350,
    334, 244, 551, 28, 14, 918,
    426, 855, 236, 819, 901, 276,
    3, 346, 797, 601, 824, 250,
    744, 720, 637, 4, 666, 457,
    704, 278, 295, 1000, 628, 611,
    299, 309, 649, 499, 806, 767,
    47, 39, 705, 332, 39, 556,
    694, 223, 405, 709, 618, 709,
    549, 675, 698, 167, 940, 65,
    953, 404, 76, 440, 964, 361,
    586, 130, 2, 815, 677, 947,
    // Bin capacity
    1455, 1458, 1520, 1503, 1807, 1640
  };
  const int i6_18_6_11_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    339, 803, 618, 313, 368, 790,
    732, 547, 695, 931, 227, 811,
    329, 17, 714, 76, 222, 524,
    923, 325, 186, 144, 374, 477,
    337, 479, 418, 246, 940, 451,
    830, 342, 259, 419, 848, 841,
    124, 607, 203, 808, 983, 441,
    147, 785, 557, 458, 41, 509,
    554, 716, 914, 647, 591, 652,
    235, 449, 89, 97, 534, 697,
    312, 493, 113, 569, 132, 320,
    438, 682, 47, 576, 477, 344,
    450, 728, 360, 3, 893, 964,
    770, 386, 886, 631, 591, 318,
    91, 632, 558, 933, 725, 484,
    33, 906, 168, 666, 25, 574,
    384, 467, 154, 283, 406, 238,
    696, 226, 364, 623, 968, 457,
    // Bin capacity
    1429, 1775, 1352, 1559, 1729, 1831
  };
  const int i6_18_6_12_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    492, 458, 243, 739, 22, 164,
    900, 584, 950, 169, 248, 923,
    771, 760, 65, 193, 467, 545,
    480, 433, 662, 38, 219, 315,
    980, 59, 393, 684, 634, 592,
    862, 440, 478, 831, 595, 416,
    383, 735, 202, 822, 691, 112,
    552, 189, 470, 382, 77, 815,
    596, 528, 792, 788, 298, 380,
    514, 603, 247, 631, 706, 409,
    652, 640, 346, 305, 599, 982,
    983, 630, 41, 748, 262, 984,
    288, 474, 308, 758, 467, 173,
    211, 456, 588, 838, 843, 925,
    771, 532, 654, 717, 979, 7,
    563, 624, 39, 400, 885, 591,
    100, 899, 642, 391, 609, 673,
    362, 982, 613, 195, 440, 83,
    // Bin capacity
    1953, 1872, 1444, 1798, 1688, 1697
  };
  const int i6_18_6_13_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    650, 470, 484, 350, 49, 146,
    425, 391, 549, 399, 974, 976,
    644, 772, 979, 219, 296, 163,
    475, 744, 391, 857, 990, 517,
    198, 292, 310, 64, 393, 487,
    729, 936, 369, 4, 587, 354,
    558, 847, 423, 634, 696, 162,
    344, 642, 690, 853, 278, 858,
    421, 973, 27, 672, 483, 593,
    935, 431, 163, 761, 62, 840,
    467, 454, 959, 384, 623, 723,
    383, 134, 576, 491, 828, 961,
    904, 264, 661, 434, 761, 275,
    508, 864, 306, 492, 852, 753,
    836, 829, 480, 317, 669, 55,
    82, 166, 344, 274, 559, 519,
    647, 616, 644, 230, 49, 662,
    835, 736, 619, 536, 889, 758,
    // Bin capacity
    1892, 1989, 1691, 1502, 1891, 1847
  };
  const int i6_18_6_14_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    900, 24, 475, 673, 485, 464,
    468, 431, 956, 430, 690, 319,
    451, 160, 585, 812, 264, 374,
    719, 605, 643, 12, 897, 662,
    126, 726, 299, 256, 743, 415,
    557, 169, 498, 671, 571, 436,
    779, 819, 377, 174, 416, 371,
    931, 845, 185, 667, 327, 343,
    227, 133, 803, 468, 992, 899,
    872, 576, 937, 406, 432, 977,
    777, 563, 537, 55, 424, 989,
    383, 295, 49, 265, 696, 611,
    412, 598, 588, 611, 852, 78,
    160, 298, 976, 449, 714, 306,
    313, 597, 121, 13, 302, 771,
    914, 933, 264, 868, 391, 449,
    537, 291, 195, 278, 463, 469,
    723, 206, 129, 991, 972, 658,
    // Bin capacity
    1948, 1572, 1638, 1539, 2020, 1823
  };
  const int i6_18_6_15_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    291, 579, 193, 983, 677, 512,
    82, 403, 516, 239, 429, 41,
    398, 859, 425, 43, 617, 699,
    882, 189, 296, 878, 407, 454,
    537, 783, 134, 273, 785, 269,
    195, 564, 29, 11, 797, 252,
    528, 32, 56, 734, 892, 271,
    304, 810, 435, 940, 590, 539,
    286, 589, 926, 445, 401, 530,
    203, 505, 561, 151, 185, 523,
    451, 347, 272, 436, 887, 497,
    694, 119, 413, 275, 500, 691,
    762, 651, 344, 479, 111, 353,
    590, 411, 15, 125, 253, 18,
    425, 175, 816, 335, 5, 313,
    953, 598, 118, 34, 178, 911,
    563, 666, 134, 866, 93, 265,
    951, 869, 11, 21, 773, 523,
    // Bin capacity
    1744, 1754, 1092, 1394, 1645, 1469
  };
  const int i6_18_6_16_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    619, 881, 1000, 13, 847, 654,
    133, 684, 176, 825, 406, 276,
    453, 518, 903, 613, 818, 425,
    333, 226, 528, 655, 465, 465,
    779, 261, 850, 989, 823, 343,
    778, 403, 272, 110, 941, 852,
    604, 696, 672, 246, 578, 473,
    742, 902, 428, 111, 329, 99,
    885, 457, 897, 477, 790, 159,
    925, 358, 60, 570, 498, 456,
    672, 181, 790, 415, 11, 925,
    774, 635, 890, 154, 667, 897,
    794, 869, 460, 42, 707, 90,
    427, 315, 318, 370, 466, 472,
    221, 769, 678, 174, 195, 437,
    863, 677, 145, 261, 0, 283,
    235, 768, 631, 538, 43, 768,
    401, 804, 530, 774, 566, 50,
    // Bin capacity
    2057, 2012, 1978, 1419, 1769, 1571
  };
  const int i6_18_6_17_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    729, 11, 541, 133, 222, 40,
    540, 154, 747, 621, 396, 625,
    165, 417, 696, 21, 152, 941,
    879, 650, 579, 666, 931, 420,
    302, 772, 242, 792, 539, 290,
    711, 468, 760, 666, 315, 259,
    652, 532, 325, 328, 381, 230,
    517, 184, 761, 646, 925, 612,
    790, 103, 616, 613, 146, 155,
    500, 333, 700, 414, 886, 405,
    361, 87, 820, 341, 542, 522,
    881, 114, 173, 176, 260, 439,
    695, 587, 72, 780, 322, 188,
    47, 833, 744, 546, 924, 385,
    1, 441, 514, 72, 849, 875,
    519, 930, 576, 988, 425, 875,
    446, 241, 241, 812, 68, 375,
    441, 869, 455, 857, 345, 886,
    // Bin capacity
    1790, 1507, 1865, 1848, 1683, 1662
  };
  const int i6_18_6_18_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    197, 983, 458, 968, 613, 528,
    333, 318, 658, 102, 297, 280,
    677, 234, 91, 787, 538, 699,
    614, 279, 861, 164, 579, 886,
    607, 49, 312, 346, 160, 422,
    557, 641, 878, 192, 586, 572,
    128, 383, 674, 468, 278, 519,
    320, 572, 912, 516, 539, 417,
    962, 68, 491, 733, 545, 546,
    629, 669, 195, 317, 762, 511,
    242, 583, 617, 81, 945, 160,
    419, 593, 445, 42, 70, 207,
    387, 907, 318, 240, 844, 920,
    663, 27, 475, 560, 375, 247,
    990, 346, 541, 945, 269, 485,
    257, 760, 443, 68, 381, 88,
    579, 35, 48, 997, 336, 984,
    632, 216, 874, 316, 386, 873,
    // Bin capacity
    1808, 1508, 1828, 1543, 1673, 1838
  };
  const int i6_18_6_19_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    759, 567, 742, 67, 663, 195,
    948, 443, 790, 541, 940, 829,
    258, 505, 708, 810, 125, 48,
    477, 51, 980, 81, 30, 574,
    402, 760, 669, 187, 962, 2,
    942, 728, 64, 261, 684, 581,
    62, 764, 973, 647, 34, 561,
    29, 194, 421, 420, 939, 746,
    826, 978, 757, 886, 752, 80,
    690, 49, 188, 782, 977, 939,
    386, 945, 180, 370, 515, 629,
    182, 759, 256, 965, 318, 206,
    632, 478, 777, 98, 8, 668,
    907, 438, 505, 744, 869, 669,
    177, 143, 188, 803, 425, 762,
    925, 155, 511, 259, 782, 280,
    40, 980, 962, 615, 518, 320,
    1, 544, 211, 973, 590, 238,
    // Bin capacity
    1715, 1881, 1960, 1886, 2010, 1652
  };
  const int i6_18_6_20_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    907, 356, 772, 517, 721, 511,
    403, 916, 467, 470, 678, 666,
    611, 791, 111, 47, 82, 304,
    847, 771, 778, 432, 798, 17,
    966, 224, 90, 312, 339, 765,
    407, 961, 105, 677, 298, 868,
    472, 845, 973, 844, 547, 144,
    806, 453, 572, 487, 211, 608,
    729, 87, 363, 94, 958, 483,
    193, 203, 380, 841, 731, 178,
    184, 789, 957, 568, 565, 725,
    321, 761, 744, 72, 377, 963,
    818, 543, 475, 809, 725, 379,
    719, 903, 353, 279, 469, 791,
    704, 690, 610, 95, 506, 46,
    632, 181, 351, 941, 570, 242,
    770, 482, 258, 408, 472, 308,
    953, 598, 78, 524, 860, 188,
    // Bin capacity
    2289, 2111, 1688, 1684, 1982, 1638
  };
  const int i6_18_6_21_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    22, 461, 365, 769, 971, 120,
    719, 915, 743, 882, 384, 375,
    725, 636, 496, 309, 371, 358,
    319, 527, 318, 718, 675, 465,
    994, 273, 299, 797, 181, 345,
    620, 749, 814, 436, 659, 552,
    413, 517, 119, 69, 394, 698,
    933, 862, 473, 706, 647, 574,
    914, 713, 274, 561, 335, 263,
    509, 622, 974, 150, 195, 990,
    199, 497, 369, 30, 231, 818,
    44, 983, 769, 677, 556, 887,
    294, 104, 760, 454, 950, 436,
    69, 251, 376, 994, 601, 479,
    201, 817, 528, 13, 210, 478,
    156, 294, 945, 166, 786, 281,
    519, 849, 806, 783, 379, 272,
    968, 400, 593, 21, 948, 739,
    // Bin capacity
    1738, 2112, 2021, 1722, 1911, 1842
  };
  const int i6_18_6_22_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    958, 221, 299, 408, 898, 857,
    838, 595, 894, 673, 329, 871,
    438, 536, 293, 632, 743, 964,
    269, 216, 20, 173, 101, 721,
    133, 209, 926, 881, 778, 883,
    615, 780, 823, 477, 808, 595,
    563, 421, 538, 232, 98, 61,
    229, 444, 96, 754, 181, 173,
    883, 820, 528, 363, 241, 545,
    960, 935, 599, 967, 313, 417,
    579, 411, 70, 697, 136, 624,
    761, 635, 611, 352, 825, 973,
    94, 986, 577, 453, 997, 884,
    491, 796, 798, 473, 87, 221,
    513, 704, 597, 539, 987, 948,
    153, 906, 288, 184, 830, 609,
    341, 477, 450, 424, 600, 788,
    980, 10, 88, 879, 458, 261,
    // Bin capacity
    1993, 2055, 1728, 1945, 1914, 2317
  };
  const int i6_18_6_23_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    846, 393, 178, 983, 449, 157,
    400, 295, 605, 469, 589, 135,
    191, 478, 602, 675, 400, 973,
    521, 43, 569, 224, 637, 655,
    292, 269, 836, 162, 656, 713,
    284, 456, 82, 929, 740, 931,
    90, 247, 419, 704, 429, 10,
    535, 517, 444, 167, 55, 411,
    598, 598, 95, 669, 976, 378,
    253, 700, 228, 851, 893, 280,
    951, 944, 949, 956, 875, 898,
    528, 270, 120, 24, 186, 605,
    835, 739, 923, 743, 319, 603,
    173, 250, 156, 739, 667, 70,
    638, 159, 694, 841, 705, 371,
    536, 416, 870, 48, 945, 245,
    39, 222, 877, 456, 663, 371,
    356, 527, 747, 736, 548, 850,
    // Bin capacity
    1654, 1543, 1926, 2128, 2201, 1775
  };
  const int i6_18_6_24_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    108, 872, 818, 921, 832, 845,
    829, 109, 636, 302, 204, 525,
    112, 186, 962, 123, 754, 118,
    126, 773, 45, 924, 170, 992,
    903, 76, 252, 470, 899, 928,
    464, 337, 982, 33, 847, 324,
    71, 548, 188, 344, 891, 883,
    385, 788, 944, 838, 80, 33,
    967, 223, 766, 800, 935, 812,
    71, 121, 813, 888, 144, 439,
    201, 695, 906, 136, 487, 89,
    607, 896, 731, 697, 912, 939,
    740, 315, 258, 327, 2, 4,
    238, 213, 75, 795, 774, 190,
    4, 745, 310, 537, 135, 623,
    629, 154, 666, 864, 754, 649,
    416, 711, 249, 774, 1, 767,
    637, 735, 199, 390, 959, 546,
    // Bin capacity
    1552, 1757, 2026, 2101, 2022, 2006
  };
  const int i6_18_6_25_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    92, 357, 902, 532, 569, 283,
    709, 811, 685, 856, 756, 380,
    386, 350, 757, 202, 880, 805,
    535, 484, 487, 371, 756, 683,
    474, 243, 70, 916, 274, 104,
    545, 324, 74, 35, 725, 177,
    627, 282, 939, 4, 23, 541,
    632, 877, 842, 441, 99, 785,
    954, 139, 694, 877, 717, 568,
    759, 83, 343, 155, 633, 746,
    592, 54, 730, 251, 454, 188,
    359, 762, 180, 885, 961, 209,
    770, 386, 52, 753, 254, 434,
    774, 49, 842, 45, 959, 367,
    799, 323, 534, 996, 902, 875,
    424, 597, 498, 182, 116, 524,
    720, 137, 993, 99, 806, 380,
    475, 780, 269, 573, 535, 87,
    // Bin capacity
    2214, 1467, 2061, 1703, 2171, 1695
  };
  const int i6_18_6_26_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    654, 979, 151, 693, 680, 371,
    749, 843, 472, 460, 325, 535,
    306, 578, 222, 889, 958, 551,
    211, 369, 489, 741, 292, 144,
    43, 2, 94, 15, 233, 851,
    486, 384, 403, 295, 5, 658,
    397, 381, 925, 401, 984, 11,
    264, 921, 646, 411, 528, 51,
    178, 462, 421, 237, 466, 666,
    312, 25, 212, 847, 769, 358,
    111, 824, 112, 451, 649, 220,
    739, 190, 764, 509, 207, 864,
    527, 26, 124, 921, 905, 117,
    559, 560, 273, 892, 268, 45,
    97, 693, 651, 194, 557, 424,
    744, 820, 173, 603, 248, 888,
    807, 709, 91, 332, 457, 310,
    636, 260, 473, 675, 12, 608,
    // Bin capacity
    1643, 1896, 1407, 2009, 1795, 1612
  };
  const int i6_18_6_27_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    151, 567, 277, 937, 977, 667,
    512, 270, 269, 379, 275, 951,
    706, 852, 700, 78, 850, 857,
    585, 987, 546, 242, 937, 130,
    430, 762, 125, 798, 507, 136,
    647, 619, 168, 708, 7, 244,
    106, 468, 771, 251, 832, 628,
    367, 248, 3, 733, 33, 379,
    251, 230, 796, 514, 491, 95,
    28, 230, 853, 731, 92, 87,
    470, 497, 736, 220, 962, 286,
    114, 427, 922, 596, 826, 306,
    429, 754, 873, 458, 433, 697,
    16, 240, 100, 934, 14, 476,
    116, 513, 624, 416, 913, 419,
    159, 63, 819, 204, 662, 439,
    215, 547, 581, 339, 336, 492,
    698, 801, 345, 982, 745, 489,
    // Bin capacity
    1270, 1921, 2013, 2016, 2094, 1647
  };
  const int i6_18_6_28_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    498, 502, 643, 960, 882, 83,
    833, 913, 324, 948, 578, 359,
    135, 898, 436, 557, 881, 526,
    628, 241, 952, 523, 108, 148,
    431, 674, 738, 977, 839, 581,
    633, 669, 804, 437, 325, 321,
    609, 961, 516, 720, 778, 432,
    95, 939, 591, 404, 407, 655,
    556, 545, 525, 939, 201, 347,
    942, 621, 272, 290, 412, 174,
    86, 645, 110, 637, 75, 21,
    259, 510, 503, 119, 954, 716,
    518, 439, 339, 328, 858, 305,
    700, 461, 819, 468, 575, 871,
    773, 644, 75, 921, 733, 999,
    186, 963, 405, 806, 197, 257,
    41, 457, 222, 831, 878, 623,
    509, 156, 886, 616, 401, 575,
    // Bin capacity
    1799, 2398, 1955, 2450, 2151, 1706
  };
  const int i6_18_6_29_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    78, 659, 168, 160, 493, 553,
    912, 339, 765, 103, 471, 699,
    314, 803, 523, 917, 721, 169,
    846, 441, 680, 235, 6, 758,
    596, 697, 351, 880, 677, 751,
    688, 761, 873, 854, 516, 720,
    352, 401, 102, 888, 1000, 84,
    284, 792, 205, 893, 789, 113,
    525, 893, 993, 758, 18, 28,
    981, 515, 247, 416, 318, 486,
    67, 616, 865, 775, 745, 422,
    209, 657, 899, 397, 364, 926,
    272, 549, 467, 443, 88, 872,
    672, 759, 793, 530, 778, 932,
    461, 10, 889, 803, 471, 915,
    133, 279, 25, 818, 825, 838,
    243, 436, 368, 580, 447, 616,
    709, 299, 14, 987, 98, 420,
    // Bin capacity
    1794, 2130, 1984, 2459, 1898, 2215
  };
  const int i6_18_6_30_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    650, 338, 439, 878, 704, 249,
    953, 615, 529, 158, 79, 286,
    324, 451, 390, 177, 378, 7,
    716, 812, 945, 6, 411, 900,
    456, 184, 432, 969, 334, 671,
    741, 726, 914, 178, 382, 255,
    763, 353, 988, 164, 353, 192,
    161, 745, 675, 833, 665, 672,
    26, 121, 880, 740, 328, 897,
    503, 294, 303, 362, 567, 694,
    472, 700, 847, 880, 707, 722,
    309, 750, 345, 908, 408, 566,
    655, 781, 329, 787, 921, 1000,
    239, 418, 54, 815, 613, 704,
    132, 320, 284, 914, 747, 891,
    698, 764, 883, 830, 597, 806,
    205, 342, 748, 611, 935, 357,
    785, 368, 790, 344, 883, 27,
    // Bin capacity
    1905, 1968, 2335, 2287, 2170, 2145
  };
  const int i6_18_6_31_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    899, 893, 402, 91, 784, 766,
    892, 380, 660, 771, 731, 91,
    503, 750, 534, 448, 502, 959,
    998, 360, 874, 396, 332, 220,
    900, 65, 436, 37, 992, 602,
    135, 49, 274, 11, 29, 984,
    872, 825, 787, 502, 488, 187,
    332, 929, 451, 733, 159, 313,
    563, 415, 595, 787, 26, 60,
    948, 997, 579, 720, 183, 2,
    212, 640, 588, 148, 853, 356,
    977, 32, 829, 109, 691, 525,
    867, 217, 170, 774, 389, 753,
    690, 480, 439, 406, 622, 206,
    988, 887, 551, 598, 204, 418,
    556, 442, 73, 668, 379, 338,
    628, 949, 993, 638, 964, 44,
    276, 461, 327, 148, 705, 968,
    // Bin capacity
    2672, 2134, 2088, 1744, 1973, 1702
  };
  const int i6_18_6_32_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    18, 364, 626, 223, 884, 92,
    859, 974, 695, 357, 926, 147,
    268, 840, 273, 212, 820, 92,
    765, 541, 732, 834, 980, 88,
    711, 439, 427, 972, 418, 988,
    817, 248, 331, 887, 1, 505,
    72, 950, 766, 840, 46, 989,
    527, 800, 406, 308, 763, 402,
    219, 270, 850, 360, 668, 119,
    794, 938, 365, 699, 299, 916,
    170, 730, 289, 952, 924, 121,
    712, 510, 368, 401, 374, 505,
    500, 207, 558, 512, 372, 740,
    474, 929, 86, 13, 488, 10,
    175, 543, 412, 167, 397, 671,
    32, 225, 237, 590, 239, 584,
    420, 67, 400, 913, 389, 770,
    747, 910, 848, 351, 512, 564,
    // Bin capacity
    1822, 2307, 1908, 2111, 2090, 1827
  };
  const int i6_18_6_33_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    901, 676, 531, 49, 849, 635,
    295, 215, 807, 363, 769, 393,
    231, 488, 478, 87, 760, 496,
    159, 95, 693, 296, 525, 740,
    318, 674, 639, 501, 966, 169,
    437, 956, 75, 456, 900, 134,
    860, 103, 634, 651, 484, 635,
    168, 69, 768, 433, 514, 57,
    84, 593, 759, 922, 934, 197,
    412, 821, 243, 966, 129, 481,
    41, 265, 281, 318, 586, 256,
    381, 207, 988, 343, 338, 937,
    347, 208, 7, 676, 890, 510,
    633, 709, 86, 479, 273, 89,
    290, 819, 127, 754, 647, 788,
    103, 473, 386, 185, 753, 899,
    827, 30, 297, 332, 259, 434,
    974, 480, 539, 110, 780, 674,
    // Bin capacity
    1654, 1747, 1849, 1756, 2518, 1890
  };
  const int i6_18_6_34_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    419, 877, 107, 60, 315, 985,
    488, 881, 221, 104, 210, 539,
    682, 335, 756, 78, 166, 68,
    149, 201, 316, 962, 434, 514,
    266, 466, 905, 856, 459, 307,
    731, 892, 639, 788, 385, 436,
    260, 176, 133, 974, 15, 806,
    670, 830, 357, 492, 586, 24,
    542, 370, 33, 913, 262, 307,
    388, 746, 911, 679, 227, 434,
    597, 834, 18, 806, 675, 652,
    777, 417, 506, 363, 717, 576,
    312, 176, 362, 285, 527, 255,
    645, 590, 901, 599, 622, 950,
    921, 180, 278, 409, 489, 765,
    662, 366, 43, 947, 421, 78,
    356, 40, 65, 761, 59, 344,
    361, 524, 883, 69, 987, 27,
    // Bin capacity
    2061, 1988, 1661, 2266, 1688, 1802
  };
  const int i6_18_6_35_1[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    222, 496, 99, 873, 896, 464,
    26, 425, 718, 387, 834, 928,
    405, 590, 322, 819, 920, 970,
    733, 919, 882, 618, 559, 35,
    283, 812, 394, 405, 560, 875,
    319, 55, 764, 711, 346, 121,
    600, 742, 836, 11, 849, 676,
    139, 765, 478, 664, 822, 374,
    345, 814, 185, 553, 836, 97,
    404, 508, 55, 812, 397, 259,
    631, 948, 225, 353, 143, 490,
    185, 5, 791, 613, 236, 486,
    49, 533, 394, 601, 870, 217,
    20, 740, 71, 442, 979, 343,
    701, 523, 206, 180, 259, 268,
    968, 690, 285, 806, 206, 324,
    957, 275, 677, 441, 502, 318,
    689, 541, 137, 87, 696, 857,
    // Bin capacity
    1728, 2336, 1692, 2110, 2455, 1823
  };
  const int i6_18_6_0_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    209, 410, 822, 447, 354, 129,
    703, 346, 568, 353, 281, 659,
    364, 729, 461, 601, 327, 519,
    440, 587, 622, 887, 697, 606,
    407, 656, 815, 148, 847, 579,
    642, 746, 259, 461, 270, 310,
    702, 619, 384, 997, 2, 342,
    705, 158, 931, 170, 364, 577,
    265, 775, 585, 244, 729, 816,
    255, 137, 914, 158, 369, 891,
    184, 321, 541, 48, 797, 750,
    582, 361, 441, 561, 228, 179,
    122, 76, 994, 530, 17, 553,
    377, 79, 449, 939, 265, 839,
    490, 506, 748, 239, 566, 757,
    464, 767, 897, 891, 775, 408,
    767, 289, 246, 435, 560, 810,
    129, 288, 480, 938, 609, 710,
    // Bin capacity
    1302, 1309, 1860, 1508, 1343, 1739
  };
  const int i6_18_6_1_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    401, 185, 633, 470, 701, 433,
    409, 670, 15, 309, 187, 26,
    901, 211, 359, 517, 806, 320,
    47, 159, 311, 883, 333, 3,
    198, 120, 529, 426, 701, 559,
    263, 113, 463, 981, 135, 51,
    480, 421, 403, 217, 292, 932,
    177, 768, 875, 228, 149, 480,
    351, 985, 153, 264, 501, 990,
    63, 760, 572, 637, 559, 28,
    140, 854, 619, 62, 252, 201,
    512, 644, 755, 267, 125, 471,
    565, 910, 370, 953, 43, 994,
    521, 688, 339, 628, 186, 592,
    685, 400, 533, 406, 958, 459,
    750, 602, 245, 839, 492, 259,
    782, 884, 950, 613, 120, 850,
    161, 166, 464, 864, 112, 661,
    // Bin capacity
    1247, 1606, 1446, 1610, 1120, 1399
  };
  const int i6_18_6_2_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    901, 659, 854, 207, 930, 361,
    920, 457, 422, 242, 884, 596,
    411, 681, 601, 202, 50, 644,
    442, 920, 810, 778, 69, 538,
    998, 606, 460, 233, 529, 700,
    606, 121, 242, 656, 971, 840,
    983, 100, 887, 34, 742, 910,
    75, 910, 485, 668, 174, 444,
    938, 175, 894, 714, 250, 119,
    789, 767, 149, 563, 889, 462,
    95, 115, 458, 471, 402, 572,
    111, 599, 319, 298, 164, 686,
    890, 280, 15, 401, 926, 113,
    67, 917, 602, 923, 567, 767,
    557, 638, 413, 522, 989, 129,
    802, 488, 691, 531, 672, 192,
    694, 252, 970, 222, 139, 139,
    395, 168, 531, 761, 828, 544,
    // Bin capacity
    1815, 1506, 1667, 1433, 1730, 1489
  };
  const int i6_18_6_3_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    446, 873, 636, 603, 793, 868,
    362, 891, 308, 957, 537, 912,
    481, 717, 104, 9, 657, 33,
    907, 561, 923, 421, 423, 774,
    440, 686, 55, 494, 896, 568,
    492, 343, 757, 979, 341, 377,
    53, 10, 738, 299, 429, 523,
    573, 855, 271, 891, 225, 749,
    527, 429, 94, 271, 459, 149,
    959, 764, 665, 35, 733, 349,
    22, 218, 446, 161, 811, 935,
    52, 600, 345, 430, 634, 77,
    839, 148, 541, 517, 317, 316,
    909, 738, 426, 894, 361, 935,
    430, 144, 558, 914, 1, 559,
    195, 144, 288, 198, 783, 294,
    116, 612, 83, 162, 242, 589,
    988, 981, 239, 652, 747, 409,
    // Bin capacity
    1510, 1668, 1284, 1526, 1612, 1617
  };
  const int i6_18_6_4_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    463, 936, 48, 730, 531, 528,
    438, 474, 551, 817, 777, 7,
    266, 292, 240, 861, 934, 159,
    284, 483, 970, 750, 280, 488,
    286, 54, 749, 224, 795, 881,
    410, 426, 0, 365, 318, 990,
    212, 512, 928, 863, 858, 351,
    672, 628, 871, 126, 547, 891,
    292, 693, 967, 410, 670, 4,
    572, 412, 644, 493, 794, 892,
    448, 252, 286, 19, 429, 798,
    817, 254, 374, 424, 11, 987,
    255, 350, 875, 902, 924, 699,
    293, 280, 805, 293, 509, 868,
    540, 816, 884, 752, 831, 578,
    754, 127, 342, 518, 767, 842,
    194, 127, 919, 529, 513, 100,
    906, 500, 583, 952, 832, 580,
    // Bin capacity
    1405, 1321, 1913, 1739, 1963, 1845
  };
  const int i6_18_6_5_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    39, 376, 132, 648, 203, 98,
    78, 457, 240, 39, 378, 855,
    616, 17, 793, 328, 547, 391,
    890, 460, 798, 40, 288, 398,
    526, 291, 999, 911, 254, 794,
    774, 729, 273, 861, 919, 179,
    445, 492, 598, 476, 794, 856,
    958, 812, 77, 862, 110, 476,
    954, 899, 207, 249, 560, 5,
    756, 682, 182, 626, 933, 441,
    374, 675, 913, 385, 223, 922,
    36, 706, 647, 480, 186, 639,
    830, 600, 927, 59, 28, 458,
    305, 688, 86, 978, 191, 649,
    184, 598, 731, 204, 92, 627,
    538, 905, 470, 48, 617, 471,
    359, 659, 542, 703, 448, 360,
    811, 16, 251, 905, 676, 986,
    // Bin capacity
    1658, 1761, 1552, 1541, 1304, 1681
  };
  const int i6_18_6_6_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    125, 970, 593, 359, 3, 601,
    359, 294, 955, 645, 982, 968,
    936, 202, 781, 12, 854, 220,
    484, 776, 846, 110, 378, 459,
    989, 532, 125, 331, 608, 219,
    65, 344, 777, 541, 364, 854,
    414, 411, 835, 301, 0, 286,
    440, 95, 497, 598, 384, 824,
    794, 754, 816, 190, 472, 538,
    687, 375, 265, 459, 689, 458,
    504, 194, 616, 694, 921, 793,
    8, 867, 848, 429, 946, 706,
    951, 917, 851, 542, 920, 357,
    166, 591, 892, 46, 342, 272,
    658, 204, 952, 816, 91, 177,
    114, 159, 488, 8, 220, 858,
    95, 482, 35, 469, 268, 670,
    276, 607, 34, 477, 829, 125,
    // Bin capacity
    1425, 1551, 1980, 1242, 1638, 1659
  };
  const int i6_18_6_7_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    211, 410, 557, 321, 202, 76,
    817, 695, 295, 400, 286, 640,
    757, 973, 749, 522, 182, 200,
    316, 695, 178, 224, 16, 35,
    420, 41, 857, 69, 882, 843,
    152, 89, 318, 218, 16, 742,
    551, 228, 747, 725, 936, 95,
    700, 50, 274, 504, 772, 682,
    983, 875, 690, 423, 888, 430,
    154, 221, 378, 1000, 577, 604,
    89, 548, 111, 538, 677, 353,
    601, 146, 118, 655, 381, 470,
    856, 445, 122, 233, 193, 506,
    633, 147, 875, 33, 814, 940,
    479, 874, 646, 2, 404, 921,
    577, 96, 584, 594, 480, 607,
    516, 900, 669, 908, 919, 706,
    152, 620, 261, 33, 305, 845,
    // Bin capacity
    1599, 1437, 1504, 1321, 1593, 1729
  };
  const int i6_18_6_8_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    297, 115, 877, 777, 482, 446,
    320, 716, 933, 163, 872, 450,
    575, 433, 173, 230, 696, 638,
    615, 115, 247, 292, 69, 856,
    329, 915, 110, 557, 124, 528,
    755, 765, 91, 78, 661, 693,
    540, 510, 585, 803, 930, 284,
    677, 252, 158, 596, 38, 43,
    543, 313, 861, 248, 857, 877,
    577, 898, 943, 865, 99, 931,
    661, 340, 828, 152, 386, 512,
    853, 408, 668, 478, 106, 839,
    161, 673, 391, 843, 687, 864,
    656, 500, 619, 772, 765, 147,
    857, 2, 632, 896, 155, 263,
    973, 587, 910, 649, 668, 206,
    866, 557, 824, 760, 832, 677,
    252, 456, 812, 34, 717, 240,
    // Bin capacity
    1892, 1540, 1920, 1655, 1646, 1709
  };
  const int i6_18_6_9_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    956, 396, 871, 872, 895, 422,
    300, 449, 484, 952, 606, 998,
    224, 551, 743, 1000, 846, 507,
    36, 183, 430, 586, 856, 920,
    231, 108, 286, 488, 823, 934,
    877, 875, 0, 101, 593, 752,
    146, 40, 218, 760, 464, 473,
    474, 767, 488, 535, 678, 401,
    668, 653, 468, 644, 856, 11,
    731, 29, 909, 837, 345, 462,
    824, 723, 435, 836, 351, 11,
    339, 377, 35, 533, 543, 599,
    593, 662, 272, 625, 546, 831,
    724, 981, 809, 344, 723, 469,
    490, 961, 946, 650, 127, 315,
    196, 963, 936, 103, 100, 808,
    435, 361, 648, 130, 305, 595,
    120, 300, 970, 197, 281, 773,
    // Bin capacity
    1520, 1704, 1808, 1852, 1806, 1868
  };
  const int i6_18_6_10_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    981, 379, 497, 718, 771, 950,
    576, 254, 275, 754, 465, 641,
    740, 882, 467, 857, 306, 384,
    737, 804, 454, 498, 521, 525,
    329, 490, 783, 336, 556, 179,
    416, 815, 639, 564, 27, 865,
    282, 479, 0, 414, 710, 886,
    634, 133, 178, 471, 281, 473,
    666, 671, 376, 685, 188, 452,
    842, 57, 893, 53, 20, 111,
    338, 776, 95, 954, 973, 457,
    515, 903, 420, 822, 118, 687,
    59, 1, 18, 103, 224, 580,
    3, 475, 990, 667, 16, 327,
    229, 886, 763, 432, 293, 22,
    110, 861, 891, 481, 828, 684,
    685, 267, 842, 660, 978, 92,
    641, 800, 587, 481, 341, 561,
    // Bin capacity
    1611, 1822, 1681, 1825, 1397, 1628
  };
  const int i6_18_6_11_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    446, 821, 817, 845, 471, 903,
    997, 438, 294, 110, 676, 532,
    201, 708, 373, 846, 65, 411,
    429, 944, 179, 586, 899, 566,
    121, 292, 513, 939, 246, 648,
    773, 581, 277, 255, 987, 796,
    397, 628, 422, 951, 265, 62,
    289, 868, 760, 661, 211, 201,
    682, 668, 81, 411, 548, 883,
    295, 553, 536, 288, 243, 690,
    334, 93, 763, 767, 250, 993,
    706, 769, 275, 402, 632, 4,
    734, 179, 250, 700, 192, 867,
    146, 623, 934, 941, 960, 523,
    495, 154, 526, 781, 390, 153,
    752, 478, 568, 30, 237, 621,
    3, 842, 954, 504, 857, 160,
    288, 957, 117, 883, 359, 907,
    // Bin capacity
    1497, 1961, 1599, 2017, 1571, 1836
  };
  const int i6_18_6_12_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    163, 115, 943, 498, 798, 937,
    738, 141, 443, 643, 749, 684,
    682, 545, 285, 400, 329, 821,
    767, 940, 571, 70, 685, 702,
    183, 570, 813, 594, 46, 462,
    763, 201, 694, 526, 539, 244,
    771, 748, 64, 563, 596, 898,
    326, 285, 122, 19, 902, 453,
    933, 3, 609, 946, 507, 292,
    986, 296, 751, 458, 729, 755,
    117, 560, 371, 367, 68, 27,
    697, 636, 539, 801, 580, 309,
    313, 845, 424, 379, 431, 834,
    284, 123, 105, 648, 208, 700,
    981, 629, 694, 238, 987, 326,
    164, 978, 613, 381, 475, 699,
    594, 801, 724, 393, 110, 369,
    384, 220, 114, 18, 682, 969,
    // Bin capacity
    1838, 1613, 1658, 1483, 1759, 1957
  };
  const int i6_18_6_13_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    990, 600, 168, 747, 810, 280,
    520, 297, 77, 338, 426, 107,
    706, 250, 360, 541, 881, 851,
    24, 892, 539, 690, 377, 382,
    213, 804, 913, 545, 210, 907,
    82, 571, 934, 818, 667, 422,
    956, 535, 924, 955, 908, 628,
    100, 961, 559, 331, 82, 873,
    263, 849, 889, 508, 789, 505,
    283, 488, 271, 90, 152, 243,
    181, 719, 199, 14, 613, 516,
    497, 767, 167, 398, 1000, 597,
    638, 115, 485, 50, 582, 523,
    966, 779, 283, 242, 255, 824,
    81, 151, 534, 253, 525, 453,
    575, 433, 19, 304, 791, 915,
    350, 235, 653, 323, 776, 438,
    215, 614, 875, 54, 107, 636,
    // Bin capacity
    1439, 1895, 1667, 1357, 1875, 1903
  };
  const int i6_18_6_14_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    610, 670, 982, 739, 611, 61,
    388, 582, 874, 145, 861, 208,
    518, 995, 883, 638, 759, 226,
    218, 732, 689, 849, 842, 406,
    344, 523, 16, 139, 768, 167,
    386, 259, 247, 498, 190, 480,
    969, 397, 608, 958, 22, 32,
    866, 819, 106, 481, 577, 126,
    557, 952, 161, 370, 922, 67,
    47, 248, 382, 436, 324, 531,
    295, 210, 778, 878, 303, 253,
    938, 36, 308, 95, 979, 304,
    513, 969, 711, 314, 932, 31,
    338, 226, 703, 45, 351, 195,
    877, 764, 826, 672, 335, 956,
    69, 814, 411, 29, 620, 320,
    595, 103, 174, 417, 45, 35,
    846, 312, 571, 97, 737, 643,
    // Bin capacity
    1782, 1827, 1792, 1483, 1934, 958
  };
  const int i6_18_6_15_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    360, 472, 959, 434, 860, 396,
    493, 488, 320, 383, 355, 921,
    492, 28, 544, 990, 230, 375,
    228, 9, 874, 927, 744, 787,
    60, 313, 713, 456, 112, 411,
    696, 127, 169, 415, 383, 582,
    432, 257, 171, 748, 246, 282,
    233, 850, 126, 193, 171, 133,
    1, 910, 19, 130, 362, 134,
    542, 461, 587, 818, 437, 102,
    133, 980, 848, 230, 915, 538,
    802, 705, 371, 694, 474, 1,
    183, 996, 422, 810, 284, 186,
    512, 238, 378, 14, 232, 880,
    252, 949, 102, 788, 206, 389,
    858, 173, 36, 156, 247, 371,
    678, 460, 991, 11, 65, 863,
    142, 1000, 531, 982, 46, 171,
    // Bin capacity
    1361, 1805, 1565, 1760, 1221, 1442
  };
  const int i6_18_6_16_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    47, 838, 502, 786, 848, 764,
    590, 197, 188, 624, 47, 83,
    371, 626, 915, 578, 393, 989,
    938, 869, 905, 41, 173, 508,
    190, 908, 632, 323, 556, 646,
    122, 158, 571, 882, 808, 483,
    232, 538, 715, 766, 941, 149,
    382, 324, 380, 900, 469, 4,
    515, 724, 377, 754, 815, 625,
    210, 497, 553, 898, 604, 326,
    742, 75, 546, 683, 299, 258,
    984, 265, 275, 969, 475, 887,
    532, 813, 731, 78, 205, 163,
    287, 408, 663, 334, 999, 952,
    913, 274, 802, 606, 446, 140,
    837, 378, 993, 666, 455, 332,
    852, 38, 6, 971, 73, 737,
    530, 346, 169, 36, 738, 14,
    // Bin capacity
    1793, 1601, 1919, 2107, 1807, 1559
  };
  const int i6_18_6_17_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    650, 248, 10, 447, 945, 807,
    501, 359, 114, 437, 370, 694,
    830, 373, 561, 278, 631, 445,
    512, 344, 342, 719, 79, 960,
    65, 125, 433, 812, 766, 680,
    657, 993, 417, 473, 948, 390,
    93, 528, 133, 432, 449, 883,
    538, 688, 379, 810, 237, 432,
    625, 745, 428, 933, 851, 343,
    220, 252, 939, 145, 418, 997,
    625, 2, 748, 316, 139, 293,
    321, 819, 964, 499, 872, 198,
    76, 778, 341, 729, 71, 242,
    246, 539, 665, 385, 35, 989,
    195, 306, 827, 36, 606, 502,
    595, 317, 536, 13, 831, 545,
    111, 872, 155, 256, 308, 878,
    936, 357, 448, 554, 652, 632,
    // Bin capacity
    1521, 1686, 1646, 1614, 1796, 2128
  };
  const int i6_18_6_18_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    351, 685, 254, 934, 254, 50,
    959, 670, 840, 262, 823, 810,
    885, 111, 843, 113, 174, 793,
    832, 227, 252, 610, 171, 154,
    68, 291, 675, 519, 569, 985,
    738, 256, 113, 139, 134, 100,
    889, 312, 130, 858, 265, 847,
    174, 958, 266, 256, 40, 600,
    482, 118, 636, 402, 51, 5,
    383, 853, 899, 802, 111, 354,
    576, 854, 880, 987, 800, 226,
    13, 656, 884, 258, 842, 413,
    484, 248, 328, 213, 202, 956,
    983, 323, 861, 420, 637, 882,
    903, 922, 552, 668, 853, 721,
    556, 896, 678, 472, 392, 268,
    479, 645, 666, 845, 440, 322,
    729, 227, 153, 454, 957, 818,
    // Bin capacity
    2062, 1820, 1949, 1812, 1518, 1830
  };
  const int i6_18_6_19_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    802, 263, 1000, 947, 335, 697,
    223, 498, 165, 383, 883, 693,
    712, 755, 955, 526, 3, 591,
    307, 820, 410, 794, 268, 771,
    86, 553, 263, 188, 438, 839,
    474, 42, 480, 86, 711, 484,
    868, 38, 942, 564, 270, 886,
    124, 126, 614, 461, 901, 71,
    919, 411, 466, 130, 58, 260,
    599, 434, 519, 774, 64, 475,
    244, 972, 683, 230, 768, 371,
    714, 980, 585, 428, 942, 118,
    235, 602, 157, 176, 784, 649,
    844, 177, 843, 131, 969, 329,
    109, 566, 256, 749, 823, 628,
    61, 65, 10, 283, 913, 842,
    934, 725, 305, 952, 580, 183,
    664, 13, 219, 169, 874, 942,
    // Bin capacity
    1769, 1595, 1760, 1581, 2100, 1950
  };
  const int i6_18_6_20_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    801, 925, 825, 138, 791, 510,
    75, 125, 465, 26, 610, 103,
    279, 827, 681, 871, 819, 22,
    855, 171, 953, 709, 108, 506,
    299, 404, 328, 639, 313, 240,
    931, 483, 557, 466, 476, 728,
    784, 704, 591, 625, 698, 584,
    318, 620, 574, 172, 863, 226,
    252, 581, 62, 310, 330, 383,
    433, 247, 716, 806, 825, 577,
    617, 258, 238, 445, 95, 993,
    758, 489, 697, 157, 804, 562,
    635, 381, 336, 486, 23, 989,
    848, 742, 333, 99, 568, 400,
    135, 490, 730, 646, 104, 533,
    957, 212, 636, 439, 3, 679,
    554, 185, 336, 894, 689, 685,
    345, 976, 917, 267, 311, 251,
    // Bin capacity
    1976, 1764, 1995, 1639, 1686, 1795
  };
  const int i6_18_6_21_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    281, 457, 69, 6, 724, 984,
    61, 630, 914, 940, 32, 533,
    205, 401, 381, 818, 143, 967,
    473, 930, 969, 409, 371, 355,
    730, 114, 462, 498, 768, 712,
    564, 766, 742, 446, 721, 750,
    487, 660, 430, 243, 901, 566,
    429, 196, 561, 112, 685, 861,
    351, 95, 786, 622, 710, 215,
    989, 254, 410, 146, 144, 40,
    911, 118, 609, 320, 530, 626,
    609, 786, 739, 492, 238, 3,
    733, 245, 911, 515, 924, 70,
    707, 962, 624, 398, 185, 78,
    805, 140, 395, 180, 605, 213,
    837, 982, 490, 367, 111, 485,
    847, 328, 88, 70, 245, 431,
    96, 325, 808, 778, 19, 864,
    // Bin capacity
    2040, 1692, 2095, 1485, 1625, 1766
  };
  const int i6_18_6_22_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    587, 69, 563, 610, 980, 641,
    128, 756, 242, 584, 842, 115,
    39, 720, 187, 905, 322, 131,
    109, 593, 327, 551, 15, 821,
    26, 119, 430, 482, 657, 160,
    232, 76, 458, 122, 401, 53,
    429, 864, 141, 297, 308, 207,
    905, 770, 118, 39, 187, 467,
    636, 317, 158, 688, 112, 631,
    412, 306, 332, 514, 275, 315,
    455, 505, 670, 39, 709, 826,
    429, 533, 594, 18, 420, 706,
    527, 823, 538, 460, 624, 931,
    360, 49, 526, 538, 282, 987,
    585, 821, 797, 879, 477, 474,
    150, 334, 356, 544, 818, 882,
    805, 518, 280, 418, 690, 654,
    782, 669, 120, 824, 514, 236,
    // Bin capacity
    1545, 1798, 1391, 1731, 1756, 1879
  };
  const int i6_18_6_23_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    507, 628, 383, 101, 676, 506,
    789, 131, 146, 305, 876, 523,
    358, 4, 12, 660, 426, 215,
    74, 941, 342, 218, 282, 149,
    806, 640, 927, 584, 169, 799,
    791, 55, 78, 577, 226, 55,
    372, 441, 538, 342, 922, 132,
    900, 110, 554, 689, 104, 756,
    262, 345, 248, 904, 266, 352,
    530, 225, 910, 697, 894, 206,
    806, 419, 229, 457, 516, 794,
    67, 84, 150, 890, 148, 194,
    688, 388, 300, 237, 820, 78,
    404, 114, 107, 554, 781, 846,
    6, 641, 140, 964, 220, 135,
    76, 889, 385, 536, 134, 172,
    397, 641, 307, 491, 260, 694,
    658, 12, 212, 330, 940, 453,
    // Bin capacity
    1741, 1376, 1224, 1955, 1776, 1448
  };
  const int i6_18_6_24_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    537, 572, 340, 846, 845, 504,
    961, 71, 397, 788, 309, 494,
    256, 423, 630, 477, 372, 424,
    284, 292, 721, 282, 668, 176,
    75, 768, 589, 834, 393, 407,
    200, 183, 249, 842, 222, 785,
    803, 877, 414, 441, 179, 268,
    713, 35, 414, 106, 175, 691,
    683, 674, 208, 203, 962, 668,
    514, 800, 121, 251, 697, 720,
    706, 322, 647, 750, 181, 86,
    510, 712, 342, 576, 907, 794,
    79, 991, 123, 123, 281, 150,
    857, 170, 459, 598, 404, 659,
    947, 990, 405, 561, 341, 456,
    618, 408, 901, 182, 380, 432,
    861, 263, 918, 954, 655, 747,
    836, 13, 749, 724, 999, 225,
    // Bin capacity
    2158, 1770, 1783, 1972, 1854, 1796
  };
  const int i6_18_6_25_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    569, 156, 221, 238, 890, 389,
    660, 970, 668, 786, 16, 949,
    359, 824, 453, 214, 55, 431,
    698, 353, 528, 541, 252, 97,
    746, 289, 172, 100, 164, 254,
    860, 191, 660, 59, 897, 141,
    365, 596, 887, 484, 420, 831,
    105, 744, 120, 124, 711, 331,
    985, 477, 734, 870, 606, 78,
    440, 393, 656, 387, 710, 643,
    808, 835, 846, 387, 755, 489,
    191, 917, 314, 18, 257, 881,
    306, 416, 107, 914, 489, 537,
    663, 283, 369, 208, 96, 796,
    591, 400, 717, 376, 466, 465,
    161, 399, 971, 401, 963, 378,
    931, 305, 834, 127, 528, 628,
    825, 722, 473, 64, 478, 150,
    // Bin capacity
    2139, 1932, 2028, 1313, 1824, 1765
  };
  const int i6_18_6_26_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    126, 923, 47, 320, 303, 739,
    742, 971, 792, 721, 638, 349,
    227, 946, 190, 507, 540, 702,
    48, 973, 217, 887, 764, 108,
    181, 257, 853, 410, 264, 897,
    927, 385, 613, 487, 496, 413,
    226, 301, 953, 751, 849, 730,
    807, 585, 321, 653, 441, 323,
    97, 14, 763, 420, 362, 139,
    534, 706, 21, 676, 559, 893,
    429, 968, 888, 420, 151, 791,
    728, 158, 564, 887, 650, 649,
    217, 590, 34, 521, 475, 926,
    321, 705, 221, 877, 49, 433,
    814, 638, 525, 992, 976, 990,
    629, 129, 923, 32, 104, 582,
    207, 993, 287, 477, 986, 181,
    617, 859, 866, 448, 673, 860,
    // Bin capacity
    1655, 2332, 1907, 2203, 1949, 2249
  };
  const int i6_18_6_27_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    699, 204, 358, 469, 531, 52,
    609, 513, 696, 343, 937, 113,
    31, 527, 430, 447, 370, 254,
    495, 827, 150, 465, 688, 99,
    203, 232, 481, 913, 338, 344,
    174, 533, 336, 2, 335, 550,
    370, 102, 556, 3, 30, 100,
    347, 126, 837, 328, 626, 855,
    65, 753, 184, 302, 456, 401,
    427, 877, 991, 173, 948, 805,
    981, 266, 820, 750, 505, 727,
    159, 490, 154, 747, 795, 967,
    925, 636, 650, 868, 229, 546,
    643, 790, 607, 568, 96, 186,
    245, 150, 977, 529, 371, 70,
    706, 154, 401, 218, 953, 760,
    911, 194, 445, 787, 985, 871,
    360, 583, 279, 552, 16, 615,
    // Bin capacity
    1768, 1685, 1980, 1792, 1950, 1761
  };
  const int i6_18_6_28_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    901, 13, 992, 331, 538, 506,
    183, 700, 332, 151, 260, 733,
    30, 869, 333, 422, 456, 238,
    25, 222, 521, 395, 38, 375,
    837, 528, 697, 943, 230, 517,
    796, 998, 93, 186, 763, 536,
    531, 390, 607, 472, 699, 694,
    748, 266, 801, 829, 637, 6,
    485, 235, 316, 439, 578, 182,
    840, 917, 985, 90, 991, 972,
    101, 82, 598, 228, 381, 363,
    646, 957, 775, 705, 743, 586,
    245, 229, 214, 547, 207, 17,
    536, 868, 346, 319, 611, 781,
    216, 291, 339, 523, 772, 139,
    926, 652, 786, 981, 617, 875,
    363, 152, 626, 582, 245, 136,
    792, 982, 852, 375, 149, 48,
    // Bin capacity
    1963, 1995, 2179, 1818, 1902, 1644
  };
  const int i6_18_6_29_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    546, 463, 231, 198, 176, 562,
    492, 982, 410, 166, 974, 522,
    487, 98, 802, 974, 230, 643,
    6, 726, 82, 652, 598, 519,
    133, 571, 504, 798, 148, 647,
    855, 384, 181, 694, 884, 960,
    644, 527, 950, 883, 949, 431,
    405, 277, 539, 857, 182, 891,
    544, 312, 274, 848, 571, 221,
    275, 943, 452, 77, 23, 87,
    575, 851, 754, 465, 317, 797,
    69, 992, 73, 381, 268, 253,
    801, 440, 68, 873, 227, 986,
    474, 779, 210, 567, 779, 663,
    513, 366, 68, 99, 131, 362,
    632, 637, 776, 379, 912, 510,
    801, 4, 733, 777, 599, 550,
    187, 192, 205, 85, 338, 274,
    // Bin capacity
    1815, 2052, 1573, 2102, 1786, 2124
  };
  const int i6_18_6_30_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    9, 607, 952, 452, 477, 412,
    324, 394, 46, 910, 133, 694,
    330, 301, 602, 354, 169, 54,
    989, 519, 270, 800, 946, 188,
    169, 352, 274, 908, 851, 646,
    459, 876, 490, 887, 424, 13,
    976, 90, 967, 147, 543, 685,
    597, 43, 185, 861, 382, 569,
    741, 417, 104, 96, 233, 415,
    174, 95, 664, 412, 67, 300,
    208, 13, 685, 371, 43, 935,
    264, 557, 755, 424, 556, 738,
    335, 209, 802, 580, 913, 228,
    476, 920, 459, 423, 764, 179,
    251, 707, 77, 363, 454, 389,
    965, 193, 115, 349, 810, 182,
    775, 184, 436, 384, 940, 527,
    63, 600, 878, 21, 700, 264,
    // Bin capacity
    1757, 1534, 1899, 1895, 2038, 1608
  };
  const int i6_18_6_31_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    709, 270, 689, 81, 825, 947,
    598, 263, 542, 11, 232, 787,
    383, 393, 276, 625, 956, 754,
    225, 726, 384, 438, 351, 775,
    237, 430, 282, 206, 802, 788,
    928, 597, 53, 352, 902, 30,
    722, 147, 603, 251, 357, 433,
    278, 491, 237, 337, 184, 150,
    318, 313, 928, 104, 314, 321,
    467, 150, 507, 520, 901, 441,
    911, 413, 418, 609, 621, 538,
    823, 479, 531, 400, 955, 404,
    52, 70, 697, 573, 569, 703,
    489, 604, 143, 925, 621, 811,
    712, 901, 306, 811, 706, 110,
    752, 68, 477, 527, 326, 610,
    760, 1, 653, 322, 591, 67,
    394, 340, 442, 90, 26, 906,
    // Bin capacity
    2131, 1454, 1784, 1569, 2236, 2091
  };
  const int i6_18_6_32_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    48, 365, 880, 381, 585, 58,
    703, 233, 539, 827, 988, 124,
    775, 301, 885, 616, 931, 490,
    213, 210, 988, 501, 396, 866,
    827, 242, 950, 631, 382, 347,
    567, 319, 537, 730, 102, 873,
    760, 726, 272, 486, 661, 433,
    43, 306, 685, 489, 317, 124,
    379, 129, 289, 28, 849, 955,
    231, 764, 90, 571, 283, 94,
    256, 487, 598, 165, 931, 470,
    51, 925, 146, 59, 782, 825,
    66, 489, 709, 831, 213, 982,
    629, 834, 583, 88, 385, 874,
    65, 126, 100, 359, 440, 178,
    853, 619, 479, 385, 630, 958,
    481, 425, 987, 192, 856, 340,
    311, 581, 775, 67, 228, 700,
    // Bin capacity
    1597, 1778, 2309, 1630, 2191, 2133
  };
  const int i6_18_6_33_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    465, 187, 878, 587, 568, 999,
    446, 405, 353, 926, 700, 840,
    896, 600, 876, 441, 951, 650,
    810, 111, 411, 418, 143, 335,
    150, 83, 401, 440, 854, 601,
    770, 676, 364, 994, 950, 486,
    30, 828, 979, 332, 884, 267,
    730, 0, 251, 285, 152, 189,
    651, 668, 843, 726, 873, 989,
    249, 961, 190, 13, 771, 718,
    17, 527, 417, 679, 601, 58,
    628, 159, 715, 689, 92, 812,
    589, 798, 211, 928, 755, 566,
    336, 449, 674, 913, 523, 741,
    584, 273, 607, 489, 783, 272,
    81, 84, 886, 683, 324, 520,
    959, 399, 889, 831, 603, 557,
    397, 324, 81, 546, 731, 27,
    // Bin capacity
    1949, 1670, 2223, 2421, 2496, 2134
  };
  const int i6_18_6_34_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    472, 581, 993, 248, 42, 189,
    543, 598, 297, 932, 553, 489,
    638, 865, 650, 438, 484, 886,
    801, 258, 34, 135, 790, 49,
    415, 878, 101, 526, 500, 242,
    99, 34, 998, 395, 303, 409,
    549, 798, 931, 312, 50, 260,
    787, 791, 953, 245, 71, 432,
    592, 208, 97, 715, 357, 400,
    105, 945, 215, 485, 760, 996,
    993, 187, 675, 796, 877, 736,
    966, 802, 718, 946, 716, 329,
    573, 174, 683, 223, 990, 645,
    208, 334, 630, 789, 998, 292,
    659, 939, 684, 52, 68, 859,
    270, 640, 410, 87, 544, 537,
    173, 554, 978, 909, 566, 84,
    244, 281, 787, 366, 693, 302,
    // Bin capacity
    2030, 2204, 2420, 1921, 2091, 1818
  };
  const int i6_18_6_35_2[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    418, 823, 915, 714, 37, 235,
    243, 733, 408, 875, 693, 866,
    104, 995, 823, 839, 325, 143,
    574, 221, 993, 868, 292, 137,
    137, 850, 385, 811, 823, 378,
    263, 780, 507, 688, 257, 807,
    108, 163, 370, 277, 383, 178,
    858, 107, 575, 853, 183, 848,
    457, 851, 89, 521, 881, 924,
    99, 270, 387, 763, 292, 636,
    218, 209, 413, 519, 22, 839,
    501, 876, 396, 208, 565, 503,
    208, 783, 168, 283, 563, 173,
    288, 476, 80, 834, 487, 600,
    861, 516, 259, 942, 448, 246,
    432, 561, 692, 599, 599, 716,
    130, 445, 124, 775, 615, 579,
    465, 282, 816, 635, 51, 398,
    // Bin capacity
    1432, 2237, 1890, 2701, 1692, 2072
  };
  const int i6_18_6_0_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    374, 863, 238, 117, 660, 997,
    410, 239, 131, 322, 205, 488,
    850, 435, 758, 566, 735, 840,
    660, 473, 732, 153, 980, 882,
    532, 414, 839, 482, 794, 715,
    283, 960, 348, 432, 176, 245,
    146, 3, 169, 934, 833, 275,
    355, 645, 652, 937, 432, 351,
    658, 478, 457, 891, 121, 637,
    895, 805, 396, 243, 650, 182,
    567, 775, 828, 333, 643, 531,
    383, 477, 376, 455, 703, 751,
    74, 645, 593, 630, 895, 209,
    766, 567, 874, 852, 474, 847,
    573, 194, 513, 334, 240, 698,
    429, 646, 281, 536, 219, 543,
    859, 194, 597, 651, 86, 914,
    309, 456, 740, 142, 254, 839,
    // Bin capacity
    1521, 1545, 1587, 1502, 1517, 1824
  };
  const int i6_18_6_1_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    364, 841, 254, 357, 712, 145,
    20, 271, 852, 317, 962, 120,
    729, 152, 582, 302, 610, 145,
    322, 997, 227, 981, 631, 646,
    470, 50, 145, 897, 67, 280,
    32, 362, 640, 87, 191, 704,
    324, 661, 656, 842, 565, 696,
    753, 735, 954, 605, 978, 499,
    856, 789, 915, 938, 130, 353,
    772, 744, 778, 689, 760, 876,
    493, 253, 180, 462, 710, 415,
    817, 416, 269, 718, 151, 593,
    438, 898, 653, 191, 508, 893,
    343, 594, 440, 494, 142, 349,
    855, 597, 551, 994, 794, 251,
    174, 657, 325, 896, 963, 770,
    424, 976, 725, 632, 480, 647,
    36, 571, 233, 14, 954, 707,
    // Bin capacity
    1385, 1779, 1579, 1754, 1736, 1530
  };
  const int i6_18_6_2_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    114, 116, 556, 916, 598, 363,
    651, 800, 489, 804, 663, 742,
    727, 987, 391, 460, 710, 572,
    940, 130, 320, 238, 992, 651,
    455, 924, 351, 771, 666, 922,
    49, 730, 892, 731, 382, 40,
    557, 966, 411, 69, 407, 152,
    788, 185, 722, 31, 388, 652,
    959, 871, 120, 332, 840, 539,
    143, 536, 15, 836, 155, 816,
    548, 274, 612, 653, 924, 360,
    872, 574, 812, 280, 147, 173,
    102, 803, 973, 524, 76, 575,
    34, 713, 350, 779, 251, 4,
    678, 280, 120, 813, 72, 703,
    265, 759, 179, 113, 669, 449,
    159, 65, 5, 93, 340, 489,
    770, 905, 376, 899, 422, 63,
    // Bin capacity
    1498, 1806, 1308, 1589, 1480, 1406
  };
  const int i6_18_6_3_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    322, 137, 898, 198, 822, 190,
    750, 332, 192, 818, 756, 608,
    110, 138, 321, 738, 674, 91,
    29, 518, 819, 752, 989, 40,
    11, 673, 351, 25, 622, 614,
    147, 238, 847, 249, 806, 361,
    332, 277, 341, 971, 853, 97,
    728, 292, 41, 20, 22, 147,
    573, 753, 861, 895, 507, 505,
    712, 882, 996, 253, 836, 555,
    246, 833, 963, 513, 10, 328,
    854, 991, 529, 976, 551, 40,
    961, 570, 44, 3, 325, 851,
    538, 903, 270, 515, 513, 19,
    941, 60, 282, 714, 543, 143,
    302, 506, 940, 174, 97, 421,
    854, 516, 114, 484, 459, 741,
    669, 298, 185, 59, 597, 258,
    // Bin capacity
    1559, 1531, 1544, 1435, 1714, 1032
  };
  const int i6_18_6_4_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    777, 654, 46, 322, 293, 912,
    460, 804, 473, 981, 202, 363,
    339, 396, 255, 703, 50, 859,
    771, 448, 284, 374, 848, 534,
    372, 241, 570, 827, 568, 199,
    486, 638, 167, 155, 385, 858,
    425, 139, 579, 628, 793, 466,
    502, 309, 380, 710, 431, 19,
    385, 758, 429, 702, 491, 761,
    168, 816, 369, 665, 910, 435,
    920, 978, 999, 823, 723, 931,
    211, 670, 519, 929, 56, 125,
    750, 615, 523, 852, 635, 883,
    693, 675, 171, 541, 292, 722,
    822, 193, 996, 709, 798, 903,
    900, 721, 485, 934, 284, 64,
    65, 223, 804, 906, 485, 467,
    278, 860, 44, 884, 585, 378,
    // Bin capacity
    1617, 1758, 1403, 2192, 1531, 1713
  };
  const int i6_18_6_5_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    109, 443, 923, 478, 869, 913,
    503, 685, 596, 696, 71, 77,
    391, 313, 444, 566, 523, 394,
    528, 971, 538, 105, 648, 835,
    857, 97, 587, 769, 226, 504,
    882, 465, 163, 975, 589, 599,
    134, 572, 381, 224, 904, 69,
    28, 736, 216, 919, 685, 506,
    381, 623, 782, 938, 594, 283,
    642, 877, 248, 81, 867, 751,
    302, 325, 974, 817, 173, 333,
    252, 566, 924, 747, 623, 446,
    164, 540, 37, 587, 411, 937,
    417, 766, 827, 923, 710, 873,
    841, 785, 141, 604, 324, 982,
    917, 428, 74, 860, 863, 384,
    204, 827, 706, 301, 10, 306,
    193, 256, 949, 350, 221, 580,
    // Bin capacity
    1356, 1799, 1665, 1915, 1630, 1711
  };
  const int i6_18_6_6_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    285, 716, 695, 272, 828, 991,
    930, 569, 933, 113, 892, 254,
    207, 15, 865, 341, 362, 973,
    485, 831, 881, 293, 200, 1,
    221, 936, 23, 180, 65, 166,
    905, 606, 509, 632, 543, 793,
    444, 420, 708, 440, 143, 239,
    280, 287, 138, 466, 531, 287,
    619, 915, 27, 612, 122, 727,
    792, 1000, 647, 956, 591, 259,
    844, 569, 294, 439, 92, 20,
    731, 149, 709, 301, 23, 486,
    68, 620, 749, 665, 29, 778,
    730, 577, 559, 716, 763, 86,
    618, 377, 104, 17, 974, 719,
    288, 361, 783, 598, 735, 120,
    592, 486, 598, 795, 868, 263,
    29, 771, 104, 178, 94, 852,
    // Bin capacity
    1603, 1803, 1648, 1416, 1388, 1416
  };
  const int i6_18_6_7_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    241, 889, 543, 341, 903, 69,
    604, 744, 732, 571, 144, 548,
    165, 780, 993, 141, 763, 916,
    362, 635, 340, 957, 756, 816,
    679, 754, 142, 983, 243, 296,
    361, 127, 355, 289, 129, 422,
    774, 760, 589, 929, 470, 895,
    133, 317, 503, 137, 395, 577,
    439, 153, 760, 159, 680, 54,
    497, 655, 318, 157, 409, 829,
    336, 399, 393, 572, 270, 509,
    196, 496, 627, 521, 335, 663,
    759, 312, 521, 829, 612, 262,
    876, 419, 788, 909, 646, 461,
    891, 340, 42, 176, 673, 892,
    553, 124, 459, 496, 76, 966,
    997, 514, 715, 310, 3, 21,
    302, 763, 965, 374, 324, 585,
    // Bin capacity
    1635, 1638, 1745, 1579, 1397, 1745
  };
  const int i6_18_6_8_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    619, 375, 555, 131, 701, 501,
    288, 649, 767, 334, 211, 483,
    439, 208, 34, 241, 909, 67,
    662, 312, 39, 111, 550, 855,
    440, 911, 157, 346, 720, 604,
    701, 384, 691, 140, 844, 615,
    781, 836, 961, 840, 255, 771,
    362, 660, 655, 894, 849, 477,
    322, 98, 633, 594, 712, 877,
    851, 739, 919, 417, 102, 643,
    197, 913, 695, 925, 347, 535,
    138, 633, 274, 226, 16, 410,
    771, 879, 514, 95, 164, 594,
    857, 300, 188, 154, 722, 206,
    669, 72, 547, 927, 742, 571,
    771, 164, 561, 238, 763, 37,
    94, 65, 622, 644, 587, 276,
    542, 243, 928, 991, 572, 699,
    // Bin capacity
    1711, 1520, 1754, 1485, 1758, 1660
  };
  const int i6_18_6_9_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    311, 50, 663, 455, 274, 35,
    156, 193, 697, 787, 527, 335,
    16, 766, 55, 284, 470, 318,
    653, 12, 639, 71, 964, 511,
    699, 271, 261, 123, 272, 621,
    904, 197, 208, 81, 477, 20,
    619, 953, 815, 273, 908, 460,
    124, 613, 818, 688, 734, 20,
    29, 913, 276, 660, 854, 226,
    779, 438, 615, 640, 620, 494,
    962, 796, 12, 85, 632, 6,
    422, 943, 784, 348, 717, 290,
    352, 187, 625, 543, 382, 148,
    637, 259, 762, 314, 746, 27,
    224, 611, 921, 331, 122, 421,
    988, 889, 424, 867, 391, 257,
    53, 402, 671, 994, 253, 867,
    906, 864, 929, 960, 967, 884,
    // Bin capacity
    1605, 1700, 1849, 1545, 1873, 1080
  };
  const int i6_18_6_10_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    140, 14, 929, 57, 999, 255,
    448, 182, 200, 633, 971, 596,
    919, 133, 430, 685, 582, 659,
    754, 711, 985, 967, 439, 845,
    313, 71, 267, 988, 811, 300,
    497, 203, 36, 709, 83, 642,
    540, 980, 376, 670, 624, 559,
    831, 829, 630, 672, 630, 992,
    377, 862, 93, 720, 602, 214,
    212, 426, 497, 373, 248, 881,
    469, 181, 455, 129, 643, 910,
    190, 326, 852, 677, 973, 349,
    984, 512, 691, 870, 502, 114,
    306, 705, 959, 106, 269, 517,
    139, 16, 436, 673, 726, 938,
    557, 285, 508, 86, 543, 349,
    828, 791, 384, 592, 608, 777,
    309, 98, 4, 309, 874, 908,
    // Bin capacity
    1616, 1343, 1601, 1818, 2040, 1981
  };
  const int i6_18_6_11_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    683, 927, 693, 955, 110, 394,
    825, 503, 624, 25, 373, 559,
    2, 205, 474, 556, 282, 484,
    792, 567, 923, 334, 574, 129,
    532, 700, 569, 162, 201, 214,
    113, 444, 829, 862, 413, 44,
    857, 665, 457, 511, 847, 253,
    207, 5, 387, 266, 109, 598,
    229, 8, 221, 153, 29, 798,
    681, 864, 469, 801, 56, 969,
    937, 634, 386, 406, 837, 325,
    559, 771, 95, 558, 216, 734,
    896, 794, 481, 648, 85, 845,
    317, 49, 469, 180, 171, 24,
    117, 329, 256, 1, 161, 278,
    58, 652, 816, 478, 617, 563,
    288, 515, 195, 759, 681, 632,
    882, 68, 916, 281, 87, 201,
    // Bin capacity
    1661, 1610, 1714, 1469, 1083, 1489
  };
  const int i6_18_6_12_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    429, 264, 589, 359, 97, 956,
    991, 599, 52, 96, 651, 61,
    587, 860, 2, 112, 842, 790,
    735, 854, 833, 528, 104, 735,
    543, 239, 84, 156, 944, 859,
    62, 49, 910, 764, 584, 823,
    690, 980, 901, 848, 58, 59,
    854, 163, 636, 595, 787, 51,
    112, 202, 505, 755, 865, 406,
    558, 556, 446, 225, 522, 581,
    248, 992, 802, 708, 979, 583,
    900, 110, 637, 37, 71, 543,
    744, 154, 838, 47, 499, 125,
    94, 557, 855, 183, 33, 163,
    995, 588, 61, 659, 589, 76,
    359, 778, 807, 118, 592, 292,
    174, 431, 409, 115, 841, 277,
    501, 79, 31, 840, 464, 174,
    // Bin capacity
    1788, 1579, 1755, 1334, 1778, 1411
  };
  const int i6_18_6_13_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    409, 227, 436, 514, 99, 495,
    737, 782, 833, 720, 139, 659,
    594, 117, 941, 160, 47, 258,
    542, 243, 706, 271, 630, 492,
    444, 409, 794, 761, 789, 586,
    440, 196, 331, 156, 231, 849,
    579, 213, 595, 265, 316, 908,
    162, 609, 990, 780, 952, 620,
    16, 675, 522, 644, 254, 549,
    660, 507, 652, 133, 174, 129,
    42, 656, 566, 693, 332, 713,
    4, 619, 707, 684, 51, 146,
    979, 206, 849, 965, 646, 815,
    25, 188, 89, 405, 80, 25,
    765, 938, 174, 512, 182, 562,
    218, 745, 396, 407, 84, 12,
    337, 602, 726, 266, 500, 90,
    521, 344, 838, 581, 742, 778,
    // Bin capacity
    1408, 1559, 2099, 1680, 1177, 1636
  };
  const int i6_18_6_14_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    323, 789, 835, 925, 496, 924,
    908, 213, 747, 823, 838, 500,
    294, 904, 180, 995, 966, 94,
    594, 164, 125, 415, 945, 310,
    562, 96, 676, 380, 39, 915,
    544, 717, 715, 223, 510, 492,
    379, 88, 236, 869, 742, 616,
    239, 408, 439, 106, 576, 570,
    349, 455, 737, 535, 898, 106,
    434, 804, 767, 777, 414, 781,
    472, 424, 377, 634, 42, 903,
    555, 751, 389, 921, 64, 133,
    359, 883, 153, 684, 814, 494,
    987, 343, 763, 543, 584, 159,
    848, 875, 192, 769, 320, 525,
    267, 147, 735, 108, 516, 821,
    731, 509, 226, 180, 278, 603,
    955, 691, 202, 263, 382, 320,
    // Bin capacity
    1863, 1760, 1614, 1929, 1791, 1761
  };
  const int i6_18_6_15_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    37, 35, 490, 507, 897, 869,
    813, 827, 258, 335, 548, 822,
    747, 46, 370, 608, 460, 596,
    199, 836, 770, 770, 454, 803,
    70, 113, 650, 676, 67, 302,
    226, 556, 40, 349, 160, 641,
    408, 395, 170, 968, 859, 947,
    569, 505, 639, 400, 834, 439,
    701, 202, 914, 513, 761, 606,
    76, 510, 491, 813, 24, 136,
    657, 7, 800, 223, 698, 29,
    384, 300, 774, 864, 94, 292,
    499, 472, 908, 779, 912, 919,
    825, 515, 690, 790, 380, 870,
    552, 668, 933, 913, 612, 711,
    685, 948, 839, 771, 29, 424,
    649, 294, 217, 445, 899, 574,
    241, 804, 303, 884, 242, 422,
    // Bin capacity
    1599, 1540, 1966, 2225, 1712, 1994
  };
  const int i6_18_6_16_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    666, 539, 270, 616, 566, 583,
    792, 145, 258, 170, 948, 234,
    784, 632, 864, 166, 302, 308,
    554, 193, 172, 563, 770, 894,
    842, 58, 574, 844, 389, 243,
    538, 716, 231, 3, 700, 43,
    834, 854, 116, 266, 772, 715,
    158, 623, 455, 941, 481, 675,
    220, 701, 198, 748, 737, 480,
    152, 269, 746, 677, 847, 971,
    546, 398, 704, 744, 414, 402,
    460, 61, 810, 598, 364, 523,
    234, 45, 857, 965, 660, 293,
    13, 426, 336, 790, 103, 938,
    139, 537, 675, 840, 155, 826,
    35, 632, 220, 21, 746, 21,
    48, 274, 40, 628, 822, 63,
    790, 711, 818, 139, 207, 565,
    // Bin capacity
    1509, 1511, 1614, 1880, 1931, 1697
  };
  const int i6_18_6_17_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    985, 496, 113, 590, 575, 648,
    620, 837, 835, 658, 225, 228,
    529, 248, 132, 902, 957, 302,
    341, 900, 992, 142, 579, 942,
    415, 304, 274, 242, 302, 177,
    248, 653, 355, 94, 442, 118,
    290, 536, 380, 503, 865, 964,
    697, 932, 917, 766, 363, 629,
    407, 93, 744, 142, 781, 791,
    614, 957, 159, 532, 264, 675,
    549, 646, 144, 809, 362, 63,
    187, 849, 905, 67, 447, 236,
    689, 349, 394, 964, 858, 80,
    251, 777, 682, 701, 598, 98,
    709, 470, 960, 41, 595, 927,
    945, 536, 857, 842, 728, 758,
    601, 581, 498, 244, 10, 708,
    306, 936, 691, 203, 148, 124,
    // Bin capacity
    1830, 2165, 1957, 1647, 1775, 1652
  };
  const int i6_18_6_18_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    780, 469, 543, 287, 920, 26,
    678, 319, 939, 829, 987, 563,
    384, 150, 189, 377, 235, 572,
    461, 477, 307, 847, 693, 92,
    854, 323, 183, 707, 192, 150,
    606, 747, 780, 539, 699, 148,
    17, 62, 237, 725, 818, 329,
    159, 265, 535, 310, 590, 956,
    300, 446, 245, 303, 650, 396,
    526, 915, 765, 372, 317, 339,
    670, 962, 116, 584, 634, 69,
    827, 124, 711, 443, 434, 265,
    247, 330, 31, 223, 696, 256,
    346, 226, 460, 671, 950, 229,
    43, 538, 174, 933, 462, 571,
    195, 178, 383, 446, 678, 654,
    993, 365, 524, 947, 310, 164,
    223, 199, 217, 743, 282, 776,
    // Bin capacity
    1635, 1396, 1444, 2023, 2075, 1290
  };
  const int i6_18_6_19_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    744, 294, 794, 780, 259, 107,
    755, 129, 684, 725, 903, 65,
    885, 501, 213, 931, 259, 512,
    628, 874, 946, 560, 736, 645,
    966, 502, 607, 623, 666, 497,
    451, 863, 432, 346, 521, 758,
    216, 235, 245, 316, 816, 564,
    742, 649, 424, 703, 211, 110,
    959, 616, 833, 393, 323, 586,
    770, 977, 883, 989, 631, 243,
    309, 923, 532, 582, 377, 974,
    103, 734, 142, 251, 914, 917,
    465, 183, 159, 368, 606, 903,
    41, 19, 121, 620, 663, 668,
    781, 562, 229, 225, 725, 915,
    832, 474, 744, 728, 333, 781,
    150, 753, 676, 717, 188, 276,
    47, 214, 1, 671, 193, 174,
    // Bin capacity
    1953, 1885, 1719, 2089, 1850, 1923
  };
  const int i6_18_6_20_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    435, 147, 454, 285, 822, 495,
    660, 409, 353, 180, 33, 903,
    618, 293, 493, 529, 791, 455,
    847, 342, 252, 882, 860, 966,
    449, 928, 91, 477, 752, 5,
    981, 591, 298, 762, 860, 691,
    725, 696, 967, 82, 487, 647,
    228, 861, 191, 377, 904, 765,
    369, 191, 585, 645, 254, 838,
    676, 434, 277, 791, 859, 218,
    877, 174, 680, 440, 986, 759,
    475, 107, 881, 566, 807, 60,
    713, 434, 270, 727, 949, 559,
    444, 90, 510, 320, 477, 308,
    133, 836, 337, 78, 70, 145,
    97, 372, 441, 749, 907, 190,
    971, 348, 287, 797, 466, 943,
    395, 586, 34, 750, 595, 188,
    // Bin capacity
    2019, 1568, 1481, 1888, 2376, 1827
  };
  const int i6_18_6_21_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    487, 500, 519, 655, 129, 348,
    863, 332, 217, 628, 250, 155,
    478, 393, 158, 127, 170, 607,
    961, 72, 966, 170, 886, 413,
    164, 780, 687, 430, 583, 870,
    751, 238, 92, 245, 903, 687,
    509, 980, 773, 683, 350, 303,
    404, 351, 719, 89, 233, 530,
    540, 188, 516, 268, 478, 300,
    158, 983, 513, 795, 854, 46,
    921, 956, 558, 236, 821, 660,
    157, 799, 267, 199, 579, 433,
    229, 156, 743, 40, 469, 714,
    640, 886, 333, 307, 569, 311,
    128, 344, 829, 357, 787, 737,
    911, 304, 692, 92, 504, 154,
    630, 457, 164, 293, 432, 535,
    297, 493, 763, 837, 866, 242,
    // Bin capacity
    1861, 1858, 1918, 1301, 1990, 1623
  };
  const int i6_18_6_22_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    436, 42, 250, 728, 549, 560,
    757, 907, 401, 337, 694, 123,
    152, 464, 149, 535, 618, 587,
    235, 1, 843, 961, 67, 164,
    174, 437, 10, 720, 589, 560,
    469, 90, 345, 303, 420, 735,
    113, 319, 165, 509, 535, 264,
    58, 560, 609, 523, 928, 383,
    477, 824, 940, 111, 688, 366,
    316, 342, 193, 972, 613, 942,
    617, 39, 501, 3, 300, 414,
    30, 264, 424, 917, 716, 810,
    939, 516, 615, 747, 907, 890,
    480, 61, 322, 974, 557, 454,
    993, 969, 996, 818, 419, 819,
    782, 52, 379, 701, 132, 484,
    982, 869, 447, 570, 649, 654,
    957, 665, 722, 870, 516, 313,
    // Bin capacity
    1824, 1509, 1690, 2298, 2013, 1937
  };
  const int i6_18_6_23_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    545, 899, 142, 979, 853, 266,
    240, 197, 601, 611, 431, 666,
    740, 424, 351, 364, 658, 524,
    555, 897, 9, 46, 357, 611,
    664, 70, 344, 277, 673, 871,
    941, 313, 719, 705, 40, 74,
    89, 757, 487, 728, 308, 460,
    446, 854, 444, 358, 659, 901,
    321, 408, 729, 829, 658, 17,
    230, 475, 895, 981, 398, 837,
    392, 814, 154, 353, 725, 383,
    476, 691, 642, 425, 158, 482,
    966, 194, 779, 694, 336, 428,
    553, 278, 721, 197, 350, 30,
    504, 268, 991, 629, 184, 728,
    923, 561, 82, 470, 809, 33,
    44, 160, 644, 240, 765, 932,
    848, 262, 298, 621, 359, 975,
    // Bin capacity
    1943, 1748, 1852, 1949, 1788, 1890
  };
  const int i6_18_6_24_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    102, 708, 260, 178, 338, 63,
    259, 492, 881, 214, 773, 15,
    130, 556, 988, 742, 874, 437,
    218, 921, 675, 288, 466, 168,
    972, 478, 346, 820, 642, 500,
    223, 426, 365, 547, 614, 23,
    684, 266, 300, 329, 225, 465,
    954, 830, 41, 925, 611, 875,
    392, 965, 261, 612, 208, 760,
    174, 316, 427, 968, 838, 714,
    511, 569, 592, 468, 102, 856,
    915, 340, 397, 527, 70, 247,
    731, 796, 23, 888, 675, 170,
    28, 614, 664, 45, 885, 284,
    378, 493, 230, 398, 709, 675,
    759, 573, 618, 173, 682, 222,
    716, 95, 878, 326, 60, 357,
    124, 446, 847, 86, 678, 493,
    // Bin capacity
    1710, 2043, 1818, 1764, 1953, 1514
  };
  const int i6_18_6_25_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    533, 711, 576, 963, 977, 831,
    73, 428, 295, 112, 648, 576,
    759, 800, 731, 656, 324, 757,
    695, 194, 250, 46, 854, 249,
    161, 164, 160, 662, 364, 118,
    90, 612, 298, 188, 913, 12,
    428, 410, 567, 811, 166, 640,
    676, 722, 798, 609, 997, 717,
    650, 550, 569, 442, 200, 220,
    758, 62, 219, 916, 722, 495,
    134, 282, 941, 173, 538, 849,
    446, 522, 119, 940, 333, 77,
    373, 167, 630, 219, 845, 910,
    887, 887, 809, 179, 14, 850,
    824, 637, 657, 26, 441, 548,
    50, 966, 905, 156, 939, 908,
    887, 894, 418, 60, 808, 106,
    223, 206, 755, 288, 481, 167,
    // Bin capacity
    1802, 1920, 2021, 1552, 2201, 1882
  };
  const int i6_18_6_26_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    202, 196, 463, 986, 605, 640,
    805, 23, 394, 96, 597, 647,
    905, 754, 923, 444, 134, 927,
    242, 999, 363, 547, 416, 271,
    454, 452, 547, 49, 934, 467,
    287, 499, 136, 368, 993, 407,
    709, 733, 891, 908, 777, 912,
    868, 3, 264, 914, 731, 76,
    178, 255, 37, 383, 434, 91,
    588, 681, 226, 677, 239, 302,
    895, 771, 555, 694, 58, 542,
    322, 671, 362, 441, 766, 639,
    487, 489, 346, 399, 532, 876,
    223, 680, 306, 255, 6, 702,
    638, 171, 701, 363, 931, 343,
    25, 353, 618, 188, 653, 90,
    978, 534, 235, 934, 862, 51,
    152, 71, 307, 931, 526, 828,
    // Bin capacity
    1882, 1751, 1612, 2012, 2141, 1851
  };
  const int i6_18_6_27_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    641, 595, 905, 409, 264, 306,
    721, 116, 14, 978, 393, 35,
    19, 881, 457, 943, 5, 473,
    463, 232, 625, 314, 344, 87,
    125, 755, 667, 902, 754, 161,
    351, 238, 187, 908, 725, 144,
    894, 513, 933, 204, 123, 438,
    724, 809, 208, 576, 137, 521,
    98, 149, 515, 279, 407, 209,
    356, 167, 831, 700, 439, 399,
    26, 454, 248, 92, 727, 805,
    821, 491, 550, 141, 646, 606,
    610, 528, 672, 213, 417, 548,
    187, 416, 426, 714, 663, 70,
    444, 355, 292, 494, 20, 853,
    827, 39, 40, 54, 603, 459,
    274, 210, 268, 745, 115, 768,
    676, 754, 119, 295, 575, 681,
    // Bin capacity
    1748, 1631, 1685, 1897, 1558, 1601
  };
  const int i6_18_6_28_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    222, 473, 577, 825, 568, 937,
    477, 443, 385, 915, 10, 458,
    816, 345, 337, 796, 196, 778,
    556, 929, 922, 833, 832, 439,
    330, 867, 317, 59, 540, 551,
    996, 592, 19, 876, 168, 444,
    388, 860, 293, 256, 389, 274,
    239, 657, 155, 682, 949, 247,
    79, 779, 394, 657, 802, 273,
    707, 260, 258, 463, 968, 92,
    399, 742, 311, 625, 693, 239,
    332, 248, 768, 610, 482, 89,
    507, 435, 249, 536, 693, 435,
    400, 779, 819, 616, 16, 83,
    286, 487, 257, 155, 161, 664,
    866, 374, 704, 22, 691, 331,
    965, 70, 150, 528, 371, 136,
    687, 739, 865, 904, 923, 77,
    // Bin capacity
    1974, 2151, 1660, 2210, 2017, 1397
  };
  const int i6_18_6_29_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    203, 727, 750, 211, 838, 55,
    481, 114, 295, 307, 955, 331,
    605, 79, 827, 146, 654, 965,
    993, 854, 244, 327, 201, 73,
    293, 654, 568, 919, 375, 703,
    136, 302, 18, 875, 150, 860,
    953, 266, 719, 789, 505, 981,
    538, 890, 191, 242, 477, 172,
    446, 455, 72, 959, 340, 392,
    293, 193, 131, 20, 602, 345,
    109, 903, 483, 526, 807, 336,
    47, 41, 385, 763, 936, 243,
    345, 762, 27, 926, 556, 70,
    10, 228, 363, 698, 872, 69,
    531, 197, 418, 230, 740, 264,
    622, 758, 168, 708, 634, 951,
    954, 170, 618, 995, 587, 238,
    225, 685, 707, 191, 155, 61,
    // Bin capacity
    1674, 1780, 1502, 2114, 2233, 1529
  };
  const int i6_18_6_30_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    841, 345, 276, 138, 723, 656,
    277, 74, 564, 177, 386, 954,
    352, 308, 370, 668, 387, 251,
    152, 859, 968, 760, 262, 429,
    119, 366, 511, 972, 316, 340,
    843, 502, 610, 716, 919, 701,
    704, 794, 429, 787, 417, 26,
    471, 278, 950, 48, 902, 639,
    224, 49, 166, 461, 28, 685,
    169, 870, 674, 268, 610, 410,
    755, 133, 906, 15, 553, 595,
    339, 359, 993, 116, 794, 378,
    812, 613, 918, 164, 32, 558,
    26, 436, 770, 629, 286, 212,
    28, 308, 832, 601, 293, 374,
    521, 433, 586, 953, 550, 309,
    396, 607, 321, 407, 487, 617,
    351, 202, 748, 633, 598, 623,
    // Bin capacity
    1599, 1633, 2512, 1845, 1851, 1898
  };
  const int i6_18_6_31_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    235, 196, 18, 609, 144, 506,
    300, 392, 278, 498, 339, 376,
    294, 660, 546, 927, 453, 328,
    283, 86, 19, 131, 518, 543,
    958, 272, 758, 86, 498, 966,
    681, 67, 636, 847, 114, 1,
    395, 62, 542, 635, 325, 323,
    225, 362, 652, 111, 385, 454,
    723, 100, 485, 277, 596, 73,
    497, 492, 309, 1000, 343, 53,
    470, 134, 170, 475, 637, 4,
    619, 42, 831, 26, 788, 197,
    512, 673, 491, 516, 291, 136,
    45, 601, 63, 448, 372, 705,
    907, 431, 414, 203, 644, 798,
    349, 437, 690, 469, 370, 660,
    577, 6, 754, 279, 148, 466,
    425, 405, 376, 511, 172, 296,
    // Bin capacity
    1855, 1183, 1754, 1758, 1559, 1504
  };
  const int i6_18_6_32_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    179, 550, 536, 548, 115, 509,
    118, 853, 782, 835, 333, 631,
    409, 758, 423, 347, 9, 238,
    502, 42, 875, 241, 411, 540,
    342, 312, 550, 883, 512, 380,
    74, 25, 301, 13, 673, 511,
    133, 824, 760, 784, 155, 42,
    46, 497, 652, 738, 4, 280,
    222, 256, 173, 79, 120, 33,
    633, 648, 385, 524, 456, 226,
    708, 583, 309, 285, 289, 405,
    411, 440, 804, 478, 871, 750,
    859, 344, 427, 136, 551, 489,
    364, 640, 28, 655, 95, 756,
    820, 33, 249, 335, 725, 843,
    929, 172, 233, 219, 788, 923,
    76, 582, 148, 674, 177, 986,
    276, 636, 608, 210, 266, 930,
    // Bin capacity
    1563, 1803, 1814, 1757, 1441, 2084
  };
  const int i6_18_6_33_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    916, 396, 524, 860, 311, 508,
    494, 485, 683, 959, 767, 693,
    841, 809, 814, 856, 89, 777,
    671, 293, 0, 879, 301, 801,
    771, 162, 39, 293, 247, 982,
    893, 586, 653, 302, 740, 608,
    275, 478, 626, 731, 140, 454,
    672, 909, 775, 287, 527, 4,
    879, 197, 191, 748, 949, 626,
    109, 485, 253, 91, 951, 580,
    534, 994, 539, 953, 590, 24,
    399, 242, 345, 894, 774, 2,
    760, 143, 673, 941, 901, 72,
    548, 495, 715, 727, 701, 712,
    953, 712, 832, 949, 296, 222,
    438, 279, 672, 760, 700, 348,
    276, 297, 179, 529, 115, 487,
    981, 80, 98, 118, 194, 984,
    // Bin capacity
    2530, 1783, 1909, 2633, 2060, 1970
  };
  const int i6_18_6_34_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    635, 11, 785, 420, 310, 566,
    392, 119, 825, 20, 796, 430,
    719, 615, 529, 754, 368, 573,
    356, 803, 23, 163, 950, 715,
    697, 848, 863, 967, 561, 936,
    271, 338, 387, 558, 292, 287,
    703, 237, 771, 393, 474, 207,
    984, 1000, 260, 719, 584, 713,
    73, 493, 682, 639, 293, 311,
    144, 330, 99, 997, 899, 189,
    8, 732, 320, 343, 531, 769,
    30, 939, 514, 25, 532, 850,
    300, 143, 682, 493, 621, 689,
    911, 442, 18, 462, 475, 81,
    199, 73, 360, 35, 710, 584,
    153, 971, 911, 144, 981, 276,
    551, 633, 358, 592, 303, 15,
    615, 43, 841, 498, 332, 686,
    // Bin capacity
    1729, 1959, 2061, 1837, 2237, 1983
  };
  const int i6_18_6_35_3[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    63, 398, 124, 592, 727, 82,
    840, 187, 724, 192, 830, 57,
    862, 272, 959, 331, 470, 657,
    553, 517, 470, 26, 536, 612,
    935, 567, 123, 700, 614, 882,
    299, 269, 149, 710, 157, 543,
    65, 83, 485, 787, 662, 945,
    897, 818, 782, 826, 511, 891,
    55, 759, 361, 696, 759, 688,
    427, 941, 985, 982, 998, 227,
    991, 789, 989, 711, 704, 83,
    928, 542, 98, 573, 544, 104,
    808, 275, 879, 725, 218, 193,
    683, 644, 993, 336, 343, 135,
    292, 631, 311, 62, 466, 156,
    541, 913, 175, 65, 566, 450,
    912, 128, 118, 260, 113, 953,
    550, 805, 71, 465, 698, 713,
    // Bin capacity
    2408, 2147, 1980, 2034, 2232, 1884
  };
  const int i6_18_6_0_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    403, 282, 624, 386, 96, 361,
    644, 376, 516, 238, 671, 605,
    346, 578, 686, 548, 146, 93,
    256, 193, 650, 618, 537, 829,
    508, 526, 478, 37, 17, 118,
    878, 229, 114, 915, 976, 669,
    698, 466, 986, 313, 632, 629,
    501, 550, 340, 841, 72, 125,
    131, 387, 415, 279, 620, 614,
    546, 934, 516, 65, 454, 491,
    394, 587, 612, 617, 985, 806,
    822, 848, 376, 164, 397, 663,
    930, 641, 665, 116, 360, 753,
    796, 855, 503, 723, 342, 593,
    283, 974, 442, 313, 310, 615,
    791, 224, 124, 264, 320, 871,
    922, 905, 619, 644, 880, 156,
    116, 696, 99, 793, 285, 225,
    // Bin capacity
    1661, 1709, 1461, 1313, 1350, 1536
  };
  const int i6_18_6_1_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    203, 612, 247, 954, 665, 135,
    897, 108, 614, 973, 42, 932,
    140, 473, 71, 342, 274, 144,
    242, 885, 72, 458, 622, 342,
    678, 871, 818, 991, 848, 102,
    410, 136, 857, 108, 645, 361,
    953, 540, 176, 54, 87, 740,
    855, 710, 113, 90, 952, 837,
    230, 195, 484, 253, 873, 279,
    639, 774, 313, 898, 136, 744,
    28, 889, 916, 82, 63, 294,
    344, 328, 704, 508, 982, 658,
    574, 100, 580, 660, 773, 930,
    690, 764, 89, 528, 736, 203,
    416, 542, 228, 436, 559, 203,
    67, 718, 667, 5, 818, 411,
    822, 283, 345, 927, 605, 408,
    269, 773, 741, 537, 632, 345,
    // Bin capacity
    1424, 1634, 1353, 1483, 1736, 1359
  };
  const int i6_18_6_2_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    159, 633, 2, 102, 489, 999,
    971, 378, 118, 985, 585, 43,
    258, 782, 397, 257, 965, 316,
    433, 708, 337, 707, 698, 828,
    0, 295, 404, 182, 298, 383,
    668, 830, 497, 859, 914, 322,
    342, 232, 590, 139, 818, 290,
    544, 121, 840, 341, 162, 707,
    339, 101, 190, 106, 969, 626,
    228, 521, 253, 48, 401, 603,
    563, 770, 766, 77, 433, 503,
    736, 927, 351, 763, 294, 832,
    747, 488, 832, 804, 507, 295,
    583, 503, 132, 599, 353, 573,
    26, 570, 83, 665, 504, 485,
    332, 541, 956, 562, 239, 844,
    286, 960, 605, 557, 796, 608,
    813, 971, 12, 751, 77, 376,
    // Bin capacity
    1365, 1757, 1253, 1446, 1616, 1638
  };
  const int i6_18_6_3_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    51, 62, 89, 683, 47, 162,
    777, 603, 177, 37, 833, 205,
    132, 141, 499, 843, 819, 75,
    237, 703, 105, 723, 722, 876,
    839, 768, 473, 495, 20, 231,
    904, 207, 398, 90, 61, 538,
    27, 820, 211, 687, 518, 743,
    554, 909, 975, 222, 90, 910,
    75, 124, 35, 544, 647, 652,
    625, 230, 939, 826, 534, 0,
    565, 327, 855, 784, 886, 469,
    936, 858, 53, 213, 231, 734,
    523, 76, 268, 257, 124, 164,
    809, 25, 119, 134, 502, 927,
    412, 918, 571, 251, 242, 305,
    34, 465, 791, 577, 731, 148,
    402, 22, 183, 53, 814, 754,
    129, 231, 226, 136, 981, 358,
    // Bin capacity
    1379, 1286, 1197, 1297, 1512, 1417
  };
  const int i6_18_6_4_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    919, 122, 19, 537, 313, 423,
    863, 474, 919, 336, 396, 367,
    161, 541, 526, 593, 638, 728,
    55, 586, 78, 336, 60, 643,
    44, 632, 538, 750, 81, 777,
    978, 72, 259, 232, 214, 806,
    136, 713, 10, 293, 163, 391,
    62, 301, 23, 825, 773, 546,
    430, 994, 78, 978, 712, 731,
    595, 273, 39, 268, 926, 971,
    891, 460, 573, 21, 557, 289,
    3, 772, 496, 252, 580, 514,
    506, 901, 559, 397, 883, 105,
    636, 248, 989, 146, 886, 24,
    517, 928, 459, 502, 10, 613,
    945, 237, 198, 304, 501, 818,
    350, 131, 184, 408, 386, 786,
    295, 601, 717, 631, 857, 321,
    // Bin capacity
    1454, 1558, 1156, 1354, 1549, 1708
  };
  const int i6_18_6_5_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    345, 998, 406, 277, 83, 427,
    773, 306, 531, 898, 365, 272,
    848, 138, 17, 979, 672, 66,
    306, 5, 637, 268, 653, 461,
    531, 943, 274, 230, 86, 449,
    766, 425, 83, 895, 400, 180,
    926, 350, 678, 124, 886, 666,
    107, 950, 320, 452, 157, 941,
    941, 313, 829, 540, 433, 561,
    157, 700, 351, 113, 965, 513,
    395, 634, 439, 787, 729, 855,
    733, 420, 32, 887, 425, 792,
    613, 801, 58, 578, 100, 399,
    190, 62, 1000, 728, 237, 592,
    26, 968, 184, 79, 17, 151,
    100, 882, 764, 578, 621, 212,
    652, 596, 899, 139, 775, 835,
    242, 232, 39, 481, 47, 590,
    // Bin capacity
    1514, 1702, 1320, 1581, 1339, 1569
  };
  const int i6_18_6_6_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    879, 625, 614, 450, 246, 551,
    78, 761, 14, 80, 184, 876,
    563, 333, 876, 628, 648, 2,
    239, 211, 533, 965, 501, 361,
    709, 922, 761, 195, 237, 384,
    508, 352, 330, 464, 493, 75,
    532, 611, 350, 505, 937, 872,
    425, 928, 918, 308, 981, 665,
    749, 189, 269, 399, 289, 991,
    204, 475, 936, 287, 572, 75,
    254, 352, 209, 67, 730, 264,
    685, 612, 709, 348, 439, 82,
    116, 315, 31, 832, 169, 334,
    510, 529, 689, 914, 420, 532,
    721, 955, 522, 741, 665, 386,
    883, 880, 490, 702, 775, 506,
    628, 818, 41, 807, 1000, 442,
    200, 248, 718, 609, 877, 24,
    // Bin capacity
    1570, 1788, 1592, 1644, 1796, 1312
  };
  const int i6_18_6_7_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    240, 223, 932, 273, 79, 893,
    979, 336, 978, 189, 805, 405,
    433, 87, 71, 957, 769, 361,
    988, 505, 620, 816, 472, 103,
    58, 672, 619, 879, 576, 979,
    739, 196, 314, 387, 524, 724,
    804, 131, 713, 359, 351, 900,
    466, 963, 9, 304, 182, 561,
    506, 524, 696, 869, 428, 791,
    886, 385, 759, 861, 709, 744,
    419, 563, 828, 899, 696, 360,
    706, 245, 140, 41, 298, 382,
    286, 92, 30, 448, 616, 521,
    19, 146, 375, 32, 475, 736,
    771, 931, 118, 448, 853, 136,
    678, 743, 226, 892, 205, 777,
    666, 164, 353, 526, 372, 98,
    951, 501, 869, 545, 487, 470,
    // Bin capacity
    1890, 1321, 1543, 1735, 1587, 1773
  };
  const int i6_18_6_8_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    294, 982, 35, 646, 324, 320,
    723, 11, 591, 286, 901, 579,
    402, 442, 407, 949, 900, 961,
    668, 232, 843, 39, 864, 835,
    718, 433, 830, 404, 187, 237,
    865, 580, 384, 172, 319, 221,
    816, 449, 831, 844, 217, 216,
    913, 674, 909, 852, 71, 902,
    218, 288, 79, 383, 249, 955,
    184, 174, 884, 27, 600, 628,
    126, 56, 604, 485, 653, 293,
    129, 767, 904, 946, 887, 175,
    200, 281, 654, 29, 489, 526,
    899, 944, 294, 159, 700, 472,
    612, 874, 101, 611, 913, 615,
    93, 967, 331, 316, 277, 337,
    182, 364, 965, 987, 40, 852,
    695, 740, 891, 54, 849, 473,
    // Bin capacity
    1573, 1667, 1897, 1475, 1700, 1728
  };
  const int i6_18_6_9_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    284, 502, 93, 588, 714, 313,
    926, 842, 500, 303, 797, 856,
    827, 164, 279, 628, 670, 681,
    727, 5, 407, 10, 95, 269,
    554, 806, 592, 145, 315, 93,
    32, 184, 794, 853, 153, 927,
    794, 989, 882, 55, 99, 594,
    345, 738, 957, 818, 632, 450,
    879, 742, 297, 997, 266, 388,
    833, 883, 413, 516, 819, 532,
    318, 506, 829, 591, 345, 442,
    317, 217, 606, 715, 390, 412,
    289, 252, 254, 646, 896, 626,
    42, 806, 296, 653, 259, 406,
    37, 957, 974, 993, 108, 154,
    43, 950, 968, 704, 277, 621,
    816, 579, 606, 784, 887, 255,
    391, 626, 719, 99, 163, 218,
    // Bin capacity
    1536, 1953, 1902, 1835, 1433, 1497
  };
  const int i6_18_6_10_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    906, 733, 660, 451, 878, 625,
    559, 606, 983, 892, 242, 741,
    564, 189, 965, 696, 870, 760,
    776, 325, 626, 955, 57, 737,
    345, 935, 584, 244, 286, 820,
    526, 652, 723, 990, 739, 586,
    508, 191, 790, 585, 454, 398,
    134, 30, 536, 50, 619, 68,
    837, 856, 102, 162, 134, 510,
    742, 59, 671, 28, 749, 10,
    697, 494, 354, 361, 591, 832,
    539, 822, 272, 65, 985, 162,
    342, 632, 167, 315, 741, 399,
    778, 683, 867, 496, 982, 157,
    935, 464, 317, 174, 926, 456,
    944, 589, 875, 874, 939, 859,
    973, 511, 486, 344, 521, 828,
    828, 206, 763, 76, 310, 156,
    // Bin capacity
    2188, 1646, 1970, 1423, 2021, 1670
  };
  const int i6_18_6_11_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    655, 733, 945, 236, 391, 35,
    253, 330, 452, 792, 409, 694,
    589, 301, 668, 685, 935, 177,
    849, 165, 548, 29, 806, 303,
    128, 248, 153, 530, 259, 647,
    357, 814, 516, 203, 224, 253,
    902, 322, 65, 746, 909, 172,
    194, 463, 536, 32, 573, 832,
    662, 601, 48, 717, 938, 446,
    500, 688, 80, 582, 466, 580,
    681, 994, 303, 611, 105, 302,
    732, 114, 425, 799, 37, 875,
    464, 152, 391, 179, 96, 627,
    80, 198, 26, 951, 62, 127,
    155, 632, 579, 21, 221, 397,
    554, 75, 875, 466, 385, 55,
    638, 636, 894, 448, 603, 996,
    273, 604, 378, 145, 892, 603,
    // Bin capacity
    1604, 1493, 1459, 1512, 1538, 1503
  };
  const int i6_18_6_12_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    113, 564, 130, 921, 214, 684,
    21, 679, 106, 445, 462, 547,
    514, 190, 572, 405, 381, 418,
    601, 384, 69, 628, 94, 651,
    777, 492, 185, 865, 677, 606,
    661, 154, 819, 586, 229, 459,
    344, 633, 583, 741, 303, 494,
    118, 349, 924, 108, 850, 26,
    820, 97, 938, 556, 329, 366,
    403, 535, 197, 702, 190, 700,
    818, 927, 654, 666, 900, 195,
    368, 741, 346, 75, 687, 619,
    63, 440, 94, 23, 43, 354,
    707, 184, 695, 564, 338, 256,
    843, 802, 440, 808, 704, 938,
    225, 553, 178, 140, 655, 608,
    184, 159, 675, 963, 611, 310,
    59, 49, 758, 924, 786, 383,
    // Bin capacity
    1426, 1481, 1562, 1890, 1578, 1608
  };
  const int i6_18_6_13_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    563, 380, 737, 639, 444, 397,
    85, 788, 351, 620, 776, 24,
    57, 144, 573, 258, 793, 431,
    89, 494, 62, 183, 973, 998,
    744, 914, 760, 985, 101, 369,
    865, 715, 31, 743, 486, 172,
    638, 512, 936, 940, 629, 180,
    7, 460, 281, 2, 177, 686,
    198, 112, 560, 166, 878, 370,
    729, 264, 567, 24, 918, 535,
    812, 338, 436, 812, 802, 519,
    352, 242, 318, 224, 230, 653,
    804, 997, 806, 930, 59, 186,
    899, 442, 508, 29, 849, 218,
    917, 387, 949, 349, 976, 370,
    269, 705, 726, 838, 995, 698,
    816, 692, 664, 267, 211, 89,
    276, 443, 320, 171, 71, 286,
    // Bin capacity
    1718, 1701, 1806, 1541, 1953, 1353
  };
  const int i6_18_6_14_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    72, 16, 814, 712, 941, 164,
    556, 901, 805, 757, 686, 101,
    578, 15, 651, 255, 586, 962,
    859, 265, 92, 530, 771, 486,
    881, 922, 230, 932, 828, 131,
    253, 615, 47, 384, 913, 383,
    107, 88, 953, 162, 296, 875,
    506, 534, 1, 112, 876, 628,
    249, 988, 169, 974, 602, 699,
    299, 507, 55, 809, 652, 223,
    130, 382, 861, 466, 371, 729,
    922, 446, 176, 160, 132, 924,
    600, 169, 605, 909, 11, 701,
    450, 134, 113, 777, 249, 253,
    947, 331, 712, 738, 245, 834,
    977, 222, 650, 287, 696, 872,
    979, 746, 220, 97, 862, 635,
    289, 294, 454, 740, 218, 958,
    // Bin capacity
    1835, 1440, 1446, 1863, 1888, 2007
  };
  const int i6_18_6_15_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    800, 233, 395, 391, 617, 372,
    910, 929, 177, 742, 491, 11,
    576, 107, 618, 32, 563, 927,
    849, 604, 27, 973, 923, 353,
    890, 487, 47, 86, 216, 402,
    99, 792, 83, 945, 845, 473,
    884, 717, 335, 499, 75, 983,
    668, 195, 562, 207, 57, 972,
    530, 861, 892, 963, 800, 189,
    148, 892, 627, 771, 346, 826,
    258, 51, 743, 91, 91, 334,
    661, 184, 846, 339, 903, 579,
    148, 547, 518, 644, 962, 261,
    165, 548, 14, 92, 931, 367,
    879, 778, 539, 989, 851, 806,
    523, 190, 434, 466, 124, 609,
    628, 510, 802, 601, 714, 658,
    536, 26, 806, 332, 122, 654,
    // Bin capacity
    1946, 1659, 1623, 1757, 1846, 1874
  };
  const int i6_18_6_16_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    132, 720, 598, 331, 882, 263,
    408, 117, 222, 307, 321, 242,
    799, 406, 500, 970, 866, 747,
    942, 487, 809, 639, 720, 952,
    209, 936, 812, 453, 458, 942,
    224, 234, 823, 380, 612, 856,
    826, 889, 400, 631, 909, 195,
    991, 26, 558, 613, 685, 570,
    546, 781, 14, 89, 471, 14,
    493, 440, 318, 251, 631, 266,
    999, 129, 856, 266, 928, 234,
    978, 348, 61, 696, 702, 877,
    514, 204, 763, 147, 385, 940,
    984, 824, 888, 600, 752, 811,
    134, 156, 994, 490, 817, 38,
    132, 24, 136, 979, 205, 21,
    927, 317, 231, 157, 815, 724,
    721, 329, 45, 447, 160, 879,
    // Bin capacity
    2119, 1425, 1746, 1633, 2189, 1851
  };
  const int i6_18_6_17_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    151, 349, 358, 665, 231, 460,
    349, 738, 690, 409, 278, 193,
    450, 366, 194, 520, 614, 724,
    566, 281, 81, 425, 665, 34,
    750, 611, 419, 531, 641, 908,
    603, 250, 478, 556, 81, 272,
    469, 842, 532, 616, 419, 995,
    509, 420, 331, 2, 111, 264,
    977, 3, 432, 827, 625, 718,
    161, 365, 141, 345, 718, 119,
    120, 386, 997, 18, 795, 188,
    756, 193, 694, 673, 281, 401,
    40, 84, 694, 58, 482, 708,
    307, 8, 753, 217, 104, 637,
    994, 624, 444, 144, 823, 919,
    479, 388, 168, 760, 642, 422,
    977, 355, 578, 578, 231, 476,
    166, 358, 540, 500, 796, 462,
    // Bin capacity
    1721, 1292, 1663, 1530, 1665, 1736
  };
  const int i6_18_6_18_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    673, 634, 333, 594, 311, 353,
    124, 719, 153, 216, 818, 730,
    728, 994, 251, 834, 497, 115,
    131, 148, 41, 652, 988, 218,
    194, 957, 494, 185, 701, 264,
    13, 453, 598, 273, 295, 436,
    569, 935, 728, 49, 373, 860,
    266, 909, 148, 647, 662, 77,
    632, 623, 286, 12, 395, 828,
    106, 241, 360, 243, 13, 738,
    975, 25, 973, 804, 370, 538,
    585, 625, 597, 906, 148, 665,
    91, 107, 914, 82, 406, 498,
    718, 82, 618, 446, 419, 735,
    805, 783, 984, 216, 302, 71,
    429, 409, 839, 452, 869, 539,
    753, 310, 303, 585, 930, 917,
    732, 72, 233, 582, 465, 750,
    // Bin capacity
    1677, 1776, 1742, 1530, 1763, 1836
  };
  const int i6_18_6_19_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    976, 499, 251, 425, 110, 394,
    926, 390, 750, 414, 576, 620,
    250, 613, 167, 997, 362, 518,
    561, 368, 310, 542, 411, 884,
    207, 813, 807, 160, 394, 453,
    766, 179, 143, 690, 767, 796,
    938, 50, 684, 968, 312, 841,
    682, 109, 835, 361, 75, 878,
    317, 269, 464, 336, 193, 184,
    676, 37, 25, 909, 299, 894,
    312, 206, 641, 739, 830, 38,
    433, 475, 947, 48, 911, 422,
    207, 124, 859, 129, 191, 728,
    337, 961, 317, 792, 489, 994,
    219, 446, 475, 257, 328, 458,
    400, 316, 580, 979, 553, 487,
    770, 105, 912, 708, 824, 547,
    877, 59, 815, 35, 728, 377,
    // Bin capacity
    1955, 1194, 1980, 1882, 1657, 2086
  };
  const int i6_18_6_20_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    184, 846, 679, 19, 580, 429,
    496, 180, 594, 579, 428, 287,
    166, 267, 905, 629, 967, 684,
    322, 713, 140, 844, 448, 464,
    439, 617, 461, 640, 383, 718,
    19, 501, 485, 682, 289, 319,
    174, 970, 676, 145, 881, 364,
    142, 946, 850, 588, 298, 265,
    166, 875, 613, 557, 863, 590,
    702, 162, 664, 475, 999, 903,
    147, 125, 35, 347, 113, 857,
    984, 123, 559, 560, 597, 927,
    540, 47, 197, 21, 347, 208,
    43, 947, 287, 722, 710, 333,
    343, 122, 441, 788, 289, 796,
    775, 737, 861, 225, 674, 38,
    528, 381, 691, 380, 369, 594,
    201, 58, 806, 247, 208, 870,
    // Bin capacity
    1275, 1724, 1989, 1690, 1889, 1930
  };
  const int i6_18_6_21_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    990, 673, 563, 936, 876, 312,
    21, 401, 976, 338, 357, 1000,
    219, 819, 922, 452, 182, 616,
    311, 472, 393, 666, 397, 983,
    262, 756, 240, 125, 938, 487,
    879, 967, 865, 755, 56, 529,
    547, 443, 903, 622, 46, 101,
    867, 83, 166, 402, 299, 18,
    654, 590, 422, 123, 936, 926,
    680, 401, 17, 987, 194, 553,
    259, 733, 189, 490, 890, 11,
    906, 174, 396, 525, 45, 319,
    925, 444, 685, 760, 274, 189,
    169, 192, 190, 301, 766, 769,
    675, 956, 385, 780, 946, 51,
    633, 101, 315, 629, 17, 806,
    718, 107, 13, 587, 367, 852,
    679, 142, 565, 295, 82, 103,
    // Bin capacity
    2097, 1705, 1655, 1971, 1547, 1740
  };
  const int i6_18_6_22_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    218, 439, 999, 756, 816, 965,
    811, 541, 450, 53, 80, 620,
    370, 700, 44, 143, 658, 1000,
    162, 331, 857, 921, 580, 47,
    438, 308, 810, 325, 423, 495,
    740, 833, 324, 778, 365, 659,
    47, 713, 92, 139, 334, 306,
    24, 161, 586, 225, 405, 291,
    209, 458, 686, 960, 463, 138,
    406, 178, 756, 899, 206, 297,
    941, 952, 830, 264, 442, 907,
    669, 429, 396, 629, 587, 875,
    471, 390, 807, 234, 289, 626,
    382, 241, 876, 847, 907, 474,
    500, 805, 120, 964, 881, 839,
    571, 554, 680, 130, 3, 498,
    989, 822, 406, 874, 964, 990,
    916, 677, 107, 582, 432, 407,
    // Bin capacity
    1803, 1939, 1998, 1978, 1797, 2122
  };
  const int i6_18_6_23_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    730, 472, 39, 675, 157, 943,
    780, 939, 651, 156, 443, 656,
    614, 539, 490, 217, 80, 985,
    554, 547, 348, 639, 860, 926,
    104, 444, 953, 965, 386, 77,
    281, 776, 29, 420, 84, 543,
    258, 868, 768, 450, 83, 139,
    22, 274, 471, 255, 675, 447,
    651, 762, 625, 492, 620, 340,
    705, 293, 630, 591, 102, 65,
    195, 776, 458, 773, 304, 241,
    822, 624, 525, 894, 27, 58,
    273, 929, 359, 628, 21, 946,
    163, 523, 400, 929, 797, 646,
    665, 519, 817, 868, 796, 579,
    794, 627, 763, 912, 941, 11,
    246, 111, 1, 319, 874, 557,
    797, 833, 246, 125, 290, 218,
    // Bin capacity
    1775, 2226, 1758, 2114, 1546, 1718
  };
  const int i6_18_6_24_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    989, 845, 737, 432, 976, 553,
    260, 728, 468, 719, 676, 176,
    271, 81, 883, 836, 310, 335,
    821, 437, 812, 745, 560, 556,
    343, 245, 487, 231, 977, 503,
    264, 575, 981, 941, 408, 342,
    853, 595, 349, 996, 619, 449,
    70, 648, 35, 431, 858, 910,
    67, 917, 979, 258, 389, 121,
    786, 316, 292, 701, 170, 89,
    922, 581, 796, 132, 476, 186,
    12, 903, 751, 819, 22, 483,
    617, 775, 11, 545, 475, 816,
    386, 384, 309, 261, 260, 290,
    837, 445, 1, 500, 388, 366,
    843, 962, 338, 526, 595, 130,
    128, 835, 184, 855, 42, 612,
    784, 211, 113, 975, 502, 168,
    // Bin capacity
    1913, 2167, 1763, 2254, 1799, 1465
  };
  const int i6_18_6_25_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    580, 263, 893, 436, 55, 309,
    87, 860, 467, 894, 842, 545,
    33, 897, 845, 921, 75, 249,
    389, 319, 265, 448, 169, 947,
    349, 960, 631, 917, 552, 159,
    534, 899, 72, 256, 538, 395,
    586, 615, 567, 650, 453, 49,
    936, 943, 602, 25, 354, 269,
    943, 125, 257, 218, 885, 703,
    832, 918, 587, 792, 754, 899,
    112, 707, 61, 675, 848, 833,
    37, 762, 796, 191, 845, 584,
    228, 235, 101, 348, 887, 445,
    725, 1000, 246, 767, 950, 701,
    101, 228, 877, 124, 293, 465,
    361, 739, 231, 176, 103, 638,
    151, 985, 439, 420, 147, 502,
    625, 276, 815, 685, 384, 372,
    // Bin capacity
    1586, 2444, 1824, 1864, 1903, 1889
  };
  const int i6_18_6_26_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    373, 831, 249, 792, 803, 647,
    730, 890, 970, 46, 403, 515,
    436, 160, 562, 928, 349, 604,
    140, 220, 476, 427, 842, 147,
    46, 416, 543, 157, 803, 747,
    845, 911, 891, 826, 453, 364,
    465, 872, 19, 994, 151, 910,
    533, 801, 467, 50, 625, 9,
    538, 819, 963, 36, 658, 741,
    913, 424, 636, 522, 218, 205,
    557, 674, 656, 427, 101, 963,
    253, 524, 153, 907, 648, 435,
    843, 636, 84, 735, 334, 757,
    101, 900, 314, 534, 680, 59,
    888, 364, 977, 38, 571, 605,
    656, 23, 680, 869, 143, 809,
    495, 761, 879, 924, 494, 55,
    961, 861, 764, 729, 580, 727,
    // Bin capacity
    2053, 2329, 2160, 2088, 1860, 1953
  };
  const int i6_18_6_27_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    645, 755, 418, 379, 510, 301,
    370, 603, 847, 481, 990, 481,
    288, 413, 852, 468, 881, 140,
    631, 992, 426, 371, 122, 119,
    311, 41, 549, 272, 98, 610,
    951, 405, 886, 374, 233, 433,
    941, 145, 453, 758, 395, 609,
    13, 539, 857, 490, 703, 389,
    738, 183, 111, 867, 597, 689,
    497, 473, 162, 246, 176, 382,
    127, 153, 740, 436, 375, 326,
    638, 9, 445, 456, 650, 362,
    734, 714, 963, 363, 248, 662,
    941, 490, 990, 284, 57, 269,
    561, 104, 639, 803, 176, 860,
    985, 925, 711, 121, 903, 436,
    22, 328, 583, 96, 842, 398,
    269, 417, 203, 740, 539, 879,
    // Bin capacity
    2046, 1628, 2294, 1695, 1799, 1767
  };
  const int i6_18_6_28_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    29, 168, 696, 57, 554, 351,
    31, 637, 542, 829, 631, 100,
    675, 594, 348, 685, 962, 712,
    626, 235, 601, 112, 262, 975,
    383, 888, 482, 522, 121, 284,
    23, 593, 549, 133, 840, 767,
    572, 696, 201, 760, 940, 50,
    210, 183, 266, 9, 220, 614,
    220, 432, 880, 62, 725, 567,
    603, 997, 573, 245, 341, 540,
    302, 711, 915, 189, 740, 415,
    303, 944, 376, 935, 701, 704,
    468, 11, 91, 909, 93, 811,
    392, 224, 391, 931, 963, 210,
    0, 356, 494, 994, 845, 884,
    677, 586, 15, 876, 671, 185,
    533, 814, 892, 602, 878, 236,
    70, 440, 998, 669, 425, 232,
    // Bin capacity
    1305, 2029, 1987, 2031, 2328, 1843
  };
  const int i6_18_6_29_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    89, 353, 323, 627, 828, 773,
    666, 856, 654, 759, 811, 240,
    52, 342, 843, 419, 56, 321,
    985, 934, 11, 174, 335, 112,
    505, 629, 220, 908, 400, 302,
    213, 666, 311, 718, 660, 537,
    542, 933, 622, 597, 172, 952,
    57, 819, 339, 124, 774, 729,
    27, 623, 497, 429, 543, 414,
    530, 953, 837, 955, 360, 500,
    310, 568, 75, 629, 919, 47,
    588, 846, 434, 845, 236, 345,
    820, 546, 34, 551, 394, 207,
    854, 217, 221, 69, 642, 636,
    453, 839, 904, 90, 466, 692,
    735, 97, 667, 999, 216, 39,
    880, 914, 660, 867, 538, 803,
    897, 691, 222, 145, 412, 853,
    // Bin capacity
    1979, 2543, 1693, 2130, 1884, 1828
  };
  const int i6_18_6_30_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    735, 534, 575, 619, 981, 163,
    904, 70, 787, 427, 204, 699,
    244, 57, 736, 503, 691, 930,
    329, 814, 461, 243, 854, 472,
    184, 576, 355, 753, 926, 413,
    515, 243, 261, 81, 489, 867,
    755, 183, 133, 677, 993, 986,
    227, 361, 866, 638, 843, 986,
    631, 218, 990, 990, 656, 567,
    944, 986, 664, 767, 209, 939,
    911, 829, 605, 702, 181, 397,
    799, 849, 323, 165, 975, 945,
    941, 862, 158, 605, 691, 53,
    953, 641, 133, 45, 403, 655,
    309, 846, 532, 820, 984, 146,
    788, 886, 680, 749, 178, 715,
    176, 810, 952, 340, 688, 317,
    720, 528, 28, 122, 260, 596,
    // Bin capacity
    2398, 2231, 2002, 2004, 2428, 2350
  };
  const int i6_18_6_31_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    149, 765, 803, 278, 42, 405,
    753, 645, 652, 852, 594, 115,
    439, 320, 857, 401, 984, 7,
    866, 240, 250, 205, 699, 377,
    577, 170, 256, 792, 126, 251,
    549, 794, 764, 306, 230, 707,
    19, 943, 366, 371, 43, 190,
    941, 30, 326, 529, 709, 207,
    196, 916, 20, 174, 179, 890,
    939, 677, 509, 642, 532, 32,
    377, 807, 237, 847, 583, 229,
    57, 215, 345, 97, 934, 328,
    711, 994, 538, 975, 443, 302,
    938, 535, 568, 70, 823, 377,
    424, 876, 144, 948, 934, 295,
    808, 652, 101, 154, 171, 803,
    457, 720, 225, 519, 249, 529,
    51, 82, 16, 761, 268, 581,
    // Bin capacity
    2020, 2267, 1524, 1948, 1866, 1447
  };
  const int i6_18_6_32_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    765, 424, 772, 182, 958, 925,
    257, 339, 397, 336, 3, 937,
    674, 83, 905, 335, 197, 958,
    370, 397, 910, 100, 532, 470,
    966, 158, 701, 200, 904, 658,
    276, 331, 14, 309, 921, 132,
    492, 461, 529, 788, 956, 112,
    324, 649, 30, 719, 266, 63,
    470, 822, 632, 971, 376, 813,
    610, 739, 589, 511, 20, 36,
    29, 272, 407, 443, 871, 726,
    941, 948, 827, 111, 760, 478,
    884, 528, 812, 213, 53, 751,
    141, 1, 244, 874, 107, 242,
    752, 410, 61, 741, 793, 614,
    666, 495, 473, 347, 394, 729,
    622, 600, 373, 582, 663, 669,
    30, 194, 480, 805, 575, 542,
    // Bin capacity
    2040, 1728, 2015, 1885, 2057, 2169
  };
  const int i6_18_6_33_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    306, 2, 592, 840, 39, 317,
    256, 156, 393, 216, 621, 979,
    667, 995, 821, 187, 169, 106,
    93, 584, 234, 19, 886, 981,
    551, 665, 314, 716, 902, 320,
    315, 8, 275, 707, 716, 656,
    752, 476, 674, 631, 180, 924,
    452, 250, 600, 31, 603, 582,
    636, 655, 357, 112, 909, 284,
    471, 441, 738, 569, 971, 355,
    255, 148, 262, 703, 686, 854,
    832, 580, 515, 1000, 316, 782,
    419, 289, 863, 606, 319, 483,
    539, 706, 680, 224, 456, 836,
    310, 500, 372, 306, 907, 568,
    53, 581, 570, 416, 502, 917,
    692, 84, 761, 95, 256, 384,
    56, 437, 420, 149, 173, 82,
    // Bin capacity
    1697, 1676, 2093, 1669, 2131, 2308
  };
  const int i6_18_6_34_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    647, 474, 647, 309, 886, 886,
    674, 973, 952, 377, 168, 984,
    617, 779, 132, 543, 706, 923,
    885, 273, 417, 742, 644, 648,
    960, 129, 952, 525, 491, 428,
    441, 119, 31, 585, 567, 135,
    405, 872, 371, 237, 474, 613,
    753, 521, 335, 356, 455, 618,
    679, 667, 698, 646, 397, 531,
    588, 656, 532, 404, 410, 620,
    297, 339, 773, 23, 931, 621,
    537, 934, 555, 669, 10, 70,
    642, 872, 187, 60, 513, 466,
    285, 950, 929, 368, 651, 467,
    341, 310, 366, 471, 886, 12,
    613, 861, 361, 134, 386, 158,
    224, 606, 224, 700, 380, 104,
    500, 22, 375, 949, 39, 433,
    // Bin capacity
    2253, 2314, 1974, 1809, 2009, 1947
  };
  const int i6_18_6_35_4[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    609, 959, 797, 534, 168, 283,
    206, 116, 961, 70, 411, 33,
    116, 82, 341, 105, 186, 711,
    675, 431, 363, 803, 860, 985,
    482, 806, 112, 57, 435, 826,
    330, 276, 954, 957, 707, 340,
    98, 399, 245, 173, 197, 309,
    895, 693, 912, 350, 119, 462,
    611, 812, 615, 337, 243, 535,
    740, 402, 524, 233, 497, 676,
    661, 927, 138, 912, 174, 622,
    657, 692, 367, 102, 833, 947,
    946, 430, 939, 930, 1000, 719,
    131, 779, 597, 133, 220, 757,
    76, 180, 358, 72, 224, 284,
    992, 820, 832, 300, 987, 692,
    445, 480, 397, 963, 442, 535,
    741, 303, 678, 16, 137, 312,
    // Bin capacity
    2118, 2158, 2280, 1586, 1764, 2257
  };
  const int i6_18_6_0_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    686, 695, 317, 303, 474, 653,
    656, 4, 244, 893, 628, 621,
    3, 765, 360, 37, 160, 861,
    554, 577, 25, 704, 425, 273,
    125, 587, 581, 636, 293, 953,
    279, 271, 416, 595, 178, 93,
    860, 334, 593, 87, 494, 181,
    320, 257, 575, 212, 280, 206,
    259, 21, 967, 604, 373, 395,
    180, 459, 285, 533, 671, 594,
    742, 832, 467, 725, 600, 75,
    709, 252, 89, 911, 530, 568,
    556, 563, 283, 251, 935, 137,
    47, 597, 274, 7, 170, 684,
    22, 505, 202, 721, 405, 623,
    934, 843, 526, 388, 197, 687,
    289, 272, 463, 726, 898, 220,
    510, 672, 254, 846, 294, 237,
    // Bin capacity
    1289, 1418, 1154, 1530, 1335, 1344
  };
  const int i6_18_6_1_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    801, 260, 756, 655, 344, 910,
    451, 454, 335, 437, 357, 468,
    784, 100, 873, 12, 968, 258,
    915, 447, 255, 99, 501, 357,
    876, 601, 464, 616, 311, 795,
    334, 446, 309, 706, 543, 883,
    590, 385, 583, 98, 143, 318,
    981, 837, 749, 509, 653, 600,
    885, 220, 920, 295, 9, 807,
    931, 261, 858, 343, 836, 896,
    631, 959, 863, 888, 172, 238,
    247, 275, 332, 519, 309, 901,
    775, 583, 236, 913, 880, 88,
    95, 32, 664, 246, 733, 750,
    716, 976, 407, 213, 445, 130,
    691, 355, 412, 234, 227, 992,
    184, 640, 156, 278, 726, 556,
    732, 947, 659, 992, 238, 302,
    // Bin capacity
    1956, 1478, 1655, 1356, 1414, 1726
  };
  const int i6_18_6_2_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    326, 909, 52, 511, 652, 719,
    370, 714, 939, 96, 597, 542,
    884, 929, 240, 444, 560, 490,
    894, 653, 117, 277, 722, 599,
    292, 866, 94, 371, 507, 267,
    590, 332, 628, 513, 768, 550,
    630, 206, 499, 435, 967, 273,
    393, 116, 964, 490, 875, 581,
    843, 769, 954, 359, 851, 707,
    552, 97, 825, 72, 465, 400,
    617, 293, 529, 153, 163, 450,
    485, 221, 717, 289, 785, 887,
    861, 971, 234, 731, 184, 626,
    593, 596, 616, 373, 584, 253,
    383, 309, 537, 109, 328, 871,
    978, 691, 491, 678, 303, 83,
    35, 472, 50, 953, 965, 19,
    971, 879, 92, 834, 295, 9,
    // Bin capacity
    1819, 1704, 1459, 1307, 1798, 1416
  };
  const int i6_18_6_3_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    227, 777, 639, 648, 747, 526,
    474, 637, 274, 655, 372, 558,
    226, 64, 974, 601, 74, 252,
    885, 40, 930, 32, 232, 423,
    761, 242, 710, 265, 919, 378,
    742, 662, 800, 559, 808, 273,
    236, 141, 347, 182, 938, 864,
    910, 573, 203, 409, 835, 761,
    631, 225, 993, 761, 375, 967,
    67, 656, 997, 414, 612, 691,
    257, 521, 476, 727, 712, 380,
    1000, 56, 378, 341, 483, 991,
    270, 729, 763, 197, 704, 0,
    907, 265, 796, 810, 566, 765,
    541, 765, 621, 920, 551, 43,
    956, 938, 182, 243, 611, 420,
    126, 366, 10, 308, 861, 579,
    654, 452, 747, 549, 548, 825,
    // Bin capacity
    1695, 1393, 1861, 1480, 1880, 1665
  };
  const int i6_18_6_4_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    175, 382, 607, 234, 590, 721,
    108, 328, 777, 532, 615, 691,
    878, 769, 532, 790, 305, 660,
    139, 497, 93, 56, 429, 99,
    923, 474, 510, 264, 429, 236,
    197, 539, 891, 525, 295, 745,
    446, 922, 262, 617, 303, 826,
    449, 168, 666, 635, 329, 900,
    777, 789, 31, 4, 140, 748,
    288, 196, 289, 316, 402, 653,
    500, 859, 333, 597, 455, 395,
    580, 983, 742, 587, 474, 955,
    603, 143, 116, 118, 666, 60,
    240, 719, 6, 823, 458, 598,
    481, 905, 430, 339, 530, 832,
    72, 874, 238, 873, 835, 105,
    172, 102, 781, 645, 177, 776,
    315, 269, 412, 555, 352, 243,
    // Bin capacity
    1273, 1720, 1338, 1476, 1350, 1776
  };
  const int i6_18_6_5_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    124, 866, 555, 281, 763, 312,
    524, 315, 821, 75, 558, 172,
    467, 187, 147, 23, 243, 56,
    306, 965, 447, 730, 743, 156,
    762, 262, 883, 818, 208, 133,
    492, 713, 331, 199, 542, 65,
    185, 800, 546, 365, 866, 575,
    477, 207, 109, 667, 917, 169,
    331, 986, 519, 597, 779, 186,
    837, 211, 642, 746, 726, 181,
    707, 867, 951, 926, 66, 813,
    678, 296, 344, 814, 499, 932,
    888, 33, 203, 601, 67, 749,
    482, 869, 707, 138, 764, 536,
    153, 845, 11, 756, 874, 8,
    656, 424, 419, 160, 628, 626,
    611, 476, 351, 667, 707, 793,
    825, 32, 34, 816, 405, 148,
    // Bin capacity
    1664, 1637, 1404, 1642, 1813, 1157
  };
  const int i6_18_6_6_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    567, 342, 105, 797, 796, 7,
    378, 37, 437, 358, 800, 675,
    545, 256, 575, 979, 800, 360,
    434, 628, 903, 539, 207, 377,
    830, 221, 579, 518, 739, 969,
    810, 790, 205, 880, 630, 477,
    925, 307, 787, 7, 240, 613,
    283, 946, 767, 501, 365, 691,
    607, 581, 938, 912, 484, 236,
    768, 533, 212, 820, 951, 825,
    315, 823, 241, 844, 921, 347,
    29, 139, 846, 638, 864, 268,
    11, 869, 118, 820, 984, 604,
    742, 874, 413, 895, 603, 870,
    28, 556, 418, 984, 393, 509,
    109, 169, 277, 970, 571, 472,
    768, 12, 993, 327, 971, 266,
    160, 630, 856, 707, 299, 67,
    // Bin capacity
    1468, 1540, 1709, 2208, 2053, 1526
  };
  const int i6_18_6_7_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    676, 775, 880, 609, 318, 703,
    622, 583, 60, 7, 397, 852,
    79, 643, 988, 22, 528, 109,
    865, 162, 461, 658, 910, 953,
    13, 613, 201, 52, 729, 277,
    85, 115, 315, 317, 649, 231,
    163, 974, 959, 780, 105, 130,
    416, 31, 110, 691, 779, 547,
    410, 536, 687, 648, 591, 22,
    410, 934, 370, 283, 786, 240,
    218, 720, 86, 357, 158, 203,
    982, 432, 25, 753, 24, 132,
    125, 951, 85, 751, 274, 446,
    224, 722, 953, 51, 620, 902,
    890, 124, 381, 43, 741, 299,
    577, 739, 195, 14, 837, 812,
    13, 45, 427, 541, 45, 294,
    504, 778, 671, 306, 708, 881,
    // Bin capacity
    1297, 1762, 1401, 1228, 1641, 1433
  };
  const int i6_18_6_8_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    772, 745, 280, 306, 730, 513,
    303, 914, 8, 933, 677, 18,
    854, 707, 731, 247, 60, 101,
    372, 578, 921, 226, 97, 997,
    970, 107, 681, 510, 325, 511,
    151, 698, 892, 515, 311, 659,
    408, 260, 590, 246, 252, 573,
    51, 984, 29, 253, 325, 6,
    0, 9, 700, 979, 159, 177,
    969, 675, 723, 924, 983, 665,
    161, 692, 415, 702, 10, 865,
    236, 368, 813, 723, 540, 552,
    948, 246, 354, 702, 258, 17,
    256, 783, 665, 267, 972, 943,
    119, 588, 878, 574, 549, 392,
    25, 79, 876, 282, 950, 64,
    769, 765, 5, 455, 299, 860,
    201, 690, 334, 353, 207, 599,
    // Bin capacity
    1362, 1780, 1782, 1656, 1387, 1533
  };
  const int i6_18_6_9_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    367, 281, 149, 83, 59, 532,
    330, 267, 982, 738, 988, 43,
    357, 805, 28, 1000, 872, 235,
    137, 58, 242, 938, 425, 272,
    268, 366, 613, 662, 258, 140,
    333, 579, 33, 140, 51, 648,
    724, 45, 456, 905, 95, 819,
    420, 709, 53, 771, 583, 985,
    976, 815, 865, 634, 409, 638,
    901, 367, 455, 126, 845, 711,
    246, 111, 623, 674, 270, 338,
    175, 667, 345, 367, 783, 75,
    817, 690, 734, 540, 839, 993,
    861, 730, 198, 298, 873, 475,
    501, 911, 665, 153, 826, 563,
    442, 212, 487, 295, 877, 859,
    132, 849, 558, 336, 643, 485,
    751, 478, 40, 484, 11, 6,
    // Bin capacity
    1588, 1625, 1368, 1662, 1764, 1602
  };
  const int i6_18_6_10_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    200, 385, 360, 577, 108, 601,
    86, 94, 129, 365, 193, 430,
    835, 15, 1000, 12, 880, 21,
    141, 573, 709, 542, 507, 844,
    401, 244, 654, 364, 938, 814,
    795, 258, 734, 851, 537, 870,
    136, 472, 423, 169, 74, 805,
    956, 312, 923, 854, 390, 365,
    463, 987, 877, 184, 201, 427,
    360, 671, 568, 833, 244, 984,
    209, 890, 603, 514, 20, 852,
    821, 158, 149, 969, 875, 347,
    567, 371, 248, 159, 651, 841,
    338, 526, 525, 838, 423, 276,
    22, 974, 547, 733, 525, 414,
    15, 662, 672, 741, 21, 788,
    37, 225, 875, 936, 381, 381,
    598, 598, 357, 731, 671, 868,
    // Bin capacity
    1280, 1543, 1899, 1902, 1401, 2004
  };
  const int i6_18_6_11_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    285, 594, 159, 554, 13, 509,
    109, 274, 538, 317, 641, 234,
    25, 286, 41, 994, 590, 1,
    383, 539, 737, 365, 908, 777,
    272, 28, 741, 229, 180, 70,
    76, 18, 136, 710, 632, 675,
    557, 953, 667, 659, 108, 117,
    846, 550, 107, 249, 764, 678,
    616, 924, 509, 66, 382, 258,
    394, 167, 624, 516, 917, 891,
    627, 625, 777, 505, 976, 198,
    510, 848, 53, 387, 388, 176,
    639, 522, 585, 832, 509, 888,
    539, 387, 674, 909, 235, 787,
    639, 18, 63, 212, 328, 222,
    127, 995, 105, 297, 207, 266,
    596, 906, 508, 842, 980, 700,
    504, 515, 132, 315, 590, 494,
    // Bin capacity
    1433, 1693, 1324, 1658, 1730, 1470
  };
  const int i6_18_6_12_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    731, 737, 503, 155, 30, 66,
    463, 820, 166, 66, 97, 242,
    314, 847, 194, 865, 293, 313,
    583, 838, 361, 964, 785, 481,
    400, 871, 410, 257, 43, 170,
    260, 124, 614, 480, 994, 296,
    806, 337, 743, 386, 15, 171,
    289, 429, 330, 260, 714, 302,
    37, 663, 93, 442, 26, 623,
    698, 307, 418, 720, 369, 965,
    636, 425, 416, 355, 263, 453,
    549, 601, 712, 672, 937, 310,
    814, 804, 577, 555, 753, 687,
    685, 353, 833, 181, 884, 962,
    717, 246, 29, 365, 319, 102,
    488, 531, 353, 569, 697, 829,
    281, 310, 695, 45, 778, 32,
    233, 445, 328, 510, 359, 881,
    // Bin capacity
    1678, 1809, 1452, 1465, 1560, 1472
  };
  const int i6_18_6_13_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    305, 661, 705, 797, 565, 798,
    986, 790, 732, 578, 6, 661,
    166, 927, 226, 727, 283, 244,
    954, 342, 909, 473, 247, 473,
    101, 478, 511, 830, 178, 85,
    301, 827, 465, 54, 359, 854,
    934, 930, 978, 474, 182, 443,
    525, 302, 639, 198, 63, 417,
    821, 525, 338, 482, 415, 654,
    572, 711, 236, 323, 102, 853,
    376, 531, 214, 62, 603, 313,
    34, 786, 482, 112, 492, 567,
    721, 317, 461, 256, 345, 589,
    361, 75, 165, 270, 810, 880,
    648, 854, 265, 811, 535, 442,
    963, 281, 31, 584, 140, 382,
    852, 859, 989, 229, 108, 844,
    280, 81, 608, 839, 304, 243,
    // Bin capacity
    1865, 1936, 1687, 1526, 1081, 1835
  };
  const int i6_18_6_14_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    634, 373, 66, 289, 712, 553,
    218, 461, 95, 916, 65, 800,
    474, 968, 463, 774, 341, 588,
    731, 816, 437, 222, 673, 467,
    835, 627, 702, 800, 286, 56,
    507, 64, 105, 607, 189, 817,
    595, 747, 32, 741, 65, 467,
    724, 25, 87, 765, 767, 73,
    848, 679, 98, 9, 400, 690,
    822, 877, 835, 868, 568, 711,
    200, 290, 652, 260, 836, 361,
    609, 52, 487, 972, 316, 531,
    739, 83, 585, 638, 503, 933,
    368, 396, 877, 390, 259, 941,
    336, 509, 485, 269, 250, 981,
    19, 403, 812, 81, 605, 484,
    38, 928, 594, 370, 393, 172,
    737, 538, 616, 619, 949, 734,
    // Bin capacity
    1793, 1679, 1526, 1823, 1554, 1969
  };
  const int i6_18_6_15_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    10, 37, 723, 886, 62, 341,
    979, 390, 55, 661, 614, 523,
    168, 206, 142, 937, 375, 562,
    478, 289, 33, 178, 888, 619,
    456, 919, 820, 627, 883, 298,
    240, 300, 8, 328, 692, 204,
    973, 315, 214, 552, 422, 305,
    516, 541, 559, 908, 867, 995,
    633, 394, 321, 643, 534, 859,
    991, 61, 580, 778, 468, 680,
    849, 341, 520, 137, 156, 456,
    756, 235, 426, 234, 251, 193,
    786, 954, 993, 661, 626, 617,
    520, 381, 336, 741, 790, 29,
    243, 111, 431, 691, 323, 943,
    426, 998, 327, 961, 122, 721,
    466, 515, 374, 77, 205, 701,
    416, 227, 145, 905, 479, 228,
    // Bin capacity
    1899, 1383, 1344, 2091, 1679, 1778
  };
  const int i6_18_6_16_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    538, 387, 609, 118, 146, 472,
    232, 703, 436, 177, 90, 953,
    652, 286, 926, 809, 843, 179,
    256, 906, 960, 691, 236, 454,
    529, 660, 600, 97, 142, 967,
    683, 929, 24, 322, 410, 139,
    696, 723, 533, 198, 330, 536,
    890, 748, 61, 440, 294, 596,
    148, 670, 338, 651, 13, 742,
    917, 49, 880, 419, 439, 498,
    240, 92, 894, 550, 276, 261,
    358, 225, 839, 485, 72, 335,
    580, 742, 517, 815, 228, 454,
    838, 949, 382, 4, 236, 131,
    891, 322, 848, 603, 944, 731,
    211, 411, 810, 874, 148, 653,
    638, 459, 721, 482, 539, 810,
    422, 887, 107, 447, 853, 636,
    // Bin capacity
    1880, 1962, 2028, 1582, 1207, 1846
  };
  const int i6_18_6_17_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    62, 786, 763, 681, 9, 105,
    61, 633, 643, 728, 143, 719,
    969, 633, 507, 702, 587, 511,
    129, 684, 604, 925, 24, 453,
    616, 505, 184, 280, 111, 709,
    774, 982, 898, 944, 843, 126,
    98, 115, 679, 44, 416, 358,
    155, 617, 227, 539, 44, 113,
    113, 992, 173, 976, 967, 194,
    327, 863, 682, 175, 433, 936,
    629, 955, 544, 509, 264, 502,
    641, 4, 197, 341, 964, 517,
    581, 632, 10, 783, 194, 778,
    467, 580, 168, 593, 481, 967,
    237, 228, 691, 345, 184, 866,
    71, 162, 264, 137, 879, 159,
    516, 577, 927, 622, 175, 459,
    611, 235, 676, 819, 309, 318,
    // Bin capacity
    1377, 1986, 1724, 1978, 1371, 1715
  };
  const int i6_18_6_18_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    838, 158, 137, 499, 864, 217,
    327, 591, 601, 189, 126, 608,
    200, 596, 343, 234, 836, 119,
    335, 472, 706, 247, 85, 294,
    647, 356, 332, 118, 539, 818,
    709, 649, 817, 923, 673, 589,
    883, 410, 272, 807, 955, 986,
    864, 641, 510, 302, 224, 182,
    323, 952, 634, 125, 214, 85,
    761, 506, 494, 354, 306, 435,
    646, 898, 270, 758, 540, 740,
    29, 942, 327, 940, 660, 457,
    687, 76, 766, 147, 693, 537,
    121, 317, 579, 557, 227, 349,
    776, 803, 221, 992, 511, 328,
    468, 736, 638, 726, 361, 420,
    986, 550, 249, 357, 996, 315,
    47, 699, 48, 89, 545, 149,
    // Bin capacity
    1898, 2036, 1563, 1645, 1840, 1501
  };
  const int i6_18_6_19_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    64, 961, 343, 870, 123, 664,
    779, 546, 483, 915, 317, 741,
    721, 459, 517, 908, 939, 844,
    312, 476, 479, 484, 959, 90,
    814, 417, 971, 97, 364, 622,
    930, 160, 940, 866, 275, 346,
    308, 594, 461, 149, 360, 406,
    717, 743, 395, 678, 33, 430,
    279, 580, 52, 236, 49, 577,
    843, 59, 93, 395, 253, 905,
    526, 527, 69, 170, 435, 859,
    175, 865, 593, 275, 365, 416,
    444, 928, 549, 203, 479, 957,
    218, 30, 790, 402, 481, 550,
    460, 55, 307, 75, 407, 967,
    85, 591, 260, 729, 780, 480,
    235, 653, 389, 823, 789, 255,
    356, 808, 308, 754, 851, 86,
    // Bin capacity
    1640, 1875, 1587, 1791, 1639, 2023
  };
  const int i6_18_6_20_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    622, 195, 984, 735, 87, 649,
    710, 978, 213, 274, 209, 754,
    878, 312, 49, 160, 482, 904,
    416, 85, 126, 252, 45, 205,
    88, 944, 536, 896, 489, 823,
    38, 106, 888, 837, 239, 665,
    987, 612, 670, 621, 457, 813,
    76, 296, 398, 390, 145, 143,
    25, 828, 801, 905, 618, 395,
    825, 516, 530, 689, 435, 335,
    398, 187, 21, 213, 417, 962,
    279, 739, 943, 219, 582, 960,
    861, 878, 394, 894, 525, 810,
    417, 994, 992, 943, 367, 5,
    232, 872, 859, 803, 430, 415,
    211, 382, 996, 905, 823, 111,
    816, 361, 766, 666, 568, 457,
    393, 167, 189, 937, 608, 327,
    // Bin capacity
    1655, 1891, 2071, 2268, 1506, 1947
  };
  const int i6_18_6_21_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    599, 790, 691, 775, 959, 175,
    28, 300, 21, 120, 525, 553,
    864, 307, 632, 516, 200, 634,
    426, 436, 89, 841, 354, 364,
    886, 993, 438, 106, 141, 550,
    543, 600, 167, 707, 847, 961,
    764, 627, 625, 62, 213, 523,
    967, 903, 852, 636, 446, 990,
    407, 511, 472, 746, 297, 659,
    581, 42, 980, 505, 383, 778,
    90, 704, 68, 227, 539, 515,
    208, 990, 920, 559, 33, 799,
    414, 357, 733, 546, 117, 430,
    890, 597, 283, 58, 417, 57,
    785, 977, 952, 665, 519, 4,
    341, 740, 134, 605, 992, 204,
    942, 652, 140, 557, 469, 686,
    243, 892, 135, 351, 909, 227,
    // Bin capacity
    2013, 2303, 1681, 1731, 1686, 1837
  };
  const int i6_18_6_22_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    44, 854, 903, 801, 741, 85,
    395, 134, 793, 893, 939, 416,
    916, 475, 394, 250, 645, 982,
    318, 783, 646, 502, 995, 839,
    721, 614, 907, 286, 994, 905,
    853, 400, 650, 756, 27, 606,
    169, 243, 376, 124, 545, 550,
    866, 650, 538, 92, 194, 948,
    721, 842, 921, 401, 10, 436,
    592, 208, 987, 678, 461, 502,
    677, 942, 804, 929, 1, 673,
    584, 697, 198, 776, 899, 993,
    570, 893, 422, 349, 666, 116,
    422, 471, 457, 899, 239, 461,
    701, 750, 207, 173, 891, 339,
    865, 116, 833, 865, 175, 154,
    731, 697, 654, 738, 223, 782,
    201, 421, 322, 433, 205, 673,
    // Bin capacity
    2104, 2072, 2240, 2023, 1800, 2127
  };
  const int i6_18_6_23_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    674, 607, 807, 458, 188, 699,
    227, 22, 12, 133, 426, 288,
    991, 86, 614, 486, 181, 560,
    115, 841, 318, 940, 89, 841,
    126, 422, 974, 415, 820, 749,
    912, 222, 512, 262, 574, 861,
    541, 679, 682, 838, 184, 154,
    20, 373, 696, 136, 219, 323,
    26, 657, 259, 525, 657, 608,
    504, 693, 902, 337, 502, 547,
    872, 910, 904, 735, 895, 421,
    350, 277, 68, 277, 438, 915,
    877, 67, 736, 782, 563, 169,
    808, 212, 794, 7, 455, 265,
    660, 873, 391, 716, 988, 916,
    135, 265, 721, 392, 747, 326,
    995, 327, 287, 303, 383, 877,
    94, 319, 578, 327, 113, 678,
    // Bin capacity
    1831, 1610, 2103, 1655, 1727, 2091
  };
  const int i6_18_6_24_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    363, 662, 340, 724, 549, 78,
    95, 188, 675, 477, 102, 851,
    116, 535, 532, 756, 398, 994,
    992, 221, 99, 421, 760, 781,
    528, 151, 885, 82, 604, 39,
    167, 231, 598, 304, 349, 318,
    161, 529, 331, 14, 172, 233,
    904, 632, 174, 418, 960, 88,
    5, 145, 657, 193, 204, 908,
    963, 444, 321, 168, 203, 175,
    257, 608, 644, 80, 634, 804,
    741, 379, 214, 618, 750, 149,
    539, 900, 911, 786, 545, 747,
    721, 956, 532, 983, 631, 135,
    195, 297, 943, 857, 460, 201,
    244, 239, 468, 685, 796, 817,
    17, 233, 329, 710, 765, 406,
    8, 316, 333, 227, 934, 930,
    // Bin capacity
    1450, 1585, 1858, 1758, 2029, 1789
  };
  const int i6_18_6_25_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    449, 469, 882, 222, 490, 226,
    186, 628, 621, 485, 179, 374,
    643, 325, 207, 186, 960, 322,
    644, 948, 976, 76, 65, 213,
    938, 187, 31, 629, 64, 867,
    239, 509, 460, 299, 735, 671,
    354, 620, 136, 67, 158, 802,
    911, 959, 787, 438, 881, 408,
    44, 736, 517, 620, 19, 747,
    622, 274, 97, 391, 375, 290,
    403, 90, 652, 590, 205, 800,
    170, 788, 358, 221, 227, 3,
    570, 941, 687, 366, 836, 230,
    333, 552, 832, 360, 693, 798,
    326, 871, 129, 81, 31, 887,
    354, 429, 242, 428, 851, 352,
    665, 731, 870, 467, 973, 685,
    339, 115, 631, 423, 176, 648,
    // Bin capacity
    1707, 2120, 1899, 1323, 1650, 1943
  };
  const int i6_18_6_26_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    822, 647, 334, 104, 364, 98,
    637, 418, 892, 962, 137, 179,
    162, 268, 506, 188, 38, 735,
    364, 980, 770, 116, 76, 237,
    452, 877, 583, 206, 334, 18,
    449, 227, 193, 844, 486, 946,
    788, 607, 709, 228, 617, 735,
    893, 657, 265, 595, 29, 543,
    503, 193, 34, 901, 807, 522,
    740, 642, 813, 945, 287, 595,
    675, 93, 438, 940, 83, 489,
    90, 918, 825, 720, 116, 185,
    262, 620, 55, 47, 456, 102,
    510, 107, 488, 825, 800, 749,
    988, 670, 996, 506, 981, 714,
    831, 671, 746, 63, 905, 535,
    445, 104, 862, 540, 189, 127,
    854, 193, 557, 504, 966, 818,
    // Bin capacity
    2198, 1868, 2114, 1940, 1611, 1749
  };
  const int i6_18_6_27_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    486, 97, 982, 806, 607, 140,
    933, 989, 751, 602, 621, 333,
    771, 586, 884, 307, 55, 46,
    242, 331, 888, 625, 784, 520,
    656, 104, 450, 262, 149, 820,
    859, 360, 807, 839, 289, 204,
    620, 70, 362, 424, 994, 89,
    918, 956, 123, 885, 732, 618,
    857, 402, 888, 74, 226, 438,
    114, 198, 278, 360, 18, 20,
    548, 55, 296, 744, 837, 59,
    921, 131, 545, 796, 972, 125,
    975, 792, 398, 594, 276, 425,
    301, 39, 594, 816, 726, 345,
    650, 924, 197, 515, 814, 285,
    830, 558, 609, 714, 97, 896,
    261, 948, 303, 81, 267, 733,
    914, 680, 973, 138, 538, 463,
    // Bin capacity
    2510, 1740, 2187, 2029, 1906, 1389
  };
  const int i6_18_6_28_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    435, 321, 254, 768, 748, 995,
    294, 180, 894, 546, 464, 6,
    454, 699, 570, 329, 556, 5,
    642, 449, 534, 359, 555, 54,
    581, 266, 995, 712, 921, 706,
    400, 517, 817, 900, 852, 752,
    423, 503, 603, 838, 352, 944,
    703, 350, 896, 699, 740, 544,
    82, 602, 205, 627, 59, 830,
    257, 890, 899, 355, 589, 28,
    533, 943, 189, 853, 874, 735,
    569, 447, 37, 136, 961, 205,
    405, 936, 388, 930, 273, 684,
    223, 336, 731, 972, 319, 606,
    44, 482, 905, 477, 226, 287,
    948, 30, 679, 296, 937, 97,
    535, 307, 422, 604, 615, 246,
    290, 532, 864, 939, 528, 426,
    // Bin capacity
    1668, 1876, 2322, 2420, 2255, 1739
  };
  const int i6_18_6_29_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    127, 193, 982, 455, 333, 149,
    56, 844, 13, 116, 730, 734,
    93, 60, 954, 666, 979, 14,
    183, 460, 251, 460, 356, 825,
    67, 381, 172, 374, 816, 552,
    68, 565, 789, 834, 703, 809,
    966, 987, 385, 752, 736, 980,
    13, 511, 899, 237, 218, 74,
    770, 18, 768, 290, 359, 136,
    591, 555, 266, 586, 403, 849,
    389, 60, 54, 68, 937, 896,
    647, 284, 907, 270, 808, 133,
    468, 529, 100, 207, 447, 605,
    752, 876, 800, 384, 572, 716,
    525, 501, 564, 501, 576, 709,
    593, 15, 419, 95, 60, 749,
    494, 50, 991, 477, 382, 348,
    701, 499, 179, 736, 370, 43,
    // Bin capacity
    1614, 1589, 2041, 1615, 2104, 2005
  };
  const int i6_18_6_30_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    751, 285, 918, 686, 714, 629,
    723, 876, 296, 839, 701, 135,
    994, 676, 2, 88, 198, 433,
    202, 699, 476, 733, 408, 398,
    339, 407, 125, 745, 237, 188,
    899, 293, 63, 434, 861, 935,
    914, 544, 223, 77, 944, 588,
    924, 573, 364, 234, 117, 668,
    83, 879, 840, 534, 618, 145,
    619, 447, 966, 199, 128, 84,
    756, 145, 611, 743, 553, 91,
    789, 664, 199, 939, 859, 66,
    284, 141, 559, 304, 350, 7,
    926, 893, 663, 973, 520, 541,
    197, 37, 21, 296, 78, 767,
    665, 316, 197, 371, 237, 938,
    164, 531, 828, 806, 421, 841,
    524, 472, 810, 379, 586, 34,
    // Bin capacity
    2330, 1924, 1769, 2033, 1849, 1623
  };
  const int i6_18_6_31_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    202, 495, 876, 778, 553, 25,
    756, 54, 919, 646, 222, 740,
    635, 401, 303, 830, 225, 694,
    754, 9, 888, 696, 74, 414,
    51, 29, 938, 647, 939, 526,
    379, 231, 319, 869, 188, 739,
    620, 511, 708, 118, 427, 863,
    721, 288, 570, 870, 334, 29,
    888, 834, 475, 297, 412, 979,
    126, 790, 814, 476, 678, 214,
    606, 89, 241, 426, 820, 545,
    827, 580, 587, 276, 732, 116,
    657, 356, 247, 398, 303, 382,
    177, 706, 134, 29, 28, 690,
    162, 837, 680, 234, 295, 338,
    674, 430, 435, 383, 118, 184,
    246, 760, 486, 294, 121, 614,
    641, 112, 232, 58, 998, 691,
    // Bin capacity
    1992, 1641, 2152, 1818, 1631, 1918
  };
  const int i6_18_6_32_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    145, 511, 970, 986, 883, 346,
    636, 530, 519, 574, 978, 159,
    927, 718, 407, 311, 198, 732,
    365, 945, 92, 451, 121, 437,
    490, 619, 762, 49, 739, 199,
    825, 277, 218, 765, 814, 770,
    267, 434, 207, 859, 311, 311,
    742, 843, 676, 234, 0, 733,
    442, 744, 225, 63, 940, 38,
    12, 491, 393, 43, 101, 244,
    718, 340, 213, 672, 54, 733,
    602, 853, 369, 801, 497, 132,
    551, 353, 603, 612, 80, 646,
    759, 701, 450, 789, 253, 12,
    65, 742, 472, 446, 766, 665,
    967, 710, 592, 982, 947, 956,
    183, 89, 641, 957, 331, 558,
    208, 386, 565, 174, 359, 750,
    // Bin capacity
    1959, 2263, 1843, 2149, 1842, 1853
  };
  const int i6_18_6_33_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    938, 553, 285, 756, 52, 643,
    610, 434, 661, 411, 271, 686,
    106, 327, 847, 354, 265, 933,
    586, 376, 243, 30, 104, 111,
    244, 36, 765, 660, 273, 725,
    526, 398, 585, 497, 328, 854,
    130, 672, 657, 795, 723, 742,
    412, 420, 868, 554, 612, 58,
    574, 895, 218, 756, 339, 257,
    53, 17, 502, 756, 670, 969,
    932, 172, 809, 236, 337, 459,
    202, 668, 97, 914, 122, 282,
    359, 778, 591, 571, 558, 226,
    170, 472, 761, 324, 817, 844,
    295, 914, 126, 953, 397, 838,
    298, 359, 607, 826, 904, 159,
    740, 606, 198, 594, 60, 92,
    247, 474, 435, 867, 483, 966,
    // Bin capacity
    1646, 1900, 2052, 2406, 1622, 2183
  };
  const int i6_18_6_34_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    249, 265, 393, 126, 189, 516,
    213, 575, 547, 933, 191, 581,
    325, 146, 847, 588, 703, 403,
    772, 504, 322, 758, 259, 217,
    576, 868, 632, 143, 163, 191,
    832, 576, 398, 995, 491, 349,
    788, 259, 353, 759, 818, 551,
    605, 67, 872, 494, 153, 247,
    255, 954, 245, 449, 982, 136,
    662, 626, 94, 223, 826, 596,
    767, 760, 241, 399, 999, 952,
    53, 410, 617, 913, 999, 621,
    872, 430, 829, 5, 447, 984,
    988, 13, 443, 560, 999, 590,
    769, 295, 503, 936, 170, 701,
    512, 624, 327, 505, 58, 428,
    966, 651, 957, 995, 931, 545,
    224, 464, 484, 949, 959, 23,
    // Bin capacity
    2329, 1896, 2034, 2397, 2309, 1928
  };
  const int i6_18_6_35_5[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    724, 681, 707, 334, 92, 188,
    1000, 175, 602, 491, 616, 437,
    947, 803, 25, 790, 261, 484,
    190, 152, 301, 217, 62, 538,
    160, 253, 800, 274, 206, 193,
    122, 843, 581, 182, 332, 856,
    328, 881, 717, 820, 477, 41,
    984, 236, 731, 959, 183, 126,
    783, 841, 43, 77, 916, 253,
    727, 333, 494, 691, 684, 387,
    9, 11, 827, 464, 326, 218,
    992, 732, 871, 89, 882, 69,
    248, 693, 434, 436, 506, 682,
    800, 239, 256, 724, 906, 530,
    57, 365, 928, 434, 842, 21,
    622, 339, 139, 890, 555, 581,
    406, 543, 349, 733, 250, 495,
    431, 943, 891, 193, 737, 577,
    // Bin capacity
    2145, 2040, 2182, 1980, 1988, 1503
  };
  const int i6_18_6_0_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    793, 793, 399, 125, 977, 322,
    382, 827, 215, 866, 913, 252,
    62, 918, 549, 111, 133, 465,
    295, 49, 348, 73, 528, 210,
    698, 773, 298, 480, 980, 365,
    693, 258, 430, 317, 606, 896,
    806, 59, 523, 54, 516, 129,
    275, 995, 706, 590, 535, 256,
    478, 488, 550, 511, 2, 204,
    475, 63, 516, 122, 458, 305,
    330, 706, 379, 814, 455, 314,
    757, 429, 567, 151, 175, 88,
    248, 115, 61, 831, 219, 836,
    772, 833, 576, 605, 824, 701,
    884, 40, 381, 607, 918, 585,
    20, 459, 935, 474, 680, 348,
    945, 676, 757, 507, 708, 942,
    601, 37, 652, 771, 558, 627,
    // Bin capacity
    1586, 1420, 1474, 1335, 1698, 1308
  };
  const int i6_18_6_1_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    17, 964, 712, 699, 430, 531,
    185, 364, 71, 498, 937, 207,
    668, 888, 831, 804, 229, 402,
    518, 400, 325, 513, 745, 374,
    168, 927, 131, 599, 313, 27,
    405, 777, 566, 435, 401, 423,
    79, 959, 642, 320, 601, 112,
    423, 269, 844, 122, 627, 835,
    709, 689, 332, 782, 684, 528,
    164, 983, 507, 900, 832, 76,
    295, 304, 374, 47, 699, 679,
    553, 278, 275, 661, 221, 115,
    803, 256, 60, 267, 910, 167,
    16, 697, 931, 319, 798, 31,
    560, 221, 944, 247, 963, 891,
    0, 450, 1000, 277, 715, 222,
    976, 756, 671, 285, 772, 240,
    451, 462, 286, 840, 31, 337,
    // Bin capacity
    1177, 1792, 1600, 1451, 1837, 1044
  };
  const int i6_18_6_2_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    506, 762, 152, 745, 681, 946,
    417, 401, 514, 319, 410, 242,
    266, 794, 780, 607, 23, 934,
    557, 600, 785, 283, 146, 491,
    774, 361, 903, 659, 401, 450,
    459, 261, 275, 34, 916, 270,
    474, 992, 814, 626, 671, 756,
    675, 648, 460, 198, 894, 47,
    179, 431, 549, 139, 318, 500,
    457, 422, 745, 947, 279, 581,
    625, 961, 233, 633, 975, 612,
    772, 186, 414, 525, 845, 991,
    917, 312, 276, 805, 666, 843,
    621, 705, 111, 311, 984, 154,
    431, 779, 481, 300, 959, 771,
    396, 310, 292, 746, 279, 132,
    153, 132, 470, 323, 753, 738,
    565, 106, 202, 893, 468, 613,
    // Bin capacity
    1572, 1558, 1438, 1546, 1814, 1713
  };
  const int i6_18_6_3_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    749, 738, 25, 672, 674, 457,
    758, 518, 20, 467, 186, 952,
    631, 687, 944, 888, 990, 191,
    353, 385, 238, 395, 626, 325,
    854, 691, 959, 690, 735, 428,
    332, 763, 468, 97, 142, 502,
    610, 619, 800, 198, 810, 85,
    235, 515, 70, 679, 812, 567,
    542, 200, 686, 310, 829, 196,
    277, 989, 601, 423, 92, 708,
    492, 384, 953, 443, 838, 272,
    21, 326, 318, 642, 746, 127,
    246, 512, 872, 936, 173, 794,
    856, 430, 825, 544, 569, 142,
    111, 425, 969, 292, 880, 656,
    243, 350, 744, 864, 888, 24,
    172, 684, 546, 595, 960, 100,
    126, 326, 126, 655, 294, 33,
    // Bin capacity
    1307, 1639, 1745, 1681, 1931, 1126
  };
  const int i6_18_6_4_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    670, 249, 545, 508, 918, 137,
    303, 643, 509, 94, 601, 979,
    932, 369, 271, 116, 902, 542,
    235, 247, 862, 316, 857, 654,
    322, 535, 96, 719, 412, 23,
    895, 633, 691, 291, 180, 148,
    833, 199, 765, 349, 491, 923,
    617, 117, 798, 195, 713, 857,
    905, 655, 42, 217, 242, 502,
    828, 852, 67, 880, 697, 891,
    862, 454, 1, 799, 815, 199,
    75, 545, 665, 673, 584, 187,
    837, 434, 300, 788, 1, 791,
    379, 33, 539, 248, 417, 97,
    689, 817, 485, 952, 788, 813,
    471, 204, 761, 762, 824, 512,
    421, 221, 347, 543, 14, 487,
    666, 148, 132, 253, 257, 13,
    // Bin capacity
    1897, 1275, 1366, 1509, 1684, 1518
  };
  const int i6_18_6_5_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    123, 374, 59, 498, 85, 611,
    23, 892, 193, 843, 167, 552,
    611, 632, 205, 251, 966, 17,
    99, 816, 88, 921, 481, 173,
    401, 697, 618, 508, 618, 388,
    249, 726, 183, 936, 222, 867,
    582, 310, 534, 448, 556, 59,
    157, 278, 179, 796, 724, 361,
    235, 398, 590, 777, 876, 510,
    410, 880, 576, 89, 842, 231,
    568, 540, 644, 984, 444, 883,
    599, 60, 97, 548, 156, 45,
    608, 346, 577, 921, 902, 304,
    703, 373, 737, 740, 449, 764,
    983, 734, 862, 697, 536, 103,
    527, 623, 929, 304, 744, 263,
    692, 777, 322, 944, 406, 363,
    418, 804, 599, 664, 197, 625,
    // Bin capacity
    1398, 1796, 1399, 2078, 1640, 1246
  };
  const int i6_18_6_6_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    327, 485, 275, 534, 672, 767,
    642, 752, 299, 534, 333, 252,
    839, 269, 877, 792, 805, 366,
    647, 280, 706, 470, 451, 620,
    695, 50, 918, 330, 186, 868,
    120, 198, 95, 91, 24, 759,
    608, 74, 731, 900, 626, 634,
    907, 974, 262, 602, 517, 914,
    682, 166, 880, 245, 808, 676,
    363, 102, 597, 835, 153, 462,
    535, 588, 918, 71, 276, 277,
    641, 635, 522, 356, 879, 244,
    958, 408, 676, 189, 937, 206,
    73, 522, 281, 611, 935, 711,
    908, 292, 439, 686, 936, 254,
    934, 815, 638, 910, 454, 960,
    350, 461, 136, 514, 817, 189,
    847, 52, 555, 796, 146, 297,
    // Bin capacity
    1957, 1259, 1733, 1673, 1759, 1671
  };
  const int i6_18_6_7_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    806, 337, 776, 575, 801, 553,
    652, 364, 778, 746, 849, 877,
    573, 549, 422, 283, 612, 82,
    991, 337, 76, 861, 420, 863,
    869, 726, 611, 810, 746, 808,
    359, 549, 806, 825, 637, 40,
    672, 243, 257, 683, 374, 609,
    794, 785, 710, 393, 410, 399,
    900, 839, 375, 745, 477, 465,
    488, 880, 413, 308, 602, 989,
    44, 936, 466, 305, 831, 251,
    297, 557, 944, 830, 25, 270,
    409, 964, 587, 912, 500, 620,
    538, 133, 595, 511, 364, 729,
    892, 160, 160, 785, 210, 91,
    625, 186, 376, 464, 746, 838,
    317, 664, 906, 584, 165, 723,
    844, 939, 184, 624, 78, 229,
    // Bin capacity
    1975, 1810, 1684, 2006, 1578, 1683
  };
  const int i6_18_6_8_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    449, 731, 839, 35, 233, 597,
    433, 979, 474, 821, 382, 120,
    771, 528, 719, 128, 923, 459,
    334, 567, 941, 780, 845, 403,
    356, 235, 692, 71, 339, 558,
    363, 707, 175, 803, 720, 4,
    185, 485, 622, 709, 456, 600,
    133, 85, 604, 604, 416, 427,
    824, 227, 927, 837, 926, 916,
    818, 162, 403, 118, 203, 4,
    928, 822, 924, 230, 947, 819,
    239, 218, 296, 333, 99, 438,
    865, 900, 459, 272, 891, 866,
    252, 872, 597, 946, 114, 916,
    897, 37, 990, 862, 907, 275,
    928, 592, 856, 989, 112, 904,
    378, 821, 160, 140, 348, 754,
    249, 469, 964, 500, 568, 61,
    // Bin capacity
    1693, 1699, 2096, 1653, 1698, 1642
  };
  const int i6_18_6_9_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    916, 889, 194, 128, 594, 570,
    706, 432, 675, 3, 714, 884,
    552, 843, 69, 798, 477, 282,
    814, 265, 687, 774, 90, 849,
    413, 445, 867, 673, 429, 345,
    303, 261, 546, 362, 550, 240,
    637, 292, 999, 922, 419, 154,
    215, 979, 4, 624, 779, 219,
    505, 976, 625, 641, 590, 279,
    694, 478, 137, 471, 415, 4,
    255, 180, 526, 8, 108, 902,
    933, 127, 476, 336, 965, 91,
    975, 254, 736, 521, 31, 591,
    732, 237, 54, 44, 371, 171,
    257, 557, 458, 616, 865, 266,
    784, 997, 688, 493, 931, 29,
    762, 597, 193, 912, 568, 827,
    840, 47, 603, 391, 592, 248,
    // Bin capacity
    2052, 1609, 1551, 1584, 1724, 1263
  };
  const int i6_18_6_10_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    188, 262, 535, 677, 374, 448,
    429, 400, 458, 632, 603, 284,
    742, 348, 945, 924, 79, 622,
    895, 724, 421, 807, 856, 911,
    619, 487, 589, 489, 949, 252,
    606, 190, 497, 290, 649, 714,
    714, 395, 691, 556, 936, 445,
    761, 402, 211, 575, 163, 785,
    374, 856, 640, 516, 67, 164,
    563, 169, 70, 2, 765, 249,
    739, 974, 540, 735, 213, 440,
    311, 867, 591, 756, 661, 994,
    637, 55, 877, 523, 410, 827,
    342, 265, 881, 817, 443, 22,
    509, 390, 396, 403, 297, 310,
    148, 437, 945, 48, 505, 671,
    864, 702, 737, 474, 820, 475,
    534, 63, 500, 988, 646, 476,
    // Bin capacity
    1829, 1465, 1930, 1873, 1730, 1667
  };
  const int i6_18_6_11_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    110, 693, 591, 853, 128, 718,
    386, 366, 734, 421, 43, 808,
    422, 589, 146, 371, 165, 517,
    988, 515, 214, 811, 919, 833,
    908, 434, 134, 419, 584, 538,
    48, 88, 901, 488, 873, 284,
    649, 457, 761, 317, 995, 734,
    817, 748, 515, 586, 207, 267,
    604, 208, 580, 900, 595, 801,
    651, 278, 45, 42, 970, 540,
    494, 182, 939, 262, 644, 559,
    183, 209, 190, 585, 372, 469,
    793, 38, 278, 645, 496, 917,
    152, 322, 626, 94, 855, 946,
    168, 751, 278, 235, 408, 132,
    84, 35, 973, 447, 916, 653,
    586, 496, 430, 776, 489, 312,
    331, 292, 949, 587, 878, 86,
    // Bin capacity
    1550, 1240, 1718, 1636, 1950, 1872
  };
  const int i6_18_6_12_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    207, 440, 585, 777, 258, 178,
    343, 373, 856, 10, 785, 852,
    341, 594, 627, 777, 882, 657,
    616, 938, 432, 571, 592, 219,
    834, 712, 96, 625, 539, 901,
    434, 188, 782, 357, 810, 733,
    702, 476, 759, 25, 278, 279,
    219, 896, 606, 227, 125, 778,
    270, 337, 957, 500, 68, 642,
    8, 375, 501, 854, 660, 202,
    3, 16, 416, 146, 858, 152,
    388, 266, 662, 807, 488, 381,
    971, 587, 355, 793, 406, 721,
    132, 481, 586, 407, 602, 525,
    250, 622, 830, 772, 540, 981,
    800, 506, 320, 557, 559, 669,
    96, 567, 259, 131, 501, 286,
    384, 64, 232, 758, 294, 96,
    // Bin capacity
    1307, 1576, 1841, 1698, 1726, 1728
  };
  const int i6_18_6_13_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    82, 577, 680, 41, 61, 951,
    151, 843, 937, 590, 815, 409,
    913, 808, 383, 662, 138, 699,
    54, 650, 358, 93, 355, 518,
    85, 864, 803, 763, 565, 963,
    794, 467, 802, 813, 195, 744,
    86, 905, 927, 527, 835, 518,
    289, 61, 29, 139, 955, 574,
    430, 352, 195, 372, 572, 944,
    620, 802, 363, 701, 761, 580,
    707, 95, 677, 732, 803, 779,
    124, 979, 285, 247, 825, 564,
    865, 683, 33, 326, 830, 772,
    547, 780, 484, 8, 858, 692,
    976, 12, 239, 652, 688, 193,
    410, 192, 558, 976, 652, 978,
    794, 600, 874, 403, 737, 425,
    775, 1, 773, 217, 493, 893,
    // Bin capacity
    1639, 1822, 1771, 1557, 2098, 2297
  };
  const int i6_18_6_14_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    250, 925, 490, 615, 911, 129,
    253, 258, 400, 892, 28, 326,
    451, 791, 339, 859, 972, 165,
    877, 240, 79, 499, 966, 129,
    250, 245, 601, 282, 270, 293,
    566, 578, 742, 390, 585, 197,
    712, 833, 716, 830, 232, 412,
    520, 520, 81, 601, 441, 427,
    478, 566, 900, 756, 652, 403,
    9, 84, 522, 185, 162, 451,
    572, 334, 987, 661, 826, 1,
    927, 143, 630, 282, 843, 326,
    987, 722, 655, 596, 60, 602,
    256, 465, 228, 971, 455, 855,
    916, 10, 500, 905, 757, 748,
    90, 148, 871, 455, 588, 587,
    51, 687, 861, 161, 204, 256,
    110, 10, 945, 49, 633, 251,
    // Bin capacity
    1573, 1437, 2004, 1898, 1822, 1247
  };
  const int i6_18_6_15_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    376, 53, 987, 492, 976, 932,
    990, 682, 424, 931, 796, 934,
    101, 658, 75, 375, 847, 953,
    134, 524, 780, 230, 556, 422,
    886, 857, 511, 804, 902, 440,
    285, 291, 358, 644, 10, 211,
    89, 669, 157, 927, 170, 465,
    463, 994, 196, 275, 310, 170,
    896, 847, 541, 62, 86, 369,
    953, 303, 403, 124, 406, 649,
    75, 847, 878, 59, 563, 40,
    573, 198, 546, 629, 581, 126,
    254, 721, 243, 171, 739, 390,
    563, 233, 447, 660, 278, 460,
    41, 453, 358, 641, 221, 664,
    606, 595, 199, 801, 549, 252,
    964, 16, 895, 777, 548, 611,
    87, 66, 58, 922, 595, 142,
    // Bin capacity
    1598, 1727, 1545, 1826, 1751, 1578
  };
  const int i6_18_6_16_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    420, 310, 4, 788, 176, 62,
    369, 242, 891, 749, 157, 331,
    963, 845, 287, 364, 884, 305,
    566, 293, 389, 58, 172, 55,
    801, 462, 779, 264, 378, 583,
    293, 155, 871, 49, 934, 109,
    873, 51, 101, 741, 842, 569,
    118, 522, 978, 271, 109, 107,
    360, 558, 645, 548, 602, 15,
    339, 737, 771, 126, 333, 145,
    287, 949, 872, 747, 77, 926,
    104, 103, 367, 299, 45, 608,
    268, 869, 124, 461, 595, 305,
    993, 868, 357, 684, 4, 601,
    653, 26, 728, 374, 432, 231,
    290, 117, 904, 809, 90, 989,
    463, 788, 989, 940, 483, 810,
    717, 333, 25, 893, 109, 984,
    // Bin capacity
    1717, 1591, 1950, 1772, 1242, 1496
  };
  const int i6_18_6_17_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    989, 755, 456, 730, 797, 122,
    565, 743, 886, 900, 941, 566,
    562, 815, 430, 456, 458, 988,
    943, 68, 800, 609, 911, 501,
    413, 406, 4, 552, 842, 536,
    574, 52, 586, 603, 393, 732,
    19, 271, 93, 174, 287, 986,
    863, 459, 516, 406, 66, 796,
    99, 459, 169, 443, 676, 934,
    946, 340, 456, 761, 62, 45,
    212, 119, 120, 902, 923, 179,
    466, 414, 278, 243, 720, 425,
    616, 345, 264, 32, 887, 349,
    144, 851, 554, 687, 856, 435,
    357, 26, 561, 721, 337, 98,
    701, 910, 970, 61, 943, 87,
    645, 979, 927, 453, 823, 180,
    413, 961, 942, 967, 924, 464,
    // Bin capacity
    1858, 1750, 1758, 1892, 2310, 1643
  };
  const int i6_18_6_18_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    944, 494, 776, 648, 626, 498,
    714, 123, 832, 850, 907, 368,
    478, 172, 775, 726, 526, 279,
    237, 328, 600, 455, 67, 261,
    843, 420, 864, 381, 637, 941,
    263, 968, 221, 148, 974, 859,
    183, 357, 770, 383, 908, 349,
    702, 59, 850, 368, 920, 305,
    753, 839, 481, 954, 869, 19,
    310, 755, 413, 581, 411, 38,
    20, 651, 665, 676, 176, 699,
    283, 669, 901, 878, 924, 645,
    133, 142, 940, 511, 483, 818,
    824, 495, 798, 319, 443, 930,
    595, 664, 541, 273, 4, 706,
    484, 721, 747, 870, 4, 55,
    166, 513, 751, 752, 389, 162,
    664, 892, 908, 26, 793, 244,
    // Bin capacity
    1691, 1822, 2524, 1928, 1979, 1608
  };
  const int i6_18_6_19_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    712, 282, 571, 466, 303, 186,
    755, 380, 980, 961, 103, 311,
    405, 188, 752, 813, 572, 33,
    275, 61, 146, 37, 97, 767,
    779, 256, 489, 257, 324, 45,
    548, 750, 39, 750, 322, 13,
    624, 402, 147, 37, 564, 4,
    335, 217, 571, 235, 397, 129,
    121, 722, 916, 257, 149, 847,
    213, 615, 68, 895, 74, 531,
    817, 828, 850, 622, 987, 553,
    255, 422, 762, 770, 378, 534,
    9, 624, 782, 506, 304, 39,
    340, 490, 218, 826, 110, 557,
    176, 988, 883, 648, 914, 297,
    478, 930, 985, 224, 984, 325,
    175, 297, 621, 142, 980, 875,
    983, 791, 527, 649, 601, 192,
    // Bin capacity
    1587, 1834, 2045, 1804, 1619, 1238
  };
  const int i6_18_6_20_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    212, 719, 641, 998, 784, 790,
    178, 94, 37, 414, 594, 838,
    707, 113, 806, 149, 264, 829,
    819, 541, 320, 207, 22, 775,
    931, 24, 43, 628, 496, 396,
    948, 264, 733, 555, 296, 834,
    854, 891, 110, 381, 14, 8,
    346, 210, 792, 987, 133, 642,
    483, 197, 388, 107, 178, 573,
    595, 485, 496, 349, 723, 401,
    162, 826, 107, 730, 667, 855,
    707, 45, 617, 664, 533, 372,
    155, 432, 512, 578, 180, 115,
    968, 707, 164, 324, 523, 751,
    806, 787, 573, 90, 77, 735,
    731, 276, 728, 166, 30, 560,
    134, 158, 530, 386, 214, 343,
    415, 598, 609, 716, 565, 574,
    // Bin capacity
    2031, 1474, 1642, 1686, 1259, 2079
  };
  const int i6_18_6_21_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    521, 196, 110, 900, 34, 550,
    483, 466, 970, 877, 947, 618,
    572, 292, 611, 526, 110, 235,
    699, 765, 377, 129, 284, 742,
    821, 546, 447, 393, 981, 657,
    940, 568, 768, 473, 157, 524,
    115, 778, 815, 982, 499, 423,
    902, 620, 932, 963, 383, 282,
    586, 621, 698, 426, 650, 469,
    967, 777, 413, 8, 551, 823,
    413, 956, 92, 168, 980, 657,
    560, 886, 61, 608, 73, 878,
    491, 61, 221, 319, 211, 98,
    439, 908, 13, 452, 361, 967,
    926, 663, 461, 717, 966, 685,
    673, 987, 461, 833, 72, 987,
    732, 479, 322, 228, 748, 371,
    967, 965, 139, 248, 746, 490,
    // Bin capacity
    2382, 2327, 1596, 1866, 1766, 2109
  };
  const int i6_18_6_22_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    320, 202, 162, 210, 956, 622,
    989, 623, 329, 221, 776, 226,
    458, 174, 35, 436, 488, 659,
    580, 63, 358, 454, 463, 919,
    610, 627, 282, 642, 614, 42,
    801, 65, 678, 185, 772, 559,
    777, 880, 526, 632, 382, 445,
    901, 901, 341, 728, 186, 487,
    58, 650, 937, 588, 553, 358,
    620, 845, 381, 837, 634, 976,
    413, 825, 483, 692, 817, 395,
    142, 373, 735, 689, 514, 971,
    990, 456, 137, 103, 839, 774,
    925, 549, 631, 630, 252, 771,
    14, 882, 551, 713, 351, 688,
    747, 897, 688, 822, 562, 279,
    79, 216, 305, 929, 439, 710,
    874, 645, 402, 139, 142, 519,
    // Bin capacity
    2094, 2008, 1619, 1963, 1981, 2115
  };
  const int i6_18_6_23_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    373, 768, 777, 384, 663, 573,
    529, 824, 331, 658, 996, 42,
    364, 959, 659, 268, 261, 549,
    803, 405, 778, 790, 314, 874,
    838, 304, 664, 219, 783, 580,
    972, 244, 3, 781, 109, 134,
    911, 324, 893, 523, 237, 617,
    706, 533, 623, 372, 450, 419,
    630, 589, 39, 242, 132, 406,
    95, 524, 534, 992, 561, 405,
    188, 654, 412, 495, 970, 114,
    776, 756, 612, 192, 739, 937,
    547, 667, 646, 239, 700, 735,
    721, 947, 975, 411, 68, 619,
    590, 264, 70, 614, 192, 765,
    404, 361, 714, 567, 757, 663,
    636, 346, 298, 696, 574, 339,
    255, 317, 661, 146, 636, 13,
    // Bin capacity
    2120, 2007, 1987, 1761, 1875, 1801
  };
  const int i6_18_6_24_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    514, 377, 129, 445, 185, 663,
    433, 134, 5, 485, 427, 224,
    652, 100, 849, 585, 527, 794,
    47, 720, 558, 349, 196, 818,
    32, 939, 353, 781, 878, 793,
    597, 942, 229, 256, 891, 503,
    472, 291, 211, 865, 255, 321,
    197, 497, 983, 505, 347, 493,
    447, 688, 190, 838, 401, 68,
    71, 217, 846, 498, 387, 506,
    267, 857, 752, 911, 601, 540,
    649, 912, 899, 907, 659, 260,
    32, 717, 285, 19, 523, 379,
    594, 12, 875, 651, 10, 839,
    863, 238, 327, 440, 225, 592,
    949, 446, 56, 383, 780, 368,
    1, 14, 569, 594, 843, 372,
    915, 794, 613, 455, 335, 333,
    // Bin capacity
    1598, 1839, 1804, 2060, 1751, 1833
  };
  const int i6_18_6_25_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    294, 631, 767, 909, 122, 931,
    575, 222, 390, 283, 489, 868,
    620, 836, 940, 873, 387, 451,
    319, 840, 115, 678, 57, 789,
    278, 182, 37, 594, 452, 827,
    248, 148, 308, 589, 523, 532,
    28, 175, 258, 610, 240, 807,
    9, 172, 988, 795, 39, 693,
    34, 247, 169, 18, 372, 282,
    970, 832, 658, 298, 313, 310,
    381, 377, 311, 876, 408, 580,
    909, 42, 461, 769, 415, 847,
    731, 220, 879, 94, 516, 291,
    32, 523, 921, 572, 328, 199,
    765, 571, 697, 282, 63, 6,
    186, 815, 551, 761, 127, 362,
    320, 452, 995, 659, 569, 951,
    116, 95, 82, 221, 522, 409,
    // Bin capacity
    1420, 1538, 1985, 2059, 1238, 2112
  };
  const int i6_18_6_26_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    5, 542, 893, 119, 738, 349,
    253, 593, 623, 353, 589, 805,
    798, 254, 198, 459, 618, 457,
    916, 128, 168, 42, 526, 1000,
    171, 558, 875, 601, 412, 827,
    738, 122, 104, 458, 633, 583,
    541, 344, 753, 951, 512, 632,
    614, 884, 457, 443, 966, 635,
    980, 151, 210, 726, 819, 329,
    492, 353, 721, 465, 274, 0,
    175, 877, 534, 100, 651, 37,
    492, 552, 379, 569, 162, 785,
    489, 493, 104, 852, 133, 209,
    603, 144, 298, 923, 768, 661,
    185, 792, 502, 576, 462, 742,
    612, 716, 974, 539, 938, 441,
    16, 733, 910, 70, 630, 908,
    823, 468, 505, 716, 967, 102,
    // Bin capacity
    1870, 1828, 1934, 1883, 2268, 1996
  };
  const int i6_18_6_27_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    508, 735, 714, 552, 182, 671,
    395, 465, 732, 134, 122, 455,
    847, 938, 725, 166, 144, 355,
    719, 377, 150, 896, 140, 847,
    10, 847, 830, 926, 12, 478,
    257, 90, 228, 994, 931, 221,
    824, 826, 121, 409, 334, 451,
    498, 13, 219, 931, 172, 563,
    675, 914, 480, 140, 326, 212,
    77, 366, 898, 652, 942, 172,
    288, 237, 782, 190, 117, 774,
    84, 471, 687, 15, 989, 585,
    353, 272, 658, 575, 14, 388,
    253, 369, 128, 583, 627, 163,
    120, 731, 460, 463, 985, 649,
    465, 930, 408, 638, 354, 759,
    463, 907, 792, 140, 138, 595,
    181, 802, 886, 270, 551, 331,
    // Bin capacity
    1486, 2179, 2096, 1836, 1499, 1835
  };
  const int i6_18_6_28_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    123, 283, 252, 157, 128, 840,
    521, 167, 903, 73, 598, 275,
    22, 872, 750, 82, 751, 13,
    971, 378, 608, 205, 305, 662,
    974, 24, 297, 319, 507, 913,
    33, 865, 369, 198, 875, 951,
    548, 409, 232, 325, 517, 896,
    335, 314, 208, 143, 375, 288,
    995, 315, 615, 974, 636, 733,
    418, 201, 249, 956, 271, 244,
    474, 531, 106, 820, 693, 278,
    8, 705, 11, 596, 17, 758,
    505, 569, 414, 61, 214, 987,
    64, 786, 332, 811, 930, 967,
    637, 739, 148, 745, 10, 803,
    175, 176, 637, 677, 155, 521,
    727, 992, 441, 488, 45, 8,
    294, 20, 75, 23, 994, 866,
    // Bin capacity
    1670, 1781, 1419, 1633, 1712, 2348
  };
  const int i6_18_6_29_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    289, 180, 926, 240, 642, 236,
    18, 513, 625, 139, 178, 687,
    702, 600, 28, 999, 199, 243,
    725, 921, 97, 150, 929, 623,
    205, 929, 341, 525, 212, 832,
    22, 269, 881, 649, 507, 270,
    803, 293, 101, 344, 155, 812,
    507, 453, 428, 854, 30, 344,
    382, 12, 403, 327, 163, 805,
    216, 752, 924, 803, 340, 631,
    264, 318, 231, 547, 597, 255,
    55, 542, 11, 440, 251, 112,
    178, 678, 217, 425, 304, 67,
    391, 22, 603, 406, 33, 529,
    835, 380, 446, 534, 94, 784,
    289, 823, 416, 314, 108, 512,
    693, 724, 750, 128, 972, 685,
    119, 144, 435, 892, 558, 326,
    // Bin capacity
    1439, 1839, 1691, 1874, 1349, 1882
  };
  const int i6_18_6_30_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    26, 419, 102, 862, 86, 962,
    691, 587, 382, 518, 342, 232,
    231, 746, 470, 715, 987, 384,
    860, 225, 51, 151, 733, 693,
    185, 512, 967, 348, 326, 479,
    295, 43, 927, 514, 446, 205,
    156, 536, 601, 937, 180, 987,
    803, 837, 915, 17, 959, 884,
    907, 931, 27, 683, 355, 685,
    80, 89, 543, 693, 225, 695,
    707, 99, 295, 331, 625, 832,
    736, 493, 911, 868, 431, 496,
    832, 746, 509, 983, 891, 650,
    421, 749, 57, 825, 461, 87,
    758, 487, 825, 585, 355, 653,
    711, 493, 455, 513, 745, 773,
    449, 401, 39, 164, 395, 404,
    336, 815, 150, 450, 879, 757,
    // Bin capacity
    1990, 1996, 1783, 2201, 2042, 2353
  };
  const int i6_18_6_31_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    487, 799, 725, 409, 365, 843,
    178, 121, 115, 397, 642, 137,
    220, 96, 6, 297, 201, 15,
    826, 446, 173, 833, 725, 494,
    938, 136, 804, 29, 76, 99,
    855, 626, 836, 48, 632, 521,
    982, 140, 910, 146, 211, 391,
    728, 949, 909, 190, 437, 622,
    648, 54, 990, 64, 588, 173,
    568, 271, 137, 946, 619, 911,
    665, 564, 194, 120, 798, 830,
    258, 104, 572, 484, 175, 194,
    965, 773, 674, 313, 492, 70,
    862, 289, 272, 577, 138, 519,
    405, 567, 730, 470, 623, 954,
    101, 687, 484, 270, 136, 649,
    844, 957, 219, 787, 632, 285,
    78, 902, 690, 685, 205, 615,
    // Bin capacity
    2317, 1852, 2062, 1543, 1681, 1817
  };
  const int i6_18_6_32_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    973, 204, 959, 912, 723, 170,
    938, 427, 456, 536, 547, 151,
    941, 950, 951, 862, 996, 996,
    491, 539, 832, 243, 270, 193,
    927, 107, 794, 347, 222, 121,
    409, 654, 953, 953, 147, 80,
    147, 668, 567, 819, 36, 4,
    265, 946, 824, 445, 721, 667,
    911, 105, 235, 611, 50, 487,
    619, 593, 116, 67, 614, 744,
    151, 983, 528, 554, 296, 8,
    406, 224, 230, 354, 243, 0,
    198, 880, 791, 548, 993, 921,
    307, 867, 39, 514, 483, 241,
    810, 396, 690, 116, 104, 444,
    213, 797, 357, 848, 789, 867,
    727, 522, 867, 923, 549, 539,
    861, 829, 515, 11, 252, 817,
    // Bin capacity
    2265, 2353, 2355, 2126, 1768, 1639
  };
  const int i6_18_6_33_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    67, 379, 568, 908, 293, 948,
    891, 379, 457, 534, 90, 93,
    488, 276, 571, 29, 681, 489,
    274, 724, 748, 891, 64, 996,
    56, 40, 446, 764, 205, 609,
    4, 930, 877, 414, 225, 775,
    90, 325, 214, 671, 988, 62,
    463, 725, 913, 652, 328, 378,
    19, 910, 388, 816, 822, 864,
    565, 319, 745, 213, 461, 427,
    973, 394, 320, 802, 416, 762,
    114, 357, 120, 784, 614, 213,
    38, 594, 833, 142, 472, 249,
    739, 214, 870, 584, 251, 556,
    11, 170, 629, 235, 21, 378,
    364, 189, 109, 586, 277, 784,
    921, 393, 139, 284, 174, 330,
    247, 314, 14, 294, 33, 45,
    // Bin capacity
    1402, 1692, 1987, 2129, 1422, 1986
  };
  const int i6_18_6_34_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    55, 53, 258, 10, 540, 115,
    661, 985, 870, 772, 963, 654,
    76, 319, 36, 665, 662, 762,
    353, 72, 488, 555, 743, 797,
    997, 203, 224, 270, 478, 875,
    889, 546, 660, 187, 295, 595,
    989, 119, 985, 553, 376, 984,
    224, 162, 522, 588, 662, 811,
    674, 823, 338, 612, 385, 77,
    519, 496, 298, 141, 88, 92,
    402, 170, 11, 410, 695, 480,
    36, 341, 809, 725, 12, 994,
    364, 358, 712, 272, 110, 514,
    284, 143, 492, 23, 149, 975,
    778, 223, 544, 434, 97, 655,
    589, 93, 725, 195, 131, 746,
    388, 163, 512, 590, 512, 997,
    890, 922, 558, 142, 497, 935,
    // Bin capacity
    2048, 1383, 2020, 1596, 1652, 2693
  };
  const int i6_18_6_35_6[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    991, 949, 319, 739, 781, 529,
    876, 92, 338, 812, 819, 54,
    871, 744, 307, 540, 751, 340,
    557, 959, 565, 70, 810, 606,
    92, 378, 914, 983, 91, 471,
    619, 449, 573, 261, 752, 44,
    548, 937, 840, 835, 523, 994,
    693, 371, 135, 63, 332, 930,
    770, 949, 820, 343, 873, 955,
    446, 17, 687, 487, 465, 878,
    343, 223, 527, 843, 68, 871,
    184, 622, 401, 22, 891, 8,
    764, 504, 707, 438, 568, 215,
    28, 132, 535, 111, 901, 986,
    640, 746, 90, 284, 47, 760,
    798, 199, 656, 657, 474, 14,
    733, 775, 110, 285, 939, 830,
    639, 203, 322, 349, 834, 68,
    // Bin capacity
    2384, 2082, 1991, 1828, 2457, 2150
  };
  const int i6_18_6_0_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    691, 526, 858, 166, 29, 491,
    110, 813, 802, 334, 551, 867,
    168, 587, 582, 605, 837, 390,
    926, 522, 461, 334, 241, 852,
    1000, 462, 817, 986, 509, 119,
    431, 652, 152, 199, 985, 823,
    98, 203, 895, 675, 573, 983,
    659, 504, 141, 696, 588, 704,
    163, 308, 631, 717, 13, 267,
    387, 201, 98, 629, 891, 774,
    778, 735, 333, 958, 907, 398,
    893, 855, 556, 553, 212, 15,
    866, 371, 923, 552, 212, 653,
    384, 829, 285, 726, 372, 539,
    150, 894, 530, 183, 441, 660,
    760, 954, 161, 966, 838, 74,
    588, 71, 441, 462, 648, 714,
    80, 158, 922, 709, 609, 528,
    // Bin capacity
    1522, 1608, 1598, 1742, 1576, 1642
  };
  const int i6_18_6_1_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    358, 788, 992, 719, 708, 94,
    597, 621, 244, 290, 586, 479,
    669, 335, 729, 709, 570, 174,
    281, 80, 512, 124, 92, 305,
    98, 961, 892, 944, 107, 419,
    489, 511, 525, 657, 310, 566,
    352, 863, 335, 674, 158, 127,
    459, 230, 213, 946, 306, 10,
    634, 354, 463, 78, 183, 252,
    709, 649, 163, 165, 851, 542,
    214, 242, 175, 25, 418, 894,
    310, 617, 346, 913, 203, 755,
    870, 207, 39, 427, 310, 622,
    39, 313, 345, 251, 568, 222,
    648, 119, 907, 251, 582, 755,
    913, 946, 632, 471, 191, 592,
    853, 593, 353, 223, 570, 750,
    963, 429, 1000, 17, 415, 422,
    // Bin capacity
    1592, 1492, 1493, 1328, 1200, 1344
  };
  const int i6_18_6_2_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    537, 149, 60, 905, 605, 348,
    193, 157, 695, 24, 62, 466,
    666, 770, 839, 615, 968, 497,
    695, 939, 205, 88, 144, 515,
    83, 305, 48, 551, 343, 594,
    213, 614, 898, 307, 415, 878,
    505, 119, 259, 956, 893, 805,
    735, 553, 954, 572, 926, 296,
    596, 429, 759, 458, 11, 692,
    258, 553, 321, 48, 102, 705,
    878, 821, 838, 336, 647, 119,
    49, 424, 635, 788, 878, 193,
    746, 828, 547, 801, 497, 102,
    891, 158, 797, 696, 575, 611,
    5, 309, 839, 386, 748, 345,
    250, 578, 821, 393, 111, 849,
    581, 398, 222, 348, 781, 743,
    286, 321, 791, 700, 976, 491,
    // Bin capacity
    1389, 1433, 1790, 1526, 1646, 1573
  };
  const int i6_18_6_3_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    871, 592, 377, 478, 428, 380,
    907, 767, 85, 593, 462, 143,
    510, 738, 586, 778, 866, 88,
    945, 220, 200, 441, 952, 625,
    842, 668, 477, 83, 997, 63,
    619, 401, 974, 957, 688, 958,
    833, 71, 875, 197, 90, 988,
    452, 403, 815, 131, 169, 401,
    912, 971, 197, 27, 867, 194,
    684, 374, 249, 4, 640, 526,
    884, 357, 673, 608, 41, 535,
    617, 657, 327, 164, 694, 116,
    186, 913, 32, 368, 600, 298,
    730, 535, 225, 846, 855, 554,
    450, 497, 307, 829, 166, 914,
    208, 491, 227, 930, 922, 603,
    495, 165, 973, 920, 493, 564,
    434, 516, 98, 560, 448, 731,
    // Bin capacity
    1988, 1603, 1322, 1531, 1782, 1491
  };
  const int i6_18_6_4_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    203, 553, 114, 876, 813, 348,
    873, 241, 398, 987, 562, 262,
    596, 605, 667, 435, 82, 420,
    225, 637, 979, 430, 301, 789,
    985, 503, 81, 104, 821, 636,
    46, 312, 89, 955, 941, 807,
    553, 810, 577, 885, 854, 37,
    841, 512, 141, 427, 844, 661,
    454, 603, 219, 479, 743, 620,
    777, 134, 250, 582, 886, 72,
    87, 339, 422, 485, 510, 179,
    714, 40, 493, 868, 237, 410,
    597, 586, 530, 451, 756, 276,
    898, 539, 851, 749, 989, 226,
    691, 274, 453, 810, 882, 483,
    706, 345, 609, 495, 60, 815,
    653, 103, 588, 298, 740, 474,
    317, 16, 711, 108, 275, 870,
    // Bin capacity
    1771, 1240, 1417, 1807, 1958, 1454
  };
  const int i6_18_6_5_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    361, 202, 55, 888, 580, 471,
    385, 224, 765, 503, 132, 592,
    150, 36, 769, 716, 730, 872,
    539, 403, 779, 673, 543, 751,
    149, 272, 418, 363, 557, 597,
    520, 93, 849, 832, 846, 544,
    231, 442, 681, 248, 448, 262,
    773, 32, 507, 36, 673, 304,
    831, 693, 445, 19, 377, 840,
    517, 717, 256, 739, 894, 661,
    693, 676, 885, 793, 9, 691,
    870, 37, 512, 131, 796, 161,
    205, 808, 302, 66, 385, 461,
    72, 150, 429, 381, 713, 206,
    296, 259, 255, 937, 8, 139,
    500, 744, 920, 152, 882, 523,
    390, 416, 864, 638, 83, 859,
    508, 481, 5, 724, 429, 274,
    // Bin capacity
    1399, 1170, 1697, 1547, 1590, 1612
  };
  const int i6_18_6_6_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    480, 559, 499, 502, 36, 376,
    462, 440, 999, 465, 385, 191,
    7, 222, 97, 787, 278, 83,
    86, 560, 670, 875, 561, 885,
    995, 358, 903, 911, 818, 204,
    694, 850, 591, 879, 207, 566,
    91, 130, 196, 678, 868, 63,
    913, 824, 742, 739, 890, 965,
    898, 858, 61, 135, 777, 389,
    750, 564, 183, 381, 944, 823,
    101, 941, 704, 91, 279, 281,
    360, 652, 617, 721, 879, 926,
    639, 731, 529, 884, 473, 767,
    730, 803, 247, 698, 216, 263,
    965, 77, 229, 594, 954, 131,
    258, 435, 944, 178, 799, 451,
    765, 187, 759, 398, 486, 481,
    789, 104, 686, 961, 448, 385,
    // Bin capacity
    1764, 1643, 1706, 1922, 1820, 1454
  };
  const int i6_18_6_7_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    49, 786, 676, 837, 531, 360,
    981, 347, 312, 905, 803, 428,
    17, 705, 927, 104, 67, 599,
    290, 85, 447, 663, 78, 832,
    753, 747, 375, 986, 861, 760,
    485, 919, 847, 159, 36, 529,
    255, 939, 871, 707, 819, 975,
    820, 925, 307, 703, 794, 595,
    635, 31, 219, 662, 415, 317,
    714, 159, 689, 825, 463, 75,
    427, 982, 471, 190, 399, 477,
    488, 67, 361, 923, 542, 500,
    7, 501, 268, 990, 978, 606,
    173, 589, 960, 474, 636, 584,
    847, 275, 966, 357, 948, 614,
    415, 400, 219, 224, 343, 60,
    890, 985, 647, 134, 299, 146,
    921, 638, 272, 731, 853, 247,
    // Bin capacity
    1635, 1798, 1754, 1886, 1760, 1553
  };
  const int i6_18_6_8_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    552, 702, 968, 936, 395, 203,
    562, 920, 801, 443, 86, 396,
    912, 209, 230, 292, 980, 680,
    88, 350, 608, 879, 196, 850,
    810, 71, 988, 171, 551, 938,
    478, 852, 710, 825, 468, 783,
    117, 360, 957, 972, 197, 802,
    484, 358, 633, 496, 289, 11,
    273, 34, 907, 212, 930, 510,
    255, 700, 315, 802, 235, 218,
    136, 552, 401, 387, 704, 452,
    273, 646, 1, 525, 269, 276,
    305, 530, 469, 209, 315, 14,
    994, 841, 233, 212, 486, 91,
    903, 261, 705, 96, 576, 333,
    904, 912, 840, 176, 172, 397,
    397, 182, 193, 114, 969, 940,
    396, 7, 520, 205, 71, 338,
    // Bin capacity
    1592, 1528, 1887, 1432, 1421, 1482
  };
  const int i6_18_6_9_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    885, 474, 133, 726, 508, 645,
    362, 228, 76, 207, 562, 722,
    963, 444, 730, 231, 752, 595,
    285, 709, 464, 549, 400, 305,
    94, 484, 626, 290, 757, 729,
    573, 90, 159, 799, 757, 428,
    504, 349, 449, 664, 946, 879,
    663, 857, 263, 287, 461, 561,
    666, 968, 607, 614, 539, 980,
    384, 36, 283, 93, 36, 58,
    146, 775, 179, 314, 200, 522,
    341, 794, 876, 319, 270, 476,
    18, 423, 286, 118, 828, 214,
    849, 878, 835, 35, 478, 504,
    789, 107, 818, 204, 39, 894,
    922, 970, 872, 403, 882, 66,
    867, 190, 303, 531, 14, 648,
    305, 571, 377, 938, 264, 282,
    // Bin capacity
    1747, 1699, 1515, 1331, 1580, 1728
  };
  const int i6_18_6_10_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    50, 724, 848, 109, 2, 336,
    912, 877, 941, 329, 48, 704,
    129, 542, 424, 78, 36, 843,
    362, 482, 8, 277, 780, 991,
    416, 226, 963, 832, 642, 511,
    887, 752, 17, 178, 168, 194,
    212, 795, 981, 688, 514, 475,
    872, 595, 758, 600, 210, 834,
    642, 70, 11, 936, 902, 258,
    669, 309, 131, 439, 730, 796,
    574, 222, 434, 909, 587, 936,
    455, 607, 786, 634, 857, 942,
    310, 409, 134, 293, 513, 769,
    926, 327, 732, 475, 225, 330,
    434, 650, 648, 973, 443, 751,
    265, 60, 143, 120, 318, 699,
    62, 467, 633, 519, 2, 292,
    109, 400, 803, 0, 931, 926,
    // Bin capacity
    1520, 1561, 1723, 1538, 1450, 2125
  };
  const int i6_18_6_11_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    907, 584, 829, 55, 424, 766,
    905, 101, 649, 688, 282, 774,
    84, 710, 781, 495, 152, 487,
    333, 256, 210, 275, 807, 692,
    932, 484, 646, 73, 503, 631,
    118, 564, 665, 955, 460, 551,
    376, 692, 389, 780, 676, 300,
    560, 731, 353, 225, 926, 463,
    164, 38, 737, 539, 920, 279,
    871, 862, 335, 770, 198, 913,
    394, 266, 208, 850, 491, 793,
    18, 657, 905, 833, 107, 114,
    473, 521, 819, 966, 633, 56,
    718, 459, 997, 706, 140, 788,
    522, 571, 865, 938, 915, 799,
    381, 461, 933, 448, 768, 595,
    312, 872, 270, 400, 154, 458,
    878, 105, 576, 513, 781, 124,
    // Bin capacity
    1656, 1653, 2066, 1945, 1728, 1773
  };
  const int i6_18_6_12_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    456, 47, 332, 616, 736, 966,
    706, 207, 658, 286, 2, 248,
    647, 365, 317, 847, 17, 970,
    438, 553, 203, 293, 202, 502,
    676, 130, 298, 657, 344, 973,
    54, 637, 313, 297, 134, 873,
    5, 563, 893, 275, 593, 159,
    80, 882, 99, 669, 571, 489,
    648, 682, 623, 482, 22, 569,
    864, 80, 987, 274, 843, 992,
    214, 999, 625, 978, 105, 128,
    413, 184, 929, 239, 681, 895,
    789, 759, 542, 438, 633, 698,
    379, 330, 891, 277, 371, 180,
    242, 832, 772, 679, 989, 389,
    151, 916, 614, 377, 75, 542,
    906, 787, 850, 605, 0, 480,
    444, 600, 243, 800, 175, 365,
    // Bin capacity
    1515, 1784, 1902, 1697, 1213, 1945
  };
  const int i6_18_6_13_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    385, 906, 855, 716, 895, 394,
    836, 585, 71, 448, 220, 374,
    547, 912, 156, 321, 221, 74,
    592, 730, 772, 530, 684, 906,
    875, 810, 318, 512, 650, 595,
    66, 645, 468, 252, 866, 946,
    159, 787, 472, 464, 868, 773,
    292, 191, 242, 791, 449, 962,
    748, 916, 329, 959, 953, 698,
    298, 682, 24, 475, 153, 682,
    513, 808, 165, 709, 342, 707,
    775, 301, 639, 127, 233, 239,
    550, 630, 884, 670, 796, 753,
    749, 272, 424, 212, 830, 787,
    736, 824, 130, 34, 524, 994,
    823, 562, 595, 770, 699, 694,
    256, 401, 123, 586, 46, 705,
    862, 801, 31, 424, 501, 749,
    // Bin capacity
    1896, 2216, 1262, 1695, 1871, 2267
  };
  const int i6_18_6_14_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    13, 865, 104, 411, 771, 764,
    148, 733, 102, 367, 910, 837,
    988, 764, 733, 737, 154, 406,
    362, 840, 39, 935, 570, 442,
    119, 990, 347, 740, 861, 938,
    954, 297, 565, 33, 791, 683,
    560, 328, 243, 510, 983, 350,
    23, 877, 953, 204, 508, 151,
    501, 552, 732, 226, 536, 939,
    599, 883, 754, 258, 81, 602,
    654, 997, 10, 889, 858, 643,
    9, 454, 395, 422, 170, 374,
    40, 596, 307, 373, 848, 140,
    324, 869, 827, 864, 164, 396,
    77, 254, 147, 139, 801, 210,
    687, 396, 925, 625, 451, 962,
    193, 33, 309, 735, 425, 597,
    334, 721, 428, 92, 523, 210,
    // Bin capacity
    1252, 2176, 1505, 1627, 1977, 1833
  };
  const int i6_18_6_15_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    983, 685, 706, 817, 402, 119,
    247, 2, 375, 579, 912, 773,
    361, 137, 207, 802, 292, 547,
    760, 17, 726, 401, 314, 845,
    9, 489, 185, 375, 65, 227,
    637, 898, 315, 855, 822, 377,
    37, 943, 477, 913, 696, 335,
    939, 290, 389, 709, 775, 788,
    819, 526, 247, 442, 23, 106,
    852, 44, 124, 543, 132, 126,
    14, 763, 184, 25, 247, 147,
    672, 877, 875, 395, 530, 729,
    782, 391, 321, 465, 529, 451,
    790, 476, 335, 638, 533, 927,
    695, 938, 638, 417, 383, 364,
    797, 350, 365, 500, 133, 319,
    778, 378, 914, 860, 894, 966,
    179, 811, 505, 606, 976, 942,
    // Bin capacity
    1984, 1728, 1512, 1983, 1660, 1742
  };
  const int i6_18_6_16_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    418, 704, 565, 199, 72, 93,
    914, 975, 479, 904, 169, 149,
    135, 599, 772, 361, 174, 894,
    309, 253, 442, 148, 22, 843,
    747, 673, 446, 982, 840, 435,
    370, 686, 73, 883, 264, 384,
    657, 542, 504, 431, 238, 940,
    829, 709, 917, 561, 126, 46,
    883, 910, 893, 573, 222, 10,
    440, 210, 709, 667, 345, 773,
    898, 991, 334, 836, 53, 475,
    118, 863, 883, 327, 825, 73,
    187, 160, 277, 868, 752, 954,
    20, 413, 787, 778, 738, 53,
    885, 53, 621, 918, 595, 908,
    928, 77, 191, 143, 645, 414,
    324, 978, 559, 527, 478, 275,
    705, 818, 770, 627, 764, 941,
    // Bin capacity
    1889, 2053, 1977, 2076, 1416, 1675
  };
  const int i6_18_6_17_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    697, 242, 141, 613, 990, 761,
    59, 829, 34, 307, 853, 796,
    290, 507, 551, 971, 599, 106,
    11, 73, 736, 518, 553, 588,
    259, 79, 436, 826, 552, 385,
    773, 466, 221, 494, 807, 679,
    799, 15, 847, 357, 949, 314,
    931, 953, 675, 350, 221, 306,
    514, 168, 619, 966, 462, 962,
    104, 678, 286, 444, 462, 258,
    863, 854, 699, 570, 209, 678,
    29, 48, 341, 408, 363, 887,
    902, 764, 119, 123, 520, 758,
    540, 196, 834, 837, 973, 668,
    166, 767, 349, 61, 531, 960,
    265, 8, 673, 975, 966, 731,
    285, 228, 494, 891, 156, 137,
    93, 373, 225, 55, 777, 920,
    // Bin capacity
    1479, 1414, 1615, 1905, 2134, 2125
  };
  const int i6_18_6_18_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    726, 155, 408, 825, 788, 635,
    481, 31, 262, 4, 129, 760,
    382, 914, 684, 76, 810, 998,
    851, 416, 560, 488, 470, 105,
    860, 530, 850, 73, 290, 419,
    603, 143, 550, 329, 173, 589,
    842, 159, 827, 115, 967, 159,
    201, 435, 135, 723, 109, 739,
    709, 709, 561, 63, 646, 679,
    752, 599, 776, 192, 285, 601,
    779, 257, 571, 420, 532, 601,
    249, 227, 878, 230, 987, 21,
    456, 633, 301, 649, 839, 388,
    876, 268, 342, 548, 483, 68,
    699, 177, 812, 111, 777, 843,
    485, 368, 845, 925, 38, 600,
    217, 28, 902, 109, 443, 609,
    552, 598, 343, 742, 107, 688,
    // Bin capacity
    2109, 1308, 2087, 1303, 1746, 1869
  };
  const int i6_18_6_19_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    331, 133, 877, 179, 814, 706,
    757, 139, 62, 788, 191, 888,
    171, 857, 893, 351, 374, 423,
    958, 778, 704, 540, 818, 925,
    742, 885, 448, 164, 456, 449,
    788, 781, 73, 127, 73, 78,
    316, 486, 93, 178, 478, 521,
    685, 540, 996, 463, 201, 358,
    556, 797, 357, 15, 162, 172,
    726, 778, 179, 19, 426, 110,
    429, 231, 849, 835, 677, 367,
    794, 888, 344, 362, 236, 642,
    592, 430, 365, 652, 678, 449,
    10, 878, 779, 4, 796, 932,
    745, 625, 927, 901, 346, 865,
    276, 607, 30, 297, 229, 985,
    184, 991, 531, 110, 763, 310,
    285, 253, 279, 878, 341, 349,
    // Bin capacity
    1854, 2197, 1743, 1362, 1599, 1890
  };
  const int i6_18_6_20_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    670, 604, 911, 675, 419, 923,
    294, 750, 275, 344, 991, 354,
    391, 768, 305, 82, 265, 199,
    366, 303, 740, 104, 397, 388,
    609, 130, 231, 59, 185, 490,
    748, 331, 422, 148, 603, 971,
    451, 340, 324, 634, 697, 391,
    600, 804, 569, 402, 28, 259,
    537, 49, 264, 380, 625, 192,
    777, 766, 826, 80, 969, 107,
    346, 102, 573, 4, 211, 947,
    624, 537, 750, 360, 894, 641,
    347, 921, 966, 669, 858, 374,
    805, 871, 703, 999, 573, 616,
    52, 960, 588, 266, 856, 989,
    659, 470, 205, 30, 731, 681,
    869, 334, 198, 777, 831, 69,
    733, 162, 820, 730, 478, 523,
    // Bin capacity
    1976, 1841, 1934, 1349, 2123, 1823
  };
  const int i6_18_6_21_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    124, 112, 278, 76, 517, 886,
    828, 641, 829, 571, 119, 951,
    940, 165, 331, 410, 118, 615,
    38, 126, 162, 224, 468, 599,
    3, 808, 112, 916, 97, 368,
    477, 501, 774, 270, 668, 775,
    350, 421, 847, 259, 870, 555,
    715, 661, 289, 450, 156, 178,
    328, 412, 677, 906, 102, 955,
    949, 152, 43, 202, 313, 958,
    901, 530, 653, 562, 921, 243,
    273, 740, 175, 787, 328, 835,
    594, 144, 357, 831, 950, 624,
    708, 971, 822, 226, 741, 484,
    412, 470, 471, 301, 714, 105,
    816, 422, 411, 572, 20, 541,
    102, 990, 697, 140, 591, 349,
    882, 107, 785, 684, 288, 642,
    // Bin capacity
    1904, 1689, 1758, 1692, 1610, 2151
  };
  const int i6_18_6_22_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    811, 606, 601, 760, 118, 19,
    568, 703, 512, 151, 60, 437,
    347, 556, 630, 385, 480, 873,
    64, 955, 375, 937, 307, 326,
    436, 367, 684, 108, 465, 881,
    812, 86, 340, 412, 793, 641,
    838, 566, 806, 379, 175, 505,
    13, 653, 861, 822, 892, 593,
    999, 52, 627, 7, 605, 514,
    443, 863, 72, 500, 566, 496,
    916, 559, 653, 926, 670, 106,
    870, 868, 523, 274, 765, 827,
    597, 664, 800, 323, 90, 517,
    701, 11, 948, 286, 658, 209,
    822, 967, 947, 106, 888, 552,
    931, 312, 420, 656, 385, 147,
    547, 156, 235, 790, 498, 487,
    958, 520, 815, 457, 66, 620,
    // Bin capacity
    2374, 1925, 2206, 1684, 1725, 1780
  };
  const int i6_18_6_23_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    771, 96, 863, 828, 51, 420,
    856, 482, 978, 74, 409, 196,
    526, 237, 932, 651, 110, 903,
    81, 453, 317, 169, 301, 712,
    410, 342, 755, 918, 718, 7,
    628, 405, 251, 771, 977, 152,
    345, 179, 814, 190, 150, 367,
    135, 691, 272, 855, 913, 936,
    370, 389, 870, 712, 908, 825,
    483, 62, 620, 7, 111, 603,
    317, 701, 875, 689, 986, 557,
    117, 5, 665, 312, 256, 894,
    758, 244, 409, 441, 610, 195,
    972, 190, 43, 751, 18, 173,
    549, 950, 127, 897, 681, 847,
    516, 308, 895, 41, 236, 516,
    410, 491, 813, 997, 96, 521,
    740, 348, 537, 936, 277, 624,
    // Bin capacity
    1842, 1348, 2263, 2099, 1601, 1937
  };
  const int i6_18_6_24_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    45, 784, 234, 339, 350, 967,
    134, 514, 165, 605, 601, 698,
    865, 24, 767, 802, 493, 921,
    554, 790, 425, 676, 423, 469,
    921, 446, 14, 347, 316, 989,
    106, 758, 598, 989, 14, 571,
    914, 553, 233, 451, 299, 745,
    971, 864, 441, 77, 55, 1,
    846, 623, 444, 420, 408, 63,
    846, 383, 86, 738, 416, 965,
    667, 128, 948, 51, 181, 974,
    126, 272, 28, 234, 178, 7,
    529, 704, 626, 472, 817, 613,
    178, 335, 608, 65, 576, 594,
    816, 999, 796, 19, 928, 816,
    879, 634, 120, 860, 276, 168,
    912, 178, 656, 100, 909, 407,
    394, 232, 982, 639, 237, 424,
    // Bin capacity
    2212, 1906, 1689, 1630, 1546, 2148
  };
  const int i6_18_6_25_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    45, 968, 636, 59, 773, 525,
    993, 710, 998, 172, 424, 740,
    46, 843, 786, 326, 311, 219,
    68, 888, 286, 461, 387, 970,
    623, 156, 895, 444, 565, 144,
    274, 382, 636, 28, 218, 334,
    350, 775, 176, 917, 266, 741,
    700, 178, 197, 464, 20, 231,
    836, 407, 314, 909, 761, 971,
    596, 700, 832, 690, 401, 841,
    165, 381, 959, 842, 123, 35,
    943, 229, 670, 848, 190, 102,
    502, 149, 321, 302, 517, 615,
    79, 754, 594, 236, 780, 38,
    28, 225, 298, 71, 280, 866,
    254, 390, 306, 705, 976, 234,
    393, 52, 817, 456, 388, 995,
    686, 898, 737, 294, 775, 376,
    // Bin capacity
    1580, 1893, 2179, 1714, 1699, 1871
  };
  const int i6_18_6_26_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    44, 252, 462, 479, 422, 313,
    346, 891, 815, 352, 89, 438,
    434, 335, 212, 710, 171, 986,
    196, 635, 678, 260, 352, 516,
    674, 92, 284, 913, 116, 247,
    97, 481, 909, 167, 524, 975,
    686, 928, 334, 614, 228, 734,
    431, 634, 981, 719, 959, 704,
    661, 938, 964, 38, 434, 930,
    879, 849, 704, 640, 37, 469,
    789, 924, 586, 12, 669, 168,
    587, 85, 879, 393, 577, 370,
    88, 465, 189, 298, 916, 627,
    334, 626, 728, 292, 771, 791,
    116, 581, 564, 822, 583, 51,
    193, 168, 836, 709, 426, 621,
    10, 798, 866, 849, 243, 825,
    461, 238, 452, 491, 326, 493,
    // Bin capacity
    1476, 2084, 2404, 1840, 1648, 2155
  };
  const int i6_18_6_27_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    281, 735, 617, 610, 779, 624,
    678, 538, 404, 779, 864, 39,
    666, 58, 885, 271, 895, 982,
    10, 481, 448, 415, 633, 641,
    340, 800, 109, 575, 208, 784,
    595, 787, 569, 619, 396, 924,
    688, 110, 848, 874, 53, 530,
    65, 820, 788, 62, 74, 986,
    647, 868, 112, 201, 404, 23,
    255, 178, 147, 930, 756, 131,
    227, 949, 358, 940, 744, 538,
    529, 871, 5, 360, 556, 123,
    557, 531, 185, 853, 63, 208,
    913, 390, 331, 903, 389, 845,
    928, 305, 357, 12, 763, 245,
    352, 385, 877, 488, 467, 587,
    550, 391, 405, 502, 502, 870,
    311, 24, 6, 549, 117, 263,
    // Bin capacity
    1819, 1952, 1578, 2105, 1834, 1978
  };
  const int i6_18_6_28_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    61, 251, 41, 701, 137, 170,
    665, 909, 566, 994, 515, 652,
    592, 238, 775, 630, 271, 516,
    726, 244, 400, 146, 416, 200,
    110, 118, 723, 768, 542, 800,
    401, 418, 864, 915, 16, 520,
    661, 141, 724, 503, 811, 829,
    88, 373, 918, 744, 334, 311,
    546, 599, 10, 101, 1, 761,
    314, 126, 214, 971, 36, 732,
    557, 457, 192, 356, 104, 666,
    751, 589, 630, 771, 247, 970,
    301, 298, 570, 10, 716, 773,
    588, 345, 397, 957, 904, 608,
    84, 684, 905, 58, 917, 710,
    104, 322, 684, 908, 927, 657,
    124, 641, 11, 715, 143, 855,
    846, 787, 461, 776, 446, 211,
    // Bin capacity
    1605, 1609, 1939, 2352, 1597, 2335
  };
  const int i6_18_6_29_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    952, 551, 819, 361, 79, 904,
    943, 206, 607, 116, 881, 414,
    806, 746, 393, 521, 193, 432,
    131, 77, 607, 268, 60, 433,
    703, 610, 983, 14, 206, 664,
    555, 555, 224, 101, 846, 423,
    43, 288, 476, 185, 911, 126,
    549, 349, 40, 463, 975, 207,
    287, 657, 130, 991, 642, 622,
    910, 380, 415, 595, 4, 828,
    251, 300, 784, 308, 334, 65,
    835, 392, 76, 816, 725, 317,
    288, 743, 25, 614, 290, 120,
    35, 765, 252, 872, 284, 65,
    295, 15, 868, 445, 868, 446,
    851, 367, 747, 957, 730, 448,
    29, 672, 780, 538, 325, 925,
    275, 856, 174, 708, 169, 720,
    // Bin capacity
    1879, 1834, 1806, 1908, 1833, 1755
  };
  const int i6_18_6_30_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    394, 723, 522, 202, 987, 889,
    88, 609, 204, 452, 912, 482,
    717, 978, 802, 423, 328, 836,
    962, 34, 768, 796, 748, 267,
    14, 973, 595, 512, 365, 310,
    925, 833, 553, 1000, 918, 790,
    709, 454, 356, 706, 730, 35,
    880, 149, 568, 917, 141, 628,
    987, 788, 389, 381, 584, 442,
    112, 93, 449, 596, 286, 469,
    720, 981, 759, 450, 897, 509,
    928, 516, 387, 160, 617, 21,
    497, 921, 936, 425, 315, 882,
    384, 958, 609, 636, 620, 883,
    915, 953, 900, 838, 825, 525,
    199, 20, 98, 464, 887, 639,
    373, 327, 97, 77, 967, 505,
    94, 312, 371, 534, 573, 316,
    // Bin capacity
    2145, 2302, 2029, 2074, 2535, 2043
  };
  const int i6_18_6_31_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    522, 414, 985, 475, 367, 472,
    555, 703, 202, 802, 677, 201,
    904, 206, 507, 951, 979, 198,
    105, 697, 506, 236, 59, 874,
    756, 133, 791, 175, 552, 959,
    599, 900, 559, 995, 388, 536,
    722, 809, 750, 933, 616, 773,
    502, 567, 816, 548, 186, 277,
    727, 116, 893, 344, 892, 109,
    865, 655, 871, 567, 71, 849,
    943, 620, 269, 74, 324, 36,
    294, 542, 765, 733, 335, 360,
    890, 883, 828, 389, 552, 91,
    275, 269, 401, 952, 334, 521,
    206, 344, 802, 704, 429, 951,
    264, 504, 427, 974, 176, 390,
    184, 943, 11, 414, 534, 792,
    483, 295, 679, 879, 219, 626,
    // Bin capacity
    2139, 2096, 2416, 2434, 1679, 1969
  };
  const int i6_18_6_32_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    258, 735, 361, 374, 615, 978,
    789, 747, 206, 138, 425, 75,
    173, 526, 232, 923, 906, 827,
    880, 233, 251, 675, 314, 442,
    125, 511, 323, 268, 161, 992,
    184, 628, 57, 341, 360, 260,
    305, 336, 880, 692, 509, 219,
    309, 345, 555, 261, 891, 442,
    932, 80, 187, 578, 61, 245,
    734, 625, 586, 159, 211, 118,
    115, 245, 243, 916, 340, 302,
    371, 511, 285, 465, 391, 785,
    104, 110, 427, 793, 986, 517,
    103, 575, 5, 894, 52, 346,
    410, 144, 306, 9, 824, 917,
    25, 830, 459, 932, 467, 949,
    615, 261, 939, 10, 993, 398,
    726, 343, 414, 26, 789, 570,
    // Bin capacity
    1575, 1713, 1478, 1860, 2045, 2065
  };
  const int i6_18_6_33_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    184, 208, 425, 681, 257, 957,
    688, 129, 209, 750, 320, 917,
    751, 352, 652, 414, 931, 16,
    250, 191, 391, 202, 58, 715,
    633, 289, 142, 250, 378, 83,
    650, 560, 998, 234, 423, 65,
    531, 343, 479, 558, 187, 832,
    858, 771, 250, 980, 866, 344,
    285, 562, 441, 20, 170, 29,
    553, 432, 126, 465, 730, 619,
    388, 853, 563, 690, 405, 450,
    615, 935, 273, 544, 996, 671,
    673, 939, 339, 815, 546, 236,
    857, 469, 192, 336, 203, 421,
    419, 950, 218, 745, 708, 172,
    763, 993, 602, 399, 929, 764,
    168, 239, 849, 193, 867, 876,
    418, 651, 23, 537, 481, 416,
    // Bin capacity
    2147, 2187, 1590, 1954, 2096, 1903
  };
  const int i6_18_6_34_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    964, 529, 347, 100, 791, 236,
    361, 337, 728, 21, 948, 436,
    741, 458, 791, 250, 203, 626,
    756, 429, 612, 11, 151, 669,
    691, 726, 418, 719, 240, 838,
    697, 174, 863, 721, 871, 208,
    264, 672, 783, 310, 190, 457,
    251, 775, 49, 574, 985, 927,
    616, 97, 211, 325, 945, 721,
    419, 207, 77, 979, 220, 225,
    531, 60, 265, 794, 607, 732,
    632, 682, 245, 962, 961, 365,
    616, 516, 637, 877, 283, 986,
    927, 165, 102, 718, 434, 227,
    119, 544, 807, 800, 829, 249,
    631, 426, 130, 120, 683, 516,
    604, 117, 480, 867, 599, 166,
    516, 259, 885, 219, 402, 472,
    // Bin capacity
    2309, 1602, 1883, 2092, 2310, 2023
  };
  const int i6_18_6_35_7[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    766, 311, 895, 296, 95, 948,
    272, 805, 882, 439, 271, 984,
    120, 533, 526, 677, 887, 832,
    456, 608, 61, 137, 718, 547,
    780, 809, 657, 906, 130, 68,
    170, 90, 780, 198, 308, 545,
    328, 860, 859, 225, 7, 844,
    469, 898, 972, 817, 897, 400,
    793, 306, 939, 529, 953, 300,
    527, 550, 81, 562, 275, 74,
    162, 425, 670, 116, 954, 822,
    274, 61, 209, 318, 566, 312,
    733, 592, 466, 12, 118, 305,
    62, 274, 288, 748, 783, 422,
    866, 36, 603, 156, 59, 210,
    502, 190, 601, 78, 922, 956,
    336, 649, 108, 734, 568, 406,
    623, 867, 994, 727, 468, 92,
    // Bin capacity
    1854, 1995, 2383, 1727, 2021, 2041
  };
  const int i6_18_6_0_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    726, 205, 901, 268, 408, 52,
    153, 866, 788, 21, 915, 698,
    631, 891, 963, 277, 596, 505,
    974, 637, 734, 992, 887, 305,
    770, 388, 294, 868, 181, 842,
    116, 923, 189, 481, 42, 260,
    693, 309, 573, 87, 922, 231,
    431, 805, 736, 559, 992, 63,
    481, 272, 756, 745, 146, 649,
    895, 434, 597, 673, 463, 236,
    139, 436, 861, 33, 616, 345,
    270, 551, 159, 352, 909, 501,
    236, 613, 948, 116, 131, 938,
    627, 96, 604, 365, 723, 518,
    610, 558, 256, 290, 749, 480,
    417, 585, 257, 128, 33, 931,
    919, 669, 667, 30, 294, 20,
    831, 269, 946, 876, 423, 912,
    // Bin capacity
    1654, 1585, 1872, 1194, 1572, 1415
  };
  const int i6_18_6_1_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    555, 237, 520, 238, 537, 662,
    694, 220, 807, 337, 485, 549,
    147, 908, 343, 830, 288, 321,
    983, 414, 757, 191, 267, 468,
    139, 940, 525, 800, 22, 354,
    363, 36, 86, 324, 303, 4,
    34, 521, 758, 248, 36, 537,
    887, 894, 426, 102, 888, 716,
    454, 505, 931, 10, 944, 985,
    797, 776, 410, 246, 705, 477,
    755, 157, 420, 208, 857, 445,
    82, 679, 88, 364, 436, 45,
    262, 117, 313, 924, 558, 150,
    743, 61, 183, 673, 179, 423,
    593, 771, 752, 246, 967, 578,
    307, 634, 800, 811, 396, 74,
    686, 68, 331, 46, 299, 633,
    29, 151, 780, 372, 385, 590,
    // Bin capacity
    1433, 1362, 1554, 1174, 1440, 1349
  };
  const int i6_18_6_2_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    384, 501, 974, 608, 293, 779,
    711, 947, 720, 636, 133, 742,
    151, 668, 215, 870, 619, 350,
    759, 118, 91, 533, 424, 499,
    15, 19, 816, 825, 755, 953,
    596, 977, 670, 300, 79, 771,
    66, 956, 525, 169, 664, 107,
    741, 830, 909, 647, 81, 748,
    840, 472, 185, 227, 879, 249,
    747, 962, 943, 46, 465, 740,
    747, 237, 442, 913, 607, 755,
    288, 795, 817, 520, 695, 512,
    312, 354, 511, 805, 364, 226,
    130, 214, 123, 279, 225, 1,
    141, 298, 921, 186, 287, 992,
    847, 243, 599, 238, 967, 518,
    796, 663, 47, 845, 600, 804,
    134, 945, 22, 470, 380, 49,
    // Bin capacity
    1429, 1734, 1621, 1550, 1448, 1666
  };
  const int i6_18_6_3_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    825, 391, 184, 445, 18, 513,
    408, 721, 633, 748, 921, 571,
    606, 245, 906, 22, 132, 593,
    948, 211, 137, 863, 104, 346,
    308, 889, 315, 112, 134, 721,
    999, 764, 175, 628, 694, 61,
    304, 558, 786, 813, 941, 497,
    741, 736, 617, 772, 855, 991,
    661, 646, 907, 38, 737, 170,
    108, 503, 73, 626, 714, 67,
    327, 339, 393, 454, 246, 888,
    222, 759, 372, 690, 921, 202,
    590, 835, 766, 429, 921, 825,
    891, 733, 274, 834, 620, 732,
    578, 679, 60, 776, 479, 431,
    113, 777, 479, 945, 83, 196,
    278, 745, 36, 678, 235, 645,
    910, 100, 644, 174, 159, 669,
    // Bin capacity
    1686, 1825, 1332, 1725, 1531, 1566
  };
  const int i6_18_6_4_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    220, 49, 40, 149, 383, 517,
    384, 876, 505, 925, 16, 674,
    187, 311, 130, 671, 212, 156,
    775, 485, 524, 21, 408, 500,
    579, 472, 473, 997, 662, 700,
    715, 264, 976, 680, 960, 346,
    353, 305, 665, 264, 94, 680,
    893, 256, 856, 937, 472, 462,
    742, 962, 426, 883, 588, 15,
    52, 282, 384, 731, 642, 211,
    951, 884, 951, 683, 732, 958,
    226, 734, 856, 803, 995, 444,
    94, 457, 753, 279, 397, 513,
    756, 178, 948, 705, 56, 865,
    395, 403, 224, 82, 993, 586,
    853, 421, 292, 633, 134, 322,
    918, 99, 657, 501, 278, 769,
    447, 710, 641, 501, 401, 307,
    // Bin capacity
    1654, 1413, 1786, 1811, 1460, 1565
  };
  const int i6_18_6_5_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    960, 315, 705, 473, 323, 688,
    554, 810, 127, 748, 498, 731,
    549, 775, 25, 663, 786, 838,
    685, 784, 251, 281, 697, 488,
    698, 932, 421, 924, 723, 642,
    21, 724, 898, 671, 939, 68,
    40, 707, 737, 551, 380, 536,
    452, 270, 620, 524, 791, 876,
    410, 291, 498, 487, 530, 336,
    473, 972, 181, 579, 767, 614,
    823, 226, 901, 692, 481, 445,
    854, 503, 487, 517, 45, 36,
    190, 764, 34, 874, 803, 857,
    757, 595, 54, 475, 920, 21,
    725, 486, 768, 138, 357, 768,
    168, 921, 679, 375, 107, 560,
    703, 558, 415, 611, 965, 45,
    322, 20, 508, 851, 125, 701,
    // Bin capacity
    1643, 1865, 1455, 1826, 1792, 1619
  };
  const int i6_18_6_6_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    33, 266, 999, 474, 408, 123,
    719, 524, 501, 168, 551, 921,
    655, 504, 880, 641, 534, 870,
    111, 510, 203, 346, 18, 4,
    322, 65, 903, 978, 786, 252,
    293, 317, 582, 421, 409, 179,
    793, 836, 878, 400, 418, 61,
    347, 233, 207, 769, 901, 860,
    72, 429, 58, 756, 582, 276,
    770, 51, 630, 951, 896, 333,
    150, 719, 450, 863, 492, 607,
    110, 743, 173, 922, 603, 914,
    859, 314, 975, 319, 739, 149,
    446, 283, 586, 272, 955, 828,
    854, 770, 628, 289, 707, 383,
    485, 274, 57, 639, 802, 79,
    527, 76, 151, 829, 386, 725,
    151, 640, 516, 797, 49, 243,
    // Bin capacity
    1360, 1335, 1657, 1915, 1809, 1380
  };
  const int i6_18_6_7_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    369, 289, 580, 766, 130, 839,
    469, 543, 319, 307, 468, 224,
    218, 827, 24, 125, 515, 738,
    828, 924, 262, 200, 592, 385,
    702, 590, 917, 947, 357, 637,
    241, 738, 692, 544, 407, 295,
    135, 162, 542, 795, 88, 460,
    855, 279, 129, 327, 446, 786,
    423, 486, 783, 783, 575, 15,
    856, 694, 90, 809, 970, 379,
    77, 461, 434, 238, 610, 222,
    756, 392, 29, 906, 954, 35,
    537, 724, 212, 424, 529, 387,
    647, 337, 421, 370, 141, 125,
    545, 754, 585, 826, 742, 535,
    551, 727, 663, 999, 598, 87,
    309, 828, 614, 668, 166, 740,
    801, 712, 776, 991, 94, 84,
    // Bin capacity
    1662, 1867, 1440, 1967, 1495, 1244
  };
  const int i6_18_6_8_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    288, 501, 50, 937, 860, 540,
    773, 84, 907, 484, 280, 759,
    306, 388, 872, 688, 828, 791,
    597, 287, 131, 698, 870, 41,
    642, 103, 969, 162, 643, 764,
    236, 753, 217, 588, 725, 145,
    971, 414, 256, 151, 907, 575,
    814, 959, 726, 876, 159, 726,
    65, 710, 566, 189, 760, 447,
    622, 668, 498, 290, 744, 274,
    432, 566, 282, 588, 434, 826,
    813, 545, 125, 767, 734, 701,
    477, 528, 401, 480, 631, 59,
    42, 465, 621, 44, 977, 765,
    536, 25, 592, 970, 910, 185,
    503, 982, 814, 30, 479, 410,
    846, 407, 801, 847, 888, 500,
    530, 42, 136, 101, 263, 639,
    // Bin capacity
    1709, 1517, 1614, 1601, 2177, 1647
  };
  const int i6_18_6_9_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    956, 59, 820, 889, 361, 246,
    783, 727, 668, 495, 229, 906,
    175, 952, 804, 182, 494, 975,
    922, 221, 702, 673, 605, 721,
    309, 335, 850, 774, 433, 107,
    35, 108, 560, 687, 373, 236,
    241, 574, 818, 916, 873, 424,
    290, 825, 739, 909, 879, 12,
    302, 74, 652, 921, 680, 820,
    231, 844, 343, 639, 25, 882,
    398, 791, 487, 866, 981, 882,
    551, 521, 823, 931, 284, 603,
    47, 270, 329, 499, 685, 133,
    224, 19, 181, 354, 767, 149,
    612, 676, 853, 50, 837, 611,
    871, 898, 964, 486, 145, 949,
    405, 996, 89, 119, 830, 397,
    656, 124, 941, 859, 85, 825,
    // Bin capacity
    1455, 1638, 2112, 2044, 1738, 1795
  };
  const int i6_18_6_10_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    757, 651, 750, 128, 704, 261,
    483, 134, 676, 528, 882, 299,
    877, 850, 445, 680, 768, 385,
    791, 862, 97, 196, 60, 921,
    516, 697, 959, 178, 58, 245,
    296, 914, 334, 386, 716, 976,
    256, 987, 249, 519, 350, 514,
    366, 179, 863, 242, 959, 894,
    776, 624, 79, 350, 933, 174,
    742, 94, 645, 503, 967, 603,
    58, 553, 698, 47, 740, 666,
    125, 150, 638, 895, 919, 304,
    67, 590, 788, 574, 239, 402,
    351, 730, 916, 549, 59, 169,
    203, 731, 931, 144, 509, 857,
    962, 969, 591, 119, 123, 220,
    464, 709, 504, 337, 722, 70,
    871, 710, 358, 581, 303, 194,
    // Bin capacity
    1643, 2042, 1929, 1276, 1836, 1495
  };
  const int i6_18_6_11_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    592, 622, 833, 207, 307, 318,
    674, 518, 769, 640, 272, 17,
    337, 432, 757, 410, 326, 364,
    709, 487, 18, 664, 957, 937,
    524, 752, 465, 34, 298, 799,
    570, 480, 157, 819, 218, 809,
    100, 441, 627, 433, 97, 272,
    39, 601, 983, 419, 697, 728,
    453, 500, 769, 951, 303, 50,
    652, 1000, 426, 334, 878, 824,
    204, 548, 276, 917, 49, 148,
    972, 974, 244, 861, 435, 648,
    95, 95, 542, 289, 396, 624,
    163, 180, 473, 447, 497, 898,
    894, 824, 484, 757, 290, 283,
    702, 695, 842, 841, 429, 757,
    290, 824, 559, 210, 712, 75,
    484, 987, 203, 982, 195, 423,
    // Bin capacity
    1564, 2028, 1744, 1890, 1361, 1661
  };
  const int i6_18_6_12_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    110, 164, 448, 523, 5, 764,
    712, 571, 401, 667, 96, 612,
    395, 467, 290, 139, 390, 731,
    664, 10, 221, 596, 559, 92,
    640, 495, 794, 146, 976, 311,
    32, 473, 639, 466, 568, 580,
    672, 453, 550, 294, 109, 18,
    460, 996, 328, 684, 187, 174,
    915, 74, 769, 183, 161, 919,
    229, 415, 9, 58, 96, 876,
    437, 964, 924, 598, 551, 370,
    299, 791, 898, 9, 112, 110,
    187, 318, 788, 850, 767, 858,
    431, 801, 566, 724, 705, 145,
    952, 279, 149, 498, 581, 880,
    141, 546, 941, 603, 763, 849,
    599, 691, 975, 991, 219, 452,
    11, 903, 164, 447, 874, 703,
    // Bin capacity
    1473, 1757, 1840, 1583, 1441, 1763
  };
  const int i6_18_6_13_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    530, 22, 92, 903, 62, 897,
    582, 890, 20, 550, 763, 989,
    375, 447, 735, 820, 593, 886,
    751, 923, 251, 805, 84, 726,
    71, 348, 241, 247, 866, 138,
    862, 726, 382, 887, 207, 343,
    389, 7, 171, 589, 615, 849,
    134, 608, 204, 120, 323, 571,
    910, 151, 85, 89, 398, 352,
    999, 9, 588, 171, 275, 643,
    2, 453, 964, 954, 431, 10,
    275, 700, 402, 254, 35, 636,
    271, 756, 125, 310, 306, 756,
    72, 136, 77, 81, 881, 691,
    522, 879, 528, 88, 415, 690,
    382, 604, 702, 397, 624, 718,
    391, 20, 606, 64, 826, 235,
    829, 197, 908, 92, 814, 971,
    // Bin capacity
    1573, 1484, 1334, 1398, 1605, 2091
  };
  const int i6_18_6_14_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    187, 703, 753, 259, 148, 902,
    827, 581, 417, 658, 727, 119,
    721, 328, 906, 535, 730, 595,
    317, 402, 700, 796, 146, 932,
    247, 955, 441, 67, 967, 341,
    138, 492, 961, 967, 717, 380,
    382, 951, 903, 760, 275, 976,
    634, 741, 767, 71, 255, 927,
    111, 254, 559, 577, 674, 593,
    394, 879, 821, 484, 438, 693,
    557, 763, 240, 789, 327, 79,
    134, 316, 29, 290, 599, 322,
    339, 873, 254, 969, 440, 133,
    145, 617, 565, 239, 373, 122,
    92, 485, 47, 714, 766, 148,
    49, 582, 298, 324, 569, 339,
    848, 578, 657, 541, 356, 949,
    760, 973, 86, 656, 472, 298,
    // Bin capacity
    1308, 2180, 1787, 1843, 1707, 1682
  };
  const int i6_18_6_15_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    581, 833, 50, 282, 153, 314,
    352, 223, 638, 794, 871, 878,
    995, 853, 206, 509, 399, 907,
    695, 730, 677, 333, 426, 618,
    878, 313, 454, 340, 331, 896,
    232, 421, 834, 384, 410, 150,
    740, 294, 624, 311, 759, 481,
    243, 675, 808, 107, 75, 626,
    174, 856, 560, 788, 636, 708,
    838, 462, 10, 162, 469, 440,
    839, 300, 735, 825, 964, 578,
    529, 991, 359, 601, 967, 241,
    910, 151, 185, 90, 229, 999,
    508, 108, 429, 856, 762, 44,
    360, 253, 794, 660, 96, 484,
    437, 53, 589, 577, 315, 725,
    864, 968, 16, 18, 185, 704,
    276, 189, 736, 170, 693, 36,
    // Bin capacity
    2004, 1663, 1669, 1497, 1676, 1884
  };
  const int i6_18_6_16_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    962, 647, 413, 860, 898, 536,
    565, 970, 346, 504, 283, 807,
    864, 901, 264, 15, 809, 493,
    487, 545, 403, 264, 925, 890,
    168, 862, 647, 984, 152, 436,
    992, 839, 326, 856, 714, 50,
    823, 952, 228, 230, 281, 955,
    305, 929, 184, 978, 578, 113,
    273, 731, 464, 515, 22, 252,
    750, 745, 611, 595, 863, 315,
    578, 522, 440, 893, 503, 735,
    663, 528, 927, 348, 57, 916,
    418, 560, 11, 93, 568, 427,
    484, 320, 585, 727, 337, 202,
    479, 300, 120, 788, 861, 558,
    247, 507, 604, 547, 202, 768,
    426, 242, 964, 951, 774, 790,
    90, 845, 378, 338, 465, 529,
    // Bin capacity
    1851, 2310, 1531, 2028, 1797, 1890
  };
  const int i6_18_6_17_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    827, 1, 476, 452, 587, 394,
    499, 733, 646, 160, 795, 872,
    714, 866, 185, 246, 990, 17,
    695, 224, 972, 233, 588, 323,
    747, 462, 904, 855, 462, 868,
    749, 742, 504, 204, 292, 363,
    856, 209, 945, 713, 864, 842,
    644, 28, 813, 956, 760, 596,
    464, 365, 122, 319, 252, 791,
    254, 601, 554, 828, 641, 170,
    65, 668, 489, 122, 229, 569,
    414, 88, 636, 211, 133, 872,
    737, 682, 846, 271, 262, 260,
    405, 713, 252, 319, 575, 314,
    987, 400, 542, 708, 389, 124,
    133, 528, 759, 972, 506, 55,
    472, 894, 660, 988, 622, 881,
    67, 395, 141, 137, 552, 108,
    // Bin capacity
    1898, 1677, 2037, 1696, 1853, 1642
  };
  const int i6_18_6_18_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    912, 83, 321, 725, 167, 163,
    500, 720, 759, 31, 630, 630,
    657, 60, 545, 447, 468, 128,
    915, 549, 894, 70, 694, 143,
    890, 249, 214, 481, 841, 62,
    493, 883, 78, 983, 588, 911,
    729, 71, 719, 609, 110, 592,
    298, 712, 488, 939, 663, 1,
    241, 919, 994, 631, 197, 297,
    873, 707, 878, 106, 297, 62,
    457, 998, 860, 999, 290, 625,
    722, 358, 554, 913, 540, 499,
    424, 930, 598, 453, 903, 968,
    634, 3, 59, 267, 877, 938,
    956, 294, 213, 870, 578, 20,
    565, 250, 51, 303, 46, 948,
    985, 927, 263, 278, 616, 880,
    377, 916, 810, 193, 123, 32,
    // Bin capacity
    2287, 1894, 1829, 1829, 1697, 1554
  };
  const int i6_18_6_19_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    384, 164, 280, 154, 835, 849,
    632, 120, 99, 897, 647, 540,
    905, 295, 847, 88, 45, 227,
    490, 926, 337, 543, 896, 744,
    743, 366, 464, 116, 276, 524,
    87, 804, 97, 304, 873, 207,
    25, 379, 271, 941, 664, 881,
    14, 543, 908, 789, 402, 563,
    415, 73, 224, 252, 793, 105,
    228, 456, 805, 751, 756, 97,
    996, 851, 600, 175, 958, 200,
    885, 840, 918, 881, 810, 430,
    728, 4, 975, 322, 807, 203,
    465, 937, 578, 72, 372, 811,
    267, 865, 634, 471, 593, 283,
    286, 775, 542, 851, 47, 473,
    22, 578, 913, 317, 612, 400,
    118, 839, 638, 133, 584, 583,
    // Bin capacity
    1526, 1947, 2010, 1598, 2176, 1611
  };
  const int i6_18_6_20_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    469, 753, 354, 325, 98, 717,
    575, 899, 593, 585, 696, 563,
    720, 808, 156, 50, 697, 666,
    344, 633, 335, 567, 14, 633,
    552, 379, 352, 506, 133, 112,
    409, 320, 454, 711, 180, 144,
    371, 69, 135, 268, 221, 704,
    40, 46, 536, 816, 371, 947,
    964, 463, 760, 825, 387, 75,
    946, 746, 387, 947, 541, 223,
    669, 57, 919, 89, 155, 661,
    366, 625, 74, 291, 395, 192,
    676, 3, 4, 872, 66, 371,
    914, 532, 344, 801, 495, 387,
    94, 464, 581, 936, 607, 781,
    374, 541, 68, 782, 234, 965,
    746, 322, 546, 660, 982, 221,
    901, 594, 365, 453, 9, 782,
    // Bin capacity
    2026, 1651, 1393, 2097, 1257, 1829
  };
  const int i6_18_6_21_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    573, 958, 993, 86, 559, 356,
    918, 671, 594, 37, 962, 921,
    6, 830, 708, 247, 699, 86,
    870, 918, 634, 297, 732, 826,
    775, 865, 227, 917, 908, 496,
    694, 825, 533, 854, 681, 331,
    67, 302, 142, 773, 110, 158,
    357, 144, 344, 542, 558, 757,
    671, 501, 186, 278, 74, 862,
    336, 415, 676, 909, 91, 249,
    449, 74, 669, 840, 401, 181,
    258, 675, 210, 846, 922, 558,
    3, 774, 765, 478, 987, 569,
    850, 230, 322, 74, 974, 458,
    568, 88, 517, 816, 78, 958,
    672, 107, 304, 492, 735, 71,
    186, 34, 826, 180, 602, 541,
    857, 648, 388, 187, 89, 390,
    // Bin capacity
    1838, 1827, 1823, 1786, 2050, 1769
  };
  const int i6_18_6_22_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    218, 894, 882, 348, 218, 648,
    258, 788, 339, 707, 841, 169,
    123, 571, 674, 174, 334, 169,
    65, 658, 569, 439, 773, 418,
    81, 578, 612, 35, 861, 24,
    394, 569, 831, 879, 602, 460,
    217, 963, 575, 978, 38, 722,
    878, 599, 230, 365, 244, 624,
    549, 378, 104, 885, 787, 460,
    918, 829, 393, 619, 553, 118,
    193, 424, 474, 548, 231, 644,
    585, 297, 322, 970, 371, 801,
    745, 419, 798, 150, 604, 344,
    824, 991, 222, 450, 807, 402,
    862, 375, 469, 881, 637, 731,
    858, 595, 45, 776, 909, 0,
    710, 803, 823, 829, 847, 750,
    130, 93, 775, 67, 97, 224,
    // Bin capacity
    1751, 2201, 1858, 2054, 1984, 1568
  };
  const int i6_18_6_23_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    830, 70, 97, 762, 494, 272,
    646, 830, 522, 100, 89, 28,
    102, 661, 219, 869, 129, 993,
    333, 789, 990, 879, 945, 27,
    764, 201, 939, 799, 179, 875,
    448, 502, 912, 579, 691, 341,
    871, 609, 128, 797, 949, 479,
    335, 836, 678, 46, 766, 411,
    954, 888, 944, 117, 533, 467,
    868, 135, 924, 228, 354, 357,
    384, 962, 307, 846, 68, 650,
    155, 893, 34, 530, 720, 290,
    656, 728, 310, 364, 93, 914,
    18, 344, 3, 492, 274, 615,
    411, 255, 662, 981, 688, 840,
    727, 970, 889, 173, 328, 257,
    964, 171, 927, 157, 897, 867,
    560, 154, 777, 586, 35, 350,
    // Bin capacity
    2056, 2050, 2104, 1908, 1688, 1852
  };
  const int i6_18_6_24_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    374, 857, 506, 336, 265, 117,
    63, 500, 467, 435, 859, 101,
    894, 726, 997, 346, 93, 615,
    4, 960, 9, 454, 214, 494,
    624, 147, 511, 777, 153, 430,
    998, 487, 426, 803, 259, 375,
    790, 13, 195, 385, 118, 764,
    564, 171, 188, 412, 621, 319,
    729, 984, 101, 857, 511, 371,
    433, 111, 486, 368, 269, 328,
    636, 533, 975, 295, 972, 844,
    706, 532, 94, 511, 1, 943,
    375, 690, 48, 581, 374, 699,
    83, 140, 510, 762, 515, 939,
    814, 338, 559, 752, 298, 131,
    853, 999, 102, 871, 162, 34,
    538, 939, 674, 835, 445, 260,
    978, 619, 402, 19, 408, 436,
    // Bin capacity
    2161, 2015, 1499, 2026, 1351, 1695
  };
  const int i6_18_6_25_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    960, 360, 377, 891, 63, 541,
    442, 62, 978, 584, 105, 225,
    559, 387, 361, 492, 743, 712,
    568, 636, 834, 711, 680, 147,
    593, 554, 540, 726, 480, 778,
    149, 880, 554, 449, 33, 363,
    25, 897, 112, 7, 983, 749,
    446, 650, 376, 712, 854, 969,
    180, 354, 220, 580, 359, 322,
    331, 154, 323, 513, 592, 350,
    867, 721, 163, 43, 454, 252,
    102, 945, 328, 67, 719, 130,
    406, 964, 654, 49, 379, 650,
    735, 962, 199, 93, 439, 619,
    186, 373, 546, 538, 70, 433,
    530, 504, 715, 793, 479, 155,
    466, 640, 599, 987, 254, 444,
    466, 993, 242, 783, 160, 293,
    // Bin capacity
    1669, 2300, 1692, 1879, 1635, 1695
  };
  const int i6_18_6_26_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    329, 553, 623, 898, 39, 213,
    265, 95, 460, 204, 225, 544,
    319, 588, 914, 233, 513, 550,
    495, 296, 7, 791, 841, 692,
    128, 125, 653, 391, 608, 251,
    442, 53, 568, 163, 722, 719,
    398, 530, 685, 813, 622, 769,
    480, 428, 382, 626, 620, 635,
    593, 503, 116, 87, 401, 401,
    200, 740, 2, 920, 440, 551,
    358, 750, 745, 993, 169, 317,
    431, 61, 344, 334, 725, 657,
    14, 589, 13, 333, 201, 869,
    921, 257, 326, 399, 125, 294,
    665, 407, 458, 780, 620, 606,
    437, 440, 994, 426, 396, 753,
    400, 62, 807, 971, 729, 343,
    928, 500, 891, 384, 276, 914,
    // Bin capacity
    1639, 1466, 1888, 2047, 1738, 2117
  };
  const int i6_18_6_27_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    152, 949, 474, 85, 283, 328,
    856, 414, 324, 910, 431, 0,
    586, 677, 147, 944, 966, 694,
    753, 270, 399, 671, 933, 409,
    933, 169, 230, 216, 17, 308,
    35, 316, 674, 768, 241, 195,
    651, 761, 266, 974, 293, 532,
    933, 455, 980, 919, 997, 627,
    448, 788, 826, 340, 858, 404,
    947, 997, 401, 453, 358, 762,
    719, 770, 712, 164, 852, 77,
    904, 988, 503, 961, 238, 890,
    102, 832, 719, 615, 877, 675,
    134, 752, 820, 833, 452, 269,
    348, 240, 399, 305, 356, 377,
    368, 225, 132, 667, 65, 553,
    820, 450, 404, 573, 318, 973,
    241, 669, 15, 396, 427, 941,
    // Bin capacity
    2102, 2270, 1784, 2285, 1897, 1908
  };
  const int i6_18_6_28_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    409, 887, 921, 102, 110, 84,
    243, 348, 813, 60, 229, 52,
    780, 354, 250, 699, 955, 941,
    411, 596, 644, 208, 3, 990,
    390, 335, 609, 683, 76, 809,
    591, 35, 860, 593, 50, 692,
    388, 597, 701, 311, 96, 112,
    988, 311, 511, 973, 766, 497,
    176, 51, 1, 174, 367, 750,
    252, 697, 449, 120, 235, 189,
    640, 352, 10, 450, 246, 331,
    275, 493, 188, 465, 9, 70,
    470, 6, 178, 806, 148, 385,
    363, 447, 854, 288, 27, 928,
    248, 381, 633, 609, 65, 847,
    768, 578, 261, 23, 131, 235,
    312, 514, 988, 270, 226, 940,
    666, 80, 657, 696, 758, 706,
    // Bin capacity
    1786, 1507, 2033, 1607, 960, 2040
  };
  const int i6_18_6_29_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    618, 496, 700, 289, 853, 917,
    55, 75, 383, 912, 398, 431,
    911, 454, 576, 423, 504, 329,
    149, 437, 22, 502, 202, 433,
    27, 515, 533, 856, 114, 745,
    364, 65, 102, 761, 596, 207,
    96, 440, 977, 448, 542, 496,
    115, 877, 429, 334, 674, 395,
    984, 49, 47, 168, 995, 30,
    718, 49, 116, 595, 338, 280,
    387, 150, 901, 941, 243, 383,
    860, 284, 248, 505, 487, 461,
    197, 575, 289, 666, 127, 899,
    740, 322, 303, 311, 698, 483,
    839, 825, 31, 390, 408, 554,
    507, 925, 239, 750, 846, 582,
    347, 367, 909, 851, 129, 195,
    951, 224, 791, 408, 587, 90,
    // Bin capacity
    1906, 1533, 1634, 2174, 1880, 1701
  };
  const int i6_18_6_30_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    188, 130, 205, 961, 872, 461,
    116, 60, 161, 511, 370, 36,
    419, 13, 8, 28, 281, 934,
    927, 525, 934, 553, 415, 933,
    156, 220, 136, 449, 815, 982,
    20, 151, 23, 905, 436, 218,
    308, 674, 93, 173, 636, 767,
    892, 441, 392, 965, 882, 841,
    257, 946, 49, 440, 979, 314,
    252, 873, 835, 353, 577, 158,
    48, 194, 81, 599, 897, 499,
    443, 994, 762, 114, 261, 897,
    621, 1000, 695, 978, 526, 52,
    533, 384, 990, 515, 540, 653,
    5, 750, 427, 787, 970, 160,
    394, 628, 604, 855, 235, 383,
    737, 123, 532, 138, 604, 556,
    812, 611, 430, 494, 13, 415,
    // Bin capacity
    1545, 1889, 1595, 2128, 2234, 2007
  };
  const int i6_18_6_31_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    669, 644, 66, 507, 49, 134,
    626, 990, 677, 405, 796, 311,
    21, 422, 230, 383, 671, 48,
    632, 662, 988, 880, 66, 619,
    118, 630, 272, 376, 516, 22,
    481, 325, 402, 197, 801, 789,
    125, 346, 533, 566, 699, 96,
    563, 536, 190, 368, 557, 757,
    734, 515, 995, 558, 489, 484,
    350, 664, 458, 623, 709, 857,
    568, 914, 526, 160, 660, 282,
    211, 302, 706, 711, 639, 991,
    860, 822, 788, 488, 272, 672,
    979, 583, 701, 731, 258, 244,
    297, 881, 243, 490, 197, 276,
    200, 590, 649, 187, 795, 456,
    945, 405, 815, 947, 139, 847,
    508, 625, 805, 143, 775, 813,
    // Bin capacity
    1941, 2371, 2193, 1904, 1985, 1900
  };
  const int i6_18_6_32_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    367, 512, 33, 485, 989, 589,
    249, 859, 212, 452, 952, 825,
    263, 930, 510, 607, 197, 61,
    673, 13, 270, 852, 981, 762,
    70, 350, 28, 924, 596, 340,
    277, 642, 409, 326, 40, 781,
    954, 122, 599, 941, 655, 961,
    314, 447, 423, 914, 440, 241,
    473, 812, 172, 479, 125, 541,
    528, 979, 781, 768, 80, 918,
    422, 526, 947, 367, 684, 754,
    539, 384, 427, 32, 938, 435,
    928, 918, 406, 609, 230, 893,
    411, 313, 825, 754, 553, 79,
    153, 938, 379, 772, 739, 9,
    820, 254, 88, 525, 397, 315,
    931, 261, 375, 899, 751, 130,
    598, 185, 16, 255, 579, 351,
    // Bin capacity
    1974, 2078, 1518, 2412, 2184, 1977
  };
  const int i6_18_6_33_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    595, 27, 600, 493, 765, 474,
    640, 547, 291, 501, 64, 956,
    306, 346, 930, 82, 701, 181,
    556, 56, 546, 857, 664, 45,
    338, 877, 269, 735, 19, 704,
    635, 104, 807, 207, 384, 952,
    564, 75, 136, 669, 635, 438,
    984, 455, 84, 540, 541, 369,
    468, 5, 254, 723, 980, 6,
    273, 209, 142, 999, 258, 394,
    442, 602, 294, 520, 237, 243,
    587, 126, 656, 67, 518, 677,
    717, 185, 444, 977, 392, 771,
    538, 299, 911, 176, 946, 36,
    158, 698, 889, 942, 969, 727,
    690, 89, 366, 737, 398, 385,
    783, 197, 683, 826, 149, 996,
    502, 916, 60, 20, 862, 859,
    // Bin capacity
    2168, 1289, 1854, 2233, 2102, 2043
  };
  const int i6_18_6_34_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    172, 183, 283, 909, 896, 488,
    13, 469, 608, 846, 490, 639,
    76, 303, 599, 648, 385, 513,
    847, 700, 725, 520, 296, 530,
    640, 656, 326, 789, 759, 999,
    922, 815, 14, 665, 422, 95,
    234, 180, 404, 640, 891, 137,
    813, 891, 831, 671, 547, 314,
    973, 575, 533, 586, 517, 765,
    593, 117, 863, 820, 723, 311,
    960, 674, 41, 683, 614, 718,
    431, 404, 953, 567, 959, 378,
    691, 543, 71, 902, 120, 751,
    794, 481, 212, 493, 118, 694,
    660, 342, 480, 261, 58, 295,
    853, 376, 633, 257, 499, 351,
    225, 470, 944, 117, 941, 70,
    296, 670, 580, 259, 812, 473,
    // Bin capacity
    2277, 1977, 2033, 2375, 2244, 1904
  };
  const int i6_18_6_35_8[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    888, 380, 364, 179, 658, 609,
    41, 531, 894, 93, 916, 876,
    491, 335, 831, 346, 987, 238,
    559, 483, 536, 507, 639, 834,
    647, 944, 177, 421, 449, 603,
    386, 626, 806, 664, 368, 437,
    198, 483, 594, 829, 927, 503,
    982, 196, 260, 857, 451, 274,
    433, 962, 115, 640, 854, 353,
    177, 656, 264, 410, 477, 758,
    480, 875, 20, 303, 471, 886,
    57, 597, 822, 489, 764, 3,
    282, 850, 755, 857, 124, 502,
    626, 27, 129, 315, 707, 101,
    282, 816, 474, 511, 360, 659,
    134, 543, 995, 13, 889, 204,
    433, 637, 111, 201, 139, 964,
    590, 512, 422, 264, 573, 617,
    // Bin capacity
    1730, 2352, 1929, 1778, 2420, 2120
  };
  const int i6_18_6_0_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    754, 569, 300, 970, 455, 542,
    191, 512, 649, 454, 530, 327,
    254, 533, 77, 930, 99, 571,
    505, 192, 792, 795, 28, 327,
    715, 770, 739, 734, 210, 73,
    123, 847, 879, 923, 406, 26,
    949, 829, 155, 261, 58, 443,
    640, 574, 252, 666, 725, 774,
    660, 748, 138, 136, 917, 267,
    896, 134, 345, 537, 197, 815,
    839, 394, 32, 992, 534, 125,
    308, 434, 340, 89, 456, 62,
    432, 377, 267, 971, 135, 895,
    312, 439, 218, 362, 885, 651,
    3, 932, 764, 962, 588, 673,
    839, 209, 360, 558, 267, 817,
    56, 155, 924, 649, 876, 114,
    74, 170, 124, 161, 811, 984,
    // Bin capacity
    1425, 1470, 1226, 1859, 1363, 1415
  };
  const int i6_18_6_1_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    464, 510, 232, 914, 283, 490,
    311, 49, 1, 97, 0, 919,
    855, 852, 996, 222, 954, 87,
    969, 921, 451, 419, 114, 932,
    605, 860, 83, 257, 956, 533,
    920, 393, 831, 208, 824, 961,
    962, 296, 493, 723, 438, 481,
    970, 117, 140, 309, 351, 118,
    240, 425, 205, 964, 940, 50,
    28, 25, 201, 84, 734, 49,
    443, 208, 156, 637, 318, 227,
    443, 244, 94, 291, 437, 899,
    40, 48, 863, 427, 724, 801,
    724, 915, 84, 632, 547, 940,
    972, 915, 419, 454, 968, 973,
    759, 536, 496, 738, 161, 7,
    462, 958, 568, 487, 502, 621,
    33, 948, 468, 244, 126, 577,
    // Bin capacity
    1717, 1553, 1142, 1365, 1579, 1627
  };
  const int i6_18_6_2_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    695, 796, 444, 299, 344, 334,
    293, 160, 910, 899, 170, 245,
    652, 386, 832, 787, 469, 565,
    815, 704, 962, 59, 791, 862,
    896, 403, 389, 391, 50, 695,
    807, 183, 347, 240, 926, 920,
    448, 325, 148, 318, 50, 304,
    765, 302, 789, 984, 157, 728,
    948, 419, 346, 882, 303, 413,
    937, 826, 556, 465, 591, 869,
    99, 165, 548, 424, 809, 673,
    317, 773, 295, 568, 728, 798,
    234, 620, 364, 853, 182, 775,
    563, 250, 303, 768, 615, 64,
    75, 930, 266, 802, 871, 2,
    465, 411, 40, 256, 639, 42,
    498, 218, 642, 545, 549, 116,
    606, 524, 402, 556, 982, 905,
    // Bin capacity
    1720, 1428, 1460, 1717, 1569, 1583
  };
  const int i6_18_6_3_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    817, 935, 919, 24, 13, 605,
    866, 726, 540, 339, 172, 721,
    271, 250, 760, 904, 635, 916,
    675, 851, 743, 299, 538, 302,
    102, 85, 700, 748, 979, 741,
    665, 928, 530, 768, 952, 35,
    340, 753, 434, 967, 779, 291,
    986, 568, 888, 741, 889, 637,
    933, 883, 774, 554, 593, 333,
    776, 325, 793, 381, 680, 249,
    326, 794, 834, 745, 366, 621,
    550, 813, 785, 784, 40, 101,
    447, 384, 695, 660, 960, 140,
    288, 726, 395, 368, 191, 32,
    868, 596, 471, 387, 214, 678,
    788, 699, 834, 818, 294, 883,
    758, 333, 324, 518, 554, 866,
    204, 975, 190, 416, 352, 295,
    // Bin capacity
    1830, 1996, 1993, 1789, 1580, 1450
  };
  const int i6_18_6_4_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    470, 128, 439, 419, 927, 991,
    203, 928, 595, 134, 378, 137,
    82, 623, 809, 128, 545, 536,
    198, 982, 669, 89, 872, 984,
    635, 786, 660, 166, 168, 632,
    846, 951, 340, 237, 369, 59,
    820, 571, 648, 554, 911, 405,
    940, 51, 593, 922, 61, 673,
    769, 798, 147, 788, 153, 293,
    987, 195, 140, 15, 315, 308,
    794, 202, 183, 229, 12, 687,
    193, 487, 384, 382, 220, 674,
    901, 260, 999, 882, 239, 505,
    979, 944, 866, 457, 271, 485,
    137, 209, 248, 972, 76, 231,
    250, 744, 939, 395, 715, 827,
    159, 40, 403, 789, 11, 807,
    728, 615, 571, 32, 229, 706,
    // Bin capacity
    1750, 1650, 1670, 1316, 1122, 1723
  };
  const int i6_18_6_5_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    146, 403, 682, 629, 230, 901,
    667, 44, 752, 162, 170, 807,
    190, 795, 599, 605, 87, 258,
    533, 627, 408, 259, 852, 685,
    827, 656, 648, 464, 249, 258,
    265, 995, 580, 590, 188, 241,
    679, 791, 23, 41, 881, 838,
    382, 405, 287, 979, 20, 149,
    266, 628, 306, 857, 975, 519,
    123, 414, 959, 803, 485, 493,
    231, 686, 534, 372, 933, 129,
    512, 401, 85, 824, 653, 27,
    379, 727, 615, 248, 766, 477,
    615, 533, 99, 47, 122, 613,
    591, 421, 984, 855, 512, 467,
    769, 256, 280, 152, 808, 28,
    973, 229, 457, 469, 636, 48,
    584, 626, 920, 17, 4, 769,
    // Bin capacity
    1529, 1687, 1614, 1466, 1500, 1349
  };
  const int i6_18_6_6_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    862, 363, 782, 252, 111, 986,
    759, 234, 64, 726, 231, 550,
    59, 705, 568, 829, 400, 778,
    969, 415, 629, 608, 877, 331,
    625, 724, 531, 586, 802, 449,
    421, 69, 723, 368, 352, 819,
    943, 537, 918, 361, 586, 974,
    863, 911, 775, 842, 685, 848,
    684, 965, 600, 206, 121, 374,
    44, 799, 792, 231, 875, 471,
    482, 801, 111, 944, 249, 968,
    733, 661, 792, 619, 880, 412,
    603, 906, 439, 74, 291, 157,
    60, 591, 999, 101, 808, 724,
    871, 462, 489, 362, 105, 517,
    630, 94, 214, 13, 730, 745,
    457, 94, 973, 960, 474, 988,
    445, 675, 702, 176, 929, 776,
    // Bin capacity
    1857, 1768, 1962, 1459, 1680, 2097
  };
  const int i6_18_6_7_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    491, 970, 222, 567, 124, 52,
    625, 474, 546, 249, 440, 336,
    970, 71, 933, 503, 732, 198,
    467, 2, 157, 597, 618, 500,
    61, 974, 695, 733, 285, 84,
    274, 296, 624, 126, 675, 165,
    915, 906, 814, 99, 311, 995,
    686, 462, 711, 255, 723, 156,
    369, 572, 475, 389, 64, 596,
    801, 480, 134, 104, 907, 449,
    383, 305, 542, 550, 887, 857,
    801, 45, 976, 53, 898, 676,
    308, 214, 22, 431, 465, 500,
    764, 54, 824, 420, 313, 924,
    646, 735, 829, 417, 964, 161,
    907, 164, 406, 559, 371, 906,
    499, 433, 315, 696, 437, 286,
    512, 886, 869, 809, 715, 401,
    // Bin capacity
    1869, 1435, 1801, 1348, 1771, 1470
  };
  const int i6_18_6_8_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    977, 67, 437, 104, 714, 370,
    787, 528, 283, 634, 649, 175,
    455, 655, 393, 273, 233, 756,
    584, 322, 684, 829, 708, 831,
    872, 421, 463, 504, 780, 105,
    433, 686, 839, 850, 901, 858,
    602, 3, 351, 447, 278, 556,
    349, 97, 653, 368, 805, 930,
    913, 216, 849, 752, 924, 350,
    810, 593, 619, 307, 453, 788,
    40, 760, 970, 137, 135, 443,
    607, 688, 623, 616, 793, 98,
    707, 912, 814, 41, 543, 949,
    903, 940, 600, 848, 818, 662,
    755, 707, 522, 327, 234, 293,
    659, 433, 635, 639, 723, 11,
    611, 108, 260, 962, 539, 501,
    771, 128, 806, 245, 100, 675,
    // Bin capacity
    2131, 1488, 1945, 1599, 1860, 1684
  };
  const int i6_18_6_9_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    559, 859, 888, 718, 58, 225,
    655, 155, 154, 919, 420, 110,
    772, 453, 720, 751, 403, 82,
    778, 945, 278, 267, 103, 45,
    476, 768, 915, 792, 404, 132,
    911, 860, 455, 199, 364, 968,
    951, 243, 908, 224, 959, 461,
    382, 37, 225, 211, 137, 779,
    346, 141, 14, 87, 817, 946,
    925, 829, 59, 701, 615, 520,
    478, 344, 568, 172, 595, 978,
    107, 704, 641, 367, 757, 430,
    7, 273, 226, 886, 551, 224,
    133, 570, 765, 134, 129, 257,
    832, 913, 866, 398, 787, 741,
    440, 346, 817, 327, 364, 438,
    350, 526, 134, 34, 887, 57,
    111, 196, 299, 201, 604, 548,
    // Bin capacity
    1674, 1665, 1623, 1343, 1627, 1443
  };
  const int i6_18_6_10_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    549, 675, 893, 637, 864, 138,
    500, 146, 927, 777, 625, 519,
    624, 1000, 249, 434, 252, 246,
    602, 2, 201, 59, 967, 588,
    210, 35, 165, 886, 471, 31,
    4, 724, 609, 410, 459, 595,
    240, 26, 183, 620, 773, 267,
    634, 334, 360, 36, 329, 487,
    160, 729, 900, 382, 582, 309,
    584, 780, 821, 963, 749, 207,
    116, 930, 826, 88, 833, 974,
    127, 434, 962, 570, 489, 389,
    991, 660, 972, 728, 308, 888,
    533, 322, 69, 134, 729, 677,
    175, 560, 134, 752, 956, 486,
    433, 672, 361, 217, 420, 209,
    975, 462, 442, 181, 476, 352,
    21, 363, 318, 255, 38, 356,
    // Bin capacity
    1371, 1624, 1722, 1491, 1893, 1415
  };
  const int i6_18_6_11_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    192, 192, 592, 930, 172, 179,
    55, 0, 419, 63, 698, 509,
    131, 283, 733, 233, 65, 971,
    973, 856, 458, 354, 122, 973,
    675, 124, 345, 46, 122, 981,
    239, 484, 532, 954, 203, 626,
    630, 484, 93, 915, 805, 18,
    354, 733, 460, 281, 783, 274,
    624, 36, 689, 73, 509, 840,
    531, 850, 23, 955, 816, 59,
    724, 402, 786, 86, 782, 389,
    431, 40, 387, 790, 490, 570,
    876, 829, 911, 796, 86, 822,
    342, 219, 988, 644, 882, 525,
    5, 138, 865, 241, 612, 123,
    294, 614, 512, 841, 26, 54,
    756, 341, 994, 684, 75, 418,
    839, 213, 414, 53, 711, 49,
    // Bin capacity
    1605, 1266, 1888, 1654, 1473, 1551
  };
  const int i6_18_6_12_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    971, 79, 469, 76, 26, 114,
    399, 72, 615, 505, 341, 612,
    927, 447, 653, 998, 667, 398,
    763, 116, 431, 757, 35, 355,
    785, 988, 944, 556, 5, 737,
    799, 953, 568, 106, 633, 422,
    247, 340, 328, 430, 851, 204,
    910, 668, 202, 818, 750, 436,
    204, 844, 719, 514, 965, 809,
    57, 666, 224, 568, 581, 787,
    526, 540, 2, 167, 357, 181,
    253, 542, 133, 437, 187, 95,
    724, 725, 737, 809, 654, 74,
    898, 99, 593, 87, 778, 482,
    588, 227, 146, 393, 168, 950,
    614, 709, 18, 432, 89, 308,
    63, 632, 369, 778, 807, 112,
    763, 21, 891, 326, 543, 903,
    // Bin capacity
    1959, 1619, 1502, 1635, 1575, 1490
  };
  const int i6_18_6_13_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    737, 488, 256, 331, 422, 100,
    764, 842, 786, 950, 951, 233,
    993, 986, 727, 769, 66, 744,
    582, 581, 418, 67, 470, 573,
    823, 296, 766, 686, 42, 933,
    785, 886, 502, 267, 624, 978,
    136, 957, 514, 328, 642, 133,
    825, 127, 565, 779, 768, 329,
    718, 47, 366, 148, 579, 287,
    180, 986, 291, 96, 789, 401,
    112, 917, 480, 616, 789, 989,
    204, 976, 295, 362, 529, 447,
    665, 660, 182, 272, 547, 769,
    888, 980, 855, 147, 983, 863,
    155, 479, 945, 69, 306, 784,
    863, 365, 148, 893, 67, 619,
    476, 264, 44, 490, 760, 605,
    290, 800, 777, 318, 712, 114,
    // Bin capacity
    1921, 2192, 1680, 1430, 1892, 1865
  };
  const int i6_18_6_14_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    529, 768, 663, 606, 235, 757,
    52, 356, 340, 330, 746, 120,
    118, 772, 327, 440, 960, 534,
    171, 143, 524, 466, 429, 826,
    451, 610, 746, 600, 417, 347,
    982, 185, 819, 448, 142, 122,
    174, 90, 184, 246, 853, 801,
    685, 457, 347, 765, 814, 573,
    876, 445, 571, 417, 844, 222,
    165, 32, 759, 198, 339, 985,
    903, 432, 827, 851, 625, 397,
    754, 261, 226, 407, 615, 381,
    33, 153, 175, 177, 693, 661,
    37, 894, 324, 738, 115, 982,
    557, 844, 801, 386, 97, 851,
    707, 758, 504, 258, 758, 729,
    142, 646, 511, 997, 628, 495,
    526, 73, 987, 188, 118, 674,
    // Bin capacity
    1494, 1505, 1831, 1619, 1792, 1987
  };
  const int i6_18_6_15_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    487, 709, 227, 329, 71, 676,
    963, 440, 694, 567, 657, 944,
    494, 948, 983, 181, 317, 876,
    832, 337, 253, 718, 102, 809,
    567, 537, 860, 229, 721, 351,
    0, 510, 840, 391, 721, 290,
    970, 39, 64, 250, 495, 673,
    897, 374, 316, 891, 806, 811,
    310, 219, 840, 77, 665, 920,
    75, 90, 433, 934, 652, 706,
    89, 477, 322, 370, 285, 452,
    693, 772, 818, 770, 122, 692,
    988, 303, 960, 709, 621, 450,
    865, 240, 128, 210, 267, 48,
    383, 36, 156, 217, 256, 74,
    404, 19, 930, 794, 889, 952,
    253, 291, 264, 894, 953, 339,
    127, 35, 986, 912, 303, 139,
    // Bin capacity
    1802, 1223, 1931, 1810, 1707, 1956
  };
  const int i6_18_6_16_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    380, 362, 672, 408, 371, 557,
    369, 230, 753, 31, 361, 693,
    466, 684, 631, 293, 552, 785,
    244, 351, 225, 716, 630, 598,
    603, 95, 630, 550, 347, 679,
    76, 849, 863, 202, 805, 432,
    480, 384, 925, 151, 1000, 975,
    625, 382, 465, 177, 794, 264,
    12, 47, 456, 571, 530, 873,
    837, 125, 982, 534, 391, 754,
    347, 218, 864, 3, 327, 360,
    491, 842, 623, 183, 661, 221,
    489, 530, 445, 825, 598, 400,
    666, 975, 558, 364, 463, 469,
    464, 647, 207, 172, 153, 540,
    249, 803, 113, 247, 558, 805,
    637, 916, 888, 287, 888, 148,
    353, 773, 563, 940, 76, 294,
    // Bin capacity
    1506, 1782, 2101, 1287, 1838, 1904
  };
  const int i6_18_6_17_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    7, 423, 888, 931, 974, 842,
    596, 888, 512, 793, 601, 566,
    264, 906, 645, 508, 830, 766,
    322, 480, 86, 921, 572, 995,
    341, 62, 862, 335, 376, 417,
    86, 281, 385, 149, 258, 225,
    329, 67, 486, 779, 89, 460,
    841, 219, 216, 179, 555, 758,
    259, 527, 948, 456, 692, 615,
    193, 114, 25, 112, 176, 901,
    923, 766, 29, 698, 463, 508,
    95, 482, 453, 278, 396, 445,
    167, 649, 285, 767, 268, 517,
    492, 308, 884, 228, 186, 508,
    690, 171, 823, 249, 880, 121,
    127, 543, 68, 157, 639, 181,
    397, 100, 116, 809, 872, 134,
    27, 463, 597, 357, 570, 617,
    // Bin capacity
    1201, 1453, 1621, 1698, 1833, 1868
  };
  const int i6_18_6_18_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    545, 102, 617, 231, 341, 687,
    664, 147, 451, 810, 200, 824,
    129, 377, 780, 416, 658, 782,
    949, 181, 208, 166, 330, 612,
    537, 478, 840, 378, 330, 628,
    673, 240, 843, 844, 166, 141,
    146, 678, 858, 853, 606, 623,
    198, 852, 749, 744, 33, 879,
    504, 847, 222, 523, 988, 8,
    265, 483, 272, 906, 69, 858,
    42, 744, 147, 523, 173, 425,
    683, 614, 882, 651, 46, 534,
    863, 623, 148, 985, 276, 168,
    14, 750, 199, 9, 890, 608,
    537, 709, 153, 689, 773, 622,
    845, 626, 943, 843, 48, 229,
    776, 963, 311, 450, 784, 136,
    501, 87, 768, 301, 191, 48,
    // Bin capacity
    1745, 1869, 1847, 2030, 1358, 1734
  };
  const int i6_18_6_19_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    372, 637, 5, 979, 498, 396,
    535, 760, 487, 667, 699, 555,
    6, 787, 192, 947, 954, 156,
    750, 713, 926, 521, 207, 967,
    593, 502, 530, 366, 889, 832,
    453, 428, 696, 800, 719, 936,
    297, 145, 149, 779, 666, 615,
    242, 840, 328, 782, 271, 144,
    480, 921, 642, 775, 366, 173,
    880, 954, 692, 761, 345, 701,
    666, 952, 861, 442, 97, 127,
    809, 925, 469, 956, 390, 341,
    773, 924, 697, 812, 891, 92,
    400, 84, 9, 316, 813, 102,
    337, 447, 145, 349, 462, 207,
    662, 899, 322, 381, 837, 687,
    452, 701, 828, 9, 46, 498,
    968, 920, 106, 324, 494, 813,
    // Bin capacity
    1919, 2487, 1604, 2175, 1913, 1655
  };
  const int i6_18_6_20_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    43, 932, 988, 781, 476, 244,
    167, 730, 371, 41, 823, 813,
    205, 260, 527, 836, 556, 25,
    395, 756, 218, 98, 756, 258,
    843, 969, 626, 271, 987, 181,
    245, 849, 666, 346, 211, 430,
    834, 818, 789, 154, 437, 83,
    226, 485, 163, 572, 899, 511,
    909, 405, 405, 34, 130, 994,
    497, 651, 523, 357, 664, 815,
    157, 938, 263, 981, 554, 504,
    64, 816, 714, 788, 239, 410,
    563, 746, 233, 239, 812, 229,
    219, 865, 168, 767, 281, 364,
    131, 428, 373, 860, 612, 628,
    623, 188, 344, 753, 205, 112,
    360, 476, 373, 178, 222, 960,
    157, 863, 61, 522, 200, 531,
    // Bin capacity
    1328, 2435, 1561, 1716, 1813, 1619
  };
  const int i6_18_6_21_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    885, 243, 737, 201, 971, 461,
    412, 329, 133, 749, 631, 884,
    573, 40, 208, 864, 234, 898,
    280, 242, 236, 730, 287, 554,
    680, 244, 592, 622, 512, 582,
    936, 673, 46, 569, 788, 843,
    66, 586, 854, 640, 994, 402,
    452, 277, 48, 546, 691, 802,
    941, 653, 482, 469, 258, 49,
    12, 290, 135, 677, 984, 233,
    263, 43, 219, 358, 196, 311,
    147, 864, 874, 760, 22, 124,
    88, 769, 105, 769, 240, 851,
    1, 964, 898, 25, 279, 577,
    774, 290, 91, 449, 812, 30,
    175, 335, 712, 800, 646, 202,
    935, 896, 889, 506, 914, 131,
    183, 797, 635, 269, 369, 954,
    // Bin capacity
    1574, 1722, 1592, 2018, 1982, 1793
  };
  const int i6_18_6_22_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    798, 187, 829, 281, 615, 35,
    662, 49, 841, 746, 699, 85,
    312, 179, 108, 634, 764, 251,
    229, 942, 280, 137, 560, 726,
    864, 913, 168, 872, 109, 919,
    212, 278, 387, 752, 952, 419,
    302, 156, 25, 489, 439, 932,
    228, 553, 471, 342, 896, 620,
    480, 113, 528, 603, 374, 231,
    692, 855, 60, 22, 605, 200,
    327, 984, 201, 638, 853, 924,
    816, 660, 909, 228, 6, 316,
    902, 609, 385, 825, 974, 355,
    888, 952, 180, 685, 885, 892,
    154, 258, 511, 386, 672, 458,
    905, 844, 426, 844, 115, 608,
    211, 410, 34, 932, 129, 365,
    669, 422, 2, 642, 440, 871,
    // Bin capacity
    1963, 1905, 1291, 2046, 2052, 1873
  };
  const int i6_18_6_23_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    568, 239, 460, 970, 76, 310,
    11, 615, 290, 812, 586, 50,
    992, 26, 715, 752, 188, 900,
    731, 15, 471, 630, 379, 900,
    146, 361, 944, 640, 604, 516,
    574, 746, 106, 921, 838, 898,
    945, 617, 631, 980, 94, 172,
    383, 362, 126, 357, 529, 610,
    429, 959, 572, 327, 686, 144,
    958, 999, 360, 401, 925, 608,
    733, 947, 458, 990, 631, 85,
    815, 80, 587, 984, 806, 49,
    422, 914, 610, 812, 805, 767,
    604, 71, 78, 173, 842, 927,
    323, 989, 21, 667, 980, 531,
    394, 853, 760, 878, 154, 681,
    462, 211, 447, 533, 624, 617,
    298, 914, 705, 663, 819, 784,
    // Bin capacity
    2007, 2034, 1710, 2561, 2167, 1958
  };
  const int i6_18_6_24_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    167, 865, 845, 263, 372, 280,
    838, 410, 234, 134, 408, 749,
    846, 480, 190, 945, 449, 8,
    495, 447, 462, 691, 912, 431,
    331, 31, 475, 827, 583, 963,
    4, 286, 558, 335, 709, 576,
    341, 0, 602, 112, 869, 232,
    1, 286, 245, 98, 457, 454,
    391, 699, 843, 632, 686, 292,
    808, 868, 277, 541, 596, 840,
    934, 96, 134, 472, 668, 89,
    987, 264, 683, 17, 423, 945,
    188, 835, 265, 827, 408, 12,
    994, 327, 985, 113, 227, 512,
    226, 730, 574, 935, 104, 613,
    28, 223, 323, 607, 694, 356,
    822, 451, 117, 3, 562, 297,
    155, 619, 791, 61, 604, 86,
    // Bin capacity
    1769, 1637, 1778, 1574, 2012, 1599
  };
  const int i6_18_6_25_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    330, 664, 67, 975, 267, 325,
    163, 258, 896, 232, 609, 828,
    721, 226, 30, 681, 539, 277,
    900, 898, 920, 150, 496, 350,
    61, 595, 624, 80, 972, 39,
    747, 77, 53, 418, 9, 115,
    398, 137, 567, 584, 600, 96,
    885, 337, 415, 386, 378, 634,
    768, 906, 145, 521, 616, 160,
    478, 645, 886, 790, 767, 146,
    1000, 680, 540, 192, 940, 277,
    887, 170, 177, 850, 550, 796,
    285, 934, 10, 491, 729, 129,
    668, 27, 51, 695, 703, 46,
    537, 580, 526, 730, 675, 503,
    428, 913, 833, 351, 620, 805,
    156, 694, 27, 196, 652, 255,
    386, 383, 743, 328, 121, 460,
    // Bin capacity
    2042, 1901, 1565, 1803, 2134, 1301
  };
  const int i6_18_6_26_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    166, 618, 777, 395, 954, 50,
    822, 208, 329, 972, 766, 522,
    612, 582, 0, 66, 903, 134,
    241, 257, 249, 860, 376, 660,
    877, 36, 180, 579, 125, 111,
    417, 658, 271, 896, 26, 584,
    878, 498, 903, 108, 940, 101,
    138, 258, 103, 1000, 148, 564,
    424, 710, 799, 443, 963, 886,
    815, 147, 553, 659, 624, 543,
    741, 397, 625, 16, 976, 433,
    562, 771, 348, 91, 935, 488,
    691, 17, 630, 168, 841, 613,
    670, 223, 234, 790, 802, 754,
    29, 649, 856, 60, 767, 874,
    845, 411, 792, 722, 540, 695,
    651, 845, 639, 54, 735, 856,
    79, 560, 560, 742, 832, 32,
    // Bin capacity
    2029, 1648, 1859, 1811, 2574, 1869
  };
  const int i6_18_6_27_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    116, 103, 250, 373, 794, 603,
    788, 271, 340, 193, 200, 152,
    398, 584, 92, 411, 811, 250,
    762, 483, 140, 558, 375, 601,
    650, 467, 858, 127, 769, 975,
    363, 65, 490, 400, 921, 658,
    433, 561, 233, 217, 947, 266,
    391, 816, 177, 714, 701, 405,
    503, 349, 405, 607, 547, 914,
    395, 450, 882, 933, 59, 482,
    1000, 180, 940, 825, 876, 418,
    901, 878, 235, 390, 237, 609,
    64, 544, 946, 883, 173, 145,
    693, 958, 197, 163, 739, 267,
    928, 404, 787, 970, 859, 678,
    443, 992, 329, 416, 996, 265,
    466, 169, 886, 745, 295, 470,
    33, 585, 240, 956, 614, 724,
    // Bin capacity
    1975, 1876, 1784, 2092, 2310, 1881
  };
  const int i6_18_6_28_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    368, 597, 492, 635, 968, 201,
    869, 997, 494, 955, 893, 919,
    820, 306, 520, 615, 171, 946,
    911, 975, 824, 136, 621, 774,
    421, 332, 195, 498, 288, 223,
    926, 94, 359, 163, 890, 368,
    567, 302, 922, 14, 946, 349,
    556, 421, 996, 452, 844, 442,
    493, 168, 476, 692, 833, 357,
    915, 479, 427, 262, 623, 321,
    158, 277, 570, 27, 20, 548,
    826, 31, 503, 243, 224, 474,
    338, 866, 68, 306, 149, 194,
    387, 367, 172, 354, 575, 314,
    483, 351, 88, 833, 56, 787,
    448, 649, 160, 798, 416, 14,
    172, 956, 150, 53, 274, 381,
    139, 189, 421, 165, 120, 990,
    // Bin capacity
    2091, 1783, 1672, 1537, 1902, 1836
  };
  const int i6_18_6_29_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    232, 296, 366, 360, 639, 99,
    471, 708, 247, 88, 87, 226,
    15, 563, 639, 496, 179, 862,
    794, 105, 674, 53, 397, 225,
    571, 807, 925, 629, 291, 253,
    375, 638, 660, 481, 798, 980,
    806, 215, 265, 743, 828, 965,
    832, 995, 231, 198, 131, 575,
    332, 327, 943, 175, 96, 993,
    756, 34, 485, 698, 408, 794,
    8, 918, 337, 374, 520, 454,
    357, 187, 528, 812, 171, 199,
    32, 888, 443, 449, 367, 951,
    702, 112, 477, 467, 537, 962,
    8, 498, 415, 199, 739, 838,
    577, 938, 266, 867, 21, 489,
    129, 196, 749, 336, 580, 607,
    777, 611, 10, 166, 162, 764,
    // Bin capacity
    1672, 1943, 1862, 1633, 1495, 2416
  };
  const int i6_18_6_30_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    912, 64, 883, 969, 272, 841,
    130, 79, 633, 289, 900, 183,
    65, 727, 295, 618, 438, 107,
    309, 575, 850, 124, 228, 106,
    200, 538, 983, 382, 567, 840,
    304, 396, 776, 531, 67, 984,
    472, 366, 654, 658, 654, 327,
    156, 408, 224, 556, 820, 54,
    15, 570, 21, 832, 752, 486,
    67, 285, 257, 591, 572, 695,
    519, 640, 452, 78, 48, 708,
    537, 441, 65, 850, 752, 124,
    410, 493, 769, 15, 759, 646,
    278, 482, 138, 471, 771, 554,
    848, 377, 388, 576, 616, 247,
    292, 879, 532, 124, 683, 562,
    745, 886, 366, 921, 895, 390,
    369, 228, 88, 1, 881, 404,
    // Bin capacity
    1437, 1828, 1815, 1861, 2313, 1790
  };
  const int i6_18_6_31_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    117, 23, 891, 481, 768, 161,
    726, 416, 148, 180, 293, 321,
    0, 376, 546, 223, 248, 984,
    663, 877, 161, 576, 792, 482,
    782, 710, 92, 846, 832, 312,
    58, 43, 195, 145, 245, 755,
    501, 179, 801, 577, 775, 2,
    999, 657, 937, 11, 596, 736,
    878, 786, 465, 409, 214, 922,
    415, 437, 968, 361, 90, 442,
    69, 775, 460, 968, 204, 708,
    175, 288, 487, 564, 637, 95,
    676, 493, 91, 663, 545, 36,
    343, 202, 842, 400, 792, 269,
    546, 362, 916, 210, 764, 38,
    638, 481, 649, 124, 562, 735,
    567, 363, 48, 280, 548, 347,
    570, 240, 147, 732, 864, 136,
    // Bin capacity
    1905, 1683, 1931, 1693, 2133, 1634
  };
  const int i6_18_6_32_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    982, 578, 152, 63, 209, 523,
    637, 131, 222, 345, 719, 99,
    407, 993, 718, 592, 662, 232,
    50, 227, 714, 325, 329, 905,
    793, 35, 955, 761, 537, 788,
    729, 675, 669, 198, 415, 857,
    450, 112, 741, 244, 992, 771,
    132, 908, 786, 854, 200, 330,
    461, 473, 170, 499, 375, 543,
    196, 649, 441, 203, 969, 818,
    56, 913, 848, 274, 893, 470,
    325, 514, 510, 119, 135, 343,
    452, 689, 718, 922, 765, 752,
    603, 426, 482, 6, 803, 355,
    936, 327, 978, 878, 681, 253,
    165, 113, 657, 721, 564, 530,
    139, 774, 490, 947, 67, 100,
    21, 847, 601, 899, 820, 355,
    // Bin capacity
    1658, 2065, 2388, 1947, 2230, 1986
  };
  const int i6_18_6_33_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    2, 546, 150, 725, 523, 346,
    285, 603, 598, 405, 511, 154,
    538, 707, 892, 320, 243, 390,
    559, 498, 957, 835, 344, 911,
    422, 559, 214, 561, 447, 583,
    605, 934, 737, 864, 22, 475,
    784, 35, 532, 746, 941, 859,
    18, 661, 661, 364, 285, 912,
    839, 646, 647, 136, 65, 605,
    41, 922, 326, 893, 96, 550,
    175, 790, 87, 26, 15, 297,
    468, 113, 446, 440, 756, 555,
    631, 205, 888, 103, 418, 345,
    631, 519, 318, 162, 963, 783,
    351, 741, 679, 218, 644, 779,
    702, 68, 860, 966, 385, 278,
    442, 342, 134, 349, 711, 248,
    181, 72, 445, 117, 443, 682,
    // Bin capacity
    1702, 1987, 2122, 1825, 1732, 2162
  };
  const int i6_18_6_34_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    541, 396, 976, 451, 338, 388,
    842, 642, 249, 170, 72, 45,
    921, 265, 689, 466, 998, 601,
    741, 845, 846, 315, 931, 914,
    959, 507, 13, 461, 766, 860,
    531, 883, 906, 425, 271, 358,
    440, 505, 939, 267, 169, 477,
    492, 661, 520, 207, 117, 81,
    63, 154, 485, 618, 318, 298,
    275, 971, 744, 311, 774, 250,
    481, 575, 746, 711, 497, 373,
    666, 227, 725, 865, 111, 126,
    291, 890, 506, 628, 704, 14,
    409, 542, 137, 919, 555, 542,
    860, 371, 9, 489, 634, 940,
    678, 160, 270, 201, 486, 111,
    361, 523, 369, 670, 469, 606,
    430, 360, 582, 703, 124, 88,
    // Bin capacity
    2230, 2117, 2169, 1983, 1862, 1580
  };
  const int i6_18_6_35_9[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    235, 490, 852, 401, 89, 354,
    359, 683, 531, 114, 779, 461,
    221, 459, 447, 854, 913, 409,
    535, 390, 575, 401, 676, 592,
    933, 603, 617, 164, 933, 556,
    799, 758, 402, 946, 957, 514,
    448, 109, 807, 989, 185, 526,
    477, 487, 679, 689, 237, 82,
    814, 155, 378, 102, 688, 39,
    431, 562, 321, 970, 29, 864,
    698, 327, 921, 238, 202, 596,
    887, 714, 949, 419, 490, 912,
    135, 758, 240, 729, 132, 758,
    814, 546, 613, 194, 14, 593,
    390, 796, 346, 48, 887, 153,
    768, 941, 685, 627, 276, 723,
    823, 818, 617, 35, 724, 728,
    804, 404, 541, 661, 714, 9,
    // Bin capacity
    2379, 2250, 2368, 1931, 2009, 1996
  };
  const int i6_18_6_0_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    717, 83, 544, 742, 597, 500,
    502, 515, 403, 290, 503, 538,
    961, 551, 625, 789, 573, 886,
    310, 165, 304, 690, 129, 720,
    831, 925, 805, 965, 412, 663,
    408, 222, 77, 111, 338, 120,
    416, 832, 988, 493, 123, 379,
    221, 822, 193, 592, 704, 431,
    848, 567, 31, 218, 238, 205,
    984, 537, 68, 686, 75, 881,
    963, 103, 831, 847, 391, 437,
    335, 285, 849, 436, 602, 395,
    832, 246, 93, 601, 113, 123,
    934, 375, 787, 354, 515, 281,
    741, 640, 180, 536, 737, 976,
    891, 0, 874, 975, 781, 681,
    328, 62, 688, 184, 428, 412,
    56, 797, 435, 779, 114, 711,
    // Bin capacity
    1880, 1288, 1463, 1715, 1229, 1557
  };
  const int i6_18_6_1_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    372, 605, 855, 278, 381, 570,
    699, 535, 937, 220, 416, 81,
    629, 812, 514, 519, 855, 638,
    858, 265, 916, 633, 62, 655,
    511, 470, 236, 913, 849, 176,
    636, 606, 853, 174, 406, 910,
    169, 572, 293, 478, 81, 753,
    311, 552, 150, 134, 314, 193,
    122, 906, 891, 392, 473, 560,
    101, 277, 307, 421, 490, 711,
    486, 601, 675, 455, 396, 283,
    113, 357, 88, 923, 3, 215,
    740, 69, 730, 51, 287, 731,
    187, 553, 945, 308, 823, 950,
    735, 186, 638, 133, 968, 323,
    550, 251, 400, 870, 601, 971,
    230, 665, 203, 755, 541, 230,
    27, 899, 459, 360, 687, 390,
    // Bin capacity
    1259, 1546, 1699, 1350, 1454, 1573
  };
  const int i6_18_6_2_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    45, 191, 747, 955, 140, 251,
    60, 743, 490, 145, 363, 889,
    452, 243, 236, 981, 438, 100,
    537, 333, 783, 3, 920, 701,
    293, 531, 319, 709, 581, 636,
    385, 788, 558, 324, 217, 903,
    811, 648, 251, 581, 859, 530,
    217, 624, 549, 228, 758, 947,
    331, 241, 719, 846, 293, 764,
    990, 203, 687, 228, 757, 843,
    389, 70, 835, 362, 386, 525,
    761, 596, 221, 750, 20, 141,
    51, 243, 568, 388, 997, 470,
    878, 426, 599, 546, 129, 694,
    375, 107, 736, 814, 425, 932,
    869, 336, 208, 832, 982, 822,
    7, 977, 930, 785, 496, 85,
    844, 501, 492, 351, 376, 165,
    // Bin capacity
    1411, 1327, 1688, 1671, 1554, 1768
  };
  const int i6_18_6_3_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    719, 948, 72, 298, 7, 185,
    330, 366, 188, 736, 52, 7,
    260, 936, 668, 792, 242, 873,
    856, 627, 131, 511, 664, 74,
    211, 850, 629, 676, 214, 273,
    736, 38, 907, 114, 520, 35,
    327, 112, 369, 984, 387, 982,
    440, 238, 261, 130, 602, 941,
    641, 425, 694, 57, 413, 399,
    859, 846, 44, 12, 179, 957,
    670, 877, 35, 958, 1000, 820,
    347, 710, 157, 579, 512, 185,
    505, 772, 935, 322, 390, 721,
    314, 846, 557, 346, 808, 189,
    208, 466, 709, 426, 77, 436,
    333, 597, 164, 313, 358, 657,
    549, 856, 835, 173, 501, 111,
    269, 857, 477, 614, 120, 661,
    // Bin capacity
    1472, 1952, 1345, 1381, 1210, 1461
  };
  const int i6_18_6_4_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    320, 303, 692, 583, 983, 104,
    974, 666, 205, 95, 914, 315,
    319, 231, 859, 390, 665, 697,
    762, 976, 366, 691, 347, 813,
    567, 346, 179, 467, 344, 250,
    986, 846, 103, 213, 959, 548,
    34, 228, 997, 590, 284, 952,
    915, 175, 913, 255, 464, 47,
    541, 527, 154, 880, 145, 405,
    517, 263, 610, 468, 925, 890,
    657, 435, 476, 824, 553, 483,
    82, 552, 384, 250, 129, 552,
    696, 722, 46, 218, 831, 264,
    127, 533, 640, 280, 748, 280,
    514, 511, 856, 58, 156, 310,
    375, 134, 121, 871, 738, 477,
    288, 76, 732, 976, 610, 611,
    18, 489, 961, 734, 194, 288,
    // Bin capacity
    1507, 1389, 1611, 1533, 1732, 1437
  };
  const int i6_18_6_5_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    677, 291, 917, 422, 621, 204,
    181, 560, 416, 926, 963, 925,
    749, 871, 122, 160, 161, 965,
    584, 404, 576, 138, 509, 982,
    684, 242, 800, 158, 67, 67,
    816, 432, 271, 262, 274, 763,
    830, 674, 656, 166, 238, 131,
    730, 220, 877, 698, 3, 499,
    899, 613, 425, 353, 472, 743,
    705, 444, 843, 642, 413, 145,
    327, 190, 504, 76, 998, 776,
    921, 928, 690, 365, 59, 462,
    656, 537, 943, 459, 495, 545,
    743, 151, 664, 785, 939, 637,
    10, 296, 964, 795, 565, 151,
    123, 859, 980, 960, 853, 343,
    248, 538, 694, 461, 190, 908,
    42, 31, 435, 888, 746, 806,
    // Bin capacity
    1737, 1450, 2061, 1525, 1500, 1760
  };
  const int i6_18_6_6_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    453, 492, 234, 433, 315, 927,
    988, 284, 169, 855, 773, 909,
    30, 506, 65, 47, 191, 973,
    631, 467, 233, 586, 684, 106,
    550, 943, 592, 519, 986, 934,
    6, 58, 284, 117, 842, 576,
    721, 433, 595, 282, 613, 564,
    457, 297, 449, 992, 613, 358,
    550, 820, 867, 743, 628, 286,
    162, 665, 841, 641, 261, 704,
    11, 683, 897, 96, 300, 633,
    701, 18, 232, 85, 308, 406,
    553, 546, 158, 675, 511, 802,
    770, 696, 25, 164, 604, 151,
    575, 330, 648, 995, 796, 833,
    961, 327, 179, 669, 831, 555,
    914, 75, 165, 664, 384, 812,
    146, 640, 276, 334, 174, 28,
    // Bin capacity
    1622, 1463, 1221, 1572, 1734, 1866
  };
  const int i6_18_6_7_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    294, 481, 720, 849, 830, 137,
    753, 297, 202, 72, 91, 87,
    95, 627, 6, 161, 983, 118,
    884, 84, 634, 698, 476, 370,
    985, 864, 642, 123, 210, 338,
    84, 10, 94, 725, 397, 795,
    818, 290, 702, 364, 172, 26,
    463, 385, 277, 922, 972, 215,
    863, 619, 748, 499, 649, 810,
    289, 75, 589, 339, 895, 407,
    642, 606, 921, 165, 529, 745,
    731, 622, 985, 18, 278, 390,
    470, 332, 947, 435, 376, 494,
    590, 530, 363, 35, 455, 769,
    135, 987, 998, 814, 664, 227,
    372, 683, 942, 665, 470, 420,
    629, 251, 512, 836, 744, 641,
    375, 917, 650, 170, 77, 890,
    // Bin capacity
    1690, 1545, 1950, 1408, 1653, 1406
  };
  const int i6_18_6_8_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    800, 218, 859, 156, 626, 742,
    598, 364, 540, 365, 588, 39,
    562, 739, 930, 394, 567, 681,
    680, 717, 673, 586, 26, 972,
    554, 720, 268, 853, 758, 142,
    804, 932, 820, 86, 238, 150,
    504, 385, 662, 816, 761, 700,
    877, 615, 848, 640, 783, 98,
    941, 877, 127, 292, 945, 355,
    206, 947, 909, 526, 706, 781,
    39, 914, 585, 219, 292, 191,
    523, 321, 781, 510, 983, 222,
    488, 719, 167, 796, 83, 528,
    415, 11, 971, 554, 188, 605,
    86, 356, 576, 727, 3, 222,
    520, 298, 250, 116, 655, 65,
    560, 942, 680, 907, 959, 592,
    948, 519, 526, 542, 639, 520,
    // Bin capacity
    1819, 1907, 2011, 1636, 1764, 1369
  };
  const int i6_18_6_9_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    996, 167, 656, 984, 763, 87,
    864, 513, 68, 298, 477, 129,
    189, 225, 59, 704, 763, 309,
    304, 694, 106, 674, 265, 243,
    278, 496, 849, 641, 958, 214,
    505, 897, 525, 837, 282, 547,
    894, 824, 728, 427, 15, 782,
    858, 209, 598, 855, 147, 280,
    599, 424, 71, 434, 619, 621,
    268, 22, 839, 885, 876, 690,
    163, 207, 277, 508, 265, 701,
    423, 949, 199, 118, 865, 227,
    688, 809, 866, 858, 663, 207,
    307, 226, 160, 394, 144, 256,
    349, 808, 451, 449, 559, 280,
    815, 635, 892, 354, 325, 686,
    516, 498, 556, 542, 505, 95,
    455, 557, 79, 601, 134, 484,
    // Bin capacity
    1721, 1665, 1450, 1919, 1567, 1243
  };
  const int i6_18_6_10_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    841, 439, 617, 391, 853, 149,
    202, 366, 148, 636, 646, 931,
    426, 25, 563, 115, 550, 423,
    308, 876, 391, 643, 940, 439,
    314, 365, 400, 398, 883, 403,
    306, 4, 778, 643, 675, 209,
    899, 55, 196, 916, 249, 207,
    351, 996, 80, 126, 181, 61,
    989, 867, 78, 141, 793, 747,
    854, 673, 800, 732, 348, 760,
    330, 327, 322, 995, 442, 263,
    938, 396, 806, 171, 695, 193,
    22, 50, 68, 40, 2, 239,
    804, 398, 874, 405, 65, 960,
    169, 675, 817, 670, 621, 862,
    296, 990, 488, 456, 549, 214,
    970, 493, 962, 951, 475, 729,
    538, 548, 425, 699, 128, 503,
    // Bin capacity
    1753, 1567, 1616, 1674, 1668, 1521
  };
  const int i6_18_6_11_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    641, 160, 389, 429, 715, 912,
    352, 769, 814, 556, 502, 890,
    547, 166, 417, 170, 683, 766,
    312, 95, 365, 292, 951, 519,
    371, 487, 485, 230, 258, 749,
    503, 778, 662, 686, 310, 657,
    717, 299, 6, 115, 488, 865,
    532, 636, 383, 947, 896, 376,
    1000, 82, 256, 907, 26, 125,
    883, 638, 794, 524, 697, 674,
    599, 770, 970, 613, 789, 374,
    587, 323, 364, 425, 591, 711,
    807, 958, 57, 609, 650, 207,
    191, 872, 799, 106, 871, 694,
    967, 276, 55, 100, 802, 907,
    91, 676, 1, 305, 725, 690,
    993, 24, 978, 766, 231, 934,
    118, 679, 764, 711, 156, 57,
    // Bin capacity
    1890, 1608, 1584, 1571, 1914, 2055
  };
  const int i6_18_6_12_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    125, 679, 676, 596, 910, 643,
    602, 849, 37, 435, 298, 568,
    506, 495, 103, 437, 50, 177,
    178, 472, 544, 17, 651, 785,
    202, 533, 717, 506, 251, 45,
    671, 859, 190, 622, 629, 304,
    982, 988, 11, 411, 700, 123,
    645, 725, 242, 45, 842, 490,
    147, 615, 455, 158, 450, 223,
    728, 427, 231, 600, 93, 812,
    137, 32, 934, 144, 497, 465,
    533, 257, 168, 326, 442, 588,
    612, 425, 663, 348, 597, 793,
    484, 212, 370, 276, 108, 309,
    532, 796, 426, 451, 629, 965,
    866, 374, 20, 857, 59, 44,
    228, 699, 866, 165, 355, 836,
    765, 611, 617, 713, 0, 310,
    // Bin capacity
    1670, 1876, 1358, 1327, 1412, 1583
  };
  const int i6_18_6_13_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    586, 995, 841, 552, 873, 982,
    128, 638, 704, 545, 32, 330,
    718, 311, 508, 163, 906, 409,
    996, 901, 622, 477, 532, 744,
    936, 519, 35, 903, 555, 911,
    346, 891, 365, 348, 924, 539,
    795, 42, 594, 544, 722, 271,
    504, 56, 714, 345, 64, 55,
    256, 623, 338, 815, 748, 200,
    896, 694, 80, 125, 201, 510,
    715, 878, 834, 390, 795, 258,
    856, 502, 654, 982, 401, 679,
    827, 6, 832, 742, 937, 430,
    296, 779, 134, 212, 134, 688,
    617, 274, 320, 333, 736, 357,
    831, 454, 647, 414, 298, 938,
    204, 894, 69, 97, 670, 172,
    382, 74, 502, 94, 350, 793,
    // Bin capacity
    2051, 1796, 1657, 1522, 1861, 1746
  };
  const int i6_18_6_14_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    432, 798, 751, 366, 58, 357,
    700, 450, 406, 761, 444, 914,
    821, 205, 583, 343, 547, 831,
    965, 126, 995, 454, 33, 722,
    26, 373, 171, 289, 650, 747,
    711, 895, 227, 630, 550, 192,
    362, 130, 189, 315, 780, 95,
    558, 755, 168, 717, 987, 561,
    231, 854, 369, 199, 220, 615,
    809, 128, 553, 785, 74, 131,
    943, 34, 440, 817, 200, 625,
    617, 117, 571, 755, 221, 689,
    806, 123, 488, 196, 444, 777,
    143, 846, 970, 525, 398, 873,
    33, 732, 110, 206, 708, 298,
    520, 887, 474, 244, 926, 885,
    951, 364, 21, 297, 188, 679,
    204, 889, 904, 626, 902, 198,
    // Bin capacity
    1869, 1655, 1595, 1620, 1583, 1936
  };
  const int i6_18_6_15_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    445, 923, 561, 88, 20, 154,
    336, 783, 829, 505, 52, 202,
    329, 248, 743, 585, 77, 997,
    432, 57, 689, 598, 378, 837,
    592, 217, 910, 264, 152, 253,
    205, 503, 895, 156, 356, 208,
    206, 210, 722, 623, 249, 536,
    568, 977, 954, 547, 193, 944,
    734, 409, 833, 758, 292, 698,
    658, 148, 899, 87, 977, 438,
    69, 645, 759, 402, 112, 842,
    438, 611, 575, 136, 341, 273,
    124, 805, 840, 760, 364, 458,
    292, 434, 559, 6, 586, 711,
    405, 152, 562, 447, 413, 376,
    403, 448, 936, 698, 415, 344,
    650, 332, 860, 619, 224, 809,
    76, 18, 409, 52, 10, 900,
    // Bin capacity
    1335, 1518, 2595, 1406, 999, 1913
  };
  const int i6_18_6_16_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    803, 562, 763, 435, 825, 775,
    849, 290, 593, 957, 14, 306,
    155, 403, 873, 338, 451, 212,
    139, 926, 964, 979, 43, 240,
    952, 964, 941, 646, 528, 967,
    63, 735, 779, 145, 113, 697,
    993, 425, 107, 463, 872, 37,
    519, 526, 836, 391, 772, 636,
    14, 483, 350, 26, 670, 69,
    318, 786, 895, 912, 663, 55,
    44, 78, 302, 121, 634, 155,
    969, 718, 542, 446, 777, 932,
    127, 738, 686, 634, 456, 820,
    809, 628, 945, 307, 74, 598,
    113, 523, 585, 371, 1, 82,
    485, 484, 350, 321, 557, 599,
    581, 519, 785, 337, 105, 831,
    9, 868, 321, 671, 641, 86,
    // Bin capacity
    1536, 2061, 2246, 1644, 1585, 1566
  };
  const int i6_18_6_17_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    720, 17, 822, 672, 680, 307,
    48, 776, 908, 949, 836, 788,
    100, 900, 686, 111, 628, 748,
    206, 506, 52, 141, 501, 66,
    270, 490, 695, 305, 117, 840,
    224, 584, 869, 989, 574, 134,
    339, 671, 935, 415, 74, 152,
    947, 855, 910, 670, 905, 918,
    477, 925, 7, 869, 447, 172,
    363, 632, 955, 561, 739, 963,
    600, 248, 635, 353, 742, 10,
    657, 619, 76, 259, 312, 707,
    830, 497, 419, 339, 119, 136,
    875, 532, 419, 697, 327, 920,
    799, 476, 751, 798, 640, 978,
    655, 229, 676, 89, 309, 307,
    940, 137, 122, 213, 44, 55,
    973, 914, 957, 35, 255, 109,
    // Bin capacity
    1955, 1952, 2125, 1651, 1609, 1621
  };
  const int i6_18_6_18_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    88, 837, 207, 242, 255, 208,
    5, 991, 340, 725, 139, 880,
    621, 508, 741, 769, 920, 898,
    944, 589, 170, 328, 835, 993,
    189, 665, 234, 302, 384, 337,
    735, 504, 562, 428, 358, 702,
    169, 365, 310, 56, 973, 298,
    997, 175, 327, 953, 617, 530,
    696, 635, 544, 500, 681, 547,
    776, 617, 804, 363, 634, 881,
    844, 794, 100, 525, 221, 211,
    633, 594, 986, 504, 884, 771,
    110, 947, 125, 493, 941, 548,
    8, 922, 265, 727, 23, 241,
    147, 915, 772, 314, 234, 262,
    447, 104, 382, 967, 921, 777,
    996, 761, 881, 173, 129, 648,
    892, 707, 433, 521, 958, 766,
    // Bin capacity
    1829, 2288, 1610, 1749, 1988, 2065
  };
  const int i6_18_6_19_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    574, 458, 839, 842, 775, 944,
    838, 658, 878, 535, 360, 910,
    57, 636, 662, 746, 762, 803,
    682, 53, 279, 450, 681, 343,
    996, 1, 165, 67, 467, 330,
    629, 535, 274, 558, 968, 315,
    950, 534, 787, 530, 595, 399,
    2, 794, 923, 301, 406, 614,
    592, 365, 253, 344, 708, 409,
    932, 970, 710, 863, 977, 429,
    773, 78, 333, 701, 267, 468,
    959, 146, 730, 850, 318, 710,
    376, 227, 115, 889, 394, 535,
    661, 156, 892, 35, 897, 927,
    581, 770, 736, 626, 174, 423,
    410, 44, 140, 639, 339, 28,
    296, 800, 893, 369, 97, 529,
    279, 390, 816, 977, 459, 839,
    // Bin capacity
    2100, 1511, 2068, 2048, 1913, 1975
  };
  const int i6_18_6_20_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    336, 918, 321, 230, 359, 233,
    756, 358, 621, 949, 372, 552,
    56, 837, 438, 626, 702, 627,
    748, 155, 92, 700, 950, 333,
    181, 978, 338, 977, 428, 687,
    942, 884, 961, 159, 791, 125,
    418, 509, 568, 440, 6, 201,
    530, 215, 90, 498, 635, 970,
    509, 948, 82, 218, 35, 110,
    204, 641, 45, 744, 759, 255,
    301, 143, 969, 34, 877, 683,
    902, 305, 150, 403, 415, 289,
    397, 202, 563, 665, 887, 661,
    491, 271, 473, 506, 505, 230,
    467, 696, 728, 114, 355, 507,
    847, 863, 898, 950, 169, 76,
    934, 817, 99, 903, 951, 770,
    54, 725, 393, 979, 268, 153,
    // Bin capacity
    1815, 2093, 1566, 2019, 1893, 1493
  };
  const int i6_18_6_21_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    585, 174, 444, 52, 137, 144,
    279, 348, 709, 51, 217, 150,
    314, 650, 285, 819, 418, 298,
    750, 276, 53, 287, 922, 610,
    165, 895, 452, 357, 777, 242,
    188, 654, 489, 62, 505, 236,
    683, 535, 703, 213, 827, 557,
    475, 483, 114, 491, 59, 604,
    927, 495, 430, 544, 880, 147,
    756, 36, 982, 166, 947, 548,
    794, 632, 502, 302, 76, 277,
    105, 839, 642, 213, 619, 735,
    205, 214, 665, 99, 69, 403,
    68, 297, 458, 413, 231, 683,
    689, 668, 639, 573, 123, 696,
    426, 814, 708, 583, 664, 285,
    181, 118, 229, 708, 818, 551,
    20, 66, 462, 322, 844, 963,
    // Bin capacity
    1535, 1653, 1809, 1262, 1842, 1640
  };
  const int i6_18_6_22_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    605, 823, 507, 322, 15, 331,
    628, 193, 862, 974, 502, 458,
    578, 834, 923, 364, 373, 548,
    365, 291, 449, 506, 453, 730,
    328, 762, 622, 760, 324, 600,
    196, 450, 498, 735, 708, 259,
    917, 244, 346, 946, 54, 668,
    18, 558, 570, 288, 174, 212,
    258, 504, 924, 486, 561, 433,
    545, 857, 568, 341, 76, 993,
    808, 955, 931, 80, 313, 222,
    228, 718, 130, 942, 110, 675,
    72, 840, 869, 167, 915, 358,
    636, 745, 905, 558, 793, 642,
    999, 969, 51, 495, 835, 854,
    776, 398, 419, 425, 179, 252,
    551, 49, 204, 230, 408, 20,
    554, 372, 53, 484, 732, 427,
    // Bin capacity
    1843, 2148, 1999, 1851, 1531, 1766
  };
  const int i6_18_6_23_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    305, 670, 506, 237, 763, 734,
    169, 59, 931, 449, 619, 806,
    304, 190, 179, 978, 676, 528,
    197, 137, 38, 141, 968, 801,
    920, 927, 445, 838, 940, 318,
    139, 832, 736, 445, 192, 9,
    619, 172, 769, 888, 317, 353,
    467, 37, 366, 123, 861, 123,
    81, 697, 370, 941, 707, 569,
    747, 519, 517, 26, 964, 755,
    767, 259, 845, 263, 434, 440,
    33, 183, 308, 898, 430, 693,
    486, 791, 547, 344, 541, 786,
    148, 56, 463, 555, 28, 730,
    277, 413, 394, 522, 228, 963,
    130, 110, 138, 749, 870, 294,
    854, 953, 33, 580, 50, 689,
    115, 908, 160, 175, 254, 583,
    // Bin capacity
    1386, 1623, 1588, 1877, 2018, 2086
  };
  const int i6_18_6_24_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    818, 188, 335, 901, 720, 622,
    308, 285, 853, 353, 568, 622,
    660, 597, 33, 776, 975, 735,
    569, 788, 796, 519, 270, 702,
    639, 880, 963, 554, 691, 610,
    388, 149, 124, 700, 864, 775,
    291, 165, 64, 303, 125, 345,
    288, 394, 261, 352, 633, 807,
    761, 359, 403, 481, 282, 9,
    717, 13, 695, 106, 798, 615,
    362, 349, 111, 787, 5, 778,
    295, 882, 233, 302, 152, 639,
    97, 695, 366, 823, 877, 277,
    457, 699, 947, 172, 870, 719,
    888, 568, 767, 304, 37, 622,
    844, 628, 314, 515, 617, 115,
    481, 713, 407, 350, 247, 458,
    274, 400, 974, 254, 664, 685,
    // Bin capacity
    1889, 1809, 1787, 1768, 1942, 2095
  };
  const int i6_18_6_25_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    312, 116, 922, 962, 819, 960,
    66, 341, 790, 466, 109, 609,
    547, 624, 821, 972, 928, 280,
    520, 249, 47, 996, 934, 647,
    928, 631, 225, 868, 239, 272,
    23, 509, 918, 747, 980, 150,
    636, 869, 901, 217, 324, 791,
    868, 741, 100, 436, 178, 991,
    981, 485, 64, 754, 760, 326,
    476, 862, 452, 918, 85, 969,
    951, 493, 597, 77, 300, 735,
    55, 425, 374, 295, 193, 889,
    115, 206, 603, 858, 107, 796,
    79, 294, 625, 946, 446, 523,
    564, 872, 897, 417, 300, 665,
    8, 576, 568, 936, 518, 280,
    54, 19, 644, 602, 735, 566,
    381, 840, 777, 458, 920, 154,
    // Bin capacity
    1576, 1907, 2152, 2485, 1849, 2209
  };
  const int i6_18_6_26_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    140, 5, 636, 840, 952, 465,
    160, 159, 216, 823, 961, 587,
    27, 226, 866, 871, 562, 165,
    282, 925, 431, 223, 640, 652,
    603, 251, 898, 586, 782, 168,
    979, 880, 648, 348, 236, 495,
    274, 503, 209, 424, 164, 800,
    947, 200, 180, 140, 85, 666,
    776, 795, 973, 481, 57, 453,
    359, 646, 329, 449, 792, 234,
    837, 709, 992, 992, 37, 211,
    966, 337, 512, 678, 382, 811,
    785, 366, 63, 716, 540, 587,
    890, 615, 246, 65, 149, 881,
    109, 623, 438, 211, 948, 644,
    116, 425, 727, 140, 431, 486,
    847, 631, 709, 63, 563, 210,
    636, 593, 83, 629, 465, 927,
    // Bin capacity
    2044, 1867, 1923, 1823, 1837, 1983
  };
  const int i6_18_6_27_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    369, 434, 669, 617, 607, 33,
    194, 313, 937, 477, 29, 835,
    443, 649, 711, 530, 516, 185,
    754, 353, 493, 20, 163, 356,
    124, 424, 726, 910, 92, 561,
    616, 392, 678, 216, 871, 498,
    719, 54, 704, 924, 936, 880,
    741, 42, 85, 272, 428, 76,
    242, 37, 523, 684, 80, 414,
    163, 54, 636, 802, 165, 969,
    109, 134, 474, 889, 186, 314,
    424, 577, 696, 733, 251, 271,
    484, 913, 903, 849, 349, 322,
    84, 511, 938, 289, 727, 32,
    189, 472, 241, 731, 240, 634,
    321, 507, 533, 910, 204, 982,
    762, 193, 269, 700, 155, 826,
    172, 125, 5, 339, 734, 202,
    // Bin capacity
    1463, 1309, 2164, 2306, 1426, 1776
  };
  const int i6_18_6_28_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    340, 426, 395, 539, 767, 155,
    518, 880, 391, 511, 423, 93,
    494, 429, 176, 359, 30, 384,
    532, 765, 198, 357, 102, 430,
    430, 159, 649, 465, 591, 887,
    522, 153, 476, 741, 771, 364,
    766, 416, 336, 992, 273, 670,
    912, 454, 225, 323, 754, 958,
    725, 795, 414, 893, 798, 223,
    484, 356, 839, 325, 173, 426,
    923, 570, 47, 220, 926, 213,
    630, 342, 793, 45, 777, 433,
    888, 834, 239, 331, 864, 590,
    869, 283, 338, 504, 50, 682,
    767, 360, 776, 713, 929, 496,
    156, 782, 846, 268, 73, 646,
    40, 563, 488, 852, 680, 660,
    273, 14, 833, 461, 769, 112,
    // Bin capacity
    2191, 1831, 1805, 1899, 2080, 1797
  };
  const int i6_18_6_29_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    458, 845, 74, 583, 653, 227,
    813, 74, 801, 498, 1000, 44,
    154, 151, 850, 340, 947, 279,
    336, 316, 400, 499, 641, 360,
    72, 809, 766, 963, 879, 667,
    333, 533, 258, 849, 213, 138,
    952, 732, 715, 806, 562, 286,
    44, 649, 740, 310, 511, 420,
    857, 717, 164, 976, 336, 619,
    956, 363, 841, 228, 312, 679,
    131, 535, 370, 345, 770, 592,
    620, 581, 763, 14, 832, 597,
    595, 2, 895, 289, 916, 79,
    996, 385, 884, 694, 644, 378,
    955, 46, 832, 105, 523, 803,
    29, 121, 782, 76, 612, 380,
    441, 256, 35, 240, 398, 772,
    208, 547, 735, 724, 79, 136,
    // Bin capacity
    1925, 1648, 2345, 1836, 2329, 1604
  };
  const int i6_18_6_30_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    803, 404, 693, 812, 915, 372,
    40, 69, 704, 8, 629, 17,
    712, 39, 8, 659, 21, 617,
    444, 786, 783, 871, 826, 38,
    635, 756, 319, 846, 757, 878,
    501, 548, 872, 758, 319, 525,
    638, 920, 179, 938, 288, 696,
    42, 342, 819, 347, 770, 370,
    691, 624, 813, 430, 725, 918,
    286, 21, 509, 785, 661, 242,
    172, 877, 630, 973, 345, 523,
    894, 826, 764, 780, 933, 37,
    431, 308, 43, 896, 98, 14,
    839, 668, 384, 400, 0, 514,
    410, 389, 627, 716, 297, 898,
    460, 34, 527, 79, 849, 25,
    680, 635, 334, 138, 522, 969,
    646, 433, 217, 290, 541, 995,
    // Bin capacity
    2021, 1881, 1999, 2324, 2058, 1874
  };
  const int i6_18_6_31_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    237, 126, 898, 580, 336, 27,
    544, 340, 610, 700, 590, 321,
    316, 259, 243, 797, 234, 16,
    176, 173, 160, 321, 247, 869,
    482, 834, 405, 902, 637, 914,
    425, 962, 192, 709, 616, 784,
    5, 465, 58, 318, 481, 525,
    489, 605, 240, 346, 467, 571,
    237, 669, 43, 860, 812, 895,
    892, 325, 876, 992, 305, 357,
    888, 166, 315, 348, 168, 612,
    158, 612, 59, 974, 958, 194,
    59, 717, 311, 385, 873, 229,
    860, 788, 279, 830, 841, 13,
    510, 736, 542, 590, 484, 236,
    574, 515, 114, 923, 438, 526,
    596, 749, 891, 931, 423, 645,
    444, 135, 861, 924, 900, 230,
    // Bin capacity
    1724, 2004, 1550, 2714, 2142, 1739
  };
  const int i6_18_6_32_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    236, 677, 620, 331, 593, 988,
    827, 328, 501, 6, 423, 731,
    470, 710, 644, 516, 402, 550,
    189, 617, 189, 652, 923, 786,
    429, 966, 277, 493, 873, 626,
    971, 645, 205, 769, 87, 542,
    277, 693, 287, 757, 192, 149,
    663, 989, 264, 452, 907, 429,
    329, 307, 121, 92, 231, 865,
    503, 540, 363, 535, 811, 462,
    17, 829, 359, 354, 579, 937,
    6, 904, 326, 707, 503, 856,
    261, 858, 500, 281, 63, 184,
    554, 729, 415, 256, 536, 86,
    904, 8, 949, 339, 627, 908,
    981, 571, 125, 405, 828, 607,
    351, 497, 684, 614, 636, 539,
    708, 665, 703, 511, 76, 627,
    // Bin capacity
    1909, 2538, 1658, 1776, 2044, 2392
  };
  const int i6_18_6_33_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    489, 758, 846, 302, 994, 708,
    115, 435, 70, 689, 307, 615,
    40, 470, 704, 27, 494, 115,
    859, 671, 724, 719, 88, 535,
    679, 735, 547, 292, 584, 847,
    278, 833, 105, 936, 609, 690,
    690, 17, 196, 815, 344, 352,
    975, 634, 636, 24, 659, 831,
    888, 828, 424, 632, 20, 132,
    949, 55, 4, 691, 285, 867,
    803, 107, 797, 324, 997, 349,
    465, 469, 827, 26, 619, 3,
    144, 851, 250, 287, 49, 99,
    779, 573, 661, 451, 724, 104,
    307, 844, 118, 173, 755, 484,
    182, 287, 611, 261, 1000, 473,
    958, 392, 435, 305, 613, 740,
    164, 416, 543, 109, 95, 935,
    // Bin capacity
    2165, 2079, 1884, 1566, 2048, 1969
  };
  const int i6_18_6_34_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    883, 278, 701, 317, 337, 633,
    741, 495, 356, 816, 80, 718,
    639, 935, 930, 449, 772, 535,
    652, 190, 969, 929, 748, 669,
    601, 25, 622, 550, 467, 326,
    926, 928, 787, 876, 590, 865,
    337, 466, 296, 966, 713, 543,
    438, 858, 55, 638, 482, 5,
    218, 926, 834, 230, 434, 255,
    877, 764, 801, 591, 838, 680,
    376, 410, 110, 644, 122, 930,
    863, 749, 360, 522, 162, 429,
    116, 148, 643, 266, 477, 597,
    787, 243, 479, 153, 677, 879,
    180, 969, 486, 954, 520, 242,
    397, 920, 674, 211, 780, 268,
    707, 644, 171, 35, 468, 671,
    67, 538, 175, 589, 962, 997,
    // Bin capacity
    2190, 2342, 2111, 2175, 2151, 2288
  };
  const int i6_18_6_35_10[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    159, 141, 603, 959, 794, 709,
    250, 428, 545, 715, 778, 980,
    549, 780, 330, 378, 102, 902,
    286, 918, 204, 468, 749, 904,
    693, 666, 386, 647, 777, 737,
    366, 889, 726, 504, 337, 891,
    459, 136, 805, 169, 671, 195,
    682, 944, 544, 747, 882, 797,
    52, 992, 11, 147, 477, 472,
    304, 304, 447, 26, 587, 737,
    338, 655, 893, 825, 552, 372,
    799, 102, 550, 426, 868, 870,
    683, 336, 992, 410, 48, 266,
    512, 223, 442, 667, 889, 931,
    531, 655, 593, 212, 174, 366,
    274, 738, 0, 74, 494, 119,
    258, 868, 190, 187, 377, 269,
    929, 994, 717, 617, 402, 942,
    // Bin capacity
    1828, 2424, 2021, 1841, 2241, 2579
  };
  const int i6_18_6_0_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    699, 33, 513, 671, 115, 450,
    401, 246, 126, 661, 515, 284,
    899, 719, 796, 475, 158, 179,
    139, 758, 193, 896, 102, 769,
    840, 584, 730, 581, 947, 923,
    446, 742, 567, 303, 988, 761,
    4, 298, 52, 450, 694, 904,
    12, 542, 314, 30, 235, 337,
    853, 245, 352, 560, 257, 469,
    683, 685, 167, 947, 24, 95,
    193, 210, 112, 587, 931, 933,
    142, 620, 113, 217, 44, 119,
    982, 628, 62, 222, 114, 595,
    98, 444, 952, 364, 542, 922,
    627, 321, 968, 767, 651, 663,
    940, 146, 470, 8, 148, 574,
    246, 287, 981, 924, 405, 305,
    89, 410, 46, 76, 116, 642,
    // Bin capacity
    1383, 1320, 1253, 1457, 1165, 1654
  };
  const int i6_18_6_1_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    160, 969, 86, 788, 890, 504,
    151, 94, 351, 929, 817, 785,
    832, 413, 132, 196, 75, 343,
    407, 565, 278, 332, 260, 566,
    849, 218, 936, 427, 44, 13,
    95, 921, 53, 592, 404, 63,
    913, 451, 215, 504, 147, 61,
    952, 580, 917, 547, 593, 337,
    718, 359, 896, 598, 617, 582,
    692, 275, 526, 722, 939, 38,
    319, 467, 96, 106, 419, 688,
    158, 761, 237, 400, 912, 890,
    247, 936, 790, 967, 949, 448,
    557, 647, 760, 421, 268, 473,
    847, 526, 379, 212, 196, 342,
    62, 433, 968, 937, 406, 214,
    72, 447, 344, 395, 237, 537,
    669, 782, 16, 429, 54, 894,
    // Bin capacity
    1465, 1658, 1344, 1600, 1385, 1310
  };
  const int i6_18_6_2_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    117, 360, 357, 201, 757, 390,
    225, 209, 11, 440, 218, 297,
    614, 543, 265, 878, 368, 753,
    274, 689, 256, 403, 274, 625,
    287, 597, 835, 517, 195, 245,
    191, 2, 999, 443, 675, 576,
    790, 390, 613, 173, 384, 988,
    759, 549, 275, 639, 571, 399,
    658, 636, 673, 696, 124, 5,
    770, 118, 953, 218, 603, 781,
    606, 967, 841, 342, 221, 305,
    928, 688, 960, 594, 455, 480,
    343, 374, 729, 238, 23, 59,
    919, 734, 558, 185, 745, 726,
    311, 249, 72, 950, 179, 594,
    689, 108, 155, 185, 941, 694,
    10, 102, 383, 268, 881, 675,
    809, 0, 168, 606, 860, 111,
    // Bin capacity
    1581, 1244, 1548, 1356, 1441, 1480
  };
  const int i6_18_6_3_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    545, 727, 172, 257, 572, 57,
    500, 413, 745, 680, 220, 362,
    874, 283, 436, 33, 546, 593,
    850, 73, 137, 987, 892, 127,
    442, 408, 583, 930, 249, 405,
    383, 528, 542, 663, 335, 216,
    887, 670, 504, 258, 1, 875,
    492, 602, 394, 97, 709, 823,
    147, 744, 61, 63, 196, 794,
    905, 793, 303, 491, 177, 620,
    429, 143, 611, 829, 596, 530,
    182, 169, 760, 311, 76, 970,
    215, 550, 712, 209, 158, 509,
    156, 267, 639, 525, 934, 995,
    445, 743, 498, 744, 487, 189,
    805, 79, 388, 681, 466, 636,
    493, 158, 982, 922, 335, 736,
    661, 126, 589, 759, 567, 537,
    // Bin capacity
    1616, 1284, 1555, 1621, 1291, 1713
  };
  const int i6_18_6_4_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    963, 531, 363, 820, 425, 115,
    731, 191, 744, 695, 951, 352,
    977, 992, 793, 17, 39, 344,
    86, 30, 587, 650, 735, 563,
    299, 598, 204, 816, 623, 351,
    163, 848, 362, 802, 223, 822,
    921, 484, 449, 663, 101, 522,
    894, 632, 496, 445, 258, 270,
    793, 210, 616, 734, 53, 37,
    769, 806, 412, 86, 217, 852,
    650, 595, 18, 461, 92, 476,
    240, 406, 169, 212, 389, 741,
    508, 837, 728, 660, 678, 831,
    402, 253, 506, 527, 978, 696,
    521, 846, 716, 712, 933, 748,
    545, 301, 298, 919, 263, 272,
    286, 314, 20, 600, 158, 487,
    778, 608, 481, 673, 707, 401,
    // Bin capacity
    1825, 1644, 1381, 1819, 1356, 1540
  };
  const int i6_18_6_5_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    66, 451, 506, 586, 995, 457,
    996, 640, 320, 79, 456, 784,
    351, 853, 964, 704, 266, 682,
    487, 50, 186, 141, 138, 77,
    922, 849, 229, 452, 654, 468,
    740, 565, 832, 998, 416, 767,
    718, 620, 886, 319, 589, 632,
    109, 723, 534, 216, 572, 954,
    264, 384, 721, 540, 902, 126,
    285, 89, 424, 142, 240, 321,
    163, 914, 928, 525, 458, 51,
    273, 254, 295, 703, 184, 146,
    31, 808, 789, 157, 636, 430,
    718, 111, 408, 321, 249, 612,
    491, 58, 833, 879, 889, 985,
    198, 558, 862, 952, 506, 793,
    564, 398, 503, 327, 299, 805,
    119, 469, 668, 391, 85, 698,
    // Bin capacity
    1312, 1539, 1906, 1476, 1494, 1713
  };
  const int i6_18_6_6_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    104, 195, 969, 757, 791, 394,
    406, 889, 443, 176, 108, 176,
    299, 538, 625, 434, 257, 848,
    988, 190, 644, 468, 970, 92,
    10, 835, 393, 742, 86, 32,
    21, 103, 13, 840, 111, 552,
    488, 281, 662, 590, 737, 268,
    211, 200, 645, 809, 462, 643,
    378, 907, 236, 958, 374, 237,
    947, 964, 441, 500, 990, 883,
    396, 744, 770, 197, 516, 851,
    542, 64, 961, 529, 719, 402,
    198, 665, 335, 660, 671, 568,
    712, 206, 739, 620, 774, 668,
    614, 15, 549, 191, 404, 595,
    655, 668, 93, 5, 46, 790,
    982, 374, 872, 320, 297, 537,
    816, 231, 941, 986, 574, 185,
    // Bin capacity
    1549, 1426, 1826, 1729, 1571, 1541
  };
  const int i6_18_6_7_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    252, 134, 308, 479, 63, 520,
    475, 293, 917, 222, 789, 519,
    936, 757, 34, 305, 211, 192,
    394, 231, 233, 191, 350, 280,
    598, 797, 624, 900, 708, 864,
    73, 609, 828, 532, 73, 574,
    727, 345, 897, 742, 775, 765,
    989, 792, 284, 900, 731, 795,
    577, 219, 291, 773, 415, 386,
    910, 271, 422, 757, 780, 418,
    364, 150, 512, 690, 4, 241,
    293, 492, 430, 344, 439, 420,
    220, 182, 594, 359, 898, 127,
    343, 942, 493, 782, 343, 160,
    543, 865, 98, 747, 699, 312,
    784, 809, 456, 426, 168, 570,
    141, 148, 553, 202, 598, 510,
    742, 445, 730, 500, 436, 83,
    // Bin capacity
    1670, 1513, 1553, 1757, 1513, 1380
  };
  const int i6_18_6_8_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    920, 898, 843, 537, 285, 105,
    205, 780, 363, 511, 894, 604,
    730, 954, 589, 797, 849, 289,
    46, 931, 71, 517, 85, 689,
    684, 383, 252, 627, 166, 850,
    636, 412, 147, 531, 80, 768,
    568, 717, 494, 986, 138, 634,
    960, 343, 126, 208, 817, 177,
    351, 142, 859, 67, 940, 664,
    26, 377, 782, 169, 885, 993,
    358, 285, 842, 666, 161, 47,
    670, 674, 366, 836, 127, 88,
    992, 754, 516, 722, 985, 706,
    544, 475, 286, 328, 271, 402,
    147, 502, 693, 765, 964, 527,
    994, 934, 310, 124, 501, 235,
    114, 598, 63, 835, 800, 512,
    142, 203, 739, 465, 92, 48,
    // Bin capacity
    1636, 1866, 1502, 1745, 1628, 1501
  };
  const int i6_18_6_9_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    154, 97, 240, 315, 746, 678,
    650, 709, 219, 844, 955, 758,
    616, 740, 623, 644, 6, 665,
    895, 136, 606, 205, 562, 785,
    862, 697, 369, 467, 141, 10,
    391, 330, 165, 479, 106, 239,
    431, 978, 163, 813, 917, 237,
    949, 702, 804, 22, 442, 619,
    506, 880, 740, 347, 237, 648,
    641, 415, 978, 457, 643, 871,
    487, 741, 777, 379, 429, 60,
    465, 129, 131, 988, 590, 980,
    292, 398, 287, 138, 432, 871,
    84, 126, 949, 367, 521, 392,
    183, 514, 288, 801, 250, 59,
    989, 609, 121, 287, 939, 875,
    23, 918, 371, 239, 282, 8,
    88, 238, 222, 569, 149, 752,
    // Bin capacity
    1582, 1700, 1463, 1519, 1517, 1728
  };
  const int i6_18_6_10_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    882, 113, 858, 171, 222, 182,
    726, 200, 34, 459, 559, 336,
    748, 434, 118, 75, 950, 375,
    798, 866, 813, 606, 437, 251,
    938, 682, 994, 758, 527, 362,
    192, 462, 742, 40, 579, 648,
    845, 230, 553, 844, 83, 363,
    678, 638, 618, 639, 651, 876,
    210, 667, 859, 118, 752, 294,
    658, 524, 613, 422, 899, 0,
    862, 440, 574, 702, 15, 407,
    985, 857, 752, 227, 108, 432,
    571, 511, 659, 602, 491, 316,
    788, 403, 350, 946, 932, 150,
    335, 18, 765, 997, 240, 75,
    876, 641, 496, 585, 455, 64,
    52, 246, 523, 438, 971, 700,
    491, 253, 979, 872, 824, 683,
    // Bin capacity
    2134, 1501, 2072, 1742, 1778, 1195
  };
  const int i6_18_6_11_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    87, 441, 69, 223, 937, 542,
    97, 930, 824, 545, 691, 225,
    422, 930, 877, 128, 615, 781,
    80, 678, 86, 870, 8, 924,
    35, 10, 892, 18, 636, 118,
    989, 916, 379, 677, 764, 980,
    190, 297, 825, 447, 410, 16,
    356, 840, 515, 118, 888, 815,
    533, 801, 676, 41, 915, 706,
    900, 271, 848, 294, 953, 937,
    831, 457, 52, 877, 478, 21,
    147, 971, 291, 373, 205, 545,
    609, 348, 168, 749, 795, 229,
    946, 10, 74, 972, 780, 689,
    301, 381, 322, 651, 775, 668,
    964, 270, 326, 589, 451, 498,
    467, 808, 943, 650, 234, 20,
    500, 730, 543, 850, 515, 314,
    // Bin capacity
    1564, 1867, 1612, 1679, 2045, 1671
  };
  const int i6_18_6_12_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    867, 252, 581, 274, 544, 290,
    235, 663, 789, 27, 409, 39,
    180, 323, 697, 653, 74, 253,
    527, 718, 956, 903, 384, 568,
    356, 846, 680, 906, 193, 706,
    302, 233, 424, 595, 7, 822,
    18, 927, 947, 795, 666, 690,
    712, 203, 838, 845, 227, 72,
    252, 388, 891, 663, 764, 78,
    893, 369, 730, 278, 586, 705,
    563, 517, 64, 581, 879, 89,
    713, 88, 37, 235, 525, 599,
    504, 762, 417, 49, 742, 226,
    879, 73, 728, 971, 520, 623,
    260, 477, 509, 899, 448, 482,
    993, 386, 972, 201, 811, 751,
    924, 372, 286, 477, 819, 838,
    692, 937, 89, 839, 366, 736,
    // Bin capacity
    1843, 1594, 1986, 1903, 1674, 1600
  };
  const int i6_18_6_13_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    33, 975, 290, 874, 555, 239,
    640, 47, 864, 387, 463, 187,
    314, 885, 68, 70, 521, 459,
    391, 80, 138, 895, 546, 700,
    578, 815, 519, 257, 497, 10,
    130, 461, 270, 265, 225, 18,
    822, 299, 340, 864, 12, 70,
    994, 10, 884, 325, 955, 336,
    108, 747, 28, 147, 664, 642,
    11, 917, 774, 549, 85, 962,
    107, 376, 437, 38, 186, 398,
    574, 98, 524, 705, 133, 778,
    226, 459, 883, 830, 900, 174,
    791, 75, 51, 167, 826, 799,
    990, 504, 453, 463, 304, 242,
    136, 129, 227, 615, 195, 418,
    535, 189, 384, 445, 789, 259,
    623, 253, 742, 15, 668, 511,
    // Bin capacity
    1508, 1379, 1484, 1490, 1606, 1357
  };
  const int i6_18_6_14_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    531, 216, 845, 869, 928, 351,
    48, 366, 341, 660, 907, 409,
    764, 742, 868, 977, 15, 414,
    120, 803, 534, 978, 772, 836,
    924, 715, 998, 17, 958, 636,
    274, 208, 182, 780, 58, 761,
    994, 490, 208, 357, 53, 837,
    20, 603, 378, 618, 814, 368,
    437, 893, 17, 626, 449, 990,
    652, 353, 408, 542, 211, 157,
    957, 556, 73, 115, 440, 120,
    305, 694, 631, 44, 516, 980,
    78, 0, 794, 409, 333, 284,
    324, 758, 348, 360, 393, 872,
    223, 46, 636, 633, 569, 416,
    739, 286, 882, 46, 241, 769,
    233, 652, 138, 495, 95, 69,
    977, 26, 474, 953, 443, 450,
    // Bin capacity
    1635, 1598, 1664, 1802, 1558, 1847
  };
  const int i6_18_6_15_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    956, 613, 575, 749, 931, 453,
    279, 629, 149, 706, 410, 666,
    187, 227, 422, 760, 163, 422,
    69, 170, 327, 54, 420, 159,
    952, 912, 716, 333, 420, 429,
    110, 247, 300, 995, 958, 238,
    28, 354, 617, 391, 862, 891,
    997, 235, 242, 302, 140, 385,
    109, 679, 937, 246, 365, 37,
    570, 815, 153, 541, 730, 339,
    933, 909, 23, 970, 38, 249,
    537, 772, 506, 309, 896, 206,
    847, 250, 766, 883, 397, 70,
    443, 229, 517, 436, 481, 966,
    884, 175, 4, 577, 695, 95,
    88, 945, 314, 584, 386, 61,
    49, 147, 660, 836, 726, 751,
    300, 633, 427, 270, 794, 114,
    // Bin capacity
    1599, 1714, 1468, 1906, 1881, 1252
  };
  const int i6_18_6_16_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    973, 824, 91, 989, 29, 79,
    906, 574, 620, 157, 833, 932,
    923, 833, 40, 401, 747, 254,
    410, 663, 397, 488, 770, 852,
    184, 730, 314, 459, 457, 588,
    873, 608, 323, 323, 600, 96,
    794, 912, 736, 204, 321, 134,
    163, 573, 548, 320, 559, 367,
    892, 197, 101, 912, 563, 33,
    162, 284, 219, 60, 580, 617,
    19, 28, 912, 130, 428, 926,
    724, 183, 580, 687, 750, 915,
    996, 864, 918, 210, 324, 134,
    684, 773, 31, 256, 177, 555,
    85, 323, 892, 243, 243, 671,
    267, 165, 876, 119, 265, 462,
    792, 400, 174, 197, 423, 790,
    205, 994, 856, 796, 424, 200,
    // Bin capacity
    1944, 1920, 1669, 1344, 1642, 1664
  };
  const int i6_18_6_17_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    182, 281, 845, 849, 32, 137,
    494, 813, 726, 576, 308, 972,
    481, 714, 746, 563, 563, 593,
    503, 319, 275, 389, 930, 85,
    584, 459, 176, 84, 167, 790,
    221, 153, 901, 664, 607, 337,
    141, 473, 201, 650, 999, 44,
    297, 689, 896, 751, 720, 124,
    949, 64, 413, 811, 581, 76,
    849, 524, 181, 254, 876, 442,
    706, 374, 670, 391, 477, 294,
    126, 855, 13, 798, 836, 131,
    793, 234, 291, 821, 892, 136,
    602, 555, 723, 260, 902, 184,
    916, 489, 560, 231, 823, 682,
    471, 342, 982, 282, 72, 724,
    782, 51, 80, 522, 311, 13,
    204, 819, 377, 90, 185, 289,
    // Bin capacity
    1814, 1601, 1766, 1753, 2005, 1181
  };
  const int i6_18_6_18_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    91, 75, 122, 496, 712, 18,
    924, 560, 416, 800, 633, 760,
    257, 840, 529, 988, 577, 904,
    326, 995, 71, 750, 0, 878,
    717, 813, 878, 112, 127, 824,
    239, 215, 933, 144, 69, 544,
    209, 774, 719, 987, 80, 952,
    723, 567, 996, 726, 681, 879,
    31, 649, 25, 366, 776, 947,
    771, 459, 327, 946, 741, 946,
    505, 71, 105, 923, 635, 607,
    571, 809, 235, 525, 429, 305,
    157, 393, 402, 821, 789, 201,
    945, 717, 716, 752, 254, 826,
    143, 784, 904, 330, 178, 308,
    347, 972, 464, 1, 208, 987,
    960, 551, 413, 141, 343, 15,
    334, 83, 73, 504, 677, 614,
    // Bin capacity
    1623, 2031, 1638, 2029, 1556, 2265
  };
  const int i6_18_6_19_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    161, 386, 109, 227, 894, 157,
    798, 231, 774, 259, 646, 511,
    681, 450, 848, 858, 847, 855,
    163, 347, 648, 498, 796, 630,
    577, 772, 347, 505, 304, 834,
    889, 119, 182, 856, 12, 667,
    297, 276, 822, 489, 61, 214,
    527, 766, 340, 156, 555, 268,
    188, 699, 482, 941, 856, 466,
    451, 412, 958, 356, 547, 775,
    228, 336, 517, 349, 843, 854,
    963, 707, 255, 96, 669, 293,
    310, 907, 128, 447, 967, 47,
    797, 975, 804, 298, 396, 456,
    620, 158, 932, 508, 969, 230,
    21, 217, 878, 339, 538, 377,
    37, 872, 115, 995, 636, 45,
    660, 410, 269, 632, 347, 145,
    // Bin capacity
    1660, 1793, 1866, 1748, 2159, 1552
  };
  const int i6_18_6_20_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    67, 885, 48, 539, 184, 164,
    213, 130, 597, 45, 248, 87,
    799, 822, 250, 117, 878, 588,
    404, 15, 294, 519, 343, 711,
    218, 57, 564, 769, 112, 543,
    664, 779, 957, 670, 217, 349,
    117, 136, 315, 767, 995, 30,
    787, 856, 327, 117, 792, 937,
    694, 792, 877, 195, 971, 241,
    731, 633, 99, 835, 919, 884,
    631, 595, 864, 918, 990, 751,
    253, 157, 146, 317, 286, 714,
    362, 285, 960, 858, 174, 634,
    697, 752, 607, 676, 974, 401,
    880, 688, 772, 276, 262, 939,
    959, 943, 756, 465, 124, 637,
    433, 825, 551, 677, 107, 459,
    281, 513, 935, 42, 910, 434,
    // Bin capacity
    1838, 1973, 1984, 1761, 1898, 1901
  };
  const int i6_18_6_21_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    13, 393, 591, 323, 104, 136,
    527, 993, 478, 885, 316, 368,
    35, 399, 198, 710, 371, 312,
    468, 954, 395, 193, 542, 947,
    777, 17, 535, 34, 238, 830,
    144, 194, 921, 147, 70, 560,
    494, 799, 795, 695, 542, 159,
    904, 168, 519, 802, 820, 286,
    275, 241, 337, 260, 934, 359,
    374, 406, 327, 77, 159, 697,
    501, 525, 927, 570, 150, 378,
    168, 297, 763, 67, 389, 924,
    483, 830, 108, 747, 257, 537,
    650, 407, 589, 802, 674, 10,
    56, 146, 89, 874, 101, 429,
    88, 984, 339, 54, 572, 582,
    233, 124, 372, 998, 293, 778,
    111, 488, 859, 480, 175, 131,
    // Bin capacity
    1271, 1687, 1844, 1759, 1353, 1699
  };
  const int i6_18_6_22_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    184, 9, 187, 944, 138, 156,
    498, 926, 686, 919, 850, 597,
    296, 935, 628, 750, 492, 618,
    885, 309, 660, 121, 581, 13,
    832, 199, 590, 570, 985, 669,
    184, 487, 976, 817, 280, 483,
    129, 210, 209, 254, 642, 879,
    601, 810, 505, 435, 717, 868,
    545, 887, 41, 791, 381, 648,
    365, 642, 3, 306, 554, 665,
    519, 184, 241, 65, 703, 221,
    51, 696, 452, 959, 17, 774,
    391, 515, 801, 985, 481, 986,
    376, 159, 952, 12, 222, 907,
    516, 224, 864, 231, 198, 296,
    54, 528, 982, 456, 674, 325,
    235, 890, 436, 36, 315, 114,
    669, 55, 369, 804, 576, 376,
    // Bin capacity
    1491, 1762, 1949, 1923, 1791, 1951
  };
  const int i6_18_6_23_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    885, 598, 347, 400, 461, 546,
    871, 371, 386, 312, 523, 549,
    716, 174, 479, 145, 61, 263,
    646, 267, 238, 753, 542, 792,
    666, 705, 889, 165, 727, 253,
    649, 783, 882, 152, 983, 304,
    430, 134, 280, 852, 140, 535,
    392, 772, 686, 635, 278, 855,
    649, 903, 758, 389, 164, 272,
    177, 352, 393, 287, 83, 1,
    33, 298, 415, 1, 276, 30,
    877, 818, 128, 253, 125, 419,
    562, 599, 784, 403, 358, 904,
    251, 280, 173, 219, 946, 563,
    265, 841, 991, 365, 885, 844,
    309, 312, 562, 628, 291, 7,
    415, 437, 239, 425, 294, 360,
    880, 154, 156, 514, 388, 711,
    // Bin capacity
    1983, 1804, 1802, 1415, 1543, 1683
  };
  const int i6_18_6_24_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    472, 438, 852, 623, 207, 288,
    526, 870, 123, 190, 88, 78,
    737, 513, 626, 142, 331, 385,
    925, 908, 161, 607, 788, 792,
    221, 401, 646, 5, 450, 894,
    589, 690, 357, 325, 625, 783,
    339, 58, 550, 215, 32, 146,
    512, 517, 682, 328, 863, 591,
    519, 709, 572, 263, 727, 476,
    68, 755, 895, 155, 357, 560,
    63, 313, 739, 771, 800, 400,
    93, 521, 614, 75, 989, 912,
    837, 633, 628, 91, 222, 825,
    59, 24, 319, 710, 335, 536,
    906, 746, 575, 412, 303, 989,
    157, 110, 93, 571, 782, 414,
    733, 2, 984, 247, 207, 933,
    214, 197, 472, 766, 599, 966,
    // Bin capacity
    1648, 1738, 2044, 1343, 1800, 2267
  };
  const int i6_18_6_25_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    962, 133, 499, 736, 268, 861,
    128, 277, 518, 536, 450, 126,
    344, 750, 631, 799, 322, 311,
    392, 698, 974, 270, 424, 465,
    114, 512, 44, 369, 410, 246,
    250, 350, 19, 620, 830, 970,
    423, 362, 737, 963, 350, 93,
    744, 781, 850, 725, 193, 447,
    752, 641, 651, 699, 436, 961,
    251, 666, 483, 731, 321, 865,
    998, 866, 621, 35, 805, 538,
    744, 571, 364, 758, 707, 18,
    563, 330, 988, 61, 576, 923,
    87, 867, 863, 665, 586, 613,
    650, 584, 170, 459, 132, 244,
    70, 326, 935, 458, 452, 155,
    169, 798, 333, 323, 179, 833,
    908, 104, 263, 169, 475, 579,
    // Bin capacity
    1782, 2004, 2072, 1954, 1650, 1927
  };
  const int i6_18_6_26_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    752, 634, 53, 831, 465, 764,
    680, 794, 977, 329, 487, 645,
    768, 469, 569, 789, 901, 906,
    483, 86, 271, 407, 411, 972,
    903, 421, 399, 352, 51, 888,
    892, 535, 190, 56, 536, 592,
    44, 286, 374, 653, 801, 88,
    568, 38, 437, 339, 289, 13,
    645, 79, 224, 377, 446, 125,
    444, 889, 561, 176, 379, 217,
    30, 832, 430, 163, 540, 583,
    119, 55, 395, 120, 594, 173,
    448, 171, 443, 419, 862, 757,
    552, 689, 92, 575, 13, 885,
    738, 813, 435, 674, 761, 681,
    26, 721, 122, 669, 359, 786,
    206, 422, 398, 223, 691, 85,
    938, 179, 311, 135, 750, 696,
    // Bin capacity
    1940, 1704, 1404, 1531, 1961, 2070
  };
  const int i6_18_6_27_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    781, 913, 376, 562, 297, 901,
    616, 453, 975, 781, 689, 631,
    207, 661, 701, 712, 555, 535,
    988, 179, 511, 792, 798, 378,
    455, 712, 797, 866, 110, 797,
    402, 29, 63, 556, 82, 741,
    787, 259, 982, 568, 908, 249,
    670, 463, 679, 421, 667, 206,
    138, 394, 452, 694, 513, 653,
    223, 723, 216, 234, 812, 569,
    959, 98, 632, 219, 175, 194,
    156, 809, 489, 412, 558, 193,
    544, 628, 669, 23, 387, 129,
    144, 436, 529, 782, 85, 107,
    93, 144, 830, 657, 496, 232,
    663, 838, 427, 352, 816, 84,
    854, 663, 40, 126, 210, 494,
    748, 44, 6, 693, 530, 811,
    // Bin capacity
    1996, 1788, 1985, 2001, 1839, 1674
  };
  const int i6_18_6_28_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    473, 644, 424, 111, 541, 880,
    434, 272, 344, 928, 125, 739,
    381, 304, 458, 528, 411, 504,
    413, 863, 956, 257, 948, 165,
    9, 534, 863, 393, 480, 48,
    37, 539, 964, 780, 383, 141,
    970, 863, 894, 108, 914, 405,
    623, 97, 709, 260, 615, 440,
    374, 282, 97, 31, 719, 883,
    716, 88, 300, 87, 276, 56,
    601, 340, 318, 183, 47, 678,
    926, 649, 44, 958, 16, 480,
    20, 189, 508, 823, 155, 80,
    382, 519, 383, 883, 842, 440,
    751, 918, 881, 320, 981, 234,
    204, 520, 485, 206, 864, 93,
    199, 317, 439, 83, 136, 714,
    390, 70, 195, 936, 310, 234,
    // Bin capacity
    1686, 1709, 1976, 1680, 1870, 1539
  };
  const int i6_18_6_29_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    119, 283, 264, 888, 333, 172,
    812, 500, 585, 396, 720, 571,
    335, 99, 633, 895, 174, 324,
    207, 334, 301, 826, 181, 955,
    32, 51, 951, 62, 424, 339,
    127, 836, 407, 627, 315, 592,
    280, 889, 541, 962, 648, 119,
    296, 804, 925, 87, 315, 769,
    329, 108, 165, 150, 873, 862,
    936, 495, 966, 419, 947, 285,
    167, 580, 360, 94, 801, 529,
    239, 299, 483, 561, 898, 161,
    128, 582, 794, 811, 379, 936,
    486, 904, 403, 267, 26, 844,
    280, 85, 523, 732, 385, 952,
    7, 408, 408, 66, 413, 187,
    84, 982, 387, 92, 658, 181,
    582, 963, 783, 386, 925, 58,
    // Bin capacity
    1171, 1979, 2124, 1790, 2025, 1900
  };
  const int i6_18_6_30_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    955, 525, 45, 498, 1, 699,
    171, 190, 98, 15, 927, 364,
    512, 103, 714, 184, 126, 783,
    861, 464, 579, 973, 359, 614,
    699, 816, 290, 787, 392, 942,
    17, 573, 822, 985, 274, 609,
    146, 983, 350, 789, 155, 703,
    671, 561, 647, 123, 640, 426,
    402, 714, 805, 585, 355, 59,
    739, 658, 76, 855, 395, 722,
    194, 600, 118, 359, 628, 457,
    229, 652, 748, 209, 147, 499,
    765, 759, 757, 110, 549, 17,
    954, 457, 849, 999, 344, 150,
    962, 952, 146, 417, 984, 480,
    950, 226, 288, 699, 137, 174,
    954, 352, 941, 84, 563, 296,
    482, 148, 918, 446, 5, 159,
    // Bin capacity
    2311, 2109, 1992, 1976, 1513, 1767
  };
  const int i6_18_6_31_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    538, 714, 352, 232, 391, 826,
    323, 602, 253, 201, 594, 747,
    31, 25, 907, 497, 983, 61,
    955, 887, 574, 826, 350, 226,
    415, 550, 337, 901, 632, 264,
    234, 228, 992, 838, 384, 487,
    586, 735, 129, 779, 481, 558,
    8, 704, 492, 400, 962, 355,
    614, 225, 354, 492, 812, 69,
    560, 678, 385, 884, 429, 746,
    910, 188, 819, 394, 943, 595,
    169, 100, 81, 736, 903, 45,
    907, 365, 240, 391, 5, 643,
    260, 515, 728, 951, 67, 189,
    8, 313, 900, 162, 733, 55,
    109, 998, 534, 436, 44, 636,
    132, 275, 805, 258, 711, 9,
    687, 22, 160, 980, 407, 311,
    // Bin capacity
    1626, 1774, 1975, 2262, 2147, 1490
  };
  const int i6_18_6_32_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    796, 903, 18, 620, 961, 889,
    475, 968, 196, 601, 568, 296,
    989, 986, 402, 67, 126, 95,
    549, 589, 585, 496, 12, 215,
    599, 609, 949, 925, 422, 873,
    747, 827, 277, 627, 648, 851,
    92, 344, 541, 798, 727, 576,
    973, 864, 252, 204, 284, 488,
    441, 754, 855, 340, 628, 351,
    786, 592, 616, 814, 749, 40,
    834, 771, 346, 775, 812, 582,
    830, 376, 378, 711, 534, 70,
    976, 868, 132, 829, 133, 182,
    677, 182, 958, 948, 129, 769,
    99, 46, 685, 519, 378, 684,
    272, 91, 353, 247, 701, 423,
    207, 280, 825, 483, 725, 219,
    677, 910, 657, 165, 928, 614,
    // Bin capacity
    2425, 2412, 1986, 2238, 2083, 1808
  };
  const int i6_18_6_33_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    716, 477, 368, 104, 128, 592,
    423, 675, 454, 612, 406, 716,
    152, 379, 20, 54, 496, 638,
    249, 895, 979, 783, 876, 849,
    47, 3, 937, 268, 492, 15,
    671, 332, 637, 100, 879, 631,
    321, 268, 818, 982, 656, 870,
    764, 302, 151, 427, 756, 127,
    743, 723, 573, 678, 133, 206,
    265, 594, 610, 521, 599, 877,
    675, 116, 491, 157, 388, 274,
    484, 575, 229, 698, 911, 888,
    174, 170, 862, 251, 846, 229,
    869, 975, 775, 968, 533, 632,
    668, 693, 211, 863, 527, 564,
    897, 197, 587, 278, 616, 410,
    145, 650, 711, 715, 308, 993,
    486, 961, 525, 182, 258, 612,
    // Bin capacity
    1940, 1992, 2203, 1916, 2175, 2244
  };
  const int i6_18_6_34_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    13, 905, 288, 301, 986, 238,
    453, 766, 398, 405, 679, 396,
    372, 278, 354, 492, 853, 474,
    711, 324, 570, 494, 827, 510,
    121, 838, 704, 755, 283, 238,
    187, 489, 143, 224, 89, 852,
    467, 18, 327, 313, 464, 791,
    575, 614, 403, 814, 305, 235,
    425, 964, 293, 407, 579, 832,
    255, 533, 146, 789, 760, 599,
    526, 578, 861, 383, 994, 198,
    677, 834, 365, 478, 855, 378,
    314, 225, 855, 670, 943, 346,
    774, 661, 361, 803, 903, 791,
    45, 440, 189, 459, 203, 523,
    58, 79, 275, 942, 983, 481,
    118, 216, 330, 277, 56, 466,
    346, 917, 35, 590, 71, 107,
    // Bin capacity
    1438, 2162, 1541, 2144, 2420, 1889
  };
  const int i6_18_6_35_11[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    646, 857, 439, 962, 481, 473,
    68, 875, 860, 632, 71, 580,
    492, 81, 370, 998, 649, 360,
    510, 515, 188, 236, 776, 556,
    239, 192, 715, 984, 297, 787,
    84, 776, 596, 287, 748, 116,
    105, 273, 512, 78, 321, 219,
    574, 607, 38, 498, 19, 768,
    732, 431, 675, 123, 951, 608,
    585, 684, 716, 636, 742, 425,
    847, 226, 105, 227, 371, 120,
    308, 64, 256, 48, 387, 227,
    539, 257, 398, 378, 873, 126,
    565, 975, 80, 881, 830, 672,
    822, 565, 442, 663, 114, 413,
    821, 796, 771, 467, 356, 174,
    342, 735, 316, 250, 386, 550,
    903, 757, 258, 937, 964, 111,
    // Bin capacity
    2066, 2175, 1741, 2090, 2101, 1640
  };
  const int i6_18_6_0_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    48, 705, 240, 737, 878, 364,
    13, 808, 663, 802, 809, 338,
    251, 707, 217, 204, 969, 905,
    143, 695, 968, 444, 647, 485,
    241, 301, 105, 170, 744, 107,
    250, 144, 512, 665, 884, 663,
    655, 206, 834, 789, 945, 362,
    646, 701, 522, 600, 118, 625,
    47, 918, 983, 698, 195, 343,
    118, 158, 209, 487, 620, 335,
    906, 354, 971, 430, 9, 194,
    321, 824, 340, 783, 530, 22,
    734, 485, 89, 768, 713, 736,
    746, 702, 368, 89, 951, 575,
    91, 999, 161, 829, 875, 568,
    788, 43, 217, 527, 331, 352,
    211, 481, 791, 731, 812, 95,
    60, 263, 780, 742, 426, 501,
    // Bin capacity
    1045, 1583, 1495, 1750, 1910, 1262
  };
  const int i6_18_6_1_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    114, 737, 481, 546, 639, 525,
    811, 720, 955, 325, 370, 293,
    516, 822, 449, 10, 171, 530,
    361, 312, 459, 439, 261, 791,
    63, 296, 760, 646, 33, 676,
    527, 913, 847, 914, 821, 674,
    735, 502, 297, 597, 514, 978,
    700, 571, 738, 935, 333, 373,
    967, 292, 606, 559, 890, 159,
    93, 628, 103, 83, 568, 908,
    668, 797, 946, 306, 855, 594,
    368, 472, 263, 55, 136, 904,
    91, 400, 567, 701, 303, 0,
    695, 137, 503, 162, 566, 88,
    195, 545, 252, 279, 111, 240,
    421, 673, 817, 433, 75, 280,
    889, 907, 542, 35, 270, 791,
    241, 262, 403, 158, 756, 352,
    // Bin capacity
    1424, 1681, 1682, 1210, 1292, 1542
  };
  const int i6_18_6_2_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    375, 11, 849, 415, 221, 669,
    83, 124, 412, 292, 468, 387,
    326, 272, 145, 597, 974, 921,
    681, 326, 899, 296, 785, 0,
    521, 849, 870, 778, 116, 426,
    544, 593, 96, 260, 63, 233,
    122, 187, 802, 839, 182, 685,
    401, 408, 892, 740, 702, 494,
    67, 434, 116, 301, 61, 64,
    365, 738, 468, 151, 541, 994,
    171, 451, 468, 935, 72, 602,
    650, 262, 429, 926, 774, 428,
    596, 978, 76, 49, 394, 22,
    581, 298, 700, 736, 963, 673,
    605, 816, 624, 836, 475, 555,
    368, 647, 1, 371, 668, 247,
    474, 964, 642, 127, 281, 446,
    433, 202, 52, 215, 76, 255,
    // Bin capacity
    1252, 1456, 1452, 1507, 1329, 1378
  };
  const int i6_18_6_3_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    171, 153, 850, 308, 381, 173,
    522, 292, 90, 180, 301, 775,
    635, 885, 799, 700, 778, 154,
    323, 122, 883, 879, 239, 295,
    398, 820, 939, 230, 13, 13,
    19, 236, 184, 863, 63, 904,
    601, 2, 988, 924, 26, 916,
    954, 308, 249, 599, 355, 435,
    958, 57, 990, 256, 246, 279,
    22, 760, 708, 212, 835, 635,
    445, 919, 25, 958, 759, 42,
    140, 825, 760, 33, 344, 120,
    157, 752, 776, 786, 168, 836,
    480, 2, 466, 217, 820, 509,
    317, 931, 502, 233, 680, 473,
    333, 61, 155, 988, 374, 860,
    195, 919, 845, 781, 151, 822,
    232, 400, 60, 643, 735, 520,
    // Bin capacity
    1185, 1450, 1763, 1681, 1248, 1504
  };
  const int i6_18_6_4_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    952, 881, 784, 511, 615, 501,
    358, 621, 907, 307, 599, 223,
    956, 948, 118, 715, 358, 739,
    829, 415, 406, 696, 907, 290,
    925, 436, 501, 778, 196, 537,
    988, 134, 459, 247, 21, 11,
    355, 459, 119, 55, 898, 304,
    191, 517, 140, 745, 530, 893,
    79, 233, 383, 37, 675, 944,
    938, 29, 207, 808, 747, 368,
    846, 996, 561, 301, 913, 559,
    577, 647, 590, 360, 897, 635,
    415, 99, 283, 411, 767, 1,
    265, 69, 175, 529, 657, 68,
    122, 882, 244, 319, 911, 899,
    135, 738, 827, 397, 924, 902,
    375, 958, 401, 292, 353, 439,
    969, 148, 715, 776, 347, 975,
    // Bin capacity
    1781, 1597, 1356, 1436, 1962, 1610
  };
  const int i6_18_6_5_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    67, 396, 133, 262, 416, 116,
    331, 767, 412, 630, 389, 441,
    106, 850, 390, 50, 158, 445,
    81, 376, 934, 855, 618, 597,
    548, 495, 50, 497, 312, 104,
    106, 874, 329, 962, 145, 441,
    862, 333, 480, 177, 320, 588,
    979, 398, 336, 904, 732, 164,
    560, 324, 994, 537, 462, 594,
    457, 879, 713, 962, 255, 597,
    458, 856, 314, 493, 962, 117,
    34, 737, 483, 664, 916, 789,
    608, 251, 96, 122, 580, 388,
    838, 151, 732, 736, 938, 62,
    35, 795, 486, 455, 386, 924,
    166, 526, 477, 565, 216, 282,
    653, 507, 549, 423, 17, 793,
    448, 884, 626, 147, 298, 712,
    // Bin capacity
    1284, 1820, 1494, 1653, 1421, 1427
  };
  const int i6_18_6_6_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    219, 111, 340, 690, 418, 738,
    832, 12, 795, 768, 551, 873,
    58, 13, 554, 652, 943, 815,
    291, 488, 552, 167, 526, 887,
    786, 852, 992, 845, 787, 798,
    862, 409, 161, 208, 458, 783,
    630, 625, 610, 238, 130, 830,
    471, 82, 472, 108, 341, 262,
    592, 563, 985, 156, 544, 830,
    102, 803, 915, 879, 877, 262,
    191, 759, 523, 307, 232, 19,
    349, 315, 717, 751, 311, 482,
    755, 370, 892, 98, 824, 738,
    655, 660, 556, 568, 166, 929,
    62, 360, 704, 180, 503, 257,
    14, 148, 36, 455, 141, 288,
    360, 773, 830, 894, 596, 847,
    231, 757, 239, 872, 2, 617,
    // Bin capacity
    1318, 1431, 1921, 1562, 1476, 1989
  };
  const int i6_18_6_7_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    557, 925, 217, 978, 591, 464,
    930, 286, 34, 741, 950, 122,
    185, 334, 830, 625, 128, 832,
    463, 825, 582, 366, 745, 653,
    236, 892, 773, 983, 533, 720,
    239, 213, 33, 432, 247, 545,
    910, 421, 124, 162, 489, 559,
    693, 951, 142, 71, 932, 990,
    703, 13, 4, 171, 463, 757,
    476, 383, 342, 590, 758, 609,
    647, 92, 336, 146, 425, 589,
    371, 414, 677, 17, 338, 626,
    867, 582, 439, 814, 192, 360,
    654, 128, 300, 15, 521, 508,
    301, 822, 569, 782, 14, 362,
    445, 181, 274, 847, 328, 756,
    948, 573, 675, 331, 692, 294,
    160, 753, 802, 71, 248, 357,
    // Bin capacity
    1745, 1568, 1276, 1452, 1533, 1802
  };
  const int i6_18_6_8_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    530, 695, 80, 719, 194, 713,
    261, 447, 764, 177, 720, 439,
    382, 570, 754, 698, 66, 0,
    897, 941, 681, 296, 187, 673,
    605, 467, 386, 874, 725, 881,
    632, 154, 93, 644, 97, 769,
    992, 976, 458, 957, 238, 197,
    561, 420, 315, 650, 563, 69,
    341, 903, 459, 668, 71, 782,
    984, 440, 303, 953, 514, 151,
    418, 315, 447, 949, 550, 520,
    729, 728, 866, 521, 476, 121,
    186, 171, 533, 415, 650, 347,
    213, 592, 250, 803, 165, 374,
    798, 93, 264, 587, 70, 325,
    558, 571, 914, 351, 955, 646,
    645, 990, 375, 236, 369, 509,
    819, 77, 217, 212, 844, 288,
    // Bin capacity
    1900, 1719, 1469, 1928, 1342, 1405
  };
  const int i6_18_6_9_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    659, 1, 33, 13, 782, 129,
    909, 545, 594, 755, 61, 969,
    726, 870, 595, 532, 165, 308,
    520, 150, 53, 990, 550, 127,
    941, 301, 0, 192, 937, 355,
    624, 310, 680, 222, 599, 309,
    303, 157, 939, 457, 299, 663,
    413, 464, 14, 299, 90, 348,
    568, 583, 464, 913, 422, 960,
    230, 493, 698, 438, 869, 980,
    370, 775, 430, 727, 899, 379,
    272, 128, 974, 489, 741, 259,
    739, 886, 509, 756, 223, 578,
    274, 330, 28, 723, 820, 384,
    20, 702, 443, 529, 502, 73,
    282, 48, 16, 938, 841, 645,
    557, 472, 727, 651, 244, 620,
    501, 492, 284, 221, 74, 629,
    // Bin capacity
    1619, 1401, 1360, 1789, 1657, 1584
  };
  const int i6_18_6_10_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    145, 204, 291, 659, 673, 706,
    85, 965, 147, 950, 725, 582,
    228, 222, 608, 976, 864, 92,
    160, 902, 631, 666, 647, 212,
    327, 778, 683, 96, 999, 379,
    86, 422, 132, 291, 400, 74,
    699, 100, 512, 518, 690, 90,
    610, 126, 555, 868, 923, 986,
    220, 374, 426, 909, 133, 966,
    955, 323, 684, 574, 932, 958,
    861, 796, 905, 530, 800, 826,
    640, 590, 329, 368, 624, 839,
    482, 20, 674, 344, 68, 693,
    438, 458, 495, 632, 936, 453,
    670, 728, 445, 889, 804, 275,
    490, 371, 294, 251, 781, 266,
    647, 338, 853, 722, 818, 746,
    522, 614, 960, 2, 8, 78,
    // Bin capacity
    1516, 1528, 1765, 1879, 2168, 1691
  };
  const int i6_18_6_11_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    91, 443, 762, 753, 659, 786,
    596, 594, 366, 487, 814, 276,
    779, 859, 831, 363, 219, 788,
    165, 591, 784, 1000, 413, 221,
    533, 21, 943, 585, 241, 237,
    652, 622, 817, 695, 674, 370,
    993, 168, 535, 200, 763, 220,
    127, 954, 189, 734, 37, 920,
    867, 231, 693, 204, 720, 46,
    580, 250, 954, 382, 309, 298,
    338, 447, 911, 926, 293, 246,
    595, 591, 693, 22, 468, 779,
    139, 761, 987, 518, 529, 39,
    367, 630, 695, 177, 653, 601,
    230, 941, 802, 513, 919, 273,
    570, 972, 717, 819, 816, 878,
    547, 793, 879, 126, 427, 536,
    418, 846, 666, 395, 456, 643,
    // Bin capacity
    1589, 1983, 2447, 1647, 1741, 1510
  };
  const int i6_18_6_12_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    652, 879, 321, 611, 413, 939,
    424, 661, 841, 769, 956, 846,
    121, 31, 683, 82, 609, 949,
    428, 366, 203, 555, 619, 858,
    527, 360, 37, 427, 48, 184,
    309, 971, 630, 137, 71, 911,
    864, 622, 62, 877, 304, 54,
    608, 358, 256, 58, 937, 539,
    298, 128, 212, 841, 877, 710,
    130, 116, 94, 601, 529, 621,
    642, 477, 986, 949, 815, 13,
    557, 398, 900, 118, 607, 428,
    921, 212, 810, 37, 340, 606,
    198, 603, 709, 781, 518, 323,
    53, 815, 363, 972, 840, 656,
    723, 852, 648, 682, 294, 291,
    663, 608, 462, 119, 551, 39,
    190, 53, 997, 731, 833, 492,
    // Bin capacity
    1551, 1589, 1720, 1745, 1897, 1766
  };
  const int i6_18_6_13_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    428, 642, 763, 832, 791, 118,
    393, 561, 791, 739, 562, 776,
    492, 468, 545, 285, 482, 421,
    390, 754, 868, 569, 633, 19,
    122, 412, 241, 4, 134, 554,
    549, 434, 203, 514, 892, 394,
    606, 102, 450, 998, 416, 742,
    927, 344, 449, 785, 95, 393,
    761, 378, 604, 521, 859, 202,
    865, 5, 200, 255, 258, 392,
    668, 830, 224, 539, 816, 160,
    644, 905, 510, 915, 515, 417,
    687, 634, 884, 425, 579, 799,
    629, 740, 236, 830, 56, 407,
    948, 587, 580, 199, 44, 347,
    439, 816, 379, 676, 859, 745,
    618, 961, 184, 271, 828, 953,
    288, 589, 937, 966, 376, 170,
    // Bin capacity
    1969, 1914, 1705, 1945, 1732, 1509
  };
  const int i6_18_6_14_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    276, 176, 8, 170, 440, 74,
    33, 959, 480, 787, 647, 593,
    457, 58, 824, 170, 515, 893,
    108, 629, 424, 48, 7, 650,
    157, 162, 99, 378, 860, 279,
    907, 516, 344, 50, 660, 831,
    75, 271, 108, 512, 638, 164,
    633, 328, 155, 559, 314, 743,
    581, 57, 13, 710, 348, 440,
    338, 904, 922, 400, 575, 368,
    429, 754, 465, 832, 538, 12,
    238, 109, 721, 36, 976, 391,
    159, 769, 239, 445, 80, 797,
    885, 53, 417, 896, 12, 712,
    384, 892, 731, 263, 658, 43,
    869, 398, 8, 266, 385, 483,
    907, 408, 829, 33, 293, 837,
    691, 796, 52, 316, 392, 544,
    // Bin capacity
    1545, 1566, 1300, 1306, 1585, 1683
  };
  const int i6_18_6_15_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    872, 419, 819, 156, 694, 826,
    354, 530, 812, 938, 2, 827,
    66, 858, 700, 162, 364, 50,
    445, 83, 762, 314, 219, 422,
    405, 71, 860, 590, 207, 934,
    327, 40, 606, 619, 139, 99,
    499, 538, 934, 581, 325, 716,
    122, 145, 707, 166, 617, 867,
    553, 994, 31, 785, 642, 329,
    875, 193, 191, 211, 495, 27,
    411, 228, 715, 62, 774, 280,
    843, 792, 108, 918, 264, 894,
    482, 900, 647, 417, 238, 240,
    424, 685, 29, 299, 164, 68,
    316, 334, 665, 667, 945, 984,
    592, 807, 960, 588, 966, 62,
    969, 968, 210, 401, 435, 508,
    384, 990, 932, 731, 880, 944,
    // Bin capacity
    1714, 1836, 2049, 1650, 1605, 1740
  };
  const int i6_18_6_16_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    520, 321, 565, 355, 285, 857,
    994, 690, 200, 305, 406, 245,
    970, 112, 650, 106, 453, 889,
    969, 950, 973, 254, 607, 291,
    317, 736, 691, 494, 712, 913,
    430, 824, 989, 147, 438, 601,
    867, 526, 284, 553, 826, 693,
    895, 830, 399, 891, 753, 361,
    853, 799, 442, 597, 919, 783,
    595, 176, 382, 453, 201, 104,
    614, 233, 543, 857, 794, 998,
    958, 34, 889, 32, 397, 919,
    80, 545, 346, 817, 60, 312,
    4, 516, 790, 167, 302, 999,
    896, 636, 4, 930, 477, 160,
    227, 591, 201, 623, 324, 31,
    813, 352, 247, 167, 91, 701,
    582, 318, 625, 660, 856, 954,
    // Bin capacity
    2240, 1777, 1783, 1626, 1721, 2091
  };
  const int i6_18_6_17_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    625, 321, 396, 327, 587, 967,
    498, 202, 837, 473, 586, 513,
    312, 834, 462, 514, 441, 12,
    935, 729, 975, 508, 438, 881,
    746, 392, 689, 951, 711, 850,
    345, 734, 473, 483, 853, 217,
    994, 688, 458, 146, 260, 204,
    350, 198, 374, 87, 8, 101,
    639, 430, 431, 49, 471, 211,
    831, 199, 551, 231, 374, 798,
    458, 637, 197, 392, 766, 485,
    48, 737, 257, 908, 24, 528,
    803, 67, 766, 631, 156, 285,
    300, 901, 912, 1, 749, 697,
    533, 735, 266, 48, 617, 344,
    145, 65, 375, 253, 425, 932,
    144, 299, 400, 662, 155, 917,
    571, 319, 89, 651, 102, 87,
    // Bin capacity
    1810, 1655, 1738, 1427, 1506, 1761
  };
  const int i6_18_6_18_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    729, 755, 472, 28, 345, 956,
    161, 8, 610, 837, 531, 396,
    387, 767, 764, 604, 262, 783,
    3, 646, 934, 542, 635, 19,
    226, 486, 792, 803, 472, 68,
    99, 888, 106, 327, 375, 567,
    112, 985, 833, 330, 887, 603,
    620, 723, 687, 539, 159, 194,
    36, 884, 526, 562, 636, 295,
    384, 200, 103, 357, 919, 673,
    2, 780, 281, 180, 707, 10,
    585, 776, 796, 262, 686, 101,
    238, 73, 832, 655, 419, 403,
    578, 388, 895, 824, 706, 927,
    37, 353, 358, 152, 715, 113,
    347, 753, 497, 393, 171, 804,
    53, 980, 107, 670, 644, 195,
    903, 442, 702, 524, 980, 695,
    // Bin capacity
    1082, 2142, 2025, 1690, 2016, 1535
  };
  const int i6_18_6_19_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    168, 197, 357, 660, 110, 645,
    174, 308, 445, 157, 89, 612,
    759, 285, 373, 22, 943, 430,
    348, 713, 221, 627, 15, 880,
    397, 745, 182, 426, 996, 466,
    630, 871, 271, 90, 17, 489,
    744, 867, 44, 387, 786, 811,
    699, 845, 383, 194, 249, 113,
    1, 675, 850, 978, 304, 336,
    258, 270, 375, 331, 773, 898,
    55, 228, 120, 221, 974, 248,
    881, 236, 441, 968, 546, 484,
    296, 299, 936, 562, 908, 937,
    513, 964, 496, 925, 535, 930,
    180, 916, 11, 531, 83, 584,
    892, 933, 619, 731, 350, 448,
    694, 107, 217, 109, 843, 203,
    86, 551, 449, 90, 597, 138,
    // Bin capacity
    1543, 1986, 1347, 1589, 1809, 1915
  };
  const int i6_18_6_20_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    552, 857, 403, 173, 281, 214,
    350, 521, 540, 896, 534, 236,
    804, 766, 576, 594, 490, 663,
    146, 603, 70, 26, 198, 472,
    567, 786, 95, 560, 915, 765,
    574, 196, 967, 405, 440, 674,
    30, 408, 272, 107, 420, 667,
    504, 359, 223, 17, 416, 34,
    298, 456, 391, 918, 874, 939,
    840, 109, 270, 137, 906, 909,
    836, 852, 485, 592, 625, 600,
    341, 516, 643, 630, 107, 427,
    793, 424, 382, 553, 701, 359,
    442, 636, 284, 931, 437, 962,
    703, 347, 708, 70, 37, 395,
    433, 529, 63, 232, 505, 324,
    834, 22, 301, 66, 576, 453,
    259, 700, 393, 250, 633, 551,
    // Bin capacity
    1862, 1818, 1414, 1432, 1819, 1929
  };
  const int i6_18_6_21_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    233, 355, 572, 588, 772, 397,
    759, 911, 377, 434, 886, 787,
    220, 379, 312, 746, 290, 523,
    752, 602, 782, 998, 888, 741,
    609, 470, 390, 369, 337, 252,
    19, 313, 436, 827, 649, 520,
    97, 39, 903, 678, 309, 90,
    838, 259, 605, 722, 826, 736,
    384, 466, 100, 568, 415, 368,
    1, 374, 898, 79, 982, 376,
    114, 829, 500, 641, 805, 276,
    723, 765, 98, 926, 549, 585,
    945, 668, 247, 471, 954, 156,
    69, 345, 162, 839, 822, 348,
    572, 502, 39, 960, 24, 296,
    245, 811, 900, 687, 199, 394,
    850, 726, 636, 695, 537, 294,
    12, 626, 263, 543, 975, 458,
    // Bin capacity
    1501, 1904, 1658, 2374, 2263, 1533
  };
  const int i6_18_6_22_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    254, 768, 688, 616, 898, 411,
    493, 200, 490, 587, 467, 840,
    614, 284, 436, 687, 890, 655,
    173, 658, 159, 383, 708, 852,
    528, 904, 897, 441, 813, 914,
    487, 722, 801, 22, 174, 22,
    210, 914, 609, 827, 936, 715,
    228, 275, 219, 981, 672, 395,
    735, 211, 275, 138, 625, 105,
    220, 485, 546, 488, 407, 153,
    828, 573, 850, 963, 405, 979,
    888, 403, 888, 637, 135, 309,
    130, 101, 267, 448, 547, 634,
    576, 620, 881, 971, 413, 973,
    986, 343, 253, 114, 60, 96,
    890, 153, 912, 137, 426, 49,
    639, 147, 861, 122, 228, 655,
    49, 173, 636, 84, 246, 566,
    // Bin capacity
    1816, 1614, 2170, 1759, 1841, 1896
  };
  const int i6_18_6_23_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    452, 890, 341, 18, 149, 951,
    95, 509, 603, 216, 471, 163,
    726, 601, 572, 807, 151, 784,
    635, 619, 894, 430, 441, 515,
    108, 118, 774, 359, 344, 219,
    146, 58, 793, 775, 596, 38,
    558, 619, 997, 415, 920, 554,
    880, 846, 579, 104, 221, 741,
    916, 40, 241, 670, 857, 642,
    112, 210, 657, 412, 192, 51,
    684, 905, 450, 576, 505, 971,
    309, 429, 403, 519, 579, 889,
    863, 678, 682, 44, 430, 535,
    465, 902, 532, 490, 146, 597,
    456, 191, 121, 81, 155, 910,
    143, 587, 0, 627, 437, 985,
    914, 137, 684, 918, 107, 282,
    22, 715, 64, 57, 478, 853,
    // Bin capacity
    1740, 1857, 1925, 1542, 1472, 2190
  };
  const int i6_18_6_24_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    433, 66, 388, 420, 205, 960,
    580, 770, 902, 611, 345, 299,
    685, 849, 877, 32, 156, 354,
    615, 448, 327, 834, 650, 196,
    649, 76, 140, 32, 171, 791,
    122, 980, 511, 723, 699, 824,
    157, 375, 469, 292, 274, 757,
    240, 620, 519, 632, 244, 129,
    884, 822, 746, 210, 143, 390,
    976, 889, 78, 336, 650, 129,
    236, 803, 325, 596, 444, 537,
    257, 853, 114, 753, 389, 236,
    9, 144, 480, 926, 285, 578,
    35, 641, 808, 404, 610, 871,
    887, 277, 296, 304, 35, 947,
    506, 233, 534, 296, 623, 498,
    468, 224, 599, 511, 915, 912,
    673, 495, 649, 424, 590, 665,
    // Bin capacity
    1739, 1977, 1811, 1723, 1536, 2082
  };
  const int i6_18_6_25_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    804, 802, 564, 564, 357, 890,
    416, 790, 9, 732, 727, 385,
    90, 90, 561, 274, 994, 855,
    673, 483, 512, 682, 535, 165,
    145, 711, 724, 966, 965, 27,
    776, 557, 271, 447, 195, 290,
    651, 24, 919, 793, 790, 930,
    775, 410, 684, 798, 499, 523,
    590, 934, 602, 847, 816, 900,
    529, 347, 996, 787, 923, 908,
    486, 449, 100, 73, 5, 939,
    74, 62, 960, 448, 655, 831,
    989, 763, 271, 920, 355, 822,
    795, 456, 352, 254, 565, 127,
    814, 690, 634, 358, 629, 858,
    129, 143, 83, 534, 648, 714,
    688, 919, 943, 539, 102, 156,
    476, 238, 387, 879, 957, 801,
    // Bin capacity
    2063, 1848, 1995, 2270, 2233, 2317
  };
  const int i6_18_6_26_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    757, 147, 159, 936, 147, 683,
    689, 804, 975, 375, 115, 512,
    23, 917, 893, 304, 441, 37,
    597, 727, 555, 334, 909, 715,
    230, 115, 194, 697, 509, 432,
    431, 368, 768, 577, 187, 661,
    330, 241, 92, 573, 169, 18,
    354, 256, 835, 107, 533, 875,
    111, 905, 281, 232, 202, 631,
    274, 527, 531, 303, 866, 558,
    159, 489, 125, 315, 16, 520,
    38, 100, 658, 312, 415, 320,
    813, 834, 10, 79, 383, 668,
    361, 746, 384, 9, 475, 457,
    121, 425, 842, 652, 818, 439,
    815, 625, 575, 674, 67, 619,
    51, 427, 680, 852, 64, 27,
    44, 426, 349, 868, 579, 554,
    // Bin capacity
    1302, 1907, 1871, 1722, 1448, 1833
  };
  const int i6_18_6_27_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    410, 688, 975, 705, 356, 675,
    361, 131, 448, 853, 823, 155,
    593, 314, 283, 687, 526, 794,
    50, 727, 916, 470, 206, 547,
    682, 295, 436, 14, 505, 765,
    987, 94, 933, 937, 727, 184,
    323, 612, 500, 288, 644, 8,
    791, 72, 901, 906, 625, 875,
    63, 93, 666, 333, 88, 278,
    529, 439, 50, 881, 162, 536,
    291, 462, 476, 765, 683, 736,
    864, 422, 835, 497, 598, 668,
    684, 148, 861, 40, 935, 882,
    681, 416, 421, 461, 58, 502,
    423, 577, 438, 482, 912, 562,
    148, 29, 933, 921, 1, 470,
    360, 606, 918, 776, 587, 830,
    771, 312, 264, 338, 431, 383,
    // Bin capacity
    1908, 1363, 2383, 2192, 1877, 2085
  };
  const int i6_18_6_28_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    122, 206, 961, 931, 211, 610,
    306, 665, 208, 743, 5, 16,
    561, 348, 249, 386, 531, 396,
    521, 314, 339, 858, 354, 514,
    570, 249, 972, 329, 647, 334,
    987, 96, 995, 560, 99, 368,
    825, 803, 820, 238, 670, 471,
    558, 537, 439, 33, 10, 721,
    168, 477, 128, 823, 645, 275,
    864, 527, 387, 948, 645, 816,
    228, 489, 922, 978, 718, 261,
    91, 778, 183, 11, 842, 310,
    960, 779, 744, 652, 11, 820,
    625, 371, 248, 921, 144, 727,
    372, 135, 417, 385, 471, 214,
    491, 436, 137, 943, 300, 309,
    107, 667, 658, 424, 372, 528,
    714, 437, 281, 163, 340, 101,
    // Bin capacity
    1935, 1774, 1939, 2203, 1497, 1663
  };
  const int i6_18_6_29_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    918, 622, 695, 280, 191, 13,
    75, 972, 480, 514, 900, 556,
    86, 985, 74, 676, 193, 413,
    445, 786, 818, 226, 402, 253,
    82, 372, 730, 840, 403, 376,
    369, 342, 597, 508, 800, 281,
    608, 407, 224, 719, 86, 436,
    708, 211, 765, 69, 966, 981,
    868, 39, 402, 595, 822, 863,
    104, 353, 633, 681, 387, 120,
    111, 914, 922, 796, 308, 778,
    789, 127, 67, 742, 636, 672,
    950, 933, 284, 842, 334, 65,
    611, 614, 922, 145, 782, 952,
    430, 875, 761, 245, 337, 639,
    693, 575, 491, 370, 109, 329,
    170, 60, 752, 706, 871, 756,
    424, 200, 490, 804, 933, 5,
    // Bin capacity
    1815, 2019, 2174, 2098, 2034, 1825
  };
  const int i6_18_6_30_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    178, 819, 622, 893, 65, 548,
    62, 514, 737, 725, 387, 507,
    621, 11, 243, 158, 302, 293,
    356, 590, 25, 604, 327, 740,
    148, 531, 578, 691, 128, 918,
    279, 58, 798, 205, 615, 372,
    640, 994, 648, 95, 981, 365,
    532, 593, 7, 984, 357, 188,
    378, 867, 91, 614, 825, 843,
    71, 832, 142, 255, 362, 160,
    482, 403, 47, 153, 969, 962,
    283, 450, 751, 441, 623, 110,
    660, 400, 683, 824, 96, 16,
    327, 618, 308, 855, 724, 55,
    980, 954, 926, 521, 940, 31,
    263, 695, 918, 60, 790, 986,
    322, 850, 278, 387, 551, 113,
    365, 830, 311, 202, 433, 58,
    // Bin capacity
    1506, 2386, 1758, 1878, 2053, 1575
  };
  const int i6_18_6_31_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    780, 259, 130, 256, 45, 433,
    610, 760, 216, 22, 79, 315,
    261, 188, 372, 270, 36, 812,
    454, 505, 876, 201, 982, 953,
    659, 706, 220, 917, 34, 290,
    669, 33, 478, 679, 117, 820,
    938, 931, 797, 828, 689, 773,
    658, 605, 375, 422, 553, 622,
    631, 636, 808, 381, 481, 197,
    80, 421, 747, 940, 581, 580,
    391, 864, 253, 501, 881, 331,
    631, 215, 166, 564, 541, 65,
    244, 926, 291, 588, 759, 982,
    7, 823, 665, 71, 770, 571,
    110, 442, 390, 223, 220, 926,
    943, 555, 277, 836, 633, 379,
    88, 908, 170, 444, 73, 4,
    212, 480, 822, 675, 360, 766,
    // Bin capacity
    1827, 2240, 1759, 1926, 1711, 2144
  };
  const int i6_18_6_32_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    878, 381, 20, 111, 750, 156,
    787, 18, 993, 336, 41, 497,
    514, 885, 821, 531, 539, 613,
    531, 653, 325, 503, 780, 783,
    379, 978, 866, 818, 788, 634,
    355, 454, 559, 657, 213, 864,
    150, 993, 581, 745, 959, 751,
    909, 713, 222, 919, 402, 755,
    558, 151, 301, 261, 420, 161,
    23, 616, 996, 828, 946, 553,
    231, 233, 970, 70, 720, 197,
    494, 611, 718, 997, 57, 25,
    318, 929, 125, 216, 882, 665,
    885, 935, 421, 422, 815, 835,
    930, 596, 902, 884, 203, 983,
    504, 703, 916, 725, 627, 419,
    978, 908, 525, 577, 784, 577,
    806, 432, 129, 691, 48, 898,
    // Bin capacity
    2251, 2462, 2286, 2265, 2195, 2281
  };
  const int i6_18_6_33_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    441, 494, 494, 203, 72, 183,
    404, 803, 383, 854, 30, 671,
    658, 677, 762, 15, 589, 768,
    595, 598, 216, 752, 97, 219,
    344, 140, 966, 139, 133, 419,
    956, 299, 389, 238, 645, 59,
    187, 672, 643, 567, 959, 295,
    896, 908, 269, 151, 636, 846,
    98, 945, 160, 504, 812, 366,
    222, 116, 829, 251, 814, 38,
    421, 301, 984, 609, 418, 706,
    609, 667, 746, 811, 185, 368,
    995, 765, 792, 910, 369, 819,
    247, 93, 56, 955, 125, 961,
    149, 834, 339, 353, 178, 567,
    852, 184, 687, 630, 998, 100,
    96, 456, 484, 413, 625, 810,
    261, 421, 506, 806, 823, 448,
    // Bin capacity
    1869, 2078, 2152, 2031, 1886, 1916
  };
  const int i6_18_6_34_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    52, 145, 334, 535, 428, 727,
    143, 657, 527, 103, 168, 448,
    537, 583, 113, 704, 458, 116,
    177, 101, 527, 209, 469, 828,
    952, 992, 331, 873, 424, 265,
    633, 814, 635, 858, 742, 659,
    957, 565, 37, 42, 704, 19,
    833, 70, 422, 814, 903, 474,
    447, 827, 536, 393, 930, 540,
    849, 836, 147, 937, 13, 538,
    677, 932, 125, 234, 272, 982,
    94, 587, 153, 561, 995, 228,
    933, 644, 728, 869, 227, 390,
    873, 493, 662, 793, 670, 679,
    770, 796, 454, 49, 445, 591,
    771, 126, 561, 734, 793, 22,
    121, 883, 951, 18, 634, 106,
    370, 719, 446, 828, 523, 892,
    // Bin capacity
    2276, 2406, 1718, 2134, 2189, 1900
  };
  const int i6_18_6_35_12[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    422, 266, 56, 585, 354, 481,
    125, 300, 206, 415, 434, 535,
    760, 763, 999, 742, 322, 446,
    581, 914, 888, 884, 309, 375,
    863, 599, 345, 656, 904, 909,
    939, 342, 477, 146, 234, 832,
    436, 573, 201, 847, 286, 581,
    655, 741, 962, 831, 48, 117,
    928, 343, 510, 251, 648, 768,
    737, 543, 55, 959, 142, 758,
    631, 527, 587, 233, 810, 689,
    115, 250, 711, 359, 962, 553,
    20, 245, 628, 208, 925, 620,
    375, 230, 863, 657, 279, 967,
    515, 654, 564, 410, 304, 854,
    136, 874, 675, 526, 785, 574,
    546, 438, 814, 569, 74, 775,
    635, 99, 697, 869, 711, 792,
    // Bin capacity
    2120, 1958, 2304, 2284, 1920, 2616
  };
  const int i6_18_6_0_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    680, 681, 679, 533, 283, 893,
    161, 42, 759, 246, 685, 323,
    47, 811, 323, 615, 918, 658,
    666, 766, 122, 961, 394, 627,
    36, 956, 807, 14, 849, 802,
    54, 916, 116, 653, 753, 453,
    210, 760, 963, 536, 175, 501,
    926, 693, 493, 618, 759, 485,
    241, 664, 697, 156, 240, 323,
    53, 454, 696, 348, 118, 369,
    689, 162, 407, 355, 742, 151,
    837, 946, 891, 265, 65, 139,
    942, 805, 565, 196, 328, 433,
    312, 15, 575, 675, 374, 322,
    834, 397, 553, 199, 422, 17,
    376, 63, 740, 598, 903, 809,
    38, 202, 648, 985, 913, 46,
    544, 165, 793, 535, 794, 937,
    // Bin capacity
    1275, 1583, 1805, 1415, 1620, 1382
  };
  const int i6_18_6_1_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    961, 42, 547, 813, 62, 634,
    526, 992, 984, 210, 458, 509,
    85, 236, 925, 597, 655, 195,
    9, 737, 964, 206, 580, 618,
    598, 64, 265, 963, 768, 193,
    313, 694, 108, 983, 788, 217,
    656, 114, 590, 723, 515, 657,
    891, 217, 133, 484, 285, 643,
    301, 816, 734, 902, 504, 79,
    839, 58, 391, 564, 893, 100,
    614, 97, 695, 408, 388, 606,
    898, 969, 621, 133, 53, 839,
    428, 778, 329, 450, 152, 826,
    776, 34, 6, 555, 768, 111,
    160, 435, 504, 858, 850, 710,
    140, 188, 122, 316, 264, 511,
    205, 422, 844, 652, 257, 577,
    893, 162, 269, 691, 91, 403,
    // Bin capacity
    1565, 1188, 1521, 1769, 1403, 1419
  };
  const int i6_18_6_2_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    127, 477, 954, 388, 324, 853,
    169, 456, 145, 866, 302, 1,
    72, 53, 151, 642, 625, 523,
    923, 795, 53, 48, 896, 504,
    303, 421, 395, 28, 324, 987,
    597, 986, 689, 115, 292, 478,
    92, 82, 108, 730, 190, 786,
    302, 111, 986, 316, 528, 431,
    968, 35, 318, 648, 899, 134,
    385, 662, 100, 240, 718, 406,
    909, 606, 81, 299, 240, 917,
    462, 749, 985, 624, 502, 977,
    719, 955, 67, 499, 121, 348,
    676, 156, 621, 11, 639, 822,
    344, 173, 259, 753, 785, 868,
    680, 628, 317, 24, 0, 325,
    83, 687, 894, 239, 851, 113,
    161, 343, 21, 169, 719, 773,
    // Bin capacity
    1356, 1424, 1215, 1129, 1523, 1742
  };
  const int i6_18_6_3_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    293, 221, 609, 133, 509, 901,
    20, 188, 50, 593, 957, 651,
    662, 91, 425, 419, 723, 277,
    591, 488, 735, 607, 997, 700,
    857, 820, 855, 233, 850, 928,
    340, 895, 269, 414, 365, 257,
    670, 566, 333, 35, 352, 703,
    175, 849, 937, 573, 866, 537,
    917, 563, 582, 230, 344, 313,
    224, 18, 545, 539, 401, 15,
    886, 68, 442, 624, 297, 715,
    639, 835, 870, 137, 228, 116,
    232, 99, 150, 59, 69, 659,
    218, 732, 7, 297, 847, 96,
    311, 672, 931, 799, 792, 531,
    254, 92, 920, 372, 959, 792,
    794, 867, 424, 203, 499, 982,
    140, 301, 214, 51, 591, 148,
    // Bin capacity
    1412, 1436, 1597, 1085, 1828, 1601
  };
  const int i6_18_6_4_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    187, 26, 452, 961, 866, 416,
    63, 164, 502, 793, 34, 208,
    329, 590, 299, 907, 309, 859,
    811, 735, 619, 98, 967, 196,
    238, 930, 70, 532, 927, 771,
    50, 319, 386, 881, 785, 179,
    444, 976, 677, 482, 253, 200,
    629, 174, 891, 103, 242, 667,
    229, 926, 66, 988, 44, 186,
    30, 735, 960, 879, 984, 575,
    327, 234, 329, 406, 818, 762,
    936, 288, 403, 906, 60, 150,
    821, 300, 796, 829, 278, 273,
    638, 597, 242, 349, 712, 136,
    407, 681, 919, 823, 286, 103,
    495, 151, 210, 673, 450, 587,
    522, 529, 815, 448, 517, 4,
    299, 752, 524, 920, 487, 719,
    // Bin capacity
    1293, 1579, 1588, 2077, 1564, 1212
  };
  const int i6_18_6_5_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    806, 452, 890, 944, 294, 387,
    346, 365, 962, 763, 386, 806,
    562, 957, 39, 581, 711, 207,
    910, 658, 964, 336, 985, 473,
    817, 342, 759, 969, 3, 377,
    34, 451, 633, 124, 205, 380,
    739, 568, 994, 513, 958, 908,
    184, 861, 356, 770, 657, 621,
    366, 551, 904, 286, 863, 456,
    772, 11, 316, 854, 254, 991,
    45, 760, 119, 103, 822, 475,
    734, 900, 547, 364, 481, 913,
    678, 167, 718, 195, 857, 384,
    597, 294, 324, 31, 951, 382,
    145, 518, 359, 975, 827, 61,
    760, 696, 199, 94, 328, 994,
    168, 389, 967, 123, 929, 150,
    758, 880, 158, 489, 280, 590,
    // Bin capacity
    1649, 1719, 1787, 1490, 1889, 1673
  };
  const int i6_18_6_6_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    793, 538, 202, 619, 630, 137,
    853, 433, 273, 59, 22, 701,
    200, 432, 706, 305, 850, 443,
    710, 983, 938, 346, 99, 967,
    953, 213, 730, 129, 404, 644,
    198, 748, 709, 392, 787, 14,
    346, 990, 393, 554, 294, 457,
    782, 749, 276, 964, 755, 192,
    736, 113, 586, 577, 354, 610,
    472, 533, 528, 862, 216, 664,
    806, 980, 216, 506, 623, 326,
    190, 119, 740, 483, 521, 79,
    711, 681, 16, 863, 783, 564,
    956, 427, 193, 529, 430, 148,
    361, 452, 411, 536, 311, 194,
    799, 903, 259, 101, 649, 791,
    358, 809, 148, 469, 513, 616,
    627, 609, 529, 457, 943, 696,
    // Bin capacity
    1918, 1893, 1388, 1547, 1623, 1457
  };
  const int i6_18_6_7_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    586, 917, 261, 459, 130, 616,
    709, 10, 403, 228, 728, 590,
    919, 984, 407, 295, 856, 635,
    992, 436, 775, 238, 726, 899,
    898, 369, 280, 195, 970, 215,
    528, 463, 366, 423, 390, 665,
    848, 982, 507, 136, 198, 717,
    43, 501, 26, 588, 812, 904,
    870, 153, 32, 863, 422, 548,
    461, 846, 105, 731, 102, 894,
    350, 716, 927, 450, 970, 163,
    987, 525, 240, 826, 870, 325,
    746, 856, 111, 397, 341, 607,
    526, 46, 365, 874, 423, 760,
    226, 354, 924, 953, 169, 195,
    376, 574, 293, 18, 219, 301,
    839, 4, 530, 312, 512, 354,
    657, 378, 215, 985, 878, 48,
    // Bin capacity
    2062, 1626, 1207, 1600, 1733, 1683
  };
  const int i6_18_6_8_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    950, 731, 340, 993, 120, 588,
    482, 94, 260, 466, 656, 555,
    931, 299, 934, 361, 375, 113,
    945, 74, 434, 89, 315, 23,
    570, 708, 651, 177, 358, 345,
    363, 753, 60, 450, 754, 342,
    581, 349, 513, 299, 420, 707,
    670, 227, 709, 245, 452, 504,
    49, 369, 50, 55, 805, 616,
    36, 199, 4, 357, 244, 838,
    903, 531, 885, 722, 777, 323,
    469, 814, 60, 315, 113, 617,
    480, 587, 102, 584, 122, 281,
    737, 441, 472, 749, 900, 16,
    237, 991, 737, 766, 329, 407,
    669, 239, 50, 393, 859, 122,
    166, 480, 818, 295, 410, 689,
    59, 125, 766, 967, 594, 481,
    // Bin capacity
    1674, 1442, 1413, 1491, 1549, 1363
  };
  const int i6_18_6_9_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    152, 632, 546, 478, 746, 714,
    923, 423, 133, 710, 740, 946,
    133, 668, 635, 544, 37, 248,
    855, 96, 814, 657, 839, 612,
    147, 568, 610, 317, 124, 933,
    809, 389, 232, 33, 373, 689,
    109, 743, 921, 840, 364, 657,
    798, 991, 375, 332, 465, 727,
    567, 599, 772, 774, 29, 539,
    253, 680, 994, 156, 170, 148,
    401, 189, 201, 254, 940, 320,
    859, 938, 801, 575, 347, 641,
    690, 259, 746, 987, 528, 978,
    195, 791, 31, 806, 363, 616,
    968, 510, 913, 457, 841, 964,
    48, 267, 319, 200, 494, 999,
    902, 550, 661, 185, 790, 234,
    465, 414, 19, 59, 539, 38,
    // Bin capacity
    1685, 1764, 1767, 1520, 1586, 1999
  };
  const int i6_18_6_10_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    141, 977, 715, 291, 96, 734,
    117, 64, 871, 562, 266, 280,
    18, 304, 357, 949, 457, 975,
    425, 566, 197, 377, 946, 961,
    61, 966, 933, 534, 649, 278,
    935, 354, 984, 44, 726, 45,
    685, 166, 604, 809, 997, 443,
    848, 698, 695, 3, 410, 929,
    315, 744, 180, 473, 941, 768,
    261, 471, 223, 198, 167, 57,
    873, 530, 236, 840, 370, 341,
    375, 556, 818, 609, 664, 277,
    25, 227, 352, 57, 894, 480,
    175, 777, 735, 882, 181, 576,
    817, 317, 345, 716, 725, 955,
    86, 230, 672, 919, 621, 80,
    796, 701, 529, 164, 551, 382,
    526, 118, 322, 582, 33, 282,
    // Bin capacity
    1372, 1608, 1791, 1652, 1778, 1622
  };
  const int i6_18_6_11_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    751, 551, 380, 862, 263, 503,
    308, 608, 52, 271, 46, 599,
    582, 357, 586, 310, 405, 195,
    661, 68, 35, 718, 192, 332,
    710, 731, 695, 399, 445, 607,
    469, 257, 838, 743, 671, 804,
    806, 464, 81, 713, 64, 224,
    648, 296, 112, 998, 425, 68,
    54, 173, 135, 627, 630, 688,
    20, 586, 552, 878, 726, 347,
    355, 979, 638, 94, 57, 311,
    674, 667, 539, 511, 93, 654,
    781, 341, 939, 169, 583, 213,
    71, 184, 745, 109, 455, 73,
    962, 338, 744, 170, 358, 301,
    674, 277, 70, 503, 457, 270,
    498, 600, 543, 819, 234, 319,
    275, 280, 288, 971, 445, 403,
    // Bin capacity
    1721, 1436, 1475, 1826, 1212, 1279
  };
  const int i6_18_6_12_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    256, 654, 482, 610, 386, 73,
    729, 846, 165, 933, 687, 810,
    62, 483, 869, 68, 295, 825,
    1, 730, 248, 768, 364, 669,
    738, 435, 672, 685, 292, 271,
    60, 937, 157, 545, 253, 586,
    812, 74, 219, 544, 48, 302,
    107, 487, 709, 69, 863, 370,
    713, 444, 997, 854, 262, 934,
    318, 901, 401, 741, 209, 308,
    343, 263, 618, 561, 248, 182,
    602, 778, 134, 874, 748, 944,
    387, 349, 433, 2, 401, 548,
    958, 988, 458, 505, 941, 324,
    669, 199, 93, 656, 6, 238,
    464, 382, 270, 23, 690, 202,
    925, 267, 222, 582, 5, 111,
    933, 59, 544, 826, 560, 80,
    // Bin capacity
    1695, 1732, 1436, 1838, 1355, 1452
  };
  const int i6_18_6_13_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    969, 89, 277, 353, 939, 491,
    36, 651, 852, 27, 600, 944,
    523, 739, 663, 107, 771, 531,
    6, 670, 962, 293, 777, 59,
    510, 29, 87, 717, 735, 252,
    579, 81, 288, 274, 832, 242,
    254, 899, 268, 115, 65, 128,
    570, 165, 307, 182, 456, 300,
    64, 474, 389, 915, 842, 57,
    194, 579, 416, 88, 7, 8,
    579, 532, 289, 314, 277, 286,
    562, 457, 287, 948, 869, 203,
    677, 322, 67, 847, 248, 639,
    722, 61, 451, 990, 506, 168,
    646, 545, 322, 123, 242, 336,
    327, 475, 988, 471, 955, 553,
    835, 596, 212, 706, 410, 941,
    884, 73, 278, 133, 455, 644,
    // Bin capacity
    1684, 1401, 1395, 1432, 1881, 1278
  };
  const int i6_18_6_14_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    1, 871, 718, 658, 143, 311,
    167, 60, 638, 140, 848, 794,
    234, 159, 434, 220, 249, 531,
    108, 529, 779, 262, 807, 901,
    863, 352, 109, 427, 950, 455,
    456, 487, 640, 748, 672, 341,
    231, 538, 575, 551, 919, 540,
    472, 548, 636, 978, 360, 353,
    255, 534, 638, 666, 880, 777,
    201, 623, 213, 703, 341, 593,
    299, 509, 242, 542, 383, 857,
    830, 952, 277, 58, 408, 176,
    4, 900, 414, 197, 322, 640,
    661, 472, 703, 688, 935, 497,
    512, 931, 249, 810, 752, 943,
    568, 322, 727, 312, 860, 284,
    209, 561, 692, 490, 668, 260,
    685, 572, 164, 718, 642, 82,
    // Bin capacity
    1284, 1885, 1682, 1742, 2117, 1774
  };
  const int i6_18_6_15_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    638, 893, 127, 359, 134, 863,
    849, 32, 337, 380, 615, 778,
    910, 327, 585, 368, 930, 961,
    841, 937, 264, 861, 963, 869,
    295, 418, 931, 17, 840, 277,
    241, 876, 671, 361, 854, 758,
    104, 544, 234, 111, 751, 274,
    24, 76, 83, 972, 108, 977,
    140, 910, 926, 163, 826, 946,
    548, 521, 75, 970, 694, 378,
    581, 169, 703, 948, 789, 799,
    480, 978, 74, 297, 866, 537,
    413, 552, 337, 120, 882, 498,
    213, 634, 628, 558, 9, 892,
    976, 180, 700, 406, 587, 44,
    462, 209, 761, 123, 547, 135,
    668, 959, 394, 772, 684, 381,
    421, 116, 424, 416, 258, 318,
    // Bin capacity
    1688, 1789, 1583, 1573, 2173, 2048
  };
  const int i6_18_6_16_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    856, 14, 38, 487, 296, 422,
    124, 357, 126, 91, 276, 417,
    929, 588, 967, 497, 681, 564,
    849, 2, 698, 775, 797, 268,
    777, 575, 889, 514, 884, 762,
    812, 781, 744, 307, 62, 597,
    828, 197, 167, 354, 69, 794,
    713, 555, 326, 778, 42, 568,
    307, 455, 733, 447, 830, 621,
    39, 615, 535, 838, 894, 331,
    905, 654, 187, 756, 451, 211,
    130, 114, 841, 833, 523, 465,
    819, 770, 938, 941, 264, 574,
    582, 900, 527, 637, 816, 369,
    574, 896, 487, 132, 829, 863,
    625, 1, 367, 512, 229, 640,
    84, 192, 773, 744, 763, 786,
    693, 609, 964, 520, 604, 48,
    // Bin capacity
    2059, 1600, 1993, 1965, 1800, 1798
  };
  const int i6_18_6_17_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    656, 891, 578, 80, 725, 137,
    685, 34, 896, 520, 797, 875,
    13, 981, 769, 458, 151, 584,
    304, 868, 82, 418, 754, 863,
    883, 888, 905, 783, 670, 499,
    281, 152, 627, 654, 582, 154,
    926, 307, 295, 243, 45, 56,
    540, 841, 134, 678, 270, 783,
    931, 483, 149, 389, 298, 24,
    666, 508, 596, 292, 490, 211,
    292, 637, 185, 357, 68, 446,
    774, 663, 625, 220, 147, 416,
    937, 741, 237, 674, 500, 205,
    606, 730, 559, 484, 369, 516,
    241, 87, 368, 183, 32, 374,
    894, 148, 258, 159, 633, 687,
    200, 387, 919, 607, 861, 173,
    607, 5, 16, 332, 846, 652,
    // Bin capacity
    2036, 1824, 1599, 1469, 1607, 1493
  };
  const int i6_18_6_18_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    758, 783, 279, 862, 754, 625,
    243, 641, 11, 626, 811, 299,
    456, 476, 812, 975, 443, 154,
    452, 271, 625, 407, 655, 159,
    878, 623, 979, 259, 893, 126,
    984, 538, 544, 454, 716, 659,
    755, 400, 245, 22, 966, 44,
    111, 386, 201, 413, 791, 156,
    236, 362, 339, 97, 294, 179,
    908, 234, 662, 986, 994, 692,
    269, 228, 541, 10, 294, 917,
    879, 7, 489, 292, 861, 730,
    11, 757, 522, 351, 989, 61,
    655, 758, 675, 962, 909, 223,
    831, 4, 889, 868, 747, 200,
    910, 736, 465, 895, 722, 787,
    234, 423, 242, 765, 675, 712,
    134, 283, 920, 114, 959, 617,
    // Bin capacity
    1909, 1556, 1857, 1841, 2650, 1444
  };
  const int i6_18_6_19_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    378, 28, 920, 338, 786, 421,
    4, 906, 689, 219, 224, 872,
    32, 634, 227, 106, 982, 68,
    777, 476, 625, 572, 889, 504,
    62, 357, 785, 476, 894, 761,
    762, 726, 468, 658, 522, 771,
    805, 68, 89, 488, 92, 401,
    199, 106, 214, 689, 668, 533,
    495, 122, 404, 614, 104, 918,
    960, 453, 695, 653, 674, 979,
    442, 9, 473, 760, 537, 921,
    964, 988, 5, 804, 90, 949,
    2, 814, 904, 729, 273, 102,
    780, 787, 409, 63, 948, 354,
    134, 386, 812, 411, 185, 213,
    914, 656, 311, 799, 912, 317,
    101, 934, 747, 964, 90, 479,
    32, 863, 978, 22, 411, 48,
    // Bin capacity
    1556, 1848, 1935, 1858, 1841, 1907
  };
  const int i6_18_6_20_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    803, 243, 79, 538, 14, 849,
    589, 324, 802, 908, 310, 767,
    718, 690, 414, 642, 489, 613,
    347, 424, 144, 626, 55, 663,
    563, 283, 769, 24, 214, 266,
    835, 950, 935, 724, 244, 512,
    493, 819, 472, 823, 919, 711,
    29, 691, 853, 684, 289, 842,
    729, 362, 783, 772, 497, 18,
    113, 392, 725, 305, 10, 442,
    97, 202, 363, 747, 555, 259,
    150, 638, 751, 481, 403, 562,
    241, 203, 124, 373, 827, 514,
    929, 651, 275, 713, 504, 82,
    357, 954, 328, 950, 108, 550,
    468, 472, 169, 710, 498, 971,
    538, 974, 511, 120, 73, 931,
    918, 874, 660, 155, 802, 65,
    // Bin capacity
    1784, 2030, 1832, 2059, 1363, 1924
  };
  const int i6_18_6_21_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    410, 527, 197, 743, 734, 544,
    261, 861, 569, 141, 961, 869,
    885, 475, 856, 77, 465, 468,
    878, 65, 865, 11, 713, 156,
    516, 485, 641, 425, 310, 538,
    619, 839, 748, 893, 479, 316,
    479, 961, 418, 453, 190, 883,
    991, 125, 594, 868, 409, 712,
    336, 459, 733, 97, 715, 313,
    358, 477, 505, 739, 619, 661,
    207, 116, 873, 913, 210, 531,
    212, 973, 607, 740, 206, 711,
    266, 848, 311, 725, 582, 40,
    484, 268, 806, 985, 850, 139,
    415, 834, 960, 52, 641, 200,
    314, 992, 111, 405, 454, 181,
    55, 859, 899, 930, 2, 512,
    101, 966, 857, 260, 920, 491,
    // Bin capacity
    1571, 2245, 2330, 1908, 1908, 1667
  };
  const int i6_18_6_22_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    875, 897, 910, 896, 442, 688,
    500, 144, 618, 141, 452, 99,
    136, 795, 813, 505, 482, 462,
    303, 694, 246, 54, 488, 883,
    102, 152, 674, 210, 86, 434,
    758, 264, 225, 727, 273, 89,
    12, 975, 708, 9, 262, 126,
    109, 224, 831, 24, 228, 961,
    514, 40, 757, 388, 43, 485,
    548, 91, 437, 394, 351, 499,
    42, 506, 662, 533, 766, 453,
    75, 221, 367, 418, 771, 699,
    613, 293, 357, 885, 25, 317,
    949, 176, 828, 600, 398, 903,
    201, 318, 456, 769, 298, 545,
    816, 369, 658, 171, 621, 91,
    41, 754, 138, 984, 21, 617,
    836, 342, 248, 434, 188, 945,
    // Bin capacity
    1511, 1476, 2020, 1656, 1260, 1891
  };
  const int i6_18_6_23_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    477, 799, 845, 561, 781, 582,
    554, 132, 1, 305, 70, 98,
    141, 511, 653, 810, 948, 870,
    279, 33, 321, 122, 100, 713,
    613, 605, 367, 358, 807, 227,
    850, 546, 778, 825, 689, 532,
    571, 781, 836, 326, 601, 916,
    606, 223, 238, 642, 973, 860,
    810, 383, 461, 80, 434, 134,
    431, 801, 829, 108, 343, 763,
    389, 650, 270, 489, 220, 383,
    423, 907, 916, 936, 483, 787,
    635, 424, 362, 995, 600, 32,
    992, 72, 145, 655, 888, 744,
    823, 89, 226, 294, 762, 240,
    302, 299, 224, 923, 900, 461,
    741, 250, 212, 105, 9, 540,
    792, 560, 948, 594, 152, 972,
    // Bin capacity
    2138, 1654, 1770, 1872, 2001, 2021
  };
  const int i6_18_6_24_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    904, 492, 155, 589, 65, 459,
    92, 604, 443, 438, 725, 931,
    1, 539, 216, 176, 569, 410,
    655, 797, 177, 447, 326, 811,
    719, 596, 558, 208, 614, 764,
    819, 497, 352, 487, 764, 110,
    987, 849, 720, 413, 767, 871,
    141, 948, 920, 926, 56, 549,
    683, 293, 389, 704, 850, 516,
    431, 635, 921, 346, 222, 485,
    512, 538, 35, 755, 321, 730,
    656, 467, 101, 591, 216, 777,
    82, 617, 105, 359, 996, 30,
    103, 503, 450, 265, 905, 171,
    136, 839, 676, 207, 417, 62,
    943, 49, 340, 334, 579, 434,
    923, 840, 479, 183, 272, 161,
    622, 874, 293, 557, 73, 324,
    // Bin capacity
    1945, 2269, 1515, 1651, 1806, 1777
  };
  const int i6_18_6_25_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    387, 996, 710, 912, 753, 484,
    135, 529, 524, 61, 293, 126,
    316, 54, 14, 122, 982, 296,
    228, 172, 87, 990, 125, 220,
    78, 623, 303, 615, 497, 536,
    337, 183, 512, 501, 259, 924,
    930, 36, 461, 889, 454, 124,
    185, 464, 118, 385, 905, 625,
    69, 807, 898, 889, 709, 839,
    440, 33, 139, 614, 794, 33,
    160, 118, 233, 325, 184, 829,
    190, 698, 698, 955, 992, 783,
    59, 221, 457, 356, 96, 434,
    361, 246, 609, 224, 750, 637,
    567, 835, 316, 847, 950, 180,
    388, 615, 685, 521, 343, 791,
    330, 545, 143, 771, 334, 844,
    967, 742, 926, 99, 511, 642,
    // Bin capacity
    1277, 1650, 1632, 2100, 2069, 1948
  };
  const int i6_18_6_26_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    874, 579, 262, 980, 212, 176,
    145, 99, 677, 276, 375, 450,
    131, 289, 658, 631, 139, 674,
    419, 659, 589, 411, 559, 663,
    822, 711, 483, 601, 312, 962,
    816, 623, 512, 800, 645, 139,
    122, 383, 323, 738, 709, 205,
    305, 19, 904, 105, 766, 682,
    158, 237, 487, 394, 402, 235,
    373, 344, 122, 402, 770, 878,
    584, 723, 499, 724, 517, 259,
    180, 476, 173, 209, 823, 30,
    294, 664, 241, 710, 367, 400,
    645, 863, 904, 141, 853, 678,
    78, 62, 510, 869, 873, 464,
    862, 138, 622, 171, 338, 92,
    343, 339, 216, 815, 791, 383,
    294, 506, 595, 844, 586, 425,
    // Bin capacity
    1564, 1620, 1844, 2063, 2108, 1637
  };
  const int i6_18_6_27_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    565, 882, 261, 412, 394, 850,
    728, 349, 44, 297, 61, 81,
    230, 619, 39, 178, 455, 856,
    752, 142, 436, 92, 386, 353,
    607, 733, 521, 471, 122, 835,
    997, 240, 322, 907, 328, 253,
    813, 324, 97, 691, 10, 612,
    387, 234, 711, 938, 12, 246,
    254, 297, 57, 789, 499, 296,
    372, 38, 920, 974, 483, 45,
    694, 354, 11, 479, 147, 636,
    436, 177, 202, 97, 21, 457,
    452, 854, 507, 878, 178, 378,
    830, 558, 941, 593, 775, 882,
    215, 331, 477, 521, 661, 647,
    255, 343, 992, 914, 249, 249,
    796, 774, 888, 88, 416, 413,
    771, 249, 762, 683, 665, 480,
    // Bin capacity
    2150, 1588, 1734, 2118, 1241, 1814
  };
  const int i6_18_6_28_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    404, 733, 610, 925, 719, 38,
    462, 841, 727, 490, 889, 78,
    225, 67, 324, 105, 866, 321,
    383, 399, 930, 637, 0, 386,
    955, 87, 880, 298, 517, 0,
    55, 873, 925, 949, 146, 254,
    542, 687, 881, 706, 233, 279,
    358, 342, 542, 614, 514, 917,
    439, 31, 459, 467, 306, 504,
    596, 896, 791, 218, 724, 540,
    357, 982, 389, 545, 847, 420,
    44, 154, 668, 736, 844, 402,
    66, 16, 710, 967, 76, 48,
    994, 630, 837, 438, 399, 671,
    376, 108, 654, 424, 29, 162,
    176, 709, 426, 595, 578, 77,
    497, 600, 519, 614, 924, 599,
    55, 995, 836, 839, 424, 85,
    // Bin capacity
    1490, 1952, 2584, 2255, 1928, 1234
  };
  const int i6_18_6_29_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    567, 1000, 712, 370, 255, 321,
    279, 142, 767, 973, 981, 862,
    853, 617, 526, 160, 144, 677,
    345, 543, 914, 428, 106, 846,
    404, 688, 166, 424, 730, 953,
    635, 550, 312, 859, 63, 818,
    384, 169, 734, 849, 463, 576,
    117, 478, 905, 349, 88, 590,
    355, 713, 77, 993, 558, 685,
    416, 594, 706, 88, 357, 93,
    180, 59, 360, 165, 850, 929,
    818, 899, 594, 509, 322, 161,
    671, 644, 733, 207, 205, 291,
    65, 493, 976, 535, 568, 735,
    715, 713, 178, 708, 280, 245,
    761, 494, 154, 175, 655, 268,
    190, 182, 300, 991, 57, 553,
    674, 127, 114, 342, 81, 1,
    // Bin capacity
    1813, 1958, 1985, 1962, 1455, 2065
  };
  const int i6_18_6_30_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    102, 902, 303, 400, 17, 204,
    214, 543, 572, 9, 487, 36,
    409, 703, 272, 770, 250, 522,
    544, 171, 34, 178, 95, 633,
    245, 382, 77, 769, 380, 718,
    381, 238, 353, 504, 341, 7,
    743, 62, 970, 605, 388, 891,
    134, 219, 320, 507, 308, 10,
    555, 271, 374, 739, 486, 480,
    259, 623, 572, 166, 201, 596,
    828, 701, 812, 716, 544, 459,
    492, 145, 666, 501, 376, 373,
    372, 627, 127, 5, 494, 967,
    488, 991, 598, 48, 724, 393,
    409, 777, 660, 122, 563, 543,
    988, 680, 360, 722, 222, 540,
    666, 112, 73, 54, 409, 578,
    46, 699, 550, 220, 576, 503,
    // Bin capacity
    1707, 1917, 1667, 1525, 1487, 1832
  };
  const int i6_18_6_31_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    174, 413, 95, 294, 776, 496,
    530, 116, 83, 9, 467, 305,
    540, 585, 523, 342, 77, 861,
    957, 270, 907, 501, 693, 863,
    30, 804, 981, 612, 347, 938,
    117, 914, 329, 363, 31, 717,
    995, 260, 364, 801, 993, 78,
    319, 715, 312, 935, 356, 930,
    491, 883, 897, 96, 607, 167,
    435, 668, 381, 802, 140, 503,
    739, 319, 838, 621, 704, 603,
    485, 270, 712, 363, 799, 317,
    615, 368, 687, 177, 973, 392,
    308, 557, 290, 557, 626, 738,
    646, 635, 707, 520, 418, 279,
    541, 701, 83, 867, 247, 945,
    651, 373, 547, 194, 614, 509,
    858, 733, 901, 184, 776, 915,
    // Bin capacity
    2060, 2093, 2105, 1799, 2106, 2305
  };
  const int i6_18_6_32_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    401, 919, 868, 159, 589, 961,
    957, 979, 600, 908, 501, 946,
    635, 114, 641, 844, 624, 65,
    426, 287, 141, 79, 526, 679,
    584, 254, 729, 521, 584, 34,
    19, 562, 398, 647, 843, 468,
    118, 128, 215, 451, 969, 301,
    505, 299, 516, 449, 438, 681,
    950, 712, 335, 931, 693, 324,
    522, 234, 614, 201, 948, 84,
    73, 555, 587, 686, 302, 811,
    53, 365, 682, 647, 27, 688,
    186, 537, 776, 596, 59, 239,
    929, 636, 100, 553, 884, 283,
    551, 518, 925, 410, 931, 531,
    901, 581, 982, 831, 38, 999,
    518, 125, 321, 368, 120, 32,
    582, 371, 85, 695, 794, 403,
    // Bin capacity
    1961, 1799, 2094, 2195, 2172, 1877
  };
  const int i6_18_6_33_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    511, 573, 369, 512, 94, 127,
    759, 571, 396, 171, 915, 307,
    568, 889, 863, 924, 449, 877,
    278, 822, 679, 148, 381, 69,
    754, 893, 270, 395, 600, 622,
    415, 959, 372, 202, 682, 567,
    210, 957, 486, 19, 596, 65,
    452, 214, 872, 690, 8, 360,
    418, 747, 508, 864, 751, 374,
    686, 294, 753, 467, 405, 995,
    297, 774, 190, 122, 996, 735,
    972, 15, 89, 442, 360, 455,
    61, 286, 447, 7, 89, 867,
    875, 445, 424, 279, 458, 947,
    241, 381, 286, 751, 666, 924,
    865, 58, 485, 685, 346, 65,
    910, 445, 177, 907, 566, 322,
    829, 935, 141, 748, 887, 523,
    // Bin capacity
    2240, 2274, 1731, 1848, 2051, 2040
  };
  const int i6_18_6_34_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    502, 376, 877, 785, 690, 754,
    801, 628, 310, 151, 365, 644,
    625, 822, 134, 241, 944, 515,
    745, 107, 741, 921, 224, 847,
    797, 586, 685, 449, 587, 952,
    453, 34, 127, 492, 766, 34,
    397, 558, 619, 650, 961, 312,
    934, 759, 535, 502, 168, 755,
    510, 921, 451, 653, 16, 582,
    291, 838, 401, 71, 122, 350,
    933, 810, 458, 205, 732, 372,
    436, 675, 813, 742, 652, 577,
    66, 926, 350, 130, 649, 802,
    651, 321, 599, 968, 87, 175,
    833, 183, 967, 874, 41, 827,
    176, 876, 977, 424, 709, 827,
    1000, 89, 334, 382, 246, 565,
    861, 942, 465, 129, 116, 974,
    // Bin capacity
    2460, 2335, 2199, 1959, 1804, 2427
  };
  const int i6_18_6_35_13[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    521, 123, 985, 207, 910, 391,
    88, 375, 137, 536, 520, 873,
    182, 166, 598, 139, 214, 359,
    784, 652, 936, 860, 70, 88,
    731, 915, 573, 975, 741, 361,
    917, 235, 513, 742, 481, 188,
    941, 777, 379, 498, 225, 394,
    902, 911, 72, 531, 357, 940,
    187, 426, 491, 659, 911, 306,
    768, 309, 931, 898, 713, 624,
    127, 475, 842, 452, 707, 949,
    970, 320, 599, 870, 864, 812,
    831, 880, 267, 615, 241, 973,
    275, 888, 627, 475, 361, 920,
    322, 765, 43, 361, 380, 805,
    300, 446, 804, 759, 509, 597,
    676, 196, 801, 109, 557, 986,
    160, 89, 112, 356, 106, 300,
    // Bin capacity
    2179, 2014, 2185, 2260, 1996, 2445
  };
  const int i6_18_6_0_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    562, 10, 720, 454, 564, 610,
    831, 186, 710, 117, 505, 878,
    166, 725, 35, 311, 873, 136,
    364, 230, 760, 116, 60, 830,
    261, 660, 72, 573, 404, 396,
    969, 416, 925, 854, 497, 820,
    258, 80, 832, 302, 721, 976,
    833, 288, 887, 81, 848, 871,
    204, 589, 717, 102, 804, 182,
    915, 376, 308, 509, 936, 611,
    519, 32, 32, 618, 502, 21,
    614, 730, 319, 854, 223, 762,
    861, 621, 294, 156, 833, 363,
    19, 407, 30, 841, 964, 528,
    972, 309, 741, 740, 357, 973,
    362, 752, 829, 393, 607, 93,
    599, 764, 801, 957, 683, 949,
    691, 940, 169, 293, 703, 998,
    // Bin capacity
    1667, 1353, 1531, 1379, 1848, 1833
  };
  const int i6_18_6_1_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    214, 904, 656, 213, 766, 2,
    705, 23, 217, 767, 159, 721,
    44, 730, 546, 333, 879, 424,
    155, 806, 719, 942, 555, 379,
    98, 589, 664, 229, 411, 981,
    172, 188, 966, 4, 95, 972,
    169, 725, 91, 863, 135, 950,
    623, 551, 899, 424, 926, 122,
    920, 572, 427, 718, 79, 156,
    267, 180, 166, 577, 588, 909,
    439, 308, 42, 183, 24, 20,
    171, 213, 49, 38, 514, 342,
    174, 801, 425, 228, 937, 589,
    566, 975, 918, 688, 871, 284,
    523, 618, 747, 919, 467, 85,
    477, 152, 982, 425, 986, 434,
    940, 532, 505, 381, 837, 66,
    123, 719, 696, 19, 28, 196,
    // Bin capacity
    1142, 1614, 1636, 1339, 1559, 1285
  };
  const int i6_18_6_2_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    988, 379, 489, 68, 698, 796,
    823, 650, 496, 557, 149, 95,
    594, 811, 387, 89, 740, 681,
    402, 706, 607, 158, 529, 297,
    154, 438, 583, 737, 996, 896,
    522, 802, 589, 638, 275, 993,
    752, 990, 932, 405, 200, 702,
    365, 677, 917, 778, 881, 517,
    871, 70, 843, 442, 202, 651,
    499, 426, 303, 391, 747, 354,
    715, 938, 845, 338, 326, 539,
    216, 657, 635, 830, 632, 29,
    665, 25, 734, 359, 682, 809,
    509, 353, 19, 467, 131, 909,
    933, 575, 709, 760, 977, 804,
    638, 89, 518, 490, 709, 67,
    351, 202, 768, 829, 843, 106,
    232, 452, 519, 893, 767, 273,
    // Bin capacity
    1739, 1571, 1852, 1569, 1783, 1619
  };
  const int i6_18_6_3_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    485, 319, 108, 977, 170, 237,
    317, 191, 249, 272, 289, 877,
    336, 808, 692, 402, 650, 64,
    670, 739, 691, 284, 930, 868,
    666, 733, 827, 317, 168, 894,
    168, 386, 440, 559, 107, 579,
    908, 753, 266, 663, 964, 730,
    608, 815, 933, 185, 122, 271,
    454, 94, 241, 949, 407, 570,
    384, 768, 942, 802, 821, 417,
    971, 329, 432, 676, 731, 887,
    111, 510, 599, 489, 47, 836,
    1000, 341, 333, 921, 126, 414,
    647, 50, 919, 396, 291, 477,
    820, 85, 849, 140, 737, 526,
    352, 857, 511, 833, 75, 129,
    828, 961, 312, 738, 390, 877,
    476, 911, 935, 710, 135, 124,
    // Bin capacity
    1752, 1657, 1765, 1771, 1230, 1679
  };
  const int i6_18_6_4_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    731, 170, 12, 702, 526, 428,
    83, 389, 610, 395, 668, 356,
    756, 260, 163, 34, 700, 328,
    373, 491, 583, 835, 921, 955,
    72, 227, 996, 619, 112, 267,
    998, 84, 68, 322, 186, 952,
    498, 520, 711, 716, 174, 369,
    966, 153, 434, 842, 610, 768,
    722, 166, 164, 505, 570, 815,
    441, 146, 455, 580, 600, 444,
    155, 491, 294, 105, 317, 26,
    889, 79, 947, 894, 576, 594,
    924, 802, 994, 118, 812, 265,
    710, 539, 997, 248, 88, 103,
    817, 562, 720, 920, 817, 695,
    42, 642, 500, 165, 879, 47,
    963, 650, 466, 850, 673, 575,
    927, 982, 963, 959, 111, 713,
    // Bin capacity
    1919, 1275, 1747, 1701, 1619, 1508
  };
  const int i6_18_6_5_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    681, 575, 932, 310, 345, 512,
    509, 507, 683, 650, 164, 772,
    707, 295, 526, 358, 755, 374,
    667, 637, 305, 710, 136, 267,
    191, 419, 383, 286, 729, 911,
    173, 675, 368, 781, 687, 6,
    566, 844, 725, 816, 879, 642,
    79, 543, 580, 362, 151, 92,
    300, 86, 506, 339, 721, 821,
    704, 334, 158, 987, 422, 160,
    190, 18, 757, 802, 929, 352,
    472, 901, 119, 954, 760, 105,
    203, 640, 154, 935, 339, 246,
    579, 50, 567, 678, 251, 986,
    59, 522, 384, 117, 199, 73,
    128, 945, 825, 839, 370, 401,
    545, 624, 24, 8, 629, 652,
    678, 614, 695, 9, 457, 477,
    // Bin capacity
    1301, 1616, 1521, 1740, 1562, 1374
  };
  const int i6_18_6_6_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    630, 12, 724, 293, 947, 300,
    98, 158, 852, 567, 513, 598,
    375, 431, 881, 936, 58, 336,
    8, 752, 339, 107, 327, 379,
    262, 854, 408, 128, 786, 364,
    533, 58, 533, 879, 362, 418,
    694, 220, 109, 725, 407, 167,
    730, 523, 107, 499, 619, 802,
    655, 111, 690, 997, 173, 205,
    582, 708, 832, 864, 999, 47,
    546, 997, 427, 564, 358, 894,
    626, 692, 684, 964, 73, 678,
    27, 401, 29, 732, 912, 993,
    815, 413, 968, 206, 172, 379,
    416, 20, 380, 999, 257, 710,
    339, 169, 652, 658, 780, 132,
    44, 607, 701, 722, 752, 834,
    647, 56, 625, 355, 41, 177,
    // Bin capacity
    1419, 1269, 1757, 1978, 1509, 1487
  };
  const int i6_18_6_7_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    560, 146, 509, 127, 280, 696,
    7, 128, 203, 262, 771, 537,
    700, 870, 623, 9, 329, 834,
    736, 768, 666, 198, 446, 818,
    227, 699, 529, 62, 197, 584,
    596, 564, 473, 964, 825, 345,
    976, 659, 850, 986, 478, 569,
    667, 171, 216, 853, 859, 476,
    185, 509, 884, 751, 252, 25,
    14, 789, 127, 690, 102, 130,
    473, 311, 753, 166, 600, 97,
    613, 917, 413, 616, 815, 591,
    109, 604, 374, 563, 794, 98,
    515, 33, 840, 889, 466, 444,
    674, 727, 409, 258, 582, 563,
    842, 167, 554, 935, 428, 625,
    716, 923, 423, 756, 188, 579,
    402, 253, 974, 754, 706, 328,
    // Bin capacity
    1608, 1648, 1752, 1755, 1627, 1488
  };
  const int i6_18_6_8_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    493, 723, 674, 67, 151, 329,
    695, 248, 715, 47, 481, 317,
    872, 151, 800, 773, 963, 564,
    715, 986, 767, 307, 277, 334,
    515, 583, 934, 98, 267, 778,
    459, 378, 407, 668, 660, 139,
    762, 820, 476, 179, 770, 752,
    392, 252, 985, 577, 175, 518,
    229, 406, 250, 471, 647, 120,
    152, 426, 1, 994, 670, 884,
    551, 559, 0, 281, 299, 274,
    885, 400, 64, 463, 244, 261,
    107, 724, 367, 877, 573, 397,
    951, 719, 12, 996, 316, 87,
    525, 895, 45, 396, 449, 803,
    360, 186, 434, 775, 304, 603,
    274, 76, 310, 674, 428, 569,
    363, 531, 522, 979, 112, 522,
    // Bin capacity
    1674, 1632, 1398, 1732, 1402, 1486
  };
  const int i6_18_6_9_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    945, 85, 273, 505, 702, 62,
    196, 712, 996, 642, 141, 51,
    520, 905, 526, 777, 915, 810,
    732, 586, 197, 295, 588, 42,
    844, 832, 618, 732, 990, 842,
    830, 373, 310, 169, 639, 117,
    834, 172, 326, 400, 720, 789,
    896, 882, 11, 112, 980, 505,
    652, 47, 2, 588, 689, 701,
    300, 770, 663, 708, 685, 573,
    879, 729, 516, 277, 938, 283,
    130, 599, 248, 314, 674, 817,
    593, 985, 529, 441, 703, 104,
    622, 174, 12, 396, 330, 130,
    951, 604, 750, 197, 75, 19,
    358, 679, 709, 298, 777, 389,
    684, 14, 260, 233, 819, 466,
    764, 129, 973, 100, 361, 143,
    // Bin capacity
    2131, 1686, 1439, 1306, 2131, 1244
  };
  const int i6_18_6_10_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    133, 335, 374, 488, 344, 285,
    240, 858, 250, 920, 451, 883,
    261, 525, 535, 366, 916, 822,
    206, 54, 158, 763, 49, 674,
    752, 338, 670, 805, 798, 872,
    112, 828, 611, 483, 734, 192,
    829, 87, 633, 157, 541, 468,
    189, 450, 117, 14, 560, 374,
    982, 242, 680, 412, 370, 631,
    35, 396, 773, 599, 649, 327,
    762, 18, 861, 366, 613, 901,
    147, 434, 534, 686, 807, 772,
    422, 421, 545, 351, 940, 772,
    634, 98, 500, 166, 721, 885,
    938, 19, 224, 142, 139, 775,
    666, 769, 702, 469, 993, 134,
    765, 455, 458, 255, 527, 790,
    386, 827, 250, 559, 385, 23,
    // Bin capacity
    1551, 1312, 1628, 1467, 1932, 1940
  };
  const int i6_18_6_11_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    833, 507, 735, 896, 536, 793,
    745, 715, 335, 333, 668, 483,
    359, 334, 351, 917, 430, 114,
    451, 622, 930, 106, 362, 467,
    990, 338, 682, 411, 483, 780,
    689, 668, 567, 243, 461, 979,
    695, 309, 714, 942, 76, 795,
    905, 185, 703, 895, 424, 897,
    539, 55, 655, 993, 438, 606,
    253, 661, 113, 944, 996, 492,
    802, 775, 495, 179, 697, 124,
    782, 20, 88, 62, 967, 602,
    11, 817, 882, 853, 42, 121,
    553, 478, 365, 988, 757, 781,
    507, 716, 466, 753, 929, 650,
    620, 161, 403, 949, 198, 907,
    223, 75, 892, 71, 289, 625,
    428, 348, 639, 746, 802, 171,
    // Bin capacity
    1922, 1441, 1853, 2087, 1768, 1922
  };
  const int i6_18_6_12_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    342, 891, 964, 94, 845, 824,
    590, 83, 419, 953, 6, 274,
    126, 581, 570, 855, 434, 235,
    373, 534, 566, 986, 910, 248,
    295, 61, 474, 459, 683, 546,
    563, 488, 487, 338, 65, 459,
    140, 549, 272, 438, 222, 653,
    204, 78, 606, 48, 922, 101,
    304, 928, 740, 130, 697, 205,
    429, 860, 561, 342, 960, 130,
    912, 260, 349, 238, 816, 966,
    67, 75, 643, 467, 13, 291,
    277, 128, 707, 581, 779, 307,
    515, 966, 186, 249, 567, 514,
    127, 972, 24, 291, 255, 893,
    65, 165, 361, 361, 159, 378,
    404, 377, 0, 547, 73, 555,
    207, 982, 292, 501, 702, 732,
    // Bin capacity
    1109, 1676, 1535, 1471, 1701, 1552
  };
  const int i6_18_6_13_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    529, 634, 409, 258, 94, 628,
    30, 956, 119, 698, 173, 128,
    533, 587, 683, 428, 439, 892,
    248, 26, 317, 459, 859, 245,
    457, 917, 127, 261, 794, 926,
    688, 762, 905, 503, 768, 219,
    504, 499, 529, 407, 440, 374,
    805, 469, 924, 338, 294, 550,
    468, 831, 226, 994, 988, 780,
    117, 451, 912, 7, 536, 932,
    457, 133, 402, 184, 75, 109,
    317, 630, 911, 481, 595, 533,
    203, 225, 180, 438, 831, 605,
    722, 207, 725, 338, 933, 870,
    741, 214, 145, 225, 554, 268,
    540, 870, 193, 984, 564, 940,
    683, 383, 459, 649, 880, 101,
    982, 597, 350, 430, 21, 903,
    // Bin capacity
    1700, 1769, 1604, 1523, 1853, 1884
  };
  const int i6_18_6_14_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    539, 612, 446, 537, 599, 265,
    395, 385, 162, 637, 419, 291,
    502, 553, 766, 663, 410, 453,
    523, 102, 332, 503, 808, 650,
    985, 861, 802, 29, 587, 170,
    884, 720, 629, 761, 667, 333,
    44, 667, 822, 448, 422, 773,
    741, 88, 938, 480, 820, 822,
    126, 0, 89, 821, 515, 440,
    974, 607, 175, 429, 850, 242,
    840, 335, 299, 190, 726, 318,
    98, 356, 298, 344, 277, 93,
    896, 303, 106, 564, 785, 137,
    855, 435, 828, 749, 459, 599,
    332, 378, 69, 102, 874, 842,
    667, 636, 735, 129, 235, 845,
    972, 427, 33, 870, 268, 337,
    577, 238, 815, 245, 197, 40,
    // Bin capacity
    2081, 1464, 1586, 1616, 1885, 1454
  };
  const int i6_18_6_15_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    807, 77, 142, 470, 820, 383,
    839, 815, 304, 456, 545, 532,
    677, 966, 282, 580, 263, 297,
    276, 444, 344, 418, 180, 25,
    324, 888, 128, 707, 426, 814,
    413, 287, 902, 944, 589, 510,
    681, 54, 122, 368, 737, 847,
    304, 282, 724, 458, 446, 990,
    743, 884, 207, 698, 755, 998,
    883, 932, 625, 720, 352, 833,
    775, 38, 647, 853, 779, 18,
    180, 634, 542, 16, 926, 721,
    894, 124, 136, 771, 835, 519,
    202, 622, 730, 829, 120, 360,
    653, 632, 676, 197, 158, 258,
    7, 860, 872, 486, 679, 409,
    754, 858, 65, 470, 465, 717,
    568, 684, 316, 739, 142, 69,
    // Bin capacity
    1913, 1933, 1489, 1952, 1767, 1783
  };
  const int i6_18_6_16_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    489, 172, 646, 767, 107, 755,
    306, 142, 614, 500, 55, 257,
    100, 18, 111, 35, 508, 606,
    228, 268, 88, 59, 631, 18,
    852, 290, 825, 831, 813, 949,
    310, 342, 590, 53, 898, 313,
    424, 424, 974, 625, 216, 947,
    129, 896, 779, 798, 316, 159,
    107, 959, 139, 996, 226, 275,
    669, 770, 942, 186, 340, 511,
    358, 664, 954, 754, 786, 233,
    101, 511, 777, 678, 582, 306,
    361, 350, 319, 813, 653, 237,
    803, 808, 221, 170, 304, 336,
    11, 966, 424, 910, 395, 493,
    863, 150, 241, 488, 327, 17,
    234, 988, 888, 984, 336, 561,
    623, 274, 849, 869, 77, 214,
    // Bin capacity
    1348, 1739, 2007, 2034, 1464, 1390
  };
  const int i6_18_6_17_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    835, 0, 428, 647, 95, 157,
    262, 198, 585, 721, 703, 640,
    458, 631, 940, 335, 709, 512,
    382, 502, 482, 383, 879, 659,
    501, 908, 880, 668, 771, 428,
    534, 824, 12, 348, 495, 612,
    861, 484, 298, 579, 695, 870,
    486, 53, 863, 252, 110, 750,
    207, 98, 160, 864, 331, 156,
    427, 176, 730, 166, 252, 299,
    290, 910, 452, 412, 546, 392,
    853, 138, 41, 864, 433, 755,
    464, 744, 648, 15, 878, 298,
    796, 687, 454, 477, 752, 503,
    511, 303, 419, 868, 813, 620,
    473, 605, 594, 846, 380, 641,
    551, 421, 627, 94, 95, 355,
    227, 586, 276, 754, 618, 475,
    // Bin capacity
    1779, 1613, 1734, 1813, 1864, 1779
  };
  const int i6_18_6_18_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    623, 990, 104, 572, 231, 794,
    459, 847, 585, 743, 426, 999,
    665, 142, 650, 756, 487, 864,
    464, 210, 377, 509, 737, 360,
    937, 586, 68, 827, 545, 858,
    402, 665, 918, 213, 830, 582,
    591, 332, 912, 947, 707, 512,
    827, 718, 354, 75, 903, 888,
    244, 523, 675, 101, 288, 792,
    39, 75, 873, 228, 520, 498,
    268, 90, 285, 800, 675, 158,
    352, 125, 440, 556, 456, 844,
    848, 128, 814, 574, 392, 277,
    388, 787, 785, 29, 722, 341,
    161, 582, 177, 699, 239, 704,
    30, 736, 696, 121, 339, 367,
    482, 822, 271, 795, 955, 482,
    171, 310, 707, 192, 164, 601,
    // Bin capacity
    1564, 1705, 1906, 1719, 1892, 2148
  };
  const int i6_18_6_19_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    283, 751, 817, 728, 964, 50,
    251, 602, 287, 537, 367, 893,
    244, 625, 900, 126, 189, 677,
    102, 765, 596, 741, 395, 501,
    440, 781, 247, 302, 331, 576,
    714, 572, 665, 400, 835, 270,
    423, 348, 294, 558, 6, 946,
    274, 316, 677, 177, 189, 874,
    966, 236, 742, 327, 46, 567,
    477, 708, 457, 857, 902, 788,
    306, 478, 898, 659, 63, 76,
    698, 932, 687, 375, 385, 875,
    106, 752, 898, 268, 535, 275,
    456, 936, 342, 657, 852, 647,
    317, 526, 718, 767, 94, 37,
    39, 242, 947, 514, 107, 951,
    423, 420, 946, 89, 868, 848,
    397, 544, 298, 712, 42, 791,
    // Bin capacity
    1372, 2090, 2265, 1745, 1423, 2111
  };
  const int i6_18_6_20_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    99, 539, 893, 138, 163, 857,
    899, 359, 227, 550, 933, 890,
    277, 88, 955, 19, 910, 136,
    375, 418, 815, 802, 994, 935,
    613, 248, 698, 301, 89, 547,
    515, 539, 619, 99, 491, 586,
    951, 576, 299, 926, 30, 32,
    433, 146, 542, 987, 481, 22,
    395, 892, 298, 747, 956, 718,
    625, 59, 407, 401, 658, 155,
    683, 210, 268, 513, 296, 305,
    17, 707, 661, 406, 786, 501,
    626, 936, 221, 135, 888, 637,
    53, 41, 519, 628, 985, 627,
    920, 343, 372, 556, 478, 781,
    803, 983, 770, 606, 413, 964,
    956, 317, 309, 560, 813, 832,
    398, 334, 742, 353, 568, 802,
    // Bin capacity
    1928, 1547, 1923, 1746, 2187, 2066
  };
  const int i6_18_6_21_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    483, 161, 833, 982, 606, 558,
    955, 611, 926, 254, 916, 356,
    299, 258, 792, 793, 830, 952,
    259, 348, 705, 478, 474, 830,
    507, 303, 103, 45, 686, 377,
    298, 666, 876, 300, 531, 446,
    312, 182, 270, 446, 86, 61,
    854, 846, 644, 147, 114, 554,
    634, 585, 226, 983, 442, 773,
    264, 125, 1, 811, 636, 562,
    330, 136, 652, 973, 155, 71,
    309, 56, 812, 733, 986, 649,
    597, 357, 438, 501, 83, 678,
    581, 113, 6, 699, 995, 456,
    458, 133, 576, 534, 605, 783,
    140, 942, 144, 769, 561, 738,
    520, 680, 387, 540, 333, 71,
    839, 733, 324, 999, 231, 978,
    // Bin capacity
    1743, 1460, 1758, 2216, 1870, 1996
  };
  const int i6_18_6_22_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    212, 592, 324, 395, 382, 858,
    907, 178, 846, 770, 571, 759,
    823, 532, 556, 903, 728, 846,
    833, 475, 406, 110, 618, 704,
    573, 731, 105, 359, 760, 350,
    564, 982, 927, 779, 605, 585,
    779, 316, 311, 311, 574, 786,
    183, 879, 176, 597, 38, 180,
    918, 916, 729, 970, 83, 861,
    153, 531, 624, 250, 779, 82,
    345, 966, 345, 119, 107, 984,
    17, 374, 773, 694, 330, 660,
    351, 815, 945, 77, 852, 407,
    331, 438, 858, 622, 970, 364,
    207, 45, 270, 511, 455, 410,
    357, 711, 137, 278, 162, 401,
    646, 165, 606, 638, 292, 870,
    10, 724, 374, 83, 188, 431,
    // Bin capacity
    1670, 2109, 1894, 1722, 1728, 2143
  };
  const int i6_18_6_23_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    887, 598, 758, 65, 904, 889,
    386, 491, 115, 880, 848, 541,
    395, 650, 969, 409, 943, 467,
    231, 442, 560, 985, 826, 794,
    229, 369, 459, 754, 480, 835,
    612, 443, 401, 690, 115, 192,
    159, 599, 527, 882, 65, 243,
    508, 791, 57, 226, 630, 979,
    773, 560, 808, 98, 434, 418,
    424, 840, 848, 689, 618, 634,
    968, 140, 690, 546, 883, 429,
    266, 190, 563, 40, 623, 738,
    641, 86, 412, 803, 563, 42,
    871, 114, 353, 55, 495, 377,
    99, 446, 989, 700, 398, 673,
    393, 546, 343, 615, 686, 244,
    655, 540, 877, 441, 200, 939,
    867, 510, 161, 752, 115, 205,
    // Bin capacity
    1920, 1713, 2028, 1975, 2015, 1976
  };
  const int i6_18_6_24_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    689, 206, 156, 932, 632, 799,
    807, 899, 430, 84, 576, 668,
    47, 135, 808, 79, 940, 116,
    261, 363, 98, 271, 23, 395,
    713, 26, 371, 600, 792, 419,
    146, 73, 545, 647, 26, 868,
    668, 431, 738, 814, 522, 658,
    717, 260, 784, 748, 106, 766,
    254, 132, 451, 7, 232, 377,
    381, 298, 917, 466, 203, 204,
    617, 105, 910, 3, 142, 240,
    438, 702, 671, 623, 809, 549,
    316, 100, 394, 44, 827, 73,
    6, 609, 844, 1, 21, 19,
    421, 376, 394, 318, 184, 167,
    764, 668, 837, 879, 597, 965,
    433, 907, 388, 92, 618, 908,
    98, 711, 762, 100, 380, 594,
    // Bin capacity
    1608, 1447, 2170, 1387, 1577, 1816
  };
  const int i6_18_6_25_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    724, 971, 263, 488, 890, 800,
    565, 746, 317, 162, 133, 547,
    412, 986, 973, 671, 134, 499,
    853, 536, 841, 712, 183, 810,
    484, 713, 272, 167, 901, 789,
    162, 791, 448, 336, 206, 947,
    602, 375, 662, 509, 2, 9,
    358, 563, 88, 32, 964, 181,
    301, 392, 661, 481, 644, 390,
    882, 128, 317, 186, 553, 172,
    106, 218, 674, 8, 960, 317,
    773, 508, 978, 680, 948, 822,
    112, 874, 333, 561, 383, 649,
    828, 321, 323, 420, 69, 772,
    77, 132, 314, 357, 63, 480,
    938, 139, 542, 853, 69, 303,
    111, 536, 422, 728, 184, 474,
    239, 680, 767, 281, 573, 202,
    // Bin capacity
    1777, 2002, 1916, 1590, 1638, 1909
  };
  const int i6_18_6_26_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    897, 351, 761, 838, 674, 587,
    792, 231, 638, 659, 347, 316,
    445, 699, 204, 703, 596, 783,
    683, 504, 670, 112, 856, 59,
    885, 320, 685, 720, 98, 358,
    317, 705, 808, 540, 846, 193,
    899, 895, 627, 693, 770, 151,
    340, 779, 286, 578, 382, 89,
    252, 525, 410, 275, 545, 29,
    147, 686, 842, 713, 212, 974,
    280, 338, 404, 524, 554, 672,
    134, 572, 861, 170, 980, 313,
    878, 493, 561, 222, 630, 655,
    524, 547, 988, 799, 347, 526,
    60, 642, 668, 444, 50, 16,
    784, 52, 821, 285, 188, 261,
    276, 896, 707, 982, 471, 174,
    939, 267, 829, 709, 909, 920,
    // Bin capacity
    2002, 1996, 2472, 2093, 1986, 1486
  };
  const int i6_18_6_27_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    933, 63, 102, 292, 612, 373,
    960, 171, 680, 830, 311, 407,
    384, 867, 296, 419, 167, 180,
    170, 859, 273, 866, 727, 28,
    968, 660, 793, 383, 282, 498,
    627, 518, 557, 402, 495, 413,
    798, 322, 815, 533, 577, 411,
    129, 785, 507, 325, 669, 319,
    777, 529, 942, 60, 362, 755,
    941, 842, 338, 703, 99, 544,
    421, 877, 60, 941, 14, 196,
    550, 12, 145, 101, 603, 972,
    178, 371, 289, 278, 485, 593,
    897, 1000, 147, 227, 206, 231,
    71, 42, 999, 324, 272, 708,
    289, 279, 541, 995, 842, 228,
    325, 751, 348, 797, 439, 26,
    269, 492, 910, 764, 498, 271,
    // Bin capacity
    2051, 1999, 1851, 1956, 1622, 1515
  };
  const int i6_18_6_28_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    895, 296, 839, 104, 758, 577,
    730, 514, 759, 535, 389, 744,
    744, 672, 665, 762, 480, 46,
    3, 854, 95, 191, 423, 857,
    272, 292, 297, 256, 100, 666,
    1000, 701, 280, 924, 878, 718,
    299, 105, 309, 671, 667, 597,
    100, 369, 842, 961, 440, 914,
    166, 562, 340, 509, 90, 65,
    133, 430, 586, 633, 613, 551,
    807, 8, 772, 269, 71, 132,
    492, 468, 838, 963, 790, 384,
    103, 370, 898, 19, 680, 17,
    216, 548, 406, 393, 281, 719,
    854, 470, 211, 671, 582, 330,
    548, 64, 723, 62, 75, 572,
    820, 799, 877, 466, 452, 911,
    798, 742, 503, 229, 858, 758,
    // Bin capacity
    1916, 1763, 2185, 1839, 1841, 2040
  };
  const int i6_18_6_29_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    856, 134, 951, 144, 909, 623,
    243, 53, 311, 473, 604, 282,
    639, 1000, 443, 618, 994, 298,
    782, 110, 897, 783, 311, 847,
    831, 590, 886, 510, 921, 467,
    948, 762, 542, 673, 469, 864,
    634, 426, 778, 654, 883, 966,
    562, 259, 209, 575, 646, 18,
    282, 902, 977, 2, 400, 433,
    225, 368, 312, 666, 565, 854,
    829, 144, 998, 760, 543, 261,
    806, 66, 15, 808, 39, 662,
    464, 361, 761, 171, 503, 787,
    678, 487, 667, 150, 94, 522,
    707, 973, 711, 987, 840, 491,
    387, 35, 436, 742, 547, 144,
    145, 286, 353, 19, 991, 116,
    793, 697, 837, 519, 426, 674,
    // Bin capacity
    2325, 1646, 2384, 1990, 2298, 2002
  };
  const int i6_18_6_30_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    385, 630, 489, 492, 849, 301,
    398, 161, 657, 741, 11, 37,
    389, 199, 917, 348, 380, 329,
    305, 153, 630, 7, 126, 321,
    507, 610, 966, 879, 716, 592,
    901, 118, 312, 206, 708, 631,
    942, 711, 354, 172, 873, 120,
    625, 733, 90, 895, 507, 648,
    708, 191, 533, 829, 850, 982,
    3, 59, 485, 619, 973, 500,
    510, 21, 775, 398, 122, 281,
    23, 583, 341, 328, 800, 953,
    62, 661, 532, 569, 310, 165,
    246, 27, 633, 851, 947, 625,
    256, 217, 657, 745, 571, 423,
    226, 702, 664, 549, 874, 18,
    575, 781, 793, 318, 518, 563,
    61, 878, 714, 262, 849, 417,
    // Bin capacity
    1544, 1611, 2285, 1996, 2380, 1713
  };
  const int i6_18_6_31_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    758, 904, 368, 983, 368, 536,
    916, 337, 520, 360, 639, 489,
    215, 655, 428, 906, 430, 767,
    249, 96, 236, 618, 425, 218,
    709, 374, 345, 617, 263, 901,
    302, 215, 184, 632, 659, 115,
    339, 778, 455, 639, 388, 691,
    866, 520, 474, 40, 465, 863,
    306, 404, 367, 875, 344, 597,
    536, 573, 352, 66, 437, 300,
    496, 904, 163, 527, 995, 902,
    875, 410, 515, 454, 540, 552,
    828, 881, 59, 566, 203, 354,
    503, 411, 156, 81, 918, 441,
    685, 151, 606, 98, 61, 69,
    632, 878, 730, 962, 12, 625,
    198, 805, 21, 128, 734, 674,
    407, 254, 71, 59, 812, 205,
    // Bin capacity
    2145, 2086, 1321, 1881, 1898, 2031
  };
  const int i6_18_6_32_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    16, 501, 622, 271, 720, 226,
    9, 373, 355, 887, 260, 787,
    288, 198, 438, 179, 237, 464,
    125, 696, 800, 426, 708, 637,
    735, 485, 234, 718, 331, 186,
    990, 980, 262, 150, 460, 617,
    601, 349, 965, 737, 633, 669,
    505, 384, 70, 735, 393, 788,
    292, 352, 443, 286, 437, 59,
    62, 157, 652, 348, 30, 234,
    179, 918, 433, 687, 352, 719,
    760, 687, 484, 342, 983, 583,
    744, 774, 654, 415, 445, 732,
    503, 296, 922, 288, 28, 484,
    836, 63, 957, 144, 344, 961,
    326, 183, 699, 262, 977, 631,
    521, 486, 847, 135, 884, 982,
    336, 955, 680, 653, 3, 453,
    // Bin capacity
    1723, 1945, 2314, 1686, 1810, 2247
  };
  const int i6_18_6_33_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    17, 431, 962, 571, 666, 798,
    664, 861, 207, 483, 251, 893,
    48, 343, 937, 762, 504, 429,
    845, 468, 361, 123, 480, 632,
    328, 350, 142, 345, 259, 896,
    758, 31, 118, 363, 842, 685,
    454, 916, 561, 469, 651, 321,
    446, 327, 173, 262, 956, 104,
    208, 208, 545, 742, 691, 936,
    938, 265, 783, 238, 927, 992,
    668, 21, 324, 629, 968, 914,
    936, 961, 872, 27, 930, 479,
    173, 671, 162, 700, 733, 917,
    262, 814, 13, 608, 125, 468,
    888, 830, 292, 604, 965, 244,
    735, 671, 127, 429, 89, 817,
    433, 563, 511, 558, 76, 593,
    829, 368, 522, 365, 941, 919,
    // Bin capacity
    2135, 2017, 1688, 1835, 2451, 2669
  };
  const int i6_18_6_34_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    82, 353, 753, 43, 304, 544,
    601, 974, 879, 825, 136, 365,
    378, 704, 155, 949, 206, 874,
    84, 229, 216, 195, 965, 262,
    600, 636, 39, 554, 156, 607,
    453, 924, 487, 744, 165, 764,
    222, 812, 232, 760, 462, 45,
    199, 878, 496, 389, 665, 904,
    777, 556, 595, 985, 568, 86,
    218, 982, 586, 207, 573, 532,
    64, 81, 860, 95, 579, 272,
    812, 467, 724, 522, 743, 696,
    810, 143, 553, 661, 310, 501,
    201, 75, 441, 532, 146, 693,
    328, 468, 204, 280, 930, 970,
    645, 573, 161, 302, 541, 760,
    657, 904, 818, 697, 275, 616,
    504, 806, 413, 137, 123, 289,
    // Bin capacity
    1706, 2360, 1924, 1983, 1753, 2185
  };
  const int i6_18_6_35_14[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    966, 784, 198, 839, 234, 145,
    735, 66, 691, 575, 173, 937,
    24, 153, 433, 832, 760, 101,
    307, 663, 519, 591, 767, 134,
    189, 889, 91, 977, 423, 547,
    803, 386, 986, 598, 530, 81,
    990, 336, 329, 826, 773, 587,
    586, 651, 276, 920, 461, 182,
    455, 499, 601, 517, 716, 524,
    331, 154, 970, 222, 589, 716,
    470, 33, 676, 784, 155, 541,
    711, 467, 997, 633, 231, 493,
    109, 792, 397, 412, 702, 191,
    113, 432, 746, 53, 786, 832,
    433, 328, 121, 644, 927, 353,
    215, 978, 889, 315, 108, 188,
    872, 266, 99, 822, 766, 555,
    547, 703, 977, 49, 374, 683,
    // Bin capacity
    1993, 1931, 2250, 2388, 2132, 1753
  };
  const int i6_18_6_0_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    0, 878, 678, 606, 939, 429,
    464, 670, 470, 189, 807, 475,
    683, 344, 713, 654, 445, 714,
    639, 346, 751, 934, 370, 320,
    81, 8, 67, 60, 858, 761,
    19, 766, 783, 693, 953, 511,
    433, 140, 413, 91, 293, 101,
    555, 606, 236, 337, 255, 348,
    43, 294, 128, 922, 222, 293,
    203, 854, 820, 941, 75, 409,
    281, 39, 842, 101, 512, 218,
    472, 308, 998, 670, 102, 857,
    832, 129, 915, 441, 184, 718,
    84, 277, 880, 180, 560, 923,
    706, 574, 806, 590, 370, 443,
    367, 519, 503, 884, 795, 480,
    674, 202, 246, 763, 691, 256,
    399, 407, 301, 752, 215, 72,
    // Bin capacity
    1156, 1227, 1759, 1635, 1441, 1388
  };
  const int i6_18_6_1_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    939, 622, 369, 178, 803, 692,
    337, 997, 26, 698, 363, 614,
    405, 335, 788, 50, 97, 856,
    621, 291, 19, 238, 303, 530,
    550, 659, 817, 142, 154, 920,
    124, 538, 412, 813, 870, 222,
    241, 833, 21, 12, 865, 113,
    717, 500, 264, 662, 976, 931,
    117, 732, 538, 268, 716, 353,
    602, 747, 556, 351, 912, 73,
    872, 820, 645, 268, 194, 269,
    750, 178, 135, 544, 781, 400,
    84, 518, 141, 646, 132, 275,
    203, 682, 269, 738, 771, 45,
    407, 569, 197, 679, 232, 113,
    123, 345, 398, 298, 298, 835,
    934, 139, 144, 362, 862, 716,
    615, 535, 206, 419, 925, 548,
    // Bin capacity
    1455, 1691, 1001, 1240, 1727, 1432
  };
  const int i6_18_6_2_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    254, 497, 161, 775, 603, 610,
    363, 488, 582, 45, 54, 727,
    665, 992, 819, 226, 885, 167,
    415, 683, 93, 103, 471, 459,
    170, 910, 853, 161, 773, 49,
    550, 474, 871, 860, 677, 242,
    907, 688, 923, 525, 3, 840,
    75, 509, 725, 822, 126, 240,
    42, 373, 457, 69, 855, 477,
    480, 448, 945, 58, 86, 903,
    672, 932, 961, 991, 671, 637,
    601, 852, 486, 420, 9, 922,
    20, 48, 548, 304, 822, 945,
    911, 800, 953, 724, 57, 390,
    708, 617, 712, 404, 810, 562,
    804, 455, 675, 849, 594, 743,
    183, 134, 401, 774, 149, 256,
    818, 671, 115, 845, 827, 56,
    // Bin capacity
    1469, 1798, 1918, 1523, 1441, 1569
  };
  const int i6_18_6_3_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    541, 127, 773, 996, 629, 758,
    630, 253, 211, 989, 86, 174,
    712, 81, 638, 420, 750, 638,
    432, 92, 424, 614, 868, 838,
    595, 203, 306, 237, 195, 660,
    731, 998, 804, 874, 141, 13,
    170, 679, 175, 45, 166, 764,
    621, 813, 525, 553, 195, 139,
    350, 297, 12, 510, 414, 846,
    937, 42, 903, 400, 549, 132,
    12, 468, 977, 244, 922, 126,
    132, 472, 345, 93, 172, 573,
    80, 463, 690, 697, 623, 246,
    850, 183, 597, 141, 269, 358,
    723, 74, 630, 343, 457, 269,
    850, 836, 953, 23, 779, 674,
    215, 420, 928, 647, 459, 272,
    488, 939, 102, 759, 818, 973,
    // Bin capacity
    1557, 1278, 1716, 1474, 1458, 1452
  };
  const int i6_18_6_4_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    469, 653, 813, 344, 375, 109,
    795, 971, 680, 638, 564, 528,
    582, 163, 712, 574, 653, 850,
    907, 594, 634, 802, 426, 753,
    46, 954, 420, 192, 269, 26,
    235, 283, 833, 58, 496, 26,
    981, 29, 479, 775, 796, 142,
    310, 435, 165, 13, 459, 141,
    967, 554, 169, 105, 65, 201,
    561, 154, 327, 531, 498, 154,
    433, 410, 456, 144, 341, 415,
    14, 307, 553, 25, 279, 601,
    172, 544, 426, 908, 481, 308,
    931, 565, 892, 754, 160, 112,
    927, 830, 632, 929, 179, 42,
    584, 338, 463, 432, 912, 586,
    857, 786, 529, 828, 781, 942,
    540, 582, 761, 647, 645, 536,
    // Bin capacity
    1788, 1587, 1724, 1508, 1453, 1122
  };
  const int i6_18_6_5_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    191, 420, 687, 15, 50, 543,
    531, 74, 212, 843, 830, 603,
    165, 413, 9, 440, 870, 353,
    823, 548, 978, 24, 805, 17,
    903, 288, 992, 641, 873, 116,
    518, 145, 953, 160, 125, 489,
    52, 774, 684, 626, 748, 655,
    985, 995, 569, 432, 15, 931,
    361, 790, 547, 769, 314, 473,
    715, 212, 818, 124, 642, 674,
    386, 526, 112, 937, 4, 40,
    13, 68, 926, 838, 571, 754,
    498, 657, 142, 671, 658, 945,
    802, 18, 902, 651, 234, 615,
    140, 410, 169, 331, 534, 157,
    780, 915, 333, 774, 1000, 674,
    938, 874, 994, 686, 524, 460,
    821, 162, 833, 93, 506, 767,
    // Bin capacity
    1684, 1451, 1901, 1585, 1629, 1622
  };
  const int i6_18_6_6_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    907, 836, 490, 304, 938, 388,
    39, 234, 354, 173, 682, 809,
    267, 541, 286, 59, 48, 351,
    784, 884, 299, 710, 96, 617,
    433, 320, 805, 401, 32, 109,
    161, 178, 21, 255, 498, 715,
    338, 867, 641, 425, 364, 824,
    941, 172, 463, 525, 767, 960,
    335, 796, 819, 4, 277, 970,
    782, 968, 585, 962, 497, 300,
    830, 70, 686, 750, 891, 725,
    558, 441, 110, 20, 292, 302,
    432, 949, 508, 335, 77, 828,
    153, 921, 147, 699, 217, 666,
    806, 892, 928, 254, 31, 265,
    812, 835, 613, 449, 315, 77,
    6, 928, 986, 923, 781, 773,
    861, 679, 44, 808, 484, 758,
    // Bin capacity
    1669, 2034, 1553, 1424, 1288, 1844
  };
  const int i6_18_6_7_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    28, 867, 391, 960, 812, 927,
    467, 658, 176, 885, 177, 911,
    90, 989, 825, 439, 36, 88,
    508, 44, 512, 960, 782, 983,
    534, 992, 427, 507, 331, 514,
    722, 312, 692, 548, 248, 769,
    360, 575, 642, 86, 594, 526,
    29, 249, 237, 174, 960, 954,
    514, 182, 385, 541, 556, 546,
    962, 419, 686, 978, 258, 710,
    156, 153, 645, 129, 308, 300,
    898, 669, 950, 773, 938, 486,
    14, 956, 105, 834, 949, 137,
    164, 898, 253, 666, 443, 867,
    790, 732, 19, 689, 822, 567,
    295, 678, 873, 133, 335, 124,
    198, 286, 656, 90, 505, 142,
    564, 278, 724, 511, 736, 982,
    // Bin capacity
    1301, 1773, 1641, 1767, 1746, 1879
  };
  const int i6_18_6_8_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    490, 647, 560, 829, 118, 212,
    602, 543, 735, 424, 539, 835,
    529, 764, 703, 824, 481, 531,
    679, 182, 350, 569, 421, 135,
    247, 132, 10, 349, 722, 471,
    138, 59, 284, 491, 107, 410,
    850, 185, 347, 669, 952, 286,
    26, 784, 914, 947, 597, 369,
    981, 77, 933, 520, 169, 889,
    415, 587, 988, 295, 736, 931,
    198, 459, 845, 510, 2, 294,
    61, 779, 769, 133, 324, 806,
    584, 718, 487, 136, 414, 92,
    948, 479, 131, 607, 28, 134,
    810, 213, 77, 523, 221, 555,
    651, 462, 216, 386, 228, 744,
    273, 321, 406, 896, 465, 195,
    977, 106, 511, 190, 558, 888,
    // Bin capacity
    1703, 1350, 1668, 1674, 1275, 1580
  };
  const int i6_18_6_9_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    869, 770, 138, 162, 739, 477,
    675, 57, 69, 87, 606, 761,
    253, 447, 942, 740, 481, 753,
    284, 774, 456, 643, 815, 319,
    639, 891, 313, 977, 160, 520,
    645, 756, 924, 654, 627, 203,
    760, 77, 449, 69, 795, 720,
    757, 510, 164, 641, 847, 944,
    165, 547, 452, 7, 96, 419,
    527, 874, 522, 353, 484, 445,
    76, 892, 500, 509, 523, 500,
    452, 756, 180, 176, 214, 160,
    199, 847, 211, 416, 256, 162,
    375, 98, 49, 965, 277, 343,
    830, 720, 507, 35, 653, 937,
    2, 114, 198, 935, 768, 947,
    871, 170, 6, 895, 879, 943,
    418, 222, 375, 92, 131, 346,
    // Bin capacity
    1599, 1730, 1173, 1519, 1699, 1799
  };
  const int i6_18_6_10_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    313, 405, 648, 998, 819, 88,
    855, 986, 621, 570, 218, 912,
    375, 54, 414, 881, 976, 390,
    163, 370, 719, 66, 50, 433,
    597, 558, 304, 1000, 136, 540,
    730, 773, 151, 924, 756, 251,
    944, 555, 849, 420, 886, 689,
    976, 618, 813, 81, 946, 566,
    929, 695, 621, 123, 769, 142,
    261, 563, 242, 930, 63, 921,
    525, 100, 827, 82, 866, 558,
    653, 881, 844, 864, 483, 611,
    546, 521, 686, 27, 727, 954,
    803, 303, 687, 483, 666, 945,
    694, 83, 634, 695, 533, 821,
    218, 808, 713, 788, 836, 791,
    176, 143, 994, 155, 118, 223,
    826, 534, 79, 187, 163, 282,
    // Bin capacity
    1941, 1641, 1989, 1701, 1836, 1855
  };
  const int i6_18_6_11_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    630, 991, 610, 770, 989, 616,
    123, 562, 66, 97, 529, 80,
    443, 17, 717, 472, 804, 874,
    862, 86, 831, 61, 932, 987,
    399, 280, 551, 440, 264, 124,
    936, 712, 328, 146, 642, 730,
    867, 7, 76, 510, 554, 611,
    609, 114, 852, 681, 306, 74,
    959, 118, 503, 438, 286, 699,
    921, 491, 38, 55, 789, 587,
    213, 165, 749, 519, 239, 879,
    417, 78, 441, 71, 93, 314,
    116, 248, 176, 852, 487, 277,
    82, 273, 819, 116, 806, 116,
    657, 694, 493, 837, 327, 621,
    759, 446, 296, 658, 513, 738,
    745, 646, 505, 595, 907, 546,
    812, 813, 747, 141, 199, 39,
    // Bin capacity
    1952, 1248, 1628, 1380, 1789, 1649
  };
  const int i6_18_6_12_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    906, 750, 18, 381, 571, 209,
    361, 743, 241, 682, 914, 302,
    459, 360, 220, 971, 565, 451,
    18, 536, 719, 259, 435, 385,
    563, 41, 682, 360, 426, 829,
    638, 582, 149, 330, 270, 141,
    942, 58, 212, 966, 379, 457,
    559, 135, 400, 353, 504, 966,
    428, 369, 766, 892, 122, 349,
    868, 550, 917, 525, 929, 565,
    653, 624, 51, 450, 85, 48,
    366, 176, 914, 89, 915, 628,
    501, 802, 92, 251, 854, 834,
    433, 809, 110, 693, 298, 480,
    171, 24, 701, 284, 682, 453,
    305, 504, 126, 304, 484, 813,
    161, 942, 237, 107, 626, 270,
    327, 16, 85, 400, 548, 729,
    // Bin capacity
    1617, 1498, 1240, 1549, 1794, 1664
  };
  const int i6_18_6_13_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    238, 402, 328, 915, 449, 657,
    280, 54, 989, 443, 644, 995,
    204, 552, 1000, 992, 928, 437,
    69, 152, 57, 590, 727, 221,
    773, 234, 990, 109, 277, 925,
    193, 996, 653, 278, 563, 507,
    833, 156, 434, 135, 885, 841,
    584, 276, 73, 141, 964, 454,
    16, 485, 229, 135, 71, 296,
    614, 117, 587, 642, 210, 64,
    33, 509, 508, 844, 430, 506,
    519, 956, 603, 509, 542, 206,
    780, 47, 221, 313, 133, 310,
    544, 662, 192, 889, 285, 263,
    469, 847, 26, 315, 995, 491,
    571, 293, 983, 11, 210, 264,
    259, 117, 445, 677, 74, 294,
    138, 356, 933, 429, 465, 230,
    // Bin capacity
    1341, 1359, 1743, 1576, 1668, 1500
  };
  const int i6_18_6_14_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    237, 24, 10, 450, 7, 555,
    21, 11, 751, 651, 403, 57,
    583, 687, 932, 989, 414, 605,
    669, 33, 44, 123, 868, 758,
    902, 98, 407, 699, 196, 178,
    600, 187, 731, 599, 742, 983,
    981, 744, 442, 74, 898, 462,
    141, 460, 977, 990, 101, 180,
    383, 989, 404, 351, 58, 703,
    401, 430, 643, 89, 536, 312,
    491, 901, 453, 45, 291, 177,
    86, 358, 713, 139, 849, 797,
    980, 228, 130, 765, 66, 530,
    268, 939, 342, 116, 55, 908,
    560, 845, 245, 232, 3, 559,
    56, 535, 315, 620, 739, 824,
    56, 589, 98, 462, 474, 943,
    451, 428, 873, 250, 412, 204,
    // Bin capacity
    1495, 1613, 1617, 1453, 1352, 1850
  };
  const int i6_18_6_15_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    960, 339, 517, 675, 579, 756,
    745, 954, 14, 596, 102, 786,
    21, 896, 398, 622, 428, 512,
    32, 66, 579, 749, 318, 428,
    706, 833, 960, 924, 447, 597,
    490, 223, 112, 3, 729, 79,
    350, 259, 704, 977, 403, 487,
    49, 932, 766, 999, 703, 333,
    567, 98, 592, 600, 772, 641,
    856, 146, 60, 128, 341, 634,
    788, 266, 256, 315, 34, 1000,
    215, 168, 621, 473, 585, 610,
    19, 315, 330, 228, 409, 122,
    604, 976, 158, 633, 850, 129,
    782, 691, 596, 347, 280, 350,
    758, 115, 298, 796, 963, 860,
    654, 296, 445, 432, 863, 843,
    5, 637, 905, 267, 346, 984,
    // Bin capacity
    1649, 1574, 1593, 1872, 1755, 1946
  };
  const int i6_18_6_16_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    860, 756, 493, 118, 408, 818,
    128, 364, 771, 510, 28, 789,
    287, 852, 819, 457, 511, 636,
    545, 15, 301, 827, 700, 484,
    284, 96, 678, 223, 969, 571,
    26, 636, 218, 937, 992, 847,
    542, 12, 998, 27, 64, 96,
    649, 569, 432, 150, 968, 848,
    314, 99, 570, 17, 909, 588,
    891, 393, 540, 769, 149, 285,
    712, 662, 899, 207, 624, 206,
    638, 927, 258, 34, 789, 918,
    792, 558, 97, 2, 857, 380,
    590, 146, 87, 329, 215, 984,
    546, 705, 953, 856, 278, 191,
    248, 9, 617, 949, 318, 857,
    545, 252, 487, 826, 82, 384,
    922, 833, 411, 126, 951, 623,
    // Bin capacity
    1841, 1525, 1862, 1424, 1897, 2031
  };
  const int i6_18_6_17_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    951, 609, 467, 810, 153, 57,
    72, 457, 392, 272, 556, 823,
    154, 115, 934, 825, 936, 940,
    291, 528, 752, 593, 357, 24,
    981, 736, 341, 444, 911, 491,
    272, 683, 107, 949, 733, 148,
    566, 716, 544, 205, 764, 297,
    506, 358, 369, 108, 213, 345,
    885, 860, 391, 819, 674, 929,
    922, 883, 161, 721, 311, 328,
    267, 713, 589, 521, 757, 994,
    454, 770, 68, 658, 883, 896,
    642, 71, 126, 35, 274, 610,
    153, 670, 208, 847, 161, 629,
    295, 528, 810, 762, 102, 526,
    259, 845, 214, 775, 32, 882,
    585, 188, 549, 189, 297, 673,
    95, 682, 882, 901, 906, 261,
    // Bin capacity
    1629, 2031, 1542, 2035, 1759, 1922
  };
  const int i6_18_6_18_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    905, 305, 13, 524, 166, 786,
    42, 646, 71, 144, 147, 321,
    601, 463, 215, 920, 187, 460,
    970, 36, 387, 780, 182, 77,
    630, 396, 248, 517, 153, 711,
    988, 935, 287, 926, 110, 590,
    334, 732, 332, 418, 196, 97,
    659, 36, 173, 735, 311, 757,
    775, 401, 92, 381, 976, 464,
    822, 338, 768, 400, 888, 165,
    220, 796, 461, 688, 786, 75,
    57, 0, 362, 179, 193, 37,
    333, 651, 627, 601, 634, 865,
    9, 893, 908, 648, 63, 919,
    792, 298, 442, 427, 492, 947,
    371, 646, 302, 649, 680, 513,
    668, 880, 260, 491, 597, 694,
    547, 573, 283, 482, 829, 210,
    // Bin capacity
    1913, 1775, 1226, 1949, 1493, 1709
  };
  const int i6_18_6_19_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    781, 360, 699, 753, 909, 645,
    190, 277, 763, 734, 756, 322,
    454, 113, 710, 448, 199, 830,
    731, 406, 530, 179, 983, 410,
    105, 620, 313, 764, 675, 416,
    366, 526, 345, 616, 625, 190,
    726, 544, 230, 789, 761, 519,
    706, 646, 146, 540, 704, 104,
    928, 580, 857, 689, 907, 953,
    587, 287, 21, 617, 880, 911,
    390, 877, 751, 119, 31, 737,
    926, 24, 17, 313, 603, 932,
    646, 171, 125, 977, 141, 24,
    614, 387, 743, 397, 80, 509,
    363, 90, 131, 107, 997, 608,
    157, 112, 721, 786, 899, 936,
    173, 297, 113, 485, 239, 484,
    187, 378, 633, 426, 951, 490,
    // Bin capacity
    1791, 1328, 1557, 1932, 2250, 1988
  };
  const int i6_18_6_20_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    517, 730, 152, 361, 515, 601,
    80, 687, 910, 807, 0, 942,
    994, 895, 238, 797, 413, 898,
    109, 557, 431, 219, 22, 141,
    178, 321, 382, 47, 853, 377,
    447, 785, 951, 201, 952, 319,
    652, 927, 895, 939, 240, 306,
    99, 763, 899, 92, 766, 342,
    718, 236, 633, 888, 941, 358,
    142, 847, 380, 147, 309, 716,
    399, 351, 313, 320, 119, 614,
    463, 682, 770, 589, 483, 421,
    463, 797, 32, 443, 760, 156,
    54, 889, 422, 289, 377, 785,
    219, 365, 454, 775, 696, 624,
    547, 925, 512, 762, 286, 426,
    681, 865, 124, 590, 183, 716,
    403, 23, 703, 788, 530, 885,
    // Bin capacity
    1433, 2329, 1841, 1811, 1689, 1926
  };
  const int i6_18_6_21_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    624, 819, 653, 448, 738, 306,
    37, 244, 499, 407, 158, 699,
    321, 216, 730, 256, 659, 638,
    537, 393, 124, 362, 327, 972,
    117, 379, 707, 82, 371, 40,
    738, 92, 450, 839, 854, 810,
    816, 96, 628, 268, 850, 142,
    817, 742, 482, 856, 152, 903,
    887, 459, 667, 596, 734, 529,
    874, 640, 969, 806, 668, 966,
    703, 171, 715, 591, 464, 851,
    993, 67, 866, 345, 779, 606,
    837, 95, 185, 838, 231, 381,
    159, 15, 262, 363, 479, 754,
    31, 400, 995, 847, 241, 201,
    168, 831, 549, 487, 712, 452,
    332, 791, 127, 949, 103, 436,
    991, 699, 661, 775, 24, 841,
    // Bin capacity
    2014, 1442, 2071, 2040, 1724, 2123
  };
  const int i6_18_6_22_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    313, 725, 967, 348, 928, 770,
    573, 835, 980, 924, 308, 387,
    149, 99, 874, 197, 757, 802,
    577, 306, 382, 380, 909, 960,
    461, 724, 669, 417, 808, 603,
    417, 14, 152, 382, 431, 368,
    31, 252, 904, 12, 240, 735,
    26, 462, 635, 361, 543, 217,
    754, 327, 613, 930, 901, 850,
    280, 944, 56, 963, 878, 360,
    124, 985, 168, 255, 6, 171,
    269, 207, 820, 212, 672, 626,
    864, 945, 317, 982, 516, 145,
    837, 894, 206, 577, 286, 455,
    201, 618, 92, 106, 61, 308,
    601, 217, 608, 391, 879, 749,
    621, 349, 131, 950, 350, 745,
    432, 436, 289, 40, 441, 791,
    // Bin capacity
    1532, 1899, 1803, 1714, 2016, 2042
  };
  const int i6_18_6_23_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    396, 383, 856, 123, 165, 677,
    935, 17, 599, 342, 511, 171,
    337, 711, 281, 867, 796, 267,
    239, 225, 344, 609, 845, 291,
    182, 152, 298, 781, 117, 399,
    714, 264, 381, 977, 258, 112,
    412, 955, 94, 403, 86, 146,
    956, 51, 598, 542, 630, 973,
    628, 109, 601, 303, 23, 797,
    896, 187, 398, 839, 738, 132,
    177, 807, 270, 223, 379, 474,
    783, 872, 782, 240, 290, 297,
    940, 641, 16, 28, 117, 577,
    698, 599, 322, 415, 812, 324,
    367, 922, 949, 502, 846, 277,
    663, 630, 936, 748, 407, 33,
    177, 387, 121, 110, 515, 685,
    612, 954, 756, 105, 146, 970,
    // Bin capacity
    2073, 1818, 1764, 1673, 1575, 1559
  };
  const int i6_18_6_24_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    679, 47, 84, 731, 65, 221,
    964, 443, 814, 154, 34, 739,
    997, 782, 524, 769, 412, 558,
    452, 376, 133, 19, 998, 199,
    874, 141, 503, 425, 41, 76,
    42, 257, 791, 743, 522, 469,
    580, 781, 155, 192, 699, 226,
    442, 134, 354, 703, 656, 139,
    233, 70, 501, 622, 894, 452,
    789, 512, 631, 586, 565, 236,
    32, 418, 743, 239, 650, 558,
    408, 246, 928, 532, 327, 318,
    197, 484, 36, 711, 371, 281,
    826, 932, 730, 190, 68, 77,
    323, 797, 756, 172, 623, 648,
    972, 116, 515, 123, 128, 651,
    133, 40, 519, 873, 678, 66,
    139, 568, 452, 629, 960, 930,
    // Bin capacity
    1877, 1477, 1895, 1739, 1797, 1415
  };
  const int i6_18_6_25_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    694, 214, 268, 462, 715, 789,
    835, 353, 101, 424, 193, 190,
    794, 630, 23, 476, 932, 320,
    482, 787, 179, 210, 230, 350,
    285, 731, 658, 435, 972, 306,
    210, 276, 535, 939, 347, 684,
    735, 381, 842, 191, 204, 813,
    119, 875, 750, 962, 180, 166,
    930, 272, 761, 805, 771, 755,
    268, 665, 748, 574, 819, 169,
    789, 427, 10, 671, 917, 921,
    211, 942, 86, 883, 154, 445,
    622, 332, 92, 424, 222, 261,
    456, 82, 779, 881, 887, 922,
    660, 368, 149, 459, 860, 472,
    564, 342, 973, 32, 991, 424,
    447, 501, 90, 412, 614, 36,
    172, 741, 199, 558, 936, 950,
    // Bin capacity
    1932, 1859, 1509, 2042, 2280, 1870
  };
  const int i6_18_6_26_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    842, 715, 886, 161, 312, 21,
    72, 935, 997, 438, 433, 707,
    501, 336, 100, 840, 957, 445,
    172, 16, 419, 626, 130, 490,
    79, 947, 495, 680, 535, 732,
    24, 79, 115, 160, 168, 962,
    814, 260, 271, 367, 44, 681,
    517, 303, 960, 809, 418, 281,
    658, 214, 380, 421, 784, 957,
    208, 568, 892, 652, 891, 365,
    565, 377, 826, 569, 370, 774,
    449, 345, 689, 387, 925, 552,
    408, 340, 675, 218, 580, 256,
    835, 611, 580, 973, 710, 118,
    175, 161, 698, 367, 110, 914,
    293, 515, 663, 65, 252, 643,
    564, 999, 858, 232, 530, 528,
    568, 46, 242, 820, 868, 264,
    // Bin capacity
    1627, 1632, 2257, 1845, 1894, 2035
  };
  const int i6_18_6_27_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    117, 957, 602, 867, 145, 265,
    355, 862, 3, 478, 910, 11,
    31, 702, 338, 971, 228, 538,
    160, 869, 382, 293, 90, 847,
    623, 698, 888, 72, 512, 227,
    84, 418, 527, 298, 396, 794,
    233, 996, 436, 211, 594, 167,
    306, 323, 38, 227, 74, 16,
    347, 721, 102, 432, 464, 473,
    232, 850, 249, 771, 613, 86,
    835, 602, 103, 509, 714, 469,
    442, 300, 383, 636, 886, 772,
    594, 125, 670, 925, 52, 563,
    90, 985, 632, 81, 693, 362,
    566, 549, 519, 591, 551, 178,
    245, 948, 950, 286, 516, 51,
    225, 285, 83, 879, 290, 213,
    718, 849, 501, 19, 465, 888,
    // Bin capacity
    1313, 2549, 1568, 1809, 1735, 1465
  };
  const int i6_18_6_28_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    436, 215, 294, 519, 337, 79,
    8, 640, 103, 994, 855, 527,
    202, 672, 793, 991, 30, 503,
    951, 304, 335, 613, 191, 753,
    555, 7, 904, 786, 47, 381,
    644, 789, 900, 25, 797, 636,
    676, 229, 681, 807, 623, 480,
    918, 350, 250, 476, 927, 909,
    354, 422, 715, 232, 280, 598,
    865, 110, 820, 457, 718, 997,
    456, 412, 630, 533, 423, 898,
    567, 268, 840, 424, 565, 864,
    238, 878, 844, 240, 56, 531,
    560, 170, 94, 227, 538, 48,
    186, 204, 660, 670, 543, 251,
    662, 551, 445, 38, 856, 19,
    614, 976, 556, 837, 541, 134,
    468, 904, 559, 189, 713, 846,
    // Bin capacity
    1997, 1729, 2224, 1933, 1929, 2017
  };
  const int i6_18_6_29_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    902, 636, 838, 752, 157, 240,
    465, 606, 114, 484, 566, 27,
    222, 967, 214, 545, 468, 753,
    883, 567, 220, 785, 691, 950,
    721, 654, 715, 182, 896, 569,
    287, 325, 161, 970, 120, 261,
    689, 918, 934, 780, 417, 110,
    823, 407, 884, 390, 725, 280,
    603, 206, 687, 795, 852, 312,
    217, 44, 904, 156, 136, 762,
    34, 286, 973, 221, 616, 796,
    906, 746, 586, 661, 124, 771,
    239, 426, 97, 510, 146, 868,
    770, 248, 424, 88, 791, 827,
    919, 537, 848, 380, 449, 96,
    350, 555, 831, 350, 167, 835,
    162, 39, 833, 559, 535, 604,
    763, 438, 756, 510, 192, 552,
    // Bin capacity
    2141, 1851, 2370, 1961, 1731, 2067
  };
  const int i6_18_6_30_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    777, 510, 14, 423, 739, 81,
    407, 630, 23, 619, 627, 24,
    1000, 749, 784, 846, 443, 677,
    100, 539, 870, 510, 217, 308,
    765, 758, 774, 573, 350, 35,
    525, 200, 229, 450, 172, 198,
    984, 60, 783, 808, 804, 305,
    418, 714, 294, 172, 955, 759,
    712, 909, 690, 555, 694, 158,
    538, 555, 273, 798, 690, 310,
    12, 720, 630, 412, 470, 673,
    695, 181, 664, 878, 141, 81,
    213, 240, 187, 20, 36, 954,
    676, 153, 516, 459, 699, 688,
    479, 93, 561, 223, 169, 47,
    281, 540, 108, 860, 517, 648,
    245, 283, 869, 386, 655, 262,
    836, 261, 988, 332, 210, 760,
    // Bin capacity
    2094, 1754, 2006, 2021, 1861, 1510
  };
  const int i6_18_6_31_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    530, 675, 929, 629, 516, 9,
    282, 587, 121, 781, 405, 356,
    658, 334, 516, 219, 164, 318,
    83, 326, 624, 770, 0, 627,
    965, 896, 690, 160, 355, 918,
    789, 963, 4, 369, 108, 858,
    18, 487, 561, 14, 885, 597,
    362, 894, 867, 293, 133, 892,
    849, 381, 830, 459, 851, 417,
    242, 520, 794, 177, 11, 683,
    7, 762, 626, 672, 21, 158,
    255, 729, 974, 283, 770, 609,
    981, 417, 111, 193, 602, 609,
    342, 194, 482, 891, 332, 822,
    556, 525, 537, 466, 563, 28,
    771, 914, 985, 122, 969, 424,
    1, 597, 891, 79, 201, 950,
    393, 875, 769, 826, 910, 8,
    // Bin capacity
    1766, 2419, 2470, 1617, 1703, 2027
  };
  const int i6_18_6_32_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    310, 573, 180, 251, 717, 621,
    964, 810, 104, 710, 13, 482,
    78, 241, 7, 308, 843, 127,
    260, 798, 605, 282, 317, 349,
    620, 901, 545, 160, 67, 310,
    757, 657, 134, 385, 768, 540,
    500, 608, 599, 80, 140, 182,
    482, 363, 207, 505, 708, 709,
    177, 968, 484, 203, 817, 179,
    926, 165, 596, 994, 872, 433,
    967, 56, 549, 516, 646, 657,
    299, 801, 194, 470, 874, 380,
    228, 700, 146, 745, 399, 319,
    602, 439, 197, 919, 274, 339,
    965, 388, 76, 931, 347, 481,
    428, 442, 938, 112, 993, 409,
    930, 65, 445, 814, 105, 955,
    192, 593, 989, 864, 241, 11,
    // Bin capacity
    2131, 2105, 1539, 2035, 2012, 1647
  };
  const int i6_18_6_33_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    550, 431, 876, 229, 433, 337,
    329, 276, 85, 599, 822, 817,
    437, 131, 790, 839, 395, 861,
    123, 19, 881, 809, 699, 392,
    793, 965, 499, 858, 937, 179,
    481, 241, 391, 549, 989, 715,
    413, 77, 520, 826, 515, 167,
    446, 761, 877, 888, 33, 879,
    832, 368, 240, 744, 749, 231,
    233, 978, 927, 438, 411, 200,
    140, 310, 395, 38, 495, 855,
    25, 611, 212, 187, 313, 832,
    106, 892, 490, 185, 880, 782,
    994, 324, 928, 993, 770, 702,
    335, 646, 636, 343, 237, 57,
    39, 65, 402, 327, 985, 632,
    446, 141, 170, 491, 1000, 761,
    308, 596, 982, 857, 415, 921,
    // Bin capacity
    1559, 1737, 2284, 2261, 2456, 2288
  };
  const int i6_18_6_34_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    803, 232, 812, 417, 542, 949,
    629, 29, 330, 904, 778, 315,
    928, 254, 32, 258, 830, 160,
    324, 814, 255, 136, 931, 386,
    812, 431, 628, 809, 32, 576,
    983, 662, 859, 743, 181, 57,
    560, 377, 979, 298, 434, 386,
    590, 835, 33, 741, 996, 587,
    15, 19, 839, 143, 131, 917,
    564, 667, 258, 604, 155, 851,
    555, 775, 327, 140, 213, 824,
    299, 623, 435, 51, 635, 523,
    258, 243, 130, 48, 126, 557,
    195, 829, 628, 894, 64, 627,
    466, 291, 647, 230, 742, 37,
    361, 408, 900, 179, 903, 799,
    240, 450, 908, 123, 453, 8,
    550, 194, 413, 153, 14, 228,
    // Bin capacity
    2040, 1817, 2103, 1535, 1823, 1963
  };
  const int i6_18_6_35_15[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    756, 578, 478, 243, 597, 648,
    526, 331, 38, 779, 431, 233,
    324, 773, 171, 948, 243, 66,
    728, 864, 244, 119, 503, 139,
    998, 277, 745, 578, 76, 976,
    307, 888, 498, 624, 565, 979,
    28, 983, 912, 729, 844, 857,
    190, 741, 777, 958, 794, 559,
    211, 187, 454, 400, 253, 398,
    677, 286, 196, 912, 722, 472,
    519, 915, 451, 928, 514, 177,
    20, 109, 752, 294, 933, 717,
    276, 859, 99, 187, 43, 305,
    572, 644, 930, 670, 920, 217,
    261, 229, 403, 382, 58, 408,
    272, 829, 71, 736, 485, 457,
    509, 900, 846, 601, 791, 544,
    211, 924, 22, 251, 363, 596,
    // Bin capacity
    1662, 2547, 1820, 2327, 2056, 1969
  };
  const int i6_18_6_0_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    333, 131, 600, 971, 737, 916,
    250, 734, 188, 631, 705, 59,
    282, 542, 293, 169, 384, 697,
    240, 202, 688, 15, 283, 686,
    829, 641, 697, 633, 23, 899,
    315, 353, 884, 747, 269, 11,
    907, 207, 554, 881, 557, 289,
    320, 569, 810, 445, 980, 522,
    309, 618, 620, 458, 109, 926,
    140, 716, 511, 727, 124, 658,
    496, 20, 10, 114, 66, 75,
    657, 929, 854, 798, 694, 473,
    173, 707, 168, 589, 756, 23,
    688, 388, 838, 880, 284, 356,
    483, 912, 889, 575, 538, 713,
    190, 961, 213, 258, 976, 688,
    818, 422, 465, 927, 468, 164,
    728, 803, 445, 302, 121, 794,
    // Bin capacity
    1360, 1643, 1622, 1687, 1346, 1492
  };
  const int i6_18_6_1_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    690, 398, 831, 397, 673, 646,
    734, 925, 266, 763, 926, 581,
    125, 673, 239, 445, 324, 77,
    356, 328, 950, 966, 550, 767,
    179, 945, 214, 728, 101, 961,
    438, 112, 67, 282, 935, 257,
    408, 356, 631, 219, 106, 325,
    555, 430, 28, 316, 455, 829,
    224, 668, 508, 145, 676, 917,
    915, 318, 342, 144, 390, 206,
    607, 488, 892, 522, 354, 448,
    908, 473, 81, 878, 67, 344,
    316, 141, 161, 182, 472, 807,
    281, 964, 372, 340, 171, 861,
    396, 701, 135, 611, 354, 806,
    475, 710, 941, 172, 468, 861,
    448, 748, 862, 676, 421, 999,
    166, 24, 482, 322, 842, 910,
    // Bin capacity
    1384, 1583, 1348, 1365, 1395, 1954
  };
  const int i6_18_6_2_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    855, 921, 312, 304, 519, 876,
    565, 100, 994, 628, 738, 101,
    163, 804, 328, 328, 381, 861,
    698, 517, 98, 350, 531, 267,
    828, 254, 360, 924, 129, 117,
    455, 802, 268, 91, 929, 361,
    734, 299, 774, 467, 54, 993,
    246, 148, 919, 695, 451, 654,
    677, 649, 889, 936, 807, 667,
    313, 1, 537, 76, 651, 29,
    960, 856, 896, 780, 286, 371,
    724, 640, 962, 546, 942, 89,
    854, 32, 325, 79, 829, 290,
    365, 506, 400, 843, 219, 889,
    695, 695, 27, 324, 789, 493,
    41, 382, 1, 0, 416, 417,
    39, 741, 36, 391, 920, 556,
    150, 159, 280, 984, 684, 746,
    // Bin capacity
    1592, 1447, 1430, 1487, 1747, 1493
  };
  const int i6_18_6_3_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    33, 961, 447, 739, 699, 510,
    494, 35, 599, 177, 951, 734,
    81, 711, 330, 117, 806, 789,
    998, 348, 162, 668, 506, 558,
    326, 887, 151, 375, 724, 735,
    238, 397, 691, 998, 86, 313,
    428, 366, 4, 583, 528, 165,
    785, 219, 589, 729, 907, 112,
    625, 464, 568, 941, 905, 445,
    568, 713, 386, 488, 248, 775,
    291, 486, 962, 330, 109, 567,
    669, 443, 787, 220, 78, 421,
    896, 916, 540, 653, 172, 774,
    762, 210, 902, 826, 395, 772,
    618, 555, 834, 877, 690, 858,
    251, 918, 9, 483, 390, 229,
    179, 862, 456, 602, 322, 120,
    911, 711, 856, 381, 470, 638,
    // Bin capacity
    1572, 1752, 1592, 1749, 1543, 1634
  };
  const int i6_18_6_4_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    950, 418, 601, 13, 801, 165,
    920, 505, 419, 424, 981, 248,
    597, 354, 49, 396, 307, 6,
    618, 724, 825, 512, 341, 635,
    304, 509, 927, 183, 333, 204,
    372, 124, 373, 943, 44, 523,
    706, 821, 254, 39, 208, 182,
    828, 531, 848, 257, 661, 863,
    214, 305, 917, 695, 744, 46,
    477, 877, 686, 996, 400, 911,
    120, 673, 233, 562, 944, 655,
    931, 470, 473, 76, 36, 969,
    112, 327, 325, 441, 275, 453,
    214, 417, 517, 905, 438, 312,
    714, 447, 24, 850, 662, 801,
    800, 333, 629, 656, 162, 590,
    947, 152, 812, 120, 340, 76,
    556, 38, 777, 875, 235, 182,
    // Bin capacity
    1800, 1391, 1680, 1551, 1372, 1356
  };
  const int i6_18_6_5_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    185, 890, 477, 514, 511, 228,
    484, 489, 759, 892, 920, 869,
    5, 482, 364, 944, 665, 853,
    14, 224, 656, 208, 666, 376,
    153, 463, 381, 875, 999, 793,
    967, 199, 584, 589, 106, 621,
    530, 597, 647, 604, 887, 484,
    769, 976, 60, 344, 310, 892,
    232, 425, 984, 576, 221, 76,
    534, 458, 705, 233, 608, 346,
    345, 570, 763, 662, 567, 546,
    147, 721, 134, 225, 900, 210,
    981, 77, 931, 279, 855, 287,
    428, 375, 201, 980, 236, 666,
    459, 272, 937, 552, 799, 226,
    433, 695, 349, 668, 577, 653,
    873, 621, 81, 192, 389, 646,
    229, 555, 310, 968, 588, 948,
    // Bin capacity
    1360, 1591, 1632, 1804, 1891, 1701
  };
  const int i6_18_6_6_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    540, 66, 136, 570, 416, 232,
    156, 270, 249, 293, 625, 261,
    631, 570, 180, 100, 598, 820,
    747, 267, 390, 734, 638, 969,
    218, 253, 561, 80, 761, 460,
    487, 438, 821, 296, 222, 494,
    115, 150, 706, 253, 551, 521,
    885, 603, 425, 20, 51, 317,
    179, 645, 755, 768, 761, 636,
    234, 582, 189, 281, 753, 304,
    636, 550, 323, 510, 22, 954,
    936, 626, 744, 383, 999, 12,
    395, 40, 262, 524, 647, 907,
    575, 96, 671, 832, 803, 795,
    25, 832, 424, 621, 39, 675,
    784, 596, 816, 817, 823, 553,
    400, 865, 630, 505, 600, 144,
    972, 772, 897, 380, 897, 15,
    // Bin capacity
    1575, 1453, 1622, 1408, 1804, 1603
  };
  const int i6_18_6_7_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    817, 103, 85, 602, 305, 867,
    235, 107, 394, 194, 722, 260,
    939, 907, 837, 945, 490, 942,
    396, 904, 265, 570, 209, 991,
    97, 71, 725, 636, 704, 152,
    151, 262, 153, 137, 390, 848,
    723, 481, 295, 181, 403, 85,
    113, 426, 446, 682, 121, 678,
    153, 305, 420, 703, 109, 878,
    315, 676, 460, 320, 774, 870,
    760, 864, 315, 660, 11, 223,
    190, 471, 235, 331, 110, 478,
    100, 288, 345, 215, 161, 436,
    185, 972, 544, 749, 989, 858,
    215, 955, 107, 460, 688, 634,
    689, 891, 208, 886, 124, 307,
    285, 932, 846, 448, 694, 170,
    35, 645, 90, 958, 645, 659,
    // Bin capacity
    1141, 1830, 1208, 1726, 1365, 1844
  };
  const int i6_18_6_8_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    149, 992, 273, 360, 961, 944,
    981, 21, 792, 691, 266, 208,
    600, 779, 741, 565, 789, 592,
    817, 733, 39, 30, 104, 45,
    601, 832, 905, 159, 659, 217,
    459, 774, 700, 954, 110, 909,
    316, 544, 625, 417, 55, 431,
    236, 770, 900, 327, 556, 683,
    375, 279, 583, 547, 350, 510,
    188, 285, 487, 373, 652, 661,
    463, 317, 429, 853, 960, 953,
    907, 563, 360, 932, 617, 89,
    744, 689, 187, 741, 988, 45,
    535, 402, 340, 782, 536, 645,
    937, 77, 946, 395, 321, 296,
    829, 746, 325, 296, 294, 33,
    649, 779, 939, 249, 34, 639,
    965, 32, 197, 996, 567, 904,
    // Bin capacity
    1936, 1731, 1759, 1741, 1588, 1585
  };
  const int i6_18_6_9_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    719, 623, 219, 273, 414, 462,
    10, 39, 580, 333, 806, 200,
    36, 600, 956, 89, 928, 615,
    549, 356, 787, 806, 275, 900,
    372, 598, 291, 344, 766, 982,
    853, 830, 856, 617, 757, 610,
    940, 9, 438, 547, 243, 232,
    385, 456, 452, 841, 927, 661,
    504, 760, 909, 45, 977, 686,
    76, 57, 571, 27, 408, 424,
    885, 8, 875, 982, 790, 84,
    63, 479, 160, 736, 6, 213,
    654, 88, 289, 165, 513, 527,
    107, 336, 399, 343, 115, 238,
    504, 958, 328, 537, 278, 160,
    193, 515, 860, 946, 838, 64,
    64, 383, 702, 454, 977, 559,
    111, 117, 752, 504, 939, 351,
    // Bin capacity
    1277, 1311, 1894, 1561, 1991, 1448
  };
  const int i6_18_6_10_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    539, 668, 53, 408, 267, 961,
    226, 765, 152, 41, 77, 176,
    14, 892, 426, 871, 955, 848,
    915, 802, 824, 570, 841, 295,
    62, 508, 801, 927, 801, 818,
    699, 934, 782, 79, 304, 296,
    609, 324, 270, 430, 119, 539,
    634, 498, 51, 203, 982, 119,
    327, 952, 545, 448, 23, 613,
    825, 405, 793, 808, 887, 887,
    256, 734, 957, 592, 256, 191,
    642, 726, 293, 446, 370, 426,
    213, 261, 646, 936, 57, 941,
    942, 607, 974, 390, 677, 270,
    742, 658, 180, 812, 600, 922,
    873, 340, 44, 540, 764, 577,
    511, 382, 96, 839, 179, 466,
    537, 751, 5, 119, 579, 700,
    // Bin capacity
    1754, 2055, 1447, 1735, 1602, 1842
  };
  const int i6_18_6_11_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    1000, 997, 595, 477, 773, 638,
    697, 940, 200, 192, 512, 732,
    169, 739, 241, 872, 885, 723,
    697, 262, 207, 940, 632, 255,
    499, 128, 288, 275, 449, 583,
    558, 144, 685, 622, 961, 277,
    864, 90, 908, 227, 863, 874,
    831, 78, 357, 124, 332, 946,
    820, 557, 680, 139, 666, 648,
    943, 842, 578, 922, 462, 809,
    570, 777, 96, 56, 831, 870,
    397, 393, 315, 516, 289, 850,
    946, 202, 193, 26, 463, 415,
    718, 383, 531, 329, 616, 167,
    352, 75, 494, 616, 332, 268,
    688, 238, 869, 730, 52, 261,
    491, 171, 78, 259, 13, 18,
    341, 640, 475, 617, 928, 141,
    // Bin capacity
    2143, 1417, 1442, 1469, 1861, 1753
  };
  const int i6_18_6_12_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    438, 967, 555, 339, 772, 292,
    124, 918, 48, 143, 734, 520,
    307, 860, 37, 934, 494, 752,
    406, 898, 778, 472, 515, 432,
    336, 351, 901, 692, 395, 565,
    358, 476, 881, 21, 973, 202,
    468, 701, 82, 277, 828, 818,
    760, 255, 385, 493, 122, 170,
    616, 623, 155, 779, 311, 211,
    822, 930, 26, 954, 390, 476,
    745, 576, 750, 164, 560, 509,
    706, 480, 657, 6, 692, 962,
    525, 51, 701, 13, 750, 354,
    538, 983, 237, 625, 526, 832,
    135, 705, 547, 138, 905, 694,
    408, 81, 922, 506, 503, 435,
    997, 870, 31, 485, 221, 180,
    692, 632, 334, 485, 855, 534,
    // Bin capacity
    1752, 2120, 1499, 1405, 1969, 1669
  };
  const int i6_18_6_13_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    658, 721, 719, 962, 559, 722,
    654, 627, 948, 958, 321, 154,
    1, 639, 907, 230, 347, 129,
    148, 971, 767, 805, 70, 365,
    847, 903, 216, 783, 437, 542,
    581, 415, 182, 254, 238, 804,
    584, 141, 746, 709, 759, 660,
    799, 428, 486, 520, 898, 688,
    271, 283, 98, 66, 427, 397,
    914, 690, 430, 683, 635, 229,
    688, 256, 88, 629, 688, 477,
    309, 804, 985, 930, 529, 615,
    29, 881, 540, 690, 865, 89,
    833, 925, 744, 403, 714, 650,
    613, 623, 857, 601, 684, 405,
    533, 313, 716, 110, 505, 73,
    882, 365, 369, 393, 545, 508,
    412, 83, 928, 332, 963, 133,
    // Bin capacity
    1838, 1897, 2021, 1895, 1918, 1439
  };
  const int i6_18_6_14_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    36, 647, 604, 45, 185, 931,
    375, 226, 87, 691, 302, 673,
    514, 491, 28, 327, 410, 931,
    986, 841, 174, 969, 478, 477,
    410, 983, 8, 754, 135, 532,
    948, 575, 692, 547, 922, 944,
    638, 736, 595, 793, 844, 956,
    815, 968, 978, 844, 731, 701,
    194, 228, 487, 248, 19, 486,
    333, 24, 823, 844, 891, 716,
    256, 637, 573, 555, 532, 901,
    644, 354, 838, 644, 209, 326,
    583, 846, 321, 214, 892, 807,
    208, 386, 652, 618, 687, 134,
    678, 348, 977, 183, 948, 821,
    886, 469, 375, 958, 40, 818,
    579, 950, 56, 411, 856, 627,
    967, 368, 275, 974, 521, 225,
    // Bin capacity
    1910, 1915, 1624, 2018, 1825, 2282
  };
  const int i6_18_6_15_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    696, 800, 137, 486, 232, 796,
    419, 495, 729, 704, 216, 933,
    11, 39, 851, 411, 958, 896,
    826, 935, 576, 476, 412, 927,
    870, 732, 395, 590, 505, 250,
    398, 995, 323, 379, 557, 518,
    579, 431, 469, 118, 524, 11,
    540, 66, 205, 683, 483, 12,
    488, 150, 867, 519, 184, 545,
    46, 553, 743, 170, 40, 509,
    822, 372, 744, 571, 853, 356,
    10, 510, 592, 367, 739, 677,
    244, 755, 856, 246, 190, 298,
    644, 171, 364, 37, 935, 647,
    930, 481, 805, 220, 507, 86,
    779, 519, 946, 544, 446, 107,
    493, 741, 681, 56, 314, 726,
    48, 838, 481, 714, 368, 863,
    // Bin capacity
    1695, 1837, 2064, 1398, 1623, 1756
  };
  const int i6_18_6_16_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    620, 130, 519, 581, 6, 912,
    311, 14, 601, 436, 98, 432,
    612, 875, 911, 431, 228, 264,
    667, 91, 667, 488, 822, 67,
    128, 756, 509, 331, 245, 169,
    393, 623, 231, 25, 815, 393,
    658, 283, 150, 331, 863, 321,
    924, 504, 779, 978, 15, 649,
    805, 439, 238, 843, 560, 18,
    876, 962, 173, 153, 856, 453,
    125, 984, 648, 911, 738, 213,
    619, 960, 811, 980, 603, 555,
    922, 297, 184, 679, 732, 353,
    826, 50, 568, 284, 219, 101,
    242, 647, 194, 18, 224, 536,
    599, 299, 582, 838, 524, 87,
    698, 280, 158, 361, 59, 898,
    336, 658, 794, 373, 286, 327,
    // Bin capacity
    2004, 1712, 1686, 1748, 1526, 1305
  };
  const int i6_18_6_17_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    77, 973, 316, 504, 37, 9,
    468, 31, 400, 5, 147, 53,
    572, 377, 828, 733, 623, 15,
    771, 435, 826, 812, 469, 721,
    235, 264, 440, 576, 669, 514,
    937, 965, 745, 473, 897, 695,
    786, 19, 459, 857, 885, 143,
    199, 666, 745, 954, 668, 408,
    960, 937, 994, 963, 770, 251,
    755, 640, 880, 952, 360, 91,
    438, 895, 547, 217, 849, 299,
    604, 656, 887, 115, 47, 138,
    745, 280, 290, 467, 637, 595,
    956, 1, 314, 967, 985, 206,
    862, 944, 667, 733, 309, 767,
    463, 289, 808, 633, 368, 323,
    45, 423, 304, 332, 328, 841,
    80, 619, 676, 457, 424, 599,
    // Bin capacity
    1941, 1836, 2170, 2097, 1848, 1301
  };
  const int i6_18_6_18_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    655, 523, 814, 858, 43, 901,
    47, 806, 284, 583, 792, 379,
    873, 237, 57, 857, 138, 589,
    496, 374, 987, 448, 699, 30,
    305, 12, 690, 930, 364, 995,
    920, 917, 659, 197, 118, 410,
    926, 920, 586, 369, 328, 777,
    962, 649, 753, 878, 407, 861,
    379, 822, 730, 838, 895, 928,
    698, 165, 536, 382, 429, 709,
    842, 120, 439, 537, 636, 557,
    602, 158, 799, 797, 566, 876,
    827, 725, 536, 285, 571, 970,
    267, 132, 202, 886, 746, 629,
    742, 291, 253, 802, 733, 89,
    635, 931, 356, 292, 841, 152,
    809, 612, 624, 158, 143, 826,
    918, 27, 788, 418, 191, 436,
    // Bin capacity
    2341, 1657, 1985, 2068, 1700, 2186
  };
  const int i6_18_6_19_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    965, 450, 671, 302, 632, 399,
    1, 352, 137, 28, 936, 482,
    4, 786, 67, 390, 381, 124,
    883, 242, 584, 403, 766, 652,
    199, 645, 161, 49, 915, 813,
    156, 168, 855, 650, 515, 408,
    115, 701, 841, 909, 342, 86,
    832, 782, 382, 34, 665, 367,
    616, 817, 209, 767, 933, 793,
    616, 608, 613, 550, 101, 232,
    475, 340, 4, 163, 128, 477,
    256, 260, 631, 415, 505, 676,
    469, 89, 944, 965, 50, 712,
    952, 343, 147, 570, 671, 69,
    640, 339, 165, 768, 786, 288,
    610, 468, 933, 673, 766, 568,
    20, 442, 419, 809, 817, 161,
    662, 888, 240, 805, 220, 135,
    // Bin capacity
    1681, 1730, 1588, 1835, 2009, 1476
  };
  const int i6_18_6_20_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    192, 961, 569, 264, 727, 859,
    383, 112, 150, 76, 344, 825,
    135, 351, 145, 809, 998, 13,
    679, 179, 724, 332, 594, 812,
    354, 126, 162, 650, 446, 326,
    632, 267, 149, 21, 718, 152,
    974, 730, 704, 129, 56, 732,
    530, 14, 331, 709, 349, 499,
    93, 403, 354, 609, 367, 648,
    532, 969, 151, 547, 824, 211,
    25, 807, 171, 608, 62, 904,
    282, 308, 885, 50, 205, 315,
    299, 228, 997, 89, 988, 126,
    596, 615, 453, 541, 972, 18,
    987, 289, 937, 565, 211, 2,
    203, 954, 977, 104, 773, 299,
    843, 406, 330, 725, 790, 439,
    771, 252, 910, 967, 442, 372,
    // Bin capacity
    1702, 1595, 1820, 1559, 1974, 1511
  };
  const int i6_18_6_21_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    807, 239, 796, 319, 488, 388,
    386, 885, 658, 519, 380, 857,
    448, 425, 659, 147, 55, 837,
    345, 690, 343, 449, 573, 939,
    575, 464, 513, 340, 375, 314,
    258, 562, 503, 653, 328, 87,
    100, 527, 845, 920, 970, 317,
    628, 310, 411, 229, 97, 685,
    906, 787, 579, 229, 387, 198,
    617, 410, 583, 453, 429, 687,
    838, 677, 409, 856, 417, 673,
    302, 703, 270, 609, 336, 601,
    399, 351, 695, 632, 859, 774,
    511, 897, 753, 353, 690, 603,
    581, 688, 235, 141, 595, 962,
    759, 794, 423, 179, 991, 269,
    988, 1, 808, 46, 580, 236,
    612, 730, 879, 736, 674, 230,
    // Bin capacity
    2029, 2045, 2090, 1576, 1861, 1948
  };
  const int i6_18_6_22_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    673, 239, 310, 447, 621, 434,
    322, 523, 694, 440, 29, 639,
    750, 520, 787, 83, 890, 658,
    167, 300, 24, 684, 69, 798,
    335, 468, 385, 409, 358, 748,
    57, 327, 586, 813, 821, 829,
    657, 422, 361, 41, 526, 220,
    689, 19, 304, 539, 26, 881,
    513, 566, 453, 226, 428, 500,
    963, 4, 662, 728, 16, 148,
    647, 582, 839, 36, 481, 593,
    850, 209, 96, 478, 598, 138,
    583, 448, 478, 376, 680, 992,
    956, 106, 278, 968, 569, 835,
    123, 372, 237, 473, 140, 597,
    327, 910, 531, 763, 757, 501,
    428, 295, 346, 178, 331, 330,
    229, 868, 952, 289, 335, 153,
    // Bin capacity
    1885, 1460, 1693, 1621, 1561, 2033
  };
  const int i6_18_6_23_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    27, 269, 111, 67, 376, 864,
    209, 29, 101, 383, 632, 558,
    589, 794, 836, 195, 513, 617,
    960, 14, 134, 747, 237, 48,
    82, 520, 916, 206, 992, 15,
    424, 738, 330, 642, 854, 456,
    490, 606, 106, 765, 56, 444,
    994, 969, 308, 548, 489, 947,
    12, 87, 446, 849, 138, 356,
    666, 517, 848, 455, 145, 715,
    217, 828, 909, 13, 567, 533,
    298, 134, 700, 481, 156, 565,
    679, 54, 593, 607, 269, 411,
    92, 896, 24, 687, 999, 781,
    601, 432, 959, 314, 395, 552,
    638, 679, 582, 324, 738, 904,
    7, 169, 198, 42, 770, 63,
    122, 722, 234, 79, 184, 755,
    // Bin capacity
    1457, 1734, 1709, 1518, 1745, 1965
  };
  const int i6_18_6_24_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    968, 754, 10, 888, 471, 828,
    248, 537, 469, 814, 908, 651,
    219, 523, 396, 439, 117, 838,
    604, 823, 672, 8, 742, 455,
    780, 729, 625, 230, 198, 155,
    985, 69, 900, 387, 771, 111,
    532, 691, 597, 801, 358, 96,
    358, 491, 465, 207, 117, 439,
    497, 932, 2, 692, 974, 977,
    331, 228, 82, 308, 670, 686,
    528, 998, 902, 741, 893, 30,
    940, 244, 846, 865, 173, 818,
    143, 686, 462, 162, 182, 896,
    289, 565, 664, 866, 329, 851,
    408, 462, 714, 841, 424, 393,
    807, 709, 763, 905, 792, 343,
    20, 589, 693, 684, 445, 784,
    404, 334, 941, 844, 958, 221,
    // Bin capacity
    1873, 2142, 2109, 2208, 1968, 1979
  };
  const int i6_18_6_25_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    844, 43, 634, 685, 979, 414,
    999, 391, 116, 263, 653, 475,
    352, 722, 12, 771, 384, 643,
    714, 884, 897, 152, 704, 641,
    632, 90, 644, 922, 945, 516,
    83, 279, 179, 486, 249, 607,
    863, 36, 683, 846, 700, 277,
    329, 867, 475, 771, 407, 477,
    467, 572, 482, 978, 294, 529,
    900, 33, 740, 153, 946, 689,
    883, 23, 357, 236, 834, 194,
    222, 791, 537, 344, 680, 589,
    240, 456, 729, 921, 286, 554,
    12, 2, 691, 663, 126, 524,
    245, 438, 687, 519, 388, 903,
    284, 453, 288, 290, 216, 58,
    748, 252, 167, 525, 876, 418,
    195, 586, 645, 929, 1, 817,
    // Bin capacity
    1878, 1442, 1868, 2178, 2015, 1943
  };
  const int i6_18_6_26_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    666, 791, 592, 695, 794, 365,
    199, 732, 99, 934, 685, 337,
    862, 957, 609, 944, 877, 422,
    427, 95, 487, 496, 31, 106,
    51, 45, 235, 878, 306, 171,
    947, 271, 818, 12, 86, 90,
    467, 88, 455, 660, 823, 569,
    918, 706, 278, 131, 738, 433,
    149, 792, 344, 75, 286, 211,
    844, 26, 926, 5, 994, 758,
    782, 869, 740, 920, 546, 392,
    190, 33, 261, 212, 667, 372,
    899, 11, 92, 316, 321, 201,
    730, 65, 511, 703, 699, 484,
    3, 480, 213, 765, 638, 172,
    667, 557, 607, 500, 324, 646,
    636, 70, 712, 38, 992, 196,
    709, 521, 664, 625, 780, 987,
    // Bin capacity
    2131, 1493, 1816, 1871, 2224, 1452
  };
  const int i6_18_6_27_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    870, 840, 541, 539, 132, 497,
    395, 524, 866, 150, 221, 416,
    78, 754, 64, 683, 829, 94,
    179, 206, 905, 254, 816, 177,
    917, 711, 53, 214, 506, 365,
    417, 931, 586, 564, 819, 777,
    364, 319, 243, 849, 172, 724,
    200, 917, 704, 217, 566, 851,
    332, 160, 674, 611, 992, 187,
    827, 716, 270, 98, 929, 263,
    319, 15, 246, 717, 157, 65,
    791, 609, 347, 563, 844, 350,
    680, 765, 412, 282, 506, 924,
    856, 627, 96, 294, 652, 822,
    696, 116, 351, 305, 400, 377,
    404, 99, 877, 267, 586, 682,
    141, 823, 441, 177, 373, 34,
    475, 402, 321, 573, 963, 720,
    // Bin capacity
    1893, 2019, 1693, 1558, 2215, 1763
  };
  const int i6_18_6_28_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    403, 881, 965, 211, 653, 638,
    105, 717, 412, 416, 226, 909,
    634, 254, 960, 888, 365, 690,
    258, 237, 419, 340, 554, 226,
    159, 350, 696, 859, 852, 431,
    609, 850, 817, 366, 366, 13,
    964, 333, 661, 519, 472, 744,
    364, 615, 479, 695, 856, 120,
    959, 612, 446, 769, 269, 392,
    866, 482, 135, 161, 50, 142,
    885, 819, 708, 436, 608, 846,
    793, 649, 279, 334, 125, 472,
    791, 95, 989, 663, 485, 881,
    123, 298, 79, 765, 358, 875,
    911, 281, 424, 145, 53, 837,
    271, 219, 356, 568, 118, 798,
    996, 250, 372, 685, 534, 182,
    210, 650, 952, 474, 53, 303,
    // Bin capacity
    2198, 1833, 2166, 1983, 1493, 2027
  };
  const int i6_18_6_29_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    259, 776, 214, 524, 135, 543,
    619, 890, 579, 403, 326, 692,
    33, 509, 404, 413, 930, 685,
    325, 412, 648, 265, 487, 136,
    860, 358, 121, 697, 361, 193,
    719, 349, 565, 38, 361, 432,
    345, 796, 257, 518, 860, 97,
    696, 613, 698, 223, 425, 241,
    300, 740, 746, 270, 478, 834,
    51, 889, 737, 348, 704, 6,
    863, 368, 889, 641, 575, 624,
    448, 669, 385, 77, 945, 38,
    134, 816, 826, 86, 398, 249,
    286, 459, 686, 96, 711, 755,
    145, 785, 598, 462, 778, 876,
    242, 735, 588, 190, 183, 492,
    453, 563, 211, 453, 439, 422,
    677, 311, 412, 354, 318, 359,
    // Bin capacity
    1603, 2374, 2057, 1303, 2025, 1650
  };
  const int i6_18_6_30_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    725, 903, 325, 504, 433, 614,
    979, 473, 791, 876, 318, 112,
    219, 131, 86, 183, 152, 298,
    663, 731, 610, 454, 592, 836,
    745, 85, 993, 547, 238, 34,
    498, 649, 489, 290, 736, 48,
    591, 93, 814, 208, 706, 153,
    259, 685, 986, 390, 495, 728,
    436, 322, 183, 341, 71, 70,
    844, 846, 583, 972, 101, 921,
    209, 21, 8, 74, 383, 24,
    281, 856, 177, 867, 641, 894,
    136, 963, 736, 372, 766, 624,
    718, 970, 640, 644, 688, 441,
    479, 904, 409, 202, 642, 931,
    754, 446, 543, 261, 846, 510,
    489, 842, 963, 723, 130, 596,
    224, 323, 502, 744, 813, 295,
    // Bin capacity
    2004, 2220, 2132, 1875, 1897, 1762
  };
  const int i6_18_6_31_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    691, 772, 897, 460, 642, 606,
    305, 453, 139, 301, 362, 465,
    981, 331, 696, 574, 91, 381,
    605, 783, 152, 267, 46, 914,
    733, 894, 469, 173, 682, 132,
    837, 589, 51, 137, 736, 383,
    757, 978, 225, 715, 401, 590,
    596, 327, 150, 492, 86, 255,
    525, 693, 177, 817, 421, 330,
    347, 105, 810, 692, 160, 667,
    35, 776, 333, 378, 288, 426,
    704, 231, 933, 489, 259, 316,
    326, 452, 968, 151, 776, 637,
    567, 144, 629, 60, 840, 987,
    921, 745, 219, 398, 452, 793,
    816, 892, 328, 665, 36, 919,
    510, 762, 546, 510, 367, 130,
    447, 888, 322, 151, 609, 102,
    // Bin capacity
    2337, 2362, 1757, 1623, 1584, 1973
  };
  const int i6_18_6_32_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    145, 499, 752, 42, 676, 822,
    203, 828, 77, 230, 868, 749,
    406, 25, 206, 552, 669, 416,
    3, 459, 592, 862, 413, 579,
    169, 181, 361, 728, 139, 426,
    860, 549, 838, 145, 585, 450,
    342, 487, 206, 614, 561, 517,
    880, 948, 414, 911, 492, 333,
    506, 803, 327, 980, 489, 879,
    600, 728, 50, 17, 300, 225,
    881, 820, 900, 699, 16, 805,
    927, 154, 689, 169, 33, 144,
    218, 882, 21, 179, 280, 604,
    152, 308, 300, 323, 836, 668,
    196, 98, 984, 243, 646, 19,
    472, 727, 109, 253, 647, 897,
    125, 285, 974, 345, 676, 578,
    106, 403, 964, 574, 976, 414,
    // Bin capacity
    1583, 2021, 1929, 1731, 2047, 2096
  };
  const int i6_18_6_33_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    547, 246, 50, 943, 527, 344,
    672, 128, 157, 381, 845, 613,
    647, 276, 936, 737, 236, 42,
    56, 786, 200, 741, 131, 908,
    797, 428, 250, 142, 442, 239,
    695, 868, 140, 814, 371, 32,
    820, 383, 884, 210, 886, 968,
    5, 820, 49, 716, 828, 534,
    231, 350, 659, 153, 898, 763,
    777, 161, 21, 643, 762, 85,
    502, 701, 530, 203, 29, 942,
    862, 745, 286, 316, 465, 32,
    739, 600, 879, 145, 897, 472,
    455, 148, 999, 356, 580, 676,
    925, 447, 392, 662, 196, 552,
    151, 630, 248, 10, 870, 577,
    336, 706, 931, 46, 697, 140,
    826, 863, 539, 567, 649, 668,
    // Bin capacity
    2227, 2059, 1807, 1726, 2286, 1904
  };
  const int i6_18_6_34_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    984, 912, 446, 739, 946, 784,
    982, 5, 740, 476, 588, 46,
    429, 443, 513, 321, 58, 89,
    821, 403, 977, 298, 98, 936,
    614, 305, 145, 648, 538, 692,
    820, 190, 187, 465, 897, 437,
    470, 121, 971, 205, 281, 310,
    464, 151, 337, 110, 391, 286,
    935, 280, 104, 894, 732, 301,
    935, 103, 101, 45, 989, 143,
    452, 92, 737, 44, 116, 864,
    936, 779, 914, 583, 703, 471,
    25, 571, 432, 176, 806, 750,
    954, 459, 343, 303, 67, 448,
    340, 12, 287, 651, 525, 510,
    474, 674, 361, 451, 66, 733,
    912, 174, 802, 163, 61, 429,
    611, 210, 166, 825, 534, 666,
    // Bin capacity
    2716, 1315, 1913, 1652, 1876, 1987
  };
  const int i6_18_6_35_16[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    261, 77, 1000, 968, 608, 75,
    753, 195, 662, 243, 394, 465,
    377, 38, 252, 889, 343, 735,
    496, 700, 904, 537, 130, 517,
    27, 396, 683, 839, 666, 737,
    428, 126, 938, 449, 92, 882,
    780, 861, 14, 677, 82, 590,
    799, 375, 598, 52, 800, 428,
    952, 968, 540, 539, 277, 733,
    409, 388, 847, 735, 800, 744,
    526, 715, 578, 310, 241, 480,
    935, 878, 268, 712, 996, 802,
    544, 235, 722, 173, 457, 16,
    874, 759, 173, 522, 913, 531,
    360, 325, 710, 466, 289, 49,
    309, 522, 667, 353, 500, 690,
    124, 216, 655, 805, 974, 88,
    991, 362, 884, 544, 121, 547,
    // Bin capacity
    2238, 1831, 2497, 2208, 1954, 2050
  };
  const int i6_18_6_0_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    974, 312, 299, 647, 588, 903,
    444, 238, 382, 556, 150, 186,
    288, 614, 999, 581, 389, 966,
    113, 354, 607, 762, 586, 148,
    575, 111, 556, 97, 224, 850,
    525, 498, 322, 781, 794, 999,
    73, 555, 72, 685, 929, 802,
    1, 246, 123, 849, 946, 467,
    992, 828, 426, 854, 107, 172,
    510, 810, 27, 60, 953, 180,
    406, 896, 677, 406, 693, 8,
    189, 953, 626, 200, 493, 877,
    941, 507, 16, 759, 945, 104,
    360, 731, 480, 597, 890, 785,
    472, 121, 586, 91, 529, 528,
    385, 756, 883, 453, 867, 446,
    189, 612, 344, 828, 646, 227,
    512, 87, 585, 86, 206, 466,
    // Bin capacity
    1325, 1539, 1335, 1549, 1823, 1519
  };
  const int i6_18_6_1_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    580, 503, 152, 261, 183, 842,
    627, 678, 413, 717, 524, 997,
    470, 974, 602, 80, 21, 434,
    929, 802, 1, 439, 536, 701,
    333, 769, 646, 952, 911, 418,
    785, 446, 203, 472, 195, 387,
    6, 62, 316, 814, 30, 816,
    868, 980, 498, 652, 848, 711,
    632, 950, 239, 189, 632, 218,
    922, 220, 668, 650, 545, 163,
    346, 970, 254, 120, 963, 753,
    643, 888, 487, 953, 521, 206,
    226, 778, 12, 835, 970, 446,
    557, 842, 360, 768, 893, 753,
    21, 921, 628, 924, 541, 733,
    571, 291, 347, 233, 491, 8,
    862, 554, 361, 754, 618, 160,
    390, 986, 936, 935, 621, 361,
    // Bin capacity
    1645, 2124, 1200, 1810, 1691, 1534
  };
  const int i6_18_6_2_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    843, 862, 367, 741, 799, 214,
    177, 353, 690, 672, 324, 360,
    290, 711, 16, 712, 645, 151,
    134, 773, 26, 926, 77, 400,
    854, 396, 578, 36, 488, 580,
    738, 158, 996, 523, 92, 486,
    901, 224, 812, 427, 727, 461,
    235, 783, 4, 809, 953, 508,
    136, 991, 291, 403, 746, 242,
    132, 583, 525, 829, 180, 296,
    241, 17, 869, 957, 416, 154,
    641, 414, 783, 718, 206, 667,
    428, 13, 42, 79, 422, 904,
    416, 988, 21, 279, 871, 568,
    671, 732, 700, 921, 426, 182,
    212, 506, 683, 487, 565, 730,
    127, 112, 358, 731, 865, 638,
    704, 273, 318, 101, 927, 862,
    // Bin capacity
    1340, 1512, 1374, 1760, 1654, 1429
  };
  const int i6_18_6_3_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    260, 964, 88, 929, 908, 875,
    541, 742, 251, 176, 195, 487,
    517, 333, 505, 905, 415, 424,
    121, 259, 952, 342, 15, 375,
    338, 707, 84, 300, 841, 476,
    918, 651, 567, 821, 322, 755,
    893, 977, 261, 929, 529, 296,
    228, 121, 837, 545, 629, 126,
    57, 937, 426, 235, 557, 122,
    848, 544, 843, 390, 142, 428,
    288, 193, 457, 708, 758, 281,
    629, 140, 324, 340, 26, 169,
    270, 334, 404, 285, 352, 122,
    458, 59, 948, 186, 6, 589,
    862, 769, 112, 513, 121, 398,
    232, 804, 864, 542, 758, 487,
    774, 450, 553, 772, 811, 722,
    869, 163, 141, 808, 743, 867,
    // Bin capacity
    1563, 1571, 1480, 1670, 1396, 1374
  };
  const int i6_18_6_4_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    812, 449, 822, 784, 543, 851,
    520, 163, 667, 777, 681, 850,
    436, 562, 477, 428, 240, 231,
    372, 695, 956, 774, 223, 597,
    605, 901, 672, 865, 669, 777,
    795, 943, 289, 74, 908, 779,
    893, 142, 545, 916, 752, 954,
    591, 207, 526, 29, 702, 368,
    299, 443, 874, 39, 576, 790,
    232, 340, 437, 426, 771, 698,
    912, 798, 101, 289, 905, 489,
    841, 273, 658, 780, 878, 987,
    284, 790, 545, 600, 172, 543,
    528, 791, 39, 195, 430, 126,
    649, 732, 566, 850, 323, 650,
    274, 697, 27, 113, 835, 592,
    79, 401, 655, 742, 341, 657,
    313, 649, 240, 954, 301, 882,
    // Bin capacity
    1636, 1730, 1577, 1671, 1777, 2049
  };
  const int i6_18_6_5_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    248, 745, 760, 709, 983, 355,
    194, 912, 501, 959, 650, 218,
    835, 193, 987, 187, 201, 540,
    217, 160, 553, 245, 549, 377,
    648, 410, 589, 378, 499, 975,
    133, 115, 327, 332, 341, 105,
    40, 105, 583, 374, 881, 317,
    562, 176, 10, 170, 762, 999,
    329, 568, 988, 16, 414, 627,
    299, 737, 908, 483, 852, 366,
    530, 58, 800, 548, 27, 629,
    986, 141, 308, 964, 114, 873,
    184, 217, 703, 449, 634, 538,
    625, 779, 620, 879, 304, 189,
    993, 32, 513, 628, 860, 185,
    782, 963, 988, 777, 679, 880,
    589, 68, 294, 642, 430, 38,
    179, 116, 294, 362, 275, 730,
    // Bin capacity
    1466, 1137, 1878, 1593, 1655, 1565
  };
  const int i6_18_6_6_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    93, 64, 543, 22, 481, 634,
    239, 150, 144, 511, 864, 825,
    711, 100, 747, 924, 13, 804,
    537, 578, 128, 272, 316, 936,
    137, 223, 931, 163, 680, 669,
    102, 917, 284, 636, 392, 911,
    906, 128, 594, 121, 700, 147,
    773, 628, 596, 362, 594, 528,
    48, 56, 141, 203, 179, 564,
    644, 100, 167, 836, 841, 783,
    749, 593, 986, 527, 590, 522,
    41, 114, 910, 237, 947, 205,
    265, 262, 0, 919, 63, 545,
    876, 111, 229, 218, 627, 699,
    234, 628, 205, 274, 768, 480,
    189, 613, 545, 356, 823, 684,
    207, 606, 681, 900, 381, 236,
    311, 939, 703, 770, 89, 789,
    // Bin capacity
    1248, 1204, 1508, 1458, 1652, 1937
  };
  const int i6_18_6_7_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    666, 956, 964, 789, 514, 590,
    227, 505, 644, 390, 52, 989,
    9, 725, 223, 196, 18, 276,
    198, 236, 584, 279, 300, 924,
    121, 640, 544, 535, 814, 504,
    412, 893, 72, 810, 304, 201,
    643, 939, 128, 522, 950, 425,
    504, 90, 126, 707, 467, 761,
    459, 398, 474, 481, 345, 772,
    551, 47, 839, 916, 497, 951,
    143, 27, 274, 360, 807, 222,
    393, 449, 752, 821, 384, 896,
    264, 256, 862, 147, 849, 972,
    14, 609, 934, 762, 156, 848,
    986, 449, 204, 376, 843, 25,
    644, 6, 219, 590, 608, 682,
    402, 238, 160, 484, 714, 631,
    536, 78, 971, 257, 946, 784,
    // Bin capacity
    1280, 1345, 1601, 1681, 1707, 2043
  };
  const int i6_18_6_8_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    792, 365, 711, 777, 977, 995,
    991, 692, 24, 145, 359, 765,
    353, 704, 868, 771, 407, 312,
    97, 859, 500, 329, 321, 62,
    135, 135, 423, 836, 315, 879,
    450, 175, 496, 675, 784, 593,
    589, 719, 999, 820, 920, 370,
    974, 445, 159, 411, 412, 349,
    632, 493, 408, 524, 88, 108,
    596, 48, 55, 176, 233, 504,
    18, 297, 488, 688, 474, 156,
    157, 773, 364, 57, 914, 390,
    81, 579, 554, 96, 495, 408,
    179, 200, 132, 473, 22, 967,
    769, 107, 20, 515, 632, 389,
    562, 38, 124, 859, 704, 514,
    274, 774, 36, 666, 692, 508,
    513, 477, 638, 490, 743, 443,
    // Bin capacity
    1470, 1419, 1260, 1676, 1709, 1569
  };
  const int i6_18_6_9_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    311, 591, 18, 257, 860, 351,
    856, 349, 490, 259, 886, 137,
    18, 140, 674, 358, 172, 665,
    869, 382, 193, 317, 975, 320,
    575, 13, 240, 665, 556, 36,
    119, 930, 26, 928, 922, 2,
    510, 123, 816, 57, 528, 559,
    829, 799, 418, 67, 270, 259,
    106, 940, 885, 44, 966, 533,
    767, 364, 944, 373, 419, 746,
    275, 824, 758, 634, 978, 267,
    141, 119, 424, 907, 173, 603,
    833, 739, 231, 450, 175, 178,
    464, 697, 4, 89, 784, 553,
    90, 893, 280, 885, 545, 675,
    734, 823, 945, 737, 904, 395,
    802, 785, 841, 908, 380, 66,
    28, 0, 483, 409, 197, 650,
    // Bin capacity
    1513, 1728, 1576, 1516, 1943, 1271
  };
  const int i6_18_6_10_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    713, 633, 481, 664, 166, 764,
    834, 987, 656, 639, 924, 811,
    949, 592, 774, 922, 501, 980,
    680, 959, 737, 993, 667, 798,
    506, 494, 857, 75, 224, 302,
    416, 355, 434, 444, 906, 550,
    470, 499, 747, 883, 489, 36,
    967, 860, 97, 50, 324, 419,
    921, 678, 724, 552, 610, 760,
    600, 928, 328, 220, 963, 433,
    467, 353, 771, 291, 579, 884,
    535, 347, 91, 724, 291, 259,
    147, 872, 207, 270, 1, 551,
    72, 800, 540, 256, 247, 862,
    930, 2, 388, 368, 613, 58,
    438, 163, 190, 907, 126, 344,
    520, 841, 20, 655, 884, 620,
    427, 824, 65, 491, 613, 513,
    // Bin capacity
    1942, 2051, 1487, 1725, 1674, 1824
  };
  const int i6_18_6_11_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    389, 50, 545, 283, 535, 204,
    324, 626, 640, 522, 826, 259,
    516, 889, 959, 20, 559, 868,
    572, 928, 674, 346, 803, 589,
    157, 367, 969, 852, 862, 472,
    253, 881, 317, 456, 277, 915,
    932, 505, 983, 213, 166, 948,
    600, 279, 962, 119, 815, 625,
    614, 335, 91, 857, 397, 61,
    97, 790, 730, 790, 405, 956,
    745, 755, 450, 391, 791, 623,
    587, 777, 91, 377, 518, 327,
    829, 308, 459, 335, 512, 540,
    388, 292, 62, 54, 545, 798,
    761, 521, 658, 917, 156, 486,
    900, 521, 4, 403, 617, 60,
    237, 610, 577, 277, 137, 96,
    737, 953, 697, 667, 885, 601,
    // Bin capacity
    1784, 1922, 1826, 1458, 1815, 1745
  };
  const int i6_18_6_12_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    854, 712, 214, 514, 917, 821,
    373, 436, 949, 264, 709, 219,
    974, 916, 751, 57, 781, 915,
    967, 737, 752, 106, 169, 891,
    863, 569, 163, 91, 346, 108,
    326, 470, 873, 489, 645, 607,
    931, 219, 37, 397, 180, 737,
    287, 389, 365, 920, 82, 144,
    885, 638, 778, 721, 356, 584,
    771, 908, 812, 85, 639, 516,
    7, 385, 953, 223, 120, 38,
    835, 135, 765, 769, 411, 682,
    545, 802, 644, 958, 812, 721,
    802, 530, 752, 140, 452, 536,
    286, 526, 552, 293, 540, 57,
    72, 857, 269, 956, 149, 739,
    371, 748, 702, 842, 531, 295,
    648, 574, 852, 101, 851, 720,
    // Bin capacity
    2016, 1970, 2088, 1480, 1623, 1742
  };
  const int i6_18_6_13_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    631, 873, 226, 243, 693, 953,
    744, 759, 217, 341, 6, 661,
    625, 647, 731, 333, 624, 164,
    508, 659, 349, 539, 661, 218,
    750, 864, 69, 434, 922, 138,
    129, 905, 357, 719, 495, 197,
    481, 964, 253, 594, 388, 963,
    517, 512, 6, 871, 474, 424,
    192, 743, 17, 685, 913, 669,
    810, 960, 678, 136, 443, 931,
    13, 105, 817, 983, 350, 526,
    94, 559, 772, 140, 882, 481,
    110, 451, 278, 570, 130, 975,
    859, 780, 520, 104, 763, 209,
    165, 630, 441, 754, 735, 689,
    635, 581, 323, 599, 606, 649,
    438, 976, 585, 136, 351, 892,
    939, 50, 127, 406, 502, 291,
    // Bin capacity
    1628, 2264, 1275, 1618, 1872, 1889
  };
  const int i6_18_6_14_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    285, 587, 300, 660, 505, 645,
    733, 412, 237, 532, 249, 148,
    360, 101, 865, 375, 787, 242,
    335, 241, 256, 445, 510, 405,
    292, 627, 469, 271, 101, 745,
    781, 728, 875, 95, 725, 619,
    832, 198, 369, 829, 258, 907,
    966, 603, 602, 987, 612, 496,
    702, 284, 526, 636, 597, 97,
    639, 564, 733, 745, 220, 291,
    26, 926, 850, 258, 875, 208,
    866, 601, 97, 855, 26, 903,
    425, 973, 813, 29, 105, 896,
    838, 240, 994, 429, 411, 526,
    305, 496, 980, 272, 153, 404,
    647, 283, 192, 411, 731, 783,
    521, 193, 504, 442, 27, 546,
    872, 181, 663, 9, 562, 904,
    // Bin capacity
    1981, 1566, 1962, 1574, 1417, 1856
  };
  const int i6_18_6_15_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    549, 111, 919, 651, 932, 106,
    706, 740, 305, 992, 605, 605,
    425, 933, 390, 229, 299, 921,
    771, 281, 336, 745, 998, 455,
    955, 180, 303, 551, 49, 755,
    121, 738, 279, 264, 513, 875,
    209, 534, 705, 9, 989, 4,
    886, 466, 976, 348, 27, 393,
    140, 867, 385, 113, 45, 491,
    720, 933, 61, 841, 64, 503,
    463, 794, 362, 841, 961, 219,
    79, 58, 220, 527, 43, 64,
    421, 823, 767, 8, 280, 976,
    359, 878, 382, 29, 690, 456,
    61, 742, 900, 1000, 292, 39,
    861, 199, 444, 130, 761, 916,
    885, 256, 4, 963, 722, 908,
    310, 603, 653, 505, 54, 710,
    // Bin capacity
    1710, 1943, 1609, 1677, 1596, 1801
  };
  const int i6_18_6_16_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    50, 996, 333, 986, 513, 643,
    763, 400, 77, 494, 899, 977,
    627, 499, 316, 312, 192, 101,
    632, 872, 263, 49, 773, 443,
    336, 855, 644, 477, 538, 168,
    4, 937, 377, 594, 284, 200,
    306, 217, 924, 85, 211, 955,
    63, 161, 96, 788, 970, 553,
    563, 767, 919, 301, 155, 893,
    76, 322, 944, 502, 329, 909,
    373, 266, 558, 315, 688, 189,
    803, 417, 793, 517, 389, 947,
    113, 582, 294, 740, 803, 385,
    993, 395, 324, 850, 436, 702,
    59, 314, 846, 331, 930, 964,
    847, 222, 152, 740, 526, 952,
    350, 294, 305, 738, 207, 684,
    661, 101, 361, 88, 80, 851,
    // Bin capacity
    1474, 1666, 1649, 1723, 1726, 2227
  };
  const int i6_18_6_17_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    601, 573, 837, 973, 375, 103,
    98, 915, 285, 907, 667, 368,
    165, 949, 52, 502, 474, 578,
    541, 305, 415, 319, 481, 472,
    479, 672, 346, 617, 543, 366,
    732, 337, 284, 651, 338, 305,
    704, 347, 194, 437, 351, 897,
    153, 713, 752, 360, 405, 130,
    387, 230, 295, 49, 573, 350,
    204, 967, 102, 184, 684, 727,
    767, 599, 847, 758, 10, 218,
    809, 995, 607, 470, 473, 230,
    773, 972, 381, 218, 542, 44,
    993, 767, 565, 432, 939, 712,
    190, 403, 42, 160, 28, 428,
    882, 880, 533, 951, 711, 423,
    29, 744, 906, 332, 393, 896,
    324, 462, 52, 188, 918, 209,
    // Bin capacity
    1723, 2307, 1462, 1660, 1737, 1454
  };
  const int i6_18_6_18_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    575, 797, 726, 261, 515, 223,
    108, 427, 785, 430, 560, 788,
    629, 569, 610, 541, 36, 647,
    691, 112, 935, 135, 899, 77,
    878, 34, 561, 36, 851, 487,
    763, 329, 604, 569, 959, 545,
    656, 128, 622, 328, 285, 769,
    168, 738, 110, 672, 319, 902,
    440, 242, 297, 602, 639, 355,
    281, 325, 655, 677, 896, 386,
    82, 449, 931, 414, 24, 415,
    362, 753, 513, 704, 643, 434,
    127, 776, 187, 98, 809, 463,
    232, 414, 179, 953, 594, 870,
    223, 815, 683, 602, 303, 936,
    625, 47, 453, 798, 941, 132,
    721, 329, 1000, 627, 803, 520,
    631, 775, 607, 568, 568, 475,
    // Bin capacity
    1612, 1585, 2057, 1773, 2094, 1854
  };
  const int i6_18_6_19_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    389, 614, 256, 25, 806, 826,
    158, 918, 217, 332, 752, 519,
    863, 594, 745, 550, 453, 547,
    991, 486, 577, 86, 450, 59,
    700, 412, 977, 130, 446, 736,
    564, 807, 936, 385, 413, 706,
    454, 523, 530, 656, 95, 188,
    648, 245, 966, 521, 453, 675,
    638, 511, 526, 985, 341, 841,
    863, 674, 125, 431, 644, 312,
    32, 828, 367, 384, 413, 443,
    321, 312, 541, 549, 630, 356,
    636, 966, 884, 369, 616, 763,
    426, 417, 554, 925, 417, 114,
    364, 322, 416, 421, 950, 643,
    621, 397, 745, 556, 686, 968,
    686, 14, 262, 608, 863, 312,
    164, 294, 712, 634, 894, 802,
    // Bin capacity
    1888, 1852, 2050, 1696, 2048, 1946
  };
  const int i6_18_6_20_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    837, 944, 825, 897, 416, 696,
    603, 926, 688, 627, 489, 532,
    631, 858, 200, 417, 770, 492,
    286, 29, 361, 304, 383, 734,
    80, 983, 636, 619, 230, 184,
    694, 346, 615, 660, 188, 175,
    763, 633, 690, 477, 443, 806,
    413, 229, 25, 76, 272, 403,
    66, 378, 911, 477, 37, 29,
    533, 545, 136, 66, 794, 270,
    409, 349, 776, 943, 968, 566,
    23, 755, 202, 421, 234, 994,
    930, 67, 845, 491, 586, 784,
    483, 841, 645, 926, 605, 285,
    814, 320, 338, 701, 443, 208,
    339, 936, 60, 901, 265, 958,
    244, 147, 277, 987, 36, 366,
    352, 911, 548, 386, 780, 391,
    // Bin capacity
    1700, 2040, 1756, 2076, 1588, 1775
  };
  const int i6_18_6_21_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    375, 680, 352, 917, 909, 277,
    981, 816, 753, 203, 1000, 983,
    113, 608, 395, 406, 68, 122,
    945, 983, 91, 548, 518, 188,
    165, 314, 122, 975, 30, 121,
    137, 650, 43, 237, 587, 747,
    928, 16, 775, 610, 877, 13,
    475, 523, 478, 667, 375, 766,
    205, 701, 153, 276, 267, 586,
    898, 178, 216, 603, 797, 267,
    802, 408, 81, 166, 418, 31,
    221, 413, 712, 590, 684, 220,
    367, 107, 891, 490, 798, 30,
    800, 172, 617, 331, 125, 642,
    802, 159, 110, 57, 539, 918,
    385, 449, 879, 482, 268, 162,
    405, 5, 861, 242, 225, 860,
    757, 828, 998, 571, 675, 454,
    // Bin capacity
    1969, 1616, 1720, 1689, 1848, 1490
  };
  const int i6_18_6_22_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    218, 294, 58, 76, 639, 22,
    329, 664, 309, 242, 234, 240,
    449, 155, 193, 350, 502, 317,
    21, 513, 222, 581, 53, 309,
    37, 659, 370, 168, 644, 915,
    559, 240, 900, 270, 199, 598,
    647, 686, 362, 218, 458, 144,
    577, 257, 957, 182, 28, 147,
    44, 709, 869, 341, 470, 373,
    898, 389, 344, 708, 42, 372,
    974, 474, 6, 915, 532, 892,
    287, 935, 801, 781, 869, 868,
    607, 665, 928, 663, 481, 980,
    206, 739, 555, 913, 855, 8,
    42, 152, 680, 723, 426, 1,
    322, 757, 28, 324, 80, 981,
    357, 758, 996, 838, 607, 846,
    180, 681, 104, 748, 648, 102,
    // Bin capacity
    1374, 1978, 1766, 1839, 1580, 1651
  };
  const int i6_18_6_23_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    857, 900, 622, 690, 139, 370,
    329, 815, 631, 62, 109, 511,
    313, 77, 964, 786, 387, 702,
    319, 238, 380, 438, 7, 884,
    31, 939, 601, 798, 743, 950,
    772, 152, 848, 278, 708, 506,
    926, 615, 424, 685, 372, 439,
    804, 275, 190, 434, 773, 911,
    163, 636, 655, 275, 252, 512,
    631, 649, 341, 42, 865, 654,
    264, 62, 284, 10, 791, 455,
    399, 363, 227, 662, 352, 330,
    96, 367, 466, 895, 990, 136,
    797, 289, 843, 184, 314, 417,
    17, 544, 846, 264, 377, 13,
    626, 357, 892, 271, 493, 283,
    660, 353, 776, 928, 112, 900,
    618, 714, 578, 329, 244, 168,
    // Bin capacity
    1768, 1711, 2167, 1647, 1646, 1874
  };
  const int i6_18_6_24_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    117, 990, 185, 757, 846, 198,
    33, 813, 523, 281, 65, 477,
    21, 296, 99, 91, 423, 366,
    669, 572, 900, 204, 139, 18,
    604, 659, 178, 248, 785, 361,
    378, 61, 574, 578, 720, 126,
    750, 500, 724, 825, 48, 847,
    707, 855, 69, 791, 442, 476,
    27, 225, 338, 254, 860, 31,
    433, 362, 69, 65, 386, 20,
    763, 89, 4, 483, 533, 782,
    469, 461, 838, 803, 808, 411,
    3, 255, 538, 606, 909, 161,
    169, 176, 103, 385, 935, 177,
    879, 72, 822, 488, 199, 906,
    587, 313, 628, 504, 904, 765,
    331, 356, 734, 60, 460, 694,
    448, 582, 292, 359, 426, 979,
    // Bin capacity
    1527, 1579, 1575, 1609, 2044, 1611
  };
  const int i6_18_6_25_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    916, 42, 393, 658, 20, 193,
    608, 421, 18, 574, 196, 559,
    528, 115, 558, 80, 24, 923,
    971, 444, 435, 484, 307, 657,
    210, 760, 920, 514, 444, 465,
    616, 930, 59, 384, 928, 270,
    841, 171, 235, 707, 719, 666,
    554, 265, 121, 79, 398, 841,
    491, 456, 65, 535, 330, 153,
    781, 983, 802, 504, 533, 870,
    969, 390, 996, 297, 980, 598,
    98, 532, 944, 54, 602, 570,
    723, 516, 821, 241, 772, 694,
    609, 847, 59, 368, 816, 376,
    630, 967, 347, 347, 586, 699,
    294, 813, 113, 898, 256, 542,
    293, 138, 891, 384, 143, 631,
    350, 967, 379, 300, 587, 614,
    // Bin capacity
    2184, 2033, 1700, 1544, 1801, 2151
  };
  const int i6_18_6_26_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    942, 720, 609, 421, 104, 577,
    854, 697, 671, 461, 773, 668,
    933, 923, 952, 588, 599, 450,
    533, 605, 753, 828, 940, 542,
    901, 919, 672, 55, 198, 176,
    509, 731, 396, 259, 688, 531,
    741, 948, 295, 694, 922, 449,
    573, 517, 404, 871, 844, 498,
    754, 978, 258, 868, 960, 70,
    537, 848, 122, 306, 379, 34,
    308, 496, 481, 384, 256, 889,
    392, 994, 226, 391, 933, 827,
    428, 707, 377, 906, 722, 593,
    72, 259, 880, 347, 564, 990,
    221, 850, 594, 745, 729, 187,
    354, 470, 444, 505, 69, 186,
    489, 840, 285, 453, 708, 222,
    928, 907, 417, 467, 704, 553,
    // Bin capacity
    2199, 2816, 1856, 2006, 2330, 1773
  };
  const int i6_18_6_27_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    892, 382, 970, 841, 773, 462,
    841, 331, 222, 736, 793, 253,
    654, 248, 781, 277, 400, 647,
    166, 779, 35, 328, 464, 23,
    412, 996, 701, 314, 423, 961,
    846, 588, 477, 672, 213, 956,
    872, 422, 46, 223, 669, 859,
    73, 380, 876, 872, 398, 541,
    306, 177, 226, 746, 488, 512,
    79, 975, 981, 354, 563, 670,
    977, 572, 699, 379, 970, 911,
    898, 584, 274, 349, 436, 987,
    229, 516, 803, 557, 329, 741,
    413, 580, 345, 244, 949, 2,
    579, 740, 105, 337, 100, 860,
    13, 551, 468, 398, 208, 974,
    373, 552, 473, 449, 310, 685,
    607, 94, 582, 338, 424, 868,
    // Bin capacity
    1954, 2004, 1919, 1781, 1886, 2522
  };
  const int i6_18_6_28_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    623, 300, 54, 537, 17, 731,
    552, 227, 188, 830, 771, 181,
    399, 544, 660, 644, 398, 552,
    128, 349, 400, 765, 181, 853,
    536, 208, 654, 401, 567, 102,
    49, 1000, 90, 742, 585, 896,
    720, 722, 285, 504, 459, 453,
    167, 680, 25, 942, 394, 767,
    824, 227, 323, 745, 654, 829,
    852, 376, 815, 844, 93, 707,
    699, 72, 145, 732, 819, 258,
    469, 874, 103, 119, 37, 431,
    25, 171, 184, 812, 885, 756,
    750, 808, 931, 653, 397, 381,
    251, 947, 945, 284, 230, 509,
    818, 329, 659, 594, 287, 159,
    192, 793, 925, 816, 562, 755,
    128, 833, 490, 57, 865, 762,
    // Bin capacity
    1746, 2019, 1681, 2352, 1750, 2151
  };
  const int i6_18_6_29_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    117, 142, 676, 885, 13, 864,
    973, 888, 544, 625, 752, 77,
    61, 48, 434, 665, 545, 292,
    541, 966, 709, 55, 379, 100,
    688, 31, 257, 789, 192, 970,
    866, 744, 159, 868, 980, 272,
    698, 804, 63, 273, 346, 977,
    140, 353, 924, 198, 878, 903,
    768, 711, 435, 315, 616, 535,
    729, 408, 912, 685, 390, 504,
    990, 337, 429, 721, 607, 149,
    535, 787, 446, 701, 200, 163,
    980, 769, 406, 984, 597, 878,
    922, 64, 371, 767, 665, 666,
    841, 465, 438, 169, 767, 508,
    723, 819, 230, 134, 126, 588,
    442, 894, 815, 903, 534, 24,
    56, 643, 658, 381, 11, 866,
    // Bin capacity
    2381, 2123, 1915, 2176, 1849, 2008
  };
  const int i6_18_6_30_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    76, 239, 31, 148, 165, 692,
    416, 536, 609, 120, 40, 437,
    756, 798, 215, 353, 484, 906,
    609, 691, 494, 129, 588, 74,
    366, 740, 397, 218, 977, 556,
    424, 580, 55, 452, 472, 91,
    829, 302, 849, 269, 808, 685,
    468, 867, 776, 324, 829, 705,
    237, 828, 754, 74, 397, 437,
    127, 235, 578, 915, 192, 580,
    180, 350, 324, 638, 372, 202,
    524, 938, 655, 508, 601, 708,
    639, 75, 150, 661, 828, 633,
    179, 379, 617, 350, 397, 76,
    156, 265, 418, 243, 436, 234,
    579, 831, 867, 79, 447, 593,
    759, 404, 2, 472, 721, 41,
    599, 434, 188, 427, 394, 97,
    // Bin capacity
    1717, 2057, 1729, 1383, 1983, 1679
  };
  const int i6_18_6_31_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    968, 586, 636, 243, 339, 173,
    861, 150, 943, 248, 145, 363,
    590, 727, 345, 678, 978, 245,
    181, 137, 726, 673, 551, 295,
    62, 164, 92, 996, 502, 896,
    681, 450, 455, 303, 605, 298,
    825, 259, 546, 341, 718, 889,
    71, 287, 735, 921, 546, 722,
    443, 465, 380, 991, 431, 50,
    334, 151, 265, 951, 473, 22,
    928, 430, 351, 903, 749, 228,
    754, 31, 223, 547, 540, 466,
    673, 745, 388, 910, 568, 557,
    541, 593, 310, 567, 702, 996,
    419, 282, 445, 991, 490, 265,
    630, 281, 703, 578, 188, 170,
    458, 870, 647, 700, 864, 882,
    892, 942, 852, 518, 931, 944,
    // Bin capacity
    2252, 1649, 1975, 2633, 2254, 1848
  };
  const int i6_18_6_32_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    853, 850, 566, 490, 330, 600,
    550, 452, 453, 925, 119, 185,
    695, 862, 983, 91, 72, 182,
    170, 297, 541, 115, 626, 501,
    293, 108, 401, 284, 827, 112,
    953, 156, 692, 844, 89, 796,
    978, 586, 987, 688, 574, 196,
    605, 421, 798, 640, 504, 93,
    425, 526, 733, 491, 989, 969,
    242, 396, 197, 518, 650, 449,
    288, 657, 1, 389, 644, 231,
    540, 675, 248, 739, 568, 726,
    554, 873, 961, 56, 462, 713,
    798, 923, 278, 120, 476, 487,
    457, 619, 182, 912, 430, 44,
    952, 264, 687, 656, 22, 546,
    38, 485, 348, 954, 716, 744,
    764, 497, 505, 44, 681, 231,
    // Bin capacity
    2235, 2123, 2104, 1971, 1932, 1718
  };
  const int i6_18_6_33_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    81, 761, 872, 337, 181, 777,
    591, 387, 135, 601, 36, 369,
    744, 193, 755, 520, 788, 456,
    175, 174, 827, 56, 838, 700,
    523, 268, 522, 554, 44, 615,
    43, 744, 236, 302, 993, 870,
    535, 649, 409, 801, 601, 950,
    365, 625, 300, 969, 949, 161,
    768, 629, 63, 994, 410, 752,
    652, 707, 114, 394, 211, 587,
    984, 469, 361, 113, 983, 294,
    93, 84, 164, 730, 150, 908,
    127, 4, 867, 719, 133, 659,
    447, 843, 861, 222, 548, 782,
    11, 777, 638, 634, 884, 128,
    13, 515, 47, 501, 915, 403,
    565, 172, 350, 989, 386, 365,
    883, 393, 112, 860, 599, 953,
    // Bin capacity
    1685, 1861, 1692, 2283, 2139, 2379
  };
  const int i6_18_6_34_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    660, 66, 527, 359, 978, 916,
    328, 32, 556, 187, 691, 376,
    524, 283, 440, 613, 89, 214,
    971, 663, 450, 7, 771, 511,
    26, 728, 344, 842, 129, 622,
    610, 739, 886, 778, 940, 827,
    84, 310, 784, 868, 88, 748,
    922, 831, 857, 790, 547, 223,
    507, 181, 164, 600, 40, 585,
    694, 915, 656, 261, 269, 800,
    323, 653, 206, 572, 842, 494,
    421, 381, 845, 139, 960, 255,
    664, 979, 703, 98, 843, 123,
    903, 982, 396, 838, 679, 502,
    370, 6, 696, 330, 557, 938,
    780, 547, 371, 207, 902, 248,
    327, 385, 907, 198, 81, 832,
    516, 221, 753, 856, 315, 13,
    // Bin capacity
    2151, 1989, 2355, 1908, 2172, 2061
  };
  const int i6_18_6_35_17[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    819, 36, 915, 157, 334, 983,
    224, 605, 597, 10, 713, 250,
    673, 796, 318, 897, 194, 847,
    453, 817, 394, 717, 523, 906,
    696, 416, 604, 958, 544, 554,
    914, 431, 920, 186, 44, 519,
    148, 724, 848, 574, 345, 360,
    524, 533, 448, 332, 744, 320,
    144, 96, 337, 576, 579, 775,
    239, 719, 645, 584, 383, 247,
    733, 76, 619, 237, 472, 338,
    254, 20, 963, 119, 155, 898,
    363, 194, 908, 246, 965, 423,
    619, 106, 205, 325, 560, 130,
    463, 594, 307, 968, 357, 997,
    476, 881, 586, 70, 272, 276,
    785, 520, 362, 808, 163, 338,
    235, 857, 378, 542, 652, 124,
    // Bin capacity
    1972, 1895, 2330, 1869, 1800, 2090
  };
  const int i6_18_6_0_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    982, 518, 775, 753, 557, 355,
    676, 40, 488, 91, 608, 828,
    435, 225, 585, 935, 261, 392,
    330, 188, 14, 104, 436, 165,
    623, 220, 242, 657, 569, 949,
    169, 232, 583, 395, 186, 464,
    114, 556, 621, 561, 688, 215,
    253, 709, 623, 568, 807, 382,
    899, 73, 869, 481, 670, 33,
    62, 61, 126, 628, 250, 571,
    934, 979, 646, 696, 333, 192,
    273, 408, 952, 267, 215, 820,
    410, 560, 745, 724, 564, 836,
    533, 522, 955, 400, 948, 849,
    342, 988, 638, 951, 653, 776,
    800, 275, 662, 907, 734, 597,
    159, 897, 27, 116, 471, 41,
    731, 534, 89, 408, 377, 833,
    // Bin capacity
    1455, 1331, 1607, 1607, 1555, 1550
  };
  const int i6_18_6_1_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    224, 113, 657, 86, 357, 436,
    190, 76, 715, 577, 967, 166,
    302, 368, 16, 420, 31, 617,
    64, 983, 596, 431, 702, 562,
    301, 79, 495, 877, 699, 442,
    320, 677, 698, 903, 381, 347,
    881, 962, 281, 739, 70, 609,
    382, 381, 779, 122, 985, 38,
    361, 613, 459, 69, 380, 172,
    463, 621, 366, 589, 864, 64,
    501, 721, 150, 724, 125, 238,
    535, 600, 681, 68, 981, 268,
    5, 717, 417, 201, 336, 932,
    613, 627, 964, 730, 965, 553,
    269, 575, 387, 5, 494, 75,
    996, 638, 310, 393, 838, 759,
    54, 829, 964, 564, 346, 35,
    138, 652, 682, 593, 924, 728,
    // Bin capacity
    1111, 1723, 1619, 1362, 1759, 1186
  };
  const int i6_18_6_2_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    930, 76, 92, 149, 997, 110,
    270, 966, 178, 782, 439, 385,
    309, 703, 182, 449, 384, 408,
    228, 450, 806, 159, 100, 318,
    93, 389, 661, 917, 165, 560,
    690, 832, 576, 611, 777, 943,
    943, 816, 434, 49, 648, 48,
    436, 12, 714, 186, 900, 243,
    966, 220, 694, 448, 49, 553,
    994, 288, 721, 128, 615, 572,
    900, 952, 733, 923, 546, 304,
    746, 643, 284, 277, 597, 347,
    338, 622, 475, 387, 118, 463,
    532, 402, 415, 25, 801, 306,
    121, 276, 270, 977, 699, 930,
    19, 700, 413, 788, 621, 797,
    385, 758, 828, 860, 502, 279,
    246, 107, 222, 729, 559, 580,
    // Bin capacity
    1555, 1567, 1479, 1504, 1618, 1385
  };
  const int i6_18_6_3_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    759, 974, 983, 383, 152, 636,
    31, 282, 95, 35, 246, 220,
    563, 946, 583, 912, 989, 981,
    429, 230, 799, 256, 825, 280,
    611, 466, 242, 297, 720, 939,
    549, 59, 454, 49, 305, 848,
    541, 281, 324, 91, 526, 598,
    513, 330, 330, 424, 444, 53,
    69, 755, 600, 922, 88, 889,
    554, 918, 307, 312, 531, 155,
    228, 78, 69, 466, 997, 528,
    187, 238, 273, 273, 550, 880,
    90, 133, 758, 420, 681, 574,
    242, 1000, 149, 202, 656, 916,
    726, 911, 929, 764, 20, 854,
    359, 77, 831, 224, 889, 189,
    913, 273, 327, 888, 920, 552,
    510, 389, 514, 430, 511, 802,
    // Bin capacity
    1352, 1432, 1471, 1262, 1726, 1871
  };
  const int i6_18_6_4_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    587, 663, 448, 146, 339, 832,
    40, 302, 906, 863, 24, 480,
    528, 670, 655, 115, 19, 723,
    87, 607, 532, 939, 644, 489,
    989, 753, 599, 263, 649, 342,
    368, 396, 891, 678, 520, 386,
    287, 557, 526, 773, 679, 642,
    753, 211, 576, 87, 585, 803,
    172, 314, 77, 502, 938, 466,
    845, 580, 273, 870, 829, 196,
    957, 856, 771, 88, 50, 903,
    373, 174, 879, 442, 786, 723,
    671, 288, 626, 30, 617, 713,
    6, 437, 87, 917, 625, 397,
    747, 4, 777, 352, 835, 646,
    579, 26, 57, 711, 145, 909,
    542, 962, 710, 885, 249, 821,
    420, 560, 880, 751, 395, 43,
    // Bin capacity
    1552, 1450, 1781, 1632, 1548, 1823
  };
  const int i6_18_6_5_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    416, 906, 121, 569, 798, 347,
    338, 475, 446, 295, 327, 581,
    978, 195, 828, 947, 658, 469,
    953, 814, 477, 585, 474, 307,
    632, 281, 875, 302, 309, 805,
    888, 722, 596, 765, 741, 29,
    935, 390, 74, 374, 604, 354,
    978, 438, 270, 980, 35, 833,
    254, 505, 400, 824, 45, 181,
    140, 8, 335, 651, 795, 381,
    91, 257, 579, 837, 443, 341,
    842, 323, 125, 429, 435, 623,
    288, 853, 753, 973, 289, 161,
    783, 561, 599, 122, 718, 731,
    276, 47, 351, 366, 804, 210,
    996, 459, 892, 81, 817, 276,
    223, 770, 319, 914, 464, 443,
    242, 693, 919, 125, 658, 653,
    // Bin capacity
    1795, 1522, 1568, 1775, 1648, 1352
  };
  const int i6_18_6_6_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    18, 858, 706, 363, 859, 955,
    835, 203, 548, 208, 834, 705,
    604, 505, 755, 659, 601, 400,
    70, 298, 998, 244, 583, 296,
    954, 645, 268, 281, 202, 208,
    241, 776, 408, 210, 888, 896,
    523, 452, 643, 386, 694, 696,
    11, 771, 398, 326, 752, 568,
    177, 908, 928, 364, 805, 763,
    617, 740, 440, 310, 471, 208,
    931, 453, 298, 152, 103, 229,
    872, 276, 483, 797, 408, 598,
    367, 30, 945, 866, 313, 312,
    845, 59, 984, 456, 147, 671,
    348, 175, 479, 254, 943, 299,
    913, 281, 675, 240, 285, 455,
    801, 353, 997, 62, 89, 591,
    428, 357, 348, 202, 142, 492,
    // Bin capacity
    1689, 1439, 1997, 1128, 1612, 1651
  };
  const int i6_18_6_7_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    863, 422, 439, 698, 883, 75,
    446, 902, 443, 353, 466, 194,
    866, 704, 832, 203, 150, 107,
    664, 912, 682, 794, 891, 8,
    779, 355, 1, 116, 245, 400,
    989, 134, 39, 267, 872, 714,
    487, 860, 806, 768, 810, 25,
    309, 597, 607, 492, 390, 428,
    743, 244, 371, 387, 770, 190,
    308, 265, 759, 684, 37, 662,
    288, 640, 484, 212, 326, 389,
    70, 699, 973, 526, 94, 192,
    308, 807, 684, 848, 430, 84,
    564, 573, 669, 853, 881, 951,
    347, 814, 774, 601, 612, 846,
    805, 169, 196, 224, 337, 760,
    625, 932, 339, 908, 943, 447,
    562, 831, 724, 438, 729, 838,
    // Bin capacity
    1788, 1937, 1752, 1672, 1760, 1304
  };
  const int i6_18_6_8_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    596, 912, 365, 429, 888, 426,
    77, 452, 236, 93, 623, 62,
    281, 295, 622, 64, 855, 718,
    340, 756, 564, 548, 696, 66,
    5, 356, 620, 89, 497, 516,
    528, 291, 227, 345, 237, 39,
    97, 302, 827, 282, 323, 781,
    683, 889, 740, 135, 548, 205,
    4, 915, 954, 651, 844, 33,
    853, 577, 253, 391, 821, 530,
    590, 561, 677, 294, 80, 294,
    289, 299, 620, 457, 323, 135,
    657, 816, 403, 733, 407, 207,
    409, 276, 688, 519, 304, 876,
    348, 77, 209, 559, 895, 880,
    884, 409, 360, 974, 306, 812,
    840, 426, 387, 102, 282, 974,
    407, 587, 996, 134, 225, 153,
    // Bin capacity
    1420, 1656, 1755, 1224, 1648, 1388
  };
  const int i6_18_6_9_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    790, 877, 666, 598, 649, 618,
    546, 601, 218, 8, 904, 11,
    316, 397, 118, 199, 858, 379,
    723, 594, 545, 574, 472, 686,
    380, 623, 666, 180, 119, 791,
    545, 977, 978, 51, 978, 45,
    956, 527, 382, 584, 762, 729,
    271, 2, 555, 50, 384, 17,
    218, 455, 108, 55, 311, 272,
    380, 327, 942, 34, 331, 687,
    26, 601, 294, 479, 112, 206,
    299, 139, 277, 483, 594, 780,
    582, 19, 277, 364, 515, 641,
    405, 595, 942, 253, 70, 317,
    167, 240, 211, 370, 50, 805,
    393, 604, 196, 400, 592, 500,
    750, 115, 768, 44, 367, 789,
    534, 841, 915, 16, 407, 834,
    // Bin capacity
    1505, 1551, 1646, 862, 1540, 1655
  };
  const int i6_18_6_10_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    332, 892, 29, 41, 642, 93,
    167, 896, 746, 924, 174, 424,
    984, 75, 160, 466, 949, 506,
    62, 676, 720, 163, 51, 56,
    579, 742, 300, 947, 943, 867,
    980, 201, 949, 399, 885, 903,
    896, 539, 968, 5, 828, 654,
    175, 178, 517, 781, 185, 470,
    733, 495, 604, 239, 862, 344,
    987, 853, 505, 342, 604, 393,
    624, 636, 970, 937, 84, 831,
    771, 921, 688, 916, 914, 827,
    35, 568, 749, 54, 351, 315,
    432, 813, 839, 488, 244, 855,
    94, 329, 71, 458, 429, 389,
    886, 227, 711, 930, 253, 943,
    376, 143, 280, 852, 878, 98,
    873, 443, 502, 870, 256, 943,
    // Bin capacity
    1831, 1765, 1890, 1799, 1748, 1818
  };
  const int i6_18_6_11_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    536, 660, 289, 482, 350, 737,
    593, 554, 395, 904, 25, 588,
    653, 4, 610, 420, 350, 734,
    319, 1000, 626, 293, 589, 197,
    889, 163, 689, 306, 412, 336,
    847, 60, 80, 305, 613, 448,
    603, 398, 432, 648, 493, 336,
    243, 220, 502, 195, 51, 817,
    862, 133, 620, 675, 560, 874,
    10, 397, 680, 729, 496, 673,
    991, 681, 102, 841, 118, 283,
    370, 893, 386, 599, 392, 180,
    454, 151, 360, 259, 349, 720,
    839, 353, 932, 313, 242, 209,
    704, 114, 311, 556, 202, 338,
    509, 427, 673, 691, 927, 759,
    573, 277, 913, 820, 81, 787,
    113, 605, 294, 863, 178, 751,
    // Bin capacity
    1870, 1312, 1646, 1832, 1190, 1807
  };
  const int i6_18_6_12_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    620, 551, 626, 830, 827, 349,
    628, 836, 997, 869, 550, 530,
    29, 288, 113, 591, 588, 434,
    525, 688, 708, 902, 122, 766,
    419, 419, 140, 422, 421, 38,
    567, 384, 85, 99, 809, 293,
    227, 668, 520, 580, 199, 914,
    847, 776, 341, 336, 235, 868,
    757, 95, 893, 402, 757, 73,
    302, 290, 312, 729, 902, 486,
    356, 51, 254, 472, 784, 557,
    646, 407, 462, 518, 173, 727,
    409, 508, 791, 998, 948, 223,
    643, 41, 816, 764, 91, 700,
    885, 805, 615, 888, 13, 153,
    191, 353, 905, 149, 543, 159,
    413, 726, 186, 194, 594, 51,
    306, 310, 413, 777, 104, 165,
    // Bin capacity
    1638, 1530, 1714, 1964, 1617, 1398
  };
  const int i6_18_6_13_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    307, 136, 849, 301, 382, 307,
    537, 445, 901, 960, 812, 194,
    76, 115, 407, 203, 580, 825,
    108, 450, 259, 432, 100, 651,
    164, 554, 940, 724, 312, 751,
    215, 156, 491, 374, 423, 296,
    684, 833, 19, 519, 172, 108,
    907, 515, 941, 331, 783, 56,
    347, 933, 656, 77, 231, 985,
    501, 928, 98, 272, 889, 341,
    354, 581, 451, 853, 662, 150,
    763, 95, 686, 991, 665, 57,
    303, 769, 376, 15, 335, 748,
    347, 275, 377, 756, 491, 671,
    369, 818, 692, 806, 173, 496,
    207, 630, 78, 419, 662, 617,
    667, 84, 203, 493, 375, 46,
    794, 681, 576, 508, 421, 677,
    // Bin capacity
    1441, 1695, 1695, 1702, 1595, 1503
  };
  const int i6_18_6_14_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    682, 504, 514, 647, 386, 264,
    277, 292, 722, 630, 14, 45,
    880, 673, 200, 141, 531, 181,
    772, 736, 490, 521, 800, 105,
    402, 480, 213, 57, 242, 397,
    757, 485, 113, 493, 255, 2,
    369, 893, 83, 206, 140, 737,
    547, 707, 271, 772, 314, 533,
    121, 582, 782, 27, 47, 981,
    362, 106, 161, 86, 348, 320,
    855, 742, 648, 773, 154, 66,
    436, 469, 598, 48, 369, 764,
    80, 62, 321, 430, 183, 97,
    120, 105, 560, 972, 56, 630,
    981, 853, 310, 676, 572, 493,
    863, 831, 129, 562, 411, 28,
    407, 167, 917, 350, 639, 501,
    556, 149, 48, 637, 127, 532,
    // Bin capacity
    1799, 1679, 1346, 1526, 1062, 1269
  };
  const int i6_18_6_15_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    891, 27, 158, 521, 419, 28,
    142, 365, 661, 377, 93, 742,
    333, 106, 726, 569, 299, 75,
    592, 677, 242, 224, 454, 822,
    989, 253, 86, 603, 873, 410,
    549, 21, 383, 325, 855, 818,
    284, 409, 517, 691, 999, 217,
    371, 212, 732, 159, 457, 555,
    691, 381, 414, 607, 811, 477,
    388, 926, 771, 231, 364, 321,
    350, 953, 798, 517, 838, 947,
    595, 54, 938, 461, 97, 795,
    205, 550, 149, 98, 697, 599,
    458, 956, 606, 79, 896, 941,
    79, 224, 984, 409, 297, 691,
    903, 503, 731, 106, 890, 4,
    881, 449, 492, 456, 805, 633,
    779, 382, 694, 380, 188, 702,
    // Bin capacity
    1817, 1428, 1933, 1306, 1981, 1874
  };
  const int i6_18_6_16_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    150, 84, 207, 913, 486, 649,
    592, 295, 875, 833, 687, 522,
    155, 80, 289, 484, 247, 339,
    721, 610, 559, 610, 490, 994,
    801, 906, 667, 922, 660, 784,
    823, 604, 299, 29, 563, 158,
    388, 998, 535, 676, 216, 153,
    560, 617, 93, 513, 964, 784,
    242, 518, 101, 973, 943, 338,
    556, 867, 119, 64, 75, 101,
    507, 620, 708, 724, 756, 236,
    883, 538, 772, 620, 276, 91,
    94, 254, 779, 968, 498, 175,
    71, 660, 202, 882, 84, 332,
    775, 609, 880, 557, 978, 472,
    921, 895, 134, 33, 327, 431,
    471, 289, 850, 986, 120, 13,
    234, 328, 228, 849, 107, 380,
    // Bin capacity
    1730, 1890, 1605, 2250, 1639, 1345
  };
  const int i6_18_6_17_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    820, 325, 718, 909, 325, 993,
    298, 953, 347, 207, 554, 225,
    132, 12, 263, 230, 26, 594,
    427, 536, 816, 741, 523, 431,
    911, 909, 911, 399, 22, 778,
    566, 252, 689, 572, 703, 362,
    504, 364, 873, 671, 587, 753,
    347, 866, 780, 436, 723, 156,
    148, 859, 823, 479, 107, 456,
    863, 983, 170, 203, 291, 65,
    124, 310, 567, 155, 275, 803,
    148, 276, 738, 151, 11, 229,
    774, 679, 57, 634, 39, 268,
    906, 710, 926, 434, 813, 852,
    238, 83, 346, 45, 215, 526,
    86, 920, 267, 158, 605, 570,
    189, 119, 960, 492, 177, 405,
    659, 858, 323, 746, 918, 239,
    // Bin capacity
    1588, 1953, 2062, 1495, 1349, 1698
  };
  const int i6_18_6_18_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    804, 877, 754, 111, 588, 69,
    422, 320, 499, 442, 185, 299,
    774, 598, 456, 567, 994, 768,
    625, 62, 9, 126, 96, 507,
    637, 166, 961, 951, 448, 676,
    530, 82, 775, 484, 794, 184,
    955, 412, 101, 58, 193, 308,
    677, 43, 303, 548, 34, 487,
    762, 164, 887, 979, 442, 489,
    712, 169, 766, 705, 573, 598,
    196, 754, 709, 218, 123, 610,
    387, 552, 676, 75, 8, 849,
    948, 651, 884, 27, 257, 603,
    802, 99, 422, 30, 392, 954,
    972, 416, 92, 817, 897, 574,
    762, 181, 703, 378, 133, 204,
    108, 596, 25, 580, 188, 243,
    810, 237, 775, 629, 745, 684,
    // Bin capacity
    2337, 1255, 1927, 1520, 1395, 1791
  };
  const int i6_18_6_19_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    168, 468, 783, 419, 466, 649,
    439, 285, 795, 42, 94, 983,
    69, 211, 436, 682, 132, 517,
    77, 221, 401, 0, 357, 489,
    780, 74, 46, 695, 478, 697,
    521, 118, 381, 78, 148, 531,
    725, 44, 16, 576, 104, 359,
    906, 796, 477, 568, 100, 519,
    237, 279, 106, 331, 638, 649,
    61, 195, 471, 559, 555, 107,
    825, 658, 252, 263, 114, 635,
    741, 504, 74, 288, 642, 244,
    239, 335, 194, 944, 325, 484,
    430, 256, 385, 600, 468, 396,
    661, 9, 55, 83, 36, 454,
    251, 872, 427, 406, 374, 398,
    162, 469, 380, 136, 923, 789,
    942, 42, 631, 681, 651, 322,
    // Bin capacity
    1634, 1158, 1252, 1458, 1310, 1830
  };
  const int i6_18_6_20_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    963, 482, 311, 993, 711, 12,
    197, 298, 585, 95, 314, 545,
    917, 248, 338, 985, 162, 905,
    495, 781, 858, 995, 987, 383,
    561, 151, 358, 451, 381, 722,
    890, 304, 637, 605, 505, 576,
    51, 444, 536, 250, 60, 269,
    371, 997, 301, 395, 681, 7,
    739, 210, 288, 630, 254, 506,
    571, 301, 352, 134, 303, 783,
    610, 632, 408, 525, 570, 351,
    28, 604, 36, 226, 122, 480,
    360, 637, 96, 592, 530, 82,
    762, 828, 667, 254, 598, 656,
    868, 725, 681, 610, 990, 965,
    571, 34, 448, 271, 106, 521,
    585, 212, 181, 330, 951, 815,
    903, 887, 920, 817, 707, 531,
    // Bin capacity
    2089, 1755, 1601, 1832, 1787, 1822
  };
  const int i6_18_6_21_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    767, 915, 414, 122, 855, 414,
    233, 647, 621, 259, 364, 462,
    905, 355, 550, 473, 171, 342,
    533, 737, 27, 614, 796, 937,
    107, 187, 358, 955, 393, 461,
    67, 784, 293, 49, 698, 121,
    764, 693, 72, 940, 683, 945,
    707, 14, 945, 358, 192, 725,
    506, 956, 829, 168, 259, 774,
    436, 986, 19, 288, 793, 42,
    706, 212, 859, 339, 910, 928,
    317, 912, 507, 557, 199, 228,
    309, 588, 947, 41, 778, 903,
    760, 157, 536, 635, 268, 282,
    214, 900, 504, 804, 463, 712,
    915, 479, 681, 610, 163, 360,
    558, 140, 709, 48, 565, 322,
    132, 487, 977, 349, 630, 911,
    // Bin capacity
    1803, 2047, 1987, 1535, 1852, 1991
  };
  const int i6_18_6_22_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    795, 161, 410, 869, 757, 541,
    871, 63, 117, 613, 504, 469,
    212, 829, 965, 528, 1, 621,
    607, 413, 929, 715, 141, 761,
    194, 311, 545, 633, 323, 983,
    433, 720, 779, 842, 477, 218,
    517, 493, 700, 679, 970, 737,
    491, 703, 310, 537, 804, 334,
    160, 188, 206, 464, 739, 240,
    262, 698, 409, 868, 147, 781,
    703, 133, 820, 416, 520, 775,
    706, 78, 923, 425, 369, 261,
    830, 953, 458, 733, 153, 554,
    86, 487, 438, 135, 986, 955,
    155, 229, 713, 222, 333, 848,
    845, 699, 899, 855, 701, 0,
    694, 325, 125, 974, 490, 368,
    274, 727, 236, 79, 398, 841,
    // Bin capacity
    1797, 1670, 2030, 2153, 1792, 2092
  };
  const int i6_18_6_23_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    748, 260, 330, 404, 632, 593,
    845, 984, 788, 691, 171, 439,
    294, 777, 420, 907, 116, 818,
    653, 699, 53, 911, 790, 528,
    412, 422, 974, 954, 222, 49,
    355, 511, 140, 494, 945, 246,
    786, 707, 585, 936, 187, 463,
    212, 25, 56, 949, 824, 482,
    31, 185, 929, 578, 301, 278,
    839, 325, 754, 497, 190, 999,
    490, 991, 184, 552, 798, 863,
    535, 902, 944, 87, 695, 5,
    208, 532, 318, 657, 225, 407,
    14, 777, 88, 54, 870, 820,
    935, 344, 291, 670, 591, 181,
    314, 615, 676, 746, 244, 407,
    221, 745, 479, 927, 20, 709,
    642, 273, 688, 353, 840, 101,
    // Bin capacity
    1750, 2066, 1783, 2331, 1776, 1720
  };
  const int i6_18_6_24_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    692, 829, 427, 432, 279, 15,
    449, 589, 909, 691, 467, 291,
    449, 453, 990, 22, 545, 301,
    317, 188, 296, 925, 460, 503,
    120, 803, 742, 803, 766, 60,
    468, 997, 314, 762, 250, 635,
    728, 994, 271, 491, 203, 639,
    864, 310, 801, 845, 853, 387,
    400, 783, 953, 933, 951, 205,
    868, 733, 433, 767, 938, 149,
    797, 16, 625, 615, 533, 909,
    359, 782, 787, 163, 272, 186,
    401, 445, 865, 494, 609, 49,
    351, 773, 366, 837, 972, 541,
    141, 616, 355, 93, 446, 397,
    455, 625, 655, 897, 266, 748,
    373, 446, 769, 10, 903, 432,
    275, 469, 51, 162, 979, 716,
    // Bin capacity
    1759, 2243, 2193, 2055, 2210, 1481
  };
  const int i6_18_6_25_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    832, 296, 883, 790, 100, 703,
    250, 864, 392, 232, 490, 594,
    239, 210, 456, 581, 484, 19,
    269, 811, 817, 725, 563, 979,
    120, 604, 252, 56, 182, 682,
    205, 513, 10, 209, 208, 828,
    771, 465, 476, 140, 311, 991,
    593, 261, 668, 841, 482, 78,
    907, 686, 289, 82, 637, 395,
    479, 23, 122, 459, 506, 752,
    99, 1, 758, 27, 426, 102,
    712, 334, 256, 792, 529, 176,
    612, 727, 259, 631, 661, 838,
    974, 633, 615, 770, 214, 7,
    875, 718, 519, 466, 555, 415,
    663, 407, 984, 664, 204, 905,
    789, 100, 507, 36, 867, 995,
    753, 502, 381, 536, 823, 962,
    // Bin capacity
    2113, 1699, 1801, 1675, 1718, 2172
  };
  const int i6_18_6_26_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    464, 965, 525, 712, 47, 674,
    416, 626, 903, 187, 415, 554,
    465, 965, 955, 719, 186, 924,
    95, 399, 298, 834, 7, 183,
    859, 200, 720, 316, 297, 406,
    371, 783, 849, 975, 6, 252,
    360, 61, 821, 136, 203, 803,
    499, 219, 444, 682, 244, 832,
    134, 162, 512, 604, 6, 770,
    999, 825, 168, 760, 201, 418,
    95, 215, 178, 290, 631, 527,
    300, 30, 22, 875, 615, 700,
    660, 416, 613, 617, 998, 860,
    604, 383, 238, 382, 748, 896,
    957, 206, 283, 885, 210, 695,
    85, 347, 413, 742, 328, 373,
    575, 573, 579, 55, 600, 145,
    794, 312, 10, 305, 576, 926,
    // Bin capacity
    1834, 1615, 1792, 2116, 1327, 2297
  };
  const int i6_18_6_27_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    899, 762, 442, 403, 373, 185,
    411, 90, 795, 277, 571, 863,
    890, 656, 312, 786, 327, 325,
    406, 337, 165, 969, 38, 826,
    654, 701, 475, 808, 214, 158,
    725, 893, 275, 409, 435, 671,
    270, 342, 539, 17, 48, 603,
    400, 556, 897, 528, 155, 995,
    695, 111, 413, 612, 260, 288,
    600, 213, 777, 693, 785, 608,
    672, 263, 760, 234, 278, 450,
    124, 17, 701, 350, 887, 424,
    738, 600, 508, 102, 113, 604,
    461, 870, 111, 589, 247, 846,
    629, 500, 982, 636, 964, 251,
    362, 342, 892, 275, 240, 878,
    494, 292, 221, 662, 9, 418,
    948, 488, 30, 33, 289, 289,
    // Bin capacity
    2197, 1701, 1968, 1775, 1320, 2050
  };
  const int i6_18_6_28_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    924, 866, 162, 916, 92, 768,
    539, 961, 923, 491, 965, 82,
    139, 514, 241, 484, 635, 938,
    155, 602, 74, 608, 198, 578,
    397, 461, 205, 659, 113, 272,
    741, 219, 446, 931, 742, 8,
    847, 969, 730, 547, 735, 952,
    159, 411, 169, 716, 209, 310,
    90, 232, 690, 68, 580, 640,
    734, 928, 69, 757, 363, 589,
    437, 866, 335, 978, 468, 900,
    768, 189, 470, 207, 825, 562,
    638, 955, 485, 41, 779, 791,
    657, 10, 643, 246, 845, 420,
    661, 30, 109, 909, 828, 418,
    866, 14, 0, 779, 1, 39,
    150, 547, 107, 801, 637, 904,
    738, 514, 219, 650, 450, 823,
    // Bin capacity
    2057, 1982, 1297, 2302, 2020, 2133
  };
  const int i6_18_6_29_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    366, 875, 6, 159, 191, 639,
    257, 568, 485, 805, 851, 107,
    94, 990, 757, 291, 667, 603,
    626, 495, 672, 666, 508, 940,
    377, 847, 123, 656, 694, 995,
    39, 823, 478, 457, 322, 699,
    653, 412, 40, 420, 524, 661,
    583, 886, 845, 339, 503, 221,
    117, 197, 57, 336, 784, 842,
    726, 505, 688, 105, 844, 249,
    270, 561, 676, 814, 269, 747,
    94, 913, 933, 722, 434, 547,
    914, 485, 431, 10, 584, 225,
    476, 429, 448, 147, 493, 432,
    804, 633, 152, 328, 377, 780,
    510, 364, 929, 166, 341, 127,
    573, 140, 188, 851, 974, 972,
    44, 51, 506, 648, 431, 804,
    // Bin capacity
    1618, 2188, 1810, 1703, 2106, 2277
  };
  const int i6_18_6_30_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    190, 96, 242, 664, 831, 856,
    582, 962, 301, 80, 836, 951,
    568, 372, 374, 613, 587, 765,
    638, 707, 337, 481, 235, 296,
    371, 184, 92, 162, 459, 44,
    574, 566, 343, 573, 9, 618,
    925, 640, 80, 81, 685, 464,
    476, 143, 443, 120, 688, 798,
    588, 793, 381, 890, 943, 427,
    53, 542, 537, 898, 727, 962,
    299, 149, 978, 135, 487, 974,
    546, 965, 860, 353, 478, 167,
    421, 605, 539, 307, 521, 489,
    804, 25, 727, 461, 721, 348,
    456, 772, 157, 774, 43, 991,
    411, 764, 584, 678, 559, 481,
    378, 813, 304, 17, 483, 211,
    1, 897, 143, 88, 524, 632,
    // Bin capacity
    1795, 2166, 1609, 1598, 2127, 2270
  };
  const int i6_18_6_31_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    18, 753, 585, 569, 412, 357,
    155, 822, 226, 745, 344, 936,
    343, 899, 462, 994, 947, 212,
    222, 601, 405, 217, 943, 151,
    540, 289, 911, 938, 114, 622,
    20, 898, 421, 736, 310, 200,
    804, 520, 47, 635, 362, 301,
    312, 306, 604, 538, 703, 96,
    333, 835, 272, 510, 286, 47,
    959, 882, 513, 384, 377, 903,
    403, 218, 33, 421, 739, 155,
    226, 972, 851, 819, 640, 754,
    799, 771, 315, 415, 505, 618,
    117, 999, 409, 841, 908, 465,
    119, 470, 419, 372, 390, 59,
    42, 17, 743, 823, 654, 806,
    332, 396, 619, 311, 412, 95,
    136, 283, 907, 58, 187, 785,
    // Bin capacity
    1284, 2387, 1909, 2255, 2016, 1652
  };
  const int i6_18_6_32_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    783, 61, 415, 49, 977, 464,
    645, 848, 193, 83, 472, 566,
    67, 801, 783, 102, 580, 856,
    931, 745, 460, 824, 389, 93,
    490, 475, 648, 144, 265, 720,
    119, 160, 746, 273, 895, 850,
    89, 722, 818, 627, 669, 927,
    498, 807, 44, 382, 81, 466,
    336, 240, 142, 832, 611, 468,
    894, 377, 321, 933, 333, 769,
    14, 589, 33, 320, 987, 782,
    642, 809, 415, 951, 435, 676,
    936, 658, 481, 760, 144, 659,
    769, 704, 477, 498, 211, 473,
    182, 861, 718, 728, 282, 44,
    902, 87, 339, 89, 674, 638,
    876, 230, 508, 977, 889, 539,
    845, 313, 562, 673, 616, 905,
    // Bin capacity
    2204, 2088, 1783, 2034, 2093, 2397
  };
  const int i6_18_6_33_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    101, 68, 287, 379, 645, 186,
    30, 735, 468, 488, 35, 426,
    771, 386, 857, 848, 157, 482,
    835, 294, 737, 838, 253, 145,
    644, 581, 367, 405, 320, 449,
    994, 104, 135, 30, 30, 987,
    211, 989, 70, 234, 779, 761,
    535, 949, 196, 728, 857, 919,
    835, 650, 222, 616, 322, 475,
    848, 489, 811, 894, 979, 333,
    956, 482, 537, 86, 502, 542,
    7, 634, 677, 944, 638, 372,
    749, 613, 912, 797, 652, 300,
    134, 292, 300, 941, 948, 798,
    96, 221, 811, 113, 278, 945,
    975, 566, 883, 532, 550, 214,
    559, 48, 269, 475, 478, 24,
    81, 206, 919, 394, 164, 829,
    // Bin capacity
    2076, 1842, 2097, 2160, 1904, 2037
  };
  const int i6_18_6_34_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    297, 960, 351, 342, 227, 958,
    573, 185, 655, 758, 765, 498,
    282, 811, 896, 291, 264, 135,
    461, 540, 812, 268, 651, 306,
    241, 507, 475, 930, 736, 157,
    159, 684, 295, 167, 314, 677,
    979, 375, 17, 295, 891, 500,
    561, 965, 456, 304, 65, 758,
    323, 278, 169, 347, 867, 374,
    839, 402, 97, 474, 728, 73,
    96, 440, 479, 127, 511, 315,
    609, 912, 696, 186, 276, 622,
    277, 69, 384, 882, 821, 613,
    211, 625, 245, 566, 469, 808,
    406, 464, 23, 39, 681, 829,
    346, 736, 149, 728, 173, 257,
    960, 902, 460, 145, 607, 425,
    996, 637, 169, 633, 911, 736,
    // Bin capacity
    1925, 2344, 1525, 1671, 2224, 2020
  };
  const int i6_18_6_35_18[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    500, 174, 128, 534, 752, 455,
    775, 162, 0, 785, 639, 370,
    889, 8, 70, 357, 746, 574,
    109, 323, 189, 799, 177, 236,
    259, 469, 715, 894, 469, 682,
    15, 70, 397, 953, 1000, 682,
    904, 513, 935, 814, 357, 162,
    179, 397, 182, 734, 288, 991,
    204, 923, 424, 813, 288, 950,
    264, 74, 435, 398, 273, 399,
    194, 240, 483, 230, 727, 956,
    364, 695, 425, 607, 971, 303,
    682, 711, 244, 143, 275, 579,
    730, 911, 624, 124, 408, 111,
    138, 817, 652, 851, 611, 504,
    627, 224, 300, 812, 641, 89,
    929, 136, 682, 29, 838, 881,
    611, 708, 684, 643, 135, 541,
    // Bin capacity
    1884, 1700, 1704, 2368, 2159, 2130
  };
  const int i6_18_6_0_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    719, 837, 622, 552, 529, 716,
    326, 709, 294, 347, 949, 983,
    993, 780, 741, 70, 282, 517,
    893, 21, 948, 545, 773, 506,
    161, 963, 581, 551, 192, 497,
    550, 127, 662, 121, 719, 885,
    434, 730, 118, 662, 664, 910,
    543, 216, 544, 68, 86, 952,
    253, 232, 536, 598, 435, 668,
    525, 248, 71, 716, 496, 551,
    587, 601, 134, 572, 777, 317,
    53, 889, 86, 115, 136, 858,
    168, 264, 10, 112, 536, 45,
    136, 926, 264, 164, 840, 904,
    801, 309, 505, 591, 257, 871,
    290, 197, 685, 580, 921, 533,
    804, 458, 311, 596, 486, 323,
    222, 597, 626, 366, 255, 909,
    // Bin capacity
    1410, 1518, 1290, 1221, 1556, 1991
  };
  const int i6_18_6_1_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    10, 688, 460, 733, 287, 817,
    804, 844, 303, 611, 679, 157,
    254, 285, 672, 923, 46, 807,
    694, 309, 665, 570, 513, 209,
    936, 670, 889, 464, 97, 998,
    376, 409, 124, 244, 194, 717,
    634, 558, 111, 838, 360, 910,
    253, 96, 387, 504, 130, 923,
    182, 50, 252, 149, 818, 72,
    830, 140, 54, 21, 751, 365,
    208, 723, 63, 721, 462, 329,
    150, 843, 875, 278, 676, 651,
    394, 54, 157, 403, 776, 368,
    431, 869, 225, 289, 236, 992,
    705, 61, 350, 547, 440, 252,
    3, 708, 457, 646, 161, 235,
    562, 243, 446, 927, 972, 807,
    277, 614, 775, 201, 982, 770,
    // Bin capacity
    1297, 1375, 1223, 1527, 1445, 1748
  };
  const int i6_18_6_2_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    720, 660, 574, 785, 833, 395,
    752, 976, 566, 609, 581, 245,
    902, 813, 542, 223, 594, 363,
    213, 636, 163, 415, 904, 666,
    208, 890, 594, 597, 228, 346,
    575, 834, 836, 562, 604, 252,
    239, 207, 341, 124, 725, 464,
    174, 327, 625, 466, 142, 614,
    881, 788, 733, 723, 275, 444,
    749, 660, 318, 160, 830, 963,
    631, 809, 588, 425, 829, 342,
    494, 164, 682, 683, 40, 796,
    560, 425, 926, 73, 726, 189,
    28, 48, 45, 297, 95, 359,
    401, 357, 419, 288, 817, 429,
    884, 276, 241, 149, 42, 207,
    336, 568, 366, 674, 857, 897,
    274, 577, 542, 627, 168, 585,
    // Bin capacity
    1534, 1703, 1548, 1340, 1580, 1455
  };
  const int i6_18_6_3_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    4, 988, 61, 590, 624, 270,
    274, 368, 414, 413, 731, 498,
    490, 8, 97, 652, 199, 164,
    778, 127, 963, 921, 408, 106,
    672, 787, 404, 676, 172, 757,
    643, 750, 174, 14, 988, 876,
    638, 305, 870, 519, 225, 646,
    510, 638, 978, 220, 491, 574,
    660, 13, 509, 183, 152, 51,
    444, 682, 578, 622, 705, 819,
    481, 223, 184, 664, 33, 761,
    557, 141, 670, 223, 721, 28,
    460, 56, 701, 814, 916, 549,
    412, 814, 867, 775, 954, 16,
    8, 470, 932, 989, 547, 353,
    176, 13, 360, 572, 662, 459,
    730, 85, 96, 834, 450, 10,
    160, 979, 567, 54, 661, 686,
    // Bin capacity
    1390, 1279, 1618, 1672, 1655, 1309
  };
  const int i6_18_6_4_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    969, 580, 64, 903, 443, 289,
    972, 303, 333, 401, 303, 471,
    328, 148, 441, 733, 319, 908,
    503, 408, 30, 805, 379, 119,
    360, 451, 324, 543, 811, 515,
    977, 517, 622, 951, 827, 635,
    38, 935, 938, 603, 642, 553,
    146, 346, 207, 509, 849, 912,
    761, 438, 207, 119, 204, 71,
    844, 138, 604, 955, 298, 721,
    886, 356, 731, 948, 946, 974,
    447, 294, 330, 167, 349, 919,
    499, 299, 332, 935, 110, 95,
    533, 109, 950, 907, 764, 427,
    151, 587, 917, 45, 65, 726,
    579, 623, 811, 278, 727, 910,
    742, 222, 241, 399, 846, 851,
    459, 696, 710, 143, 64, 878,
    // Bin capacity
    1767, 1292, 1524, 1793, 1551, 1903
  };
  const int i6_18_6_5_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    93, 914, 568, 960, 844, 198,
    144, 35, 121, 538, 277, 937,
    154, 384, 169, 311, 473, 37,
    238, 273, 133, 242, 221, 998,
    952, 138, 350, 844, 488, 142,
    458, 970, 117, 310, 610, 866,
    729, 370, 398, 372, 396, 974,
    628, 568, 419, 43, 122, 196,
    343, 323, 93, 705, 890, 308,
    663, 461, 965, 643, 939, 704,
    870, 213, 706, 272, 835, 792,
    285, 210, 354, 231, 558, 197,
    581, 953, 804, 281, 68, 727,
    265, 151, 407, 31, 327, 859,
    455, 923, 861, 854, 269, 13,
    186, 610, 628, 586, 856, 137,
    737, 231, 891, 599, 70, 736,
    386, 304, 363, 376, 593, 577,
    // Bin capacity
    1430, 1406, 1461, 1435, 1547, 1645
  };
  const int i6_18_6_6_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    217, 98, 804, 342, 291, 507,
    530, 247, 439, 935, 824, 98,
    73, 470, 843, 326, 13, 88,
    327, 736, 734, 27, 108, 32,
    587, 223, 571, 390, 912, 502,
    458, 450, 456, 855, 822, 760,
    67, 752, 43, 385, 475, 399,
    459, 144, 641, 302, 377, 574,
    662, 626, 584, 298, 999, 460,
    411, 759, 718, 423, 470, 493,
    208, 845, 957, 673, 545, 137,
    494, 883, 260, 372, 587, 468,
    126, 829, 445, 632, 944, 202,
    868, 828, 118, 601, 749, 147,
    58, 883, 847, 864, 534, 309,
    887, 971, 398, 549, 258, 91,
    260, 665, 979, 215, 571, 798,
    582, 123, 170, 818, 390, 775,
    // Bin capacity
    1286, 1861, 1768, 1592, 1744, 1209
  };
  const int i6_18_6_7_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    876, 571, 793, 825, 859, 374,
    235, 863, 253, 390, 50, 830,
    294, 874, 352, 227, 677, 68,
    232, 619, 214, 18, 634, 313,
    2, 169, 87, 305, 72, 108,
    934, 314, 570, 129, 870, 980,
    621, 553, 588, 196, 894, 804,
    267, 610, 184, 564, 343, 636,
    346, 561, 404, 942, 953, 580,
    769, 963, 225, 53, 586, 76,
    219, 203, 122, 945, 757, 281,
    634, 885, 612, 28, 798, 620,
    778, 188, 422, 115, 525, 855,
    443, 910, 706, 595, 503, 455,
    369, 119, 638, 895, 728, 626,
    620, 959, 502, 645, 92, 74,
    429, 966, 543, 477, 124, 706,
    737, 22, 809, 708, 189, 14,
    // Bin capacity
    1571, 1846, 1431, 1437, 1722, 1498
  };
  const int i6_18_6_8_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    850, 95, 420, 328, 65, 70,
    92, 707, 537, 668, 569, 501,
    220, 937, 831, 336, 966, 459,
    517, 858, 434, 466, 224, 361,
    278, 552, 154, 35, 748, 458,
    50, 749, 146, 590, 454, 903,
    389, 188, 142, 447, 151, 917,
    395, 188, 435, 784, 661, 167,
    161, 747, 109, 401, 780, 836,
    909, 266, 395, 970, 423, 990,
    654, 532, 19, 348, 478, 112,
    397, 908, 914, 973, 545, 27,
    657, 238, 197, 918, 718, 819,
    271, 287, 165, 494, 219, 941,
    982, 640, 840, 501, 493, 520,
    747, 114, 504, 333, 0, 187,
    110, 337, 900, 313, 99, 470,
    319, 193, 539, 854, 857, 860,
    // Bin capacity
    1440, 1537, 1383, 1757, 1521, 1728
  };
  const int i6_18_6_9_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    690, 782, 492, 407, 982, 506,
    308, 522, 403, 788, 499, 458,
    708, 366, 908, 87, 128, 248,
    224, 600, 569, 9, 453, 788,
    958, 300, 45, 687, 169, 918,
    763, 849, 815, 253, 238, 975,
    122, 234, 472, 211, 634, 82,
    13, 630, 779, 499, 815, 493,
    162, 701, 727, 330, 275, 703,
    208, 471, 80, 816, 669, 743,
    667, 256, 801, 638, 438, 54,
    177, 386, 855, 458, 591, 829,
    516, 950, 704, 626, 438, 4,
    56, 644, 253, 279, 399, 823,
    490, 735, 666, 118, 958, 294,
    673, 30, 580, 244, 600, 506,
    10, 919, 484, 71, 691, 830,
    232, 219, 339, 881, 157, 774,
    // Bin capacity
    1268, 1743, 1812, 1345, 1660, 1822
  };
  const int i6_18_6_10_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    905, 185, 724, 59, 988, 32,
    567, 422, 12, 81, 59, 285,
    126, 515, 378, 792, 866, 608,
    553, 156, 806, 69, 122, 601,
    368, 849, 499, 353, 966, 688,
    905, 692, 271, 94, 247, 330,
    120, 801, 268, 556, 947, 486,
    149, 862, 813, 40, 185, 117,
    284, 989, 480, 474, 39, 869,
    276, 755, 871, 681, 886, 707,
    798, 415, 153, 877, 588, 938,
    156, 863, 153, 505, 976, 67,
    893, 655, 766, 478, 282, 333,
    787, 392, 981, 710, 257, 742,
    115, 897, 144, 347, 425, 333,
    822, 311, 736, 768, 108, 66,
    748, 834, 536, 715, 540, 593,
    803, 168, 487, 613, 586, 87,
    // Bin capacity
    1719, 1973, 1665, 1506, 1663, 1446
  };
  const int i6_18_6_11_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    261, 936, 668, 87, 563, 377,
    4, 118, 234, 691, 210, 651,
    757, 210, 160, 780, 24, 856,
    863, 861, 681, 970, 934, 84,
    192, 766, 987, 496, 760, 73,
    205, 573, 182, 421, 263, 475,
    167, 300, 752, 737, 848, 404,
    757, 293, 225, 71, 743, 110,
    237, 224, 991, 115, 492, 672,
    452, 726, 552, 131, 278, 178,
    722, 956, 153, 871, 207, 343,
    765, 928, 363, 976, 976, 490,
    786, 47, 190, 428, 208, 918,
    860, 587, 692, 402, 90, 124,
    102, 908, 383, 7, 972, 332,
    32, 338, 162, 398, 224, 347,
    515, 443, 180, 295, 833, 686,
    261, 878, 927, 565, 856, 144,
    // Bin capacity
    1469, 1868, 1570, 1562, 1754, 1344
  };
  const int i6_18_6_12_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    347, 563, 810, 266, 842, 717,
    782, 195, 852, 177, 788, 167,
    503, 704, 151, 995, 344, 858,
    979, 704, 270, 843, 874, 361,
    562, 996, 678, 280, 284, 820,
    556, 443, 76, 942, 603, 963,
    431, 456, 782, 46, 166, 905,
    440, 343, 364, 817, 447, 137,
    621, 232, 803, 592, 33, 499,
    12, 728, 564, 69, 56, 115,
    41, 758, 911, 183, 485, 339,
    176, 747, 971, 562, 960, 394,
    807, 256, 399, 677, 128, 354,
    262, 607, 252, 981, 781, 386,
    568, 263, 479, 436, 102, 210,
    402, 270, 535, 216, 792, 144,
    735, 450, 1, 124, 360, 550,
    384, 271, 454, 323, 282, 576,
    // Bin capacity
    1607, 1678, 1746, 1593, 1555, 1586
  };
  const int i6_18_6_13_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    795, 848, 400, 561, 477, 736,
    180, 887, 256, 129, 712, 102,
    176, 648, 357, 715, 261, 357,
    536, 524, 874, 1, 388, 0,
    132, 322, 806, 154, 377, 347,
    435, 970, 372, 126, 258, 773,
    526, 14, 563, 339, 750, 94,
    426, 805, 369, 938, 546, 761,
    878, 917, 162, 420, 124, 421,
    27, 224, 322, 947, 531, 573,
    53, 642, 692, 491, 390, 721,
    757, 153, 971, 675, 747, 457,
    373, 151, 320, 309, 253, 976,
    1, 77, 190, 424, 312, 36,
    654, 782, 874, 341, 391, 478,
    751, 527, 759, 910, 437, 669,
    384, 923, 45, 622, 398, 455,
    134, 137, 182, 581, 518, 734,
    // Bin capacity
    1360, 1799, 1604, 1636, 1483, 1637
  };
  const int i6_18_6_14_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    21, 923, 92, 951, 629, 526,
    412, 69, 973, 716, 486, 959,
    135, 934, 423, 440, 818, 979,
    401, 134, 91, 784, 653, 549,
    268, 696, 943, 593, 129, 242,
    336, 475, 546, 920, 973, 673,
    660, 981, 396, 25, 27, 11,
    46, 812, 953, 86, 267, 568,
    274, 420, 631, 527, 919, 194,
    617, 281, 77, 284, 762, 635,
    43, 334, 267, 536, 267, 126,
    958, 390, 892, 72, 465, 531,
    5, 962, 392, 674, 951, 241,
    829, 695, 995, 780, 707, 597,
    436, 688, 179, 589, 26, 210,
    924, 71, 280, 215, 612, 459,
    163, 167, 283, 149, 585, 915,
    288, 764, 225, 657, 690, 583,
    // Bin capacity
    1296, 1862, 1642, 1710, 1894, 1710
  };
  const int i6_18_6_15_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    387, 917, 806, 582, 663, 734,
    475, 628, 9, 675, 496, 713,
    373, 508, 856, 961, 362, 991,
    716, 7, 582, 699, 516, 576,
    839, 311, 974, 309, 10, 481,
    74, 921, 132, 136, 494, 476,
    121, 103, 23, 249, 369, 456,
    969, 690, 887, 149, 472, 218,
    112, 543, 766, 256, 879, 224,
    552, 810, 122, 256, 910, 481,
    904, 305, 850, 27, 255, 1000,
    132, 154, 434, 361, 295, 286,
    214, 106, 82, 106, 220, 480,
    812, 15, 988, 297, 831, 725,
    942, 258, 590, 470, 282, 148,
    88, 138, 190, 584, 507, 51,
    688, 56, 396, 227, 429, 459,
    927, 858, 925, 16, 80, 283,
    // Bin capacity
    1788, 1405, 1843, 1219, 1547, 1684
  };
  const int i6_18_6_16_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    678, 445, 89, 102, 554, 267,
    606, 411, 882, 437, 165, 164,
    740, 266, 966, 391, 950, 877,
    686, 231, 667, 411, 878, 724,
    602, 402, 21, 749, 363, 984,
    608, 162, 807, 969, 624, 787,
    563, 19, 492, 713, 126, 204,
    555, 298, 297, 33, 179, 211,
    789, 493, 276, 115, 587, 249,
    79, 926, 678, 377, 199, 102,
    470, 462, 227, 94, 30, 954,
    82, 526, 841, 745, 783, 702,
    392, 393, 294, 43, 507, 10,
    202, 565, 744, 605, 265, 873,
    683, 278, 534, 524, 462, 616,
    142, 651, 747, 288, 26, 266,
    253, 766, 595, 197, 46, 414,
    227, 604, 411, 240, 518, 834,
    // Bin capacity
    1616, 1527, 1850, 1360, 1404, 1787
  };
  const int i6_18_6_17_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    393, 559, 3, 851, 538, 58,
    762, 878, 854, 426, 105, 613,
    315, 695, 658, 730, 633, 82,
    973, 5, 867, 700, 642, 6,
    680, 633, 920, 835, 663, 950,
    278, 268, 893, 552, 941, 744,
    193, 229, 759, 750, 926, 755,
    869, 55, 743, 6, 302, 559,
    772, 998, 460, 596, 422, 857,
    648, 386, 493, 696, 24, 802,
    200, 555, 492, 596, 335, 351,
    85, 164, 542, 400, 554, 269,
    777, 235, 262, 426, 965, 978,
    115, 810, 538, 430, 652, 901,
    773, 996, 842, 302, 499, 731,
    494, 515, 156, 383, 981, 476,
    578, 991, 276, 34, 784, 775,
    568, 909, 987, 752, 36, 477,
    // Bin capacity
    1848, 1927, 2096, 1846, 1951, 2025
  };
  const int i6_18_6_18_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    13, 809, 348, 227, 859, 583,
    195, 793, 535, 380, 817, 275,
    206, 936, 491, 781, 639, 930,
    759, 552, 3, 378, 504, 887,
    621, 128, 441, 251, 858, 709,
    725, 173, 168, 915, 612, 236,
    97, 320, 133, 56, 941, 814,
    545, 267, 561, 614, 840, 180,
    847, 10, 465, 902, 244, 997,
    518, 955, 40, 700, 711, 916,
    111, 184, 612, 354, 608, 884,
    545, 524, 53, 454, 219, 428,
    750, 168, 682, 663, 111, 758,
    677, 53, 674, 773, 663, 293,
    825, 954, 256, 296, 358, 997,
    452, 313, 322, 923, 868, 461,
    528, 471, 273, 92, 42, 363,
    390, 317, 444, 929, 778, 712,
    // Bin capacity
    1732, 1559, 1279, 1906, 2099, 2247
  };
  const int i6_18_6_19_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    711, 874, 808, 906, 860, 809,
    569, 382, 262, 585, 265, 421,
    67, 95, 700, 651, 499, 62,
    645, 945, 782, 224, 71, 599,
    506, 871, 474, 207, 587, 931,
    82, 466, 676, 171, 5, 943,
    5, 229, 213, 533, 288, 994,
    578, 858, 245, 140, 702, 918,
    752, 480, 681, 565, 901, 593,
    553, 364, 651, 687, 974, 140,
    469, 990, 809, 724, 397, 847,
    612, 983, 72, 170, 738, 767,
    819, 592, 594, 307, 843, 951,
    879, 933, 644, 697, 961, 155,
    530, 507, 963, 943, 442, 983,
    112, 896, 867, 549, 230, 809,
    412, 614, 606, 943, 242, 455,
    749, 723, 107, 33, 546, 262,
    // Bin capacity
    1795, 2341, 2014, 1792, 1895, 2309
  };
  const int i6_18_6_20_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    485, 303, 704, 783, 475, 249,
    643, 802, 919, 141, 672, 15,
    544, 195, 442, 249, 270, 682,
    471, 234, 177, 354, 365, 232,
    394, 867, 320, 835, 523, 454,
    724, 799, 682, 556, 492, 730,
    628, 728, 260, 166, 54, 756,
    693, 856, 327, 242, 802, 670,
    390, 75, 259, 217, 561, 100,
    924, 767, 82, 787, 239, 487,
    21, 285, 692, 768, 705, 652,
    437, 251, 743, 115, 165, 127,
    681, 276, 95, 281, 201, 916,
    243, 721, 723, 294, 293, 385,
    459, 506, 123, 729, 936, 173,
    164, 125, 479, 632, 436, 176,
    671, 5, 191, 996, 599, 375,
    861, 290, 560, 252, 647, 246,
    // Bin capacity
    1887, 1617, 1556, 1680, 1687, 1485
  };
  const int i6_18_6_21_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    954, 733, 408, 541, 322, 811,
    580, 540, 527, 382, 30, 269,
    508, 158, 901, 434, 11, 104,
    362, 85, 674, 460, 379, 230,
    178, 549, 557, 280, 733, 823,
    722, 90, 657, 952, 384, 853,
    535, 900, 414, 775, 218, 117,
    256, 478, 343, 179, 712, 856,
    829, 37, 652, 711, 143, 387,
    500, 945, 799, 536, 702, 950,
    8, 653, 868, 305, 10, 740,
    371, 945, 111, 657, 425, 514,
    837, 964, 740, 743, 283, 300,
    275, 948, 452, 508, 70, 427,
    991, 731, 600, 83, 193, 324,
    66, 731, 383, 861, 120, 295,
    773, 584, 357, 469, 108, 434,
    254, 506, 981, 133, 911, 212,
    // Bin capacity
    1815, 2134, 2103, 1817, 1161, 1744
  };
  const int i6_18_6_22_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    402, 202, 193, 939, 928, 89,
    594, 303, 458, 990, 214, 746,
    487, 567, 540, 192, 232, 4,
    242, 811, 496, 487, 888, 395,
    741, 129, 379, 930, 628, 775,
    37, 615, 875, 808, 632, 404,
    241, 202, 158, 121, 113, 788,
    819, 361, 599, 546, 152, 942,
    208, 796, 457, 934, 174, 549,
    163, 851, 180, 754, 394, 54,
    896, 768, 399, 512, 815, 315,
    712, 641, 759, 736, 708, 530,
    455, 544, 811, 880, 4, 122,
    454, 817, 241, 663, 777, 560,
    934, 781, 437, 890, 613, 725,
    878, 879, 113, 424, 667, 69,
    511, 45, 981, 136, 340, 971,
    91, 577, 161, 857, 303, 553,
    // Bin capacity
    1803, 2011, 1675, 2400, 1746, 1747
  };
  const int i6_18_6_23_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    424, 939, 486, 252, 371, 65,
    675, 328, 343, 422, 112, 579,
    596, 293, 418, 474, 213, 711,
    511, 431, 13, 323, 956, 10,
    222, 781, 639, 410, 74, 812,
    436, 850, 415, 924, 489, 280,
    129, 487, 983, 687, 530, 118,
    722, 555, 505, 961, 656, 713,
    527, 919, 897, 46, 95, 867,
    969, 408, 729, 636, 747, 99,
    438, 714, 660, 600, 813, 778,
    947, 609, 994, 583, 269, 519,
    600, 929, 867, 46, 41, 94,
    919, 152, 924, 835, 625, 868,
    326, 136, 949, 817, 72, 150,
    295, 130, 935, 307, 410, 483,
    638, 376, 48, 558, 862, 282,
    308, 766, 734, 936, 761, 842,
    // Bin capacity
    1985, 2010, 2366, 2013, 1660, 1696
  };
  const int i6_18_6_24_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    462, 556, 974, 114, 31, 858,
    881, 857, 197, 232, 742, 432,
    880, 969, 329, 699, 462, 7,
    642, 200, 408, 733, 151, 556,
    740, 252, 761, 249, 564, 756,
    813, 818, 123, 719, 106, 24,
    640, 957, 167, 677, 774, 548,
    860, 414, 637, 642, 492, 782,
    445, 841, 866, 248, 31, 472,
    903, 657, 577, 369, 658, 759,
    325, 338, 896, 922, 218, 978,
    217, 180, 70, 228, 531, 70,
    154, 429, 523, 428, 83, 39,
    495, 463, 634, 441, 281, 414,
    478, 661, 608, 114, 922, 940,
    893, 106, 230, 122, 15, 484,
    738, 163, 461, 748, 681, 768,
    722, 469, 252, 561, 793, 972,
    // Bin capacity
    2333, 1929, 1801, 1705, 1558, 2038
  };
  const int i6_18_6_25_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    607, 215, 685, 159, 765, 989,
    421, 40, 88, 916, 517, 845,
    962, 682, 83, 304, 5, 286,
    533, 847, 711, 238, 560, 646,
    362, 888, 917, 377, 955, 45,
    340, 527, 675, 551, 587, 979,
    793, 257, 28, 47, 935, 491,
    789, 41, 379, 341, 812, 962,
    849, 0, 886, 28, 692, 59,
    822, 293, 599, 897, 965, 57,
    635, 907, 47, 103, 977, 57,
    959, 454, 733, 73, 324, 161,
    214, 75, 883, 822, 564, 952,
    162, 114, 532, 916, 754, 38,
    304, 90, 405, 903, 812, 10,
    876, 172, 536, 784, 972, 343,
    696, 557, 450, 456, 812, 428,
    904, 498, 383, 863, 839, 975,
    // Bin capacity
    2340, 1387, 1880, 1829, 2677, 1734
  };
  const int i6_18_6_26_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    273, 233, 749, 635, 79, 514,
    782, 136, 440, 581, 895, 282,
    330, 512, 513, 994, 123, 329,
    223, 875, 7, 848, 936, 282,
    231, 155, 101, 164, 39, 697,
    787, 867, 647, 746, 152, 259,
    853, 748, 88, 628, 971, 186,
    272, 100, 362, 589, 667, 375,
    813, 529, 885, 714, 970, 222,
    173, 332, 987, 46, 683, 187,
    929, 659, 671, 527, 973, 586,
    360, 342, 699, 691, 198, 860,
    0, 371, 20, 836, 482, 383,
    172, 262, 554, 193, 108, 435,
    12, 592, 530, 429, 426, 901,
    210, 657, 143, 701, 637, 364,
    321, 583, 453, 251, 215, 226,
    369, 430, 452, 956, 268, 914,
    // Bin capacity
    1494, 1761, 1744, 2212, 1853, 1681
  };
  const int i6_18_6_27_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    426, 512, 333, 925, 416, 107,
    235, 441, 179, 789, 920, 503,
    74, 89, 972, 642, 700, 846,
    466, 963, 149, 346, 686, 201,
    399, 943, 274, 334, 914, 71,
    221, 261, 498, 575, 318, 134,
    164, 567, 783, 303, 236, 223,
    765, 703, 548, 78, 884, 837,
    973, 74, 330, 119, 40, 9,
    858, 236, 296, 31, 350, 328,
    396, 470, 38, 284, 107, 804,
    91, 283, 715, 649, 20, 906,
    889, 107, 866, 806, 292, 357,
    324, 770, 904, 508, 519, 203,
    286, 680, 626, 462, 21, 380,
    61, 524, 5, 637, 231, 78,
    735, 523, 890, 954, 47, 497,
    345, 297, 507, 657, 404, 557,
    // Bin capacity
    1632, 1788, 1887, 1926, 1504, 1491
  };
  const int i6_18_6_28_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    109, 252, 125, 158, 616, 635,
    457, 973, 813, 672, 682, 979,
    324, 344, 983, 735, 889, 245,
    60, 527, 306, 180, 149, 499,
    174, 283, 438, 21, 229, 704,
    600, 180, 381, 228, 471, 494,
    782, 676, 636, 845, 981, 266,
    255, 432, 783, 880, 860, 451,
    158, 524, 391, 392, 115, 2,
    559, 104, 169, 698, 865, 728,
    909, 887, 138, 196, 861, 547,
    474, 848, 496, 924, 772, 282,
    728, 904, 554, 287, 624, 504,
    399, 682, 575, 372, 496, 833,
    346, 115, 689, 132, 421, 845,
    628, 773, 769, 854, 689, 192,
    816, 379, 243, 68, 203, 20,
    500, 911, 441, 944, 589, 528,
    // Bin capacity
    1766, 2090, 1906, 1832, 2243, 1868
  };
  const int i6_18_6_29_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    9, 731, 763, 930, 982, 311,
    78, 213, 77, 44, 699, 852,
    859, 149, 259, 300, 541, 67,
    68, 426, 478, 420, 473, 386,
    145, 103, 283, 286, 414, 58,
    401, 616, 83, 945, 253, 72,
    624, 695, 916, 467, 367, 112,
    803, 976, 550, 603, 684, 74,
    707, 59, 813, 69, 517, 121,
    556, 3, 202, 682, 101, 120,
    599, 80, 433, 113, 147, 212,
    376, 400, 931, 754, 909, 711,
    365, 644, 40, 877, 124, 681,
    278, 714, 507, 897, 724, 512,
    772, 432, 104, 945, 254, 117,
    304, 704, 936, 767, 877, 343,
    186, 798, 807, 486, 237, 908,
    146, 502, 136, 182, 380, 463,
    // Bin capacity
    1565, 1773, 1789, 2100, 1867, 1316
  };
  const int i6_18_6_30_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    690, 5, 461, 878, 873, 777,
    306, 443, 470, 999, 141, 144,
    776, 626, 489, 771, 817, 281,
    507, 994, 779, 524, 886, 183,
    760, 230, 734, 837, 200, 902,
    407, 757, 394, 285, 967, 964,
    914, 161, 911, 156, 107, 80,
    49, 550, 831, 343, 102, 870,
    905, 623, 610, 867, 119, 139,
    565, 262, 755, 345, 942, 503,
    478, 405, 419, 227, 406, 350,
    360, 4, 56, 930, 803, 46,
    756, 446, 581, 754, 962, 433,
    874, 849, 537, 273, 918, 969,
    837, 373, 643, 499, 888, 573,
    503, 711, 985, 299, 457, 717,
    186, 598, 17, 312, 278, 4,
    462, 480, 80, 880, 417, 560,
    // Bin capacity
    2240, 1846, 2113, 2206, 2228, 1841
  };
  const int i6_18_6_31_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    432, 923, 224, 355, 532, 348,
    643, 149, 50, 237, 467, 351,
    422, 998, 298, 66, 498, 155,
    163, 810, 587, 506, 454, 307,
    757, 165, 995, 39, 814, 321,
    818, 696, 131, 56, 127, 208,
    354, 694, 387, 89, 376, 235,
    31, 345, 176, 759, 910, 784,
    746, 661, 106, 962, 165, 78,
    252, 854, 280, 110, 780, 965,
    855, 792, 861, 769, 637, 48,
    109, 456, 788, 650, 920, 213,
    837, 706, 377, 915, 805, 362,
    793, 3, 934, 53, 512, 298,
    317, 495, 742, 875, 953, 309,
    984, 703, 230, 268, 697, 907,
    52, 476, 656, 11, 259, 326,
    69, 185, 577, 10, 592, 997,
    // Bin capacity
    1886, 2208, 1834, 1470, 2293, 1575
  };
  const int i6_18_6_32_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    439, 673, 915, 603, 878, 426,
    245, 609, 121, 358, 682, 674,
    462, 654, 31, 567, 559, 837,
    424, 801, 753, 564, 52, 983,
    166, 244, 472, 928, 279, 65,
    695, 13, 244, 401, 182, 729,
    518, 958, 978, 624, 433, 926,
    736, 203, 848, 351, 872, 668,
    500, 312, 647, 771, 761, 8,
    729, 340, 633, 580, 472, 523,
    186, 267, 193, 52, 765, 925,
    603, 478, 617, 11, 939, 27,
    461, 657, 479, 142, 725, 482,
    544, 539, 548, 267, 905, 158,
    17, 289, 0, 150, 82, 732,
    50, 310, 873, 78, 839, 371,
    895, 647, 347, 620, 692, 210,
    676, 134, 618, 646, 640, 507,
    // Bin capacity
    1837, 1789, 2050, 1697, 2367, 2036
  };
  const int i6_18_6_33_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    488, 166, 184, 1000, 81, 761,
    887, 986, 221, 386, 466, 910,
    301, 843, 699, 44, 519, 289,
    923, 557, 392, 352, 558, 794,
    821, 224, 955, 548, 465, 808,
    951, 871, 350, 351, 419, 394,
    911, 190, 854, 535, 736, 142,
    823, 798, 434, 66, 552, 606,
    705, 720, 107, 402, 984, 836,
    481, 378, 226, 180, 165, 507,
    719, 798, 709, 388, 165, 297,
    497, 607, 568, 17, 102, 720,
    67, 719, 691, 354, 794, 447,
    550, 514, 750, 714, 575, 408,
    526, 814, 521, 903, 473, 247,
    221, 811, 40, 160, 466, 1000,
    59, 830, 63, 834, 463, 846,
    363, 913, 281, 467, 125, 385,
    // Bin capacity
    2282, 2603, 1784, 1708, 1798, 2305
  };
  const int i6_18_6_34_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    761, 833, 136, 466, 573, 781,
    15, 821, 48, 70, 242, 736,
    577, 381, 26, 183, 471, 871,
    682, 897, 937, 367, 753, 288,
    453, 293, 871, 846, 116, 334,
    260, 645, 603, 502, 503, 729,
    924, 552, 590, 487, 768, 590,
    636, 80, 784, 16, 1000, 141,
    371, 11, 799, 612, 472, 68,
    639, 232, 851, 764, 253, 733,
    626, 712, 73, 801, 750, 222,
    988, 46, 500, 713, 15, 898,
    33, 991, 752, 34, 663, 506,
    434, 455, 299, 833, 380, 603,
    775, 839, 13, 703, 630, 182,
    876, 158, 563, 854, 473, 601,
    701, 314, 628, 615, 793, 948,
    37, 120, 645, 1, 594, 920,
    // Bin capacity
    2186, 1872, 2037, 1981, 2111, 2268
  };
  const int i6_18_6_35_19[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    13, 85, 72, 395, 24, 281,
    405, 259, 925, 251, 376, 29,
    700, 573, 115, 920, 729, 828,
    122, 233, 680, 326, 371, 574,
    405, 856, 631, 689, 333, 596,
    723, 594, 245, 133, 169, 366,
    836, 980, 971, 791, 471, 404,
    118, 706, 664, 892, 246, 134,
    755, 109, 770, 262, 350, 561,
    129, 921, 393, 873, 134, 849,
    340, 894, 339, 726, 369, 68,
    166, 689, 0, 42, 410, 449,
    55, 589, 308, 339, 428, 327,
    435, 284, 556, 882, 135, 594,
    500, 280, 346, 86, 696, 404,
    262, 757, 812, 406, 916, 218,
    996, 776, 36, 160, 920, 610,
    947, 752, 179, 474, 166, 48,
    // Bin capacity
    1780, 2326, 1810, 1946, 1630, 1652
  };
  const int i6_18_6_0_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    166, 759, 410, 263, 477, 140,
    327, 80, 905, 507, 356, 51,
    855, 430, 890, 164, 180, 220,
    57, 717, 760, 414, 939, 121,
    914, 84, 345, 784, 819, 334,
    893, 742, 704, 122, 899, 343,
    210, 361, 762, 737, 559, 482,
    60, 648, 830, 718, 203, 477,
    771, 780, 148, 606, 217, 691,
    214, 822, 30, 669, 328, 220,
    290, 48, 539, 28, 689, 225,
    111, 660, 193, 538, 910, 213,
    541, 348, 391, 673, 178, 128,
    550, 702, 839, 42, 668, 293,
    872, 615, 425, 826, 383, 532,
    519, 392, 359, 192, 614, 129,
    404, 253, 904, 727, 579, 29,
    801, 655, 875, 347, 334, 559,
    // Bin capacity
    1426, 1516, 1719, 1393, 1556, 865
  };
  const int i6_18_6_1_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    282, 526, 384, 268, 321, 185,
    320, 785, 285, 801, 371, 658,
    604, 327, 658, 93, 840, 811,
    851, 932, 701, 767, 888, 165,
    218, 427, 319, 534, 655, 766,
    453, 329, 786, 373, 385, 85,
    24, 154, 22, 199, 28, 97,
    23, 463, 128, 396, 210, 616,
    19, 329, 871, 230, 68, 146,
    318, 280, 619, 704, 197, 861,
    273, 894, 481, 464, 367, 32,
    792, 240, 218, 214, 595, 264,
    588, 768, 355, 527, 644, 310,
    811, 577, 287, 353, 425, 284,
    251, 503, 930, 2, 185, 961,
    228, 899, 69, 455, 824, 599,
    669, 396, 79, 459, 20, 297,
    302, 614, 551, 733, 830, 820,
    // Bin capacity
    1183, 1590, 1304, 1275, 1322, 1340
  };
  const int i6_18_6_2_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    376, 680, 598, 111, 439, 131,
    428, 867, 761, 302, 534, 306,
    968, 215, 497, 422, 323, 208,
    148, 113, 330, 192, 332, 364,
    232, 417, 684, 45, 63, 248,
    203, 658, 271, 384, 629, 687,
    593, 299, 911, 663, 127, 955,
    461, 176, 312, 222, 905, 129,
    258, 171, 214, 84, 598, 696,
    59, 757, 540, 292, 344, 801,
    656, 398, 229, 689, 594, 256,
    205, 474, 386, 754, 488, 676,
    27, 187, 742, 104, 389, 642,
    48, 733, 702, 349, 655, 734,
    126, 485, 394, 393, 30, 606,
    778, 97, 876, 95, 769, 375,
    709, 15, 968, 287, 439, 457,
    469, 813, 763, 392, 573, 413,
    // Bin capacity
    1147, 1285, 1731, 983, 1400, 1477
  };
  const int i6_18_6_3_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    893, 866, 758, 961, 837, 303,
    913, 384, 338, 371, 435, 165,
    21, 955, 245, 539, 729, 836,
    621, 79, 718, 406, 858, 284,
    775, 728, 888, 60, 179, 940,
    755, 654, 815, 978, 899, 515,
    328, 241, 908, 126, 9, 849,
    253, 450, 708, 909, 651, 593,
    89, 946, 362, 874, 459, 869,
    59, 747, 640, 694, 320, 783,
    930, 870, 999, 412, 351, 482,
    649, 497, 598, 237, 432, 246,
    50, 359, 773, 223, 575, 237,
    126, 75, 670, 460, 934, 397,
    279, 542, 27, 557, 979, 129,
    366, 549, 787, 36, 329, 225,
    214, 683, 995, 773, 477, 817,
    365, 108, 923, 537, 327, 234,
    // Bin capacity
    1320, 1671, 2087, 1572, 1679, 1529
  };
  const int i6_18_6_4_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    83, 705, 145, 391, 693, 16,
    2, 866, 404, 988, 429, 514,
    178, 331, 78, 574, 248, 656,
    662, 866, 699, 76, 936, 484,
    574, 957, 963, 401, 55, 968,
    304, 284, 457, 603, 16, 701,
    671, 767, 510, 71, 650, 231,
    246, 181, 796, 107, 494, 685,
    798, 940, 416, 370, 952, 105,
    998, 423, 400, 621, 461, 121,
    571, 435, 299, 962, 38, 565,
    280, 422, 322, 320, 463, 200,
    703, 839, 597, 332, 778, 394,
    618, 902, 17, 382, 492, 99,
    397, 532, 70, 524, 848, 463,
    799, 248, 882, 545, 746, 283,
    547, 329, 1000, 567, 100, 971,
    646, 785, 178, 602, 764, 751,
    // Bin capacity
    1574, 1875, 1428, 1463, 1589, 1423
  };
  const int i6_18_6_5_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    275, 195, 215, 675, 485, 396,
    740, 311, 550, 78, 792, 127,
    9, 556, 753, 839, 506, 323,
    72, 941, 519, 310, 92, 359,
    391, 447, 146, 728, 26, 281,
    381, 396, 812, 852, 949, 755,
    326, 1000, 499, 46, 876, 893,
    535, 449, 368, 418, 165, 943,
    337, 40, 65, 502, 170, 158,
    308, 565, 237, 607, 907, 523,
    627, 321, 418, 267, 549, 763,
    887, 688, 813, 997, 261, 469,
    108, 230, 804, 841, 442, 917,
    731, 446, 105, 420, 87, 236,
    459, 266, 988, 353, 229, 669,
    767, 306, 236, 260, 555, 932,
    915, 668, 719, 79, 703, 667,
    860, 454, 669, 555, 402, 842,
    // Bin capacity
    1528, 1449, 1561, 1545, 1435, 1795
  };
  const int i6_18_6_6_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    467, 313, 518, 986, 840, 657,
    347, 1000, 939, 924, 231, 651,
    157, 383, 190, 354, 478, 825,
    346, 255, 113, 716, 271, 3,
    735, 431, 207, 35, 867, 746,
    524, 592, 356, 666, 787, 172,
    734, 33, 525, 572, 604, 246,
    574, 909, 897, 351, 601, 769,
    683, 716, 299, 810, 74, 312,
    522, 3, 314, 908, 525, 830,
    527, 721, 991, 895, 292, 796,
    79, 900, 374, 691, 78, 740,
    406, 986, 165, 152, 425, 495,
    612, 889, 998, 390, 273, 725,
    97, 214, 643, 315, 984, 175,
    771, 218, 307, 736, 293, 912,
    134, 787, 46, 982, 802, 373,
    168, 706, 181, 603, 241, 370,
    // Bin capacity
    1393, 1777, 1425, 1959, 1531, 1731
  };
  const int i6_18_6_7_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    556, 204, 206, 378, 437, 668,
    850, 251, 816, 507, 780, 575,
    443, 264, 592, 430, 2, 355,
    461, 818, 996, 851, 551, 934,
    65, 896, 810, 363, 670, 286,
    328, 865, 954, 200, 188, 948,
    47, 134, 94, 552, 377, 447,
    622, 398, 34, 752, 136, 290,
    412, 350, 831, 122, 240, 556,
    335, 603, 742, 887, 104, 176,
    343, 111, 211, 200, 984, 968,
    763, 540, 449, 174, 470, 28,
    221, 465, 214, 219, 54, 254,
    865, 708, 708, 344, 141, 62,
    618, 333, 264, 38, 139, 502,
    28, 714, 196, 982, 195, 161,
    13, 625, 194, 412, 513, 301,
    330, 435, 839, 707, 289, 881,
    // Bin capacity
    1302, 1554, 1632, 1448, 1119, 1497
  };
  const int i6_18_6_8_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    450, 90, 325, 746, 349, 326,
    917, 11, 280, 117, 390, 463,
    232, 200, 451, 77, 342, 398,
    277, 10, 12, 159, 372, 962,
    194, 305, 152, 755, 712, 822,
    491, 185, 268, 751, 590, 41,
    107, 164, 560, 292, 979, 285,
    563, 721, 428, 197, 299, 760,
    158, 826, 877, 620, 399, 86,
    134, 389, 266, 101, 428, 451,
    951, 69, 201, 267, 40, 472,
    916, 555, 389, 858, 285, 821,
    298, 8, 406, 525, 562, 760,
    92, 885, 717, 295, 619, 24,
    881, 760, 688, 535, 37, 380,
    241, 435, 289, 713, 265, 52,
    60, 886, 500, 91, 554, 339,
    436, 642, 561, 411, 530, 996,
    // Bin capacity
    1332, 1286, 1327, 1352, 1396, 1519
  };
  const int i6_18_6_9_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    731, 770, 882, 135, 485, 967,
    273, 465, 223, 296, 644, 27,
    618, 524, 740, 637, 24, 504,
    256, 0, 827, 630, 262, 745,
    398, 199, 63, 762, 547, 415,
    854, 770, 391, 830, 260, 384,
    212, 308, 739, 774, 528, 832,
    970, 789, 971, 311, 447, 543,
    670, 437, 931, 998, 390, 581,
    537, 866, 500, 695, 83, 432,
    783, 51, 146, 580, 805, 182,
    54, 967, 534, 239, 644, 24,
    620, 556, 722, 168, 423, 1000,
    72, 228, 211, 467, 163, 690,
    388, 307, 225, 644, 888, 442,
    440, 897, 488, 840, 688, 362,
    839, 160, 908, 481, 349, 46,
    623, 459, 437, 696, 105, 877,
    // Bin capacity
    1697, 1591, 1806, 1850, 1406, 1645
  };
  const int i6_18_6_10_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    524, 458, 979, 912, 701, 923,
    350, 37, 530, 825, 54, 279,
    362, 250, 226, 1000, 395, 976,
    995, 274, 496, 241, 950, 682,
    341, 402, 898, 721, 416, 337,
    747, 657, 799, 27, 686, 808,
    727, 765, 595, 657, 553, 74,
    143, 508, 448, 321, 305, 903,
    946, 311, 789, 685, 577, 563,
    473, 780, 329, 397, 331, 33,
    852, 650, 749, 903, 964, 784,
    210, 152, 464, 576, 969, 375,
    726, 213, 91, 234, 643, 621,
    119, 261, 300, 415, 113, 527,
    416, 419, 339, 280, 416, 718,
    0, 799, 945, 588, 758, 385,
    217, 455, 970, 724, 56, 675,
    16, 685, 369, 563, 759, 579,
    // Bin capacity
    1497, 1481, 1892, 1846, 1769, 1878
  };
  const int i6_18_6_11_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    649, 53, 776, 738, 231, 175,
    922, 537, 675, 333, 32, 322,
    765, 898, 528, 380, 243, 437,
    61, 578, 905, 620, 277, 501,
    119, 361, 715, 942, 537, 729,
    302, 686, 942, 894, 34, 587,
    407, 917, 379, 639, 897, 568,
    182, 837, 71, 568, 357, 107,
    198, 589, 780, 555, 898, 935,
    775, 909, 902, 652, 268, 747,
    622, 2, 108, 409, 626, 549,
    64, 658, 58, 84, 302, 128,
    814, 28, 24, 950, 46, 818,
    551, 867, 475, 474, 399, 157,
    774, 357, 915, 410, 779, 732,
    947, 336, 499, 523, 986, 877,
    900, 160, 802, 935, 327, 21,
    104, 516, 368, 60, 961, 174,
    // Bin capacity
    1694, 1719, 1836, 1881, 1517, 1585
  };
  const int i6_18_6_12_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    637, 628, 913, 456, 573, 149,
    44, 521, 534, 629, 411, 927,
    927, 458, 977, 348, 760, 309,
    989, 430, 717, 90, 74, 933,
    391, 483, 735, 640, 452, 820,
    220, 608, 143, 161, 263, 389,
    452, 883, 688, 710, 129, 145,
    733, 404, 961, 261, 116, 851,
    298, 797, 802, 308, 265, 909,
    710, 782, 590, 900, 715, 464,
    265, 455, 431, 992, 936, 412,
    552, 810, 278, 565, 6, 433,
    334, 408, 345, 478, 710, 606,
    188, 23, 394, 55, 147, 720,
    132, 287, 957, 355, 611, 409,
    967, 319, 31, 952, 240, 530,
    42, 345, 383, 971, 350, 80,
    730, 837, 232, 402, 285, 332,
    // Bin capacity
    1608, 1770, 1888, 1731, 1315, 1759
  };
  const int i6_18_6_13_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    158, 51, 18, 673, 550, 512,
    899, 518, 484, 461, 830, 27,
    887, 338, 585, 79, 189, 755,
    710, 171, 423, 416, 80, 142,
    91, 819, 1000, 885, 545, 895,
    608, 208, 900, 693, 177, 398,
    788, 960, 189, 419, 486, 918,
    497, 86, 261, 965, 242, 202,
    631, 614, 404, 578, 957, 65,
    86, 857, 751, 321, 146, 165,
    576, 262, 650, 669, 514, 594,
    84, 974, 409, 536, 65, 236,
    833, 83, 637, 361, 720, 121,
    900, 47, 379, 544, 401, 802,
    451, 847, 222, 576, 809, 618,
    173, 601, 311, 811, 292, 36,
    718, 778, 154, 595, 987, 121,
    228, 414, 320, 161, 378, 383,
    // Bin capacity
    1755, 1625, 1525, 1835, 1576, 1317
  };
  const int i6_18_6_14_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    711, 756, 38, 342, 227, 738,
    249, 447, 292, 538, 637, 263,
    824, 765, 825, 809, 985, 643,
    908, 145, 624, 376, 566, 534,
    27, 682, 23, 338, 924, 441,
    425, 424, 188, 919, 20, 202,
    249, 904, 952, 123, 675, 460,
    730, 60, 679, 125, 192, 285,
    732, 906, 86, 992, 384, 43,
    382, 918, 923, 157, 918, 54,
    696, 617, 331, 392, 318, 40,
    87, 828, 204, 197, 453, 902,
    237, 0, 687, 144, 625, 897,
    13, 370, 876, 226, 275, 467,
    408, 97, 582, 505, 389, 486,
    610, 128, 90, 901, 567, 901,
    827, 452, 128, 722, 335, 937,
    583, 173, 388, 154, 155, 215,
    // Bin capacity
    1653, 1648, 1505, 1513, 1643, 1617
  };
  const int i6_18_6_15_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    397, 634, 223, 784, 765, 26,
    802, 763, 3, 810, 188, 93,
    127, 644, 178, 538, 464, 271,
    907, 734, 794, 509, 895, 375,
    76, 194, 492, 271, 384, 365,
    398, 649, 133, 8, 908, 598,
    434, 625, 953, 483, 351, 334,
    374, 58, 205, 212, 59, 743,
    642, 688, 171, 843, 912, 323,
    512, 355, 893, 311, 946, 365,
    794, 856, 597, 267, 945, 189,
    487, 119, 33, 252, 952, 669,
    39, 310, 31, 361, 158, 340,
    941, 469, 876, 280, 930, 751,
    261, 748, 161, 748, 226, 950,
    491, 298, 40, 337, 665, 615,
    55, 495, 397, 572, 774, 265,
    888, 285, 48, 554, 944, 870,
    // Bin capacity
    1654, 1711, 1194, 1561, 2198, 1561
  };
  const int i6_18_6_16_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    665, 853, 189, 941, 720, 933,
    966, 25, 447, 482, 713, 704,
    437, 438, 93, 115, 817, 160,
    297, 346, 814, 176, 870, 218,
    283, 206, 665, 647, 321, 997,
    156, 687, 24, 364, 512, 660,
    32, 447, 887, 636, 886, 329,
    138, 840, 860, 934, 647, 647,
    498, 69, 916, 16, 595, 137,
    763, 667, 65, 546, 114, 523,
    584, 556, 603, 572, 734, 667,
    692, 879, 913, 184, 915, 756,
    720, 351, 624, 365, 944, 807,
    308, 518, 111, 893, 739, 647,
    90, 380, 202, 887, 918, 380,
    563, 589, 857, 16, 829, 109,
    540, 955, 53, 106, 544, 665,
    996, 69, 553, 902, 421, 463,
    // Bin capacity
    1688, 1716, 1717, 1698, 2367, 1896
  };
  const int i6_18_6_17_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    940, 281, 943, 283, 274, 991,
    84, 500, 556, 88, 815, 833,
    205, 87, 653, 396, 688, 821,
    897, 730, 167, 1, 662, 610,
    800, 318, 577, 325, 626, 24,
    47, 691, 357, 276, 347, 200,
    857, 104, 907, 361, 220, 534,
    937, 141, 830, 358, 726, 69,
    145, 877, 647, 894, 587, 550,
    600, 733, 50, 54, 829, 292,
    131, 178, 138, 930, 322, 125,
    560, 91, 867, 787, 13, 215,
    680, 624, 131, 800, 460, 746,
    544, 759, 727, 142, 705, 955,
    401, 516, 952, 733, 623, 276,
    271, 175, 876, 187, 279, 412,
    389, 276, 541, 558, 965, 806,
    266, 134, 67, 342, 836, 326,
    // Bin capacity
    1708, 1407, 1948, 1466, 1946, 1714
  };
  const int i6_18_6_18_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    166, 882, 211, 941, 338, 328,
    443, 625, 479, 556, 651, 961,
    51, 976, 483, 161, 116, 696,
    20, 147, 414, 469, 567, 3,
    89, 267, 676, 143, 769, 440,
    207, 4, 723, 377, 136, 329,
    479, 589, 333, 494, 387, 724,
    19, 838, 51, 86, 613, 187,
    838, 739, 607, 864, 429, 409,
    568, 330, 238, 239, 935, 785,
    810, 482, 369, 316, 843, 627,
    447, 882, 5, 569, 291, 861,
    147, 409, 747, 449, 153, 493,
    127, 544, 623, 281, 775, 247,
    960, 691, 965, 724, 746, 542,
    134, 506, 42, 346, 873, 646,
    813, 629, 181, 78, 17, 379,
    677, 391, 393, 126, 238, 486,
    // Bin capacity
    1376, 1954, 1483, 1420, 1746, 1799
  };
  const int i6_18_6_19_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    781, 347, 709, 936, 907, 564,
    114, 437, 695, 672, 391, 297,
    31, 798, 463, 801, 893, 829,
    922, 135, 702, 895, 887, 101,
    224, 672, 532, 155, 447, 96,
    736, 712, 194, 384, 856, 744,
    252, 268, 584, 137, 268, 182,
    826, 274, 152, 963, 912, 983,
    185, 463, 723, 117, 663, 400,
    934, 690, 628, 444, 669, 976,
    664, 425, 952, 396, 412, 554,
    126, 270, 223, 742, 932, 315,
    636, 925, 378, 360, 873, 623,
    49, 618, 742, 928, 610, 530,
    231, 94, 910, 405, 651, 378,
    438, 200, 689, 464, 827, 734,
    680, 632, 808, 751, 866, 844,
    31, 813, 921, 215, 288, 890,
    // Bin capacity
    1559, 1740, 2183, 1937, 2450, 1992
  };
  const int i6_18_6_20_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    113, 834, 608, 136, 151, 396,
    457, 417, 729, 117, 51, 313,
    143, 311, 474, 102, 350, 660,
    555, 759, 807, 149, 244, 475,
    999, 415, 900, 566, 678, 381,
    368, 535, 510, 93, 611, 286,
    970, 120, 431, 155, 615, 611,
    145, 685, 264, 669, 660, 241,
    989, 744, 988, 925, 829, 936,
    24, 727, 133, 176, 729, 742,
    368, 334, 406, 111, 233, 582,
    604, 905, 141, 955, 398, 885,
    730, 929, 755, 397, 966, 18,
    462, 431, 555, 50, 397, 688,
    689, 993, 404, 639, 602, 690,
    246, 107, 54, 810, 157, 936,
    910, 615, 911, 165, 82, 843,
    761, 106, 428, 360, 736, 636,
    // Bin capacity
    1907, 1994, 1900, 1315, 1698, 2064
  };
  const int i6_18_6_21_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    899, 700, 708, 539, 429, 233,
    563, 736, 625, 469, 812, 882,
    675, 115, 16, 653, 274, 981,
    682, 205, 551, 284, 271, 126,
    850, 381, 662, 87, 28, 495,
    221, 162, 202, 227, 234, 687,
    753, 751, 320, 597, 970, 872,
    586, 845, 856, 991, 348, 301,
    999, 104, 709, 389, 821, 273,
    454, 712, 134, 528, 555, 136,
    789, 525, 894, 504, 659, 119,
    77, 800, 84, 749, 521, 140,
    911, 870, 430, 511, 553, 770,
    516, 63, 964, 584, 497, 592,
    950, 761, 965, 505, 758, 600,
    683, 273, 38, 727, 964, 996,
    867, 878, 792, 498, 216, 917,
    161, 418, 953, 680, 6, 711,
    // Bin capacity
    2347, 1876, 1998, 1921, 1799, 1983
  };
  const int i6_18_6_22_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    67, 723, 908, 0, 44, 260,
    178, 934, 190, 7, 725, 232,
    494, 415, 506, 133, 184, 262,
    169, 279, 373, 832, 466, 135,
    608, 885, 926, 123, 602, 691,
    348, 705, 560, 488, 632, 702,
    526, 199, 86, 420, 852, 349,
    379, 561, 492, 171, 225, 483,
    302, 788, 974, 121, 143, 437,
    74, 767, 847, 424, 832, 460,
    721, 256, 837, 413, 633, 211,
    448, 979, 141, 632, 161, 48,
    689, 405, 234, 943, 875, 578,
    124, 828, 604, 134, 292, 505,
    128, 896, 711, 85, 770, 430,
    566, 353, 915, 217, 403, 371,
    314, 74, 55, 687, 53, 571,
    318, 840, 780, 597, 440, 796,
    // Bin capacity
    1313, 2214, 2062, 1307, 1695, 1530
  };
  const int i6_18_6_23_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    676, 261, 410, 572, 497, 577,
    687, 897, 865, 54, 413, 812,
    35, 622, 594, 676, 945, 954,
    316, 437, 205, 833, 761, 732,
    343, 119, 498, 846, 370, 373,
    430, 529, 498, 85, 812, 213,
    411, 962, 252, 358, 395, 203,
    240, 389, 779, 315, 527, 51,
    507, 275, 986, 31, 522, 507,
    380, 599, 723, 605, 779, 592,
    12, 504, 235, 613, 819, 217,
    878, 593, 282, 724, 725, 928,
    238, 484, 648, 513, 23, 278,
    658, 556, 308, 523, 134, 647,
    81, 856, 142, 911, 726, 422,
    56, 674, 476, 974, 642, 196,
    735, 784, 96, 824, 219, 883,
    490, 527, 575, 333, 732, 630,
    // Bin capacity
    1471, 2064, 1758, 2007, 2059, 1890
  };
  const int i6_18_6_24_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    597, 16, 208, 421, 547, 563,
    190, 577, 191, 78, 828, 359,
    794, 833, 553, 42, 192, 804,
    775, 628, 423, 518, 272, 218,
    640, 657, 167, 86, 820, 947,
    965, 451, 634, 548, 154, 999,
    716, 29, 585, 846, 823, 210,
    878, 389, 508, 233, 72, 280,
    302, 425, 423, 457, 303, 17,
    706, 142, 965, 557, 495, 773,
    274, 239, 693, 865, 692, 202,
    381, 554, 285, 45, 549, 1,
    922, 416, 94, 203, 979, 29,
    329, 950, 927, 223, 483, 16,
    570, 798, 107, 910, 762, 396,
    563, 428, 262, 226, 625, 247,
    927, 303, 195, 856, 796, 109,
    982, 799, 383, 523, 543, 512,
    // Bin capacity
    2379, 1785, 1572, 1579, 2054, 1381
  };
  const int i6_18_6_25_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    859, 839, 577, 572, 28, 950,
    854, 776, 740, 603, 740, 878,
    903, 155, 254, 345, 186, 448,
    470, 294, 140, 738, 633, 800,
    917, 299, 902, 3, 120, 798,
    426, 758, 920, 469, 368, 941,
    794, 303, 702, 324, 774, 348,
    964, 907, 876, 622, 136, 630,
    769, 616, 561, 673, 684, 363,
    594, 662, 928, 157, 486, 822,
    550, 102, 621, 213, 349, 330,
    899, 594, 552, 686, 661, 612,
    593, 694, 685, 519, 345, 6,
    413, 986, 462, 634, 846, 398,
    893, 222, 581, 221, 71, 186,
    228, 369, 86, 788, 922, 839,
    959, 343, 541, 380, 146, 147,
    338, 985, 356, 86, 933, 668,
    // Bin capacity
    2589, 2064, 2185, 1674, 1756, 2118
  };
  const int i6_18_6_26_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    751, 22, 179, 300, 281, 166,
    376, 293, 376, 513, 476, 223,
    343, 662, 918, 399, 937, 973,
    432, 592, 924, 667, 480, 838,
    194, 747, 895, 9, 615, 638,
    421, 868, 659, 373, 285, 549,
    740, 94, 329, 512, 768, 793,
    982, 441, 85, 857, 841, 429,
    544, 66, 382, 874, 153, 858,
    407, 630, 164, 398, 443, 196,
    5, 980, 335, 683, 132, 477,
    190, 683, 2, 476, 912, 523,
    217, 180, 414, 578, 51, 815,
    836, 211, 534, 901, 194, 289,
    563, 917, 455, 361, 32, 802,
    991, 600, 480, 618, 283, 135,
    646, 262, 865, 913, 676, 594,
    212, 993, 110, 412, 107, 157,
    // Bin capacity
    1859, 1941, 1703, 2068, 1610, 1986
  };
  const int i6_18_6_27_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    807, 1000, 543, 330, 367, 112,
    852, 935, 631, 786, 96, 246,
    700, 460, 946, 719, 488, 269,
    310, 294, 928, 112, 249, 411,
    635, 68, 66, 134, 348, 731,
    335, 309, 556, 435, 303, 56,
    843, 971, 85, 603, 460, 336,
    233, 719, 837, 594, 652, 939,
    885, 643, 911, 500, 391, 83,
    450, 727, 263, 880, 556, 537,
    112, 152, 837, 116, 969, 516,
    186, 836, 604, 351, 614, 647,
    668, 907, 337, 738, 322, 684,
    807, 474, 842, 162, 422, 207,
    582, 522, 37, 956, 827, 694,
    274, 791, 60, 802, 797, 99,
    175, 684, 22, 584, 455, 822,
    937, 28, 948, 142, 658, 302,
    // Bin capacity
    2073, 2227, 2001, 1894, 1900, 1628
  };
  const int i6_18_6_28_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    467, 841, 60, 576, 326, 536,
    777, 83, 843, 638, 995, 958,
    656, 5, 20, 974, 924, 8,
    852, 857, 83, 579, 217, 304,
    815, 869, 253, 299, 23, 809,
    141, 656, 508, 538, 813, 988,
    110, 498, 328, 77, 448, 548,
    145, 255, 734, 998, 288, 280,
    690, 129, 568, 270, 791, 781,
    420, 46, 254, 991, 1, 538,
    910, 832, 909, 625, 759, 500,
    307, 280, 727, 545, 450, 780,
    725, 759, 531, 425, 579, 265,
    4, 963, 513, 733, 152, 649,
    105, 24, 1000, 321, 717, 928,
    152, 109, 689, 850, 208, 881,
    835, 380, 302, 617, 280, 435,
    482, 274, 594, 948, 30, 626,
    // Bin capacity
    1834, 1677, 1903, 2348, 1707, 2307
  };
  const int i6_18_6_29_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    980, 467, 413, 209, 331, 276,
    979, 722, 400, 193, 538, 463,
    12, 990, 476, 806, 666, 987,
    539, 386, 237, 535, 214, 952,
    460, 383, 99, 299, 740, 965,
    875, 372, 285, 484, 70, 530,
    156, 221, 304, 167, 892, 651,
    730, 38, 292, 138, 866, 311,
    343, 486, 646, 889, 347, 239,
    786, 194, 693, 464, 65, 123,
    281, 688, 995, 95, 191, 225,
    899, 340, 169, 299, 335, 879,
    760, 591, 205, 253, 516, 608,
    689, 904, 30, 945, 161, 339,
    694, 438, 883, 243, 202, 452,
    997, 172, 543, 900, 942, 143,
    386, 140, 449, 70, 702, 848,
    428, 356, 699, 374, 85, 151,
    // Bin capacity
    2364, 1696, 1681, 1584, 1691, 1966
  };
  const int i6_18_6_30_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    42, 849, 916, 977, 141, 698,
    764, 794, 256, 556, 551, 191,
    646, 290, 604, 238, 420, 418,
    840, 195, 925, 484, 184, 802,
    263, 293, 217, 709, 108, 840,
    507, 772, 892, 957, 191, 478,
    250, 52, 530, 645, 236, 588,
    775, 663, 225, 976, 416, 959,
    640, 466, 654, 888, 299, 604,
    261, 481, 269, 645, 665, 20,
    48, 48, 819, 592, 261, 257,
    382, 195, 447, 95, 681, 165,
    730, 664, 544, 606, 912, 132,
    190, 172, 423, 48, 267, 561,
    233, 342, 356, 804, 63, 336,
    572, 808, 309, 204, 766, 563,
    210, 969, 96, 183, 950, 184,
    468, 79, 952, 62, 537, 205,
    // Bin capacity
    1695, 1762, 2045, 2095, 1658, 1734
  };
  const int i6_18_6_31_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    927, 805, 728, 565, 313, 179,
    771, 863, 543, 119, 720, 326,
    37, 498, 353, 771, 642, 5,
    202, 733, 676, 59, 552, 771,
    61, 428, 932, 460, 162, 174,
    113, 47, 714, 688, 619, 493,
    163, 811, 820, 896, 662, 983,
    765, 540, 548, 744, 780, 632,
    95, 863, 991, 623, 1000, 899,
    952, 425, 633, 830, 417, 33,
    338, 200, 954, 329, 151, 171,
    296, 574, 446, 238, 660, 427,
    438, 657, 7, 67, 702, 82,
    611, 85, 341, 523, 77, 573,
    979, 316, 471, 192, 528, 421,
    861, 433, 859, 754, 542, 541,
    153, 324, 924, 249, 397, 579,
    759, 682, 879, 661, 646, 475,
    // Bin capacity
    1861, 2028, 2581, 1915, 2090, 1696
  };
  const int i6_18_6_32_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    592, 464, 245, 708, 837, 317,
    779, 165, 408, 647, 494, 436,
    82, 268, 191, 165, 519, 377,
    401, 671, 850, 651, 993, 36,
    969, 602, 253, 481, 547, 678,
    705, 680, 562, 455, 86, 572,
    935, 646, 413, 330, 268, 961,
    524, 986, 225, 369, 349, 643,
    20, 436, 887, 692, 471, 371,
    863, 492, 773, 956, 572, 633,
    210, 695, 208, 830, 52, 500,
    735, 829, 143, 55, 217, 583,
    459, 820, 270, 163, 807, 269,
    732, 893, 418, 381, 934, 451,
    527, 97, 349, 822, 117, 626,
    366, 506, 339, 925, 46, 665,
    445, 967, 676, 121, 94, 389,
    694, 807, 945, 303, 799, 397,
    // Bin capacity
    2209, 2426, 1795, 1992, 1805, 1959
  };
  const int i6_18_6_33_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    843, 298, 438, 117, 418, 55,
    74, 231, 42, 896, 142, 467,
    895, 993, 407, 499, 935, 760,
    905, 106, 531, 170, 457, 691,
    724, 575, 790, 221, 468, 302,
    943, 913, 505, 203, 625, 484,
    669, 458, 465, 73, 248, 235,
    140, 107, 778, 686, 621, 125,
    447, 538, 405, 705, 513, 983,
    677, 918, 348, 313, 835, 625,
    934, 475, 494, 934, 941, 485,
    272, 517, 620, 304, 496, 404,
    103, 896, 882, 774, 223, 826,
    886, 161, 926, 530, 221, 795,
    803, 831, 480, 575, 647, 947,
    272, 925, 852, 434, 226, 847,
    60, 220, 545, 960, 406, 863,
    778, 931, 298, 269, 637, 394,
    // Bin capacity
    2311, 2238, 2174, 1921, 2009, 2281
  };
  const int i6_18_6_34_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    403, 307, 832, 858, 795, 124,
    795, 479, 785, 756, 317, 390,
    77, 24, 137, 166, 669, 274,
    398, 269, 575, 236, 328, 157,
    187, 367, 806, 944, 943, 324,
    270, 167, 734, 458, 29, 41,
    301, 98, 782, 30, 776, 947,
    667, 567, 944, 285, 166, 822,
    155, 280, 873, 342, 721, 357,
    631, 413, 9, 631, 481, 413,
    244, 521, 328, 241, 324, 414,
    779, 217, 600, 64, 406, 448,
    209, 415, 389, 519, 581, 13,
    548, 667, 272, 776, 733, 268,
    862, 552, 143, 837, 502, 586,
    496, 730, 527, 403, 138, 891,
    621, 856, 94, 710, 804, 941,
    834, 747, 8, 625, 112, 527,
    // Bin capacity
    1894, 1715, 1974, 1984, 1971, 1773
  };
  const int i6_18_6_35_20[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    105, 37, 30, 991, 903, 357,
    617, 310, 691, 239, 357, 450,
    96, 552, 317, 304, 703, 384,
    510, 371, 868, 797, 866, 503,
    790, 109, 79, 444, 344, 22,
    100, 151, 615, 742, 962, 178,
    135, 645, 345, 773, 443, 453,
    332, 399, 135, 316, 539, 545,
    368, 15, 483, 623, 452, 661,
    155, 424, 493, 958, 283, 260,
    181, 332, 600, 365, 760, 706,
    255, 263, 803, 410, 231, 522,
    189, 718, 1, 990, 271, 670,
    254, 786, 354, 536, 286, 644,
    822, 292, 518, 223, 750, 164,
    239, 37, 420, 572, 985, 637,
    660, 823, 40, 868, 282, 18,
    23, 102, 829, 931, 70, 154,
    // Bin capacity
    1312, 1433, 1715, 2494, 2135, 1649
  };
  const int i6_18_6_0_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    856, 802, 323, 590, 826, 183,
    180, 112, 589, 117, 289, 323,
    890, 666, 681, 973, 890, 412,
    71, 653, 725, 9, 866, 859,
    932, 122, 731, 773, 583, 673,
    850, 32, 192, 548, 349, 768,
    836, 723, 787, 936, 883, 949,
    84, 533, 326, 746, 382, 336,
    404, 646, 155, 448, 710, 821,
    955, 641, 778, 619, 604, 374,
    529, 787, 34, 256, 452, 871,
    312, 873, 658, 483, 856, 130,
    861, 379, 765, 806, 429, 597,
    669, 10, 948, 122, 801, 298,
    652, 659, 852, 966, 143, 558,
    686, 0, 131, 361, 668, 654,
    507, 623, 58, 941, 770, 845,
    860, 689, 873, 389, 603, 719,
    // Bin capacity
    1856, 1492, 1601, 1681, 1851, 1729
  };
  const int i6_18_6_1_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    242, 223, 272, 584, 209, 338,
    492, 616, 180, 766, 671, 646,
    377, 583, 714, 385, 483, 301,
    346, 137, 411, 384, 577, 658,
    793, 413, 752, 835, 0, 760,
    456, 95, 284, 900, 581, 296,
    1, 897, 615, 3, 273, 709,
    14, 514, 462, 427, 509, 711,
    308, 445, 377, 956, 940, 32,
    45, 164, 380, 916, 217, 492,
    377, 251, 621, 192, 418, 577,
    103, 574, 646, 97, 943, 683,
    914, 567, 211, 293, 396, 163,
    641, 65, 400, 560, 857, 177,
    788, 50, 187, 350, 307, 217,
    644, 256, 454, 618, 549, 639,
    529, 128, 920, 892, 862, 612,
    261, 324, 624, 707, 367, 258,
    // Bin capacity
    1235, 1061, 1433, 1661, 1542, 1392
  };
  const int i6_18_6_2_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    3, 890, 44, 198, 241, 488,
    836, 803, 288, 956, 316, 280,
    130, 677, 186, 966, 366, 531,
    588, 104, 755, 567, 509, 910,
    644, 947, 312, 115, 93, 614,
    874, 788, 216, 918, 942, 352,
    428, 26, 715, 608, 127, 34,
    90, 360, 200, 313, 908, 170,
    540, 756, 940, 809, 898, 141,
    642, 681, 838, 519, 670, 922,
    906, 882, 788, 254, 616, 740,
    133, 710, 513, 158, 720, 618,
    182, 774, 176, 856, 511, 142,
    250, 721, 350, 899, 340, 533,
    895, 634, 565, 905, 462, 524,
    442, 601, 910, 496, 658, 944,
    942, 253, 412, 40, 747, 760,
    733, 837, 413, 738, 63, 655,
    // Bin capacity
    1574, 1946, 1466, 1754, 1562, 1591
  };
  const int i6_18_6_3_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    3, 9, 488, 732, 175, 572,
    278, 276, 981, 168, 765, 707,
    805, 442, 314, 807, 186, 148,
    550, 976, 350, 795, 812, 875,
    353, 182, 945, 486, 803, 583,
    107, 908, 952, 81, 408, 305,
    58, 282, 305, 233, 334, 172,
    367, 652, 651, 183, 523, 389,
    374, 222, 618, 280, 982, 558,
    807, 735, 179, 43, 951, 929,
    832, 278, 553, 562, 114, 171,
    359, 19, 617, 76, 890, 168,
    978, 681, 858, 423, 195, 25,
    734, 917, 823, 190, 743, 285,
    622, 724, 85, 862, 252, 377,
    292, 96, 214, 67, 332, 648,
    826, 242, 545, 896, 992, 729,
    147, 371, 179, 554, 642, 638,
    // Bin capacity
    1458, 1376, 1658, 1277, 1734, 1422
  };
  const int i6_18_6_4_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    359, 80, 84, 92, 114, 146,
    967, 134, 30, 671, 233, 474,
    268, 466, 301, 965, 756, 180,
    63, 927, 112, 58, 588, 71,
    523, 859, 785, 270, 795, 172,
    842, 162, 922, 712, 52, 997,
    233, 231, 235, 198, 211, 225,
    959, 335, 561, 504, 977, 247,
    686, 371, 692, 79, 207, 251,
    737, 414, 665, 0, 617, 522,
    597, 259, 762, 8, 463, 260,
    79, 31, 997, 1, 158, 571,
    576, 24, 112, 658, 562, 568,
    783, 290, 745, 942, 819, 249,
    124, 15, 239, 563, 511, 995,
    181, 428, 446, 960, 182, 195,
    489, 512, 327, 411, 910, 879,
    332, 494, 603, 518, 432, 253,
    // Bin capacity
    1525, 1046, 1494, 1320, 1489, 1258
  };
  const int i6_18_6_5_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    160, 463, 555, 260, 498, 485,
    90, 619, 445, 823, 557, 753,
    490, 195, 785, 678, 399, 259,
    469, 240, 258, 189, 646, 686,
    598, 630, 54, 241, 681, 484,
    106, 258, 247, 61, 507, 829,
    202, 653, 743, 197, 441, 497,
    178, 960, 219, 911, 9, 923,
    905, 818, 249, 320, 822, 528,
    394, 3, 684, 383, 334, 818,
    108, 459, 958, 666, 198, 501,
    108, 988, 641, 96, 181, 288,
    839, 226, 393, 375, 120, 176,
    973, 146, 618, 411, 410, 223,
    816, 245, 986, 772, 573, 750,
    787, 150, 161, 173, 524, 965,
    727, 952, 531, 472, 579, 329,
    565, 459, 766, 115, 266, 495,
    // Bin capacity
    1491, 1482, 1627, 1251, 1356, 1749
  };
  const int i6_18_6_6_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    178, 839, 140, 669, 997, 711,
    169, 319, 257, 399, 422, 929,
    336, 756, 316, 414, 782, 391,
    954, 493, 683, 45, 364, 661,
    875, 155, 863, 353, 966, 108,
    160, 810, 559, 162, 620, 156,
    494, 126, 47, 333, 79, 347,
    96, 500, 737, 985, 901, 666,
    430, 341, 579, 527, 170, 516,
    1, 971, 45, 128, 943, 299,
    616, 290, 721, 752, 206, 463,
    680, 241, 168, 285, 913, 485,
    914, 397, 988, 663, 412, 347,
    116, 213, 275, 969, 191, 996,
    251, 527, 669, 446, 525, 364,
    345, 622, 473, 234, 368, 659,
    646, 237, 512, 686, 122, 677,
    615, 957, 243, 117, 78, 170,
    // Bin capacity
    1392, 1554, 1462, 1443, 1601, 1581
  };
  const int i6_18_6_7_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    883, 268, 227, 821, 307, 488,
    319, 798, 238, 586, 173, 856,
    250, 26, 936, 574, 957, 247,
    574, 237, 581, 239, 41, 961,
    204, 928, 546, 959, 624, 842,
    883, 130, 478, 739, 429, 782,
    971, 788, 809, 93, 433, 565,
    749, 417, 237, 94, 790, 477,
    773, 277, 73, 410, 713, 728,
    516, 838, 98, 833, 250, 529,
    354, 779, 509, 177, 351, 38,
    523, 302, 119, 642, 124, 826,
    427, 14, 368, 124, 165, 317,
    646, 475, 110, 324, 669, 141,
    84, 555, 333, 216, 178, 42,
    160, 410, 249, 303, 885, 983,
    952, 175, 464, 281, 924, 611,
    219, 326, 393, 482, 429, 829,
    // Bin capacity
    1692, 1381, 1207, 1409, 1506, 1831
  };
  const int i6_18_6_8_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    467, 824, 90, 846, 551, 935,
    421, 708, 509, 678, 717, 334,
    553, 67, 412, 942, 906, 183,
    108, 405, 845, 745, 484, 627,
    730, 727, 261, 139, 408, 618,
    716, 306, 730, 539, 114, 450,
    93, 380, 95, 155, 494, 108,
    663, 895, 105, 923, 789, 290,
    437, 542, 301, 344, 839, 340,
    658, 94, 92, 419, 476, 46,
    103, 565, 320, 603, 807, 544,
    921, 261, 319, 441, 535, 347,
    868, 514, 429, 615, 906, 532,
    821, 429, 60, 956, 691, 711,
    719, 109, 517, 289, 321, 225,
    520, 526, 897, 207, 240, 299,
    731, 259, 210, 240, 363, 345,
    510, 967, 426, 649, 564, 666,
    // Bin capacity
    1808, 1545, 1192, 1752, 1837, 1368
  };
  const int i6_18_6_9_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    14, 328, 786, 29, 111, 865,
    456, 952, 875, 827, 275, 172,
    238, 469, 449, 88, 258, 472,
    879, 935, 42, 576, 921, 384,
    297, 642, 660, 937, 721, 698,
    877, 74, 897, 325, 672, 842,
    915, 433, 162, 83, 857, 992,
    187, 800, 334, 549, 878, 888,
    515, 712, 838, 235, 161, 471,
    444, 555, 140, 430, 583, 583,
    294, 657, 441, 75, 37, 402,
    840, 377, 488, 729, 229, 519,
    228, 110, 498, 426, 549, 394,
    859, 91, 390, 813, 208, 947,
    876, 435, 324, 986, 595, 870,
    876, 771, 215, 590, 575, 144,
    948, 337, 65, 522, 707, 727,
    464, 631, 130, 279, 249, 175,
    // Bin capacity
    1855, 1692, 1406, 1544, 1560, 1916
  };
  const int i6_18_6_10_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    482, 119, 795, 817, 981, 928,
    756, 966, 564, 12, 636, 281,
    756, 386, 90, 414, 994, 930,
    639, 553, 877, 793, 328, 374,
    700, 314, 558, 661, 581, 964,
    729, 147, 301, 74, 797, 494,
    218, 274, 684, 714, 130, 753,
    745, 120, 463, 545, 12, 295,
    440, 264, 302, 707, 47, 541,
    76, 695, 998, 264, 193, 827,
    372, 182, 107, 71, 121, 747,
    556, 216, 94, 76, 246, 44,
    257, 777, 84, 576, 891, 793,
    315, 77, 359, 682, 701, 547,
    376, 551, 785, 370, 663, 469,
    295, 72, 874, 96, 340, 575,
    752, 355, 127, 841, 528, 260,
    271, 560, 758, 733, 712, 27,
    // Bin capacity
    1602, 1216, 1617, 1549, 1632, 1806
  };
  const int i6_18_6_11_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    70, 536, 542, 488, 391, 639,
    918, 842, 454, 725, 245, 394,
    184, 120, 415, 941, 970, 106,
    893, 313, 1000, 888, 113, 81,
    989, 338, 642, 692, 391, 538,
    633, 367, 278, 625, 109, 80,
    384, 135, 663, 185, 80, 253,
    857, 724, 118, 761, 308, 710,
    229, 357, 339, 355, 866, 406,
    48, 916, 552, 747, 899, 409,
    657, 32, 947, 910, 289, 656,
    967, 481, 241, 673, 253, 955,
    792, 259, 67, 780, 727, 269,
    695, 894, 874, 68, 328, 382,
    284, 875, 230, 797, 384, 941,
    809, 259, 567, 173, 498, 920,
    715, 992, 946, 143, 718, 765,
    380, 830, 658, 706, 792, 99,
    // Bin capacity
    1944, 1715, 1764, 1972, 1547, 1592
  };
  const int i6_18_6_12_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    280, 115, 93, 482, 839, 671,
    481, 190, 520, 742, 103, 554,
    792, 584, 50, 39, 952, 346,
    502, 50, 432, 764, 933, 764,
    670, 223, 395, 280, 52, 800,
    640, 269, 450, 334, 876, 568,
    515, 5, 616, 5, 319, 808,
    420, 42, 815, 188, 981, 760,
    560, 557, 343, 334, 831, 31,
    892, 24, 807, 600, 504, 30,
    815, 619, 863, 416, 790, 516,
    973, 501, 313, 266, 517, 511,
    484, 920, 376, 237, 970, 719,
    509, 24, 908, 405, 581, 184,
    80, 707, 746, 765, 992, 513,
    41, 515, 722, 333, 538, 799,
    133, 560, 989, 110, 413, 809,
    810, 905, 870, 797, 954, 214,
    // Bin capacity
    1792, 1272, 1925, 1325, 2268, 1792
  };
  const int i6_18_6_13_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    402, 274, 968, 619, 845, 148,
    785, 712, 360, 552, 363, 346,
    824, 221, 785, 502, 977, 886,
    748, 169, 978, 930, 590, 893,
    331, 436, 508, 177, 425, 756,
    624, 894, 843, 598, 137, 550,
    216, 108, 959, 189, 296, 708,
    647, 829, 518, 780, 961, 342,
    187, 170, 519, 116, 634, 8,
    901, 194, 64, 88, 107, 737,
    312, 889, 173, 536, 865, 678,
    479, 452, 958, 71, 593, 892,
    809, 369, 682, 388, 907, 841,
    457, 686, 926, 801, 169, 178,
    739, 480, 997, 298, 469, 723,
    963, 818, 325, 675, 640, 207,
    879, 886, 374, 487, 797, 229,
    849, 138, 898, 39, 806, 615,
    // Bin capacity
    2101, 1644, 2229, 1478, 1993, 1834
  };
  const int i6_18_6_14_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    644, 89, 314, 429, 853, 326,
    909, 227, 300, 387, 227, 499,
    701, 403, 516, 118, 263, 593,
    343, 451, 538, 223, 22, 760,
    436, 523, 248, 106, 786, 182,
    310, 103, 343, 83, 211, 79,
    37, 935, 590, 293, 56, 47,
    512, 963, 635, 931, 873, 629,
    627, 878, 658, 671, 361, 854,
    286, 118, 590, 82, 945, 121,
    230, 623, 175, 912, 591, 189,
    832, 559, 22, 791, 524, 926,
    148, 50, 509, 664, 182, 9,
    387, 57, 413, 349, 946, 842,
    392, 581, 325, 200, 912, 733,
    302, 660, 743, 586, 771, 286,
    826, 673, 64, 991, 278, 603,
    891, 71, 41, 994, 236, 303,
    // Bin capacity
    1675, 1514, 1335, 1674, 1718, 1517
  };
  const int i6_18_6_15_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    204, 212, 790, 397, 100, 659,
    634, 429, 122, 583, 963, 794,
    59, 529, 769, 930, 802, 656,
    461, 644, 793, 811, 402, 137,
    956, 177, 755, 852, 173, 18,
    581, 543, 552, 731, 942, 738,
    889, 304, 431, 954, 351, 795,
    147, 256, 526, 170, 883, 961,
    769, 472, 222, 265, 520, 131,
    427, 115, 288, 50, 425, 540,
    844, 924, 986, 96, 640, 271,
    803, 248, 915, 334, 292, 435,
    94, 239, 741, 162, 430, 860,
    141, 370, 26, 680, 288, 293,
    294, 858, 313, 467, 425, 87,
    33, 454, 504, 654, 854, 876,
    369, 125, 252, 308, 8, 325,
    91, 773, 930, 744, 164, 784,
    // Bin capacity
    1495, 1471, 1901, 1762, 1661, 1794
  };
  const int i6_18_6_16_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    985, 923, 137, 591, 809, 656,
    317, 970, 277, 265, 616, 318,
    85, 386, 468, 358, 502, 715,
    118, 311, 226, 909, 548, 383,
    379, 876, 69, 228, 476, 457,
    845, 793, 734, 441, 51, 711,
    257, 709, 103, 659, 637, 997,
    363, 90, 974, 612, 542, 783,
    108, 178, 325, 188, 805, 328,
    49, 150, 752, 595, 592, 218,
    120, 185, 356, 613, 129, 905,
    848, 622, 741, 889, 124, 517,
    821, 996, 943, 951, 334, 8,
    414, 527, 931, 429, 945, 587,
    7, 162, 366, 2, 483, 965,
    76, 395, 261, 268, 913, 188,
    736, 874, 80, 894, 719, 177,
    516, 190, 717, 99, 390, 704,
    // Bin capacity
    1362, 1806, 1636, 1739, 1859, 1860
  };
  const int i6_18_6_17_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    572, 814, 408, 400, 145, 772,
    281, 749, 576, 127, 325, 271,
    658, 511, 688, 0, 738, 342,
    881, 10, 26, 366, 321, 826,
    940, 163, 104, 784, 585, 911,
    679, 440, 236, 354, 374, 515,
    719, 632, 79, 964, 880, 905,
    546, 236, 315, 598, 961, 417,
    369, 222, 99, 984, 527, 893,
    740, 560, 177, 198, 483, 218,
    172, 274, 530, 380, 13, 386,
    783, 560, 33, 385, 920, 802,
    755, 793, 790, 344, 398, 49,
    879, 515, 548, 772, 822, 599,
    711, 703, 940, 444, 655, 17,
    821, 400, 751, 126, 226, 10,
    637, 718, 801, 391, 528, 230,
    577, 906, 431, 858, 872, 449,
    // Bin capacity
    2286, 1796, 1469, 1653, 1906, 1680
  };
  const int i6_18_6_18_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    512, 416, 760, 177, 595, 569,
    318, 191, 310, 103, 535, 384,
    734, 622, 913, 248, 380, 51,
    695, 906, 270, 241, 886, 975,
    532, 21, 632, 379, 468, 430,
    510, 718, 241, 348, 522, 502,
    137, 740, 674, 42, 307, 258,
    208, 927, 134, 898, 765, 289,
    341, 571, 706, 529, 744, 484,
    377, 961, 662, 739, 936, 956,
    485, 403, 732, 914, 535, 995,
    808, 771, 826, 691, 460, 481,
    395, 970, 88, 524, 530, 752,
    438, 985, 689, 643, 934, 381,
    807, 107, 167, 283, 368, 335,
    931, 180, 23, 967, 509, 622,
    25, 989, 639, 282, 969, 66,
    545, 72, 458, 277, 155, 354,
    // Bin capacity
    1731, 2075, 1756, 1630, 2085, 1748
  };
  const int i6_18_6_19_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    99, 444, 706, 661, 620, 542,
    796, 530, 596, 454, 425, 736,
    707, 602, 536, 389, 261, 294,
    206, 742, 214, 445, 995, 647,
    266, 676, 369, 560, 412, 617,
    135, 196, 36, 11, 511, 409,
    73, 29, 439, 520, 865, 969,
    354, 205, 71, 520, 791, 263,
    750, 806, 107, 642, 885, 24,
    863, 999, 900, 302, 977, 765,
    622, 935, 105, 846, 32, 46,
    237, 371, 409, 887, 52, 561,
    504, 735, 317, 228, 397, 319,
    596, 188, 377, 323, 771, 320,
    455, 632, 145, 665, 819, 976,
    261, 686, 841, 607, 193, 495,
    445, 608, 323, 893, 46, 377,
    401, 414, 810, 10, 958, 855,
    // Bin capacity
    1542, 1944, 1449, 1778, 1986, 1828
  };
  const int i6_18_6_20_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    555, 168, 336, 318, 376, 814,
    543, 488, 224, 383, 116, 480,
    396, 963, 965, 369, 347, 258,
    426, 449, 656, 854, 87, 933,
    296, 418, 377, 154, 163, 193,
    97, 254, 869, 384, 795, 395,
    498, 313, 164, 834, 640, 962,
    18, 982, 317, 235, 575, 564,
    941, 907, 46, 683, 434, 270,
    525, 134, 459, 89, 97, 742,
    912, 534, 539, 760, 828, 451,
    647, 325, 101, 262, 676, 738,
    412, 555, 266, 924, 894, 814,
    817, 715, 613, 312, 597, 967,
    816, 726, 924, 545, 687, 114,
    680, 718, 439, 917, 444, 284,
    998, 582, 48, 70, 905, 807,
    628, 656, 392, 376, 91, 862,
    // Bin capacity
    2041, 1978, 1547, 1694, 1751, 2130
  };
  const int i6_18_6_21_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    994, 495, 803, 812, 468, 720,
    407, 984, 402, 474, 757, 567,
    936, 12, 338, 333, 819, 92,
    976, 668, 428, 0, 427, 713,
    361, 242, 932, 801, 367, 43,
    713, 572, 799, 513, 111, 317,
    988, 552, 603, 81, 335, 717,
    368, 603, 876, 577, 841, 705,
    256, 864, 374, 858, 466, 789,
    524, 196, 101, 712, 704, 447,
    339, 279, 424, 679, 49, 922,
    293, 905, 328, 189, 454, 752,
    276, 587, 661, 565, 606, 904,
    719, 955, 215, 280, 369, 638,
    144, 114, 62, 236, 214, 701,
    958, 272, 536, 476, 320, 619,
    292, 381, 349, 568, 961, 778,
    754, 480, 123, 420, 378, 972,
    // Bin capacity
    2077, 1848, 1685, 1730, 1744, 2299
  };
  const int i6_18_6_22_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    309, 517, 136, 783, 551, 998,
    103, 727, 204, 581, 737, 973,
    544, 58, 459, 375, 854, 468,
    430, 304, 874, 259, 905, 561,
    288, 663, 492, 208, 81, 439,
    403, 110, 115, 964, 501, 856,
    405, 75, 486, 864, 874, 559,
    854, 19, 501, 658, 943, 43,
    557, 747, 763, 30, 702, 581,
    559, 490, 674, 319, 309, 404,
    854, 133, 549, 389, 648, 530,
    823, 422, 29, 822, 827, 445,
    861, 386, 740, 873, 539, 244,
    596, 98, 182, 628, 612, 207,
    549, 732, 256, 657, 184, 885,
    737, 622, 59, 830, 493, 264,
    23, 607, 903, 12, 43, 816,
    886, 933, 489, 619, 142, 691,
    // Bin capacity
    1989, 1555, 1609, 2008, 2023, 2027
  };
  const int i6_18_6_23_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    566, 579, 690, 867, 196, 726,
    283, 273, 223, 948, 589, 553,
    305, 379, 881, 823, 582, 430,
    800, 962, 344, 640, 546, 114,
    766, 973, 245, 184, 762, 464,
    878, 708, 849, 101, 428, 343,
    584, 45, 905, 137, 700, 260,
    425, 447, 369, 322, 775, 52,
    481, 699, 54, 658, 33, 929,
    474, 491, 790, 286, 295, 195,
    803, 530, 207, 22, 429, 964,
    838, 332, 371, 795, 902, 946,
    334, 608, 928, 696, 124, 184,
    551, 768, 314, 671, 238, 1,
    105, 677, 755, 810, 724, 91,
    662, 452, 200, 757, 77, 969,
    544, 913, 597, 715, 934, 189,
    341, 163, 302, 143, 709, 834,
    // Bin capacity
    1997, 2050, 1850, 1963, 1854, 1691
  };
  const int i6_18_6_24_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    371, 746, 470, 478, 918, 126,
    926, 2, 990, 436, 599, 203,
    692, 460, 602, 238, 599, 401,
    331, 938, 689, 125, 906, 480,
    144, 37, 169, 42, 473, 603,
    959, 397, 10, 220, 162, 67,
    223, 645, 380, 665, 377, 670,
    293, 697, 949, 499, 522, 376,
    102, 118, 942, 239, 839, 128,
    448, 73, 409, 296, 267, 166,
    36, 127, 545, 183, 155, 795,
    841, 435, 164, 845, 916, 31,
    551, 783, 743, 81, 9, 949,
    154, 985, 115, 410, 846, 874,
    150, 209, 656, 875, 574, 568,
    778, 71, 69, 530, 191, 730,
    554, 60, 524, 158, 880, 263,
    742, 775, 181, 125, 26, 570,
    // Bin capacity
    1715, 1562, 1779, 1332, 1914, 1654
  };
  const int i6_18_6_25_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    3, 819, 461, 756, 569, 172,
    382, 264, 606, 12, 71, 286,
    940, 368, 951, 422, 77, 965,
    1000, 399, 347, 708, 805, 518,
    922, 990, 268, 120, 882, 816,
    500, 800, 613, 289, 518, 407,
    337, 165, 560, 533, 672, 894,
    111, 714, 970, 432, 434, 245,
    946, 83, 457, 453, 267, 508,
    914, 93, 562, 555, 897, 383,
    860, 86, 284, 148, 84, 697,
    329, 412, 755, 97, 442, 800,
    21, 219, 212, 878, 909, 35,
    563, 130, 907, 208, 671, 523,
    991, 893, 583, 39, 434, 403,
    366, 875, 989, 19, 288, 370,
    56, 655, 738, 368, 500, 145,
    1000, 227, 823, 658, 402, 817,
    // Bin capacity
    2134, 1707, 2310, 1395, 1859, 1872
  };
  const int i6_18_6_26_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    981, 666, 624, 935, 173, 882,
    374, 956, 57, 11, 510, 450,
    799, 840, 693, 605, 38, 63,
    141, 165, 24, 590, 224, 130,
    131, 329, 807, 489, 685, 171,
    671, 154, 281, 675, 454, 340,
    590, 333, 108, 176, 766, 542,
    40, 787, 830, 791, 488, 208,
    254, 931, 709, 830, 193, 746,
    574, 720, 564, 978, 497, 884,
    156, 843, 710, 706, 131, 765,
    862, 531, 433, 829, 723, 511,
    988, 189, 308, 157, 783, 61,
    557, 911, 629, 285, 283, 823,
    770, 897, 750, 95, 569, 142,
    789, 218, 688, 731, 494, 928,
    947, 725, 12, 156, 459, 61,
    966, 378, 823, 416, 967, 489,
    // Bin capacity
    2224, 2221, 1901, 1986, 1772, 1722
  };
  const int i6_18_6_27_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    888, 648, 641, 248, 204, 217,
    202, 492, 904, 174, 713, 840,
    928, 849, 595, 279, 444, 284,
    187, 645, 67, 980, 482, 15,
    791, 219, 721, 361, 619, 537,
    175, 313, 571, 990, 374, 217,
    306, 927, 399, 505, 849, 120,
    654, 93, 418, 234, 194, 648,
    496, 86, 159, 889, 338, 383,
    788, 409, 371, 77, 258, 269,
    772, 287, 273, 855, 587, 178,
    735, 636, 628, 430, 259, 12,
    185, 192, 133, 750, 706, 979,
    90, 512, 951, 317, 75, 453,
    75, 400, 391, 616, 213, 420,
    633, 329, 404, 347, 669, 586,
    294, 37, 925, 449, 847, 862,
    318, 104, 500, 868, 241, 455,
    // Bin capacity
    1803, 1520, 1916, 1984, 1709, 1583
  };
  const int i6_18_6_28_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    133, 219, 279, 846, 542, 159,
    898, 942, 800, 817, 145, 512,
    924, 988, 355, 667, 250, 226,
    446, 837, 412, 275, 291, 657,
    593, 660, 763, 358, 308, 463,
    992, 5, 732, 265, 827, 111,
    615, 125, 581, 280, 351, 968,
    473, 885, 564, 315, 473, 870,
    544, 670, 245, 553, 578, 300,
    292, 529, 11, 327, 332, 179,
    376, 176, 451, 790, 850, 773,
    120, 336, 631, 204, 315, 141,
    798, 664, 802, 58, 816, 112,
    440, 384, 851, 502, 332, 219,
    322, 559, 982, 392, 210, 279,
    852, 586, 187, 53, 585, 269,
    220, 419, 137, 114, 757, 479,
    241, 470, 792, 33, 48, 395,
    // Bin capacity
    1980, 2017, 2043, 1462, 1709, 1518
  };
  const int i6_18_6_29_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    870, 596, 190, 330, 702, 33,
    498, 973, 852, 428, 372, 779,
    268, 465, 743, 731, 505, 372,
    708, 633, 924, 938, 476, 694,
    981, 902, 183, 901, 636, 800,
    134, 506, 564, 492, 602, 856,
    864, 349, 478, 957, 719, 416,
    363, 458, 692, 951, 450, 371,
    424, 767, 713, 658, 645, 941,
    518, 227, 662, 527, 763, 734,
    728, 269, 584, 349, 432, 735,
    507, 550, 174, 982, 179, 355,
    931, 352, 646, 126, 118, 690,
    745, 375, 198, 357, 227, 636,
    897, 114, 406, 444, 701, 330,
    83, 446, 195, 577, 772, 160,
    900, 708, 366, 927, 836, 97,
    543, 868, 397, 424, 891, 366,
    // Bin capacity
    2357, 2055, 1928, 2387, 2156, 2014
  };
  const int i6_18_6_30_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    928, 687, 626, 117, 188, 861,
    479, 464, 921, 995, 121, 196,
    466, 171, 1, 716, 322, 384,
    961, 106, 832, 877, 70, 86,
    441, 863, 376, 554, 956, 259,
    254, 127, 566, 870, 805, 991,
    274, 967, 926, 414, 898, 463,
    63, 75, 24, 258, 649, 46,
    442, 448, 5, 727, 862, 946,
    661, 273, 303, 412, 889, 849,
    424, 960, 363, 516, 79, 289,
    112, 707, 85, 624, 163, 798,
    907, 484, 762, 878, 198, 375,
    975, 921, 344, 826, 564, 591,
    340, 406, 699, 294, 400, 511,
    694, 480, 450, 794, 190, 535,
    909, 92, 837, 482, 17, 767,
    641, 799, 491, 1, 582, 252,
    // Bin capacity
    2161, 1957, 1866, 2244, 1724, 1994
  };
  const int i6_18_6_31_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    149, 193, 998, 77, 370, 846,
    777, 946, 832, 589, 499, 237,
    852, 116, 267, 902, 428, 391,
    166, 240, 352, 308, 122, 961,
    263, 424, 752, 968, 0, 497,
    504, 64, 584, 593, 906, 132,
    397, 981, 154, 452, 903, 554,
    486, 548, 324, 831, 452, 880,
    963, 920, 901, 469, 796, 801,
    443, 304, 383, 656, 350, 386,
    599, 778, 474, 818, 429, 645,
    855, 702, 378, 441, 820, 160,
    779, 598, 625, 85, 221, 187,
    418, 372, 261, 30, 284, 313,
    654, 189, 863, 346, 702, 400,
    964, 83, 898, 817, 296, 797,
    274, 215, 574, 52, 214, 695,
    430, 302, 383, 429, 877, 776,
    // Bin capacity
    2178, 1742, 2184, 1936, 1893, 2109
  };
  const int i6_18_6_32_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    694, 192, 981, 923, 953, 973,
    3, 521, 520, 83, 668, 548,
    499, 255, 499, 67, 137, 518,
    933, 192, 230, 133, 735, 606,
    725, 171, 272, 466, 802, 933,
    916, 767, 259, 678, 939, 65,
    746, 563, 792, 607, 674, 522,
    913, 662, 476, 193, 535, 912,
    52, 335, 821, 54, 503, 394,
    909, 857, 772, 429, 813, 913,
    174, 547, 933, 36, 605, 986,
    844, 146, 850, 538, 66, 666,
    265, 516, 33, 94, 980, 693,
    168, 603, 125, 630, 411, 85,
    767, 783, 281, 159, 813, 543,
    794, 349, 782, 695, 891, 159,
    748, 607, 214, 603, 967, 490,
    579, 230, 498, 954, 617, 172,
    // Bin capacity
    2361, 1826, 2055, 1616, 2664, 2240
  };
  const int i6_18_6_33_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    281, 403, 625, 719, 494, 978,
    70, 530, 427, 514, 142, 363,
    166, 736, 603, 539, 233, 799,
    559, 22, 887, 592, 890, 458,
    753, 218, 80, 33, 990, 266,
    109, 186, 482, 899, 995, 980,
    597, 560, 62, 913, 777, 115,
    939, 857, 697, 160, 107, 980,
    74, 272, 451, 962, 638, 227,
    485, 783, 567, 225, 69, 496,
    633, 965, 839, 162, 308, 181,
    807, 693, 102, 909, 134, 659,
    259, 708, 830, 982, 612, 813,
    979, 401, 180, 999, 558, 263,
    264, 295, 178, 790, 855, 182,
    359, 98, 33, 509, 550, 69,
    469, 572, 155, 235, 56, 2,
    601, 711, 607, 49, 973, 576,
    // Bin capacity
    1863, 1998, 1731, 2260, 2080, 1864
  };
  const int i6_18_6_34_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    930, 65, 98, 782, 998, 975,
    538, 333, 636, 85, 453, 594,
    767, 898, 920, 259, 222, 981,
    679, 444, 307, 549, 253, 724,
    252, 446, 686, 780, 365, 900,
    303, 499, 426, 32, 934, 829,
    775, 432, 458, 89, 447, 857,
    305, 874, 88, 394, 509, 85,
    339, 295, 369, 912, 100, 295,
    284, 458, 473, 669, 670, 249,
    734, 74, 888, 382, 579, 637,
    911, 277, 134, 989, 939, 770,
    156, 423, 591, 247, 156, 855,
    736, 206, 791, 188, 637, 699,
    718, 632, 913, 744, 715, 986,
    436, 449, 354, 653, 859, 244,
    61, 142, 953, 346, 682, 36,
    650, 50, 61, 283, 195, 789,
    // Bin capacity
    2139, 1563, 2043, 1873, 2170, 2570
  };
  const int i6_18_6_35_21[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    943, 887, 324, 673, 59, 309,
    39, 7, 763, 461, 552, 703,
    630, 153, 332, 975, 341, 248,
    770, 276, 809, 393, 831, 890,
    172, 499, 351, 405, 326, 883,
    576, 161, 245, 945, 920, 686,
    962, 395, 462, 667, 278, 177,
    489, 933, 87, 395, 169, 957,
    160, 708, 245, 644, 465, 388,
    897, 52, 220, 328, 981, 362,
    464, 530, 436, 883, 539, 748,
    747, 463, 739, 498, 51, 36,
    375, 149, 678, 60, 845, 327,
    921, 484, 177, 712, 765, 374,
    924, 365, 194, 869, 146, 951,
    60, 655, 596, 175, 53, 118,
    991, 995, 199, 355, 839, 898,
    704, 330, 274, 279, 771, 205,
    // Bin capacity
    2436, 1810, 1605, 2187, 2010, 2084
  };
  const int i6_18_6_0_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    341, 405, 248, 280, 272, 178,
    571, 177, 175, 965, 81, 471,
    667, 321, 680, 765, 288, 453,
    533, 497, 817, 475, 529, 379,
    814, 322, 639, 320, 226, 167,
    288, 908, 83, 312, 292, 523,
    351, 291, 661, 854, 657, 635,
    209, 385, 464, 56, 92, 112,
    629, 253, 986, 157, 487, 315,
    289, 842, 659, 901, 59, 695,
    245, 326, 339, 106, 293, 287,
    583, 692, 448, 157, 116, 935,
    809, 236, 56, 511, 171, 41,
    563, 508, 95, 819, 94, 226,
    514, 470, 938, 917, 235, 584,
    367, 683, 638, 913, 584, 807,
    276, 82, 149, 157, 676, 309,
    25, 986, 654, 364, 151, 623,
    // Bin capacity
    1346, 1398, 1455, 1505, 884, 1290
  };
  const int i6_18_6_1_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    178, 354, 347, 78, 31, 551,
    896, 91, 832, 448, 472, 456,
    960, 833, 153, 207, 923, 473,
    352, 569, 803, 557, 661, 414,
    136, 242, 341, 335, 863, 403,
    160, 795, 831, 2, 217, 269,
    315, 607, 708, 652, 269, 356,
    497, 474, 223, 203, 832, 489,
    874, 200, 519, 398, 915, 131,
    302, 815, 891, 605, 994, 256,
    402, 772, 297, 685, 512, 424,
    854, 687, 690, 622, 389, 70,
    178, 581, 256, 108, 140, 857,
    251, 910, 858, 248, 492, 821,
    700, 254, 436, 357, 330, 361,
    800, 340, 973, 186, 962, 930,
    584, 34, 168, 808, 406, 679,
    734, 271, 127, 761, 499, 677,
    // Bin capacity
    1545, 1487, 1592, 1223, 1668, 1451
  };
  const int i6_18_6_2_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    7, 591, 837, 500, 320, 838,
    113, 633, 164, 370, 518, 238,
    653, 492, 87, 72, 41, 802,
    341, 403, 232, 618, 913, 538,
    184, 615, 427, 732, 475, 718,
    213, 834, 535, 650, 50, 381,
    983, 605, 421, 513, 442, 563,
    501, 844, 638, 257, 846, 150,
    442, 598, 750, 940, 780, 265,
    318, 792, 547, 52, 660, 62,
    357, 156, 537, 679, 943, 733,
    31, 573, 441, 883, 924, 561,
    761, 633, 791, 363, 548, 775,
    105, 652, 925, 151, 44, 952,
    252, 659, 152, 177, 955, 796,
    703, 476, 333, 450, 705, 589,
    710, 363, 471, 998, 388, 852,
    508, 45, 455, 376, 8, 555,
    // Bin capacity
    1221, 1694, 1487, 1493, 1626, 1763
  };
  const int i6_18_6_3_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    867, 496, 289, 185, 488, 860,
    180, 990, 252, 44, 304, 91,
    765, 935, 625, 255, 676, 975,
    314, 822, 993, 75, 839, 189,
    525, 119, 542, 592, 332, 945,
    467, 248, 477, 52, 644, 247,
    711, 804, 21, 257, 460, 134,
    881, 740, 98, 131, 598, 718,
    719, 193, 383, 757, 137, 228,
    676, 464, 524, 819, 391, 939,
    333, 927, 553, 313, 471, 131,
    469, 384, 503, 510, 159, 147,
    275, 736, 963, 758, 300, 217,
    776, 175, 24, 656, 21, 513,
    214, 501, 531, 161, 305, 635,
    618, 817, 130, 347, 239, 817,
    901, 528, 75, 41, 308, 509,
    345, 38, 20, 563, 666, 640,
    // Bin capacity
    1723, 1703, 1203, 1119, 1260, 1534
  };
  const int i6_18_6_4_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    304, 59, 8, 567, 858, 660,
    867, 179, 22, 301, 954, 334,
    658, 458, 285, 230, 912, 888,
    110, 669, 346, 329, 132, 288,
    427, 243, 446, 932, 778, 487,
    900, 670, 787, 923, 245, 999,
    440, 599, 966, 960, 964, 408,
    25, 402, 211, 866, 198, 122,
    876, 28, 940, 63, 301, 100,
    810, 932, 137, 69, 697, 72,
    883, 869, 132, 568, 585, 450,
    527, 760, 419, 44, 547, 803,
    835, 330, 976, 926, 912, 227,
    276, 751, 28, 646, 634, 819,
    364, 132, 350, 607, 422, 0,
    255, 791, 663, 357, 33, 443,
    873, 288, 18, 2, 771, 725,
    174, 966, 427, 798, 455, 82,
    // Bin capacity
    1665, 1582, 1242, 1593, 1803, 1371
  };
  const int i6_18_6_5_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    567, 426, 641, 725, 302, 904,
    462, 945, 257, 99, 459, 309,
    30, 70, 538, 838, 735, 955,
    917, 387, 697, 898, 502, 998,
    747, 869, 865, 618, 216, 251,
    389, 499, 245, 519, 975, 692,
    355, 600, 890, 393, 523, 628,
    728, 356, 963, 389, 111, 586,
    92, 535, 366, 474, 474, 863,
    654, 21, 459, 436, 803, 421,
    86, 145, 854, 471, 86, 321,
    627, 40, 193, 175, 401, 636,
    800, 313, 810, 18, 990, 412,
    704, 198, 663, 466, 262, 291,
    621, 597, 722, 91, 994, 540,
    645, 691, 299, 852, 844, 148,
    890, 757, 266, 879, 216, 129,
    606, 708, 593, 36, 729, 129,
    // Bin capacity
    1736, 1428, 1807, 1466, 1684, 1613
  };
  const int i6_18_6_6_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    513, 117, 957, 516, 731, 766,
    941, 375, 826, 64, 102, 873,
    483, 67, 789, 34, 120, 371,
    520, 664, 204, 198, 63, 883,
    602, 374, 590, 802, 340, 309,
    757, 273, 606, 269, 75, 928,
    899, 482, 388, 827, 383, 36,
    318, 211, 338, 448, 966, 851,
    506, 698, 525, 648, 263, 239,
    600, 387, 812, 616, 801, 141,
    988, 30, 303, 334, 299, 299,
    887, 833, 884, 817, 716, 819,
    56, 237, 414, 223, 255, 863,
    399, 129, 529, 732, 887, 350,
    337, 372, 224, 839, 361, 339,
    826, 691, 476, 855, 289, 416,
    788, 77, 148, 544, 211, 989,
    955, 798, 149, 252, 505, 967,
    // Bin capacity
    2010, 1204, 1619, 1594, 1302, 1845
  };
  const int i6_18_6_7_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    756, 341, 522, 802, 21, 166,
    56, 199, 995, 847, 70, 501,
    911, 68, 445, 71, 635, 715,
    537, 624, 838, 906, 457, 22,
    430, 980, 396, 236, 125, 136,
    831, 331, 908, 946, 825, 798,
    565, 775, 366, 659, 805, 947,
    392, 365, 623, 227, 222, 912,
    827, 9, 883, 711, 631, 456,
    384, 681, 298, 278, 790, 506,
    191, 939, 936, 862, 196, 322,
    997, 319, 586, 754, 721, 833,
    909, 862, 278, 575, 639, 317,
    568, 105, 230, 976, 32, 642,
    695, 627, 83, 979, 988, 546,
    431, 80, 193, 608, 493, 853,
    66, 594, 684, 163, 893, 600,
    156, 525, 267, 744, 415, 366,
    // Bin capacity
    1731, 1503, 1700, 2024, 1598, 1719
  };
  const int i6_18_6_8_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    265, 167, 399, 141, 317, 315,
    112, 781, 714, 116, 811, 968,
    956, 112, 214, 163, 646, 906,
    174, 139, 932, 919, 902, 578,
    778, 155, 258, 812, 154, 157,
    261, 999, 254, 240, 554, 879,
    869, 886, 309, 635, 717, 963,
    842, 752, 659, 752, 756, 165,
    27, 906, 702, 554, 841, 321,
    607, 473, 546, 408, 348, 440,
    193, 335, 612, 28, 993, 144,
    376, 179, 1000, 995, 76, 848,
    161, 738, 885, 444, 76, 6,
    246, 460, 679, 941, 628, 182,
    940, 287, 974, 488, 635, 373,
    818, 979, 816, 379, 429, 192,
    606, 276, 706, 923, 872, 147,
    269, 441, 1000, 620, 511, 141,
    // Bin capacity
    1530, 1632, 2099, 1721, 1848, 1391
  };
  const int i6_18_6_9_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    462, 654, 138, 732, 876, 920,
    861, 122, 533, 485, 845, 54,
    266, 578, 332, 986, 413, 187,
    929, 246, 383, 607, 653, 586,
    938, 428, 996, 879, 931, 538,
    161, 518, 176, 375, 529, 244,
    780, 985, 207, 866, 670, 289,
    970, 439, 100, 261, 937, 870,
    14, 820, 972, 182, 892, 734,
    442, 400, 380, 251, 140, 143,
    169, 7, 165, 881, 266, 343,
    970, 841, 398, 962, 183, 805,
    924, 133, 975, 147, 853, 691,
    251, 847, 128, 933, 42, 199,
    949, 961, 783, 858, 336, 523,
    835, 721, 411, 814, 283, 888,
    248, 973, 515, 638, 570, 585,
    663, 382, 441, 231, 128, 758,
    // Bin capacity
    1968, 1827, 1460, 2015, 1735, 1700
  };
  const int i6_18_6_10_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    813, 354, 401, 640, 846, 749,
    990, 704, 524, 555, 405, 221,
    926, 430, 113, 959, 62, 879,
    680, 644, 693, 525, 214, 373,
    187, 984, 419, 886, 726, 132,
    99, 656, 361, 838, 949, 541,
    350, 589, 339, 744, 90, 555,
    177, 685, 417, 734, 820, 445,
    624, 997, 728, 487, 72, 880,
    425, 829, 873, 871, 928, 645,
    957, 848, 121, 996, 337, 47,
    172, 917, 601, 946, 941, 353,
    198, 99, 354, 413, 13, 951,
    449, 79, 561, 529, 141, 548,
    457, 63, 22, 919, 397, 6,
    869, 586, 998, 103, 504, 764,
    190, 814, 170, 657, 489, 814,
    197, 511, 363, 562, 716, 541,
    // Bin capacity
    1606, 1978, 1478, 2267, 1586, 1732
  };
  const int i6_18_6_11_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    188, 121, 721, 391, 867, 84,
    938, 470, 869, 506, 632, 132,
    313, 650, 91, 356, 142, 900,
    841, 357, 416, 436, 977, 24,
    673, 839, 308, 226, 591, 665,
    577, 317, 963, 707, 901, 29,
    881, 893, 528, 983, 737, 546,
    251, 589, 2, 500, 572, 324,
    340, 931, 199, 848, 227, 710,
    379, 837, 102, 128, 628, 789,
    955, 819, 458, 670, 341, 294,
    928, 41, 594, 185, 616, 419,
    466, 645, 665, 528, 514, 614,
    437, 471, 452, 863, 54, 6,
    265, 157, 436, 366, 697, 339,
    788, 173, 920, 764, 214, 996,
    450, 931, 639, 119, 919, 7,
    184, 540, 0, 958, 313, 210,
    // Bin capacity
    1823, 1810, 1548, 1764, 1840, 1312
  };
  const int i6_18_6_12_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    895, 564, 576, 586, 472, 428,
    10, 278, 116, 521, 210, 114,
    563, 319, 528, 321, 23, 312,
    379, 425, 539, 247, 370, 651,
    579, 54, 407, 8, 258, 384,
    516, 421, 730, 156, 990, 392,
    644, 997, 639, 480, 786, 960,
    806, 934, 479, 98, 201, 687,
    65, 154, 98, 888, 887, 446,
    511, 771, 319, 276, 13, 570,
    909, 921, 275, 74, 698, 71,
    408, 123, 367, 60, 443, 584,
    141, 623, 875, 85, 918, 875,
    716, 973, 577, 840, 509, 860,
    235, 150, 8, 328, 628, 449,
    681, 572, 901, 190, 835, 498,
    412, 226, 613, 943, 65, 331,
    59, 558, 188, 636, 949, 920,
    // Bin capacity
    1593, 1692, 1538, 1258, 1728, 1780
  };
  const int i6_18_6_13_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    625, 2, 411, 472, 61, 849,
    604, 614, 365, 298, 660, 347,
    746, 427, 345, 622, 570, 164,
    430, 278, 417, 640, 367, 965,
    806, 891, 913, 360, 493, 303,
    507, 296, 873, 32, 67, 765,
    237, 588, 695, 163, 181, 940,
    841, 198, 500, 662, 631, 516,
    152, 604, 197, 910, 100, 464,
    970, 66, 48, 866, 772, 792,
    130, 345, 706, 322, 525, 904,
    350, 299, 246, 404, 785, 215,
    573, 255, 313, 137, 34, 889,
    683, 406, 64, 95, 219, 975,
    768, 239, 187, 732, 287, 201,
    278, 57, 306, 182, 131, 596,
    485, 802, 246, 887, 830, 295,
    857, 345, 64, 576, 152, 413,
    // Bin capacity
    1892, 1265, 1299, 1575, 1293, 1996
  };
  const int i6_18_6_14_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    442, 663, 112, 442, 419, 808,
    864, 692, 550, 990, 956, 395,
    749, 565, 737, 629, 544, 642,
    567, 922, 941, 478, 366, 147,
    944, 932, 325, 801, 980, 149,
    545, 198, 822, 429, 189, 123,
    604, 737, 921, 444, 47, 940,
    515, 30, 849, 387, 157, 582,
    340, 243, 854, 375, 902, 353,
    634, 709, 720, 483, 218, 370,
    561, 370, 709, 879, 426, 380,
    680, 224, 116, 976, 436, 992,
    813, 121, 606, 828, 143, 521,
    20, 989, 225, 864, 484, 246,
    156, 25, 584, 787, 904, 786,
    724, 608, 977, 176, 473, 472,
    206, 444, 561, 567, 385, 237,
    774, 10, 419, 651, 512, 79,
    // Bin capacity
    1927, 1612, 2096, 2126, 1623, 1563
  };
  const int i6_18_6_15_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    72, 859, 952, 765, 764, 675,
    873, 213, 941, 917, 519, 267,
    468, 901, 14, 433, 711, 824,
    852, 671, 824, 951, 437, 160,
    124, 234, 232, 751, 729, 880,
    772, 365, 501, 165, 341, 805,
    566, 507, 58, 375, 267, 861,
    62, 978, 556, 455, 277, 285,
    823, 475, 914, 851, 334, 177,
    48, 239, 878, 229, 183, 829,
    638, 173, 8, 629, 257, 33,
    365, 825, 394, 278, 296, 459,
    211, 285, 713, 480, 412, 148,
    945, 778, 52, 369, 784, 951,
    189, 327, 321, 756, 474, 379,
    746, 242, 882, 86, 42, 281,
    956, 6, 115, 67, 349, 997,
    669, 459, 653, 760, 762, 593,
    // Bin capacity
    1798, 1637, 1727, 1786, 1522, 1841
  };
  const int i6_18_6_16_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    585, 578, 940, 818, 976, 824,
    941, 388, 40, 346, 428, 899,
    42, 850, 22, 824, 365, 940,
    629, 988, 635, 572, 255, 610,
    837, 15, 821, 946, 953, 476,
    579, 465, 57, 599, 425, 800,
    441, 463, 423, 841, 675, 402,
    539, 512, 488, 672, 902, 751,
    199, 205, 576, 434, 581, 250,
    96, 116, 310, 480, 615, 785,
    66, 305, 963, 885, 972, 773,
    535, 324, 234, 782, 170, 255,
    393, 540, 849, 229, 538, 378,
    302, 720, 320, 130, 582, 160,
    485, 304, 973, 141, 674, 498,
    179, 346, 285, 828, 987, 760,
    950, 233, 681, 669, 815, 152,
    124, 504, 965, 975, 674, 731,
    // Bin capacity
    1532, 1519, 1853, 2160, 2241, 2020
  };
  const int i6_18_6_17_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    683, 402, 735, 774, 191, 5,
    701, 83, 979, 166, 211, 461,
    390, 635, 786, 179, 309, 719,
    716, 252, 790, 965, 509, 37,
    915, 675, 889, 845, 75, 110,
    271, 272, 848, 423, 865, 949,
    599, 661, 843, 455, 355, 252,
    720, 37, 311, 963, 880, 817,
    17, 947, 960, 781, 292, 912,
    988, 648, 61, 512, 94, 299,
    23, 797, 923, 565, 562, 284,
    269, 279, 99, 366, 525, 563,
    92, 595, 445, 673, 68, 567,
    939, 45, 163, 142, 273, 819,
    691, 826, 35, 83, 578, 425,
    690, 308, 213, 650, 465, 478,
    674, 917, 684, 425, 291, 210,
    673, 52, 207, 626, 259, 816,
    // Bin capacity
    1960, 1645, 1945, 1871, 1327, 1701
  };
  const int i6_18_6_18_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    61, 157, 919, 945, 88, 454,
    863, 558, 237, 635, 836, 303,
    895, 567, 308, 429, 247, 854,
    491, 658, 105, 527, 827, 54,
    209, 740, 134, 40, 711, 317,
    49, 156, 267, 644, 355, 937,
    946, 650, 180, 24, 15, 679,
    152, 337, 297, 811, 135, 193,
    950, 432, 796, 852, 723, 273,
    13, 59, 286, 782, 305, 50,
    663, 751, 937, 761, 834, 146,
    538, 160, 401, 333, 541, 263,
    382, 895, 963, 860, 30, 420,
    606, 999, 66, 975, 409, 415,
    395, 163, 633, 630, 47, 202,
    137, 246, 461, 726, 933, 938,
    658, 211, 848, 548, 423, 594,
    8, 468, 637, 729, 7, 198,
    // Bin capacity
    1577, 1615, 1667, 2213, 1469, 1434
  };
  const int i6_18_6_19_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    154, 445, 705, 767, 317, 887,
    702, 350, 855, 793, 513, 725,
    355, 531, 403, 671, 845, 960,
    800, 394, 891, 348, 804, 445,
    917, 463, 795, 171, 376, 145,
    329, 1, 576, 182, 598, 66,
    674, 113, 394, 711, 761, 172,
    141, 470, 366, 845, 861, 317,
    689, 218, 616, 638, 93, 194,
    550, 423, 466, 845, 275, 707,
    854, 751, 524, 655, 648, 409,
    357, 251, 94, 60, 473, 587,
    176, 601, 884, 555, 989, 287,
    226, 834, 504, 257, 365, 290,
    880, 182, 62, 940, 381, 55,
    949, 159, 199, 7, 667, 510,
    589, 601, 369, 446, 880, 597,
    974, 495, 289, 363, 931, 41,
    // Bin capacity
    2047, 1445, 1784, 1836, 2138, 1467
  };
  const int i6_18_6_20_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    574, 800, 506, 945, 631, 779,
    759, 974, 208, 261, 902, 843,
    379, 968, 99, 645, 756, 852,
    992, 54, 289, 422, 613, 481,
    190, 370, 766, 475, 611, 381,
    633, 620, 514, 960, 320, 89,
    40, 401, 21, 800, 579, 1,
    656, 625, 569, 410, 635, 605,
    741, 821, 756, 124, 747, 178,
    985, 972, 227, 15, 697, 131,
    26, 244, 359, 821, 774, 628,
    387, 713, 309, 298, 277, 217,
    31, 435, 867, 131, 45, 427,
    392, 440, 927, 540, 671, 889,
    519, 808, 677, 22, 546, 312,
    350, 893, 739, 23, 122, 424,
    980, 954, 376, 24, 845, 738,
    210, 775, 712, 297, 278, 862,
    // Bin capacity
    1769, 2374, 1785, 1443, 2010, 1768
  };
  const int i6_18_6_21_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    327, 45, 910, 145, 446, 892,
    858, 660, 660, 970, 934, 170,
    347, 276, 444, 648, 660, 179,
    789, 230, 716, 265, 840, 692,
    714, 571, 491, 243, 468, 370,
    324, 657, 893, 937, 728, 5,
    336, 721, 567, 272, 410, 824,
    757, 68, 821, 176, 4, 240,
    100, 421, 728, 934, 745, 150,
    354, 570, 517, 788, 452, 824,
    139, 853, 555, 186, 670, 98,
    776, 452, 469, 156, 626, 161,
    39, 898, 79, 83, 943, 910,
    86, 891, 812, 123, 164, 642,
    336, 889, 918, 13, 189, 961,
    746, 455, 822, 407, 297, 126,
    716, 47, 819, 89, 383, 164,
    806, 178, 789, 813, 25, 396,
    // Bin capacity
    1725, 1792, 2423, 1462, 1812, 1574
  };
  const int i6_18_6_22_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    955, 472, 80, 693, 351, 814,
    383, 595, 330, 33, 780, 458,
    814, 377, 323, 584, 867, 2,
    506, 238, 994, 955, 395, 559,
    259, 552, 472, 100, 531, 873,
    122, 654, 966, 754, 914, 387,
    732, 831, 564, 919, 720, 203,
    606, 742, 124, 137, 128, 652,
    579, 963, 967, 675, 772, 103,
    79, 872, 316, 198, 704, 493,
    537, 616, 590, 960, 63, 642,
    182, 323, 642, 31, 909, 516,
    113, 1000, 352, 648, 183, 585,
    705, 595, 592, 891, 783, 322,
    397, 191, 108, 896, 525, 861,
    165, 747, 761, 977, 753, 654,
    182, 184, 346, 382, 206, 606,
    548, 158, 385, 947, 178, 159,
    // Bin capacity
    1600, 2056, 1813, 2192, 1985, 1808
  };
  const int i6_18_6_23_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    496, 177, 147, 283, 413, 946,
    771, 559, 990, 419, 154, 231,
    149, 613, 621, 917, 677, 877,
    541, 910, 293, 840, 477, 737,
    334, 733, 984, 195, 985, 763,
    63, 932, 608, 921, 748, 451,
    108, 102, 688, 652, 141, 392,
    141, 781, 967, 986, 652, 889,
    877, 335, 36, 719, 136, 711,
    973, 868, 488, 241, 164, 650,
    141, 999, 581, 895, 873, 683,
    827, 743, 678, 44, 814, 100,
    532, 654, 129, 809, 880, 360,
    63, 330, 995, 946, 596, 189,
    192, 220, 368, 613, 119, 531,
    558, 710, 824, 612, 567, 724,
    117, 114, 513, 227, 425, 838,
    644, 860, 125, 447, 955, 256,
    // Bin capacity
    1544, 2182, 2058, 2208, 2005, 2118
  };
  const int i6_18_6_24_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    78, 729, 376, 147, 648, 397,
    402, 155, 0, 687, 723, 835,
    732, 512, 827, 688, 127, 231,
    303, 131, 576, 893, 933, 762,
    516, 145, 857, 617, 275, 96,
    864, 258, 877, 336, 200, 716,
    596, 719, 95, 699, 109, 970,
    611, 659, 749, 843, 578, 492,
    629, 774, 407, 734, 728, 454,
    121, 525, 719, 924, 661, 646,
    501, 350, 777, 37, 919, 362,
    248, 4, 733, 626, 779, 649,
    975, 525, 396, 359, 569, 368,
    328, 455, 615, 60, 701, 306,
    827, 509, 847, 716, 375, 484,
    561, 828, 782, 737, 603, 952,
    780, 334, 600, 364, 687, 231,
    490, 94, 508, 941, 891, 963,
    // Bin capacity
    1977, 1593, 2220, 2151, 2172, 2049
  };
  const int i6_18_6_25_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    25, 321, 630, 13, 101, 119,
    875, 895, 366, 923, 296, 697,
    133, 181, 960, 268, 970, 29,
    497, 884, 797, 864, 429, 95,
    708, 973, 276, 151, 379, 48,
    43, 817, 354, 984, 685, 716,
    929, 320, 918, 940, 900, 883,
    369, 319, 249, 856, 647, 137,
    429, 536, 103, 102, 445, 960,
    369, 558, 566, 160, 720, 461,
    921, 95, 471, 354, 447, 616,
    10, 701, 216, 601, 799, 153,
    110, 288, 828, 476, 201, 400,
    582, 223, 399, 672, 504, 743,
    837, 446, 858, 743, 197, 88,
    90, 258, 847, 680, 474, 261,
    893, 282, 347, 816, 569, 548,
    599, 210, 249, 782, 136, 770,
    // Bin capacity
    1754, 1731, 1966, 2164, 1854, 1610
  };
  const int i6_18_6_26_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    319, 674, 577, 823, 556, 616,
    977, 264, 889, 675, 830, 768,
    363, 253, 32, 425, 660, 377,
    960, 389, 491, 954, 172, 900,
    618, 602, 654, 766, 819, 847,
    62, 416, 924, 470, 575, 601,
    996, 504, 609, 244, 402, 484,
    275, 619, 701, 506, 632, 290,
    261, 781, 897, 812, 221, 297,
    832, 370, 778, 814, 91, 470,
    211, 947, 232, 221, 456, 829,
    122, 692, 421, 606, 871, 162,
    740, 80, 346, 335, 411, 154,
    374, 74, 608, 279, 349, 73,
    195, 992, 895, 883, 951, 738,
    34, 9, 428, 718, 160, 141,
    576, 957, 57, 233, 567, 781,
    324, 531, 933, 875, 264, 90,
    // Bin capacity
    1731, 1923, 2200, 2235, 1888, 1810
  };
  const int i6_18_6_27_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    847, 820, 880, 985, 615, 980,
    306, 745, 278, 201, 117, 422,
    919, 575, 301, 291, 248, 996,
    513, 345, 170, 915, 524, 978,
    22, 507, 146, 895, 734, 748,
    612, 403, 592, 547, 217, 249,
    434, 291, 691, 998, 235, 146,
    991, 222, 420, 493, 640, 174,
    287, 310, 926, 713, 982, 150,
    75, 928, 602, 381, 79, 965,
    529, 72, 752, 25, 753, 378,
    244, 325, 995, 640, 442, 711,
    381, 765, 846, 415, 355, 742,
    464, 647, 306, 889, 381, 313,
    181, 512, 537, 905, 885, 921,
    121, 518, 791, 614, 210, 137,
    302, 819, 459, 923, 931, 439,
    843, 342, 240, 61, 284, 609,
    // Bin capacity
    1709, 1936, 2103, 2306, 1828, 2129
  };
  const int i6_18_6_28_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    357, 223, 899, 272, 411, 866,
    273, 335, 674, 316, 453, 789,
    731, 373, 586, 739, 368, 964,
    816, 353, 321, 424, 166, 427,
    128, 855, 61, 80, 102, 565,
    844, 175, 957, 540, 772, 307,
    300, 67, 947, 576, 993, 663,
    158, 32, 125, 706, 874, 246,
    113, 292, 504, 966, 263, 534,
    406, 787, 711, 337, 470, 184,
    850, 842, 550, 813, 408, 943,
    541, 522, 465, 483, 739, 183,
    276, 802, 650, 558, 363, 306,
    592, 221, 73, 738, 528, 512,
    564, 263, 489, 478, 610, 458,
    788, 409, 50, 404, 916, 991,
    778, 398, 985, 95, 460, 411,
    270, 700, 614, 533, 904, 264,
    // Bin capacity
    1875, 1632, 2062, 1933, 2091, 2051
  };
  const int i6_18_6_29_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    631, 281, 978, 489, 275, 956,
    423, 443, 828, 431, 488, 129,
    374, 146, 220, 163, 682, 250,
    540, 907, 670, 130, 116, 285,
    960, 270, 47, 420, 568, 665,
    72, 999, 464, 112, 871, 932,
    179, 170, 357, 765, 295, 923,
    503, 145, 255, 542, 786, 756,
    929, 487, 491, 574, 426, 47,
    106, 431, 868, 141, 55, 226,
    237, 543, 651, 277, 413, 871,
    51, 55, 147, 954, 368, 656,
    555, 702, 596, 718, 525, 381,
    460, 841, 575, 531, 559, 912,
    28, 199, 802, 124, 711, 427,
    92, 919, 507, 694, 719, 344,
    687, 811, 461, 233, 809, 821,
    738, 816, 616, 643, 338, 305,
    // Bin capacity
    1627, 1971, 2050, 1708, 1936, 2126
  };
  const int i6_18_6_30_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    998, 212, 921, 347, 176, 936,
    314, 944, 78, 739, 871, 316,
    292, 901, 957, 116, 863, 500,
    539, 496, 186, 326, 461, 222,
    603, 993, 340, 904, 478, 956,
    222, 10, 246, 716, 725, 201,
    11, 779, 772, 319, 500, 176,
    789, 976, 127, 857, 816, 719,
    564, 534, 338, 854, 812, 884,
    448, 173, 546, 946, 189, 984,
    864, 692, 577, 500, 195, 770,
    428, 695, 988, 131, 440, 500,
    123, 945, 610, 128, 807, 873,
    476, 864, 352, 866, 391, 219,
    385, 898, 815, 772, 42, 878,
    677, 976, 871, 674, 563, 446,
    988, 967, 278, 891, 898, 364,
    868, 678, 63, 492, 38, 690,
    // Bin capacity
    2078, 2759, 1965, 2292, 2008, 2305
  };
  const int i6_18_6_31_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    441, 646, 960, 813, 67, 921,
    450, 943, 868, 760, 727, 840,
    718, 97, 456, 254, 802, 660,
    259, 273, 36, 412, 361, 710,
    113, 831, 777, 769, 467, 516,
    87, 148, 995, 351, 604, 7,
    858, 235, 562, 771, 974, 710,
    161, 63, 931, 630, 655, 307,
    247, 940, 999, 142, 634, 188,
    819, 444, 863, 519, 1000, 565,
    549, 817, 417, 841, 45, 411,
    897, 83, 190, 224, 404, 228,
    670, 905, 493, 224, 602, 688,
    334, 760, 497, 840, 686, 939,
    944, 487, 898, 527, 731, 934,
    315, 496, 721, 886, 382, 401,
    906, 659, 47, 869, 213, 613,
    831, 119, 320, 390, 262, 762,
    // Bin capacity
    2096, 1954, 2409, 2232, 2100, 2271
  };
  const int i6_18_6_32_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    264, 760, 11, 312, 475, 447,
    994, 866, 763, 373, 601, 858,
    446, 93, 482, 109, 445, 104,
    115, 773, 881, 533, 776, 598,
    84, 323, 868, 80, 77, 9,
    479, 178, 947, 944, 314, 389,
    747, 827, 858, 301, 888, 21,
    666, 502, 915, 568, 745, 649,
    677, 151, 835, 394, 268, 637,
    565, 504, 519, 512, 107, 932,
    566, 843, 454, 58, 171, 650,
    741, 804, 927, 962, 764, 720,
    891, 891, 720, 499, 61, 629,
    957, 425, 704, 392, 34, 181,
    678, 804, 503, 925, 899, 576,
    507, 564, 213, 591, 846, 910,
    221, 526, 957, 515, 793, 976,
    179, 125, 467, 291, 854, 231,
    // Bin capacity
    2151, 2191, 2646, 1839, 2006, 2094
  };
  const int i6_18_6_33_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    162, 49, 650, 839, 929, 527,
    549, 104, 283, 728, 642, 332,
    254, 791, 116, 436, 241, 490,
    814, 483, 404, 557, 281, 714,
    514, 144, 982, 231, 326, 150,
    103, 929, 362, 638, 281, 743,
    722, 274, 278, 353, 335, 521,
    644, 699, 242, 465, 636, 551,
    175, 49, 590, 527, 961, 139,
    491, 4, 172, 172, 941, 615,
    494, 147, 541, 68, 751, 448,
    662, 981, 320, 759, 678, 177,
    927, 621, 104, 90, 755, 946,
    161, 226, 353, 284, 781, 373,
    787, 529, 510, 806, 359, 498,
    650, 177, 474, 675, 124, 58,
    60, 727, 293, 557, 31, 663,
    508, 642, 279, 368, 746, 513,
    // Bin capacity
    1924, 1680, 1542, 1896, 2172, 1875
  };
  const int i6_18_6_34_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    619, 142, 204, 833, 894, 747,
    932, 887, 732, 317, 633, 651,
    729, 343, 502, 250, 932, 280,
    445, 733, 53, 34, 786, 318,
    851, 967, 874, 638, 469, 151,
    595, 10, 838, 904, 412, 129,
    131, 632, 470, 759, 306, 56,
    853, 465, 806, 670, 134, 884,
    746, 903, 319, 293, 222, 505,
    503, 68, 103, 159, 591, 769,
    765, 306, 42, 155, 323, 556,
    951, 31, 419, 625, 609, 129,
    29, 742, 623, 747, 637, 471,
    654, 346, 117, 696, 997, 878,
    138, 820, 871, 344, 351, 85,
    681, 177, 430, 60, 253, 845,
    772, 80, 927, 866, 688, 62,
    58, 449, 126, 199, 707, 354,
    // Bin capacity
    2335, 1810, 1889, 1910, 2221, 1758
  };
  const int i6_18_6_35_22[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    747, 131, 910, 262, 327, 503,
    727, 314, 79, 595, 837, 874,
    70, 922, 853, 943, 288, 793,
    284, 95, 827, 990, 879, 136,
    108, 378, 168, 592, 496, 363,
    532, 395, 759, 129, 796, 223,
    132, 862, 820, 331, 862, 553,
    827, 673, 265, 95, 780, 134,
    36, 239, 106, 377, 376, 161,
    933, 243, 126, 225, 577, 331,
    749, 48, 188, 517, 927, 253,
    208, 135, 729, 276, 536, 178,
    962, 606, 78, 588, 819, 317,
    211, 246, 105, 453, 308, 838,
    751, 951, 479, 287, 404, 809,
    811, 620, 44, 342, 187, 828,
    937, 778, 380, 998, 506, 564,
    81, 714, 169, 964, 904, 118,
    // Bin capacity
    2049, 1879, 1595, 2017, 2433, 1795
  };
  const int i6_18_6_0_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    823, 409, 429, 830, 419, 731,
    153, 440, 86, 52, 890, 722,
    133, 523, 264, 883, 843, 220,
    161, 400, 956, 798, 521, 810,
    132, 237, 40, 226, 252, 293,
    750, 100, 223, 418, 275, 280,
    269, 811, 271, 259, 673, 114,
    270, 908, 550, 602, 711, 767,
    822, 87, 67, 88, 157, 882,
    224, 322, 372, 952, 786, 374,
    97, 925, 119, 763, 79, 1,
    615, 499, 555, 347, 330, 835,
    23, 371, 110, 801, 594, 949,
    800, 51, 833, 132, 34, 56,
    449, 873, 892, 423, 610, 947,
    73, 919, 248, 543, 117, 488,
    360, 823, 139, 131, 343, 883,
    120, 687, 878, 472, 803, 299,
    // Bin capacity
    1046, 1565, 1172, 1454, 1407, 1609
  };
  const int i6_18_6_1_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    496, 509, 796, 936, 252, 61,
    218, 31, 452, 643, 935, 152,
    898, 586, 639, 720, 463, 346,
    217, 5, 13, 656, 624, 734,
    350, 124, 84, 401, 894, 985,
    68, 820, 423, 719, 141, 62,
    222, 296, 428, 885, 595, 935,
    579, 811, 407, 251, 805, 874,
    839, 380, 943, 78, 993, 362,
    708, 126, 11, 304, 454, 238,
    110, 436, 460, 820, 995, 59,
    425, 567, 487, 240, 513, 512,
    724, 687, 336, 368, 797, 683,
    79, 460, 901, 853, 32, 852,
    909, 471, 430, 863, 86, 430,
    267, 822, 340, 963, 475, 106,
    165, 514, 212, 716, 217, 325,
    870, 940, 316, 21, 17, 526,
    // Bin capacity
    1371, 1446, 1293, 1757, 1564, 1388
  };
  const int i6_18_6_2_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    42, 729, 334, 364, 620, 135,
    318, 561, 236, 317, 370, 877,
    518, 832, 671, 298, 815, 877,
    753, 420, 668, 311, 489, 142,
    407, 464, 936, 578, 963, 906,
    482, 103, 610, 584, 65, 932,
    240, 929, 506, 188, 240, 533,
    619, 943, 63, 928, 421, 725,
    898, 65, 725, 121, 642, 190,
    234, 711, 964, 794, 307, 285,
    899, 258, 27, 499, 930, 522,
    388, 240, 637, 328, 103, 90,
    468, 830, 724, 756, 154, 842,
    616, 797, 637, 435, 89, 772,
    37, 764, 598, 183, 563, 283,
    319, 338, 966, 858, 792, 810,
    440, 795, 624, 938, 816, 177,
    519, 763, 313, 682, 620, 906,
    // Bin capacity
    1394, 1793, 1741, 1558, 1530, 1701
  };
  const int i6_18_6_3_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    398, 484, 330, 392, 502, 310,
    346, 244, 61, 811, 767, 945,
    997, 156, 656, 974, 492, 121,
    390, 400, 671, 391, 196, 903,
    248, 353, 775, 679, 727, 248,
    419, 888, 906, 569, 590, 390,
    383, 936, 883, 492, 748, 863,
    82, 367, 526, 239, 560, 589,
    221, 763, 614, 968, 615, 293,
    47, 185, 280, 981, 629, 817,
    898, 501, 392, 354, 927, 595,
    679, 321, 176, 449, 865, 290,
    790, 164, 236, 732, 134, 326,
    364, 65, 423, 84, 16, 699,
    36, 770, 199, 315, 522, 128,
    655, 318, 451, 631, 726, 777,
    723, 81, 662, 51, 406, 264,
    806, 259, 65, 310, 366, 133,
    // Bin capacity
    1457, 1246, 1426, 1618, 1681, 1492
  };
  const int i6_18_6_4_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    755, 218, 380, 578, 615, 921,
    459, 679, 132, 341, 727, 546,
    161, 165, 723, 497, 645, 964,
    887, 962, 80, 421, 477, 409,
    902, 309, 51, 508, 483, 412,
    224, 498, 214, 266, 445, 859,
    837, 778, 855, 623, 85, 508,
    407, 793, 689, 39, 972, 308,
    552, 994, 798, 839, 977, 798,
    526, 403, 972, 485, 236, 561,
    428, 292, 500, 554, 475, 978,
    67, 937, 612, 475, 175, 5,
    441, 955, 456, 100, 735, 986,
    672, 549, 728, 609, 986, 764,
    644, 822, 504, 455, 622, 345,
    672, 934, 461, 61, 189, 812,
    220, 919, 82, 460, 19, 6,
    385, 856, 746, 467, 214, 386,
    // Bin capacity
    1602, 2091, 1558, 1349, 1574, 1832
  };
  const int i6_18_6_5_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    346, 969, 107, 738, 126, 327,
    377, 279, 54, 99, 884, 943,
    69, 937, 133, 89, 333, 495,
    416, 865, 895, 693, 166, 592,
    231, 177, 918, 667, 199, 532,
    563, 419, 503, 376, 619, 502,
    661, 140, 466, 69, 942, 218,
    774, 51, 553, 878, 436, 799,
    211, 706, 951, 123, 76, 752,
    348, 428, 629, 251, 275, 244,
    339, 736, 185, 772, 785, 908,
    913, 27, 934, 355, 22, 396,
    780, 731, 26, 5, 378, 554,
    745, 659, 624, 159, 899, 237,
    739, 344, 19, 970, 425, 208,
    822, 993, 680, 963, 676, 529,
    381, 829, 566, 441, 349, 753,
    528, 749, 488, 495, 772, 783,
    // Bin capacity
    1618, 1757, 1528, 1426, 1464, 1711
  };
  const int i6_18_6_6_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    538, 277, 111, 51, 553, 517,
    808, 796, 127, 469, 73, 533,
    357, 116, 662, 382, 240, 908,
    685, 165, 858, 630, 62, 884,
    124, 170, 114, 79, 566, 341,
    651, 250, 909, 529, 398, 452,
    428, 860, 79, 544, 773, 497,
    826, 342, 110, 928, 172, 812,
    930, 126, 289, 243, 97, 84,
    667, 333, 8, 232, 541, 477,
    714, 708, 170, 256, 809, 278,
    643, 647, 342, 678, 299, 563,
    597, 532, 330, 87, 694, 889,
    466, 682, 431, 383, 502, 698,
    505, 974, 4, 100, 13, 672,
    161, 97, 329, 220, 59, 84,
    732, 567, 157, 531, 747, 344,
    841, 7, 800, 757, 619, 998,
    // Bin capacity
    1886, 1352, 1030, 1255, 1276, 1773
  };
  const int i6_18_6_7_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    729, 586, 503, 6, 74, 145,
    580, 124, 648, 269, 420, 59,
    363, 498, 841, 862, 469, 557,
    512, 106, 148, 936, 876, 814,
    212, 593, 504, 525, 629, 317,
    913, 402, 858, 748, 58, 878,
    791, 86, 38, 539, 477, 239,
    548, 719, 359, 11, 683, 459,
    64, 900, 51, 858, 954, 576,
    989, 674, 565, 5, 124, 954,
    933, 501, 317, 879, 300, 630,
    553, 658, 491, 906, 122, 156,
    97, 542, 352, 607, 416, 207,
    76, 873, 176, 213, 253, 839,
    236, 193, 104, 883, 95, 321,
    336, 55, 533, 185, 342, 248,
    10, 376, 966, 648, 555, 743,
    631, 198, 657, 266, 577, 315,
    // Bin capacity
    1529, 1442, 1447, 1667, 1324, 1509
  };
  const int i6_18_6_8_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    360, 660, 345, 325, 512, 79,
    68, 990, 212, 287, 858, 535,
    601, 493, 631, 498, 218, 691,
    560, 730, 733, 208, 550, 287,
    463, 678, 140, 903, 653, 885,
    758, 633, 244, 103, 116, 578,
    325, 459, 133, 938, 800, 52,
    811, 907, 255, 443, 320, 623,
    506, 522, 778, 561, 113, 620,
    675, 717, 747, 746, 742, 109,
    955, 16, 848, 756, 713, 910,
    523, 426, 318, 222, 999, 861,
    701, 763, 664, 415, 945, 687,
    806, 878, 367, 729, 379, 218,
    471, 957, 342, 354, 121, 498,
    67, 179, 783, 810, 311, 533,
    400, 918, 946, 479, 407, 313,
    485, 428, 221, 685, 529, 696,
    // Bin capacity
    1717, 2044, 1568, 1704, 1672, 1652
  };
  const int i6_18_6_9_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    403, 208, 355, 29, 487, 743,
    480, 979, 111, 456, 188, 334,
    115, 255, 135, 891, 454, 543,
    591, 93, 273, 650, 400, 811,
    968, 829, 665, 382, 908, 466,
    902, 585, 732, 910, 857, 701,
    979, 144, 626, 221, 477, 962,
    151, 270, 901, 602, 43, 804,
    863, 996, 782, 882, 661, 768,
    939, 891, 736, 783, 574, 514,
    63, 941, 682, 2, 138, 109,
    336, 589, 624, 685, 414, 147,
    932, 485, 785, 614, 197, 818,
    253, 226, 888, 553, 35, 178,
    26, 691, 559, 484, 528, 85,
    263, 943, 543, 233, 912, 325,
    845, 877, 561, 201, 257, 805,
    83, 836, 569, 128, 442, 838,
    // Bin capacity
    1670, 1969, 1913, 1582, 1449, 1808
  };
  const int i6_18_6_10_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    921, 836, 759, 983, 824, 728,
    376, 181, 304, 56, 214, 148,
    408, 157, 189, 403, 949, 167,
    340, 973, 291, 710, 154, 220,
    330, 147, 444, 848, 629, 460,
    540, 694, 673, 798, 297, 523,
    834, 47, 663, 139, 740, 57,
    880, 434, 553, 428, 50, 333,
    269, 101, 745, 312, 569, 955,
    874, 610, 471, 84, 430, 478,
    366, 680, 717, 479, 67, 412,
    502, 91, 976, 132, 155, 550,
    260, 537, 543, 355, 126, 425,
    419, 540, 536, 99, 236, 52,
    65, 27, 107, 654, 115, 213,
    230, 793, 280, 559, 684, 412,
    137, 235, 258, 889, 627, 126,
    750, 396, 326, 212, 375, 251,
    // Bin capacity
    1559, 1372, 1620, 1493, 1328, 1194
  };
  const int i6_18_6_11_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    315, 51, 724, 673, 216, 351,
    30, 77, 820, 859, 428, 979,
    681, 536, 237, 843, 715, 529,
    583, 482, 952, 839, 893, 667,
    329, 116, 335, 318, 474, 669,
    358, 805, 979, 819, 630, 655,
    15, 954, 944, 191, 427, 474,
    921, 109, 771, 591, 537, 522,
    95, 879, 859, 169, 11, 290,
    528, 231, 48, 874, 167, 589,
    14, 445, 989, 140, 767, 504,
    465, 507, 950, 888, 689, 726,
    145, 410, 898, 197, 346, 990,
    688, 208, 125, 778, 399, 157,
    640, 183, 770, 409, 956, 447,
    308, 856, 879, 115, 310, 256,
    612, 169, 492, 303, 227, 102,
    317, 816, 791, 286, 272, 233,
    // Bin capacity
    1304, 1450, 2325, 1720, 1566, 1691
  };
  const int i6_18_6_12_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    872, 959, 887, 590, 909, 882,
    715, 908, 727, 70, 530, 672,
    831, 939, 989, 384, 420, 3,
    443, 453, 616, 799, 11, 465,
    490, 493, 357, 890, 74, 30,
    15, 435, 376, 767, 162, 800,
    192, 398, 778, 234, 531, 90,
    723, 860, 862, 902, 720, 747,
    158, 865, 361, 907, 921, 153,
    431, 852, 219, 262, 960, 75,
    133, 183, 217, 948, 818, 341,
    418, 121, 698, 404, 809, 941,
    253, 996, 240, 764, 132, 792,
    408, 827, 158, 132, 447, 600,
    956, 617, 246, 32, 231, 706,
    875, 419, 511, 8, 786, 243,
    670, 703, 625, 397, 323, 992,
    489, 601, 848, 938, 549, 664,
    // Bin capacity
    1694, 2171, 1814, 1760, 1743, 1717
  };
  const int i6_18_6_13_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    548, 809, 810, 240, 436, 61,
    612, 300, 193, 567, 694, 358,
    960, 277, 478, 293, 443, 779,
    188, 33, 172, 976, 717, 68,
    53, 236, 644, 959, 239, 614,
    434, 395, 770, 484, 854, 166,
    1000, 415, 806, 152, 227, 261,
    627, 488, 413, 981, 998, 213,
    68, 490, 823, 754, 849, 413,
    327, 796, 63, 87, 321, 549,
    457, 150, 130, 100, 911, 421,
    120, 825, 818, 464, 691, 834,
    309, 405, 848, 968, 586, 441,
    896, 718, 600, 856, 184, 720,
    69, 630, 451, 592, 176, 116,
    122, 586, 460, 535, 537, 162,
    764, 200, 871, 252, 807, 1000,
    488, 587, 905, 697, 147, 750,
    // Bin capacity
    1515, 1571, 1932, 1876, 1849, 1493
  };
  const int i6_18_6_14_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    240, 320, 371, 22, 497, 838,
    298, 934, 291, 355, 819, 958,
    791, 717, 795, 41, 386, 810,
    361, 327, 588, 979, 621, 933,
    479, 833, 246, 781, 644, 549,
    853, 611, 465, 301, 39, 522,
    977, 916, 195, 233, 241, 244,
    290, 184, 732, 751, 782, 206,
    183, 242, 142, 993, 495, 425,
    806, 340, 408, 717, 965, 284,
    514, 316, 459, 134, 444, 581,
    199, 79, 592, 824, 92, 311,
    423, 332, 646, 46, 94, 394,
    15, 245, 147, 329, 707, 577,
    394, 831, 638, 463, 766, 373,
    290, 638, 389, 123, 152, 857,
    175, 556, 354, 845, 882, 940,
    348, 273, 854, 597, 654, 452,
    // Bin capacity
    1451, 1652, 1580, 1622, 1764, 1949
  };
  const int i6_18_6_15_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    205, 792, 893, 942, 242, 276,
    511, 333, 572, 378, 572, 639,
    625, 780, 612, 538, 619, 580,
    895, 987, 432, 809, 245, 242,
    445, 76, 796, 592, 470, 378,
    533, 457, 944, 299, 352, 232,
    704, 191, 448, 581, 188, 743,
    162, 464, 431, 1, 330, 427,
    493, 256, 131, 959, 263, 977,
    358, 549, 55, 556, 785, 505,
    608, 985, 650, 704, 239, 931,
    861, 869, 811, 457, 318, 58,
    961, 365, 105, 689, 484, 33,
    777, 516, 780, 779, 364, 398,
    883, 936, 251, 420, 407, 729,
    437, 446, 33, 191, 131, 529,
    779, 232, 718, 418, 866, 980,
    195, 59, 400, 219, 985, 487,
    // Bin capacity
    2000, 1782, 1737, 1827, 1507, 1753
  };
  const int i6_18_6_16_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    651, 617, 294, 714, 637, 654,
    106, 122, 348, 444, 331, 397,
    387, 769, 198, 185, 435, 582,
    318, 932, 470, 744, 643, 612,
    431, 769, 73, 410, 837, 8,
    133, 662, 968, 906, 700, 600,
    561, 355, 219, 491, 953, 122,
    477, 842, 473, 152, 661, 692,
    319, 249, 909, 877, 665, 444,
    942, 874, 518, 712, 875, 924,
    545, 969, 917, 654, 631, 534,
    298, 332, 701, 565, 909, 217,
    90, 310, 637, 179, 253, 352,
    924, 425, 946, 340, 888, 741,
    189, 641, 73, 835, 328, 910,
    839, 915, 836, 955, 908, 737,
    816, 717, 311, 483, 176, 82,
    437, 894, 761, 771, 313, 800,
    // Bin capacity
    1637, 2203, 1867, 2014, 2155, 1819
  };
  const int i6_18_6_17_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    915, 215, 778, 767, 531, 107,
    178, 350, 642, 493, 245, 389,
    126, 89, 302, 917, 299, 676,
    877, 881, 469, 350, 880, 816,
    13, 523, 814, 911, 866, 643,
    755, 37, 138, 46, 235, 526,
    731, 412, 410, 27, 863, 963,
    34, 137, 335, 182, 577, 954,
    177, 631, 334, 600, 133, 793,
    464, 526, 206, 271, 535, 181,
    532, 934, 656, 953, 792, 96,
    30, 950, 119, 738, 327, 367,
    383, 846, 538, 310, 968, 97,
    53, 519, 244, 691, 595, 1000,
    706, 371, 878, 170, 689, 641,
    158, 624, 92, 191, 72, 652,
    108, 155, 758, 587, 114, 428,
    993, 796, 235, 202, 644, 628,
    // Bin capacity
    1411, 1755, 1550, 1640, 1827, 1942
  };
  const int i6_18_6_18_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    699, 47, 523, 795, 933, 295,
    187, 396, 730, 881, 530, 248,
    126, 944, 393, 370, 525, 170,
    777, 943, 303, 105, 623, 312,
    344, 415, 456, 247, 199, 988,
    224, 670, 904, 412, 619, 741,
    911, 90, 168, 78, 380, 511,
    561, 185, 677, 872, 36, 901,
    192, 174, 479, 567, 613, 223,
    927, 421, 563, 122, 56, 865,
    90, 849, 412, 666, 927, 970,
    790, 621, 696, 459, 532, 110,
    886, 250, 498, 156, 917, 642,
    784, 108, 397, 431, 207, 359,
    513, 687, 535, 582, 777, 211,
    787, 168, 863, 648, 45, 943,
    713, 795, 372, 324, 819, 601,
    794, 582, 918, 179, 6, 656,
    // Bin capacity
    2027, 1642, 1945, 1553, 1720, 1917
  };
  const int i6_18_6_19_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    62, 883, 230, 893, 72, 310,
    935, 904, 780, 653, 248, 959,
    134, 59, 277, 699, 24, 365,
    442, 464, 349, 201, 558, 968,
    856, 420, 294, 662, 176, 767,
    932, 750, 101, 356, 69, 237,
    899, 346, 675, 144, 39, 87,
    174, 837, 19, 850, 680, 269,
    316, 458, 154, 650, 864, 261,
    229, 294, 405, 717, 118, 363,
    506, 214, 313, 860, 538, 825,
    358, 307, 745, 728, 833, 386,
    637, 591, 702, 356, 275, 644,
    17, 982, 614, 830, 660, 662,
    38, 662, 526, 617, 796, 775,
    721, 966, 889, 24, 105, 183,
    566, 91, 574, 287, 928, 880,
    205, 951, 22, 944, 23, 559,
    // Bin capacity
    1593, 2019, 1522, 2077, 1390, 1885
  };
  const int i6_18_6_20_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    384, 487, 480, 837, 240, 407,
    309, 770, 817, 428, 803, 249,
    281, 578, 448, 655, 436, 18,
    978, 881, 827, 770, 968, 271,
    129, 137, 861, 440, 613, 790,
    696, 415, 134, 574, 359, 95,
    902, 919, 2, 392, 452, 132,
    442, 595, 423, 720, 620, 872,
    288, 535, 643, 498, 471, 926,
    472, 955, 15, 12, 573, 756,
    975, 210, 70, 182, 813, 243,
    262, 965, 59, 357, 339, 776,
    722, 398, 509, 73, 792, 144,
    373, 341, 922, 636, 750, 231,
    972, 920, 157, 259, 302, 201,
    614, 438, 681, 877, 960, 776,
    424, 530, 794, 84, 853, 592,
    116, 153, 198, 304, 486, 677,
    // Bin capacity
    1868, 2046, 1608, 1620, 2166, 1632
  };
  const int i6_18_6_21_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    14, 599, 642, 890, 587, 925,
    992, 988, 847, 752, 310, 824,
    695, 926, 294, 149, 747, 545,
    307, 148, 423, 358, 979, 711,
    378, 829, 663, 250, 890, 880,
    323, 614, 321, 751, 170, 552,
    273, 954, 781, 772, 256, 765,
    330, 14, 636, 833, 96, 947,
    738, 757, 845, 57, 813, 215,
    930, 192, 541, 899, 364, 405,
    567, 29, 849, 146, 506, 317,
    209, 440, 431, 817, 837, 305,
    786, 451, 953, 781, 390, 287,
    236, 261, 30, 708, 15, 473,
    815, 115, 738, 55, 96, 502,
    91, 486, 685, 415, 511, 490,
    276, 311, 801, 119, 189, 804,
    519, 863, 244, 0, 887, 381,
    // Bin capacity
    1710, 1811, 2163, 1765, 1744, 2083
  };
  const int i6_18_6_22_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    233, 68, 118, 707, 650, 183,
    487, 10, 417, 38, 438, 760,
    231, 986, 537, 178, 498, 435,
    318, 688, 303, 28, 68, 753,
    880, 718, 222, 254, 88, 568,
    471, 110, 658, 127, 951, 804,
    259, 774, 876, 669, 259, 652,
    489, 10, 545, 690, 879, 510,
    601, 908, 20, 632, 475, 150,
    687, 195, 548, 298, 796, 134,
    565, 184, 869, 162, 69, 94,
    911, 409, 823, 347, 433, 561,
    283, 661, 115, 552, 467, 210,
    667, 983, 414, 341, 613, 196,
    385, 656, 219, 220, 381, 457,
    511, 80, 409, 634, 942, 959,
    650, 410, 163, 310, 919, 993,
    379, 813, 388, 218, 863, 864,
    // Bin capacity
    1832, 1762, 1555, 1303, 1991, 1888
  };
  const int i6_18_6_23_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    353, 822, 756, 19, 463, 909,
    722, 123, 3, 864, 493, 810,
    623, 590, 974, 361, 415, 373,
    899, 354, 478, 825, 158, 444,
    610, 782, 107, 939, 820, 981,
    182, 335, 156, 692, 476, 4,
    47, 791, 23, 72, 915, 749,
    103, 934, 589, 386, 27, 121,
    2, 214, 178, 457, 483, 436,
    728, 405, 488, 772, 288, 869,
    144, 46, 858, 297, 326, 815,
    188, 748, 559, 935, 305, 596,
    54, 908, 276, 264, 145, 286,
    684, 459, 73, 541, 185, 546,
    851, 440, 258, 29, 782, 661,
    110, 225, 751, 919, 476, 915,
    479, 543, 453, 59, 996, 477,
    906, 196, 533, 409, 652, 216,
    // Bin capacity
    1576, 1828, 1541, 1813, 1724, 2093
  };
  const int i6_18_6_24_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    428, 24, 0, 270, 989, 704,
    44, 911, 14, 611, 489, 293,
    915, 226, 674, 972, 21, 187,
    336, 251, 371, 461, 725, 388,
    96, 373, 900, 133, 190, 515,
    504, 73, 505, 110, 385, 964,
    289, 864, 484, 335, 333, 912,
    169, 879, 571, 733, 171, 706,
    567, 522, 953, 526, 180, 532,
    84, 183, 907, 117, 820, 245,
    486, 404, 649, 706, 364, 523,
    896, 534, 90, 541, 550, 313,
    591, 1, 972, 161, 13, 154,
    4, 712, 809, 26, 558, 118,
    163, 133, 585, 219, 301, 403,
    329, 942, 498, 902, 287, 236,
    11, 26, 937, 435, 289, 379,
    789, 359, 208, 823, 693, 558,
    // Bin capacity
    1385, 1533, 2093, 1671, 1521, 1681
  };
  const int i6_18_6_25_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    993, 103, 191, 1, 912, 22,
    656, 253, 450, 858, 914, 817,
    204, 715, 17, 263, 820, 862,
    186, 635, 650, 131, 593, 629,
    699, 879, 398, 740, 689, 740,
    114, 347, 447, 108, 896, 35,
    144, 318, 93, 55, 75, 3,
    196, 533, 570, 744, 527, 897,
    761, 703, 149, 393, 307, 697,
    660, 833, 782, 631, 236, 291,
    287, 561, 253, 865, 307, 556,
    741, 509, 601, 987, 510, 619,
    989, 378, 462, 512, 995, 619,
    865, 68, 84, 317, 467, 68,
    227, 309, 363, 856, 867, 381,
    981, 921, 470, 740, 325, 403,
    854, 86, 943, 547, 795, 529,
    491, 513, 275, 126, 727, 646,
    // Bin capacity
    2094, 1805, 1500, 1849, 2284, 1837
  };
  const int i6_18_6_26_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    980, 64, 168, 679, 527, 257,
    648, 85, 383, 394, 447, 168,
    389, 765, 215, 933, 10, 294,
    180, 92, 385, 94, 944, 756,
    538, 913, 520, 707, 252, 962,
    408, 272, 380, 590, 735, 319,
    560, 218, 730, 67, 922, 831,
    830, 399, 31, 937, 174, 31,
    560, 908, 891, 329, 624, 127,
    456, 801, 458, 165, 884, 965,
    541, 808, 929, 121, 110, 746,
    124, 398, 394, 685, 169, 916,
    134, 747, 608, 647, 222, 86,
    26, 380, 311, 415, 594, 427,
    33, 289, 472, 170, 992, 818,
    736, 495, 820, 487, 980, 756,
    301, 303, 270, 326, 376, 968,
    215, 264, 983, 93, 65, 967,
    // Bin capacity
    1609, 1723, 1880, 1647, 1896, 2183
  };
  const int i6_18_6_27_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    974, 931, 707, 332, 376, 738,
    406, 49, 471, 486, 828, 535,
    296, 342, 341, 83, 406, 71,
    847, 934, 800, 186, 400, 627,
    116, 633, 373, 446, 21, 432,
    586, 438, 878, 635, 931, 495,
    568, 27, 176, 422, 127, 661,
    738, 665, 683, 959, 371, 813,
    756, 165, 953, 546, 473, 606,
    580, 931, 460, 537, 543, 737,
    483, 581, 442, 572, 384, 45,
    78, 36, 107, 656, 613, 862,
    455, 321, 210, 609, 340, 602,
    494, 211, 272, 628, 891, 205,
    25, 53, 507, 38, 202, 976,
    758, 644, 678, 529, 518, 323,
    475, 204, 720, 446, 868, 78,
    529, 230, 364, 374, 741, 982,
    // Bin capacity
    1940, 1566, 1936, 1796, 1912, 2072
  };
  const int i6_18_6_28_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    499, 219, 29, 6, 180, 390,
    193, 741, 435, 820, 603, 356,
    308, 263, 775, 616, 760, 51,
    641, 747, 701, 678, 240, 78,
    625, 931, 683, 145, 553, 798,
    85, 357, 732, 359, 325, 529,
    320, 968, 722, 776, 104, 485,
    929, 107, 131, 130, 723, 729,
    170, 321, 513, 955, 149, 860,
    678, 802, 934, 655, 774, 591,
    66, 213, 338, 353, 493, 779,
    761, 846, 624, 453, 733, 264,
    288, 944, 893, 933, 320, 781,
    658, 405, 92, 889, 544, 25,
    683, 353, 973, 267, 630, 642,
    657, 199, 585, 169, 668, 421,
    802, 38, 242, 662, 268, 830,
    991, 360, 272, 183, 584, 592,
    // Bin capacity
    1996, 1881, 2064, 1931, 1846, 1963
  };
  const int i6_18_6_29_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    529, 712, 985, 868, 888, 712,
    825, 607, 851, 942, 891, 17,
    888, 6, 943, 17, 718, 906,
    264, 988, 826, 18, 726, 680,
    90, 670, 437, 764, 822, 784,
    676, 597, 626, 748, 540, 147,
    131, 576, 234, 610, 20, 289,
    255, 413, 592, 517, 638, 702,
    59, 365, 883, 342, 220, 947,
    258, 115, 471, 720, 686, 58,
    718, 903, 778, 812, 604, 543,
    171, 524, 767, 181, 636, 421,
    251, 325, 505, 873, 606, 145,
    95, 173, 847, 573, 377, 721,
    271, 139, 138, 424, 389, 467,
    608, 548, 775, 477, 401, 919,
    93, 78, 683, 256, 30, 595,
    507, 210, 30, 486, 843, 229,
    // Bin capacity
    1439, 1710, 2445, 2071, 2158, 1996
  };
  const int i6_18_6_30_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    322, 754, 321, 747, 206, 83,
    960, 734, 555, 295, 502, 455,
    533, 670, 453, 311, 842, 262,
    661, 771, 634, 314, 82, 184,
    682, 665, 649, 865, 757, 605,
    624, 810, 63, 126, 887, 867,
    332, 91, 557, 664, 690, 774,
    402, 342, 294, 832, 5, 559,
    335, 563, 229, 685, 810, 721,
    293, 996, 65, 349, 886, 424,
    668, 846, 610, 277, 105, 785,
    167, 199, 663, 280, 860, 276,
    258, 71, 823, 690, 121, 716,
    574, 575, 411, 884, 530, 64,
    76, 848, 628, 392, 343, 610,
    292, 886, 222, 944, 846, 678,
    64, 953, 69, 394, 173, 577,
    911, 447, 79, 18, 574, 326,
    // Bin capacity
    1767, 2432, 1588, 1965, 1998, 1943
  };
  const int i6_18_6_31_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    871, 155, 802, 268, 81, 495,
    761, 789, 208, 850, 48, 778,
    814, 353, 124, 415, 938, 950,
    901, 109, 280, 874, 211, 820,
    390, 144, 431, 432, 62, 78,
    147, 226, 964, 94, 881, 441,
    823, 467, 753, 418, 525, 161,
    102, 322, 355, 715, 175, 821,
    74, 385, 300, 363, 424, 523,
    853, 192, 678, 178, 838, 413,
    714, 35, 748, 191, 626, 820,
    98, 443, 607, 781, 546, 997,
    894, 104, 588, 428, 385, 97,
    163, 621, 711, 856, 135, 520,
    626, 109, 11, 499, 620, 910,
    866, 529, 507, 217, 912, 757,
    980, 194, 18, 721, 986, 120,
    284, 71, 798, 521, 822, 286,
    // Bin capacity
    2263, 1146, 1940, 1926, 2012, 2181
  };
  const int i6_18_6_32_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    489, 495, 645, 530, 779, 452,
    474, 622, 531, 256, 675, 863,
    872, 293, 147, 662, 880, 559,
    701, 746, 258, 483, 709, 376,
    832, 243, 724, 58, 191, 18,
    261, 979, 8, 311, 620, 301,
    96, 256, 833, 541, 464, 879,
    94, 601, 52, 133, 384, 909,
    26, 739, 393, 478, 276, 872,
    166, 433, 72, 771, 498, 650,
    90, 910, 531, 585, 695, 508,
    687, 460, 182, 712, 386, 372,
    101, 8, 627, 795, 252, 709,
    926, 578, 346, 324, 11, 356,
    725, 383, 304, 647, 755, 116,
    466, 527, 747, 559, 296, 22,
    286, 412, 287, 576, 943, 465,
    82, 629, 120, 438, 18, 301,
    // Bin capacity
    1623, 2050, 1498, 1949, 1944, 1921
  };
  const int i6_18_6_33_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    289, 232, 907, 343, 410, 110,
    340, 487, 300, 209, 479, 362,
    878, 362, 968, 700, 108, 723,
    475, 10, 730, 794, 229, 153,
    641, 20, 60, 727, 357, 251,
    978, 870, 89, 974, 994, 19,
    320, 506, 866, 778, 246, 26,
    209, 593, 299, 782, 674, 991,
    595, 722, 364, 279, 823, 990,
    524, 512, 337, 268, 394, 649,
    783, 1000, 683, 150, 881, 538,
    810, 589, 550, 172, 454, 754,
    489, 160, 207, 580, 413, 843,
    361, 867, 846, 875, 350, 226,
    763, 295, 928, 432, 267, 22,
    1000, 661, 594, 361, 571, 480,
    527, 667, 897, 747, 666, 638,
    574, 136, 162, 391, 803, 63,
    // Bin capacity
    2340, 1927, 2170, 2120, 2022, 1738
  };
  const int i6_18_6_34_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    373, 891, 610, 89, 554, 840,
    251, 410, 177, 311, 641, 388,
    785, 509, 504, 36, 544, 291,
    179, 761, 122, 602, 249, 464,
    910, 161, 643, 682, 944, 257,
    262, 863, 782, 685, 768, 174,
    260, 293, 55, 870, 293, 522,
    338, 60, 408, 429, 914, 697,
    656, 221, 404, 948, 392, 174,
    324, 426, 451, 950, 331, 50,
    149, 137, 333, 690, 643, 787,
    693, 754, 967, 614, 505, 109,
    175, 887, 129, 793, 539, 88,
    753, 33, 720, 938, 375, 260,
    846, 604, 268, 916, 164, 14,
    817, 526, 711, 617, 516, 300,
    743, 303, 697, 426, 101, 596,
    940, 373, 791, 326, 799, 651,
    // Bin capacity
    2112, 1835, 1960, 2440, 2071, 1488
  };
  const int i6_18_6_35_23[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    683, 230, 918, 595, 404, 540,
    459, 739, 927, 453, 494, 246,
    0, 376, 258, 183, 508, 53,
    545, 341, 661, 791, 520, 426,
    931, 208, 554, 44, 159, 620,
    638, 44, 330, 384, 860, 221,
    76, 786, 755, 348, 559, 507,
    866, 6, 229, 980, 448, 343,
    673, 429, 479, 955, 847, 388,
    347, 736, 188, 361, 286, 141,
    336, 420, 11, 549, 286, 728,
    358, 228, 969, 202, 733, 154,
    78, 290, 405, 165, 716, 21,
    543, 127, 356, 21, 496, 613,
    613, 2, 529, 69, 283, 476,
    858, 726, 427, 847, 839, 396,
    268, 297, 502, 257, 797, 217,
    790, 473, 90, 205, 562, 314,
    // Bin capacity
    2039, 1454, 1933, 1668, 2205, 1441
  };
  const int i6_18_6_0_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    849, 692, 765, 120, 545, 451,
    160, 58, 243, 99, 722, 22,
    946, 442, 301, 758, 443, 859,
    721, 440, 938, 467, 746, 823,
    819, 261, 887, 997, 827, 829,
    249, 901, 738, 582, 316, 479,
    731, 293, 257, 477, 790, 293,
    731, 418, 388, 580, 597, 541,
    663, 616, 164, 337, 156, 594,
    592, 473, 171, 828, 43, 188,
    270, 321, 820, 168, 459, 32,
    630, 676, 557, 310, 761, 404,
    45, 938, 502, 490, 380, 648,
    194, 304, 765, 434, 279, 630,
    72, 308, 502, 330, 62, 71,
    305, 256, 989, 663, 384, 883,
    132, 589, 260, 346, 252, 381,
    955, 557, 288, 960, 758, 147,
    // Bin capacity
    1511, 1424, 1590, 1491, 1420, 1380
  };
  const int i6_18_6_1_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    965, 587, 746, 157, 547, 324,
    261, 156, 457, 115, 347, 116,
    159, 532, 713, 68, 51, 925,
    905, 900, 523, 506, 461, 803,
    731, 981, 487, 846, 329, 546,
    148, 513, 653, 715, 411, 315,
    719, 85, 67, 828, 523, 163,
    426, 283, 353, 651, 646, 637,
    409, 994, 273, 492, 64, 400,
    971, 588, 307, 500, 659, 7,
    133, 298, 884, 96, 80, 599,
    604, 861, 363, 839, 144, 233,
    540, 165, 201, 836, 345, 147,
    953, 384, 624, 181, 711, 525,
    753, 154, 823, 12, 163, 886,
    449, 974, 204, 623, 973, 96,
    529, 703, 668, 745, 461, 122,
    457, 541, 415, 754, 239, 655,
    // Bin capacity
    1703, 1633, 1475, 1509, 1205, 1263
  };
  const int i6_18_6_2_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    75, 401, 980, 552, 374, 434,
    768, 796, 218, 107, 757, 282,
    91, 723, 179, 537, 136, 401,
    726, 290, 656, 227, 724, 217,
    590, 777, 426, 300, 288, 405,
    212, 495, 612, 609, 503, 810,
    950, 130, 990, 548, 708, 764,
    810, 892, 446, 662, 671, 763,
    49, 612, 471, 18, 989, 224,
    651, 59, 910, 379, 2, 331,
    860, 768, 76, 270, 32, 760,
    627, 514, 307, 324, 124, 111,
    8, 724, 863, 501, 236, 459,
    186, 326, 78, 858, 389, 198,
    513, 958, 75, 363, 62, 661,
    605, 175, 696, 492, 824, 700,
    143, 967, 657, 862, 277, 256,
    209, 45, 952, 288, 984, 665,
    // Bin capacity
    1373, 1641, 1631, 1343, 1374, 1435
  };
  const int i6_18_6_3_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    372, 287, 368, 967, 87, 137,
    221, 46, 63, 68, 65, 581,
    191, 67, 919, 599, 886, 265,
    671, 262, 104, 661, 274, 1000,
    442, 550, 69, 715, 782, 402,
    782, 878, 885, 383, 322, 909,
    770, 331, 190, 333, 151, 838,
    349, 12, 548, 391, 623, 26,
    512, 486, 269, 141, 961, 4,
    264, 560, 221, 976, 101, 273,
    580, 891, 152, 49, 855, 361,
    205, 359, 145, 308, 799, 232,
    248, 341, 455, 472, 944, 797,
    329, 830, 236, 926, 46, 852,
    124, 572, 891, 580, 432, 777,
    601, 749, 818, 74, 848, 999,
    937, 190, 959, 773, 998, 460,
    41, 931, 361, 315, 717, 834,
    // Bin capacity
    1312, 1433, 1314, 1499, 1698, 1674
  };
  const int i6_18_6_4_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    855, 950, 175, 652, 460, 785,
    107, 296, 206, 827, 367, 303,
    313, 457, 349, 938, 28, 891,
    46, 290, 760, 517, 331, 458,
    905, 946, 629, 75, 160, 351,
    423, 212, 398, 558, 896, 223,
    878, 380, 236, 737, 379, 913,
    882, 340, 143, 417, 751, 930,
    589, 749, 763, 249, 948, 856,
    834, 378, 601, 7, 517, 608,
    751, 543, 117, 568, 12, 916,
    595, 581, 878, 779, 354, 451,
    456, 319, 941, 743, 302, 464,
    473, 79, 520, 106, 763, 187,
    261, 754, 4, 258, 137, 843,
    294, 941, 848, 525, 733, 3,
    925, 140, 38, 869, 298, 886,
    967, 527, 939, 118, 816, 442,
    // Bin capacity
    1830, 1540, 1482, 1551, 1431, 1822
  };
  const int i6_18_6_5_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    941, 904, 151, 595, 899, 841,
    580, 405, 175, 686, 779, 68,
    426, 157, 790, 667, 97, 167,
    439, 787, 942, 131, 247, 955,
    895, 258, 624, 67, 643, 542,
    235, 938, 820, 781, 318, 822,
    772, 493, 576, 464, 242, 22,
    402, 240, 638, 701, 422, 819,
    944, 411, 702, 940, 319, 429,
    982, 663, 906, 128, 621, 20,
    664, 729, 621, 118, 883, 927,
    866, 512, 626, 404, 744, 535,
    976, 855, 817, 231, 805, 422,
    128, 734, 976, 134, 449, 469,
    126, 474, 236, 55, 33, 36,
    460, 575, 502, 686, 277, 802,
    231, 812, 505, 605, 914, 769,
    192, 213, 805, 969, 791, 558,
    // Bin capacity
    1796, 1778, 1998, 1464, 1660, 1611
  };
  const int i6_18_6_6_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    722, 32, 710, 297, 326, 328,
    873, 4, 592, 967, 676, 445,
    478, 164, 619, 161, 798, 528,
    769, 272, 5, 560, 544, 835,
    128, 100, 996, 672, 844, 393,
    365, 339, 827, 599, 635, 707,
    521, 963, 156, 62, 995, 18,
    571, 131, 190, 157, 245, 790,
    587, 243, 302, 901, 636, 903,
    896, 292, 366, 912, 706, 370,
    427, 275, 90, 975, 581, 324,
    820, 250, 549, 713, 912, 862,
    573, 449, 558, 609, 882, 815,
    744, 751, 594, 607, 225, 767,
    407, 55, 622, 495, 644, 102,
    155, 787, 100, 54, 278, 77,
    371, 547, 999, 579, 808, 405,
    781, 939, 92, 183, 655, 471,
    // Bin capacity
    1800, 1165, 1479, 1679, 2013, 1615
  };
  const int i6_18_6_7_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    697, 518, 502, 188, 659, 846,
    602, 943, 123, 401, 706, 220,
    745, 202, 259, 606, 199, 417,
    505, 124, 748, 506, 155, 765,
    571, 195, 931, 380, 939, 566,
    271, 307, 349, 754, 612, 412,
    629, 399, 679, 367, 606, 159,
    811, 624, 34, 917, 645, 226,
    976, 249, 822, 845, 330, 836,
    605, 242, 18, 171, 593, 431,
    574, 204, 505, 639, 10, 826,
    69, 910, 525, 821, 817, 575,
    197, 527, 876, 629, 113, 112,
    854, 411, 274, 879, 300, 673,
    481, 151, 593, 425, 957, 47,
    853, 754, 1000, 245, 102, 723,
    809, 418, 877, 829, 785, 862,
    168, 758, 597, 162, 354, 346,
    // Bin capacity
    1858, 1416, 1732, 1742, 1584, 1613
  };
  const int i6_18_6_8_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    360, 410, 140, 592, 49, 420,
    464, 800, 267, 446, 693, 799,
    387, 430, 501, 627, 914, 189,
    212, 642, 745, 928, 133, 98,
    503, 283, 923, 610, 49, 113,
    449, 961, 55, 752, 760, 159,
    272, 563, 588, 541, 192, 215,
    745, 436, 505, 143, 739, 152,
    325, 266, 948, 171, 237, 548,
    41, 597, 326, 747, 386, 444,
    883, 867, 602, 520, 252, 729,
    257, 579, 17, 843, 399, 374,
    878, 461, 300, 90, 33, 98,
    776, 420, 572, 705, 670, 401,
    71, 170, 301, 362, 768, 634,
    653, 736, 387, 476, 761, 802,
    30, 865, 687, 509, 565, 371,
    330, 723, 656, 581, 734, 703,
    // Bin capacity
    1375, 1838, 1534, 1736, 1501, 1305
  };
  const int i6_18_6_9_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    32, 718, 394, 265, 927, 731,
    865, 215, 256, 656, 573, 647,
    248, 384, 224, 975, 912, 154,
    147, 244, 196, 383, 750, 787,
    2, 168, 240, 703, 724, 342,
    5, 323, 817, 15, 907, 603,
    249, 53, 447, 811, 246, 550,
    951, 622, 227, 246, 220, 877,
    875, 971, 421, 916, 261, 40,
    620, 283, 622, 518, 707, 623,
    848, 684, 475, 857, 620, 132,
    253, 287, 590, 892, 176, 246,
    839, 514, 509, 186, 558, 369,
    219, 945, 510, 246, 803, 848,
    798, 651, 90, 54, 763, 928,
    892, 108, 236, 255, 510, 395,
    341, 18, 609, 889, 877, 637,
    519, 526, 485, 919, 257, 400,
    // Bin capacity
    1582, 1402, 1335, 1778, 1961, 1692
  };
  const int i6_18_6_10_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    2, 453, 81, 493, 583, 183,
    811, 543, 804, 950, 750, 994,
    522, 161, 709, 557, 684, 367,
    807, 872, 305, 495, 503, 84,
    987, 406, 892, 933, 480, 511,
    598, 147, 735, 138, 465, 907,
    479, 817, 216, 237, 156, 28,
    129, 206, 324, 743, 697, 311,
    554, 335, 672, 796, 726, 861,
    474, 28, 631, 361, 527, 394,
    913, 130, 308, 812, 43, 804,
    468, 541, 792, 458, 630, 239,
    276, 827, 934, 888, 184, 523,
    949, 206, 377, 469, 391, 953,
    149, 767, 310, 61, 135, 59,
    237, 910, 650, 81, 295, 389,
    963, 22, 179, 176, 170, 789,
    619, 771, 303, 619, 333, 654,
    // Bin capacity
    1822, 1493, 1691, 1699, 1422, 1660
  };
  const int i6_18_6_11_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    313, 607, 751, 953, 196, 987,
    423, 647, 622, 442, 44, 929,
    414, 487, 46, 745, 109, 199,
    898, 865, 876, 11, 830, 30,
    623, 294, 760, 194, 612, 305,
    913, 723, 913, 664, 456, 514,
    374, 603, 343, 886, 548, 409,
    400, 985, 223, 836, 301, 383,
    541, 729, 935, 640, 298, 195,
    917, 893, 685, 949, 660, 999,
    290, 808, 115, 114, 660, 658,
    205, 464, 359, 280, 771, 402,
    645, 67, 460, 588, 560, 538,
    334, 711, 303, 42, 434, 894,
    236, 195, 413, 857, 276, 60,
    776, 592, 750, 939, 765, 796,
    231, 810, 844, 134, 914, 442,
    388, 691, 485, 922, 938, 427,
    // Bin capacity
    1651, 2067, 1829, 1887, 1734, 1696
  };
  const int i6_18_6_12_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    518, 764, 56, 692, 48, 489,
    841, 367, 722, 782, 606, 570,
    245, 403, 479, 353, 417, 33,
    609, 274, 321, 418, 289, 799,
    502, 18, 480, 412, 900, 106,
    527, 283, 755, 261, 955, 288,
    828, 980, 484, 377, 791, 543,
    578, 424, 865, 460, 160, 672,
    154, 912, 882, 416, 311, 65,
    816, 425, 82, 584, 844, 604,
    295, 757, 804, 49, 177, 324,
    906, 59, 887, 747, 587, 905,
    928, 113, 594, 49, 272, 889,
    361, 239, 132, 793, 363, 538,
    276, 342, 257, 13, 582, 657,
    822, 890, 388, 13, 854, 990,
    404, 90, 294, 778, 715, 159,
    289, 640, 766, 347, 541, 116,
    // Bin capacity
    1848, 1490, 1727, 1409, 1757, 1633
  };
  const int i6_18_6_13_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    584, 544, 847, 287, 109, 82,
    671, 507, 27, 807, 847, 402,
    896, 144, 493, 255, 998, 72,
    249, 463, 463, 661, 728, 220,
    874, 75, 81, 858, 924, 280,
    890, 309, 13, 240, 124, 979,
    614, 607, 93, 999, 115, 888,
    754, 16, 942, 474, 455, 268,
    378, 505, 946, 651, 943, 304,
    439, 40, 970, 458, 73, 216,
    836, 238, 194, 362, 167, 249,
    67, 483, 496, 225, 101, 155,
    715, 417, 663, 699, 482, 322,
    587, 462, 338, 638, 191, 426,
    851, 756, 229, 902, 140, 162,
    283, 650, 765, 684, 108, 515,
    126, 856, 667, 535, 174, 42,
    908, 668, 150, 45, 388, 657,
    // Bin capacity
    2020, 1458, 1578, 1842, 1331, 1176
  };
  const int i6_18_6_14_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    599, 26, 126, 560, 918, 231,
    986, 361, 231, 936, 40, 210,
    465, 328, 74, 318, 165, 361,
    364, 138, 392, 79, 515, 125,
    5, 484, 810, 668, 791, 789,
    684, 804, 303, 929, 837, 20,
    38, 284, 971, 933, 754, 80,
    290, 509, 391, 977, 971, 785,
    55, 383, 249, 933, 201, 265,
    875, 998, 472, 459, 980, 221,
    948, 795, 104, 948, 354, 190,
    979, 749, 969, 167, 334, 144,
    52, 199, 714, 803, 414, 993,
    281, 579, 867, 863, 505, 445,
    316, 106, 147, 808, 978, 791,
    506, 289, 961, 589, 669, 600,
    569, 510, 58, 327, 613, 387,
    205, 51, 816, 499, 777, 885,
    // Bin capacity
    1562, 1443, 1645, 2242, 2056, 1430
  };
  const int i6_18_6_15_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    989, 446, 422, 899, 755, 280,
    492, 146, 94, 970, 872, 885,
    335, 298, 646, 28, 94, 924,
    535, 672, 564, 48, 888, 715,
    245, 809, 844, 499, 354, 152,
    660, 967, 846, 870, 742, 870,
    158, 936, 505, 369, 597, 203,
    466, 737, 229, 260, 622, 901,
    860, 630, 43, 309, 35, 456,
    470, 397, 129, 409, 193, 27,
    825, 907, 781, 639, 678, 952,
    942, 933, 377, 727, 334, 344,
    27, 83, 643, 431, 555, 411,
    401, 754, 744, 66, 307, 186,
    450, 901, 792, 334, 761, 153,
    837, 794, 625, 736, 390, 144,
    68, 51, 224, 137, 525, 147,
    49, 475, 207, 151, 308, 658,
    // Bin capacity
    1689, 2097, 1671, 1511, 1727, 1612
  };
  const int i6_18_6_16_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    787, 771, 530, 81, 16, 800,
    638, 511, 825, 157, 42, 766,
    978, 783, 348, 272, 455, 132,
    369, 601, 725, 624, 348, 442,
    271, 267, 511, 804, 399, 284,
    861, 786, 702, 91, 976, 538,
    538, 340, 940, 34, 553, 464,
    549, 692, 131, 531, 970, 264,
    186, 288, 267, 407, 53, 423,
    183, 852, 282, 591, 669, 576,
    268, 191, 712, 976, 173, 578,
    606, 492, 930, 296, 27, 52,
    647, 523, 131, 247, 914, 936,
    626, 283, 189, 455, 357, 413,
    943, 494, 801, 930, 660, 728,
    871, 479, 933, 722, 451, 259,
    370, 640, 853, 142, 903, 913,
    558, 928, 371, 156, 519, 678,
    // Bin capacity
    1982, 1919, 1969, 1454, 1641, 1788
  };
  const int i6_18_6_17_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    421, 202, 931, 519, 621, 25,
    670, 720, 801, 461, 349, 382,
    608, 71, 380, 795, 620, 295,
    887, 73, 725, 933, 121, 859,
    642, 577, 201, 234, 89, 649,
    766, 814, 127, 221, 536, 761,
    315, 706, 170, 65, 938, 905,
    472, 181, 307, 444, 882, 555,
    682, 512, 899, 513, 361, 588,
    741, 843, 169, 709, 446, 146,
    4, 208, 242, 868, 729, 634,
    335, 685, 443, 674, 220, 119,
    917, 186, 451, 237, 438, 799,
    549, 780, 631, 162, 321, 112,
    618, 37, 354, 908, 904, 516,
    305, 345, 173, 701, 131, 724,
    148, 432, 593, 361, 732, 563,
    121, 841, 733, 846, 560, 28,
    // Bin capacity
    1795, 1602, 1625, 1882, 1755, 1689
  };
  const int i6_18_6_18_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    101, 215, 966, 141, 142, 669,
    395, 679, 359, 657, 113, 940,
    847, 625, 551, 637, 146, 26,
    367, 659, 947, 812, 473, 826,
    951, 351, 959, 963, 92, 123,
    807, 603, 851, 606, 489, 760,
    289, 898, 827, 304, 462, 139,
    443, 198, 510, 350, 58, 961,
    26, 207, 728, 51, 823, 216,
    224, 806, 243, 566, 925, 452,
    125, 928, 529, 957, 96, 963,
    207, 232, 530, 112, 680, 169,
    142, 169, 199, 507, 976, 488,
    611, 280, 536, 152, 557, 48,
    925, 769, 671, 111, 446, 488,
    159, 266, 584, 558, 259, 580,
    933, 759, 738, 333, 46, 968,
    656, 776, 678, 365, 237, 249,
    // Bin capacity
    1615, 1853, 2244, 1610, 1381, 1783
  };
  const int i6_18_6_19_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    815, 395, 103, 344, 847, 104,
    38, 535, 67, 331, 901, 89,
    698, 455, 434, 635, 99, 317,
    659, 907, 568, 387, 897, 96,
    290, 167, 564, 81, 75, 557,
    135, 696, 145, 47, 741, 910,
    278, 515, 493, 837, 745, 648,
    418, 268, 99, 131, 933, 161,
    585, 301, 824, 804, 625, 344,
    46, 458, 881, 789, 690, 958,
    862, 740, 125, 888, 612, 902,
    23, 127, 499, 492, 605, 492,
    563, 157, 684, 503, 672, 888,
    470, 20, 686, 907, 64, 559,
    421, 558, 520, 657, 387, 529,
    880, 224, 442, 118, 575, 570,
    320, 824, 151, 514, 511, 641,
    451, 430, 668, 709, 253, 178,
    // Bin capacity
    1578, 1543, 1578, 1820, 2030, 1774
  };
  const int i6_18_6_20_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    259, 677, 979, 906, 323, 741,
    793, 553, 478, 258, 489, 453,
    497, 174, 231, 271, 311, 544,
    912, 267, 523, 398, 73, 953,
    540, 244, 375, 882, 330, 960,
    488, 94, 413, 158, 250, 265,
    693, 457, 609, 772, 364, 875,
    631, 24, 482, 117, 710, 740,
    712, 638, 61, 476, 251, 616,
    85, 980, 334, 201, 945, 584,
    811, 130, 271, 818, 739, 650,
    996, 273, 355, 831, 350, 440,
    433, 810, 495, 704, 157, 579,
    131, 604, 835, 953, 330, 455,
    1, 32, 318, 741, 332, 555,
    397, 390, 80, 482, 432, 500,
    594, 304, 538, 533, 415, 369,
    483, 115, 204, 120, 663, 459,
    // Bin capacity
    1892, 1354, 1517, 1925, 1493, 2148
  };
  const int i6_18_6_21_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    534, 39, 191, 618, 220, 559,
    248, 697, 98, 987, 850, 759,
    447, 94, 295, 903, 442, 251,
    66, 204, 388, 269, 151, 857,
    556, 163, 37, 54, 192, 558,
    565, 725, 496, 218, 390, 104,
    655, 487, 506, 584, 821, 551,
    291, 816, 2, 882, 515, 617,
    887, 760, 488, 208, 411, 64,
    683, 734, 107, 126, 242, 462,
    551, 505, 365, 889, 46, 796,
    407, 653, 298, 527, 328, 568,
    78, 812, 714, 150, 819, 81,
    446, 994, 234, 446, 705, 216,
    807, 344, 30, 881, 177, 970,
    892, 628, 93, 146, 174, 298,
    964, 242, 63, 719, 166, 207,
    574, 435, 438, 309, 164, 921,
    // Bin capacity
    1947, 1882, 977, 1799, 1374, 1783
  };
  const int i6_18_6_22_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    44, 444, 486, 814, 844, 604,
    261, 36, 700, 40, 373, 247,
    769, 851, 884, 791, 185, 138,
    703, 217, 92, 665, 278, 325,
    679, 29, 906, 838, 774, 190,
    794, 514, 117, 807, 352, 440,
    487, 1000, 40, 733, 465, 921,
    601, 854, 39, 734, 967, 935,
    131, 580, 10, 459, 343, 236,
    880, 993, 87, 759, 995, 373,
    918, 208, 231, 401, 712, 244,
    627, 654, 187, 572, 259, 367,
    621, 443, 231, 698, 865, 653,
    87, 391, 346, 972, 895, 102,
    819, 968, 988, 224, 403, 359,
    677, 566, 754, 593, 857, 231,
    569, 643, 893, 194, 66, 582,
    674, 748, 914, 551, 30, 771,
    // Bin capacity
    2103, 2062, 1608, 2206, 1965, 1570
  };
  const int i6_18_6_23_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    307, 419, 33, 221, 779, 686,
    814, 383, 241, 989, 356, 185,
    217, 493, 545, 245, 862, 726,
    869, 923, 669, 525, 214, 506,
    139, 905, 695, 52, 250, 812,
    269, 319, 780, 464, 973, 946,
    898, 455, 928, 795, 172, 727,
    21, 180, 474, 40, 661, 539,
    442, 849, 619, 826, 696, 518,
    11, 375, 896, 700, 635, 265,
    128, 339, 388, 133, 468, 590,
    981, 931, 287, 33, 205, 561,
    585, 303, 846, 643, 133, 742,
    189, 751, 715, 485, 925, 729,
    732, 1, 339, 626, 161, 243,
    131, 21, 61, 90, 767, 267,
    76, 308, 518, 207, 562, 727,
    72, 882, 410, 496, 81, 67,
    // Bin capacity
    1411, 1812, 1937, 1552, 1825, 2017
  };
  const int i6_18_6_24_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    575, 831, 725, 184, 763, 612,
    89, 104, 305, 310, 430, 841,
    710, 643, 437, 800, 523, 448,
    908, 693, 70, 348, 93, 997,
    667, 54, 407, 23, 405, 52,
    963, 288, 252, 686, 509, 268,
    736, 678, 344, 417, 708, 221,
    85, 687, 685, 111, 363, 454,
    847, 933, 1000, 741, 609, 476,
    105, 895, 73, 911, 725, 928,
    409, 293, 99, 797, 69, 682,
    449, 33, 565, 862, 814, 958,
    648, 865, 109, 203, 289, 723,
    979, 936, 532, 216, 267, 154,
    819, 541, 848, 795, 90, 280,
    863, 983, 868, 856, 954, 388,
    257, 708, 568, 29, 860, 109,
    86, 819, 601, 211, 622, 639,
    // Bin capacity
    2107, 2271, 1755, 1757, 1880, 1908
  };
  const int i6_18_6_25_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    553, 809, 862, 345, 491, 821,
    264, 449, 146, 563, 223, 990,
    888, 718, 82, 140, 324, 497,
    919, 763, 879, 116, 26, 961,
    724, 121, 305, 459, 316, 920,
    796, 764, 763, 210, 938, 69,
    565, 35, 435, 70, 133, 87,
    418, 631, 453, 802, 82, 909,
    256, 896, 593, 237, 488, 834,
    721, 748, 488, 145, 94, 373,
    931, 316, 8, 920, 857, 997,
    867, 435, 411, 432, 9, 297,
    625, 981, 519, 899, 413, 866,
    207, 38, 559, 968, 239, 408,
    737, 248, 144, 286, 632, 841,
    106, 157, 537, 376, 800, 398,
    333, 807, 85, 23, 440, 327,
    602, 615, 231, 613, 118, 47,
    // Bin capacity
    2190, 1986, 1563, 1585, 1380, 2218
  };
  const int i6_18_6_26_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    991, 786, 981, 974, 785, 14,
    120, 797, 511, 791, 135, 601,
    134, 554, 935, 181, 176, 624,
    589, 726, 530, 369, 197, 119,
    547, 621, 982, 226, 260, 966,
    698, 300, 842, 696, 171, 233,
    671, 345, 813, 428, 837, 573,
    678, 603, 121, 841, 632, 146,
    163, 809, 433, 483, 781, 136,
    629, 10, 983, 78, 435, 168,
    966, 415, 144, 952, 555, 247,
    73, 295, 655, 921, 575, 307,
    904, 315, 923, 112, 85, 556,
    64, 380, 71, 616, 654, 954,
    850, 236, 307, 710, 711, 477,
    617, 881, 635, 965, 608, 510,
    75, 312, 101, 759, 796, 645,
    180, 940, 342, 537, 71, 915,
    // Bin capacity
    1880, 1959, 2165, 2235, 1778, 1721
  };
  const int i6_18_6_27_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    622, 873, 227, 293, 234, 785,
    838, 399, 687, 939, 300, 125,
    0, 259, 957, 601, 760, 726,
    75, 223, 45, 237, 535, 871,
    687, 140, 962, 177, 467, 797,
    391, 753, 925, 719, 814, 775,
    282, 311, 297, 540, 709, 651,
    58, 506, 3, 494, 647, 78,
    983, 164, 460, 357, 421, 498,
    81, 521, 67, 90, 494, 782,
    722, 300, 281, 573, 717, 381,
    494, 34, 915, 838, 535, 303,
    944, 794, 609, 75, 920, 373,
    677, 99, 220, 339, 857, 422,
    69, 164, 813, 35, 874, 558,
    318, 352, 314, 307, 598, 748,
    833, 857, 512, 163, 207, 237,
    714, 57, 565, 771, 544, 537,
    // Bin capacity
    1861, 1441, 1876, 1598, 2251, 2042
  };
  const int i6_18_6_28_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    524, 874, 404, 721, 849, 45,
    773, 957, 157, 850, 76, 490,
    767, 43, 277, 981, 724, 305,
    914, 196, 507, 701, 208, 677,
    198, 544, 780, 287, 162, 350,
    445, 426, 374, 372, 714, 631,
    455, 577, 841, 325, 470, 145,
    479, 893, 80, 423, 777, 111,
    605, 892, 85, 0, 569, 997,
    68, 407, 331, 374, 697, 230,
    322, 95, 973, 20, 755, 898,
    404, 252, 870, 759, 840, 752,
    710, 801, 890, 793, 51, 865,
    446, 407, 557, 894, 506, 262,
    498, 558, 3, 715, 471, 711,
    522, 384, 330, 897, 520, 827,
    705, 355, 225, 341, 755, 512,
    727, 938, 68, 269, 111, 276,
    // Bin capacity
    2040, 2048, 1654, 2075, 1975, 1938
  };
  const int i6_18_6_29_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    118, 578, 824, 621, 808, 705,
    41, 585, 765, 193, 530, 741,
    570, 987, 38, 350, 428, 967,
    955, 14, 676, 994, 261, 855,
    666, 355, 497, 360, 310, 535,
    883, 968, 297, 550, 546, 718,
    525, 597, 0, 91, 875, 678,
    478, 360, 976, 563, 630, 292,
    596, 780, 612, 305, 414, 758,
    691, 953, 74, 541, 619, 554,
    854, 963, 864, 732, 877, 827,
    945, 592, 603, 623, 103, 312,
    743, 834, 727, 761, 546, 516,
    552, 949, 179, 548, 416, 495,
    515, 313, 755, 493, 411, 230,
    459, 635, 795, 231, 748, 713,
    27, 659, 79, 166, 313, 719,
    948, 134, 851, 182, 636, 238,
    // Bin capacity
    2272, 2421, 2067, 1786, 2037, 2334
  };
  const int i6_18_6_30_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    602, 93, 126, 905, 946, 397,
    566, 234, 547, 337, 53, 207,
    625, 239, 514, 623, 465, 753,
    543, 172, 941, 582, 970, 256,
    691, 816, 15, 662, 947, 498,
    645, 81, 250, 713, 199, 133,
    810, 399, 103, 113, 213, 295,
    188, 56, 452, 673, 56, 948,
    191, 852, 680, 476, 738, 932,
    905, 625, 993, 453, 671, 912,
    142, 47, 822, 13, 849, 156,
    459, 274, 919, 702, 486, 986,
    833, 128, 23, 42, 12, 717,
    762, 947, 861, 672, 657, 304,
    651, 775, 250, 441, 63, 747,
    826, 347, 771, 48, 201, 408,
    720, 817, 627, 686, 598, 424,
    780, 373, 236, 208, 270, 346,
    // Bin capacity
    2371, 1577, 1979, 1809, 1819, 2041
  };
  const int i6_18_6_31_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    565, 491, 117, 227, 794, 303,
    177, 518, 564, 422, 118, 674,
    831, 141, 309, 946, 857, 243,
    61, 160, 738, 237, 166, 465,
    233, 914, 615, 325, 842, 59,
    891, 516, 635, 648, 793, 995,
    74, 949, 628, 113, 93, 690,
    213, 146, 249, 917, 462, 567,
    699, 759, 557, 239, 305, 58,
    460, 465, 935, 526, 240, 738,
    598, 958, 721, 196, 687, 910,
    605, 972, 3, 10, 313, 861,
    666, 163, 326, 16, 403, 794,
    368, 207, 961, 768, 732, 287,
    705, 42, 603, 246, 778, 534,
    17, 340, 7, 735, 985, 171,
    56, 43, 116, 380, 397, 525,
    289, 42, 86, 129, 745, 382,
    // Bin capacity
    1640, 1709, 1784, 1546, 2121, 2021
  };
  const int i6_18_6_32_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    473, 188, 292, 31, 81, 785,
    507, 542, 541, 578, 740, 662,
    309, 279, 547, 284, 216, 877,
    944, 880, 231, 634, 36, 894,
    841, 662, 386, 876, 537, 274,
    610, 506, 960, 517, 268, 211,
    130, 213, 260, 734, 91, 805,
    460, 86, 490, 729, 628, 560,
    351, 766, 779, 603, 27, 459,
    639, 418, 115, 295, 142, 945,
    816, 71, 429, 770, 339, 178,
    305, 35, 589, 791, 84, 616,
    304, 271, 954, 485, 427, 296,
    524, 340, 952, 987, 270, 576,
    106, 934, 282, 179, 937, 134,
    700, 949, 383, 13, 496, 188,
    611, 514, 922, 206, 160, 455,
    861, 770, 336, 195, 127, 375,
    // Bin capacity
    2089, 1854, 2079, 1960, 1234, 2044
  };
  const int i6_18_6_33_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    902, 543, 568, 881, 153, 306,
    685, 989, 26, 699, 660, 672,
    46, 459, 742, 827, 680, 345,
    12, 806, 15, 262, 470, 83,
    352, 502, 855, 223, 477, 784,
    735, 177, 958, 506, 671, 386,
    973, 999, 247, 608, 228, 390,
    380, 20, 585, 10, 795, 527,
    7, 2, 931, 853, 413, 263,
    89, 272, 109, 413, 22, 288,
    177, 200, 341, 192, 562, 477,
    123, 90, 916, 210, 916, 670,
    936, 588, 487, 587, 223, 617,
    777, 406, 475, 907, 689, 923,
    429, 451, 855, 783, 324, 427,
    813, 252, 775, 190, 980, 502,
    588, 201, 915, 912, 712, 128,
    692, 402, 319, 332, 689, 147,
    // Bin capacity
    1933, 1632, 2244, 2083, 2143, 1759
  };
  const int i6_18_6_34_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    550, 339, 417, 469, 713, 79,
    827, 178, 766, 342, 676, 861,
    857, 243, 271, 152, 504, 978,
    914, 57, 563, 17, 912, 496,
    190, 321, 220, 200, 566, 690,
    612, 731, 569, 393, 539, 6,
    783, 110, 573, 911, 783, 462,
    924, 79, 507, 88, 152, 839,
    587, 559, 256, 187, 960, 553,
    19, 293, 746, 220, 966, 152,
    36, 679, 495, 739, 464, 522,
    298, 371, 790, 664, 483, 22,
    382, 478, 708, 751, 449, 57,
    470, 961, 838, 2, 806, 816,
    843, 413, 317, 323, 661, 624,
    416, 50, 863, 595, 902, 51,
    732, 511, 573, 698, 780, 589,
    137, 202, 921, 338, 684, 674,
    // Bin capacity
    2139, 1469, 2322, 1584, 2681, 1892
  };
  const int i6_18_6_35_24[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    41, 846, 2, 262, 154, 164,
    306, 717, 897, 188, 890, 200,
    749, 804, 35, 553, 283, 650,
    607, 934, 12, 553, 386, 14,
    77, 456, 858, 581, 479, 879,
    27, 0, 181, 377, 119, 945,
    504, 99, 827, 254, 121, 601,
    677, 948, 267, 347, 378, 624,
    168, 332, 125, 324, 146, 57,
    908, 583, 221, 980, 930, 265,
    621, 740, 639, 158, 3, 639,
    444, 962, 650, 808, 937, 82,
    725, 448, 950, 897, 42, 29,
    419, 759, 351, 403, 575, 459,
    852, 581, 686, 861, 937, 303,
    68, 569, 761, 627, 146, 228,
    440, 942, 643, 157, 738, 871,
    720, 723, 871, 611, 538, 965,
    // Bin capacity
    1880, 2575, 2020, 2012, 1756, 1795
  };
  const int i6_18_6_0_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    108, 596, 346, 56, 75, 167,
    969, 5, 335, 402, 52, 154,
    521, 94, 114, 356, 128, 2,
    960, 307, 504, 505, 453, 358,
    490, 743, 661, 48, 568, 3,
    797, 515, 415, 585, 476, 754,
    191, 893, 217, 248, 404, 62,
    245, 360, 727, 953, 501, 999,
    300, 637, 574, 143, 987, 142,
    350, 880, 728, 870, 891, 739,
    777, 625, 926, 11, 411, 530,
    310, 429, 729, 94, 414, 470,
    260, 861, 696, 205, 385, 415,
    30, 381, 792, 154, 54, 939,
    131, 421, 289, 900, 462, 787,
    28, 17, 752, 998, 935, 502,
    164, 523, 944, 318, 827, 364,
    969, 427, 692, 288, 28, 514,
    // Bin capacity
    1267, 1453, 1741, 1189, 1342, 1317
  };
  const int i6_18_6_1_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    113, 413, 126, 710, 422, 338,
    89, 677, 715, 509, 258, 294,
    718, 785, 388, 562, 333, 112,
    757, 191, 741, 103, 309, 115,
    221, 367, 962, 609, 627, 519,
    983, 551, 185, 958, 932, 609,
    177, 919, 100, 972, 287, 683,
    318, 729, 61, 684, 923, 535,
    998, 698, 114, 871, 412, 951,
    667, 846, 867, 627, 30, 650,
    114, 229, 670, 827, 498, 943,
    876, 977, 215, 396, 494, 480,
    931, 960, 0, 501, 15, 910,
    623, 885, 35, 197, 195, 513,
    379, 900, 542, 515, 77, 321,
    773, 680, 662, 520, 562, 984,
    623, 300, 340, 892, 379, 988,
    178, 470, 96, 947, 194, 994,
    // Bin capacity
    1606, 1949, 1148, 1919, 1170, 1842
  };
  const int i6_18_6_2_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    945, 372, 597, 166, 749, 387,
    556, 556, 426, 129, 129, 345,
    500, 315, 185, 737, 256, 947,
    452, 356, 68, 697, 103, 337,
    493, 183, 716, 287, 813, 932,
    310, 314, 582, 609, 582, 486,
    20, 527, 899, 582, 109, 768,
    411, 961, 8, 698, 124, 317,
    487, 300, 542, 282, 477, 872,
    567, 168, 425, 496, 892, 342,
    721, 547, 245, 242, 113, 370,
    99, 157, 914, 723, 812, 611,
    572, 520, 647, 874, 837, 988,
    109, 916, 962, 557, 601, 980,
    688, 48, 103, 343, 475, 648,
    319, 904, 342, 942, 97, 622,
    381, 505, 849, 162, 714, 85,
    234, 178, 307, 870, 429, 697,
    // Bin capacity
    1337, 1331, 1499, 1598, 1414, 1825
  };
  const int i6_18_6_3_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    110, 485, 133, 208, 580, 387,
    266, 108, 626, 336, 701, 299,
    523, 327, 415, 747, 175, 590,
    86, 980, 928, 360, 209, 431,
    411, 517, 302, 73, 581, 384,
    348, 497, 555, 399, 41, 254,
    668, 209, 785, 725, 257, 724,
    524, 760, 93, 558, 137, 554,
    96, 107, 568, 228, 603, 320,
    745, 459, 87, 249, 17, 175,
    915, 87, 611, 10, 936, 192,
    881, 21, 149, 70, 693, 224,
    971, 226, 364, 31, 486, 221,
    861, 886, 505, 275, 914, 910,
    768, 293, 646, 521, 507, 938,
    954, 910, 203, 517, 979, 266,
    596, 565, 749, 885, 359, 564,
    682, 780, 463, 425, 766, 191,
    // Bin capacity
    1787, 1411, 1405, 1136, 1535, 1309
  };
  const int i6_18_6_4_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    277, 458, 854, 512, 374, 918,
    162, 592, 782, 871, 145, 95,
    4, 953, 103, 691, 158, 687,
    602, 907, 231, 436, 645, 133,
    383, 596, 786, 873, 896, 467,
    972, 767, 95, 229, 963, 649,
    808, 464, 875, 960, 543, 169,
    35, 240, 267, 640, 372, 749,
    840, 742, 466, 567, 708, 133,
    858, 232, 915, 40, 775, 736,
    138, 363, 363, 379, 550, 986,
    44, 957, 196, 950, 246, 770,
    771, 954, 346, 354, 584, 156,
    436, 203, 475, 221, 276, 507,
    94, 90, 499, 287, 446, 413,
    507, 415, 616, 601, 933, 537,
    488, 549, 26, 519, 782, 991,
    386, 856, 700, 749, 960, 37,
    // Bin capacity
    1353, 1792, 1490, 1713, 1796, 1584
  };
  const int i6_18_6_5_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    443, 504, 237, 758, 644, 729,
    349, 215, 647, 892, 675, 817,
    659, 363, 83, 198, 244, 321,
    921, 175, 784, 245, 57, 832,
    555, 20, 971, 118, 723, 767,
    914, 644, 346, 28, 523, 553,
    111, 154, 861, 791, 545, 282,
    229, 419, 800, 775, 441, 605,
    47, 241, 401, 987, 504, 606,
    476, 499, 841, 691, 833, 193,
    113, 341, 928, 258, 192, 674,
    60, 873, 378, 154, 710, 435,
    539, 721, 294, 671, 111, 974,
    282, 446, 357, 39, 395, 947,
    612, 480, 557, 125, 204, 165,
    723, 259, 211, 187, 124, 907,
    905, 245, 46, 505, 947, 879,
    762, 20, 828, 911, 930, 969,
    // Bin capacity
    1523, 1159, 1675, 1459, 1541, 2040
  };
  const int i6_18_6_6_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    609, 629, 202, 916, 854, 771,
    758, 231, 766, 693, 444, 629,
    173, 186, 605, 882, 20, 54,
    777, 284, 26, 850, 74, 974,
    265, 360, 791, 493, 389, 623,
    794, 411, 97, 481, 370, 974,
    352, 416, 257, 263, 375, 911,
    107, 684, 760, 65, 387, 200,
    282, 165, 369, 495, 669, 263,
    682, 987, 220, 614, 29, 6,
    909, 487, 370, 199, 227, 968,
    585, 348, 411, 869, 559, 6,
    966, 112, 322, 981, 62, 900,
    994, 518, 228, 260, 827, 82,
    463, 720, 499, 241, 658, 618,
    297, 616, 440, 832, 211, 437,
    132, 300, 502, 827, 150, 74,
    662, 819, 939, 488, 175, 901,
    // Bin capacity
    1733, 1462, 1379, 1846, 1145, 1660
  };
  const int i6_18_6_7_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    721, 301, 311, 951, 726, 164,
    664, 45, 205, 353, 962, 885,
    262, 641, 374, 320, 261, 333,
    769, 787, 547, 76, 958, 460,
    642, 734, 894, 552, 606, 242,
    304, 849, 390, 49, 219, 256,
    976, 732, 600, 972, 124, 108,
    533, 183, 445, 339, 276, 350,
    200, 423, 990, 97, 944, 741,
    337, 1, 365, 902, 36, 284,
    795, 9, 172, 447, 441, 727,
    601, 546, 280, 548, 336, 988,
    478, 976, 369, 306, 133, 198,
    404, 619, 386, 872, 891, 162,
    419, 64, 544, 169, 466, 62,
    80, 279, 943, 3, 461, 145,
    361, 46, 627, 723, 805, 908,
    271, 848, 495, 723, 300, 103,
    // Bin capacity
    1573, 1442, 1594, 1499, 1596, 1270
  };
  const int i6_18_6_8_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    639, 55, 379, 397, 583, 92,
    79, 10, 399, 272, 844, 655,
    258, 545, 224, 878, 534, 632,
    456, 871, 593, 378, 295, 693,
    654, 555, 696, 375, 438, 709,
    182, 440, 26, 453, 464, 507,
    272, 919, 767, 24, 654, 872,
    388, 380, 153, 292, 872, 943,
    764, 142, 755, 802, 380, 520,
    561, 896, 331, 700, 989, 455,
    366, 320, 86, 583, 309, 878,
    367, 559, 127, 128, 551, 597,
    546, 2, 765, 700, 747, 938,
    365, 921, 670, 958, 718, 774,
    805, 893, 259, 817, 516, 150,
    122, 862, 680, 465, 61, 221,
    189, 920, 785, 584, 499, 658,
    451, 810, 110, 499, 851, 683,
    // Bin capacity
    1344, 1818, 1405, 1675, 1855, 1976
  };
  const int i6_18_6_9_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    164, 352, 256, 557, 504, 445,
    719, 387, 193, 150, 604, 164,
    955, 751, 500, 116, 590, 409,
    812, 596, 202, 200, 179, 489,
    304, 980, 970, 486, 679, 897,
    839, 953, 839, 13, 626, 263,
    732, 337, 756, 788, 924, 798,
    933, 621, 8, 873, 951, 307,
    1000, 594, 113, 750, 18, 312,
    849, 542, 317, 218, 654, 387,
    928, 480, 563, 779, 91, 855,
    688, 774, 164, 789, 256, 659,
    841, 506, 925, 428, 113, 666,
    939, 12, 993, 342, 259, 706,
    448, 472, 886, 650, 314, 515,
    941, 71, 107, 478, 327, 260,
    701, 309, 408, 333, 454, 826,
    443, 85, 697, 274, 582, 739,
    // Bin capacity
    2405, 1603, 1617, 1495, 1477, 1762
  };
  const int i6_18_6_10_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    72, 679, 892, 145, 726, 277,
    383, 161, 844, 712, 54, 924,
    295, 194, 10, 163, 18, 797,
    485, 621, 523, 821, 273, 707,
    544, 525, 543, 646, 734, 922,
    65, 609, 568, 810, 430, 880,
    89, 590, 405, 294, 358, 623,
    336, 682, 599, 504, 7, 581,
    755, 235, 729, 149, 450, 945,
    677, 501, 223, 460, 607, 205,
    913, 499, 339, 176, 596, 679,
    166, 271, 763, 616, 217, 549,
    836, 935, 804, 450, 864, 179,
    277, 576, 446, 521, 462, 861,
    886, 708, 67, 311, 839, 911,
    273, 874, 8, 607, 151, 444,
    220, 79, 660, 919, 804, 716,
    842, 221, 653, 704, 640, 468,
    // Bin capacity
    1488, 1643, 1664, 1652, 1509, 2140
  };
  const int i6_18_6_11_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    967, 270, 470, 893, 990, 845,
    569, 721, 702, 518, 441, 215,
    465, 211, 749, 343, 799, 56,
    932, 204, 508, 50, 466, 364,
    663, 383, 596, 182, 744, 101,
    677, 461, 314, 208, 892, 70,
    172, 144, 870, 242, 561, 65,
    704, 561, 546, 837, 474, 810,
    790, 244, 414, 115, 375, 861,
    318, 310, 220, 49, 463, 363,
    666, 596, 474, 882, 221, 486,
    337, 319, 554, 505, 330, 209,
    81, 152, 511, 611, 83, 534,
    245, 934, 540, 678, 910, 990,
    646, 514, 123, 637, 818, 98,
    392, 993, 355, 504, 519, 789,
    414, 413, 618, 467, 869, 889,
    503, 165, 282, 256, 572, 945,
    // Bin capacity
    1766, 1406, 1637, 1476, 1948, 1608
  };
  const int i6_18_6_12_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    547, 133, 961, 773, 393, 524,
    918, 394, 802, 849, 734, 346,
    321, 398, 919, 591, 757, 945,
    545, 204, 763, 451, 257, 663,
    38, 584, 847, 207, 108, 614,
    70, 566, 734, 589, 580, 501,
    865, 787, 263, 275, 439, 663,
    514, 773, 156, 404, 438, 722,
    0, 183, 326, 707, 981, 941,
    352, 979, 847, 520, 823, 43,
    675, 326, 91, 439, 97, 529,
    599, 18, 626, 849, 500, 389,
    565, 43, 682, 369, 346, 505,
    539, 570, 212, 470, 530, 974,
    574, 57, 2, 291, 653, 848,
    162, 937, 349, 96, 428, 120,
    533, 521, 947, 668, 321, 875,
    894, 621, 195, 772, 279, 966,
    // Bin capacity
    1627, 1511, 1815, 1740, 1618, 2085
  };
  const int i6_18_6_13_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    698, 116, 707, 599, 961, 715,
    17, 223, 975, 567, 36, 276,
    413, 16, 632, 642, 208, 553,
    104, 850, 544, 749, 824, 556,
    110, 41, 128, 783, 236, 2,
    434, 750, 814, 712, 352, 56,
    716, 713, 902, 132, 344, 435,
    841, 411, 676, 996, 683, 785,
    844, 417, 137, 254, 796, 926,
    197, 194, 695, 925, 422, 472,
    716, 611, 636, 754, 122, 290,
    345, 204, 388, 192, 776, 927,
    704, 32, 374, 559, 416, 368,
    776, 431, 903, 270, 263, 942,
    354, 945, 961, 982, 465, 984,
    932, 595, 461, 844, 326, 202,
    111, 441, 615, 907, 797, 367,
    480, 934, 94, 506, 295, 111,
    // Bin capacity
    1656, 1493, 2005, 2142, 1568, 1689
  };
  const int i6_18_6_14_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    919, 701, 335, 436, 831, 18,
    304, 483, 803, 86, 201, 859,
    533, 217, 125, 417, 900, 728,
    427, 713, 128, 244, 14, 835,
    290, 685, 567, 480, 850, 814,
    481, 718, 466, 238, 887, 761,
    867, 910, 758, 42, 628, 297,
    30, 130, 292, 135, 295, 264,
    548, 592, 356, 464, 363, 679,
    462, 474, 774, 797, 871, 702,
    911, 367, 119, 841, 748, 502,
    904, 789, 779, 540, 682, 745,
    405, 619, 406, 728, 485, 795,
    74, 367, 797, 255, 253, 500,
    323, 813, 928, 789, 860, 229,
    812, 320, 754, 972, 640, 287,
    997, 250, 3, 460, 284, 322,
    135, 544, 372, 6, 470, 407,
    // Bin capacity
    1791, 1842, 1665, 1507, 1950, 1852
  };
  const int i6_18_6_15_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    199, 316, 9, 884, 585, 398,
    28, 472, 28, 273, 879, 850,
    186, 897, 602, 547, 472, 425,
    916, 232, 583, 339, 300, 519,
    45, 908, 877, 960, 758, 383,
    203, 686, 30, 978, 494, 849,
    702, 645, 626, 711, 132, 18,
    700, 615, 68, 304, 263, 264,
    497, 192, 282, 352, 381, 927,
    75, 657, 272, 486, 633, 197,
    958, 272, 791, 102, 939, 828,
    453, 356, 910, 684, 61, 937,
    332, 571, 95, 632, 205, 160,
    712, 177, 809, 30, 877, 825,
    387, 679, 821, 248, 899, 444,
    864, 600, 727, 67, 54, 488,
    546, 523, 705, 892, 428, 6,
    828, 954, 93, 427, 536, 65,
    // Bin capacity
    1655, 1870, 1597, 1709, 1706, 1646
  };
  const int i6_18_6_16_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    509, 500, 892, 344, 481, 246,
    331, 516, 838, 359, 72, 852,
    107, 779, 24, 958, 25, 19,
    90, 767, 739, 727, 429, 488,
    600, 725, 867, 543, 281, 830,
    497, 46, 35, 919, 68, 658,
    268, 876, 410, 410, 182, 482,
    241, 716, 977, 389, 270, 278,
    653, 987, 502, 999, 677, 311,
    759, 323, 727, 416, 11, 540,
    566, 374, 797, 973, 639, 945,
    882, 867, 25, 756, 237, 936,
    76, 392, 580, 419, 844, 824,
    873, 794, 532, 982, 300, 864,
    452, 233, 35, 761, 763, 723,
    436, 863, 550, 434, 49, 591,
    843, 337, 897, 785, 306, 225,
    128, 599, 198, 713, 303, 35,
    // Bin capacity
    1607, 2068, 1861, 2299, 1148, 1904
  };
  const int i6_18_6_17_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    280, 22, 726, 262, 363, 912,
    66, 888, 730, 488, 742, 286,
    763, 594, 748, 93, 479, 459,
    756, 418, 864, 779, 182, 382,
    988, 245, 968, 812, 645, 194,
    186, 629, 109, 420, 917, 563,
    75, 260, 642, 747, 375, 575,
    645, 39, 582, 225, 96, 581,
    993, 823, 972, 743, 213, 852,
    852, 2, 276, 212, 739, 56,
    419, 496, 561, 64, 126, 392,
    622, 303, 725, 395, 987, 967,
    310, 547, 720, 645, 825, 644,
    673, 761, 187, 573, 205, 992,
    600, 380, 945, 884, 554, 554,
    905, 464, 445, 119, 689, 161,
    594, 771, 2, 540, 163, 816,
    671, 488, 964, 775, 426, 796,
    // Bin capacity
    2028, 1586, 2178, 1712, 1702, 1986
  };
  const int i6_18_6_18_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    41, 517, 388, 538, 565, 410,
    254, 528, 855, 339, 422, 399,
    817, 564, 30, 781, 813, 475,
    212, 667, 625, 922, 559, 866,
    575, 774, 725, 401, 733, 655,
    902, 641, 708, 620, 225, 552,
    363, 858, 974, 392, 596, 986,
    682, 411, 516, 498, 451, 937,
    357, 862, 26, 489, 77, 204,
    766, 868, 947, 403, 374, 132,
    990, 669, 123, 113, 164, 380,
    725, 532, 942, 597, 518, 407,
    853, 979, 568, 498, 326, 473,
    190, 674, 654, 335, 641, 143,
    456, 205, 947, 395, 454, 761,
    544, 638, 890, 464, 81, 83,
    591, 219, 24, 214, 47, 90,
    608, 361, 417, 726, 574, 66,
    // Bin capacity
    1953, 2157, 2038, 1716, 1499, 1578
  };
  const int i6_18_6_19_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    130, 37, 490, 965, 160, 4,
    318, 247, 805, 1000, 740, 135,
    380, 241, 147, 389, 327, 729,
    881, 204, 939, 159, 139, 229,
    159, 967, 498, 484, 11, 764,
    887, 599, 328, 653, 13, 54,
    478, 246, 879, 535, 347, 573,
    580, 965, 736, 611, 26, 247,
    343, 832, 179, 10, 573, 304,
    690, 102, 763, 298, 979, 756,
    673, 89, 619, 763, 769, 202,
    761, 892, 713, 884, 195, 808,
    558, 537, 705, 369, 950, 910,
    969, 476, 41, 404, 787, 114,
    633, 684, 732, 455, 408, 379,
    288, 567, 981, 642, 599, 5,
    250, 97, 388, 969, 247, 541,
    607, 142, 412, 230, 767, 597,
    // Bin capacity
    1902, 1572, 2054, 1948, 1595, 1458
  };
  const int i6_18_6_20_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    366, 674, 839, 194, 628, 309,
    268, 563, 493, 626, 771, 830,
    490, 128, 512, 41, 103, 679,
    830, 501, 738, 210, 758, 347,
    656, 423, 187, 591, 220, 253,
    955, 550, 932, 115, 507, 3,
    572, 622, 512, 491, 136, 761,
    578, 815, 539, 638, 92, 962,
    6, 763, 417, 73, 441, 387,
    120, 558, 442, 879, 748, 236,
    646, 203, 450, 254, 629, 918,
    165, 760, 6, 202, 933, 56,
    416, 846, 669, 453, 287, 419,
    346, 715, 719, 202, 373, 750,
    433, 761, 721, 932, 709, 81,
    605, 287, 517, 937, 507, 21,
    362, 479, 521, 677, 699, 710,
    295, 531, 173, 305, 60, 722,
    // Bin capacity
    1622, 2036, 1878, 1564, 1721, 1689
  };
  const int i6_18_6_21_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    204, 521, 439, 465, 621, 248,
    9, 483, 55, 341, 617, 542,
    650, 492, 380, 407, 345, 344,
    711, 362, 757, 26, 354, 619,
    412, 595, 283, 412, 714, 675,
    312, 732, 705, 279, 318, 282,
    383, 759, 787, 476, 303, 637,
    776, 374, 703, 303, 267, 604,
    93, 781, 223, 872, 837, 859,
    34, 441, 259, 667, 343, 673,
    727, 809, 578, 302, 79, 786,
    229, 114, 841, 568, 521, 289,
    779, 532, 945, 745, 153, 622,
    8, 825, 767, 707, 824, 935,
    346, 153, 624, 50, 903, 522,
    354, 541, 787, 343, 136, 69,
    724, 359, 490, 670, 737, 439,
    549, 280, 885, 269, 344, 143,
    // Bin capacity
    1473, 1846, 2120, 1594, 1698, 1874
  };
  const int i6_18_6_22_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    155, 379, 911, 715, 2, 394,
    519, 571, 767, 278, 399, 58,
    734, 721, 352, 118, 354, 986,
    475, 225, 514, 884, 118, 410,
    89, 868, 240, 722, 822, 255,
    930, 283, 612, 138, 577, 471,
    401, 92, 407, 220, 880, 828,
    971, 406, 925, 537, 144, 848,
    196, 996, 498, 770, 770, 292,
    469, 405, 353, 715, 712, 28,
    889, 162, 323, 416, 330, 757,
    175, 124, 909, 938, 298, 836,
    257, 956, 576, 371, 153, 379,
    558, 533, 796, 375, 798, 732,
    619, 461, 280, 274, 390, 517,
    147, 189, 315, 951, 657, 29,
    432, 710, 875, 778, 572, 26,
    134, 503, 440, 45, 336, 840,
    // Bin capacity
    1658, 1746, 2053, 1880, 1691, 1767
  };
  const int i6_18_6_23_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    926, 656, 916, 578, 258, 464,
    159, 237, 595, 171, 431, 612,
    556, 870, 294, 124, 467, 685,
    387, 565, 446, 788, 982, 489,
    674, 37, 711, 478, 239, 299,
    780, 662, 958, 741, 592, 848,
    520, 568, 409, 722, 421, 751,
    346, 350, 169, 699, 969, 121,
    432, 687, 927, 228, 877, 179,
    277, 115, 574, 697, 664, 279,
    352, 147, 740, 208, 728, 1000,
    824, 289, 734, 978, 68, 448,
    715, 83, 191, 597, 463, 768,
    142, 217, 132, 345, 40, 152,
    98, 707, 187, 993, 527, 268,
    31, 644, 224, 276, 834, 203,
    803, 360, 540, 670, 685, 986,
    480, 374, 768, 336, 37, 887,
    // Bin capacity
    1743, 1552, 1951, 1974, 1903, 1935
  };
  const int i6_18_6_24_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    753, 710, 479, 126, 21, 22,
    409, 658, 49, 177, 914, 764,
    252, 616, 531, 124, 107, 212,
    276, 512, 564, 831, 295, 0,
    14, 47, 596, 77, 239, 819,
    163, 967, 500, 951, 156, 41,
    673, 65, 317, 514, 120, 330,
    399, 705, 64, 350, 260, 365,
    756, 750, 686, 699, 383, 795,
    908, 161, 793, 526, 27, 557,
    309, 75, 821, 874, 977, 40,
    753, 722, 487, 160, 993, 404,
    149, 847, 834, 683, 211, 980,
    311, 314, 50, 232, 212, 119,
    336, 702, 517, 339, 9, 752,
    912, 566, 62, 641, 171, 569,
    194, 1000, 556, 321, 163, 171,
    533, 763, 985, 416, 205, 117,
    // Bin capacity
    1674, 2104, 1838, 1662, 1130, 1459
  };
  const int i6_18_6_25_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    344, 611, 689, 379, 787, 313,
    815, 493, 982, 555, 616, 161,
    579, 920, 23, 172, 746, 688,
    202, 305, 597, 948, 573, 850,
    186, 41, 300, 903, 393, 907,
    399, 643, 357, 267, 508, 54,
    71, 179, 211, 717, 680, 242,
    304, 333, 194, 365, 936, 124,
    738, 466, 511, 828, 772, 665,
    588, 594, 844, 862, 509, 674,
    874, 526, 25, 990, 311, 75,
    883, 634, 288, 210, 681, 750,
    741, 495, 898, 731, 387, 602,
    897, 499, 629, 636, 443, 997,
    156, 767, 994, 799, 359, 299,
    336, 260, 896, 180, 407, 249,
    138, 897, 763, 444, 16, 997,
    202, 734, 383, 547, 412, 573,
    // Bin capacity
    1762, 1958, 1997, 2195, 1987, 1921
  };
  const int i6_18_6_26_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    923, 542, 337, 61, 934, 611,
    917, 586, 442, 586, 240, 735,
    450, 248, 19, 97, 614, 903,
    250, 948, 494, 87, 930, 787,
    973, 974, 646, 592, 116, 623,
    204, 133, 652, 715, 974, 670,
    958, 236, 441, 179, 330, 413,
    466, 682, 806, 775, 634, 1000,
    84, 639, 895, 361, 511, 651,
    73, 643, 77, 693, 27, 574,
    320, 840, 548, 290, 760, 539,
    580, 692, 462, 237, 181, 95,
    943, 562, 981, 135, 519, 490,
    98, 522, 436, 694, 16, 49,
    629, 905, 352, 197, 672, 769,
    337, 692, 712, 36, 886, 819,
    567, 200, 551, 372, 18, 954,
    847, 870, 92, 128, 530, 402,
    // Bin capacity
    2020, 2292, 1879, 1310, 1868, 2328
  };
  const int i6_18_6_27_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    325, 748, 731, 97, 862, 865,
    603, 788, 529, 985, 341, 539,
    472, 607, 204, 860, 36, 587,
    833, 976, 766, 105, 641, 850,
    939, 927, 558, 848, 544, 281,
    835, 412, 764, 61, 428, 946,
    446, 8, 275, 15, 349, 38,
    722, 149, 345, 828, 863, 868,
    121, 540, 141, 990, 945, 36,
    930, 624, 249, 333, 926, 676,
    179, 520, 903, 245, 843, 762,
    631, 966, 990, 391, 673, 370,
    375, 426, 326, 999, 90, 558,
    392, 694, 700, 4, 932, 698,
    928, 890, 48, 702, 346, 837,
    788, 682, 902, 280, 185, 19,
    543, 699, 458, 490, 794, 613,
    703, 46, 695, 393, 944, 903,
    // Bin capacity
    2279, 2266, 2029, 1826, 2274, 2212
  };
  const int i6_18_6_28_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    3, 122, 555, 21, 90, 521,
    479, 820, 108, 436, 338, 963,
    214, 742, 511, 232, 238, 448,
    364, 855, 381, 756, 996, 740,
    93, 68, 64, 358, 671, 855,
    39, 182, 990, 14, 52, 643,
    38, 573, 971, 416, 161, 362,
    293, 844, 824, 208, 833, 748,
    722, 915, 755, 466, 853, 886,
    276, 436, 50, 69, 610, 306,
    751, 833, 136, 257, 91, 92,
    360, 97, 836, 259, 838, 765,
    18, 939, 198, 348, 17, 76,
    964, 340, 303, 406, 907, 788,
    800, 426, 719, 368, 473, 856,
    345, 264, 852, 340, 873, 270,
    293, 818, 172, 907, 957, 988,
    105, 702, 153, 409, 554, 582,
    // Bin capacity
    1314, 2129, 1830, 1338, 2038, 2323
  };
  const int i6_18_6_29_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    406, 625, 170, 752, 902, 73,
    716, 947, 831, 82, 196, 637,
    526, 445, 845, 812, 187, 441,
    727, 344, 962, 896, 948, 895,
    427, 157, 923, 77, 786, 970,
    197, 407, 24, 95, 533, 924,
    430, 154, 983, 609, 545, 582,
    962, 537, 661, 602, 707, 317,
    217, 886, 148, 650, 583, 571,
    48, 783, 40, 525, 298, 909,
    152, 463, 718, 969, 937, 141,
    57, 573, 799, 431, 879, 937,
    44, 730, 537, 868, 758, 429,
    776, 514, 336, 198, 993, 542,
    644, 865, 249, 486, 460, 815,
    455, 760, 738, 468, 224, 760,
    271, 41, 899, 860, 20, 236,
    446, 773, 422, 122, 996, 244,
    // Bin capacity
    1613, 2151, 2212, 2043, 2355, 2241
  };
  const int i6_18_6_30_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    203, 642, 35, 272, 1000, 991,
    100, 434, 568, 799, 291, 255,
    703, 629, 675, 19, 630, 17,
    748, 32, 61, 874, 713, 797,
    71, 672, 499, 21, 698, 460,
    132, 239, 383, 748, 380, 948,
    568, 189, 717, 707, 616, 541,
    727, 167, 260, 246, 401, 940,
    987, 526, 200, 470, 590, 648,
    637, 864, 805, 1000, 716, 3,
    571, 534, 933, 703, 699, 812,
    504, 20, 656, 379, 597, 73,
    303, 226, 314, 197, 689, 452,
    61, 813, 233, 309, 917, 199,
    302, 358, 133, 620, 888, 625,
    371, 638, 735, 703, 607, 702,
    1, 299, 183, 32, 600, 130,
    212, 459, 706, 778, 99, 978,
    // Bin capacity
    1561, 1678, 1755, 1924, 2412, 2074
  };
  const int i6_18_6_31_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    762, 828, 527, 495, 47, 708,
    358, 120, 976, 876, 747, 729,
    980, 309, 0, 566, 729, 229,
    564, 546, 112, 638, 678, 474,
    119, 850, 678, 774, 717, 456,
    411, 308, 516, 88, 998, 764,
    201, 242, 536, 629, 997, 865,
    334, 258, 588, 12, 594, 151,
    265, 167, 502, 979, 172, 68,
    893, 34, 17, 54, 718, 723,
    622, 450, 683, 423, 770, 708,
    887, 378, 47, 361, 517, 630,
    139, 779, 178, 455, 843, 579,
    636, 986, 148, 993, 801, 54,
    291, 126, 726, 719, 65, 231,
    637, 67, 657, 39, 604, 331,
    936, 721, 636, 698, 222, 456,
    727, 778, 347, 886, 403, 577,
    // Bin capacity
    2132, 1736, 1720, 2115, 2320, 1907
  };
  const int i6_18_6_32_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    212, 54, 247, 892, 237, 757,
    75, 682, 146, 661, 915, 388,
    806, 414, 723, 191, 87, 674,
    833, 554, 365, 197, 720, 868,
    298, 479, 522, 327, 73, 561,
    672, 0, 299, 737, 242, 103,
    804, 185, 831, 582, 869, 452,
    120, 343, 513, 821, 967, 110,
    302, 773, 653, 139, 331, 398,
    478, 545, 252, 260, 988, 884,
    735, 997, 843, 897, 867, 733,
    565, 228, 538, 143, 308, 315,
    806, 194, 55, 889, 354, 419,
    913, 928, 230, 944, 994, 758,
    815, 150, 457, 765, 827, 108,
    259, 179, 812, 740, 847, 829,
    720, 557, 436, 874, 735, 282,
    55, 924, 43, 556, 441, 196,
    // Bin capacity
    2083, 1801, 1753, 2336, 2377, 1944
  };
  const int i6_18_6_33_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    431, 309, 592, 539, 454, 633,
    865, 103, 732, 9, 601, 858,
    84, 174, 455, 654, 448, 254,
    638, 45, 176, 794, 53, 169,
    348, 801, 525, 296, 767, 241,
    397, 773, 850, 708, 335, 202,
    722, 768, 891, 697, 432, 246,
    775, 957, 156, 557, 705, 675,
    582, 691, 225, 488, 474, 774,
    294, 347, 39, 286, 827, 260,
    946, 428, 232, 186, 720, 913,
    497, 724, 97, 975, 15, 800,
    522, 710, 779, 567, 570, 670,
    699, 105, 618, 135, 209, 236,
    790, 799, 113, 452, 708, 162,
    545, 154, 711, 625, 823, 432,
    342, 839, 328, 136, 312, 261,
    421, 29, 265, 930, 405, 108,
    // Bin capacity
    2195, 1941, 1726, 2003, 1964, 1750
  };
  const int i6_18_6_34_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    409, 156, 249, 506, 157, 900,
    579, 240, 358, 441, 509, 355,
    405, 344, 814, 341, 90, 400,
    56, 381, 72, 207, 865, 306,
    353, 386, 232, 641, 333, 56,
    384, 427, 861, 369, 492, 7,
    852, 763, 144, 6, 335, 777,
    395, 748, 322, 304, 508, 976,
    242, 144, 88, 415, 233, 425,
    270, 989, 462, 258, 712, 785,
    861, 897, 87, 645, 501, 259,
    645, 201, 511, 383, 132, 662,
    554, 249, 188, 830, 372, 712,
    409, 203, 352, 71, 281, 980,
    623, 463, 983, 850, 855, 698,
    902, 43, 330, 939, 183, 353,
    473, 1, 468, 286, 135, 748,
    774, 760, 533, 653, 1, 86,
    // Bin capacity
    2052, 1652, 1576, 1820, 1495, 2119
  };
  const int i6_18_6_35_25[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    229, 821, 55, 405, 385, 812,
    924, 692, 218, 939, 197, 243,
    12, 804, 608, 993, 294, 756,
    112, 240, 989, 992, 455, 322,
    843, 305, 163, 333, 803, 324,
    248, 924, 874, 316, 457, 858,
    638, 789, 832, 680, 101, 83,
    251, 359, 397, 370, 816, 273,
    188, 568, 772, 337, 901, 886,
    735, 812, 218, 325, 987, 436,
    303, 314, 701, 801, 260, 430,
    6, 356, 15, 905, 773, 675,
    904, 857, 967, 682, 228, 800,
    198, 787, 324, 330, 696, 507,
    204, 234, 90, 670, 389, 352,
    471, 548, 266, 865, 110, 365,
    960, 758, 378, 530, 928, 716,
    532, 705, 928, 332, 617, 868,
    // Bin capacity
    1746, 2447, 1979, 2432, 2115, 2184
  };
  const int i6_18_6_0_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    266, 348, 832, 775, 359, 240,
    73, 6, 904, 92, 598, 619,
    152, 655, 906, 906, 231, 759,
    927, 646, 135, 79, 288, 512,
    219, 225, 480, 10, 239, 203,
    570, 476, 969, 959, 568, 351,
    964, 895, 234, 521, 176, 386,
    911, 848, 135, 518, 990, 580,
    260, 862, 438, 522, 941, 956,
    254, 653, 518, 710, 56, 781,
    237, 953, 899, 894, 569, 24,
    802, 671, 596, 632, 757, 273,
    528, 348, 572, 577, 189, 621,
    181, 665, 201, 732, 793, 309,
    685, 70, 547, 788, 527, 432,
    809, 596, 242, 584, 640, 424,
    446, 649, 99, 182, 246, 735,
    380, 720, 755, 582, 49, 774,
    // Bin capacity
    1444, 1715, 1577, 1678, 1370, 1497
  };
  const int i6_18_6_1_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    529, 677, 157, 571, 39, 691,
    710, 960, 490, 111, 436, 207,
    621, 543, 613, 887, 816, 912,
    331, 19, 329, 476, 13, 747,
    764, 752, 351, 403, 5, 401,
    393, 681, 727, 754, 281, 251,
    292, 92, 301, 647, 887, 632,
    487, 730, 860, 73, 568, 184,
    193, 714, 881, 535, 574, 810,
    64, 946, 152, 696, 892, 44,
    930, 761, 652, 611, 416, 586,
    69, 381, 492, 62, 821, 322,
    1000, 335, 798, 812, 687, 978,
    275, 678, 938, 309, 611, 444,
    261, 995, 100, 611, 251, 724,
    423, 788, 375, 911, 373, 895,
    117, 507, 897, 958, 697, 136,
    522, 742, 6, 604, 88, 586,
    // Bin capacity
    1344, 1903, 1536, 1689, 1424, 1608
  };
  const int i6_18_6_2_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    746, 389, 52, 731, 367, 359,
    7, 208, 370, 525, 930, 943,
    262, 350, 255, 447, 68, 126,
    864, 416, 454, 680, 667, 366,
    724, 213, 74, 995, 707, 877,
    632, 444, 719, 759, 340, 800,
    600, 917, 237, 408, 179, 340,
    639, 110, 49, 383, 717, 551,
    675, 724, 736, 889, 91, 729,
    979, 319, 209, 478, 62, 130,
    528, 555, 614, 65, 333, 253,
    529, 470, 842, 69, 105, 377,
    774, 897, 760, 541, 381, 202,
    685, 867, 852, 756, 808, 30,
    579, 546, 718, 281, 335, 40,
    983, 559, 431, 372, 432, 645,
    857, 106, 322, 242, 954, 659,
    755, 373, 209, 347, 152, 542,
    // Bin capacity
    2010, 1439, 1344, 1525, 1297, 1355
  };
  const int i6_18_6_3_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    820, 736, 26, 876, 924, 194,
    207, 739, 806, 702, 867, 525,
    693, 425, 76, 456, 710, 129,
    186, 107, 634, 342, 311, 123,
    122, 95, 60, 452, 479, 831,
    409, 195, 337, 563, 318, 861,
    178, 176, 913, 453, 822, 393,
    993, 905, 310, 349, 202, 82,
    11, 821, 54, 404, 767, 940,
    317, 687, 117, 241, 662, 200,
    587, 925, 923, 42, 346, 595,
    930, 805, 415, 750, 945, 465,
    826, 479, 802, 87, 519, 260,
    457, 759, 865, 364, 763, 400,
    780, 325, 845, 908, 836, 108,
    845, 913, 20, 866, 777, 699,
    882, 467, 197, 295, 179, 356,
    308, 838, 468, 611, 295, 981,
    // Bin capacity
    1640, 1785, 1351, 1504, 1841, 1398
  };
  const int i6_18_6_4_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    992, 246, 51, 946, 858, 878,
    383, 5, 260, 978, 321, 335,
    269, 604, 594, 687, 219, 193,
    238, 450, 370, 78, 675, 529,
    112, 797, 576, 841, 298, 7,
    821, 421, 487, 298, 764, 972,
    33, 769, 752, 741, 498, 807,
    669, 13, 928, 228, 947, 439,
    116, 262, 392, 266, 640, 317,
    266, 608, 775, 867, 111, 6,
    868, 195, 463, 169, 398, 181,
    63, 415, 428, 257, 430, 1000,
    572, 867, 908, 711, 394, 668,
    64, 130, 398, 995, 416, 293,
    339, 681, 598, 75, 652, 789,
    277, 13, 968, 335, 7, 224,
    783, 333, 932, 472, 290, 521,
    461, 842, 690, 400, 370, 522,
    // Bin capacity
    1270, 1327, 1833, 1620, 1437, 1505
  };
  const int i6_18_6_5_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    941, 583, 517, 660, 668, 715,
    392, 738, 464, 758, 134, 927,
    410, 734, 88, 261, 126, 199,
    761, 533, 18, 36, 631, 171,
    178, 829, 557, 472, 212, 814,
    683, 295, 606, 606, 183, 707,
    532, 282, 22, 604, 945, 94,
    573, 691, 750, 412, 104, 463,
    729, 21, 50, 685, 974, 805,
    337, 276, 787, 64, 658, 606,
    871, 959, 926, 77, 784, 801,
    503, 694, 335, 887, 951, 904,
    103, 386, 432, 94, 939, 577,
    45, 297, 976, 188, 167, 498,
    583, 337, 171, 481, 966, 3,
    864, 738, 768, 631, 985, 827,
    1000, 561, 500, 880, 36, 874,
    948, 656, 504, 296, 893, 164,
    // Bin capacity
    1830, 1682, 1483, 1417, 1813, 1777
  };
  const int i6_18_6_6_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    891, 977, 113, 682, 645, 85,
    688, 688, 639, 551, 742, 107,
    576, 687, 85, 818, 51, 665,
    528, 433, 779, 789, 328, 118,
    704, 390, 7, 498, 588, 1000,
    814, 223, 664, 202, 715, 874,
    21, 708, 475, 668, 126, 303,
    483, 825, 668, 537, 712, 660,
    259, 392, 634, 421, 404, 958,
    478, 984, 348, 283, 765, 459,
    274, 950, 225, 857, 621, 553,
    301, 634, 441, 842, 463, 863,
    113, 863, 224, 263, 917, 633,
    781, 272, 380, 982, 229, 994,
    609, 347, 173, 865, 353, 803,
    424, 263, 178, 252, 581, 615,
    168, 185, 240, 213, 807, 908,
    145, 672, 757, 797, 402, 240,
    // Bin capacity
    1459, 1854, 1242, 1859, 1670, 1915
  };
  const int i6_18_6_7_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    740, 308, 63, 634, 959, 893,
    773, 318, 379, 671, 712, 669,
    988, 316, 77, 144, 156, 848,
    548, 430, 549, 434, 73, 191,
    216, 347, 186, 850, 65, 557,
    696, 871, 454, 891, 846, 753,
    219, 778, 52, 915, 793, 424,
    580, 380, 676, 395, 450, 913,
    355, 386, 152, 713, 295, 814,
    381, 871, 98, 951, 195, 891,
    708, 233, 523, 36, 506, 791,
    788, 441, 177, 411, 342, 813,
    246, 257, 531, 292, 504, 539,
    628, 304, 374, 928, 237, 615,
    414, 918, 667, 730, 925, 838,
    991, 285, 485, 175, 961, 285,
    928, 5, 824, 540, 556, 200,
    677, 712, 432, 896, 333, 887,
    // Bin capacity
    1940, 1456, 1195, 1892, 1589, 2126
  };
  const int i6_18_6_8_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    836, 560, 232, 582, 185, 286,
    344, 887, 865, 449, 320, 792,
    784, 83, 713, 746, 497, 400,
    650, 941, 998, 846, 717, 979,
    512, 660, 834, 581, 171, 670,
    856, 309, 13, 422, 717, 98,
    389, 517, 116, 875, 240, 43,
    676, 832, 329, 600, 45, 590,
    54, 692, 301, 26, 501, 839,
    222, 430, 919, 166, 448, 10,
    330, 565, 522, 678, 88, 825,
    978, 434, 968, 38, 508, 58,
    566, 904, 691, 335, 515, 629,
    321, 851, 988, 625, 635, 143,
    411, 692, 328, 382, 941, 861,
    256, 290, 926, 118, 674, 381,
    189, 729, 747, 941, 484, 894,
    841, 783, 529, 147, 893, 973,
    // Bin capacity
    1659, 2009, 1984, 1541, 1545, 1705
  };
  const int i6_18_6_9_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    905, 206, 404, 303, 740, 983,
    402, 779, 2, 610, 63, 156,
    642, 862, 108, 852, 799, 674,
    873, 364, 252, 733, 520, 61,
    751, 632, 717, 433, 862, 418,
    676, 391, 968, 916, 156, 144,
    990, 499, 51, 60, 746, 705,
    7, 89, 651, 147, 503, 582,
    240, 358, 506, 109, 30, 648,
    63, 795, 242, 545, 272, 526,
    316, 396, 950, 196, 936, 286,
    135, 582, 560, 232, 458, 43,
    871, 958, 121, 947, 935, 908,
    805, 499, 860, 617, 863, 325,
    783, 98, 202, 726, 433, 511,
    908, 705, 862, 330, 8, 769,
    320, 536, 768, 194, 671, 882,
    36, 833, 90, 725, 298, 762,
    // Bin capacity
    1767, 1741, 1511, 1576, 1689, 1705
  };
  const int i6_18_6_10_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    796, 460, 174, 263, 54, 452,
    258, 252, 761, 411, 484, 357,
    80, 892, 915, 634, 491, 283,
    957, 536, 432, 244, 160, 32,
    352, 834, 594, 595, 251, 341,
    969, 320, 195, 933, 445, 887,
    380, 110, 56, 442, 16, 564,
    482, 605, 626, 473, 763, 703,
    919, 205, 887, 73, 805, 980,
    409, 894, 844, 682, 50, 647,
    701, 812, 240, 421, 213, 45,
    270, 239, 168, 631, 515, 487,
    472, 673, 196, 940, 177, 370,
    448, 669, 227, 653, 86, 502,
    521, 579, 40, 509, 599, 273,
    39, 373, 245, 943, 685, 353,
    268, 907, 488, 681, 563, 974,
    823, 675, 486, 437, 177, 704,
    // Bin capacity
    1677, 1840, 1389, 1827, 1198, 1642
  };
  const int i6_18_6_11_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    537, 244, 894, 207, 222, 26,
    207, 827, 628, 550, 763, 884,
    164, 648, 443, 869, 341, 963,
    992, 36, 948, 185, 385, 491,
    391, 257, 580, 735, 273, 412,
    136, 111, 221, 605, 963, 723,
    989, 685, 63, 126, 626, 207,
    943, 863, 589, 791, 892, 684,
    218, 495, 928, 640, 40, 514,
    900, 751, 499, 57, 292, 16,
    798, 719, 440, 350, 769, 285,
    193, 342, 105, 25, 190, 451,
    887, 800, 77, 103, 180, 479,
    295, 740, 141, 244, 12, 975,
    748, 536, 942, 704, 351, 428,
    660, 677, 967, 355, 769, 720,
    334, 381, 397, 878, 200, 145,
    647, 660, 414, 732, 104, 560,
    // Bin capacity
    1858, 1808, 1717, 1509, 1364, 1659
  };
  const int i6_18_6_12_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    672, 92, 17, 461, 985, 955,
    578, 840, 381, 870, 686, 261,
    961, 523, 864, 94, 261, 170,
    793, 218, 445, 45, 119, 54,
    740, 418, 843, 783, 371, 400,
    714, 841, 9, 962, 847, 224,
    941, 819, 26, 517, 412, 688,
    679, 372, 727, 45, 203, 724,
    334, 762, 366, 885, 322, 76,
    99, 527, 719, 895, 333, 56,
    785, 225, 405, 715, 429, 703,
    478, 724, 655, 604, 298, 144,
    183, 322, 90, 427, 442, 535,
    115, 517, 680, 842, 554, 847,
    953, 656, 937, 537, 403, 48,
    153, 718, 366, 434, 746, 951,
    449, 442, 109, 791, 22, 915,
    111, 722, 936, 616, 995, 906,
    // Bin capacity
    1818, 1818, 1601, 1965, 1574, 1616
  };
  const int i6_18_6_13_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    364, 549, 526, 809, 964, 992,
    218, 992, 399, 2, 881, 812,
    111, 898, 787, 600, 702, 912,
    45, 815, 621, 430, 529, 452,
    884, 395, 979, 190, 844, 246,
    797, 328, 512, 27, 428, 883,
    550, 291, 57, 646, 304, 348,
    840, 482, 570, 793, 986, 661,
    229, 981, 669, 874, 270, 369,
    918, 270, 243, 107, 995, 757,
    472, 624, 543, 959, 952, 582,
    94, 724, 92, 936, 109, 920,
    216, 888, 752, 464, 706, 428,
    125, 372, 297, 701, 766, 998,
    596, 340, 11, 938, 85, 655,
    875, 950, 465, 200, 124, 358,
    249, 290, 445, 247, 561, 563,
    89, 882, 15, 763, 795, 286,
    // Bin capacity
    1445, 2086, 1504, 1825, 2072, 2114
  };
  const int i6_18_6_14_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    651, 633, 976, 728, 756, 895,
    857, 846, 331, 274, 723, 89,
    271, 116, 264, 84, 988, 977,
    816, 571, 49, 307, 113, 482,
    111, 422, 591, 545, 900, 142,
    262, 168, 58, 766, 251, 534,
    517, 967, 963, 697, 124, 23,
    549, 340, 625, 470, 976, 835,
    577, 518, 639, 180, 183, 340,
    196, 392, 786, 681, 769, 560,
    614, 33, 797, 466, 92, 386,
    617, 1000, 73, 768, 911, 708,
    7, 11, 11, 812, 567, 82,
    11, 96, 229, 746, 596, 676,
    502, 531, 754, 302, 761, 948,
    698, 618, 944, 774, 342, 207,
    17, 743, 901, 546, 175, 643,
    136, 625, 399, 709, 176, 558,
    // Bin capacity
    1408, 1640, 1785, 1873, 1787, 1727
  };
  const int i6_18_6_15_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    628, 991, 346, 747, 669, 662,
    62, 540, 727, 421, 299, 504,
    180, 289, 480, 261, 309, 736,
    318, 880, 840, 806, 691, 563,
    506, 294, 661, 79, 831, 481,
    741, 505, 820, 405, 512, 536,
    464, 352, 941, 47, 430, 652,
    41, 771, 195, 253, 362, 198,
    221, 832, 717, 93, 153, 911,
    450, 301, 122, 313, 990, 517,
    397, 344, 827, 316, 326, 719,
    923, 272, 502, 444, 276, 244,
    498, 472, 314, 6, 816, 360,
    96, 837, 790, 79, 158, 93,
    249, 4, 770, 966, 629, 404,
    57, 394, 476, 877, 637, 171,
    719, 882, 266, 427, 512, 338,
    9, 221, 190, 231, 844, 724,
    // Bin capacity
    1258, 1760, 1914, 1298, 1811, 1690
  };
  const int i6_18_6_16_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    930, 162, 491, 193, 223, 800,
    768, 586, 962, 508, 691, 348,
    565, 552, 313, 342, 979, 544,
    279, 172, 266, 380, 177, 533,
    109, 33, 86, 541, 733, 320,
    340, 531, 188, 931, 431, 528,
    126, 774, 797, 224, 771, 178,
    736, 593, 320, 548, 59, 560,
    350, 323, 252, 293, 222, 377,
    2, 588, 184, 266, 282, 438,
    651, 481, 403, 844, 72, 271,
    858, 468, 439, 856, 816, 804,
    738, 797, 528, 50, 243, 925,
    11, 235, 930, 688, 389, 277,
    606, 369, 332, 788, 979, 447,
    524, 605, 772, 398, 484, 898,
    343, 266, 398, 900, 481, 512,
    791, 248, 911, 205, 731, 950,
    // Bin capacity
    1688, 1505, 1658, 1732, 1695, 1878
  };
  const int i6_18_6_17_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    635, 130, 665, 614, 782, 380,
    921, 268, 20, 740, 525, 531,
    844, 264, 108, 114, 879, 828,
    213, 852, 737, 84, 712, 697,
    862, 260, 690, 713, 173, 19,
    674, 885, 771, 655, 152, 794,
    771, 529, 686, 332, 746, 883,
    1, 684, 361, 269, 346, 285,
    583, 488, 189, 290, 343, 477,
    11, 943, 603, 639, 402, 477,
    633, 337, 422, 856, 3, 166,
    388, 26, 804, 992, 182, 928,
    758, 408, 598, 166, 678, 478,
    411, 877, 11, 411, 30, 997,
    661, 948, 284, 533, 395, 248,
    860, 514, 212, 932, 801, 360,
    607, 200, 176, 994, 302, 63,
    215, 232, 425, 368, 29, 978,
    // Bin capacity
    1960, 1725, 1514, 1892, 1459, 1870
  };
  const int i6_18_6_18_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    977, 655, 652, 361, 141, 801,
    609, 222, 512, 778, 841, 632,
    532, 897, 269, 731, 173, 727,
    102, 616, 702, 886, 748, 541,
    512, 918, 966, 583, 15, 41,
    892, 339, 75, 633, 547, 486,
    521, 534, 466, 315, 201, 301,
    169, 403, 965, 330, 74, 894,
    851, 623, 899, 694, 933, 294,
    377, 443, 932, 207, 59, 127,
    50, 484, 218, 158, 199, 874,
    236, 21, 838, 909, 950, 37,
    48, 286, 20, 85, 650, 952,
    982, 378, 282, 570, 817, 114,
    244, 473, 231, 158, 134, 535,
    89, 566, 834, 334, 787, 619,
    759, 289, 134, 449, 763, 35,
    64, 620, 549, 399, 44, 626,
    // Bin capacity
    1577, 1725, 1877, 1688, 1589, 1699
  };
  const int i6_18_6_19_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    203, 193, 146, 967, 533, 723,
    314, 800, 154, 681, 653, 48,
    898, 101, 120, 485, 421, 381,
    28, 792, 865, 581, 303, 728,
    218, 879, 499, 998, 92, 16,
    941, 965, 56, 285, 783, 168,
    388, 550, 82, 538, 843, 310,
    903, 451, 880, 745, 780, 802,
    588, 856, 478, 169, 156, 86,
    148, 178, 148, 283, 889, 283,
    728, 456, 668, 503, 288, 592,
    386, 87, 869, 712, 604, 879,
    286, 641, 100, 854, 155, 688,
    469, 927, 619, 380, 727, 549,
    637, 347, 819, 975, 491, 358,
    112, 353, 794, 568, 837, 578,
    514, 424, 51, 234, 621, 325,
    643, 635, 373, 842, 505, 710,
    // Bin capacity
    1667, 1911, 1532, 2142, 1921, 1632
  };
  const int i6_18_6_20_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    26, 984, 175, 937, 341, 524,
    107, 277, 461, 541, 249, 234,
    585, 943, 64, 521, 262, 982,
    335, 376, 844, 55, 539, 393,
    794, 838, 38, 119, 771, 325,
    296, 994, 75, 851, 940, 672,
    641, 186, 235, 571, 761, 112,
    906, 67, 304, 862, 525, 95,
    863, 293, 315, 240, 980, 90,
    462, 110, 235, 33, 149, 893,
    959, 870, 579, 564, 667, 365,
    560, 459, 884, 908, 327, 906,
    5, 439, 744, 968, 162, 296,
    808, 370, 246, 355, 892, 372,
    124, 952, 56, 726, 290, 18,
    727, 466, 246, 193, 550, 748,
    610, 841, 495, 919, 69, 413,
    143, 79, 690, 580, 536, 375,
    // Bin capacity
    1791, 1909, 1338, 1989, 1802, 1563
  };
  const int i6_18_6_21_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    225, 943, 574, 989, 856, 374,
    418, 578, 862, 149, 562, 131,
    156, 461, 959, 230, 190, 345,
    755, 328, 844, 868, 9, 858,
    122, 139, 901, 489, 32, 850,
    814, 960, 504, 445, 788, 697,
    566, 283, 92, 150, 892, 864,
    673, 5, 515, 367, 435, 11,
    552, 16, 527, 576, 196, 629,
    557, 638, 471, 118, 8, 132,
    728, 691, 801, 126, 831, 713,
    747, 488, 55, 313, 109, 811,
    97, 496, 485, 485, 641, 754,
    830, 94, 329, 752, 874, 893,
    534, 459, 317, 532, 386, 354,
    358, 776, 959, 198, 535, 927,
    861, 59, 214, 829, 253, 138,
    55, 653, 622, 156, 540, 30,
    // Bin capacity
    1825, 1627, 2023, 1568, 1641, 1919
  };
  const int i6_18_6_22_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    463, 684, 420, 569, 952, 471,
    350, 59, 718, 113, 296, 475,
    955, 748, 850, 842, 465, 157,
    724, 140, 427, 888, 136, 555,
    360, 354, 244, 586, 924, 567,
    55, 437, 988, 176, 248, 634,
    652, 521, 936, 710, 590, 271,
    979, 707, 177, 751, 139, 874,
    630, 687, 193, 245, 326, 97,
    560, 845, 977, 687, 543, 547,
    209, 802, 797, 322, 590, 670,
    902, 990, 360, 399, 691, 821,
    540, 123, 749, 458, 415, 372,
    38, 840, 888, 942, 338, 890,
    987, 244, 542, 910, 833, 581,
    173, 542, 197, 285, 863, 980,
    549, 456, 66, 747, 545, 761,
    211, 916, 669, 594, 426, 836,
    // Bin capacity
    1899, 2053, 2074, 2079, 1896, 2147
  };
  const int i6_18_6_23_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    727, 248, 679, 764, 688, 811,
    206, 429, 199, 481, 722, 354,
    980, 999, 692, 882, 408, 117,
    698, 397, 617, 933, 380, 608,
    103, 649, 869, 697, 464, 186,
    473, 273, 207, 465, 570, 538,
    756, 26, 48, 102, 551, 485,
    653, 497, 922, 933, 394, 124,
    529, 164, 486, 827, 518, 676,
    617, 872, 173, 653, 419, 677,
    769, 432, 303, 269, 913, 802,
    826, 632, 151, 278, 385, 227,
    19, 714, 581, 606, 977, 636,
    78, 760, 338, 797, 862, 576,
    615, 625, 784, 866, 507, 274,
    503, 667, 350, 896, 860, 298,
    564, 393, 255, 292, 668, 407,
    893, 288, 561, 637, 957, 1,
    // Bin capacity
    2052, 1859, 1685, 2333, 2305, 1599
  };
  const int i6_18_6_24_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    773, 482, 70, 323, 844, 379,
    461, 906, 719, 852, 804, 780,
    663, 848, 730, 944, 602, 804,
    856, 930, 591, 934, 539, 758,
    718, 208, 381, 135, 692, 595,
    203, 156, 318, 754, 155, 148,
    476, 408, 184, 423, 154, 384,
    801, 114, 88, 189, 886, 33,
    717, 815, 177, 823, 395, 347,
    996, 736, 471, 118, 757, 470,
    982, 160, 286, 808, 467, 470,
    805, 817, 512, 534, 189, 645,
    231, 279, 74, 222, 768, 708,
    444, 852, 17, 78, 111, 255,
    773, 222, 947, 795, 88, 639,
    363, 76, 958, 344, 703, 332,
    314, 536, 124, 428, 705, 966,
    357, 163, 63, 69, 349, 605,
    // Bin capacity
    2260, 1800, 1387, 1814, 1903, 1926
  };
  const int i6_18_6_25_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    131, 132, 694, 209, 380, 851,
    415, 224, 342, 829, 127, 884,
    398, 752, 501, 509, 410, 165,
    860, 196, 506, 715, 725, 445,
    615, 193, 727, 327, 417, 243,
    310, 659, 484, 944, 802, 183,
    147, 644, 500, 910, 284, 180,
    666, 104, 633, 77, 729, 654,
    467, 70, 732, 405, 915, 424,
    874, 951, 80, 288, 298, 802,
    61, 813, 963, 874, 215, 203,
    348, 834, 4, 241, 79, 625,
    881, 650, 779, 651, 990, 303,
    887, 263, 426, 236, 491, 71,
    452, 225, 423, 837, 869, 267,
    878, 745, 655, 991, 676, 871,
    833, 354, 263, 299, 835, 490,
    584, 563, 467, 600, 905, 471,
    // Bin capacity
    2044, 1745, 1913, 2072, 2114, 1695
  };
  const int i6_18_6_26_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    267, 585, 84, 938, 768, 308,
    886, 788, 559, 608, 706, 406,
    435, 86, 170, 321, 575, 362,
    140, 461, 402, 683, 3, 177,
    720, 43, 986, 23, 72, 299,
    71, 621, 237, 849, 681, 376,
    239, 758, 306, 858, 826, 663,
    754, 393, 328, 204, 396, 312,
    269, 781, 349, 83, 911, 255,
    188, 481, 564, 909, 675, 676,
    428, 957, 908, 113, 274, 441,
    405, 249, 889, 992, 948, 524,
    614, 221, 237, 64, 489, 996,
    87, 924, 624, 116, 14, 207,
    358, 976, 25, 449, 312, 711,
    323, 159, 643, 44, 130, 809,
    718, 785, 13, 0, 115, 56,
    398, 728, 288, 36, 175, 99,
    // Bin capacity
    1533, 2100, 1599, 1531, 1695, 1613
  };
  const int i6_18_6_27_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    98, 165, 163, 379, 651, 211,
    877, 249, 721, 298, 227, 617,
    10, 387, 423, 569, 38, 881,
    280, 362, 967, 195, 266, 918,
    586, 99, 537, 473, 930, 134,
    313, 424, 185, 510, 525, 932,
    596, 986, 948, 739, 590, 754,
    407, 814, 106, 70, 761, 87,
    631, 34, 29, 799, 722, 89,
    154, 149, 149, 933, 328, 418,
    124, 464, 87, 575, 385, 596,
    129, 152, 931, 435, 236, 941,
    454, 52, 542, 229, 74, 951,
    320, 816, 182, 453, 34, 890,
    661, 134, 655, 933, 926, 31,
    863, 18, 811, 427, 111, 277,
    635, 314, 803, 565, 536, 198,
    154, 62, 686, 834, 976, 132,
    // Bin capacity
    1544, 1203, 1890, 1994, 1761, 1918
  };
  const int i6_18_6_28_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    872, 133, 291, 646, 269, 930,
    124, 43, 95, 240, 773, 979,
    603, 744, 552, 86, 345, 887,
    359, 240, 869, 509, 38, 786,
    840, 665, 16, 551, 749, 136,
    668, 160, 826, 689, 147, 209,
    502, 702, 539, 315, 33, 566,
    423, 584, 293, 971, 780, 9,
    860, 615, 704, 6, 50, 685,
    711, 419, 59, 746, 898, 360,
    485, 523, 851, 102, 534, 813,
    782, 277, 856, 531, 359, 126,
    944, 582, 758, 667, 779, 721,
    708, 366, 496, 509, 771, 623,
    641, 969, 780, 380, 271, 703,
    231, 668, 299, 322, 737, 113,
    402, 149, 245, 724, 579, 602,
    52, 153, 300, 749, 752, 766,
    // Bin capacity
    2178, 1705, 1884, 1866, 1891, 2137
  };
  const int i6_18_6_29_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    862, 722, 159, 925, 832, 302,
    517, 625, 457, 617, 401, 516,
    888, 116, 760, 363, 316, 494,
    725, 371, 246, 751, 645, 904,
    832, 141, 769, 26, 924, 317,
    157, 548, 420, 82, 223, 898,
    406, 523, 605, 553, 247, 607,
    889, 601, 210, 426, 251, 418,
    114, 866, 602, 345, 632, 412,
    151, 260, 37, 472, 515, 852,
    914, 499, 932, 206, 950, 664,
    951, 326, 473, 940, 257, 569,
    641, 886, 984, 736, 625, 848,
    502, 640, 801, 714, 249, 433,
    33, 356, 965, 763, 289, 759,
    983, 640, 511, 110, 642, 134,
    598, 103, 460, 267, 502, 30,
    895, 354, 281, 570, 193, 500,
    // Bin capacity
    2378, 1845, 2080, 1907, 1869, 2077
  };
  const int i6_18_6_30_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    285, 712, 990, 13, 489, 123,
    214, 980, 833, 723, 274, 949,
    980, 141, 898, 60, 234, 188,
    793, 662, 162, 128, 16, 669,
    594, 853, 923, 495, 760, 398,
    849, 568, 987, 515, 324, 44,
    96, 298, 740, 660, 891, 492,
    125, 954, 436, 456, 371, 119,
    735, 987, 12, 789, 569, 850,
    448, 988, 963, 983, 674, 908,
    968, 0, 241, 652, 513, 913,
    773, 268, 39, 306, 349, 68,
    86, 284, 82, 252, 882, 282,
    835, 607, 51, 867, 556, 660,
    363, 101, 508, 50, 515, 632,
    77, 975, 791, 290, 11, 927,
    835, 532, 482, 527, 876, 794,
    122, 623, 789, 615, 941, 205,
    // Bin capacity
    1989, 2283, 2151, 1816, 2004, 1998
  };
  const int i6_18_6_31_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    92, 187, 494, 248, 858, 314,
    347, 747, 677, 564, 176, 822,
    78, 541, 680, 454, 446, 306,
    800, 617, 849, 396, 212, 762,
    28, 277, 499, 663, 450, 47,
    587, 125, 374, 478, 445, 465,
    573, 560, 368, 856, 227, 511,
    602, 149, 91, 833, 645, 651,
    560, 697, 199, 940, 317, 510,
    112, 68, 739, 380, 337, 797,
    66, 838, 74, 592, 6, 769,
    476, 457, 974, 225, 770, 461,
    541, 385, 370, 966, 554, 244,
    853, 538, 785, 985, 11, 77,
    174, 470, 722, 815, 940, 331,
    305, 5, 937, 882, 858, 734,
    849, 862, 368, 610, 567, 65,
    603, 960, 28, 608, 760, 346,
    // Bin capacity
    1670, 1853, 2015, 2510, 1874, 1793
  };
  const int i6_18_6_32_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    459, 749, 888, 805, 86, 764,
    89, 100, 996, 808, 165, 824,
    784, 481, 688, 525, 366, 955,
    450, 343, 957, 832, 48, 383,
    483, 872, 503, 302, 133, 567,
    344, 6, 10, 22, 125, 149,
    869, 729, 149, 251, 696, 915,
    422, 855, 252, 848, 525, 330,
    24, 296, 345, 121, 837, 546,
    925, 570, 803, 274, 796, 332,
    174, 608, 784, 965, 441, 737,
    549, 257, 720, 880, 101, 748,
    801, 338, 198, 151, 199, 819,
    552, 881, 144, 882, 820, 769,
    917, 856, 369, 771, 820, 666,
    701, 284, 647, 442, 887, 985,
    753, 755, 509, 259, 51, 833,
    183, 980, 700, 950, 832, 587,
    // Bin capacity
    2086, 2192, 2126, 2220, 1745, 2620
  };
  const int i6_18_6_33_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    822, 193, 581, 270, 603, 25,
    696, 178, 420, 800, 464, 971,
    1000, 227, 951, 478, 990, 851,
    483, 922, 827, 629, 487, 655,
    511, 16, 460, 482, 386, 590,
    733, 891, 984, 466, 903, 572,
    474, 454, 441, 213, 995, 266,
    907, 774, 735, 203, 151, 692,
    951, 603, 445, 945, 966, 848,
    831, 254, 626, 791, 852, 886,
    788, 363, 261, 586, 854, 175,
    943, 603, 334, 651, 974, 749,
    903, 756, 181, 776, 772, 420,
    54, 177, 664, 85, 313, 677,
    742, 59, 211, 26, 604, 344,
    76, 736, 819, 313, 356, 88,
    891, 560, 304, 563, 197, 298,
    712, 594, 137, 314, 154, 372,
    // Bin capacity
    2775, 1854, 2080, 1905, 2443, 2102
  };
  const int i6_18_6_34_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    160, 186, 488, 427, 891, 487,
    660, 972, 895, 298, 164, 626,
    621, 256, 320, 601, 896, 716,
    480, 680, 191, 710, 55, 275,
    22, 52, 426, 987, 996, 368,
    149, 616, 823, 886, 189, 539,
    383, 339, 188, 738, 417, 473,
    190, 388, 340, 691, 826, 94,
    521, 122, 372, 752, 778, 670,
    674, 546, 299, 116, 318, 649,
    224, 84, 24, 953, 756, 43,
    414, 135, 666, 212, 149, 595,
    54, 310, 974, 302, 89, 64,
    517, 172, 293, 134, 686, 769,
    391, 979, 203, 21, 934, 675,
    1, 957, 807, 390, 26, 465,
    544, 676, 341, 475, 945, 889,
    736, 927, 971, 375, 349, 348,
    // Bin capacity
    1506, 1876, 1926, 2026, 2114, 1954
  };
  const int i6_18_6_35_26[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    600, 819, 217, 25, 278, 60,
    771, 168, 438, 863, 734, 610,
    316, 603, 322, 68, 163, 295,
    796, 764, 923, 103, 769, 421,
    122, 897, 42, 669, 864, 129,
    509, 137, 311, 592, 832, 968,
    851, 669, 36, 114, 614, 792,
    4, 202, 95, 303, 624, 550,
    290, 158, 789, 276, 761, 43,
    194, 166, 158, 247, 928, 997,
    520, 429, 501, 430, 807, 129,
    161, 434, 326, 957, 736, 889,
    708, 495, 914, 13, 874, 668,
    37, 15, 167, 885, 991, 305,
    328, 704, 476, 835, 768, 95,
    70, 701, 666, 508, 397, 193,
    195, 901, 584, 926, 678, 201,
    285, 325, 883, 665, 559, 738,
    // Bin capacity
    1521, 1933, 1766, 1908, 2785, 1819
  };
  const int i6_18_6_0_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    935, 580, 513, 74, 150, 480,
    810, 684, 723, 981, 811, 0,
    44, 328, 727, 125, 311, 316,
    344, 558, 871, 23, 182, 517,
    423, 669, 650, 35, 438, 570,
    535, 813, 701, 938, 327, 797,
    869, 502, 143, 625, 181, 890,
    142, 964, 23, 470, 621, 997,
    199, 30, 851, 807, 481, 339,
    465, 35, 729, 201, 726, 448,
    12, 211, 909, 491, 673, 326,
    395, 775, 97, 106, 68, 918,
    284, 887, 771, 154, 343, 512,
    155, 680, 75, 604, 981, 529,
    706, 889, 915, 812, 962, 806,
    477, 277, 160, 685, 173, 894,
    755, 806, 161, 713, 782, 582,
    252, 809, 236, 963, 443, 793,
    // Bin capacity
    1301, 1750, 1543, 1468, 1443, 1786
  };
  const int i6_18_6_1_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    879, 934, 535, 396, 194, 553,
    904, 204, 229, 5, 750, 776,
    992, 721, 659, 171, 414, 991,
    995, 178, 907, 212, 666, 470,
    666, 358, 503, 429, 589, 717,
    329, 920, 440, 726, 600, 541,
    777, 623, 657, 443, 107, 556,
    283, 121, 345, 722, 855, 500,
    247, 107, 401, 360, 859, 895,
    232, 82, 718, 76, 726, 44,
    95, 815, 425, 201, 17, 209,
    629, 600, 46, 21, 186, 726,
    478, 729, 336, 505, 419, 753,
    566, 12, 470, 514, 47, 22,
    331, 744, 811, 589, 840, 928,
    546, 205, 236, 852, 955, 581,
    502, 253, 66, 300, 860, 418,
    479, 633, 754, 478, 802, 30,
    // Bin capacity
    1672, 1387, 1438, 1179, 1665, 1635
  };
  const int i6_18_6_2_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    831, 112, 96, 422, 142, 363,
    78, 390, 110, 178, 692, 617,
    293, 718, 597, 129, 706, 830,
    853, 919, 549, 868, 106, 930,
    219, 375, 404, 618, 603, 845,
    911, 928, 888, 429, 732, 468,
    676, 952, 609, 327, 949, 271,
    637, 190, 967, 609, 491, 0,
    833, 445, 878, 644, 239, 809,
    2, 230, 559, 573, 732, 221,
    606, 363, 705, 211, 247, 682,
    627, 574, 841, 122, 911, 460,
    697, 525, 661, 9, 387, 591,
    820, 482, 345, 363, 496, 492,
    934, 687, 944, 867, 831, 172,
    850, 447, 740, 911, 702, 752,
    381, 668, 168, 271, 371, 102,
    340, 708, 704, 404, 624, 386,
    // Bin capacity
    1800, 1652, 1831, 1353, 1694, 1529
  };
  const int i6_18_6_3_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    750, 855, 261, 975, 946, 704,
    823, 567, 474, 379, 528, 365,
    562, 111, 308, 713, 268, 861,
    316, 205, 600, 312, 190, 157,
    60, 732, 910, 659, 224, 901,
    765, 774, 243, 630, 69, 290,
    597, 632, 24, 38, 672, 970,
    317, 559, 415, 745, 82, 654,
    764, 295, 171, 97, 916, 525,
    110, 411, 247, 10, 799, 210,
    98, 388, 528, 251, 363, 396,
    182, 513, 583, 567, 774, 591,
    390, 263, 570, 434, 694, 361,
    729, 483, 19, 14, 599, 383,
    365, 418, 476, 979, 474, 15,
    71, 661, 425, 929, 231, 501,
    629, 560, 228, 683, 192, 756,
    169, 954, 224, 440, 253, 456,
    // Bin capacity
    1322, 1611, 1152, 1521, 1421, 1562
  };
  const int i6_18_6_4_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    1, 145, 701, 268, 473, 547,
    426, 487, 213, 955, 336, 244,
    104, 151, 47, 513, 884, 730,
    250, 848, 694, 721, 733, 797,
    611, 658, 181, 840, 907, 116,
    332, 663, 90, 839, 852, 577,
    711, 812, 952, 968, 207, 720,
    493, 159, 175, 440, 576, 108,
    936, 321, 631, 896, 863, 367,
    777, 508, 874, 437, 73, 162,
    754, 353, 688, 848, 879, 140,
    731, 526, 114, 503, 747, 72,
    880, 691, 639, 716, 468, 231,
    266, 29, 528, 597, 94, 163,
    701, 581, 323, 996, 489, 855,
    73, 413, 982, 252, 969, 931,
    682, 520, 941, 674, 815, 683,
    507, 469, 283, 700, 544, 577,
    // Bin capacity
    1601, 1445, 1570, 2109, 1891, 1391
  };
  const int i6_18_6_5_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    520, 323, 796, 16, 92, 477,
    161, 978, 962, 471, 36, 323,
    965, 476, 188, 721, 255, 3,
    686, 279, 872, 663, 708, 596,
    528, 945, 225, 864, 755, 359,
    946, 153, 268, 414, 112, 569,
    295, 963, 517, 785, 591, 909,
    154, 960, 369, 547, 25, 73,
    973, 840, 361, 297, 910, 758,
    632, 974, 910, 921, 239, 71,
    187, 465, 452, 481, 502, 659,
    581, 7, 843, 467, 37, 455,
    985, 574, 244, 624, 319, 110,
    733, 463, 822, 269, 746, 832,
    940, 353, 945, 476, 866, 391,
    589, 560, 784, 510, 551, 299,
    919, 728, 593, 411, 970, 786,
    722, 763, 43, 364, 790, 380,
    // Bin capacity
    2016, 1891, 1784, 1628, 1489, 1409
  };
  const int i6_18_6_6_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    702, 24, 669, 87, 436, 788,
    284, 455, 927, 976, 847, 710,
    676, 116, 881, 813, 529, 28,
    291, 367, 269, 257, 959, 251,
    206, 625, 631, 568, 546, 823,
    604, 738, 226, 510, 609, 276,
    879, 499, 510, 738, 193, 559,
    883, 222, 597, 418, 100, 86,
    892, 565, 409, 517, 282, 597,
    118, 792, 846, 673, 669, 512,
    968, 827, 397, 193, 358, 190,
    763, 792, 953, 183, 638, 867,
    220, 150, 99, 124, 189, 458,
    695, 637, 662, 260, 946, 217,
    960, 985, 968, 463, 93, 327,
    580, 895, 456, 493, 970, 415,
    137, 580, 974, 648, 592, 311,
    371, 695, 221, 903, 596, 248,
    // Bin capacity
    1808, 1761, 1890, 1559, 1688, 1354
  };
  const int i6_18_6_7_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    721, 61, 599, 248, 752, 630,
    695, 892, 75, 402, 226, 63,
    711, 649, 113, 639, 428, 931,
    879, 766, 466, 930, 700, 284,
    793, 32, 351, 153, 179, 903,
    51, 307, 55, 50, 471, 127,
    510, 82, 366, 641, 829, 489,
    14, 651, 834, 521, 424, 777,
    517, 17, 500, 868, 400, 338,
    968, 667, 794, 165, 258, 973,
    331, 703, 637, 510, 69, 296,
    725, 129, 300, 674, 912, 251,
    129, 549, 868, 731, 734, 642,
    533, 437, 117, 924, 129, 573,
    540, 172, 469, 655, 73, 55,
    818, 973, 278, 112, 425, 484,
    906, 308, 256, 278, 212, 90,
    413, 376, 870, 771, 229, 914,
    // Bin capacity
    1829, 1386, 1418, 1654, 1329, 1573
  };
  const int i6_18_6_8_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    272, 263, 150, 135, 100, 502,
    402, 166, 48, 437, 814, 449,
    869, 803, 9, 346, 117, 812,
    741, 378, 140, 501, 795, 798,
    657, 0, 491, 507, 133, 677,
    485, 396, 724, 447, 147, 388,
    143, 306, 906, 469, 382, 895,
    533, 954, 202, 222, 340, 24,
    541, 576, 876, 956, 415, 690,
    100, 1, 937, 982, 188, 915,
    184, 692, 891, 510, 637, 67,
    588, 31, 576, 890, 944, 891,
    853, 509, 580, 518, 627, 646,
    552, 600, 571, 809, 79, 720,
    767, 252, 776, 592, 14, 505,
    537, 293, 352, 281, 845, 287,
    198, 762, 270, 179, 310, 116,
    463, 915, 233, 762, 257, 399,
    // Bin capacity
    1600, 1422, 1572, 1718, 1286, 1761
  };
  const int i6_18_6_9_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    661, 764, 773, 825, 55, 900,
    294, 729, 475, 306, 763, 497,
    858, 940, 934, 829, 326, 876,
    391, 674, 542, 985, 181, 397,
    954, 106, 330, 294, 787, 450,
    980, 478, 986, 678, 615, 730,
    436, 840, 286, 982, 210, 419,
    307, 595, 2, 221, 472, 686,
    63, 171, 14, 446, 413, 978,
    122, 129, 53, 894, 837, 788,
    915, 325, 951, 9, 516, 739,
    932, 865, 548, 623, 540, 930,
    812, 890, 312, 344, 825, 510,
    75, 525, 163, 22, 202, 148,
    782, 516, 84, 422, 35, 728,
    314, 946, 176, 486, 837, 10,
    989, 956, 160, 855, 980, 493,
    81, 423, 512, 681, 183, 567,
    // Bin capacity
    1811, 1976, 1327, 1799, 1595, 1971
  };
  const int i6_18_6_10_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    846, 535, 634, 462, 475, 373,
    80, 582, 341, 721, 246, 928,
    904, 706, 640, 552, 144, 65,
    293, 337, 189, 194, 986, 963,
    452, 177, 852, 615, 402, 963,
    128, 87, 262, 959, 768, 696,
    313, 489, 351, 438, 420, 196,
    695, 40, 296, 919, 74, 418,
    401, 718, 244, 137, 750, 747,
    852, 8, 762, 394, 903, 63,
    623, 277, 829, 552, 171, 916,
    304, 586, 722, 757, 726, 106,
    581, 568, 887, 511, 315, 986,
    573, 671, 292, 645, 805, 134,
    457, 765, 23, 446, 737, 669,
    849, 590, 841, 398, 873, 725,
    638, 849, 818, 401, 69, 549,
    576, 860, 439, 937, 238, 305,
    // Bin capacity
    1754, 1622, 1728, 1841, 1669, 1798
  };
  const int i6_18_6_11_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    151, 750, 988, 63, 300, 767,
    452, 135, 445, 894, 626, 773,
    952, 6, 465, 791, 433, 550,
    71, 725, 205, 327, 626, 673,
    780, 192, 816, 274, 64, 384,
    159, 384, 846, 387, 150, 141,
    626, 507, 409, 970, 577, 82,
    214, 130, 700, 105, 679, 194,
    251, 400, 346, 584, 794, 641,
    921, 888, 593, 13, 31, 990,
    743, 617, 545, 216, 983, 919,
    222, 963, 448, 708, 186, 64,
    406, 984, 327, 54, 345, 903,
    333, 921, 221, 897, 385, 358,
    889, 553, 155, 217, 295, 842,
    439, 352, 293, 253, 712, 201,
    506, 683, 661, 871, 646, 946,
    753, 961, 744, 655, 205, 172,
    // Bin capacity
    1641, 1878, 1704, 1532, 1487, 1777
  };
  const int i6_18_6_12_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    389, 802, 827, 181, 931, 738,
    611, 650, 488, 198, 323, 536,
    253, 371, 587, 254, 169, 228,
    386, 594, 407, 882, 340, 144,
    20, 806, 909, 598, 787, 691,
    821, 229, 669, 320, 811, 20,
    887, 781, 410, 71, 51, 953,
    998, 948, 196, 340, 546, 347,
    641, 241, 56, 996, 644, 546,
    28, 525, 951, 484, 44, 789,
    800, 785, 238, 849, 989, 442,
    309, 57, 769, 96, 878, 335,
    879, 223, 876, 599, 833, 990,
    93, 644, 942, 912, 612, 603,
    91, 692, 678, 359, 785, 959,
    81, 670, 478, 82, 831, 856,
    343, 156, 754, 770, 170, 343,
    358, 894, 867, 303, 452, 844,
    // Bin capacity
    1492, 1880, 2073, 1549, 1904, 1935
  };
  const int i6_18_6_13_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    254, 244, 5, 577, 634, 798,
    97, 22, 514, 729, 551, 533,
    706, 658, 897, 806, 496, 928,
    170, 557, 4, 768, 511, 257,
    340, 349, 523, 777, 102, 909,
    860, 889, 170, 595, 311, 685,
    579, 61, 859, 138, 615, 395,
    583, 232, 706, 559, 322, 209,
    267, 748, 429, 183, 379, 137,
    115, 565, 734, 88, 151, 431,
    722, 40, 838, 548, 160, 247,
    887, 98, 207, 875, 868, 491,
    362, 531, 883, 736, 26, 43,
    870, 678, 742, 843, 702, 590,
    33, 188, 320, 357, 258, 591,
    872, 845, 342, 65, 26, 511,
    860, 217, 177, 727, 97, 173,
    851, 853, 773, 779, 213, 338,
    // Bin capacity
    1776, 1465, 1719, 1912, 1210, 1557
  };
  const int i6_18_6_14_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    911, 395, 747, 54, 817, 105,
    827, 547, 726, 945, 2, 225,
    160, 627, 780, 38, 331, 839,
    979, 900, 805, 545, 589, 695,
    361, 50, 430, 206, 417, 940,
    344, 297, 508, 164, 771, 470,
    971, 443, 823, 737, 898, 982,
    846, 977, 9, 270, 104, 899,
    347, 311, 58, 352, 636, 682,
    424, 27, 861, 458, 769, 904,
    115, 138, 89, 147, 786, 689,
    343, 240, 199, 373, 970, 10,
    190, 977, 599, 18, 68, 288,
    195, 262, 334, 614, 932, 970,
    591, 761, 899, 328, 517, 464,
    466, 949, 968, 457, 137, 575,
    519, 41, 930, 287, 522, 333,
    290, 451, 328, 867, 315, 273,
    // Bin capacity
    1688, 1595, 1918, 1304, 1821, 1966
  };
  const int i6_18_6_15_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    101, 399, 429, 978, 592, 196,
    555, 796, 751, 170, 872, 233,
    78, 608, 13, 903, 210, 864,
    220, 815, 283, 105, 548, 428,
    333, 710, 471, 335, 302, 735,
    573, 869, 389, 504, 205, 682,
    818, 550, 875, 93, 282, 398,
    177, 246, 280, 258, 854, 559,
    712, 485, 632, 698, 782, 510,
    805, 873, 40, 82, 908, 272,
    749, 552, 276, 967, 599, 702,
    665, 410, 141, 396, 710, 137,
    148, 465, 850, 848, 705, 229,
    554, 506, 374, 5, 713, 999,
    898, 903, 490, 738, 149, 945,
    276, 276, 54, 314, 947, 59,
    154, 279, 912, 703, 618, 866,
    822, 658, 481, 490, 682, 687,
    // Bin capacity
    1656, 1994, 1484, 1646, 2047, 1822
  };
  const int i6_18_6_16_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    406, 481, 924, 165, 356, 38,
    354, 136, 118, 666, 33, 955,
    519, 478, 919, 732, 261, 132,
    574, 372, 448, 424, 694, 914,
    521, 10, 823, 178, 916, 599,
    393, 450, 193, 748, 641, 586,
    714, 463, 284, 567, 653, 949,
    466, 17, 530, 694, 363, 408,
    797, 552, 277, 39, 543, 469,
    562, 372, 485, 424, 68, 561,
    688, 909, 378, 921, 107, 5,
    411, 922, 849, 21, 614, 957,
    456, 809, 356, 460, 365, 340,
    672, 673, 720, 573, 484, 414,
    839, 140, 626, 785, 732, 862,
    978, 924, 862, 211, 33, 60,
    388, 31, 283, 335, 976, 204,
    77, 134, 822, 993, 534, 450,
    // Bin capacity
    1898, 1523, 1914, 1728, 1619, 1722
  };
  const int i6_18_6_17_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    173, 453, 481, 131, 155, 205,
    180, 546, 652, 790, 793, 179,
    828, 330, 902, 676, 430, 138,
    987, 170, 881, 496, 624, 566,
    107, 154, 174, 36, 563, 262,
    88, 264, 929, 150, 20, 215,
    651, 691, 611, 116, 887, 690,
    445, 18, 641, 795, 716, 40,
    452, 609, 344, 19, 990, 7,
    165, 729, 744, 525, 607, 25,
    824, 126, 675, 985, 86, 223,
    198, 748, 436, 511, 785, 641,
    791, 729, 205, 799, 367, 261,
    247, 572, 63, 189, 884, 171,
    175, 278, 33, 145, 154, 278,
    692, 892, 17, 549, 807, 257,
    602, 570, 80, 207, 797, 472,
    130, 255, 238, 588, 615, 99,
    // Bin capacity
    1509, 1587, 1581, 1503, 2005, 923
  };
  const int i6_18_6_18_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    512, 572, 243, 450, 434, 967,
    560, 147, 918, 662, 41, 315,
    450, 270, 59, 641, 585, 106,
    822, 736, 265, 253, 591, 844,
    724, 779, 944, 940, 947, 684,
    327, 183, 219, 889, 177, 612,
    151, 93, 960, 441, 212, 166,
    952, 403, 359, 15, 636, 850,
    984, 773, 55, 21, 248, 490,
    748, 288, 119, 347, 567, 589,
    12, 644, 172, 195, 129, 148,
    238, 880, 158, 781, 928, 245,
    669, 905, 107, 628, 332, 240,
    352, 3, 748, 747, 832, 471,
    527, 449, 935, 262, 72, 257,
    198, 697, 393, 810, 885, 296,
    889, 815, 988, 277, 61, 272,
    818, 978, 267, 20, 789, 794,
    // Bin capacity
    1954, 1891, 1556, 1648, 1665, 1642
  };
  const int i6_18_6_19_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    20, 667, 571, 970, 824, 794,
    638, 673, 20, 424, 817, 584,
    241, 637, 892, 589, 534, 477,
    994, 550, 517, 588, 840, 638,
    654, 892, 85, 698, 36, 679,
    205, 769, 681, 782, 845, 600,
    61, 545, 685, 675, 919, 684,
    783, 37, 545, 535, 173, 971,
    356, 526, 32, 647, 143, 605,
    369, 311, 462, 648, 152, 286,
    287, 752, 887, 78, 451, 969,
    813, 777, 364, 836, 874, 565,
    88, 322, 926, 690, 968, 337,
    510, 188, 492, 132, 983, 292,
    878, 816, 193, 579, 710, 324,
    688, 41, 461, 189, 723, 521,
    969, 108, 69, 477, 823, 35,
    352, 803, 106, 700, 279, 214,
    // Bin capacity
    1767, 1868, 1585, 2031, 2201, 1900
  };
  const int i6_18_6_20_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    269, 261, 798, 845, 512, 176,
    994, 564, 97, 906, 98, 502,
    617, 348, 290, 841, 507, 215,
    361, 651, 155, 831, 133, 69,
    588, 300, 580, 793, 825, 517,
    926, 72, 723, 688, 422, 722,
    847, 979, 383, 558, 127, 564,
    372, 307, 815, 747, 8, 349,
    686, 471, 445, 610, 500, 464,
    259, 742, 672, 901, 646, 771,
    941, 38, 648, 327, 444, 800,
    975, 436, 542, 535, 167, 38,
    214, 587, 825, 816, 548, 92,
    627, 284, 169, 585, 720, 814,
    433, 553, 940, 402, 793, 439,
    970, 210, 422, 489, 389, 510,
    101, 652, 562, 489, 355, 152,
    218, 134, 546, 246, 413, 119,
    // Bin capacity
    2080, 1518, 1923, 2322, 1522, 1463
  };
  const int i6_18_6_21_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    134, 855, 312, 406, 105, 611,
    305, 17, 57, 361, 838, 983,
    144, 515, 827, 979, 871, 410,
    480, 81, 827, 729, 314, 805,
    500, 815, 170, 291, 990, 258,
    238, 569, 944, 148, 573, 300,
    522, 999, 710, 881, 667, 20,
    590, 617, 499, 150, 462, 546,
    564, 566, 307, 948, 960, 290,
    79, 545, 588, 644, 421, 430,
    130, 721, 331, 815, 455, 436,
    540, 657, 1, 486, 191, 557,
    646, 336, 643, 678, 307, 302,
    954, 160, 761, 511, 127, 694,
    27, 740, 75, 762, 50, 746,
    288, 658, 718, 471, 14, 991,
    756, 310, 30, 68, 289, 667,
    477, 595, 671, 378, 370, 923,
    // Bin capacity
    1488, 1968, 1709, 1958, 1615, 2011
  };
  const int i6_18_6_22_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    151, 458, 764, 956, 860, 714,
    562, 215, 776, 426, 294, 22,
    64, 683, 882, 405, 807, 259,
    522, 336, 709, 863, 434, 434,
    13, 442, 590, 682, 675, 743,
    66, 755, 777, 114, 714, 818,
    433, 979, 205, 62, 178, 493,
    214, 858, 465, 596, 324, 530,
    915, 344, 576, 81, 251, 933,
    668, 907, 795, 337, 886, 395,
    606, 172, 141, 598, 814, 718,
    349, 180, 447, 162, 91, 952,
    347, 108, 128, 982, 753, 421,
    354, 377, 34, 215, 380, 743,
    164, 566, 291, 719, 154, 171,
    740, 547, 961, 401, 210, 273,
    560, 472, 966, 872, 373, 597,
    877, 140, 65, 32, 318, 602,
    // Bin capacity
    1547, 1737, 1947, 1729, 1732, 1997
  };
  const int i6_18_6_23_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    556, 599, 196, 701, 226, 47,
    274, 282, 857, 958, 412, 11,
    592, 681, 325, 243, 391, 120,
    710, 197, 477, 272, 38, 737,
    755, 295, 181, 782, 28, 575,
    392, 688, 280, 233, 677, 988,
    671, 956, 709, 305, 939, 491,
    809, 186, 29, 493, 758, 98,
    93, 947, 934, 945, 811, 576,
    812, 509, 757, 761, 192, 437,
    927, 121, 72, 155, 811, 433,
    971, 102, 824, 249, 558, 195,
    224, 297, 439, 611, 326, 818,
    399, 793, 164, 387, 838, 48,
    201, 691, 318, 999, 81, 830,
    581, 544, 567, 260, 733, 650,
    991, 241, 253, 644, 828, 140,
    13, 248, 781, 501, 355, 471,
    // Bin capacity
    2045, 1718, 1674, 1948, 1846, 1572
  };
  const int i6_18_6_24_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    586, 589, 473, 121, 333, 499,
    650, 342, 215, 871, 468, 369,
    371, 174, 192, 165, 251, 249,
    181, 669, 148, 687, 668, 576,
    686, 982, 479, 613, 583, 831,
    658, 250, 970, 342, 683, 761,
    673, 439, 285, 173, 373, 198,
    87, 405, 296, 775, 565, 852,
    435, 431, 578, 120, 61, 163,
    884, 243, 679, 779, 704, 549,
    965, 32, 548, 696, 51, 942,
    859, 326, 794, 818, 888, 113,
    418, 26, 222, 858, 687, 366,
    560, 648, 263, 942, 97, 328,
    120, 340, 865, 459, 709, 159,
    780, 2, 779, 549, 449, 212,
    512, 647, 716, 788, 731, 664,
    680, 631, 192, 731, 833, 340,
    // Bin capacity
    2089, 1484, 1797, 2168, 1888, 1689
  };
  const int i6_18_6_25_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    315, 407, 189, 437, 870, 419,
    118, 921, 670, 647, 525, 465,
    590, 584, 107, 203, 954, 715,
    294, 354, 18, 73, 477, 89,
    300, 159, 908, 850, 583, 778,
    190, 307, 954, 696, 805, 662,
    793, 730, 65, 365, 81, 840,
    963, 515, 358, 465, 701, 809,
    657, 157, 682, 654, 733, 417,
    402, 364, 904, 778, 460, 828,
    169, 834, 447, 437, 293, 411,
    150, 758, 606, 109, 975, 701,
    673, 961, 437, 144, 623, 500,
    765, 800, 626, 210, 205, 656,
    550, 665, 789, 956, 685, 332,
    751, 97, 575, 5, 196, 310,
    82, 805, 535, 130, 161, 153,
    486, 455, 995, 666, 273, 986,
    // Bin capacity
    1719, 2057, 2056, 1631, 2000, 2099
  };
  const int i6_18_6_26_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    509, 797, 122, 296, 440, 91,
    29, 543, 586, 541, 83, 991,
    86, 992, 743, 465, 558, 919,
    155, 153, 343, 92, 37, 1000,
    672, 614, 41, 998, 521, 486,
    731, 844, 37, 352, 695, 892,
    507, 643, 950, 295, 142, 315,
    120, 996, 972, 494, 721, 459,
    173, 58, 236, 168, 454, 371,
    496, 761, 998, 102, 119, 476,
    231, 808, 902, 512, 827, 265,
    84, 929, 26, 181, 224, 531,
    62, 58, 864, 232, 713, 599,
    394, 398, 221, 643, 451, 414,
    142, 588, 924, 867, 904, 921,
    305, 149, 498, 170, 155, 707,
    541, 258, 516, 192, 163, 641,
    36, 660, 637, 501, 877, 948,
    // Bin capacity
    1108, 2153, 2020, 1492, 1698, 2316
  };
  const int i6_18_6_27_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    764, 399, 255, 124, 168, 544,
    392, 585, 369, 433, 994, 348,
    39, 858, 228, 908, 673, 576,
    52, 91, 820, 457, 603, 628,
    481, 89, 959, 453, 528, 404,
    636, 541, 881, 468, 746, 275,
    966, 573, 262, 827, 313, 124,
    798, 308, 618, 875, 982, 92,
    864, 484, 532, 670, 694, 209,
    597, 702, 764, 328, 842, 975,
    323, 371, 258, 879, 506, 616,
    231, 994, 968, 100, 493, 870,
    599, 605, 431, 42, 582, 601,
    876, 156, 852, 492, 835, 989,
    956, 703, 17, 946, 770, 128,
    504, 788, 308, 437, 506, 575,
    679, 360, 689, 723, 520, 15,
    491, 294, 993, 784, 308, 491,
    // Bin capacity
    2170, 1885, 2160, 2106, 2342, 1791
  };
  const int i6_18_6_28_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    776, 735, 710, 266, 741, 518,
    541, 929, 266, 662, 905, 921,
    950, 819, 316, 66, 757, 822,
    358, 607, 557, 909, 939, 753,
    830, 799, 393, 822, 422, 315,
    995, 200, 805, 749, 890, 344,
    758, 505, 222, 116, 296, 12,
    57, 205, 799, 565, 867, 222,
    29, 50, 611, 195, 885, 877,
    251, 299, 849, 92, 377, 300,
    170, 794, 146, 489, 384, 803,
    692, 217, 829, 851, 972, 162,
    591, 950, 793, 242, 568, 932,
    931, 751, 354, 833, 814, 849,
    762, 750, 468, 550, 209, 184,
    47, 195, 210, 2, 552, 996,
    2, 358, 251, 751, 133, 594,
    767, 994, 476, 28, 514, 530,
    // Bin capacity
    2029, 2167, 1932, 1747, 2395, 2162
  };
  const int i6_18_6_29_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    212, 71, 68, 382, 382, 287,
    603, 930, 677, 5, 682, 206,
    712, 292, 216, 897, 742, 742,
    742, 267, 81, 139, 822, 360,
    605, 465, 514, 380, 403, 330,
    983, 709, 577, 722, 965, 879,
    967, 333, 138, 435, 675, 992,
    689, 149, 415, 165, 624, 95,
    822, 945, 696, 863, 70, 948,
    843, 889, 358, 406, 502, 5,
    796, 74, 875, 106, 812, 852,
    355, 42, 954, 334, 302, 411,
    169, 923, 439, 856, 160, 196,
    681, 949, 237, 274, 272, 407,
    35, 391, 934, 544, 676, 994,
    82, 604, 509, 31, 134, 550,
    939, 616, 982, 56, 899, 591,
    478, 437, 177, 308, 185, 471,
    // Bin capacity
    2304, 1954, 1903, 1485, 2002, 2003
  };
  const int i6_18_6_30_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    276, 336, 965, 185, 826, 521,
    610, 514, 972, 817, 331, 53,
    853, 979, 244, 829, 941, 838,
    47, 995, 361, 613, 239, 827,
    143, 700, 465, 634, 490, 662,
    259, 540, 863, 838, 617, 191,
    172, 845, 54, 936, 458, 754,
    175, 929, 893, 944, 45, 522,
    909, 671, 595, 946, 816, 979,
    661, 704, 342, 176, 602, 413,
    907, 453, 180, 643, 363, 174,
    589, 707, 433, 25, 459, 350,
    435, 95, 127, 457, 824, 86,
    4, 749, 457, 847, 528, 440,
    5, 268, 679, 233, 98, 943,
    34, 20, 613, 192, 411, 303,
    195, 185, 873, 681, 547, 32,
    501, 230, 407, 524, 360, 444,
    // Bin capacity
    1468, 2150, 2064, 2280, 1941, 1849
  };
  const int i6_18_6_31_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    762, 706, 347, 43, 435, 841,
    110, 400, 420, 885, 260, 757,
    839, 969, 327, 652, 409, 519,
    1000, 24, 867, 534, 260, 794,
    279, 493, 530, 333, 520, 639,
    868, 935, 139, 142, 451, 699,
    541, 277, 185, 932, 226, 740,
    689, 419, 395, 263, 555, 736,
    216, 539, 735, 610, 957, 316,
    851, 655, 988, 888, 235, 903,
    288, 876, 235, 606, 569, 360,
    163, 692, 210, 132, 796, 286,
    650, 786, 102, 926, 578, 687,
    916, 524, 646, 674, 28, 535,
    981, 360, 831, 61, 573, 69,
    299, 220, 840, 732, 914, 286,
    364, 832, 176, 36, 741, 28,
    828, 980, 693, 200, 335, 845,
    // Bin capacity
    2324, 2334, 1893, 1889, 1931, 2193
  };
  const int i6_18_6_32_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    732, 360, 8, 384, 389, 115,
    536, 612, 829, 616, 989, 543,
    236, 370, 506, 370, 127, 107,
    790, 671, 683, 220, 961, 652,
    798, 32, 169, 948, 294, 565,
    420, 508, 103, 63, 191, 326,
    642, 77, 696, 188, 450, 70,
    522, 33, 586, 312, 971, 616,
    135, 954, 468, 408, 453, 710,
    826, 972, 613, 116, 365, 829,
    876, 376, 897, 317, 376, 274,
    252, 195, 126, 164, 44, 128,
    424, 742, 811, 851, 509, 574,
    76, 234, 19, 853, 629, 780,
    31, 260, 114, 924, 414, 120,
    111, 736, 658, 120, 804, 294,
    477, 620, 377, 564, 903, 0,
    662, 57, 481, 270, 447, 564,
    // Bin capacity
    1881, 1718, 1792, 1692, 2050, 1599
  };
  const int i6_18_6_33_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    278, 223, 2, 839, 200, 701,
    9, 940, 774, 6, 60, 322,
    753, 314, 520, 60, 635, 261,
    779, 982, 768, 435, 486, 664,
    576, 767, 85, 79, 230, 773,
    739, 594, 928, 471, 112, 463,
    207, 344, 649, 788, 519, 374,
    102, 959, 777, 616, 240, 558,
    583, 1, 268, 716, 62, 586,
    478, 502, 459, 235, 341, 535,
    681, 346, 114, 812, 408, 844,
    666, 629, 835, 922, 350, 252,
    111, 73, 693, 848, 424, 626,
    823, 307, 410, 519, 456, 494,
    972, 549, 134, 147, 824, 721,
    655, 585, 887, 567, 997, 777,
    595, 177, 898, 800, 457, 286,
    34, 909, 207, 901, 101, 912,
    // Bin capacity
    2005, 2040, 2086, 2164, 1530, 2250
  };
  const int i6_18_6_34_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    404, 844, 20, 222, 317, 799,
    94, 383, 327, 428, 957, 806,
    334, 191, 475, 504, 317, 920,
    630, 580, 38, 175, 768, 556,
    823, 401, 277, 548, 4, 217,
    665, 182, 361, 548, 255, 350,
    742, 296, 953, 954, 468, 783,
    870, 152, 427, 413, 939, 543,
    567, 803, 856, 62, 983, 866,
    640, 296, 889, 477, 628, 238,
    701, 948, 860, 689, 653, 564,
    193, 937, 816, 226, 676, 668,
    776, 130, 691, 218, 568, 713,
    959, 666, 856, 715, 663, 127,
    713, 594, 7, 921, 646, 886,
    193, 850, 24, 875, 420, 921,
    455, 165, 616, 44, 113, 48,
    860, 948, 982, 873, 333, 55,
    // Bin capacity
    2372, 2092, 2117, 1986, 2169, 2247
  };
  const int i6_18_6_35_27[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    115, 980, 554, 114, 339, 330,
    493, 27, 362, 799, 65, 615,
    855, 14, 550, 768, 491, 506,
    400, 721, 624, 848, 779, 616,
    806, 54, 340, 415, 712, 151,
    914, 540, 877, 16, 478, 680,
    859, 678, 37, 887, 129, 417,
    204, 170, 879, 119, 742, 675,
    882, 582, 208, 104, 756, 49,
    701, 886, 993, 561, 249, 167,
    206, 537, 817, 256, 777, 403,
    605, 296, 860, 679, 708, 708,
    438, 44, 111, 900, 42, 833,
    408, 605, 510, 942, 466, 637,
    679, 326, 601, 45, 850, 127,
    173, 492, 794, 300, 634, 343,
    791, 853, 414, 126, 209, 482,
    925, 196, 44, 302, 195, 914,
    // Bin capacity
    2353, 1801, 2155, 1841, 1940, 1947
  };
  const int i6_18_6_0_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    321, 619, 108, 431, 42, 375,
    801, 173, 317, 812, 902, 535,
    66, 973, 849, 392, 29, 226,
    532, 200, 589, 548, 158, 10,
    245, 58, 67, 989, 381, 77,
    366, 76, 624, 398, 866, 865,
    933, 547, 881, 115, 496, 597,
    303, 954, 932, 926, 518, 265,
    218, 316, 627, 406, 652, 609,
    102, 97, 837, 96, 955, 679,
    947, 958, 624, 929, 90, 457,
    354, 141, 630, 347, 987, 231,
    667, 330, 447, 912, 206, 2,
    839, 48, 405, 36, 695, 677,
    945, 398, 911, 424, 354, 557,
    140, 990, 509, 69, 357, 921,
    390, 7, 621, 7, 284, 326,
    842, 997, 888, 300, 856, 635,
    // Bin capacity
    1502, 1314, 1811, 1357, 1472, 1341
  };
  const int i6_18_6_1_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    530, 386, 691, 776, 157, 115,
    444, 533, 866, 911, 700, 952,
    473, 661, 6, 814, 443, 486,
    534, 450, 888, 679, 148, 684,
    382, 819, 514, 331, 96, 26,
    60, 929, 505, 69, 414, 591,
    586, 510, 556, 549, 306, 438,
    277, 361, 575, 14, 897, 919,
    174, 543, 489, 998, 246, 983,
    954, 851, 704, 721, 414, 917,
    769, 462, 773, 902, 305, 711,
    63, 497, 893, 394, 874, 630,
    194, 831, 34, 918, 176, 152,
    611, 691, 105, 314, 18, 727,
    645, 806, 803, 51, 338, 423,
    938, 24, 618, 525, 557, 474,
    295, 50, 411, 965, 749, 66,
    337, 426, 653, 876, 643, 66,
    // Bin capacity
    1392, 1655, 1698, 1820, 1260, 1576
  };
  const int i6_18_6_2_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    839, 849, 383, 690, 713, 832,
    965, 547, 656, 477, 68, 324,
    207, 58, 54, 577, 105, 300,
    908, 204, 599, 353, 463, 435,
    902, 969, 965, 2, 257, 696,
    762, 949, 449, 271, 129, 725,
    442, 728, 320, 615, 241, 162,
    526, 446, 537, 1, 199, 338,
    853, 429, 110, 213, 653, 44,
    127, 283, 723, 678, 510, 349,
    439, 971, 625, 257, 922, 275,
    229, 456, 252, 382, 392, 560,
    522, 409, 842, 713, 971, 784,
    550, 293, 196, 460, 977, 188,
    651, 920, 322, 424, 139, 647,
    640, 17, 452, 301, 769, 242,
    29, 366, 352, 874, 270, 965,
    536, 503, 693, 635, 69, 772,
    // Bin capacity
    1722, 1598, 1451, 1347, 1334, 1469
  };
  const int i6_18_6_3_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    856, 556, 462, 609, 893, 523,
    682, 124, 89, 983, 888, 228,
    479, 493, 786, 652, 335, 482,
    174, 991, 724, 189, 413, 400,
    913, 455, 769, 203, 605, 838,
    929, 39, 782, 61, 651, 43,
    205, 862, 210, 595, 136, 691,
    147, 948, 666, 806, 884, 519,
    475, 841, 142, 504, 250, 8,
    696, 650, 496, 694, 965, 856,
    136, 798, 392, 712, 666, 757,
    605, 850, 48, 93, 277, 977,
    723, 311, 723, 51, 860, 758,
    805, 493, 551, 431, 535, 556,
    564, 810, 754, 876, 755, 983,
    420, 477, 444, 429, 296, 829,
    385, 305, 878, 439, 363, 473,
    892, 748, 361, 700, 876, 212,
    // Bin capacity
    1732, 1846, 1593, 1550, 1828, 1740
  };
  const int i6_18_6_4_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    788, 209, 567, 203, 764, 347,
    421, 507, 44, 561, 176, 785,
    272, 971, 453, 651, 276, 553,
    766, 203, 571, 810, 858, 249,
    514, 35, 370, 295, 534, 544,
    290, 266, 68, 794, 255, 130,
    950, 5, 788, 295, 303, 215,
    993, 21, 633, 289, 402, 815,
    42, 905, 631, 949, 64, 436,
    800, 504, 804, 440, 835, 325,
    695, 346, 177, 972, 45, 27,
    859, 324, 223, 390, 56, 94,
    668, 717, 989, 0, 452, 0,
    325, 401, 89, 903, 616, 544,
    707, 978, 240, 796, 695, 513,
    416, 567, 208, 741, 913, 440,
    263, 380, 536, 497, 586, 808,
    610, 62, 237, 392, 402, 559,
    // Bin capacity
    1800, 1283, 1323, 1730, 1427, 1280
  };
  const int i6_18_6_5_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    133, 33, 891, 494, 745, 866,
    59, 88, 226, 432, 133, 554,
    712, 786, 433, 667, 216, 850,
    953, 964, 945, 896, 293, 854,
    478, 233, 852, 56, 562, 274,
    491, 781, 682, 123, 148, 518,
    717, 885, 737, 142, 955, 979,
    190, 636, 590, 101, 20, 751,
    247, 952, 670, 924, 73, 266,
    899, 253, 553, 161, 559, 506,
    118, 969, 417, 126, 771, 266,
    555, 942, 74, 968, 598, 509,
    511, 947, 947, 635, 974, 187,
    713, 188, 273, 299, 867, 411,
    788, 866, 62, 724, 658, 661,
    930, 389, 128, 502, 47, 797,
    439, 305, 367, 205, 519, 491,
    965, 572, 297, 744, 845, 604,
    // Bin capacity
    1733, 1889, 1601, 1435, 1573, 1811
  };
  const int i6_18_6_6_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    884, 525, 747, 636, 774, 792,
    210, 650, 915, 886, 356, 778,
    939, 282, 557, 232, 564, 980,
    60, 633, 542, 130, 897, 839,
    614, 723, 789, 215, 820, 765,
    194, 964, 951, 407, 178, 872,
    618, 721, 519, 409, 76, 55,
    491, 225, 922, 831, 961, 958,
    691, 801, 565, 572, 841, 787,
    834, 293, 388, 266, 410, 717,
    973, 387, 489, 814, 25, 92,
    346, 120, 665, 338, 376, 486,
    909, 838, 428, 9, 257, 744,
    559, 682, 415, 526, 77, 270,
    34, 359, 401, 267, 926, 154,
    417, 112, 587, 743, 84, 572,
    899, 500, 891, 80, 53, 919,
    579, 342, 955, 633, 285, 73,
    // Bin capacity
    1812, 1618, 2072, 1413, 1407, 1918
  };
  const int i6_18_6_7_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    672, 187, 418, 462, 223, 872,
    400, 980, 174, 597, 592, 4,
    277, 348, 400, 964, 994, 11,
    120, 557, 725, 903, 216, 679,
    821, 123, 368, 397, 923, 650,
    925, 26, 232, 963, 11, 416,
    225, 13, 829, 159, 513, 539,
    850, 871, 343, 559, 415, 920,
    504, 492, 552, 429, 813, 263,
    34, 602, 480, 220, 84, 544,
    429, 353, 743, 894, 877, 496,
    510, 457, 666, 953, 278, 165,
    791, 817, 589, 781, 551, 414,
    731, 606, 646, 103, 78, 279,
    319, 167, 808, 438, 324, 679,
    379, 184, 914, 128, 698, 238,
    550, 608, 458, 840, 289, 305,
    808, 920, 857, 20, 682, 89,
    // Bin capacity
    1667, 1483, 1820, 1750, 1527, 1349
  };
  const int i6_18_6_8_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    793, 96, 541, 104, 557, 591,
    559, 204, 82, 293, 851, 492,
    21, 261, 456, 61, 128, 823,
    463, 975, 558, 11, 99, 861,
    406, 299, 225, 529, 908, 503,
    434, 1, 751, 704, 460, 661,
    827, 559, 150, 653, 743, 389,
    205, 386, 314, 492, 414, 698,
    74, 446, 295, 459, 529, 262,
    395, 267, 213, 353, 995, 582,
    923, 98, 92, 252, 47, 294,
    641, 817, 395, 189, 189, 840,
    576, 963, 796, 264, 915, 137,
    206, 143, 663, 762, 31, 282,
    302, 537, 148, 926, 209, 727,
    282, 451, 384, 671, 373, 93,
    257, 341, 403, 831, 591, 870,
    195, 210, 150, 773, 512, 716,
    // Bin capacity
    1361, 1270, 1191, 1499, 1540, 1768
  };
  const int i6_18_6_9_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    125, 516, 313, 1000, 239, 932,
    337, 427, 110, 90, 92, 790,
    955, 805, 18, 431, 120, 644,
    330, 608, 690, 750, 209, 274,
    558, 914, 896, 585, 889, 812,
    800, 835, 580, 536, 935, 17,
    827, 734, 601, 838, 47, 562,
    426, 299, 21, 980, 248, 803,
    296, 825, 571, 79, 544, 352,
    1000, 905, 889, 727, 446, 471,
    429, 947, 968, 63, 9, 219,
    5, 9, 136, 729, 173, 761,
    767, 797, 273, 277, 242, 465,
    263, 944, 641, 33, 223, 822,
    651, 956, 287, 624, 478, 538,
    16, 854, 455, 690, 941, 696,
    611, 467, 502, 342, 688, 347,
    355, 631, 315, 908, 350, 223,
    // Bin capacity
    1590, 2266, 1502, 1759, 1249, 1768
  };
  const int i6_18_6_10_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    988, 201, 205, 197, 488, 63,
    103, 323, 289, 539, 750, 57,
    523, 967, 956, 146, 149, 266,
    59, 429, 438, 557, 239, 768,
    537, 388, 296, 47, 304, 590,
    300, 79, 249, 570, 445, 949,
    408, 41, 817, 578, 11, 383,
    896, 275, 341, 914, 288, 94,
    326, 98, 661, 191, 30, 611,
    905, 930, 218, 2, 998, 362,
    742, 306, 532, 481, 233, 197,
    671, 592, 101, 891, 235, 330,
    692, 418, 962, 934, 605, 475,
    370, 486, 37, 266, 441, 904,
    416, 318, 431, 937, 231, 400,
    479, 341, 250, 784, 126, 799,
    297, 749, 39, 697, 32, 63,
    57, 964, 416, 653, 380, 171,
    // Bin capacity
    1608, 1450, 1327, 1721, 1098, 1372
  };
  const int i6_18_6_11_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    817, 429, 30, 908, 867, 338,
    659, 468, 584, 836, 226, 755,
    422, 870, 246, 237, 741, 220,
    695, 732, 591, 369, 483, 196,
    908, 557, 583, 31, 629, 694,
    582, 399, 816, 969, 966, 105,
    44, 551, 519, 443, 380, 622,
    566, 534, 17, 355, 433, 748,
    637, 444, 292, 621, 534, 978,
    269, 126, 152, 191, 359, 536,
    920, 851, 617, 476, 201, 477,
    703, 584, 59, 148, 307, 960,
    582, 981, 321, 709, 477, 424,
    397, 24, 536, 140, 933, 55,
    528, 263, 677, 905, 256, 343,
    627, 331, 911, 992, 421, 237,
    184, 850, 584, 93, 928, 27,
    93, 832, 233, 612, 105, 876,
    // Bin capacity
    1783, 1818, 1438, 1672, 1711, 1590
  };
  const int i6_18_6_12_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    834, 661, 221, 54, 4, 760,
    69, 653, 716, 54, 342, 713,
    649, 347, 988, 774, 268, 380,
    792, 472, 511, 84, 433, 782,
    481, 546, 440, 94, 840, 521,
    921, 486, 737, 533, 644, 33,
    320, 996, 516, 922, 60, 637,
    328, 587, 315, 36, 287, 753,
    716, 867, 370, 182, 918, 966,
    470, 459, 604, 198, 69, 71,
    147, 977, 380, 123, 48, 519,
    387, 850, 7, 152, 197, 567,
    116, 544, 142, 561, 909, 861,
    427, 595, 944, 293, 903, 341,
    596, 618, 674, 127, 912, 721,
    786, 932, 154, 506, 111, 219,
    968, 621, 893, 903, 529, 97,
    734, 632, 476, 180, 130, 766,
    // Bin capacity
    1819, 2211, 1697, 1079, 1420, 1812
  };
  const int i6_18_6_13_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    156, 627, 515, 342, 84, 624,
    797, 507, 814, 951, 929, 119,
    943, 699, 535, 737, 950, 164,
    485, 21, 785, 222, 61, 219,
    538, 636, 236, 45, 705, 19,
    413, 828, 355, 724, 334, 747,
    560, 673, 726, 739, 81, 701,
    309, 823, 598, 110, 394, 628,
    898, 128, 118, 45, 668, 724,
    322, 652, 205, 928, 687, 361,
    362, 763, 351, 543, 953, 456,
    480, 266, 77, 400, 102, 477,
    399, 55, 42, 29, 861, 228,
    413, 136, 696, 78, 802, 390,
    888, 486, 87, 812, 697, 295,
    636, 129, 42, 439, 754, 382,
    813, 571, 890, 924, 572, 885,
    453, 930, 466, 640, 564, 291,
    // Bin capacity
    1858, 1682, 1420, 1641, 1921, 1453
  };
  const int i6_18_6_14_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    854, 248, 797, 599, 954, 259,
    157, 525, 596, 911, 631, 393,
    7, 527, 537, 672, 66, 307,
    178, 678, 120, 595, 755, 370,
    308, 909, 568, 434, 557, 882,
    218, 1, 7, 223, 378, 196,
    101, 267, 62, 603, 807, 428,
    134, 162, 821, 310, 866, 444,
    134, 34, 42, 578, 439, 128,
    65, 908, 821, 492, 184, 335,
    760, 725, 92, 253, 105, 996,
    841, 776, 109, 715, 823, 436,
    930, 965, 195, 486, 67, 824,
    289, 761, 545, 680, 581, 903,
    630, 892, 26, 644, 995, 752,
    396, 716, 91, 30, 630, 141,
    728, 908, 34, 57, 521, 561,
    9, 528, 191, 170, 609, 564,
    // Bin capacity
    1281, 2001, 1075, 1606, 1894, 1695
  };
  const int i6_18_6_15_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    896, 780, 895, 866, 717, 900,
    194, 919, 718, 587, 492, 455,
    236, 972, 439, 996, 475, 827,
    726, 940, 426, 7, 564, 818,
    569, 53, 373, 239, 917, 163,
    677, 779, 150, 138, 371, 396,
    892, 913, 327, 386, 583, 144,
    288, 397, 236, 860, 780, 633,
    321, 984, 725, 103, 316, 788,
    119, 772, 998, 802, 636, 223,
    414, 346, 230, 706, 589, 315,
    308, 593, 462, 336, 623, 823,
    372, 865, 374, 692, 165, 67,
    613, 107, 821, 989, 896, 912,
    489, 35, 127, 348, 464, 815,
    757, 509, 218, 707, 217, 143,
    219, 343, 763, 828, 942, 819,
    473, 352, 103, 116, 770, 832,
    // Bin capacity
    1642, 2043, 1608, 1861, 2016, 1931
  };
  const int i6_18_6_16_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    393, 423, 80, 607, 752, 769,
    80, 262, 832, 431, 685, 488,
    354, 107, 704, 954, 313, 764,
    561, 795, 554, 78, 190, 448,
    241, 684, 27, 257, 775, 916,
    187, 930, 971, 376, 695, 679,
    723, 462, 404, 20, 532, 633,
    638, 232, 826, 836, 769, 470,
    298, 523, 573, 125, 121, 306,
    624, 887, 145, 803, 452, 261,
    457, 258, 31, 839, 546, 705,
    280, 79, 769, 215, 686, 40,
    669, 3, 421, 497, 514, 574,
    719, 140, 996, 930, 215, 117,
    303, 884, 160, 15, 560, 615,
    114, 40, 355, 740, 302, 515,
    151, 373, 596, 168, 626, 709,
    421, 908, 625, 65, 102, 421,
    // Bin capacity
    1395, 1545, 1754, 1539, 1709, 1824
  };
  const int i6_18_6_17_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    659, 798, 220, 809, 250, 265,
    541, 285, 617, 946, 169, 826,
    644, 814, 537, 452, 730, 290,
    531, 987, 226, 499, 22, 45,
    684, 785, 642, 742, 178, 850,
    430, 829, 90, 559, 318, 349,
    313, 868, 981, 154, 546, 316,
    406, 448, 144, 454, 929, 986,
    696, 23, 484, 246, 29, 598,
    865, 259, 757, 218, 204, 39,
    749, 973, 851, 751, 137, 839,
    943, 36, 139, 580, 294, 128,
    315, 423, 214, 866, 659, 310,
    477, 543, 525, 768, 407, 833,
    897, 921, 888, 732, 696, 10,
    946, 937, 528, 914, 969, 48,
    749, 485, 956, 371, 302, 785,
    936, 378, 312, 847, 753, 188,
    // Bin capacity
    2298, 2105, 1777, 2128, 1481, 1503
  };
  const int i6_18_6_18_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    726, 896, 222, 229, 807, 504,
    975, 670, 570, 548, 790, 469,
    149, 878, 91, 762, 304, 951,
    240, 136, 890, 697, 717, 488,
    753, 124, 266, 59, 568, 769,
    37, 56, 598, 267, 383, 432,
    472, 993, 343, 572, 827, 13,
    634, 101, 369, 474, 711, 583,
    649, 525, 787, 549, 775, 474,
    424, 147, 67, 52, 811, 400,
    219, 962, 354, 23, 169, 867,
    289, 939, 211, 229, 67, 286,
    926, 91, 965, 164, 816, 703,
    261, 130, 56, 736, 346, 159,
    136, 723, 69, 842, 58, 681,
    711, 675, 319, 610, 126, 873,
    749, 449, 657, 314, 731, 318,
    514, 115, 584, 377, 655, 941,
    // Bin capacity
    1744, 1694, 1459, 1476, 1900, 1950
  };
  const int i6_18_6_19_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    689, 595, 233, 294, 487, 178,
    725, 540, 262, 216, 284, 527,
    546, 959, 494, 136, 11, 777,
    374, 252, 827, 350, 248, 477,
    646, 812, 774, 256, 80, 364,
    504, 642, 665, 810, 444, 33,
    78, 328, 598, 554, 793, 207,
    798, 364, 326, 673, 512, 693,
    16, 243, 820, 634, 405, 629,
    36, 510, 396, 773, 452, 287,
    930, 327, 984, 18, 490, 889,
    467, 497, 520, 31, 316, 765,
    971, 176, 269, 900, 988, 538,
    303, 389, 423, 915, 735, 888,
    854, 402, 807, 114, 167, 1000,
    61, 671, 694, 91, 407, 803,
    784, 388, 830, 787, 293, 782,
    666, 934, 174, 74, 380, 298,
    // Bin capacity
    1874, 1791, 2003, 1513, 1486, 2011
  };
  const int i6_18_6_20_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    559, 0, 611, 834, 932, 980,
    96, 691, 456, 1000, 837, 840,
    909, 913, 89, 602, 870, 936,
    790, 647, 699, 389, 777, 251,
    441, 423, 199, 669, 333, 320,
    401, 612, 975, 671, 558, 387,
    286, 914, 309, 624, 111, 293,
    614, 195, 631, 256, 65, 728,
    435, 283, 69, 202, 332, 742,
    188, 13, 61, 568, 429, 614,
    820, 708, 617, 12, 524, 306,
    239, 525, 256, 417, 419, 938,
    510, 341, 856, 818, 980, 109,
    869, 403, 404, 498, 578, 436,
    570, 738, 253, 11, 543, 412,
    705, 284, 931, 308, 20, 921,
    835, 988, 648, 664, 313, 303,
    248, 391, 626, 595, 435, 704,
    // Bin capacity
    1903, 1814, 1738, 1828, 1812, 2044
  };
  const int i6_18_6_21_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    781, 518, 318, 310, 738, 492,
    850, 464, 829, 42, 374, 285,
    698, 844, 715, 269, 758, 148,
    875, 985, 473, 255, 236, 957,
    634, 630, 436, 385, 395, 370,
    408, 396, 533, 73, 956, 844,
    289, 958, 844, 127, 568, 138,
    893, 110, 327, 53, 889, 709,
    817, 188, 511, 589, 61, 711,
    891, 532, 258, 849, 910, 907,
    955, 941, 721, 896, 252, 562,
    551, 766, 48, 428, 848, 123,
    664, 393, 786, 967, 653, 559,
    590, 619, 698, 459, 372, 150,
    821, 327, 504, 277, 822, 727,
    272, 986, 368, 645, 499, 802,
    848, 914, 658, 228, 472, 645,
    81, 488, 391, 39, 653, 124,
    // Bin capacity
    2404, 2231, 1900, 1390, 2109, 1867
  };
  const int i6_18_6_22_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    968, 977, 937, 709, 413, 184,
    141, 716, 730, 828, 183, 303,
    453, 556, 939, 882, 975, 809,
    249, 31, 713, 895, 352, 834,
    871, 713, 208, 180, 979, 732,
    900, 394, 8, 185, 20, 634,
    625, 121, 174, 863, 630, 84,
    28, 880, 921, 574, 47, 213,
    938, 132, 544, 745, 131, 32,
    900, 812, 821, 576, 1, 790,
    601, 920, 86, 157, 112, 527,
    620, 900, 207, 897, 198, 57,
    998, 113, 73, 100, 763, 692,
    970, 907, 260, 600, 220, 387,
    290, 793, 852, 347, 456, 547,
    581, 639, 127, 865, 998, 545,
    940, 841, 395, 675, 861, 295,
    842, 176, 946, 637, 443, 746,
    // Bin capacity
    2423, 2160, 1819, 2179, 1583, 1711
  };
  const int i6_18_6_23_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    886, 505, 559, 161, 683, 33,
    303, 361, 206, 432, 657, 685,
    221, 711, 163, 358, 624, 233,
    176, 631, 111, 247, 170, 100,
    140, 660, 522, 192, 367, 240,
    356, 970, 316, 683, 84, 773,
    66, 941, 259, 463, 900, 502,
    242, 224, 991, 156, 720, 701,
    405, 784, 629, 790, 734, 185,
    437, 480, 694, 40, 381, 71,
    798, 342, 131, 652, 542, 534,
    688, 393, 787, 753, 766, 680,
    259, 66, 768, 876, 893, 74,
    424, 238, 169, 382, 104, 314,
    856, 591, 608, 375, 477, 904,
    927, 414, 303, 155, 268, 966,
    451, 670, 19, 115, 205, 63,
    827, 550, 204, 50, 412, 326,
    // Bin capacity
    1735, 1954, 1525, 1411, 1843, 1514
  };
  const int i6_18_6_24_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    511, 427, 264, 609, 158, 947,
    675, 221, 970, 832, 420, 645,
    991, 124, 464, 706, 281, 707,
    634, 894, 18, 235, 392, 651,
    296, 201, 481, 513, 172, 41,
    896, 738, 3, 3, 234, 940,
    974, 559, 369, 449, 50, 607,
    307, 66, 680, 25, 416, 745,
    361, 558, 669, 808, 796, 475,
    137, 509, 835, 962, 543, 108,
    574, 280, 931, 851, 367, 868,
    656, 516, 50, 164, 890, 250,
    64, 877, 829, 692, 946, 237,
    488, 597, 677, 798, 150, 789,
    475, 895, 120, 192, 372, 654,
    217, 67, 973, 483, 860, 584,
    485, 833, 887, 357, 848, 963,
    880, 190, 35, 106, 352, 40,
    // Bin capacity
    1989, 1768, 1913, 1816, 1705, 2119
  };
  const int i6_18_6_25_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    27, 683, 684, 644, 779, 277,
    326, 456, 855, 656, 638, 875,
    170, 694, 139, 946, 154, 670,
    339, 270, 148, 616, 572, 916,
    356, 761, 137, 591, 369, 754,
    34, 631, 902, 161, 134, 534,
    874, 918, 956, 148, 296, 62,
    313, 92, 973, 148, 147, 409,
    423, 242, 468, 677, 900, 644,
    142, 409, 493, 851, 553, 800,
    679, 582, 639, 876, 233, 684,
    797, 644, 487, 421, 784, 674,
    704, 204, 738, 220, 35, 644,
    364, 651, 752, 588, 649, 726,
    961, 158, 985, 230, 341, 699,
    142, 43, 923, 532, 410, 835,
    966, 362, 306, 252, 829, 31,
    779, 345, 51, 801, 212, 389,
    // Bin capacity
    1750, 1697, 2216, 1950, 1674, 2214
  };
  const int i6_18_6_26_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    813, 554, 921, 473, 76, 694,
    790, 711, 289, 186, 899, 883,
    803, 815, 790, 575, 531, 921,
    725, 91, 12, 909, 717, 526,
    37, 177, 733, 143, 658, 704,
    820, 486, 971, 138, 701, 121,
    204, 34, 773, 569, 676, 86,
    256, 133, 760, 209, 356, 421,
    839, 236, 558, 385, 752, 379,
    215, 470, 284, 348, 597, 797,
    842, 615, 674, 311, 489, 321,
    465, 373, 289, 832, 138, 468,
    887, 64, 734, 964, 468, 312,
    99, 737, 821, 276, 836, 103,
    926, 465, 88, 774, 648, 129,
    874, 374, 944, 473, 377, 437,
    621, 278, 834, 325, 19, 765,
    878, 652, 440, 222, 789, 92,
    // Bin capacity
    2330, 1526, 2293, 1704, 2043, 1714
  };
  const int i6_18_6_27_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    742, 766, 415, 119, 379, 270,
    804, 811, 254, 459, 725, 108,
    742, 951, 391, 261, 16, 646,
    309, 276, 900, 57, 156, 358,
    439, 908, 803, 917, 893, 511,
    748, 836, 380, 187, 593, 215,
    680, 837, 102, 625, 188, 747,
    323, 184, 26, 554, 661, 943,
    955, 992, 343, 348, 55, 986,
    72, 656, 574, 462, 447, 782,
    227, 165, 353, 517, 262, 801,
    295, 363, 310, 924, 634, 652,
    542, 922, 948, 48, 597, 858,
    904, 122, 664, 410, 810, 662,
    834, 999, 626, 729, 424, 989,
    172, 62, 39, 91, 466, 764,
    73, 325, 789, 200, 307, 997,
    655, 941, 778, 892, 183, 426,
    // Bin capacity
    2015, 2353, 1841, 1651, 1651, 2480
  };
  const int i6_18_6_28_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    775, 134, 850, 951, 152, 317,
    936, 331, 415, 467, 226, 566,
    604, 661, 678, 309, 664, 901,
    632, 985, 997, 916, 604, 633,
    113, 820, 332, 777, 508, 135,
    113, 624, 338, 756, 352, 687,
    524, 241, 797, 528, 663, 380,
    389, 380, 54, 627, 251, 401,
    816, 896, 310, 38, 983, 69,
    828, 211, 138, 237, 388, 537,
    704, 766, 246, 333, 504, 835,
    41, 491, 121, 782, 953, 612,
    482, 412, 180, 925, 419, 391,
    811, 125, 95, 675, 7, 333,
    923, 383, 929, 430, 242, 284,
    352, 353, 893, 22, 814, 631,
    938, 590, 37, 426, 839, 500,
    490, 880, 543, 35, 634, 209,
    // Bin capacity
    2234, 1981, 1697, 1970, 1964, 1797
  };
  const int i6_18_6_29_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    762, 984, 99, 377, 411, 312,
    197, 181, 876, 332, 455, 693,
    594, 992, 545, 902, 30, 696,
    696, 402, 619, 424, 147, 912,
    387, 586, 681, 227, 654, 741,
    816, 821, 718, 323, 51, 821,
    383, 239, 571, 30, 499, 172,
    308, 897, 142, 873, 786, 308,
    441, 823, 380, 7, 586, 532,
    742, 590, 542, 643, 125, 550,
    387, 362, 108, 37, 342, 668,
    75, 214, 454, 945, 213, 515,
    260, 576, 745, 841, 297, 653,
    387, 927, 827, 559, 183, 500,
    287, 391, 132, 480, 854, 805,
    587, 235, 204, 418, 18, 172,
    5, 197, 967, 454, 330, 528,
    56, 747, 342, 28, 320, 880,
    // Bin capacity
    1585, 2186, 1925, 1699, 1355, 2249
  };
  const int i6_18_6_30_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    53, 418, 939, 242, 669, 366,
    194, 423, 561, 557, 436, 551,
    984, 714, 252, 680, 875, 784,
    628, 320, 746, 593, 393, 318,
    320, 44, 146, 663, 598, 2,
    479, 474, 151, 612, 855, 983,
    8, 202, 53, 880, 511, 598,
    666, 411, 315, 793, 131, 17,
    970, 50, 295, 297, 765, 539,
    809, 401, 916, 176, 420, 620,
    682, 857, 501, 257, 399, 705,
    972, 793, 868, 149, 698, 655,
    577, 598, 213, 980, 664, 999,
    712, 665, 238, 899, 515, 753,
    411, 945, 997, 307, 461, 840,
    877, 857, 585, 18, 171, 735,
    94, 267, 184, 292, 391, 230,
    84, 149, 284, 795, 376, 523,
    // Bin capacity
    2063, 1861, 1787, 1992, 2022, 2214
  };
  const int i6_18_6_31_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    529, 25, 190, 765, 588, 105,
    368, 780, 687, 545, 420, 162,
    399, 8, 560, 777, 918, 163,
    567, 988, 740, 306, 980, 479,
    154, 304, 282, 112, 436, 209,
    822, 286, 128, 860, 55, 81,
    226, 116, 179, 839, 663, 352,
    229, 366, 179, 260, 799, 295,
    576, 654, 869, 534, 91, 617,
    875, 780, 221, 375, 253, 360,
    904, 37, 841, 894, 97, 510,
    63, 353, 820, 776, 148, 859,
    696, 342, 240, 287, 750, 427,
    132, 811, 235, 985, 282, 715,
    639, 865, 318, 955, 918, 706,
    705, 75, 890, 662, 794, 670,
    756, 579, 824, 142, 802, 644,
    677, 479, 820, 909, 218, 958,
    // Bin capacity
    2035, 1714, 1971, 2398, 2012, 1815
  };
  const int i6_18_6_32_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    520, 662, 273, 499, 916, 647,
    567, 665, 60, 683, 888, 229,
    18, 475, 344, 164, 244, 306,
    886, 994, 62, 68, 236, 44,
    462, 80, 660, 693, 235, 752,
    162, 908, 255, 735, 718, 167,
    273, 143, 724, 780, 823, 257,
    549, 681, 734, 729, 440, 84,
    83, 239, 475, 882, 289, 753,
    558, 878, 850, 464, 422, 678,
    321, 202, 994, 483, 759, 25,
    445, 398, 831, 714, 751, 9,
    668, 243, 557, 365, 767, 282,
    241, 985, 704, 812, 282, 881,
    47, 509, 955, 695, 677, 377,
    871, 232, 303, 2, 651, 159,
    938, 472, 179, 404, 796, 913,
    852, 405, 724, 232, 282, 115,
    // Bin capacity
    1862, 2018, 2131, 2069, 2239, 1470
  };
  const int i6_18_6_33_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    484, 921, 979, 650, 566, 962,
    763, 546, 782, 764, 283, 76,
    716, 214, 161, 830, 388, 455,
    707, 580, 883, 239, 453, 616,
    666, 730, 198, 37, 57, 743,
    700, 683, 593, 724, 144, 681,
    564, 232, 745, 192, 371, 961,
    729, 394, 363, 911, 264, 964,
    564, 620, 68, 645, 873, 373,
    855, 140, 177, 899, 733, 24,
    903, 650, 617, 850, 694, 109,
    606, 118, 248, 926, 387, 841,
    760, 923, 149, 913, 168, 513,
    367, 878, 814, 864, 783, 103,
    791, 303, 385, 361, 937, 960,
    678, 697, 780, 639, 849, 333,
    389, 393, 266, 948, 338, 624,
    587, 288, 539, 525, 605, 969,
    // Bin capacity
    2623, 2064, 1939, 2642, 1972, 2285
  };
  const int i6_18_6_34_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    256, 841, 682, 358, 948, 510,
    333, 319, 628, 246, 220, 996,
    849, 484, 841, 195, 930, 727,
    111, 383, 95, 240, 544, 916,
    272, 523, 350, 711, 913, 84,
    512, 773, 227, 345, 625, 834,
    497, 704, 556, 727, 853, 167,
    678, 904, 280, 460, 629, 511,
    727, 700, 85, 643, 893, 587,
    922, 746, 576, 724, 645, 239,
    149, 646, 915, 755, 449, 531,
    478, 730, 167, 353, 193, 898,
    71, 855, 697, 366, 888, 507,
    227, 810, 115, 143, 717, 127,
    356, 690, 987, 414, 614, 19,
    455, 425, 366, 964, 980, 667,
    528, 926, 834, 20, 245, 869,
    817, 993, 247, 929, 813, 465,
    // Bin capacity
    1840, 2781, 1932, 1920, 2703, 2157
  };
  const int i6_18_6_35_28[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    205, 282, 426, 770, 5, 132,
    641, 357, 3, 44, 425, 30,
    950, 276, 728, 953, 129, 396,
    585, 385, 514, 817, 794, 898,
    277, 471, 588, 653, 541, 505,
    201, 637, 484, 755, 324, 262,
    311, 284, 1000, 11, 192, 141,
    977, 909, 469, 277, 253, 380,
    962, 345, 545, 477, 419, 373,
    398, 168, 0, 27, 884, 451,
    558, 362, 63, 40, 289, 369,
    278, 127, 84, 416, 277, 64,
    769, 369, 196, 542, 383, 384,
    353, 313, 998, 483, 160, 763,
    63, 768, 354, 280, 705, 17,
    958, 238, 25, 494, 755, 362,
    883, 163, 737, 453, 420, 576,
    681, 466, 851, 820, 825, 402,
    // Bin capacity
    2262, 1557, 1815, 1871, 1751, 1464
  };
  const int i6_18_6_0_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    520, 234, 923, 785, 846, 285,
    854, 929, 950, 153, 504, 900,
    208, 11, 21, 751, 212, 684,
    32, 201, 497, 801, 950, 922,
    145, 766, 835, 187, 489, 156,
    977, 813, 273, 169, 659, 208,
    104, 157, 46, 31, 851, 162,
    838, 252, 867, 811, 365, 225,
    516, 845, 217, 244, 273, 480,
    213, 766, 358, 33, 20, 440,
    90, 147, 163, 93, 764, 474,
    888, 722, 781, 269, 616, 227,
    250, 435, 947, 826, 48, 122,
    179, 738, 6, 871, 538, 422,
    31, 576, 156, 536, 991, 214,
    611, 737, 334, 883, 642, 180,
    373, 683, 588, 64, 511, 816,
    405, 279, 924, 584, 93, 571,
    // Bin capacity
    1206, 1549, 1481, 1349, 1562, 1248
  };
  const int i6_18_6_1_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    845, 678, 290, 910, 487, 416,
    397, 417, 55, 202, 843, 892,
    658, 528, 961, 660, 59, 876,
    5, 569, 706, 257, 987, 302,
    504, 318, 115, 705, 470, 527,
    500, 461, 50, 659, 358, 120,
    657, 876, 660, 400, 94, 382,
    71, 644, 16, 715, 762, 336,
    414, 749, 639, 475, 467, 317,
    592, 5, 415, 56, 90, 219,
    988, 77, 255, 385, 414, 416,
    598, 79, 394, 712, 96, 917,
    570, 802, 25, 449, 511, 190,
    904, 620, 875, 924, 382, 509,
    706, 4, 29, 916, 771, 933,
    505, 557, 396, 392, 123, 557,
    629, 414, 835, 101, 175, 633,
    507, 137, 259, 123, 85, 437,
    // Bin capacity
    1692, 1336, 1175, 1522, 1208, 1512
  };
  const int i6_18_6_2_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    868, 974, 297, 521, 868, 266,
    609, 33, 748, 141, 761, 909,
    541, 17, 197, 614, 660, 995,
    613, 507, 619, 667, 103, 655,
    168, 244, 805, 333, 565, 784,
    303, 402, 554, 529, 321, 467,
    970, 88, 205, 878, 20, 628,
    996, 722, 174, 919, 284, 46,
    832, 486, 181, 871, 845, 41,
    91, 83, 831, 68, 536, 564,
    284, 921, 562, 974, 402, 692,
    149, 92, 521, 255, 163, 306,
    695, 704, 378, 637, 29, 982,
    213, 302, 749, 250, 824, 456,
    509, 122, 549, 940, 292, 203,
    917, 275, 986, 208, 826, 814,
    691, 672, 541, 624, 22, 381,
    601, 703, 100, 935, 556, 61,
    // Bin capacity
    1709, 1249, 1530, 1762, 1374, 1573
  };
  const int i6_18_6_3_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    440, 862, 711, 573, 582, 66,
    949, 106, 120, 358, 700, 717,
    817, 296, 690, 250, 467, 75,
    610, 100, 53, 321, 955, 376,
    353, 773, 547, 866, 410, 531,
    639, 226, 458, 264, 364, 161,
    488, 567, 597, 208, 556, 501,
    295, 91, 760, 672, 127, 234,
    841, 578, 241, 232, 610, 738,
    827, 936, 709, 232, 569, 436,
    452, 925, 73, 70, 216, 341,
    940, 178, 90, 116, 600, 615,
    133, 970, 384, 998, 145, 813,
    420, 674, 848, 412, 846, 620,
    647, 455, 150, 164, 812, 701,
    159, 931, 438, 189, 306, 766,
    306, 661, 233, 167, 373, 484,
    620, 595, 388, 782, 140, 265,
    // Bin capacity
    1706, 1704, 1286, 1181, 1507, 1449
  };
  const int i6_18_6_4_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    359, 175, 589, 806, 511, 668,
    44, 992, 705, 608, 958, 974,
    782, 678, 327, 775, 326, 663,
    795, 198, 347, 892, 292, 794,
    863, 450, 182, 195, 271, 308,
    513, 751, 725, 600, 711, 367,
    203, 834, 322, 997, 85, 74,
    590, 326, 66, 595, 644, 112,
    892, 908, 742, 920, 41, 237,
    849, 46, 747, 577, 412, 911,
    724, 849, 196, 931, 759, 323,
    845, 16, 862, 968, 496, 300,
    221, 262, 593, 298, 578, 850,
    432, 709, 11, 42, 561, 977,
    552, 996, 410, 550, 117, 506,
    982, 213, 572, 147, 414, 621,
    464, 347, 51, 433, 642, 574,
    536, 684, 332, 606, 457, 114,
    // Bin capacity
    1846, 1636, 1349, 1897, 1435, 1625
  };
  const int i6_18_6_5_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    360, 160, 55, 990, 220, 844,
    800, 777, 547, 370, 350, 790,
    326, 85, 152, 696, 936, 49,
    37, 529, 426, 831, 266, 622,
    935, 827, 809, 232, 435, 285,
    362, 231, 150, 457, 616, 908,
    312, 536, 384, 402, 880, 132,
    917, 46, 50, 47, 580, 144,
    314, 703, 771, 344, 40, 443,
    818, 399, 988, 450, 127, 337,
    76, 664, 121, 152, 548, 105,
    987, 910, 951, 52, 285, 444,
    328, 422, 390, 366, 683, 200,
    733, 646, 242, 665, 258, 797,
    671, 192, 871, 437, 333, 330,
    49, 242, 992, 808, 582, 861,
    118, 962, 1000, 187, 909, 837,
    523, 575, 687, 115, 467, 431,
    // Bin capacity
    1517, 1559, 1678, 1331, 1491, 1498
  };
  const int i6_18_6_6_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    222, 452, 120, 907, 313, 417,
    761, 311, 894, 844, 606, 643,
    668, 281, 312, 675, 742, 357,
    94, 271, 166, 277, 929, 522,
    919, 790, 62, 790, 383, 201,
    402, 435, 33, 275, 652, 905,
    589, 828, 107, 706, 42, 249,
    463, 303, 730, 361, 647, 350,
    696, 141, 538, 52, 995, 238,
    364, 777, 892, 90, 75, 466,
    199, 205, 903, 717, 397, 872,
    2, 615, 761, 479, 725, 541,
    366, 771, 38, 952, 741, 30,
    78, 113, 326, 242, 547, 597,
    668, 831, 773, 938, 546, 854,
    348, 353, 212, 559, 550, 916,
    652, 15, 389, 101, 901, 158,
    15, 316, 459, 73, 0, 497,
    // Bin capacity
    1327, 1380, 1363, 1597, 1730, 1557
  };
  const int i6_18_6_7_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    720, 353, 102, 220, 389, 337,
    803, 183, 583, 636, 220, 493,
    646, 859, 727, 378, 122, 283,
    882, 564, 429, 137, 826, 935,
    793, 619, 195, 819, 684, 61,
    142, 557, 896, 44, 628, 232,
    701, 799, 112, 470, 255, 318,
    754, 238, 237, 689, 583, 170,
    742, 49, 160, 404, 965, 14,
    740, 555, 355, 980, 848, 263,
    452, 169, 60, 192, 445, 552,
    706, 851, 738, 287, 612, 719,
    619, 916, 116, 781, 75, 258,
    526, 734, 558, 532, 519, 466,
    161, 478, 326, 989, 34, 711,
    985, 239, 756, 850, 233, 118,
    695, 185, 184, 586, 178, 929,
    797, 505, 538, 553, 61, 909,
    // Bin capacity
    2116, 1579, 1262, 1703, 1370, 1386
  };
  const int i6_18_6_8_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    32, 352, 328, 974, 221, 912,
    726, 784, 704, 960, 243, 410,
    867, 984, 8, 875, 915, 241,
    96, 204, 637, 987, 194, 425,
    979, 256, 125, 270, 753, 640,
    274, 359, 558, 324, 256, 955,
    522, 816, 622, 464, 507, 456,
    263, 467, 399, 988, 182, 84,
    458, 491, 153, 644, 105, 747,
    178, 864, 931, 619, 21, 447,
    773, 522, 717, 804, 150, 90,
    39, 210, 306, 557, 545, 918,
    884, 808, 470, 786, 524, 403,
    278, 690, 107, 834, 502, 266,
    566, 994, 657, 91, 964, 367,
    833, 661, 804, 961, 611, 185,
    651, 597, 145, 975, 472, 391,
    447, 585, 560, 70, 138, 937,
    // Bin capacity
    1596, 1916, 1482, 2193, 1315, 1598
  };
  const int i6_18_6_9_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    112, 384, 543, 13, 946, 198,
    832, 60, 137, 649, 629, 372,
    184, 133, 770, 580, 846, 307,
    32, 340, 435, 283, 376, 470,
    873, 58, 241, 459, 760, 978,
    75, 334, 788, 114, 584, 618,
    194, 649, 141, 699, 551, 320,
    876, 797, 732, 753, 233, 723,
    111, 868, 464, 403, 626, 839,
    104, 556, 185, 188, 133, 683,
    42, 195, 821, 413, 695, 374,
    307, 228, 518, 314, 205, 952,
    847, 416, 19, 910, 374, 139,
    308, 674, 621, 699, 608, 658,
    895, 524, 71, 736, 226, 439,
    878, 958, 953, 314, 790, 462,
    918, 407, 343, 870, 510, 808,
    811, 385, 398, 882, 931, 482,
    // Bin capacity
    1526, 1448, 1487, 1686, 1821, 1785
  };
  const int i6_18_6_10_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    206, 342, 280, 957, 668, 411,
    255, 613, 585, 829, 44, 482,
    883, 234, 988, 313, 780, 141,
    594, 439, 96, 130, 457, 925,
    466, 765, 764, 270, 137, 686,
    117, 894, 554, 631, 924, 939,
    603, 445, 55, 742, 480, 831,
    582, 655, 64, 488, 773, 158,
    264, 108, 382, 794, 782, 239,
    95, 669, 944, 745, 308, 455,
    658, 425, 909, 360, 133, 579,
    815, 781, 341, 149, 668, 191,
    335, 81, 796, 217, 559, 397,
    38, 437, 222, 782, 49, 269,
    429, 9, 620, 613, 382, 13,
    318, 732, 913, 729, 177, 52,
    359, 415, 599, 863, 616, 886,
    336, 857, 936, 520, 102, 664,
    // Bin capacity
    1349, 1632, 1843, 1858, 1474, 1525
  };
  const int i6_18_6_11_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    766, 305, 865, 143, 976, 34,
    352, 944, 969, 218, 416, 766,
    775, 198, 422, 912, 623, 87,
    826, 215, 622, 297, 793, 731,
    682, 899, 355, 493, 684, 683,
    250, 319, 650, 155, 686, 354,
    32, 250, 15, 569, 1000, 747,
    709, 309, 505, 880, 382, 85,
    32, 115, 827, 131, 127, 488,
    463, 178, 264, 99, 786, 258,
    874, 291, 924, 859, 553, 580,
    555, 509, 470, 992, 366, 56,
    581, 867, 299, 301, 51, 503,
    105, 160, 843, 716, 709, 800,
    463, 268, 372, 76, 916, 730,
    903, 546, 793, 583, 408, 374,
    482, 11, 830, 393, 453, 253,
    401, 823, 741, 550, 560, 640,
    // Bin capacity
    1712, 1334, 1992, 1548, 1941, 1512
  };
  const int i6_18_6_12_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    426, 530, 672, 40, 479, 846,
    354, 886, 111, 957, 543, 688,
    620, 626, 929, 101, 613, 229,
    592, 160, 214, 667, 10, 156,
    683, 182, 25, 690, 857, 238,
    228, 971, 879, 640, 979, 676,
    770, 623, 558, 387, 997, 84,
    303, 64, 782, 603, 827, 835,
    615, 792, 921, 816, 9, 213,
    904, 43, 208, 452, 789, 241,
    419, 24, 41, 730, 767, 482,
    505, 869, 699, 908, 692, 583,
    255, 219, 209, 727, 936, 646,
    976, 737, 222, 277, 557, 684,
    216, 870, 576, 156, 600, 687,
    109, 151, 496, 707, 385, 106,
    357, 99, 277, 612, 153, 170,
    847, 476, 448, 452, 391, 525,
    // Bin capacity
    1714, 1554, 1544, 1853, 1976, 1510
  };
  const int i6_18_6_13_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    203, 578, 533, 213, 675, 444,
    5, 416, 342, 292, 346, 41,
    569, 366, 735, 788, 58, 201,
    112, 556, 922, 800, 368, 885,
    750, 384, 556, 4, 704, 79,
    988, 938, 694, 872, 502, 754,
    613, 667, 489, 614, 67, 962,
    742, 589, 790, 195, 297, 529,
    858, 381, 388, 518, 85, 683,
    81, 573, 891, 423, 523, 113,
    228, 647, 34, 965, 80, 348,
    257, 853, 135, 176, 145, 74,
    13, 590, 548, 832, 420, 860,
    506, 769, 916, 382, 177, 133,
    596, 340, 313, 874, 667, 305,
    300, 846, 935, 284, 926, 891,
    999, 711, 885, 891, 707, 170,
    441, 854, 235, 452, 868, 626,
    // Bin capacity
    1556, 2083, 1948, 1804, 1435, 1526
  };
  const int i6_18_6_14_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    570, 161, 784, 829, 227, 427,
    939, 674, 697, 640, 457, 773,
    240, 449, 329, 327, 379, 581,
    431, 382, 895, 649, 39, 809,
    826, 120, 706, 31, 692, 838,
    678, 237, 645, 33, 135, 169,
    752, 418, 598, 553, 882, 805,
    11, 455, 460, 638, 584, 229,
    187, 402, 28, 3, 746, 955,
    923, 758, 397, 13, 410, 858,
    286, 509, 229, 549, 56, 290,
    40, 714, 887, 161, 203, 498,
    943, 102, 641, 701, 401, 465,
    862, 370, 744, 427, 593, 516,
    172, 646, 49, 676, 284, 742,
    312, 766, 415, 437, 438, 40,
    351, 724, 229, 455, 544, 492,
    895, 966, 872, 705, 465, 958,
    // Bin capacity
    1790, 1683, 1825, 1488, 1432, 1985
  };
  const int i6_18_6_15_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    629, 132, 608, 165, 74, 112,
    371, 152, 520, 586, 592, 595,
    863, 233, 91, 574, 929, 836,
    506, 730, 447, 878, 864, 42,
    638, 585, 420, 337, 249, 165,
    321, 234, 753, 592, 469, 889,
    574, 235, 837, 298, 156, 422,
    711, 785, 761, 594, 925, 729,
    937, 660, 910, 467, 378, 151,
    711, 921, 941, 517, 593, 563,
    597, 258, 386, 745, 100, 433,
    434, 992, 275, 19, 559, 718,
    438, 296, 975, 953, 783, 24,
    454, 792, 565, 145, 107, 489,
    127, 27, 542, 940, 310, 439,
    560, 901, 917, 825, 599, 189,
    231, 843, 345, 213, 402, 234,
    219, 23, 790, 66, 455, 705,
    // Bin capacity
    1787, 1687, 2125, 1709, 1638, 1483
  };
  const int i6_18_6_16_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    894, 672, 475, 176, 967, 429,
    566, 888, 760, 230, 926, 539,
    590, 978, 225, 757, 868, 71,
    76, 693, 650, 310, 113, 73,
    853, 396, 233, 109, 932, 837,
    795, 565, 824, 216, 816, 768,
    770, 43, 806, 604, 586, 326,
    426, 226, 370, 464, 32, 859,
    785, 466, 769, 973, 332, 74,
    335, 358, 659, 121, 354, 597,
    969, 732, 800, 318, 365, 71,
    389, 714, 272, 234, 492, 479,
    656, 625, 529, 243, 742, 649,
    320, 91, 517, 410, 178, 72,
    407, 540, 745, 7, 795, 906,
    504, 28, 165, 439, 213, 582,
    667, 182, 184, 938, 290, 216,
    621, 955, 589, 160, 403, 308,
    // Bin capacity
    2054, 1770, 1851, 1298, 1819, 1519
  };
  const int i6_18_6_17_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    564, 951, 312, 496, 453, 203,
    183, 579, 737, 500, 632, 283,
    346, 697, 927, 706, 116, 70,
    421, 274, 511, 862, 856, 285,
    430, 282, 447, 733, 724, 849,
    831, 344, 602, 689, 407, 14,
    413, 928, 673, 702, 452, 526,
    562, 60, 177, 795, 415, 249,
    398, 686, 408, 986, 240, 28,
    934, 631, 207, 971, 0, 705,
    243, 952, 611, 127, 177, 706,
    581, 741, 366, 564, 201, 124,
    736, 639, 94, 961, 888, 89,
    883, 826, 175, 456, 631, 630,
    181, 676, 476, 884, 426, 541,
    879, 811, 956, 918, 324, 726,
    716, 187, 431, 424, 141, 113,
    716, 411, 805, 799, 586, 14,
    // Bin capacity
    1954, 2082, 1739, 2452, 1496, 1201
  };
  const int i6_18_6_18_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    109, 500, 718, 331, 108, 97,
    193, 378, 916, 82, 851, 251,
    139, 432, 111, 885, 512, 437,
    182, 865, 583, 545, 99, 811,
    801, 850, 420, 637, 590, 465,
    531, 775, 542, 21, 223, 803,
    647, 960, 760, 684, 250, 237,
    400, 264, 43, 53, 641, 970,
    11, 591, 678, 378, 499, 399,
    522, 776, 292, 955, 445, 751,
    893, 256, 810, 962, 410, 483,
    170, 574, 364, 998, 29, 123,
    144, 262, 516, 895, 592, 879,
    508, 796, 745, 661, 906, 209,
    33, 44, 155, 225, 200, 388,
    551, 944, 203, 713, 183, 589,
    166, 47, 151, 657, 248, 790,
    361, 639, 729, 204, 315, 671,
    // Bin capacity
    1251, 1958, 1719, 1945, 1397, 1840
  };
  const int i6_18_6_19_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    83, 225, 572, 846, 135, 882,
    991, 217, 61, 446, 851, 534,
    831, 810, 886, 908, 767, 399,
    878, 560, 206, 929, 743, 174,
    560, 362, 965, 792, 420, 440,
    767, 137, 740, 964, 671, 428,
    263, 309, 157, 847, 97, 829,
    985, 202, 213, 545, 443, 106,
    218, 498, 871, 527, 306, 856,
    992, 971, 567, 500, 171, 798,
    271, 953, 560, 599, 504, 232,
    524, 539, 394, 155, 990, 459,
    487, 545, 120, 409, 967, 498,
    555, 906, 764, 402, 658, 785,
    632, 808, 418, 582, 657, 346,
    335, 972, 689, 951, 154, 208,
    27, 58, 172, 157, 407, 806,
    539, 311, 973, 699, 676, 936,
    // Bin capacity
    1972, 1861, 1851, 2233, 1908, 1928
  };
  const int i6_18_6_20_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    672, 111, 99, 639, 826, 960,
    605, 569, 532, 727, 418, 372,
    49, 475, 789, 190, 910, 502,
    7, 177, 920, 137, 568, 484,
    810, 956, 33, 467, 454, 958,
    83, 121, 803, 785, 159, 931,
    454, 633, 66, 266, 790, 222,
    402, 300, 389, 427, 262, 284,
    379, 940, 530, 829, 515, 39,
    870, 55, 717, 927, 520, 997,
    488, 463, 790, 318, 960, 921,
    423, 88, 508, 871, 375, 329,
    497, 298, 458, 559, 755, 389,
    227, 411, 252, 798, 255, 711,
    226, 559, 271, 746, 713, 616,
    129, 371, 886, 222, 150, 37,
    438, 588, 691, 678, 643, 27,
    852, 603, 904, 334, 383, 16,
    // Bin capacity
    1523, 1544, 1928, 1984, 1932, 1759
  };
  const int i6_18_6_21_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    790, 10, 291, 66, 473, 774,
    642, 877, 35, 787, 914, 266,
    666, 95, 107, 654, 823, 440,
    771, 796, 278, 811, 599, 209,
    589, 106, 501, 132, 699, 995,
    520, 838, 567, 523, 980, 87,
    596, 672, 550, 131, 579, 876,
    705, 741, 239, 269, 144, 473,
    192, 443, 851, 357, 716, 708,
    965, 344, 273, 424, 238, 935,
    73, 338, 381, 84, 624, 517,
    210, 252, 808, 125, 817, 895,
    860, 613, 230, 301, 320, 721,
    979, 534, 92, 370, 835, 869,
    923, 850, 368, 798, 174, 836,
    142, 34, 149, 885, 194, 501,
    956, 851, 177, 876, 209, 231,
    851, 755, 364, 456, 881, 259,
    // Bin capacity
    2306, 1846, 1263, 1624, 2061, 2137
  };
  const int i6_18_6_22_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    146, 540, 125, 482, 991, 602,
    940, 77, 75, 592, 170, 188,
    996, 138, 30, 133, 825, 886,
    662, 18, 467, 259, 705, 950,
    599, 779, 824, 243, 357, 315,
    164, 802, 818, 93, 909, 904,
    174, 626, 758, 461, 861, 901,
    97, 185, 884, 292, 244, 582,
    375, 254, 724, 550, 403, 721,
    73, 513, 614, 451, 403, 593,
    551, 907, 294, 537, 988, 336,
    431, 2, 155, 321, 119, 141,
    656, 650, 823, 844, 79, 187,
    432, 217, 130, 361, 393, 473,
    101, 926, 336, 495, 17, 808,
    520, 220, 288, 815, 351, 154,
    925, 642, 675, 280, 237, 953,
    77, 868, 771, 872, 685, 558,
    // Bin capacity
    1611, 1701, 1788, 1644, 1777, 2085
  };
  const int i6_18_6_23_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    949, 178, 151, 532, 83, 143,
    386, 931, 792, 994, 897, 758,
    844, 686, 272, 171, 5, 481,
    290, 876, 251, 738, 414, 979,
    878, 815, 232, 251, 204, 691,
    688, 350, 753, 138, 28, 103,
    569, 927, 353, 346, 461, 532,
    129, 619, 460, 349, 965, 771,
    698, 910, 875, 481, 90, 236,
    894, 347, 856, 310, 514, 911,
    547, 640, 852, 533, 337, 498,
    53, 394, 978, 867, 756, 804,
    41, 590, 332, 545, 375, 411,
    997, 625, 910, 22, 999, 146,
    898, 496, 109, 57, 911, 440,
    220, 678, 104, 369, 366, 749,
    712, 4, 801, 203, 76, 101,
    436, 16, 424, 514, 86, 791,
    // Bin capacity
    2097, 2067, 1949, 1522, 1552, 1957
  };
  const int i6_18_6_24_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    701, 201, 210, 130, 500, 100,
    869, 344, 361, 441, 853, 566,
    911, 77, 17, 134, 536, 952,
    927, 468, 911, 239, 196, 432,
    255, 526, 308, 539, 688, 712,
    5, 636, 607, 921, 204, 862,
    602, 204, 775, 612, 802, 877,
    787, 291, 342, 517, 359, 713,
    66, 850, 974, 654, 280, 932,
    620, 9, 989, 847, 149, 27,
    871, 572, 911, 349, 262, 23,
    347, 345, 819, 994, 684, 445,
    292, 333, 401, 652, 151, 372,
    620, 587, 621, 947, 22, 472,
    852, 794, 584, 581, 916, 860,
    418, 275, 395, 552, 563, 398,
    294, 439, 148, 158, 644, 819,
    200, 827, 303, 403, 952, 948,
    // Bin capacity
    1992, 1608, 2000, 1999, 1811, 2173
  };
  const int i6_18_6_25_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    937, 746, 711, 821, 304, 840,
    931, 623, 280, 454, 498, 143,
    865, 468, 961, 675, 896, 133,
    849, 970, 567, 729, 785, 672,
    336, 0, 208, 565, 66, 340,
    983, 528, 795, 56, 904, 400,
    718, 501, 404, 775, 767, 324,
    747, 283, 218, 218, 593, 497,
    988, 744, 79, 534, 1, 207,
    398, 954, 212, 792, 351, 528,
    751, 349, 579, 519, 279, 531,
    439, 256, 721, 100, 597, 444,
    534, 10, 738, 777, 841, 290,
    90, 635, 530, 377, 140, 640,
    839, 870, 34, 186, 440, 230,
    156, 615, 983, 650, 910, 143,
    193, 83, 546, 978, 731, 799,
    4, 448, 590, 57, 179, 126,
    // Bin capacity
    2242, 1893, 1908, 1930, 1934, 1519
  };
  const int i6_18_6_26_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    941, 426, 832, 264, 619, 575,
    84, 174, 783, 417, 646, 46,
    400, 452, 933, 170, 490, 661,
    788, 688, 54, 280, 129, 563,
    127, 76, 854, 302, 938, 650,
    960, 492, 635, 1000, 518, 349,
    98, 767, 80, 349, 558, 414,
    232, 127, 96, 31, 470, 767,
    187, 914, 253, 72, 110, 720,
    563, 517, 99, 521, 333, 337,
    130, 696, 534, 759, 768, 674,
    617, 580, 737, 274, 309, 739,
    546, 97, 358, 921, 814, 96,
    533, 731, 551, 62, 256, 792,
    472, 710, 870, 117, 470, 628,
    823, 632, 407, 673, 973, 323,
    254, 558, 702, 409, 326, 929,
    538, 457, 781, 238, 91, 488,
    // Bin capacity
    1742, 1910, 2008, 1441, 1852, 2048
  };
  const int i6_18_6_27_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    382, 399, 255, 990, 80, 722,
    182, 362, 611, 877, 739, 676,
    114, 867, 742, 442, 618, 41,
    598, 798, 742, 219, 245, 241,
    384, 866, 686, 325, 37, 201,
    245, 574, 320, 847, 165, 223,
    766, 799, 806, 793, 300, 650,
    315, 484, 669, 436, 148, 781,
    131, 875, 635, 80, 50, 579,
    133, 53, 160, 851, 19, 886,
    63, 396, 249, 356, 80, 890,
    806, 486, 806, 22, 448, 467,
    244, 826, 31, 569, 66, 901,
    909, 239, 713, 805, 749, 477,
    872, 957, 628, 923, 995, 6,
    524, 196, 956, 834, 16, 970,
    401, 227, 433, 414, 89, 203,
    12, 996, 925, 544, 150, 152,
    // Bin capacity
    1499, 2202, 2195, 2186, 1058, 1919
  };
  const int i6_18_6_28_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    647, 60, 295, 799, 989, 976,
    201, 197, 133, 597, 282, 977,
    298, 598, 564, 629, 928, 926,
    596, 813, 297, 18, 289, 700,
    690, 818, 396, 398, 549, 661,
    696, 585, 751, 586, 24, 45,
    375, 834, 485, 218, 989, 171,
    300, 328, 27, 76, 730, 2,
    237, 91, 468, 630, 180, 350,
    766, 376, 282, 178, 118, 100,
    516, 826, 416, 837, 637, 533,
    482, 144, 6, 832, 708, 769,
    819, 339, 376, 795, 865, 547,
    667, 812, 135, 759, 51, 445,
    821, 618, 765, 175, 295, 569,
    201, 939, 587, 541, 268, 676,
    750, 115, 507, 461, 620, 135,
    27, 983, 450, 89, 843, 471,
    // Bin capacity
    1939, 2022, 1481, 1839, 1998, 1932
  };
  const int i6_18_6_29_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    917, 544, 285, 379, 937, 366,
    72, 789, 615, 589, 983, 465,
    292, 497, 693, 50, 443, 78,
    302, 854, 172, 33, 313, 406,
    473, 173, 761, 339, 895, 334,
    451, 481, 31, 397, 805, 162,
    59, 274, 65, 237, 457, 895,
    908, 258, 987, 12, 453, 462,
    112, 725, 170, 634, 446, 686,
    187, 128, 95, 425, 831, 860,
    197, 174, 19, 903, 736, 989,
    31, 615, 391, 637, 528, 498,
    149, 881, 784, 537, 109, 461,
    747, 8, 339, 880, 565, 240,
    18, 638, 467, 314, 967, 723,
    418, 94, 994, 619, 901, 980,
    720, 407, 272, 74, 328, 746,
    856, 305, 971, 286, 575, 575,
    // Bin capacity
    1486, 1687, 1744, 1580, 2424, 2135
  };
  const int i6_18_6_30_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    13, 229, 488, 169, 199, 611,
    237, 46, 885, 164, 975, 433,
    939, 536, 9, 21, 362, 669,
    815, 943, 900, 621, 647, 544,
    982, 427, 559, 882, 82, 281,
    434, 567, 407, 559, 793, 833,
    551, 950, 551, 245, 893, 313,
    823, 826, 65, 260, 684, 66,
    759, 670, 129, 507, 255, 715,
    619, 633, 320, 603, 510, 983,
    108, 622, 113, 611, 695, 44,
    645, 683, 137, 239, 622, 399,
    737, 367, 558, 589, 359, 469,
    665, 939, 386, 240, 302, 503,
    357, 130, 402, 284, 268, 619,
    936, 228, 805, 502, 723, 177,
    696, 152, 169, 914, 738, 445,
    225, 883, 941, 635, 129, 802,
    // Bin capacity
    2284, 2131, 1696, 1744, 2002, 1930
  };
  const int i6_18_6_31_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    220, 171, 943, 292, 283, 798,
    745, 476, 791, 101, 120, 826,
    461, 758, 544, 545, 578, 576,
    860, 830, 302, 484, 98, 632,
    666, 261, 353, 608, 621, 30,
    482, 545, 860, 366, 603, 490,
    644, 868, 391, 694, 296, 194,
    461, 977, 916, 756, 19, 422,
    768, 353, 462, 682, 253, 362,
    122, 258, 345, 360, 532, 734,
    64, 66, 813, 916, 264, 344,
    728, 89, 440, 224, 925, 999,
    121, 843, 774, 75, 474, 596,
    6, 336, 265, 954, 602, 649,
    726, 764, 214, 447, 366, 466,
    724, 384, 686, 607, 866, 50,
    403, 935, 812, 315, 157, 674,
    688, 289, 951, 545, 241, 273,
    // Bin capacity
    1941, 2010, 2372, 1959, 1594, 1991
  };
  const int i6_18_6_32_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    524, 861, 778, 972, 903, 547,
    308, 831, 703, 385, 908, 549,
    994, 649, 231, 819, 435, 89,
    344, 785, 553, 27, 229, 527,
    172, 75, 438, 0, 990, 80,
    175, 18, 725, 151, 603, 955,
    862, 751, 886, 287, 612, 521,
    287, 197, 913, 727, 293, 386,
    284, 649, 302, 444, 101, 75,
    359, 546, 573, 279, 629, 723,
    42, 902, 233, 242, 633, 869,
    432, 268, 644, 980, 634, 766,
    500, 813, 389, 688, 961, 682,
    440, 281, 599, 152, 718, 277,
    810, 766, 2, 564, 114, 302,
    582, 680, 706, 770, 497, 970,
    292, 210, 336, 443, 612, 925,
    778, 893, 479, 298, 780, 428,
    // Bin capacity
    1801, 2239, 2088, 1811, 2344, 2128
  };
  const int i6_18_6_33_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    510, 887, 116, 92, 504, 358,
    881, 631, 375, 125, 907, 232,
    606, 22, 652, 261, 290, 490,
    942, 521, 835, 93, 458, 588,
    128, 701, 601, 599, 963, 167,
    615, 515, 729, 87, 677, 708,
    170, 259, 614, 764, 683, 382,
    249, 752, 312, 671, 216, 521,
    894, 440, 792, 61, 992, 680,
    910, 610, 719, 185, 691, 732,
    873, 439, 828, 865, 827, 433,
    684, 98, 465, 794, 721, 23,
    406, 723, 174, 167, 374, 176,
    176, 747, 924, 322, 52, 422,
    343, 0, 215, 478, 503, 40,
    565, 713, 487, 993, 464, 101,
    630, 749, 498, 274, 645, 479,
    884, 533, 463, 134, 745, 313,
    // Bin capacity
    2320, 2071, 2173, 1544, 2375, 1518
  };
  const int i6_18_6_34_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    956, 451, 287, 943, 872, 855,
    386, 722, 269, 475, 318, 89,
    915, 789, 770, 804, 650, 769,
    68, 347, 258, 400, 299, 601,
    91, 406, 786, 132, 485, 975,
    741, 31, 108, 192, 207, 226,
    638, 714, 486, 599, 106, 245,
    703, 859, 125, 709, 614, 358,
    726, 22, 570, 545, 927, 73,
    938, 347, 801, 348, 561, 544,
    254, 663, 156, 698, 470, 447,
    87, 601, 167, 254, 587, 646,
    530, 921, 261, 8, 474, 588,
    609, 692, 328, 126, 445, 603,
    756, 348, 10, 599, 861, 265,
    476, 544, 646, 54, 869, 371,
    5, 545, 211, 751, 54, 65,
    588, 521, 280, 444, 406, 64,
    // Bin capacity
    2115, 2127, 1456, 1805, 2056, 1739
  };
  const int i6_18_6_35_29[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    259, 656, 147, 820, 836, 570,
    211, 75, 139, 24, 665, 20,
    363, 916, 150, 488, 974, 800,
    962, 591, 978, 969, 245, 473,
    714, 589, 95, 637, 723, 741,
    524, 380, 494, 175, 146, 402,
    289, 371, 834, 927, 805, 651,
    22, 630, 14, 815, 679, 148,
    59, 408, 439, 671, 874, 167,
    522, 972, 360, 267, 130, 504,
    68, 482, 802, 714, 507, 819,
    467, 971, 18, 951, 356, 99,
    38, 283, 719, 24, 197, 59,
    470, 916, 950, 929, 78, 924,
    767, 148, 22, 677, 574, 346,
    39, 528, 98, 378, 170, 911,
    645, 969, 842, 901, 182, 10,
    269, 339, 559, 214, 758, 973,
    // Bin capacity
    1505, 2301, 1724, 2381, 2003, 1939
  };
  const int i6_18_6_0_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    193, 196, 450, 429, 107, 939,
    158, 645, 18, 651, 490, 319,
    383, 376, 510, 659, 436, 779,
    122, 8, 724, 516, 538, 402,
    268, 890, 542, 507, 948, 223,
    702, 409, 454, 951, 240, 88,
    612, 813, 147, 871, 672, 920,
    876, 98, 761, 936, 887, 960,
    901, 937, 115, 153, 308, 877,
    801, 871, 962, 58, 483, 47,
    265, 413, 477, 210, 569, 726,
    212, 676, 866, 926, 41, 700,
    118, 455, 967, 618, 354, 393,
    883, 632, 858, 362, 267, 268,
    768, 51, 182, 954, 502, 410,
    808, 141, 837, 502, 743, 465,
    790, 78, 348, 197, 251, 44,
    690, 805, 589, 849, 26, 240,
    // Bin capacity
    1592, 1416, 1635, 1725, 1311, 1467
  };
  const int i6_18_6_1_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    142, 175, 269, 165, 688, 860,
    437, 203, 928, 629, 491, 561,
    350, 663, 772, 589, 103, 675,
    840, 888, 782, 365, 246, 363,
    175, 107, 777, 817, 737, 250,
    828, 221, 10, 103, 961, 454,
    43, 700, 652, 519, 198, 559,
    210, 961, 434, 126, 771, 509,
    287, 558, 307, 945, 543, 518,
    104, 528, 252, 695, 238, 502,
    918, 305, 525, 250, 18, 782,
    696, 922, 817, 703, 258, 292,
    455, 567, 520, 989, 436, 117,
    272, 139, 874, 890, 158, 532,
    150, 61, 459, 400, 418, 649,
    194, 503, 284, 77, 23, 35,
    304, 98, 124, 305, 993, 432,
    938, 62, 247, 758, 377, 344,
    // Bin capacity
    1237, 1290, 1521, 1570, 1289, 1420
  };
  const int i6_18_6_2_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    443, 999, 676, 39, 600, 624,
    401, 327, 230, 732, 519, 402,
    991, 473, 422, 963, 391, 376,
    382, 146, 175, 737, 50, 99,
    58, 458, 367, 751, 93, 747,
    62, 477, 261, 987, 945, 255,
    70, 317, 215, 109, 552, 887,
    293, 94, 183, 900, 48, 96,
    982, 469, 218, 762, 704, 936,
    441, 12, 997, 405, 772, 89,
    56, 160, 55, 14, 911, 319,
    897, 805, 113, 449, 809, 143,
    171, 879, 80, 590, 60, 279,
    864, 87, 376, 227, 940, 917,
    873, 960, 958, 635, 929, 796,
    594, 579, 286, 490, 385, 490,
    205, 777, 717, 490, 985, 770,
    277, 633, 254, 492, 233, 757,
    // Bin capacity
    1371, 1471, 1120, 1662, 1688, 1527
  };
  const int i6_18_6_3_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    896, 975, 224, 81, 182, 438,
    987, 667, 810, 164, 613, 803,
    276, 272, 758, 712, 699, 979,
    498, 294, 582, 440, 536, 742,
    624, 367, 682, 397, 332, 641,
    728, 589, 647, 308, 529, 524,
    902, 469, 341, 449, 749, 64,
    691, 759, 577, 391, 366, 315,
    30, 345, 895, 480, 422, 974,
    155, 148, 546, 986, 797, 866,
    463, 77, 403, 267, 362, 583,
    5, 411, 883, 819, 282, 862,
    289, 119, 365, 793, 623, 157,
    953, 595, 820, 921, 196, 772,
    856, 948, 396, 976, 338, 259,
    908, 980, 905, 342, 542, 904,
    30, 806, 945, 285, 200, 481,
    503, 516, 817, 258, 205, 996,
    // Bin capacity
    1682, 1603, 1991, 1557, 1369, 1951
  };
  const int i6_18_6_4_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    568, 402, 512, 128, 938, 841,
    171, 470, 997, 820, 513, 710,
    946, 800, 691, 628, 748, 459,
    75, 676, 198, 751, 117, 920,
    948, 757, 534, 152, 928, 310,
    626, 761, 149, 746, 51, 896,
    779, 224, 49, 756, 352, 950,
    653, 759, 731, 385, 281, 119,
    308, 649, 594, 82, 22, 228,
    270, 789, 487, 179, 497, 76,
    840, 285, 235, 847, 270, 488,
    166, 826, 58, 114, 225, 184,
    785, 283, 168, 373, 14, 922,
    557, 61, 813, 14, 78, 258,
    5, 322, 10, 711, 261, 6,
    335, 649, 497, 422, 930, 935,
    435, 715, 556, 830, 608, 103,
    84, 246, 10, 836, 169, 169,
    // Bin capacity
    1483, 1677, 1264, 1521, 1214, 1487
  };
  const int i6_18_6_5_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    987, 129, 797, 393, 36, 591,
    951, 967, 772, 373, 61, 453,
    471, 858, 955, 361, 709, 755,
    674, 212, 637, 625, 757, 574,
    491, 320, 575, 390, 759, 120,
    202, 432, 854, 134, 681, 649,
    276, 88, 437, 832, 719, 252,
    13, 173, 613, 908, 140, 496,
    849, 914, 462, 958, 555, 136,
    525, 829, 365, 476, 249, 664,
    342, 275, 694, 647, 606, 188,
    187, 306, 482, 295, 784, 788,
    690, 401, 989, 421, 334, 228,
    764, 207, 137, 680, 852, 993,
    182, 747, 511, 352, 915, 684,
    943, 141, 650, 607, 745, 174,
    403, 580, 812, 839, 239, 652,
    535, 417, 552, 114, 49, 321,
    // Bin capacity
    1660, 1400, 1977, 1646, 1609, 1526
  };
  const int i6_18_6_6_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    888, 543, 704, 651, 410, 209,
    102, 155, 972, 995, 177, 203,
    11, 460, 562, 725, 635, 996,
    173, 146, 943, 762, 670, 678,
    68, 812, 276, 93, 720, 522,
    347, 622, 810, 696, 29, 981,
    415, 3, 136, 742, 490, 928,
    626, 794, 933, 53, 72, 521,
    686, 419, 393, 870, 419, 349,
    398, 720, 667, 971, 543, 824,
    454, 321, 370, 978, 814, 954,
    13, 563, 726, 495, 284, 213,
    280, 906, 420, 46, 679, 561,
    412, 247, 947, 772, 13, 736,
    651, 630, 159, 321, 17, 557,
    881, 118, 692, 39, 373, 687,
    751, 428, 818, 435, 106, 82,
    912, 816, 194, 36, 958, 239,
    // Bin capacity
    1426, 1538, 1895, 1711, 1309, 1810
  };
  const int i6_18_6_7_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    971, 467, 301, 905, 508, 152,
    797, 987, 150, 596, 437, 15,
    147, 382, 771, 818, 601, 831,
    743, 607, 692, 984, 614, 535,
    168, 169, 680, 500, 217, 860,
    906, 448, 242, 933, 609, 347,
    68, 814, 593, 895, 354, 369,
    919, 850, 369, 175, 509, 31,
    139, 732, 27, 500, 318, 331,
    84, 616, 9, 720, 210, 157,
    563, 528, 354, 502, 137, 168,
    351, 534, 46, 433, 133, 264,
    794, 838, 978, 372, 514, 720,
    389, 183, 764, 702, 257, 859,
    898, 325, 174, 972, 294, 361,
    631, 529, 186, 929, 905, 435,
    599, 0, 706, 683, 861, 186,
    379, 987, 773, 284, 790, 426,
    // Bin capacity
    1703, 1783, 1394, 2123, 1475, 1257
  };
  const int i6_18_6_8_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    142, 865, 518, 658, 724, 792,
    327, 237, 606, 1000, 630, 298,
    3, 502, 124, 831, 364, 935,
    702, 213, 320, 632, 329, 66,
    750, 87, 79, 678, 789, 478,
    749, 546, 616, 575, 531, 244,
    539, 769, 976, 192, 59, 718,
    173, 547, 739, 913, 212, 224,
    810, 60, 725, 998, 618, 460,
    494, 967, 290, 804, 418, 212,
    146, 153, 548, 544, 68, 413,
    622, 584, 695, 849, 995, 545,
    31, 370, 484, 274, 922, 576,
    384, 339, 265, 372, 391, 131,
    581, 221, 263, 706, 964, 163,
    868, 482, 157, 198, 854, 900,
    299, 135, 32, 912, 809, 788,
    436, 341, 721, 457, 239, 368,
    // Bin capacity
    1451, 1336, 1469, 2087, 1785, 1496
  };
  const int i6_18_6_9_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    797, 923, 581, 753, 798, 559,
    577, 842, 707, 689, 836, 367,
    895, 752, 144, 168, 92, 206,
    666, 241, 276, 224, 699, 311,
    731, 319, 472, 189, 6, 271,
    744, 314, 589, 753, 813, 618,
    239, 647, 121, 917, 612, 576,
    665, 85, 336, 187, 390, 630,
    733, 130, 703, 139, 443, 644,
    976, 762, 604, 237, 970, 320,
    493, 541, 965, 439, 594, 418,
    713, 579, 737, 727, 331, 188,
    633, 622, 348, 586, 762, 87,
    961, 69, 47, 743, 700, 319,
    344, 638, 479, 678, 633, 949,
    247, 75, 687, 434, 634, 640,
    604, 943, 555, 820, 294, 936,
    85, 850, 914, 169, 903, 675,
    // Bin capacity
    2018, 1696, 1684, 1609, 1910, 1584
  };
  const int i6_18_6_10_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    116, 422, 599, 10, 245, 846,
    394, 984, 761, 315, 588, 744,
    678, 676, 739, 345, 962, 759,
    198, 743, 197, 962, 43, 427,
    304, 392, 170, 452, 649, 185,
    524, 263, 541, 782, 642, 479,
    954, 462, 310, 329, 341, 18,
    345, 488, 288, 334, 794, 792,
    106, 99, 990, 987, 411, 395,
    193, 987, 311, 903, 839, 478,
    452, 746, 931, 825, 503, 112,
    83, 942, 343, 994, 477, 965,
    922, 332, 574, 361, 237, 270,
    234, 718, 708, 90, 302, 664,
    78, 26, 441, 944, 222, 321,
    232, 825, 139, 73, 20, 795,
    846, 376, 650, 678, 720, 987,
    622, 777, 739, 119, 940, 36,
    // Bin capacity
    1335, 1881, 1730, 1743, 1639, 1701
  };
  const int i6_18_6_11_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    201, 21, 149, 301, 33, 549,
    864, 519, 633, 645, 566, 45,
    611, 337, 383, 993, 296, 470,
    347, 304, 756, 907, 48, 831,
    756, 277, 151, 937, 32, 688,
    531, 652, 612, 801, 421, 546,
    430, 736, 877, 449, 446, 803,
    155, 736, 326, 20, 307, 669,
    711, 630, 293, 633, 260, 598,
    916, 571, 738, 133, 947, 498,
    880, 393, 189, 30, 465, 118,
    389, 933, 344, 664, 751, 641,
    890, 811, 139, 971, 557, 696,
    759, 475, 692, 386, 690, 242,
    583, 539, 414, 596, 686, 169,
    627, 33, 498, 83, 982, 649,
    658, 98, 810, 16, 165, 684,
    538, 172, 864, 130, 396, 347,
    // Bin capacity
    2007, 1524, 1641, 1609, 1489, 1710
  };
  const int i6_18_6_12_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    315, 97, 61, 948, 246, 45,
    763, 466, 81, 235, 982, 909,
    145, 577, 273, 802, 421, 505,
    63, 773, 437, 622, 440, 266,
    615, 149, 273, 975, 66, 877,
    859, 678, 824, 6, 81, 934,
    938, 94, 358, 602, 915, 404,
    790, 424, 905, 767, 795, 314,
    519, 738, 259, 346, 449, 135,
    626, 424, 685, 90, 99, 977,
    227, 231, 994, 835, 628, 233,
    159, 471, 377, 686, 460, 388,
    767, 138, 757, 480, 142, 568,
    630, 162, 241, 12, 916, 147,
    35, 111, 558, 409, 62, 940,
    74, 700, 493, 400, 230, 84,
    787, 464, 539, 546, 720, 33,
    399, 962, 37, 366, 635, 532,
    // Bin capacity
    1627, 1430, 1522, 1704, 1547, 1548
  };
  const int i6_18_6_13_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    817, 454, 259, 154, 76, 360,
    301, 632, 986, 454, 922, 869,
    511, 632, 705, 275, 677, 300,
    50, 936, 77, 80, 94, 814,
    920, 711, 389, 608, 167, 854,
    242, 162, 984, 880, 631, 988,
    307, 367, 807, 903, 377, 449,
    705, 524, 975, 882, 335, 143,
    901, 113, 541, 472, 147, 253,
    962, 218, 202, 142, 867, 146,
    304, 32, 884, 398, 235, 225,
    947, 895, 10, 671, 631, 876,
    168, 667, 954, 110, 555, 941,
    360, 806, 641, 70, 406, 589,
    898, 701, 894, 379, 625, 825,
    614, 604, 222, 212, 468, 857,
    291, 239, 449, 586, 281, 362,
    783, 167, 332, 758, 81, 871,
    // Bin capacity
    1899, 1669, 1942, 1514, 1427, 2020
  };
  const int i6_18_6_14_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    639, 559, 750, 463, 853, 484,
    594, 196, 982, 864, 220, 663,
    511, 970, 280, 573, 235, 906,
    402, 849, 261, 824, 973, 642,
    437, 259, 364, 573, 239, 522,
    917, 776, 446, 388, 867, 73,
    189, 882, 552, 680, 942, 733,
    805, 723, 806, 870, 540, 168,
    31, 540, 660, 25, 266, 684,
    303, 621, 209, 402, 65, 126,
    650, 360, 93, 542, 160, 99,
    565, 263, 872, 485, 279, 765,
    151, 268, 267, 699, 886, 301,
    785, 544, 121, 893, 886, 154,
    638, 503, 637, 416, 778, 201,
    753, 383, 424, 62, 528, 118,
    141, 178, 10, 239, 556, 11,
    796, 724, 870, 939, 530, 177,
    // Bin capacity
    1769, 1824, 1635, 1889, 1863, 1298
  };
  const int i6_18_6_15_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    412, 530, 887, 129, 152, 970,
    352, 859, 175, 713, 137, 456,
    97, 321, 355, 354, 773, 550,
    775, 571, 335, 820, 546, 479,
    516, 851, 384, 391, 282, 546,
    552, 72, 548, 252, 787, 888,
    89, 613, 654, 290, 72, 23,
    582, 983, 99, 200, 18, 97,
    852, 737, 230, 893, 971, 944,
    910, 684, 867, 588, 738, 282,
    28, 976, 669, 694, 930, 869,
    765, 950, 720, 145, 975, 979,
    292, 439, 6, 170, 871, 894,
    828, 630, 917, 993, 468, 663,
    616, 266, 56, 716, 385, 454,
    103, 337, 882, 704, 959, 301,
    475, 873, 198, 373, 309, 918,
    359, 699, 412, 499, 40, 775,
    // Bin capacity
    1649, 2184, 1609, 1711, 1805, 2126
  };
  const int i6_18_6_16_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    614, 564, 647, 781, 83, 107,
    772, 740, 202, 231, 713, 887,
    55, 248, 62, 651, 650, 435,
    247, 998, 225, 605, 419, 386,
    578, 859, 99, 471, 690, 453,
    847, 177, 194, 623, 495, 80,
    498, 677, 334, 476, 528, 809,
    141, 965, 911, 774, 551, 71,
    866, 605, 644, 790, 703, 883,
    427, 3, 651, 421, 197, 293,
    48, 204, 640, 975, 925, 575,
    596, 833, 693, 231, 540, 527,
    664, 565, 139, 405, 477, 581,
    613, 523, 7, 848, 36, 163,
    770, 550, 629, 410, 339, 216,
    863, 886, 48, 187, 582, 825,
    91, 892, 535, 951, 956, 714,
    491, 476, 395, 518, 363, 259,
    // Bin capacity
    1775, 2082, 1364, 2001, 1788, 1598
  };
  const int i6_18_6_17_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    226, 847, 81, 917, 852, 989,
    71, 747, 126, 590, 589, 183,
    276, 478, 886, 580, 969, 153,
    931, 197, 843, 129, 334, 620,
    521, 0, 737, 556, 387, 404,
    906, 62, 655, 319, 923, 8,
    575, 508, 428, 540, 76, 63,
    408, 987, 201, 679, 915, 219,
    698, 427, 549, 822, 612, 770,
    920, 672, 610, 774, 729, 92,
    279, 21, 523, 906, 558, 545,
    786, 378, 863, 467, 402, 61,
    402, 182, 608, 829, 71, 171,
    802, 248, 392, 753, 161, 706,
    941, 223, 917, 708, 193, 903,
    235, 594, 590, 802, 523, 441,
    61, 383, 334, 161, 19, 947,
    192, 616, 257, 99, 837, 17,
    // Bin capacity
    1800, 1477, 1872, 2074, 1785, 1422
  };
  const int i6_18_6_18_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    779, 922, 256, 512, 151, 948,
    163, 618, 227, 575, 406, 861,
    757, 881, 34, 74, 785, 516,
    357, 33, 540, 577, 392, 183,
    236, 962, 113, 931, 907, 605,
    43, 751, 107, 833, 737, 916,
    292, 595, 567, 795, 942, 340,
    335, 818, 410, 467, 281, 279,
    257, 133, 179, 776, 669, 536,
    672, 317, 743, 625, 297, 944,
    508, 260, 931, 197, 208, 277,
    274, 455, 412, 397, 684, 977,
    277, 84, 161, 512, 804, 819,
    735, 288, 920, 649, 905, 649,
    72, 970, 363, 330, 175, 156,
    635, 262, 28, 726, 781, 418,
    529, 264, 759, 865, 56, 998,
    243, 798, 134, 385, 929, 767,
    // Bin capacity
    1409, 1851, 1354, 2012, 1989, 2201
  };
  const int i6_18_6_19_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    706, 79, 973, 483, 986, 36,
    591, 586, 144, 663, 659, 747,
    598, 741, 134, 642, 392, 68,
    551, 132, 135, 982, 245, 57,
    301, 447, 281, 118, 407, 108,
    744, 759, 426, 310, 909, 797,
    797, 177, 165, 846, 836, 847,
    601, 404, 736, 463, 26, 887,
    978, 695, 584, 8, 238, 87,
    928, 216, 578, 20, 466, 849,
    439, 923, 267, 548, 782, 219,
    679, 237, 313, 274, 957, 570,
    95, 391, 741, 939, 584, 659,
    708, 844, 368, 138, 45, 712,
    982, 111, 534, 551, 776, 285,
    330, 314, 660, 934, 958, 635,
    418, 850, 871, 907, 322, 86,
    423, 28, 726, 687, 557, 981,
    // Bin capacity
    2156, 1574, 1713, 1887, 2013, 1712
  };
  const int i6_18_6_20_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    435, 788, 335, 522, 191, 290,
    808, 618, 344, 181, 658, 70,
    299, 11, 376, 169, 568, 822,
    547, 562, 53, 575, 156, 756,
    3, 510, 38, 296, 362, 999,
    386, 639, 849, 312, 613, 610,
    287, 744, 285, 35, 963, 196,
    644, 686, 156, 922, 44, 45,
    840, 423, 199, 516, 968, 384,
    829, 74, 200, 872, 899, 886,
    306, 907, 602, 705, 802, 118,
    849, 43, 265, 10, 927, 442,
    772, 226, 813, 324, 498, 529,
    654, 802, 128, 402, 271, 670,
    918, 819, 936, 270, 861, 945,
    414, 986, 336, 80, 717, 34,
    360, 531, 580, 393, 916, 934,
    168, 70, 170, 925, 504, 458,
    // Bin capacity
    1904, 1888, 1333, 1502, 2184, 1838
  };
  const int i6_18_6_21_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    29, 742, 953, 666, 257, 144,
    408, 973, 697, 261, 0, 340,
    376, 573, 124, 451, 449, 805,
    772, 67, 666, 69, 459, 197,
    859, 657, 679, 650, 671, 776,
    778, 279, 819, 295, 206, 359,
    147, 46, 997, 459, 650, 303,
    677, 493, 752, 395, 350, 148,
    477, 848, 934, 8, 218, 217,
    987, 838, 274, 935, 541, 539,
    646, 952, 726, 964, 302, 530,
    448, 349, 394, 776, 667, 900,
    787, 299, 277, 475, 175, 775,
    380, 48, 967, 874, 139, 267,
    482, 13, 70, 625, 352, 412,
    663, 570, 138, 436, 48, 430,
    351, 216, 43, 753, 842, 246,
    857, 274, 479, 383, 916, 226,
    // Bin capacity
    2042, 1662, 2015, 1911, 1461, 1536
  };
  const int i6_18_6_22_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    924, 679, 296, 114, 104, 884,
    194, 849, 494, 353, 754, 161,
    773, 633, 30, 601, 569, 87,
    30, 95, 89, 229, 557, 312,
    977, 397, 953, 104, 5, 777,
    788, 216, 995, 722, 955, 287,
    322, 757, 713, 492, 422, 666,
    835, 566, 209, 871, 497, 797,
    964, 590, 359, 548, 917, 154,
    66, 903, 251, 997, 425, 303,
    777, 439, 837, 258, 720, 278,
    666, 129, 707, 267, 848, 817,
    698, 703, 656, 42, 975, 388,
    37, 759, 522, 712, 329, 806,
    842, 264, 477, 121, 230, 53,
    592, 462, 987, 446, 340, 853,
    310, 540, 142, 238, 509, 232,
    974, 543, 159, 44, 140, 647,
    // Bin capacity
    2190, 1937, 1805, 1456, 1891, 1729
  };
  const int i6_18_6_23_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    176, 225, 266, 136, 376, 616,
    360, 836, 38, 889, 824, 633,
    825, 232, 829, 519, 789, 840,
    483, 127, 286, 557, 955, 836,
    482, 0, 961, 923, 565, 597,
    554, 212, 161, 32, 507, 326,
    713, 395, 815, 576, 703, 323,
    382, 808, 570, 178, 197, 183,
    1000, 902, 163, 502, 460, 715,
    348, 286, 356, 127, 778, 577,
    767, 824, 646, 969, 511, 867,
    462, 882, 305, 776, 728, 226,
    11, 579, 261, 67, 790, 84,
    806, 169, 77, 947, 94, 215,
    539, 119, 768, 643, 858, 694,
    999, 609, 33, 119, 649, 942,
    404, 689, 797, 41, 979, 820,
    802, 306, 136, 909, 619, 849,
    // Bin capacity
    2074, 1681, 1531, 1827, 2334, 2121
  };
  const int i6_18_6_24_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    658, 957, 513, 576, 653, 660,
    581, 441, 763, 822, 309, 552,
    436, 884, 103, 874, 71, 990,
    64, 85, 417, 647, 490, 365,
    410, 299, 505, 264, 672, 140,
    10, 384, 274, 487, 817, 4,
    948, 683, 261, 539, 615, 292,
    959, 976, 589, 974, 555, 656,
    467, 936, 858, 754, 680, 641,
    684, 267, 330, 305, 662, 56,
    549, 391, 831, 551, 92, 649,
    100, 119, 761, 96, 813, 781,
    215, 317, 559, 740, 315, 341,
    636, 72, 190, 570, 337, 859,
    527, 292, 591, 201, 831, 72,
    580, 467, 863, 821, 809, 430,
    410, 366, 268, 499, 747, 225,
    890, 819, 17, 854, 676, 492,
    // Bin capacity
    1886, 1810, 1797, 2186, 2097, 1696
  };
  const int i6_18_6_25_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    355, 333, 531, 362, 139, 659,
    707, 85, 450, 990, 974, 892,
    747, 445, 742, 540, 891, 677,
    144, 575, 931, 335, 169, 409,
    128, 797, 495, 180, 839, 558,
    371, 726, 360, 376, 384, 827,
    828, 878, 203, 959, 356, 175,
    174, 668, 797, 354, 484, 441,
    729, 393, 229, 14, 136, 608,
    455, 576, 560, 73, 403, 725,
    440, 239, 651, 892, 690, 692,
    810, 397, 287, 594, 754, 714,
    848, 205, 93, 323, 656, 17,
    3, 645, 331, 83, 240, 781,
    551, 126, 877, 32, 582, 316,
    339, 397, 993, 978, 875, 42,
    579, 900, 619, 995, 918, 591,
    743, 128, 589, 11, 19, 369,
    // Bin capacity
    1865, 1774, 2029, 1686, 1982, 1978
  };
  const int i6_18_6_26_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    587, 758, 250, 24, 891, 890,
    909, 689, 312, 473, 165, 445,
    345, 81, 919, 112, 316, 328,
    390, 631, 423, 856, 913, 309,
    991, 711, 361, 520, 419, 217,
    673, 639, 866, 423, 201, 767,
    920, 717, 833, 251, 110, 547,
    536, 300, 161, 129, 733, 946,
    173, 575, 519, 639, 836, 712,
    656, 312, 382, 894, 533, 647,
    925, 27, 515, 753, 839, 153,
    282, 637, 965, 547, 539, 554,
    844, 772, 408, 51, 934, 706,
    605, 490, 918, 890, 310, 714,
    462, 459, 43, 315, 254, 264,
    20, 955, 118, 443, 30, 203,
    701, 177, 77, 48, 454, 887,
    986, 702, 436, 743, 806, 134,
    // Bin capacity
    2312, 2023, 1787, 1704, 1950, 1979
  };
  const int i6_18_6_27_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    941, 155, 413, 58, 695, 526,
    245, 286, 531, 376, 618, 247,
    57, 856, 213, 279, 646, 340,
    781, 197, 572, 187, 687, 914,
    76, 376, 106, 231, 771, 936,
    579, 278, 622, 513, 324, 188,
    343, 279, 415, 232, 96, 239,
    452, 982, 932, 160, 921, 661,
    1000, 959, 362, 734, 426, 936,
    100, 159, 757, 1000, 555, 548,
    908, 430, 506, 338, 87, 595,
    449, 37, 445, 740, 412, 820,
    667, 38, 170, 679, 322, 588,
    656, 409, 436, 434, 927, 553,
    400, 488, 752, 251, 976, 7,
    365, 377, 872, 293, 893, 529,
    488, 776, 464, 334, 962, 945,
    936, 669, 326, 368, 931, 165,
    // Bin capacity
    1999, 1641, 1883, 1526, 2382, 2061
  };
  const int i6_18_6_28_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    717, 417, 913, 844, 372, 278,
    753, 924, 164, 169, 702, 449,
    11, 369, 410, 971, 184, 958,
    239, 306, 252, 353, 665, 848,
    189, 722, 722, 422, 241, 719,
    78, 943, 351, 993, 529, 469,
    293, 457, 685, 86, 264, 838,
    37, 781, 895, 472, 554, 292,
    479, 519, 911, 711, 914, 343,
    138, 647, 792, 975, 185, 341,
    546, 66, 853, 757, 384, 894,
    189, 755, 770, 511, 547, 519,
    3, 585, 541, 270, 176, 428,
    541, 521, 26, 761, 575, 20,
    383, 971, 846, 369, 114, 332,
    617, 632, 264, 825, 829, 584,
    240, 677, 795, 494, 524, 689,
    333, 955, 559, 352, 870, 387,
    // Bin capacity
    1235, 2400, 2294, 2205, 1841, 2003
  };
  const int i6_18_6_29_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    279, 326, 926, 210, 440, 977,
    220, 780, 706, 402, 888, 570,
    685, 302, 613, 141, 154, 118,
    645, 135, 16, 16, 453, 525,
    897, 168, 886, 98, 843, 943,
    399, 42, 564, 135, 604, 112,
    58, 311, 832, 246, 917, 359,
    913, 681, 924, 899, 626, 313,
    857, 500, 984, 56, 398, 695,
    668, 934, 221, 373, 536, 626,
    268, 176, 130, 422, 246, 457,
    427, 234, 552, 708, 410, 280,
    255, 420, 376, 801, 325, 165,
    723, 409, 919, 526, 643, 190,
    326, 485, 667, 780, 259, 650,
    838, 836, 726, 337, 90, 460,
    428, 121, 102, 93, 1, 490,
    424, 267, 451, 578, 414, 183,
    // Bin capacity
    2002, 1533, 2278, 1467, 1774, 1745
  };
  const int i6_18_6_30_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    492, 287, 690, 388, 339, 640,
    929, 126, 125, 621, 936, 734,
    726, 650, 468, 291, 426, 189,
    694, 908, 215, 991, 44, 21,
    836, 219, 437, 693, 561, 366,
    400, 144, 347, 52, 567, 896,
    169, 636, 737, 773, 472, 818,
    504, 285, 849, 903, 963, 717,
    894, 452, 925, 770, 709, 92,
    457, 805, 290, 717, 529, 682,
    140, 599, 178, 190, 876, 157,
    905, 247, 289, 592, 286, 755,
    356, 22, 29, 425, 799, 620,
    413, 4, 7, 853, 819, 735,
    924, 391, 436, 414, 396, 227,
    610, 828, 750, 953, 10, 611,
    261, 687, 150, 75, 903, 391,
    285, 858, 562, 810, 155, 142,
    // Bin capacity
    2166, 1766, 1622, 2278, 2122, 1906
  };
  const int i6_18_6_31_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    949, 519, 620, 522, 880, 445,
    474, 256, 852, 370, 755, 224,
    671, 613, 507, 739, 102, 894,
    575, 574, 597, 781, 5, 212,
    881, 610, 56, 591, 387, 833,
    840, 930, 696, 90, 343, 399,
    445, 367, 339, 219, 896, 992,
    887, 723, 246, 781, 976, 488,
    510, 839, 520, 711, 638, 735,
    782, 528, 748, 936, 648, 518,
    884, 329, 527, 444, 836, 651,
    698, 736, 189, 257, 257, 595,
    444, 323, 739, 559, 115, 461,
    0, 367, 318, 388, 85, 768,
    366, 685, 6, 220, 413, 97,
    403, 493, 793, 654, 94, 918,
    382, 414, 273, 94, 332, 244,
    517, 877, 802, 952, 859, 851,
    // Bin capacity
    2338, 2224, 1928, 2033, 1883, 2255
  };
  const int i6_18_6_32_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    824, 437, 630, 373, 734, 671,
    801, 376, 320, 393, 730, 930,
    897, 263, 380, 696, 866, 455,
    66, 718, 447, 170, 373, 82,
    855, 980, 191, 541, 602, 738,
    117, 270, 491, 413, 666, 8,
    609, 938, 561, 395, 620, 428,
    968, 644, 215, 413, 166, 156,
    617, 63, 11, 343, 754, 125,
    116, 617, 326, 210, 899, 889,
    652, 201, 718, 602, 179, 475,
    397, 723, 608, 893, 183, 77,
    684, 716, 547, 731, 790, 325,
    710, 381, 428, 498, 143, 470,
    82, 729, 591, 551, 152, 878,
    943, 639, 689, 498, 145, 673,
    292, 758, 564, 965, 981, 396,
    263, 178, 238, 921, 731, 298,
    // Bin capacity
    2177, 2119, 1751, 2114, 2138, 1777
  };
  const int i6_18_6_33_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    166, 865, 100, 184, 184, 327,
    110, 720, 165, 187, 511, 36,
    772, 823, 983, 284, 689, 775,
    313, 818, 145, 911, 577, 629,
    564, 451, 733, 700, 761, 519,
    181, 303, 224, 124, 965, 586,
    645, 367, 711, 756, 107, 742,
    235, 947, 325, 176, 965, 778,
    772, 335, 738, 361, 154, 430,
    905, 627, 764, 736, 911, 282,
    212, 93, 262, 557, 31, 927,
    503, 732, 884, 746, 786, 677,
    894, 952, 202, 288, 640, 285,
    433, 473, 688, 29, 79, 870,
    510, 335, 222, 263, 709, 203,
    462, 810, 96, 202, 763, 310,
    258, 950, 670, 990, 242, 863,
    900, 448, 298, 459, 732, 523,
    // Bin capacity
    1959, 2450, 1820, 1763, 2174, 2164
  };
  const int i6_18_6_34_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    583, 463, 606, 697, 710, 276,
    778, 869, 673, 451, 101, 976,
    160, 215, 810, 326, 645, 592,
    975, 767, 478, 191, 526, 239,
    436, 992, 121, 477, 743, 830,
    704, 742, 744, 309, 577, 537,
    909, 993, 527, 52, 422, 121,
    730, 396, 745, 542, 235, 446,
    792, 668, 724, 973, 141, 693,
    169, 165, 287, 578, 560, 966,
    571, 68, 545, 270, 585, 390,
    436, 792, 482, 655, 708, 780,
    673, 461, 556, 397, 58, 994,
    62, 694, 819, 812, 377, 251,
    731, 689, 864, 933, 544, 38,
    213, 277, 826, 500, 698, 972,
    359, 670, 113, 283, 332, 958,
    919, 391, 357, 991, 152, 514,
    // Bin capacity
    2278, 2304, 2296, 2108, 1813, 2362
  };
  const int i6_18_6_35_30[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    981, 233, 687, 603, 594, 949,
    467, 944, 319, 405, 362, 538,
    738, 549, 682, 267, 738, 340,
    173, 574, 206, 271, 317, 603,
    174, 171, 179, 410, 720, 371,
    155, 958, 722, 657, 400, 574,
    538, 77, 256, 384, 313, 142,
    174, 296, 380, 96, 390, 452,
    448, 333, 918, 423, 305, 90,
    811, 333, 9, 723, 819, 898,
    809, 236, 242, 385, 709, 60,
    609, 740, 746, 319, 589, 919,
    771, 171, 255, 941, 708, 45,
    713, 753, 507, 429, 921, 828,
    669, 457, 496, 776, 661, 267,
    26, 443, 266, 532, 739, 56,
    901, 141, 139, 899, 10, 660,
    323, 809, 218, 965, 701, 230,
    // Bin capacity
    2133, 1850, 1627, 2135, 2250, 1805
  };
  const int i6_18_6_0_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    523, 73, 389, 955, 659, 512,
    68, 882, 68, 185, 604, 561,
    235, 870, 862, 916, 887, 14,
    230, 82, 885, 734, 291, 301,
    700, 117, 245, 751, 886, 977,
    510, 723, 87, 678, 563, 621,
    122, 240, 187, 740, 128, 871,
    696, 556, 866, 226, 971, 302,
    95, 843, 269, 557, 876, 1,
    642, 3, 700, 275, 799, 74,
    853, 928, 94, 549, 196, 72,
    882, 505, 872, 699, 48, 286,
    972, 905, 874, 181, 948, 267,
    276, 156, 739, 926, 900, 999,
    201, 745, 570, 430, 708, 152,
    57, 938, 46, 431, 808, 440,
    763, 146, 319, 31, 565, 471,
    173, 525, 482, 370, 727, 332,
    // Bin capacity
    1333, 1540, 1426, 1606, 1928, 1209
  };
  const int i6_18_6_1_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    626, 480, 834, 666, 150, 161,
    119, 598, 72, 942, 129, 278,
    991, 116, 702, 405, 736, 614,
    385, 544, 181, 649, 361, 949,
    538, 177, 358, 872, 449, 675,
    143, 494, 830, 972, 54, 190,
    558, 278, 838, 284, 177, 916,
    745, 528, 685, 102, 222, 208,
    660, 313, 287, 210, 960, 595,
    2, 948, 350, 516, 873, 683,
    301, 684, 500, 945, 915, 359,
    938, 742, 918, 216, 830, 669,
    168, 377, 561, 8, 98, 11,
    161, 112, 510, 917, 415, 675,
    809, 468, 105, 473, 373, 64,
    223, 885, 944, 172, 408, 61,
    28, 140, 331, 373, 450, 670,
    865, 111, 782, 288, 517, 955,
    // Bin capacity
    1391, 1346, 1648, 1517, 1367, 1471
  };
  const int i6_18_6_2_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    315, 574, 478, 469, 414, 151,
    661, 978, 232, 38, 693, 346,
    909, 735, 234, 5, 780, 538,
    604, 133, 892, 888, 34, 122,
    151, 685, 14, 571, 823, 78,
    79, 487, 226, 17, 414, 767,
    287, 23, 359, 336, 200, 214,
    63, 56, 749, 862, 703, 998,
    568, 666, 792, 1, 431, 186,
    906, 548, 201, 236, 498, 237,
    269, 329, 533, 24, 953, 822,
    775, 463, 862, 405, 204, 581,
    538, 511, 192, 689, 693, 705,
    482, 979, 901, 311, 652, 991,
    612, 524, 94, 230, 545, 958,
    169, 640, 197, 524, 462, 801,
    625, 577, 852, 868, 577, 26,
    212, 318, 905, 112, 785, 876,
    // Bin capacity
    1399, 1569, 1482, 1120, 1677, 1598
  };
  const int i6_18_6_3_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    382, 98, 620, 282, 930, 224,
    54, 743, 420, 435, 971, 885,
    441, 659, 673, 397, 614, 465,
    189, 391, 249, 515, 78, 528,
    799, 574, 157, 129, 294, 534,
    741, 710, 139, 547, 507, 685,
    941, 491, 404, 271, 824, 977,
    296, 315, 388, 875, 57, 844,
    498, 355, 869, 44, 111, 74,
    706, 418, 999, 157, 212, 788,
    226, 531, 140, 733, 438, 884,
    787, 678, 656, 850, 949, 328,
    326, 389, 579, 973, 435, 945,
    14, 623, 258, 280, 662, 458,
    181, 156, 414, 722, 643, 464,
    901, 989, 512, 972, 717, 217,
    923, 303, 601, 722, 331, 460,
    727, 41, 640, 555, 101, 344,
    // Bin capacity
    1568, 1453, 1497, 1624, 1524, 1735
  };
  const int i6_18_6_4_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    221, 884, 752, 18, 248, 434,
    462, 798, 905, 3, 134, 292,
    514, 20, 965, 214, 852, 44,
    47, 141, 235, 294, 1, 459,
    10, 0, 985, 957, 653, 843,
    399, 804, 957, 288, 810, 70,
    61, 256, 988, 678, 59, 508,
    869, 164, 391, 600, 342, 333,
    545, 447, 31, 484, 798, 898,
    501, 470, 843, 471, 615, 470,
    236, 649, 763, 893, 431, 210,
    486, 20, 225, 947, 253, 93,
    665, 963, 223, 531, 22, 204,
    259, 119, 351, 423, 695, 817,
    734, 186, 559, 280, 989, 136,
    389, 528, 374, 253, 483, 901,
    330, 703, 918, 297, 10, 114,
    712, 716, 423, 200, 978, 462,
    // Bin capacity
    1290, 1364, 1888, 1358, 1452, 1264
  };
  const int i6_18_6_5_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    169, 671, 24, 564, 744, 527,
    82, 571, 461, 508, 529, 66,
    464, 853, 894, 601, 794, 824,
    866, 204, 988, 362, 758, 220,
    338, 748, 424, 329, 347, 411,
    711, 734, 22, 45, 482, 81,
    153, 235, 296, 555, 614, 197,
    757, 982, 563, 879, 194, 824,
    187, 544, 463, 608, 22, 84,
    159, 689, 530, 621, 747, 761,
    753, 88, 204, 940, 455, 498,
    157, 882, 255, 255, 197, 494,
    112, 479, 891, 23, 215, 187,
    98, 662, 694, 123, 208, 691,
    188, 453, 961, 506, 736, 90,
    451, 26, 605, 273, 804, 415,
    842, 617, 972, 453, 554, 738,
    708, 93, 258, 895, 886, 329,
    // Bin capacity
    1260, 1668, 1664, 1495, 1626, 1302
  };
  const int i6_18_6_6_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    671, 167, 899, 228, 384, 629,
    780, 247, 450, 876, 74, 920,
    262, 614, 70, 228, 287, 525,
    232, 95, 565, 408, 191, 768,
    19, 865, 341, 325, 964, 892,
    652, 686, 625, 154, 397, 51,
    751, 651, 897, 152, 791, 701,
    616, 195, 587, 33, 871, 94,
    231, 985, 940, 176, 739, 486,
    854, 433, 412, 659, 822, 795,
    278, 802, 41, 10, 760, 160,
    554, 191, 516, 1, 571, 21,
    278, 549, 318, 474, 642, 327,
    490, 916, 70, 65, 749, 653,
    436, 37, 553, 354, 36, 197,
    380, 211, 744, 960, 685, 41,
    312, 154, 351, 122, 575, 162,
    668, 771, 137, 641, 962, 651,
    // Bin capacity
    1496, 1514, 1505, 1037, 1855, 1427
  };
  const int i6_18_6_7_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    897, 697, 973, 459, 613, 121,
    966, 469, 686, 92, 435, 38,
    386, 555, 789, 493, 600, 509,
    910, 651, 633, 972, 452, 557,
    694, 66, 828, 790, 717, 516,
    103, 753, 793, 320, 299, 645,
    933, 198, 597, 43, 515, 808,
    747, 750, 570, 71, 746, 23,
    885, 184, 86, 30, 225, 75,
    683, 814, 799, 462, 393, 461,
    847, 516, 766, 525, 587, 583,
    184, 726, 551, 422, 840, 343,
    159, 90, 559, 605, 303, 76,
    629, 136, 864, 105, 763, 774,
    434, 652, 875, 192, 792, 816,
    850, 172, 599, 756, 232, 961,
    834, 884, 381, 595, 741, 143,
    239, 460, 94, 686, 836, 594,
    // Bin capacity
    2030, 1565, 2041, 1359, 1800, 1435
  };
  const int i6_18_6_8_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    90, 899, 22, 867, 773, 332,
    998, 731, 363, 845, 801, 503,
    143, 403, 358, 905, 57, 295,
    395, 960, 396, 262, 371, 516,
    442, 448, 80, 301, 643, 34,
    891, 264, 379, 54, 476, 541,
    136, 117, 410, 506, 560, 577,
    972, 759, 2, 675, 317, 160,
    172, 665, 466, 543, 739, 99,
    849, 212, 317, 221, 605, 484,
    341, 472, 465, 207, 747, 645,
    466, 39, 205, 985, 885, 294,
    33, 55, 255, 837, 571, 524,
    245, 765, 936, 189, 842, 34,
    740, 437, 41, 388, 533, 709,
    449, 674, 315, 910, 656, 653,
    380, 157, 911, 153, 429, 437,
    462, 55, 41, 648, 602, 32,
    // Bin capacity
    1477, 1461, 1074, 1710, 1910, 1237
  };
  const int i6_18_6_9_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    212, 672, 305, 359, 479, 153,
    799, 496, 718, 738, 701, 527,
    184, 664, 556, 219, 351, 930,
    524, 547, 913, 36, 819, 575,
    843, 973, 201, 63, 911, 86,
    427, 562, 779, 159, 239, 551,
    793, 379, 814, 748, 760, 725,
    352, 395, 196, 559, 610, 511,
    738, 289, 98, 651, 697, 552,
    897, 808, 12, 859, 95, 660,
    495, 249, 205, 55, 75, 333,
    962, 865, 357, 209, 34, 707,
    326, 259, 52, 419, 668, 580,
    660, 225, 317, 267, 614, 750,
    974, 328, 234, 93, 710, 145,
    458, 590, 317, 253, 589, 185,
    593, 368, 185, 521, 57, 447,
    511, 740, 253, 476, 667, 105,
    // Bin capacity
    1953, 1710, 1184, 1215, 1649, 1549
  };
  const int i6_18_6_10_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    456, 954, 406, 124, 703, 102,
    824, 537, 495, 619, 432, 390,
    979, 957, 879, 333, 275, 778,
    106, 110, 637, 208, 260, 33,
    198, 617, 657, 141, 229, 856,
    126, 864, 433, 632, 377, 389,
    729, 304, 140, 841, 773, 371,
    474, 935, 433, 173, 189, 992,
    925, 188, 573, 133, 437, 530,
    99, 11, 588, 63, 290, 889,
    659, 393, 354, 605, 173, 875,
    732, 1, 202, 297, 82, 262,
    956, 207, 172, 406, 78, 522,
    224, 791, 346, 571, 40, 36,
    839, 790, 695, 648, 569, 750,
    975, 309, 25, 243, 850, 355,
    501, 501, 203, 301, 295, 533,
    578, 189, 979, 351, 629, 17,
    // Bin capacity
    1904, 1588, 1507, 1227, 1225, 1592
  };
  const int i6_18_6_11_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    700, 156, 234, 651, 438, 635,
    604, 433, 772, 723, 523, 536,
    34, 700, 786, 824, 500, 144,
    706, 286, 252, 386, 113, 151,
    679, 464, 415, 961, 191, 580,
    287, 150, 710, 450, 478, 258,
    445, 328, 423, 677, 352, 436,
    850, 516, 919, 943, 912, 543,
    244, 749, 759, 252, 151, 624,
    634, 47, 207, 353, 258, 209,
    295, 179, 468, 235, 956, 413,
    218, 57, 901, 992, 553, 305,
    280, 600, 491, 659, 388, 490,
    212, 428, 324, 722, 529, 556,
    980, 368, 899, 722, 118, 652,
    727, 988, 447, 499, 239, 719,
    904, 954, 892, 628, 600, 605,
    76, 880, 798, 112, 247, 117,
    // Bin capacity
    1642, 1533, 1979, 1996, 1397, 1476
  };
  const int i6_18_6_12_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    262, 79, 150, 689, 432, 226,
    432, 513, 946, 632, 912, 951,
    85, 402, 188, 316, 543, 519,
    54, 22, 709, 270, 749, 274,
    265, 566, 489, 116, 670, 405,
    118, 485, 864, 882, 317, 147,
    611, 71, 782, 531, 431, 698,
    365, 678, 949, 452, 416, 47,
    548, 626, 89, 850, 613, 872,
    150, 113, 785, 196, 706, 528,
    468, 425, 967, 676, 850, 216,
    431, 377, 668, 876, 804, 790,
    396, 278, 77, 470, 790, 1,
    916, 347, 999, 581, 886, 505,
    264, 283, 362, 513, 717, 719,
    85, 800, 967, 422, 673, 666,
    91, 286, 57, 861, 88, 774,
    842, 791, 784, 868, 956, 689,
    // Bin capacity
    1192, 1334, 2022, 1905, 2157, 1686
  };
  const int i6_18_6_13_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    431, 118, 463, 159, 755, 345,
    137, 600, 322, 73, 487, 31,
    336, 262, 574, 266, 682, 197,
    849, 183, 353, 763, 451, 835,
    927, 228, 939, 129, 51, 395,
    12, 429, 182, 821, 346, 560,
    558, 162, 422, 24, 596, 442,
    954, 450, 8, 851, 429, 534,
    397, 912, 975, 355, 59, 166,
    769, 685, 693, 879, 75, 723,
    947, 278, 927, 63, 526, 715,
    955, 337, 340, 803, 179, 441,
    471, 381, 519, 851, 755, 824,
    634, 329, 759, 555, 959, 70,
    627, 556, 200, 313, 620, 408,
    361, 149, 327, 139, 983, 952,
    64, 452, 350, 7, 672, 853,
    321, 149, 76, 545, 60, 292,
    // Bin capacity
    1837, 1255, 1588, 1431, 1636, 1655
  };
  const int i6_18_6_14_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    822, 600, 612, 235, 115, 879,
    422, 113, 769, 460, 384, 370,
    714, 878, 848, 215, 380, 100,
    308, 279, 586, 665, 46, 950,
    837, 255, 75, 947, 230, 851,
    7, 978, 743, 618, 771, 822,
    299, 900, 978, 265, 979, 647,
    465, 852, 894, 109, 771, 152,
    609, 944, 86, 830, 823, 280,
    834, 6, 475, 742, 865, 272,
    183, 447, 954, 329, 428, 254,
    223, 299, 699, 598, 585, 256,
    689, 873, 102, 244, 720, 371,
    300, 670, 374, 777, 250, 768,
    534, 517, 516, 681, 781, 320,
    465, 140, 617, 605, 745, 409,
    76, 952, 640, 233, 657, 944,
    857, 469, 115, 509, 500, 834,
    // Bin capacity
    1643, 1933, 1916, 1722, 1906, 1802
  };
  const int i6_18_6_15_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    951, 720, 672, 781, 581, 517,
    32, 333, 631, 172, 420, 896,
    525, 533, 437, 619, 206, 528,
    764, 947, 952, 543, 698, 466,
    811, 333, 321, 590, 616, 700,
    397, 60, 932, 927, 825, 998,
    191, 460, 225, 651, 498, 902,
    129, 745, 914, 750, 92, 392,
    314, 490, 422, 371, 633, 346,
    156, 780, 390, 692, 965, 511,
    110, 491, 19, 258, 517, 947,
    756, 785, 73, 607, 566, 609,
    980, 34, 419, 945, 642, 154,
    32, 245, 117, 315, 238, 766,
    654, 68, 17, 308, 11, 781,
    95, 443, 960, 434, 322, 670,
    51, 28, 353, 57, 195, 679,
    53, 774, 439, 643, 227, 194,
    // Bin capacity
    1342, 1585, 1590, 1853, 1582, 2120
  };
  const int i6_18_6_16_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    274, 362, 388, 585, 710, 733,
    178, 81, 437, 320, 302, 33,
    460, 467, 415, 471, 45, 304,
    29, 265, 896, 951, 501, 433,
    671, 382, 384, 711, 987, 617,
    894, 429, 84, 744, 77, 712,
    246, 319, 443, 162, 640, 216,
    817, 819, 413, 747, 130, 962,
    983, 934, 960, 444, 383, 788,
    206, 802, 592, 663, 32, 91,
    233, 857, 577, 625, 137, 410,
    52, 550, 165, 695, 643, 86,
    716, 121, 75, 334, 631, 155,
    310, 683, 262, 608, 26, 503,
    67, 264, 290, 448, 425, 552,
    722, 682, 641, 960, 509, 720,
    601, 142, 654, 163, 115, 616,
    804, 297, 279, 727, 783, 501,
    // Bin capacity
    1598, 1635, 1538, 2003, 1369, 1631
  };
  const int i6_18_6_17_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    411, 355, 721, 20, 57, 429,
    881, 615, 38, 650, 705, 695,
    776, 381, 998, 890, 889, 653,
    225, 452, 562, 221, 693, 487,
    637, 99, 884, 91, 615, 101,
    264, 238, 919, 452, 84, 673,
    608, 487, 142, 890, 100, 966,
    255, 193, 41, 260, 185, 460,
    2, 884, 916, 123, 988, 789,
    473, 230, 304, 120, 883, 41,
    281, 302, 4, 367, 820, 226,
    993, 50, 343, 817, 321, 843,
    981, 344, 694, 289, 250, 645,
    644, 483, 264, 277, 160, 24,
    934, 620, 637, 770, 533, 322,
    51, 441, 68, 44, 522, 588,
    48, 206, 760, 160, 644, 158,
    836, 704, 271, 595, 119, 88,
    // Bin capacity
    1814, 1382, 1671, 1373, 1671, 1597
  };
  const int i6_18_6_18_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    537, 846, 480, 624, 9, 535,
    598, 543, 804, 708, 293, 869,
    463, 160, 338, 295, 981, 985,
    82, 787, 853, 736, 221, 817,
    808, 391, 669, 23, 675, 335,
    914, 832, 640, 488, 295, 377,
    991, 111, 622, 876, 547, 132,
    234, 131, 139, 482, 553, 523,
    252, 666, 367, 203, 708, 456,
    738, 707, 11, 859, 384, 172,
    174, 408, 510, 881, 778, 663,
    999, 831, 441, 788, 42, 152,
    628, 905, 637, 278, 458, 792,
    353, 374, 812, 723, 201, 867,
    586, 340, 615, 193, 366, 291,
    284, 530, 965, 541, 656, 576,
    210, 653, 355, 738, 621, 556,
    596, 761, 170, 549, 380, 87,
    // Bin capacity
    1858, 1962, 1855, 1964, 1607, 1807
  };
  const int i6_18_6_19_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    602, 744, 816, 406, 495, 311,
    895, 779, 454, 879, 186, 144,
    401, 558, 143, 951, 194, 892,
    461, 4, 405, 340, 145, 911,
    107, 639, 39, 665, 597, 766,
    969, 643, 388, 498, 499, 425,
    515, 579, 472, 135, 233, 360,
    564, 772, 726, 965, 670, 59,
    256, 140, 997, 397, 445, 325,
    320, 886, 645, 547, 31, 385,
    926, 726, 151, 555, 869, 731,
    232, 621, 120, 204, 416, 401,
    842, 899, 31, 994, 914, 268,
    644, 363, 762, 351, 999, 38,
    271, 628, 554, 976, 995, 432,
    5, 343, 476, 945, 579, 589,
    783, 284, 910, 224, 607, 655,
    193, 482, 745, 364, 960, 377,
    // Bin capacity
    1783, 2002, 1753, 2062, 1951, 1601
  };
  const int i6_18_6_20_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    54, 523, 364, 109, 936, 517,
    61, 342, 954, 263, 357, 582,
    381, 244, 375, 967, 17, 31,
    153, 727, 734, 537, 997, 214,
    834, 821, 957, 464, 578, 303,
    839, 668, 545, 593, 758, 183,
    233, 70, 121, 260, 202, 920,
    635, 726, 675, 209, 938, 527,
    745, 890, 38, 718, 485, 736,
    107, 802, 572, 665, 921, 446,
    164, 800, 9, 32, 307, 607,
    539, 361, 407, 727, 881, 432,
    14, 358, 921, 842, 981, 933,
    432, 668, 960, 265, 340, 358,
    989, 177, 155, 786, 799, 744,
    174, 781, 21, 93, 75, 574,
    610, 257, 534, 753, 705, 447,
    835, 329, 932, 197, 43, 555,
    // Bin capacity
    1560, 1909, 1855, 1696, 2064, 1822
  };
  const int i6_18_6_21_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    387, 280, 85, 860, 286, 997,
    974, 986, 613, 609, 228, 954,
    876, 696, 418, 803, 780, 420,
    83, 607, 662, 43, 485, 572,
    160, 172, 659, 359, 264, 370,
    470, 174, 245, 342, 233, 602,
    157, 79, 630, 928, 483, 755,
    990, 116, 704, 380, 469, 965,
    498, 776, 907, 89, 265, 564,
    391, 14, 360, 221, 390, 720,
    339, 295, 800, 790, 343, 592,
    802, 270, 169, 0, 600, 47,
    73, 831, 408, 212, 730, 679,
    763, 865, 349, 178, 691, 488,
    909, 846, 927, 416, 625, 955,
    490, 888, 86, 141, 627, 861,
    516, 222, 501, 270, 4, 863,
    257, 165, 267, 679, 745, 822,
    // Bin capacity
    1843, 1671, 1773, 1477, 1664, 2466
  };
  const int i6_18_6_22_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    600, 634, 648, 458, 289, 146,
    201, 666, 46, 418, 113, 749,
    196, 697, 657, 736, 770, 669,
    677, 624, 693, 71, 516, 526,
    348, 607, 958, 464, 240, 23,
    26, 584, 978, 865, 408, 216,
    646, 541, 1000, 79, 844, 819,
    775, 954, 915, 552, 226, 298,
    348, 394, 514, 849, 454, 160,
    900, 267, 555, 637, 718, 402,
    854, 287, 277, 408, 399, 474,
    4, 994, 559, 180, 883, 426,
    802, 424, 904, 987, 616, 906,
    60, 937, 399, 609, 728, 141,
    869, 714, 926, 144, 621, 718,
    518, 834, 485, 528, 16, 136,
    976, 639, 434, 85, 667, 549,
    764, 246, 336, 834, 838, 596,
    // Bin capacity
    1945, 2246, 2295, 1811, 1901, 1618
  };
  const int i6_18_6_23_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    620, 671, 478, 983, 841, 156,
    348, 381, 519, 167, 506, 563,
    987, 861, 435, 221, 486, 730,
    966, 55, 206, 546, 469, 140,
    672, 135, 196, 346, 849, 161,
    132, 116, 69, 329, 467, 75,
    218, 886, 59, 897, 201, 1,
    695, 138, 248, 657, 320, 655,
    967, 433, 567, 594, 84, 235,
    842, 27, 34, 645, 320, 568,
    811, 741, 9, 902, 183, 939,
    783, 659, 7, 604, 579, 25,
    950, 681, 54, 201, 197, 783,
    228, 821, 877, 326, 118, 227,
    74, 366, 707, 306, 402, 59,
    729, 927, 907, 288, 75, 482,
    396, 438, 424, 731, 112, 669,
    833, 550, 152, 915, 770, 457,
    // Bin capacity
    2307, 1822, 1220, 1980, 1431, 1420
  };
  const int i6_18_6_24_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    173, 720, 214, 505, 309, 180,
    852, 745, 899, 598, 112, 537,
    271, 638, 438, 828, 694, 696,
    102, 972, 639, 240, 839, 963,
    132, 169, 440, 261, 775, 335,
    890, 568, 674, 473, 868, 172,
    967, 497, 604, 455, 84, 610,
    95, 540, 263, 903, 396, 149,
    238, 557, 291, 446, 174, 131,
    596, 994, 976, 260, 611, 134,
    313, 336, 463, 337, 663, 168,
    66, 184, 689, 258, 584, 140,
    68, 757, 916, 500, 686, 308,
    129, 151, 488, 398, 350, 717,
    720, 44, 87, 44, 381, 658,
    247, 112, 962, 937, 599, 950,
    181, 497, 515, 576, 456, 878,
    351, 556, 556, 662, 642, 314,
    // Bin capacity
    1321, 1868, 2091, 1795, 1907, 1662
  };
  const int i6_18_6_25_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    152, 852, 804, 747, 956, 81,
    329, 685, 269, 958, 50, 764,
    52, 144, 384, 616, 382, 198,
    377, 657, 486, 597, 407, 66,
    119, 902, 932, 746, 711, 912,
    273, 614, 25, 913, 289, 377,
    840, 444, 639, 925, 982, 845,
    292, 330, 68, 42, 440, 857,
    987, 454, 245, 678, 829, 915,
    245, 438, 742, 967, 769, 590,
    727, 356, 37, 826, 9, 164,
    282, 550, 559, 203, 127, 423,
    54, 679, 663, 119, 384, 428,
    842, 121, 239, 448, 657, 5,
    623, 495, 301, 59, 318, 942,
    529, 305, 403, 652, 165, 849,
    990, 846, 251, 601, 93, 194,
    154, 936, 284, 605, 256, 146,
    // Bin capacity
    1639, 2044, 1528, 2230, 1630, 1825
  };
  const int i6_18_6_26_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    722, 194, 894, 647, 220, 50,
    337, 83, 507, 452, 505, 828,
    673, 784, 860, 301, 499, 902,
    990, 120, 638, 756, 335, 166,
    228, 700, 812, 516, 517, 782,
    355, 75, 34, 361, 91, 703,
    906, 0, 406, 824, 150, 56,
    632, 936, 821, 888, 690, 44,
    499, 888, 959, 270, 44, 337,
    635, 462, 129, 718, 853, 760,
    417, 827, 626, 139, 487, 533,
    89, 409, 699, 280, 765, 941,
    101, 345, 829, 633, 681, 248,
    456, 421, 862, 463, 468, 612,
    63, 482, 648, 77, 631, 314,
    289, 393, 581, 711, 96, 663,
    872, 244, 559, 565, 556, 286,
    154, 955, 228, 348, 739, 422,
    // Bin capacity
    1768, 1747, 2330, 1880, 1749, 1816
  };
  const int i6_18_6_27_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    372, 563, 171, 945, 969, 453,
    658, 806, 218, 91, 22, 490,
    895, 938, 717, 323, 18, 886,
    24, 178, 912, 502, 60, 470,
    232, 108, 566, 759, 445, 914,
    394, 767, 868, 529, 387, 990,
    231, 721, 912, 590, 684, 146,
    981, 9, 583, 324, 91, 497,
    427, 855, 44, 378, 401, 175,
    393, 358, 518, 350, 874, 756,
    93, 128, 89, 495, 508, 324,
    878, 593, 307, 545, 608, 167,
    520, 126, 620, 778, 122, 865,
    980, 333, 244, 577, 829, 266,
    71, 591, 142, 300, 852, 296,
    997, 198, 188, 435, 375, 474,
    602, 663, 63, 694, 647, 710,
    867, 178, 563, 129, 320, 201,
    // Bin capacity
    2036, 1718, 1636, 1851, 1739, 1922
  };
  const int i6_18_6_28_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    128, 619, 918, 452, 970, 926,
    23, 916, 531, 770, 370, 215,
    189, 439, 590, 335, 978, 128,
    707, 149, 881, 474, 945, 199,
    54, 236, 960, 863, 499, 652,
    567, 537, 197, 892, 863, 926,
    140, 326, 906, 742, 761, 904,
    935, 703, 743, 855, 891, 693,
    14, 517, 323, 127, 532, 769,
    946, 154, 224, 258, 729, 868,
    78, 51, 686, 3, 14, 164,
    979, 276, 341, 620, 250, 928,
    459, 109, 242, 888, 528, 904,
    147, 911, 220, 585, 49, 345,
    374, 522, 331, 774, 368, 908,
    321, 266, 505, 605, 266, 66,
    944, 689, 203, 316, 579, 845,
    455, 972, 238, 298, 73, 723,
    // Bin capacity
    1592, 1791, 1929, 2103, 2062, 2382
  };
  const int i6_18_6_29_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    135, 498, 661, 740, 404, 620,
    118, 241, 827, 13, 883, 465,
    752, 652, 280, 664, 248, 624,
    940, 125, 535, 34, 596, 329,
    546, 737, 474, 922, 335, 700,
    527, 181, 334, 490, 995, 258,
    5, 287, 168, 584, 842, 497,
    628, 824, 426, 81, 888, 100,
    870, 309, 8, 881, 882, 598,
    380, 242, 930, 574, 245, 337,
    923, 775, 855, 857, 38, 305,
    611, 133, 870, 897, 368, 900,
    272, 851, 328, 651, 978, 522,
    871, 837, 494, 481, 522, 77,
    706, 753, 407, 861, 971, 903,
    786, 481, 173, 971, 297, 44,
    968, 341, 322, 633, 377, 286,
    996, 771, 805, 968, 996, 775,
    // Bin capacity
    2373, 1944, 1913, 2430, 2336, 1794
  };
  const int i6_18_6_30_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    65, 834, 987, 866, 26, 50,
    257, 746, 904, 587, 913, 55,
    736, 684, 426, 881, 675, 539,
    424, 672, 665, 120, 304, 25,
    8, 768, 128, 677, 74, 4,
    45, 73, 368, 345, 872, 618,
    824, 736, 215, 190, 126, 442,
    816, 560, 25, 527, 941, 419,
    989, 149, 284, 743, 178, 364,
    799, 944, 175, 991, 420, 50,
    92, 413, 827, 419, 646, 917,
    412, 585, 19, 786, 486, 469,
    1000, 917, 957, 456, 859, 757,
    917, 272, 597, 60, 949, 766,
    610, 731, 111, 997, 57, 456,
    285, 582, 110, 426, 584, 940,
    380, 173, 64, 949, 628, 448,
    728, 315, 918, 235, 206, 534,
    // Bin capacity
    2034, 2201, 1686, 2222, 1938, 1702
  };
  const int i6_18_6_31_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    972, 684, 482, 523, 436, 725,
    841, 520, 395, 544, 293, 300,
    429, 728, 408, 596, 895, 595,
    664, 945, 194, 982, 792, 320,
    540, 538, 107, 790, 330, 687,
    918, 588, 232, 251, 409, 82,
    721, 462, 2, 730, 728, 756,
    911, 476, 62, 258, 226, 314,
    869, 415, 853, 176, 349, 842,
    901, 812, 242, 213, 692, 377,
    894, 723, 887, 761, 540, 650,
    845, 438, 464, 300, 462, 924,
    279, 752, 799, 178, 701, 224,
    887, 604, 929, 981, 132, 192,
    913, 696, 690, 49, 683, 228,
    808, 254, 999, 792, 317, 657,
    986, 554, 102, 218, 719, 403,
    158, 81, 537, 548, 653, 257,
    // Bin capacity
    2956, 2243, 1831, 1941, 2043, 1864
  };
  const int i6_18_6_32_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    351, 481, 479, 526, 855, 777,
    567, 983, 294, 596, 872, 294,
    721, 499, 669, 123, 856, 560,
    378, 706, 274, 379, 488, 451,
    966, 630, 494, 731, 883, 973,
    292, 114, 502, 388, 507, 218,
    300, 608, 309, 243, 911, 616,
    814, 871, 474, 161, 377, 802,
    328, 79, 3, 181, 758, 854,
    912, 524, 832, 543, 699, 370,
    748, 235, 569, 907, 941, 683,
    693, 290, 718, 144, 43, 374,
    876, 92, 862, 48, 997, 72,
    358, 415, 605, 618, 336, 995,
    183, 631, 736, 337, 176, 440,
    531, 490, 713, 514, 671, 304,
    564, 90, 838, 298, 759, 854,
    960, 911, 410, 721, 493, 834,
    // Bin capacity
    2320, 1903, 2152, 1641, 2557, 2304
  };
  const int i6_18_6_33_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    8, 776, 864, 619, 824, 285,
    817, 154, 264, 885, 431, 487,
    444, 413, 290, 881, 397, 363,
    608, 313, 498, 13, 936, 965,
    818, 980, 118, 126, 35, 699,
    806, 772, 224, 321, 400, 563,
    976, 97, 748, 379, 817, 426,
    813, 199, 841, 727, 880, 429,
    543, 802, 44, 950, 471, 445,
    736, 652, 64, 691, 574, 534,
    329, 518, 789, 913, 562, 220,
    786, 681, 755, 238, 798, 205,
    402, 670, 748, 269, 687, 277,
    569, 238, 941, 143, 877, 787,
    740, 55, 161, 107, 682, 741,
    803, 59, 613, 107, 578, 569,
    353, 47, 504, 366, 39, 825,
    995, 575, 116, 51, 17, 573,
    // Bin capacity
    2560, 1774, 1903, 1726, 2218, 2083
  };
  const int i6_18_6_34_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    814, 622, 602, 218, 890, 469,
    604, 741, 991, 724, 16, 963,
    515, 640, 992, 601, 715, 824,
    756, 645, 740, 74, 657, 908,
    509, 609, 134, 838, 398, 100,
    299, 67, 345, 229, 965, 978,
    288, 981, 827, 3, 514, 819,
    483, 628, 254, 535, 591, 978,
    115, 278, 739, 492, 279, 478,
    693, 483, 337, 565, 201, 6,
    395, 860, 649, 921, 432, 687,
    790, 935, 948, 708, 399, 520,
    867, 85, 531, 204, 711, 532,
    1, 276, 154, 134, 890, 922,
    497, 350, 590, 503, 462, 226,
    771, 151, 158, 195, 33, 579,
    82, 275, 428, 814, 321, 417,
    5, 207, 149, 951, 651, 536,
    // Bin capacity
    1895, 1973, 2137, 1946, 2038, 2444
  };
  const int i6_18_6_35_31[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    765, 166, 759, 181, 404, 835,
    920, 753, 443, 946, 146, 50,
    969, 20, 985, 795, 795, 277,
    371, 181, 904, 188, 626, 835,
    812, 709, 528, 730, 363, 510,
    532, 616, 169, 730, 768, 136,
    574, 665, 683, 340, 216, 94,
    872, 996, 273, 915, 294, 978,
    577, 431, 6, 228, 684, 354,
    523, 790, 958, 700, 298, 970,
    395, 746, 90, 874, 540, 1,
    682, 977, 820, 588, 588, 509,
    737, 778, 112, 452, 31, 623,
    226, 547, 296, 311, 474, 0,
    840, 718, 853, 364, 834, 491,
    397, 687, 730, 253, 471, 43,
    179, 472, 243, 615, 161, 883,
    404, 212, 840, 459, 778, 36,
    // Bin capacity
    2425, 2355, 2181, 2176, 1906, 1716
  };
  const int i6_18_6_0_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    112, 58, 455, 519, 658, 465,
    668, 864, 369, 378, 237, 105,
    632, 868, 768, 438, 277, 396,
    559, 797, 194, 624, 948, 959,
    844, 532, 237, 625, 519, 633,
    753, 497, 365, 678, 654, 290,
    794, 919, 442, 440, 320, 205,
    884, 822, 347, 704, 331, 455,
    449, 278, 295, 123, 220, 361,
    584, 47, 64, 208, 195, 629,
    874, 682, 216, 819, 142, 461,
    28, 229, 407, 462, 710, 971,
    854, 15, 744, 499, 424, 108,
    712, 135, 757, 828, 732, 455,
    622, 27, 6, 680, 222, 484,
    314, 642, 589, 5, 756, 474,
    714, 314, 104, 405, 394, 381,
    124, 889, 400, 807, 703, 602,
    // Bin capacity
    1754, 1436, 1127, 1541, 1407, 1406
  };
  const int i6_18_6_1_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    816, 425, 751, 968, 202, 615,
    450, 36, 601, 335, 956, 926,
    97, 992, 916, 629, 282, 627,
    586, 630, 152, 684, 305, 431,
    463, 452, 525, 699, 292, 181,
    288, 311, 740, 824, 465, 276,
    784, 943, 410, 690, 118, 185,
    748, 304, 226, 260, 883, 782,
    378, 505, 540, 700, 216, 222,
    467, 609, 294, 649, 600, 262,
    689, 225, 306, 167, 264, 111,
    476, 410, 160, 813, 466, 253,
    978, 621, 605, 189, 807, 897,
    965, 821, 440, 588, 897, 470,
    567, 624, 336, 135, 48, 908,
    263, 316, 545, 243, 581, 408,
    797, 976, 662, 939, 656, 531,
    816, 889, 928, 569, 429, 646,
    // Bin capacity
    1790, 1699, 1539, 1697, 1426, 1470
  };
  const int i6_18_6_2_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    645, 691, 247, 694, 558, 943,
    444, 732, 210, 236, 647, 937,
    858, 207, 716, 464, 308, 550,
    508, 532, 327, 368, 17, 521,
    647, 837, 322, 27, 746, 576,
    417, 956, 521, 380, 674, 832,
    963, 647, 944, 475, 423, 19,
    329, 990, 240, 993, 785, 561,
    376, 255, 128, 137, 246, 46,
    998, 738, 932, 68, 883, 458,
    34, 951, 142, 571, 938, 599,
    464, 543, 504, 655, 537, 981,
    732, 112, 758, 599, 54, 336,
    905, 431, 40, 733, 176, 328,
    370, 599, 677, 521, 197, 747,
    998, 664, 934, 473, 200, 139,
    982, 667, 643, 5, 287, 748,
    492, 397, 965, 596, 499, 851,
    // Bin capacity
    1898, 1862, 1573, 1360, 1390, 1730
  };
  const int i6_18_6_3_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    751, 341, 995, 756, 568, 222,
    554, 638, 354, 511, 361, 893,
    30, 584, 446, 436, 650, 199,
    451, 308, 597, 239, 42, 20,
    656, 927, 274, 848, 507, 374,
    56, 120, 70, 818, 388, 129,
    292, 995, 831, 924, 767, 725,
    590, 439, 34, 891, 130, 658,
    437, 450, 404, 534, 763, 225,
    458, 356, 490, 513, 124, 517,
    999, 324, 403, 727, 452, 28,
    598, 286, 986, 355, 796, 544,
    103, 809, 565, 536, 943, 706,
    956, 537, 109, 36, 178, 155,
    401, 766, 730, 148, 910, 678,
    5, 823, 234, 901, 936, 563,
    959, 429, 337, 859, 815, 375,
    5, 216, 164, 763, 806, 995,
    // Bin capacity
    1426, 1605, 1378, 1854, 1741, 1375
  };
  const int i6_18_6_4_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    818, 367, 812, 524, 563, 828,
    261, 963, 612, 933, 66, 206,
    383, 649, 507, 394, 852, 707,
    744, 304, 480, 881, 769, 554,
    356, 881, 506, 465, 293, 415,
    672, 195, 387, 609, 531, 887,
    25, 226, 486, 415, 862, 561,
    601, 554, 569, 252, 794, 283,
    638, 490, 467, 284, 821, 786,
    155, 356, 84, 863, 983, 364,
    753, 463, 958, 602, 696, 799,
    387, 656, 273, 991, 647, 529,
    277, 985, 221, 907, 322, 250,
    210, 552, 416, 78, 958, 857,
    39, 521, 764, 622, 840, 586,
    808, 489, 796, 623, 504, 514,
    337, 713, 599, 69, 135, 550,
    434, 428, 917, 899, 678, 637,
    // Bin capacity
    1369, 1698, 1709, 1805, 1962, 1788
  };
  const int i6_18_6_5_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    918, 987, 10, 988, 850, 532,
    806, 20, 375, 828, 748, 594,
    268, 486, 765, 135, 319, 159,
    115, 321, 186, 179, 138, 416,
    749, 410, 192, 319, 988, 648,
    893, 250, 225, 933, 979, 864,
    46, 37, 532, 143, 194, 318,
    521, 76, 741, 12, 974, 634,
    109, 824, 502, 415, 47, 964,
    877, 823, 684, 693, 489, 446,
    129, 771, 649, 993, 696, 949,
    136, 213, 251, 775, 229, 982,
    979, 474, 291, 510, 81, 37,
    54, 544, 145, 207, 309, 371,
    99, 716, 860, 242, 874, 511,
    393, 944, 42, 670, 966, 445,
    104, 935, 296, 465, 225, 471,
    112, 445, 787, 197, 540, 38,
    // Bin capacity
    1279, 1624, 1319, 1524, 1689, 1642
  };
  const int i6_18_6_6_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    837, 872, 596, 109, 694, 525,
    772, 582, 929, 323, 131, 10,
    155, 436, 752, 547, 155, 188,
    47, 54, 137, 454, 660, 819,
    689, 266, 887, 835, 453, 676,
    55, 276, 493, 487, 931, 207,
    129, 403, 125, 78, 11, 990,
    807, 357, 14, 377, 614, 913,
    454, 422, 70, 767, 429, 948,
    863, 636, 306, 79, 494, 894,
    978, 248, 119, 778, 681, 357,
    466, 608, 752, 332, 982, 193,
    133, 919, 659, 937, 31, 652,
    954, 447, 693, 629, 456, 995,
    906, 921, 13, 105, 166, 520,
    951, 325, 457, 23, 136, 324,
    749, 349, 163, 968, 364, 770,
    720, 624, 20, 784, 954, 837,
    // Bin capacity
    1885, 1545, 1270, 1522, 1474, 1912
  };
  const int i6_18_6_7_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    755, 21, 540, 150, 805, 238,
    737, 424, 373, 776, 444, 574,
    747, 498, 346, 67, 221, 137,
    931, 665, 237, 568, 907, 640,
    688, 815, 171, 81, 662, 445,
    992, 446, 501, 598, 334, 992,
    594, 636, 510, 75, 655, 64,
    489, 599, 788, 911, 139, 372,
    955, 978, 696, 95, 577, 946,
    343, 255, 262, 940, 304, 377,
    456, 172, 422, 483, 829, 936,
    191, 248, 310, 863, 421, 389,
    358, 230, 604, 193, 485, 370,
    243, 579, 457, 920, 428, 167,
    101, 773, 536, 349, 189, 991,
    492, 263, 535, 737, 369, 735,
    940, 455, 711, 974, 107, 320,
    461, 394, 29, 165, 741, 123,
    // Bin capacity
    1868, 1508, 1432, 1596, 1537, 1573
  };
  const int i6_18_6_8_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    551, 174, 490, 700, 17, 55,
    81, 153, 271, 626, 670, 605,
    446, 538, 494, 364, 194, 508,
    934, 648, 912, 498, 475, 275,
    346, 932, 57, 955, 947, 3,
    631, 472, 958, 711, 285, 701,
    683, 184, 738, 281, 964, 752,
    400, 41, 152, 629, 932, 428,
    21, 785, 700, 20, 298, 301,
    188, 950, 642, 3, 858, 208,
    440, 543, 560, 108, 681, 85,
    148, 295, 673, 27, 296, 321,
    596, 706, 66, 612, 89, 934,
    734, 294, 807, 414, 955, 775,
    51, 765, 513, 711, 848, 750,
    622, 555, 200, 437, 703, 589,
    860, 560, 918, 171, 456, 172,
    342, 903, 941, 622, 839, 414,
    // Bin capacity
    1454, 1710, 1817, 1421, 1892, 1418
  };
  const int i6_18_6_9_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    375, 694, 930, 458, 501, 699,
    760, 223, 98, 934, 219, 739,
    23, 711, 335, 577, 975, 407,
    548, 880, 48, 640, 568, 387,
    76, 484, 659, 565, 437, 706,
    365, 824, 30, 636, 986, 760,
    981, 996, 871, 283, 479, 445,
    948, 927, 175, 908, 34, 95,
    825, 203, 488, 97, 423, 206,
    102, 814, 743, 465, 189, 592,
    64, 941, 76, 791, 632, 617,
    273, 627, 39, 8, 333, 905,
    462, 867, 803, 903, 461, 72,
    82, 846, 309, 780, 128, 545,
    50, 496, 126, 677, 734, 77,
    694, 977, 402, 941, 43, 433,
    972, 708, 268, 359, 699, 881,
    570, 564, 0, 328, 896, 44,
    // Bin capacity
    1485, 2323, 1163, 1881, 1588, 1565
  };
  const int i6_18_6_10_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    43, 284, 676, 249, 984, 551,
    449, 439, 459, 999, 874, 98,
    903, 847, 393, 152, 829, 611,
    852, 119, 352, 848, 740, 671,
    439, 60, 233, 872, 685, 40,
    96, 38, 187, 120, 678, 746,
    160, 1, 507, 731, 963, 443,
    164, 459, 462, 827, 690, 540,
    547, 652, 713, 879, 969, 395,
    896, 534, 75, 985, 911, 941,
    160, 288, 125, 784, 390, 140,
    983, 365, 572, 241, 453, 386,
    725, 908, 263, 282, 374, 62,
    67, 15, 847, 921, 680, 24,
    219, 847, 987, 933, 485, 948,
    207, 755, 475, 583, 348, 66,
    956, 196, 583, 476, 512, 57,
    670, 696, 294, 810, 189, 574,
    // Bin capacity
    1565, 1376, 1504, 2144, 2155, 1338
  };
  const int i6_18_6_11_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    881, 456, 868, 890, 823, 858,
    994, 875, 271, 638, 297, 719,
    650, 531, 93, 446, 522, 560,
    299, 315, 453, 27, 522, 873,
    707, 18, 708, 516, 698, 366,
    98, 949, 111, 905, 14, 291,
    760, 576, 958, 675, 506, 348,
    990, 657, 838, 11, 566, 318,
    846, 628, 105, 129, 17, 165,
    577, 391, 435, 361, 934, 208,
    177, 832, 396, 279, 483, 384,
    382, 190, 309, 696, 690, 169,
    501, 268, 684, 535, 827, 494,
    93, 852, 130, 486, 957, 665,
    615, 925, 677, 892, 788, 915,
    13, 843, 19, 450, 203, 721,
    855, 802, 650, 979, 647, 15,
    392, 88, 700, 443, 391, 869,
    // Bin capacity
    1819, 1887, 1555, 1732, 1829, 1654
  };
  const int i6_18_6_12_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    614, 360, 374, 940, 763, 165,
    309, 529, 946, 788, 151, 945,
    886, 232, 330, 139, 676, 522,
    125, 958, 842, 832, 77, 54,
    798, 390, 827, 902, 294, 828,
    51, 853, 362, 422, 589, 720,
    119, 451, 800, 591, 989, 395,
    7, 259, 780, 429, 461, 582,
    472, 527, 844, 34, 819, 959,
    683, 233, 66, 157, 797, 466,
    872, 326, 972, 929, 261, 468,
    130, 237, 865, 958, 269, 655,
    131, 442, 39, 392, 217, 35,
    676, 753, 320, 908, 13, 421,
    657, 560, 699, 502, 955, 74,
    521, 372, 43, 690, 546, 583,
    207, 370, 559, 839, 467, 257,
    465, 893, 974, 726, 515, 205,
    // Bin capacity
    1442, 1633, 1987, 2087, 1654, 1556
  };
  const int i6_18_6_13_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    215, 965, 572, 464, 734, 592,
    91, 992, 605, 45, 527, 904,
    551, 358, 70, 653, 671, 288,
    92, 280, 640, 23, 435, 645,
    236, 631, 697, 21, 601, 518,
    445, 998, 935, 141, 895, 125,
    171, 106, 773, 454, 830, 958,
    204, 451, 216, 168, 964, 425,
    271, 756, 347, 842, 34, 171,
    108, 17, 387, 882, 884, 506,
    214, 470, 251, 548, 88, 973,
    825, 541, 475, 770, 820, 377,
    801, 780, 828, 127, 162, 5,
    211, 142, 395, 504, 764, 262,
    789, 743, 86, 117, 318, 948,
    75, 384, 669, 176, 355, 360,
    803, 361, 482, 809, 819, 582,
    591, 692, 386, 794, 15, 212,
    // Bin capacity
    1261, 1821, 1660, 1420, 1868, 1667
  };
  const int i6_18_6_14_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    110, 470, 348, 520, 149, 783,
    879, 640, 645, 73, 397, 898,
    969, 276, 344, 342, 20, 827,
    438, 188, 651, 574, 296, 396,
    28, 420, 745, 158, 860, 87,
    584, 641, 686, 381, 920, 43,
    988, 304, 508, 859, 619, 768,
    1000, 766, 635, 6, 269, 303,
    653, 250, 777, 317, 217, 914,
    604, 778, 96, 63, 593, 40,
    704, 529, 978, 1, 461, 63,
    233, 732, 303, 337, 996, 624,
    638, 975, 519, 316, 964, 385,
    902, 782, 140, 596, 118, 191,
    973, 997, 267, 187, 859, 965,
    466, 587, 916, 894, 834, 517,
    226, 346, 481, 500, 805, 902,
    173, 983, 927, 433, 923, 853,
    // Bin capacity
    2008, 2027, 1894, 1246, 1958, 1817
  };
  const int i6_18_6_15_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    927, 289, 224, 334, 96, 199,
    411, 231, 22, 134, 263, 433,
    307, 613, 418, 891, 238, 94,
    619, 450, 521, 366, 327, 582,
    642, 73, 912, 817, 383, 108,
    978, 734, 314, 907, 177, 440,
    179, 868, 987, 504, 596, 457,
    404, 899, 6, 535, 978, 533,
    244, 935, 874, 491, 545, 543,
    461, 845, 821, 767, 190, 624,
    378, 808, 805, 733, 919, 112,
    224, 72, 82, 516, 123, 345,
    696, 190, 479, 195, 772, 409,
    517, 104, 838, 202, 84, 276,
    443, 574, 544, 836, 497, 389,
    858, 986, 453, 409, 118, 853,
    344, 806, 238, 378, 934, 928,
    935, 325, 661, 490, 381, 136,
    // Bin capacity
    1834, 1879, 1764, 1822, 1461, 1431
  };
  const int i6_18_6_16_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    243, 265, 757, 866, 370, 532,
    790, 66, 459, 886, 380, 649,
    83, 479, 294, 456, 841, 200,
    908, 475, 997, 232, 464, 328,
    218, 449, 453, 406, 398, 336,
    662, 308, 802, 794, 134, 514,
    933, 836, 480, 103, 263, 838,
    595, 837, 559, 963, 486, 43,
    477, 836, 107, 430, 795, 84,
    969, 135, 542, 543, 789, 658,
    617, 711, 418, 231, 183, 221,
    420, 708, 189, 309, 800, 346,
    496, 145, 145, 118, 101, 272,
    990, 505, 312, 830, 761, 862,
    190, 347, 856, 368, 730, 799,
    133, 614, 494, 25, 139, 821,
    104, 568, 365, 213, 201, 434,
    438, 696, 972, 383, 977, 837,
    // Bin capacity
    1792, 1737, 1779, 1577, 1704, 1697
  };
  const int i6_18_6_17_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    429, 717, 26, 540, 529, 221,
    68, 248, 822, 103, 478, 461,
    69, 280, 373, 96, 136, 506,
    541, 792, 317, 674, 472, 631,
    626, 197, 547, 796, 957, 186,
    379, 643, 115, 858, 174, 913,
    883, 103, 85, 228, 565, 805,
    87, 148, 239, 456, 357, 994,
    700, 668, 747, 571, 532, 48,
    147, 425, 864, 70, 949, 589,
    452, 787, 47, 175, 457, 353,
    988, 651, 214, 131, 102, 816,
    483, 490, 842, 118, 964, 54,
    699, 320, 187, 497, 241, 964,
    872, 878, 927, 639, 843, 975,
    716, 410, 17, 130, 644, 801,
    315, 357, 498, 984, 578, 258,
    769, 316, 532, 796, 167, 400,
    // Bin capacity
    1799, 1644, 1443, 1534, 1784, 1946
  };
  const int i6_18_6_18_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    0, 369, 211, 12, 938, 93,
    970, 150, 434, 897, 576, 516,
    466, 164, 427, 474, 524, 188,
    634, 790, 902, 534, 239, 756,
    469, 856, 326, 76, 661, 128,
    981, 952, 825, 581, 68, 6,
    202, 148, 706, 212, 432, 247,
    70, 8, 245, 556, 663, 35,
    617, 241, 646, 948, 522, 492,
    479, 697, 414, 418, 842, 984,
    234, 840, 124, 332, 945, 710,
    705, 704, 605, 211, 651, 165,
    35, 256, 328, 897, 404, 404,
    991, 961, 121, 449, 697, 996,
    185, 946, 74, 942, 313, 315,
    855, 191, 400, 200, 16, 488,
    671, 419, 808, 529, 692, 94,
    964, 150, 395, 475, 484, 125,
    // Bin capacity
    1874, 1739, 1572, 1720, 1902, 1326
  };
  const int i6_18_6_19_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    376, 51, 32, 975, 348, 317,
    466, 18, 557, 930, 467, 746,
    18, 643, 354, 410, 649, 600,
    408, 609, 821, 317, 799, 917,
    75, 754, 210, 195, 813, 236,
    228, 855, 601, 666, 33, 560,
    744, 563, 196, 962, 430, 747,
    876, 662, 58, 544, 136, 829,
    687, 231, 565, 505, 611, 206,
    192, 381, 645, 765, 213, 401,
    768, 853, 62, 405, 63, 577,
    602, 91, 625, 924, 447, 901,
    196, 85, 921, 239, 909, 892,
    740, 21, 794, 489, 994, 407,
    847, 312, 91, 883, 833, 380,
    437, 433, 476, 709, 187, 381,
    992, 951, 581, 797, 186, 708,
    631, 589, 238, 147, 552, 105,
    // Bin capacity
    1842, 1607, 1553, 2155, 1720, 1966
  };
  const int i6_18_6_20_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    476, 128, 585, 720, 782, 48,
    470, 526, 307, 343, 525, 550,
    344, 514, 385, 376, 490, 886,
    244, 603, 432, 835, 444, 897,
    969, 995, 488, 591, 694, 471,
    75, 422, 348, 456, 200, 494,
    367, 906, 388, 870, 124, 921,
    52, 505, 711, 910, 145, 936,
    815, 228, 663, 417, 777, 634,
    864, 115, 139, 928, 902, 423,
    654, 639, 386, 838, 132, 4,
    603, 834, 855, 980, 871, 516,
    719, 279, 551, 601, 669, 36,
    729, 68, 205, 493, 553, 581,
    73, 978, 480, 631, 205, 297,
    253, 276, 236, 968, 388, 725,
    611, 202, 427, 936, 517, 820,
    539, 879, 272, 893, 455, 55,
    // Bin capacity
    1772, 1820, 1572, 2558, 1775, 1859
  };
  const int i6_18_6_21_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    854, 304, 999, 461, 748, 429,
    910, 840, 189, 736, 379, 442,
    56, 882, 818, 194, 565, 984,
    690, 971, 436, 186, 416, 119,
    380, 251, 320, 3, 170, 779,
    845, 593, 168, 971, 416, 389,
    587, 806, 260, 348, 705, 303,
    348, 544, 707, 6, 495, 860,
    827, 691, 965, 210, 349, 13,
    92, 424, 727, 350, 329, 760,
    592, 925, 270, 890, 418, 908,
    815, 432, 834, 967, 545, 727,
    565, 317, 628, 700, 620, 147,
    148, 349, 84, 721, 56, 23,
    925, 304, 244, 708, 600, 930,
    208, 393, 61, 142, 43, 7,
    158, 8, 752, 14, 210, 218,
    227, 633, 183, 823, 824, 468,
    // Bin capacity
    1861, 1950, 1744, 1701, 1591, 1716
  };
  const int i6_18_6_22_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    938, 783, 782, 238, 207, 702,
    440, 836, 738, 73, 131, 295,
    770, 191, 825, 192, 714, 97,
    556, 715, 276, 375, 500, 506,
    340, 942, 716, 878, 755, 948,
    748, 648, 336, 919, 408, 954,
    988, 763, 229, 906, 577, 398,
    581, 228, 901, 45, 503, 917,
    582, 771, 262, 577, 257, 961,
    400, 714, 411, 496, 459, 175,
    888, 309, 589, 863, 983, 807,
    448, 446, 165, 606, 794, 342,
    751, 792, 724, 62, 702, 477,
    459, 32, 58, 576, 36, 805,
    192, 927, 864, 655, 519, 539,
    930, 894, 852, 212, 160, 55,
    717, 916, 10, 724, 545, 53,
    334, 845, 585, 238, 794, 590,
    // Bin capacity
    2250, 2390, 1896, 1756, 1839, 1957
  };
  const int i6_18_6_23_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    252, 999, 409, 458, 568, 655,
    363, 829, 51, 722, 227, 704,
    784, 372, 364, 135, 461, 553,
    303, 982, 801, 962, 904, 766,
    995, 582, 306, 157, 405, 926,
    936, 652, 65, 463, 551, 367,
    71, 340, 69, 566, 169, 828,
    297, 315, 292, 349, 968, 19,
    138, 334, 11, 21, 265, 706,
    17, 56, 810, 724, 196, 841,
    147, 623, 337, 469, 412, 270,
    404, 403, 31, 718, 689, 829,
    713, 527, 685, 372, 865, 683,
    812, 332, 808, 297, 988, 492,
    87, 65, 137, 554, 33, 185,
    921, 442, 853, 535, 913, 661,
    426, 683, 525, 121, 658, 879,
    815, 774, 31, 831, 750, 442,
    // Bin capacity
    1739, 1909, 1350, 1734, 2055, 2216
  };
  const int i6_18_6_24_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    722, 222, 928, 976, 29, 637,
    704, 243, 148, 159, 539, 978,
    51, 402, 752, 841, 933, 332,
    605, 238, 961, 307, 88, 890,
    330, 606, 869, 53, 546, 881,
    147, 874, 76, 997, 195, 620,
    85, 25, 243, 798, 447, 957,
    419, 80, 813, 1, 233, 638,
    214, 601, 336, 222, 734, 116,
    213, 270, 291, 965, 710, 814,
    75, 335, 354, 38, 12, 797,
    583, 243, 650, 159, 961, 943,
    133, 387, 821, 777, 741, 804,
    628, 296, 617, 76, 428, 190,
    334, 748, 624, 682, 189, 259,
    662, 653, 570, 780, 972, 478,
    274, 520, 857, 806, 17, 795,
    820, 793, 116, 363, 136, 586,
    // Bin capacity
    1447, 1558, 2073, 1860, 1635, 2422
  };
  const int i6_18_6_25_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    467, 106, 553, 679, 520, 358,
    648, 298, 495, 862, 640, 679,
    89, 735, 915, 369, 495, 996,
    379, 286, 805, 266, 818, 46,
    950, 337, 177, 748, 861, 353,
    260, 286, 70, 558, 155, 156,
    328, 918, 744, 984, 674, 695,
    9, 627, 379, 142, 766, 588,
    501, 612, 500, 617, 80, 542,
    65, 164, 525, 441, 588, 6,
    745, 208, 392, 985, 693, 890,
    899, 640, 4, 191, 192, 697,
    528, 425, 13, 734, 704, 429,
    195, 370, 346, 60, 853, 695,
    131, 359, 101, 388, 193, 595,
    961, 85, 876, 721, 434, 859,
    565, 633, 212, 588, 747, 182,
    595, 588, 120, 336, 47, 816,
    // Bin capacity
    1733, 1600, 1506, 2015, 1971, 1997
  };
  const int i6_18_6_26_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    42, 767, 672, 749, 805, 243,
    294, 565, 533, 811, 273, 849,
    67, 456, 786, 12, 969, 436,
    53, 92, 292, 678, 466, 58,
    38, 498, 269, 575, 896, 705,
    392, 238, 205, 961, 449, 178,
    869, 167, 615, 348, 76, 271,
    794, 537, 313, 847, 682, 564,
    914, 728, 586, 41, 872, 133,
    882, 174, 646, 783, 887, 452,
    509, 352, 457, 6, 852, 152,
    70, 555, 898, 112, 787, 886,
    535, 186, 624, 697, 330, 502,
    952, 656, 965, 85, 960, 708,
    769, 530, 881, 799, 952, 781,
    176, 89, 982, 876, 651, 62,
    218, 887, 745, 17, 294, 976,
    389, 955, 558, 106, 678, 300,
    // Bin capacity
    1673, 1771, 2316, 1786, 2495, 1734
  };
  const int i6_18_6_27_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    279, 124, 797, 987, 250, 344,
    418, 312, 126, 574, 460, 950,
    319, 656, 393, 870, 907, 473,
    167, 614, 383, 403, 891, 40,
    373, 958, 832, 559, 575, 48,
    612, 396, 348, 674, 222, 44,
    573, 769, 822, 629, 175, 303,
    518, 983, 76, 366, 438, 230,
    202, 447, 830, 857, 202, 524,
    338, 948, 882, 203, 106, 571,
    936, 934, 222, 220, 44, 853,
    621, 206, 59, 197, 121, 954,
    404, 638, 823, 204, 996, 593,
    649, 784, 18, 255, 863, 390,
    653, 494, 883, 372, 318, 764,
    646, 778, 596, 140, 114, 141,
    926, 900, 661, 447, 129, 958,
    58, 975, 526, 776, 933, 722,
    // Bin capacity
    1840, 2523, 1964, 1849, 1640, 1885
  };
  const int i6_18_6_28_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    159, 777, 233, 738, 275, 755,
    916, 322, 295, 605, 269, 1,
    995, 369, 857, 961, 486, 163,
    510, 207, 730, 357, 179, 595,
    629, 932, 297, 334, 810, 259,
    536, 484, 735, 341, 428, 855,
    675, 43, 528, 931, 370, 944,
    956, 591, 402, 90, 143, 116,
    149, 590, 534, 339, 909, 957,
    366, 310, 985, 752, 315, 346,
    737, 651, 466, 839, 628, 878,
    784, 826, 410, 137, 26, 149,
    153, 381, 989, 339, 661, 628,
    532, 862, 358, 792, 399, 916,
    735, 319, 970, 744, 669, 135,
    979, 146, 70, 210, 742, 186,
    143, 4, 69, 794, 78, 723,
    381, 736, 138, 913, 805, 334,
    // Bin capacity
    2205, 1824, 1935, 2180, 1748, 1908
  };
  const int i6_18_6_29_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    308, 772, 525, 762, 301, 936,
    754, 641, 115, 439, 326, 786,
    177, 862, 977, 384, 453, 560,
    221, 399, 389, 67, 507, 594,
    374, 311, 347, 578, 35, 899,
    157, 691, 634, 140, 223, 470,
    796, 599, 469, 353, 867, 81,
    727, 607, 840, 592, 155, 829,
    760, 563, 296, 740, 993, 472,
    246, 540, 499, 471, 2, 309,
    567, 768, 85, 252, 664, 222,
    686, 394, 374, 190, 431, 365,
    543, 312, 716, 872, 675, 969,
    894, 631, 50, 522, 11, 931,
    720, 242, 250, 190, 332, 196,
    929, 54, 517, 989, 150, 905,
    568, 529, 795, 812, 543, 37,
    368, 302, 425, 179, 617, 792,
    // Bin capacity
    2106, 1982, 1786, 1835, 1567, 2226
  };
  const int i6_18_6_30_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    253, 501, 48, 921, 734, 482,
    822, 119, 182, 215, 558, 423,
    100, 496, 713, 208, 322, 414,
    14, 251, 522, 51, 63, 457,
    871, 772, 55, 653, 212, 478,
    983, 30, 402, 902, 924, 915,
    922, 727, 771, 675, 728, 426,
    287, 205, 513, 399, 486, 522,
    118, 281, 738, 846, 969, 470,
    146, 750, 545, 242, 696, 501,
    934, 672, 249, 37, 440, 406,
    485, 420, 725, 611, 806, 580,
    595, 210, 691, 877, 809, 217,
    692, 573, 445, 88, 672, 889,
    934, 338, 980, 115, 187, 282,
    808, 768, 538, 397, 491, 674,
    842, 889, 896, 858, 834, 798,
    175, 450, 961, 481, 961, 898,
    // Bin capacity
    2163, 1832, 2162, 1859, 2360, 2131
  };
  const int i6_18_6_31_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    942, 17, 391, 570, 623, 586,
    591, 856, 882, 728, 525, 761,
    573, 527, 19, 862, 885, 338,
    794, 369, 826, 960, 490, 859,
    766, 243, 388, 710, 27, 640,
    116, 971, 71, 388, 678, 492,
    273, 97, 476, 693, 868, 558,
    696, 905, 156, 392, 769, 151,
    476, 182, 390, 615, 827, 97,
    463, 168, 546, 767, 518, 388,
    878, 168, 475, 314, 305, 317,
    603, 319, 616, 497, 610, 704,
    263, 872, 416, 858, 190, 571,
    422, 729, 270, 268, 898, 638,
    300, 44, 802, 137, 541, 678,
    256, 625, 308, 673, 353, 917,
    497, 183, 62, 115, 79, 755,
    185, 165, 822, 515, 986, 768,
    // Bin capacity
    1986, 1625, 1729, 2197, 2221, 2231
  };
  const int i6_18_6_32_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    495, 885, 722, 643, 446, 818,
    284, 273, 569, 223, 57, 951,
    203, 844, 547, 695, 652, 918,
    389, 356, 294, 161, 429, 966,
    177, 298, 683, 778, 332, 299,
    984, 139, 263, 34, 429, 290,
    513, 931, 65, 589, 530, 372,
    831, 568, 340, 814, 720, 183,
    287, 962, 309, 478, 968, 437,
    280, 598, 720, 87, 508, 355,
    560, 763, 645, 694, 853, 759,
    331, 753, 4, 80, 8, 98,
    533, 924, 455, 736, 776, 315,
    756, 899, 325, 366, 541, 998,
    798, 282, 667, 926, 944, 467,
    962, 984, 309, 393, 967, 674,
    559, 798, 882, 918, 27, 618,
    843, 917, 854, 319, 819, 942,
    // Bin capacity
    2153, 2679, 1904, 1966, 2202, 2302
  };
  const int i6_18_6_33_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    651, 460, 841, 412, 619, 510,
    619, 555, 974, 143, 554, 850,
    307, 797, 675, 279, 722, 670,
    804, 603, 301, 642, 749, 899,
    756, 228, 270, 396, 41, 333,
    341, 68, 691, 283, 867, 611,
    113, 397, 647, 205, 588, 434,
    998, 91, 201, 607, 132, 373,
    412, 245, 899, 513, 457, 588,
    992, 832, 17, 520, 15, 114,
    461, 340, 407, 214, 183, 615,
    830, 750, 251, 313, 301, 934,
    119, 201, 22, 981, 962, 181,
    63, 77, 552, 220, 334, 752,
    455, 280, 590, 979, 942, 176,
    685, 739, 799, 314, 844, 133,
    776, 749, 348, 270, 792, 342,
    152, 187, 406, 214, 841, 405,
    // Bin capacity
    2114, 1685, 1971, 1664, 2205, 1978
  };
  const int i6_18_6_34_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    779, 65, 855, 215, 398, 582,
    680, 55, 380, 700, 634, 180,
    951, 930, 610, 408, 705, 9,
    33, 803, 584, 550, 388, 981,
    717, 165, 213, 404, 277, 317,
    345, 45, 310, 785, 965, 785,
    384, 795, 58, 143, 1, 560,
    979, 867, 730, 900, 989, 510,
    9, 438, 236, 886, 921, 902,
    177, 985, 733, 143, 220, 729,
    220, 536, 198, 771, 632, 430,
    888, 435, 714, 46, 524, 978,
    111, 534, 578, 879, 65, 920,
    120, 782, 817, 788, 899, 851,
    481, 935, 442, 630, 73, 25,
    376, 878, 161, 789, 675, 199,
    643, 157, 42, 116, 877, 597,
    406, 69, 868, 323, 940, 561,
    // Bin capacity
    1854, 2116, 1905, 2117, 2275, 2260
  };
  const int i6_18_6_35_32[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    606, 428, 799, 829, 548, 886,
    965, 942, 24, 882, 19, 466,
    698, 509, 795, 304, 538, 99,
    932, 645, 95, 49, 988, 836,
    332, 771, 148, 701, 217, 955,
    83, 584, 760, 513, 177, 481,
    577, 392, 470, 229, 388, 155,
    380, 163, 416, 846, 1, 812,
    871, 491, 852, 143, 610, 894,
    4, 480, 313, 874, 843, 517,
    221, 485, 243, 580, 406, 26,
    662, 415, 583, 755, 703, 335,
    594, 860, 189, 189, 889, 752,
    6, 598, 422, 713, 2, 879,
    984, 894, 21, 921, 988, 751,
    899, 170, 127, 102, 366, 272,
    601, 132, 121, 794, 339, 739,
    551, 542, 62, 754, 366, 251,
    // Bin capacity
    2243, 2138, 1449, 2291, 1888, 2274
  };
  const int i6_18_6_0_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    848, 212, 369, 316, 299, 309,
    858, 445, 896, 295, 350, 239,
    413, 249, 689, 344, 36, 582,
    770, 875, 396, 989, 174, 694,
    528, 739, 345, 437, 452, 508,
    174, 471, 737, 132, 292, 576,
    121, 318, 696, 339, 535, 220,
    878, 492, 327, 368, 76, 268,
    222, 560, 454, 719, 123, 673,
    592, 672, 202, 785, 448, 293,
    813, 319, 89, 701, 164, 444,
    996, 422, 236, 640, 11, 209,
    87, 365, 997, 715, 602, 853,
    668, 192, 687, 735, 123, 528,
    721, 30, 283, 602, 552, 285,
    688, 472, 971, 871, 223, 595,
    472, 907, 283, 238, 559, 842,
    570, 800, 327, 843, 435, 893,
    // Bin capacity
    1737, 1424, 1498, 1679, 909, 1502
  };
  const int i6_18_6_1_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    767, 255, 780, 186, 708, 3,
    160, 430, 701, 237, 92, 594,
    988, 931, 972, 219, 468, 561,
    971, 12, 248, 515, 891, 316,
    354, 819, 982, 755, 520, 647,
    842, 299, 24, 286, 575, 793,
    850, 862, 505, 786, 420, 188,
    968, 2, 520, 102, 920, 486,
    914, 667, 987, 576, 89, 27,
    680, 143, 551, 276, 263, 493,
    763, 611, 145, 690, 659, 664,
    786, 393, 390, 949, 527, 876,
    68, 536, 860, 630, 845, 557,
    974, 157, 914, 515, 981, 309,
    349, 426, 454, 636, 413, 779,
    915, 764, 409, 76, 26, 631,
    493, 652, 657, 294, 772, 223,
    817, 490, 25, 306, 324, 667,
    // Bin capacity
    2131, 1423, 1705, 1353, 1598, 1484
  };
  const int i6_18_6_2_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    175, 767, 575, 118, 377, 567,
    739, 319, 220, 984, 558, 326,
    107, 958, 500, 414, 194, 527,
    918, 95, 786, 404, 106, 797,
    91, 506, 18, 351, 836, 309,
    521, 688, 89, 69, 231, 862,
    156, 393, 32, 87, 583, 240,
    598, 944, 156, 543, 430, 604,
    766, 572, 557, 392, 282, 804,
    419, 457, 886, 184, 353, 150,
    974, 47, 581, 781, 216, 859,
    262, 835, 799, 968, 505, 763,
    534, 401, 677, 358, 738, 349,
    702, 686, 822, 650, 755, 878,
    78, 961, 8, 370, 33, 145,
    821, 842, 573, 160, 340, 709,
    414, 406, 372, 40, 541, 662,
    929, 289, 709, 72, 352, 985,
    // Bin capacity
    1565, 1729, 1422, 1181, 1264, 1792
  };
  const int i6_18_6_3_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    603, 485, 879, 779, 352, 390,
    520, 129, 842, 635, 991, 114,
    878, 757, 846, 969, 777, 874,
    514, 82, 264, 637, 488, 30,
    341, 467, 626, 128, 844, 182,
    457, 271, 834, 491, 569, 374,
    241, 255, 626, 731, 308, 706,
    531, 711, 501, 653, 434, 854,
    914, 798, 857, 139, 497, 990,
    780, 906, 726, 657, 22, 574,
    957, 947, 28, 245, 125, 827,
    850, 868, 457, 822, 920, 702,
    494, 546, 213, 113, 604, 268,
    588, 43, 411, 47, 620, 870,
    682, 136, 221, 115, 783, 749,
    519, 9, 827, 463, 128, 463,
    486, 888, 810, 73, 671, 541,
    191, 694, 352, 923, 925, 687,
    // Bin capacity
    1811, 1544, 1772, 1480, 1727, 1751
  };
  const int i6_18_6_4_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    522, 743, 356, 224, 218, 509,
    216, 289, 501, 194, 360, 164,
    0, 939, 337, 277, 561, 794,
    129, 710, 419, 127, 93, 419,
    951, 192, 132, 391, 80, 783,
    42, 849, 709, 389, 424, 437,
    991, 151, 423, 700, 236, 505,
    53, 284, 561, 717, 838, 229,
    354, 599, 269, 937, 282, 6,
    436, 271, 272, 790, 196, 908,
    900, 827, 802, 610, 632, 131,
    410, 33, 827, 149, 108, 359,
    897, 742, 637, 801, 52, 109,
    255, 362, 475, 121, 978, 326,
    154, 617, 750, 144, 989, 967,
    832, 993, 774, 869, 427, 450,
    182, 569, 401, 998, 760, 833,
    803, 774, 47, 967, 965, 937,
    // Bin capacity
    1409, 1724, 1507, 1631, 1422, 1537
  };
  const int i6_18_6_5_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    839, 862, 172, 995, 459, 366,
    449, 680, 422, 379, 249, 226,
    376, 258, 222, 607, 620, 680,
    518, 814, 608, 707, 779, 762,
    830, 830, 592, 377, 620, 654,
    731, 6, 277, 396, 308, 257,
    816, 612, 789, 406, 875, 417,
    904, 848, 477, 741, 293, 808,
    781, 672, 538, 879, 350, 100,
    594, 118, 659, 881, 598, 747,
    762, 498, 568, 375, 324, 159,
    731, 957, 831, 830, 845, 199,
    609, 891, 249, 491, 675, 16,
    391, 734, 991, 182, 971, 966,
    321, 784, 279, 538, 233, 422,
    300, 586, 864, 95, 49, 772,
    273, 543, 477, 29, 969, 977,
    384, 952, 485, 603, 319, 524,
    // Bin capacity
    1857, 2038, 1663, 1665, 1669, 1585
  };
  const int i6_18_6_6_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    4, 120, 723, 763, 488, 875,
    522, 593, 322, 706, 77, 40,
    588, 976, 763, 237, 702, 199,
    932, 149, 249, 495, 479, 137,
    14, 970, 226, 458, 6, 680,
    927, 199, 654, 529, 263, 163,
    143, 312, 515, 757, 991, 362,
    38, 671, 902, 81, 382, 606,
    488, 965, 892, 659, 892, 379,
    668, 387, 132, 184, 951, 368,
    447, 43, 97, 603, 395, 972,
    220, 817, 833, 430, 521, 370,
    878, 570, 542, 590, 399, 659,
    995, 252, 359, 723, 436, 836,
    306, 326, 210, 560, 653, 203,
    128, 421, 917, 350, 920, 631,
    173, 797, 70, 275, 141, 685,
    251, 743, 800, 85, 600, 564,
    // Bin capacity
    1365, 1645, 1627, 1500, 1643, 1543
  };
  const int i6_18_6_7_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    727, 506, 990, 801, 942, 230,
    487, 967, 839, 542, 365, 290,
    224, 374, 226, 705, 979, 265,
    630, 226, 163, 665, 605, 742,
    59, 684, 321, 575, 429, 385,
    621, 427, 403, 553, 423, 106,
    699, 120, 54, 130, 481, 288,
    523, 757, 312, 567, 930, 786,
    523, 602, 786, 463, 914, 750,
    250, 406, 661, 698, 524, 681,
    364, 29, 895, 408, 122, 443,
    209, 570, 595, 458, 51, 184,
    516, 421, 529, 180, 766, 7,
    279, 949, 775, 143, 100, 652,
    635, 109, 59, 231, 764, 516,
    443, 309, 519, 310, 619, 627,
    85, 225, 800, 980, 169, 328,
    714, 563, 39, 436, 48, 164,
    // Bin capacity
    1425, 1471, 1599, 1578, 1647, 1328
  };
  const int i6_18_6_8_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    805, 648, 65, 7, 666, 950,
    866, 983, 260, 681, 834, 344,
    899, 288, 633, 540, 798, 974,
    684, 692, 515, 818, 284, 366,
    276, 852, 426, 704, 7, 198,
    443, 204, 913, 31, 811, 949,
    352, 58, 87, 401, 491, 845,
    452, 44, 61, 555, 821, 294,
    745, 516, 863, 629, 878, 984,
    526, 671, 688, 884, 598, 377,
    458, 271, 474, 313, 191, 564,
    830, 540, 231, 876, 530, 572,
    559, 182, 321, 632, 132, 378,
    38, 851, 469, 694, 512, 359,
    895, 785, 75, 278, 744, 16,
    383, 782, 819, 615, 48, 30,
    409, 287, 965, 500, 735, 205,
    350, 948, 157, 52, 170, 947,
    // Bin capacity
    1795, 1729, 1444, 1658, 1665, 1684
  };
  const int i6_18_6_9_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    838, 404, 609, 640, 638, 850,
    11, 992, 711, 222, 767, 399,
    901, 296, 89, 454, 865, 263,
    800, 87, 910, 507, 925, 37,
    343, 986, 407, 145, 332, 539,
    228, 423, 331, 218, 397, 748,
    731, 497, 765, 727, 68, 972,
    738, 414, 82, 673, 742, 421,
    547, 957, 918, 586, 150, 154,
    273, 671, 362, 495, 171, 369,
    108, 32, 969, 465, 968, 558,
    22, 793, 840, 670, 414, 568,
    621, 928, 869, 122, 47, 759,
    35, 137, 719, 759, 655, 701,
    772, 465, 556, 395, 218, 544,
    866, 295, 557, 235, 986, 779,
    742, 641, 711, 206, 538, 304,
    196, 936, 137, 578, 731, 136,
    // Bin capacity
    1594, 1809, 1916, 1471, 1747, 1654
  };
  const int i6_18_6_10_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    907, 497, 767, 246, 502, 147,
    845, 192, 830, 29, 367, 651,
    955, 771, 318, 302, 678, 819,
    356, 930, 353, 436, 601, 102,
    179, 329, 913, 959, 155, 792,
    412, 868, 785, 533, 303, 360,
    203, 433, 402, 457, 289, 166,
    820, 822, 501, 197, 8, 963,
    724, 997, 717, 854, 956, 413,
    646, 725, 492, 176, 223, 111,
    448, 263, 878, 674, 675, 84,
    144, 215, 140, 890, 275, 352,
    887, 741, 83, 706, 326, 920,
    202, 445, 429, 202, 184, 944,
    208, 427, 637, 485, 362, 375,
    45, 455, 753, 461, 869, 983,
    600, 604, 189, 449, 732, 161,
    659, 102, 407, 916, 735, 936,
    // Bin capacity
    1694, 1800, 1759, 1645, 1511, 1702
  };
  const int i6_18_6_11_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    787, 707, 299, 255, 406, 352,
    666, 196, 427, 271, 884, 986,
    818, 26, 751, 792, 472, 453,
    174, 156, 759, 96, 704, 247,
    459, 57, 99, 434, 87, 726,
    108, 415, 419, 584, 662, 348,
    722, 342, 900, 390, 404, 213,
    900, 788, 768, 933, 20, 537,
    170, 411, 767, 248, 473, 691,
    274, 237, 536, 780, 138, 149,
    81, 379, 327, 611, 977, 46,
    663, 360, 293, 224, 299, 961,
    449, 780, 916, 854, 565, 615,
    174, 890, 5, 62, 263, 161,
    81, 597, 832, 947, 879, 777,
    430, 144, 975, 147, 952, 774,
    522, 765, 728, 409, 513, 29,
    803, 245, 246, 636, 459, 169,
    // Bin capacity
    1532, 1387, 1859, 1605, 1695, 1524
  };
  const int i6_18_6_12_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    953, 605, 670, 470, 403, 34,
    345, 776, 813, 681, 3, 730,
    26, 220, 43, 265, 469, 196,
    838, 804, 901, 901, 652, 861,
    536, 865, 630, 877, 561, 620,
    103, 103, 844, 193, 844, 82,
    584, 546, 224, 404, 123, 788,
    265, 742, 584, 119, 615, 704,
    778, 923, 467, 950, 178, 176,
    809, 727, 990, 388, 476, 849,
    911, 81, 817, 937, 519, 663,
    782, 112, 868, 444, 341, 500,
    835, 365, 683, 624, 820, 126,
    989, 921, 424, 808, 519, 983,
    467, 330, 246, 579, 837, 365,
    761, 600, 43, 664, 168, 88,
    637, 459, 662, 740, 640, 427,
    987, 843, 991, 633, 982, 126,
    // Bin capacity
    2167, 1871, 2035, 1994, 1709, 1553
  };
  const int i6_18_6_13_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    900, 278, 952, 356, 926, 728,
    615, 681, 52, 189, 260, 185,
    457, 725, 357, 534, 722, 894,
    305, 517, 676, 399, 835, 558,
    249, 434, 449, 44, 941, 313,
    242, 394, 892, 338, 651, 961,
    138, 722, 599, 998, 798, 197,
    50, 49, 493, 347, 137, 544,
    269, 995, 193, 15, 271, 991,
    671, 348, 253, 683, 369, 630,
    540, 944, 959, 384, 181, 815,
    852, 416, 166, 823, 106, 707,
    74, 523, 314, 407, 209, 535,
    827, 429, 308, 250, 296, 711,
    456, 525, 638, 696, 416, 343,
    660, 223, 321, 744, 452, 939,
    949, 665, 359, 768, 804, 627,
    670, 147, 260, 20, 10, 160,
    // Bin capacity
    1681, 1698, 1553, 1506, 1579, 2042
  };
  const int i6_18_6_14_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    820, 825, 472, 619, 343, 388,
    997, 911, 726, 389, 856, 877,
    292, 816, 250, 903, 802, 690,
    61, 348, 315, 152, 442, 592,
    890, 276, 776, 776, 942, 445,
    353, 311, 57, 588, 929, 789,
    53, 136, 52, 127, 861, 708,
    679, 889, 343, 937, 127, 350,
    376, 794, 326, 649, 511, 773,
    265, 828, 639, 98, 557, 739,
    538, 304, 943, 161, 91, 687,
    389, 292, 353, 998, 370, 173,
    552, 538, 288, 463, 657, 549,
    26, 205, 713, 482, 176, 469,
    429, 430, 313, 746, 497, 652,
    819, 267, 659, 619, 181, 686,
    682, 203, 456, 347, 302, 971,
    859, 638, 199, 206, 858, 97,
    // Bin capacity
    1726, 1713, 1498, 1760, 1806, 2021
  };
  const int i6_18_6_15_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    607, 565, 482, 608, 276, 982,
    869, 147, 870, 841, 42, 421,
    152, 562, 931, 402, 277, 991,
    589, 143, 715, 756, 168, 430,
    564, 510, 879, 555, 978, 925,
    958, 22, 467, 931, 789, 590,
    548, 184, 836, 401, 165, 801,
    207, 845, 15, 117, 825, 488,
    508, 374, 640, 298, 717, 234,
    500, 216, 694, 73, 762, 904,
    288, 129, 497, 665, 598, 959,
    445, 573, 648, 155, 374, 769,
    955, 633, 231, 164, 616, 146,
    416, 504, 774, 53, 467, 50,
    788, 801, 35, 280, 731, 205,
    408, 393, 853, 621, 423, 781,
    468, 858, 156, 598, 682, 914,
    881, 415, 6, 880, 641, 631,
    // Bin capacity
    1946, 1510, 1865, 1610, 1827, 2151
  };
  const int i6_18_6_16_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    53, 595, 929, 629, 802, 565,
    501, 307, 703, 373, 277, 296,
    762, 404, 524, 512, 582, 405,
    28, 488, 360, 43, 734, 865,
    685, 382, 38, 559, 1, 909,
    604, 961, 48, 437, 484, 919,
    74, 627, 329, 50, 150, 989,
    561, 685, 147, 937, 93, 494,
    192, 123, 187, 220, 306, 645,
    271, 446, 343, 402, 966, 53,
    756, 608, 606, 699, 152, 139,
    581, 397, 553, 772, 923, 34,
    846, 871, 458, 31, 722, 324,
    408, 756, 212, 541, 52, 48,
    708, 263, 450, 919, 204, 847,
    816, 171, 663, 319, 721, 631,
    35, 62, 876, 870, 582, 80,
    978, 599, 165, 394, 809, 47,
    // Bin capacity
    1713, 1691, 1468, 1684, 1655, 1603
  };
  const int i6_18_6_17_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    637, 264, 381, 854, 58, 184,
    54, 694, 926, 107, 753, 182,
    417, 932, 288, 731, 160, 781,
    695, 213, 496, 101, 64, 470,
    467, 899, 279, 695, 599, 760,
    817, 350, 903, 993, 772, 811,
    16, 401, 65, 790, 113, 625,
    913, 867, 351, 321, 50, 633,
    635, 540, 806, 569, 799, 440,
    514, 650, 325, 597, 43, 88,
    564, 160, 243, 719, 107, 321,
    295, 800, 786, 386, 850, 161,
    783, 554, 224, 305, 404, 153,
    774, 423, 33, 560, 393, 46,
    965, 486, 64, 468, 754, 964,
    23, 726, 662, 228, 72, 801,
    263, 704, 514, 990, 340, 552,
    966, 319, 756, 259, 218, 994,
    // Bin capacity
    1911, 1947, 1580, 1887, 1278, 1749
  };
  const int i6_18_6_18_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    100, 333, 359, 92, 443, 211,
    899, 204, 213, 184, 690, 219,
    483, 679, 221, 974, 168, 282,
    741, 287, 497, 136, 276, 981,
    332, 969, 9, 395, 418, 38,
    274, 279, 984, 214, 370, 577,
    842, 485, 397, 907, 249, 406,
    111, 597, 752, 282, 378, 577,
    238, 120, 967, 200, 693, 111,
    895, 528, 265, 419, 504, 24,
    277, 89, 663, 686, 917, 628,
    920, 900, 579, 365, 873, 2,
    360, 873, 965, 682, 541, 649,
    174, 909, 427, 390, 852, 544,
    302, 702, 205, 767, 960, 779,
    695, 640, 550, 882, 677, 577,
    978, 658, 285, 320, 473, 53,
    750, 889, 998, 432, 558, 669,
    // Bin capacity
    1843, 1995, 1837, 1638, 1975, 1441
  };
  const int i6_18_6_19_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    829, 175, 954, 55, 677, 424,
    388, 626, 53, 508, 595, 656,
    437, 686, 793, 982, 57, 128,
    797, 494, 87, 470, 915, 557,
    257, 429, 236, 400, 283, 496,
    420, 993, 373, 55, 381, 119,
    833, 554, 212, 609, 28, 902,
    260, 912, 832, 687, 43, 48,
    448, 22, 732, 898, 351, 558,
    761, 980, 526, 898, 10, 404,
    614, 862, 321, 463, 560, 224,
    502, 813, 115, 790, 854, 890,
    336, 183, 641, 341, 132, 695,
    102, 586, 318, 910, 527, 873,
    660, 466, 514, 54, 772, 501,
    751, 126, 73, 526, 251, 789,
    876, 295, 207, 721, 0, 998,
    952, 712, 168, 24, 55, 659,
    // Bin capacity
    2028, 1967, 1420, 1863, 1288, 1968
  };
  const int i6_18_6_20_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    665, 478, 342, 397, 559, 876,
    288, 576, 375, 624, 642, 848,
    148, 227, 228, 429, 150, 796,
    57, 900, 420, 288, 837, 811,
    317, 293, 995, 966, 410, 868,
    634, 470, 259, 183, 196, 836,
    774, 404, 842, 281, 337, 999,
    522, 84, 794, 907, 513, 311,
    165, 218, 71, 133, 352, 814,
    47, 683, 23, 555, 374, 549,
    900, 495, 552, 158, 560, 133,
    735, 480, 262, 227, 810, 727,
    868, 506, 272, 660, 919, 420,
    337, 484, 514, 63, 913, 732,
    465, 184, 764, 738, 910, 175,
    799, 372, 637, 58, 900, 954,
    771, 939, 58, 255, 439, 146,
    388, 497, 297, 57, 192, 321,
    // Bin capacity
    1776, 1658, 1541, 1396, 2003, 2264
  };
  const int i6_18_6_21_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    684, 894, 827, 411, 294, 126,
    386, 329, 159, 652, 812, 123,
    135, 99, 977, 895, 735, 397,
    610, 570, 679, 75, 39, 779,
    677, 91, 866, 727, 716, 134,
    175, 691, 124, 165, 204, 630,
    638, 950, 879, 699, 565, 526,
    329, 255, 402, 725, 182, 544,
    449, 822, 990, 177, 334, 770,
    251, 700, 366, 989, 200, 337,
    753, 406, 518, 212, 17, 1,
    179, 890, 424, 594, 675, 44,
    856, 928, 905, 905, 220, 918,
    131, 141, 608, 618, 649, 228,
    950, 738, 836, 162, 449, 696,
    333, 487, 24, 438, 1000, 978,
    476, 142, 759, 428, 919, 336,
    319, 7, 381, 559, 12, 905,
    // Bin capacity
    1681, 1844, 2163, 1902, 1618, 1709
  };
  const int i6_18_6_22_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    608, 419, 435, 701, 61, 251,
    142, 384, 85, 856, 389, 369,
    37, 225, 188, 599, 100, 23,
    384, 528, 618, 671, 104, 367,
    630, 307, 441, 133, 356, 510,
    18, 918, 197, 580, 497, 657,
    680, 580, 357, 82, 449, 716,
    792, 31, 79, 703, 406, 563,
    337, 620, 930, 908, 483, 150,
    949, 244, 471, 925, 344, 437,
    446, 549, 326, 708, 669, 179,
    876, 653, 282, 361, 286, 892,
    480, 937, 250, 173, 508, 751,
    229, 711, 894, 495, 206, 476,
    214, 400, 74, 853, 628, 369,
    70, 458, 335, 159, 392, 105,
    863, 52, 59, 152, 455, 410,
    612, 153, 575, 96, 458, 14,
    // Bin capacity
    1702, 1662, 1342, 1862, 1381, 1472
  };
  const int i6_18_6_23_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    580, 196, 752, 226, 139, 850,
    765, 765, 10, 863, 968, 297,
    242, 748, 817, 479, 33, 692,
    566, 786, 526, 138, 986, 953,
    204, 273, 378, 724, 235, 246,
    953, 111, 114, 84, 421, 516,
    25, 514, 390, 894, 778, 487,
    818, 747, 315, 335, 44, 690,
    534, 371, 729, 409, 401, 302,
    56, 497, 89, 499, 194, 159,
    247, 738, 284, 722, 692, 554,
    80, 191, 698, 627, 562, 837,
    741, 26, 121, 978, 444, 917,
    812, 130, 39, 911, 569, 336,
    434, 31, 447, 294, 192, 266,
    566, 188, 571, 285, 136, 581,
    636, 17, 310, 635, 225, 708,
    240, 997, 39, 359, 701, 902,
    // Bin capacity
    1743, 1502, 1359, 1940, 1583, 2111
  };
  const int i6_18_6_24_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    708, 487, 979, 51, 802, 550,
    673, 246, 43, 735, 182, 851,
    224, 122, 396, 971, 638, 514,
    657, 225, 257, 45, 428, 505,
    157, 379, 350, 271, 554, 780,
    678, 199, 54, 316, 651, 296,
    114, 62, 624, 107, 698, 113,
    101, 548, 537, 799, 343, 35,
    361, 780, 801, 656, 418, 530,
    450, 692, 618, 937, 173, 895,
    587, 800, 680, 590, 73, 991,
    319, 787, 120, 57, 854, 510,
    532, 201, 686, 353, 190, 983,
    717, 945, 846, 189, 439, 822,
    429, 484, 183, 980, 62, 600,
    721, 639, 549, 88, 101, 472,
    494, 541, 217, 797, 846, 837,
    583, 395, 35, 484, 519, 425,
    // Bin capacity
    1758, 1764, 1649, 1742, 1648, 2214
  };
  const int i6_18_6_25_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    989, 214, 735, 814, 381, 846,
    637, 679, 519, 576, 139, 449,
    221, 591, 693, 882, 602, 859,
    937, 746, 289, 589, 950, 455,
    587, 953, 306, 789, 531, 985,
    802, 10, 590, 477, 607, 945,
    7, 975, 712, 801, 130, 276,
    590, 640, 753, 406, 176, 228,
    905, 550, 260, 33, 74, 417,
    156, 520, 686, 437, 165, 754,
    180, 933, 997, 763, 279, 193,
    885, 151, 734, 65, 222, 144,
    752, 299, 775, 552, 471, 745,
    310, 191, 309, 528, 938, 528,
    72, 365, 753, 456, 706, 356,
    52, 976, 116, 804, 860, 334,
    479, 450, 914, 371, 105, 435,
    266, 339, 290, 828, 687, 326,
    // Bin capacity
    1839, 1997, 2174, 2119, 1672, 1933
  };
  const int i6_18_6_26_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    417, 880, 359, 232, 543, 145,
    31, 371, 998, 313, 553, 987,
    8, 803, 504, 397, 999, 221,
    259, 369, 636, 255, 638, 482,
    197, 871, 40, 129, 601, 128,
    381, 32, 522, 854, 375, 730,
    898, 194, 426, 43, 266, 765,
    323, 339, 327, 365, 644, 837,
    203, 40, 28, 420, 566, 868,
    818, 3, 199, 849, 575, 389,
    281, 697, 916, 430, 923, 53,
    198, 121, 986, 697, 565, 534,
    765, 116, 549, 745, 374, 516,
    160, 70, 777, 198, 183, 508,
    308, 167, 926, 16, 485, 732,
    238, 252, 949, 707, 236, 195,
    311, 107, 847, 278, 1, 409,
    241, 314, 780, 109, 913, 317,
    // Bin capacity
    1268, 1207, 2262, 1478, 1983, 1852
  };
  const int i6_18_6_27_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    336, 679, 35, 524, 640, 137,
    326, 32, 576, 934, 953, 136,
    730, 988, 847, 482, 575, 910,
    918, 742, 725, 376, 753, 358,
    819, 6, 29, 580, 224, 374,
    673, 101, 124, 74, 949, 59,
    523, 332, 258, 404, 367, 547,
    541, 390, 67, 697, 485, 908,
    340, 564, 169, 28, 481, 719,
    745, 912, 922, 449, 719, 292,
    118, 323, 191, 8, 490, 750,
    604, 409, 29, 784, 571, 361,
    822, 979, 758, 837, 32, 41,
    466, 544, 507, 147, 246, 533,
    428, 983, 933, 589, 469, 620,
    947, 337, 183, 881, 830, 358,
    285, 530, 40, 141, 96, 157,
    461, 750, 476, 770, 439, 885,
    // Bin capacity
    2135, 2033, 1454, 1843, 1973, 1725
  };
  const int i6_18_6_28_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    793, 846, 389, 288, 44, 931,
    421, 494, 354, 730, 532, 804,
    424, 938, 469, 38, 86, 829,
    812, 385, 862, 55, 890, 458,
    316, 318, 865, 770, 73, 728,
    30, 959, 653, 267, 652, 206,
    446, 81, 612, 732, 855, 209,
    554, 728, 710, 816, 676, 667,
    811, 264, 75, 645, 822, 919,
    106, 802, 742, 195, 513, 148,
    197, 654, 397, 540, 410, 495,
    408, 951, 839, 688, 691, 135,
    635, 264, 792, 925, 773, 675,
    250, 563, 103, 875, 551, 581,
    702, 116, 561, 123, 965, 714,
    54, 502, 657, 830, 538, 904,
    955, 562, 20, 843, 783, 208,
    581, 499, 547, 618, 763, 882,
    // Bin capacity
    1813, 2118, 2059, 2129, 2265, 2239
  };
  const int i6_18_6_29_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    26, 457, 235, 760, 438, 642,
    860, 177, 463, 350, 407, 503,
    931, 181, 762, 838, 930, 422,
    512, 796, 342, 328, 731, 300,
    700, 842, 508, 39, 426, 392,
    945, 777, 949, 534, 815, 685,
    119, 897, 52, 18, 897, 366,
    971, 935, 803, 708, 64, 281,
    259, 371, 62, 894, 400, 254,
    313, 822, 143, 486, 617, 789,
    310, 30, 58, 933, 818, 438,
    341, 293, 110, 507, 922, 561,
    806, 798, 960, 441, 403, 366,
    844, 794, 48, 156, 992, 334,
    309, 871, 465, 266, 998, 464,
    211, 452, 658, 351, 332, 281,
    651, 113, 148, 178, 435, 910,
    30, 852, 636, 914, 576, 923,
    // Bin capacity
    1965, 2249, 1592, 1871, 2409, 1916
  };
  const int i6_18_6_30_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    813, 444, 997, 236, 994, 33,
    699, 204, 671, 267, 904, 511,
    645, 405, 639, 790, 162, 731,
    82, 47, 103, 541, 217, 150,
    225, 949, 436, 697, 67, 498,
    818, 914, 364, 95, 652, 181,
    714, 511, 46, 607, 353, 337,
    897, 646, 166, 858, 645, 25,
    270, 137, 16, 195, 893, 222,
    984, 589, 722, 874, 453, 148,
    658, 918, 349, 136, 925, 593,
    571, 959, 486, 36, 515, 657,
    753, 26, 234, 514, 808, 524,
    994, 613, 293, 972, 174, 67,
    885, 829, 840, 622, 845, 574,
    823, 500, 534, 805, 856, 631,
    436, 125, 215, 825, 574, 755,
    457, 774, 477, 156, 836, 14,
    // Bin capacity
    2541, 2078, 1645, 1999, 2356, 1442
  };
  const int i6_18_6_31_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    755, 736, 196, 815, 138, 287,
    236, 619, 627, 140, 820, 285,
    804, 295, 399, 736, 199, 293,
    381, 382, 345, 796, 357, 208,
    923, 516, 522, 183, 445, 727,
    904, 874, 141, 878, 407, 330,
    422, 365, 126, 837, 545, 18,
    371, 3, 11, 989, 150, 513,
    901, 412, 551, 746, 304, 681,
    158, 409, 686, 699, 957, 536,
    860, 288, 725, 493, 8, 28,
    328, 797, 268, 632, 482, 115,
    750, 977, 773, 651, 533, 507,
    447, 76, 432, 312, 662, 971,
    622, 383, 636, 444, 842, 778,
    969, 95, 154, 547, 158, 502,
    108, 119, 473, 203, 181, 9,
    985, 896, 223, 125, 552, 543,
    // Bin capacity
    2386, 1800, 1592, 2233, 1690, 1601
  };
  const int i6_18_6_32_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    69, 141, 56, 132, 299, 113,
    222, 397, 282, 892, 799, 723,
    341, 600, 885, 383, 824, 603,
    12, 41, 921, 260, 833, 404,
    168, 211, 955, 465, 708, 820,
    276, 312, 611, 289, 493, 774,
    18, 898, 669, 710, 218, 877,
    299, 923, 795, 507, 739, 483,
    262, 252, 292, 178, 144, 29,
    467, 66, 515, 507, 599, 68,
    481, 200, 871, 375, 216, 503,
    66, 513, 892, 222, 849, 601,
    913, 37, 508, 221, 851, 786,
    886, 35, 239, 123, 651, 414,
    382, 90, 128, 595, 108, 369,
    974, 981, 691, 613, 764, 385,
    660, 74, 527, 478, 111, 752,
    770, 517, 116, 324, 753, 938,
    // Bin capacity
    1599, 1384, 2190, 1601, 2191, 2122
  };
  const int i6_18_6_33_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    768, 464, 236, 281, 734, 982,
    54, 38, 797, 446, 432, 4,
    227, 395, 545, 276, 403, 248,
    368, 436, 910, 635, 190, 753,
    319, 134, 383, 822, 18, 824,
    668, 97, 167, 648, 709, 975,
    674, 346, 580, 544, 434, 633,
    666, 844, 327, 881, 80, 82,
    772, 364, 672, 799, 155, 448,
    568, 195, 137, 410, 484, 688,
    5, 717, 197, 924, 608, 408,
    269, 780, 692, 360, 10, 73,
    753, 834, 926, 173, 283, 702,
    479, 658, 59, 526, 959, 373,
    789, 416, 473, 344, 33, 967,
    258, 544, 592, 271, 426, 931,
    25, 898, 899, 227, 280, 703,
    656, 29, 597, 741, 310, 347,
    // Bin capacity
    1844, 1816, 2037, 2064, 1452, 2248
  };
  const int i6_18_6_34_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    436, 31, 760, 496, 623, 239,
    376, 157, 520, 174, 425, 26,
    190, 130, 92, 156, 683, 939,
    166, 378, 131, 129, 722, 927,
    861, 651, 924, 32, 667, 796,
    669, 558, 959, 931, 255, 833,
    569, 772, 60, 585, 308, 516,
    692, 252, 882, 201, 816, 867,
    820, 438, 351, 497, 464, 987,
    291, 175, 106, 729, 625, 121,
    842, 385, 684, 915, 675, 365,
    637, 150, 738, 559, 133, 791,
    322, 368, 405, 379, 883, 110,
    566, 454, 576, 945, 311, 947,
    420, 304, 463, 496, 579, 10,
    756, 350, 607, 430, 160, 936,
    422, 40, 2, 915, 272, 219,
    727, 742, 841, 444, 279, 538,
    // Bin capacity
    2181, 1415, 2033, 2013, 1984, 2271
  };
  const int i6_18_6_35_33[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    363, 159, 890, 116, 445, 828,
    841, 857, 469, 489, 196, 729,
    908, 333, 368, 21, 737, 55,
    931, 823, 57, 387, 250, 858,
    603, 541, 291, 740, 196, 262,
    493, 496, 256, 64, 933, 194,
    246, 685, 392, 877, 675, 166,
    82, 918, 506, 101, 856, 895,
    49, 672, 428, 729, 619, 408,
    933, 258, 748, 544, 636, 63,
    668, 570, 827, 802, 12, 533,
    636, 310, 35, 384, 786, 445,
    324, 202, 711, 538, 780, 266,
    910, 446, 830, 158, 987, 961,
    484, 521, 656, 599, 594, 994,
    847, 729, 143, 924, 933, 746,
    299, 93, 853, 738, 683, 939,
    677, 187, 682, 187, 264, 662,
    // Bin capacity
    2317, 1980, 2057, 1890, 2381, 2251
  };
  const int i6_18_6_0_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    414, 84, 381, 892, 707, 260,
    898, 649, 753, 544, 556, 876,
    165, 233, 334, 290, 280, 728,
    531, 888, 881, 96, 932, 332,
    901, 157, 736, 527, 762, 175,
    770, 98, 648, 770, 884, 376,
    924, 277, 496, 136, 792, 769,
    668, 341, 929, 597, 423, 58,
    948, 568, 111, 466, 17, 42,
    999, 978, 120, 186, 985, 370,
    292, 752, 319, 488, 935, 8,
    843, 308, 964, 259, 93, 408,
    598, 510, 87, 768, 944, 739,
    951, 155, 305, 601, 30, 399,
    209, 333, 290, 132, 274, 791,
    511, 156, 665, 54, 951, 732,
    785, 517, 507, 735, 78, 265,
    283, 894, 312, 534, 804, 636,
    // Bin capacity
    1949, 1317, 1473, 1346, 1742, 1328
  };
  const int i6_18_6_1_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    537, 643, 948, 655, 416, 939,
    346, 25, 974, 816, 402, 209,
    672, 847, 456, 864, 226, 661,
    902, 397, 385, 32, 742, 526,
    87, 689, 545, 1000, 836, 329,
    486, 505, 20, 288, 412, 764,
    488, 15, 755, 496, 217, 881,
    711, 261, 165, 214, 371, 320,
    450, 987, 772, 947, 949, 215,
    440, 130, 427, 636, 18, 715,
    777, 610, 124, 940, 311, 727,
    301, 816, 297, 261, 296, 605,
    637, 601, 86, 404, 483, 881,
    321, 133, 215, 933, 486, 196,
    989, 867, 973, 949, 50, 22,
    618, 84, 849, 451, 998, 592,
    400, 581, 526, 290, 789, 476,
    49, 529, 941, 337, 719, 358,
    // Bin capacity
    1551, 1468, 1593, 1770, 1469, 1586
  };
  const int i6_18_6_2_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    947, 658, 121, 553, 663, 833,
    5, 201, 222, 110, 397, 990,
    906, 413, 306, 790, 964, 764,
    312, 289, 242, 324, 658, 989,
    565, 522, 609, 846, 319, 945,
    669, 697, 799, 644, 97, 131,
    306, 913, 344, 201, 328, 546,
    672, 330, 502, 137, 359, 153,
    741, 619, 911, 232, 192, 425,
    391, 856, 842, 479, 378, 766,
    543, 777, 209, 668, 566, 396,
    393, 699, 385, 279, 357, 188,
    84, 322, 339, 704, 70, 110,
    948, 205, 299, 611, 668, 481,
    737, 403, 781, 247, 10, 639,
    131, 933, 368, 666, 855, 709,
    763, 849, 996, 351, 746, 789,
    248, 601, 662, 590, 934, 148,
    // Bin capacity
    1592, 1749, 1520, 1434, 1456, 1701
  };
  const int i6_18_6_3_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    781, 736, 394, 684, 35, 655,
    767, 886, 255, 190, 52, 683,
    131, 388, 376, 33, 957, 306,
    300, 794, 729, 638, 816, 13,
    589, 850, 559, 414, 113, 914,
    294, 674, 148, 220, 805, 866,
    66, 468, 218, 162, 41, 570,
    955, 54, 762, 881, 679, 867,
    592, 590, 548, 511, 412, 24,
    523, 209, 183, 247, 893, 78,
    236, 736, 89, 854, 202, 180,
    325, 760, 296, 816, 704, 779,
    78, 939, 731, 608, 34, 752,
    118, 505, 824, 664, 943, 177,
    653, 738, 324, 62, 117, 76,
    912, 903, 238, 275, 704, 567,
    871, 750, 142, 973, 695, 247,
    564, 45, 614, 540, 607, 682,
    // Bin capacity
    1503, 1893, 1276, 1506, 1513, 1449
  };
  const int i6_18_6_4_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    379, 900, 707, 711, 444, 424,
    449, 462, 666, 167, 138, 209,
    554, 508, 774, 554, 328, 5,
    372, 475, 749, 416, 224, 280,
    144, 855, 509, 545, 597, 578,
    98, 225, 878, 516, 617, 488,
    736, 854, 503, 815, 111, 825,
    544, 990, 108, 777, 233, 191,
    901, 395, 481, 301, 453, 389,
    270, 445, 610, 569, 899, 520,
    509, 600, 219, 454, 426, 828,
    740, 775, 662, 905, 574, 160,
    491, 814, 321, 333, 318, 984,
    85, 878, 625, 467, 536, 445,
    883, 102, 330, 335, 622, 390,
    668, 292, 74, 242, 539, 709,
    128, 359, 147, 590, 326, 931,
    295, 181, 584, 685, 460, 731,
    // Bin capacity
    1430, 1753, 1551, 1627, 1360, 1576
  };
  const int i6_18_6_5_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    891, 419, 553, 481, 548, 95,
    118, 993, 818, 495, 287, 789,
    384, 62, 405, 308, 266, 578,
    941, 141, 545, 875, 425, 573,
    232, 718, 524, 564, 638, 470,
    447, 388, 791, 42, 984, 486,
    305, 546, 931, 367, 969, 258,
    491, 688, 417, 72, 342, 827,
    958, 59, 370, 160, 94, 621,
    899, 483, 3, 314, 880, 788,
    983, 859, 172, 539, 152, 298,
    481, 111, 593, 398, 330, 479,
    941, 479, 302, 764, 537, 762,
    60, 225, 327, 134, 131, 60,
    504, 432, 459, 332, 38, 377,
    591, 812, 677, 356, 196, 13,
    878, 985, 417, 199, 421, 336,
    723, 942, 437, 839, 469, 274,
    // Bin capacity
    1895, 1635, 1530, 1267, 1349, 1415
  };
  const int i6_18_6_6_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    51, 272, 362, 479, 591, 65,
    298, 900, 417, 759, 773, 736,
    947, 489, 854, 878, 244, 688,
    340, 172, 445, 800, 520, 277,
    602, 715, 206, 554, 354, 647,
    562, 868, 515, 711, 19, 815,
    68, 965, 259, 719, 498, 963,
    796, 211, 126, 579, 522, 787,
    314, 553, 420, 786, 221, 820,
    197, 458, 12, 650, 840, 170,
    376, 230, 913, 570, 201, 33,
    994, 578, 892, 275, 196, 839,
    280, 511, 328, 794, 198, 534,
    676, 924, 98, 343, 454, 337,
    954, 731, 775, 954, 800, 110,
    657, 837, 936, 157, 55, 888,
    551, 557, 958, 103, 605, 991,
    564, 767, 354, 926, 381, 27,
    // Bin capacity
    1631, 1898, 1568, 1950, 1321, 1719
  };
  const int i6_18_6_7_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    35, 47, 827, 395, 266, 762,
    99, 502, 270, 967, 214, 471,
    620, 757, 856, 86, 389, 983,
    1, 267, 783, 305, 562, 547,
    602, 259, 277, 318, 498, 874,
    393, 441, 81, 558, 78, 922,
    177, 72, 972, 203, 224, 544,
    669, 440, 235, 557, 386, 388,
    369, 202, 167, 131, 671, 835,
    859, 791, 919, 82, 259, 454,
    475, 694, 679, 305, 359, 314,
    542, 674, 544, 930, 9, 715,
    584, 187, 750, 942, 292, 480,
    919, 250, 738, 52, 491, 72,
    347, 154, 414, 496, 165, 865,
    499, 346, 775, 224, 763, 378,
    508, 109, 26, 435, 74, 425,
    211, 535, 499, 869, 571, 825,
    // Bin capacity
    1411, 1200, 1750, 1401, 1119, 1936
  };
  const int i6_18_6_8_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    422, 816, 978, 823, 286, 170,
    385, 904, 303, 470, 417, 90,
    177, 910, 428, 759, 76, 498,
    504, 124, 142, 922, 915, 629,
    808, 356, 181, 624, 545, 219,
    641, 13, 309, 570, 182, 838,
    591, 827, 456, 874, 410, 227,
    855, 587, 132, 889, 98, 754,
    171, 878, 966, 734, 597, 341,
    443, 902, 290, 95, 333, 838,
    669, 786, 598, 934, 521, 419,
    359, 393, 86, 699, 704, 827,
    775, 627, 141, 716, 936, 70,
    558, 153, 443, 653, 934, 664,
    965, 387, 481, 219, 383, 618,
    241, 177, 200, 194, 587, 5,
    553, 8, 518, 573, 241, 2,
    636, 598, 544, 883, 776, 945,
    // Bin capacity
    1756, 1701, 1296, 2094, 1610, 1468
  };
  const int i6_18_6_9_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    387, 347, 575, 173, 95, 824,
    777, 807, 261, 917, 315, 942,
    639, 17, 183, 85, 571, 156,
    575, 892, 125, 844, 319, 71,
    548, 278, 850, 900, 274, 778,
    559, 784, 984, 530, 714, 610,
    514, 655, 668, 665, 237, 647,
    13, 578, 92, 213, 100, 448,
    182, 721, 669, 56, 178, 60,
    704, 621, 776, 790, 370, 145,
    53, 113, 987, 209, 314, 971,
    399, 503, 116, 940, 26, 309,
    851, 0, 415, 964, 507, 244,
    696, 627, 929, 306, 719, 986,
    205, 698, 229, 806, 179, 982,
    551, 387, 85, 487, 188, 779,
    333, 268, 544, 760, 180, 920,
    445, 914, 960, 760, 459, 1,
    // Bin capacity
    1532, 1674, 1717, 1891, 1044, 1794
  };
  const int i6_18_6_10_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    18, 620, 530, 428, 989, 612,
    13, 499, 405, 974, 415, 790,
    608, 451, 113, 486, 919, 346,
    208, 672, 152, 670, 764, 80,
    242, 82, 873, 321, 361, 827,
    831, 839, 108, 422, 92, 234,
    251, 152, 770, 60, 192, 938,
    523, 514, 100, 647, 320, 618,
    810, 576, 300, 682, 231, 901,
    935, 444, 486, 749, 749, 881,
    597, 423, 606, 576, 157, 495,
    359, 279, 824, 37, 599, 189,
    392, 619, 755, 60, 112, 494,
    410, 946, 703, 34, 543, 147,
    364, 972, 621, 352, 10, 236,
    45, 124, 612, 391, 451, 67,
    89, 127, 174, 171, 637, 849,
    784, 707, 805, 264, 137, 771,
    // Bin capacity
    1372, 1659, 1639, 1343, 1408, 1738
  };
  const int i6_18_6_11_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    281, 77, 949, 925, 894, 30,
    835, 425, 74, 514, 424, 827,
    746, 109, 316, 695, 930, 426,
    54, 961, 957, 985, 562, 570,
    121, 206, 921, 957, 992, 312,
    437, 579, 36, 809, 730, 831,
    813, 182, 448, 367, 508, 617,
    638, 555, 226, 669, 424, 772,
    571, 950, 681, 210, 420, 245,
    582, 752, 819, 831, 297, 423,
    604, 748, 211, 820, 269, 655,
    369, 154, 236, 537, 822, 687,
    229, 821, 817, 917, 253, 645,
    895, 435, 929, 43, 616, 905,
    909, 702, 718, 471, 642, 787,
    537, 888, 540, 119, 119, 537,
    773, 733, 331, 229, 450, 555,
    20, 696, 361, 880, 734, 814,
    // Bin capacity
    1742, 1846, 1771, 2031, 1866, 1969
  };
  const int i6_18_6_12_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    752, 163, 411, 641, 541, 927,
    458, 878, 239, 995, 391, 2,
    143, 770, 861, 854, 510, 846,
    14, 534, 887, 168, 424, 398,
    237, 177, 894, 969, 724, 544,
    996, 732, 616, 653, 75, 106,
    72, 683, 509, 122, 248, 362,
    156, 859, 154, 865, 605, 565,
    392, 901, 734, 707, 27, 397,
    671, 711, 382, 317, 699, 294,
    448, 623, 459, 788, 77, 107,
    1000, 56, 233, 188, 524, 387,
    189, 92, 471, 558, 30, 464,
    121, 554, 628, 172, 715, 583,
    889, 656, 40, 719, 715, 574,
    712, 604, 153, 849, 438, 31,
    90, 320, 756, 33, 321, 17,
    207, 443, 320, 655, 79, 459,
    // Bin capacity
    1409, 1822, 1633, 1914, 1334, 1319
  };
  const int i6_18_6_13_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    263, 573, 236, 968, 870, 594,
    602, 238, 509, 682, 107, 584,
    307, 181, 468, 806, 245, 136,
    648, 515, 368, 389, 500, 395,
    339, 281, 851, 535, 462, 397,
    568, 862, 869, 259, 102, 828,
    568, 350, 897, 663, 292, 840,
    304, 46, 742, 47, 387, 622,
    412, 191, 752, 802, 263, 491,
    803, 976, 199, 520, 714, 263,
    872, 808, 146, 905, 504, 166,
    27, 310, 587, 845, 404, 290,
    250, 612, 116, 682, 421, 966,
    516, 597, 32, 287, 74, 720,
    610, 19, 80, 340, 650, 223,
    989, 213, 400, 273, 349, 357,
    530, 100, 202, 505, 573, 989,
    824, 922, 514, 138, 196, 538,
    // Bin capacity
    1777, 1468, 1501, 1817, 1340, 1771
  };
  const int i6_18_6_14_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    599, 736, 553, 786, 975, 147,
    407, 626, 886, 876, 4, 110,
    695, 809, 703, 160, 508, 305,
    294, 277, 941, 940, 156, 290,
    957, 50, 509, 140, 878, 964,
    564, 910, 81, 606, 827, 800,
    94, 288, 284, 851, 122, 372,
    352, 425, 299, 739, 781, 109,
    861, 918, 687, 861, 512, 370,
    288, 891, 577, 296, 419, 864,
    498, 380, 728, 266, 763, 587,
    241, 817, 564, 533, 394, 633,
    846, 761, 427, 345, 614, 885,
    189, 307, 529, 278, 704, 637,
    306, 986, 66, 252, 258, 320,
    101, 853, 299, 91, 781, 490,
    628, 896, 701, 252, 237, 248,
    274, 543, 895, 618, 430, 298,
    // Bin capacity
    1557, 2180, 1849, 1690, 1779, 1602
  };
  const int i6_18_6_15_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    94, 694, 241, 691, 868, 451,
    496, 332, 549, 685, 400, 116,
    383, 353, 89, 555, 410, 376,
    471, 729, 133, 300, 605, 206,
    839, 936, 539, 807, 895, 284,
    287, 62, 849, 43, 405, 316,
    877, 345, 509, 687, 274, 65,
    223, 843, 722, 328, 3, 828,
    383, 102, 284, 384, 278, 407,
    498, 527, 976, 586, 656, 43,
    986, 323, 275, 685, 930, 904,
    805, 608, 895, 6, 310, 995,
    721, 471, 291, 321, 669, 33,
    289, 215, 307, 591, 96, 673,
    888, 816, 995, 293, 263, 484,
    411, 375, 279, 37, 264, 128,
    702, 983, 460, 66, 684, 707,
    364, 478, 897, 357, 528, 486,
    // Bin capacity
    1863, 1762, 1781, 1423, 1637, 1438
  };
  const int i6_18_6_16_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    121, 231, 120, 223, 891, 190,
    741, 585, 692, 296, 626, 683,
    194, 723, 437, 673, 372, 51,
    785, 972, 101, 480, 931, 104,
    213, 974, 79, 863, 107, 725,
    208, 299, 287, 394, 817, 548,
    749, 371, 745, 226, 857, 933,
    297, 327, 52, 721, 451, 33,
    715, 506, 324, 300, 385, 271,
    540, 615, 875, 564, 560, 452,
    690, 480, 873, 161, 903, 45,
    398, 118, 568, 213, 415, 897,
    543, 735, 28, 44, 497, 979,
    485, 649, 987, 784, 703, 874,
    191, 812, 753, 503, 103, 670,
    964, 256, 621, 375, 728, 996,
    814, 847, 33, 792, 642, 181,
    452, 975, 899, 197, 161, 17,
    // Bin capacity
    1760, 2026, 1639, 1510, 1963, 1673
  };
  const int i6_18_6_17_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    229, 257, 254, 533, 491, 313,
    10, 679, 917, 478, 402, 237,
    110, 95, 897, 147, 450, 84,
    596, 368, 471, 361, 872, 462,
    317, 717, 263, 143, 61, 741,
    923, 904, 198, 239, 383, 829,
    62, 778, 897, 834, 927, 359,
    482, 259, 595, 19, 472, 892,
    883, 913, 40, 447, 686, 908,
    565, 363, 18, 856, 8, 341,
    355, 710, 352, 614, 671, 104,
    692, 342, 180, 890, 310, 554,
    117, 654, 760, 265, 475, 277,
    176, 250, 837, 420, 251, 423,
    282, 321, 664, 620, 272, 180,
    537, 376, 310, 769, 425, 62,
    344, 303, 744, 549, 125, 83,
    972, 967, 577, 303, 238, 385,
    // Bin capacity
    1493, 1805, 1750, 1655, 1467, 1411
  };
  const int i6_18_6_18_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    908, 381, 320, 208, 227, 26,
    35, 877, 99, 549, 690, 567,
    273, 727, 786, 667, 628, 97,
    991, 426, 951, 130, 117, 408,
    236, 309, 204, 774, 573, 548,
    520, 611, 177, 510, 536, 745,
    239, 21, 128, 763, 749, 694,
    932, 89, 566, 731, 915, 318,
    208, 528, 607, 547, 560, 110,
    569, 395, 145, 167, 363, 132,
    272, 11, 639, 497, 105, 932,
    562, 793, 692, 199, 401, 563,
    951, 610, 135, 694, 624, 353,
    153, 247, 777, 229, 74, 472,
    746, 343, 290, 612, 411, 709,
    531, 499, 451, 695, 879, 991,
    368, 20, 250, 238, 438, 579,
    664, 185, 832, 49, 737, 468,
    // Bin capacity
    1802, 1391, 1583, 1625, 1776, 1714
  };
  const int i6_18_6_19_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    819, 112, 770, 834, 975, 985,
    592, 223, 98, 216, 169, 813,
    930, 599, 762, 892, 282, 986,
    436, 278, 419, 115, 776, 403,
    252, 161, 359, 143, 711, 890,
    927, 508, 555, 53, 522, 639,
    656, 946, 725, 397, 857, 427,
    174, 803, 485, 979, 653, 868,
    999, 895, 441, 477, 332, 260,
    738, 405, 67, 711, 180, 442,
    647, 554, 853, 784, 438, 588,
    122, 24, 480, 576, 162, 691,
    756, 23, 44, 202, 330, 896,
    994, 647, 659, 636, 131, 770,
    732, 487, 719, 893, 772, 103,
    371, 673, 520, 459, 330, 284,
    28, 812, 659, 325, 562, 462,
    335, 780, 555, 987, 717, 196,
    // Bin capacity
    2085, 1772, 1819, 1920, 1765, 2123
  };
  const int i6_18_6_20_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    506, 966, 272, 77, 3, 133,
    288, 998, 428, 312, 869, 373,
    162, 15, 183, 94, 683, 903,
    266, 114, 708, 692, 210, 613,
    679, 138, 220, 949, 151, 782,
    930, 187, 452, 629, 554, 660,
    237, 448, 807, 388, 459, 476,
    258, 899, 809, 199, 373, 211,
    977, 389, 352, 215, 731, 903,
    745, 40, 857, 759, 558, 211,
    512, 990, 344, 44, 50, 51,
    160, 548, 523, 125, 442, 244,
    562, 15, 601, 855, 584, 732,
    952, 125, 660, 575, 205, 615,
    122, 872, 380, 265, 558, 599,
    394, 882, 654, 371, 407, 278,
    347, 586, 206, 111, 72, 683,
    100, 514, 608, 25, 182, 103,
    // Bin capacity
    1640, 1746, 1813, 1337, 1419, 1714
  };
  const int i6_18_6_21_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    214, 452, 815, 951, 311, 26,
    959, 373, 733, 961, 646, 73,
    174, 120, 46, 987, 458, 238,
    467, 172, 111, 10, 938, 292,
    616, 793, 703, 352, 327, 216,
    418, 287, 102, 824, 764, 874,
    525, 705, 793, 897, 521, 603,
    628, 733, 984, 360, 885, 19,
    446, 87, 752, 390, 973, 709,
    174, 967, 670, 690, 609, 763,
    64, 123, 453, 498, 771, 828,
    80, 226, 884, 935, 703, 999,
    976, 573, 727, 291, 476, 297,
    216, 858, 941, 132, 812, 414,
    596, 64, 98, 703, 710, 151,
    22, 429, 173, 867, 418, 500,
    998, 232, 897, 316, 237, 865,
    161, 560, 630, 162, 68, 731,
    // Bin capacity
    1560, 1564, 2120, 2083, 2144, 1734
  };
  const int i6_18_6_22_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    876, 436, 769, 809, 504, 738,
    112, 175, 668, 870, 530, 842,
    435, 399, 586, 668, 0, 268,
    791, 363, 593, 553, 841, 260,
    449, 979, 946, 861, 59, 78,
    889, 86, 426, 124, 27, 572,
    258, 404, 777, 148, 565, 951,
    537, 428, 184, 130, 210, 740,
    272, 124, 46, 153, 812, 590,
    622, 370, 926, 112, 383, 460,
    788, 432, 472, 219, 188, 399,
    348, 886, 297, 816, 241, 340,
    580, 578, 639, 181, 280, 981,
    389, 411, 671, 415, 161, 518,
    439, 337, 110, 928, 828, 105,
    366, 558, 507, 597, 330, 243,
    39, 154, 165, 806, 466, 891,
    472, 434, 952, 989, 427, 721,
    // Bin capacity
    1762, 1536, 1980, 1908, 1394, 1972
  };
  const int i6_18_6_23_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    591, 847, 970, 628, 973, 183,
    604, 267, 640, 726, 517, 177,
    83, 210, 558, 428, 364, 838,
    947, 691, 344, 686, 586, 150,
    134, 495, 358, 584, 895, 589,
    547, 713, 981, 392, 797, 57,
    789, 573, 497, 106, 379, 628,
    377, 197, 420, 975, 565, 631,
    455, 750, 852, 635, 937, 879,
    389, 755, 796, 394, 98, 511,
    549, 662, 762, 84, 0, 24,
    78, 908, 306, 601, 369, 455,
    367, 202, 922, 700, 567, 325,
    600, 411, 97, 355, 793, 661,
    984, 953, 106, 481, 454, 246,
    90, 512, 585, 932, 443, 556,
    971, 178, 579, 215, 282, 868,
    631, 757, 618, 847, 674, 636,
    // Bin capacity
    1884, 2067, 2131, 2003, 1988, 1725
  };
  const int i6_18_6_24_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    386, 155, 335, 313, 431, 85,
    846, 583, 679, 79, 147, 62,
    887, 129, 28, 61, 710, 734,
    40, 829, 977, 23, 17, 896,
    328, 790, 516, 331, 887, 336,
    706, 627, 846, 247, 124, 628,
    839, 726, 423, 728, 994, 192,
    658, 506, 371, 8, 547, 166,
    190, 72, 650, 789, 178, 907,
    641, 176, 941, 288, 765, 281,
    263, 943, 761, 326, 443, 14,
    967, 229, 263, 35, 843, 499,
    939, 447, 354, 490, 200, 674,
    776, 580, 234, 813, 633, 317,
    645, 844, 794, 138, 558, 443,
    610, 815, 62, 319, 105, 11,
    377, 500, 514, 252, 894, 640,
    584, 932, 309, 903, 577, 198,
    // Bin capacity
    2208, 2043, 1872, 1270, 1871, 1464
  };
  const int i6_18_6_25_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    32, 746, 711, 518, 259, 780,
    571, 317, 917, 296, 768, 813,
    493, 368, 740, 829, 903, 353,
    508, 849, 18, 66, 245, 535,
    315, 972, 570, 831, 974, 708,
    514, 592, 797, 418, 439, 29,
    380, 170, 515, 675, 476, 402,
    909, 106, 352, 333, 757, 482,
    894, 515, 595, 604, 998, 338,
    549, 926, 868, 856, 934, 964,
    481, 135, 203, 68, 939, 967,
    60, 32, 343, 723, 689, 533,
    273, 736, 407, 994, 881, 232,
    925, 572, 844, 371, 748, 907,
    770, 221, 776, 100, 606, 599,
    243, 851, 377, 515, 426, 787,
    946, 398, 182, 103, 39, 338,
    486, 244, 994, 830, 926, 311,
    // Bin capacity
    1948, 1823, 2127, 1903, 2502, 2100
  };
  const int i6_18_6_26_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    105, 856, 828, 60, 914, 888,
    371, 178, 772, 508, 238, 290,
    845, 727, 853, 919, 84, 35,
    547, 316, 369, 935, 769, 904,
    540, 168, 221, 910, 430, 256,
    271, 571, 108, 537, 976, 222,
    339, 953, 465, 260, 214, 338,
    996, 446, 868, 405, 787, 127,
    992, 938, 254, 320, 772, 760,
    869, 284, 356, 737, 141, 224,
    109, 837, 71, 388, 457, 902,
    185, 949, 632, 191, 36, 210,
    898, 645, 903, 444, 49, 585,
    32, 268, 369, 752, 951, 551,
    502, 44, 402, 773, 582, 418,
    217, 164, 779, 176, 551, 922,
    316, 763, 230, 209, 980, 320,
    622, 758, 369, 380, 435, 5,
    // Bin capacity
    1839, 2072, 1859, 1870, 1967, 1671
  };
  const int i6_18_6_27_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    585, 819, 436, 708, 428, 591,
    747, 50, 253, 481, 270, 300,
    545, 65, 281, 880, 530, 884,
    224, 837, 638, 894, 285, 310,
    71, 343, 59, 420, 646, 41,
    160, 417, 751, 898, 215, 799,
    989, 325, 455, 409, 577, 277,
    417, 994, 60, 61, 958, 128,
    53, 804, 357, 439, 68, 227,
    175, 933, 448, 366, 968, 276,
    198, 74, 442, 944, 655, 914,
    560, 775, 636, 577, 554, 899,
    760, 579, 666, 517, 221, 787,
    657, 490, 5, 896, 802, 588,
    293, 503, 436, 49, 951, 825,
    825, 564, 958, 592, 850, 560,
    431, 788, 814, 318, 535, 911,
    273, 306, 849, 453, 76, 766,
    // Bin capacity
    1686, 2046, 1809, 2096, 2030, 2135
  };
  const int i6_18_6_28_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    274, 873, 971, 13, 804, 840,
    531, 193, 997, 109, 684, 761,
    127, 148, 114, 912, 668, 650,
    161, 147, 597, 549, 924, 425,
    179, 20, 197, 437, 367, 27,
    597, 883, 15, 16, 988, 25,
    154, 173, 418, 5, 170, 156,
    372, 729, 866, 212, 473, 270,
    484, 644, 40, 919, 893, 42,
    773, 993, 790, 758, 420, 272,
    851, 676, 675, 959, 308, 926,
    828, 20, 620, 887, 415, 993,
    310, 116, 697, 745, 443, 669,
    930, 803, 437, 933, 797, 975,
    365, 376, 364, 928, 148, 73,
    547, 16, 894, 161, 102, 69,
    19, 639, 596, 790, 953, 55,
    597, 66, 38, 636, 197, 306,
    // Bin capacity
    1728, 1604, 1990, 2127, 2081, 1608
  };
  const int i6_18_6_29_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    580, 800, 207, 87, 404, 723,
    794, 436, 766, 778, 641, 919,
    588, 192, 393, 744, 422, 623,
    903, 207, 944, 936, 299, 684,
    277, 655, 115, 732, 77, 438,
    148, 879, 131, 860, 856, 190,
    173, 324, 916, 991, 812, 912,
    94, 138, 859, 366, 194, 939,
    624, 76, 28, 325, 330, 662,
    107, 630, 79, 87, 808, 15,
    754, 146, 724, 790, 887, 724,
    234, 720, 346, 917, 799, 872,
    889, 124, 909, 6, 909, 497,
    17, 450, 780, 500, 316, 618,
    498, 873, 327, 421, 633, 612,
    751, 957, 852, 863, 671, 115,
    665, 351, 310, 428, 65, 681,
    811, 336, 683, 489, 842, 370,
    // Bin capacity
    1916, 1784, 2015, 2219, 2143, 2278
  };
  const int i6_18_6_30_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    77, 662, 902, 538, 67, 844,
    122, 804, 653, 810, 52, 606,
    575, 945, 962, 783, 866, 509,
    344, 764, 207, 758, 332, 51,
    130, 531, 788, 870, 466, 858,
    897, 800, 324, 296, 255, 35,
    527, 796, 678, 677, 149, 872,
    548, 90, 37, 629, 76, 656,
    693, 827, 523, 530, 426, 165,
    272, 104, 192, 770, 538, 212,
    978, 673, 459, 700, 714, 763,
    788, 506, 783, 623, 664, 488,
    883, 639, 238, 753, 255, 558,
    674, 771, 450, 686, 285, 208,
    226, 241, 747, 574, 660, 529,
    636, 302, 477, 511, 690, 829,
    215, 349, 437, 935, 214, 341,
    844, 528, 447, 418, 26, 946,
    // Bin capacity
    2043, 2239, 2016, 2570, 1460, 2052
  };
  const int i6_18_6_31_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    213, 911, 206, 123, 735, 960,
    172, 443, 808, 596, 645, 783,
    592, 265, 479, 957, 890, 623,
    262, 539, 469, 632, 539, 111,
    777, 915, 907, 245, 806, 711,
    535, 881, 936, 553, 695, 585,
    151, 297, 665, 283, 626, 91,
    610, 99, 524, 446, 22, 159,
    391, 242, 804, 140, 557, 426,
    623, 289, 389, 796, 374, 603,
    543, 793, 454, 139, 171, 766,
    754, 164, 923, 412, 443, 369,
    898, 74, 243, 241, 700, 151,
    498, 359, 482, 670, 931, 910,
    563, 405, 790, 612, 238, 792,
    585, 26, 818, 593, 77, 511,
    492, 54, 94, 305, 274, 200,
    845, 378, 24, 883, 109, 9,
    // Bin capacity
    2076, 1558, 2187, 1884, 1929, 1913
  };
  const int i6_18_6_32_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    661, 162, 663, 471, 88, 472,
    755, 761, 169, 967, 833, 386,
    555, 151, 535, 101, 9, 700,
    259, 575, 320, 953, 513, 685,
    317, 951, 408, 96, 960, 532,
    548, 35, 23, 886, 688, 380,
    579, 849, 787, 38, 88, 788,
    847, 356, 622, 281, 261, 158,
    83, 34, 341, 58, 951, 808,
    160, 698, 339, 659, 739, 174,
    1, 653, 282, 771, 100, 6,
    615, 487, 973, 494, 964, 352,
    8, 134, 135, 608, 491, 906,
    566, 391, 505, 296, 411, 824,
    555, 837, 218, 138, 446, 974,
    337, 12, 613, 757, 580, 547,
    122, 229, 405, 395, 261, 629,
    577, 546, 614, 41, 402, 222,
    // Bin capacity
    1660, 1730, 1750, 1763, 1933, 2100
  };
  const int i6_18_6_33_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    13, 187, 62, 234, 175, 835,
    846, 268, 525, 102, 683, 695,
    188, 581, 34, 8, 487, 300,
    958, 295, 852, 705, 349, 774,
    452, 42, 962, 90, 661, 287,
    608, 156, 388, 142, 434, 244,
    964, 264, 322, 425, 1, 814,
    88, 538, 523, 88, 698, 214,
    393, 599, 166, 346, 831, 950,
    990, 830, 47, 80, 818, 553,
    127, 353, 98, 339, 378, 755,
    16, 456, 498, 19, 335, 758,
    79, 983, 184, 592, 312, 973,
    446, 762, 714, 917, 56, 417,
    747, 752, 930, 821, 947, 179,
    974, 336, 333, 653, 616, 528,
    89, 895, 55, 172, 445, 535,
    464, 165, 231, 204, 552, 532,
    // Bin capacity
    1872, 1876, 1535, 1317, 1946, 2293
  };
  const int i6_18_6_34_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    709, 401, 106, 545, 601, 171,
    631, 80, 254, 708, 736, 975,
    367, 647, 392, 589, 271, 276,
    76, 767, 149, 383, 570, 909,
    177, 418, 536, 760, 218, 928,
    719, 8, 47, 888, 177, 135,
    207, 50, 838, 503, 934, 722,
    347, 713, 439, 206, 432, 278,
    258, 172, 95, 854, 122, 552,
    978, 715, 469, 693, 811, 703,
    774, 883, 91, 541, 240, 205,
    155, 92, 302, 765, 258, 501,
    496, 984, 787, 864, 38, 247,
    655, 837, 389, 675, 626, 153,
    254, 120, 643, 867, 496, 998,
    143, 206, 801, 517, 590, 404,
    789, 93, 569, 951, 949, 403,
    399, 653, 794, 424, 960, 192,
    // Bin capacity
    1817, 1751, 1720, 2621, 2017, 1955
  };
  const int i6_18_6_35_34[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    237, 444, 422, 265, 990, 623,
    453, 495, 411, 288, 437, 42,
    112, 845, 94, 289, 650, 520,
    849, 334, 94, 283, 912, 224,
    707, 496, 56, 55, 19, 931,
    499, 761, 397, 73, 795, 779,
    33, 775, 735, 795, 385, 412,
    311, 655, 985, 814, 327, 885,
    924, 415, 565, 521, 793, 501,
    433, 410, 785, 349, 912, 707,
    58, 898, 966, 633, 180, 960,
    680, 53, 102, 214, 663, 410,
    140, 596, 265, 960, 859, 981,
    711, 811, 152, 132, 49, 610,
    461, 472, 809, 820, 932, 608,
    340, 978, 335, 666, 155, 785,
    492, 273, 825, 840, 644, 469,
    420, 803, 415, 48, 288, 727,
    // Bin capacity
    1769, 2366, 1893, 1811, 2248, 2515
  };
  const int i6_18_6_0_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    690, 237, 363, 544, 761, 188,
    776, 942, 762, 661, 749, 833,
    172, 559, 969, 904, 989, 184,
    86, 869, 87, 393, 858, 505,
    499, 775, 832, 42, 173, 141,
    490, 22, 103, 284, 239, 566,
    498, 194, 326, 780, 162, 211,
    3, 29, 636, 811, 547, 519,
    923, 519, 240, 558, 39, 600,
    416, 155, 995, 303, 621, 826,
    940, 780, 405, 452, 656, 917,
    247, 544, 532, 398, 823, 584,
    457, 795, 217, 46, 889, 164,
    837, 511, 903, 955, 774, 634,
    254, 284, 977, 812, 769, 304,
    210, 573, 406, 78, 536, 460,
    254, 193, 767, 19, 646, 837,
    422, 752, 33, 935, 745, 717,
    // Bin capacity
    1363, 1456, 1593, 1496, 1830, 1532
  };
  const int i6_18_6_1_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    612, 929, 664, 629, 97, 22,
    377, 146, 872, 881, 803, 234,
    197, 432, 344, 36, 269, 294,
    798, 178, 660, 810, 421, 457,
    254, 435, 335, 7, 354, 104,
    649, 467, 124, 535, 596, 524,
    948, 180, 324, 97, 306, 938,
    809, 294, 397, 893, 804, 623,
    925, 117, 27, 313, 209, 557,
    511, 888, 926, 562, 188, 200,
    595, 121, 886, 41, 371, 737,
    141, 304, 231, 944, 299, 775,
    359, 508, 726, 54, 317, 661,
    253, 498, 222, 765, 792, 452,
    648, 180, 579, 340, 955, 541,
    392, 501, 732, 143, 693, 743,
    564, 209, 702, 778, 138, 898,
    542, 148, 42, 171, 265, 18,
    // Bin capacity
    1612, 1101, 1481, 1347, 1326, 1478
  };
  const int i6_18_6_2_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    440, 542, 65, 286, 2, 863,
    171, 497, 619, 434, 192, 330,
    442, 122, 527, 773, 296, 109,
    814, 195, 63, 596, 639, 239,
    712, 653, 187, 908, 517, 178,
    419, 356, 287, 636, 235, 161,
    889, 184, 441, 964, 382, 483,
    596, 812, 666, 236, 429, 640,
    549, 895, 298, 414, 607, 346,
    393, 787, 432, 322, 397, 758,
    372, 473, 872, 303, 222, 198,
    207, 16, 673, 662, 440, 684,
    53, 75, 177, 62, 768, 548,
    566, 935, 707, 769, 273, 129,
    728, 935, 136, 604, 542, 480,
    192, 771, 325, 341, 80, 397,
    0, 280, 892, 893, 979, 641,
    389, 717, 7, 919, 399, 305,
    // Bin capacity
    1349, 1572, 1254, 1721, 1258, 1274
  };
  const int i6_18_6_3_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    531, 565, 199, 370, 996, 427,
    604, 599, 306, 77, 868, 354,
    409, 892, 970, 16, 636, 661,
    303, 703, 68, 46, 226, 287,
    805, 697, 138, 223, 950, 528,
    685, 15, 280, 481, 996, 755,
    376, 621, 907, 116, 118, 643,
    353, 545, 210, 777, 382, 359,
    859, 805, 837, 890, 994, 515,
    795, 683, 970, 220, 156, 252,
    517, 821, 707, 121, 456, 530,
    85, 595, 982, 383, 130, 229,
    556, 779, 853, 876, 760, 815,
    852, 712, 850, 987, 442, 473,
    170, 751, 74, 202, 751, 120,
    353, 638, 618, 51, 301, 497,
    69, 33, 571, 55, 160, 84,
    551, 349, 383, 504, 71, 552,
    // Bin capacity
    1524, 1855, 1704, 1098, 1613, 1388
  };
  const int i6_18_6_4_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    195, 972, 10, 635, 320, 901,
    317, 869, 179, 973, 358, 143,
    37, 514, 937, 399, 676, 457,
    86, 144, 567, 926, 332, 285,
    110, 880, 258, 719, 753, 663,
    73, 771, 391, 34, 980, 585,
    726, 130, 775, 835, 535, 215,
    214, 414, 782, 70, 736, 238,
    381, 666, 528, 548, 924, 532,
    150, 959, 593, 566, 761, 269,
    565, 104, 157, 38, 99, 201,
    479, 168, 163, 917, 403, 437,
    408, 351, 117, 211, 274, 124,
    838, 781, 39, 700, 639, 281,
    635, 374, 150, 991, 875, 15,
    554, 964, 32, 887, 746, 14,
    752, 650, 345, 41, 377, 729,
    816, 646, 801, 314, 461, 594,
    // Bin capacity
    1272, 1796, 1183, 1700, 1777, 1159
  };
  const int i6_18_6_5_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    776, 463, 267, 66, 42, 707,
    679, 606, 789, 899, 867, 845,
    228, 523, 477, 672, 682, 29,
    677, 928, 78, 577, 756, 691,
    903, 69, 176, 783, 464, 758,
    221, 12, 292, 172, 38, 33,
    500, 276, 779, 94, 299, 482,
    638, 527, 166, 621, 263, 727,
    627, 329, 52, 106, 569, 937,
    485, 546, 603, 478, 635, 283,
    10, 132, 194, 86, 313, 161,
    660, 168, 62, 555, 778, 763,
    344, 709, 559, 642, 946, 72,
    603, 28, 971, 93, 130, 977,
    299, 309, 810, 735, 667, 106,
    951, 310, 348, 15, 645, 983,
    413, 356, 156, 240, 617, 5,
    993, 861, 614, 449, 647, 639,
    // Bin capacity
    1752, 1252, 1294, 1275, 1638, 1610
  };
  const int i6_18_6_6_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    65, 484, 406, 759, 600, 150,
    406, 537, 387, 522, 33, 438,
    528, 365, 45, 289, 56, 489,
    242, 653, 363, 116, 99, 435,
    389, 294, 189, 424, 681, 9,
    672, 497, 211, 655, 102, 178,
    366, 329, 110, 750, 130, 31,
    651, 259, 671, 331, 602, 898,
    347, 76, 244, 686, 530, 694,
    804, 548, 566, 969, 850, 509,
    541, 317, 158, 322, 802, 712,
    435, 774, 315, 758, 130, 757,
    124, 844, 325, 696, 701, 494,
    835, 331, 235, 872, 961, 963,
    390, 254, 891, 307, 734, 315,
    679, 444, 813, 696, 417, 155,
    97, 266, 966, 838, 244, 640,
    805, 905, 934, 172, 392, 758,
    // Bin capacity
    1480, 1445, 1384, 1796, 1425, 1524
  };
  const int i6_18_6_7_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    6, 354, 56, 215, 95, 685,
    907, 389, 580, 640, 697, 950,
    880, 475, 570, 74, 576, 894,
    26, 234, 26, 515, 895, 438,
    199, 540, 354, 144, 935, 991,
    351, 432, 947, 793, 171, 354,
    618, 314, 11, 485, 625, 970,
    582, 360, 786, 168, 89, 312,
    73, 597, 647, 870, 973, 216,
    4, 463, 794, 211, 849, 639,
    121, 326, 438, 968, 385, 291,
    626, 993, 230, 4, 404, 321,
    908, 334, 586, 876, 631, 411,
    231, 708, 702, 149, 203, 73,
    373, 841, 228, 997, 955, 519,
    211, 436, 848, 719, 933, 280,
    488, 620, 82, 69, 889, 24,
    66, 340, 422, 808, 814, 959,
    // Bin capacity
    1190, 1562, 1482, 1553, 1983, 1664
  };
  const int i6_18_6_8_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    505, 814, 856, 409, 207, 49,
    884, 435, 491, 79, 568, 413,
    128, 570, 448, 6, 134, 276,
    242, 40, 865, 671, 929, 525,
    806, 825, 53, 102, 923, 921,
    44, 44, 885, 261, 782, 724,
    900, 999, 567, 931, 337, 185,
    292, 427, 77, 143, 653, 185,
    381, 262, 662, 627, 226, 435,
    722, 424, 589, 609, 968, 933,
    537, 646, 566, 544, 264, 214,
    22, 38, 629, 125, 54, 474,
    839, 460, 397, 510, 150, 580,
    431, 506, 959, 2, 159, 547,
    971, 317, 138, 679, 867, 186,
    649, 784, 380, 875, 818, 932,
    94, 870, 993, 736, 825, 521,
    656, 153, 699, 757, 248, 996,
    // Bin capacity
    1639, 1551, 1846, 1452, 1641, 1638
  };
  const int i6_18_6_9_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    882, 2, 898, 699, 644, 840,
    947, 660, 819, 36, 572, 937,
    70, 523, 885, 984, 641, 658,
    970, 75, 971, 290, 119, 377,
    504, 505, 233, 824, 462, 704,
    200, 316, 213, 864, 34, 632,
    319, 842, 815, 43, 203, 699,
    272, 838, 895, 396, 571, 53,
    606, 358, 850, 433, 490, 245,
    283, 776, 961, 970, 133, 813,
    368, 239, 495, 437, 993, 99,
    155, 564, 70, 834, 312, 169,
    646, 401, 234, 855, 8, 751,
    538, 220, 946, 748, 457, 515,
    885, 730, 616, 689, 865, 293,
    110, 782, 120, 265, 750, 36,
    66, 653, 577, 771, 890, 661,
    288, 990, 689, 773, 798, 509,
    // Bin capacity
    1474, 1722, 2051, 1983, 1625, 1634
  };
  const int i6_18_6_10_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    601, 676, 347, 707, 705, 468,
    502, 235, 454, 755, 758, 180,
    569, 795, 645, 630, 713, 670,
    435, 862, 715, 584, 871, 320,
    827, 51, 370, 545, 619, 766,
    713, 962, 443, 980, 649, 701,
    759, 306, 804, 202, 533, 459,
    846, 73, 930, 622, 385, 493,
    978, 318, 80, 437, 60, 963,
    424, 309, 344, 922, 419, 64,
    969, 669, 328, 29, 275, 212,
    376, 528, 296, 246, 703, 960,
    202, 956, 753, 838, 580, 507,
    430, 909, 79, 648, 111, 310,
    682, 996, 327, 892, 399, 322,
    544, 689, 708, 730, 314, 38,
    445, 873, 854, 613, 102, 216,
    229, 859, 795, 509, 748, 179,
    // Bin capacity
    1931, 2029, 1700, 1997, 1640, 1436
  };
  const int i6_18_6_11_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    769, 148, 38, 230, 822, 345,
    309, 810, 454, 156, 457, 567,
    477, 273, 882, 31, 295, 719,
    588, 527, 391, 429, 752, 116,
    397, 647, 231, 601, 214, 953,
    695, 46, 639, 989, 528, 872,
    33, 991, 294, 401, 516, 766,
    326, 610, 756, 400, 894, 897,
    280, 728, 138, 0, 777, 401,
    150, 565, 239, 752, 216, 733,
    631, 323, 495, 469, 185, 998,
    149, 664, 949, 921, 657, 4,
    517, 381, 975, 329, 674, 536,
    880, 381, 596, 189, 165, 911,
    338, 840, 927, 523, 433, 999,
    401, 451, 638, 258, 733, 331,
    191, 347, 207, 274, 758, 147,
    939, 500, 965, 613, 632, 411,
    // Bin capacity
    1493, 1708, 1816, 1400, 1796, 1981
  };
  const int i6_18_6_12_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    625, 267, 964, 718, 815, 288,
    93, 696, 912, 857, 192, 631,
    341, 754, 132, 395, 474, 445,
    273, 434, 339, 747, 120, 821,
    480, 755, 771, 961, 537, 150,
    71, 83, 144, 773, 732, 537,
    851, 491, 747, 2, 667, 519,
    912, 908, 257, 597, 733, 266,
    330, 793, 355, 256, 482, 856,
    158, 835, 977, 251, 642, 449,
    178, 118, 928, 914, 263, 488,
    601, 785, 117, 140, 299, 425,
    519, 453, 537, 816, 559, 224,
    678, 228, 799, 654, 258, 558,
    0, 282, 473, 152, 799, 793,
    883, 189, 54, 395, 445, 792,
    675, 448, 934, 359, 324, 382,
    3, 215, 388, 715, 575, 278,
    // Bin capacity
    1432, 1631, 1835, 1812, 1665, 1662
  };
  const int i6_18_6_13_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    353, 696, 550, 360, 921, 150,
    417, 756, 424, 535, 974, 540,
    492, 596, 289, 539, 217, 766,
    31, 821, 161, 407, 356, 993,
    889, 934, 392, 129, 877, 622,
    390, 473, 773, 634, 452, 741,
    623, 88, 692, 806, 657, 152,
    169, 637, 49, 623, 190, 455,
    802, 861, 240, 607, 892, 617,
    438, 927, 381, 885, 413, 866,
    412, 601, 338, 744, 135, 17,
    463, 59, 897, 774, 728, 729,
    893, 689, 936, 676, 97, 59,
    485, 336, 520, 184, 54, 75,
    348, 245, 112, 595, 589, 334,
    545, 841, 720, 748, 615, 839,
    59, 647, 324, 448, 308, 763,
    525, 336, 746, 955, 531, 368,
    // Bin capacity
    1570, 1986, 1610, 2006, 1697, 1712
  };
  const int i6_18_6_14_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    944, 878, 904, 409, 956, 299,
    876, 897, 873, 383, 247, 760,
    67, 215, 39, 977, 823, 379,
    323, 49, 428, 29, 936, 846,
    796, 818, 979, 608, 866, 668,
    756, 345, 104, 628, 46, 596,
    143, 896, 181, 909, 621, 630,
    525, 495, 290, 945, 298, 838,
    761, 109, 742, 811, 526, 359,
    276, 855, 209, 379, 212, 718,
    82, 695, 501, 116, 77, 529,
    161, 591, 462, 116, 453, 347,
    308, 48, 654, 506, 415, 619,
    649, 331, 214, 99, 619, 874,
    355, 569, 96, 28, 223, 862,
    638, 148, 103, 142, 124, 623,
    319, 337, 344, 194, 100, 501,
    900, 429, 690, 111, 100, 144,
    // Bin capacity
    1688, 1654, 1485, 1405, 1452, 2013
  };
  const int i6_18_6_15_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    117, 0, 298, 419, 420, 471,
    976, 707, 663, 889, 177, 447,
    254, 57, 26, 635, 358, 665,
    556, 814, 600, 390, 617, 399,
    691, 85, 855, 207, 995, 611,
    254, 15, 158, 232, 545, 602,
    757, 958, 647, 419, 698, 872,
    955, 442, 549, 155, 224, 359,
    52, 376, 466, 939, 780, 876,
    87, 575, 738, 588, 499, 961,
    152, 469, 936, 932, 282, 467,
    192, 730, 569, 369, 655, 871,
    207, 677, 488, 947, 558, 371,
    679, 475, 750, 949, 347, 456,
    48, 752, 810, 319, 532, 425,
    732, 265, 44, 73, 228, 281,
    506, 523, 994, 11, 550, 204,
    412, 272, 232, 681, 130, 313,
    // Bin capacity
    1462, 1571, 1883, 1755, 1648, 1850
  };
  const int i6_18_6_16_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    566, 419, 889, 286, 480, 541,
    340, 765, 262, 161, 9, 782,
    392, 824, 721, 658, 530, 627,
    714, 412, 104, 706, 240, 620,
    509, 700, 586, 897, 850, 118,
    635, 608, 424, 514, 880, 826,
    46, 483, 284, 204, 295, 519,
    532, 710, 801, 347, 877, 103,
    904, 45, 69, 591, 713, 5,
    676, 90, 961, 171, 281, 817,
    884, 714, 38, 459, 445, 156,
    130, 338, 458, 389, 924, 602,
    726, 612, 700, 929, 500, 653,
    560, 289, 453, 13, 986, 50,
    209, 360, 747, 551, 158, 199,
    438, 68, 808, 224, 985, 150,
    204, 467, 0, 935, 525, 998,
    75, 887, 994, 433, 186, 557,
    // Bin capacity
    1652, 1700, 1798, 1638, 1908, 1610
  };
  const int i6_18_6_17_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    720, 562, 710, 254, 137, 687,
    696, 56, 778, 983, 465, 878,
    964, 124, 948, 921, 183, 303,
    77, 807, 273, 823, 883, 283,
    112, 768, 854, 434, 419, 652,
    846, 199, 116, 173, 841, 258,
    657, 675, 647, 477, 401, 385,
    10, 798, 862, 892, 821, 586,
    169, 238, 255, 975, 411, 513,
    714, 509, 153, 564, 146, 544,
    268, 721, 693, 259, 520, 924,
    510, 408, 702, 327, 389, 54,
    265, 473, 6, 173, 198, 206,
    753, 107, 178, 694, 825, 221,
    320, 419, 935, 800, 213, 303,
    499, 385, 391, 74, 101, 497,
    942, 737, 516, 650, 838, 664,
    67, 383, 463, 745, 475, 253,
    // Bin capacity
    1675, 1632, 1849, 1993, 1612, 1602
  };
  const int i6_18_6_18_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    829, 18, 917, 154, 893, 515,
    116, 1, 325, 11, 336, 256,
    281, 635, 817, 854, 128, 186,
    106, 19, 610, 770, 356, 416,
    416, 771, 36, 938, 328, 524,
    345, 469, 166, 120, 92, 554,
    264, 653, 597, 347, 281, 688,
    79, 794, 235, 746, 354, 186,
    718, 55, 579, 590, 715, 259,
    0, 993, 53, 476, 538, 697,
    750, 43, 759, 284, 48, 770,
    754, 975, 285, 647, 150, 464,
    699, 982, 60, 295, 632, 236,
    565, 112, 764, 677, 554, 412,
    51, 869, 110, 992, 11, 254,
    88, 991, 3, 515, 987, 42,
    950, 987, 410, 25, 963, 467,
    241, 432, 605, 314, 203, 289,
    // Bin capacity
    1427, 1928, 1442, 1722, 1489, 1419
  };
  const int i6_18_6_19_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    276, 921, 542, 519, 868, 734,
    608, 172, 390, 971, 637, 507,
    655, 450, 896, 209, 962, 677,
    613, 461, 622, 771, 579, 192,
    225, 342, 602, 355, 264, 147,
    106, 387, 182, 593, 31, 167,
    288, 228, 351, 557, 319, 103,
    255, 410, 836, 135, 472, 34,
    671, 825, 413, 757, 425, 787,
    796, 506, 690, 310, 936, 541,
    289, 387, 170, 353, 5, 726,
    885, 728, 243, 580, 971, 138,
    581, 526, 428, 175, 293, 217,
    502, 634, 187, 118, 180, 559,
    621, 256, 314, 44, 962, 301,
    172, 182, 40, 962, 558, 64,
    499, 946, 326, 702, 998, 431,
    357, 679, 757, 569, 608, 676,
    // Bin capacity
    1666, 1793, 1585, 1722, 1997, 1389
  };
  const int i6_18_6_20_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    201, 306, 730, 826, 953, 14,
    253, 812, 527, 783, 650, 531,
    72, 951, 655, 877, 731, 792,
    850, 685, 943, 472, 61, 701,
    421, 172, 111, 571, 754, 62,
    794, 900, 400, 832, 417, 630,
    162, 403, 534, 68, 374, 874,
    265, 55, 603, 759, 691, 899,
    82, 369, 816, 819, 132, 419,
    0, 446, 639, 712, 797, 421,
    666, 315, 569, 726, 494, 700,
    608, 389, 336, 789, 598, 878,
    532, 416, 974, 636, 196, 556,
    772, 330, 297, 216, 875, 248,
    839, 51, 788, 446, 413, 712,
    393, 162, 993, 920, 624, 183,
    422, 845, 869, 513, 996, 780,
    152, 184, 921, 814, 613, 892,
    // Bin capacity
    1497, 1559, 2341, 2356, 2074, 2059
  };
  const int i6_18_6_21_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    525, 441, 349, 911, 564, 943,
    170, 194, 354, 132, 245, 58,
    995, 575, 837, 275, 738, 15,
    163, 651, 409, 414, 467, 278,
    720, 35, 837, 296, 334, 717,
    746, 45, 143, 58, 299, 504,
    611, 762, 833, 57, 716, 149,
    696, 23, 91, 966, 65, 413,
    49, 85, 954, 470, 900, 679,
    661, 726, 742, 0, 812, 528,
    507, 458, 545, 633, 267, 290,
    400, 641, 161, 404, 643, 985,
    275, 318, 198, 514, 903, 96,
    493, 901, 513, 639, 532, 434,
    361, 386, 476, 967, 760, 111,
    750, 581, 863, 538, 565, 118,
    170, 615, 238, 174, 406, 853,
    304, 854, 971, 6, 663, 277,
    // Bin capacity
    1734, 1673, 1919, 1504, 1993, 1503
  };
  const int i6_18_6_22_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    606, 976, 291, 647, 587, 86,
    316, 150, 433, 340, 172, 954,
    14, 110, 603, 112, 457, 633,
    446, 560, 680, 221, 909, 878,
    447, 49, 427, 858, 452, 899,
    853, 529, 400, 804, 247, 197,
    301, 632, 431, 196, 991, 417,
    965, 13, 200, 347, 145, 970,
    208, 352, 703, 346, 594, 234,
    96, 925, 913, 421, 140, 219,
    956, 721, 932, 61, 592, 442,
    554, 416, 184, 214, 278, 698,
    632, 572, 355, 530, 319, 283,
    251, 872, 197, 150, 485, 344,
    520, 238, 825, 852, 838, 52,
    185, 357, 368, 566, 939, 26,
    664, 297, 646, 247, 232, 183,
    209, 962, 557, 276, 96, 122,
    // Bin capacity
    1673, 1776, 1860, 1462, 1723, 1553
  };
  const int i6_18_6_23_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    195, 579, 282, 30, 210, 103,
    759, 51, 885, 826, 609, 897,
    615, 988, 510, 166, 651, 254,
    867, 709, 500, 522, 454, 479,
    640, 237, 911, 631, 283, 255,
    438, 908, 49, 254, 228, 832,
    751, 153, 854, 91, 724, 402,
    271, 199, 553, 68, 950, 224,
    596, 283, 905, 270, 236, 300,
    783, 369, 343, 230, 896, 503,
    957, 933, 723, 116, 186, 949,
    752, 964, 477, 283, 990, 887,
    257, 353, 790, 843, 653, 745,
    442, 263, 393, 790, 258, 465,
    803, 887, 831, 980, 669, 265,
    299, 875, 326, 421, 684, 229,
    187, 987, 241, 813, 878, 66,
    957, 41, 318, 642, 919, 301,
    // Bin capacity
    2167, 2005, 2028, 1636, 2148, 1672
  };
  const int i6_18_6_24_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    800, 733, 398, 575, 595, 274,
    421, 252, 399, 636, 582, 83,
    228, 319, 957, 843, 826, 31,
    159, 700, 54, 672, 527, 721,
    940, 727, 671, 286, 50, 382,
    837, 705, 692, 562, 124, 638,
    259, 663, 820, 275, 845, 974,
    967, 747, 210, 165, 498, 402,
    920, 674, 405, 217, 996, 12,
    309, 17, 779, 595, 855, 682,
    128, 760, 366, 23, 187, 736,
    5, 674, 525, 879, 611, 600,
    942, 219, 348, 431, 757, 478,
    940, 96, 733, 686, 728, 724,
    99, 911, 664, 422, 716, 163,
    402, 823, 573, 773, 327, 437,
    997, 635, 38, 542, 496, 131,
    258, 745, 787, 356, 199, 962,
    // Bin capacity
    1987, 2150, 1947, 1848, 2050, 1743
  };
  const int i6_18_6_25_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    129, 891, 248, 411, 735, 935,
    225, 549, 38, 687, 877, 616,
    836, 611, 823, 138, 180, 342,
    695, 842, 522, 376, 528, 887,
    87, 590, 903, 908, 558, 735,
    968, 10, 377, 159, 614, 238,
    772, 755, 406, 557, 342, 522,
    721, 531, 855, 716, 378, 254,
    824, 665, 175, 945, 410, 980,
    625, 878, 283, 323, 159, 804,
    750, 553, 459, 548, 725, 567,
    97, 261, 15, 87, 343, 88,
    502, 84, 759, 224, 466, 830,
    710, 545, 588, 206, 998, 749,
    250, 442, 983, 662, 556, 598,
    588, 233, 197, 894, 491, 933,
    388, 820, 263, 898, 54, 603,
    829, 135, 598, 193, 405, 61,
    // Bin capacity
    2083, 1958, 1770, 1861, 1838, 2238
  };
  const int i6_18_6_26_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    867, 197, 970, 963, 300, 132,
    937, 808, 472, 589, 751, 612,
    193, 612, 106, 461, 869, 141,
    125, 866, 824, 331, 884, 139,
    943, 803, 239, 456, 328, 497,
    37, 87, 737, 637, 957, 296,
    954, 799, 876, 771, 557, 197,
    440, 675, 910, 223, 606, 293,
    64, 309, 912, 219, 705, 727,
    652, 483, 958, 246, 981, 376,
    627, 714, 265, 508, 225, 446,
    679, 930, 435, 510, 520, 544,
    715, 542, 608, 862, 521, 131,
    37, 197, 721, 504, 675, 811,
    593, 820, 500, 541, 938, 701,
    687, 653, 632, 475, 123, 832,
    992, 976, 284, 452, 385, 247,
    234, 676, 236, 806, 741, 955,
    // Bin capacity
    2053, 2341, 2244, 2007, 2324, 1697
  };
  const int i6_18_6_27_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    414, 828, 720, 697, 613, 278,
    372, 106, 87, 358, 27, 798,
    51, 602, 360, 363, 478, 988,
    131, 73, 521, 340, 151, 26,
    520, 200, 356, 988, 750, 455,
    894, 502, 459, 344, 646, 157,
    996, 842, 438, 740, 167, 98,
    497, 151, 809, 44, 792, 220,
    1000, 518, 406, 34, 881, 455,
    809, 638, 387, 971, 450, 640,
    65, 276, 571, 895, 458, 260,
    872, 230, 937, 375, 23, 648,
    164, 113, 792, 39, 10, 542,
    281, 655, 188, 759, 236, 359,
    418, 286, 474, 278, 119, 839,
    498, 488, 258, 806, 801, 343,
    494, 581, 308, 673, 687, 661,
    883, 909, 420, 221, 95, 95,
    // Bin capacity
    1981, 1693, 1798, 1890, 1563, 1665
  };
  const int i6_18_6_28_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    182, 865, 520, 367, 750, 466,
    188, 497, 669, 11, 590, 426,
    194, 906, 686, 527, 431, 561,
    57, 460, 173, 322, 656, 149,
    438, 81, 242, 482, 48, 580,
    180, 606, 499, 591, 862, 272,
    713, 946, 438, 773, 487, 913,
    821, 55, 810, 419, 176, 947,
    33, 132, 288, 654, 421, 294,
    160, 790, 510, 690, 539, 362,
    269, 623, 618, 437, 897, 700,
    719, 269, 909, 775, 641, 246,
    598, 916, 137, 840, 764, 722,
    523, 796, 756, 524, 54, 971,
    232, 10, 984, 186, 549, 566,
    796, 849, 45, 999, 540, 632,
    546, 506, 482, 570, 165, 871,
    891, 827, 623, 916, 73, 543,
    // Bin capacity
    1609, 2162, 2003, 2152, 1844, 2181
  };
  const int i6_18_6_29_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    160, 413, 699, 567, 38, 751,
    519, 416, 230, 115, 476, 994,
    125, 741, 943, 174, 568, 298,
    755, 481, 922, 978, 748, 951,
    663, 589, 142, 336, 658, 54,
    380, 586, 549, 182, 645, 895,
    529, 723, 838, 441, 378, 412,
    21, 620, 265, 788, 610, 699,
    168, 933, 637, 157, 953, 39,
    959, 432, 85, 826, 116, 868,
    547, 86, 381, 356, 852, 701,
    348, 36, 537, 866, 922, 475,
    538, 14, 403, 863, 619, 12,
    743, 613, 521, 910, 899, 187,
    20, 42, 763, 18, 347, 960,
    589, 800, 608, 155, 219, 392,
    452, 635, 163, 171, 250, 305,
    56, 492, 857, 372, 950, 525,
    // Bin capacity
    1628, 1861, 2052, 1780, 2204, 2047
  };
  const int i6_18_6_30_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    408, 365, 634, 873, 251, 520,
    957, 587, 246, 252, 900, 588,
    417, 846, 546, 762, 294, 655,
    865, 408, 646, 986, 940, 175,
    768, 339, 675, 287, 194, 788,
    920, 105, 343, 227, 254, 902,
    737, 993, 523, 973, 66, 769,
    165, 26, 31, 392, 126, 664,
    892, 991, 777, 578, 797, 463,
    1, 320, 279, 473, 122, 751,
    888, 708, 714, 921, 604, 745,
    65, 463, 896, 199, 604, 943,
    593, 886, 21, 103, 46, 961,
    189, 371, 594, 110, 875, 849,
    150, 267, 28, 40, 51, 850,
    996, 297, 747, 855, 929, 313,
    786, 942, 662, 389, 912, 449,
    516, 881, 300, 455, 141, 508,
    // Bin capacity
    2235, 2123, 1877, 1923, 1757, 2577
  };
  const int i6_18_6_31_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    222, 679, 601, 391, 630, 307,
    203, 189, 383, 37, 532, 156,
    297, 118, 412, 100, 601, 226,
    979, 878, 172, 851, 428, 345,
    297, 233, 795, 525, 913, 6,
    622, 423, 263, 376, 832, 169,
    425, 762, 990, 61, 161, 591,
    241, 971, 168, 551, 755, 758,
    973, 311, 583, 957, 776, 318,
    284, 672, 200, 212, 525, 229,
    476, 107, 616, 276, 278, 220,
    198, 475, 806, 889, 547, 804,
    226, 504, 637, 1000, 48, 235,
    258, 920, 309, 912, 89, 164,
    950, 300, 817, 75, 638, 143,
    112, 654, 952, 600, 849, 121,
    497, 8, 868, 316, 391, 568,
    126, 451, 735, 778, 707, 572,
    // Bin capacity
    1613, 1890, 2251, 1945, 2118, 1296
  };
  const int i6_18_6_32_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    741, 404, 298, 950, 171, 920,
    147, 532, 662, 38, 194, 243,
    603, 354, 993, 354, 105, 17,
    609, 361, 808, 828, 206, 776,
    64, 564, 313, 821, 48, 619,
    813, 160, 714, 121, 177, 209,
    942, 13, 958, 687, 50, 243,
    746, 149, 531, 997, 773, 339,
    890, 757, 264, 624, 493, 63,
    618, 113, 424, 283, 433, 75,
    64, 817, 727, 375, 98, 397,
    348, 576, 504, 259, 270, 573,
    322, 82, 182, 554, 423, 399,
    757, 820, 339, 931, 805, 715,
    57, 640, 576, 853, 897, 422,
    134, 242, 101, 628, 62, 423,
    111, 557, 721, 407, 660, 35,
    971, 290, 830, 862, 275, 599,
    // Bin capacity
    1967, 1635, 2188, 2326, 1351, 1555
  };
  const int i6_18_6_33_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    917, 659, 122, 131, 923, 534,
    884, 745, 512, 521, 208, 177,
    871, 44, 478, 466, 510, 662,
    961, 805, 185, 298, 449, 946,
    520, 499, 212, 364, 281, 162,
    796, 678, 27, 740, 895, 162,
    755, 297, 939, 924, 760, 294,
    103, 843, 410, 814, 636, 446,
    495, 812, 720, 914, 806, 384,
    835, 684, 211, 385, 611, 51,
    260, 670, 42, 467, 298, 214,
    457, 124, 537, 991, 785, 283,
    895, 344, 531, 771, 479, 911,
    344, 78, 281, 652, 321, 634,
    955, 555, 877, 452, 373, 673,
    655, 447, 231, 194, 129, 977,
    836, 36, 453, 777, 292, 55,
    864, 173, 459, 822, 92, 563,
    // Bin capacity
    2750, 1883, 1602, 2369, 1962, 1802
  };
  const int i6_18_6_34_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    213, 347, 686, 244, 797, 16,
    467, 296, 722, 249, 420, 738,
    221, 673, 690, 667, 171, 172,
    403, 924, 231, 861, 544, 386,
    576, 784, 249, 364, 178, 344,
    201, 373, 467, 392, 614, 910,
    340, 105, 189, 623, 120, 318,
    750, 226, 918, 534, 947, 696,
    139, 650, 225, 871, 329, 211,
    561, 919, 483, 870, 179, 986,
    930, 809, 33, 410, 988, 727,
    53, 188, 331, 590, 83, 559,
    910, 190, 711, 674, 136, 366,
    446, 190, 925, 340, 765, 729,
    943, 677, 135, 608, 605, 344,
    323, 693, 290, 959, 224, 721,
    713, 957, 868, 681, 144, 566,
    342, 94, 165, 745, 811, 509,
    // Bin capacity
    1906, 2032, 1858, 2386, 1799, 2077
  };
  const int i6_18_6_35_35[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    458, 913, 154, 907, 720, 270,
    345, 730, 703, 564, 886, 874,
    204, 541, 746, 255, 540, 815,
    684, 523, 712, 890, 996, 43,
    526, 260, 981, 765, 819, 926,
    36, 486, 535, 627, 184, 334,
    301, 181, 449, 832, 474, 364,
    897, 948, 272, 785, 32, 174,
    37, 972, 422, 366, 733, 60,
    891, 857, 234, 854, 104, 905,
    390, 542, 195, 526, 822, 411,
    868, 9, 142, 717, 505, 460,
    488, 774, 369, 597, 651, 365,
    329, 914, 236, 279, 492, 21,
    706, 164, 813, 802, 173, 57,
    285, 512, 102, 519, 405, 507,
    611, 111, 117, 434, 482, 793,
    857, 451, 426, 848, 950, 480,
    // Bin capacity
    2006, 2225, 1712, 2603, 2243, 1769
  };
  const int i6_18_6_0_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    322, 331, 201, 903, 565, 90,
    556, 268, 221, 633, 168, 661,
    753, 947, 395, 446, 113, 163,
    803, 381, 319, 450, 826, 321,
    122, 528, 57, 543, 733, 424,
    43, 187, 901, 469, 955, 545,
    325, 524, 482, 427, 415, 699,
    353, 947, 884, 264, 529, 840,
    499, 721, 782, 372, 218, 672,
    677, 739, 711, 204, 973, 479,
    439, 68, 431, 888, 547, 289,
    574, 973, 1000, 987, 499, 370,
    880, 37, 547, 587, 580, 75,
    295, 157, 62, 183, 571, 536,
    225, 524, 106, 731, 874, 749,
    410, 379, 332, 697, 526, 949,
    444, 839, 994, 28, 247, 502,
    466, 337, 802, 677, 933, 409,
    // Bin capacity
    1365, 1482, 1538, 1582, 1712, 1463
  };
  const int i6_18_6_1_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    968, 373, 930, 108, 369, 50,
    615, 633, 559, 544, 224, 173,
    31, 565, 542, 616, 836, 669,
    320, 949, 175, 686, 3, 802,
    449, 958, 711, 41, 907, 854,
    517, 612, 54, 201, 195, 763,
    230, 250, 519, 52, 22, 652,
    863, 464, 24, 888, 592, 533,
    747, 814, 489, 348, 192, 244,
    663, 351, 959, 992, 324, 285,
    839, 836, 217, 615, 655, 264,
    163, 908, 851, 809, 378, 75,
    293, 680, 313, 585, 743, 757,
    437, 284, 874, 341, 936, 879,
    705, 939, 849, 247, 668, 156,
    717, 576, 724, 192, 195, 339,
    148, 804, 926, 221, 375, 585,
    176, 328, 843, 83, 153, 612,
    // Bin capacity
    1495, 1907, 1778, 1275, 1308, 1464
  };
  const int i6_18_6_2_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    457, 958, 618, 168, 492, 801,
    544, 857, 746, 715, 571, 825,
    484, 401, 379, 645, 895, 394,
    667, 130, 158, 496, 188, 196,
    553, 416, 779, 761, 42, 382,
    616, 468, 737, 708, 967, 265,
    94, 988, 282, 773, 170, 451,
    113, 578, 338, 939, 115, 631,
    746, 252, 428, 356, 235, 452,
    283, 813, 369, 230, 995, 58,
    822, 718, 993, 495, 357, 219,
    423, 143, 305, 982, 863, 163,
    672, 195, 971, 574, 796, 580,
    903, 310, 310, 881, 536, 477,
    853, 860, 660, 720, 100, 574,
    58, 485, 809, 168, 756, 791,
    384, 86, 313, 584, 317, 213,
    22, 669, 281, 395, 325, 894,
    // Bin capacity
    1478, 1586, 1611, 1801, 1483, 1423
  };
  const int i6_18_6_3_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    619, 13, 995, 176, 660, 241,
    925, 54, 132, 592, 375, 788,
    794, 981, 12, 621, 242, 245,
    415, 661, 86, 132, 180, 355,
    541, 422, 819, 456, 196, 532,
    716, 299, 248, 78, 474, 667,
    274, 244, 654, 538, 192, 665,
    234, 355, 801, 672, 833, 379,
    977, 587, 470, 485, 99, 239,
    235, 147, 195, 399, 877, 515,
    617, 699, 56, 48, 436, 986,
    139, 778, 118, 198, 82, 547,
    547, 564, 13, 522, 389, 559,
    513, 487, 560, 636, 867, 600,
    331, 717, 624, 662, 442, 269,
    208, 208, 25, 161, 267, 3,
    654, 827, 449, 929, 985, 833,
    68, 150, 678, 970, 80, 376,
    // Bin capacity
    1512, 1407, 1191, 1421, 1318, 1511
  };
  const int i6_18_6_4_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    845, 291, 450, 666, 173, 791,
    869, 497, 304, 218, 215, 793,
    677, 386, 823, 36, 824, 921,
    797, 555, 100, 10, 482, 520,
    470, 253, 606, 876, 214, 286,
    179, 231, 823, 169, 550, 991,
    142, 23, 633, 200, 26, 976,
    508, 325, 809, 166, 937, 877,
    317, 909, 89, 753, 719, 903,
    426, 42, 194, 923, 961, 969,
    667, 502, 101, 642, 165, 734,
    521, 998, 351, 638, 651, 387,
    697, 335, 553, 89, 886, 461,
    737, 553, 288, 681, 469, 651,
    945, 273, 399, 759, 143, 652,
    972, 281, 680, 490, 319, 588,
    144, 846, 130, 940, 955, 71,
    892, 605, 719, 396, 809, 631,
    // Bin capacity
    1873, 1371, 1396, 1500, 1647, 2116
  };
  const int i6_18_6_5_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    69, 518, 182, 93, 73, 996,
    214, 169, 877, 898, 653, 965,
    538, 991, 462, 112, 671, 672,
    917, 988, 695, 13, 675, 862,
    978, 502, 570, 913, 489, 91,
    706, 852, 353, 789, 767, 266,
    380, 157, 44, 979, 701, 393,
    321, 665, 566, 863, 901, 379,
    769, 851, 896, 602, 987, 934,
    391, 426, 734, 954, 673, 824,
    249, 937, 45, 460, 290, 371,
    54, 181, 263, 815, 862, 755,
    632, 152, 262, 800, 910, 555,
    509, 707, 70, 441, 684, 846,
    713, 784, 534, 123, 283, 62,
    823, 469, 905, 224, 418, 708,
    21, 203, 97, 585, 587, 511,
    664, 513, 345, 788, 425, 122,
    // Bin capacity
    1566, 1762, 1383, 1830, 1934, 1805
  };
  const int i6_18_6_6_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    979, 16, 496, 525, 165, 945,
    450, 76, 653, 180, 446, 522,
    121, 113, 869, 392, 381, 217,
    903, 138, 388, 572, 591, 207,
    258, 820, 463, 69, 560, 887,
    74, 211, 427, 78, 698, 789,
    478, 193, 173, 642, 243, 615,
    948, 171, 568, 203, 103, 45,
    194, 747, 734, 182, 86, 24,
    77, 301, 767, 282, 385, 960,
    113, 898, 627, 980, 391, 52,
    168, 458, 456, 371, 570, 351,
    24, 600, 443, 352, 522, 913,
    857, 250, 95, 58, 31, 200,
    478, 499, 96, 179, 251, 349,
    27, 381, 450, 946, 342, 484,
    760, 695, 463, 958, 636, 555,
    985, 264, 935, 406, 195, 334,
    // Bin capacity
    1395, 1207, 1609, 1303, 1166, 1493
  };
  const int i6_18_6_7_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    379, 263, 395, 673, 109, 266,
    193, 377, 246, 593, 817, 569,
    289, 271, 566, 166, 736, 466,
    820, 811, 793, 844, 793, 706,
    833, 56, 722, 65, 697, 423,
    689, 371, 375, 247, 652, 347,
    464, 416, 297, 256, 996, 497,
    444, 943, 724, 775, 473, 644,
    995, 605, 949, 845, 223, 898,
    464, 780, 905, 408, 918, 817,
    846, 324, 325, 758, 124, 765,
    563, 101, 80, 960, 194, 86,
    834, 171, 118, 734, 503, 502,
    247, 477, 945, 664, 196, 151,
    31, 562, 749, 78, 568, 587,
    834, 277, 483, 559, 731, 921,
    532, 221, 898, 16, 10, 102,
    407, 726, 362, 351, 493, 918,
    // Bin capacity
    1760, 1383, 1772, 1604, 1647, 1724
  };
  const int i6_18_6_8_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    549, 706, 463, 24, 861, 204,
    738, 663, 370, 737, 219, 473,
    10, 980, 605, 840, 762, 762,
    169, 73, 315, 906, 414, 725,
    295, 254, 66, 116, 675, 876,
    371, 30, 25, 281, 77, 469,
    741, 136, 662, 168, 810, 158,
    943, 727, 579, 538, 695, 329,
    539, 912, 308, 496, 491, 255,
    790, 129, 963, 442, 268, 220,
    924, 419, 490, 840, 244, 963,
    690, 59, 817, 873, 850, 818,
    70, 338, 228, 201, 104, 838,
    961, 19, 192, 848, 271, 540,
    844, 758, 710, 530, 17, 677,
    732, 960, 434, 100, 3, 323,
    715, 465, 162, 129, 679, 703,
    881, 379, 807, 630, 452, 212,
    // Bin capacity
    1974, 1442, 1476, 1566, 1421, 1719
  };
  const int i6_18_6_9_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    673, 526, 95, 820, 236, 586,
    125, 635, 407, 925, 574, 411,
    743, 180, 360, 528, 276, 299,
    225, 295, 685, 50, 528, 679,
    702, 391, 271, 871, 400, 422,
    192, 153, 544, 969, 471, 766,
    202, 477, 324, 396, 516, 668,
    989, 18, 415, 272, 836, 139,
    826, 788, 751, 381, 964, 165,
    246, 264, 772, 667, 612, 640,
    901, 738, 451, 746, 863, 286,
    264, 163, 444, 882, 355, 847,
    231, 483, 37, 677, 66, 563,
    709, 401, 929, 660, 294, 856,
    453, 780, 900, 222, 251, 751,
    527, 733, 723, 721, 358, 112,
    364, 848, 133, 597, 521, 94,
    893, 23, 329, 966, 124, 414,
    // Bin capacity
    1684, 1435, 1557, 2062, 1498, 1581
  };
  const int i6_18_6_10_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    38, 997, 226, 405, 304, 225,
    925, 376, 220, 499, 808, 298,
    760, 984, 557, 760, 190, 464,
    621, 156, 890, 52, 638, 598,
    597, 452, 312, 474, 826, 806,
    659, 861, 604, 6, 906, 771,
    692, 199, 711, 250, 907, 621,
    148, 6, 565, 343, 383, 903,
    769, 740, 284, 933, 233, 81,
    81, 523, 611, 111, 79, 266,
    9, 499, 986, 654, 972, 604,
    423, 268, 881, 565, 525, 209,
    573, 519, 966, 997, 328, 579,
    479, 737, 570, 150, 956, 548,
    283, 145, 920, 683, 575, 841,
    627, 549, 998, 946, 776, 762,
    162, 821, 445, 686, 471, 217,
    107, 850, 6, 504, 177, 421,
    // Bin capacity
    1459, 1776, 1972, 1654, 1844, 1690
  };
  const int i6_18_6_11_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    473, 807, 587, 375, 174, 662,
    436, 928, 267, 257, 756, 106,
    418, 8, 713, 268, 432, 332,
    796, 707, 914, 526, 131, 536,
    748, 32, 532, 168, 738, 455,
    438, 918, 950, 72, 38, 846,
    275, 277, 518, 996, 469, 488,
    734, 898, 818, 815, 12, 478,
    274, 161, 355, 488, 926, 751,
    653, 284, 465, 573, 729, 576,
    54, 881, 461, 944, 753, 83,
    180, 549, 169, 914, 654, 171,
    613, 985, 238, 457, 166, 849,
    781, 491, 981, 440, 322, 565,
    730, 384, 881, 190, 454, 93,
    518, 934, 453, 807, 641, 207,
    915, 795, 651, 531, 224, 907,
    882, 829, 541, 71, 214, 736,
    // Bin capacity
    1835, 2011, 1942, 1646, 1450, 1636
  };
  const int i6_18_6_12_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    183, 663, 409, 929, 127, 300,
    0, 603, 779, 96, 424, 960,
    448, 797, 425, 101, 562, 354,
    891, 31, 984, 845, 744, 270,
    968, 66, 142, 501, 515, 161,
    6, 224, 516, 570, 989, 863,
    645, 398, 412, 77, 799, 795,
    562, 207, 64, 700, 455, 803,
    218, 438, 21, 142, 970, 905,
    196, 714, 552, 349, 733, 991,
    199, 237, 582, 159, 381, 436,
    217, 528, 52, 415, 597, 811,
    779, 771, 792, 437, 962, 135,
    268, 263, 9, 1000, 865, 967,
    942, 168, 397, 878, 290, 977,
    607, 835, 47, 175, 924, 402,
    159, 368, 286, 99, 98, 118,
    737, 178, 459, 336, 478, 581,
    // Bin capacity
    1498, 1398, 1294, 1458, 2038, 2022
  };
  const int i6_18_6_13_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    635, 583, 517, 11, 236, 451,
    32, 919, 987, 49, 703, 284,
    98, 282, 864, 93, 709, 485,
    511, 321, 590, 319, 381, 817,
    428, 901, 743, 642, 478, 455,
    784, 376, 835, 506, 624, 430,
    553, 477, 773, 833, 764, 558,
    323, 9, 253, 510, 800, 915,
    349, 178, 944, 738, 685, 818,
    584, 936, 103, 499, 169, 213,
    405, 163, 696, 321, 615, 497,
    83, 11, 112, 199, 571, 565,
    755, 799, 160, 381, 784, 864,
    640, 15, 940, 698, 291, 808,
    18, 209, 954, 882, 154, 946,
    394, 45, 842, 373, 14, 778,
    147, 301, 293, 994, 306, 863,
    138, 314, 730, 692, 9, 586,
    // Bin capacity
    1296, 1289, 2135, 1647, 1562, 2135
  };
  const int i6_18_6_14_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    353, 87, 473, 155, 829, 375,
    394, 526, 943, 220, 338, 664,
    377, 13, 178, 77, 176, 987,
    227, 674, 119, 689, 732, 391,
    291, 338, 886, 94, 822, 902,
    303, 864, 990, 836, 260, 442,
    795, 106, 443, 242, 240, 95,
    846, 620, 874, 99, 717, 922,
    8, 868, 485, 223, 321, 471,
    355, 883, 83, 244, 520, 13,
    427, 699, 277, 323, 899, 496,
    570, 573, 707, 603, 957, 213,
    390, 979, 443, 287, 731, 552,
    867, 839, 741, 823, 393, 531,
    477, 161, 122, 509, 759, 193,
    867, 586, 103, 508, 849, 909,
    316, 775, 682, 400, 335, 615,
    24, 61, 849, 891, 515, 161,
    // Bin capacity
    1499, 1834, 1786, 1373, 1975, 1698
  };
  const int i6_18_6_15_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    4, 417, 422, 119, 652, 191,
    30, 569, 222, 394, 188, 435,
    91, 795, 695, 517, 890, 944,
    321, 530, 689, 662, 374, 600,
    858, 681, 320, 927, 370, 873,
    832, 465, 788, 282, 143, 81,
    914, 324, 109, 729, 658, 15,
    743, 336, 38, 57, 517, 929,
    801, 107, 735, 295, 746, 288,
    836, 873, 82, 810, 867, 940,
    757, 120, 437, 106, 467, 98,
    718, 270, 497, 338, 99, 648,
    444, 530, 162, 517, 256, 35,
    201, 187, 285, 767, 749, 850,
    763, 427, 211, 930, 394, 254,
    640, 336, 770, 567, 451, 57,
    880, 341, 587, 231, 185, 602,
    374, 397, 965, 659, 206, 30,
    // Bin capacity
    1957, 1477, 1537, 1708, 1574, 1509
  };
  const int i6_18_6_16_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    339, 423, 516, 955, 204, 108,
    491, 818, 678, 268, 6, 629,
    86, 752, 504, 453, 907, 996,
    163, 97, 72, 510, 428, 983,
    27, 49, 16, 134, 890, 195,
    131, 9, 510, 957, 442, 685,
    487, 392, 200, 975, 562, 612,
    89, 467, 340, 575, 536, 577,
    503, 705, 746, 155, 503, 289,
    740, 179, 124, 567, 268, 972,
    145, 660, 75, 58, 706, 400,
    308, 762, 795, 149, 618, 863,
    586, 386, 645, 552, 774, 624,
    138, 767, 748, 680, 980, 821,
    126, 818, 874, 879, 281, 18,
    354, 737, 943, 433, 907, 699,
    607, 585, 149, 819, 722, 933,
    821, 227, 751, 16, 815, 522,
    // Bin capacity
    1188, 1708, 1680, 1767, 2040, 2113
  };
  const int i6_18_6_17_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    143, 311, 448, 832, 491, 529,
    553, 120, 725, 764, 221, 291,
    448, 684, 481, 32, 608, 118,
    530, 979, 427, 337, 417, 853,
    993, 291, 667, 974, 617, 188,
    368, 733, 690, 631, 880, 209,
    995, 643, 280, 790, 535, 375,
    151, 454, 398, 946, 345, 989,
    901, 902, 384, 459, 950, 711,
    733, 31, 742, 306, 924, 722,
    769, 162, 775, 706, 964, 863,
    145, 522, 672, 735, 881, 109,
    114, 72, 535, 823, 393, 637,
    6, 305, 310, 84, 899, 235,
    350, 272, 106, 5, 820, 555,
    483, 61, 348, 37, 227, 59,
    925, 49, 418, 96, 934, 784,
    249, 734, 283, 674, 658, 758,
    // Bin capacity
    1727, 1429, 1695, 1801, 2294, 1753
  };
  const int i6_18_6_18_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    247, 250, 374, 71, 248, 921,
    912, 753, 527, 106, 190, 558,
    496, 112, 342, 602, 954, 87,
    816, 978, 179, 326, 464, 389,
    152, 448, 217, 114, 217, 879,
    250, 734, 838, 912, 815, 804,
    274, 354, 644, 894, 237, 508,
    699, 833, 97, 88, 243, 108,
    323, 866, 227, 756, 180, 701,
    829, 580, 899, 354, 495, 192,
    621, 412, 76, 891, 38, 131,
    275, 158, 779, 618, 859, 90,
    256, 318, 453, 908, 542, 666,
    596, 283, 761, 430, 81, 855,
    631, 319, 684, 432, 349, 951,
    618, 597, 630, 452, 757, 215,
    538, 54, 845, 291, 807, 834,
    685, 255, 368, 267, 128, 724,
    // Bin capacity
    1813, 1634, 1759, 1675, 1496, 1891
  };
  const int i6_18_6_19_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    153, 550, 624, 54, 470, 838,
    406, 25, 229, 230, 677, 427,
    91, 410, 615, 730, 336, 134,
    946, 848, 215, 684, 749, 482,
    676, 540, 830, 816, 931, 714,
    917, 50, 198, 994, 330, 111,
    576, 163, 803, 652, 929, 948,
    485, 807, 109, 583, 608, 677,
    38, 452, 786, 802, 413, 6,
    441, 969, 717, 765, 944, 639,
    293, 687, 297, 509, 323, 800,
    599, 491, 693, 916, 240, 131,
    379, 679, 284, 498, 234, 801,
    875, 918, 25, 414, 699, 754,
    931, 865, 548, 850, 430, 784,
    80, 442, 770, 273, 305, 545,
    280, 116, 159, 454, 514, 552,
    17, 831, 412, 565, 323, 784,
    // Bin capacity
    1623, 1953, 1649, 2140, 1876, 2009
  };
  const int i6_18_6_20_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    1, 553, 445, 376, 299, 921,
    941, 729, 405, 284, 525, 600,
    750, 835, 453, 209, 588, 765,
    213, 941, 570, 313, 941, 638,
    585, 723, 257, 513, 109, 642,
    250, 204, 326, 874, 88, 753,
    107, 868, 819, 719, 881, 314,
    428, 491, 543, 42, 557, 417,
    845, 267, 334, 612, 381, 793,
    673, 543, 696, 224, 122, 66,
    629, 750, 613, 602, 426, 896,
    583, 23, 783, 861, 998, 295,
    397, 624, 816, 276, 372, 544,
    319, 961, 849, 256, 845, 869,
    279, 192, 947, 80, 214, 989,
    456, 697, 703, 512, 687, 873,
    440, 654, 965, 927, 876, 332,
    559, 949, 449, 47, 376, 611,
    // Bin capacity
    1691, 2201, 2195, 1546, 1857, 2264
  };
  const int i6_18_6_21_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    426, 833, 654, 243, 194, 373,
    212, 975, 44, 21, 930, 475,
    219, 310, 32, 569, 19, 885,
    13, 925, 613, 876, 555, 764,
    966, 998, 439, 167, 441, 663,
    351, 40, 352, 629, 557, 548,
    966, 74, 613, 604, 116, 832,
    687, 125, 874, 927, 989, 593,
    375, 174, 995, 554, 2, 58,
    983, 356, 583, 875, 404, 846,
    645, 485, 478, 246, 453, 72,
    76, 621, 748, 50, 604, 870,
    300, 400, 222, 422, 116, 603,
    906, 902, 413, 520, 762, 892,
    671, 760, 214, 416, 539, 9,
    949, 261, 29, 437, 526, 67,
    231, 381, 125, 386, 186, 552,
    211, 148, 87, 631, 530, 892,
    // Bin capacity
    1853, 1769, 1516, 1729, 1598, 2016
  };
  const int i6_18_6_22_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    286, 755, 397, 752, 386, 953,
    980, 628, 895, 614, 41, 952,
    620, 197, 894, 353, 461, 69,
    367, 239, 22, 644, 204, 318,
    61, 588, 735, 950, 779, 369,
    120, 914, 966, 235, 438, 323,
    73, 431, 584, 308, 953, 349,
    519, 592, 950, 701, 278, 677,
    855, 381, 355, 635, 970, 922,
    10, 524, 667, 55, 492, 259,
    395, 570, 980, 394, 984, 502,
    638, 720, 327, 327, 333, 924,
    102, 165, 180, 949, 419, 219,
    206, 315, 345, 743, 776, 140,
    909, 972, 707, 390, 111, 775,
    250, 992, 496, 466, 189, 483,
    626, 47, 630, 217, 278, 243,
    0, 813, 481, 491, 835, 91,
    // Bin capacity
    1427, 2002, 2158, 1876, 1816, 1743
  };
  const int i6_18_6_23_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    933, 158, 156, 234, 502, 199,
    614, 400, 422, 267, 592, 309,
    887, 66, 68, 957, 339, 261,
    856, 831, 894, 710, 584, 537,
    960, 13, 5, 65, 342, 291,
    861, 783, 826, 561, 117, 983,
    760, 60, 642, 883, 218, 546,
    880, 442, 487, 729, 670, 421,
    634, 743, 411, 269, 599, 528,
    493, 65, 169, 667, 485, 193,
    800, 936, 339, 834, 23, 132,
    820, 300, 907, 600, 579, 241,
    893, 518, 525, 466, 24, 185,
    323, 117, 824, 657, 269, 974,
    97, 351, 374, 855, 473, 54,
    497, 468, 441, 639, 217, 66,
    673, 308, 328, 996, 364, 571,
    117, 899, 229, 147, 884, 686,
    // Bin capacity
    2481, 1529, 1650, 2160, 1493, 1472
  };
  const int i6_18_6_24_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    134, 691, 870, 251, 464, 962,
    537, 991, 637, 581, 774, 759,
    17, 994, 503, 496, 760, 303,
    531, 922, 317, 195, 454, 85,
    723, 672, 145, 35, 240, 931,
    153, 557, 538, 333, 546, 588,
    541, 279, 782, 661, 164, 509,
    913, 778, 158, 78, 679, 133,
    521, 180, 263, 703, 972, 802,
    369, 71, 689, 357, 367, 80,
    554, 430, 904, 55, 697, 950,
    185, 30, 998, 902, 559, 525,
    92, 843, 915, 613, 619, 602,
    579, 747, 164, 693, 724, 738,
    450, 143, 58, 96, 741, 636,
    478, 942, 769, 958, 900, 271,
    7, 105, 386, 850, 916, 554,
    804, 286, 67, 17, 782, 410,
    // Bin capacity
    1569, 1997, 1894, 1628, 2348, 2034
  };
  const int i6_18_6_25_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    79, 555, 510, 820, 207, 969,
    370, 243, 887, 501, 968, 697,
    965, 654, 321, 972, 73, 808,
    809, 552, 992, 573, 563, 90,
    43, 760, 808, 112, 891, 424,
    46, 866, 763, 655, 777, 624,
    15, 178, 602, 753, 642, 171,
    883, 552, 780, 983, 185, 801,
    689, 427, 866, 674, 89, 121,
    625, 992, 193, 503, 847, 703,
    283, 807, 177, 989, 597, 653,
    29, 225, 565, 307, 197, 183,
    478, 187, 800, 209, 18, 224,
    340, 746, 750, 460, 597, 560,
    778, 684, 541, 62, 425, 805,
    815, 663, 728, 378, 182, 691,
    494, 817, 98, 721, 932, 608,
    944, 397, 126, 680, 842, 708,
    // Bin capacity
    1810, 2147, 2189, 2157, 1882, 2050
  };
  const int i6_18_6_26_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    373, 784, 281, 82, 334, 683,
    688, 477, 882, 186, 15, 954,
    113, 553, 80, 925, 163, 875,
    82, 553, 446, 462, 284, 925,
    842, 239, 109, 720, 101, 918,
    145, 384, 663, 621, 669, 328,
    633, 489, 106, 966, 355, 367,
    954, 682, 901, 332, 700, 491,
    366, 427, 932, 901, 241, 659,
    652, 238, 576, 479, 421, 201,
    219, 199, 862, 888, 810, 925,
    741, 705, 93, 24, 20, 532,
    451, 76, 386, 457, 583, 246,
    258, 995, 670, 471, 548, 810,
    36, 364, 868, 398, 490, 281,
    842, 106, 65, 753, 407, 474,
    344, 655, 317, 319, 591, 904,
    373, 608, 44, 824, 292, 780,
    // Bin capacity
    1704, 1793, 1740, 2060, 1476, 2385
  };
  const int i6_18_6_27_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    83, 560, 833, 557, 736, 859,
    715, 328, 423, 539, 166, 428,
    890, 407, 429, 69, 276, 837,
    844, 387, 712, 234, 329, 393,
    349, 387, 803, 889, 902, 977,
    847, 728, 846, 450, 680, 135,
    49, 24, 348, 516, 228, 537,
    888, 508, 766, 309, 577, 567,
    471, 954, 750, 549, 700, 245,
    304, 641, 549, 776, 851, 744,
    397, 724, 356, 340, 139, 963,
    951, 484, 545, 249, 495, 497,
    440, 543, 903, 403, 568, 76,
    85, 528, 129, 793, 998, 465,
    534, 526, 938, 559, 258, 11,
    32, 481, 145, 756, 12, 370,
    7, 197, 526, 618, 820, 910,
    876, 782, 515, 187, 451, 363,
    // Bin capacity
    1855, 1946, 2226, 1862, 1945, 1985
  };
  const int i6_18_6_28_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    405, 770, 388, 305, 947, 65,
    572, 906, 140, 829, 755, 879,
    463, 20, 284, 571, 637, 112,
    683, 891, 917, 187, 50, 432,
    53, 964, 885, 157, 672, 758,
    906, 363, 977, 487, 25, 260,
    984, 453, 77, 672, 779, 141,
    329, 500, 757, 268, 188, 650,
    649, 752, 900, 796, 979, 528,
    840, 222, 776, 594, 405, 491,
    262, 663, 71, 919, 56, 4,
    197, 882, 87, 810, 443, 517,
    41, 743, 774, 860, 748, 379,
    595, 670, 792, 173, 326, 165,
    81, 155, 783, 860, 751, 309,
    921, 388, 919, 384, 343, 941,
    671, 353, 687, 947, 73, 724,
    14, 983, 306, 74, 312, 241,
    // Bin capacity
    1849, 2278, 2245, 2111, 1811, 1621
  };
  const int i6_18_6_29_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    933, 786, 755, 906, 6, 18,
    767, 912, 279, 876, 286, 431,
    361, 861, 357, 144, 448, 667,
    911, 504, 955, 940, 58, 694,
    829, 807, 16, 17, 301, 403,
    599, 316, 343, 584, 818, 168,
    911, 209, 179, 61, 456, 484,
    712, 886, 503, 927, 443, 728,
    207, 461, 787, 889, 614, 823,
    308, 943, 739, 826, 54, 271,
    307, 365, 872, 53, 854, 21,
    359, 928, 898, 962, 353, 208,
    427, 859, 453, 322, 315, 387,
    469, 718, 850, 125, 596, 297,
    177, 615, 909, 561, 689, 600,
    163, 867, 418, 711, 616, 510,
    667, 633, 397, 71, 324, 387,
    377, 941, 760, 985, 224, 725,
    // Bin capacity
    2040, 2712, 2252, 2142, 1603, 1682
  };
  const int i6_18_6_30_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    666, 304, 25, 642, 972, 829,
    866, 455, 619, 903, 722, 258,
    376, 374, 145, 259, 94, 48,
    207, 756, 360, 554, 742, 774,
    545, 725, 69, 947, 576, 931,
    950, 396, 35, 506, 584, 762,
    14, 198, 605, 157, 214, 22,
    66, 109, 987, 180, 208, 396,
    775, 615, 922, 648, 394, 16,
    130, 130, 65, 343, 195, 605,
    18, 883, 737, 770, 413, 352,
    492, 380, 625, 893, 976, 947,
    283, 417, 828, 990, 589, 854,
    237, 346, 373, 529, 650, 466,
    434, 191, 695, 302, 573, 518,
    903, 648, 231, 45, 739, 294,
    420, 750, 159, 620, 213, 19,
    615, 122, 618, 349, 655, 830,
    // Bin capacity
    1733, 1690, 1755, 2089, 2061, 1933
  };
  const int i6_18_6_31_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    347, 980, 167, 277, 936, 408,
    383, 143, 501, 479, 33, 230,
    878, 657, 925, 489, 398, 733,
    579, 245, 443, 116, 122, 776,
    229, 197, 264, 27, 989, 562,
    853, 599, 524, 921, 130, 508,
    3, 282, 339, 757, 973, 218,
    379, 872, 176, 647, 751, 149,
    115, 361, 785, 202, 366, 649,
    766, 176, 104, 153, 580, 893,
    134, 668, 977, 85, 239, 149,
    504, 296, 93, 557, 112, 795,
    539, 425, 467, 235, 412, 348,
    177, 661, 508, 29, 120, 466,
    278, 675, 470, 102, 838, 360,
    497, 360, 915, 763, 586, 987,
    924, 398, 943, 351, 238, 20,
    666, 627, 181, 262, 949, 759,
    // Bin capacity
    1802, 1883, 1918, 1409, 1916, 1968
  };
  const int i6_18_6_32_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    822, 130, 697, 391, 662, 820,
    555, 216, 766, 154, 414, 212,
    582, 109, 979, 556, 300, 61,
    275, 491, 245, 543, 378, 53,
    414, 769, 963, 596, 677, 648,
    200, 758, 792, 765, 801, 863,
    47, 354, 69, 397, 157, 208,
    906, 559, 513, 909, 452, 149,
    820, 38, 437, 383, 82, 245,
    834, 926, 84, 983, 953, 87,
    245, 155, 21, 248, 196, 939,
    216, 485, 441, 197, 533, 753,
    347, 326, 649, 453, 924, 943,
    353, 717, 496, 468, 623, 636,
    211, 62, 554, 178, 266, 471,
    119, 575, 240, 667, 503, 518,
    53, 873, 416, 404, 258, 670,
    35, 214, 178, 340, 468, 10,
    // Bin capacity
    1548, 1707, 1879, 1900, 1903, 1823
  };
  const int i6_18_6_33_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    952, 195, 694, 301, 971, 699,
    952, 448, 308, 997, 935, 883,
    711, 657, 843, 65, 145, 724,
    662, 116, 718, 655, 460, 891,
    849, 591, 302, 297, 514, 363,
    356, 269, 867, 203, 568, 950,
    1000, 981, 6, 286, 212, 592,
    580, 603, 981, 80, 241, 832,
    116, 492, 849, 988, 448, 407,
    106, 259, 189, 631, 373, 730,
    48, 156, 432, 346, 454, 555,
    489, 459, 233, 400, 784, 313,
    195, 133, 874, 729, 171, 478,
    188, 446, 113, 105, 739, 559,
    670, 850, 6, 58, 530, 307,
    697, 137, 219, 857, 856, 68,
    40, 466, 27, 889, 87, 253,
    677, 976, 193, 222, 723, 339,
    // Bin capacity
    2059, 1826, 1741, 1798, 2042, 2205
  };
  const int i6_18_6_34_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    139, 206, 4, 530, 462, 463,
    875, 304, 394, 503, 266, 365,
    472, 199, 189, 823, 549, 382,
    402, 395, 521, 470, 665, 585,
    204, 598, 110, 944, 163, 924,
    76, 606, 287, 821, 445, 604,
    542, 181, 479, 227, 453, 946,
    706, 989, 328, 39, 816, 564,
    319, 881, 164, 643, 308, 927,
    663, 47, 699, 506, 16, 144,
    502, 451, 924, 517, 972, 973,
    775, 212, 540, 385, 398, 233,
    757, 304, 348, 312, 276, 765,
    599, 565, 433, 580, 8, 133,
    65, 432, 565, 702, 324, 947,
    115, 643, 572, 733, 960, 71,
    482, 570, 294, 723, 110, 654,
    134, 212, 455, 492, 500, 178,
    // Bin capacity
    1749, 1741, 1632, 2223, 1718, 2202
  };
  const int i6_18_6_35_36[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    571, 72, 105, 804, 436, 662,
    946, 732, 375, 539, 41, 335,
    67, 612, 786, 248, 214, 626,
    496, 429, 510, 335, 535, 841,
    906, 866, 692, 307, 989, 323,
    151, 881, 163, 45, 623, 500,
    899, 581, 454, 349, 752, 965,
    37, 918, 986, 205, 466, 294,
    850, 743, 30, 398, 170, 302,
    857, 786, 576, 579, 255, 707,
    702, 522, 605, 846, 564, 940,
    94, 239, 817, 812, 709, 517,
    462, 665, 756, 537, 611, 676,
    627, 864, 389, 324, 105, 749,
    860, 987, 537, 100, 340, 584,
    830, 372, 80, 263, 395, 803,
    10, 367, 478, 369, 849, 563,
    943, 411, 720, 488, 570, 48,
    // Bin capacity
    2320, 2486, 2039, 1699, 1941, 2348
  };
  const int i6_18_6_0_37[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    831, 488, 559, 700, 347, 545,
    370, 474, 973, 850, 748, 580,
    229, 538, 929, 567, 25, 116,
    225, 964, 986, 921, 966, 483,
    633, 852, 903, 975, 440, 462,
    571, 284, 393, 424, 261, 927,
    197, 540, 16, 962, 759, 855,
    547, 292, 455, 125, 509, 107,
    767, 369, 496, 317, 101, 256,
    852, 178, 605, 15, 338, 676,
    287, 894, 699, 304, 86, 302,
    855, 890, 809, 145, 257, 913,
    894, 498, 172, 1000, 891, 699,
    334, 918, 576, 855, 643, 999,
    424, 678, 301, 317, 133, 229,
    672, 586, 298, 826, 942, 792,
    386, 349, 925, 753, 890, 493,
    324, 734, 849, 254, 98, 880,
    // Bin capacity
    1567, 1755, 1824, 1719, 1406, 1719
  };
  const int i6_18_6_1_37[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    163, 941, 815, 436, 59, 383,
    216, 414, 391, 988, 248, 113,
    355, 940, 171, 623, 444, 673,
    864, 123, 755, 21, 527, 15,
    799, 978, 479, 461, 916, 900,
    574, 850, 285, 919, 743, 563,
    685, 603, 495, 977, 138, 746,
    763, 30, 482, 684, 223, 997,
    428, 572, 938, 204, 827, 956,
    460, 724, 397, 897, 438, 342,
    777, 961, 685, 103, 556, 857,
    465, 877, 490, 470, 137, 507,
    33, 155, 166, 446, 992, 207,
    52, 172, 647, 580, 940, 110,
    990, 722, 25, 345, 658, 943,
    972, 946, 940, 735, 1000, 841,
    890, 860, 676, 773, 538, 232,
    780, 216, 471, 719, 305, 716,
    // Bin capacity
    1729, 1866, 1567, 1748, 1631, 1701
  };
  const int i6_18_6_2_37[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    382, 803, 459, 82, 189, 412,
    34, 948, 882, 630, 92, 163,
    385, 712, 724, 771, 76, 962,
    172, 551, 164, 190, 661, 601,
    291, 370, 654, 765, 678, 208,
    284, 354, 526, 386, 762, 155,
    74, 962, 117, 345, 290, 768,
    171, 56, 124, 27, 298, 49,
    1000, 750, 854, 688, 733, 367,
    74, 304, 772, 237, 982, 982,
    277, 336, 4, 840, 676, 32,
    780, 891, 124, 74, 644, 768,
    513, 169, 744, 102, 821, 157,
    831, 798, 813, 185, 4, 880,
    983, 76, 222, 257, 267, 587,
    485, 85, 96, 315, 184, 253,
    970, 969, 473, 873, 30, 138,
    882, 179, 35, 776, 594, 894,
    // Bin capacity
    1460, 1584, 1324, 1283, 1357, 1424
  };
  const int i6_18_6_3_37[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    944, 540, 477, 936, 807, 381,
    294, 917, 875, 692, 323, 162,
    941, 915, 669, 346, 857, 444,
    382, 405, 19, 855, 312, 929,
    629, 703, 317, 121, 388, 719,
    144, 811, 645, 50, 785, 832,
    744, 574, 520, 227, 692, 701,
    429, 475, 44, 817, 847, 342,
    875, 394, 862, 584, 137, 589,
    241, 887, 738, 353, 954, 983,
    600, 987, 838, 261, 84, 920,
    734, 300, 25, 936, 859, 160,
    734, 389, 862, 26, 25, 389,
    251, 327, 702, 306, 538, 981,
    397, 146, 810, 208, 177, 558,
    347, 178, 555, 679, 943, 151,
    86, 390, 12, 539, 881, 864,
    269, 121, 206, 841, 590, 34,
    // Bin capacity
    1553, 1624, 1576, 1507, 1751, 1741
  };
  const int i6_18_6_4_37[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    756, 645, 141, 489, 652, 698,
    916, 553, 252, 379, 37, 734,
    714, 745, 41, 153, 693, 901,
    361, 806, 606, 895, 933, 533,
    615, 210, 105, 448, 25, 783,
    507, 434, 922, 85, 623, 943,
    803, 875, 269, 55, 760, 648,
    314, 460, 232, 404, 900, 675,
    608, 543, 170, 736, 420, 97,
    731, 656, 522, 26, 107, 291,
    104, 793, 730, 75, 158, 36,
    809, 221, 381, 435, 98, 487,
    627, 699, 623, 954, 186, 919,
    787, 618, 449, 313, 764, 501,
    214, 520, 494, 683, 254, 536,
    102, 871, 828, 811, 575, 137,
    77, 887, 279, 506, 451, 697,
    5, 290, 739, 294, 384, 664,
    // Bin capacity
    1569, 1877, 1350, 1342, 1391, 1782
  };
  const int i6_18_6_5_37[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    712, 532, 694, 730, 497, 99,
    787, 955, 663, 333, 753, 535,
    559, 374, 478, 219, 50, 642,
    64, 365, 326, 567, 67, 815,
    185, 636, 246, 189, 801, 185,
    768, 102, 793, 365, 202, 514,
    273, 155, 448, 115, 813, 635,
    178, 551, 793, 236, 754, 442,
    96, 401, 492, 80, 584, 271,
    160, 116, 658, 365, 552, 56,
    242, 950, 24, 867, 464, 162,
    582, 723, 728, 127, 423, 955,
    46, 274, 749, 195, 993, 278,
    208, 957, 232, 428, 131, 841,
    298, 564, 572, 395, 147, 104,
    757, 624, 443, 8, 822, 860,
    392, 866, 498, 123, 395, 586,
    148, 692, 255, 949, 758, 708,
    // Bin capacity
    1130, 1722, 1592, 1101, 1612, 1521
  };
  const int i6_18_6_6_37[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    435, 383, 34, 311, 824, 518,
    549, 701, 124, 523, 61, 452,
    815, 553, 157, 355, 770, 429,
    315, 42, 826, 810, 407, 205,
    610, 797, 457, 57, 485, 165,
    473, 357, 413, 544, 156, 844,
    251, 823, 283, 528, 341, 966,
    738, 115, 918, 6, 409, 964,
    648, 184, 661, 345, 619, 366,
    88, 199, 367, 159, 696, 51,
    609, 138, 559, 241, 304, 376,
    986, 938, 525, 100, 700, 995,
    249, 707, 213, 157, 4, 661,
    643, 739, 122, 119, 609, 500,
    183, 487, 142, 113, 979, 210,
    476, 359, 658, 142, 644, 153,
    828, 633, 320, 952, 448, 742,
    895, 622, 674, 605, 222, 11,
    // Bin capacity
    1730, 1551, 1317, 1072, 1534, 1521
  };
  const int i6_18_6_7_37[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    264, 389, 93, 277, 968, 751,
    226, 975, 331, 91, 187, 590,
    328, 896, 430, 838, 121, 593,
    722, 326, 567, 134, 93, 81,
    667, 456, 435, 913, 912, 490,
    594, 356, 385, 524, 372, 522,
    139, 91, 309, 858, 75, 107,
    752, 937, 416, 680, 794, 304,
    888, 489, 716, 758, 506, 91,
    589, 389, 210, 693, 308, 931,
    746, 755, 534, 192, 671, 813,
    188, 157, 448, 390, 691, 576,
    600, 951, 378, 3, 747, 902,
    337, 644, 97, 859, 785, 69,
    104, 24, 127, 970, 567, 229,
    777, 278, 88, 704, 399, 707,
    407, 249, 973, 957, 530, 749,
    755, 304, 145, 9, 974, 107,
    // Bin capacity
    1620, 1546, 1192, 1757, 1730, 1536
  };
  const int i6_18_6_8_37[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    93, 102, 718, 33, 58, 815,
    99, 722, 671, 490, 760, 937,
    992, 98, 503, 733, 24, 447,
    807, 566, 604, 342, 715, 359,
    614, 992, 71, 565, 13, 356,
    830, 589, 271, 682, 560, 194,
    602, 163, 23, 480, 905, 793,
    506, 876, 478, 665, 74, 752,
    107, 475, 744, 164, 127, 840,
    189, 627, 526, 939, 151, 633,
    958, 234, 424, 351, 486, 900,
    771, 383, 499, 720, 646, 324,
    221, 617, 697, 969, 739, 968,
    832, 908, 747, 773, 83, 976,
    745, 928, 560, 296, 691, 819,
    276, 453, 378, 279, 924, 759,
    785, 238, 801, 998, 134, 919,
    243, 133, 130, 858, 377, 943,
    // Bin capacity
    1741, 1639, 1593, 1861, 1345, 2293
  };
  const int i6_18_6_9_37[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    302, 844, 570, 343, 510, 800,
    825, 217, 828, 978, 803, 887,
    561, 70, 890, 22, 226, 407,
    228, 119, 92, 982, 43, 645,
    761, 67, 975, 762, 126, 9,
    971, 557, 529, 788, 240, 885,
    972, 979, 242, 870, 105, 860,
    777, 231, 915, 479, 489, 289,
    495, 613, 298, 921, 185, 249,
    62, 429, 694, 717, 433, 542,
    180, 368, 285, 408, 518, 991,
    278, 637, 741, 82, 69, 71,
    615, 999, 285, 149, 115, 384,
    780, 61, 958, 483, 266, 987,
    42, 460, 617, 631, 934, 56,
    494, 899, 734, 938, 304, 814,
    604, 840, 582, 655, 740, 684,
    598, 513, 154, 684, 432, 537,
    // Bin capacity
    1735, 1618, 1888, 1979, 1188, 1835
  };
  const int i6_18_6_10_37[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    125, 412, 819, 163, 371, 82,
    172, 533, 213, 154, 343, 720,
    665, 784, 962, 325, 995, 967,
    80, 653, 630, 289, 18, 766,
    350, 10, 640, 620, 921, 729,
    778, 435, 984, 79, 956, 406,
    841, 901, 487, 547, 264, 871,
    178, 742, 796, 981, 705, 544,
    127, 675, 542, 205, 582, 894,
    537, 993, 503, 988, 383, 890,
    206, 148, 159, 422, 64, 959,
    484, 497, 359, 491, 159, 495,
    668, 671, 665, 867, 805, 132,
    579, 823, 344, 866, 33, 187,
    401, 135, 178, 454, 983, 769,
    927, 510, 343, 616, 744, 679,
    192, 186, 822, 699, 909, 788,
    448, 841, 49, 357, 490, 870,
    // Bin capacity
    1423, 1824, 1741, 1673, 1783, 2154
  };
  const int i6_18_6_11_37[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    333, 350, 11, 276, 7, 775,
    835, 262, 601, 409, 278, 506,
    255, 244, 573, 80, 176, 556,
    239, 138, 863, 796, 945, 53,
    346, 209, 110, 257, 150, 87,
    222, 930, 872, 131, 184, 942,
    416, 158, 951, 30, 433, 273,
    597, 231, 221, 745, 947, 148,
    963, 785, 406, 577, 34, 191,
    685, 226, 45, 661, 454, 114,
    544, 983, 258, 139, 143, 596,
    87, 86, 4, 527, 736, 722,
    458, 42, 750, 465, 533, 694,
    575, 218, 184, 140, 709, 580,
    15, 841, 530, 616, 878, 231,
    65, 906, 973, 812, 463, 572,
    854, 25, 31, 903, 829, 66,
    790, 799, 868, 557, 637, 572,
    // Bin capacity
    1532, 1376, 1527, 1503, 1580, 1421
  };
  const int i6_18_6_12_37[] = {
    // Dimension, items, bins
    6, 18, 6,
    // Item sizes
    872, 415, 476, 207, 795, 959,
    136, 526, 632, 565, 983, 423,
    411, 179, 197, 845, 636, 417,
    37, 774, 500, 829, 137, 562,
    664, 248, 226, 562, 604, 713,
    125, 721, 83, 606, 531, 346,
    663, 653, 158, 358, 991, 927,
    467, 633, 192, 295, 938, 892,
    536, 9, 751, 550, 371, 450,
    13, 787, 301, 444, 478, 57,
    427, 402, 738, 803, 504, 986,
    480, 663, 450, 136, 99, 514,
    301, 979, 483, 171, 909, 1000,
    485, 522, 17, 822, 743, 105,
    241, 308, 156, 409, 723, 66,
    211, 735, 164, 930, 728, 552,
    490, 383, 230, 181, 91, 389,
    2