/*
 * Copyright (C) 2007 Felipe Weckx <felipe.weckx@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gbemol-mpd.h"
#include "gbemol-info.h"
#include "gbemol-utils.h"

static GObjectClass *parent_class = NULL;

struct _GbemolInfoPrivate {
	GbemolMpd* mpd;

	GtkWidget* lbl_title;
	GtkWidget* lbl_track;
	GtkWidget* lbl_genre;
	GtkWidget* lbl_date;
	GtkWidget* lbl_composer;
	GtkWidget* lbl_performer;
	GtkWidget* lbl_disc;
	GtkWidget* lbl_comment;
	GtkWidget* img_info;
	GtkWidget* vbox_info;
	GtkWidget* vp;

	gboolean dispose_has_run;

	gint mode;
};

/* Constructors/Destructors */
static void gbemol_info_class_init (GObjectClass *g_class);
static void gbemol_info_init (GbemolInfo *obj);
static void gbemol_info_finalize (GObject *object);
static void gbemol_info_dispose (GObject *object);

/* Class functions */

static void gbemol_info_set_mode (GbemolInfo* info, gint mode);
static void gbemol_info_set_song_mode (GbemolInfo* info);
static void gbemol_info_set_image (GbemolInfo* info, GdkPixbuf* img);

/* Callbacks */
static void on_info_style_set (GtkWidget *widget, GtkStyle *style, GtkWidget* event);
static void on_info_style_set_vp (GtkWidget *widget, GtkStyle *style, GtkWidget* vp);
static void on_info_style_set_hbox (GtkWidget *widget, GtkStyle *style, GtkWidget* hbox);

GType
gbemol_info_get_type (void)
{
	static GType type = 0;
	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (GbemolInfoClass),
			NULL,   /* base_init */
			NULL,   /* base_finalize */
			(GClassInitFunc) gbemol_info_class_init,   /* class_init */
			NULL,   /* class_finalize */
			NULL,   /* class_data */
			sizeof (GbemolInfo),
			0,      /* n_preallocs */
			(GInstanceInitFunc) gbemol_info_init    /* instance_init */
		};

		type = g_type_register_static (GTK_TYPE_EVENT_BOX,
				"GbemolInfo",
				&info, 0);
	}
	return type;
}

static void
gbemol_info_class_init (GObjectClass *g_class)
{
	parent_class = g_type_class_peek_parent (g_class);
	g_class->finalize = gbemol_info_finalize;
	g_class->dispose = gbemol_info_dispose;	
}

static void
gbemol_info_dispose (GObject* obj)
{
	GbemolInfo* self = (GbemolInfo *) obj;

	if (self->priv->dispose_has_run)
		return;
	self->priv->dispose_has_run = TRUE;

	G_OBJECT_CLASS (parent_class)->dispose (obj);

}

static void 
gbemol_info_finalize (GObject *obj)
{
	g_free (GBEMOL_INFO (obj)->priv);
	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static void
gbemol_info_init (GbemolInfo *obj)
{
	obj->priv = g_new0 (GbemolInfoPrivate, 1);
	obj->priv->dispose_has_run = FALSE;
}

GbemolInfo*
gbemol_info_new (GbemolMpd* mpd)
{
	GbemolInfo *info;
	GtkWidget *vbox, *hbox, *scr, *event;

	info = GBEMOL_INFO (g_object_new (GBEMOL_TYPE_INFO, NULL));
	info->priv->mpd = mpd;
	info->priv->mode = 0;

	vbox = gtk_vbox_new (FALSE, 5);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 12);

	/* Set Backgound Color */
	gtk_widget_set_app_paintable (GTK_WIDGET (info), TRUE);
	g_signal_connect (G_OBJECT (vbox), "style-set", G_CALLBACK (on_info_style_set), info);
	gtk_container_add (GTK_CONTAINER (info), vbox);

	event = gtk_event_box_new ();
	gtk_widget_set_app_paintable (event, TRUE);	
	g_signal_connect (G_OBJECT (vbox), "style-set", G_CALLBACK (on_info_style_set_hbox), event);
	gtk_box_pack_start (GTK_BOX (vbox), event, FALSE, FALSE, 5);

	hbox = gtk_hbox_new (FALSE, 5);
	gtk_container_add (GTK_CONTAINER (event), hbox);

	info->priv->lbl_title = gtk_label_new ("");
	gtk_box_pack_start (GTK_BOX (hbox), info->priv->lbl_title, FALSE, FALSE, 10);

	info->priv->img_info = gtk_image_new ();
	gtk_box_pack_end (GTK_BOX (hbox), info->priv->img_info, FALSE, FALSE, 5);

	scr = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scr), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start (GTK_BOX (vbox), scr, TRUE, TRUE, 0);

	info->priv->vp = gtk_viewport_new (gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW (scr)), 
				gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (scr)));
	gtk_widget_set_app_paintable (info->priv->vp, TRUE);
	gtk_container_add (GTK_CONTAINER (scr), info->priv->vp);
	g_signal_connect (G_OBJECT (vbox), "style-set", G_CALLBACK (on_info_style_set_vp), info->priv->vp);

	info->priv->vbox_info = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (info->priv->vp), info->priv->vbox_info);

	gtk_widget_show_all (GTK_WIDGET (info));

	return info;
}

static void
gbemol_info_set_song_mode (GbemolInfo* info)
{
	GtkWidget *table, *lbl;

	table = gtk_table_new (7, 2, TRUE);
	gtk_table_set_row_spacings (GTK_TABLE (table), 3);
	gtk_table_set_col_spacings (GTK_TABLE (table), 3);

	lbl = gbemol_utils_label_new_with_markup (_("<b>Track</b>"));
	gtk_table_attach (GTK_TABLE (table), lbl, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	info->priv->lbl_track = gbemol_utils_label_new_with_markup ("");
	gtk_table_attach (GTK_TABLE (table), info->priv->lbl_track, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	lbl = gbemol_utils_label_new_with_markup (_("<b>Genre</b>"));
	gtk_table_attach (GTK_TABLE (table), lbl, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
	info->priv->lbl_genre = gbemol_utils_label_new_with_markup ("");
	gtk_table_attach (GTK_TABLE (table), info->priv->lbl_genre, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);

	lbl = gbemol_utils_label_new_with_markup (_("<b>Date</b>"));
	gtk_table_attach (GTK_TABLE (table), lbl, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
	info->priv->lbl_date = gbemol_utils_label_new_with_markup ("");
	gtk_table_attach (GTK_TABLE (table), info->priv->lbl_date, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 3, 3);

	lbl = gbemol_utils_label_new_with_markup (_("<b>Composer</b>"));
	gtk_table_attach (GTK_TABLE (table), lbl, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 3, 3);
	info->priv->lbl_composer = gbemol_utils_label_new_with_markup ("");
	gtk_table_attach (GTK_TABLE (table), info->priv->lbl_composer, 1, 2, 3, 4, GTK_FILL, GTK_FILL, 3, 3);

	lbl = gbemol_utils_label_new_with_markup (_("<b>Performer</b>"));
	gtk_table_attach (GTK_TABLE (table), lbl, 0, 1, 4, 5, GTK_FILL, GTK_FILL, 3, 3);
	info->priv->lbl_performer = gbemol_utils_label_new_with_markup ("");
	gtk_table_attach (GTK_TABLE (table), info->priv->lbl_performer, 1, 2, 4, 5, GTK_FILL, GTK_FILL, 3, 3);

	lbl = gbemol_utils_label_new_with_markup (_("<b>Disc</b>"));
	gtk_table_attach (GTK_TABLE (table), lbl, 0, 1, 5, 6, GTK_FILL, GTK_FILL, 3, 3);
	info->priv->lbl_disc = gbemol_utils_label_new_with_markup ("");
	gtk_table_attach (GTK_TABLE (table), info->priv->lbl_disc, 1, 2, 5, 6, GTK_FILL, GTK_FILL, 3, 3);

	lbl = gbemol_utils_label_new_with_markup (_("<b>Comment</b>"));
	gtk_table_attach (GTK_TABLE (table), lbl, 0, 1, 6, 7, GTK_FILL, GTK_FILL, 3, 3);
	info->priv->lbl_comment = gbemol_utils_label_new_with_markup ("");
	gtk_table_attach (GTK_TABLE (table), info->priv->lbl_comment, 1, 2, 6, 7, GTK_FILL, GTK_FILL, 3, 3);

	gtk_box_pack_start (GTK_BOX (info->priv->vbox_info), table, FALSE, FALSE, 0);
}


static void 
gbemol_info_set_mode (GbemolInfo* info, gint mode)
{
	if (mode == info->priv->mode)
		return;

	/* Destroy current widgets */
	gtk_widget_destroy (info->priv->vbox_info);

	info->priv->vbox_info = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (info->priv->vp), info->priv->vbox_info);

	switch (mode)
	{
		case INFO_MODE_SONG:
			gbemol_info_set_song_mode (info);
			break;
		case INFO_MODE_ALBUM:
			break;
		case INFO_MODE_ARTIST:
			break;
	}

	gtk_widget_show_all (info->priv->vbox_info);
}

static void 
gbemol_info_set_image (GbemolInfo* info, GdkPixbuf* img)
{
	GdkPixbuf* scaled;

	if (img)
	{
		scaled = gdk_pixbuf_scale_simple (img, 100, 100, GDK_INTERP_BILINEAR);
		gtk_image_set_from_pixbuf (GTK_IMAGE (info->priv->img_info), scaled);
		g_object_unref (img);
		g_object_unref (scaled);
		gtk_widget_show (info->priv->img_info);
	}
	else
		gtk_widget_hide (info->priv->img_info);
}

/* Set a song info in the info tab */
void
gbemol_info_set_song (GbemolInfo* info, GbemolMpdSong* song)
{
	gchar *str;

	if (!song)
		return;

	gbemol_info_set_mode (info, INFO_MODE_SONG);

	if (song->title != NULL)
		str = g_markup_printf_escaped (_("<span size=\"xx-large\" weight=\"ultrabold\">%s</span>\n"
					"by <b>%s</b> from <b>%s</b>"), song->title, song->artist, song->album);
	else
		str = g_markup_printf_escaped (_("<span size=\"xx-large\" weight=\"ultrabold\">%s</span>"), song->name);
	gtk_label_set_markup (GTK_LABEL (info->priv->lbl_title), str);
	g_free (str);

	if (song->track != NULL)
		gtk_label_set_text (GTK_LABEL (info->priv->lbl_track), song->track);
	else
		gtk_label_set_text (GTK_LABEL (info->priv->lbl_track), _("Unknown"));
	if (song->genre != NULL)
		gtk_label_set_text (GTK_LABEL (info->priv->lbl_genre), song->genre);
	else
		gtk_label_set_text (GTK_LABEL (info->priv->lbl_genre), _("Unknown"));
	if (song->date != NULL)
		gtk_label_set_text (GTK_LABEL (info->priv->lbl_date), song->date);
	else
		gtk_label_set_text (GTK_LABEL (info->priv->lbl_date), _("Unknown"));
	if (song->composer != NULL)
		gtk_label_set_text (GTK_LABEL (info->priv->lbl_composer), song->composer);
	else
		gtk_label_set_text (GTK_LABEL (info->priv->lbl_composer), _("Unknown"));
	if (song->performer != NULL)
		gtk_label_set_text (GTK_LABEL (info->priv->lbl_performer), song->performer);
	else
		gtk_label_set_text (GTK_LABEL (info->priv->lbl_performer), _("Unknown"));
	if (song->disc != NULL)
		gtk_label_set_text (GTK_LABEL (info->priv->lbl_disc), song->disc);
	else
		gtk_label_set_text (GTK_LABEL (info->priv->lbl_disc), _("Unknown"));
	if (song->comment != NULL)
		gtk_label_set_text (GTK_LABEL (info->priv->lbl_comment), song->comment);
	else
		gtk_label_set_text (GTK_LABEL (info->priv->lbl_comment), _("Unknown"));

	gbemol_info_set_image (info, gbemol_utils_get_song_image (song));
}

void
gbemol_info_set_album (GbemolInfo* info, gchar* album, gchar* artist)
{
	gchar *str;
	GList* songs;
	GtkWidget* lbl;
	GbemolMpdSong* song;
	gint time = 0;

	gbemol_info_set_mode (info, INFO_MODE_ALBUM);

	str = g_markup_printf_escaped ("<span size=\"x-large\" weight=\"heavy\">%s</span><b>  by  </b>"
					"<span size=\"x-large\" weight=\"heavy\">%s</span>", album, artist);
	gtk_label_set_markup (GTK_LABEL (info->priv->lbl_title), str);
	g_free (str);
	
	/* Get the album songs */
	gbemol_mpd_search_start (info->priv->mpd);
	gbemol_mpd_search_add_constraint (info->priv->mpd, MPD_TAG_ITEM_ARTIST, artist);
	gbemol_mpd_search_add_constraint (info->priv->mpd, MPD_TAG_ITEM_ALBUM, album);
	songs = gbemol_mpd_search_get_results (info->priv->mpd);

	do
	{
		song = songs->data;
		if (song->time != MPD_SONG_NO_TIME)
			time += song->time;
		str = g_strdup_printf ("%s %s", song->track, song->title);
		lbl = gbemol_utils_label_new_with_markup (str);
		gtk_box_pack_start (GTK_BOX (info->priv->vbox_info), lbl, FALSE, FALSE, 3);
		g_free (str);
		gtk_widget_show (lbl);
	} while ((songs = g_list_next (songs)));

	if (time > 3600)
		str = g_markup_printf_escaped (_("\n<b>Total Time: </b> %02d:%02d:%02d"), time/3600, 
				(time%3600)/60, (time%3600)%60);
	else
		str = g_markup_printf_escaped (_("\n<b>Total Time: </b> %02d:%02d"), time/60, time%60);

	lbl = gbemol_utils_label_new_with_markup (str);
	gtk_box_pack_start (GTK_BOX (info->priv->vbox_info), lbl, FALSE, FALSE, 5);
	g_free (str);
	gtk_widget_show (lbl);

	gbemol_mpd_free_song_list (songs);
}

void gbemol_info_set_artist (GbemolInfo* info, gchar* artist)
{
	GList *albums;
	GtkWidget* lbl;
	gchar *str;

	gbemol_info_set_mode (info, INFO_MODE_ARTIST);

	str = g_markup_printf_escaped ("<span size=\"xx-large\" weight=\"heavy\">%s</span>", artist);
	gtk_label_set_markup (GTK_LABEL (info->priv->lbl_title), str);
	g_free (str);

	/* Find all the albums by artist */
	gbemol_mpd_search_field_start (info->priv->mpd, MPD_TABLE_ALBUM);
	gbemol_mpd_search_add_constraint (info->priv->mpd, MPD_TABLE_ARTIST, artist);
	albums = gbemol_mpd_search_field_get_results (info->priv->mpd, MPD_TABLE_ALBUM);

	str = g_markup_printf_escaped ("<span size=\"large\" weight=\"bold\">Albums by %s\n</span>", artist);
	lbl = gbemol_utils_label_new_with_markup (str);
	gtk_box_pack_start (GTK_BOX (info->priv->vbox_info), lbl, FALSE, FALSE, 3);
	gtk_widget_show (lbl);
	g_free (str);

	do
	{
		str = albums->data;
		lbl = gbemol_utils_label_new_with_markup (str);
		gtk_box_pack_start (GTK_BOX (info->priv->vbox_info), lbl, FALSE, FALSE, 3);
		g_free (str);
		gtk_widget_show (lbl);
	} while ((albums = g_list_next (albums)));

	gbemol_utils_char_list_free (albums);
}

static void 
on_info_style_set (GtkWidget *widget, GtkStyle *style, GtkWidget* event)
{
	gtk_widget_modify_bg(event, GTK_STATE_NORMAL, &(widget->style->base[GTK_STATE_NORMAL]));
}

static void 
on_info_style_set_vp (GtkWidget *widget, GtkStyle *style, GtkWidget* vp)
{
	gtk_widget_modify_bg(vp, GTK_STATE_NORMAL, &(widget->style->base[GTK_STATE_NORMAL]));
}

static void 
on_info_style_set_hbox (GtkWidget *widget, GtkStyle *style, GtkWidget* vp)
{
	gtk_widget_modify_bg(vp, GTK_STATE_NORMAL, &(widget->style->base[GTK_STATE_SELECTED]));
}


