/* $Id: ValDeclaration.hpp 4323 2009-01-27 13:48:12Z potyra $
 *
 * ValDeclaration: abstract representation of a signal, variable or constant 
 *                 declaration.
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __VAL_DECLARATION_HPP_INCLUDDED
#define __VAL_DECLARATION_HPP_INCLUDDED

#include <string>
#include "frontend/ast/AttributableDeclaration.hpp"
#include "frontend/ast/Expression.hpp"
#include "frontend/ast/SubtypeIndication.hpp"

namespace ast {

//! A signal, variable or constant declaration.
/** This class represents a declaration of either a VHDL signal,
 *  a VHDL variable or a VHDL constant.
 */
class ValDeclaration : public AttributableDeclaration {
public:

	/** VHDL mode attribute (in, out, inout)
	 */
	enum Mode {
		MODE_IN, 	/**< access mode in */
		MODE_OUT,	/**< access mode out */
		MODE_INOUT	/**< access mode inout */
	};

	/** Storage class (signal, constant, variable) */
	enum ObjClass {
		OBJ_CLASS_SIGNAL, 	/**< signal */
		OBJ_CLASS_VARIABLE,	/**< variable */
		OBJ_CLASS_CONSTANT,	/**< constant */
		OBJ_CLASS_UNSPECIFIED	/**< unspecified */
	};

	/** mode in which this declaration is used */
	enum UsageMode {
		/** not used at all */
		USAGE_NONE 	= 0,
		/** read from VHDL */
		USAGE_READ	= 1 << 0,
		/** written to from VHDL */
		USAGE_WRITE 	= 1 << 1,
		/** signal used in a foreign component */
		USAGE_FOREIGN	= 1 << 2
	};
	
	/** c'tor 
	 *  @param accessMode mode of the Signal (in, out, inout).
	 *  @param declName the declared name 
	 *  @param varInit initializer Expression for the value.
	 *  @param subtype subtype indication of the value.
	 *  @param sc storage class for convenience.
	 *  @param loc Location of the declaration.
	 */
	ValDeclaration(
		enum Mode accessMode,
		std::string *declName, 
		Expression *varInit,
		SubtypeIndication *subtype,
		enum ObjClass sc,
		Location loc
		) : 	AttributableDeclaration(declName, loc),
			init(varInit), 
			mode(accessMode),
			subtypeIndic(subtype),
			storageClass(sc),
			usage(USAGE_NONE) {}

	/** initial value (optional) */
	Expression *init;
	/** access mode */
	enum Mode mode;
	/** subtype indication */
	SubtypeIndication *subtypeIndic;
	/** storage class for convenience */
	enum ObjClass storageClass;

	/** does the declaration denote an Unconstraint array?
	 *  @return true if the declaration denotes an unconstraint array,
	 *          false otherwise.
	 */
	bool isUnconstraint(void) const;

	/** useage of the declaration */
	int usage;

protected:
	/** Destructor */
	virtual ~ValDeclaration() {
		util::MiscUtil::terminate(init);
		util::MiscUtil::terminate(subtypeIndic);
	}
};
	
}; /* namespace ast */

#endif /* __VAL_DECLARATION_HPP_INCLUDDED */
