/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/test_list_result.cpp
 *
 * @brief [LEVEL: beta] Implementation of @ref
 * diagnostics::unittest::Test_List_Result class
 *
 * $Id: test_list_result.cpp,v 1.7 2005/06/23 09:54:26 esdentem Exp $
 * 
 * @author Christian Schallhart
 */

#include <diagnostics/unittest/test_system/test_list_result.hpp>

#include <diagnostics/unittest/test_system_exception.hpp>
#include <diagnostics/unittest/name_separators.hpp>

#include <iomanip>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

Test_List_Result::Test_List_Result(::std::string const & path,
				   int const timeout,
				   bool const is_rerunable,
				   bool const is_testable_at_prod,
				   bool const is_testable_at_debug,
				   bool const is_testable_at_audit)
    : m_test_case_path(path),
      m_timeout(timeout),
      m_is_rerunable(is_rerunable),
      m_levels((is_testable_at_prod ? (1<<LEVEL_PROD) : 0)
	       |(is_testable_at_debug ? (1<<LEVEL_DEBUG) : 0)
	       |(is_testable_at_audit ? (1<<LEVEL_AUDIT) : 0))
{
    if(!path_adheres_convention(m_test_case_path))
	throw Test_System_Exception("path does not follow the conventions: '" + path + "'");
    if(m_timeout<0)
	throw Test_System_Exception("timeout < 0");
}


Test_List_Result::Test_List_Result(Self const & other)
    : m_test_case_path(other.m_test_case_path),
      m_timeout(other.m_timeout),
      m_is_rerunable(other.m_is_rerunable),
      m_levels(other.m_levels)
{
}

bool Test_List_Result::is_testable_at(Level_t const level) const 
{
    if(level==LEVEL_TEST) 
	throw Test_System_Exception("level is not allowed to be set to LEVEL_TEST");
    return 0!=((1<<level)&m_levels); 
}


Test_List_Result & Test_List_Result::operator=(Self const & other) 
{
    m_test_case_path=other.m_test_case_path;
    m_timeout=other.m_timeout;
    m_is_rerunable=other.m_is_rerunable;
    m_levels=other.m_levels;
    return *this;
}    

bool Test_List_Result::operator==(Self const & r) const
{
    return m_test_case_path==r.m_test_case_path
	&& m_timeout==r.m_timeout
	&& m_is_rerunable==r.m_is_rerunable
	&& m_levels==r.m_levels;
}


UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;



::std::ostream & operator<<(::std::ostream & stream, ::diagnostics::unittest::Test_List_Result const & r)
{
    using namespace diagnostics;
    
    stream << ::std::setw(4) << r.timeout() 
	   << (r.is_rerunable() ? " REPEAT " : " SINGLE ")
	   << (r.is_testable_at(LEVEL_AUDIT) ? "AUD " : "    ")
	   << (r.is_testable_at(LEVEL_DEBUG) ? "DEB " : "    ")
	   << (r.is_testable_at(LEVEL_PROD)  ? "PRO " : "    ")
	   << r.test_case_path();
    return stream;
}
// vim:ts=4:sw=4
