/*
 * Copyright © 2011 Canonical Limited
 * Copyright © 2010 Codethink Limited
 *
 * This library is free software: you can redistribute it and/or modify it under the terms of version 3 of the
 * GNU Lesser General Public License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this program.  If not,
 * see <http://www.gnu.org/licenses/>.
 *
 * Author: Ryan Lortie <desrt@desrt.ca>
 */

#include "qconfschema.h"

#include <QMetaProperty>
#include <QStringList>
#include <QHash>

#include "gvdb/gvdb-reader.h"
#include "qconftypes.h"
#include <libintl.h>

struct _QConfSchemaPrivate
{
  QString     name;
  QMetaObject meta;

  QList<char *>     names;
  QList<GVariant *> values;

  const char *gettext_domain;
  const char *path;
};

/* The following function is lifted verbatim out of gsettingsschema.c in GLib.  Do not modify it without first
 * modifying GLib, since doing so would introduce incompatible behaviour for schema lookups which is likely to
 * be quite confusing for everyone.
 */
static GSList *schema_sources;

static void
initialise_schema_sources(void)
{
  static gsize initialised;

  if G_UNLIKELY (g_once_init_enter (&initialised))
    {
      const gchar * const *dir;
      const gchar *path;

      for (dir = g_get_system_data_dirs (); *dir; dir++)
        {
          gchar *filename;
          GvdbTable *table;

          filename = g_build_filename (*dir, "glib-2.0", "schemas",
                                       "gschemas.compiled", NULL);
          table = gvdb_table_new (filename, TRUE, NULL);

          if (table != NULL)
            schema_sources = g_slist_prepend (schema_sources, table);

          g_free (filename);
        }

      schema_sources = g_slist_reverse (schema_sources);

      if ((path = g_getenv ("GSETTINGS_SCHEMA_DIR")) != NULL)
        {
          gchar *filename;
          GvdbTable *table;

          filename = g_build_filename (path, "gschemas.compiled", NULL);
          table = gvdb_table_new (filename, TRUE, NULL);

          if (table != NULL)
            schema_sources = g_slist_prepend (schema_sources, table);

          g_free (filename);
        }

      g_once_init_leave (&initialised, TRUE);
    }
}

const QConfSchema *
QConfSchema::getSchema(const QString & name) {
    static QHash<QString, QConfSchema*> schemas;
    QConfSchema *schema;

    initialise_schema_sources();

    schema = schemas.value(name);


    if (schema == NULL)
      {
        schema = new QConfSchema(name);
        schemas.insert(name, schema);
      }

    return schema;
}

static const char *lookup_string(GvdbTable *table, const char *key)
{
    const char *result = NULL;
    GVariant *value;

    value = gvdb_table_get_raw_value(table, key);

    if (value != NULL) {
        result = g_variant_get_string(value, NULL);
        g_variant_unref(value);
    }

    return result;
}

static int addString(QByteArray &bytes, const QString& string) {
    int offset = bytes.size();

    bytes.append(string);
    bytes.append('\0');

    return offset;
}

/* convert 'some-key' to 'someKey' or 'SomeKey'.
 * the second form is needed for appending to 'set' for 'setSomeKey'
 */
static QString qtify_name(const char *name)
{
    bool next_cap = false;
    QString result;

    while (*name) {
        if (*name == '-') {
            next_cap = true;
        } else if (next_cap) {
            result.append(toupper(*name));
            next_cap = false;
        } else {
            result.append(*name);
        }

        name++;
    }

    return result;
}

// This function adds a string of the form 'firstSecond(type)'
static uint addSignature(QByteArray& bytes, const QString& first, const QString& second, const QString& type)
{
    uint offset = bytes.size();

    bytes.append(first);

    bytes.append(toupper(second[0].unicode()));
    bytes.append(second.mid(1));

    bytes.append("(");
    bytes.append(type);
    bytes.append(")");

    bytes.append('\0');

    return offset;
}


/* A QMetaObject contains an array of unsigned integers and an array of characters.  The character array
 * contains nul-terminated strings at offsets indicated by elements in the integer array.
 *
 * Note that the QMetaObject ABI is stable.  Any future changes to Qt should be compatible with what we do here
 * (for the same reason that these changes have to be compatible with programs compiled with old versions of
 * moc).
 *
 * The first 14 elements of the integer array are a header.  The header contains integers representing the
 * number of properties and methods (signals and slots) described by the metaobject (among other things that we
 * do not presently use).  The header also contains offsets to locations within the integer array where the
 * properties and methods are actually described.
 *
 * Method descriptions are 5 integers in length.  If the header indicates that there are 4 methods, starting at
 * position 14 then we know that positions 14 up to (not including) 34 are taken by method descriptions (14 + 4
 * * 5).  The 5 integers are:
 *
 *   ("string" means an offset into the string array at which a nul-terminated string is stored)
 *
 *   - 0: string: the signature of the method, like "method(int)"
 *   - 1: string: the comma-separated names of the arguments
 *   - 2: string: the name of the return type or "" for void
 *   - 3: string: a tag, currently unsupported by moc: always ""
 *   - 4: bitfield: the flags
 *
 * Similarly, property descriptions are 3 integers in length.  Properties can contain an optional 'Notify' flag,
 * however, to indicate that an additional integer for that property follows all of the other property
 * descriptors to indicate which signal notifies of changes to the property.  The index of the signal is given
 * (in the order that the signal is enumerated among the methods).
 *
 *   - 0: string: the name of the property
 *   - 1: string: the name of the type of the property
 *   - 2: bitfield: the flags in the low 24 bits and the QVariant::Type in the top 8
 *
 * This is all followed by a single zero value.  I'm not sure if this is needed, but moc puts it there and it
 * may have some forward-compatibility implications, so we put it there too.
 *
 * For each GSettings schema entry (e.g. 'some-key') we create 3 things:
 *
 *   - a property    'someKey'
 *   - a signal      'someKeyChanged'
 *   - a slot        'setSomeKey'
 *
 * So for 'n' keys in the schema, we will have 'n' properties, 'n' signals and 'n' slots.
 *
 * The property and signal are essential for using QConf from QML.  The slot is useful for programming with
 * traditional Qt, since it means that you can (for example) connect the 'toggled(bool)' signal on a QCheckBox
 * to the 'setSomeKey(bool)' slot on a boolean key.
 *
 * Of course, we wire 'someKeyChanged' up as the notify signal for 'someKey'.
 *
 * To that end, we lay out the integer array as follows (for 'n' keys):
 *
 * -- offset 0 ------------------------------------------
 *   header                                     (14)
 *
 * -- offset 14 -----------------------------------------
 *   'n' signal descriptions                    (5 * n)
 *
 * -- offset 14 + 5 * n ---------------------------------
 *   'n' slot descriptions                      (5 * n)
 *
 * -- offset 14 + 5 * n + 5 * n -------------------------
 *   'n' property descriptions                  (3 * n)
 *
 * -- offset 14 + 5 * n + 5 * n + 3 * n -----------------
 *   'n' notify signal indices                  (1 * n)
 *
 * -- offset 14 + 5 * n + 5 * n + 3 * n + n -------------
 *   zero                                       (1)
 *
 * -- total size 14 + 5 * n + 5 * n + 3 * n + n + 1 -----
 */


void QConfSchema::buildMetaObject() {
    uint n = count();

    // verbatim from the table above
    uint signal_descriptions_offset   = 14;
    uint slot_descriptions_offset     = 14 + 5 * n;
    uint property_descriptions_offset = 14 + 5 * n + 5 * n;
    uint notify_indices_offset        = 14 + 5 * n + 5 * n + 3 * n;
    uint zero_offset                  = 14 + 5 * n + 5 * n + 3 * n + n;
    uint total_size                   = 14 + 5 * n + 5 * n + 3 * n + n + 1;

    // some constants lifted from qmetaobject_p.h (for improved readability below)
    const uint MethodProtected    = 0x000001;
    const uint MethodPublic       = 0x000002;
    const uint MethodSignal       = 0x000004;
    const uint MethodSlot         = 0x000008;
    const uint PropertyScriptable = 0x004000;
    const uint PropertyWritable   = 0x000002;
    const uint PropertyNotify     = 0x400000;

    // the actual arrays to build into
    uint *data = new uint[total_size];
    QByteArray stringdata;

    // build the header
    {
        // version
        data[0] = 5;

        // class name
        data[1] = addString(stringdata, "QConf");

        // class info
        data[2] = 0;
        data[3] = 0;

        // methods: signals and slots
        data[4] = n + n;
        data[5] = signal_descriptions_offset;

        // properties
        data[6] = n;
        data[7] = property_descriptions_offset;

        // enums (TODO: GSettings enum conversion)
        data[8] = 0;
        data[9] = 0;

        // constructors
        data[10] = 0;
        data[11] = 0;

        // flags
        data[12] = 0;

        // signal count
        data[13] = n;
    }

    // optimisation: we use the empty string a lot, so write it once and reuse the offset
    uint emptyStringOffset = addString(stringdata, "");

    // also, the string "value"
    uint valueOffset = addString(stringdata, "value");

    // write the signal, slot and property descriptions with notify signal indices
    for (uint i = 0; i < n; i++) {
        // the offsets of the various things that we will write during this iteration
        uint *signal   = data + signal_descriptions_offset   + 5 * i; // methods are 5 ints in length
        uint *slot     = data + slot_descriptions_offset     + 5 * i;
        uint *property = data + property_descriptions_offset + 3 * i; // a property is 3 ints
        uint *notify   = data + notify_indices_offset        + i;     // a notify index is 1

        QVariant::Type type = qconf_types_convert(valueType(i));
        // these should have been filtered out already
        Q_ASSERT (type != QVariant::Invalid);
        QString typeName = QVariant::typeToName(type);

        const char *key_name = keyName(i);      // name like "some-key"
        QString qtKey = qtify_name(key_name);   // name like "someKey"

        // write the signal 'someKeyChanged(type)'
        signal[0] = addSignature(stringdata, qtKey, "Changed", typeName);       // signature
        signal[1] = valueOffset;                                                // argument names
        signal[2] = emptyStringOffset;                                          // return type
        signal[3] = emptyStringOffset;                                          // tag
        signal[4] = MethodProtected | MethodSignal;                             // flags

        // write the slot 'setSomeKey(type)'
        slot[0] = addSignature(stringdata, "set", qtKey, typeName);             // signature
        slot[1] = valueOffset;                                                  // argument names
        slot[2] = emptyStringOffset;                                            // return type
        slot[3] = emptyStringOffset;                                            // tag
        slot[4] = MethodPublic | MethodSlot;                                    // flags

        // write the property 'someKey'
        property[0] = addString(stringdata, qtKey);                             // name
        property[1] = addString(stringdata, typeName);                          // type name
        property[2] = PropertyWritable | PropertyScriptable | PropertyNotify | (type << 24);

        // the notify signal has the same index as the property
        notify[0] = i;
    }

    // and the zero at the end
    data[zero_offset] = 0;

    // set up the QMetaObject.  We need to copy the stack-allocated stringdata.
    QMetaObject meta = { { &QObject::staticMetaObject,
                           (char *) g_memdup(stringdata.constData(), stringdata.size()),
                           data, 0 } };
    priv->meta = meta;
}

QConfSchema::QConfSchema(const QString & name) {
    GvdbTable *table = NULL;
    GSList *source;
    char **list;
    int i;

    priv = new QConfSchemaPrivate;

    for (source = schema_sources; source; source = source->next) {
        GvdbTable *file = (GvdbTable *) source->data;

        if ((table = gvdb_table_get_table(file, name.toUtf8())))
            break;
    }

    if (table == NULL)
        qFatal("Settings schema '%s' is not installed\n", name.toUtf8().constData());

    list = gvdb_table_list(table, "");

    for (i = 0; list[i]; i++) {
        if (list[i][0] != '.') {
            GVariant *value;

            value = gvdb_table_get_raw_value(table, list[i]);

            // Skip values that we are incapable of supporting
            if (!g_variant_type_is_tuple(g_variant_get_type(value)) ||
                qconf_types_convert(g_variant_type_first(g_variant_get_type(value))) == QVariant::Invalid)
                continue;

            priv->names.append(g_strdup(list[i]));

            Q_ASSERT (value != NULL);

            priv->values.append(value);
        }
    }

    g_strfreev(list);

    priv->name = name;
    priv->gettext_domain = lookup_string(table, ".gettext-domain");
    priv->path = lookup_string(table, ".path");
    gvdb_table_unref(table);

    if (priv->gettext_domain != NULL)
        bind_textdomain_codeset(priv->gettext_domain, "UTF-8");

    buildMetaObject();
}

const QMetaObject *QConfSchema::metaObject() const
{
    return &priv->meta;
}

GVariant *QConfSchema::defaultValue(int id) const
{
    return g_variant_get_child_value(priv->values[id], 0);
}

const GVariantType *QConfSchema::valueType(int id) const
{
    return g_variant_type_first(g_variant_get_type(priv->values[id]));
}

const char *QConfSchema::keyName(int id) const
{
    return priv->names[id];
}

const char *QConfSchema::path() const {
    return priv->path;
}

int QConfSchema::count() const {
    Q_ASSERT (priv->names.size() == priv->values.size());

    return priv->names.size();
}

const QString& QConfSchema::name() const {
    return priv->name;
}

int QConfSchema::findKey(const char *key) const
{
    int i;

    for (i = 0; i < priv->names.size(); i++) {
        if (strcmp(priv->names[i], key) == 0) {
            return i;
        }
    }

    return -1;
}

// vim:sw=4 tw=112
