/*
 *  RpcCommandHandler.h
 *  RpcDriver
 *
 *  Created by Mikael Gransell on 1/31/06.
 *  Copyright 2006 __MyCompanyName__. All rights reserved.
 *
 */

#ifndef _RPC_COMMAND_H_
#define _RPC_COMMAND_H_

#include <boost/shared_ptr.hpp>
#include <boost/any.hpp>
#include <boost/function.hpp>

#include <string>
#include <list>

namespace rpc {
	
/// Smart pointer class for queueing commands
typedef boost::shared_ptr< std::list<boost::any> > CmdPtr;
	
/**
 * Base class for all commands that want to register for notifications
 * when commands come in to the RpcDriver.
 */
class RpcCommandHandler {

public:
	/**
	 * Create a command that corresponds to the given name.
	 */
	explicit RpcCommandHandler( const std::string& name,
								boost::function<void (int, const CmdPtr&)> sender,
								short np ) 
	: cmdSenderMethod( sender ), cmdName( name ), numParams(np)
	{ }
	
	virtual ~RpcCommandHandler() {}
		
	/**
	 * Retreive the name that this command shoul be registered for.
	 * @return Command name.
	 */
	const std::string& getCmdName() const { return cmdName; }
	
	/**
	 * Method that actually handles the command.
	 * @param clientId Id of the client that sent this command. Use this
	 *				   id when sending replies to a specific client.
	 * @param params Variant containing the parameters that were sent.
	 */
	virtual void handleCommand( int clientId, 
								const std::list<boost::any>& params ) = 0;
	
	/**
	 * Get the amount of params that this command wants.
	 * @return Number of params we want
	 */
	short getNumParams() const { return numParams; }

protected:
	/// Function pointer to a method that allows us to send
	/// commands using the pc driver.
	boost::function<void (int, const CmdPtr&)> cmdSenderMethod; 
	
private:
	/// The name that this command is registered for
	const std::string cmdName;
	/// The amount of paramams that this command should receive
	const short numParams;
};

/// Smart pointer class for simplicity
typedef boost::shared_ptr<RpcCommandHandler> RpcCommandHandlerPtr;

}

#endif
