/*

  Author:  Bob Dean
  Copyright (c) 1999, 2000


   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public Licensse as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

 */

#ifdef __cplusplus
extern "C"
{
#endif

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <limits.h>
#include <fcntl.h>
#include <signal.h>
#include <stdarg.h>
#include <sys/shm.h>
#include <glib.h>
#include <math.h>
#include <string.h>

#include <dbsoundcard.h>
#include <dbchannel.h>
#include <dbdebug.h>
#include <dbaudiolib.h>

#include "prototypes.h"
	
	/* variables to access DBMix channels shared memory */
	extern  int shmid, sysshmid;
	extern dbfsd_data * sysdata;
	extern local_channel * local_channels;
	/* static local_channel * cue_channels; */
	extern local_channel * ch;
	
	extern int errno;
	extern int debug_level; /* declared in debug module */
	
	extern float local_pitch;
	extern float sample1, sample2;
	extern float float_index;
	
	extern signed short output_buf[OUTPUT_BUFSIZE];  /* buffer used to hold main output to dbfsd */
	extern signed short cue_buf[OUTPUT_BUFSIZE];     /* buffer used to hold cue output to dbfsd */
	extern signed short conv_buf[OUTPUT_BUFSIZE];    /* buffer used to hold format converted input */
	
	extern int sampleindex;
	extern int outlen;
	extern int num_channels;
	extern int format;
	
	/*
	  DBAudio_Write - given a buf of length len, write the data to the
	  channel associated with this instance.

	  On success, the number of bytes read is returned. Otherwise
	  -1, or FAILURE, is returned, and errno is set accordingly.

	  Hopefully this function returns values in the same fashion
	  as other basic I/O functions such as read() and write()

	  Variables: 
	  count is the number of bytes written during a loop iteration
	  totalcount is the total number of bytes written
	  left_gain and right_gain are percentages used to adjust the 
	  output signal volume
	  tempbuf is a temporary pointer used in the volume operation
	  temp_chbuf is a pointer to the buffer to be written to during
	  shared memory mode.
	*/

	int DBAudio_Write(char* buf, int len)
	{
		int left_gain,right_gain;
		int left_cue_gain, right_cue_gain;
		signed short * tempbuf, *tempbuf2, * tempoutbuf;
		char * tempcharbuf;
		int incr_flag, count;
		int intindex;
		float buflen, gain1, gain2;
		int   index1,index2, output_sample;
		int i;
		int stereo_multiplier,format_multiplier;
		int tempsize;
		int sampler_flag;
		enum sampler_state_e local_sampler_state;

		/* check parameters */
		if (buf == NULL) {errno = ERROR_BAD_PARAM; return FAILURE;}
		if (ch == NULL)  {errno = ERROR_NOT_INITIALIZED; return FAILURE;}
		if (len < 0)     {errno = ERROR_BAD_PARAM; return FAILURE;}
	
		DBAudio_Handle_Message_Queue();
		
		/* remember sampler state as it may change during
		   the course of the function */
		local_sampler_state = ch->sampler_state;

		if (ch->pause)
		{
			return 0;
		}

		/* get pitch */
		local_pitch = (ch->base_pitch / 100.000) * (ch->user_pitch / 100.000);
		buflen = len / 2.0;
	
		/* calculate buffer space needed to convert the data to 
		   44.1kHz 16bit stereo*/
		switch (num_channels)
		{
			case 1:  stereo_multiplier = 2; break;
			case 2:  stereo_multiplier = 1; break;
			default: errno = ERROR_BAD_NUMCH; return FAILURE;
		}
		
		switch (format)
		{
			case AFMT_U8:     format_multiplier = 2; break;
			case AFMT_S8:     format_multiplier = 2; break;
			case AFMT_S16_LE: format_multiplier = 1; break;
			case AFMT_S16_BE: format_multiplier = 1; break;
			default: errno = ERROR_BAD_FORMAT; return FAILURE;
		}
		
		/* return error if the needed output space is greater than the
		   output buffer */
		if (ceil((buflen * (float)stereo_multiplier * 
				  (float)format_multiplier) / local_pitch) > (float)(OUTPUT_BUFSIZE))
		{
			errno = ERROR_TOO_MUCH_DATA; 
			return FAILURE;
		}
		
		ch->writing = 1;

		/* init local variables */
		intindex = 0;
		incr_flag = 0;
		sampleindex = 0;
		gain1 = gain2 = 0.0;
		sample1 = sample2 = 0.0;
		sampler_flag = 0;

		left_cue_gain = ch->cue_left_gain;
		right_cue_gain = ch->cue_right_gain;

		/* calculate gain percentages */
		if (ch->mute == TRUE)
		{
			left_gain = right_gain = 0;
		}
		else
		{
			left_gain = ch->left_gain * sysdata->left_balance;
			right_gain = ch->right_gain * sysdata->right_balance;
	
			/* cut volume if mic is being used */
			if (sysdata->talkover_enabled && !(MIC_ENABLED))
			{
				left_gain = left_gain >> DB_TALKOVER_DIVISOR_POWER;
				right_gain = right_gain >> DB_TALKOVER_DIVISOR_POWER;
			}
		}

		switch (local_sampler_state)
		{
			case SAMPLER_PLAY_SINGLE:
			case SAMPLER_PLAY_LOOP:				

				if (ch->sampler_size == 0)
				{
					ch->sampler_state = SAMPLER_OFF;
					len = 0;
					goto done;
				}
				
				/* tempsize - amount of data available in buffer to read */
				tempsize = (ch->sampler_endoffset - ch->sampler_readoffset);

				sampler_flag = 1;

				/* if we are in loop mode and loop over end of buffer, 
                   get data from start of buffer */
				if ((tempsize < len) && (local_sampler_state == SAMPLER_PLAY_LOOP))
				{
					/* copy portion at end of buffer */
					memcpy(conv_buf,(ch->sampler_buf + ch->sampler_readoffset),tempsize);
					/* copy portion at beginning of buffer */
					memcpy(conv_buf+tempsize,ch->sampler_buf+ch->sampler_startoffset,(len - tempsize));
					/* update variables */
					/* read offset is now amount to write, minus the overflow, plus the startoffset */
					ch->sampler_readoffset = len - tempsize + ch->sampler_startoffset;
					tempsize = len;
				}
				else
				{
					/* if we are in simgle play mode and out of data, reset state
                       and exit */
					if ((tempsize <= 0) && (local_sampler_state == SAMPLER_PLAY_SINGLE))
					{
						ch->sampler_state = SAMPLER_READY;
						goto done;
					}

					/* get full buffers worth of data from somewhere in middle of 
                       sampler buffer  */
					if (tempsize > len) tempsize = len;

					memcpy(conv_buf,(ch->sampler_buf + ch->sampler_readoffset),tempsize);

					ch->sampler_readoffset += tempsize;
				}


				/* update function state variables */
				buflen = (tempsize / 2);
				tempbuf = conv_buf;

				break;
			default:
				{
					/* convert input data into 44.1 KHz 16 bit stereo */
					tempbuf = (signed short *) buf;
					
					/* convert mono input to stereo */
					if (num_channels == 1)
					{
						tempbuf2 = conv_buf;
						
						if ((format == AFMT_U8) || (format == AFMT_S8))
						{
							tempcharbuf = buf;
							
							for (i = 0; i < buflen*2.0; i++)
							{
								*tempbuf2 = *tempcharbuf; tempbuf2++;
								*tempbuf2 = *tempcharbuf; tempbuf2++; tempcharbuf++;
							}
						} 
						else 
						{
							for (i = 0; i < buflen; i++)
							{
								*tempbuf2 = *tempbuf; tempbuf2++;
								*tempbuf2 = *tempbuf; tempbuf2++; tempbuf++;
							}
						}
						
						buflen *=2.0;
						tempbuf = conv_buf;
					}
					
					/* convert 8 bit input to 16 bit input */
					if ((format != AFMT_S16_LE) && (format != AFMT_S16_BE) 
						&& (format != AFMT_S16_NE))
					{
						switch (format)
						{
							case AFMT_U8:
								tempbuf2 = conv_buf;
								buflen *= 2.0;
								
								/* if data was mono, then it is already in conv_buf */
								if (num_channels == 1)
								{
									for (i = 0; i < buflen; i++)
									{
										*tempbuf = (*tempbuf2 - 127) << 8;
										tempbuf++; tempbuf2++;
									}
								}
								else
								{   /* data is 8 bit stereo, and is in buf not conv_buf*/
									tempcharbuf = buf;
									for (i = 0; i < len; i++)
									{
										*tempbuf = (*tempcharbuf - 127) << 8;
										tempbuf++; tempcharbuf++;
									}			
								}
								
								tempbuf = conv_buf;
								break;
							case AFMT_S8:
								tempbuf2 = conv_buf;
								buflen *= 2.0;
								
								/* if data was mono, then it is already in conv_buf */
								if (num_channels == 1)
								{
									for (i = 0; i < buflen; i++)
									{
										*tempbuf = *tempbuf2 << 8;
										tempbuf++; tempbuf2++;
									}
								}
								else
								{   /* data is 8 bit stereo, and is in buf not conv_buf*/
									tempcharbuf = buf;
									for (i = 0; i < len; i++)
									{
										*tempbuf = *tempcharbuf << 8;
										tempbuf++; tempcharbuf++;
									}			
								}
								
								tempbuf = conv_buf;
								break;
							default: 
								{
									errno = ERROR_BAD_FORMAT; 
									ch->writing = 0;
									return FAILURE;
								}
						}
					}	
				} /* end default case*/
		} /* end switch sampler_state */

		/* copy buffer to sample buffer if sampler state is record */
		if (local_sampler_state == SAMPLER_RECORD)
		{
			tempsize = 0;

			/* get amount of data to copy */
			if ((ch->sampler_size + (buflen * 2)) > ch->sampler_bufsize)
			{
				tempsize = ch->sampler_bufsize - ch->sampler_size;
			}
			else
			{
				tempsize = (buflen * 2);
			}

			/* change state if buffer is full */
			if (tempsize == 0)
			{
				ch->sampler_state = SAMPLER_READY;
			}
			
			/* copy data */
			memcpy(((ch->sampler_buf) + (ch->sampler_size)),tempbuf,tempsize);

			/* update sampler state variables */
			ch->sampler_size += tempsize;
			ch->sampler_endoffset = ch->sampler_size;
		}

		if ((local_pitch == 1.0) || (!(PITCH_ENABLED)))
		{
			memcpy(output_buf,tempbuf,buflen*2);
			
			outlen = buflen*2;
			tempbuf = output_buf;
			
			goto apply_gain;
		}

		/* calculate pitch shifted signal using basic linear interpolation
		   the theory is this:
		   you have two known samples, and want to calculate the value of a new sample
		   in between them.  The new sample will contain a percentage of the first sample
		   and a percentage of the second sample.  These percentages are porportional
		   to the distance between the new sample and each of the knwon samples.
		   The "position" of the new sample is determined by the float index */

		tempoutbuf = output_buf;

		while (intindex < buflen)
		{
			/* calculate sample percentages (amplitude) */
			intindex = floor(float_index);
			gain2 = float_index - intindex;
			gain1 = 1.0 - gain2;

			/* get index of first sample pair */
			intindex = intindex << 1;
			
			/* check incr_flag to see if we should be operatiing 
			   on the left or right channel sample */
			if (incr_flag) 
			{ float_index += local_pitch; incr_flag = 0; intindex++;}
			else
			{incr_flag = 1;}
			
			index1 = intindex;
			
			/* get the first "known" sample*/
			sample1 = tempbuf[index1];
			index2 = index1 + 2; 
			
			/* get the second "known" sample */
			if (index2 < (buflen))
			{sample2 = tempbuf[index2];}
			else
				/* if index2 is beyond the length of the input buffer,
                   then cheat to prevent audio pops/snaps/etc */
			{
				*tempoutbuf = sample1;
				sampleindex++;
				break;
			}
			
			/* create the new sample */
			output_sample = (((float)sample1 * gain1) + ((float)sample2 * gain2));
			
			if (output_sample > 32767)
			{output_sample = 32767;}
			if (output_sample < -32767)
			{output_sample = -32767;}
			
			*tempoutbuf = output_sample;
			tempoutbuf++;
			
			sampleindex++;
		}

		/* update global variables */
		outlen = (sampleindex-1) << 1;
		
		float_index = float_index - floor(float_index);
		
		tempbuf = output_buf;
		
		/* 	if (outlen < PIPE_BUF)
			{errno = ERROR_TOO_LITTLE_DATA; return FAILURE;} */

	apply_gain:

		/* if cueing is enabled, copy the buffer before the volume adjustment */
		if (CUE_ENABLED && ch->cue)
		{
/* 			memcpy(cue_buf,output_buf,outlen); */
			tempbuf = cue_buf;
			tempoutbuf = output_buf;

			for (i = 0; i < outlen/2; i++)
			{
				(*tempbuf) = (*tempoutbuf * left_cue_gain) >> 7;
				tempbuf++;
				tempoutbuf++;
				(*tempbuf) = (*tempoutbuf * right_cue_gain) >> 7;
				tempbuf++;
				tempoutbuf++;
			}
		}
		else
		{
			if (CUE_ENABLED)
			{
				memset(cue_buf,0,outlen);
			}
		}
	
		tempbuf = output_buf;

		/* adjust volume  - volume is adjusted here so that it is only applied 
		   to the main output buffer and not the cue buffer. If volume was 
		   adjusted in the linear interpolation loop, cue volume would be 
		   affected as well, which would be bad. */
		for (i = 0; i < outlen/2; i++)
		{
			(*tempbuf) = ((*tempbuf) * left_gain) >> 14;
			tempbuf++;
			(*tempbuf) = ((*tempbuf) * right_gain) >> 14;
			tempbuf++;
		}
		
		/* write main and cue output buffers */
		{
			int numbytes;
			char * write_cue_buf;
			char * write_output_buf; 

			write_cue_buf = (char *) cue_buf;
			write_output_buf = (char *) output_buf;

			while(outlen > 0)
			{				
				if(outlen > DB_BUFSIZE_CHAR)
				{
					numbytes = DB_BUFSIZE_CHAR;
				}
				else
				{
					numbytes = outlen;
				}
				
				outlen -= numbytes;
				
				if (CUE_ENABLED)
				{
					if(ch->cue)
					{
						if ((count = write(ch->client_cue_fd,write_cue_buf,numbytes)) < 0)
						{
							if (errno != 11)
							{
								Error("DBAudioLib: DBAudio_Write: error %d on cue pipe write.",errno);
							}
						}
					}
				}
		
				/* 	printf("numbytes: %d  buflen %f\n",numbytes,buflen); */
				
				if ((count = write(ch->client_comm_fd,write_output_buf,numbytes)) < 0)
				{
					Error("DBAudioLib: DBAudio_Write: Error on pipe write");
				}

				write_cue_buf += numbytes;
				write_output_buf += numbytes;
			}
		}
	
	done:

		ch->writing = 0;

		if (sampler_flag)
		{
			return 0;
		}
		else
		{
			return len;
		}
	}

#ifdef __cplusplus
}
#endif

