/*
 * Author: Andrei Zavada <johnhommer@gmail.com>
 *
 * License: GPL-2+
 *
 * Initial version: 2010-02-12
 *
 * CNModel runner (interpreter)
 */


#include <stdio.h>

#include "config.h"
#ifdef HAVE_LIBREADLINE
#  if defined(HAVE_READLINE_READLINE_H)
#    include <readline/readline.h>
#  elif defined(HAVE_READLINE_H)
#    include <readline.h>
#  endif /* !defined(HAVE_READLINE_H) */
#endif /* HAVE_LIBREADLINE */

#ifdef HAVE_READLINE_HISTORY
#  if defined(HAVE_READLINE_HISTORY_H)
#    include <readline/history.h>
#  elif defined(HAVE_HISTORY_H)
#    include <history.h>
#  endif
#endif /* HAVE_READLINE_HISTORY */

#include "runner.hh"
#include "libcn/model.hh"

using namespace std;
using namespace Stilton;




static char*
cnrun_null_generator( const char* text, int state)
{
//	printf( "No completion in this context\n");
	return nullptr;
}


static char*
cnrun_cmd_generator( const char* text, int state)
{
	static int list_index, len;
        const char *name;

        if ( !state ) {
		list_index = 0;
		len = strlen( text);
        }

        while ( (name = cnrun_cmd[list_index]) ) {
		list_index++;
		if ( strncmp( name, text, len) == 0 )
			return strdup( name);
        }
        return nullptr;
}

static char*
cnrun_source_types_generator( const char* text, int state)
{
	static int list_index, len;
        const char *name;

        if ( !state ) {
		list_index = 0;
		len = strlen( text);
        }

        while ( (name = __SourceTypes[list_index]) ) {
		list_index++;
		if ( strncmp( name, text, len) == 0 )
			return strdup( name);
        }
        return nullptr;
}







static char*
cnrun_neu_type_generator( const char *text, int state)
{
	static const char** neuron_types = nullptr;
	if ( !neuron_types ) {
		if ( !(neuron_types = (const char**)malloc( (NT_LAST - NT_FIRST+1+1)*sizeof(char*))) )
			abort();
		size_t n;
		for ( n = 0; n <= NT_LAST - NT_FIRST; n++ )
			neuron_types[n] = strdup( __CNUDT[NT_FIRST+n].species);  // family would do just as well
		neuron_types[n] = nullptr;
	}

	static int list_index, len;
        const char *name;
        if ( !state ) {
		list_index = 0;
		len = strlen( text);
        }
        while ( (name = neuron_types[list_index]) ) {
		list_index++;
		if ( strncmp( name, text, len) == 0 )
			return strdup( name);
        }
        return nullptr;
}



static char*
cnrun_syn_type_generator( const char *text, int state)
{
	static const char** synapse_types = nullptr;
	if ( !synapse_types ) {
		if ( !(synapse_types = (const char**)malloc( (YT_LAST - YT_FIRST+1+1)*sizeof(char*))) )
			abort();
		size_t n, i;
		for ( n = i = 0; n <= YT_LAST - YT_FIRST; n++ )
			synapse_types[i++] = strdup( __CNUDT[YT_FIRST+n].family);
		// there are fewer families than species, so we are wasting some tens of bytes here.  oh well.
		synapse_types[i] = nullptr;
	}

	static int list_index, len;
        const char *name;
        if ( !state ) {
		list_index = 0;
		len = strlen( text);
        }
        while ( (name = synapse_types[list_index]) ) {
		list_index++;
		if ( strncmp( name, text, len) == 0 )
			return strdup( name);
        }
        return nullptr;
}





bool regenerate_unit_labels = true;

#define GENERATE_NEURONS  1
#define GENERATE_SYNAPSES 2
static int restrict_generated_set = 0;

static char*
cnrun_unit_label_generator( const char *text, int state)
{
	static int list_index, len;
        const char *name;

	static char** unit_labels = nullptr;

	if ( regenerate_unit_labels ) {
		regenerate_unit_labels = false;

		if ( !Model ) {
			free( unit_labels);
			unit_labels = nullptr;
			return nullptr;
		}

		if ( !(unit_labels = (char**)realloc( unit_labels, (Model->units()+1) * sizeof(char*))) )
			abort();
		size_t n = 0;
		for_model_units (Model, U)
			if ( ((restrict_generated_set & GENERATE_NEURONS) && (*U)->is_neuron()) ||
			     ((restrict_generated_set & GENERATE_SYNAPSES) && (*U)->is_synapse()) )
				unit_labels[n++] = strdup( (*U) -> label());
		unit_labels[n] = nullptr;
	}

	if ( !unit_labels )
		return nullptr;

        if ( !state ) {
		list_index = 0;
		len = strlen( text);
        }
        while ( (name = unit_labels[list_index]) ) {
		list_index++;
		if ( strncmp( name, text, len) == 0 )
			return strdup( name);
        }
        return nullptr;
}



bool regenerate_var_names = true;

static char*
cnrun_var_names_generator( const char *text, int state)
{
	static int list_index, len;
        const char *name;

	static char** var_names = nullptr;

	if ( regenerate_var_names ) {
		regenerate_var_names = false;

		if ( current_shell_variables->size() == 0 )
			return nullptr;

		if ( !(var_names = (char**)realloc( var_names, (current_shell_variables->size()+1) * sizeof(char*))) )
			abort();
		size_t n = 0;
		for ( auto &v : *current_shell_variables )
			var_names[n++] = strdup( v.name);
		var_names[n] = nullptr;
	}

	if ( !var_names )
		return nullptr;

        if ( !state ) {
		list_index = 0;
		len = strlen( text);
        }
        while ( (name = var_names[list_index]) ) {
		list_index++;
		if ( strncmp( name, text, len) == 0 )
			return strdup( name);
        }
        return nullptr;
}





bool regenerate_source_ids = true;

static char*
cnrun_source_id_generator( const char *text, int state)
{
	static int list_index, len;
        const char *name;

	static char** source_ids = nullptr;

	if ( regenerate_source_ids ) {
		regenerate_source_ids = false;

		if ( !Model || Model->Sources.size() == 0 )
			return nullptr;

		if ( !(source_ids = (char**)realloc( source_ids, (Model->Sources.size()+1) * sizeof(char*))) )
			abort();
		size_t n = 0;
		for ( auto &v : Model->Sources )
			source_ids[n++] = strdup( v->name.c_str());
		source_ids[n] = nullptr;
	}

	if ( !source_ids )
		return nullptr;

        if ( !state ) {
		list_index = 0;
		len = strlen( text);
        }
        while ( (name = source_ids[list_index]) ) {
		list_index++;
		if ( strncmp( name, text, len) == 0 )
			return strdup( name);
        }
        return nullptr;
}




static char **parm_names = nullptr;
static char *unit_label_completing_for = nullptr;
static char *synapse_target_label_completing_for = nullptr;

static char*
cnrun_parm_names_generator( const char *text, int state)
{
	static int list_index, len;
        const char *name;

	if ( !Model )
		return nullptr;
	C_BaseSynapse *y;
	TUnitType t;
	C_BaseUnit *u1, *u2;
	if ( synapse_target_label_completing_for )
		if ( (u1 = Model->unit_by_label( unit_label_completing_for)) && u1->is_neuron() &&
		     (u2 = Model->unit_by_label( synapse_target_label_completing_for)) && u2->is_neuron() &&
		     (y = (static_cast<C_BaseNeuron*>(u1)) -> connects_via( *static_cast<C_BaseNeuron*>(u2))) )
			t = y->type();
		else
			return nullptr;
	else
		t = Model -> unit_by_label( unit_label_completing_for) -> type();
	if ( t == NT_VOID )
		return nullptr;

	if ( !(parm_names = (char**)realloc( parm_names, (__CNUDT[t].pno+1) * sizeof(char*))) )
		abort();
	size_t n, p;
	for ( n = p = 0; p < __CNUDT[t].pno; p++ )
		if ( __cn_verbosely > 5 || __CNUDT[t].stock_param_syms[p][0] != '.' )
			parm_names[n++] = strdup( __CNUDT[t].stock_param_syms[p]);
	parm_names[n] = nullptr;

	if ( !parm_names )
		return nullptr;

        if ( !state ) {
		list_index = 0;
		len = strlen( text);
        }
        while ( (name = parm_names[list_index]) ) {
		list_index++;
		if ( strncmp( name, text, len) == 0 )
			return strdup( name);
        }
        return nullptr;
}




static int
rl_point_at_word() __attribute__ ((pure));

static int
rl_point_at_word()
{
	int p = 0, delims = 0;
	while ( p < rl_point ) {
		if ( isspace(rl_line_buffer[p]) ) {
			delims++;
			do p++;
			while ( p < rl_point && isspace(rl_line_buffer[p]) );
		}
		p++;
	}
	return delims;
}



char**
cnrun_completion( const char *text, int start, int end)
{
	if ( start == 0 )
		return rl_completion_matches( text, &cnrun_cmd_generator);

	char	*line_buffer = strdupa( rl_line_buffer),
	 	*cmd = strtok( line_buffer, " \t");

	if ( strcmp( cmd, cnrun_cmd[CNCMD_add_neuron]) == 0 ) {
		switch ( rl_point_at_word() ) {
		case 1:   return rl_completion_matches( text, &cnrun_neu_type_generator);
		default:  return rl_completion_matches( text, &cnrun_null_generator);
		}

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_add_synapse]) == 0 ) {
		switch ( rl_point_at_word() ) {
		case 1:  return rl_completion_matches( text, &cnrun_syn_type_generator);
		case 2:
		case 3:  return (restrict_generated_set = 0|GENERATE_NEURONS,
				 rl_completion_matches( text, &cnrun_unit_label_generator));
		default: return rl_completion_matches( text, &cnrun_null_generator);
		}

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_load_nml]) == 0 ) {
		return nullptr; // use built-in filename completion

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_show_units]) == 0 ||
		    strcmp( cmd, cnrun_cmd[CNCMD_decimate]) == 0 ||
		    strcmp( cmd, cnrun_cmd[CNCMD_start_listen]) == 0 ||
		    strcmp( cmd, cnrun_cmd[CNCMD_stop_listen]) == 0 ||
		    strcmp( cmd, cnrun_cmd[CNCMD_start_log_spikes]) == 0 ||
		    strcmp( cmd, cnrun_cmd[CNCMD_stop_log_spikes]) == 0 ||
		    strcmp( cmd, cnrun_cmd[CNCMD_putout]) == 0 ) {
		return (rl_point_at_word() == 1) ? (restrict_generated_set = 0|GENERATE_NEURONS|GENERATE_SYNAPSES,
						    rl_completion_matches( text, &cnrun_unit_label_generator)) : nullptr;

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_show_vars])  == 0 ||
		    strcmp( cmd, cnrun_cmd[CNCMD_clear_vars]) == 0 ||
		    strcmp( cmd, cnrun_cmd[CNCMD_listen_dt])  == 0 ||
		    strcmp( cmd, cnrun_cmd[CNCMD_integration_dt_min]) == 0 ||
		    strcmp( cmd, cnrun_cmd[CNCMD_integration_dt_max]) == 0 ||
		    strcmp( cmd, cnrun_cmd[CNCMD_integration_dt_cap]) == 0 ) {
		return (rl_point_at_word() == 1) ? rl_completion_matches( text, cnrun_var_names_generator) : nullptr;

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_set_parm_neuron]) == 0 ) {
		switch ( rl_point_at_word() ) {
		case 1:	restrict_generated_set = 0|GENERATE_NEURONS;
			return rl_completion_matches( text, cnrun_unit_label_generator);
		case 2:	unit_label_completing_for = strtok( nullptr, " ");
			synapse_target_label_completing_for = nullptr;
			return rl_completion_matches( text, cnrun_parm_names_generator);
		default: return rl_completion_matches( text, cnrun_var_names_generator);
		}

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_set_parm_synapse]) == 0 ) {
		switch ( rl_point_at_word() ) {
		case 1:
		case 2:	restrict_generated_set = 0|GENERATE_NEURONS;
			return rl_completion_matches( text, cnrun_unit_label_generator);
		case 3:	unit_label_completing_for = strtok( nullptr, " ");
			synapse_target_label_completing_for = strtok( nullptr, " ");
			return rl_completion_matches( text, cnrun_parm_names_generator);
		default: return rl_completion_matches( text, cnrun_var_names_generator);
		}

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_connect_source]) == 0 ) {
		switch ( rl_point_at_word() ) {
		case 1:	return rl_completion_matches( text, &cnrun_source_id_generator);
		case 2:	restrict_generated_set = 0|GENERATE_NEURONS|GENERATE_SYNAPSES;
			return rl_completion_matches( text, &cnrun_unit_label_generator);
		case 3:	unit_label_completing_for = (strtok( nullptr, " "), strtok( nullptr, " "));
			synapse_target_label_completing_for = nullptr;
			return rl_completion_matches( text, cnrun_parm_names_generator);
		default: return rl_completion_matches( text, cnrun_null_generator);
		}

	} else if ( strcmp( cmd, cnrun_cmd[CNCMD_new_source]) == 0 ) {
		switch ( rl_point_at_word() ) {
		case 1:  return rl_completion_matches( text, cnrun_source_types_generator);
		default: return rl_completion_matches( text, cnrun_null_generator);
		}

	} else {
		return nullptr;
	}
}




// EOF
