/*
 * Author: Andrei Zavada <johnhommer@gmail.com>
 *         building on original work by Thomas Nowotny <tnowotny@ucsd.edu>
 *
 * License: GPL-2+
 *
 * Initial version: 2008-08-02
 *
 */



#ifndef LIBCN_STANDALONE_SYNAPSES_H
#define LIBCN_STANDALONE_SYNAPSES_H

#include <iostream>

#include "base-synapse.hh"
#include "base-neuron.hh"
#include "standalone-attr.hh"
#include "mx-attr.hh"

#include "config.h"


namespace CNRun {

class CModel;

class C_StandaloneSynapse
  : public C_BaseSynapse, public C_StandaloneAttributes {

    private:
	C_StandaloneSynapse();
    protected:
	C_StandaloneSynapse( TUnitType intype, C_BaseNeuron *insource, C_BaseNeuron *intarget,
			     double ing, CModel* inM, int s_mask = 0);

    public:
       ~C_StandaloneSynapse();

	double &var_value( size_t v)			{ return V[v]; }
	const double &get_var_value( size_t v) const	{ return V[v]; }
	double  S() const				{ return V[0]; }
	double &S( vector<double>&)			{ return V[0];  }

	void reset_vars()
		{
			memcpy( V.data(), __CNUDT[_type].stock_var_values,
				sizeof(double) * v_no());
			memcpy( V_next.data(), __CNUDT[_type].stock_var_values,
				sizeof(double) * v_no());
		}
};





class CSynapseMap
  : public C_StandaloneSynapse {

    private:
	CSynapseMap();

    public:
	static constexpr double fixed_dt = 0.1;

	CSynapseMap( C_BaseNeuron *insource, C_BaseNeuron *intarget,
		     double ing, CModel *inM, int s_mask = 0, TUnitType alt_type = YT_MAP);

	void preadvance();  // defined inline in model.h

	enum {
		_tau_, _delta_, _Esyn_
	};
	double Isyn( const C_BaseNeuron &with_neuron, double g) const
		{
			return -g * S() * (with_neuron.E() - P[_Esyn_]);
		}
	double Isyn( vector<double>& unused, const C_BaseNeuron &with_neuron, double g) const
		{
			return Isyn( with_neuron, g);
		}

    protected:
	bool _source_was_spiking;
};





class CSynapseMxMap
  : public CSynapseMap, public C_MultiplexingAttributes {

    public:
	static constexpr double fixed_dt = 0.1;

	CSynapseMxMap( C_BaseNeuron *insource, C_BaseNeuron *intarget,
		       double ing, CModel *inM, int s_mask = 0)
	      : CSynapseMap( insource, intarget, ing, inM, s_mask, YT_MXMAP)
		{}

	enum {
		_tau_, _delta_, _Esyn_
	};
	void preadvance();  // defined inline in model.h

    private:
	friend class CModel;
	void update_queue();
};

}

#endif

// EOF
