;;;
;;; Numerical syntax "torture test"
;;;
;;; This tries to test a lot of edge cases in Scheme's numerical syntax.
;;;
;;; Output is written so that if you run it through "grep ERROR" it will
;;; output nothing (and exit status will be nonzero) if there are no errors.
;;; If you run it through "tail -n 1" you will just get the total error summary.
;;;
;;; This code assumes that string->number accepts numbers with embedded radix
;;; specifiers (R5RS mentions that it's allowed to return #f in those cases).
;;; It also doesn't try to support Schemes which support *only* integers or
;;; *only* flonums (which is also allowed by R5RS).
;;;

(use srfi-1 ports)

(define the-nan (fp/ 0.0 0.0))
(define pos-inf (fp/ 1.0 0.0))
(define neg-inf (fp/ -1.0 0.0))
(define neg-zero (/ -1.0 +inf.0))

(define (nan? x) (and (number? x) (not (= x x))))

(define total-errors 0)

(define (check-string-against-values! str . possible-values)
  (define (none? pred) (not (any pred possible-values)))
  (let ((res (string->number str)))
    (cond
     ((none? (lambda (value)
               (or (and (not (string? value)) (equal? res value))
                   (and res (nan? res) (or (and value (nan? value)))))))
      (display "PARSE ERROR         ")
      (write (cons str possible-values))
      (display " => ") (write res) (newline)
      (set! total-errors (+ total-errors 1)))
     ((let ((re-str (and res (number->string res))))
        (and (none? (lambda (value)
                      (or (and res (string=? re-str str))
                          (and (not res) (not value))
                          (and res (string? value) (string=? re-str value)))))
             re-str))
      => (lambda (re-str)
          (display "SERIALIZATION ERROR ")
          (write (cons str possible-values))
          (display " => ") (write re-str) (newline)
          (set! total-errors (+ total-errors 1))))
     ((handle-exceptions exn
        (and res exn)
        (let ((re-read (with-input-from-string str read)))
          (and (not (symbol? re-read))
               (not (eof-object? re-read))
               (or (not res)
                   (and (not (and (nan? res) (nan? re-read)))
                        (not (equal? res re-read))))
               re-read)))
      => (lambda (obj)
           (display (if (condition? obj)
                        "READBACK EXN ERROR  "
                        "READBACK ERROR      "))
           (write (cons str possible-values))
           (display " => ")
           (if (condition? obj)
               (write ((condition-property-accessor 'exn 'message #f) obj))
               (write obj))
           (newline)
           (set! total-errors (+ total-errors 1))))
     ((let ((written&read (with-input-from-string (with-output-to-string
                                                    (lambda () (write res)))
                            read)))
        (and (not (or (and (nan? res) (nan? written&read))
                      (equal? res written&read)))
             written&read))
      => (lambda (read-back)
           (display "R/W VARIANCE ERROR  ")
           (write (cons str possible-values))
           (display " => ")
           (write read-back) (newline)
           (set! total-errors (+ total-errors 1))))
     (else (display "OK                  ")
           (write (cons str possible-values))
           (newline)))))

(define-syntax test-numbers
  (syntax-rules ()
    ((_ (str value ...) rest ...)
     (begin
       (check-string-against-values! str value ...)
       (test-numbers rest ...)))
    ((_ "no-totals") #f)
    ((_ x rest ...)
     (begin (newline) (display "-> ") (display x) (newline)
            (display "-----------------------------------------------------")
            (newline)
            (test-numbers rest ...)))
    ((_)
     (if (= 0 total-errors)
         (begin (newline)
                (display "-----> Everything OK, no errors!")
                (newline))
         (begin (newline)
                (display "-----> TOTAL ERRORS: ")
                (display total-errors)
                (newline)
                (error total-errors))))))

(test-numbers
 "Simple integers"
 ("1" 1)
 ("+1" 1 "1")
 ("-1" (- 1))
 ("#i1" 1.0 "1.0" "1.")
 ("#I1" 1.0 "1.0" "1.")
 ("#i-1" (- 1.0) "-1.0" "-1.")
 ("123\x00456" #f)
 ("-#i1" #f)
 ("+-1" #f)
 ("" #f)
 ("-" #f)
 ("+" #f)
 ("+-" #f)

 "Basic decimal notation"
 ("1.0" (exact->inexact 1) "1.")
 ("1." 1.0 "1.0" "1.")
 ("1.#" 1.0 1.5 "1.0" "1." "1.5")
 (".1" 0.1 "0.1" "100.0e-3")
 ("-.1" (- 0.1) "-0.1" "-100.0e-3")
 ;; Some Schemes don't allow negative zero. This is okay with the standard
 ("-.0" -0.0 "-0." "-0.0" "0.0" "0." ".0")
 ("-0." -0.0 "-.0" "-0.0" "0.0" "0." ".0")
 ("." #f)
 (".1." #f)
 ("..1" #f)
 ("1.." #f)
 ("#i1.0" 1.0 "1.0" "1.")
 ("#e1.0" 1 "1")
 ("#e-.0" 0 "0")
 ("#e-0." 0 "0")
 ("-#e.0" #f)

 "Decimal notation with padding"
 ("1#" 10.0 15.0 "10.0" "15.0" "10." "15.")
 ("#e1#" 10 15 "10" "15")
 ("#E1#" 10 15 "10" "15")
 ("#1" #f)
 ("#" #f)
 ("1#2" #f)
 ("1.#2" #f)
 (".#" #f)
 ("#.#" #f)
 ("#.1" #f)
 ("1#.2" #f)
 ("1#." 10.0 15.0 "10.0" "15.0" "10." "15.")

 "Decimal notation with suffix"
 ("1e2" 100.0 "100.0" "100.")
 ("1E2" 100.0 "100.0" "100.")
 ("1s2" 100.0 "100.0" "100.")
 ("1S2" 100.0 "100.0" "100.")
 ("1f2" 100.0 "100.0" "100.")
 ("1F2" 100.0 "100.0" "100.")
 ("1d2" 100.0 "100.0" "100.")
 ("1D2" 100.0 "100.0" "100.")
 ("1l2" 100.0 "100.0" "100.")
 ("1L2" 100.0 "100.0" "100.")
 ("1e2e3" #f)
 ("1e2s3" #f)
 ("1e2.0" #f)

 "Decimal notation with suffix and padding"
 ("1#e2" 1000.0 1500.0 "1000.0" "1500.0" "1000." "1500." "1.0e3" "15.0e2")
 ("1e2#" #f)

 "NaN, Inf, negative zero"
 ("+nan.0" the-nan "+NaN.0")
 ("+NAN.0" the-nan "+nan.0" "+NaN.0")
 ("+nan.1" #f)
 ("+nan.01" #f)
 ("+inf.0" pos-inf "+Inf.0")
 ("+InF.0" pos-inf "+inf.0" "+Inf.0")
 ("-inf.0" neg-inf "-Inf.0")
 ("-iNF.0" neg-inf "-inf.0" "-Inf.0")
 ("+inf.01" #f)
 ("+inf.1" #f)
 ("-inf.01" #f)
 ("-inf.1" #f)
 ("+inf.0/1" #f)
 ("1/+inf.0" #f)
 ;; Thanks to John Cowan for these
 ("#e+nan.0" #f)
 ("#e+inf.0" #f)
 ("#e-inf.0" #f)
 ("#i+nan.0" the-nan "+nan.0" "+NaN.0")
 ("#i+inf.0" pos-inf "+inf.0" "+Inf.0")
 ("#i-inf.0" neg-inf "-inf.0" "-Inf.0")
 ("-0.0" neg-zero "-.0" "-0.")
 ;; These used to be accepted but are invalid
 ("+nan" #f)
 ("+inf" #f)
 ("-inf" #f)
 ("nan.0" #f)
 ("inf.0" #f)

 "Fractions"
 ("1/2" (/ 1 2) "0.5" ".5" "500.0e-3")
 ("#e1/2" #f)
 ("10/2" 5.0 "5.0" "5.")
 ("#i10/2" 5.0 "5.0" "5.")
 ("-1/2" (- (/ 1 2)) "-0.5" "-.5" "-500.0e-3")
 ("1/-2" #f)
 ("10/0" +inf.0 "+inf.0")
 ("0/10" 0.0 "0.0" "0.")
 ("#e0/10" 0 "0")
 ("#e1#/2" 5 "5")
 ("#e1/2#" #f)
 ("1.0/2" #f)
 ("1/2.0" #f)
 ("1/2e2" #f)
 ("1/2e2" #f)
 ("1#/2" 5.0 7.5 "5.0" "5." "7.5")
 ("1/2#" 0.05 "0.05" ".05" "50.0e-3" "5.e-002")
 ("#i3/2" (/ 3.0 2.0) "1.5")
 ("1#/#" #f)
 ("1/" #f)
 ("1/+" #f)
 ("+/1" #f)
 ("/1" #f)
 ("/" #f)
 ("#i1/0" pos-inf "+inf.0" "+Inf.0")
 ("#i-1/0" neg-inf "-inf.0" "-Inf.0")
 ("#i0/0" the-nan "+nan.0" "+NaN.0")
 ;; This _could_ be valid (but isn't as pretty)
 ;("#i1/0" #f)
 ;("#i-1/0" #f)
 ;("#i0/0" #f)
 
 "Some invalid complex numbers syntax (not supported at all)"
 ("2i" #f)
 ("+-i" #f)
 ("i" #f)
 ("1+2i1" #f)
 ("1+2" #f)
 ("1#+#i" #f)

 "Base prefixes"
 ("#x11" 17 "17")
 ("#X11" 17 "17")
 ("#d11" 11 "11")
 ("#D11" 11 "11")
 ("#o11" 9 "9")
 ("#O11" 9 "9")
 ("#b11" 3 "3")
 ("#B11" 3 "3")
 ("#da1" #f)
 ("#o8" #f)
 ("#b2" #f)
 ("#o7" 7 "7")
 ("#xa" 10 "10")
 ("#xA" 10 "10")
 ("#xf" 15 "15")
 ("#xg" #f)
 ("#x-10" -16 "-16")
 ("#d-10" -10 "-10")
 ("#o-10" -8 "-8")
 ("#b-10" -2 "-2")
 ("-#x10" #f)
 ("-#d10" #f)
 ("-#o10" #f)
 ("-#b10" #f)
 ("#x-" #f)
 ("#x" #f)
 ("#d" #f)
 ("#d-" #f)
 ("#d+" #f)
 ("#o" #f)
 ("#o-" #f)
 ("#b" #f)
 ("#b-" #f)
 ("#e" #f)
 ("#e-" #f)
 ("#i" #f)
 ("#i-" #f)

 "Combination of prefixes"
 ("#x#x11" #f)
 ("#x#b11" #f)
 ("#b#o11" #f)
 ("#e#x10" 16 "16")
 ("#i#x10" 16.0 "16.0" "16.")
 ("#e#e10" #f)
 ("#e#e#x10" #f)
 ("#E#e#X10" #f)
 ("#i#e#x10" #f)
 ("#e#x#e10" #f)
 ("#x#x#e10" #f)
 ("#x#e#x10" #f)

 "Base prefixes with padding"
 ("#x1#0" #f)
 ("#d1#0" #f)
 ("#o1#0" #f)
 ("#b1#0" #f)
 ("#x1#" 16.0 24.0 "16.0" "24.0" "16." "24.")
 ("#d1#" 10.0 15.0 "10.0" "15.0" "10." "15.")
 ("#o1#" 8.0 12.0 "8.0" "12.0" "8." "12.")
 ("#b1#" 2.0 3.0 "2.0" "3.0" "2." "3.")

 "(Attempted) decimal notation with base prefixes"
 ("#x1.0" #f)
 ("#d1.0" 1.0 "1.0" "1.")
 ("#o1.0" #f)
 ("#b1.0" #f)
 ("#x1.#" #f)
 ("#d1.#" 1.0 1.5 "1.0" "1.5" "1.")
 ("#o1.#" #f)
 ("#b1.#" #f)
 ("#x1." #f)
 ("#d1." 1.0 "1.0" "1.")
 ("#o1." #f)
 ("#b1." #f)
 ("#x.1" #f)
 ("#d.1" 0.1 "0.1" ".1" "100.0e-3")
 ("#o.1" #f)
 ("#b.1" #f)
 ("#x1e2" 482 "482")
 ("#d1e2" 100.0 "100.0" "100.")
 ("#o1e2" #f)
 ("#b1e2" #f)

 "Fractions with prefixes"
 ("#x10/2" 8.0 "8.0" "8.")
 ("#x11/2" 8.5 "8.5")
 ("#d11/2" 5.5 "5.5")
 ("#o11/2" 4.5 "4.5")
 ("#b11/10" 1.5 "1.5")
 ("#b11/2" #f)
 ("#x10/#o10" #f)
 ("10/#o10" #f)
 ("#x1#/2" 8.0 12.0 "8.0" "8." "12.0" "12.")
 ("#d1#/2" 5.0 7.5 "5.0" "5." "7.5")
 ("#o1#/2" 4.0 6.0 "4.0" "4." "6.0" "6.")
 ("#b1#/2" #f)
 ("#b1#/10" 1.0 1.5 "1.0" "1." "1.5")
 )

;; #1272 - Bases not in [2,36] throw errors.
(let ((check-base (lambda (b)
                    (string->number "123" b)
                    (error "No error on invalid base" b))))
  (condition-case (check-base 1)  ((exn type) 'ok))
  (condition-case (check-base 37) ((exn type) 'ok)))
