/* Copyright (c) 2021 Michael Baeuerle
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY
 * SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 * SPDX-License-Identifier: ICU
 */

/* Execute authentication according to RFC 8315 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libcanlock-3/canlock.h>
#include <libcanlock-3/canlock-hp.h>


/*
 * To extract the fields from the header the function cl_hp_get_field() can
 * be used.
 *
 * The following header fields are already unfolded.
 * The function cl_hp_unfold_field() can be used if this is not the case.
 */

/* Body of the Cancel-Key header field of a cancel or supersede article */
static const char *keys =
   "Cancel-Key: sha1:xRi1jfCrgA6kCyGbk4ZUhddVdT0= (does not match) "
   "sha256:Vy+nHHZZybqxFC/uWmes153KPUpt/FoAMmMij8Ks2dM=";

/* Body of the Cancel-Lock header field of the target article */
static const char *locks =
   "Cancel-Lock: sha1:NkCyGbk4ZUhgExRi1jhCrgAVdT0= "
   "sha256:x0JAxz+Ln+ZB2pC+uPKvcka6kfK5ojebnVxEosgOujQ=";


int error_handler(const char *msg)
{
   fprintf(stderr, "Error: %s\n", msg);
   exit(EXIT_FAILURE);
}


/* Create an (initial) Cancel-Lock header field */
int main(void)
{
   char *parsed_keys, *parsed_locks;

   /* Parse header field bodies */
   parsed_keys = cl_hp_parse_field(keys, strlen(keys));
   if (NULL == parsed_keys)
      error_handler("Parser failed");
   parsed_locks = cl_hp_parse_field(locks, strlen(locks));
   if (NULL == parsed_locks)
   {
      free(parsed_keys);
      error_handler("Parser failed");
   }

   printf("Keys : %s\n", parsed_keys);
   printf("Locks: %s\n", parsed_locks);

   /*
    * Verify whether at least one key matches a lock
    *
    * No reject list will accept all hash algorithms supported by
    * libcanlock for <scheme>.
    */
   {
      int failed = cl_verify_multi(NULL, parsed_keys, parsed_locks);
      if (failed)
      {
         free(parsed_locks);
         free(parsed_keys);
         error_handler("Authentication failed");
      }
   }

   printf("Good\n");

   /* Release memory allocated by libcanlock-hp */
   free(parsed_locks);
   free(parsed_keys);

   exit(EXIT_SUCCESS);
}


/* EOF */
